use strict;
use warnings;
use lib qw(lib);

use Wurm qw(mob let);
use Wurm::Grub::REST;
use Data::UUID;
use Encode;
use JSON::XS;
use Tenjin;

my $grub = Wurm::Grub::REST->new
->get(
  sub {
    my $meal = shift;

    my $text = $meal->mind->{meld}{$meal->grit->{id}};
    return
      unless defined $text;

    $meal->grit->{text} = $text;
    return $meal->vent->{json}
      ? to_json($meal)
      : to_html($meal, 'item.html')
    ;
  },
  sub {
    my $meal = shift;

    $meal->grit->{items} = $meal->mind->{meld};
    return $meal->vent->{json}
      ? to_json($meal)
      : to_html($meal, 'index.html')
    ;
  },
)
->post(sub {
  my $meal = shift;

  my $text = $meal->req->parameters->{text};
  return
    unless defined $text;

  $meal->grit->{id} = gen_uuid($meal);
  $meal->mind->{meld}{$meal->grit->{id}} = $text;
  return Wurm::_302($meal->env->{PATH_INFO});
})
->patch(sub {
  my $meal = shift;

  return
    unless exists $meal->mind->{meld}{$meal->grit->{id}};

  my $text = $meal->req->parameters->{text};
  return
    unless defined $text;

  $meal->mind->{meld}{$meal->grit->{id}} = $text;
  return Wurm::_204;
})
->delete(sub {
  my $meal = shift;

  return
    unless exists $meal->mind->{meld}{$meal->grit->{id}};

  delete $meal->mind->{meld}{$meal->grit->{id}};
  return Wurm::_204;
})
;

sub gen_uuid {
  my $meal = shift;
  my $uuid = $meal->mind->{uuid};
  return lc $uuid->to_string($uuid->create);
}

sub to_html {
  my $meal = shift;
  my $file = shift;
  my $html = $meal->mind->{html}->render($file, $meal->grit);
  return Wurm::_200('text/html', Encode::encode('UTF-8', $html));
}

sub to_json {
  my $meal = shift;
  my $json = $meal->mind->{json}->encode($meal->grit);
  return Wurm::_200('application/json', $json);
}

my $mind = {
  meld => {'Wurm::Grub::REST' => 'Hello, Wurm!'},
  uuid => Data::UUID->new,
  json => JSON::XS->new->utf8,
  html => Tenjin->new({
    path => ['./examples/html/rest'], strict => 1, cache => 0
  }),
};

my $app = Wurm::wrapp($grub->molt, $mind);
$app
