package Bot::Backbone::Service::Role::Chat;
BEGIN {
  $Bot::Backbone::Service::Role::Chat::VERSION = '0.112320';
}
use v5.10;
use Moose::Role;

# ABSTRACT: Chat services must implement this role


has chat_consumers => (
    is          => 'ro',
    isa         => 'ArrayRef', # 'ArrayRef[DOES Bot::Backbone::Service::Role::ChatConsumer]',
    required    => 1,
    default     => sub { [] },
    traits      => [ 'Array' ],
    handles     => {
        register_chat_consumer => 'push',
        list_chat_consumers    => 'elements',
    },
);


requires qw( send_reply send_message );


sub resend_message {
    my ($self, $message) = @_;

    for my $consumer ($self->list_chat_consumers) {
        $consumer->receive_message($message);
    }
}

1;

__END__
=pod

=head1 NAME

Bot::Backbone::Service::Role::Chat - Chat services must implement this role

=head1 VERSION

version 0.112320

=head1 DESCRIPTION

A chat service is one that sends and receives messages to other entities.

See L<Bot::Backbone::Service::ConsoleChat> and
L<Bot::Backbone::Service::JabberChat>.

=head1 ATTRIBUTES

=head2 chat_consumers

This is a list of L<Bot::Backbone::Service::Role::ChatConsumer>s that hvae registered to
receive messages from this chat service. A chat consumer is registered using the
C<register_chat_cnosumer> method. A C<list_chat_consumers> method is provided to
list the registered consumers.

=head1 REQUIRED METHODS

=head2 send_reply

  $chat->send_reply($message, $text);

Given a message generated by this chat service, this should send a reply to the
origin of the message, whether that be a group or individual or other entity.

=head2 send_message

  # Send a direct message
  $chat->send_message(
      to   => $to_username,
      text => 'blah blah blah',
  );

  # Send a group message
  $chat->send_message(
      group => $to_group,
      text  => 'blah blah blah',
  );

Sends a message to a group or individual using this chat service. This is used
when the message is not being made as a direct reply to a message received from
the chat service.

=head1 METHODS

=head2 resend_message

  $chat->resend_message($message);

This should be called whenever a message is received from the chat service. This
message willb e forwarded to all of the registered L</chat_consumers>.

=head1 AUTHOR

Andrew Sterling Hanenkamp <hanenkamp@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2011 by Qubling Software LLC.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut

