/*
 * Decompiled with CFR 0.152.
 */
package org.apache.logging.log4j.layout.template.json.resolver;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import org.apache.logging.log4j.layout.template.json.resolver.TemplateResolver;
import org.apache.logging.log4j.layout.template.json.resolver.TemplateResolverConfig;
import org.apache.logging.log4j.layout.template.json.resolver.TemplateResolverContext;
import org.apache.logging.log4j.layout.template.json.resolver.TemplateResolverFactory;
import org.apache.logging.log4j.layout.template.json.resolver.TemplateResolverInterceptor;
import org.apache.logging.log4j.layout.template.json.resolver.TemplateResolverStringSubstitutor;
import org.apache.logging.log4j.layout.template.json.util.JsonReader;
import org.apache.logging.log4j.layout.template.json.util.JsonWriter;

public final class TemplateResolvers {
    private static final String RESOLVER_FIELD_NAME = "$resolver";
    private static final TemplateResolver<?> EMPTY_ARRAY_RESOLVER = new EmptyArrayResolver();
    private static final TemplateResolver<?> EMPTY_OBJECT_RESOLVER = new EmptyObjectResolver();
    private static final TemplateResolver<?> NULL_RESOLVER = new NullResolver();

    private TemplateResolvers() {
    }

    public static <V, C extends TemplateResolverContext<V, C>> TemplateResolver<V> ofTemplate(C context, String template) {
        Object node;
        Objects.requireNonNull(context, "context");
        Objects.requireNonNull(template, "template");
        try {
            node = JsonReader.read(template);
        }
        catch (Exception error) {
            String message = String.format("failed parsing template (template=%s)", template);
            throw new RuntimeException(message, error);
        }
        List<TemplateResolverInterceptor<V, C>> interceptors = context.getResolverInterceptors();
        for (int interceptorIndex = 0; interceptorIndex < interceptors.size(); ++interceptorIndex) {
            TemplateResolverInterceptor<V, C> interceptor = interceptors.get(interceptorIndex);
            node = interceptor.processTemplateBeforeResolverInjection(context, node);
        }
        return TemplateResolvers.ofObject(context, node);
    }

    static <V, C extends TemplateResolverContext<V, C>> TemplateResolver<V> ofObject(C context, Object object) {
        if (object == null) {
            TemplateResolver<?> nullResolver = NULL_RESOLVER;
            return nullResolver;
        }
        if (object instanceof List) {
            List list = (List)object;
            return TemplateResolvers.ofList(context, list);
        }
        if (object instanceof Map) {
            Map map = (Map)object;
            return TemplateResolvers.ofMap(context, map);
        }
        if (object instanceof String) {
            String string = (String)object;
            return TemplateResolvers.ofString(context, string);
        }
        if (object instanceof Number) {
            Number number = (Number)object;
            return TemplateResolvers.ofNumber(number);
        }
        if (object instanceof Boolean) {
            boolean value = (Boolean)object;
            return TemplateResolvers.ofBoolean(value);
        }
        String message = String.format("invalid JSON node type (class=%s)", object.getClass().getName());
        throw new IllegalArgumentException(message);
    }

    private static <V, C extends TemplateResolverContext<V, C>> TemplateResolver<V> ofList(C context, List<Object> list) {
        List itemResolvers = list.stream().map(item -> {
            TemplateResolver itemResolver = TemplateResolvers.ofObject(context, item);
            if (itemResolver.isFlattening()) {
                throw new IllegalArgumentException("flattening resolvers are not allowed in lists");
            }
            return itemResolver;
        }).collect(Collectors.toList());
        if (itemResolvers.isEmpty()) {
            TemplateResolver<?> emptyArrayResolver = EMPTY_ARRAY_RESOLVER;
            return emptyArrayResolver;
        }
        return new ArrayResolver(itemResolvers);
    }

    private static <V, C extends TemplateResolverContext<V, C>> TemplateResolver<V> ofMap(C context, Map<String, Object> map) {
        if (map.containsKey(RESOLVER_FIELD_NAME)) {
            return TemplateResolvers.ofResolver(context, map);
        }
        List<FieldResolverContext<V>> fieldResolverContexts = TemplateResolvers.populateFieldResolverMethods(context, map);
        int fieldCount = fieldResolverContexts.size();
        if (fieldCount == 0) {
            TemplateResolver<?> emptyObjectResolver = EMPTY_OBJECT_RESOLVER;
            return emptyObjectResolver;
        }
        return new MapResolver(fieldResolverContexts);
    }

    private static <V, C extends TemplateResolverContext<V, C>> List<FieldResolverContext<V>> populateFieldResolverMethods(C context, Map<String, Object> map) {
        ArrayList fieldNames = new ArrayList();
        ArrayList fieldResolvers = new ArrayList();
        map.forEach((fieldName, fieldValue) -> {
            TemplateResolver fieldResolver = TemplateResolvers.ofObject(context, fieldValue);
            boolean resolvable = fieldResolver.isResolvable();
            if (resolvable) {
                fieldNames.add(fieldName);
                fieldResolvers.add(fieldResolver);
            }
        });
        List fieldPrefixes = fieldNames.stream().map(fieldName -> {
            try (JsonWriter jsonWriter = context.getJsonWriter();){
                jsonWriter.writeString((CharSequence)fieldName);
                jsonWriter.getStringBuilder().append(':');
                String string = jsonWriter.getStringBuilder().toString();
                return string;
            }
        }).collect(Collectors.toList());
        int fieldCount = fieldNames.size();
        return IntStream.range(0, fieldCount).mapToObj(fieldIndex -> {
            FieldResolverMethod fieldResolverMethod;
            TemplateResolver fieldResolver = (TemplateResolver)fieldResolvers.get(fieldIndex);
            boolean flattening = fieldResolver.isFlattening();
            if (flattening) {
                fieldResolverMethod = new FlatteningFieldResolverMethod(fieldResolver);
            } else {
                String fieldPrefix = (String)fieldPrefixes.get(fieldIndex);
                fieldResolverMethod = new PrefixedFieldResolverMethod(fieldPrefix, fieldResolver);
            }
            return new FieldResolverContext(fieldResolver, fieldResolverMethod);
        }).collect(Collectors.toList());
    }

    private static <V, C extends TemplateResolverContext<V, C>> TemplateResolver<V> ofResolver(C context, Map<String, Object> configMap) {
        Objects.requireNonNull(context, "context");
        Objects.requireNonNull(configMap, "configMap");
        Object resolverNameObject = configMap.get(RESOLVER_FIELD_NAME);
        if (!(resolverNameObject instanceof String)) {
            throw new IllegalArgumentException("invalid resolver name: " + resolverNameObject);
        }
        String resolverName = (String)resolverNameObject;
        TemplateResolverFactory<V, C> resolverFactory = context.getResolverFactoryByName().get(resolverName);
        if (resolverFactory == null) {
            throw new IllegalArgumentException("unknown resolver: " + resolverName);
        }
        TemplateResolverConfig resolverConfig = new TemplateResolverConfig(configMap);
        return resolverFactory.create(context, resolverConfig);
    }

    private static <V, C extends TemplateResolverContext<V, C>> TemplateResolver<V> ofString(C context, String fieldValue) {
        boolean substitutionNeeded = fieldValue.contains("${");
        JsonWriter contextJsonWriter = context.getJsonWriter();
        if (substitutionNeeded) {
            TemplateResolverStringSubstitutor<V> substitutor = context.getSubstitutor();
            if (substitutor.isStable()) {
                String replacedText = substitutor.replace(null, fieldValue);
                if (replacedText == null) {
                    TemplateResolver<?> resolver = NULL_RESOLVER;
                    return resolver;
                }
                String escapedReplacedText = contextJsonWriter.use(() -> contextJsonWriter.writeString(replacedText));
                return new RawStringResolver(escapedReplacedText);
            }
            return new SubstitutingStringResolver(substitutor, fieldValue);
        }
        String escapedFieldValue = contextJsonWriter.use(() -> contextJsonWriter.writeString(fieldValue));
        return new RawStringResolver(escapedFieldValue);
    }

    private static <V> TemplateResolver<V> ofNumber(Number number) {
        return new NumberResolver(number);
    }

    private static <V> TemplateResolver<V> ofBoolean(boolean value) {
        return new BooleanResolver(value);
    }

    private static final class ArrayResolver<V>
    implements TemplateResolver<V> {
        private final List<TemplateResolver<V>> itemResolvers;

        private ArrayResolver(List<TemplateResolver<V>> itemResolvers) {
            this.itemResolvers = itemResolvers;
        }

        @Override
        public void resolve(V value, JsonWriter jsonWriter) {
            jsonWriter.writeArrayStart();
            for (int itemResolverIndex = 0; itemResolverIndex < this.itemResolvers.size(); ++itemResolverIndex) {
                if (itemResolverIndex > 0) {
                    jsonWriter.writeSeparator();
                }
                TemplateResolver<V> itemResolver = this.itemResolvers.get(itemResolverIndex);
                itemResolver.resolve(value, jsonWriter);
            }
            jsonWriter.writeArrayEnd();
        }
    }

    private static final class MapResolver<V>
    implements TemplateResolver<V> {
        private final List<FieldResolverContext<V>> fieldResolverContexts;

        private MapResolver(List<FieldResolverContext<V>> fieldResolverContexts) {
            this.fieldResolverContexts = fieldResolverContexts;
        }

        @Override
        public boolean isResolvable() {
            return true;
        }

        @Override
        public boolean isResolvable(V value) {
            int fieldCount = this.fieldResolverContexts.size();
            for (int fieldIndex = 0; fieldIndex < fieldCount; ++fieldIndex) {
                TemplateResolver fieldResolver = ((FieldResolverContext)this.fieldResolverContexts.get(fieldIndex)).resolver;
                boolean resolvable = fieldResolver.isResolvable(value);
                if (!resolvable) continue;
                return true;
            }
            return false;
        }

        @Override
        public void resolve(V value, JsonWriter jsonWriter) {
            jsonWriter.writeObjectStart();
            int fieldCount = this.fieldResolverContexts.size();
            int resolvedFieldCount = 0;
            for (int fieldIndex = 0; fieldIndex < fieldCount; ++fieldIndex) {
                FieldResolverContext<V> fieldResolverContext = this.fieldResolverContexts.get(fieldIndex);
                boolean resolvable = ((FieldResolverContext)fieldResolverContext).resolver.isResolvable(value);
                if (!resolvable) continue;
                boolean succeedingEntry = resolvedFieldCount > 0;
                boolean resolved = ((FieldResolverContext)fieldResolverContext).resolverMethod.resolve(value, jsonWriter, succeedingEntry);
                if (!resolved) continue;
                ++resolvedFieldCount;
            }
            jsonWriter.writeObjectEnd();
        }
    }

    private static final class RawStringResolver<V>
    implements TemplateResolver<V> {
        private final String rawString;

        private RawStringResolver(String rawString) {
            this.rawString = rawString;
        }

        @Override
        public void resolve(V ignored, JsonWriter jsonWriter) {
            jsonWriter.writeRawString(this.rawString);
        }
    }

    private static final class SubstitutingStringResolver<V>
    implements TemplateResolver<V> {
        private final TemplateResolverStringSubstitutor<V> substitutor;
        private final String string;

        private SubstitutingStringResolver(TemplateResolverStringSubstitutor<V> substitutor, String string) {
            this.substitutor = substitutor;
            this.string = string;
        }

        @Override
        public void resolve(V value, JsonWriter jsonWriter) {
            String replacedString = this.substitutor.replace(value, this.string);
            jsonWriter.writeString(replacedString);
        }
    }

    private static final class NumberResolver<V>
    implements TemplateResolver<V> {
        private final String numberString;

        private NumberResolver(Number number) {
            this.numberString = String.valueOf(number);
        }

        @Override
        public void resolve(V ignored, JsonWriter jsonWriter) {
            jsonWriter.writeRawString(this.numberString);
        }
    }

    private static final class BooleanResolver<V>
    implements TemplateResolver<V> {
        private final boolean value;

        private BooleanResolver(boolean value) {
            this.value = value;
        }

        @Override
        public void resolve(V ignored, JsonWriter jsonWriter) {
            jsonWriter.writeBoolean(this.value);
        }
    }

    private static final class FlatteningFieldResolverMethod<V>
    implements FieldResolverMethod<V> {
        private final TemplateResolver<V> fieldResolver;

        private FlatteningFieldResolverMethod(TemplateResolver<V> fieldResolver) {
            this.fieldResolver = fieldResolver;
        }

        @Override
        public boolean resolve(V value, JsonWriter jsonWriter, boolean succeedingEntry) {
            boolean resolvable = this.fieldResolver.isResolvable(value);
            if (!resolvable) {
                return false;
            }
            StringBuilder jsonWriterStringBuilder = jsonWriter.getStringBuilder();
            int initLength = jsonWriterStringBuilder.length();
            this.fieldResolver.resolve(value, jsonWriter, succeedingEntry);
            return jsonWriterStringBuilder.length() > initLength;
        }
    }

    private static final class PrefixedFieldResolverMethod<V>
    implements FieldResolverMethod<V> {
        private final String fieldPrefix;
        private final TemplateResolver<V> fieldResolver;

        private PrefixedFieldResolverMethod(String fieldPrefix, TemplateResolver<V> fieldResolver) {
            this.fieldPrefix = fieldPrefix;
            this.fieldResolver = fieldResolver;
        }

        @Override
        public boolean resolve(V value, JsonWriter jsonWriter, boolean succeedingEntry) {
            boolean resolvable = this.fieldResolver.isResolvable(value);
            if (!resolvable) {
                return false;
            }
            if (succeedingEntry) {
                jsonWriter.writeSeparator();
            }
            jsonWriter.writeRawString(this.fieldPrefix);
            this.fieldResolver.resolve(value, jsonWriter, succeedingEntry);
            return true;
        }
    }

    private static final class FieldResolverContext<V> {
        private final TemplateResolver<V> resolver;
        private final FieldResolverMethod<V> resolverMethod;

        private FieldResolverContext(TemplateResolver<V> resolver, FieldResolverMethod<V> resolverMethod) {
            this.resolver = resolver;
            this.resolverMethod = resolverMethod;
        }
    }

    @FunctionalInterface
    private static interface FieldResolverMethod<V> {
        public boolean resolve(V var1, JsonWriter var2, boolean var3);
    }

    private static final class EmptyArrayResolver
    extends UnresolvableTemplateResolver {
        private EmptyArrayResolver() {
        }

        @Override
        public void resolve(Object value, JsonWriter jsonWriter) {
            jsonWriter.writeArrayStart();
            jsonWriter.writeArrayEnd();
        }
    }

    private static final class EmptyObjectResolver
    extends UnresolvableTemplateResolver {
        private EmptyObjectResolver() {
        }

        @Override
        public void resolve(Object value, JsonWriter jsonWriter) {
            jsonWriter.writeObjectStart();
            jsonWriter.writeObjectEnd();
        }
    }

    private static final class NullResolver
    extends UnresolvableTemplateResolver {
        private NullResolver() {
        }

        @Override
        public void resolve(Object value, JsonWriter jsonWriter) {
            jsonWriter.writeNull();
        }
    }

    private static abstract class UnresolvableTemplateResolver
    implements TemplateResolver<Object> {
        private UnresolvableTemplateResolver() {
        }

        @Override
        public final boolean isResolvable() {
            return false;
        }

        @Override
        public final boolean isResolvable(Object value) {
            return false;
        }
    }
}

