/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iceberg.hadoop;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.UUID;
import java.util.concurrent.RejectedExecutionException;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.iceberg.LocationProviders;
import org.apache.iceberg.LockManager;
import org.apache.iceberg.TableMetadata;
import org.apache.iceberg.TableMetadataParser;
import org.apache.iceberg.TableOperations;
import org.apache.iceberg.encryption.EncryptionManager;
import org.apache.iceberg.exceptions.CommitFailedException;
import org.apache.iceberg.exceptions.CommitStateUnknownException;
import org.apache.iceberg.exceptions.RuntimeIOException;
import org.apache.iceberg.exceptions.ValidationException;
import org.apache.iceberg.hadoop.Util;
import org.apache.iceberg.io.FileIO;
import org.apache.iceberg.io.LocationProvider;
import org.apache.iceberg.relocated.com.google.common.annotations.VisibleForTesting;
import org.apache.iceberg.relocated.com.google.common.base.Preconditions;
import org.apache.iceberg.relocated.com.google.common.collect.Lists;
import org.apache.iceberg.relocated.com.google.common.collect.Sets;
import org.apache.iceberg.util.Pair;
import org.apache.iceberg.util.Tasks;
import org.apache.iceberg.util.ThreadPools;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class HadoopTableOperations
implements TableOperations {
    private static final Logger LOG = LoggerFactory.getLogger(HadoopTableOperations.class);
    private static final Pattern VERSION_PATTERN = Pattern.compile("v([^\\.]*)\\..*");
    private static final TableMetadataParser.Codec[] TABLE_METADATA_PARSER_CODEC_VALUES = TableMetadataParser.Codec.values();
    private final Configuration conf;
    private final Path location;
    private final FileIO fileIO;
    private final LockManager lockManager;
    private TableMetadata currentMetadata = null;
    private volatile Integer version = null;
    private volatile boolean shouldRefresh = true;

    protected HadoopTableOperations(Path location, FileIO fileIO, Configuration conf, LockManager lockManager) {
        this.conf = conf;
        this.location = location;
        this.fileIO = fileIO;
        this.lockManager = lockManager;
    }

    @Override
    public TableMetadata current() {
        if (this.shouldRefresh) {
            return this.refresh();
        }
        return this.currentMetadata;
    }

    private synchronized Pair<Integer, TableMetadata> versionAndMetadata() {
        return Pair.of(this.version, this.currentMetadata);
    }

    private synchronized void updateVersionAndMetadata(int newVersion, String metadataFile) {
        if (this.version == null || this.version != newVersion) {
            this.version = newVersion;
            this.currentMetadata = HadoopTableOperations.checkUUID(this.currentMetadata, TableMetadataParser.read(this.io(), metadataFile));
        }
    }

    @Override
    public TableMetadata refresh() {
        int ver = this.version != null ? this.version.intValue() : this.findVersion();
        try {
            Path metadataFile = this.getMetadataFile(ver);
            if (this.version == null && metadataFile == null && ver == 0) {
                return null;
            }
            if (metadataFile == null) {
                throw new ValidationException("Metadata file for version %d is missing", ver);
            }
            Path nextMetadataFile = this.getMetadataFile(ver + 1);
            while (nextMetadataFile != null) {
                metadataFile = nextMetadataFile;
                nextMetadataFile = this.getMetadataFile(++ver + 1);
            }
            this.updateVersionAndMetadata(ver, metadataFile.toString());
            this.shouldRefresh = false;
            return this.currentMetadata;
        }
        catch (IOException e) {
            throw new RuntimeIOException(e, "Failed to refresh the table", new Object[0]);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void commit(TableMetadata base, TableMetadata metadata) {
        Pair<Integer, TableMetadata> current = this.versionAndMetadata();
        if (base != current.second()) {
            throw new CommitFailedException("Cannot commit changes based on stale table metadata", new Object[0]);
        }
        if (base == metadata) {
            LOG.info("Nothing to commit.");
            return;
        }
        Preconditions.checkArgument(base == null || base.location().equals(metadata.location()), "Hadoop path-based tables cannot be relocated");
        Preconditions.checkArgument(!metadata.properties().containsKey("write.metadata.path"), "Hadoop path-based tables cannot relocate metadata");
        String codecName = metadata.property("write.metadata.compression-codec", "none");
        TableMetadataParser.Codec codec = TableMetadataParser.Codec.fromName(codecName);
        String fileExtension = TableMetadataParser.getFileExtension(codec);
        Path tempMetadataFile = this.metadataPath(String.valueOf(UUID.randomUUID()) + fileExtension);
        TableMetadataParser.write(metadata, this.io().newOutputFile(tempMetadataFile.toString()));
        int nextVersion = (current.first() != null ? current.first() : 0) + 1;
        Path finalMetadataFile = this.metadataFilePath(nextVersion, codec);
        FileSystem fs = this.getFileSystem(tempMetadataFile, this.conf);
        boolean versionCommitSuccess = false;
        boolean useObjectStore = metadata.propertyAsBoolean("write.object-storage.enabled", false);
        int previousVersionsMax = metadata.propertyAsInt("write.metadata.previous-versions-max", 100);
        boolean supportGlobalLocking = useObjectStore;
        try {
            this.tryLock(tempMetadataFile, this.metadataRoot());
            versionCommitSuccess = this.commitNewVersion(fs, tempMetadataFile, finalMetadataFile, nextVersion, supportGlobalLocking);
            if (!versionCommitSuccess) {
                throw new CommitFailedException("Can not commit newMetaData because version [%s] has already been committed. tempMetaData=[%s],finalMetaData=[%s].Are there other clients running in parallel with the current task?", nextVersion, tempMetadataFile, finalMetadataFile);
            }
            this.validate(supportGlobalLocking, previousVersionsMax, nextVersion, fs, finalMetadataFile);
            this.shouldRefresh = true;
            LOG.info("Committed a new metadata file {}", (Object)finalMetadataFile);
            this.writeVersionHint(fs, nextVersion);
            this.deleteRemovedMetadataFiles(base, metadata);
        }
        catch (CommitStateUnknownException e) {
            this.shouldRefresh = true;
            throw e;
        }
        catch (Exception e) {
            this.shouldRefresh = versionCommitSuccess;
            if (!versionCommitSuccess) {
                this.tryDelete(tempMetadataFile);
                throw new CommitFailedException(e);
            }
        }
        finally {
            this.unlock(tempMetadataFile, this.metadataRoot());
        }
    }

    private void tryDelete(Path path) {
        try {
            this.io().deleteFile(path.toString());
        }
        catch (Exception exception) {
            // empty catch block
        }
    }

    @VisibleForTesting
    void tryLock(Path src, Path dst) {
        if (!this.lockManager.acquire(dst.toString(), src.toString())) {
            throw new CommitFailedException("Failed to acquire lock on file: %s with owner: %s", dst, src);
        }
    }

    void unlock(Path src, Path dst) {
        try {
            if (!this.lockManager.release(dst.toString(), src.toString())) {
                LOG.warn("Failed to release lock on file: {} with owner: {}", (Object)dst, (Object)src);
            }
        }
        catch (Exception exception) {
            // empty catch block
        }
    }

    private void validate(boolean supportGlobalLocking, int previousVersionsMax, int nextVersion, FileSystem fs, Path finalMetadataFile) throws IOException {
        if (!supportGlobalLocking) {
            this.fastFailIfDirtyCommit(previousVersionsMax, nextVersion, fs, finalMetadataFile);
            this.cleanAllTooOldDirtyCommit(fs, previousVersionsMax);
        }
    }

    @VisibleForTesting
    void fastFailIfDirtyCommit(int previousVersionsMax, int nextVersion, FileSystem fs, Path finalMetadataFile) throws IOException {
        int currentMaxVersion = this.findVersionWithOutVersionHint(fs);
        if (currentMaxVersion - nextVersion > previousVersionsMax && fs.exists(finalMetadataFile)) {
            this.tryDelete(finalMetadataFile);
            throw new CommitStateUnknownException(new RejectedExecutionException(String.format("Commit rejected by server!The current commit version [%d] is much smaller than the latest version [%d].Are there other clients running in parallel with the current task?", nextVersion, currentMaxVersion)));
        }
    }

    void cleanAllTooOldDirtyCommit(FileSystem fs, int previousVersionsMax) throws IOException {
        FileStatus[] files = fs.listStatus(this.metadataRoot(), name -> VERSION_PATTERN.matcher(name.getName()).matches());
        ArrayList<Path> dirtyCommits = Lists.newArrayList();
        int currentMaxVersion = this.findVersionWithOutVersionHint(fs);
        long now = System.currentTimeMillis();
        long ttl = 604800000L;
        for (FileStatus file : files) {
            long modificationTime = file.getModificationTime();
            Path path = file.getPath();
            if (currentMaxVersion - this.version(path.getName()) <= previousVersionsMax || now - modificationTime <= ttl) continue;
            dirtyCommits.add(path);
        }
        for (Path dirtyCommit : dirtyCommits) {
            this.io().deleteFile(dirtyCommit.toString());
        }
    }

    @Override
    public FileIO io() {
        return this.fileIO;
    }

    @Override
    public LocationProvider locationProvider() {
        return LocationProviders.locationsFor(this.current().location(), this.current().properties());
    }

    @Override
    public String metadataFileLocation(String fileName) {
        return this.metadataPath(fileName).toString();
    }

    @Override
    public TableOperations temp(final TableMetadata uncommittedMetadata) {
        return new TableOperations(){

            @Override
            public TableMetadata current() {
                return uncommittedMetadata;
            }

            @Override
            public TableMetadata refresh() {
                throw new UnsupportedOperationException("Cannot call refresh on temporary table operations");
            }

            @Override
            public void commit(TableMetadata base, TableMetadata metadata) {
                throw new UnsupportedOperationException("Cannot call commit on temporary table operations");
            }

            @Override
            public String metadataFileLocation(String fileName) {
                return HadoopTableOperations.this.metadataFileLocation(fileName);
            }

            @Override
            public LocationProvider locationProvider() {
                return LocationProviders.locationsFor(uncommittedMetadata.location(), uncommittedMetadata.properties());
            }

            @Override
            public FileIO io() {
                return HadoopTableOperations.this.io();
            }

            @Override
            public EncryptionManager encryption() {
                return HadoopTableOperations.this.encryption();
            }

            @Override
            public long newSnapshotId() {
                return HadoopTableOperations.this.newSnapshotId();
            }
        };
    }

    @VisibleForTesting
    Path getMetadataFile(int metadataVersion) throws IOException {
        for (TableMetadataParser.Codec codec : TABLE_METADATA_PARSER_CODEC_VALUES) {
            Path metadataFile = this.metadataFilePath(metadataVersion, codec);
            FileSystem fs = this.getFileSystem(metadataFile, this.conf);
            if (fs.exists(metadataFile)) {
                return metadataFile;
            }
            if (!codec.equals((Object)TableMetadataParser.Codec.GZIP) || !(fs = this.getFileSystem(metadataFile = this.oldMetadataFilePath(metadataVersion, codec), this.conf)).exists(metadataFile)) continue;
            return metadataFile;
        }
        return null;
    }

    private Path metadataFilePath(int metadataVersion, TableMetadataParser.Codec codec) {
        return this.metadataPath("v" + metadataVersion + TableMetadataParser.getFileExtension(codec));
    }

    private Path oldMetadataFilePath(int metadataVersion, TableMetadataParser.Codec codec) {
        return this.metadataPath("v" + metadataVersion + TableMetadataParser.getOldFileExtension(codec));
    }

    private Path metadataPath(String filename) {
        return new Path(this.metadataRoot(), filename);
    }

    private Path metadataRoot() {
        return new Path(this.location, "metadata");
    }

    private int version(String fileName) {
        Matcher matcher = VERSION_PATTERN.matcher(fileName);
        if (!matcher.matches()) {
            return -1;
        }
        String versionNumber = matcher.group(1);
        try {
            return Integer.parseInt(versionNumber);
        }
        catch (NumberFormatException ne) {
            return -1;
        }
    }

    @VisibleForTesting
    Path versionHintFile() {
        return this.metadataPath("version-hint.text");
    }

    @VisibleForTesting
    void writeVersionHint(FileSystem fs, Integer versionToWrite) throws IOException {
        Path versionHintFile = this.versionHintFile();
        Path tempVersionHintFile = this.metadataPath(String.valueOf(UUID.randomUUID()) + "-version-hint.temp");
        try {
            this.writeVersionToPath(fs, tempVersionHintFile, versionToWrite);
            fs.rename(tempVersionHintFile, versionHintFile);
        }
        catch (IOException e) {
            if (fs.exists(tempVersionHintFile)) {
                this.io().deleteFile(tempVersionHintFile.toString());
            }
            throw e;
        }
    }

    @VisibleForTesting
    boolean nextVersionIsLatest(int nextVersion, int currentMaxVersion) {
        return nextVersion == currentMaxVersion + 1;
    }

    private void writeVersionToPath(FileSystem fs, Path path, int versionToWrite) {
        try (FSDataOutputStream out = fs.create(path, false);){
            out.write(String.valueOf(versionToWrite).getBytes(StandardCharsets.UTF_8));
        }
        catch (IOException e) {
            throw new RuntimeIOException(e);
        }
    }

    @VisibleForTesting
    int findVersionByUsingVersionHint(FileSystem fs, Path versionHintFile) throws IOException {
        try (InputStreamReader fsr = new InputStreamReader((InputStream)fs.open(versionHintFile), StandardCharsets.UTF_8);){
            int n;
            try (BufferedReader in = new BufferedReader(fsr);){
                n = Integer.parseInt(in.readLine().replace("\n", ""));
            }
            return n;
        }
    }

    @VisibleForTesting
    int findVersionWithOutVersionHint(FileSystem fs) {
        try {
            if (!fs.exists(this.metadataRoot())) {
                LOG.warn("Metadata for table not found in directory [{}]", (Object)this.metadataRoot());
                return 0;
            }
            FileStatus[] files = fs.listStatus(this.metadataRoot(), name -> VERSION_PATTERN.matcher(name.getName()).matches());
            int maxVersion = 0;
            for (FileStatus file : files) {
                int currentVersion = this.version(file.getPath().getName());
                if (currentVersion <= maxVersion || this.getMetadataFile(currentVersion) == null) continue;
                maxVersion = currentVersion;
            }
            return maxVersion;
        }
        catch (Exception e) {
            throw new RuntimeIOException(new IOException(e));
        }
    }

    @VisibleForTesting
    int findVersion() {
        Path versionHintFile = this.versionHintFile();
        FileSystem fs = this.getFileSystem(versionHintFile, this.conf);
        try {
            return fs.exists(versionHintFile) ? this.findVersionByUsingVersionHint(fs, versionHintFile) : this.findVersionWithOutVersionHint(fs);
        }
        catch (Exception e) {
            return this.findVersionWithOutVersionHint(fs);
        }
    }

    @VisibleForTesting
    boolean commitNewVersion(FileSystem fs, Path src, Path dst, Integer nextVersion, boolean supportGlobalLocking) throws IOException {
        int maxVersion;
        if (fs.exists(dst)) {
            throw new CommitFailedException("Version %d already exists: %s", nextVersion, dst);
        }
        int n = maxVersion = supportGlobalLocking ? this.findVersion() : this.findVersionWithOutVersionHint(fs);
        if (!this.nextVersionIsLatest(nextVersion, maxVersion)) {
            if (!supportGlobalLocking) {
                this.io().deleteFile(this.versionHintFile().toString());
            }
            throw new CommitFailedException("Cannot commit version [%d] because it is smaller or much larger than the current latest version [%d].Are there other clients running in parallel with the current task?", nextVersion, maxVersion);
        }
        this.io().deleteFile(this.versionHintFile().toString());
        return this.renameMetaDataFileAndCheck(fs, src, dst, supportGlobalLocking);
    }

    protected FileSystem getFileSystem(Path path, Configuration hadoopConf) {
        return Util.getFs(path, hadoopConf);
    }

    @VisibleForTesting
    boolean checkMetaDataFileRenameSuccess(FileSystem fs, Path tempMetaDataFile, Path finalMetaDataFile, boolean supportGlobalLocking) throws IOException {
        if (!supportGlobalLocking) {
            return fs.exists(finalMetaDataFile) && !fs.exists(tempMetaDataFile);
        }
        return fs.exists(finalMetaDataFile);
    }

    @VisibleForTesting
    boolean renameMetaDataFile(FileSystem fs, Path tempMetaDataFile, Path finalMetaDataFile) throws IOException {
        return fs.rename(tempMetaDataFile, finalMetaDataFile);
    }

    private boolean renameCheck(FileSystem fs, Path tempMetaDataFile, Path finalMetaDataFile, Throwable rootError, boolean supportGlobalLocking) {
        try {
            return this.checkMetaDataFileRenameSuccess(fs, tempMetaDataFile, finalMetaDataFile, supportGlobalLocking);
        }
        catch (Exception e) {
            throw new CommitStateUnknownException(rootError != null ? rootError : e);
        }
    }

    @VisibleForTesting
    boolean renameMetaDataFileAndCheck(FileSystem fs, Path tempMetaDataFile, Path finalMetaDataFile, boolean supportGlobalLocking) {
        try {
            return this.renameMetaDataFile(fs, tempMetaDataFile, finalMetaDataFile);
        }
        catch (IOException e) {
            return this.renameCheck(fs, tempMetaDataFile, finalMetaDataFile, e, supportGlobalLocking);
        }
        catch (Exception e) {
            throw new CommitStateUnknownException(e);
        }
    }

    @VisibleForTesting
    void deleteRemovedMetadataFiles(TableMetadata base, TableMetadata metadata) {
        if (base == null) {
            return;
        }
        boolean deleteAfterCommit = metadata.propertyAsBoolean("write.metadata.delete-after-commit.enabled", false);
        if (deleteAfterCommit) {
            HashSet<TableMetadata.MetadataLogEntry> removedPreviousMetadataFiles = Sets.newHashSet(base.previousFiles());
            removedPreviousMetadataFiles.removeAll(metadata.previousFiles());
            Tasks.foreach(removedPreviousMetadataFiles).executeWith(ThreadPools.getWorkerPool()).noRetry().suppressFailureWhenFinished().onFailure((previousMetadataFile, exc) -> LOG.warn("Delete failed for previous metadata file: {}", previousMetadataFile, (Object)exc)).run(previousMetadataFile -> this.io().deleteFile(previousMetadataFile.file()));
        }
    }

    private static TableMetadata checkUUID(TableMetadata currentMetadata, TableMetadata newMetadata) {
        String newUUID = newMetadata.uuid();
        if (currentMetadata != null && currentMetadata.uuid() != null && newUUID != null) {
            Preconditions.checkState(newUUID.equals(currentMetadata.uuid()), "Table UUID does not match: current=%s != refreshed=%s", (Object)currentMetadata.uuid(), (Object)newUUID);
        }
        return newMetadata;
    }
}

