/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.runtime.net;

import java.io.IOException;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.NetworkInterface;
import java.net.Socket;
import java.net.SocketAddress;
import java.net.UnknownHostException;
import java.time.Duration;
import java.util.Arrays;
import java.util.Collections;
import java.util.Enumeration;
import java.util.List;
import java.util.UUID;
import org.apache.flink.runtime.leaderretrieval.LeaderRetrievalException;
import org.apache.flink.runtime.leaderretrieval.LeaderRetrievalListener;
import org.apache.flink.runtime.rpc.RpcSystemUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ConnectionUtils {
    private static final Logger LOG = LoggerFactory.getLogger(ConnectionUtils.class);
    private static final long MIN_SLEEP_TIME = 50L;
    private static final long MAX_SLEEP_TIME = 20000L;

    public static InetAddress findConnectingAddress(InetSocketAddress targetAddress, long maxWaitMillis, long startLoggingAfter) throws IOException {
        if (targetAddress == null) {
            throw new NullPointerException("targetAddress must not be null");
        }
        if (maxWaitMillis <= 0L) {
            throw new IllegalArgumentException("Max wait time must be positive");
        }
        long startTimeNanos = System.nanoTime();
        long currentSleepTime = 50L;
        long elapsedTimeMillis = 0L;
        List<AddressDetectionState> strategies = Collections.unmodifiableList(Arrays.asList(AddressDetectionState.LOCAL_HOST, AddressDetectionState.ADDRESS, AddressDetectionState.FAST_CONNECT, AddressDetectionState.SLOW_CONNECT));
        while (elapsedTimeMillis < maxWaitMillis) {
            boolean logging;
            boolean bl = logging = elapsedTimeMillis >= startLoggingAfter;
            if (logging) {
                LOG.info("Trying to connect to " + targetAddress);
            }
            for (AddressDetectionState strategy : strategies) {
                InetAddress address = ConnectionUtils.findAddressUsingStrategy(strategy, targetAddress, logging);
                if (address == null) continue;
                return address;
            }
            elapsedTimeMillis = (System.nanoTime() - startTimeNanos) / 1000000L;
            long toWait = Math.min(maxWaitMillis - elapsedTimeMillis, currentSleepTime);
            if (toWait > 0L) {
                if (logging) {
                    LOG.info("Could not connect. Waiting for {} msecs before next attempt", (Object)toWait);
                } else {
                    LOG.debug("Could not connect. Waiting for {} msecs before next attempt", (Object)toWait);
                }
                try {
                    Thread.sleep(toWait);
                }
                catch (InterruptedException e) {
                    throw new IOException("Connection attempts have been interrupted.");
                }
            }
            currentSleepTime = Math.min(2L * currentSleepTime, 20000L);
        }
        LOG.warn("Could not connect to {}. Selecting a local address using heuristics.", (Object)targetAddress);
        InetAddress heuristic = ConnectionUtils.findAddressUsingStrategy(AddressDetectionState.HEURISTIC, targetAddress, true);
        if (heuristic != null) {
            return heuristic;
        }
        LOG.warn("Could not find any IPv4 address that is not loopback or link-local. Using localhost address.");
        return InetAddress.getLocalHost();
    }

    private static InetAddress tryLocalHostBeforeReturning(InetAddress preliminaryResult, SocketAddress targetAddress, boolean logging) throws IOException {
        InetAddress localhostName = InetAddress.getLocalHost();
        if (preliminaryResult.equals(localhostName)) {
            return preliminaryResult;
        }
        if (ConnectionUtils.tryToConnect(localhostName, targetAddress, AddressDetectionState.SLOW_CONNECT.getTimeout(), logging)) {
            LOG.debug("Preferring {} (InetAddress.getLocalHost()) for local bind point over previous candidate {}", (Object)localhostName, (Object)preliminaryResult);
            return localhostName;
        }
        return preliminaryResult;
    }

    private static InetAddress findAddressUsingStrategy(AddressDetectionState strategy, InetSocketAddress targetAddress, boolean logging) throws IOException {
        if (strategy == AddressDetectionState.LOCAL_HOST) {
            InetAddress localhostName;
            try {
                localhostName = InetAddress.getLocalHost();
            }
            catch (UnknownHostException uhe) {
                LOG.warn("Could not resolve local hostname to an IP address: {}", (Object)uhe.getMessage());
                return null;
            }
            if (ConnectionUtils.tryToConnect(localhostName, targetAddress, strategy.getTimeout(), logging)) {
                LOG.debug("Using InetAddress.getLocalHost() immediately for the connecting address");
                return localhostName;
            }
            return null;
        }
        InetAddress address = targetAddress.getAddress();
        if (address == null) {
            return null;
        }
        byte[] targetAddressBytes = address.getAddress();
        Enumeration<NetworkInterface> e = NetworkInterface.getNetworkInterfaces();
        while (e.hasMoreElements()) {
            NetworkInterface netInterface = e.nextElement();
            Enumeration<InetAddress> ee = netInterface.getInetAddresses();
            block8: while (ee.hasMoreElements()) {
                InetAddress interfaceAddress = ee.nextElement();
                switch (strategy) {
                    case ADDRESS: {
                        if (!ConnectionUtils.hasCommonPrefix(targetAddressBytes, interfaceAddress.getAddress())) continue block8;
                        LOG.debug("Target address {} and local address {} share prefix - trying to connect.", (Object)targetAddress, (Object)interfaceAddress);
                        if (!ConnectionUtils.tryToConnect(interfaceAddress, targetAddress, strategy.getTimeout(), logging)) continue block8;
                        return ConnectionUtils.tryLocalHostBeforeReturning(interfaceAddress, targetAddress, logging);
                    }
                    case FAST_CONNECT: 
                    case SLOW_CONNECT: {
                        LOG.debug("Trying to connect to {} from local address {} with timeout {}", new Object[]{targetAddress, interfaceAddress, strategy.getTimeout()});
                        if (!ConnectionUtils.tryToConnect(interfaceAddress, targetAddress, strategy.getTimeout(), logging)) continue block8;
                        return ConnectionUtils.tryLocalHostBeforeReturning(interfaceAddress, targetAddress, logging);
                    }
                    case HEURISTIC: {
                        if (LOG.isDebugEnabled()) {
                            LOG.debug("Choosing InetAddress.getLocalHost() address as a heuristic.");
                        }
                        return InetAddress.getLocalHost();
                    }
                }
                throw new RuntimeException("Unsupported strategy: " + (Object)((Object)strategy));
            }
        }
        return null;
    }

    private static boolean hasCommonPrefix(byte[] address, byte[] address2) {
        return address[0] == address2[0] && address[1] == address2[1];
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private static boolean tryToConnect(InetAddress fromAddress, SocketAddress toSocket, int timeout, boolean logFailed) throws IOException {
        String detailedMessage = String.format("connect to [%s] from local address [%s] with timeout [%s]", toSocket, fromAddress, timeout);
        if (LOG.isDebugEnabled()) {
            LOG.debug("Trying to " + detailedMessage);
        }
        try (Socket socket = new Socket();){
            InetSocketAddress bindP = new InetSocketAddress(fromAddress, 0);
            socket.bind(bindP);
            socket.connect(toSocket, timeout);
            boolean bl = true;
            return bl;
        }
        catch (Exception ex) {
            String message = "Failed to " + detailedMessage + " due to: " + ex.getMessage();
            if (LOG.isDebugEnabled()) {
                LOG.debug(message, (Throwable)ex);
                return false;
            }
            if (!logFailed) return false;
            LOG.info(message);
            return false;
        }
    }

    public static class LeaderConnectingAddressListener
    implements LeaderRetrievalListener {
        private static final Duration defaultLoggingDelay = Duration.ofMillis(400L);
        private final RpcSystemUtils rpcSystemUtils;
        private final Object retrievalLock = new Object();
        private String rpcURL;
        private LeaderRetrievalState retrievalState = LeaderRetrievalState.NOT_RETRIEVED;
        private Exception exception;

        public LeaderConnectingAddressListener(RpcSystemUtils rpcSystemUtils) {
            this.rpcSystemUtils = rpcSystemUtils;
        }

        public InetAddress findConnectingAddress(Duration timeout) throws LeaderRetrievalException {
            return this.findConnectingAddress(timeout, defaultLoggingDelay);
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public InetAddress findConnectingAddress(Duration timeout, Duration startLoggingAfter) throws LeaderRetrievalException {
            long startTimeNanos = System.nanoTime();
            long currentSleepTime = 50L;
            long elapsedTimeMillis = 0L;
            InetSocketAddress targetAddress = null;
            try {
                while (elapsedTimeMillis < timeout.toMillis()) {
                    long maxTimeout = timeout.toMillis() - elapsedTimeMillis;
                    Object object = this.retrievalLock;
                    synchronized (object) {
                        if (this.exception != null) {
                            throw this.exception;
                        }
                        if (this.retrievalState == LeaderRetrievalState.NOT_RETRIEVED) {
                            try {
                                this.retrievalLock.wait(maxTimeout);
                            }
                            catch (InterruptedException e) {
                                throw new Exception("Finding connecting address was interruptedwhile waiting for the leader retrieval.");
                            }
                        } else if (this.retrievalState == LeaderRetrievalState.NEWLY_RETRIEVED) {
                            targetAddress = this.rpcSystemUtils.getInetSocketAddressFromRpcUrl(this.rpcURL);
                            LOG.debug("Retrieved new target address {} for RPC URL {}.", (Object)targetAddress, (Object)this.rpcURL);
                            this.retrievalState = LeaderRetrievalState.RETRIEVED;
                            currentSleepTime = 50L;
                        } else {
                            currentSleepTime = Math.min(2L * currentSleepTime, 20000L);
                        }
                    }
                    if (targetAddress != null) {
                        boolean logging;
                        AddressDetectionState strategy = AddressDetectionState.LOCAL_HOST;
                        boolean bl = logging = elapsedTimeMillis >= startLoggingAfter.toMillis();
                        if (logging) {
                            LOG.info("Trying to connect to address {}", (Object)targetAddress);
                        }
                        do {
                            InetAddress address;
                            if ((address = ConnectionUtils.findAddressUsingStrategy(strategy, targetAddress, logging)) != null) {
                                return address;
                            }
                            switch (strategy) {
                                case LOCAL_HOST: {
                                    strategy = AddressDetectionState.ADDRESS;
                                    break;
                                }
                                case ADDRESS: {
                                    strategy = AddressDetectionState.FAST_CONNECT;
                                    break;
                                }
                                case FAST_CONNECT: {
                                    strategy = AddressDetectionState.SLOW_CONNECT;
                                    break;
                                }
                                case SLOW_CONNECT: {
                                    strategy = null;
                                    break;
                                }
                                default: {
                                    throw new RuntimeException("Unsupported strategy: " + (Object)((Object)strategy));
                                }
                            }
                        } while (strategy != null);
                    }
                    elapsedTimeMillis = (System.nanoTime() - startTimeNanos) / 1000000L;
                    long timeToWait = Math.min(Math.max(timeout.toMillis() - elapsedTimeMillis, 0L), currentSleepTime);
                    if (timeToWait <= 0L) continue;
                    Object object2 = this.retrievalLock;
                    synchronized (object2) {
                        try {
                            this.retrievalLock.wait(timeToWait);
                        }
                        catch (InterruptedException e) {
                            throw new Exception("Finding connecting address was interrupted while pausing.");
                        }
                    }
                    elapsedTimeMillis = (System.nanoTime() - startTimeNanos) / 1000000L;
                }
                InetAddress heuristic = null;
                if (targetAddress != null) {
                    LOG.warn("Could not connect to {}. Selecting a local address using heuristics.", targetAddress);
                    heuristic = ConnectionUtils.findAddressUsingStrategy(AddressDetectionState.HEURISTIC, targetAddress, true);
                }
                if (heuristic != null) {
                    return heuristic;
                }
                LOG.warn("Could not find any IPv4 address that is not loopback or link-local. Using localhost address.");
                return InetAddress.getLocalHost();
            }
            catch (Exception e) {
                throw new LeaderRetrievalException("Could not retrieve the connecting address to the current leader with the pekko URL " + this.rpcURL + ".", e);
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void notifyLeaderAddress(String leaderAddress, UUID leaderSessionID) {
            if (leaderAddress != null && !leaderAddress.isEmpty()) {
                Object object = this.retrievalLock;
                synchronized (object) {
                    this.rpcURL = leaderAddress;
                    this.retrievalState = LeaderRetrievalState.NEWLY_RETRIEVED;
                    this.retrievalLock.notifyAll();
                }
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void handleError(Exception exception) {
            Object object = this.retrievalLock;
            synchronized (object) {
                this.exception = exception;
                this.retrievalLock.notifyAll();
            }
        }

        private static enum LeaderRetrievalState {
            NOT_RETRIEVED,
            RETRIEVED,
            NEWLY_RETRIEVED;

        }
    }

    private static enum AddressDetectionState {
        LOCAL_HOST(200),
        ADDRESS(50),
        FAST_CONNECT(50),
        SLOW_CONNECT(1000),
        HEURISTIC(0);

        private final int timeout;

        private AddressDetectionState(int timeout) {
            this.timeout = timeout;
        }

        public int getTimeout() {
            return this.timeout;
        }
    }
}

