/*
 * Decompiled with CFR 0.152.
 */
package org.apache.sshd.client.session.proxy;

import java.io.IOException;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.PasswordAuthentication;
import java.nio.charset.StandardCharsets;
import java.text.MessageFormat;
import java.util.Arrays;
import java.util.function.Function;
import java.util.logging.Level;
import org.apache.sshd.client.proxy.ProxyData;
import org.apache.sshd.client.session.proxy.AbstractProxyConnector;
import org.apache.sshd.client.session.proxy.AuthenticationHandler;
import org.apache.sshd.client.session.proxy.BasicAuthentication;
import org.apache.sshd.client.session.proxy.GssApiAuthentication;
import org.apache.sshd.client.session.proxy.GssApiMechanisms;
import org.apache.sshd.common.io.IoWriteFuture;
import org.apache.sshd.common.util.Readable;
import org.apache.sshd.common.util.buffer.Buffer;
import org.apache.sshd.common.util.buffer.BufferException;
import org.apache.sshd.common.util.buffer.BufferUtils;
import org.apache.sshd.common.util.buffer.ByteArrayBuffer;
import org.apache.sshd.common.util.io.functors.IOFunction;
import org.apache.sshd.common.util.logging.LoggingUtils;
import org.apache.sshd.common.util.logging.SimplifiedLog;
import org.ietf.jgss.GSSContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Socks5ProxyConnector
extends AbstractProxyConnector {
    private static final Logger LOG = LoggerFactory.getLogger(Socks5ProxyConnector.class);
    private static final SimplifiedLog SIMPLE = LoggingUtils.wrap(LOG);
    private ProtocolState state = ProtocolState.NONE;
    private AuthenticationHandler<Buffer, Buffer> authenticator;
    private GSSContext context;
    private byte[] authenticationProposals;

    public Socks5ProxyConnector(ProxyData proxy, InetSocketAddress remoteAddress, IOFunction<Buffer, IoWriteFuture> send, Function<InetSocketAddress, PasswordAuthentication> passwordAuth) {
        super(proxy, remoteAddress, send, passwordAuth);
    }

    @Override
    public void start() throws IOException {
        if (LOG.isDebugEnabled()) {
            LOG.debug("SOCKS5 starting connection to {}", (Object)this.remoteAddress);
        }
        ByteArrayBuffer buffer = new ByteArrayBuffer(5, false);
        ((Buffer)buffer).putByte((byte)5);
        this.context = Socks5ProxyConnector.getGSSContext(this.remoteAddress);
        this.authenticationProposals = this.getAuthenticationProposals();
        ((Buffer)buffer).putByte((byte)this.authenticationProposals.length);
        buffer.putRawBytes(this.authenticationProposals);
        this.state = ProtocolState.INIT;
        this.write(buffer);
    }

    private byte[] getAuthenticationProposals() {
        byte[] proposals = new byte[3];
        int i = 0;
        proposals[i++] = SocksAuthenticationMethod.ANONYMOUS.getValue();
        proposals[i++] = SocksAuthenticationMethod.PASSWORD.getValue();
        if (this.context != null) {
            proposals[i++] = SocksAuthenticationMethod.GSSAPI.getValue();
        }
        if (i == proposals.length) {
            return proposals;
        }
        return Arrays.copyOf(proposals, i);
    }

    private void sendConnectInfo() throws IOException {
        byte type;
        GssApiMechanisms.closeContextSilently(this.context);
        if (LOG.isDebugEnabled()) {
            LOG.debug("SOCKS5 authenticated, requesting connection to {}", (Object)this.remoteAddress);
        }
        byte[] rawAddress = Socks5ProxyConnector.getRawAddress(this.remoteAddress);
        byte[] remoteName = null;
        int length = 0;
        if (rawAddress == null) {
            remoteName = this.remoteAddress.getHostString().getBytes(StandardCharsets.US_ASCII);
            if (remoteName == null || remoteName.length == 0) {
                throw new IOException("No remote host name");
            }
            if (remoteName.length > 255) {
                throw new IOException("Proxy host name too long for SOCKS (at most 255 characters): " + this.remoteAddress.getHostString());
            }
            type = 3;
            length = remoteName.length + 1;
        } else {
            length = rawAddress.length;
            type = length == 4 ? (byte)1 : 4;
        }
        ByteArrayBuffer buffer = new ByteArrayBuffer(4 + length + 2, false);
        ((Buffer)buffer).putByte((byte)5);
        ((Buffer)buffer).putByte((byte)1);
        ((Buffer)buffer).putByte((byte)0);
        ((Buffer)buffer).putByte(type);
        if (remoteName != null) {
            ((Buffer)buffer).putByte((byte)remoteName.length);
            buffer.putRawBytes(remoteName);
        } else {
            buffer.putRawBytes(rawAddress);
        }
        int port = this.remoteAddress.getPort();
        if (port <= 0) {
            port = 22;
        }
        ((Buffer)buffer).putByte((byte)(port >> 8 & 0xFF));
        ((Buffer)buffer).putByte((byte)(port & 0xFF));
        this.state = ProtocolState.CONNECTING;
        this.write(buffer);
    }

    private void doPasswordAuth() throws Exception {
        GssApiMechanisms.closeContextSilently(this.context);
        this.authenticator = new SocksBasicAuthentication();
        this.startAuth();
    }

    private void doGssApiAuth() throws Exception {
        this.authenticator = new SocksGssApiAuthentication();
        this.startAuth();
    }

    @Override
    public void close() {
        super.close();
        AuthenticationHandler<Buffer, Buffer> handler = this.authenticator;
        this.authenticator = null;
        if (handler != null) {
            handler.close();
        }
    }

    private void startAuth() throws Exception {
        this.authenticator.setParams(null);
        this.authenticator.start();
        Buffer buffer = this.authenticator.getToken();
        if (buffer == null) {
            throw new IOException("No data for authenticating at proxy " + this.proxyAddress);
        }
        this.state = ProtocolState.AUTHENTICATING;
        this.write(buffer).addListener(f -> buffer.clear(true));
    }

    private void authStep(Buffer input) throws Exception {
        this.authenticator.setParams(input);
        this.authenticator.process();
        Buffer buffer = this.authenticator.getToken();
        if (buffer != null) {
            this.write(buffer).addListener(f -> buffer.clear(true));
        }
        if (this.authenticator.isDone()) {
            this.sendConnectInfo();
        }
    }

    private void establishConnection(Buffer data) throws IOException {
        byte reply = data.getByte();
        switch (reply) {
            case 0: {
                try {
                    reply = data.getByte();
                    if (reply != 0) {
                        LOG.warn("SOCKS5 proxy for connection to {} returned success with reserved byte != 0", (Object)this.remoteAddress);
                    }
                    this.skipAddress(data);
                    int port = data.getUShort();
                    if (port <= 0) {
                        LOG.warn("SOCKS5 proxy for connection to {} returned success with invalid bind port {}", (Object)this.remoteAddress, (Object)Integer.toString(port));
                    }
                }
                catch (BufferException e) {
                    data.rpos(0);
                    LOG.warn("SOCKS5 proxy for connection to {} returned malformed success reply; got {}", (Object)this.remoteAddress, (Object)e.toString());
                    BufferUtils.dumpHex(SIMPLE, Level.WARNING, "remaining", ' ', 32, data.array(), data.rpos(), data.available());
                    data.rpos(data.wpos());
                }
                if (LOG.isDebugEnabled()) {
                    LOG.debug("SOCKS5 connected to {}", (Object)this.remoteAddress);
                }
                this.state = ProtocolState.CONNECTED;
                this.done = true;
                return;
            }
            case 1: {
                throw new IOException("Tunneling failed for proxy " + this.proxyAddress);
            }
            case 2: {
                throw new IOException(MessageFormat.format("Forbidden tunnel at proxy {0} to {1}", this.proxyAddress, this.remoteAddress));
            }
            case 3: {
                throw new IOException(MessageFormat.format("Network unreachable at proxy {0} for tunnel to {1}", this.proxyAddress, this.remoteAddress));
            }
            case 4: {
                throw new IOException(MessageFormat.format("Host unreachable at proxy {0} for tunnel to {1}", this.proxyAddress, this.remoteAddress));
            }
            case 5: {
                throw new IOException(MessageFormat.format("Connection refused at proxy {0} for tunnel to {1}", this.proxyAddress, this.remoteAddress));
            }
            case 6: {
                throw new IOException(MessageFormat.format("TTL expired at proxy {0} for tunnel to {1}", this.proxyAddress, this.remoteAddress));
            }
            case 7: {
                throw new IOException("Unsupported command at proxy " + this.proxyAddress);
            }
            case 8: {
                throw new IOException("Unsupported address at proxy " + this.proxyAddress);
            }
        }
        throw new IOException("Unspecified failure at proxy " + this.proxyAddress);
    }

    private void skipAddress(Buffer data) {
        int skip = -1;
        int type = data.getUByte();
        switch (type) {
            case 1: {
                skip = 4;
                break;
            }
            case 4: {
                skip = 16;
                break;
            }
            case 3: {
                skip = data.getUByte();
                break;
            }
            default: {
                throw new BufferException("Invalid bind address type " + type);
            }
        }
        if (data.available() < skip) {
            throw new BufferException("Not enough data; need " + skip + ", have " + data.available());
        }
        data.rpos(data.rpos() + skip);
    }

    @Override
    public Buffer received(Readable buffer) throws Exception {
        try {
            ByteArrayBuffer data = new ByteArrayBuffer(buffer.available(), false);
            data.putBuffer(buffer);
            this.state.handleMessage(this, data);
            return data;
        }
        catch (Exception e) {
            this.state = ProtocolState.FAILED;
            if (this.authenticator != null) {
                this.authenticator.close();
                this.authenticator = null;
            }
            this.done = true;
            throw e;
        }
    }

    private void versionCheck(byte version) throws IOException {
        if (version != 5) {
            throw new IOException("Unexpected SOCK version " + (version & 0xFF));
        }
    }

    private SocksAuthenticationMethod getAuthMethod(byte value) {
        if (value != SocksAuthenticationMethod.NONE_ACCEPTABLE.getValue()) {
            for (byte proposed : this.authenticationProposals) {
                if (proposed != value) continue;
                for (SocksAuthenticationMethod method : SocksAuthenticationMethod.values()) {
                    if (method.getValue() != value) continue;
                    return method;
                }
                break;
            }
        }
        return SocksAuthenticationMethod.NONE_ACCEPTABLE;
    }

    private static byte[] getRawAddress(InetSocketAddress address) {
        InetAddress ipAddress = GssApiMechanisms.resolve(address);
        return ipAddress == null ? null : ipAddress.getAddress();
    }

    private static GSSContext getGSSContext(InetSocketAddress address) {
        if (!GssApiMechanisms.getSupportedMechanisms().contains(GssApiMechanisms.KERBEROS_5)) {
            return null;
        }
        return GssApiMechanisms.createContext(GssApiMechanisms.KERBEROS_5, GssApiMechanisms.getCanonicalName(address));
    }

    private static enum SocksAuthenticationMethod {
        ANONYMOUS(0),
        GSSAPI(1),
        PASSWORD(2),
        NONE_ACCEPTABLE(255);

        private final byte value;

        private SocksAuthenticationMethod(int value) {
            this.value = (byte)value;
        }

        public byte getValue() {
            return this.value;
        }
    }

    private static enum ProtocolState {
        NONE,
        INIT{

            @Override
            public void handleMessage(Socks5ProxyConnector connector, Buffer data) throws Exception {
                connector.versionCheck(data.getByte());
                SocksAuthenticationMethod authMethod = connector.getAuthMethod(data.getByte());
                switch (authMethod) {
                    case ANONYMOUS: {
                        connector.sendConnectInfo();
                        break;
                    }
                    case PASSWORD: {
                        connector.doPasswordAuth();
                        break;
                    }
                    case GSSAPI: {
                        connector.doGssApiAuth();
                        break;
                    }
                    default: {
                        throw new IOException("Cannot authenticate at SOCKS proxy " + connector.proxyAddress);
                    }
                }
            }
        }
        ,
        AUTHENTICATING{

            @Override
            public void handleMessage(Socks5ProxyConnector connector, Buffer data) throws Exception {
                connector.authStep(data);
            }
        }
        ,
        CONNECTING{

            @Override
            public void handleMessage(Socks5ProxyConnector connector, Buffer data) throws Exception {
                if (data.available() != 4) {
                    connector.versionCheck(data.getByte());
                    connector.establishConnection(data);
                }
            }
        }
        ,
        CONNECTED,
        FAILED;


        public void handleMessage(Socks5ProxyConnector connector, Buffer data) throws Exception {
            throw new IOException(MessageFormat.format("Unexpected reply from SOCKS proxy {0}: {1}", connector.proxyAddress, BufferUtils.toHex(data.array())));
        }
    }

    private class SocksBasicAuthentication
    extends BasicAuthentication<Buffer, Buffer> {
        private static final byte SOCKS_BASIC_PROTOCOL_VERSION = 1;
        private static final byte SOCKS_BASIC_AUTH_SUCCESS = 0;

        SocksBasicAuthentication() {
            super(Socks5ProxyConnector.this.proxyAddress, Socks5ProxyConnector.this.proxyUser, Socks5ProxyConnector.this.proxyPassword);
        }

        @Override
        public void process() throws Exception {
            this.done = true;
            if (((Buffer)this.params).getByte() != 1 || ((Buffer)this.params).getByte() != 0) {
                throw new IOException("SOCKS BASIC authentication failed at proxy " + this.proxy);
            }
        }

        @Override
        protected PasswordAuthentication getCredentials() {
            return Socks5ProxyConnector.this.passwordAuthentication();
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public Buffer getToken() throws IOException {
            if (this.done) {
                return null;
            }
            try {
                byte[] rawUser = this.user.getBytes(StandardCharsets.UTF_8);
                if (rawUser.length > 255) {
                    throw new IOException(MessageFormat.format("User name too long for proxy {0}: {1} bytes (max 255): {2}", this.proxy, Integer.toString(rawUser.length), this.user));
                }
                if (this.password.length > 255) {
                    throw new IOException(MessageFormat.format("Password too long for proxy {0}: {1} bytes (max 255)", this.proxy, Integer.toString(this.password.length)));
                }
                ByteArrayBuffer buffer = new ByteArrayBuffer(3 + rawUser.length + this.password.length, false);
                buffer.putByte((byte)1);
                buffer.putByte((byte)rawUser.length);
                buffer.putRawBytes(rawUser);
                buffer.putByte((byte)this.password.length);
                buffer.putRawBytes(this.password);
                ByteArrayBuffer byteArrayBuffer = buffer;
                return byteArrayBuffer;
            }
            finally {
                this.clearPassword();
                this.done = true;
            }
        }
    }

    private class SocksGssApiAuthentication
    extends GssApiAuthentication<Buffer, Buffer> {
        private static final byte SOCKS5_GSSAPI_VERSION = 1;
        private static final byte SOCKS5_GSSAPI_TOKEN = 1;
        private static final int SOCKS5_GSSAPI_FAILURE = 255;

        SocksGssApiAuthentication() {
            super(Socks5ProxyConnector.this.proxyAddress);
        }

        @Override
        protected GSSContext createContext() throws Exception {
            return Socks5ProxyConnector.this.context;
        }

        @Override
        public Buffer getToken() throws IOException {
            if (this.token == null) {
                return null;
            }
            ByteArrayBuffer buffer = new ByteArrayBuffer(4 + this.token.length, false);
            ((Buffer)buffer).putByte((byte)1);
            ((Buffer)buffer).putByte((byte)1);
            ((Buffer)buffer).putByte((byte)(this.token.length >> 8 & 0xFF));
            ((Buffer)buffer).putByte((byte)(this.token.length & 0xFF));
            buffer.putRawBytes(this.token);
            return buffer;
        }

        @Override
        protected byte[] extractToken(Buffer input) throws Exception {
            if (Socks5ProxyConnector.this.context == null) {
                return null;
            }
            int version = input.getUByte();
            if (version != 1) {
                throw new IOException(MessageFormat.format("Wrong GSS-API version at SOCKS proxy {0}: {1}", this.proxy, Integer.toString(version)));
            }
            int msgType = input.getUByte();
            if (msgType == 255) {
                throw new IOException(MessageFormat.format("SOCKS Proxy {0} reported GSS-API failure", this.proxy));
            }
            if (msgType != 1) {
                throw new IOException(MessageFormat.format("Unknown GSS-API message {1} from SOCKS proxy {0}", this.proxy, Integer.toHexString(msgType & 0xFF)));
            }
            if (input.available() >= 2) {
                int length = (input.getUByte() << 8) + input.getUByte();
                if (input.available() >= length) {
                    byte[] value = new byte[length];
                    if (length > 0) {
                        input.getRawBytes(value);
                    }
                    return value;
                }
            }
            throw new IOException("Message too short from proxy" + this.proxy);
        }
    }
}

