package org.apache.jetspeed.portlet;

import java.io.PrintWriter;
import java.io.IOException;

import org.apache.jetspeed.portlet.event.ActionListener;

/**
 ** The <CODE>PortletResponse</CODE> interface represents the response
 ** sent to the client device.
 **
 ** <P>
 ** It is up to the portlet container to enhance, extend, or modify
 ** the response as required before it is actually sent to the client
 ** device.
 **
 ** @see   PortletRequest
 **
 * @author <A HREF="mailto:shesmer@apache.org">Stephan Hesmer</A>
 **/

public interface PortletResponse extends javax.servlet.http.HttpServletResponse
{
    /**
     ** Returns the writer object that can be used to contribute
     ** markup to the portlet response.
     **
     ** <P>
     ** It is important to note that the markup generated by the
     ** portlet is either a full document or just a fragment of the
     ** markup. A corresponding setting in the deployment descriptor
     ** has to be made.
     *
     ** @return   the print writer
     **
     ** @exception   IOException
     **              if the I/O operation fails
     **/

    public PrintWriter getWriter () throws IOException;

    /**
     ** Returns the content type that can be used to contribute
     ** markup to the portlet response.
     **
     ** @return   the content type
     **/

    public String getContentType ();

    /**
     * Returns the name of the charset used for
     * the MIME body sent in this response.
     *
     * <p>If no charset has been assigned, it is implicitly
     * set to <code>ISO-8859-1</code> (<code>Latin-1</code>).
     *
     * <p>See RFC 2047 (http://ds.internic.net/rfc/rfc2045.txt)
     * for more information about character encoding and MIME.
     *
     * @return		a <code>String</code> specifying the
     *			name of the charset, for
     *			example, <code>ISO-8859-1</code>
     *
     */
  
    public String getCharacterEncoding();

    /**
     * Returns the character set that can be used to contribute
     * markup to the portlet response.
     * 
     * @return the character set
     * @deprecated use <CODE>getCharacterEncoding</CODE> since version 1.1
     */
    public String getCharacterSet ();

    /**
     * Creates a portlet URI pointing to the current portlet mode.
     * The returned URI can be further extended by adding
     * portlet-specific parameters and by attaching an action. Also,
     * the portlet window will stay in the same state as before.
     * 
     * @return a portlet URI
     */
    public PortletURI createURI ();

    /**
     * Creates a portlet URI pointing to the current portlet mode
     * and given portlet window state. The returned URI can be
     * further extended by adding portlet-specific parameters and
     * by attaching an action.
     * 
     * @param state
     *               the portlet window state
     * @return a portlet URI
     */
    public PortletURI createURI (PortletWindow.State state);

    /**
     * Creates a portlet URI pointing at the referrer of the portlet. This is
     * normally the previous screen / URL shown in the browser.
     * The returned URI can be further extended by adding
     * portlet-specific parameters and/or by attaching an action.
     * This method is very useful to return from the EDIT mode,
     * for example.
     * 
     * @return a portlet URI
     */
    public PortletURI createReturnURI ();

    /**
     * Returns the encoded URI of the resource at the given path.
     * Encoding may include prefixing, conversion to absolute URL, etc.
     * 
     * <P>
     * To generate unprotected URI's like an image use the direct
     * path inside of you portlet (e.g. /myportlet/myimage.gif).
     * Otherwise, to generate protected URI's like a property file
     * use the path inside of your portlet prefixed with /WEB-INF/
     * (e.g. /WEB-INF/myportlet/myportlet.jsp)
     * 
     * @param path
     *               the path to the resource
     * @return the resource URI as string
     * @deprecated use <CODE>encodeURL</CODE> since version 1.1
     */
    public String encodeURI (String path);

    /**
     ** Returns the encoded URI of the resource at the given path.
     ** Encoding may include prefixing, conversion to absolute URL, etc.
     **
     ** <P>
     ** To generate unprotected URI's like an image use the direct
     ** path inside of you portlet (e.g. /myportlet/myimage.gif).
     ** Otherwise, to generate protected URI's like a property file
     ** use the path inside of your portlet prefixed with /WEB-INF/
     ** (e.g. /WEB-INF/myportlet/myportlet.jsp)
     **
     ** @param   path
     **          the path to the resource
     **
     ** @return   the resource URI as string
     **/

    public String encodeURL (String path);

    /**
     ** Maps the given string value into this portlet's namespace.
     ** Should be applied to every variable or name used in the
     ** output stream. For example, a parameter used in a form,
     ** or a function name in javascript.
     **
     ** @param   aValue
     **          the name to be mapped
     **
     ** @return   the mapped name
     **/

    public String encodeNamespace (String aValue);

    /**
     * Adds the specified cookie to the response.  This method can be called
     * multiple times to set more than one cookie.
     * <P>
     * Since cookies are used to communicate with other web applications, the portlet
     * container does not encode the cookie name in the concrete portlet instances'
     * namespace. If a portlet wants to use the cookie only inside itself, the portlet
     * developer has to take care of the namespace encoding on his/her own by using
     * <CODE>encodeNamespace()</CODE>.
     * 
     * @param cookie the Cookie to return to the client
     * @exception IllegalStateException
     *                   if the portlet tries to access this function
     *                   outside of the <CODE>beginPage</CODE> method
     *                   of the <CODE>PortletPageListener</CODE>
     */

    public void addCookie(javax.servlet.http.Cookie cookie);

    /**
     * Returns a boolean indicating whether the named response header 
     * has already been set.
     * 
     * @param	name	the header name
     * @return		<code>true</code> if the named response header 
     *			has already been set; 
     * 			<code>false</code> otherwise
     */

    public boolean containsHeader(String name);

    /**
     * 
     * Sets a response header with the given name and
     * date-value.  The date is specified in terms of
     * milliseconds since the epoch.  If the header had already
     * been set, the new value overwrites the previous one.  The
     * <code>containsHeader</code> method can be used to test for the
     * presence of a header before setting its value.
     * <P>
     * <I>We need to think about all header methods and how to solve nameclashes, etc.</I>
     * 
     * @param	name	the name of the header to set
     * @param	value	the assigned date value
     * 
     * @see #containsHeader
     * @see #addDateHeader
     *
     * @exception IllegalStateException
     *                   if the portlet tries to access this function outside of the
     *                   <CODE>beginPage</CODE> method of the <CODE>PortletPageListener</CODE> or
     *                   if the portlet tries to set an portal-wide header
     */

    public void setDateHeader(String name, long date);
    
    /**
     * Adds a response header with the given name and
     * date-value.  The date is specified in terms of
     * milliseconds since the epoch.  This method allows response headers
     * to have multiple values.
     * <P>
     * <I>We need to think about all header methods and how to solve nameclashes, etc.</I>
     * 
     * @param name   the name of the header to set
     * @param date
     * @exception IllegalStateException
     *                   if the portlet tries to access this function outside of the
     *                   <CODE>beginPage</CODE> method of the <CODE>PortletPageListener</CODE> or
     *                   if the portlet tries to set an portal-wide header
     * @see #setDateHeader
     */

    public void addDateHeader(String name, long date);
    
    /**
     * Sets a response header with the given name and value.
     * If the header had already been set, the new value overwrites the
     * previous one.  The <code>containsHeader</code> method can be
     * used to test for the presence of a header before setting its
     * value.
     * <P>
     * <I>We need to think about all header methods and how to solve nameclashes, etc.</I>
     * 
     * @param name   the name of the header
     * @param value  the header value
     * @see #containsHeader
     * @see #addHeader
     *
     * @exception IllegalStateException
     *                   if the portlet tries to access this function outside of the
     *                   <CODE>beginPage</CODE> method of the <CODE>PortletPageListener</CODE> or
     *                   if the portlet tries to set an portal-wide header
     */

    public void setHeader(String name, String value);
    
    /**
     * Adds a response header with the given name and value.
     * This method allows response headers to have multiple values.
     * <P>
     * <I>We need to think about all header methods and how to solve nameclashes, etc.</I>
     * 
     * @param	name	the name of the header
     * @param	value	the additional header value
     *
     * @see #setHeader
     *
     * @exception IllegalStateException
     *                   if the portlet tries to access this function outside of the
     *                   <CODE>beginPage</CODE> method of the <CODE>PortletPageListener</CODE> or
     *                   if the portlet tries to set an portal-wide header
     */

    public void addHeader(String name, String value);

    /**
     * Sets a response header with the given name and
     * integer value.  If the header had already been set, the new value
     * overwrites the previous one.  The <code>containsHeader</code>
     * method can be used to test for the presence of a header before
     * setting its value.
     * <P>
     * <I>We need to think about all header methods and how to solve nameclashes, etc.</I>
     *
     * @param	name	the name of the header
     * @param	value	the assigned integer value
     *
     * @see #containsHeader
     * @see #addIntHeader
     *
     * @exception IllegalStateException
     *                   if the portlet tries to access this function outside of the
     *                   <CODE>beginPage</CODE> method of the <CODE>PortletPageListener</CODE> or
     *                   if the portlet tries to set an portal-wide header
     */

    public void setIntHeader(String name, int value);

    /**
     * Adds a response header with the given name and
     * integer value.  This method allows response headers to have multiple
     * values.
     * <P>
     * <I>We need to think about all header methods and how to solve nameclashes, etc.</I>
     * 
     * @param name   the name of the header
     * @param value  the assigned integer value
     * @exception IllegalStateException
     *                   if the portlet tries to access this function outside of the
     *                   <CODE>beginPage</CODE> method of the <CODE>PortletPageListener</CODE> or
     *                   if the portlet tries to set an portal-wide header
     * @see #setIntHeader
     */

    public void addIntHeader(String name, int value);

}


