/* ====================================================================
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2000-2001 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Apache" and "Apache Software Foundation" and
 *     "Apache Jetspeed" must not be used to endorse or promote products
 *    derived from this software without prior written permission. For
 *    written permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache" or
 *    "Apache Jetspeed", nor may "Apache" appear in their name, without
 *    prior written permission of the Apache Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */

package org.apache.jetspeed.portlets;

//standard java stuff
import java.io.BufferedReader;
import java.io.IOException;
import java.io.Reader;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.StringReader;
import java.util.Hashtable;
import java.util.Enumeration;


//standard Jetspeed stuff
import org.apache.jetspeed.portlet.*;
import org.apache.jetspeed.portlet.service.*;
import org.apache.jetspeed.portlets.util.PortletEntityResolver;



//xerces support
import org.apache.xerces.parsers.*;

//XML stuff
import org.w3c.dom.*;
import org.xml.sax.*;
import javax.xml.transform.stream.*;

import java.net.*;

/**
<p>Portlet which renders RDF Site Summary.</p>
<p>This portlet uses XML stylesheet for transforming the RSS
content into display markup depending on the MimeType requested
by the user-agent</p>
<p>It accepts the following parameters :
<dl>
<dt>itemDisplayed</dt>
<dd>The number of items from the RSS file to display on screen. Default 15 for HTML, 5 for WML</dd>
<dt>showDescription</dt>
<dd>Should the portlet show the item descriptions. Must be true or false. Default: true for HTML, false for WML</dd>
<dt>showTitle</dt>
<dd>Should the portlet show the channel description. Must be true or false. Default: true for HTML, false for WML</dd>
<dt>stylesheet[.<mime>]</dt>
<dd>The stylesheet URL. If a mime-type is specified, the stylesheet
is only used for this mime-type</dd>
</dl>
@author <A HREF="mailto:raphael@apache.org">Raphal Luta</A>
@version $Id: RSSPortlet.java,v 1.6 2001/03/07 06:47:09 taylor Exp $
*/
public class RSSPortlet extends TransformPortlet
{

    private final static String EXTERNAL_URL = "://";
    public final static String ERROR_NOT_VALID = "This does not appear to be an RSS document";
    public final static String INVALID_TYPE = "Unable to display for this browser";
    public final static String RSS_URL_KEY = "url";
    public final static String DEFAULT_MARKUP = "html";

    private NodeList list = null;
    private Hashtable stylesheets = null;
    private Hashtable params = null;

    /**
    This method loads the init parameters and
    parse the document tied to this portlet
    */
    public void init (PortletConfig portletConfig) throws UnavailableException
    {
        super.init(portletConfig);


        if (this.getPortletLog().isInfoEnabled())
        {
            this.getPortletLog().info("RSSPortlet:init! started");
        }

    }


    public void doView (PortletRequest request, PortletResponse response)
    throws PortletException, IOException
    {
        if (this.list==null)
        {
            Document document = null;
            DOMParser parser = null;
            String url = null;


            // load stylesheets available
            try
            {

                stylesheets = new Hashtable();
                params = new Hashtable();
                Enumeration en = request.getPortletSettings().getAttributeNames();

                String name;
                String base;

                while (en.hasMoreElements())
                {
                    name = (String)en.nextElement();
                    base = DEFAULT_MARKUP;
                    if (name.startsWith("stylesheet"))
                    {
                        int idx=-1;
                        if ((idx=name.indexOf(".")) > -1)
                        {
                            base = name.substring(idx+1,name.length());
                        }
                        stylesheets.put(base, request.getPortletSettings().getAttribute(name));
                    }
                    else
                    {
                        if (name != this.RSS_URL_KEY)
                        {
                            params.put(name.toLowerCase(), request.getPortletSettings().getAttribute(name));
                        }
                        else
                        {
                            url = request.getPortletSettings().getAttribute(RSS_URL_KEY);
                        }

                    }
                }

                // add base url as style sheet parameter
                params.put("base",response.encodeURI("/"));

                // read content, clean it, parse it and cache the DOM
                parser = new DOMParser();

                if (url == null)
                {
                    url = request.getPortletSettings().getAttribute(RSS_URL_KEY);
                }

                if (this.getPortletLog().isInfoEnabled())
                {
                    this.getPortletLog().info("RSSPortlet:RSSContent url=" + url);
                }


                String content = null;
                if ( url != null)
                {
                    if ( url.indexOf(EXTERNAL_URL) >= 0 )
                    {
                        if (this.getPortletLog().isInfoEnabled())
                            this.getPortletLog().info("RSSPortlet: loading content from external url");
                        content = getRSSContent(url, request, response);

                    }
                    else
                    {
                        if (this.getPortletLog().isInfoEnabled())
                            this.getPortletLog().info("RSSPortlet: loading content from local url");
                        content = getLocaleRSSContent(url);

                    }
                }
                else
                {
                    this.getPortletLog().error("RSSPortlet:  No url defined.");
                    throw new PortletException("RSSPortlet:  No url defined.");
                }


                PortletEntityResolver resolver = new PortletEntityResolver();
                resolver.SetContentService( (ContentAccessService) this.getConfig().getContext().getService(
                                                                                                           org.apache.jetspeed.portlet.service.ContentAccessService.class) ,request, response);

                parser.setEntityResolver(resolver);

                InputSource isrc = new InputSource( this.cleanse( content ) );
                isrc.setEncoding("UTF-8");

                parser.parse( isrc );

                document = parser.getDocument();

            }
            catch ( Throwable t )
            {
                //Locator locator = parser.getLocator();



                this.getPortletLog().error("RSSPortlet:  Couldn't parse out XML document -> " + url, t );
                throw new UnavailableException( t.getMessage() );
            }

            //Determine title and description for this portlet
            list = document.getElementsByTagName( "channel" );

            if ( list.getLength() != 1 )
            {
                throw new UnavailableException( ERROR_NOT_VALID );
            }
        }

        String stylesheet = (String)this.stylesheets.get(request.getClient().getMimeType());
        if (stylesheet != null)
        {
            try
            {
                int i = 0;
                while (i < this.list.getLength())
                {
                    /*
                    response.getWriter().println( transform(
                        new StreamSource(this.list.item(i)),
                        new StreamSource(this.getPortletConfig().getContext().getResourceAsStream(stylesheet)),
                        this.params));
                        */
                    i++;
                    throw new SAXException ("Code commented out -- please check!");
                }
            }
            catch (SAXException exc)
            {
                String message = "RSSPortlet: Could not Parse content from " +
                                 request.getPortletSettings().getAttribute(RSS_URL_KEY) +
                                 " with stylesheet " +
                                 stylesheet;
                this.getPortletLog().error(message , exc);
                throw new PortletException(message , exc);
            }

        }
        else
        {
            if (this.getPortletLog().isInfoEnabled())
            {
                this.getPortletLog().info("RSSPortlet: No stylesheet for mimeType" + request.getClient().getMimeType());
            }
        }

    }

    private String getRSSContent(String url, PortletRequest request, PortletResponse response)
    throws PortletException
    {
        URL contentURL = null;

        try
        {
            PortletContext context = getPortletConfig().getContext();
            ContentAccessService contentAccessservice = (ContentAccessService)context.getService(
                                                                                                org.apache.jetspeed.portlet.service.ContentAccessService.class);
            contentURL = contentAccessservice.getURL( url, request, response );
        }
        catch (MalformedURLException e)
        {
            this.getPortletLog().error("RSSPortlet: Malformed URL.",e);
            throw new PortletException("Malformed URL");
        }
        catch (PortletServiceNotFoundException e)
        {
            getPortletLog().error("RSSPortlet: ContentAccessPortletService not found",e);
            throw new PortletException("ContentAccessPortletService not found");
        }
        catch (PortletServiceUnavailableException e)
        {
            getPortletLog().error("RSSPortlet: ContentAccessPortletService unavailable",e);
            throw new PortletException("ContentAccessPortletService unavailable");
        }

        String content = "";
        try
        {
            HttpURLConnection conn = (HttpURLConnection)contentURL.openConnection();
            conn.setFollowRedirects(false);
            InputStream inStream = conn.getInputStream();
            int rc = 0;
            byte [] buffer = new byte[4096];
            while (rc >= 0)
            {
                if (rc>0)
                    content = content + new String(buffer,0,rc);
                rc = inStream.read(buffer);
            }
        }
        catch (java.io.IOException e)
        {
            this.getPortletLog().error("RSSPortlet.getRSSContent(String): "+e.toString(),e);
            return "";
        }
        return content;
/*
    try
    {
            if(this.getPortletLog().isDebugEnabled())
        {
        this.getPortletLog().debug("RSSPortlet:RSSContent " + JetspeedDiskCache.getInstance().getEntry( url ).getData());
        }
        return JetspeedDiskCache.getInstance().getEntry( url ).getData();
    }
    catch(Exception exc)
    {
        throw new RuntimeException(exc.getMessage());
    }
*/

    }


    private String getLocaleRSSContent(String url) throws PortletException
    {
        StringBuffer sbuf = new StringBuffer();
        String line = null;

        BufferedReader buf = new BufferedReader (
                                                new InputStreamReader( this.getPortletConfig().getContext().getResourceAsStream(url)));

        if (buf != null)
        {
            try
            {
                line = buf.readLine();
                while (line != null)
                {
                    sbuf.append( line );
                    line = buf.readLine();
                }
            }
            catch (IOException exc)
            {
                this.getPortletLog().error("RSSPortlet: IOException occurred by access on file " + url);
                throw new PortletException ("RSSPortlet: IOException occurred by access on file " + url);
            }

        }
        else
        {
            this.getPortletLog().error("RSSPortlet: Unable to access resource " + url);
            throw new PortletException ("RSSPortlet: Unable to access resource " + url);
        }
        return sbuf.toString();

    }



    /**
    Given a URL to some content, clean the content to Xerces can handle it
    better.  Right now this involves:
    <ul>
        <li>
            If the document doesn't begin with "<?xml version=" truncate the
            content until this is the first line
        </li>

    </ul>

    */
    private Reader cleanse( String content ) throws IOException {

        String filtered = null;

        //specify the XML declaration to search for... this is just a subset
        //of the content but it will always exist.
        String XMLDECL = "<?xml version=";


        int start = content.indexOf( XMLDECL );

        if ( start <= 0 )
        {
            filtered = content;
        }
        else
        {
            filtered = content.substring( start, content.length() );
        }


        //Hack should be removed later
        String sub1;
        String sub2;
        int start2;


        start = filtered.indexOf ("<!DOCTYPE");

        if (start > 0)
        {
            sub1 = filtered.substring(0 , start );
            sub2 = sub1 + filtered.substring(filtered.indexOf(">",start)+1);
            filtered = sub2;

        }

        //End of hack

        if (this.getPortletLog().isDebugEnabled())
        {
            this.getPortletLog().debug("RSSPortlet:Filtered RSSContent " + filtered);
        }

        return new StringReader( filtered );
    }
}

