package modules

import (
	"encoding/json"
	"fmt"
	"io/ioutil"
	"os"
	"path/filepath"
	"sync"
)

// Switcher control the module whether enabled
type Switcher struct {
	Enabled     bool `json:"enabled"`
	Appearance  bool `json:"appearance"`
	Background  bool `json:"background"`
	Dock        bool `json:"dock"`
	Launcher    bool `json:"launcher"`
	Network     bool `json:"network"`
	Peripherals bool `json:"peripherals"`
	Power       bool `json:"power"`
	ScreenEdge  bool `json:"screen_edge"`
	Screensaver bool `json:"screensaver"`
	Audio       bool `json:"audio"`
	Updater     bool `json:"updater"`

	changed bool
	locker  sync.RWMutex
}

const (
	ConfigDir          = "/var/lib/deepin-sync-daemon"
	ConfigSwitcherFile = ConfigDir + "/switcher.json"

	switcherEnabled     = "enabled"
	switcherAppearance  = "appearance"
	switcherBackground  = "background"
	switcherDock        = "dock"
	switcherLauncher    = "launcher"
	switcherNetwork     = "network"
	switcherPeripherals = "peripherals"
	switcherPower       = "power"
	switcherScreenEdge  = "screen_edge"
	switcherScreensaver = "screensaver"
	switcherAudio       = "audio"
	switcherUpdater     = "updater"
)

var (
	localSwitcherFile = filepath.Join(os.Getenv("HOME"), ".config",
		"deepin", "deepin-sync-daemon", "switcher.json")
)

func newSwitcher(filename string) (*Switcher, error) {
	var s = Switcher{
		Enabled:     true,
		Appearance:  true,
		Audio:       true,
		Background:  true,
		Dock:        true,
		Launcher:    true,
		Network:     true,
		Peripherals: true,
		Power:       true,
		ScreenEdge:  true,
		Screensaver: true,
		Updater:     true,
	}
	contents, err := ioutil.ReadFile(filename)
	if err != nil {
		return &s, err
	}
	err = json.Unmarshal(contents, &s)
	return &s, err
}

// Fixed merge user configuration to system
func (s *Switcher) Fixed(loc *Switcher) {
	doEnableSwitcher(&s.Enabled, loc.Enabled)
	doEnableSwitcher(&s.Appearance, loc.Appearance)
	doEnableSwitcher(&s.Audio, loc.Audio)
	doEnableSwitcher(&s.Background, loc.Background)
	doEnableSwitcher(&s.Dock, loc.Dock)
	doEnableSwitcher(&s.Launcher, loc.Launcher)
	doEnableSwitcher(&s.Network, loc.Network)
	doEnableSwitcher(&s.Peripherals, loc.Peripherals)
	doEnableSwitcher(&s.Power, loc.Power)
	doEnableSwitcher(&s.ScreenEdge, loc.ScreenEdge)
	doEnableSwitcher(&s.Screensaver, loc.Screensaver)
	doEnableSwitcher(&s.Updater, loc.Updater)
}

// Enable set module switcher
func (s *Switcher) Enable(name string, enabled bool) (bool, error) {
	s.locker.RLock()
	defer s.locker.RUnlock()
	switch name {
	case switcherEnabled:
		s.changed = doEnableSwitcher(&s.Enabled, enabled)
	case switcherAppearance:
		s.changed = doEnableSwitcher(&s.Appearance, enabled)
	case switcherAudio:
		s.changed = doEnableSwitcher(&s.Audio, enabled)
	case switcherBackground:
		s.changed = doEnableSwitcher(&s.Background, enabled)
	case switcherDock:
		s.changed = doEnableSwitcher(&s.Dock, enabled)
	case switcherLauncher:
		s.changed = doEnableSwitcher(&s.Launcher, enabled)
	case switcherNetwork:
		s.changed = doEnableSwitcher(&s.Network, enabled)
	case switcherPeripherals:
		s.changed = doEnableSwitcher(&s.Peripherals, enabled)
	case switcherPower:
		s.changed = doEnableSwitcher(&s.Power, enabled)
	case switcherScreenEdge:
		s.changed = doEnableSwitcher(&s.ScreenEdge, enabled)
	case switcherScreensaver:
		s.changed = doEnableSwitcher(&s.Screensaver, enabled)
	case switcherUpdater:
		s.changed = doEnableSwitcher(&s.Updater, enabled)
	default:
		return false, fmt.Errorf("Invalid switcher name: %s", name)
	}
	return s.changed, nil
}

// IsEnabled return the module enabled status
func (s *Switcher) IsEnabled(name string) bool {
	s.locker.RLock()
	defer s.locker.RUnlock()
	switch name {
	case switcherEnabled:
		return s.Enabled
	case switcherAppearance:
		return s.Appearance
	case switcherAudio:
		return s.Audio
	case switcherBackground:
		return s.Background
	case switcherDock:
		return s.Dock
	case switcherLauncher:
		return s.Launcher
	case switcherNetwork:
		return s.Network
	case switcherPeripherals:
		return s.Peripherals
	case switcherPower:
		return s.Power
	case switcherScreenEdge:
		return s.ScreenEdge
	case switcherScreensaver:
		return s.Screensaver
	case switcherUpdater:
		return s.Updater
	}
	return false
}

// Bytes encode Switcher to bytes
func (s *Switcher) Bytes() ([]byte, error) {
	s.locker.RLock()
	defer s.locker.RUnlock()
	return json.Marshal(s)
}

// WriteFile save data to user configuration
func (s *Switcher) WriteFile(filename string) error {
	// fmt.Println("[Switcher] write:", filename, s.changed)
	if !s.changed {
		return nil
	}
	data, err := s.Bytes()
	if err != nil {
		return err
	}

	s.locker.Lock()
	defer s.locker.Unlock()
	s.changed = false
	err = os.MkdirAll(filepath.Dir(filename), 0755)
	if err != nil {
		return err
	}
	return ioutil.WriteFile(filename, data, 0644)
}

func doEnableSwitcher(s *bool, enabled bool) bool {
	if *s == enabled {
		return false
	}
	*s = enabled
	return true
}
