/* Copyright (C) 2009 to 2011 Chris Vine

The library comprised in this file or of which this file is part is
distributed by Chris Vine under the GNU Lesser General Public
License as follows:

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public License
   as published by the Free Software Foundation; either version 2.1 of
   the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License, version 2.1, for more details.

   You should have received a copy of the GNU Lesser General Public
   License, version 2.1, along with this library (see the file LGPL.TXT
   which came with this source code package in the c++-gtk-utils
   sub-directory); if not, write to the Free Software Foundation, Inc.,
   59 Temple Place - Suite 330, Boston, MA, 02111-1307, USA.

However, it is not intended that the object code of a program whose
source code instantiates a template from this file or uses macros or
inline functions (of any length) should by reason only of that
instantiation or use be subject to the restrictions of use in the GNU
Lesser General Public License.  With that in mind, the words "and
macros, inline functions and instantiations of templates (of any
length)" shall be treated as substituted for the words "and small
macros and small inline functions (ten lines or less in length)" in
the fourth paragraph of section 5 of that licence.  This does not
affect any other reason why object code may be subject to the
restrictions in that licence (nor for the avoidance of doubt does it
affect the application of section 2 of that licence to modifications
of the source code in this file).

*/

#ifndef CGU_EMITTER_H
#define CGU_EMITTER_H

/**
 * @file emitter.h
 * @brief This file provides a thread-safe signal/slot mechanism, with
 * automatic disconnection.
 *
 * An EmitterArg object is a list of Callback::FunctorArg objects.
 * Callback::FunctorArg objects may be "connected" to the EmitterArg
 * object, and all functors so connected will be executed when the
 * operator()() or emit() member functions of the EmitterArg object
 * concerned is called.  They will be called in the order in which
 * they were connected.  Emitter is a typedef for EmitterArg\<void\>.
 * The generalised EmitterArg\<T\> type contains
 * Callback::FunctorArg\<T\> objects (type T being the unbound
 * argument of a Callback::CallbackArg\<T\> callback or the type
 * container of unbound types - see Cgu::Callback for further
 * details, and "Usage" below for examples.)  The Emitter type holds
 * Callback::Functor (namely Callback::FunctorArg\<void\>) objects.
 *
 * The main advantage of an emitter object as opposed to storing a
 * functor object directly, apart from the fact that more than one
 * functor can be dispatched by a single call to EmitterArg<>::emit()
 * or EmitterArg<>::operator()(), is that it provides for automatic
 * disconnection of a functor if the object whose member function it
 * represents has ceased to exist.
 *
 * Where automatic disconnection is wanted, the object whose method is
 * to be encapsulated by a functor must have a Releaser object as a
 * public member function.  The Releaser object should be passed as
 * the second argument of EmitterArg<>::connect().  As well as a
 * Releaser object informing an emitter object when it has ceased to
 * exist, an emitter object will do the same to the Releaser object if
 * the emitter object happens to be destroyed before an object whose
 * members it references (and therefore before the Releaser object).
 * Automatic disconnection is mainly useful for non-static member
 * functions, but it can be employed for static member functions if
 * wanted (that will in effect bind the lifetime of the functor to
 * that of the object to whose Releaser the functor has been
 * attached.)
 *
 * It is safe for a connected function (i) to delete the EmitterArg
 * object to which it is connected, even if there are other functors
 * still to execute in the same emission (which will execute normally
 * provided they do not try to call any of the emitter's functions),
 * (ii) to call 'delete this' nothwithstanding that the connected
 * function is protected by a Releaser object (assuming all the other
 * restraints on calling 'delete this' are met), provided that no
 * other access would be made to the deleted object in a function call
 * connected to the same emitter which is due to execute subsequently
 * in the same emission, and (iii) to disconnect itself from the
 * EmitterArg object.  This design approach has a trade-off: if a
 * connected function tries to block, unblock or disconnect another
 * function connected to the same EmitterArg object which is due to
 * execute subsequently in the same emission (or to block, unblock or
 * disconnect itself when it is due to execute again subsequently in
 * the same emission), the attempted block, unblock or disconnection
 * will not have any effect on that emission (it will only have effect
 * on a subsequent emission).  In addition, a connected function may
 * not destroy an object whose non-static method is connected to the
 * same emitter and which would execute subsequently in the same
 * emission, even if that object is protected by a Releaser object
 * (the non-static method will unsuccessfully attempt to execute
 * notwithstanding the destruction of the object it would be operating
 * on).
 *
 * The SafeEmitterArg classes are the same as their EmitterArg
 * counterparts except that they contain Callback::SafeFunctorArg
 * objects, and their emit(), operator()(), connect(), disconnect(),
 * block(), unblock() and destructor methods are protected by a mutex
 * so that different threads can call these methods on the same
 * emitter object, or create and delete the object.
 *
 * Note that the mutexes are released when the operator()() methods of
 * the relevent Callback::SafeFunctorArg objects are called, as
 * SafeEmitterArg objects have no idea what the referenced callbacks
 * will do so if they were not released deadlocks could arise from
 * recursive or out-of-order locking of the SafeEmitterArg mutex.  It
 * is therefore for users to provide additional synchronisation if the
 * functions encapsulated by the relevant functors themselves need
 * additional protection.  Note also the subsidiary thread-safety
 * points mentioned below.
 *
 * The Releaser class is intrinsically thread safe (the overhead of
 * locking is so low that it is pointless having a separate
 * unprotected class).  This means that if a program is
 * multi-threaded, you can use the plain EmitterArg classes provided
 * that only the thread which creates a particular EmitterArg object
 * calls connect(), block(), unblock((), emit() or operator()() on it,
 * or deletes it, or calls disconnect() on it (either directly or
 * through a Releaser object being destroyed).  Where more than one
 * thread might do that in relation to any one emitter object, use
 * SafeEmitterArg.
 *
 * @b Alternatives
 *
 * These classes are intended as a lightweight thread-safe signal/slot
 * mechanism for GUI programming.  For more demanding usage libsigc++
 * is a good choice, except that it is not thread-safe.  An
 * alternative to libsigc++ is the boost::signal2 module, which is
 * thread-safe.
 *
 * @b Subsidiary @b thread-safety @b points
 *
 * As mentioned, the SafeEmitterArg classes are thread safe, and their
 * methods can be called in different threads without ill effect.
 * However, there are some things that cannot be done.  Users should
 * observe two points.
 *
 * First, it has been mentioned that if a connected function blocks,
 * unblocks or disconnects another function connected to the same
 * emitter object and due to execute subsequently in the same
 * emission, the blocking, unblocking or disconnection will not have
 * effect in that emission, and that a connected function may not
 * delete an object whose non-static method is due to execute
 * subsequently in the same emission.  The same outcome would result
 * if another thread tries to do any of these things while an emission
 * is under way.  Another thread should therefore leave alone objects
 * connected to a SafeEmitterArg object from the time of operator()()
 * or emit() beginning to the time of it ending, and not try to
 * interfere.
 *
 * Secondly, when a Releaser object is passed as the second argument
 * to the connect() method of a SafeEmitterArg object, the Releaser
 * object must remain in existence until the connect() method returns
 * or the emitter may be left in an inconsistent state.
 *
 * @anchor AssignmentAnchor
 * @b Assignment
 *
 * EmitterArg and SafeEmitterArg objects cannot be copied.  Releaser
 * objects can be (we do not want to make a class uncopiable just
 * because it has the safety feature of having a Releaser object as a
 * member).
 *
 * So how should assignment of a Releaser object and of a class which
 * has a Releaser as a member be handled?  An object which has a
 * Releaser as a member and which is being assigned to (the assignee)
 * could keep all its existing pre-assignment emitter connections - so
 * far as the Releaser object is concerned, it will have to do so
 * where the connections are not protected by the Releaser object, and
 * we could do the same in relation to protected connections, in which
 * case we would make operator=() of Releaser do nothing: that is,
 * just return - a default assignment would always be wrong as it
 * would take the assignor's Releaser state but inherit none of its
 * connections, which the assignee cannot inherit as they depend on a
 * remote emitter object or objects.
 *
 * However, the state of the assignee after assignment may not be such
 * as to permit the inheriting of all the assignor's state except its
 * connections.  Accordingly, the default strategy adopted here is for
 * the Releaser object to become a blank sheet on assignment.  After
 * assignment, an assignee which has a Releaser object as a member
 * will no longer have any of the emitter connections which were,
 * prior to assignment, protected by the Releaser object.  If in a
 * particular case the user does not want this behaviour, she should
 * provide an assignment operator in the class which has Releaser as a
 * member and leave Releaser alone in the assignment operator.
 *
 * @b Usage
 *
 * For a class my_obj of type MyObj, with a method void
 * MyObj::my_method(int, const char*), usage for a fully bound functor
 * and emitter would be:
 *
 * @code
 *   using namespace Cgu;
 *   int arg = 1;
 *   Emitter e;
 *   e.connect(Callback::make(my_obj, &MyObj::my_method, arg, "Hello\n"));
 *   e();
 *
 *   SafeEmitter se;
 *   se.connect(Callback::make(my_obj, &MyObj::my_method, arg, "Hello\n"));
 *   se();
 * @endcode
 *
 * Or for a partially bound functor and emitter:
 *
 * @code
 *   using namespace Cgu;
 *   int arg = 1;
 *   EmitterArg<const char*> e;
 *   e.connect(Callback::make(my_obj, &MyObj::my_method, arg));
 *   e("Hello\n");
 *
 *   SafeEmitterArg<const char*> se;
 *   se.connect(Callback::make(my_obj, &MyObj::my_method, arg));
 *   se("Hello\n");
 * @endcode
 *
 * To provide for two or three unbound arguments, from version 1.2.10
 * of the library the Cgu::TypeTuple struct is used (this struct has
 * no members and is never instantiated, so it does not impose any
 * overhead: see @ref TypeTuple for further information).  As in the
 * case of a single unbound argument, if there are bound arguments
 * these multiple unbound arguments must be the last (trailing)
 * arguments of the function to be called.  For a class object my_obj
 * of type MyObj, with a method void MyObj::my_method2(int, int, int,
 * const char*), usage with two unbound arguments would be:
 *
 * @code 
 *   int arg1 = 1, arg2 = 2, arg3 = 3;
 *   Cgu::EmitterArg<Cgu::TypeTuple<int, const char*> > e;
 *   e.connect(Cgu::Callback::make(my_obj, &MyObj::my_method2, arg1, arg2));
 *   e(arg3, "Hello\n");
 * @endcode
 *
 * and for three unbound arguments:
 *
 * @code 
 *   int arg1 = 1, arg2 = 2, arg3 = 3;
 *   Cgu::EmitterArg<Cgu::TypeTuple<int, int, const char*> > e;
 *   e.connect(Cgu::Callback::make(my_obj, &MyObj::my_method2, arg1));
 *   e(arg2, arg3, "Hello\n");
 * @endcode
 *
 * EmitterArg classes do not provide for a return value.  If a
 * result is wanted, users should pass an unbound argument by
 * reference or pointer (or pointer to pointer).
 *
 * Although only two bound and three unbound arguments are provided
 * for, as any of those arguments can be a struct, any number of
 * arguments can be passed as members of a struct (or, in C++0x, a
 * std::tuple).
 *
 * @b Exception @b safety
 *
 * Apart from the emit()/operator()() and connect() methods, nothing
 * done to an EmitterArg object should cause an exception to be
 * thrown.  This is because other methods only iterate through a
 * std::list object using std::for_each(), std::find() or by hand, and
 * the only things done by std::for_each() or after a std::find() or
 * iteration is to remove a functor from the list (copying a functor
 * and comparing functors never throw, nor does destroying a functor
 * provided the destructors of any bound argument type do not throw).
 * Thus, an EmitterArg/SafeEmitterArg and Releaser object should never
 * get into an inconsistent state.
 *
 * The connect() method could throw a std::bad_alloc exception, either
 * on creating new functors or on pushing the functors onto the list.
 * However, were it to do so, the method has strong exception safety
 * (assuming merely iterating over a list does not throw, as it should
 * not).
 *
 * The emit()/operator()() methods could throw std::bad_alloc, and so
 * far as that is concerned emission of all the connected functions
 * will either all succeed or all fail.  In addition, the connected
 * functions referenced by the functors held by the emitter might
 * throw when executed.  emit()/operator()() do not attempt to catch
 * these exceptions as there is nothing they could do with them.  This
 * means that although a throwing connected function will not leave
 * the EmitterArg/SafeEmitterArg object in an inconsistent state, any
 * other connected functions due to execute subsequently on that same
 * emission will not execute.  If that is important in any particular
 * case, the user must incorporate logic in the connected functions to
 * cater for an exception causing only part execution, or must connect
 * only one function to any one signal and "chain" emissions by hand
 * so as to do the right thing.
 */

/*
  Mutex locking heirarchy:

  Some out-of-order locking must take place because of the
  relationship between the Releaser and SafeEmitterArg<> classes.  The
  mutex of Releaser is given the higher priority.  This means that a
  plain EmitterArg<> object will not take any hit from the fact that
  Releaser is also useable with SafeEmitterArg<> objects.

  One consequence is that to avoid deadlocks, it is the
  SafeEmitterArg<> functions which must yield when a deadlock would
  otherwise arise.  Yielding could occur in
  SafeEmitterArg<>::~SafeEmitterArg() and
  SafeEmitterArg<>::disconnect().
*/

#ifdef CGU_USE_SCHED_YIELD
#include <sched.h>
#else
#include <unistd.h>
#endif

#include <list>
#include <algorithm>
#include <functional>

#include <c++-gtk-utils/callback.h>
#include <c++-gtk-utils/param.h>
#include <c++-gtk-utils/mutex.h>
#include <c++-gtk-utils/cgu_config.h>

namespace Cgu {

/* The four basic emitter types */

template <class FreeArg> class EmitterArg;
template <class FreeArg> class SafeEmitterArg;
typedef EmitterArg<void> Emitter;
typedef SafeEmitterArg<void> SafeEmitter;

/** 
 * @class Releaser emitter.h c++-gtk-utils/emitter.h
 * @brief A class used for tracking EmitterArg and SafeEmitterArg
 * connections.
 * @sa EmitterArg SafeEmitterArg
 * @sa emitter.h
 * @sa Callback namespace
 *
 * This class provides tracking of EmitterArg and SafeEmitterArg
 * connections.  It should be a public member of any target class
 * which wants functors representing any of its methods to be
 * disconnected automatically from an EmitterArg or SafeEmitterArg
 * object when the target class object is destroyed, and is passed as
 * one of the arguments to the connect() method of EmitterArg or
 * SafeEmitterArg.
 *
 * All its methods are thread-safe.
 * 
 * For further background, read this: emitter.h
 */

class Releaser {

  // TODO: In the 2.0 series we use std::unordered_set instead of
  // std::list in Releaser.  At a suitable binary break I think it
  // would be useful to convert to std::set in this 1.2 series.  We
  // can't do that for EmitterArg/SafeEmitterArg objects, as they need
  // to execute connected functors in the order in which they were
  // connected.
  std::list<Callback::SafeFunctor> disconnect_list;
  Thread::Mutex mutex;

  // only an EmitterArg or SafeEmitterArg object can access add(), remove and try_remove()
  void add(const Callback::SafeFunctor&);
  void remove(Callback::SafeFunctor);
  void try_remove(Callback::SafeFunctor, int*);
public:
  template <class T> friend class EmitterArg;
  template <class T> friend class SafeEmitterArg;

  // operator=() and the copy constructor should copy nothing from the
  // assignor, because disconnect_list should be empty in the
  // sssignee, as any class containing us does not acquire as assignee
  // any emitter functors representing any of its methods

/**
 * See notes on @ref AssignmentAnchor "assignment" to see how this
 * operates.  This does not throw provided that the destructors of any
 * bound arguments of a functor managed by this Releaser object prior
 * to assignment do not throw (as they should not do), and assuming
 * that merely iterating through a list does not throw (as it would
 * not on any sane implementation).
 * @param r The assignee.
 */
  Releaser& operator=(const Releaser& r);

/**
 * This does not copy anything from the Releaser object passed as an
 * argument - see the notes on @ref AssignmentAnchor "assignment" for
 * an explanation of why.  This does not throw.
 * @param r A Releaser object.
 * @exception std::bad_alloc This constructor might throw
 * std::bad_alloc if memory is exhausted and the system throws in that
 * case.
 * @exception Thread::MutexError This constructor might throw
 * Thread::MutexError if initialisation of the contained mutex fails.
 * (It is often not worth checking for this, as it means either memory
 * is exhausted or pthread has run out of other resources to create
 * new mutexes.)
 */
  Releaser(const Releaser& r) {}

/**
 * @exception std::bad_alloc The default constructor might throw
 * std::bad_alloc if memory is exhausted and the system throws in that
 * case.
 * @exception Thread::MutexError The default constructor might throw
 * Thread::MutexError if initialisation of the contained mutex fails.
 * (It is often not worth checking for this, as it means either memory
 * is exhausted or pthread has run out of other resources to create
 * new mutexes.)
 */
  Releaser() {}

/**
 * The destructor does not throw provided that the destructors of any
 * bound arguments of a functor managed by this Releaser object do not
 * throw (as they should not do), and assuming that merely iterating
 * through a list does not throw (as it would not on any sane
 * implementation).
 */
  ~Releaser();

/* Only has effect if --with-glib-memory-slices-compat or
 * --with-glib-memory-slices-no-compat option picked */
  CGU_GLIB_MEMORY_SLICES_FUNCS
};

/* the emitter classes */

/**
 * @class EmitterArg emitter.h c++-gtk-utils/emitter.h
 * @brief A class to execute callbacks connected to it, with provision
 * for automatic disconnection.
 * @sa SafeEmitterArg Releaser
 * @sa emitter.h
 * @sa Callback namespace
 *
 * Callback::FunctorArg objects may be connected to Emitter classes,
 * and will be executed when EmitterArg<>::emit() or
 * EmitterArg<>::operator()() are called.
 *
 * One version of the connect() method takes a Releaser object as an
 * argument.  Such a Releaser object should be a public member of any
 * target class which wants functors representing any of its methods
 * to be disconnected automatically from the EmitterArg object when
 * the target class object is destroyed.
 *
 * A connection may be explicitly disconnected by calling the
 * disconnect() method, and may also be temporarily blocked and
 * subsequently unblocked with the block() and unblock() methods.
 *
 * The template type is the type of the unbound argument, if any.
 * EmitterArg<void> is typedef'ed to Emitter.
 *
 * @b Usage
 *
 * For a class my_obj of type MyObj, with a method void
 * MyObj::my_method(int, const char*), usage for a fully bound functor
 * and emitter would be:
 *
 * @code
 *   using namespace Cgu;
 *   int arg = 1;
 *   Emitter e;
 *   e.connect(Callback::make(my_obj, &MyObj::my_method, arg, "Hello\n"));
 *   e();
 * @endcode
 *
 * Or for a partially bound functor and emitter:
 *
 * @code
 *   using namespace Cgu;
 *   int arg = 1;
 *   EmitterArg<const char*> e;
 *   e.connect(Callback::make(my_obj, &MyObj::my_method, arg));
 *   e("Hello\n");
 * @endcode
 * 
 * For further background, including about thread-safety, exception
 * safety, the use of the Cgu::TypeTuple struct for calls involving
 * two or three unbound arguments and other matters, read this:
 * emitter.h
 */

template <class FreeArg>
class EmitterArg {

#ifndef DOXYGEN_PARSING
  // f1 is the functor we execute when we emit()
  // f2 is the functor we execute in our destructor if we are destroyed
  // before the remote object is
  struct ListItem {
    Callback::FunctorArg<FreeArg> f1;
    Callback::Functor f2;
    bool blocked;
    ListItem(Callback::FunctorArg<FreeArg> f1_, Callback::Functor f2_):
      f1(f1_), f2(f2_), blocked(false) {}
  };
#endif

  std::list<ListItem> emission_list;

  // only Releaser objects can access this
  void tracking_disconnect(Callback::FunctorArg<FreeArg>);

  // emitters cannot be copied
  EmitterArg(const EmitterArg&);
  EmitterArg& operator=(const EmitterArg&);
public:
  friend class Releaser;

/**
 * This will execute the connected functors.
 * @param arg The argument to be passed to the referenced function or
 * class method, if any.
 * @exception std::bad_alloc The method might throw std::bad_alloc if
 * memory is exhausted and the system throws in that case.  In
 * addition, it will throw if the function or class methods
 * referenced by the functors throw (or if the assignment operator of
 * the free or a bound argument throws and it is not a reference
 * argument).
 * @note This function is specialised for Emitter (ie
 * EmitterArg<void>) to take no argument, and for
 * EmitterArg<TypeTuple<T1, T2> > and EmitterArg<TypeTuple<T1, T2, T3>
 * > to take two and three arguments respectively.
 */
  void operator()(typename Cgu::Param<FreeArg>::ParamType arg) const {emit(arg);}

/**
 * This will execute the connected functors.
 * @param arg The argument to be passed to the referenced function or
 * class method, if any.
 * @exception std::bad_alloc The method might throw std::bad_alloc if
 * memory is exhausted and the system throws in that case.  In
 * addition, it will throw if the function or class methods
 * referenced by the functors throw (or if the assignment operator of
 * the free or a bound argument throws and it is not a reference
 * argument).
 * @note This function is specialised for Emitter (ie
 * EmitterArg<void>) to take no argument, and for
 * EmitterArg<TypeTuple<T1, T2> > and EmitterArg<TypeTuple<T1, T2, T3>
 * > to take two and three arguments respectively.
 */
  void emit(typename Cgu::Param<FreeArg>::ParamType arg) const;

/**
 * This will execute the connected functors, but it also reports
 * whether in fact there were any connected functors to execute.  (It
 * is not necessary to use this function just because it is not known
 * whether a functor is connected - if the standard emit() function is
 * called when no functor is connected, nothing will happen.  The
 * feature of this method is that it will report the outcome.)
 * @param arg The argument to be passed to the connected functions or
 * class methods, if any.
 * @return Returns false if there were no functors to execute, or true
 * if functors have been executed.
 * @exception std::bad_alloc The method might throw std::bad_alloc if
 * memory is exhausted and the system throws in that case.  In
 * addition, it will throw if the function or class methods
 * referenced by the functors throw (or if the assignment operator of
 * the free or a bound argument throws and it is not a reference
 * argument).
 * @note This function is specialised for Emitter (ie
 * EmitterArg<void>) to take no argument, and for
 * EmitterArg<TypeTuple<T1, T2> > and EmitterArg<TypeTuple<T1, T2, T3>
 * > to take two and three arguments respectively.
 */
  bool test_emit(typename Cgu::Param<FreeArg>::ParamType arg) const;

/**
 * Connects a functor.
 * @param f The functor to connect.
 * @return The functor connected.
 * @exception std::bad_alloc The method might throw std::bad_alloc if
 * memory is exhausted and the system throws in that case.
 * @note This function is specialised for Emitter (ie
 * EmitterArg<void>) to take and return a Callback::Functor argument,
 * and for EmitterArg<TypeTuple<T1, T2> > and EmitterArg<TypeTuple<T1,
 * T2, T3> > to take and return a Callback::FunctorArg<TypeTuple<T1,
 * T2> > and Callback::FunctorArg<TypeTuple<T1, T2, T3> > argument
 * respectively.
 */
  Callback::FunctorArg<FreeArg> connect(const Callback::FunctorArg<FreeArg>& f);

/**
 * Connects a functor.
 * @param f The functor to connect.
 * @param r A Releaser object for automatic disconnection of the
 * functor if the object whose method it represents is destroyed.
 * @return The functor connected.
 * @exception std::bad_alloc The method might throw std::bad_alloc if
 * memory is exhausted and the system throws in that case.
 * @note This function is specialised for Emitter (ie
 * EmitterArg<void>) to take and return a Callback::Functor argument,
 * and for EmitterArg<TypeTuple<T1, T2> > and EmitterArg<TypeTuple<T1,
 * T2, T3> > to take and return a Callback::FunctorArg<TypeTuple<T1,
 * T2> > and Callback::FunctorArg<TypeTuple<T1, T2, T3> > argument
 * respectively.
 */
  Callback::FunctorArg<FreeArg> connect(const Callback::FunctorArg<FreeArg>& f, Releaser& r);

/**
 * Disconnects a functor previously connected. This does not throw
 * provided that the destructors of any bound arguments do not throw
 * (as they should not do), and assuming that merely iterating through
 * a list does not throw (as it would not on any sane implementation).
 * @param f The functor to disconnect.
 * @note 1. If the same functor has been connected more than once to
 * the same EmitterArg object, this call will disconnect all of them.
 * @note 2. This function is specialised for Emitter (ie
 * EmitterArg<void>) to take a Callback::Functor argument, and for
 * EmitterArg<TypeTuple<T1, T2> > and EmitterArg<TypeTuple<T1, T2, T3>
 * > to take a Callback::FunctorArg<TypeTuple<T1, T2> > and
 * Callback::FunctorArg<TypeTuple<T1, T2, T3> > argument respectively.
 */
  void disconnect(const Callback::FunctorArg<FreeArg>& f);

/**
 * Blocks a connected functor from executing when emit() or
 * operator()() is called until unblock() is called.  This method does
 * not throw (assuming that merely iterating through a list does not
 * throw, as it would not on any sane implementation).
 * @param f The functor to block.
 * @note 1. If the same functor has been connected more than once to
 * the same EmitterArg object, this call will block all of them.
 * @note 2. This function is specialised for Emitter (ie
 * EmitterArg<void>) to take a Callback::Functor argument, and for
 * EmitterArg<TypeTuple<T1, T2> > and EmitterArg<TypeTuple<T1, T2, T3>
 * > to take a Callback::FunctorArg<TypeTuple<T1, T2> > and
 * Callback::FunctorArg<TypeTuple<T1, T2, T3> > argument respectively.
 */
  void block(const Callback::FunctorArg<FreeArg>& f);

/**
 * Unblocks a previously blocked functor.  This method does not throw
 * (assuming that merely iterating through a list does not throw, as
 * it would not on any sane implementation).
 * @param f The functor to unblock.
 * @note 1. If the same functor has been connected more than once to
 * the same EmitterArg object, this call will unblock all of them.
 * @note 2. This function is specialised for Emitter (ie
 * EmitterArg<void>) to take a Callback::Functor argument, and for
 * EmitterArg<TypeTuple<T1, T2> > and EmitterArg<TypeTuple<T1, T2, T3>
 * > to take a Callback::FunctorArg<TypeTuple<T1, T2> > and
 * Callback::FunctorArg<TypeTuple<T1, T2, T3> > argument respectively.
 */
  void unblock(const Callback::FunctorArg<FreeArg>& f);

/**
 * @exception std::bad_alloc The constructor might throw
 * std::bad_alloc if memory is exhausted and the system throws in that
 * case.
 */
  EmitterArg() {}

/**
 * The destructor does not throw provided that the destructors of any
 * bound arguments do not throw (as they should not do), and assuming
 * that merely iterating through a list does not throw (as it would
 * not on any sane implementation).
 */
  ~EmitterArg();

/* Only has effect if --with-glib-memory-slices-compat or
 * --with-glib-memory-slices-no-compat option picked */
  CGU_GLIB_MEMORY_SLICES_FUNCS
};

template <class FreeArg>
EmitterArg<FreeArg>::~EmitterArg() {

  struct DisconnectReleaserItem {
    static void exec(const ListItem& l) {(l.f2)();}
  };
  std::for_each(emission_list.begin(), emission_list.end(),
		DisconnectReleaserItem::exec);
}

template <class FreeArg>
void EmitterArg<FreeArg>::emit(typename Cgu::Param<FreeArg>::ParamType arg) const {

  // create a local copy of emission_list, to enable a connected
  // function (i) to delete the EmitterArg<> object to which it is
  // connected, even if there are other functors still to execute in
  // the same emission (which will execute normally provided they do
  // not try to call any of the emitter's functions), (ii) to call
  // 'delete this' nothwithstanding that the connected function is
  // protected by a Releaser object (assuming all the other restraints
  // on calling 'delete this' are met), provided that no other access
  // would be made to the deleted object in a function call connected
  // to the same emitter which is due to execute subsequently in the
  // same emission, and (iii) to disconnect itself from the
  // EmitterArg<> object.  This design approach has a trade-off: if a
  // connected function tries to block, unblock or disconnect another
  // function connected to the same EmitterArg<> object which is due
  // to execute subsequently in the same emission (or to block,
  // unblock or disconnect itself when it is due to execute again
  // subsequently in the same emission), the attempted block, unblock
  // or disconnection will not have any effect on that emission (it
  // will only have effect on a subsequent emission).  In addition, a
  // connected function may not destroy an object whose non-static
  // method is connected to the same emitter and which would execute
  // subsequently in the same emission, even if that object is
  // protected by a Releaser object (the non-static method will
  // unsuccessfully attempt to execute notwithstanding the destruction
  // of the object it would be operating on).

  std::list<ListItem> local_list = emission_list;

  // don't use std::bind2nd()/std::ptr_fun() with std::for_each()
  // or we cannot use reference arguments - iterate by hand
  typename std::list<ListItem>::const_iterator iter;
  for (iter = local_list.begin(); iter != local_list.end(); ++iter) {
    if (!iter->blocked) iter->f1(arg);
  }
}

template <class FreeArg>
bool EmitterArg<FreeArg>::test_emit(typename Cgu::Param<FreeArg>::ParamType arg) const {
  if (emission_list.empty()) return false;
  emit(arg);
  return true;
}

template <class FreeArg>
Callback::FunctorArg<FreeArg> EmitterArg<FreeArg>::connect(const Callback::FunctorArg<FreeArg>& f1) {
  emission_list.push_back(ListItem(f1, Callback::Functor()));
  return f1;
}

template <class FreeArg>
Callback::FunctorArg<FreeArg> EmitterArg<FreeArg>::connect(const Callback::FunctorArg<FreeArg>& f1, Releaser& r) {
  // In this method:
  // f1 is the functor we execute when we emit()
  // f2 is the functor we execute in our destructor if we are destroyed before the
  // remote object is
  // f3 is the functor the remote object executes in its Releaser if it is destroyed
  // before we are, or if Releaser::operator=() is called

  Callback::SafeFunctor f3(Callback::make_val(*this, &EmitterArg<FreeArg>::tracking_disconnect, f1));
  Callback::Functor f2(Callback::make_val(r, &Releaser::remove, f3));
  r.add(f3);
  try {
    emission_list.push_back(ListItem(f1, f2));
  }
  catch (...) {
    r.remove(f3);
    throw;
  }
  return f1;
}

template <class FreeArg>
void EmitterArg<FreeArg>::disconnect(const Callback::FunctorArg<FreeArg>& arg) {
  // we don't have lambdas in C++03, so use a struct in function scope
  struct Pred {
    // we can't pass const reference types as we bind with std::bind2nd below
    static bool pred(ListItem p, Callback::FunctorArg<FreeArg> f) {
      return (p.f1 == f);
    }
  };

  // in theory, we could have connected the same functor object
  // more than once, so cater for that
  typename std::list<ListItem>::iterator iter = emission_list.begin();
  for (;;) {
    iter = std::find_if(iter, emission_list.end(),
			std::bind2nd(std::ptr_fun(Pred::pred), arg));
    if (iter != emission_list.end()) {
      // remove ourselves from the remote Releaser object
      (iter->f2)();

      // remove this item from emission_list
      iter = emission_list.erase(iter);
    }
    else break;
  }
}

// tracking disconnect() is the same as disconnect(), except that we do not
// execute f2 as the remote Releaser object will destroy its own functors
// in that case
template <class FreeArg>
void EmitterArg<FreeArg>::tracking_disconnect(Callback::FunctorArg<FreeArg> arg) {
  // we don't have lambdas in C++03, so use a struct in function scope
  struct Pred {
    // we can't pass const reference types as we bind with std::bind2nd below
    static bool pred(ListItem p, Callback::FunctorArg<FreeArg> f) {
      return (p.f1 == f);
    }
  };

  typename std::list<ListItem>::iterator iter = emission_list.begin();
  for (;;) {
    iter = std::find_if(iter, emission_list.end(),
			std::bind2nd(std::ptr_fun(Pred::pred), arg));
    if (iter != emission_list.end()) {
      // remove this item from emission_list
      iter = emission_list.erase(iter);
    }
    else break;
  }
}

template <class FreeArg>
void EmitterArg<FreeArg>::block(const Callback::FunctorArg<FreeArg>& arg) {
  // we don't have lambdas in C++03, so use a struct in function scope
  struct Pred {
    // we can't pass const reference types as we bind with std::bind2nd below
    static bool pred(ListItem p, Callback::FunctorArg<FreeArg> f) {
      return (p.f1 == f);
    }
  };

  // in theory, we could have connected the same functor object
  // more than once, so cater for that
  typename std::list<ListItem>::iterator iter = emission_list.begin();
  for (;;) {
    iter = std::find_if(iter, emission_list.end(),
			std::bind2nd(std::ptr_fun(Pred::pred), arg));
    if (iter != emission_list.end()) {
      iter->blocked = true;
      ++iter;
    }
    else break;
  }
}

template <class FreeArg>
void EmitterArg<FreeArg>::unblock(const Callback::FunctorArg<FreeArg>& arg) {
  // we don't have lambdas in C++03, so use a struct in function scope
  struct Pred {
    // we can't pass const reference types as we bind with std::bind2nd below
    static bool pred(ListItem p, Callback::FunctorArg<FreeArg> f) {
      return (p.f1 == f);
    }
  };

  // in theory, we could have connected the same functor object
  // more than once, so cater for that
  typename std::list<ListItem>::iterator iter = emission_list.begin();
  for (;;) {
    iter = std::find_if(iter, emission_list.end(),
			std::bind2nd(std::ptr_fun(Pred::pred), arg));
    if (iter != emission_list.end()) {
      iter->blocked = false;
      ++iter;
    }
    else break;
  }
}

/**
 * @class SafeEmitterArg emitter.h c++-gtk-utils/emitter.h
 * @brief A thread-safe class to execute callbacks connected to it,
 * with provision for automatic disconnection.
 * @sa EmitterArg Releaser
 * @sa emitter.h
 * @sa Callback namespace
 *
 * This is a thread-safe version of the EmitterArg class.
 * Callback::SafeFunctorArg objects may be connected to SafeEmitter
 * classes, and will be executed when SafeEmitterArg<>::emit() or
 * SafeEmitterArg<>::operator()() are called.
 *
 * One version of the connect() method takes a Releaser object as an
 * argument.  Such a Releaser object should be a public member of any
 * target class which wants functors representing any of its methods
 * to be disconnected automatically from the SafeEmitterArg object
 * when the target class object is destroyed.
 *
 * A connection may be explicitly disconnected by calling the
 * disconnect() method, and may also be temporarily blocked and
 * subsequently unblocked with the block() and unblock() methods.
 *
 * The template type is the type of the unbound argument, if any.
 * SafeEmitterArg<void> is typedef'ed to SafeEmitter.
 *
 * @b Usage
 *
 * For a class my_obj of type MyObj, with a method void
 * MyObj::my_method(int, const char*), usage for a fully bound functor
 * and emitter would be:
 *
 * @code
 *   using namespace Cgu;
 *   int arg = 1;
 *   SafeEmitter se;
 *   se.connect(Callback::make(my_obj, &MyObj::my_method, arg, "Hello\n"));
 *   se();
 * @endcode
 *
 * Or for a partially bound functor and emitter:
 *
 * @code
 *   using namespace Cgu;
 *   int arg = 1;
 *   SafeEmitterArg<const char*> se;
 *   se.connect(Callback::make(my_obj, &MyObj::my_method, arg));
 *   se("Hello\n");
 * @endcode
 * 
 * For further background, including about thread-safety, exception
 * safety, the use of the Cgu::TypeTuple struct for calls involving
 * two or three unbound arguments and other matters, read this:
 * emitter.h
 */

template <class FreeArg>
class SafeEmitterArg {

#ifndef DOXYGEN_PARSING
  // f1 is the functor we execute when we emit()
  // f2 is the functor we execute in our destructor if we are destroyed
  // before the remote object is
  struct ListItem {
    Callback::SafeFunctorArg<FreeArg> f1;
    Callback::SafeFunctorArg<int*> f2;
    bool blocked;
    ListItem(Callback::SafeFunctorArg<FreeArg> f1_, Callback::SafeFunctorArg<int*> f2_):
      f1(f1_), f2(f2_), blocked(false) {}
  };
#endif
  
  std::list<ListItem> emission_list;
  mutable Thread::Mutex mutex; // make this mutable so we can lock/unlock in const methods

  // only Releaser objects can access this
  void tracking_disconnect(Callback::SafeFunctorArg<FreeArg>);

  // emitters cannot be copied
  SafeEmitterArg(const SafeEmitterArg&);
  SafeEmitterArg& operator=(const SafeEmitterArg&);
public:
  friend class Releaser;

/**
 * This will execute the connected functors.  It is thread safe if the
 * functions or class methods referenced by the connected functors are
 * thread safe.
 * @param arg The argument, if any, to be passed to the functions or
 * class methods referenced by the connected functors.
 * @exception std::bad_alloc The method might throw std::bad_alloc if
 * memory is exhausted and the system throws in that case.  In
 * addition, it will throw if the functions or class methods
 * referenced by the functors throw (or if the assignment operator of
 * the free or a bound argument throws and it is not a reference
 * argument).
 * @note This function is specialised for SafeEmitter (ie
 * SafeEmitterArg<void>) to take no argument, and for
 * SafeEmitterArg<TypeTuple<T1, T2> > and SafeEmitterArg<TypeTuple<T1,
 * T2, T3> > to take two and three arguments respectively.
 */
  void operator()(typename Cgu::Param<FreeArg>::ParamType arg) const {emit(arg);}

/**
 * This will execute the connected functors.  It is thread safe if the
 * functions or class methods referenced by the connected functors are
 * thread safe.
 * @param arg The argument to be passed to the referenced function or
 * class method, if any.
 * @exception std::bad_alloc The method might throw std::bad_alloc if
 * memory is exhausted and the system throws in that case.  In
 * addition, it will throw if the function or class methods
 * referenced by the functors throw (or if the assignment operator of
 * the free or a bound argument throws and it is not a reference
 * argument).
 * @note This function is specialised for SafeEmitter (ie
 * SafeEmitterArg<void>) to take no argument, and for
 * SafeEmitterArg<TypeTuple<T1, T2> > and SafeEmitterArg<TypeTuple<T1,
 * T2, T3> > to take two and three arguments respectively.
 */
  void emit(typename Cgu::Param<FreeArg>::ParamType arg) const;

/**
 * This will execute the connected functors, but it also reports
 * whether in fact there were any connected functors to execute.  It
 * is thread safe if the functions or class methods referenced by the
 * connected functors are thread safe.  (It is not necessary to use
 * this function just because it is not known whether a functor is
 * connected - if the standard emit() function is called when no
 * functor is connected, nothing will happen.  The feature of this
 * method is that it will report the outcome.)
 * @param arg The argument to be passed to the connected functions or
 * class methods, if any.
 * @return Returns false if there were no functors to execute, or true
 * if functors have been executed.
 * @exception std::bad_alloc The method might throw std::bad_alloc if
 * memory is exhausted and the system throws in that case.  In
 * addition, it will throw if the function or class methods
 * referenced by the functors throw (or if the assignment operator of
 * the free or a bound argument throws and it is not a reference
 * argument).
 * @note This function is specialised for SafeEmitter (ie
 * SafeEmitterArg<void>) to take no argument, and for
 * SafeEmitterArg<TypeTuple<T1, T2> > and SafeEmitterArg<TypeTuple<T1,
 * T2, T3> > to take two and three arguments respectively.
 */
  bool test_emit(typename Cgu::Param<FreeArg>::ParamType arg) const;

/**
 * Connects a functor.  It is thread safe.
 * @param f The functor to connect.
 * @return The functor connected.
 * @exception std::bad_alloc The method might throw std::bad_alloc if
 * memory is exhausted and the system throws in that case.
 * @note This function is specialised for SafeEmitter (ie
 * SafeEmitterArg<void>) to take and return a Callback::SafeFunctor
 * argument, and for SafeEmitterArg<TypeTuple<T1, T2> > and
 * SafeEmitterArg<TypeTuple<T1, T2, T3> > to take and return a
 * Callback::SafeFunctorArg<TypeTuple<T1, T2> > and
 * Callback::SafeFunctorArg<TypeTuple<T1, T2, T3> > argument
 * respectively.
 */
  Callback::SafeFunctorArg<FreeArg> connect(const Callback::SafeFunctorArg<FreeArg>& f);

/**
 * Connects a functor.  It is thread safe.
 * @param f The functor to connect.
 * @param r A Releaser object for automatic disconnection of the
 * functor if the object whose method it represents is destroyed.
 * @return The functor connected.
 * @exception std::bad_alloc The method might throw std::bad_alloc if
 * memory is exhausted and the system throws in that case.
 * @note This function is specialised for SafeEmitter (ie
 * SafeEmitterArg<void>) to take and return a Callback::SafeFunctor
 * argument, and for SafeEmitterArg<TypeTuple<T1, T2> > and
 * SafeEmitterArg<TypeTuple<T1, T2, T3> > to take and return a
 * Callback::SafeFunctorArg<TypeTuple<T1, T2> > and
 * Callback::SafeFunctorArg<TypeTuple<T1, T2, T3> > argument
 * respectively.
 */
  Callback::SafeFunctorArg<FreeArg> connect(const Callback::SafeFunctorArg<FreeArg>& f, Releaser& r);

/**
 * Disconnects a functor previously connected. This does not throw
 * provided that the destructors of any bound arguments do not throw
 * (as they should not do), and assuming that merely iterating through
 * a list does not throw (as it would not on any sane implementation).
 * It is thread safe.
 * @param f The functor to disconnect.
 * @note 1. If the same functor has been connected more than once to
 * the same SafeEmitterArg object, this call will disconnect all of
 * them.
 * @note 2. This function is specialised for SafeEmitter (ie
 * SafeEmitterArg<void>) to take a Callback::SafeFunctor argument, and
 * for SafeEmitterArg<TypeTuple<T1, T2> > and
 * SafeEmitterArg<TypeTuple<T1, T2, T3> > to take a
 * Callback::SafeFunctorArg<TypeTuple<T1, T2> > and
 * Callback::SafeFunctorArg<TypeTuple<T1, T2, T3> > argument
 * respectively.
 */
  void disconnect(const Callback::SafeFunctorArg<FreeArg>& f);

/**
 * Blocks a connected functor from executing when emit() or
 * operator()() is called until unblock() is called.  This method does
 * not throw (assuming that merely iterating through a list does not
 * throw, as it would not on any sane implementation).  It is thread
 * safe.
 * @param f The functor to block.
 * @note 1. If the same functor has been connected more than once to
 * the same SafeEmitterArg object, this call will block all of them.
 * @note 2. This function is specialised for SafeEmitter (ie
 * SafeEmitterArg<void>) to take a Callback::SafeFunctor argument, and
 * for SafeEmitterArg<TypeTuple<T1, T2> > and
 * SafeEmitterArg<TypeTuple<T1, T2, T3> > to take a
 * Callback::SafeFunctorArg<TypeTuple<T1, T2> > and
 * Callback::SafeFunctorArg<TypeTuple<T1, T2, T3> > argument
 * respectively.
 */
  void block(const Callback::SafeFunctorArg<FreeArg>& f);

/**
 * Unblocks a previously blocked functor.  This method does not throw
 * (assuming that merely iterating through a list does not throw, as
 * it would not on any sane implementation).  It is thread safe.
 * @param f The functor to unblock.
 * @note 1. If the same functor has been connected more than once to
 * the same SafeEmitterArg object, this call will unblock all of them.
 * @note 2. This function is specialised for SafeEmitter (ie
 * SafeEmitterArg<void>) to take a Callback::SafeFunctor argument, and
 * for SafeEmitterArg<TypeTuple<T1, T2> > and
 * SafeEmitterArg<TypeTuple<T1, T2, T3> > to take a
 * Callback::SafeFunctorArg<TypeTuple<T1, T2> > and
 * Callback::SafeFunctorArg<TypeTuple<T1, T2, T3> > argument
 * respectively.
 */
  void unblock(const Callback::SafeFunctorArg<FreeArg>& f);

/**
 * @exception std::bad_alloc The constructor might throw
 * std::bad_alloc if memory is exhausted and the system throws in that
 * case.
 * @exception Thread::MutexError The default constructor might throw
 * Thread::MutexError if initialisation of the contained mutex fails.
 * (It is often not worth checking for this, as it means either memory
 * is exhausted or pthread has run out of other resources to create
 * new mutexes.)
 */
  SafeEmitterArg() {}

/**
 * The destructor does not throw provided that the destructors of any
 * bound arguments do not throw (as they should not do), and assuming
 * that merely iterating through a list does not throw (as it would
 * not on any sane implementation).  It is thread-safe as regards the
 * dropping of any connected functors and of any relevant Releaser
 * objects.
 */
  ~SafeEmitterArg();

/* Only has effect if --with-glib-memory-slices-compat or
 * --with-glib-memory-slices-no-compat option picked */
  CGU_GLIB_MEMORY_SLICES_FUNCS
};

template <class FreeArg>
SafeEmitterArg<FreeArg>::~SafeEmitterArg() {

  // go through emission_list() item by item, popping off the front and erasing
  // as we go in case Releaser::try_remove() fails to acquire the lock on one
  // of the iterations
  Thread::Mutex::Lock lock(mutex);
  while (!emission_list.empty()) {
    typename std::list<ListItem>::iterator iter = emission_list.begin();
    int result = 0; // f2 might be a no-op
    // remove ourselves from the remote Releaser object
    (iter->f2)(&result);
    if (!result) { // we got the Releaser mutex lock or no-op
      // now remove this item from emission_list
      emission_list.erase(iter);
    }
    else {
      mutex.unlock();
      // spin nicely
#ifdef CGU_USE_SCHED_YIELD
      sched_yield();
#else
      usleep(10);
#endif
      mutex.lock();
    }
  }
}

template <class FreeArg>
void SafeEmitterArg<FreeArg>::emit(typename Cgu::Param<FreeArg>::ParamType arg) const {

  // create a local copy of emission_list, to enable a connected
  // function (i) to delete the EmitterArg<> object to which it is
  // connected, even if there are other functors still to execute in
  // the same emission (which will execute normally provided they do
  // not try to call any of the emitter's functions), (ii) to call
  // 'delete this' nothwithstanding that the connected function is
  // protected by a Releaser object (assuming all the other restraints
  // on calling 'delete this' are met), provided that no other access
  // would be made to the deleted object in a function call connected
  // to the same emitter which is due to execute subsequently in the
  // same emission, and (iii) to disconnect itself from the
  // EmitterArg<> object.  This design approach has a trade-off: if a
  // connected function tries to block, unblock or disconnect another
  // function connected to the same EmitterArg<> object which is due
  // to execute subsequently in the same emission (or to block,
  // unblock or disconnect itself when it is due to execute again
  // subsequently in the same emission), the attempted block, unblock
  // or disconnection will not have any effect on that emission (it
  // will only have effect on a subsequent emission).  In addition, a
  // connected function may not destroy an object whose non-static
  // method is connected to the same emitter and which would execute
  // subsequently in the same emission, even if that object is
  // protected by a Releaser object (the non-static method will
  // unsuccessfully attempt to execute notwithstanding the destruction
  // of the object it would be operating on).

  // SafeFunctorArg<> usage has the additional point that while an
  // emission is in course, another thread should not try to do any of
  // those things, or the same outcome will result.  Another thread
  // should leave alone objects connected to a SafeEmitterArg<> object
  // from the time of operator()() or emit() beginning to the time of
  // it ending, and not try to interfere.

  // a side effect of having a local list is that, as required, we
  // will not be holding our mutex when executing the functors it
  // contains.  It is OK having the functors in two different lists
  // which are potentially (when our mutex is released) in two
  // different threads, because the functors hold their
  // Callback::Callback objects by SharedLockPtr so their reference
  // count is protected (they are SafeFunctorArg<> functors).

  std::list<ListItem> local_list;
  { // scope block for mutex lock
    Thread::Mutex::Lock lock(mutex);
    local_list = emission_list;
  }

  // don't use std::bind2nd()/std::ptr_fun() with std::for_each()
  // or we cannot use reference arguments - iterate by hand
  typename std::list<ListItem>::const_iterator iter;
  for (iter = local_list.begin(); iter != local_list.end(); ++iter) {
    if (!iter->blocked) iter->f1(arg);
  }
}

template <class FreeArg>
bool SafeEmitterArg<FreeArg>::test_emit(typename Cgu::Param<FreeArg>::ParamType arg) const {

  std::list<ListItem> local_list;
  { // scope block for mutex lock
    Thread::Mutex::Lock lock(mutex);
    if (emission_list.empty()) return false;
    local_list = emission_list;
  }

  // don't use std::bind2nd()/std::ptr_fun() with std::for_each()
  // or we cannot use reference arguments - iterate by hand
  typename std::list<ListItem>::const_iterator iter;
  for (iter = local_list.begin(); iter != local_list.end(); ++iter) {
    if (!iter->blocked) iter->f1(arg);
  }
  return true;
}

template <class FreeArg>
Callback::SafeFunctorArg<FreeArg> SafeEmitterArg<FreeArg>::connect(const Callback::SafeFunctorArg<FreeArg>& f1) {
  Thread::Mutex::Lock lock(mutex);
  emission_list.push_back(ListItem(f1, Callback::SafeFunctorArg<int*>()));
  return f1;
}

template <class FreeArg>
Callback::SafeFunctorArg<FreeArg> SafeEmitterArg<FreeArg>::connect(const Callback::SafeFunctorArg<FreeArg>& f1, Releaser& r) {
  // In this method:
  // f1 is the functor we execute when we emit()
  // f2 is the functor we execute in our destructor if we are destroyed before the
  // remote object is
  // f3 is the functor the remote object executes in its Releaser if it is destroyed
  // before we are, or if Releaser::operator=() is called

  Callback::SafeFunctor f3(Callback::make_val(*this, &SafeEmitterArg<FreeArg>::tracking_disconnect, f1));
  Callback::SafeFunctorArg<int*> f2(Callback::make_val(r, &Releaser::try_remove, f3));
  // we can't call Releaser::add() when holding our mutex or we will
  // get out of order locking, as Releaser's mutex is held when it
  // calls SafeEmitterArg<FreeArg>::tracking_disconnect() via f3, and
  // we don't need to do so
  r.add(f3);
  Thread::Mutex::Lock lock(mutex);
  try {
    emission_list.push_back(ListItem(f1, f2));
  }
  catch (...) {
    mutex.unlock();
    r.remove(f3);
    mutex.lock();
    throw;
  }
  return f1;
}

template <class FreeArg>
void SafeEmitterArg<FreeArg>::disconnect(const Callback::SafeFunctorArg<FreeArg>& arg) {
  // we don't have lambdas in C++03, so use a struct in function scope
  struct Pred {
    // we can't pass const reference types as we bind with std::bind2nd below
    static bool pred(ListItem p, Callback::SafeFunctorArg<FreeArg> f) {
      return (p.f1 == f);
    }
  };

  // in theory, we could have connected the same functor object more than
  // once, so cater for that as well as Releaser::try_remove() failing
  Thread::Mutex::Lock lock(mutex);
  typename std::list<ListItem>::iterator iter = emission_list.begin();
  for(;;) {
    iter = std::find_if(iter, emission_list.end(),
                        std::bind2nd(std::ptr_fun(Pred::pred), arg));
    if (iter != emission_list.end()) {
      int result = 0; // f2 might be a no-op
      // remove ourselves from the remote Releaser object
      (iter->f2)(&result);
      if (!result) { // we got the Releaser mutex lock or no-op
	// now remove this item from emission_list
        iter = emission_list.erase(iter);
      }
      else {
        mutex.unlock();
        // spin nicely
#ifdef CGU_USE_SCHED_YIELD
        sched_yield();
#else
        usleep(10);
#endif
	mutex.lock();
	// start again at the beginning - we have released the mutex
	// so our iterator may have become invalid
	iter = emission_list.begin();
      }
    }
    else break;
  }
}

// tracking disconnect() is the same as disconnect(), except that we do not
// execute f2 as the remote Releaser object will destroy its own functors
// in that case
template <class FreeArg>
void SafeEmitterArg<FreeArg>::tracking_disconnect(Callback::SafeFunctorArg<FreeArg> arg) {
  // we don't have lambdas in C++03, so use a struct in function scope
  struct Pred {
    // we can't pass const reference types as we bind with std::bind2nd below
    static bool pred(ListItem p, Callback::SafeFunctorArg<FreeArg> f) {
      return (p.f1 == f);
    }
  };

  Thread::Mutex::Lock lock(mutex);
  typename std::list<ListItem>::iterator iter = emission_list.begin();
  for (;;) {
    iter = std::find_if(iter, emission_list.end(),
			std::bind2nd(std::ptr_fun(Pred::pred), arg));
    if (iter != emission_list.end()) {
      // remove this item from emission_list
      iter = emission_list.erase(iter);
    }
    else break;
  }
}

template <class FreeArg>
void SafeEmitterArg<FreeArg>::block(const Callback::SafeFunctorArg<FreeArg>& arg) {
  // we don't have lambdas in C++03, so use a struct in function scope
  struct Pred {
    // we can't pass const reference types as we bind with std::bind2nd below
    static bool pred(ListItem p, Callback::SafeFunctorArg<FreeArg> f) {
      return (p.f1 == f);
    }
  };

  // in theory, we could have connected the same functor object
  // more than once, so cater for that
  Thread::Mutex::Lock lock(mutex);
  typename std::list<ListItem>::iterator iter = emission_list.begin();
  for (;;) {
    iter = std::find_if(iter, emission_list.end(),
			std::bind2nd(std::ptr_fun(Pred::pred), arg));
    if (iter != emission_list.end()) {
      iter->blocked = true;
      ++iter;
    }
    else break;
  }
}

template <class FreeArg>
void SafeEmitterArg<FreeArg>::unblock(const Callback::SafeFunctorArg<FreeArg>& arg) {
  // we don't have lambdas in C++03, so use a struct in function scope
  struct Pred {
    // we can't pass const reference types as we bind with std::bind2nd below
    static bool pred(ListItem p, Callback::SafeFunctorArg<FreeArg> f) {
      return (p.f1 == f);
    }
  };

  // in theory, we could have connected the same functor object
  // more than once, so cater for that
  Thread::Mutex::Lock lock(mutex);
  typename std::list<ListItem>::iterator iter = emission_list.begin();
  for (;;) {
    iter = std::find_if(iter, emission_list.end(),
			std::bind2nd(std::ptr_fun(Pred::pred), arg));
    if (iter != emission_list.end()) {
      iter->blocked = false;
      ++iter;
    }
    else break;
  }
}

#ifndef DOXYGEN_PARSING

/* specialisations of EmitterArg and SafeEmitterArg for 0, 2 and 3
   unbound arguments */

/* TODO: we could implement this better by passing all the ListItem
 * operations (including in the connect(), disconnect(), block() and
 * unblock() methods and so forth) into a separate templated class
 * which could be incorporated by composition or inheritance into
 * EmitterArg, which would avoid much of the code duplication arising
 * from the specialisations for 2 and 3 unbound arguments as added in
 * version 1.2.10: the only functions we actually need to specialise
 * for these additional arguments are the emit() ones.  However, as
 * these classes are not PODSs this would give rise to binary breakage
 * of 1.2.9 and earlier, so we have to await a suitable ABI breakage
 * release to clean this up.
 */

template <>
class EmitterArg<void> {
  struct ListItem {
    Callback::Functor f1;
    Callback::Functor f2;
    bool blocked;
    ListItem(Callback::Functor f1_, Callback::Functor f2_):
      f1(f1_), f2(f2_), blocked(false) {}
  };
  
  std::list<ListItem> emission_list;

  void tracking_disconnect(Callback::Functor);

  EmitterArg(const EmitterArg&);
  EmitterArg& operator=(const EmitterArg&);
public:
  friend class Releaser;

  void operator()() const {emit();}
  void emit() const;

  bool test_emit() const;

  Callback::Functor connect(const Callback::Functor&);
  Callback::Functor connect(const Callback::Functor&, Releaser&);
  void disconnect(const Callback::Functor&);

  void block(const Callback::Functor&);
  void unblock(const Callback::Functor&);

  EmitterArg() {}
  ~EmitterArg();

/* Only has effect if --with-glib-memory-slices-compat or
 * --with-glib-memory-slices-no-compat option picked */
  CGU_GLIB_MEMORY_SLICES_FUNCS
};

template <class FreeArg1, class FreeArg2>
class EmitterArg <TypeTuple<FreeArg1, FreeArg2> > {
  struct ListItem {
    Callback::FunctorArg<TypeTuple<FreeArg1, FreeArg2> > f1;
    Callback::Functor f2;
    bool blocked;
    ListItem(Callback::FunctorArg<TypeTuple<FreeArg1, FreeArg2> > f1_, Callback::Functor f2_):
      f1(f1_), f2(f2_), blocked(false) {}
  };

  std::list<ListItem> emission_list;

  void tracking_disconnect(Callback::FunctorArg<TypeTuple<FreeArg1, FreeArg2> >);

  EmitterArg(const EmitterArg&);
  EmitterArg& operator=(const EmitterArg&);
public:
  friend class Releaser;

  void operator()(typename Cgu::Param<FreeArg1>::ParamType arg1,
		  typename Cgu::Param<FreeArg2>::ParamType arg2) const {
    emit(arg1, arg2);
  }
  void emit(typename Cgu::Param<FreeArg1>::ParamType arg1,
	    typename Cgu::Param<FreeArg2>::ParamType arg2) const;

  bool test_emit(typename Cgu::Param<FreeArg1>::ParamType arg1,
		 typename Cgu::Param<FreeArg2>::ParamType arg2) const;

  Callback::FunctorArg<TypeTuple<FreeArg1, FreeArg2> > connect(const Callback::FunctorArg<TypeTuple<FreeArg1, FreeArg2> >& f);
  Callback::FunctorArg<TypeTuple<FreeArg1, FreeArg2> > connect(const Callback::FunctorArg<TypeTuple<FreeArg1, FreeArg2> >& f, Releaser& r);
  void disconnect(const Callback::FunctorArg<TypeTuple<FreeArg1, FreeArg2> >& f);

  void block(const Callback::FunctorArg<TypeTuple<FreeArg1, FreeArg2> >& f);
  void unblock(const Callback::FunctorArg<TypeTuple<FreeArg1, FreeArg2> >& f);

  EmitterArg() {}
  ~EmitterArg();

/* Only has effect if --with-glib-memory-slices-compat or
 * --with-glib-memory-slices-no-compat option picked */
  CGU_GLIB_MEMORY_SLICES_FUNCS
};

template <class FreeArg1, class FreeArg2>
EmitterArg<TypeTuple<FreeArg1, FreeArg2> >::~EmitterArg() {

  struct DisconnectReleaserItem {
    static void exec(const ListItem& l) {(l.f2)();}
  };
  std::for_each(emission_list.begin(), emission_list.end(),
		DisconnectReleaserItem::exec);
}

template <class FreeArg1, class FreeArg2>
void EmitterArg<TypeTuple<FreeArg1, FreeArg2> >::emit(typename Cgu::Param<FreeArg1>::ParamType arg1,
						      typename Cgu::Param<FreeArg2>::ParamType arg2) const {
  std::list<ListItem> local_list = emission_list;

  typename std::list<ListItem>::const_iterator iter;
  for (iter = local_list.begin(); iter != local_list.end(); ++iter) {
    if (!iter->blocked) iter->f1(arg1, arg2);
  }
}

template <class FreeArg1, class FreeArg2>
bool EmitterArg<TypeTuple<FreeArg1, FreeArg2> >::test_emit(typename Cgu::Param<FreeArg1>::ParamType arg1,
							   typename Cgu::Param<FreeArg2>::ParamType arg2) const {
  if (emission_list.empty()) return false;
  emit(arg1, arg2);
  return true;
}

template <class FreeArg1, class FreeArg2>
Callback::FunctorArg<TypeTuple<FreeArg1, FreeArg2> > EmitterArg<TypeTuple<FreeArg1, FreeArg2> >::connect(const Callback::FunctorArg<TypeTuple<FreeArg1, FreeArg2> >& f1) {
  emission_list.push_back(ListItem(f1, Callback::Functor()));
  return f1;
}

template <class FreeArg1, class FreeArg2>
Callback::FunctorArg<TypeTuple<FreeArg1, FreeArg2> > EmitterArg<TypeTuple<FreeArg1, FreeArg2> >::connect(const Callback::FunctorArg<TypeTuple<FreeArg1, FreeArg2> >& f1, Releaser& r) {
  // In this method:
  // f1 is the functor we execute when we emit()
  // f2 is the functor we execute in our destructor if we are destroyed before the
  // remote object is
  // f3 is the functor the remote object executes in its Releaser if it is destroyed
  // before we are, or if Releaser::operator=() is called

  Callback::SafeFunctor f3(Callback::make_val(*this, &EmitterArg<TypeTuple<FreeArg1, FreeArg2> >::tracking_disconnect, f1));
  Callback::Functor f2(Callback::make_val(r, &Releaser::remove, f3));
  r.add(f3);
  try {
    emission_list.push_back(ListItem(f1, f2));
  }
  catch (...) {
    r.remove(f3);
    throw;
  }
  return f1;
}

template <class FreeArg1, class FreeArg2>
void EmitterArg<TypeTuple<FreeArg1, FreeArg2> >::disconnect(const Callback::FunctorArg<TypeTuple<FreeArg1, FreeArg2> >& arg) {
  struct Pred {
    static bool pred(ListItem p, Callback::FunctorArg<TypeTuple<FreeArg1, FreeArg2> > f) {
      return (p.f1 == f);
    }
  };

  typename std::list<ListItem>::iterator iter = emission_list.begin();
  for (;;) {
    iter = std::find_if(iter, emission_list.end(),
			std::bind2nd(std::ptr_fun(Pred::pred), arg));
    if (iter != emission_list.end()) {
      (iter->f2)();
      iter = emission_list.erase(iter);
    }
    else break;
  }
}

template <class FreeArg1, class FreeArg2>
void EmitterArg<TypeTuple<FreeArg1, FreeArg2> >::tracking_disconnect(Callback::FunctorArg<TypeTuple<FreeArg1, FreeArg2> > arg) {
  struct Pred {
    static bool pred(ListItem p, Callback::FunctorArg<TypeTuple<FreeArg1, FreeArg2> > f) {
      return (p.f1 == f);
    }
  };

  typename std::list<ListItem>::iterator iter = emission_list.begin();
  for (;;) {
    iter = std::find_if(iter, emission_list.end(),
			std::bind2nd(std::ptr_fun(Pred::pred), arg));
    if (iter != emission_list.end()) {
      iter = emission_list.erase(iter);
    }
    else break;
  }
}

template <class FreeArg1, class FreeArg2>
void EmitterArg<TypeTuple<FreeArg1, FreeArg2> >::block(const Callback::FunctorArg<TypeTuple<FreeArg1, FreeArg2> >& arg) {
  struct Pred {
    static bool pred(ListItem p, Callback::FunctorArg<TypeTuple<FreeArg1, FreeArg2> > f) {
      return (p.f1 == f);
    }
  };

  typename std::list<ListItem>::iterator iter = emission_list.begin();
  for (;;) {
    iter = std::find_if(iter, emission_list.end(),
			std::bind2nd(std::ptr_fun(Pred::pred), arg));
    if (iter != emission_list.end()) {
      iter->blocked = true;
      ++iter;
    }
    else break;
  }
}

template <class FreeArg1, class FreeArg2>
void EmitterArg<TypeTuple<FreeArg1, FreeArg2> >::unblock(const Callback::FunctorArg<TypeTuple<FreeArg1, FreeArg2> >& arg) {
  struct Pred {
    static bool pred(ListItem p, Callback::FunctorArg<TypeTuple<FreeArg1, FreeArg2> > f) {
      return (p.f1 == f);
    }
  };

  typename std::list<ListItem>::iterator iter = emission_list.begin();
  for (;;) {
    iter = std::find_if(iter, emission_list.end(),
			std::bind2nd(std::ptr_fun(Pred::pred), arg));
    if (iter != emission_list.end()) {
      iter->blocked = false;
      ++iter;
    }
    else break;
  }
}

template <class FreeArg1, class FreeArg2, class FreeArg3>
class EmitterArg <TypeTuple<FreeArg1, FreeArg2, FreeArg3> > {
  struct ListItem {
    Callback::FunctorArg<TypeTuple<FreeArg1, FreeArg2, FreeArg3> > f1;
    Callback::Functor f2;
    bool blocked;
    ListItem(Callback::FunctorArg<TypeTuple<FreeArg1, FreeArg2, FreeArg3> > f1_, Callback::Functor f2_):
      f1(f1_), f2(f2_), blocked(false) {}
  };

  std::list<ListItem> emission_list;

  void tracking_disconnect(Callback::FunctorArg<TypeTuple<FreeArg1, FreeArg2, FreeArg3> >);

  EmitterArg(const EmitterArg&);
  EmitterArg& operator=(const EmitterArg&);
public:
  friend class Releaser;

  void operator()(typename Cgu::Param<FreeArg1>::ParamType arg1,
		  typename Cgu::Param<FreeArg2>::ParamType arg2,
		  typename Cgu::Param<FreeArg3>::ParamType arg3) const {
    emit(arg1, arg2, arg3);
  }
  void emit(typename Cgu::Param<FreeArg1>::ParamType arg1,
	    typename Cgu::Param<FreeArg2>::ParamType arg2,
	    typename Cgu::Param<FreeArg3>::ParamType arg3) const;

  bool test_emit(typename Cgu::Param<FreeArg1>::ParamType arg1,
		 typename Cgu::Param<FreeArg2>::ParamType arg2,
		 typename Cgu::Param<FreeArg3>::ParamType arg3) const;

  Callback::FunctorArg<TypeTuple<FreeArg1, FreeArg2, FreeArg3> > connect(const Callback::FunctorArg<TypeTuple<FreeArg1, FreeArg2, FreeArg3> >& f);
  Callback::FunctorArg<TypeTuple<FreeArg1, FreeArg2, FreeArg3> > connect(const Callback::FunctorArg<TypeTuple<FreeArg1, FreeArg2, FreeArg3> >& f, Releaser& r);
  void disconnect(const Callback::FunctorArg<TypeTuple<FreeArg1, FreeArg2, FreeArg3> >& f);

  void block(const Callback::FunctorArg<TypeTuple<FreeArg1, FreeArg2, FreeArg3> >& f);
  void unblock(const Callback::FunctorArg<TypeTuple<FreeArg1, FreeArg2, FreeArg3> >& f);

  EmitterArg() {}
  ~EmitterArg();

/* Only has effect if --with-glib-memory-slices-compat or
 * --with-glib-memory-slices-no-compat option picked */
  CGU_GLIB_MEMORY_SLICES_FUNCS
};

template <class FreeArg1, class FreeArg2, class FreeArg3>
EmitterArg<TypeTuple<FreeArg1, FreeArg2, FreeArg3> >::~EmitterArg() {

  struct DisconnectReleaserItem {
    static void exec(const ListItem& l) {(l.f2)();}
  };
  std::for_each(emission_list.begin(), emission_list.end(),
		DisconnectReleaserItem::exec);
}

template <class FreeArg1, class FreeArg2, class FreeArg3>
void EmitterArg<TypeTuple<FreeArg1, FreeArg2, FreeArg3> >::emit(typename Cgu::Param<FreeArg1>::ParamType arg1,
								typename Cgu::Param<FreeArg2>::ParamType arg2,
								typename Cgu::Param<FreeArg3>::ParamType arg3) const {
  std::list<ListItem> local_list = emission_list;

  typename std::list<ListItem>::const_iterator iter;
  for (iter = local_list.begin(); iter != local_list.end(); ++iter) {
    if (!iter->blocked) iter->f1(arg1, arg2, arg3);
  }
}

template <class FreeArg1, class FreeArg2, class FreeArg3>
bool EmitterArg<TypeTuple<FreeArg1, FreeArg2, FreeArg3> >::test_emit(typename Cgu::Param<FreeArg1>::ParamType arg1,
								     typename Cgu::Param<FreeArg2>::ParamType arg2,
								     typename Cgu::Param<FreeArg3>::ParamType arg3) const {
  if (emission_list.empty()) return false;
  emit(arg1, arg2, arg3);
  return true;
}

template <class FreeArg1, class FreeArg2, class FreeArg3>
Callback::FunctorArg<TypeTuple<FreeArg1, FreeArg2, FreeArg3> > EmitterArg<TypeTuple<FreeArg1, FreeArg2, FreeArg3> >::connect(const Callback::FunctorArg<TypeTuple<FreeArg1, FreeArg2, FreeArg3> >& f1) {
  emission_list.push_back(ListItem(f1, Callback::Functor()));
  return f1;
}

template <class FreeArg1, class FreeArg2, class FreeArg3>
Callback::FunctorArg<TypeTuple<FreeArg1, FreeArg2, FreeArg3> > EmitterArg<TypeTuple<FreeArg1, FreeArg2, FreeArg3> >::connect(const Callback::FunctorArg<TypeTuple<FreeArg1, FreeArg2, FreeArg3> >& f1, Releaser& r) {
  // In this method:
  // f1 is the functor we execute when we emit()
  // f2 is the functor we execute in our destructor if we are destroyed before the
  // remote object is
  // f3 is the functor the remote object executes in its Releaser if it is destroyed
  // before we are, or if Releaser::operator=() is called

  Callback::SafeFunctor f3(Callback::make_val(*this, &EmitterArg<TypeTuple<FreeArg1, FreeArg2, FreeArg3> >::tracking_disconnect, f1));
  Callback::Functor f2(Callback::make_val(r, &Releaser::remove, f3));
  r.add(f3);
  try {
    emission_list.push_back(ListItem(f1, f2));
  }
  catch (...) {
    r.remove(f3);
    throw;
  }
  return f1;
}

template <class FreeArg1, class FreeArg2, class FreeArg3>
void EmitterArg<TypeTuple<FreeArg1, FreeArg2, FreeArg3> >::disconnect(const Callback::FunctorArg<TypeTuple<FreeArg1, FreeArg2, FreeArg3> >& arg) {
  struct Pred {
    static bool pred(ListItem p, Callback::FunctorArg<TypeTuple<FreeArg1, FreeArg2, FreeArg3> > f) {
      return (p.f1 == f);
    }
  };

  typename std::list<ListItem>::iterator iter = emission_list.begin();
  for (;;) {
    iter = std::find_if(iter, emission_list.end(),
			std::bind2nd(std::ptr_fun(Pred::pred), arg));
    if (iter != emission_list.end()) {
      (iter->f2)();
      iter = emission_list.erase(iter);
    }
    else break;
  }
}

template <class FreeArg1, class FreeArg2, class FreeArg3>
void EmitterArg<TypeTuple<FreeArg1, FreeArg2, FreeArg3> >::tracking_disconnect(Callback::FunctorArg<TypeTuple<FreeArg1, FreeArg2, FreeArg3> > arg) {
  struct Pred {
    static bool pred(ListItem p, Callback::FunctorArg<TypeTuple<FreeArg1, FreeArg2, FreeArg3> > f) {
      return (p.f1 == f);
    }
  };

  typename std::list<ListItem>::iterator iter = emission_list.begin();
  for (;;) {
    iter = std::find_if(iter, emission_list.end(),
			std::bind2nd(std::ptr_fun(Pred::pred), arg));
    if (iter != emission_list.end()) {
      iter = emission_list.erase(iter);
    }
    else break;
  }
}

template <class FreeArg1, class FreeArg2, class FreeArg3>
void EmitterArg<TypeTuple<FreeArg1, FreeArg2, FreeArg3> >::block(const Callback::FunctorArg<TypeTuple<FreeArg1, FreeArg2, FreeArg3> >& arg) {
  struct Pred {
    static bool pred(ListItem p, Callback::FunctorArg<TypeTuple<FreeArg1, FreeArg2, FreeArg3> > f) {
      return (p.f1 == f);
    }
  };

  typename std::list<ListItem>::iterator iter = emission_list.begin();
  for (;;) {
    iter = std::find_if(iter, emission_list.end(),
			std::bind2nd(std::ptr_fun(Pred::pred), arg));
    if (iter != emission_list.end()) {
      iter->blocked = true;
      ++iter;
    }
    else break;
  }
}

template <class FreeArg1, class FreeArg2, class FreeArg3>
void EmitterArg<TypeTuple<FreeArg1, FreeArg2, FreeArg3> >::unblock(const Callback::FunctorArg<TypeTuple<FreeArg1, FreeArg2, FreeArg3> >& arg) {
  struct Pred {
    static bool pred(ListItem p, Callback::FunctorArg<TypeTuple<FreeArg1, FreeArg2, FreeArg3> > f) {
      return (p.f1 == f);
    }
  };

  typename std::list<ListItem>::iterator iter = emission_list.begin();
  for (;;) {
    iter = std::find_if(iter, emission_list.end(),
			std::bind2nd(std::ptr_fun(Pred::pred), arg));
    if (iter != emission_list.end()) {
      iter->blocked = false;
      ++iter;
    }
    else break;
  }
}

/* SafeEmitterArg specialisations */

template <>
class SafeEmitterArg<void> {
  struct ListItem {
    Callback::SafeFunctor f1;
    Callback::SafeFunctorArg<int*> f2;
    bool blocked;
    ListItem(Callback::SafeFunctor f1_, Callback::SafeFunctorArg<int*> f2_):
      f1(f1_), f2(f2_), blocked(false) {}
  };
  
  std::list<ListItem> emission_list;
  mutable Thread::Mutex mutex;

  void tracking_disconnect(Callback::SafeFunctor);

  SafeEmitterArg(const SafeEmitterArg&);
  SafeEmitterArg& operator=(const SafeEmitterArg&);
public:
  friend class Releaser;

  void operator()() const {emit();}
  void emit() const;

  bool test_emit() const;

  Callback::SafeFunctor connect(const Callback::SafeFunctor&);
  Callback::SafeFunctor connect(const Callback::SafeFunctor&, Releaser&);
  void disconnect(const Callback::SafeFunctor&);

  void block(const Callback::SafeFunctor&);
  void unblock(const Callback::SafeFunctor&);

  SafeEmitterArg() {}
  ~SafeEmitterArg();

/* Only has effect if --with-glib-memory-slices-compat or
 * --with-glib-memory-slices-no-compat option picked */
  CGU_GLIB_MEMORY_SLICES_FUNCS
};

template <class FreeArg1, class FreeArg2>
class SafeEmitterArg <TypeTuple<FreeArg1, FreeArg2> > {
  struct ListItem {
    Callback::SafeFunctorArg<TypeTuple<FreeArg1, FreeArg2> > f1;
    Callback::SafeFunctorArg<int*> f2;
    bool blocked;
    ListItem(Callback::SafeFunctorArg<TypeTuple<FreeArg1, FreeArg2> > f1_, Callback::SafeFunctorArg<int*> f2_):
      f1(f1_), f2(f2_), blocked(false) {}
  };
  
  std::list<ListItem> emission_list;
  mutable Thread::Mutex mutex;

  void tracking_disconnect(Callback::SafeFunctorArg<TypeTuple<FreeArg1, FreeArg2> >);

  SafeEmitterArg(const SafeEmitterArg&);
  SafeEmitterArg& operator=(const SafeEmitterArg&);
public:
  friend class Releaser;

  void operator()(typename Cgu::Param<FreeArg1>::ParamType arg1,
		  typename Cgu::Param<FreeArg2>::ParamType arg2) const {
    emit(arg1, arg2);
  }
  void emit(typename Cgu::Param<FreeArg1>::ParamType arg1,
	    typename Cgu::Param<FreeArg2>::ParamType arg2) const;

  bool test_emit(typename Cgu::Param<FreeArg1>::ParamType arg1,
		 typename Cgu::Param<FreeArg2>::ParamType arg2) const;

  Callback::SafeFunctorArg<TypeTuple<FreeArg1, FreeArg2> > connect(const Callback::SafeFunctorArg<TypeTuple<FreeArg1, FreeArg2> >& f);
  Callback::SafeFunctorArg<TypeTuple<FreeArg1, FreeArg2> > connect(const Callback::SafeFunctorArg<TypeTuple<FreeArg1, FreeArg2> >& f, Releaser& r);
  void disconnect(const Callback::SafeFunctorArg<TypeTuple<FreeArg1, FreeArg2> >& f);

  void block(const Callback::SafeFunctorArg<TypeTuple<FreeArg1, FreeArg2> >& f);
  void unblock(const Callback::SafeFunctorArg<TypeTuple<FreeArg1, FreeArg2> >& f);

  SafeEmitterArg() {}
  ~SafeEmitterArg();

/* Only has effect if --with-glib-memory-slices-compat or
 * --with-glib-memory-slices-no-compat option picked */
  CGU_GLIB_MEMORY_SLICES_FUNCS
};

template <class FreeArg1, class FreeArg2>
SafeEmitterArg<TypeTuple<FreeArg1, FreeArg2> >::~SafeEmitterArg() {
  Thread::Mutex::Lock lock(mutex);
  while (!emission_list.empty()) {
    typename std::list<ListItem>::iterator iter = emission_list.begin();
    int result = 0;
    (iter->f2)(&result);
    if (!result) {
      emission_list.erase(iter);
    }
    else {
      mutex.unlock();
#ifdef CGU_USE_SCHED_YIELD
      sched_yield();
#else
      usleep(10);
#endif
      mutex.lock();
    }
  }
}

template <class FreeArg1, class FreeArg2>
void SafeEmitterArg<TypeTuple<FreeArg1, FreeArg2> >::emit(typename Cgu::Param<FreeArg1>::ParamType arg1,
							  typename Cgu::Param<FreeArg2>::ParamType arg2) const {
  std::list<ListItem> local_list;
  {
    Thread::Mutex::Lock lock(mutex);
    local_list = emission_list;
  }

  typename std::list<ListItem>::const_iterator iter;
  for (iter = local_list.begin(); iter != local_list.end(); ++iter) {
    if (!iter->blocked) iter->f1(arg1, arg2);
  }
}

template <class FreeArg1, class FreeArg2>
bool SafeEmitterArg<TypeTuple<FreeArg1, FreeArg2> >::test_emit(typename Cgu::Param<FreeArg1>::ParamType arg1,
							       typename Cgu::Param<FreeArg2>::ParamType arg2) const {
  std::list<ListItem> local_list;
  {
    Thread::Mutex::Lock lock(mutex);
    if (emission_list.empty()) return false;
    local_list = emission_list;
  }

  typename std::list<ListItem>::const_iterator iter;
  for (iter = local_list.begin(); iter != local_list.end(); ++iter) {
    if (!iter->blocked) iter->f1(arg1, arg2);
  }
  return true;
}

template <class FreeArg1, class FreeArg2>
Callback::SafeFunctorArg<TypeTuple<FreeArg1, FreeArg2> > SafeEmitterArg<TypeTuple<FreeArg1, FreeArg2> >::connect(const Callback::SafeFunctorArg<TypeTuple<FreeArg1, FreeArg2> >& f1) {
  Thread::Mutex::Lock lock(mutex);
  emission_list.push_back(ListItem(f1, Callback::SafeFunctorArg<int*>()));
  return f1;
}

template <class FreeArg1, class FreeArg2>
Callback::SafeFunctorArg<TypeTuple<FreeArg1, FreeArg2> > SafeEmitterArg<TypeTuple<FreeArg1, FreeArg2> >::connect(const Callback::SafeFunctorArg<TypeTuple<FreeArg1, FreeArg2> >& f1, Releaser& r) {
  // In this method:
  // f1 is the functor we execute when we emit()
  // f2 is the functor we execute in our destructor if we are destroyed before the
  // remote object is
  // f3 is the functor the remote object executes in its Releaser if it is destroyed
  // before we are, or if Releaser::operator=() is called

  Callback::SafeFunctor f3(Callback::make_val(*this, &SafeEmitterArg<TypeTuple<FreeArg1, FreeArg2> >::tracking_disconnect, f1));
  Callback::SafeFunctorArg<int*> f2(Callback::make_val(r, &Releaser::try_remove, f3));
  r.add(f3);
  Thread::Mutex::Lock lock(mutex);
  try {
    emission_list.push_back(ListItem(f1, f2));
  }
  catch (...) {
    mutex.unlock();
    r.remove(f3);
    mutex.lock();
    throw;
  }
  return f1;
}

template <class FreeArg1, class FreeArg2>
void SafeEmitterArg<TypeTuple<FreeArg1, FreeArg2> >::disconnect(const Callback::SafeFunctorArg<TypeTuple<FreeArg1, FreeArg2> >& arg) {
  struct Pred {
    static bool pred(ListItem p, Callback::SafeFunctorArg<TypeTuple<FreeArg1, FreeArg2> > f) {
      return (p.f1 == f);
    }
  };

  Thread::Mutex::Lock lock(mutex);
  typename std::list<ListItem>::iterator iter = emission_list.begin();
  for(;;) {
    iter = std::find_if(iter, emission_list.end(),
                        std::bind2nd(std::ptr_fun(Pred::pred), arg));
    if (iter != emission_list.end()) {
      int result = 0;
      (iter->f2)(&result);
      if (!result) {
        iter = emission_list.erase(iter);
      }
      else {
        mutex.unlock();
#ifdef CGU_USE_SCHED_YIELD
        sched_yield();
#else
        usleep(10);
#endif
	mutex.lock();
	iter = emission_list.begin();
      }
    }
    else break;
  }
}

template <class FreeArg1, class FreeArg2>
void SafeEmitterArg<TypeTuple<FreeArg1, FreeArg2> >::tracking_disconnect(Callback::SafeFunctorArg<TypeTuple<FreeArg1, FreeArg2> > arg) {
  struct Pred {
    static bool pred(ListItem p, Callback::SafeFunctorArg<TypeTuple<FreeArg1, FreeArg2> > f) {
      return (p.f1 == f);
    }
  };

  Thread::Mutex::Lock lock(mutex);
  typename std::list<ListItem>::iterator iter = emission_list.begin();
  for (;;) {
    iter = std::find_if(iter, emission_list.end(),
			std::bind2nd(std::ptr_fun(Pred::pred), arg));
    if (iter != emission_list.end()) {
      iter = emission_list.erase(iter);
    }
    else break;
  }
}

template <class FreeArg1, class FreeArg2>
void SafeEmitterArg<TypeTuple<FreeArg1, FreeArg2> >::block(const Callback::SafeFunctorArg<TypeTuple<FreeArg1, FreeArg2> >& arg) {
  struct Pred {
    static bool pred(ListItem p, Callback::SafeFunctorArg<TypeTuple<FreeArg1, FreeArg2> > f) {
      return (p.f1 == f);
    }
  };

  Thread::Mutex::Lock lock(mutex);
  typename std::list<ListItem>::iterator iter = emission_list.begin();
  for (;;) {
    iter = std::find_if(iter, emission_list.end(),
			std::bind2nd(std::ptr_fun(Pred::pred), arg));
    if (iter != emission_list.end()) {
      iter->blocked = true;
      ++iter;
    }
    else break;
  }
}

template <class FreeArg1, class FreeArg2>
void SafeEmitterArg<TypeTuple<FreeArg1, FreeArg2> >::unblock(const Callback::SafeFunctorArg<TypeTuple<FreeArg1, FreeArg2> >& arg) {
  struct Pred {
    static bool pred(ListItem p, Callback::SafeFunctorArg<TypeTuple<FreeArg1, FreeArg2> > f) {
      return (p.f1 == f);
    }
  };

  Thread::Mutex::Lock lock(mutex);
  typename std::list<ListItem>::iterator iter = emission_list.begin();
  for (;;) {
    iter = std::find_if(iter, emission_list.end(),
			std::bind2nd(std::ptr_fun(Pred::pred), arg));
    if (iter != emission_list.end()) {
      iter->blocked = false;
      ++iter;
    }
    else break;
  }
}

template <class FreeArg1, class FreeArg2, class FreeArg3>
class SafeEmitterArg <TypeTuple<FreeArg1, FreeArg2, FreeArg3> > {
  struct ListItem {
    Callback::SafeFunctorArg<TypeTuple<FreeArg1, FreeArg2, FreeArg3> > f1;
    Callback::SafeFunctorArg<int*> f2;
    bool blocked;
    ListItem(Callback::SafeFunctorArg<TypeTuple<FreeArg1, FreeArg2, FreeArg3> > f1_, Callback::SafeFunctorArg<int*> f2_):
      f1(f1_), f2(f2_), blocked(false) {}
  };
  
  std::list<ListItem> emission_list;
  mutable Thread::Mutex mutex;

  void tracking_disconnect(Callback::SafeFunctorArg<TypeTuple<FreeArg1, FreeArg2, FreeArg3> >);

  SafeEmitterArg(const SafeEmitterArg&);
  SafeEmitterArg& operator=(const SafeEmitterArg&);
public:
  friend class Releaser;

  void operator()(typename Cgu::Param<FreeArg1>::ParamType arg1,
		  typename Cgu::Param<FreeArg2>::ParamType arg2,
		  typename Cgu::Param<FreeArg3>::ParamType arg3) const {
    emit(arg1, arg2, arg3);
  }
  void emit(typename Cgu::Param<FreeArg1>::ParamType arg1,
	    typename Cgu::Param<FreeArg2>::ParamType arg2,
	    typename Cgu::Param<FreeArg3>::ParamType arg3) const;

  bool test_emit(typename Cgu::Param<FreeArg1>::ParamType arg1,
		 typename Cgu::Param<FreeArg2>::ParamType arg2,
		 typename Cgu::Param<FreeArg3>::ParamType arg3) const;

  Callback::SafeFunctorArg<TypeTuple<FreeArg1, FreeArg2, FreeArg3> > connect(const Callback::SafeFunctorArg<TypeTuple<FreeArg1, FreeArg2, FreeArg3> >& f);
  Callback::SafeFunctorArg<TypeTuple<FreeArg1, FreeArg2, FreeArg3> > connect(const Callback::SafeFunctorArg<TypeTuple<FreeArg1, FreeArg2, FreeArg3> >& f, Releaser& r);
  void disconnect(const Callback::SafeFunctorArg<TypeTuple<FreeArg1, FreeArg2, FreeArg3> >& f);

  void block(const Callback::SafeFunctorArg<TypeTuple<FreeArg1, FreeArg2, FreeArg3> >& f);
  void unblock(const Callback::SafeFunctorArg<TypeTuple<FreeArg1, FreeArg2, FreeArg3> >& f);

  SafeEmitterArg() {}
  ~SafeEmitterArg();

/* Only has effect if --with-glib-memory-slices-compat or
 * --with-glib-memory-slices-no-compat option picked */
  CGU_GLIB_MEMORY_SLICES_FUNCS
};

template <class FreeArg1, class FreeArg2, class FreeArg3>
SafeEmitterArg<TypeTuple<FreeArg1, FreeArg2, FreeArg3> >::~SafeEmitterArg() {
  Thread::Mutex::Lock lock(mutex);
  while (!emission_list.empty()) {
    typename std::list<ListItem>::iterator iter = emission_list.begin();
    int result = 0;
    (iter->f2)(&result);
    if (!result) {
      emission_list.erase(iter);
    }
    else {
      mutex.unlock();
#ifdef CGU_USE_SCHED_YIELD
      sched_yield();
#else
      usleep(10);
#endif
      mutex.lock();
    }
  }
}

template <class FreeArg1, class FreeArg2, class FreeArg3>
void SafeEmitterArg<TypeTuple<FreeArg1, FreeArg2, FreeArg3> >::emit(typename Cgu::Param<FreeArg1>::ParamType arg1,
								    typename Cgu::Param<FreeArg2>::ParamType arg2,
								    typename Cgu::Param<FreeArg3>::ParamType arg3) const {
  std::list<ListItem> local_list;
  {
    Thread::Mutex::Lock lock(mutex);
    local_list = emission_list;
  }

  typename std::list<ListItem>::const_iterator iter;
  for (iter = local_list.begin(); iter != local_list.end(); ++iter) {
    if (!iter->blocked) iter->f1(arg1, arg2, arg3);
  }
}

template <class FreeArg1, class FreeArg2, class FreeArg3>
bool SafeEmitterArg<TypeTuple<FreeArg1, FreeArg2, FreeArg3> >::test_emit(typename Cgu::Param<FreeArg1>::ParamType arg1,
									 typename Cgu::Param<FreeArg2>::ParamType arg2,
									 typename Cgu::Param<FreeArg3>::ParamType arg3) const {
  std::list<ListItem> local_list;
  {
    Thread::Mutex::Lock lock(mutex);
    if (emission_list.empty()) return false;
    local_list = emission_list;
  }

  typename std::list<ListItem>::const_iterator iter;
  for (iter = local_list.begin(); iter != local_list.end(); ++iter) {
    if (!iter->blocked) iter->f1(arg1, arg2, arg3);
  }
  return true;
}

template <class FreeArg1, class FreeArg2, class FreeArg3>
Callback::SafeFunctorArg<TypeTuple<FreeArg1, FreeArg2, FreeArg3> > SafeEmitterArg<TypeTuple<FreeArg1, FreeArg2, FreeArg3> >::connect(const Callback::SafeFunctorArg<TypeTuple<FreeArg1, FreeArg2, FreeArg3> >& f1) {
  Thread::Mutex::Lock lock(mutex);
  emission_list.push_back(ListItem(f1, Callback::SafeFunctorArg<int*>()));
  return f1;
}

template <class FreeArg1, class FreeArg2, class FreeArg3>
Callback::SafeFunctorArg<TypeTuple<FreeArg1, FreeArg2, FreeArg3> > SafeEmitterArg<TypeTuple<FreeArg1, FreeArg2, FreeArg3> >::connect(const Callback::SafeFunctorArg<TypeTuple<FreeArg1, FreeArg2, FreeArg3> >& f1, Releaser& r) {
  // In this method:
  // f1 is the functor we execute when we emit()
  // f2 is the functor we execute in our destructor if we are destroyed before the
  // remote object is
  // f3 is the functor the remote object executes in its Releaser if it is destroyed
  // before we are, or if Releaser::operator=() is called

  Callback::SafeFunctor f3(Callback::make_val(*this, &SafeEmitterArg<TypeTuple<FreeArg1, FreeArg2, FreeArg3> >::tracking_disconnect, f1));
  Callback::SafeFunctorArg<int*> f2(Callback::make_val(r, &Releaser::try_remove, f3));
  r.add(f3);
  Thread::Mutex::Lock lock(mutex);
  try {
    emission_list.push_back(ListItem(f1, f2));
  }
  catch (...) {
    mutex.unlock();
    r.remove(f3);
    mutex.lock();
    throw;
  }
  return f1;
}

template <class FreeArg1, class FreeArg2, class FreeArg3>
void SafeEmitterArg<TypeTuple<FreeArg1, FreeArg2, FreeArg3> >::disconnect(const Callback::SafeFunctorArg<TypeTuple<FreeArg1, FreeArg2, FreeArg3> >& arg) {
  struct Pred {
    static bool pred(ListItem p, Callback::SafeFunctorArg<TypeTuple<FreeArg1, FreeArg2, FreeArg3> > f) {
      return (p.f1 == f);
    }
  };

  Thread::Mutex::Lock lock(mutex);
  typename std::list<ListItem>::iterator iter = emission_list.begin();
  for(;;) {
    iter = std::find_if(iter, emission_list.end(),
                        std::bind2nd(std::ptr_fun(Pred::pred), arg));
    if (iter != emission_list.end()) {
      int result = 0;
      (iter->f2)(&result);
      if (!result) {
        iter = emission_list.erase(iter);
      }
      else {
        mutex.unlock();
#ifdef CGU_USE_SCHED_YIELD
        sched_yield();
#else
        usleep(10);
#endif
	mutex.lock();
	iter = emission_list.begin();
      }
    }
    else break;
  }
}

template <class FreeArg1, class FreeArg2, class FreeArg3>
void SafeEmitterArg<TypeTuple<FreeArg1, FreeArg2, FreeArg3> >::tracking_disconnect(Callback::SafeFunctorArg<TypeTuple<FreeArg1, FreeArg2, FreeArg3> > arg) {
  struct Pred {
    static bool pred(ListItem p, Callback::SafeFunctorArg<TypeTuple<FreeArg1, FreeArg2, FreeArg3> > f) {
      return (p.f1 == f);
    }
  };

  Thread::Mutex::Lock lock(mutex);
  typename std::list<ListItem>::iterator iter = emission_list.begin();
  for (;;) {
    iter = std::find_if(iter, emission_list.end(),
			std::bind2nd(std::ptr_fun(Pred::pred), arg));
    if (iter != emission_list.end()) {
      iter = emission_list.erase(iter);
    }
    else break;
  }
}

template <class FreeArg1, class FreeArg2, class FreeArg3>
void SafeEmitterArg<TypeTuple<FreeArg1, FreeArg2, FreeArg3> >::block(const Callback::SafeFunctorArg<TypeTuple<FreeArg1, FreeArg2, FreeArg3> >& arg) {
  struct Pred {
    static bool pred(ListItem p, Callback::SafeFunctorArg<TypeTuple<FreeArg1, FreeArg2, FreeArg3> > f) {
      return (p.f1 == f);
    }
  };

  Thread::Mutex::Lock lock(mutex);
  typename std::list<ListItem>::iterator iter = emission_list.begin();
  for (;;) {
    iter = std::find_if(iter, emission_list.end(),
			std::bind2nd(std::ptr_fun(Pred::pred), arg));
    if (iter != emission_list.end()) {
      iter->blocked = true;
      ++iter;
    }
    else break;
  }
}

template <class FreeArg1, class FreeArg2, class FreeArg3>
void SafeEmitterArg<TypeTuple<FreeArg1, FreeArg2, FreeArg3> >::unblock(const Callback::SafeFunctorArg<TypeTuple<FreeArg1, FreeArg2, FreeArg3> >& arg) {
  struct Pred {
    static bool pred(ListItem p, Callback::SafeFunctorArg<TypeTuple<FreeArg1, FreeArg2, FreeArg3> > f) {
      return (p.f1 == f);
    }
  };

  Thread::Mutex::Lock lock(mutex);
  typename std::list<ListItem>::iterator iter = emission_list.begin();
  for (;;) {
    iter = std::find_if(iter, emission_list.end(),
			std::bind2nd(std::ptr_fun(Pred::pred), arg));
    if (iter != emission_list.end()) {
      iter->blocked = false;
      ++iter;
    }
    else break;
  }
}

#endif // DOXYGEN_PARSING

} // namespace Cgu

#endif // EMITTER_H
