/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | Copyright (C) 2012-2016 OpenFOAM Foundation
     \\/     M anipulation  | Copyright (C) 2018 OpenCFD Ltd.
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

\*---------------------------------------------------------------------------*/

#include "vtkUnstructuredReader.H"
#include "labelIOField.H"
#include "scalarIOField.H"
#include "stringIOList.H"
#include "cellModel.H"
#include "vectorIOField.H"
#include "triPointRef.H"

/* * * * * * * * * * * * * * * Static Member Data  * * * * * * * * * * * * * */

namespace Foam
{
    defineTypeNameAndDebug(vtkUnstructuredReader, 1);
}

const Foam::Enum
<
    Foam::vtkUnstructuredReader::vtkDataType
>
Foam::vtkUnstructuredReader::vtkDataTypeNames
({
    { vtkDataType::VTK_INT, "int" },
    { vtkDataType::VTK_UINT, "unsigned_int" },
    { vtkDataType::VTK_LONG, "long" },
    { vtkDataType::VTK_ULONG, "unsigned_long" },
    { vtkDataType::VTK_FLOAT, "float" },
    { vtkDataType::VTK_DOUBLE, "double" },
    { vtkDataType::VTK_STRING, "string" },
    { vtkDataType::VTK_ID, "vtkIdType" },
});


const Foam::Enum
<
    Foam::vtkUnstructuredReader::vtkDataSetType
>
Foam::vtkUnstructuredReader::vtkDataSetTypeNames
({
    { vtkDataSetType::VTK_FIELD, "FIELD" },
    { vtkDataSetType::VTK_SCALARS, "SCALARS" },
    { vtkDataSetType::VTK_VECTORS, "VECTORS" },
});


const Foam::Enum
<
    Foam::vtkUnstructuredReader::parseMode
>
Foam::vtkUnstructuredReader::parseModeNames
({
    { parseMode::NOMODE, "NOMODE" },
    { parseMode::UNSTRUCTURED_GRID, "UNSTRUCTURED_GRID" },
    { parseMode::POLYDATA, "POLYDATA" },
    { parseMode::CELL_DATA, "CELL_DATA" },
    { parseMode::POINT_DATA, "POINT_DATA" },
});


// * * * * * * * * * * * * * Private Member Functions  * * * * * * * * * * * //

void Foam::vtkUnstructuredReader::warnUnhandledType
(
    Istream& inFile,
    const label type,
    labelHashSet& warningGiven
) const
{
    if (warningGiven.insert(type))
    {
        IOWarningInFunction(inFile)
            << "Skipping unknown cell type " << type << nl;
    }
}


void Foam::vtkUnstructuredReader::extractCells
(
    Istream& inFile,
    const labelList& cellTypes,
    const labelList& cellVertData
)
{
    const cellModel& hex = cellModel::ref(cellModel::HEX);
    const cellModel& prism = cellModel::ref(cellModel::PRISM);
    const cellModel& pyr = cellModel::ref(cellModel::PYR);
    const cellModel& tet = cellModel::ref(cellModel::TET);

    labelList tetPoints(4);
    labelList pyrPoints(5);
    labelList prismPoints(6);
    labelList hexPoints(8);

    label celli = cells_.size();
    cells_.setSize(celli+cellTypes.size());
    cellMap_.setSize(cells_.size(), -1);

    label facei = faces_.size();
    faces_.setSize(facei+cellTypes.size());
    faceMap_.setSize(faces_.size(), -1);

    label lineI = lines_.size();
    lines_.setSize(lineI+cellTypes.size());
    lineMap_.setSize(lines_.size(), -1);

    label dataIndex = 0;


    // To mark whether unhandled type has been visited.
    labelHashSet warningGiven;

    forAll(cellTypes, i)
    {
        switch (cellTypes[i])
        {
            case vtk::cellType::VTK_VERTEX:
            {
                warnUnhandledType(inFile, cellTypes[i], warningGiven);
                label nRead = cellVertData[dataIndex++];
                if (nRead != 1)
                {
                    FatalIOErrorInFunction(inFile)
                        << "Expected size 1 for VTK_VERTEX but found "
                        << nRead << exit(FatalIOError);
                }
                dataIndex += nRead;
            }
            break;

            case vtk::cellType::VTK_POLY_VERTEX:
            {
                warnUnhandledType(inFile, cellTypes[i], warningGiven);
                label nRead = cellVertData[dataIndex++];
                dataIndex += nRead;
            }
            break;

            case vtk::cellType::VTK_LINE:
            {
                //warnUnhandledType(inFile, cellTypes[i], warningGiven);
                label nRead = cellVertData[dataIndex++];
                if (nRead != 2)
                {
                    FatalIOErrorInFunction(inFile)
                        << "Expected size 2 for VTK_LINE but found "
                        << nRead << exit(FatalIOError);
                }
                lineMap_[lineI] = i;
                labelList& segment = lines_[lineI++];
                segment.setSize(2);
                segment[0] = cellVertData[dataIndex++];
                segment[1] = cellVertData[dataIndex++];
            }
            break;

            case vtk::cellType::VTK_POLY_LINE:
            {
                //warnUnhandledType(inFile, cellTypes[i], warningGiven);
                label nRead = cellVertData[dataIndex++];
                lineMap_[lineI] = i;
                labelList& segment = lines_[lineI++];
                segment.setSize(nRead);
                forAll(segment, i)
                {
                    segment[i] = cellVertData[dataIndex++];
                }
            }
            break;

            case vtk::cellType::VTK_TRIANGLE:
            {
                faceMap_[facei] = i;
                face& f = faces_[facei++];
                f.setSize(3);
                label nRead = cellVertData[dataIndex++];
                if (nRead != 3)
                {
                    FatalIOErrorInFunction(inFile)
                        << "Expected size 3 for VTK_TRIANGLE but found "
                        << nRead << exit(FatalIOError);
                }
                f[0] = cellVertData[dataIndex++];
                f[1] = cellVertData[dataIndex++];
                f[2] = cellVertData[dataIndex++];
            }
            break;

            case vtk::cellType::VTK_QUAD:
            {
                faceMap_[facei] = i;
                face& f = faces_[facei++];
                f.setSize(4);
                label nRead = cellVertData[dataIndex++];
                if (nRead != 4)
                {
                    FatalIOErrorInFunction(inFile)
                        << "Expected size 4 for VTK_QUAD but found "
                        << nRead << exit(FatalIOError);
                }
                f[0] = cellVertData[dataIndex++];
                f[1] = cellVertData[dataIndex++];
                f[2] = cellVertData[dataIndex++];
                f[3] = cellVertData[dataIndex++];
            }
            break;

            case vtk::cellType::VTK_POLYGON:
            {
                faceMap_[facei] = i;
                face& f = faces_[facei++];
                label nRead = cellVertData[dataIndex++];
                f.setSize(nRead);
                forAll(f, fp)
                {
                    f[fp] = cellVertData[dataIndex++];
                }
            }
            break;

            case vtk::cellType::VTK_TETRA:
            {
                label nRead = cellVertData[dataIndex++];
                if (nRead != 4)
                {
                    FatalIOErrorInFunction(inFile)
                        << "Expected size 4 for VTK_TETRA but found "
                        << nRead << exit(FatalIOError);
                }
                tetPoints[0] = cellVertData[dataIndex++];
                tetPoints[1] = cellVertData[dataIndex++];
                tetPoints[2] = cellVertData[dataIndex++];
                tetPoints[3] = cellVertData[dataIndex++];
                cellMap_[celli] = i;
                cells_[celli++] = cellShape(tet, tetPoints, true);
            }
            break;

            case vtk::cellType::VTK_PYRAMID:
            {
                label nRead = cellVertData[dataIndex++];
                if (nRead != 5)
                {
                    FatalIOErrorInFunction(inFile)
                        << "Expected size 5 for VTK_PYRAMID but found "
                        << nRead << exit(FatalIOError);
                }
                pyrPoints[0] = cellVertData[dataIndex++];
                pyrPoints[1] = cellVertData[dataIndex++];
                pyrPoints[2] = cellVertData[dataIndex++];
                pyrPoints[3] = cellVertData[dataIndex++];
                pyrPoints[4] = cellVertData[dataIndex++];
                cellMap_[celli] = i;
                cells_[celli++] = cellShape(pyr, pyrPoints, true);
            }
            break;

            case vtk::cellType::VTK_WEDGE:
            {
                label nRead = cellVertData[dataIndex++];
                if (nRead != 6)
                {
                    FatalIOErrorInFunction(inFile)
                        << "Expected size 6 for VTK_WEDGE but found "
                        << nRead << exit(FatalIOError);
                }
                //- From mesh description in vtk documentation
                prismPoints[0] = cellVertData[dataIndex++];
                prismPoints[2] = cellVertData[dataIndex++];
                prismPoints[1] = cellVertData[dataIndex++];
                prismPoints[3] = cellVertData[dataIndex++];
                prismPoints[5] = cellVertData[dataIndex++];
                prismPoints[4] = cellVertData[dataIndex++];
                cellMap_[celli] = i;
                cells_[celli++] = cellShape(prism, prismPoints, true);
            }
            break;

            case vtk::cellType::VTK_HEXAHEDRON:
            {
                label nRead = cellVertData[dataIndex++];
                if (nRead != 8)
                {
                    FatalIOErrorInFunction(inFile)
                        << "Expected size 8 for VTK_HEXAHEDRON but found "
                        << nRead << exit(FatalIOError);
                }
                hexPoints[0] = cellVertData[dataIndex++];
                hexPoints[1] = cellVertData[dataIndex++];
                hexPoints[2] = cellVertData[dataIndex++];
                hexPoints[3] = cellVertData[dataIndex++];
                hexPoints[4] = cellVertData[dataIndex++];
                hexPoints[5] = cellVertData[dataIndex++];
                hexPoints[6] = cellVertData[dataIndex++];
                hexPoints[7] = cellVertData[dataIndex++];
                cellMap_[celli] = i;
                cells_[celli++] = cellShape(hex, hexPoints, true);
            }
            break;

            default:
                warnUnhandledType(inFile, cellTypes[i], warningGiven);
                label nRead = cellVertData[dataIndex++];
                dataIndex += nRead;
        }
    }

    DebugInfo
        << "Read " << celli << " cells;" << facei << " faces." << nl;

    cells_.setSize(celli);
    cellMap_.setSize(celli);
    faces_.setSize(facei);
    faceMap_.setSize(facei);
    lines_.setSize(lineI);
    lineMap_.setSize(lineI);
}


void Foam::vtkUnstructuredReader::readField
(
    ISstream& inFile,
    objectRegistry& obj,
    const word& arrayName,
    const word& dataType,
    const label size
) const
{
    if (vtkDataTypeNames.found(dataType))
    {
        switch (vtkDataTypeNames[dataType])
        {
            case VTK_INT:
            case VTK_UINT:
            case VTK_LONG:
            case VTK_ULONG:
            case VTK_ID:
            {
                auto fieldVals = autoPtr<labelIOField>::New
                (
                    IOobject(arrayName, "", obj),
                    size
                );
                readBlock(inFile, fieldVals().size(), fieldVals());
                regIOobject::store(fieldVals);
            }
            break;

            case VTK_FLOAT:
            case VTK_DOUBLE:
            {
                auto fieldVals = autoPtr<scalarIOField>::New
                (
                    IOobject(arrayName, "", obj),
                    size
                );
                readBlock(inFile, fieldVals().size(), fieldVals());
                regIOobject::store(fieldVals);
            }
            break;

            case VTK_STRING:
            {
                DebugInfo
                    << "Reading strings:" << size << nl;

                auto fieldVals = autoPtr<stringIOList>::New
                (
                    IOobject(arrayName, "", obj),
                    size
                );
                // Consume current line.
                inFile.getLine(fieldVals()[0]);

                // Read without parsing
                forAll(fieldVals(), i)
                {
                    inFile.getLine(fieldVals()[i]);
                }
                regIOobject::store(fieldVals);
            }
            break;

            default:
            {
                IOWarningInFunction(inFile)
                    << "Unhandled type " << dataType << nl
                    << "Skipping " << size
                    << " words." << nl;
                scalarField fieldVals;
                readBlock(inFile, size, fieldVals);
            }
            break;
        }
    }
    else
    {
        IOWarningInFunction(inFile)
            << "Unhandled type " << dataType << nl
            << "Skipping " << size
            << " words." << nl;
        scalarField fieldVals;
        readBlock(inFile, size, fieldVals);
    }
}


Foam::wordList Foam::vtkUnstructuredReader::readFieldArray
(
    ISstream& inFile,
    objectRegistry& obj,
    const label wantedSize
) const
{
    DynamicList<word> fields;

    word dataName(inFile);

    DebugInfo
        << "dataName:" << dataName << nl;

    label numArrays(readLabel(inFile));
    if (debug)
    {
        Pout<< "numArrays:" << numArrays << nl;
    }
    for (label i = 0; i < numArrays; i++)
    {
        word arrayName(inFile);
        label numComp(readLabel(inFile));
        label numTuples(readLabel(inFile));
        word dataType(inFile);

        DebugInfo
            << "Reading field " << arrayName
            << " of " << numTuples << " tuples of rank " << numComp << nl;

        if (wantedSize != -1 && numTuples != wantedSize)
        {
            FatalIOErrorInFunction(inFile)
                << "Expected " << wantedSize << " tuples but only have "
                << numTuples << exit(FatalIOError);
        }

        readField
        (
            inFile,
            obj,
            arrayName,
            dataType,
            numTuples*numComp
        );
        fields.append(arrayName);
    }
    return fields.shrink();
}


Foam::objectRegistry& Foam::vtkUnstructuredReader::selectRegistry
(
    const parseMode readMode
)
{
    if (readMode == CELL_DATA)
    {
        return cellData_;
    }
    else if (readMode == POINT_DATA)
    {
        return pointData_;
    }

    return otherData_;
}


// * * * * * * * * * * * * * * * * Constructors  * * * * * * * * * * * * * * //

Foam::vtkUnstructuredReader::vtkUnstructuredReader
(
    const objectRegistry& obr,
    ISstream& inFile
)
:
    cellData_(IOobject("cellData", obr)),
    pointData_(IOobject("pointData", obr)),
    otherData_(IOobject("otherData", obr))
{
    read(inFile);
}


void Foam::vtkUnstructuredReader::read(ISstream& inFile)
{
    inFile.getLine(header_);
    DebugInfo<< "Header   : " << header_ << nl;

    inFile.getLine(title_);
    DebugInfo<< "Title    : " << title_ << nl;

    inFile.getLine(dataType_);
    DebugInfo<< "dataType : " << dataType_ << nl;

    if (dataType_ == "BINARY")
    {
        FatalIOErrorInFunction(inFile)
            << "Binary reading not supported"
            << exit(FatalIOError);
    }

    parseMode readMode = NOMODE;
    label wantedSize = -1;


    // Temporary storage for vertices of cells.
    labelList cellVerts;

    token tok;

    while (inFile.read(tok).good() && tok.isWord())
    {
        const word tag = tok.wordToken();

        DebugInfo
            << "line:" << inFile.lineNumber()
            << " tag:" << tag << nl;

        if (tag == "DATASET")
        {
            word geomType(inFile);
            DebugInfo<< "geomType : " << geomType << nl;

            readMode = parseModeNames[geomType];
            wantedSize = -1;
        }
        else if (tag == "POINTS")
        {
            label nPoints(readLabel(inFile));
            points_.setSize(nPoints);    ///3);

            DebugInfo
                << "Reading " << nPoints << " numbers representing "
                << points_.size() << " coordinates." << nl;

            word primitiveTag(inFile);
            if (primitiveTag != "float" && primitiveTag != "double")
            {
                FatalIOErrorInFunction(inFile)
                    << "Expected 'float' entry but found "
                    << primitiveTag
                    << exit(FatalIOError);
            }
            for (point& p : points_)
            {
                inFile >> p.x() >> p.y() >> p.z();
            }
        }
        else if (tag == "CELLS")
        {
            label nCells(readLabel(inFile));
            label nNumbers(readLabel(inFile));
            DebugInfo
                << "Reading " << nCells << " cells or faces." << nl;

            readBlock(inFile, nNumbers, cellVerts);
        }
        else if (tag == "CELL_TYPES")
        {
            label nCellTypes(readLabel(inFile));

            labelList cellTypes;
            readBlock(inFile, nCellTypes, cellTypes);

            if (cellTypes.size() > 0 && cellVerts.size() == 0)
            {
                FatalIOErrorInFunction(inFile)
                    << "Found " << cellTypes.size()
                    << " cellTypes but no cells."
                    << exit(FatalIOError);
            }

            extractCells(inFile, cellTypes, cellVerts);
            cellVerts.clear();
        }
        else if (tag == "LINES")
        {
            label nLines(readLabel(inFile));
            label nNumbers(readLabel(inFile));
            DebugInfo
                << "Reading " << nLines << " lines." << nl;

            labelList lineVerts;
            readBlock(inFile, nNumbers, lineVerts);

            label lineI = lines_.size();
            lines_.setSize(lineI+nLines);
            lineMap_.setSize(lines_.size());

            label elemI = 0;
            for (label i = 0; i < nLines; i++)
            {
                lineMap_[lineI] = lineI;
                labelList& f = lines_[lineI];
                f.setSize(lineVerts[elemI++]);
                forAll(f, fp)
                {
                    f[fp] = lineVerts[elemI++];
                }
                lineI++;
            }
        }
        else if (tag == "POLYGONS")
        {
            // If in polydata mode

            label nFaces(readLabel(inFile));
            label nNumbers(readLabel(inFile));
            DebugInfo
                << "Reading " << nFaces << " faces." << nl;

            labelList faceVerts;
            readBlock(inFile, nNumbers, faceVerts);

            label facei = faces_.size();
            faces_.setSize(facei+nFaces);
            faceMap_.setSize(faces_.size());

            label elemI = 0;
            for (label i = 0; i < nFaces; i++)
            {
                faceMap_[facei] = facei;
                face& f = faces_[facei];
                f.setSize(faceVerts[elemI++]);
                forAll(f, fp)
                {
                    f[fp] = faceVerts[elemI++];
                }
                facei++;
            }
        }
        else if (tag == "POINT_DATA")
        {
            // 'POINT_DATA 24'
            readMode = POINT_DATA;
            wantedSize = points_.size();

            label nPoints(readLabel(inFile));
            if (nPoints != wantedSize)
            {
                FatalIOErrorInFunction(inFile)
                    << "Reading POINT_DATA : expected " << wantedSize
                    << " but read " << nPoints << exit(FatalIOError);
            }
        }
        else if (tag == "CELL_DATA")
        {
            readMode = CELL_DATA;
            wantedSize = cells_.size()+faces_.size()+lines_.size();

            label nCells(readLabel(inFile));
            if (nCells != wantedSize)
            {
                FatalIOErrorInFunction(inFile)
                    << "Reading CELL_DATA : expected "
                    << wantedSize
                    << " but read " << nCells << exit(FatalIOError);
            }
        }
        else if (tag == "FIELD")
        {
            // wantedSize already set according to type we expected to read.
            readFieldArray(inFile, selectRegistry(readMode), wantedSize);
        }
        else if (tag == "SCALARS")
        {
            string line;
            inFile.getLine(line);
            IStringStream is(line);
            word dataName(is);
            word dataType(is);
            //label numComp(readLabel(inFile));

            DebugInfo
                << "Reading scalar " << dataName
                << " of type " << dataType
                << " from lookup table" << nl;

            word lookupTableTag(inFile);
            if (lookupTableTag != "LOOKUP_TABLE")
            {
                FatalIOErrorInFunction(inFile)
                    << "Expected tag LOOKUP_TABLE but read "
                    << lookupTableTag
                    << exit(FatalIOError);
            }

            word lookupTableName(inFile);

            readField
            (
                inFile,
                selectRegistry(readMode),
                dataName,
                dataType,
                wantedSize//*numComp
            );
        }
        else if (tag == "VECTORS" || tag == "NORMALS")
        {
            // 'NORMALS Normals float'
            string line;
            inFile.getLine(line);
            IStringStream is(line);
            word dataName(is);
            word dataType(is);
            DebugInfo
                << "Reading vector " << dataName
                << " of type " << dataType << nl;

            objectRegistry& reg = selectRegistry(readMode);

            readField
            (
                inFile,
                reg,
                dataName,
                dataType,
                3*wantedSize
            );

            if
            (
                vtkDataTypeNames[dataType] == VTK_FLOAT
             || vtkDataTypeNames[dataType] == VTK_DOUBLE
            )
            {
                objectRegistry::iterator iter = reg.find(dataName);
                scalarField s(*dynamic_cast<const scalarField*>(iter()));
                reg.erase(iter);
                auto fieldVals = autoPtr<vectorIOField>::New
                (
                    IOobject(dataName, "", reg),
                    s.size()/3
                );

                label elemI = 0;
                forAll(fieldVals(), i)
                {
                    fieldVals()[i].x() = s[elemI++];
                    fieldVals()[i].y() = s[elemI++];
                    fieldVals()[i].z() = s[elemI++];
                }
                regIOobject::store(fieldVals);
            }
        }
        else if (tag == "TEXTURE_COORDINATES")
        {
            // 'TEXTURE_COORDINATES TCoords 2 float'
            string line;
            inFile.getLine(line);
            IStringStream is(line);
            word dataName(is);          //"Tcoords"
            label dim(readLabel(is));
            word dataType(is);

            DebugInfo
                << "Reading texture coords " << dataName
                << " dimension " << dim
                << " of type " << dataType << nl;

            scalarField coords(dim*points_.size());
            readBlock(inFile, coords.size(), coords);
        }
        else if (tag == "TRIANGLE_STRIPS")
        {
            label nStrips(readLabel(inFile));
            label nNumbers(readLabel(inFile));
            DebugInfo
                << "Reading " << nStrips << " triangle strips." << nl;

            labelList faceVerts;
            readBlock(inFile, nNumbers, faceVerts);

            // Count number of triangles
            label elemI = 0;
            label nTris = 0;
            for (label i = 0; i < nStrips; i++)
            {
                label nVerts = faceVerts[elemI++];
                nTris += nVerts-2;
                elemI += nVerts;
            }


            // Store
            label facei = faces_.size();
            faces_.setSize(facei+nTris);
            faceMap_.setSize(faces_.size());
            elemI = 0;
            for (label i = 0; i < nStrips; i++)
            {
                label nVerts = faceVerts[elemI++];
                label nTris = nVerts-2;

                // Read first triangle
                faceMap_[facei] = facei;
                face& f = faces_[facei++];
                f.setSize(3);
                f[0] = faceVerts[elemI++];
                f[1] = faceVerts[elemI++];
                f[2] = faceVerts[elemI++];
                for (label triI = 1; triI < nTris; triI++)
                {
                    faceMap_[facei] = facei;
                    face& f = faces_[facei++];
                    f.setSize(3);
                    f[0] = faceVerts[elemI-1];
                    f[1] = faceVerts[elemI-2];
                    f[2] = faceVerts[elemI++];
                }
            }
        }
        else if (tag == "METADATA")
        {
            word infoTag(inFile);
            if (infoTag != "INFORMATION")
            {
                FatalIOErrorInFunction(inFile)
                    << "Unsupported tag "
                    << infoTag << exit(FatalIOError);
            }
            label nInfo(readLabel(inFile));
            DebugInfo
                << "Consuming " << nInfo << " metadata information." << nl;

            string line;
            // Consume rest of line
            inFile.getLine(line);
            for (label i = 0; i < 2*nInfo; i++)
            {
                inFile.getLine(line);
            }
        }
        else
        {
            FatalIOErrorInFunction(inFile)
                << "Unsupported tag "
                << tag << exit(FatalIOError);
        }
    }


    // There is some problem with orientation of prisms - the point
    // ordering seems to be different for some exports (e.g. of cgns)
    {
        const cellModel& prism = cellModel::ref(cellModel::PRISM);

        label nSwapped = 0;

        for (cellShape& shape : cells_)
        {
            if (shape.model() == prism)
            {
                const triPointRef bottom
                (
                    points_[shape[0]],
                    points_[shape[1]],
                    points_[shape[2]]
                );
                const triPointRef top
                (
                    points_[shape[3]],
                    points_[shape[4]],
                    points_[shape[5]]
                );

                const point bottomCc(bottom.centre());
                const vector bottomNormal(bottom.areaNormal());
                const point topCc(top.centre());

                if (((topCc - bottomCc) & bottomNormal) < 0)
                {
                    // Flip top and bottom
                    Swap(shape[0], shape[3]);
                    Swap(shape[1], shape[4]);
                    Swap(shape[2], shape[5]);
                    nSwapped++;
                }
            }
        }
        if (nSwapped > 0)
        {
            WarningInFunction << "Swapped " << nSwapped << " prismatic cells"
                << nl;
        }
    }

    if (debug)
    {
        Info<< "Read points:" << points_.size()
            << " cellShapes:" << cells_.size()
            << " faces:" << faces_.size()
            << " lines:" << lines_.size()
            << nl << nl;

        Info<< "Cell fields:" << nl;
        printFieldStats<vectorIOField>(cellData_);
        printFieldStats<scalarIOField>(cellData_);
        printFieldStats<labelIOField>(cellData_);
        printFieldStats<stringIOList>(cellData_);
        Info<< nl << nl;

        Info<< "Point fields:" << nl;
        printFieldStats<vectorIOField>(pointData_);
        printFieldStats<scalarIOField>(pointData_);
        printFieldStats<labelIOField>(pointData_);
        printFieldStats<stringIOList>(pointData_);
        Info<< nl << nl;

        Info<< "Other fields:" << nl;
        printFieldStats<vectorIOField>(otherData_);
        printFieldStats<scalarIOField>(otherData_);
        printFieldStats<labelIOField>(otherData_);
        printFieldStats<stringIOList>(otherData_);
    }
}


// ************************************************************************* //
