// for finding memory leaks in debug mode with Visual Studio 
#if defined _DEBUG && defined _MSC_VER
#include <crtdbg.h>
#endif

#include <SDL2/SDL.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdint.h>
#include <ctype.h> // toupper()
#ifndef _WIN32
#include <unistd.h>
#else
#define WIN32_MEAN_AND_LEAN
#include <windows.h>
#endif
#include "pt_helpers.h"
#include "pt_header.h"
#include "pt_tables.h"
#include "pt_palette.h"

extern SDL_Window *window; // pt_main.c

// used for Windows usleep() implementation
#ifdef _WIN32
static NTSTATUS (__stdcall *NtDelayExecution)(BOOL Alertable, PLARGE_INTEGER DelayInterval);
#endif

// usleep() implementation for Windows
#ifdef _WIN32
void usleep(uint32_t usec)
{
    LARGE_INTEGER lpDueTime;

    if (NtDelayExecution == NULL)
    {
        // NtDelayExecution() is not available (shouldn't happen), use regular sleep()
        Sleep((uint32_t)((usec / 1000.0) + 0.5));
    }
    else
    {
        // this prevents a 64-bit MUL (will not overflow with typical values anyway)
        lpDueTime.HighPart = 0xFFFFFFFF;
        lpDueTime.LowPart  = (DWORD)(-10 * (int32_t)(usec));

        NtDelayExecution(false, &lpDueTime);
    }
}

void setupWin32Usleep(void)
{
    NtDelayExecution = (NTSTATUS (__stdcall *)(BOOL, PLARGE_INTEGER))(GetProcAddress(GetModuleHandle("ntdll.dll"), "NtDelayExecution"));
    timeBeginPeriod(0); // enter highest timer resolution
}

void freeWin32Usleep(void)
{
    timeEndPeriod(0); // exit highest timer resolution
}
#endif

void showErrorMsgBox(const char *fmt, ...)
{
    char strBuf[1024];
    va_list args;

    // format the text string
    va_start(args, fmt);
    vsnprintf(strBuf, sizeof (strBuf), fmt, args);
    va_end(args);

    // window can be NULL here, no problem...
    SDL_ShowSimpleMessageBox(SDL_MESSAGEBOX_ERROR, "Critical Error", strBuf, window);
}

#ifndef _WIN32
int8_t changePathToHome(void)
{
    char *homePath;

    homePath = getenv("HOME");
    if ((homePath != NULL) && (chdir(homePath) == 0))
        return (true);

    return (false);
}
#endif

int8_t sampleNameIsEmpty(char *name)
{
    uint8_t i, n;

    if (name == NULL)
        return (true);

    n = 0;
    for (i = 0; i < 22; ++i)
    {
        if (name[i] == '\0')
            ++n;
    }

    return (n == 22);
}

int8_t moduleNameIsEmpty(char *name)
{
    uint8_t i, n;

    if (name == NULL)
        return (true);

    n = 0;
    for (i = 0; i < 20; ++i)
    {
        if (name[i] == '\0')
            ++n;
    }

    return (n == 20);
}

void updateWindowTitle(int8_t modified)
{
    char titleTemp[128];

    if (modified)
        modEntry->modified = true;

    if (modEntry->head.moduleTitle[0] != '\0')
    {
        if (modified)
        {
            if (ptConfig.modDot)
                sprintf(titleTemp, "ProTracker v2.3D clone (beta #%d) - \"mod.%s\" (unsaved)", BETA_VERSION, modEntry->head.moduleTitle);
            else
                sprintf(titleTemp, "ProTracker v2.3D clone (beta #%d) - \"%s.mod\" (unsaved)", BETA_VERSION, modEntry->head.moduleTitle);
        }
        else
        {
            if (ptConfig.modDot)
                sprintf(titleTemp, "ProTracker v2.3D clone (beta #%d) - \"mod.%s\"", BETA_VERSION, modEntry->head.moduleTitle);
            else
                sprintf(titleTemp, "ProTracker v2.3D clone (beta #%d) - \"%s.mod\"", BETA_VERSION, modEntry->head.moduleTitle);
        }
    }
    else
    {
        if (modified)
        {
            if (ptConfig.modDot)
                sprintf(titleTemp, "ProTracker v2.3D clone (beta #%d) - \"mod.untitled\" (unsaved)", BETA_VERSION);
            else
                sprintf(titleTemp, "ProTracker v2.3D clone (beta #%d) - \"untitled.mod\" (unsaved)", BETA_VERSION);
        }
        else
        {
            if (ptConfig.modDot)
                sprintf(titleTemp, "ProTracker v2.3D clone (beta #%d) - \"mod.untitled\"", BETA_VERSION);
            else
                sprintf(titleTemp, "ProTracker v2.3D clone (beta #%d) - \"untitled.mod\"", BETA_VERSION);
        }
    }

     SDL_SetWindowTitle(window, titleTemp);
}

void recalcChordLength(void)
{
    int8_t note;
    int32_t len;
    moduleSample_t *s;

    s = &modEntry->samples[editor.currSample];

    if (editor.chordLengthMin)
    {
        note = MAX(MAX((editor.note1 == 36) ? -1 : editor.note1,
                       (editor.note2 == 36) ? -1 : editor.note2),
                   MAX((editor.note3 == 36) ? -1 : editor.note3,
                       (editor.note4 == 36) ? -1 : editor.note4));
    }
    else
    {
        note = MIN(MIN(editor.note1, editor.note2), MIN(editor.note3, editor.note4));
    }

    if ((note < 0) || (note > 35))
    {
        editor.chordLength = 0;
    }
    else
    {
        MY_ASSERT(editor.tuningNote < 36);

        if (editor.tuningNote < 36)
        {
            len = (s->length * periodTable[(37 * s->fineTune) + note]) / periodTable[editor.tuningNote];
            if (len > MAX_SAMPLE_LEN)
                len = MAX_SAMPLE_LEN;

            editor.chordLength = len & 0xFFFE;
        }
    }

    if (editor.ui.editOpScreenShown && (editor.ui.editOpScreen == 3))
        editor.ui.updateLengthText = true;
}
