import { InterpolateLinear } from '../constants';
import { AnimationUtils } from './AnimationUtils';
import { InterpolateSmooth, InterpolateDiscrete } from '../constants';
import { CubicInterpolant } from '../math/interpolants/CubicInterpolant';
import { LinearInterpolant } from '../math/interpolants/LinearInterpolant';
import { DiscreteInterpolant } from '../math/interpolants/DiscreteInterpolant';

var KeyframeTrackPrototype;

KeyframeTrackPrototype = {

	TimeBufferType: Float32Array,
	ValueBufferType: Float32Array,

	DefaultInterpolation: InterpolateLinear,

	InterpolantFactoryMethodDiscrete: function( result ) {

		return new DiscreteInterpolant(
				this.times, this.values, this.getValueSize(), result );

	},

	InterpolantFactoryMethodLinear: function( result ) {

		return new LinearInterpolant(
				this.times, this.values, this.getValueSize(), result );

	},

	InterpolantFactoryMethodSmooth: function( result ) {

		return new CubicInterpolant(
				this.times, this.values, this.getValueSize(), result );

	},

	setInterpolation: function( interpolation ) {

		var factoryMethod;

		switch ( interpolation ) {

			case InterpolateDiscrete:

				factoryMethod = this.InterpolantFactoryMethodDiscrete;

				break;

			case InterpolateLinear:

				factoryMethod = this.InterpolantFactoryMethodLinear;

				break;

			case InterpolateSmooth:

				factoryMethod = this.InterpolantFactoryMethodSmooth;

				break;

		}

		if ( factoryMethod === undefined ) {

			var message = "unsupported interpolation for " +
					this.ValueTypeName + " keyframe track named " + this.name;

			if ( this.createInterpolant === undefined ) {

				// fall back to default, unless the default itself is messed up
				if ( interpolation !== this.DefaultInterpolation ) {

					this.setInterpolation( this.DefaultInterpolation );

				} else {

					throw new Error( message ); // fatal, in this case

				}

			}

			console.warn( message );
			return;

		}

		this.createInterpolant = factoryMethod;

	},

	getInterpolation: function() {

		switch ( this.createInterpolant ) {

			case this.InterpolantFactoryMethodDiscrete:

				return InterpolateDiscrete;

			case this.InterpolantFactoryMethodLinear:

				return InterpolateLinear;

			case this.InterpolantFactoryMethodSmooth:

				return InterpolateSmooth;

		}

	},

	getValueSize: function() {

		return this.values.length / this.times.length;

	},

	// move all keyframes either forwards or backwards in time
	shift: function( timeOffset ) {

		if( timeOffset !== 0.0 ) {

			var times = this.times;

			for( var i = 0, n = times.length; i !== n; ++ i ) {

				times[ i ] += timeOffset;

			}

		}

		return this;

	},

	// scale all keyframe times by a factor (useful for frame <-> seconds conversions)
	scale: function( timeScale ) {

		if( timeScale !== 1.0 ) {

			var times = this.times;

			for( var i = 0, n = times.length; i !== n; ++ i ) {

				times[ i ] *= timeScale;

			}

		}

		return this;

	},

	// removes keyframes before and after animation without changing any values within the range [startTime, endTime].
	// IMPORTANT: We do not shift around keys to the start of the track time, because for interpolated keys this will change their values
	trim: function( startTime, endTime ) {

		var times = this.times,
			nKeys = times.length,
			from = 0,
			to = nKeys - 1;

		while ( from !== nKeys && times[ from ] < startTime ) ++ from;
		while ( to !== -1 && times[ to ] > endTime ) -- to;

		++ to; // inclusive -> exclusive bound

		if( from !== 0 || to !== nKeys ) {

			// empty tracks are forbidden, so keep at least one keyframe
			if ( from >= to ) to = Math.max( to , 1 ), from = to - 1;

			var stride = this.getValueSize();
			this.times = AnimationUtils.arraySlice( times, from, to );
			this.values = AnimationUtils.
					arraySlice( this.values, from * stride, to * stride );

		}

		return this;

	},

	// ensure we do not get a GarbageInGarbageOut situation, make sure tracks are at least minimally viable
	validate: function() {

		var valid = true;

		var valueSize = this.getValueSize();
		if ( valueSize - Math.floor( valueSize ) !== 0 ) {

			console.error( "invalid value size in track", this );
			valid = false;

		}

		var times = this.times,
			values = this.values,

			nKeys = times.length;

		if( nKeys === 0 ) {

			console.error( "track is empty", this );
			valid = false;

		}

		var prevTime = null;

		for( var i = 0; i !== nKeys; i ++ ) {

			var currTime = times[ i ];

			if ( typeof currTime === 'number' && isNaN( currTime ) ) {

				console.error( "time is not a valid number", this, i, currTime );
				valid = false;
				break;

			}

			if( prevTime !== null && prevTime > currTime ) {

				console.error( "out of order keys", this, i, currTime, prevTime );
				valid = false;
				break;

			}

			prevTime = currTime;

		}

		if ( values !== undefined ) {

			if ( AnimationUtils.isTypedArray( values ) ) {

				for ( var i = 0, n = values.length; i !== n; ++ i ) {

					var value = values[ i ];

					if ( isNaN( value ) ) {

						console.error( "value is not a valid number", this, i, value );
						valid = false;
						break;

					}

				}

			}

		}

		return valid;

	},

	// removes equivalent sequential keys as common in morph target sequences
	// (0,0,0,0,1,1,1,0,0,0,0,0,0,0) --> (0,0,1,1,0,0)
	optimize: function() {

		var times = this.times,
			values = this.values,
			stride = this.getValueSize(),

			writeIndex = 1;

		for( var i = 1, n = times.length - 1; i <= n; ++ i ) {

			var keep = false;

			var time = times[ i ];
			var timeNext = times[ i + 1 ];

			// remove adjacent keyframes scheduled at the same time

			if ( time !== timeNext && ( i !== 1 || time !== time[ 0 ] ) ) {

				// remove unnecessary keyframes same as their neighbors
				var offset = i * stride,
					offsetP = offset - stride,
					offsetN = offset + stride;

				for ( var j = 0; j !== stride; ++ j ) {

					var value = values[ offset + j ];

					if ( value !== values[ offsetP + j ] ||
							value !== values[ offsetN + j ] ) {

						keep = true;
						break;

					}

				}

			}

			// in-place compaction

			if ( keep ) {

				if ( i !== writeIndex ) {

					times[ writeIndex ] = times[ i ];

					var readOffset = i * stride,
						writeOffset = writeIndex * stride;

					for ( var j = 0; j !== stride; ++ j ) {

						values[ writeOffset + j ] = values[ readOffset + j ];

					}


				}

				++ writeIndex;

			}

		}

		if ( writeIndex !== times.length ) {

			this.times = AnimationUtils.arraySlice( times, 0, writeIndex );
			this.values = AnimationUtils.arraySlice( values, 0, writeIndex * stride );

		}

		return this;

	}

}

export { KeyframeTrackPrototype };