// SPDX-FileCopyrightText: 2024 UnionTech Software Technology Co., Ltd.
//
// SPDX-License-Identifier: GPL-3.0-or-later

#include "ovplugin.h"
#include "ovmodelconfig.h"
#include "ovembproxy.h"

#include <QDir>

#include <openvino/openvino.hpp>

#include <iostream>

#define COMPARE_ARCH(src, trgt) QString::compare(src, trgt, Qt::CaseInsensitive) == 0

GLOBAL_USE_NAMESPACE

OVPlugin::OVPlugin(QObject *parent) : QObject(parent), InferencePlugin()
{

}

OVPlugin::~OVPlugin()
{
    delete ovCore;
    ovCore = nullptr;
}

bool OVPlugin::initialize(const QVariantHash &params)
{
    if (ovCore)
        return true;

    ovCore = new ov::Core;

    std::cerr << ov::get_openvino_version() << std::endl;
    std::vector<std::string> availableDevices = ovCore->get_available_devices();
    std::cerr << "Available devices: " << std::endl;
    for (auto&& device : availableDevices) {
        // Query supported properties and print all of them
        std::cerr << device << " SUPPORTED_PROPERTIES: " << std::endl;
        auto supported_properties = ovCore->get_property(device, ov::supported_properties);
        for (auto&& property : supported_properties) {
            if (property != ov::supported_properties.name()) {
                std::cerr  << "\t" << (property.is_mutable() ? "Mutable: " : "Immutable: ") << property <<
                           " : " << ovCore->get_property(device, property).as<std::string>() << std::endl;
            }
        }
    }

    return true;
}

ModelProxy *OVPlugin::loadModel(const QString &name, const QString &imgDir, const QVariantHash &params)
{
    ModelProxy *ret = nullptr;
    QDir dir(imgDir);
    OVModelConfig cfg (dir.absoluteFilePath("config.json"));

    for (const QString &arch : cfg.architectures()) {
        if (COMPARE_ARCH(arch, "LLM")) {

        } else if (COMPARE_ARCH(arch, "Embedding")) {
            auto model = new OVEmbProxy(name.toStdString(), ovCore);
            if (!model->initialize(imgDir + "/" + cfg.model(), imgDir + "/" + cfg.tokenizer(), params)) {
                delete model;
                model = nullptr;
            }
            ret = model;
        }
    }

    return ret;
}
