// SPDX-FileCopyrightText: 2024 UnionTech Software Technology Co., Ltd.
//
// SPDX-License-Identifier: GPL-3.0-or-later

#include "modelserver.h"
#include "httpserver.h"

#include "runtimestate.h"
#include "modelproxy.h"
#include "embeddingproxy.h"
#include "llmproxy.h"
#include "modeltasks.h"

#include <QFile>
#include <QDateTime>
#include <QCoreApplication>
#include <QDebug>
#include <QJsonDocument>
#include <QJsonArray>
#include <QJsonObject>
#include <QTimer>
#include <QMutex>

#include <iostream>

#include <unistd.h>


GLOBAL_USE_NAMESPACE

ModelServer::ModelServer(QObject *parent) : QObject(parent)
{

}

ModelServer::~ModelServer()
{
    stop();
}

bool ModelServer::run(ModelRunner *mr)
{
    Q_ASSERT(mr);
    if (stateFile)
        return false;

    runner = mr;
    RuntimeState::mkpath();
    state = new RuntimeState(QString::fromStdString(mr->modelProxy->name()));
    stateFile = new QFile(state->stateFile());
    if (!stateFile->open(QFile::ReadWrite| QFile::Truncate)) {
        std::cerr << QString("cant not create state file %0").arg(state->stateFile()).toStdString() << std::endl;
        delete state;
        state = nullptr;

        delete stateFile;
        stateFile = nullptr;

        return false;
    }

    http = new HttpServer(this);

    const QString host = uHost.isEmpty() ? "127.0.0.1" : uHost;
    const int port = uPort > 0 ? uPort : HttpServer::randomPort();

    if (!http->initialize(host, port))
        return false;

    // must block this signal
    connect(http, &HttpServer::newRequset, this, &ModelServer::onReqeust, Qt::BlockingQueuedConnection);
    http->start();
    runner->start();

    {
        QVariantHash st;
        st.insert("model", QString::fromStdString(mr->modelProxy->name()));
        st.insert("pid", getpid());
        st.insert("host", host);
        st.insert("port", port);
        st.insert("starttime", QDateTime::currentDateTime().toString("yyyy-MM-dd hh:mm:ss"));
        state->writeSate(stateFile, st);
    }

    http->registerAPI(HttpServer::Post, "/embeddings");
    http->registerAPI(HttpServer::Post, "/chat/completions");
    http->registerAPI(HttpServer::Get, "/stop");

    resetIdle();
    return true;
}

void ModelServer::stop()
{
    if (stateFile) {
        stateFile->remove();
        delete stateFile;
        stateFile = nullptr;
    }

    if (state) {
        delete state;
        state = nullptr;
    }

    if (http) {
        delete http;
        http = nullptr;
    }
}

void ModelServer::setIdle(int s)
{
    if (s < 10 && idleTimer) {
        std::cerr << "idle time is invalid:" << s << std::endl;
        delete idleTimer;
        idleTimer = nullptr;
        return;
    }

    if (idleTimer) {
        idleTimer->setInterval(s * 1000);
        return;
    }

    idleTimer = new QTimer(this);
    idleTimer->setInterval(s * 1000);
    idleTimer->setSingleShot(true);
    connect(idleTimer, &QTimer::timeout, this, &ModelServer::onIdle);
}

void ModelServer::resetIdle()
{
    if (http)
        http->setLastWorkTime(QDateTime::currentSecsSinceEpoch());

    if (!idleTimer)
        return;

    if (QThread::currentThread() == idleTimer->thread())
        idleTimer->start();
    else
        QMetaObject::invokeMethod(idleTimer, "start");
}

void ModelServer::stopIdle()
{
    if (http)
        http->setLastWorkTime(-1);

    if (!idleTimer)
        return;

    if (QThread::currentThread() == idleTimer->thread())
        idleTimer->stop();
    else
        QMetaObject::invokeMethod(idleTimer, "stop");
}

int ModelServer::instance(const QString &model)
{
    if (model.isEmpty())
        return ::getpid();

    RuntimeState st(model);
    return st.pid();
}

void ModelServer::onReqeust(void *ptr)
{
    // new requset and stop idle
    stopIdle();

    HttpContext *ctx = static_cast<HttpContext *>(ptr);
    const QString api = HttpServer::getPath(ctx);
    const QString reqBody = HttpServer::getBody(ctx);

    if (api == "/stop") {
        qApp->quit();
    } else if (api == "/embeddings") {
        if (auto emb = dynamic_cast<EmbeddingProxy *>(runner->modelProxy.data())) {
            QSharedPointer<ModelTask> task(new EmbTask(runner, reqBody, emb, ctx));
            runner->postTask(task);
            runner->recvTask(task);
        } else {
            HttpServer::setStatus(ctx, 403);
            QJsonObject obj;
            obj.insert("invalid_request_error", QString("The API is not supported by model %0").arg(QString::fromStdString(runner->modelProxy->name())));
            HttpServer::setContent(ctx, QString::fromUtf8(QJsonDocument(obj).toJson(QJsonDocument::Compact)));
        }
    } else if (api == "/chat/completions") {
        if (auto llm = dynamic_cast<LLMProxy *>(runner->modelProxy.data())) {
            QJsonParseError er;
            auto doc = QJsonDocument::fromJson(reqBody.toUtf8(), &er);
            if (er.error == QJsonParseError::NoError) {
                auto root = doc.object().toVariantHash();
                if (root.value("stream").toBool()) {
                    QSharedPointer<ModelTask> task(new ChatStreamTask(runner, root, llm, nullptr));
                    runner->postTask(task);

                    auto chunk = [task](QString &out, bool &stop) {
                        ChatStreamTask *cst = dynamic_cast<ChatStreamTask *>(task.data());

                        QMutexLocker lk(&cst->genMtx);
                        if (!cst->stop)
                            cst->con.wait(&cst->genMtx);

                        QString buff = cst->text;
                        cst->text.clear();
                        stop = cst->stop;
                        lk.unlock();

                        QVariantHash hash;
                        hash.insert("finish_reason",  stop ? QVariant("stop") : QVariant());
                        {
                            QVariantList choices;
                            QVariantHash c1;
                            c1.insert("index", 0);
                            QVariantHash delta;
                            delta.insert("role", "assistant");
                            delta.insert("content", buff);
                            c1.insert("delta", delta);
                            choices.append(c1);
                            hash.insert("choices", choices);
                        }

                        out = QJsonDocument(QJsonObject::fromVariantHash(hash)).toJson(QJsonDocument::Compact);
                        out = QString("data:%0\n").arg(out);
                        return true;
                    };

                    auto onComplete = [this, task](bool) {
                        ChatStreamTask *cst = dynamic_cast<ChatStreamTask *>(task.data());
                        {
                            QMutexLocker lk(&cst->genMtx);
                            cst->stop = true;
                        }

                        //relese task
                        runner->recvTask(task);
                        this->resetIdle();
                    };

                    HttpServer::setChunckProvider(ctx, chunk, onComplete);

                    // must be return
                    return;
                } else {
                    QSharedPointer<ModelTask> task(new ChatCompletionsTask(runner, root, llm, ctx));
                    runner->postTask(task);
                    runner->recvTask(task);
                }
            } else {
                HttpServer::setStatus(ctx, 403);
                QJsonObject obj;
                obj.insert("invalid_request_error", "Invalid input content");
                HttpServer::setContent(ctx, QString::fromUtf8(QJsonDocument(obj).toJson(QJsonDocument::Compact)));
            }
        } else {
            HttpServer::setStatus(ctx, 500);
            QJsonObject obj;
            obj.insert("error", QString("API /chat/completions not supported by model %0").arg(QString::fromStdString(runner->modelProxy->name())));
            HttpServer::setContent(ctx, QString::fromUtf8(QJsonDocument(obj).toJson(QJsonDocument::Compact)));
        }
    } else {
        HttpServer::setStatus(ctx, 403);
        QJsonObject obj;
        obj.insert("invalid_request_error", "The API is not open");
        HttpServer::setContent(ctx, QString::fromUtf8(QJsonDocument(obj).toJson(QJsonDocument::Compact)));
    }

    resetIdle();
}

void ModelServer::onIdle()
{
    std::cerr << "server exit by idle" << std::endl;
    qApp->quit();
}
