//
// Created by uos on 2022/2/16.
//

#include "Device.h"
#include "utils/Utils.h"
#include "utils/global.h"
#include "Process.h"
#include "FsTab.h"
#include <QDebug>
#include <QRegExp>
#include <QDir>
#include <QFileInfo>
#include <QJsonParseError>
#include <QJsonObject>
#include <QJsonArray>
#include <QJsonValue>
#include <QUuid>
#include <QSet>


DeviceInfoList Device::m_allDevice;

Device::Device()
{

}

Device::Device(const QString &devNameOrUUID)
{
    m_pDevice = findDevice(devNameOrUUID);
}

Device::~Device()
{

}

QString Device::getLsblkByOutput()
{
    QStringList args;
    args << "--bytes" << "--pairs" << "--output";
    args << "NAME,KNAME,LABEL,UUID,TYPE,FSTYPE,SIZE,FSUSED,FSAVAIL,LOG-SEC,PHY-SEC,MOUNTPOINT,MODEL,RO,"
            "HOTPLUG,MAJ:MIN,PARTLABEL,PARTUUID,PKNAME,PATH,VENDOR,SERIAL,REV,ROTA";
    QString out;
    Process::spawnCmd("lsblk", args, out);

    return out;
}

DeviceInfoList Device::getDeviceInfoByLsblkOut(const QString &lsblkOut)
{
    DeviceInfoList allDeviceInfo;
    QRegExp regExp(R"lit(NAME="(.*)" KNAME="(.*)" LABEL="(.*)" UUID="(.*)" TYPE="(.*)" FSTYPE="(.*)" SIZE="(.*)" FSUSED="(.*)" FSAVAIL="(.*)" LOG-SEC="(.*)" PHY-SEC="(.*)" MOUNTPOINT="(.*)" MODEL="(.*)" RO="([0-9]+)" HOTPLUG="([0-9]+)" MAJ[_:]MIN="([0-9:]+)" PARTLABEL="(.*)" PARTUUID="(.*)" PKNAME="(.*)" PATH="(.*)" VENDOR="(.*)" SERIAL="(.*)" REV="(.*)" ROTA="(.*)")lit");

#if QT_VERSION >= QT_VERSION_CHECK(5,15,0)
    QStringList lsblkOutList = lsblkOut.split("\n", Qt::SkipEmptyParts);
#else
    QStringList lsblkOutList = lsblkOut.split("\n", QString::SkipEmptyParts);
#endif

    for (QString &line : lsblkOutList) {
        if (!regExp.exactMatch(line)) {
            continue;
        }

        //qInfo() << line;
        bool ok = false;
        int pos = 1;
        DeviceInfoPtr device(new DeviceInfo);
        device->name = regExp.cap(pos++).trimmed();
        device->kname = regExp.cap(pos++).trimmed();
        device->label = regExp.cap(pos++).trimmed();
        device->uuid = regExp.cap(pos++).trimmed();
        device->type = regExp.cap(pos++).trimmed();
        device->fsType = regExp.cap(pos++).trimmed().toLower();
        device->fsType = (device->fsType == "crypto_luks") ? "luks" : device->fsType;
        device->fsType = (device->fsType == "lvm2_member") ? "lvm2" : device->fsType;
        device->sizeBytes = regExp.cap(pos++).trimmed().toLongLong();
        QString usedSize = regExp.cap(pos++).trimmed();
        if (!usedSize.isEmpty()) {
            device->usedBytes = usedSize.toULongLong();
        }

        QString availableSize = regExp.cap(pos++).trimmed();
        if (!availableSize.isEmpty()) {
            device->availableBytes = availableSize.toULongLong();
        }

        QString logSectorSize = regExp.cap(pos++).trimmed();
        if (!logSectorSize.isEmpty()) {
            device->logSectorSize = logSectorSize.toULongLong();
        }

        QString phySectorSize = regExp.cap(pos++).trimmed();
        if (!phySectorSize.isEmpty()) {
            device->phySectorSize = phySectorSize.toULongLong();
        }

        device->mountPoint = regExp.cap(pos++).trimmed();
        device->mode = regExp.cap(pos++).trimmed();
        device->readOnly = (regExp.cap(pos++).trimmed() == "1");
        device->removable = (regExp.cap(pos++).trimmed() == "1");
        device->majMin = regExp.cap(pos++).trimmed();
        int index = device->majMin.indexOf(":");
        device->major = device->majMin.left(index).toInt(&ok);
        if (!ok) {
            device->major = -1;
        }

        device->minor = device->majMin.right(device->majMin.length() - index - 1).toInt(&ok);
        if (!ok) {
            device->minor = -1;
        }
        // partlabel可能以空格开始或结束,不要trimmed()
        device->partLabel = regExp.cap(pos++);
        device->partuuid = regExp.cap(pos++).trimmed();
        device->pkname = regExp.cap(pos++).trimmed();
        device->path = regExp.cap(pos++).trimmed();
        device->vendor = regExp.cap(pos++).trimmed();
        device->serial = regExp.cap(pos++).trimmed();
        device->revision = regExp.cap(pos++).trimmed();
        device->rota = regExp.cap(pos++).trimmed().toInt(&ok);
        if (!ok) {
            device->rota = -1;
        }
        device->deviceName = QString("/dev/%1").arg(device->kname);
        if (!device->uuid.isEmpty()) {
            device->deviceByUUID = QString("/dev/disk/by-uuid/%1").arg(device->uuid);
        }
        if (!device->label.isEmpty()) {
            device->deviceByLabel = QString("/dev/disk/by-label/%1").arg(device->label);
        }
        if (!device->partuuid.isEmpty()) {
            device->deviceByPartUUID = QString("/dev/disk/by-partuuid/%1").arg(device->partuuid);
        }
        if (!device->partLabel.isEmpty()) {
            device->deviceByPartLabel = QString("/dev/disk/by-partlabel/%1").arg(device->partLabel);
        }

        allDeviceInfo.append(device);
    }

    return allDeviceInfo;
}

DeviceInfoList Device::getOrganizedDiskList(DeviceInfoList &diskList, DeviceInfoList &cryptDevList,
    DeviceInfoList &noCryptDevList, DeviceInfoList &partitionList, DeviceInfoList& lvmList)
{
    DeviceInfoList &fullFamilyDiskList = diskList;
    for (auto &diskDev : fullFamilyDiskList) {
        for (auto &cryptDev : cryptDevList) {
            if (!(cryptDev->type == "part" && cryptDev->fsType == "luks" && cryptDev->pkname == diskDev->name)) {
                continue;
            }
            cryptDev->parent = diskDev;

            for (auto &lvm2Dev : cryptDevList) {
                if (!(lvm2Dev->type == "crypt" && lvm2Dev->fsType == "lvm2" && lvm2Dev->pkname == cryptDev->kname)) {
                    continue;
                }
                lvm2Dev->parent = cryptDev;

                for (auto &lvmMember : lvmList) {
                    if (lvmMember->pkname == lvm2Dev->kname) {
                        lvmMember->parent = lvm2Dev;
                        lvm2Dev->children.append(lvmMember);
                    }
                }

                if (lvm2Dev->children.size() > 1) {
                    std::sort(lvm2Dev->children.begin(), lvm2Dev->children.end(), deviceInfoChildrenCompare);
                }
                cryptDev->children.append(lvm2Dev);
            }
            diskDev->children.append(cryptDev);
        }

        for (auto &noCryptDev : noCryptDevList) {
            if (noCryptDev->pkname != diskDev->name) {
                continue;
            }
            noCryptDev->parent = diskDev;

            for (auto &lvmMember : lvmList) {
                if (lvmMember->pkname == noCryptDev->kname) {
                    lvmMember->parent = noCryptDev;
                    noCryptDev->children.append(lvmMember);
                }
            }
            if (noCryptDev->children.size() > 1) {
                std::sort(noCryptDev->children.begin(), noCryptDev->children.end(), deviceInfoChildrenCompare);
            }
            diskDev->children.append(noCryptDev);
        }

        for (auto &partitionDev : partitionList) {
            if (partitionDev->pkname == diskDev->name) {
                partitionDev->parent = diskDev;
                diskDev->children.append(partitionDev);
            }
        }
        if (diskDev->children.size() > 1) {
            std::sort(diskDev->children.begin(), diskDev->children.end(), deviceInfoChildrenCompare);
        }
    }

    return fullFamilyDiskList;
}

bool deviceInfoChildrenCompare(const DeviceInfoPtr left, const DeviceInfoPtr right)
{
    if (nullptr == left || nullptr == right) {
        return false;
    }

    if (left->major == right->major && left->minor < right->minor) {
        return true;
    }

    return false;
}

DeviceInfoList Device::getAllDeviceByLsblk()
{
    QString lsblkOut = Device::getLsblkByOutput();
    DeviceInfoList allDeviceInfo = Device::getDeviceInfoByLsblkOut(lsblkOut);

    //lvm 从/dev/mapper/  中获取映射名
    QDir mapperDir("/dev/mapper/");
    for (auto file : mapperDir.entryInfoList()) {
        if (file.fileName() == "control") {
            continue;
        }

        if (file.isSymLink()) {
            QString mapperDevice = file.symLinkTarget();
            //qInfo() << "mapperDevice=" << mapperDevice;
            for (auto dev : allDeviceInfo) {
                if (dev->deviceName == mapperDevice) {
                    dev->mappedName = file.fileName();
                    dev->symlink = file.absoluteFilePath();
                    //qInfo() << "mapperdName" << dev->mappedName;
                    //qInfo() << "symlink" << dev->symlink;
                }
            }
        }
    }

    Device::findChildByDmsetup(allDeviceInfo);

    return allDeviceInfo;
}

QStringList Device::getSysDevicesByFstab(const QString &fstabFile)
{
    static FSTabInfoList fstabList = FSTab::getFSTabFromFile(fstabFile);
    static QStringList sysDeviceUuidList = FSTab::getUuidListFromFstab(fstabList);

    DeviceInfoList diskList;
    DeviceInfoList cryptDevList;
    DeviceInfoList noCryptDevList;
    DeviceInfoList partitionList;
    DeviceInfoList lvmList;
    Device::getAllTypeDevice(diskList, cryptDevList, noCryptDevList, partitionList, lvmList);

    QStringList sysDeviceList;
    for (auto &diskDev : diskList) {
        for (auto &cryptDev : cryptDevList) {
            if (!(cryptDev->type == "part" && cryptDev->fsType == "luks" && cryptDev->pkname == diskDev->name)) {
                continue;
            }

            for (auto &lvm2Dev : cryptDevList) {
                if (!(lvm2Dev->type == "crypt" && lvm2Dev->fsType == "lvm2" && lvm2Dev->pkname == cryptDev->kname)) {
                    continue;
                }

                for (auto &lvmMember : lvmList) {
                    if (lvmMember->pkname == lvm2Dev->kname) {
                        if (sysDeviceUuidList.contains(lvmMember->uuid) && !sysDeviceList.contains(diskDev->name)) {
                            sysDeviceList.append(diskDev->name);
                            break;
                        }
                    }
                }
            }
        }

        for (auto &noCryptDev : noCryptDevList) {
            if (noCryptDev->pkname != diskDev->name) {
                continue;
            }

            for (auto &lvmMember : lvmList) {
                if (lvmMember->pkname == noCryptDev->kname) {
                    if (sysDeviceUuidList.contains(lvmMember->uuid) && !sysDeviceList.contains(diskDev->name)) {
                        sysDeviceList.append(diskDev->name);
                        break;
                    }
                }
            }
        }

        for (auto &partitionDev : partitionList) {
            if (partitionDev->pkname == diskDev->name) {
                if (sysDeviceUuidList.contains(partitionDev->uuid) && !sysDeviceList.contains(diskDev->name)) {
                    sysDeviceList.append(diskDev->name);
                }
            }
        }
    }

    sysDeviceList.removeDuplicates();

    return sysDeviceList;
}

DeviceInfoList Device::getDeviceByLsblk()
{
    m_allDevice.clear();
    m_allDevice = Device::getAllDeviceByLsblk();
    return m_allDevice;
}

bool Device::exportPartitionInfoByLsblk(const QString &filepath, const QStringList &deviceList, QString &err)
{
    // 使用lsblk获取信息，从json解析分区信息
    QJsonArray deviceArray;
    if (!Utils::getLsblkJsonReturn("lsblk -JOb", deviceArray, err)) {
        return false;
    }

    if (Utils::filterDevice(deviceArray) == 0) {
        return false;
    }

    err = "";
    QMap<SVGInfo, QList<SLVMInfo>> lvmInfos;
//    QMap<QString, SVGInfo> vgNames;
//    QMap<QString, SLVMInfo> lvNames;
//    if (!Device::getVGNamesInfo(vgNames, err)) {
//        err = "exportPartitionInfoByLsblk getVGnames failed vgNames is empty !";
//        return false;
//    }

    DeviceInfoList diskList;
    DeviceInfoList cryptDevList;
    DeviceInfoList noCryptDevList;
    DeviceInfoList partitionList;
    DeviceInfoList lvmList;
    Device::getAllTypeDevice(diskList, cryptDevList, noCryptDevList, partitionList, lvmList);

    QMap<QString, QString> crypt2Partition;  // </dev/mapper/luks_crypt0, /dev/sda3>
    for (auto &crypt : cryptDevList) {
        for (auto &luks : cryptDevList) {
            if (luks->pkname == crypt->kname) {
                crypt2Partition[luks->path] = crypt->path;
                break;
            }
        }
    }

    QStringList vgNameList;
    if (!Device::getVGNames(vgNameList)) {
        qCritical()<<"exportPartitionInfoByLsblk getVGNames failed";
        return false;
    }

    QMap<QString, QString> pvVgMap;
    if (!Device::getPvDisplay(pvVgMap)) {
        qCritical()<<"exportPartitionInfoByLsblk getPvDisplay failed";
        return false;
    }

    QString fstabFile = "/etc/fstab";
    QStringList sysDevList = Device::getSysDevicesByFstab(fstabFile);

    QMap<QString, SVGInfo> vgNames;
    for (const QString &vg : vgNameList) {
        for (const QString &pv : pvVgMap.keys()) {
            if (vg != pvVgMap[pv]) {
                continue;
            }
            if (pv.startsWith("/dev/mapper/luks_crypt")) {
                SVGInfo &vgInfo = vgNames[vg];
                vgInfo.name = vg;
                vgInfo.pvIds.append(crypt2Partition[pv]);
                continue;
            }
            // deal with no encrypt partition
            for (const QString &sysDev : sysDevList) {
                if (pv.startsWith("/dev/" + sysDev)) {
                    SVGInfo &vgInfo = vgNames[vg];
                    vgInfo.name = vg;
                    vgInfo.pvIds.append(pv);
                    break;
                }
            }
        }
    }

    QMap<QString, SLVMInfo> lvNames;
    if (!cryptDevList.isEmpty()) {
        for (const QString &vg : vgNameList) {
            for (auto lvmPartition : lvmList) {
                QString vgName = lvmPartition->name.left(lvmPartition->name.indexOf("-"));
                if (vg == vgName) {
                    SVGInfo &vgInfo = vgNames[vg];
                    vgInfo.name = vg;
                    vgInfo.size += lvmPartition->sizeBytes / MiB;

                    SLVMInfo &lvmInfo = lvNames[lvmPartition->name];
                    lvmInfo.label = lvmPartition->label;
                    if (lvmInfo.label.isEmpty()) {
                        if ("swap" == lvmPartition->fsType) {
                            lvmInfo.label = "SWAP"; // 分区label是空的会导致调用安装器分区工具失败
                        }
                    }
                    lvmInfo.size = lvmPartition->sizeBytes / MiB;
                    lvmInfo.mountPoint = lvmPartition->mountPoint;
                    lvmInfo.filesystem = lvmPartition->fsType;
                    lvmInfo.usage = true;
                    lvmInfo.vgName = vg;
                }
            }
        }
    }

    QMap<QString, QList<SPartitionInfo>> devicesInfos;
    for (int i = 0; i < deviceArray.size(); i++) {
        // 初始化SDeviceInfo、SPartitionInfo列表
        QJsonObject deviceObj = deviceArray.at(i).toObject();
        QString devicePath = deviceObj.value("path").toString();
        QList<SPartitionInfo> partInfoItems;
        QJsonArray devicePartArray = deviceObj.value("children").toArray();
        for (int j = 0; j < devicePartArray.size(); j++ ) {
            QJsonObject devicePartObj = devicePartArray.at(j).toObject();
            SPartitionInfo partInfoItem = Device::newPartitionByJObj(devicePath, devicePartObj);
            partInfoItem.device = devicePath;
            partInfoItem.deviceSize = deviceObj.value("size").toVariant().toLongLong() / MiB;

            // 更新vgnames和lvnames
            Device::updateVGNames(devicePartObj, vgNames, lvNames);
            Device::updatePartIsMergeVG(vgNames, partInfoItem);

            partInfoItems.append(partInfoItem);
        }

        // 使用fdisk命令获取分区的起止位置信息
        if (!Device::updatePartStartEndPoint(devicePath, partInfoItems)) {
            err = "exportPartitionInfoByLsblk updatePartStartEndPoint failed !";
            return false;
        }

        if (!Device::updatePartType(devicePath, partInfoItems)) {
            err = "exportPartitionInfoByLsblk updatePartType failed !";
            return false;
        }

        // 根据分区起始位置做一次排序处理
        devicesInfos.insert(devicePath, partInfoItems);
    }

    // 根据vgnames和lvnames更新m_lvmInfos
    Device::updateLvmInfos(devicesInfos, vgNames, lvNames, lvmInfos);

    if (deviceList.size() == 0) {
        // 根据fstab将当前系统用到的设备信息选取出来
        Device::updateDeviceInfoMapByFstab(devicesInfos);
    } else {
        for (QString deviceName : devicesInfos.keys()) {
            if (deviceList.indexOf(deviceName) == -1) {
                devicesInfos.remove(deviceName);
            }
        }
    }

    // 将分区信息导出到partition_policy.json文件
    Device::writeDeviceInfo(filepath, devicesInfos, lvmInfos);
    return true;
}

void Device::calculateDiskSpace()
{
    QStringList args = {"-T", "-B1"};
    QString out;
    if (Process::spawnCmd("df", args, out)) {

        /*  sample output
            文件系统       类型            1B-块         已用         可用 已用% 挂载点
            udev           devtmpfs   8074424320            0   8074424320    0% /dev
            tmpfs          tmpfs      1623896064      3158016   1620738048    1% /run
            /dev/nvme0n1p5 btrfs    147065929728 122901000192  23372861440   85% /
            tmpfs          tmpfs      8119472128    238022656   7881449472    3% /dev/shm
            tmpfs          tmpfs         5242880         4096      5238784    1% /run/lock
            tmpfs          tmpfs      8119472128            0   8119472128    0% /sys/fs/cgroup
            /dev/nvme0n1p3 ext4       1551745024    131887104   1322553344   10% /boot
            tmpfs          tmpfs      1623891968        40960   1623851008    1% /run/user/1000
            /dev/sda3      fuseblk  355953799168 224915378176 131038420992   64% /media/uos/本地磁盘
            /dev/sda1      fuseblk  322123595776  56340246528 265783349248   18% /media/uos/0BD418FA279A9BA3
            /dev/sda2      fuseblk  322123591680  20298776576 301824815104    7% /media/uos/8B78E6D0A657E529
            /dev/nvme0n1p1 fuseblk  106680532992  32996925440  73683607552   31% /media/uos/44BA472DBA471B36
         * */
        auto lines = out.split("\n");
        int linuNum = 0;
        for (const auto& line : lines) {
        //    qInfo() << line;
            if (++linuNum == 1 || line.trimmed().length() == 0) {
                continue;
            }
#if QT_VERSION >= QT_VERSION_CHECK(5,15,0)
            auto cols = line.split(" ", Qt::SkipEmptyParts);
#else
            auto cols = line.split(" ", QString::SkipEmptyParts);
#endif
            if (cols.size() != 7) {
                continue;
            }
            auto deviceInfo = findDevice(cols[0].trimmed());
            if (deviceInfo.isNull()) {
                continue;
            }
            int i = 1;
            for (auto col : cols) {
                switch (i++) {
                    case 1:
                    case 2:
                        break;
                    case 3:
                        deviceInfo->sizeBytes = col.trimmed().toLongLong();
                        break;
                    case 4:
                        deviceInfo->usedBytes = col.trimmed().toLongLong();
                        break;
                    case 5:
                        deviceInfo->availableBytes = col.trimmed().toLongLong();
                        break;
                    case 6:
                        deviceInfo->usedPercent = col.trimmed();
                        break;
                    case 7:
                        break;
                }

            }

        }
    }
}

bool Device::mount(const QString &mountPoint)
{
    if (m_pDevice.isNull() || m_pDevice->type != "part") {
        return false;
    }

    if (!m_pDevice->mountPoint.isEmpty() && m_pDevice->mountPoint != mountPoint) {
        return false;
    }
    QString out, err;
    bool ret = Process::spawnCmd("mount",
                      QStringList() << m_pDevice->deviceName << mountPoint,
                      out, err);
    if (ret) {
        m_pDevice->mountPoint = mountPoint;
    }
    return ret;
}

bool Device::umount()
{
    if (m_pDevice.isNull() || m_pDevice->type != "part") {
        return false;
    }
    if (m_pDevice->mountPoint.isEmpty()) {
        return true;
    }
    QString out, err;
    bool ret = Process::spawnCmd("umount",
                                 QStringList() << m_pDevice->mountPoint,
                                 out, err);
    if (ret) {
        m_pDevice->mountPoint.clear();
    }
    return ret;
}

void Device::getDeviceByName(const QString &name)
{
    auto deviceInfo = findDevice(name);
    if (!deviceInfo.isNull()) {
        m_pDevice = deviceInfo;
    }

}

void Device::getDeviceByUUID(const QString &uuid)
{
    auto deviceInfo = findDevice(uuid);
    if (!deviceInfo.isNull()) {
        m_pDevice = deviceInfo;
    }
}


void Device::findChildByDmsetup(DeviceInfoList &deviceList)
{
    QString out;
    QStringList args = {"deps", "-o", "blkdevname"};
    if (Process::spawnCmd("dmsetup", args, out)) {
        for (auto line : out.split("\n")) {
            //qInfo() << line;
            if (line.trimmed().length() == 0) {
                continue;
            }
            QRegExp regExp(R"lit(([^:]*)\:.*\((.*)\))lit");
            if (!regExp.exactMatch(line)) {
                continue;
            }
            //qInfo() << "regExp.cap(1)" << regExp.cap(1).trimmed();
            //qInfo() << "regExp.cap(2)" << regExp.cap(2).trimmed();
            QString childName = regExp.cap(1).trimmed();
            QString parentName = regExp.cap(2).trimmed();
            DeviceInfoPtr parent;
            auto parentIter = std::find_if(m_allDevice.begin(), m_allDevice.end(), [=](DeviceInfoPtr devItem){ return devItem->kname == parentName; });
            if (parentIter != m_allDevice.end()) {
                parent = (*parentIter);
            }

            DeviceInfoPtr child;
            auto childIter = std::find_if(m_allDevice.begin(), m_allDevice.end(), [=](DeviceInfoPtr devItem){ return devItem->mappedName == childName; });
            if (childIter != m_allDevice.end()) {
                child = (*childIter);
            }

            if (!parent.isNull() && !child.isNull()) {
                child->pkname = parent->kname;
            //    qInfo() << "child.kname=" << child->kname;
            //   qInfo() << "child.pkname=" << parent->kname;
            }
        }
    }
}

DeviceInfoPtr Device::findDevice(const QString &devAilas)
{
    for (auto dev : m_allDevice) {
        if (dev->deviceName.startsWith("/dev/loop")) {
            continue; // dim 转换异常场景下，会导致判断root uuid错误，识别到loop里的uuid
        }

        if (dev->deviceName == devAilas) {
            return dev;
        } else if (dev->name == devAilas) {
            return dev;
        } else if (dev->kname == devAilas) {
            return dev;
        } else if (dev->uuid == devAilas) {
            return dev;
        } else if (dev->label == devAilas) {
            return dev;
        } else if (dev->partuuid == devAilas) {
            return dev;
        } else if (dev->partLabel == devAilas) {
            return dev;
        } else if (dev->deviceByUUID == devAilas) {
            return dev;
        } else if (dev->deviceByLabel == devAilas) {
            return dev;
        } else if (dev->deviceByPartLabel == devAilas) {
            return dev;
        } else if (dev->deviceByPartUUID == devAilas) {
            return dev;
        } else if (dev->mappedName == devAilas) {
            return dev;
        } else if (dev->mountPoint == devAilas) {
            return dev;
        }

    }
    return DeviceInfoPtr();
}

QString Device::size()
{
    if (m_pDevice->sizeBytes < GiB) {
        return QString().asprintf("%.1f MB", m_pDevice->sizeBytes / MiB);
    } else if (m_pDevice->sizeBytes > 0) {
        return QString().asprintf("%.1f GB", m_pDevice->sizeBytes / GiB);
    }
    return QString();
}

QString Device::used()
{
    return (m_pDevice->usedBytes == 0) ? "" : QString().asprintf("%.1f GB", m_pDevice->usedBytes / GiB);
}

QString Device::free()
{
    return (m_pDevice->availableBytes == 0) ? "" : QString().asprintf("%.1f GB", m_pDevice->availableBytes / GiB);
}

DeviceInfoPtr Device::getDeviceInfo()
{
    return m_pDevice;
}

bool Device::isLinuxFilesystem(const DeviceInfoPtr &dev)
{
    QStringList fstype = {"ext2", "ext3", "ext4", "f2fs", "reiserfs", "reiser4",
                          "xfs", "jfs", "zfs", "zfs_member", "btrfs", "lvm", "lvm2_member",
                          "luks", "crypt", "crypto_luks"};

    return fstype.contains(dev->fsType);
}

bool Device::isFsTypeSupported(const QString &fsType)
{
    // 系统备份支持的文件系统类型
    const QStringList fstype = {"ext4", "btrfs", "xfs", "reiserfs"};

    return fstype.contains(fsType);
}

bool Device::isFsTypeSupportedDataBackup(const QString &fsType)
{
    // 数据备份支持的文件系统类型
    const QStringList fstype = {"ext4", "btrfs", "xfs", "reiserfs"};

    return fstype.contains(fsType);
}

bool Device::isDisk(const DeviceInfoPtr &dev)
{
    if (dev->type == "disk" && !dev->mode.isEmpty()) {
        return true;
    }
    return false;
}

SPartitionInfo Device::newPartitionByJObj(const QString &deviceName, const QJsonObject &partitionInfoObj)
{
    SPartitionInfo partInfoItem;
    partInfoItem.size           = partitionInfoObj.value("size").toVariant().toLongLong() / MiB;
    partInfoItem.id             = QUuid::createUuid().toString();
    partInfoItem.label          = partitionInfoObj.value("label").toString();
    partInfoItem.usage          = true;
    partInfoItem.partitionType  = EPartitionType::PartitionTypePrimary;
    partInfoItem.device         = deviceName;
    partInfoItem.index          = partitionInfoObj.value("path").toString().replace(deviceName, "").replace(QRegExp("[A-Za-z]"), "").toInt();
    partInfoItem.path           = partitionInfoObj.value("path").toString();
    partInfoItem.filesystem     = partitionInfoObj.value("fstype").toString();
    partInfoItem.mountPoint     = partitionInfoObj.value("mountpoint").toString();
    partInfoItem.startPoint     = 0;
    partInfoItem.endPoint       = 0;
    partInfoItem.sectorSize     = partitionInfoObj.value("log-sec").toVariant().toLongLong();
    partInfoItem.phySectorSize  = partitionInfoObj.value("phy-sec").toVariant().toLongLong();
    partInfoItem.isStartPoint   = true;
    partInfoItem.deviceSize     = 0;
    partInfoItem.isEFIPartition = false;
    partInfoItem.isMergeVG      = false;
    return partInfoItem;
}

bool Device::updatePartStartEndPoint(const QString &deviceInfo, QList<SPartitionInfo> &partitionInfos)
{
    QString devicePartInfoString = "";
    QString cmd = QString("fdisk -l -o Device,Start,End,Type | grep %1").arg(deviceInfo);
    if (!Process::spawnCmd("/bin/bash", {"-c", cmd}, devicePartInfoString)) {
        qWarning() << QString("Command: %1 failed. %2").arg(cmd).arg(devicePartInfoString);
        return false;
    }

#if QT_VERSION >= QT_VERSION_CHECK(5,15,0)
    QStringList partInfoStringList = devicePartInfoString.split("\n", Qt::SkipEmptyParts);
#else
    QStringList partInfoStringList = devicePartInfoString.split("\n", QString::SkipEmptyParts);
#endif

    // 更新SPartitionInfo列表中的信息
    for (int i = 0; i < partitionInfos.size(); i++) {
        for (QString partInfoItem : partInfoStringList) {
            if (partInfoItem.contains(partitionInfos.at(i).path)) {
#if QT_VERSION >= QT_VERSION_CHECK(5,15,0)
                QStringList partInfoListItem = partInfoItem.split(" ", Qt::SkipEmptyParts);
#else
                QStringList partInfoListItem = partInfoItem.split(" ", QString::SkipEmptyParts);
#endif
                partInfoItem = partInfoListItem.join(" ");

                int pathIndex = partInfoItem.indexOf(" ");
                int startPointIndex = partInfoItem.indexOf(" ", pathIndex + 1);
                if (startPointIndex != -1) {
                    bool toLongLongOK = false;
                    QString startPointStr = partInfoItem.left(startPointIndex);
                    startPointStr = startPointStr.right(startPointStr.length() - pathIndex - 1);
                    partitionInfos[i].startPoint = startPointStr.toLongLong(&toLongLongOK);
                    if (!toLongLongOK) {
                        qWarning() << QString("set startPoint failed: %1 to toLongLong failed.").arg(startPointStr);
                    }

                    int endPointIndex = partInfoItem.indexOf(" ", startPointIndex + startPointStr.length() + 1);
                    if (endPointIndex != -1) {
                        QString endPointStr = partInfoItem.left(endPointIndex);
                        endPointStr = endPointStr.right(endPointStr.length() - startPointIndex - 1);
                        partitionInfos[i].endPoint = endPointStr.toLongLong(&toLongLongOK);
                        if (!toLongLongOK) {
                            qWarning() << QString("set endPoint failed: %1 to toLongLong failed.").arg(endPointStr);
                        }
                    }
                }

                // 设置是否是EFI分区
                if (partInfoItem.contains("EFI")) {
                    partitionInfos[i].isEFIPartition = true;
                }

                break;
            }
        }
    }

    return true;
}

void Device::updateLvmInfos(QMap<QString, QList<SPartitionInfo>> devs, QMap<QString, SVGInfo> &vgNames, QMap<QString, SLVMInfo> &lvNames, QMap<SVGInfo, QList<SLVMInfo>> &lvmInfos)
{
    // 更新每个vg下的lvm分区信息，和空闲空间信息
    for (QString vgName : vgNames.keys()) {
        qint64 lvTotalSize = 0;
        vgNames[vgName].id = QUuid::createUuid().toString();
        // 更新vg的pvids信息
        QMap<QString, QList<SPartitionInfo>>::iterator deviceIter;
        for (deviceIter = devs.begin(); deviceIter != devs.end(); ++deviceIter) {
            for (SPartitionInfo partInfo : deviceIter.value()) {
                if (vgNames[vgName].pvIds.indexOf(partInfo.path) != -1) {
                    vgNames[vgName].pvIds.removeOne(partInfo.path);
                    vgNames[vgName].pvIds.append(partInfo.id);
                }
            }
        }

        // 找到vg对应的lvm分区信息
        for (QString lvName : lvNames.keys()) {
            if (lvName.contains(vgName)) {
                lvNames[lvName].id     = QUuid::createUuid().toString();
                lvNames[lvName].label  = lvNames[lvName].label.replace(QString("%1-").arg(vgName), "");
                lvNames[lvName].vgName = vgName;
                lvTotalSize += lvNames[lvName].size;
                lvmInfos[vgNames[vgName]].append(lvNames[lvName]);
            }
        }

        // 获取vg中的pe的size，将其过滤掉不作为空闲空间使用
        qint64 peSize = 1;
        QString err;
        if (!getLVMPESize(vgName, peSize, err)) {
            peSize = 1;
        }

        // 添加vg空闲空间信息, 空间大小以M为单位
        qint64 vgFreeSize = vgNames[vgName].size - lvTotalSize;
        if (vgFreeSize > peSize) {
            SLVMInfo freeSpaceInfo;
            freeSpaceInfo.id         = QUuid::createUuid().toString();
            freeSpaceInfo.size       = vgFreeSize;
            freeSpaceInfo.label      = "";
            freeSpaceInfo.usage      = false;
            freeSpaceInfo.vgName     = vgName;
            freeSpaceInfo.filesystem = "";
            freeSpaceInfo.mountPoint = "";
            lvmInfos[vgNames[vgName]].append(freeSpaceInfo);
        }
    }
}

bool Device::updateDeviceInfoMapByFstab(QMap<QString, QList<SPartitionInfo>> &devicesInfos)
{
    QString fstabFile = "/etc/fstab";
    static QStringList sysDevices = Device::getSysDevicesByFstab(fstabFile);

    for (QString deviceName : devicesInfos.keys()) {
        QString devName = deviceName;
        if (sysDevices.indexOf(devName.replace("/dev/", "")) == -1) {
            devicesInfos.remove(deviceName);
        }
    }

    return true;
}

bool Device::getVGNamesInfo(QMap<QString, SVGInfo> &vgNames, QString &err)
{
    QString vgInfoString = "";
    QString cmd = "vgdisplay | grep \"VG Name\" | awk -F \" \" '{print $3}'";
    if (!Process::spawnCmd("/bin/bash", {"-c", cmd}, vgInfoString)) {
        err = QString("Command: %1 failed. %2").arg(cmd).arg(vgInfoString);
        return false;
    }
#if QT_VERSION >= QT_VERSION_CHECK(5,15,0)
    QStringList vgInfoList = vgInfoString.split("\n", Qt::SkipEmptyParts);
#else
    QStringList vgInfoList = vgInfoString.split("\n", QString::SkipEmptyParts);
#endif

    QString fstabFile = "/etc/fstab";
    static QStringList sysDevices = Device::getSysDevicesByFstab(fstabFile);

    for (QString vgInfoItem : vgInfoList) {
        cmd = QString("pvdisplay -C | grep %1 | awk -F \" \" '{print $1}'").arg(vgInfoItem);
        QString pvInfoString = "";
        if (!Process::spawnCmd("/bin/bash", {"-c", cmd}, pvInfoString)) {
            err = QString("Command: %1 failed. %2").arg(cmd).arg(pvInfoString);
            return false;
        }

#if QT_VERSION >= QT_VERSION_CHECK(5,15,0)
        QStringList pvInfoList = pvInfoString.split("\n", Qt::SkipEmptyParts);
#else
        QStringList pvInfoList = pvInfoString.split("\n", QString::SkipEmptyParts);
#endif
        for (QString pvInfoItem : pvInfoList) {
            for (QString sysDev : sysDevices) {
                if (pvInfoItem.startsWith("/dev/" + sysDev)) {
                    vgNames[vgInfoItem].name = vgInfoItem;
                    vgNames[vgInfoItem].pvIds.append(pvInfoItem);
                    break;
                }
            }
        }
    }
    return true;
}

void Device::updatePartIsMergeVG(QMap<QString, SVGInfo> &vgNames, SPartitionInfo &partitionInfo)
{
    for (QString vgNameItem : vgNames.keys()) {
        if (vgNames[vgNameItem].pvIds.contains(partitionInfo.path)) {
            partitionInfo.isMergeVG = true;
            break;
        }
    }
}

void Device::updateVGNames(const QJsonObject &partInfoObj, QMap<QString, SVGInfo> &vgNames, QMap<QString, SLVMInfo> &lvNames)
{
    if (partInfoObj.value("fstype").toString().compare("LVM2_member")) {
        return;
    }

    for (QString vgName : vgNames.keys()) {
        if (vgNames[vgName].pvIds.indexOf(partInfoObj.value("path").toString()) != -1) {
            vgNames[vgName].size += partInfoObj.value("size").toVariant().toLongLong() / MiB;
        }
    }

    // 添加lvm分区信息到lvnames
    QJsonArray lvmPartArray = partInfoObj.value("children").toArray();
    for (int h = 0; h < lvmPartArray.size(); h++) {
        QJsonObject lvmPartObj = lvmPartArray.at(h).toObject();
        QString lvmPath = lvmPartObj.value("name").toString();
        lvNames[lvmPath].label      = lvmPath;
        lvNames[lvmPath].filesystem = lvmPartObj.value("fstype").toString();
        lvNames[lvmPath].mountPoint = lvmPartObj.value("mountpoint").toString();
        lvNames[lvmPath].size       = lvmPartObj.value("size").toVariant().toLongLong() / MiB;
        lvNames[lvmPath].usage      = true;
    }
}

bool Device::getLVMPESize(const QString &vgName, qint64 &peSize, QString &err)
{
    QString cmd = QString("vgdisplay %1 | grep -iE \'PE[ ]{0,}Size\'").arg(vgName);

    QString out;
    if (!Process::spawnCmd("/bin/bash", {"-c", cmd}, out)) {
        err = QString("Command: %1 failed. %2").arg(cmd, out);
        return false;
    }

    qInfo() << QString("Command: %1 succeed. out: %2").arg(cmd, out);
    QRegExp rx("(\\d+\\.\\d+|(\\d+))");
    if (rx.indexIn(out) != -1) {
        peSize = qint64(rx.cap().toDouble());
    } else {
        peSize = 0;
    }
    qInfo() << "vgName: " << vgName << "  PE Size:" << peSize;
    return true;
}

bool Device::updatePartType(const QString &deviceInfo, QList<SPartitionInfo> &partitionInfos)
{
    QString devicePartInfoString = "";
    QString cmd = QString("parted %1 print | grep -E \"primary|extended|logical\"").arg(deviceInfo);
    if (!Process::spawnCmd("/bin/bash", {"-c", cmd}, devicePartInfoString)) {
        qWarning() << QString("Command: %1 failed. %2").arg(cmd).arg(devicePartInfoString);
        return false;
    }

#if QT_VERSION >= QT_VERSION_CHECK(5,15,0)
    QStringList partInfoStringList = devicePartInfoString.split("\n", Qt::SkipEmptyParts);
#else
    QStringList partInfoStringList = devicePartInfoString.split("\n", QString::SkipEmptyParts);
#endif

    // 将解析的分区type信息设置到SPartitionInfo列表中
    for (int i = 0; i < partitionInfos.size(); i++) {
        for (QString partInfoItem : partInfoStringList) {
#if QT_VERSION >= QT_VERSION_CHECK(5,15,0)
            QStringList partInfoListItem = partInfoItem.split(" ", Qt::SkipEmptyParts);
#else
            QStringList partInfoListItem = partInfoItem.split(" ", QString::SkipEmptyParts);
#endif
            partInfoItem = partInfoListItem.join(" ");

            int pathIndex = partInfoItem.indexOf(" ");
            QString partIndexStr = partInfoItem.left(pathIndex);
            bool toIntOK = false;
            int partIndex = partIndexStr.toInt(&toIntOK);
            if (toIntOK && (partIndex == partitionInfos.at(i).index)) {
                if (partInfoItem.contains("primary")) {
                    partitionInfos[i].partitionType = EPartitionType::PartitionTypePrimary;
                } else if (partInfoItem.contains("extended")) {
                    partitionInfos[i].partitionType = EPartitionType::PartitionTypeExtension;
                    // 计算扩展分区的大小
                    qint64 oneMSectors =  1 * MiB / partitionInfos[i].sectorSize; //CommonFunc::mToSectors(1, partitionInfos[i].sectorSize);
                    qint64 subSectors = partitionInfos.at(i).endPoint - partitionInfos.at(i).startPoint - 1;
                    partitionInfos[i].size = subSectors / oneMSectors;
                } else if (partInfoItem.contains("logical")) {
                    partitionInfos[i].partitionType = EPartitionType::PartitionTypeLogical;
                }
            }
        }
    }

    return true;
}

bool Device::writeDeviceInfo(const QString &filepath, const QMap<QString, QList<SPartitionInfo>> &devicesInfos, QMap<SVGInfo, QList<SLVMInfo>> &lvmInfos)
{
    QJsonArray partitionOperate;
    for (QString deviceKey : devicesInfos.keys()) {
        for (SPartitionInfo partInfo : devicesInfos.value(deviceKey)) {
            QJsonObject partitionItem;
            partitionItem.insert("id", partInfo.id);
            partitionItem.insert("type", "partition");
            partitionItem.insert("operate", "new");
            partitionItem.insert("device", deviceKey);
            if (!partInfo.filesystem.compare("LVM2_member")) {
                partitionItem.insert("filesystem", "lvm2 pv");
            } else if (!partInfo.filesystem.compare("crypto_LUKS")) {
                partitionItem.insert("filesystem", "crypto_luks");
            } else {
                partitionItem.insert("filesystem", partInfo.filesystem);
            }
            partitionItem.insert("mountPoint", partInfo.mountPoint);
            partitionItem.insert("label", partInfo.label);
            partitionItem.insert("startPoint", partInfo.startPoint);
            partitionItem.insert("endPoint", partInfo.endPoint);
            partitionItem.insert("sectorSize", partInfo.sectorSize);
            partitionItem.insert("phySectorSize", partInfo.phySectorSize);
            partitionItem.insert("deviceSize", partInfo.deviceSize);
            partitionItem.insert("isstartpoint", true);
            partitionItem.insert("size", partInfo.size);
            partitionItem.insert("index", -1);
            switch (partInfo.partitionType) {
                case EPartitionType::PartitionTypeLogical: {
                    partitionItem.insert("partType", "logical");
                }
                break;
                case EPartitionType::PartitionTypeExtension: {
                    partitionItem.insert("partType", "extended");
                }
                break;
                case EPartitionType::PartitionTypePrimary: {
                    partitionItem.insert("partType", "primary");
                }
                break;
                default: {
                    partitionItem.insert("partType", "primary");
                }
                break;
            }
            partitionOperate.append(partitionItem);
        }
    }

    // 保存lvm分区信息
    for (SVGInfo vgInfo : lvmInfos.keys()) {
        QJsonObject vgPartItem;
        vgPartItem.insert("id", vgInfo.id);
        vgPartItem.insert("name", vgInfo.name);
        vgPartItem.insert("operate", "new");
        QJsonArray pvs;
        for (QString pvId : vgInfo.pvIds) {
            pvs.append(pvId);
        }
        vgPartItem.insert("pv", pvs);
        vgPartItem.insert("size", vgInfo.size);
        vgPartItem.insert("type", "VG");
        partitionOperate.append(vgPartItem);

        for (SLVMInfo lvInfo : lvmInfos.value(vgInfo)) {
            // 要过滤掉空闲空间
            if (!lvInfo.usage) {
                continue;
            }
            QJsonObject lvPartItem;
            lvPartItem.insert("filesystem", lvInfo.filesystem);
            lvPartItem.insert("id", lvInfo.id);
            lvPartItem.insert("label", lvInfo.label);
            lvPartItem.insert("mountPoint", lvInfo.mountPoint);
            lvPartItem.insert("operate", "new");
            lvPartItem.insert("size", lvInfo.size);
            lvPartItem.insert("type", "LVM");
            lvPartItem.insert("vg", lvInfo.vgName);
            partitionOperate.append(lvPartItem);
        }
    }

    QJsonDocument devicePartInfo(partitionOperate);
    QFile partPolicyInfoJson(filepath);
    if (!partPolicyInfoJson.open(QFile::WriteOnly)) {
        return false;
    }

    partPolicyInfoJson.write(devicePartInfo.toJson());
    partPolicyInfoJson.close();

    return true;
}

int deviceCompare (const DevicePtr left, const DevicePtr right)
{
    if (nullptr == left->getDeviceInfo() || nullptr == right->getDeviceInfo()) {
        return 0;
    }

    if (left->getDeviceInfo()->availableBytes < right->getDeviceInfo()->availableBytes) {
        return 1;
    }

    return -1;
}

bool Device::getMountPointByUUID(const QString &uuid, QString &mountPointPath)
{
    bool retCode = false;
    QStringList args;
    args <<"-c"<< "lsblk --output UUID,MOUNTPOINT | grep " + uuid;
    QString out;
    retCode = Process::spawnCmd("/bin/bash", args, out);
    if (!retCode) {
        qInfo()<<Q_FUNC_INFO<<", spawnCmd failed, args = "<<args<<", out = "<<out;
        return retCode;
    }

#if QT_VERSION >= QT_VERSION_CHECK(5,15,0)
    QStringList outList = out.split("\n", Qt::SkipEmptyParts);
#else
    QStringList outList = out.split("\n", QString::SkipEmptyParts);
#endif
    for (QString line : outList) {
        line = line.trimmed();
        int index = line.indexOf("/");
        // 挂载点可能带空格
        mountPointPath = line.right(line.length() - index);
        retCode = true;
        break;
    }

    return retCode;
}

bool Device::getAllMediaMountPoint(QList<QString> &mountPointList)
{
    bool retCode = false;
    mountPointList.clear();
    QStringList args;
    args <<"-c"<< "lsblk --output MOUNTPOINT | grep /media/";
    QString out;
    retCode = Process::spawnCmd("/bin/bash", args, out);
    if (!retCode) {
        qInfo()<<Q_FUNC_INFO<<", spawnCmd failed, args = "<<args<<", out = "<<out;
        return retCode;
    }

    foreach(QString line, out.split("\n")) {
        QString mountPoint = line.trimmed();
        if (mountPoint.isEmpty()) {
            continue;
        }
        mountPointList.push_back(mountPoint);
//        qInfo() << "line = "<<line;
        retCode = true;
    }

    return retCode;
}

bool Device::isDeviceMountedReadOnly(const QString &path)
{
    QString cmd = QString("mount | grep /dev/ | grep \"(ro,\" | awk '{print $3}'");
    QStringList args;
    args <<"-c"<< cmd;
    QString output;
    QString error;
    if (!Process::spawnCmd("/bin/bash", args, output, error)) {
        return false;
    }

#if QT_VERSION >= QT_VERSION_CHECK(5,15,0)
    QStringList roDeviceList = output.split("\n", Qt::SkipEmptyParts);
#else
    QStringList roDeviceList = output.split("\n", QString::SkipEmptyParts);
#endif

    for (const QString &roDev : roDeviceList) {
        if ((roDev != "/") && path.startsWith(roDev)) {
            return true;
        }
    }

    return false;
}

int deviceInfoCompareMore (const DeviceInfoPtr left, const DeviceInfoPtr right)
{
    return right->availableBytes < left->availableBytes;
}

int deviceInfoTotalSizeCompare(const DeviceInfoPtr &left, const DeviceInfoPtr &right)
{
    return left->sizeBytes < right->sizeBytes;
}

bool Device::findSuitablePartition(quint64 dimTotalBytes, QString &partitionMountPoint, quint64 thresholdBytes,
                                   bool removable)
{
    partitionMountPoint = "";
    QString innerSpacePath = ""; // 内置系统磁盘分区，带空格的路径
    QString innerMediaNoSpacePath = ""; // 内置非系统磁盘，分区路径不带空格
    QString innerMediaSpacePath = ""; // 内置非系统磁盘，分区路径带空格
    QString mediaNoSpacePath = ""; // 外接磁盘，分区路径不带空格
    QString mediaSpacePath = ""; // 外接磁盘，分区路径带空格
    quint64 needBytes = dimTotalBytes + thresholdBytes;
    DeviceInfoList devList = Device::getDeviceByLsblk();
    // 先按照可用空间排序，优先选择可用空间大的
    std::sort(devList.begin(), devList.end(), deviceInfoCompareMore);
    for (auto dev : devList) {
        if (dev->availableBytes < needBytes || !Device::isFsTypeSupported(dev->fsType)) {
            continue;
        }

        if (!removable) { // 默认过滤掉外接设备, hotplug
            if (dev->removable) {
                continue;
            }
        }

        QString devMountPoint = dev->mountPoint;
        if (devMountPoint.startsWith("/media/")) {
            if (!Device::isDeviceMountedReadOnly(devMountPoint)) {
                if (devMountPoint.contains(" ")) {
                    if (dev->removable) {
                        if (mediaSpacePath.isEmpty()) {
                            mediaSpacePath = devMountPoint;
                        }
                    } else {
                        if (innerMediaSpacePath.isEmpty()) {
                            innerMediaSpacePath = devMountPoint;
                        }
                    }
                } else {
                    if (dev->removable) {
                        if (mediaNoSpacePath.isEmpty()) {
                            mediaNoSpacePath = devMountPoint;
                        }
                    } else {
                        if (innerMediaNoSpacePath.isEmpty()) {
                            innerMediaNoSpacePath = devMountPoint;
                        }
                    }
                }
            }
            continue;
        } else {
            if(devMountPoint.contains(" ")) {
                if (innerSpacePath.isEmpty()) {
                    innerSpacePath = devMountPoint;
                    //    partitionMountPoint = innerSpacePath; // 测试带空格的分区路径
                    //    return true;
                }
                continue;
            } else if ("/" == devMountPoint || "/boot" == devMountPoint) {
                continue;
            }
        }

        // 找到系统磁盘分区，且分区挂载点路径不带空格，最佳选择
        partitionMountPoint = devMountPoint; // best choose, inner disk partition, has no empty space
        return true;
    }

    /* 匹配策略：
     * 1. 内置系统盘分区挂载点路径不带空格的，第1优先级
     * 2. 内置磁盘优先级 > 外接磁盘，并且内置磁盘分区挂载点路径不带空格的 > 带空格的
     * 3. 内置磁盘分区挂载点路径都带空格的场景下：系统盘 > 内置挂载的磁盘
     * 4. 外接磁盘：分区挂载点路径不带空格的 > 带空格的
     * 5. 各种场景只保存分区可用空间最大的挂载点路径
    */
    if (!innerMediaNoSpacePath.isEmpty()) {
        partitionMountPoint = innerMediaNoSpacePath;
    } else if (!innerSpacePath.isEmpty()) {
        partitionMountPoint = innerSpacePath;
    } else if (!innerMediaSpacePath.isEmpty()) {
        partitionMountPoint = innerMediaSpacePath;
    } else if (!mediaNoSpacePath.isEmpty()) {
        partitionMountPoint = mediaNoSpacePath;
    } else if (!mediaSpacePath.isEmpty()) {
        partitionMountPoint = mediaSpacePath;
    }

    return !partitionMountPoint.isEmpty();
}

bool Device::getVGNames(QStringList &vgNameList)
{
    vgNameList.clear();
    QString error;
    QString out;
    QString cmd = "vgdisplay | grep \"VG Name\" | awk -F \" \" '{print $3}'";
    if (!Process::spawnCmd("/bin/bash", {"-c", cmd}, out, error)) {
        qCritical()<<"Device::getVGNames failed, cmd = "<<cmd<<", error = "<<error;
        return false;
    }

    if (!error.isEmpty()) {
        qCritical()<<"Device::getVGNames error = "<<error;
        return false;
    }

    out = out.trimmed();
    if (out.isEmpty()) {
        return true;
    }

    /*
     * root@zxm-PC:~# vgdisplay | grep "VG Name"
           VG Name               group6
           VG Name               group4
     */

#if QT_VERSION >= QT_VERSION_CHECK(5,15,0)
    vgNameList = out.split("\n", Qt::SkipEmptyParts);
#else
    vgNameList = out.split("\n", QString::SkipEmptyParts);
#endif

    return true;
}

bool Device::getPvDisplay(QMap<QString, QString> &pvVgMaps)
{
    pvVgMaps.clear();
    QString error;
    QString out;
    QString cmd = "pvdisplay -C";
    if (!Process::spawnCmd("/bin/bash", {"-c", cmd}, out, error)) {
        qCritical()<<"Device::getPvDisplay failed, cmd = "<<cmd<<", error = "<<error;
        return false;
    }

    if (!error.isEmpty()) {
        qCritical()<<"Device::getPvDisplay error = "<<error;
        return false;
    }

    out = out.trimmed();
    if (out.isEmpty()) {
        return true;
    }

#if QT_VERSION >= QT_VERSION_CHECK(5,15,0)
    QStringList pvList = out.split("\n", Qt::SkipEmptyParts);
#else
    QStringList pvList = out.split("\n", QString::SkipEmptyParts);
#endif

    /*
     *     PV         VG      Fmt   Attr   PSize   PFree
        /dev/sda3  group1    lvm2   a--   68.20g    0
        /dev/sdb1  group2    lvm2   a--  <88.00g    0
        /dev/sdd1            lvm2   ---  <78.00g  <78.00g
     */
    for (QString line : pvList) {
        line = line.trimmed();
        if (!line.startsWith("/dev/")) {
            continue;
        }

        int index = line.indexOf(" ");
        if (-1 == index) {
            continue;
        }
        QString pv = line.left(index);
        line = line.right(line.length() - index).trimmed();

        index = line.indexOf(" ");
        if (-1 == index) {
            continue;
        }
        QString vg = line.left(index);
        // check vg is invalid
        if (vg.startsWith("lvm")) {
            continue;
        }
        pvVgMaps.insert(pv, vg);
    }

    return true;
}

bool Device::getLvPathsByVgName(const QString &vgName, QStringList &lvPathList)
{
    lvPathList.clear();
    QString error;
    QString out;
    QString lvPath = "LV Path";
    lvPath = QString("\"%1\"").arg(lvPath);
    QString cmd = QString("lvdisplay /dev/%1 | grep %2").arg(vgName).arg(lvPath);
    if (!Process::spawnCmd("/bin/bash", {"-c", cmd}, out, error)) {
        qCritical()<<"Device::getLvPathByVgName failed, cmd = "<<cmd<<", error = "<<error;
        return false;
    }

    if (!error.isEmpty()) {
        qCritical()<<"Device::getLvPathByVgName error = "<<error;
        return false;
    }

    out = out.trimmed();
    if (out.isEmpty()) {
        return false;
    }

#if QT_VERSION >= QT_VERSION_CHECK(5,15,0)
    QStringList lvList = out.split("\n", Qt::SkipEmptyParts);
#else
    QStringList lvList = out.split("\n", QString::SkipEmptyParts);
#endif

    /*
     * root@zxm-PC:~# lvdisplay /dev/group6 |grep "LV Path"
         LV Path                /dev/group6/sdd_atm1
         LV Path                /dev/group6/sdd_bb1
         LV Path                /dev/group6/sdd_bu1
     */
    for (QString line : lvList) {
        int index = line.indexOf("/");
        if (-1 != index) {
            line = line.right(line.length() - index).trimmed();
            lvPathList.append(line);
        }
    }

    return true;
}

bool Device::lvRemove(const QString &lvPath)
{
    QString error;
    QString out;
    QString cmd = QString("lvremove %1 --yes").arg(lvPath);
    if (!Process::spawnCmd("/bin/bash", {"-c", cmd}, out, error)) {
        qCritical()<<"Device::lvRemove failed, cmd = "<<cmd<<", error = "<<error;
        return false;
    }

    if (!error.isEmpty()) {
        qCritical()<<"Device::lvRemove error = "<<error;
        return false;
    }

    return true;
}

bool Device::deleteLVM(const QStringList &devices, const QMap<QString, QString> &pvVgMaps)
{
    if (devices.isEmpty()) {
        return true;
    }

    for (const QString &device : devices) {
        for (const QString &pv : pvVgMaps.keys()) {
            if (!pv.startsWith(device)) {
                continue;
            }

            QStringList lvPathList;
            if (!Device::getLvPathsByVgName(pvVgMaps[pv], lvPathList)) {
                return false;
            }

            for (const QString &lvPath : lvPathList) {
                if (!Device::lvRemove(lvPath)) {
                    return false;
                }
            }
        }
    }

    return true;
}

bool Device::vgRemove(const QString &vgName)
{
    QString error;
    QString out;
    QString cmd = QString("vgremove %1 --yes").arg(vgName);
    if (!Process::spawnCmd("/bin/bash", {"-c", cmd}, out, error)) {
        qCritical()<<"Device::vgRemove failed, cmd = "<<cmd<<", error = "<<error;
        return false;
    }

    if (!error.isEmpty()) {
        qCritical()<<"Device::vgRemove error = "<<error;
        return false;
    }

    return true;
}

bool Device::deleteVG(const QStringList &devices, const QMap<QString, QString> &pvVgMaps)
{
    if (devices.isEmpty() || pvVgMaps.isEmpty()) {
        return true;
    }

    for (const QString &device : devices) {
        for (const QString &pv : pvVgMaps.keys()) {
            if (!pv.startsWith(device)) {
                continue;
            }

            if (!Device::vgRemove(pvVgMaps[pv])) {
                return false;
            }
        }
    }

    return true;
}

bool Device::renameVG(const QStringList &sysVgNameList, QMap<QString, QString> &vgMap)
{
    bool needRename = false;
    int num = 1;
    QString vgName = "group";
    for (auto vgKey : vgMap.keys()) {
        if (sysVgNameList.contains(vgKey)) {
            do {
                vgName = QString("group%2").arg(num++);
            } while (sysVgNameList.contains(vgName));
            vgMap[vgKey] = vgName;
            needRename = true;
        }
    }

    return needRename;
}

void Device::getAllTypeDevice(DeviceInfoList &diskList, DeviceInfoList &cryptDevList,
                      DeviceInfoList &noCryptDevList, DeviceInfoList &partitionList, DeviceInfoList &lvmList)
{
    diskList.clear();
    cryptDevList.clear();
    noCryptDevList.clear();
    partitionList.clear();
    lvmList.clear();

    DeviceInfoList usedDevices = Device::getAllDeviceByLsblk();
    for (auto &dev : usedDevices) {
        if (dev->type == "disk") {
            diskList.append(dev);
        } else if ((dev->type == "part" && dev->fsType == "luks") || (dev->type == "crypt" && dev->fsType == "lvm2")) {
            cryptDevList.append(dev); // 全盘加密场景
        } else if (dev->type == "part" && dev->fsType == "lvm2") {
            noCryptDevList.append(dev); // 普通LVM
        } else if (dev->type == "part") {
            partitionList.append(dev);
        } else if (dev->type == "lvm") {
            lvmList.append(dev);
        }
    }
}

bool Device::isMultiFullDiskEncrypt(const QString &fstabFile)
{
    static FSTabInfoList fstabList = FSTab::getFSTabFromFile(fstabFile);
    QSet<QString> mountBindPointSets;
    for (auto &info : fstabList) {
        if (info->isEmptyLine || info->isComment) {
            continue;
        }

        if (info->options.contains("bind")) {
            QString mountBindPoint = info->device.left(info->device.lastIndexOf("/"));
            mountBindPointSets.insert(mountBindPoint);
        }
    }

    if (1 != mountBindPointSets.size()) {
        return false; // not full disk install
    }

    static QStringList sysDeviceUuidList = FSTab::getUuidListFromFstab(fstabList);
    DeviceInfoList diskList;
    DeviceInfoList cryptDevList;
    DeviceInfoList noCryptDevList;
    DeviceInfoList partitionList;
    DeviceInfoList lvmList;
    Device::getAllTypeDevice(diskList, cryptDevList, noCryptDevList, partitionList, lvmList);

    QStringList sysDeviceList;
    for (auto &diskDev : diskList) {
        for (auto &cryptDev : cryptDevList) {
            if (!(cryptDev->type == "part" && cryptDev->fsType == "luks" && cryptDev->pkname == diskDev->name)) {
                continue;
            }

            for (auto &lvm2Dev : cryptDevList) {
                if (!(lvm2Dev->type == "crypt" && lvm2Dev->fsType == "lvm2" && lvm2Dev->pkname == cryptDev->kname)) {
                    continue;
                }

                for (auto &lvmMember : lvmList) {
                    if (lvmMember->pkname == lvm2Dev->kname) {
                        if (sysDeviceUuidList.contains(lvmMember->uuid) && !sysDeviceList.contains(diskDev->name)) {
                            sysDeviceList.append(diskDev->name);
                            break;
                        }
                    }
                }
            }
        }
    }

    sysDeviceList.removeDuplicates();

    return sysDeviceList.size() > 1;
}

QJsonObject GhostDiskInfo::marshal()
{
    QJsonObject jsonObject;
    jsonObject.insert("deviceName", deviceName);
    jsonObject.insert("vendor", vendor);
    jsonObject.insert("mode", mode);
    jsonObject.insert("totalSizeBytes", QString("%1").arg(totalSizeBytes));
    jsonObject.insert("allocTotalSizeBytes", QString("%1").arg(allocTotalSizeBytes));
    jsonObject.insert("usedSizeBytes", QString("%1").arg(usedSizeBytes));
    jsonObject.insert("lastPartitionTotalSizeBytes", QString("%1").arg(lastPartitionTotalSizeBytes));
    jsonObject.insert("lastPartitionUsedSizeBytes", QString("%1").arg(lastPartitionUsedSizeBytes));
    jsonObject.insert("mediaType", mediaType);
    jsonObject.insert("isEncrypt", isEncrypt);
    jsonObject.insert("isLvm", isLvm);
    jsonObject.insert("isVirtualDisk", isVirtualDisk);
    jsonObject.insert("isRemovable", isRemovable);

    return jsonObject;
}

void GhostDiskInfo::unmarshal(const QJsonObject &jsonObject)
{
    bool ok = false;
    deviceName = jsonObject.value("deviceName").toString();
    vendor = jsonObject.value("vendor").toString();
    mode = jsonObject.value("mode").toString();
    totalSizeBytes = jsonObject.value("totalSizeBytes").toString().toULongLong(&ok);
    if (!ok) {
        totalSizeBytes = 0;
    }

    allocTotalSizeBytes = jsonObject.value("allocTotalSizeBytes").toString().toULongLong(&ok);
    if (!ok) {
        allocTotalSizeBytes = 0;
    }

    usedSizeBytes = jsonObject.value("usedSizeBytes").toString().toULongLong(&ok);
    if (!ok) {
        usedSizeBytes = 0;
    }

    lastPartitionTotalSizeBytes = jsonObject.value("lastPartitionTotalSizeBytes").toString().toULongLong(&ok);
    if (!ok) {
        lastPartitionTotalSizeBytes = 0;
    }

    lastPartitionUsedSizeBytes = jsonObject.value("lastPartitionUsedSizeBytes").toString().toULongLong(&ok);
    if (!ok) {
        lastPartitionUsedSizeBytes = 0;
    }

    mediaType = jsonObject.value("mediaType").toInt(-1);
    isEncrypt = jsonObject.value("isEncrypt").toBool();
    isLvm = jsonObject.value("isLvm").toBool();
    isVirtualDisk = jsonObject.value("isVirtualDisk").toBool();
    isRemovable = jsonObject.value("isRemovable").toBool();
}

bool GhostDiskInfo::operator<(const GhostDiskInfo &diskInfo)
{
    return this->totalSizeBytes < diskInfo.totalSizeBytes;
}

void Device::fillGhostDiskInfo(const DeviceInfoPtr &devInfoPtr, GhostDiskInfo &diskInfo)
{
    diskInfo.totalSizeBytes = devInfoPtr->sizeBytes;
    diskInfo.deviceName = devInfoPtr->deviceName;
    diskInfo.isRemovable = devInfoPtr->removable;
    if (devInfoPtr->removable) {
        if (devInfoPtr->vendor.startsWith("VBOX")) {
            diskInfo.isVirtualDisk = true;
        }
    } else {
        if (devInfoPtr->mode.startsWith("VBOX")) {
            diskInfo.isVirtualDisk = true;
        }
    }
    diskInfo.mediaType = devInfoPtr->rota;
    diskInfo.vendor = devInfoPtr->vendor;
    diskInfo.mode = devInfoPtr->mode;
    diskInfo.lastPartitionTotalSizeBytes = devInfoPtr->sizeBytes; // need refresh
    diskInfo.lastPartitionUsedSizeBytes = devInfoPtr->usedBytes; // need refresh
    diskInfo.allocTotalSizeBytes = 0; // need refresh
    diskInfo.usedSizeBytes = 0; // need refresh
    diskInfo.isEncrypt = false; // need refresh
    diskInfo.isLvm = false; // need refresh
}

void Device::getGhostDiskInfo(const QString &fstabFile, QList<GhostDiskInfo> &diskInfoList)
{
    diskInfoList.clear();
    static FSTabInfoList fstabList = FSTab::getFSTabFromFile(fstabFile);
    static QStringList sysDeviceUuidList = FSTab::getUuidListFromFstab(fstabList);

    DeviceInfoList diskList;
    DeviceInfoList cryptDevList;
    DeviceInfoList noCryptDevList;
    DeviceInfoList partitionList;
    DeviceInfoList lvmList;
    Device::getAllTypeDevice(diskList, cryptDevList, noCryptDevList, partitionList, lvmList);
    DeviceInfoList orgDevList = Device::getOrganizedDiskList(diskList, cryptDevList, noCryptDevList, partitionList, lvmList);

    for (auto &diskDev : diskList) {
        GhostDiskInfo diskInfo;
        quint64 allocTotalSizeBytes = 0;
        quint64 usedSizeBytes = diskDev->usedBytes;
        quint64 lastPartitionTotalSizeBytes = 0;
        quint64 lastPartitionUsedSizeBytes = 0;
        bool isEncrypt = false;
        bool isLvm = false;
        bool isSysDisk = false;
        for (auto &cryptDev : cryptDevList) {
            if (!(cryptDev->type == "part" && cryptDev->fsType == "luks" && cryptDev->pkname == diskDev->name)) {
                continue;
            }

            for (auto &lvm2Dev : cryptDevList) {
                if (!(lvm2Dev->type == "crypt" && lvm2Dev->fsType == "lvm2" && lvm2Dev->pkname == cryptDev->kname)) {
                    continue;
                }

                for (auto &lvmMember : lvmList) {
                    if (lvmMember->pkname == lvm2Dev->kname) {
                        if (sysDeviceUuidList.contains(lvmMember->uuid)) {
                            isSysDisk = true;
                            allocTotalSizeBytes += lvmMember->sizeBytes;
                            usedSizeBytes += lvmMember->usedBytes;
                            isEncrypt = true;
                            isLvm = true;
                        }
                    }
                }
            }
        }

        for (auto &noCryptDev : noCryptDevList) {
            if (noCryptDev->pkname != diskDev->name) {
                continue;
            }

            for (auto &lvmMember : lvmList) {
                if (lvmMember->pkname == noCryptDev->kname) {
                    if (sysDeviceUuidList.contains(lvmMember->uuid)) {
                        isSysDisk = true;
                        allocTotalSizeBytes += lvmMember->sizeBytes;
                        isLvm = true;
                    }
                }
            }
        }

        for (auto &partitionDev : partitionList) {
            if (partitionDev->pkname == diskDev->name) {
                if (sysDeviceUuidList.contains(partitionDev->uuid)) {
                    isSysDisk = true;
                    allocTotalSizeBytes += partitionDev->sizeBytes;
                }
            }
        }

        if (isSysDisk) {
            if (!diskDev->children.isEmpty()) {
                DeviceInfoPtr &lastPartition = diskDev->children.last();
                while (nullptr != lastPartition) {
                    if (lastPartition->children.isEmpty()) {
                        break;
                    }
                    lastPartition = lastPartition->children.last();
                }
                lastPartitionTotalSizeBytes = lastPartition->sizeBytes;
                lastPartitionUsedSizeBytes = lastPartition->usedBytes;
            }

            Device::fillGhostDiskInfo(diskDev, diskInfo);
            diskInfo.allocTotalSizeBytes = allocTotalSizeBytes;
            diskInfo.usedSizeBytes = usedSizeBytes;
            diskInfo.lastPartitionTotalSizeBytes = lastPartitionTotalSizeBytes;
            diskInfo.lastPartitionUsedSizeBytes = lastPartitionUsedSizeBytes;
            diskInfo.isEncrypt = isEncrypt;
            diskInfo.isLvm = isLvm;
            diskInfoList.append(diskInfo);
        }
    }
}
