#!/usr/bin/env python

import http.client
http.client._MAXHEADERS = 1000

import argparse
import copy
from datetime import datetime
import json
import logging
import sys
import timestring

from blackduck.HubRestApi import HubInstance, object_id

#
# Example usage:
#	To get all the vulnerabilities (first time) and save the last run date/time,
#		python examples/get_bom_component_vulnerability_info.py struts2-showcase 2.6-SNAPSHOT -s > vulnerabilities.json
#
#	Having saved the last run date/time, use it to view any newly published vulnerabilities that have come out since the last run,
#		python examples/get_bom_component_vulnerability_info.py struts2-showcase 2.6-SNAPSHOT -n `cat .last_run` > vulnerabilities_since.json
#
#	Having saved the last run date/time, use it to view any newly published vulnerabilities that have come out since the last run
#	and update the last run date/time,
#		python examples/get_bom_component_vulnerability_info.py struts2-showcase 2.6-SNAPSHOT -s -n `cat .last_run` > vulnerabilities_since.json
#
#	Use --newer_than (aka -n) to specify your own date (or date/time),
#		python examples/get_bom_component_vulnerability_info.py struts2-showcase 2.6-SNAPSHOT -n "2017" > vulnerabilities_since_2017.json
#		python examples/get_bom_component_vulnerability_info.py struts2-showcase 2.6-SNAPSHOT -n "July 1 2018" > vulnerabilities_since_July_1_2018.json
#		python examples/get_bom_component_vulnerability_info.py struts2-showcase 2.6-SNAPSHOT -n "July 1 2018 5:30 pm" > vulnerabilities_since_July_1_2018_1730.json


parser = argparse.ArgumentParser("Retreive BOM component vulnerability information for the given project and version")
parser.add_argument("project_name")
parser.add_argument("version")
parser.add_argument("-u", "--include_updated_vulns", 
    action='store_true', 
    help="If set, will also retrieve vulnerabilities whose update date/time is later than the newer_than date/time")
parser.add_argument("-n", "--newer_than", 
	default=None, 
	type=str,
	help="Set this option to see all vulnerabilities published since the given date/time.")
parser.add_argument("-s", "--save_dt", 
	action='store_true', 
	help="If set, the date/time will be saved to a file named '.last_run' in the current directory which can be used later with the -n option to see vulnerabilities published since the last run.")
parser.add_argument("-l", "--limit",
    default=9999,
    help="Set limit on number of vulnerabilitties to retrieve (default 9999)")
parser.add_argument("-nd", "--nodetails",
    action='store_true',
    help="If set, disables retrieving details for each vulnerability to reduce execution time")
args = parser.parse_args()


if args.newer_than:
	newer_than = timestring.Date(args.newer_than).date
else:
	newer_than = None

if args.save_dt:
	with open(".last_run", "w") as f:
		f.write(datetime.now().isoformat())

logging.basicConfig(format='%(asctime)s:%(levelname)s:%(message)s', stream=sys.stderr, level=logging.DEBUG)
logging.getLogger("requests").setLevel(logging.WARNING)
logging.getLogger("urllib3").setLevel(logging.WARNING)

hub = HubInstance()

project = hub.get_project_by_name(args.project_name)

version = hub.get_version_by_name(project, args.version)
version_id = object_id(version)

vulnerablity_limit = "?limit={}".format(args.limit)

vulnerable_components_url = hub.get_link(version, "vulnerable-components") + vulnerablity_limit
custom_headers = {'Accept':'application/vnd.blackducksoftware.bill-of-materials-6+json'}
response = hub.execute_get(vulnerable_components_url, custom_headers=custom_headers)
vulnerable_bom_components = response.json().get('items', [])

bdsa_records = set()
cve_records = set()


if args.nodetails==False:
    for i, vuln in enumerate(vulnerable_bom_components):
        source = vuln['vulnerabilityWithRemediation']['source']
        vuln_name = vuln['vulnerabilityWithRemediation']['vulnerabilityName']

        # Retrieve additional details about the vulnerability
        #

        update_guidance_url = vuln['componentVersion'] + "/upgrade-guidance"
        update_guidance_results = hub.execute_get(update_guidance_url).json()
        vuln['update_guidance'] = update_guidance_results

        logging.debug("Retrieving additional details regarding vuln {}, i={}".format(vuln_name, i))
        vuln_url = hub.get_apibase() + "/vulnerabilities/{}".format(vuln_name)
        vuln_details_response = hub.execute_get(vuln_url, custom_headers={'Accept': 'application/json'})
        vuln_details = vuln_details_response.json()

        vuln['additional_vuln_info'] = vuln_details

        if source == 'BDSA':
            bdsa_records.add(vuln_name)

            # get related vulnerability info, i.e. CVE
            # note: not all BDSA records will have a corresponding CVE record
            cve_url = hub.get_link(vuln_details, "related-vulnerability")
            if cve_url:
                cve_details_response = hub.execute_get(cve_url, custom_headers={'Accept': 'application/json'})
                cve_details = cve_details_response.json()
                vuln['related_vulnerability'] = cve_details
                cve_records.add(cve_details['name'])
        elif source == "NVD":
            cve_records.add(vuln_name)
        else:
            logging.warning(f"source {source} was not recognized")

if vulnerable_bom_components:
    vulnerable_bom_components = sorted(
        vulnerable_bom_components, 
        key = lambda k: k['vulnerabilityWithRemediation']['vulnerabilityPublishedDate'])
    if newer_than:
        if args.include_updated_vulns:
            vulnerable_bom_components = [v for v in vulnerable_bom_components 
                if timestring.Date(v['vulnerabilityWithRemediation']['vulnerabilityPublishedDate']) > newer_than 
                or timestring.Date(v['vulnerabilityWithRemediation']['vulnerabilityUpdatedDate']) > newer_than ]
        else:
        	vulnerable_bom_components = [v for v in vulnerable_bom_components 
    			if timestring.Date(v['vulnerabilityWithRemediation']['vulnerabilityPublishedDate']) > newer_than ]
else:
	logging.debug("Did not find any vulnerable BOM components in project {}, version {}".format(args.project_name, args.version))

# Combine counts and vulnerabilities (aka vulnerable bom components)
#
num_bdsa_records = len(list(filter(lambda v: v['vulnerabilityWithRemediation']['source'] == 'BDSA', 
    vulnerable_bom_components)))
num_nvd_records = len(list(filter(lambda v: v['vulnerabilityWithRemediation']['source'] == 'NVD', 
    vulnerable_bom_components)))

remediation_counts = {}
remediation_statuses = [v['vulnerabilityWithRemediation']['remediationStatus'] for v in vulnerable_bom_components]
for status in remediation_statuses:
    remediation_counts[status] = remediation_counts.get(status, 0) + 1

counts = {
    'BDSA': num_bdsa_records,
    'NVD': num_nvd_records,
    'all_vulns': num_bdsa_records + num_nvd_records,
    'by_remediation_status': remediation_counts
}

if args.nodetails==False:
    everything = {
        'counts': counts,
        'vulnerabilities': vulnerable_bom_components,
        'bdsa_records': list(bdsa_records),
        'cve_records': list(cve_records),
    }
else:
    everything = {
        'counts': counts,
        'vulnerabilities': vulnerable_bom_components,
    }

print(json.dumps(everything))

