// Copyright (c) 2017,2020 Tigera, Inc. All rights reserved.

// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package clientv3_test

import (
	"context"
	"time"

	. "github.com/onsi/ginkgo"
	. "github.com/onsi/ginkgo/extensions/table"
	. "github.com/onsi/gomega"
	apiv3 "github.com/projectcalico/api/pkg/apis/projectcalico/v3"
	"github.com/projectcalico/api/pkg/lib/numorstring"
	k8sv1 "k8s.io/api/core/v1"
	metav1 "k8s.io/apimachinery/pkg/apis/meta/v1"

	"github.com/projectcalico/calico/libcalico-go/lib/apiconfig"
	"github.com/projectcalico/calico/libcalico-go/lib/backend"
	"github.com/projectcalico/calico/libcalico-go/lib/clientv3"
	"github.com/projectcalico/calico/libcalico-go/lib/options"
	"github.com/projectcalico/calico/libcalico-go/lib/testutils"
	"github.com/projectcalico/calico/libcalico-go/lib/watch"
)

var _ = testutils.E2eDatastoreDescribe("BGPPeer tests", testutils.DatastoreAll, func(config apiconfig.CalicoAPIConfig) {
	ctx := context.Background()
	name1 := "bgppeer-1"
	name2 := "bgppeer-2"
	name3 := "bgppeer-3"
	spec1 := apiv3.BGPPeerSpec{
		Node:     "node1",
		PeerIP:   "10.0.0.1",
		ASNumber: numorstring.ASNumber(6512),
	}
	spec2 := apiv3.BGPPeerSpec{
		Node:     "node2",
		PeerIP:   "20.0.0.1",
		ASNumber: numorstring.ASNumber(6511),
		Password: &apiv3.BGPPassword{
			SecretKeyRef: &k8sv1.SecretKeySelector{
				LocalObjectReference: k8sv1.LocalObjectReference{
					Name: "bgp-passwords",
				},
				Key: "p2",
			},
		},
	}
	spec3 := apiv3.BGPPeerSpec{
		NodeSelector: "has(routeReflectorClusterID)",
		PeerSelector: "has(routeReflectorClusterID)",
	}
	nameWithFilter1 := "bgppeer-with-filter-1"
	specWithFilter1 := apiv3.BGPPeerSpec{
		Node:     "node1",
		PeerIP:   "10.0.0.1",
		ASNumber: numorstring.ASNumber(6512),
		Filters:  []string{"bgp-filter-1"},
	}
	filterName1 := "bgp-filter-1"
	filterSpec1 := apiv3.BGPFilterSpec{
		ExportV4: []apiv3.BGPFilterRuleV4{
			{
				CIDR:          "10.10.10.0/24",
				MatchOperator: apiv3.In,
				Action:        apiv3.Accept,
			},
		},
	}
	filterName2 := "bgp-filter-2"
	filterSpec2 := apiv3.BGPFilterSpec{
		ImportV4: []apiv3.BGPFilterRuleV4{
			{
				CIDR:          "11.11.11.0/24",
				MatchOperator: apiv3.NotIn,
				Action:        apiv3.Reject,
			},
		},
	}

	DescribeTable("BGPPeer e2e CRUD tests",
		func(name1, name2 string, spec1, spec2 apiv3.BGPPeerSpec) {
			c, err := clientv3.New(config)
			Expect(err).NotTo(HaveOccurred())

			be, err := backend.NewClient(config)
			Expect(err).NotTo(HaveOccurred())
			be.Clean()

			By("Updating the BGPPeer before it is created")
			_, outError := c.BGPPeers().Update(ctx, &apiv3.BGPPeer{
				ObjectMeta: metav1.ObjectMeta{Name: name1, ResourceVersion: "1234", CreationTimestamp: metav1.Now(), UID: uid},
				Spec:       spec1,
			}, options.SetOptions{})
			Expect(outError).To(HaveOccurred())
			Expect(outError.Error()).To(ContainSubstring("resource does not exist: BGPPeer(" + name1 + ") with error:"))

			By("Attempting to create a new BGPPeer with name1/spec1 and a non-empty ResourceVersion")
			_, outError = c.BGPPeers().Create(ctx, &apiv3.BGPPeer{
				ObjectMeta: metav1.ObjectMeta{Name: name1, ResourceVersion: "12345"},
				Spec:       spec1,
			}, options.SetOptions{})
			Expect(outError).To(HaveOccurred())
			Expect(outError.Error()).To(Equal("error with field Metadata.ResourceVersion = '12345' (field must not be set for a Create request)"))

			By("Creating a new BGPPeer with name1/spec1")
			res1, outError := c.BGPPeers().Create(ctx, &apiv3.BGPPeer{
				ObjectMeta: metav1.ObjectMeta{Name: name1},
				Spec:       spec1,
			}, options.SetOptions{})
			Expect(outError).NotTo(HaveOccurred())
			Expect(res1).To(MatchResource(apiv3.KindBGPPeer, testutils.ExpectNoNamespace, name1, spec1))

			// Track the version of the original data for name1.
			rv1_1 := res1.ResourceVersion

			By("Attempting to create the same BGPPeer with name1 but with spec2")
			_, outError = c.BGPPeers().Create(ctx, &apiv3.BGPPeer{
				ObjectMeta: metav1.ObjectMeta{Name: name1},
				Spec:       spec2,
			}, options.SetOptions{})
			Expect(outError).To(HaveOccurred())
			Expect(outError.Error()).To(Equal("resource already exists: BGPPeer(" + name1 + ")"))

			By("Getting BGPPeer (name1) and comparing the output against spec1")
			res, outError := c.BGPPeers().Get(ctx, name1, options.GetOptions{})
			Expect(outError).NotTo(HaveOccurred())
			Expect(res).To(MatchResource(apiv3.KindBGPPeer, testutils.ExpectNoNamespace, name1, spec1))
			Expect(res.ResourceVersion).To(Equal(res1.ResourceVersion))

			By("Getting BGPPeer (name2) before it is created")
			_, outError = c.BGPPeers().Get(ctx, name2, options.GetOptions{})
			Expect(outError).To(HaveOccurred())
			Expect(outError.Error()).To(ContainSubstring("resource does not exist: BGPPeer(" + name2 + ") with error:"))

			By("Listing all the BGPPeers, expecting a single result with name1/spec1")
			outList, outError := c.BGPPeers().List(ctx, options.ListOptions{})
			Expect(outError).NotTo(HaveOccurred())
			Expect(outList.Items).To(ConsistOf(
				testutils.Resource(apiv3.KindBGPPeer, testutils.ExpectNoNamespace, name1, spec1),
			))

			By("Creating a new BGPPeer with name2/spec2")
			res2, outError := c.BGPPeers().Create(ctx, &apiv3.BGPPeer{
				ObjectMeta: metav1.ObjectMeta{Name: name2},
				Spec:       spec2,
			}, options.SetOptions{})
			Expect(outError).NotTo(HaveOccurred())
			Expect(res2).To(MatchResource(apiv3.KindBGPPeer, testutils.ExpectNoNamespace, name2, spec2))

			By("Getting BGPPeer (name2) and comparing the output against spec2")
			res, outError = c.BGPPeers().Get(ctx, name2, options.GetOptions{})
			Expect(outError).NotTo(HaveOccurred())
			Expect(res2).To(MatchResource(apiv3.KindBGPPeer, testutils.ExpectNoNamespace, name2, spec2))
			Expect(res.ResourceVersion).To(Equal(res2.ResourceVersion))

			By("Listing all the BGPPeers, expecting two results with name1/spec1 and name2/spec2")
			outList, outError = c.BGPPeers().List(ctx, options.ListOptions{})
			Expect(outError).NotTo(HaveOccurred())
			Expect(outList.Items).To(ConsistOf(
				testutils.Resource(apiv3.KindBGPPeer, testutils.ExpectNoNamespace, name1, spec1),
				testutils.Resource(apiv3.KindBGPPeer, testutils.ExpectNoNamespace, name2, spec2),
			))

			By("Updating BGPPeer name1 with spec2")
			res1.Spec = spec2
			res1, outError = c.BGPPeers().Update(ctx, res1, options.SetOptions{})
			Expect(outError).NotTo(HaveOccurred())
			Expect(res1).To(MatchResource(apiv3.KindBGPPeer, testutils.ExpectNoNamespace, name1, spec2))

			By("Attempting to update the BGPPeer without a Creation Timestamp")
			res, outError = c.BGPPeers().Update(ctx, &apiv3.BGPPeer{
				ObjectMeta: metav1.ObjectMeta{Name: name1, ResourceVersion: "1234", UID: uid},
				Spec:       spec1,
			}, options.SetOptions{})
			Expect(outError).To(HaveOccurred())
			Expect(res).To(BeNil())
			Expect(outError.Error()).To(Equal("error with field Metadata.CreationTimestamp = '0001-01-01 00:00:00 +0000 UTC' (field must be set for an Update request)"))

			By("Attempting to update the BGPPeer without a UID")
			res, outError = c.BGPPeers().Update(ctx, &apiv3.BGPPeer{
				ObjectMeta: metav1.ObjectMeta{Name: name1, ResourceVersion: "1234", CreationTimestamp: metav1.Now()},
				Spec:       spec1,
			}, options.SetOptions{})
			Expect(outError).To(HaveOccurred())
			Expect(res).To(BeNil())
			Expect(outError.Error()).To(Equal("error with field Metadata.UID = '' (field must be set for an Update request)"))

			// Track the version of the updated name1 data.
			rv1_2 := res1.ResourceVersion

			By("Updating BGPPeer name1 without specifying a resource version")
			res1.Spec = spec1
			res1.ObjectMeta.ResourceVersion = ""
			_, outError = c.BGPPeers().Update(ctx, res1, options.SetOptions{})
			Expect(outError).To(HaveOccurred())
			Expect(outError.Error()).To(Equal("error with field Metadata.ResourceVersion = '' (field must be set for an Update request)"))

			By("Updating BGPPeer name1 using the previous resource version")
			res1.Spec = spec1
			res1.ResourceVersion = rv1_1
			_, outError = c.BGPPeers().Update(ctx, res1, options.SetOptions{})
			Expect(outError).To(HaveOccurred())
			Expect(outError.Error()).To(Equal("update conflict: BGPPeer(" + name1 + ")"))

			if config.Spec.DatastoreType != apiconfig.Kubernetes {
				By("Getting BGPPeer (name1) with the original resource version and comparing the output against spec1")
				res, outError = c.BGPPeers().Get(ctx, name1, options.GetOptions{ResourceVersion: rv1_1})
				Expect(outError).NotTo(HaveOccurred())
				Expect(res).To(MatchResource(apiv3.KindBGPPeer, testutils.ExpectNoNamespace, name1, spec1))
				Expect(res.ResourceVersion).To(Equal(rv1_1))
			}

			By("Getting BGPPeer (name1) with the updated resource version and comparing the output against spec2")
			res, outError = c.BGPPeers().Get(ctx, name1, options.GetOptions{ResourceVersion: rv1_2})
			Expect(outError).NotTo(HaveOccurred())
			Expect(res).To(MatchResource(apiv3.KindBGPPeer, testutils.ExpectNoNamespace, name1, spec2))
			Expect(res.ResourceVersion).To(Equal(rv1_2))

			if config.Spec.DatastoreType != apiconfig.Kubernetes {
				By("Listing BGPPeers with the original resource version and checking for a single result with name1/spec1")
				outList, outError = c.BGPPeers().List(ctx, options.ListOptions{ResourceVersion: rv1_1})
				Expect(outError).NotTo(HaveOccurred())
				Expect(outList.Items).To(ConsistOf(
					testutils.Resource(apiv3.KindBGPPeer, testutils.ExpectNoNamespace, name1, spec1),
				))
			}

			By("Listing BGPPeers with the latest resource version and checking for two results with name1/spec2 and name2/spec2")
			outList, outError = c.BGPPeers().List(ctx, options.ListOptions{})
			Expect(outError).NotTo(HaveOccurred())
			Expect(outList.Items).To(ConsistOf(
				testutils.Resource(apiv3.KindBGPPeer, testutils.ExpectNoNamespace, name1, spec2),
				testutils.Resource(apiv3.KindBGPPeer, testutils.ExpectNoNamespace, name2, spec2),
			))

			if config.Spec.DatastoreType != apiconfig.Kubernetes {
				By("Deleting BGPPeer (name1) with the old resource version")
				_, outError = c.BGPPeers().Delete(ctx, name1, options.DeleteOptions{ResourceVersion: rv1_1})
				Expect(outError).To(HaveOccurred())
				Expect(outError.Error()).To(Equal("update conflict: BGPPeer(" + name1 + ")"))
			}

			By("Deleting BGPPeer (name1) with the new resource version")
			dres, outError := c.BGPPeers().Delete(ctx, name1, options.DeleteOptions{ResourceVersion: rv1_2})
			Expect(outError).NotTo(HaveOccurred())
			Expect(dres).To(MatchResource(apiv3.KindBGPPeer, testutils.ExpectNoNamespace, name1, spec2))

			if config.Spec.DatastoreType != apiconfig.Kubernetes {
				By("Updating BGPPeer name2 with a 2s TTL and waiting for the entry to be deleted")
				_, outError = c.BGPPeers().Update(ctx, res2, options.SetOptions{TTL: 2 * time.Second})
				Expect(outError).NotTo(HaveOccurred())
				time.Sleep(1 * time.Second)
				_, outError = c.BGPPeers().Get(ctx, name2, options.GetOptions{})
				Expect(outError).NotTo(HaveOccurred())
				time.Sleep(2 * time.Second)
				_, outError = c.BGPPeers().Get(ctx, name2, options.GetOptions{})
				Expect(outError).To(HaveOccurred())
				Expect(outError.Error()).To(ContainSubstring("resource does not exist: BGPPeer(" + name2 + ") with error:"))

				By("Creating BGPPeer name2 with a 2s TTL and waiting for the entry to be deleted")
				_, outError = c.BGPPeers().Create(ctx, &apiv3.BGPPeer{
					ObjectMeta: metav1.ObjectMeta{Name: name2},
					Spec:       spec2,
				}, options.SetOptions{TTL: 2 * time.Second})
				Expect(outError).NotTo(HaveOccurred())
				time.Sleep(1 * time.Second)
				_, outError = c.BGPPeers().Get(ctx, name2, options.GetOptions{})
				Expect(outError).NotTo(HaveOccurred())
				time.Sleep(2 * time.Second)
				_, outError = c.BGPPeers().Get(ctx, name2, options.GetOptions{})
				Expect(outError).To(HaveOccurred())
				Expect(outError.Error()).To(ContainSubstring("resource does not exist: BGPPeer(" + name2 + ") with error:"))
			}

			if config.Spec.DatastoreType == apiconfig.Kubernetes {
				By("Attempting to deleting BGPPeer (name2)")
				dres, outError = c.BGPPeers().Delete(ctx, name2, options.DeleteOptions{})
				Expect(outError).NotTo(HaveOccurred())
				Expect(dres).To(MatchResource(apiv3.KindBGPPeer, testutils.ExpectNoNamespace, name2, spec2))
			}

			By("Attempting to delete BGPPeer (name2) again")
			_, outError = c.BGPPeers().Delete(ctx, name2, options.DeleteOptions{})
			Expect(outError).To(HaveOccurred())
			Expect(outError.Error()).To(ContainSubstring("resource does not exist: BGPPeer(" + name2 + ") with error:"))

			By("Listing all BGPPeers and expecting no items")
			outList, outError = c.BGPPeers().List(ctx, options.ListOptions{})
			Expect(outError).NotTo(HaveOccurred())
			Expect(outList.Items).To(HaveLen(0))

			By("Getting BGPPeer (name2) and expecting an error")
			_, outError = c.BGPPeers().Get(ctx, name2, options.GetOptions{})
			Expect(outError).To(HaveOccurred())
			Expect(outError.Error()).To(ContainSubstring("resource does not exist: BGPPeer(" + name2 + ") with error:"))
		},

		Entry("BGPPeerSpecs 1,2", name1, name2, spec1, spec2),
		Entry("BGPPeerSpecs 2,3", name2, name3, spec2, spec3),
		Entry("BGPPeerSpecs 3,1", name3, name1, spec3, spec1),
	)

	Describe("BGPPeer watch functionality", func() {
		It("should handle watch events for different resource versions and event types", func() {
			c, err := clientv3.New(config)
			Expect(err).NotTo(HaveOccurred())

			be, err := backend.NewClient(config)
			Expect(err).NotTo(HaveOccurred())
			be.Clean()

			By("Listing BGPPeers with the latest resource version and checking for two results with name1/spec2 and name2/spec2")
			outList, outError := c.BGPPeers().List(ctx, options.ListOptions{})
			Expect(outError).NotTo(HaveOccurred())
			Expect(outList.Items).To(HaveLen(0))
			rev0 := outList.ResourceVersion

			By("Configuring a BGPPeer name1/spec1 and storing the response")
			outRes1, err := c.BGPPeers().Create(
				ctx,
				&apiv3.BGPPeer{
					ObjectMeta: metav1.ObjectMeta{Name: name1},
					Spec:       spec1,
				},
				options.SetOptions{},
			)
			rev1 := outRes1.ResourceVersion

			By("Configuring a BGPPeer name2/spec2 and storing the response")
			outRes2, err := c.BGPPeers().Create(
				ctx,
				&apiv3.BGPPeer{
					ObjectMeta: metav1.ObjectMeta{Name: name2},
					Spec:       spec2,
				},
				options.SetOptions{},
			)

			By("Starting a watcher from revision rev1 - this should skip the first creation")
			w, err := c.BGPPeers().Watch(ctx, options.ListOptions{ResourceVersion: rev1})
			Expect(err).NotTo(HaveOccurred())
			testWatcher1 := testutils.NewTestResourceWatch(config.Spec.DatastoreType, w)
			defer testWatcher1.Stop()

			By("Deleting res1")
			_, err = c.BGPPeers().Delete(ctx, name1, options.DeleteOptions{})
			Expect(err).NotTo(HaveOccurred())

			By("Checking for two events, create res2 and delete re1")
			testWatcher1.ExpectEvents(apiv3.KindBGPPeer, []watch.Event{
				{
					Type:   watch.Added,
					Object: outRes2,
				},
				{
					Type:     watch.Deleted,
					Previous: outRes1,
				},
			})
			testWatcher1.Stop()

			By("Starting a watcher from rev0 - this should get all events")
			w, err = c.BGPPeers().Watch(ctx, options.ListOptions{ResourceVersion: rev0})
			Expect(err).NotTo(HaveOccurred())
			testWatcher2 := testutils.NewTestResourceWatch(config.Spec.DatastoreType, w)
			defer testWatcher2.Stop()

			By("Modifying res2")
			outRes3, err := c.BGPPeers().Update(
				ctx,
				&apiv3.BGPPeer{
					ObjectMeta: outRes2.ObjectMeta,
					Spec:       spec1,
				},
				options.SetOptions{},
			)
			Expect(err).NotTo(HaveOccurred())
			testWatcher2.ExpectEvents(apiv3.KindBGPPeer, []watch.Event{
				{
					Type:   watch.Added,
					Object: outRes1,
				},
				{
					Type:   watch.Added,
					Object: outRes2,
				},
				{
					Type:     watch.Deleted,
					Previous: outRes1,
				},
				{
					Type:     watch.Modified,
					Previous: outRes2,
					Object:   outRes3,
				},
			})
			testWatcher2.Stop()

			// Only etcdv3 supports watching a specific instance of a resource.
			if config.Spec.DatastoreType == apiconfig.EtcdV3 {
				By("Starting a watcher from rev0 watching name1 - this should get all events for name1")
				w, err = c.BGPPeers().Watch(ctx, options.ListOptions{Name: name1, ResourceVersion: rev0})
				Expect(err).NotTo(HaveOccurred())
				testWatcher2_1 := testutils.NewTestResourceWatch(config.Spec.DatastoreType, w)
				defer testWatcher2_1.Stop()
				testWatcher2_1.ExpectEvents(apiv3.KindBGPPeer, []watch.Event{
					{
						Type:   watch.Added,
						Object: outRes1,
					},
					{
						Type:     watch.Deleted,
						Previous: outRes1,
					},
				})
				testWatcher2_1.Stop()
			}

			By("Starting a watcher not specifying a rev - expect the current snapshot")
			w, err = c.BGPPeers().Watch(ctx, options.ListOptions{})
			Expect(err).NotTo(HaveOccurred())
			testWatcher3 := testutils.NewTestResourceWatch(config.Spec.DatastoreType, w)
			defer testWatcher3.Stop()
			testWatcher3.ExpectEvents(apiv3.KindBGPPeer, []watch.Event{
				{
					Type:   watch.Added,
					Object: outRes3,
				},
			})
			testWatcher3.Stop()

			By("Configuring BGPPeer name1/spec1 again and storing the response")
			outRes1, err = c.BGPPeers().Create(
				ctx,
				&apiv3.BGPPeer{
					ObjectMeta: metav1.ObjectMeta{Name: name1},
					Spec:       spec1,
				},
				options.SetOptions{},
			)

			By("Starting a watcher not specifying a rev - expect the current snapshot")
			w, err = c.BGPPeers().Watch(ctx, options.ListOptions{})
			Expect(err).NotTo(HaveOccurred())
			testWatcher4 := testutils.NewTestResourceWatch(config.Spec.DatastoreType, w)
			defer testWatcher4.Stop()
			testWatcher4.ExpectEventsAnyOrder(apiv3.KindBGPPeer, []watch.Event{
				{
					Type:   watch.Added,
					Object: outRes1,
				},
				{
					Type:   watch.Added,
					Object: outRes3,
				},
			})

			By("Cleaning the datastore and expecting deletion events for each configured resource (tests prefix deletes results in individual events for each key)")
			be.Clean()
			testWatcher4.ExpectEventsAnyOrder(apiv3.KindBGPPeer, []watch.Event{
				{
					Type:     watch.Deleted,
					Previous: outRes1,
				},
				{
					Type:     watch.Deleted,
					Previous: outRes3,
				},
			})
			testWatcher4.Stop()
		})
	})

	Describe("BGPPeer validation", func() {
		It("should validate if a BGPFilter exists when it is specified in a BGPPeer", func() {
			c, err := clientv3.New(config)
			Expect(err).NotTo(HaveOccurred())

			be, err := backend.NewClient(config)
			Expect(err).NotTo(HaveOccurred())
			be.Clean()

			By("Attempting to create a new BGPPeer with nameWithFilter1/specWithFilter1 before creating the BGPFilter")
			_, outError := c.BGPPeers().Create(ctx, &apiv3.BGPPeer{
				ObjectMeta: metav1.ObjectMeta{Name: nameWithFilter1},
				Spec:       specWithFilter1,
			}, options.SetOptions{})
			Expect(outError).To(HaveOccurred())
			Expect(outError.Error()).To(Equal("error with field BGPPeer.Spec.Filters = '[bgp-filter-1]' (BGPFilter(s) not found)"))

			By("Creating the BGPFilter first and then creating the BGPPeer afterwards")
			_, outError = c.BGPFilter().Create(ctx, &apiv3.BGPFilter{
				ObjectMeta: metav1.ObjectMeta{Name: filterName1},
				Spec:       filterSpec1,
			}, options.SetOptions{})
			Expect(outError).NotTo(HaveOccurred())
			peerRes, outError := c.BGPPeers().Create(ctx, &apiv3.BGPPeer{
				ObjectMeta: metav1.ObjectMeta{Name: nameWithFilter1},
				Spec:       specWithFilter1,
			}, options.SetOptions{})
			Expect(outError).NotTo(HaveOccurred())

			By("Attempting to update the BGPPeer with a nonexistent BGPFilter")
			peerRes.Spec.Filters = []string{"bgp-filter-1", "bgp-filter-2"}
			_, outError = c.BGPPeers().Update(ctx, peerRes, options.SetOptions{})
			Expect(outError).To(HaveOccurred())
			Expect(outError.Error()).To(Equal("error with field BGPPeer.Spec.Filters = '[bgp-filter-2]' (BGPFilter(s) not found)"))

			By("Creating the second BGPFilter first and then updating the BGPPeer afterwards")
			_, outError = c.BGPFilter().Create(ctx, &apiv3.BGPFilter{
				ObjectMeta: metav1.ObjectMeta{Name: filterName2},
				Spec:       filterSpec2,
			}, options.SetOptions{})
			Expect(outError).NotTo(HaveOccurred())
			_, outError = c.BGPPeers().Update(ctx, peerRes, options.SetOptions{})
			Expect(outError).NotTo(HaveOccurred())
		})
	})
})
