// Copyright 2018 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef ASH_APP_MENU_NOTIFICATION_MENU_CONTROLLER_H_
#define ASH_APP_MENU_NOTIFICATION_MENU_CONTROLLER_H_

#include "ash/app_menu/app_menu_export.h"
#include "ash/app_menu/notification_menu_view.h"
#include "base/memory/raw_ptr.h"
#include "base/scoped_observation.h"
#include "ui/message_center/message_center.h"
#include "ui/message_center/message_center_observer.h"
#include "ui/views/animation/slide_out_controller_delegate.h"

namespace views {
class MenuItemView;
}

namespace ash {

class AppMenuModelAdapter;

// Handles adding/removing NotificationMenuView from the root MenuItemView,
// adding the container model entry, and updating the NotificationMenuView
// as notifications come and go.
class APP_MENU_EXPORT NotificationMenuController
    : public message_center::MessageCenterObserver,
      public views::SlideOutControllerDelegate,
      public NotificationMenuView::Delegate {
 public:
  NotificationMenuController(const std::string& app_id,
                             views::MenuItemView* root_menu,
                             AppMenuModelAdapter* app_menu_model_adapter);

  NotificationMenuController(const NotificationMenuController&) = delete;
  NotificationMenuController& operator=(const NotificationMenuController&) =
      delete;

  ~NotificationMenuController() override;

  // message_center::MessageCenterObserver overrides:
  void OnNotificationAdded(const std::string& notification_id) override;
  void OnNotificationUpdated(const std::string& notification_id) override;
  void OnNotificationRemoved(const std::string& notification_id,
                             bool by_user) override;

  // views::SlideOutControllerDelegate overrides:
  ui::Layer* GetSlideOutLayer() override;
  void OnSlideChanged(bool in_progress) override;
  void OnSlideOut() override;

  // NotificationMenuView::Delegate overrides:
  void OnOverflowAddedOrRemoved() override;
  void ActivateNotificationAndClose(
      const std::string& notification_id) override;

 private:
  // Adds a container MenuItemView to |root_menu_|, adds NOTIFICATION_CONTAINER
  // to |model_|, creates and initializes NotificationMenuView, and adds
  // NotificationMenuView to the container MenuItemView.
  void InitializeNotificationMenuView();

  // Identifies the application the menu is for.
  const std::string app_id_;

  // The top level MenuItemView. Owned by |AppMenuModelAdapter::menu_runner_|.
  const raw_ptr<views::MenuItemView, DanglingUntriaged> root_menu_;

  // Manages showing the menu. Owned by the view requesting a menu.
  const raw_ptr<AppMenuModelAdapter, DanglingUntriaged> app_menu_model_adapter_;

  // The view which shows all active notifications for |app_id_|. Owned by the
  // views hierarchy.
  raw_ptr<NotificationMenuView, DanglingUntriaged> notification_menu_view_ = nullptr;

  base::ScopedObservation<message_center::MessageCenter,
                          message_center::MessageCenterObserver>
      message_center_observation_{this};
};

}  // namespace ash

#endif  // ASH_APP_MENU_NOTIFICATION_MENU_CONTROLLER_H_
