/*
    Copyright 2004-2010 Luigi Auriemma

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA

    http://www.gnu.org/licenses/gpl-2.0.txt
*/

//#define NOLFS
#ifndef NOLFS   // 64 bit file support not really needed since the tool uses signed 32 bits at the moment, anyway I leave it enabled
    #define _LARGE_FILES        // if it's not supported the tool will work
    #define __USE_LARGEFILE64   // without support for large files
    #define __USE_FILE_OFFSET64
    #define _LARGEFILE_SOURCE
    #define _LARGEFILE64_SOURCE
    #define _FILE_OFFSET_BITS   64
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <zlib.h>           // -lz

#ifdef WIN32
    #include <winsock.h>
    #include "winerr.h"

    #define close   closesocket
    #define sleep   Sleep
    #define MYPIPENAME  "\\\\.\\pipe\\onlywebs"
    #define PIPE_T  HANDLE
    #define RUNME   "start %s"
#else
    #include <unistd.h>
    #include <sys/socket.h>
    #include <sys/types.h>
    #include <arpa/inet.h>
    #include <netinet/in.h>
    #include <netdb.h>
    #include <pthread.h>    // -lpthread
    #include <fcntl.h>

    #define MYPIPENAME  "/tmp/onlywebs"
    #define PIPE_T      int
    #define RUNME   "%s &"
#endif

#ifdef WIN32
    #define quick_thread(NAME, ARG) DWORD WINAPI NAME(ARG)
    #define thread_id   DWORD
#else
    #define quick_thread(NAME, ARG) void *NAME(ARG)
    #define thread_id   pthread_t
#endif

#if defined(_LARGE_FILES)
    #if defined(__APPLE__)
        #define fseek   fseeko
        #define ftell   ftello
    #elif defined(__FreeBSD__)
    #elif !defined(NOLFS)       // use -DNOLFS if this tool can't be compiled on your OS!
        #define off_t   off64_t
        #define fopen   fopen64
        #define fseek   fseeko64
        #define ftell   ftello64
    #endif
#endif

thread_id quick_threadx(void *func, void *data) {
    thread_id       tid;
#ifdef WIN32
    if(!CreateThread(NULL, 0, func, data, 0, &tid)) return(0);
#else
    pthread_attr_t  attr;
    pthread_attr_init(&attr);
    pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_DETACHED);
    if(pthread_create(&tid, &attr, func, data)) return(0);
#endif
    return(tid);
}

typedef unsigned char   u8;
typedef unsigned short  u16;
typedef unsigned int    u32;



#define VER             "0.3"
#define PORT            80
#define BUFFSZ          1024
#define MAX_NAME        272
#define MAXZIPLEN(n)    ((n)+(((n)/1000)+1)+12)



int read_pipe(u8 *buff, int buffsz, PIPE_T mypipe, int *connected);
PIPE_T create_pipe(int buffsz, u8 *pipe_name);
u8 *string_to_execute(u8 *str, u8 *name, int sd);
quick_thread(client, int sock);
int get_num(u8 *str);
int timeout(int sock, int secs);
void std_err(void);



PIPE_T  onlywebs_pipe   = 0;
int     do_headers      = 1,
        do_headers_cl   = 1,
        file_offset     = 0,
        file_size       = -1,
        force_chunks    = 0,
        zchunks         = 0,
        force_cl        = 0;
u8      *filename       = NULL,
        *cmd            = NULL,
        *execstring     = NULL;



int main(int argc, char *argv[]) {
    struct  sockaddr_in peer;
    int     i,
            sd,
            sa,
            psz,
            on          = 1,
            pipefile    = 0;
    u16     port        = PORT;

#ifdef WIN32
    WSADATA    wsadata;
    WSAStartup(MAKEWORD(1,0), &wsadata);
#endif

    fputs("\n"
        "One file only web server "VER"\n"
        "by Luigi Auriemma\n"
        "e-mail: aluigi@autistici.org\n"
        "web:    aluigi.org\n"
        "\n", stdout);

    if(argc < 2) {
        printf("\n"
            "Usage: %s [options] <file>\n"
            "\n"
            "Options:\n"
            "-p PORT  port to bind (%d)\n"
            "-x       the input file will be sent as is without the additional HTTP headers\n"
            "         useful for testing the own custom headers\n"
            "-X       don't wait the client HTTP request, send the data immediately\n"
            "-o OFF   offset of the file from which starting the sending\n"
            "-s SIZE  amount of byte of the file to send\n"
            "-P       file will be considered a named pipe created by this tool so that any\n"
            "         program can send data to the client simply writing in file, example:\n"
            "         onlywebs -P "MYPIPENAME"\n"
            "         (when a client connects) echo hello > "MYPIPENAME"\n"
            "-c       force chunked mode used automatically with empty files, this mode is\n"
            "         good for sending pieces of data at time like when using stdin (-)\n"
            "-z       enable the deflate compression when in chunk mode (default is none)\n"
            "-C       if the file is a named pipe or stdin use a Content-Length set to max\n"
            "-r \"P\"   execute a specific program each time a client connects, for example\n"
            "         it's good to run mencoder with the named pipe as output file, use\n"
            "         #NAME to replace them with the file requested by the client\n"
            "\n"
            "Note that this tool has been created for testing purpose and not real usage\n"
            "\n", argv[0], port);
        exit(1);
    }

    argc--;
    for(i = 1; i < argc; i++) {
        if(((argv[i][0] != '-') && (argv[i][0] != '/')) || (strlen(argv[i]) != 2)) {
            printf("\nError: wrong argument (%s)\n", argv[i]);
            exit(1);
        }
        switch(argv[i][1]) {
            case 'p': port          = get_num(argv[++i]);   break;
            case 'x': do_headers    = 0;                    break;
            case 'X': do_headers_cl = 0;                    break;
            case 'o': file_offset   = get_num(argv[++i]);   break;
            case 's': file_size     = get_num(argv[++i]);   break;
            case 'P': pipefile      = 1;                    break;
            case 'c': force_chunks  = 1;                    break;
            case 'z': zchunks       = 1;                    break;
            case 'C': force_cl      = 1;                    break;
            case 'r': execstring    = argv[++i];            break;
            default: {
                printf("\nError: wrong argument (%s)\n", argv[i]);
                exit(1);
            }
        }
    }
    filename = argv[argc];

    if(pipefile) {
        onlywebs_pipe = create_pipe(BUFFSZ, filename);
        force_chunks = 1;
    }

    peer.sin_addr.s_addr = INADDR_ANY;
    peer.sin_port        = htons(port);
    peer.sin_family      = AF_INET;

    sd = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP);
    if(sd < 0) std_err();
    if(setsockopt(sd, SOL_SOCKET, SO_REUSEADDR, (char *)&on, sizeof(on))
      < 0) std_err();
    if(bind(sd, (struct sockaddr *)&peer, sizeof(struct sockaddr_in))
      < 0) std_err();
    if(listen(sd, SOMAXCONN)
      < 0) std_err();

    printf("\n- waiting connections on port %u:\n\n", port);

    for(;;) {
        psz = sizeof(struct sockaddr_in);
        sa = accept(sd, (struct sockaddr *)&peer, &psz);
        if(sa < 0) std_err();

        printf("%s : %hu\n",
            inet_ntoa(peer.sin_addr), ntohs(peer.sin_port));

        if(!quick_threadx(client, (void *)sa)) close(sa);
    }

    close(sd);
    return(0);
}



int deflate_compress(u8 *in, int insz, u8 *out, int outsz, z_stream *z, int step) {
    int     ret;

    if(step < 0) {
        if(z->state) {
            deflateEnd(z);
            z->state = NULL;
        }
        return(0);
    }
    if(!z->state) {
        z->zalloc = Z_NULL;
        z->zfree  = Z_NULL;
        z->opaque = Z_NULL;
        if(deflateInit2(z, 9, Z_DEFLATED, -15, 9, Z_DEFAULT_STRATEGY)) {
            printf("\nError: zlib initialization error\n");
            return(-1);
        }
    }

    z->next_in   = in;
    z->avail_in  = insz;
    z->next_out  = out;
    z->avail_out = outsz;
    ret = deflate(z, step ? Z_FINISH : Z_SYNC_FLUSH);   // with Z_NO_FLUSH it gives problems
    if(ret == Z_OK) {
        ret = outsz - z->avail_out;
    } else if(ret == Z_STREAM_END) {
        ret = outsz - z->avail_out;
        deflateEnd(z);
        z->state = NULL;
    } else {
        printf("\nError: the compressed zlib/deflate input is wrong or incomplete (%d)\n", ret);
        return(-1);
    }
    return(ret);
}



quick_thread(client, int sd) {
    z_stream    z;
    struct  stat    xstat;
    FILE    *fd         = NULL;
    int     t           = 0,
            myeof       = 0,
            tot         = 0,
            len         = 0,
            buffsz      = 0,
            zbuffsz     = 0,
            chunks      = 0,
            blah        = 0;
    u8      tmp[32]     = "",
            *buff       = NULL,
            *zbuff      = NULL,
            *p          = NULL,
            *req_name   = NULL,
            *tmpexec    = NULL;            

    buffsz = BUFFSZ;
    buff = malloc(buffsz + 1);
    if(!buff) goto quit;
    memset(&z, 0, sizeof(z));
    memset(&xstat, 0, sizeof(xstat));

    if(do_headers_cl) {
        do {
            if(buffsz <= len) { // so in any case buff is allocated with some bytes
                buffsz += BUFFSZ;
                buff = realloc(buff, buffsz + 1);
                if(!buff) goto quit;
            }
            if(timeout(sd, 10) < 0) goto quit;
            t = recv(sd, buff + len, buffsz - len, 0);
            if(t <= 0) goto quit;
            len += t;
            buff[len] = 0;
        } while(!strstr(buff, "\r\n\r\n") && !strstr(buff, "\n\n"));

        for(p = buff; *p && (*p != '\r') && (*p != '\n'); p++);
        *p = 0;
        printf("  %s\n", buff);

        for(--p; (p >= buff) && (*p > ' '); p--)
        *p = 0;
        while((p >= buff) && (*p <= ' ')) *p-- = 0;
        req_name = strchr(buff, '/');
        if(!req_name) req_name = strchr(buff, '\\');
        if(req_name) {
            req_name++;
        } else {
            req_name = "\"\"";
        }
        p = strrchr(req_name, '\\');
        if(!p) p = strrchr(req_name, '/');
        if(p) req_name = p + 1;
        for(p = req_name; *p; p++) {
            if(strchr(":;%&\\//", *p)) *p = 0;
        }
        if(strlen(req_name) > MAX_NAME) req_name[MAX_NAME] = 0;
        req_name = strdup(req_name);    // needed
    }

    if(onlywebs_pipe) {
        if(file_offset) {
            for(tot = 0; tot < file_offset; tot++) {
                read_pipe(buff, 1, onlywebs_pipe, &blah);
            }
        }
    } else if(!filename || !filename[0]) {
        fd = NULL;
    } else {
        if(!strcmp(filename, "-")) {
            fd = stdin;
        } else {
            fd = fopen(filename, "rb"); // reload the file all the times, this is a test tool
            if(!fd) std_err();
        }
        if(file_offset && fd) {
            if(fd == stdin) {
                for(tot = 0; tot < file_offset; tot++) {
                    if(fgetc(fd) < 0) goto quit;
                }
            } else {
                if(fseek(fd, file_offset, SEEK_SET)) std_err();
            }
        }
    }

    if(do_headers) {
        len = sprintf(buff,
            "HTTP/1.1 200 OK\r\n" \
            "Content-Type: application/octet-stream\r\n" \
            "Connection: close\r\n");
        xstat.st_size = 0;
        if(force_cl) {
            xstat.st_size = 0x7fffffff;
            force_chunks  = 0;
        } else {
            if(fd) {
                if(fstat(fileno(fd), &xstat) < 0) std_err();
            }
        }
        if((int)xstat.st_size && !force_chunks) {
            len += sprintf(buff + len,
                "Content-Length: %u\r\n",
                (int)xstat.st_size);
        } else {    // chunked dynamic size
            len += sprintf(buff + len,
                "Content-Encoding: %s\r\n"
                "Transfer-Encoding: chunked\r\n",
                zchunks ? "deflate" : "identity");
            chunks = 1;
        }
        buff[len++] = '\r';
        buff[len++] = '\n';
        if(send(sd, buff, len, 0) != len) goto quit;
    }

    if(execstring) {
        tmpexec = string_to_execute(execstring, req_name, sd);
        fprintf(stderr, "   Execute: \"%s\"\n", tmpexec);
        system(tmpexec);
        free(tmpexec);
    }

    z.state = NULL; // init
    if(chunks) {
        zbuffsz = MAXZIPLEN(buffsz);
        zbuff = malloc(zbuffsz);
        if(!zbuff) goto quit;
        buff = realloc(buff, zbuffsz + 32); // space for CHUNKSZ\r\n+\r\n
        if(!buff) goto quit;
    }

    myeof = 0;
    tot = 0;
    for(;;) {
        if(!myeof) {
            if(onlywebs_pipe) {
                // don't use buffsz otherwise the sending is not in real-time
                //len = read_pipe(buff, 1 /*buffsz*/, onlywebs_pipe, &blah);
                if(chunks) {    // otherwise there will be chunks of 1 byte!
                    len = read_pipe(buff, buffsz, onlywebs_pipe, &blah);
                } else {
                    len = read_pipe(buff, 1, onlywebs_pipe, &blah);
                }
            } else if(!fd) {
                len = 0;
            } else {
                len = fread(buff, 1, buffsz, fd);
            }
            if(len <= 0) {
                if(!chunks) break;
                myeof = 1;
            }
            if((file_size >= 0) && ((tot + len) > file_size)) {
                len = file_size - tot;
                buffsz = 0; // trick for breaking automatically
                myeof = 1;
            }
            tot += len;
        } else {
            if(chunks && !z.state) break;   // end reached
            if(!chunks) break;
            len = 0;
        }

        if(chunks) {    // fast solution
            if(zchunks) {
                len = deflate_compress(buff, len, zbuff, zbuffsz, &z, myeof);
                if(!len) continue;  // with Z_NO_FLUSH you can have 0 returned
                if(len < 0) goto quit;
                t = sprintf(buff, "%x\r\n", len);
                memcpy(buff + t, zbuff, len);
            } else {
                t = sprintf(tmp, "%x\r\n", len);
                for(p = buff + len - 1; p >= buff; p--) {
                    *(p + t) = *p;
                }
                memcpy(buff, tmp, t);
            }
            len += t;
            buff[len++] = '\r';
            buff[len++] = '\n';
        }
        if(send(sd, buff, len, 0) != len) goto quit;
    }

    if(chunks) {
        deflate_compress(NULL, 0, NULL, 0, &z, -1);
        if(send(sd, "0\r\n\r\n", 5, 0) != 5) goto quit;
    }

quit:
    if(fd && (fd != stdin)) fclose(fd);
    if(buff) free(buff);
    if(zbuff) free(zbuff);
    if(req_name) free(req_name);
    close(sd);
    return(0);
}



int read_pipe(u8 *buff, int buffsz, PIPE_T mypipe, int *connected) {
    int     len = 0;

#ifdef WIN32
    DWORD   tmp;
    if(*connected) goto lame;
    while(ConnectNamedPipe(mypipe, NULL)) {
        *connected = 1;
        lame:
        while(ReadFile(mypipe, buff + len, buffsz - len, &tmp, NULL)) {
            len += tmp;
            if(len >= buffsz) {
                //DisconnectNamedPipe(mypipe);    // not needed here
                return(len);
            }
        }
        DisconnectNamedPipe(mypipe);
        *connected = 0;
    }
#else
    len = read(mypipe, buff, buffsz);
    //if(len <= 0) break;
    return(len);
#endif
    return(0);
}



PIPE_T create_pipe(int buffsz, u8 *pipe_name) {
    PIPE_T  mypipe;

    printf("- create named pipe file: %s\n", pipe_name);
#ifdef WIN32
    mypipe = CreateNamedPipe(
        pipe_name,
        PIPE_ACCESS_DUPLEX,
        PIPE_TYPE_BYTE | PIPE_READMODE_BYTE | PIPE_WAIT,
        PIPE_UNLIMITED_INSTANCES,
        buffsz,
        buffsz,
        NMPWAIT_USE_DEFAULT_WAIT,
        NULL);
    if(mypipe == INVALID_HANDLE_VALUE) {
        fprintf(stderr, "\nError: unable to create the named pipe %s\n", pipe_name);
        exit(1);
    }
#else
    mypipe = mkfifo(pipe_name, 0755);
    if((mypipe < 0) && (errno != EEXIST)) {
        fprintf(stderr, "\nError: unable to create the named pipe %s\n", pipe_name);
        exit(1);
    }
    mypipe = open(pipe_name, O_RDWR);
    if(mypipe < 0) {
        fprintf(stderr, "\nError: unable to open the named pipe %s\n", pipe_name);
        exit(1);
    }
#endif
    return(mypipe);
}



u32 get_peer_ip_port(int sd, u16 *port) {
    struct sockaddr_in  peer;
    int     psz;

    psz = sizeof(struct sockaddr_in);

    if(getpeername(sd, (struct sockaddr *)&peer, &psz) < 0) {
        peer.sin_addr.s_addr = 0;                   // avoids possible problems
        peer.sin_port        = 0;
    }

    if(port) *port = ntohs(peer.sin_port);

    return(peer.sin_addr.s_addr);
}



u8 *string_to_execute(u8 *str, u8 *name, int sd) {
#define QUICK_ALLOC(X) \
            if((newsz + X) >= totsz) { \
                totsz = newsz + X + 1024; \
                new = realloc(new, totsz + 1); \
            }
    u32     ip;
    u16     port;
    int     newsz   = 0,
            totsz   = 0,
            namelen;
    u8      *p,
            *new    = NULL;

    ip = get_peer_ip_port(sd, &port);
    if(!name) name = "";

    totsz = strlen(str) + 1024;
    new = malloc(totsz + 1);
#ifdef WIN32
    newsz = sprintf(new, "start ");
#endif
    namelen = strlen(name);
    for(p = str; *p;) {
        if(!strncmp(p, "#NAME", 5)) {
            QUICK_ALLOC(namelen)
            memcpy(new + newsz, name, namelen);
            newsz += namelen;
            p += 5;
        } else if(!strncmp(p, "#IP", 3)) {
            QUICK_ALLOC(32)
            newsz += sprintf(new + newsz, "%u.%u.%u.%u", (ip)&0xff, (ip>>8)&0xff, (ip>>16)&0xff, (ip>>24)&0xff);
            p += 3;
        } else if(!strncmp(p, "#PORT", 5)) {
            QUICK_ALLOC(32)
            newsz += sprintf(new + newsz, "%u", port);
            p += 5;
        } else {
            QUICK_ALLOC(1)
            new[newsz++] = *p;
            p++;
        }
    }
    new = realloc(new, newsz + 2 + 1);
    strcpy(new + newsz, " &");  // valid for both Win and others
    return(new);
}



int get_num(u8 *str) {
    int     offset;

    if(!strncmp(str, "0x", 2) || !strncmp(str, "0X", 2)) {
        sscanf(str + 2, "%x", &offset);
    } else {
        sscanf(str, "%u", &offset);
    }
    return(offset);
}



int timeout(int sock, int secs) {
    struct  timeval tout;
    fd_set  fd_read;

    tout.tv_sec  = secs;
    tout.tv_usec = 0;
    FD_ZERO(&fd_read);
    FD_SET(sock, &fd_read);
    if(select(sock + 1, &fd_read, NULL, NULL, &tout)
      <= 0) return(-1);
    return(0);
}



#ifndef WIN32
    void std_err(void) {
        perror("\nError");
        exit(1);
    }
#endif


