#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#define pwdLength	32

unsigned char constantShort[] = {0x09, 0x02, 0x13, 0x45, 0x07, 0x04, 0x13, 0x44, 
								 0x0C, 0x08, 0x13, 0x5A, 0x32, 0x15, 0x13, 0x5D, 
								 0xD2, 0x17, 0xEA, 0xD3, 0xB5, 0xDF, 0x55, 0x63, 
								 0x22, 0xE9, 0xA1, 0x4A, 0x99, 0x4B, 0x0F, 0x88};
unsigned char constantLong[]  = {0xB1, 0x56, 0x35, 0x1A, 0x9C, 0x98, 0x80, 0x84, 
								 0x37, 0xA7, 0x3D, 0x61, 0x7F, 0x2E, 0xE8, 0x76, 
								 0x2A, 0xF2, 0xA5, 0x84, 0x07, 0xC7, 0xEC, 0x27, 
								 0x6F, 0x7D, 0x04, 0xCD, 0x52, 0x1E, 0xCD, 0x5B, 
								 0xB3, 0x29, 0x76, 0x66, 0xD9, 0x5E, 0x4B, 0xCA, 
								 0x63, 0x72, 0x6F, 0xD2, 0xFD, 0x25, 0xE6, 0x7B, 
								 0xC5, 0x66, 0xB3, 0xD3, 0x45, 0x9A, 0xAF, 0xDA, 
								 0x29, 0x86, 0x22, 0x6E, 0xB8, 0x03, 0x62, 0xBC};

typedef unsigned char p_block[pwdLength];

typedef struct __passlist {
	p_block pass;
	struct __passlist *next;
} PASSLIST;

void encode (char *argv);
int decode (char *argv);
void hex_display (char *msg, unsigned char *data, unsigned long length);
void encode_long (p_block pass, unsigned char j);
void usage (char *argv);
int a_to_pb (const char *instr, p_block out);
int decode_long (PASSLIST **pListHead, p_block pass, unsigned char j);
void free_passlist (PASSLIST *pl);

/* Arguments:
 * -e encode ASCII password into 32-byte password block
 * -d decode 32-byte password block into ASCII password
 */

int main (int argc, char *argv[])
{
	p_block pass;

	fprintf(stdout, "\nPalmOS Password Codec\n");
	fprintf(stdout, "kingpin@atstake.com\n");
	fprintf(stdout, "@stake Research Labs\n");
	fprintf(stdout, "http://www.atstake.com/research\n");
	fprintf(stdout, "August 2000\n");

	if (argc > 1)
	{
		if (argv[2] == NULL) usage(argv[0]);
		if (stricmp (argv[1], "-e") == 0)
			encode (argv[2]);
		else if (stricmp (argv[1], "-d") == 0)
		{
			if (a_to_pb (argv[2], pass) == 0) usage(argv[0]);
			decode (pass);
		}
		else
			usage(argv[0]);
	}
	else
		usage(argv[0]);

	return 0;
}

void encode (char *argv)
{
	int i, j, len;
	p_block input, pass;
        
	strncpy (input, argv, pwdLength - 1);
	input[pwdLength - 1] = 0x00; 

	if (strlen(input) <= 4)
	{
		// compute index into constant
		j = (input[0] + strlen(input)) % pwdLength;

		for (i = 0; i < pwdLength; ++i)
		{
			if (j == pwdLength) j = 0; // wrap around to beginning

			pass[i] = input[i] ^ constantShort[j];
			++j;
		}
	}
	else
	{
		strncpy (pass, input, pwdLength);
		len = strlen(input);

		// pad out 32-bytes with copies of input string (incrementing ASCII value by len)
		while (len < pwdLength)
		{
			for (i = len; i < len * 2; ++i) 
				pass[i] = pass[i - len] + len; // increment each character by len

			len = len * 2;
		}

		encode_long (pass, 2); 
		encode_long (pass, 16);
		encode_long (pass, 24);
		encode_long (pass, 8); 
	}

	hex_display("", &pass[0], 8);
	hex_display("", &pass[8], 8);
	hex_display("", &pass[16], 8);
	hex_display("", &pass[24], 8);
	fprintf(stdout, "\n\n");
}

int decode (char *argv)
{
	unsigned char temp;
	int i, j;
	p_block pass;
	PASSLIST *newpl,*oldpl,*entry;

	for (j = 0; j < pwdLength; ++j) // prevent an endless loop if bytes do not match
	{
		if (memcmp(&argv[4], &constantShort[4], 28) != 0)
		{
			// rotate constant block by 1 byte
			temp = constantShort[0];
			for (i = 1; i < pwdLength; ++i)
				constantShort[i - 1] = constantShort[i]; 
			constantShort[pwdLength - 1] = temp;
		}
	}

	// short password
	if (memcmp(&argv[4], &constantShort[4], 28) == 0) // password is 4 chars or less
	{
		pass[0] = argv[0] ^ constantShort[0];
		pass[1] = argv[1] ^ constantShort[1];
		pass[2] = argv[2] ^ constantShort[2];
		pass[3] = argv[3] ^ constantShort[3];
		pass[4] = argv[4] ^ constantShort[4]; // 0x00

		hex_display("", pass, strlen(pass));
		fprintf(stdout, "\n\n");
	}

	// long password, output feedback w/ no key
	newpl = NULL;
	oldpl = NULL;

	// Pass 1
	decode_long (&newpl, argv, 8);
	oldpl = newpl;
	newpl = NULL;

	// Pass 2
	entry = oldpl;
	while (entry != NULL) {
		decode_long (&newpl, entry->pass, 24);
		entry = entry->next;
	}
	free_passlist (oldpl);
	oldpl = newpl;
	newpl = NULL;
	
	// Pass 3	
	entry = oldpl;
	while (entry != NULL) {
		decode_long (&newpl, entry->pass, 16);
		entry = entry->next;
	}
	free_passlist (oldpl);
	oldpl = newpl;
	newpl = NULL;

	// Pass 4
	entry = oldpl;
	while (entry != NULL) {
		decode_long (&newpl, entry->pass, 2);
		entry = entry->next;
	}
	free_passlist (oldpl);
	oldpl = newpl;
	newpl = NULL;

	// Test each entry in oldpl
	while (oldpl != NULL) {
		for (i = 5; i < pwdLength; ++i) // pass is greater than 4 chars
		{
			for (j = 0; j + i < pwdLength; ++j)
			{

				if ((oldpl->pass[j+i] - i) != oldpl->pass[j]) 
					break;

				if (j + i == pwdLength - 1) { // Done comparing?
					// Success
					strncpy (pass, oldpl->pass, i);
					pass[i] = 0x00; 
					hex_display("", pass, strlen(pass));
					fprintf(stdout, "\n");
					return 0;
				}
			}
		}

		oldpl = oldpl->next;
	}

	free_passlist(oldpl);
	return 0;
}

void free_passlist (PASSLIST *pl)
{
	PASSLIST *next;

	while (pl != NULL) {
		next = pl->next;
		free (pl);
		pl = next;
	}
}

int decode_long (PASSLIST **pListHead, p_block pass, unsigned char j)
{
	int i;
	unsigned char index, shift, j_temp, index_temp;
	unsigned short temp;
	p_block pass_temp;

	// 512 possible values for temp
	for (index = 0; index < 64; ++index)
	{
		for (shift = 0; shift < 8; ++shift)
		{
			memcpy (pass_temp, pass, pwdLength);
			j_temp = j;
			index_temp = index;

			for (i = 0; i < pwdLength; ++i)
			{
				if (j_temp == pwdLength) j_temp = 0; // wrap around to beginning
				if (index_temp == 64) index_temp = 0; // wrap around to beginning

				temp = constantLong[index_temp]; // xy
				temp <<= 8;
				temp |= constantLong[index_temp]; // xyxy

				temp >>= shift;

				pass_temp[j_temp] ^= (unsigned char) temp;
				++j_temp;
				++index_temp;
			}

			if ((index == ((pass_temp[j] + pass_temp[j+1]) & 0x3F)) &&
			    (shift == ((pass_temp[j+2] + pass_temp[j+3]) & 0x7)))
			{
				// Success!
				PASSLIST *entry = (PASSLIST *) malloc(sizeof(PASSLIST));
				memcpy (&(entry->pass), pass_temp, pwdLength);
				entry->next = *pListHead;
				*pListHead = entry;
			}
		}
	}

	return 0;
}

int a_to_pb (const char *instr, p_block out)
{
        unsigned char b;
        char c;
        int i;

        i=0;
        b=0;
        while(i<64) {
                b<<=4;
                c=instr[i];
                if(c==0)
                        return 0;
                if((c>='0') && (c<='9')) b+=(c-'0');
                else if((c>='a') && (c<='f')) b+=(c-'a'+10);
                else if((c>='A') && (c<='F')) b+=(c-'A'+10);
                if((i%2)==1) {
                        out[i/2]=b;
                        b=0;
                }
                i++;
        }
        return 1;
}

void encode_long (p_block pass, unsigned char j)
{
	int i;
	unsigned char index, shift;
	unsigned short temp;

	index = (pass[j] + pass[j+1]) & 0x3F; // 6 LSB
	shift = (pass[j+2] + pass[j+3]) & 0x7; // 3 LSB
		
	for (i = 0; i < pwdLength; ++i)
	{
		if (j == pwdLength) j = 0; // wrap around to beginning
		if (index == 64) index = 0; // wrap around to beginning

		temp = constantLong[index]; // xy
		temp <<= 8;
		temp |= constantLong[index]; // xyxy

		temp >>= shift;

		pass[j] ^= (unsigned char) temp;
		++j;
		++index;
	}
}

void hex_display (char *msg, unsigned char *data, unsigned long length)
{
    unsigned long i, n;

    fprintf(stdout, "\n%s", msg );

    while( length > 0 )
    {
        n = (length > 8) ? 8 : length;
        for( i = 0; i < n; i++ )
            fprintf(stdout, "0x%02X ", data[i] );

		while( i++ < 8 ) printf( "     " );

        fprintf(stdout, " [" );
        for( i = 0; i < n; i++, data++ )
            fprintf(stdout, "%c", (*data < 32 || *data > 127) ? '.' : *data );

        while( i++ < 8 ) fprintf(stdout, " " );
        fprintf(stdout, "]" );

        length -= n;
        if( length > 0 )
        {
            fprintf(stdout, "\n" );
            for( n = strlen(msg); n > 0; n-- ) fprintf(stdout, " " );
        }
    }
}

void usage (char *argv)
{
	fprintf(stdout, "\nUsage: %s -[e | d] <ASCII | password block>\n\n", argv);
	exit(0);
}
