from consts import *
import sys, struct

#The data dumped from the given physical address
g_translation_table_dump = None

#the physical base address of the dumped data
g_dump_phys_base = None

def read_qword(phys_addr):
  '''
  Reads a single DWORD from the given physical address (little endian).
  '''
  start_off = phys_addr - g_dump_phys_base
  return struct.unpack("<Q", g_translation_table_dump[start_off:start_off+8])[0]

def is_fld_block(fld): 
  '''
  Returns true IFF the given first-level descriptor is a block
  '''
  return (fld & 0b11) == 0b01

def is_fld_table(fld):
  '''
  Returns true IFF the given first-level descriptor is a table
  '''
  return (fld & 0b11) == 0b11

def is_sld_block(sld):
  '''
  Returns true IFF the given second-level descriptor is a block
  '''
  return (sld & 0b11) == 0b01

def is_sld_table(sld):
  '''
  Returns true IFF the given second-level descriptor is a table
  '''
  return (sld & 0b11) == 0b11

def dump_first_level_block(fld):
  '''
  Dumps the given first-level block descriptor
  '''
  phys_addr_start = fld & FLD_BLOCK_OA_MASK
  phys_addr_size = PHYS_ADDR_SIZE / NUM_FIRST_LEVEL_ENTRIES
  phys_addr_end = phys_addr_start + phys_addr_size
  s2ap = (fld >> S2AP_BIT_OFFSET) & 0b11
  xn = (fld >> XN_BIT_OFFSET) & 0b1
  print "0x%08x-0x%08x: AP=%s, XN=%d" % (phys_addr_start, phys_addr_end, bin(s2ap)[2:], xn)

def dump_second_level_block(sld):
  '''
  Dumps the given second-level block descriptor
  '''
  phys_addr_start = sld & SLD_BLOCK_OA_MASK
  phys_addr_size = PHYS_ADDR_SIZE / NUM_FIRST_LEVEL_ENTRIES / NUM_SECOND_LEVEL_ENTRIES
  phys_addr_end = phys_addr_start + phys_addr_size
  s2ap = (sld >> S2AP_BIT_OFFSET) & 0b11
  xn = (sld >> XN_BIT_OFFSET) & 0b1
  print "0x%08x-0x%08x: AP=%s, XN=%d" % (phys_addr_start, phys_addr_end, bin(s2ap)[2:], xn)

def dump_third_level_descriptor(tld):
  '''
  Dumps the given third-level descriptor
  '''
  phys_addr_start = tld & TLD_BLOCK_OA_MASK
  phys_addr_size = PHYS_ADDR_SIZE / NUM_FIRST_LEVEL_ENTRIES / NUM_SECOND_LEVEL_ENTRIES / NUM_THIRD_LEVEL_ENTRIES
  phys_addr_end = phys_addr_start + phys_addr_size
  s2ap = (tld >> S2AP_BIT_OFFSET) & 0b11
  xn = (tld >> XN_BIT_OFFSET) & 0b1
  print "0x%08x-0x%08x: AP=%s, XN=%d" % (phys_addr_start, phys_addr_end, bin(s2ap)[2:], xn)

def dump_second_level_table(sld):
  '''
  Dumps the given second-level table
  '''
  next_level_table_phys = sld & TABLE_OA_MASK
  for i in range(0, NUM_THIRD_LEVEL_ENTRIES):
    tld = read_qword(next_level_table_phys + i*QWORD_SIZE)

    #Is this a page?
    if tld & 0b11 == 0b11:
      dump_third_level_descriptor(tld)

def dump_first_level_table(fld):
  '''
  Dumps the given first-level table
  '''
  next_level_table_phys = fld & TABLE_OA_MASK
  for i in range(0, NUM_SECOND_LEVEL_ENTRIES):

    sld = read_qword(next_level_table_phys + i*QWORD_SIZE)

    #Is this a block?
    if is_sld_block(sld):
      dump_second_level_block(sld)

    #Is this another table?
    if is_sld_table(sld):
      dump_second_level_table(sld)

    #Otherwise - this isn't a valid descriptor, so skip

def dump_translation_table(vttbr):
  '''
  Dumps the translation table at the given physical address
  '''

  #Going over the translation tables and print out the result of walking each valid entry
  for first_level_idx in range(0, NUM_FIRST_LEVEL_ENTRIES):

    fld = read_qword(vttbr + first_level_idx*QWORD_SIZE)

    #Is this a block?
    if is_fld_block(fld):
      dump_first_level_block(fld)

    #Is this a table?
    elif is_fld_table(fld):
      dump_first_level_table(fld)

    #Otherwise - this isn't a valid descriptor, so skip

def main():

  global g_translation_table_dump
  global g_dump_phys_base

  #Reading the command-line arguments
  if len(sys.argv) != 3:
    print "USAGE: %s <TRANSLATION_TABLE_BINARY> 0x<DUMP_PHYS_BASE>" % sys.argv[0]
    return
  g_translation_table_dump = open(sys.argv[1], 'rb').read()
  g_dump_phys_base = int(sys.argv[2], 16)

  #Dumping all the data in the translation table
  dump_translation_table(g_dump_phys_base)

if __name__ == "__main__":
  main()
