
#include "stdafx.h"

// User text input limits
#define MIN_PASSWORD			1		// Minimum possible password length
#define MAX_PASSWORD			64		// Maximum possible password length
#define MAX_PIM				10		// Maximum allowed digits in a PIM (enough for 32-bit value)
#define MAX_BOOT_PIM			5		// Maximum allowed digits in a PIM for boot (enough for 16-bit value)
#define MAX_BOOT_PIM_VALUE	65535

#define PASSWORD_LEN_WARNING	20		// Display a warning when a password is shorter than this

typedef struct
{
	// Modifying this structure can introduce incompatibility with previous versions
	unsigned __int32 Length;
	unsigned char Text[MAX_PASSWORD + 1];
	char Pad[3]; // keep 64-bit alignment
} Password;

/* WARNING: Modifying the following values or their meanings can introduce incompatibility with previous versions. */

#define TC_IOCTL(CODE) (CTL_CODE (FILE_DEVICE_UNKNOWN, 0x800 + (CODE), METHOD_BUFFERED, FILE_ANY_ACCESS))

#define TC_IOCTL_MOUNT_VOLUME							TC_IOCTL (3)
#define TC_IOCTL_DISMOUNT_VOLUME						TC_IOCTL (4)

/* Start of driver interface structures, the size of these structures may
change between versions; so make sure you first send DRIVER_VERSION to
check that it's the correct device driver */

#define TC_MAX_PATH		260

#pragma pack (push)
#pragma pack(1)

typedef struct
{
	int nReturnCode;					/* Return code back from driver */
	BOOL FilesystemDirty;
	BOOL VolumeMountedReadOnlyAfterAccessDenied;
	BOOL VolumeMountedReadOnlyAfterDeviceWriteProtected;
	wchar_t wszVolume[TC_MAX_PATH];		/* Volume to be mounted */
	Password VolumePassword;			/* User password */
	BOOL bCache;						/* Cache passwords in driver */
	int nDosDriveNo;					/* Drive number to mount */
	ULONG BytesPerSector;
	BOOL bMountReadOnly;				/* Mount volume in read-only mode */
	BOOL bMountRemovable;				/* Mount volume as removable media */
	BOOL bExclusiveAccess;				/* Open host file/device in exclusive access mode */
	BOOL bMountManager;					/* Announce volume to mount manager */
	BOOL bPreserveTimestamp;			/* Preserve file container timestamp */
	BOOL bPartitionInInactiveSysEncScope;		/* If TRUE, we are to attempt to mount a partition located on an encrypted system drive without pre-boot authentication. */
	int nPartitionInInactiveSysEncScopeDriveNo;	/* If bPartitionInInactiveSysEncScope is TRUE, this contains the drive number of the system drive on which the partition is located. */
	BOOL SystemFavorite;
	// Hidden volume protection
	BOOL bProtectHiddenVolume;			/* TRUE if the user wants the hidden volume within this volume to be protected against being overwritten (damaged) */
	Password ProtectedHidVolPassword;	/* Password to the hidden volume to be protected against overwriting */
	BOOL UseBackupHeader;
	BOOL RecoveryMode;
	int pkcs5_prf;
	int ProtectedHidVolPkcs5Prf;
	BOOL bTrueCryptMode;
	ULONG BytesPerPhysicalSector;
	int VolumePim;
	int ProtectedHidVolPim;
} MOUNT_STRUCT;

typedef struct
{
	int nDosDriveNo;	/* Drive letter to unmount */
	BOOL ignoreOpenFiles;
	BOOL HiddenVolumeProtectionTriggered;
	int nReturnCode;	/* Return code back from driver */
} UNMOUNT_STRUCT;

#pragma pack(pop)


FARPROC GetProcAddressNT(LPCSTR lpName);

const int ProcessDeviceMap = 23;

typedef NTSTATUS(NTAPI *fNtSetInformationProcess)(
	IN HANDLE               ProcessHandle,
	IN int ProcessInformationClass,
	IN PVOID                ProcessInformation,
	IN ULONG                ProcessInformationLength);

typedef struct _PROCESS_DEVICEMAP_INFORMATION { // Information Class 23
	union {
		struct {
			HANDLE DirectoryHandle;
		} Set;
		struct {
			ULONG DriveMap;
			UCHAR DriveType[32];
		} Query;
	};
} PROCESS_DEVICEMAP_INFORMATION, *PPROCESS_DEVICEMAP_INFORMATION;


#include <string>
#include <strsafe.h>
#include "FileOpLock.h"
#include "CommonUtils.h"

void LockCallback(void* arg)
{
	CloseHandle(arg);
}

bool SetDosDevices(HANDLE hDir)
{
	fNtSetInformationProcess pfNtSetInformationProcess = (fNtSetInformationProcess)GetProcAddressNT("NtSetInformationProcess");

	NTSTATUS status = pfNtSetInformationProcess(GetCurrentProcess(), ProcessDeviceMap, &hDir, sizeof(hDir));

	printf("SetDosDevices: %08X\n", status);

	return status == 0;
}

bool MountVolume(HANDLE hDevice, int nDriveNo, const std::wstring& password, const std::wstring& volume)
{
	MOUNT_STRUCT mount = { 0 };
	
	StringCchPrintfA(reinterpret_cast<char*>(mount.VolumePassword.Text), 
		sizeof(mount.VolumePassword.Text), "%ls", password.c_str());
	mount.VolumePassword.Length = (unsigned int)strlen(reinterpret_cast<char*>(mount.VolumePassword.Text));

	mount.nDosDriveNo = nDriveNo;
	memcpy(mount.wszVolume, volume.c_str(), (volume.size() + 1) * sizeof(wchar_t));

	if (DeviceIoControl(hDevice, TC_IOCTL_MOUNT_VOLUME, &mount, sizeof(mount), &mount, sizeof(mount), nullptr, nullptr))
	{
		if (mount.nReturnCode == 0)
		{
			return true;
		}
		else
		{
			printf("Error in mount, driver: %d\n", mount.nReturnCode);
		}
	}
	else
	{
		printf("Error in mount: %d\n", GetLastError());
	}

	return false;
}

bool UmountVolume(HANDLE hDevice, int nDriveNo)
{
	UNMOUNT_STRUCT umount = { 0 };
	umount.ignoreOpenFiles = TRUE;
	umount.nDosDriveNo = nDriveNo;

	if (DeviceIoControl(hDevice, TC_IOCTL_DISMOUNT_VOLUME, 
		&umount, sizeof(umount), &umount, sizeof(umount), nullptr, nullptr))
	{
		return true;
	}
	else
	{
		printf("Error Umounting: %d\n", GetLastError());
		return false;
	}
}

std::wstring GetVolumePath(LPCWSTR path)
{
	WCHAR buf[MAX_PATH];

	if (GetFullPathName(path, MAX_PATH, buf, nullptr) == 0)
	{
		printf("Error getting full path %d\n", GetLastError());
		exit(1);
	}

	return buf;
}

std::wstring GetDosPath(const std::wstring& path)
{
	WCHAR buf[MAX_PATH];
	if (QueryDosDevice(path.substr(0, 2).c_str(), buf, MAX_PATH) == 0)
	{
		printf("Error getting DOS path %d\n", GetLastError());
		exit(1);
	}

	return buf;
}

int wmain(int argc, wchar_t** argv)
{
	if (argc < 4)
	{
		printf("Usage: drive: disk.hc password\n");
		return 1;
	}

	std::wstring drive = argv[1];
	std::wstring volpath = GetVolumePath(argv[2]);
	std::wstring password = argv[3];

	// Construct a path which doesn't use the drive symlink
	std::wstring volume = L"GLOBALROOT";
	volume += GetDosPath(volpath) + volpath.substr(2);

	if ((drive.length() != 2) || !isalpha(drive[0]) || drive[1] != ':')
	{
		printf("Drive must be in form of X:\n");
		return 1;
	}
	
	HANDLE hDriver = CreateFile(L"\\\\.\\VeraCrypt", GENERIC_READ | GENERIC_WRITE, 0, nullptr, OPEN_EXISTING, 0, 0);
	if (hDriver != INVALID_HANDLE_VALUE)
	{
		int nDriveNo = toupper(drive[0]) - 'A';
		HANDLE hDosDevice = CreateObjectDirectory(nullptr, nullptr, nullptr);
		HANDLE hFakeDrive = CreateObjectDirectory(hDosDevice, drive.c_str(), nullptr);

		FileOpLock* oplock = FileOpLock::CreateLock(volpath.c_str(), L"", LockCallback, hFakeDrive);
		if (oplock == nullptr)
		{
			return 1;
		}

		if (!SetDosDevices(hDosDevice))
		{
			return 1;
		}

		// Mount volume getting symlink into fake device map
		if (MountVolume(hDriver, nDriveNo, password, volume))
		{
			// Switch device map to be the global object directory
			if (!SetDosDevices(OpenObjectDirectory(nullptr, L"\\GLOBAL??")))
			{
				return 1;
			}

			// Umount which will delete the global symlink
			UmountVolume(hDriver, nDriveNo);

			// Remount as the new drive
			if (MountVolume(hDriver, nDriveNo, password, volume))
			{
				printf("Done\n");
				getchar();
				UmountVolume(hDriver, nDriveNo);
			}
		}
	}
	else
	{
		printf("Error opening device %d\n", GetLastError());
	}

    return 0;
}

