# Module containing common HTTP testing methods
# Author: Gary O'leary-Steele
"""
sec1httplib.requestbuilder

Request and Response classes for performing HTTP requests and building mulitstage
request sequences.


"""

import socket

# timeout in seconds
timeout = 120
socket.setdefaulttimeout(timeout)


import re
import urllib2
import urllib
import cookielib
import urlparse
import copy
import time
import threading
from .fuzzing import FuzzingToolsMixin


def p(req):
    print "----%s----" % req.__class__
    for x in req.__dict__:
        print x,"->" ,req.__dict__[x]

class CookieJar():
    """
    CookieJar class designed to maintain the global cookie jar and custom cookie
    jars allocated to individual requests objects.

    The global cookie jar is set to 1 on statup, all additional cookie jars
    are allocated an individual id on an incremental basis.

    Cookie jars are accessed via the Requestobj() instance method get_cookiejar()

    Setting per request cookie jars:

    Per request cookies can by initilised by calling Requestobj() instance method
    set_custom_cookie(). This function accepts three optional arguments cookiestr,
    copyjar and copyglobaljar.
    
    cookiestr       --  A cookie string. e.g "cookie=val; cookie2=val"
    copyjar         --  A cookielib cookiejar to copy
    copyglobaljar   --  A flag set to 1 or 0. Setting to 1 copies the global jar

    """
    cj_lock = threading.Lock()
    cookie_jars = []
    cookiejar_id = 1 # Start cookie jar id

    def __init__(self,url=""):
        """Create an instance and add it to CookieJar.cookie_jars[]. An id is
        allocated automatically.

        Each cookie jar is stored within CookieJar.cookie_jars[] as a dictionary
        with the following keys:

        jar     --      The cookielib cookie jar
        jarid   --      An ID value used to identify and retrieve cookie jars
        url     --      An optional value to state the URL the cookie jar is attached to
        """
        with CookieJar.cj_lock:
            cj = {"jar":cookielib.LWPCookieJar(),"jarid":CookieJar.cookiejar_id,"url":url}
            CookieJar.cookie_jars.append(cj)
            self.id = CookieJar.cookiejar_id
            CookieJar.cookiejar_id +=1

    @staticmethod
    def get_jar_byid(id):
        for jar in CookieJar.cookie_jars:
            if jar["jarid"] == id:
                return jar["jar"]

    def get_jar(self):
        for jar in CookieJar.cookie_jars:
            if jar["jarid"] == self.id:
                return jar["jar"]

    @staticmethod
    def set_jar_byid(id,newjar):
        "Replace a cookie jar by id."
        i=0
        for jar in CookieJar.cookie_jars:
            if jar["jarid"] == id:
                CookieJar.cookie_jars[i]["jar"] = newjar
            i += 1

    @staticmethod
    def reload_global():
        CookieJar.cookie_jars[0]["jar"] = cookielib.LWPCookieJar()

    @staticmethod
    def reload_byid(id):
        i=0
        for jar in CookieJar.cookie_jars:
            if jar["jarid"] == id:
                CookieJar.cookie_jars[i]["jar"] = cookielib.LWPCookieJar()
            i += 1


class DataAnalysisTools(object):
    """Tools for searching response data. Designed to be inherited into
    request/response objects.
    """

    def _get_querystring(self):
        "Return current querystring"
        return urlparse.urlsplit(self.url).query

    def _set_querystring(self,querystring):
        "set current querystring"
        url_parts = self.url.split("?")
        if not len(url_parts) <= 1:
            self.url = url_parts[0] + "?" + querystring

    def _get_querystring_parambyname(self,name):
        pass

    def _get_host(self):
        return urlparse.urlsplit(self.url).netloc

    def _get_path(self):
        return urlparse.urlsplit(self.url).path

    def _viewstate_encode(self,string_to_encode):
        string_to_encode = self._urlencode(string_to_encode)
        string_to_encode = string_to_encode.replace("/","%2f")
        return string_to_encode


    def _urlencode(self,string_to_encode,decode_first=1):
        """
        Url encode string.
        If the string is a list encode each string within the list
        """

        if isinstance(string_to_encode,str):
            if decode_first ==1:
                string_to_encode = urllib.unquote(string_to_encode)
            return urllib.quote(string_to_encode)

        elif isinstance(string_to_encode,list):
            encoded = []
            for s in string_to_encode:
                if decode_first ==1:
                    s = urllib.unquote(s)
                encoded.append(urllib.quote(s))
            return encoded

    def urlencode_to_dict(self,qs,delim="&"):
        "Convert urlencoded sting to a python Dict()"
        pairs = {}
        for pair in qs.split(delim):
            # avoid blank param pairs when theres a trailing &
            if pair =="":
                continue
            if "=" in pair:
                pairs.update(dict([pair.split("=")]))
            else:
                pairs.update({pair:""})
        return pairs

    def urlencode_to_list(self,qs,delim="&"):
        "Convert urlencoded sting to a python List()"
        pairs =[]
        for pair in qs.split(delim):
            # avoid blank param pairs when theres a trailing &
            if pair =="":
                continue
            if "=" in pair:
                pairs.append(tuple(pair.split("=")))
            else:
                pairs.append((pair,""))
        return pairs

    def is_json(self,value):
        """Detect if a value is JSON encoded data"""
        import json
        import urllib
        import re
        # Test to see if this is URL encoded JSON
        url_encoded_json = re.compile("^%5B.+%5D$|^%7B.+%7D$",re.IGNORECASE)
        if re.match(url_encoded_json,value):
            value = urllib.unquote(value)
        try:
            j = json.loads(value)
            if type(j) == type({}) or type(j) == type([]):
                return True
            else:
                return False
        except:
            return False


    def _get_host(self):
        "Return current host"
        return urlparse.urlsplit(self.url).netloc

    def _get_path(self):
        "Return current request path"
        return urlparse.urlsplit(self.url).path

    def get_queuelen(self):
        """Returns the request queue length. i.e Total length of the pre-request
        queue, exit-request queue + this request"""

        result = 1
        if hasattr(self,"prerequest_queue"):
            result = result + len(self.prerequest_queue)
        if hasattr(self,"exitrequest_queue"):
            result = result + len(self.exitrequest_queue)
        return result
    
    def get_total_responselength(self):
        "Return total body length all the requests responses in the request_history"
        l = 0
        
        for response in self.request_history:
            try:
                l += len(response.body)
            except:
                print "[i] Error getting length"
        return l

    def get_response_times(self):
        """
        Get the response time for each request in the request_history.
        Returns a Python list
        """

        times = []
        offset = 0
        for response in self.request_history:
            #print response.__class__,response.code
            
            time = response.request_time
            times.append({"request_history_offset":offset,"response_time":time,"url":response.url})
            offset = offset + 1
        return times

    def get_response_time(self):
        total_time = 0
        times = self.get_response_times()
        for time in times:
            total_time += time['response_time']
        return total_time



    def find_data(self,data,regex=1,look_in="mainbody|mainheaders|redirectbody|redirectheaders"):
        """
        Searches the body and headers in the current response and all responses
        encountered within redirects.

        Arguments:
        
        data    --  The regular expression or data string to search for.

        regex   --  Set to 1 if the search data passed to the "data" argument is
                    a regular expression. Set to 0 if its an exact string match (default: 1)
                    
        look_in --  A pipe (|) seperated list of areas to search within response
                    data to search (default: mainbody|mainheaders|redirectbody|redirectheaders)

        Examples:

        response.find_data("SQL Error")

        Returns a python list() of results. Each result item is a python dict() with the following keys "found_in","searched_data","findstr" and "url"

        """
        locations =[]
        findstr = re.compile(data,re.DOTALL | re.IGNORECASE)
        #A list of locations that the data was located
        locations =[]

        # Search Body, Headers in final destination
        if ("mainbody" in look_in) and (((regex==1) and findstr.search(self.body))
            or ((regex==0) and (data.lower() in self.body.lower()))):

            locations.append({"found_in":"mainbody","searched_data": self.body,"findstr": data,"url":self.geturl()})

        if ("mainheaders" in look_in) and (((regex==1) and findstr.search(str(self.headers)))
            or ((regex==0) and (data.lower() in str(self.headers).lower()))):

            locations.append({"found_in":"mainheaders","searched_data": str(self.headers),"findstr": data,"url":self.geturl()})

        # Search Body, Headers in redirect queue
        for redir_object in self.redir_history:

            if ("redirectbody" in look_in) and ((regex==1) and (findstr.search(redir_object.body))
                or ((regex==0) and (data.lower() in redir_object.body.lower()))):

                locations.append({"found_in":"redirectbody","searched_data": redir_object.body,"findstr": data,"url":redir_object.geturl()})

            if ("redirectheaders" in look_in) and ((regex==1) and (findstr.search(str(redir_object.headers)))
                or ((regex==0) and (data.lower() in str(redir_object.headers).lower()))):

                locations.append({"found_in":"redirectheaders","searched_data": str(redir_object.headers),"findstr": data,"url":redir_object.geturl()})

        return locations

    def find_data_in_request_history(self,matchdata,regex=1):
        """Performs a find_data() call for all response objects within the
        request_history.

        Use when you need to search all responses within a sequence.
        
        For more information see find_data() documentation"""


        # This function is available in both request and response objects
        found=[]
        for response in self.request_history:
            for findresult in response.find_data(matchdata,regex):
                found.append(findresult)
        return found


    def extractViewState(self):
        "Extract VIEWSTATE from the response"
        try:
            viewstate = re.findall("input type=\"hidden\" name=\"__VIEWSTATE\" id=\"__VIEWSTATE\" value=\"(.+?)\"",self.body)
            return viewstate[0]
        except:
            return ""

    def extract_all(self,regex):
        """Extract data based on a regular expression. Returns a list of results

        Arguments:

        regex       --      A regular expression to extract data.

        Example:

        The following extract_all() syntax will extract <a href> links
        response.extract_all("<a href=['\"](.+?)['\"]")
        """
        
        regex = re.compile(regex,re.DOTALL | re.IGNORECASE)
        data = []
        for response in self.request_history:
            data.extend(re.findall(regex,response.body))
            data.extend(re.findall(regex,str(response.headers)))
            for redir_response in response.redir_history:
                data.extend(re.findall(regex,redir_response.body))
                data.extend(re.findall(regex,str(redir_response.headers)))
        return data


    def extract_data_body(self,regex):
        """Exract data from the response body based on a regular expression passed as the first paramater"""
        try:
            regex = re.compile(regex,re.DOTALL | re.IGNORECASE)
            data = re.findall(regex,self.body)
            return data
        except Exception as err:
            print "Data Extraction Error",err


def create_basic_auth_opener(request_obj):
    "Create a urllib2 basic auth opener"
    passman = urllib2.HTTPPasswordMgrWithDefaultRealm()
    passman.add_password(None, request_obj.url, request_obj.credentials["username"], request_obj.credentials["password"])
    handler = urllib2.HTTPBasicAuthHandler(passman)
    # create "opener" (OpenerDirector instance)
    return handler


def execute_httpreq(requestobj_inst, cookielib="",disable_cookies=0):
    """Build urllib2.request and set relevant options. Called from Requestobj()
    instances via the makerequest() function"""

    proxy_handler = urllib2.ProxyHandler(Requestobj.proxy)
    
    cookielib = urllib2.HTTPCookieProcessor(cookielib)
    
    try:
        #store a list of openers in here, start with defaults of redir and cookies
        if disable_cookies ==0:
            openers = [urllib2.HTTPHandler,RedirectHandler,cookielib,proxy_handler]
        else:
       
            openers = [urllib2.HTTPHandler,RedirectHandler,proxy_handler]

        if requestobj_inst.credentials:
            openers.append(create_basic_auth_opener(requestobj_inst))

        req = urllib2.Request(requestobj_inst.url)
        
        req.follow_redirects = requestobj_inst.follow_redirects

        opener = urllib2.build_opener(*openers)

        #Set method
        req.get_method = lambda: requestobj_inst.method

        # Set headers
        for header in requestobj_inst.headers.items():
            req.add_header(*header)

        if requestobj_inst.postdata and requestobj_inst.method !="GET":
            req.add_data(requestobj_inst.postdata)

        # Make request and return response object
        if Requestobj.verbose == 1:
            print "[i] Sending %s request to %s" %(req.get_method(),req.get_full_url())

        
        resp = opener.open(req)
        
        request_response  = Responseobj(base=resp)
        
        
    except urllib2.HTTPError as err:
        if err.code != 500 and err.code !=404 and err.code !=401 and err.code !=403:
            print "HTTP Error:%s" % err.code
        
        if hasattr(err,"read") and hasattr(err,"code") and hasattr(err,"url") and hasattr(err,"headers"):
            return Responseobj(base=err)
        else:
            ob = Responseobj(code=err.code,status="HTTP Error: %s" % err.msg)
            return Responseobj(base=ob)

    except urllib2.URLError as err:

        print "URLError Occured: %s" % err
        ob = Responseobj(code=500,body="",status="URL Error: %s" % err.reason)

        ob.comunication_error = 1
        return ob

    except Exception as err:
        print "[i] Execute_http error", err
        
        ob = Responseobj(code=500,body="",status="Misc HTTP Error")
        return Responseobj(base=ob)

    return request_response


def checkscope(host):
    """Check the scope of the redirect before following"""
    return True


class Responseobj(DataAnalysisTools):
    """A class to hold response data and perform common analysis functions.

    Arguments:
    body    --  HTTP Response body
    code    --  HTTP Response code
    headers --  HTTP Response headers
    url     --  HTTP Request URL
    base    --  URLLib2 reponse objecct to copy properties from
    """
    def __init__(self,*args,**kargs):
        self.body = kargs.get("body","")
        self.code = kargs.get("code",0)
        self.headers = kargs.get("headers","")
        self.url = kargs.get("url","")
        self.request = "" # Here we store a reference to the origional request
        self.comunication_error = 0
        self.request_time = 0

        self.redir_history = []

        # If we pass in a urllib.addinfourl copy its attributes
        self.baseobj = kargs.get("base",None)

        if self.baseobj:
            self.process_urllib2responseobject(self.baseobj)
            # Now clear this due to deepcopy issues
            self.baseobj = None

    def response_length(self):
        try:
            return self.get_total_responselength()
        except Exception as err:
            print "Error executing self.get_total_responselength()",err
            return len(self.body)
        

    def geturl(self):
        """Return the URL"""
        try:
            return self.url
        except:
            return "No URL"

    def getcode(self):
        "Return HTTP response code"
        try:
            return self.code
        except:
            return 0

    def process_urllib2responseobject(self,response):
        """Process the response from urllib2 and populate common properites such as body, redirect_history,code,url and headers"""
        
        try:
            try:
                self.body = response.read()
            except:
                # Pass and leave as "" or the body defined within the constructor
                pass

            self.redir_history = getattr(response,"redir_history",[])
            self.code = response.code
            self.url = response.url
            self.headers = response.headers

        except Exception as err:
            print "[i] process_urllib2responseobject error",err
            print response.code
    

class RedirectHandler(urllib2.HTTPRedirectHandler):
    """By default urllib2 will follow redirects. This class overides that behaviour
    each response is stored within the request_history that is attached to the final returned response object.
    """
    def redirector(self, req, fp, code, msg, headers):
        #req will be a Request object,
        #fp will be a file-like object with the HTTP error body, code will
        #be the three-digit code of the error,
        #msg will be the user-visible explanation of the code and
        #headers will be a mapping object with the headers of the error.

        # Redirect target
        location = headers.get('Location','')
        # where it come from:
        origin = req._Request__original
        if Requestobj.verbose == 1:
            print "[i] Redirecting:", origin,"->",location

        # Build response object to store in redirect history
        
        resp = Responseobj(base=fp)

        if not hasattr(req,"follow_redirects"):
            req.follow_redirects = "scope"

        if req.follow_redirects == "scope" and checkscope(location):
            pass
        elif req.follow_redirects == "never":
            # Return the 302 response and dont follow it
            return fp
        else:
            pass
        
        # Continue on.
        
        result = urllib2.HTTPRedirectHandler.http_error_301(self, req, fp, code, msg, headers)
        result.status = code

        # add redir to history
        if hasattr(result,"redir_history"):
            result.redir_history = result.redir_history + [resp]
            
        else:
            result.redir_history = [resp]
            
        
        return result
    
    # set all redirects codes to use the same handler
    http_error_301 = http_error_302 = http_error_303 = http_error_307 = redirector



class Requestobj(DataAnalysisTools,FuzzingToolsMixin):

    CookieJar() # Init the global cookiejar

    verbose = 1
    proxy = {}

    def __init__(self,*args,**kargs):
        self.args = args
        self.kargs = kargs
        self.url = self.args[0]
        self.cookiejar_id=1 # 1 is the global cookie jar
        self.dynamic_cookiemods = []    # List to hold dynamic cookie modifiers
        self.disable_cookies=0          # Set to 1 to disable cookies
        self.credentials = ""
        self.follow_redirects = "scope" #options are: scope | onsite | all

        try:
            self.path = urlparse.urlparse(self.url).path
        except:
            self.path = "/"
        
        # Fuzzer notes
        self.fuzz_notes = ""
        
        self.fuzz_pos = 0
        self.fuzzed_param = ""

        self.querystring = self._get_querystring()

        self.prerequest_queue = []
        self.exitrequest_queue = []
        self.content_type="application/x-www-form-urlencoded"

        # Set this to detect a string which will indicate a failiure
        # THIS SETTING MUST BE ENABLED ON EACH INDIVIDUAL REQUEST
        self.indicate_fail_regex=""
        # optionally execute this function on fail. Gets passed self
        self.execute_on_fail = ""
        # Maximum number of retries as a result of a fail
        self.maxretries = 1
        self.failed = 0 # Set this to 1 if we cant resolve the fail

        #Function to execute before each request is executed
        self.execute_before_request=""

        self.start_time = 0
        self.end_time = 0
        #self.request_time = 0
        
        self.method = "GET"
        self.postdata = ""

        # used for CSRF extractor
        self.extract_from_previous = []
        self.extract_from_previous_GET = []
        self.dontprocess = [] # dont process these pages for csrf
        self.request_history = []

        # Some windows apps want to see a valid viewstate
        self.viewstate = ""

        # Set default headers
        self.headers = {'User-Agent':'Mozilla/4.0 (compatible; MSIE 8.0; Windows NT 6.1; WOW64; Trident/4.0; GTB6.5; SLCC2; .NET CLR 2.0.50727; .NET CLR 3.5.30729; .NET CLR 3.0.30729; Media Center PC 6.0; MDDC)'}
        
        # Track __VIEWSTATE etc for mult request objects
        param = "__VIEWSTATE"
        regex = "input type=\"hidden\" name=\"__VIEWSTATE\" id=\"__VIEWSTATE\" value=\"(.+?)\""
        self.add_extract_from_previous_regex({param:regex})
        
        param = "__EVENTVALIDATION"
        regex = "input type=\"hidden\" name=\"__EVENTVALIDATION\" id=\"__EVENTVALIDATION\" value=\"(.+?)\""
        self.add_extract_from_previous_regex({param:regex})

    
    def set_custom_cookie(self,cookiestr="",copyjar="",copyglobaljar=0):
        """
        Set a custom cookie for this request object.

        Attributes:

        cookiestr       --      Optional string to create the cookie
        copyjar         --      Provide an existing cookiejar to copy from
        copyglobaljar   --      Copy the global cookiejar

        """
        self.cookiejar_id = CookieJar(url=self.url).id
        if cookiestr:
            self.set_cookie_fromstr(cookiestr)

        if copyjar:

            for c in copyjar:
                self.get_cookiejar().set_cookie(c)

        if copyglobaljar:
            for c in CookieJar.get_jar_byid(1):
                self.get_cookiejar().set_cookie(c)

        return self.cookiejar_id
    
    def get_cookiejar(self):
        return CookieJar.get_jar_byid(self.cookiejar_id)

    def set_dynamic_cookiemod(self,cookie_name,mod_value,append=1,replace=0):
        """Set a dynamic cookie modifier. This is implemented at runtime and
        modifies a cookie with a given value
        """
        self.dynamic_cookiemods.append({"name":cookie_name,"mod_value":mod_value,
                                        "append":append,"replace":replace})


    def apply_dynamic_cookiemod(self):
        "Return a cookie lib object witha a modified cookie"
        if self.dynamic_cookiemods:
            # Make a copy of the cookie jar
            tmp_cookiejar = copy.copy(self.get_cookiejar())
            for cookie in tmp_cookiejar:
                for cookiemod in self.dynamic_cookiemods:
                    if cookie.name.lower() == cookiemod["name"].lower():
                        #print "Modifying cookie value {0}".format(cookie.name.lower())
                        if cookiemod["append"] == 1:
                            cookie.value = cookie.value + cookiemod["mod_value"]
                        else:
                            cookie.value = cookiemod["mod_value"]

            return tmp_cookiejar

        else:
            # No dynamic cookie modifiers set
            return False


    @staticmethod
    def set_proxy(ip,port):
        p = '{0}:{1}'.format(ip,port)
        Requestobj.proxy = {'http':p,'https':p}



    def __len__(self):
        return len(self.prerequest_queue) + len(self.exitrequest_queue) + 1


    def return_request_queue_list(self):
        requests = []
        for pre in self.prerequest_queue:
            requests.append({"method":pre.method,"url":pre.url})
        requests.append({"method":self.method,"url":self.url})
        for xit in self.exitrequest_queue:
            requests.append({"method":xit.method,"url":xit.url})
        return requests

    def set_cookie_fromstr(self,cookiestr):
        """Set cookies from a cookie string e.g.: cookie1=a; cookie2=b"""
        cookies = cookiestr.split(";")
        for cookie in cookies:
            try:
                cookiesandvalues=cookie.split("=")
                name=cookiesandvalues[0].strip()
                if len(cookiesandvalues) > 1:
                    value = "=".join(cookiesandvalues[1:]).strip()
                else:
                    value = ""
            except Exception as err:
                 print "[i] Set Cookie error",err
                 
            if name:
                #print "Setting name:%s value:%s" % (name,value)
                self.set_cookie(name,value)


    def reload_cookie(self):
        "Reload the cookie jar for this Requestobj. Not the global"
        if self.cookiejar_id != 1:
            CookieJar.reload_byid(self.cookiejar_id)
        else:
            print "[i] The global cookie jar must be set through CookieJar.reload_global()"

    def set_cookie(self,cookiename,cookievalue):
        """
        Sets cookies via the cookielib cookie jar
        accepts cookiename and cookievalue as params
        """
        try:
            import Cookie
        except:
            print "Import 'Cookie' error"

        parsed_url = urlparse.urlparse(self.url)
        urldomain = parsed_url.netloc

        c = cookielib.Cookie(0,name=cookiename,
                            value=cookievalue,
                            port=None,
                            port_specified=False,
                            domain=urldomain,
                            domain_specified=False,
                            domain_initial_dot=False,
                            path="/",
                            path_specified=True,
                            secure=False,
                            expires=None,
                            discard=True,
                            comment=None,
                            comment_url=None,
                            rest={},
                            rfc2109=False)
        # Sets the local cookie jar or the global one within the object
        
        self.get_cookiejar().set_cookie(c)

       
    def basic_auth(self,username,password):

        """
        passman = urllib2.HTTPPasswordMgrWithDefaultRealm()
        passman.add_password(None, self.url, username, password)
        handler = urllib2.HTTPBasicAuthHandler(passman)
        # create "opener" (OpenerDirector instance)
        self.openers.append(handler)
        """

        self.credentials = {"username":username,"password":password}


    def rawpostdata(self,method,data,content_type="application/x-www-form-urlencoded"):
        """Add POST data.requires method rawpostData('POST','a=b&c=d')
        Arguments:

        method      --  The HTTP request method
        data        --  The HTTP post data
        content_type       --  The HTTP content-type (Default: application/x-www-form-urlencoded)
        """

        self.setmethod(method)
        self.postdata=data
        self.content_type=content_type
        self.add_header("Content-Type",content_type)
        
        

    def postdata(self,data):
        "Accepts a dict and creates url encoded post data"
        if isinstance(data,dict):
            data = urllib.urlencode(data)
        self.rawpostdata("POST",data)
        

    def put_file(self,filedata):
        """
        Convert request object to a PUT request
        and upload data.
        """
        self.method = 'PUT'
        self.rawpostdata(self.method,filedata)

    def add_header(self,header,value):
        self.headers[header]=value
        
        
    def set_headers(self,req):
        for header in self.headers.items():
            req.add_header(*header)

    def add_prerequest(self,requestob):
        """Adds a request object to execute before the main request"""
        #Make a copy
        #cp = copy.deepcopy(requestob)
        cp = copy.deepcopy(requestob)
        self.prerequest_queue.append(cp)

    def add_exitrequest(self,requestob):
        """Adds a request object to execute after the main request"""
        cp = copy.deepcopy(requestob)
        self.exitrequest_queue.append(cp)

    def setmethod(self,method):
        self.method = method
        
    def copy_basic_attributes(self,obj):
        # copy the following attributes from the given object into the current
        # one. Implemented to copy basic request data when fuzzing

        copyme = ['headers','postdata','method','prerequest_queue','exitrequest_queue']
        for attr in copyme:
            # skip built in.
            if "__" in attr:
                continue
            setattr(self,attr,getattr(obj,attr))


    def _viewstate_encode(self,string_to_encode):
        string_to_encode = self._urlencode(string_to_encode)
        string_to_encode = string_to_encode.replace("/","%2f")
        return string_to_encode

    def _replace(self,find,replace,delim="&"):
        if self.content_type=="application/x-www-form-urlencoded":
            postdata = self.urlencode_to_dict(self.postdata)
            for key,data in postdata.items():
                postdata[key]=self._urlencode(data.replace(find,replace))
            ps = delim.join(["=".join(x) for x in postdata.items()])
            self.postdata = ps
        else:
            self.postdata = self.postdata.replace(find,replace)

        # Replace querystring
        querystring = self._get_querystring()
        pairs = self.urlencode_to_dict(querystring)
        for key,data in pairs.items():
            pairs[key]=self._urlencode(data.replace(find,replace))
        new_qs = delim.join(["=".join(x) for x in pairs.items()])
        if "?" in self.url:
            url = self.url.split("?")[0]
            self.url = url + "?" + new_qs

    def replace_all(self,find,replace):

        for pre in self.prerequest_queue:
            pre._replace(find,replace)
        self._replace(find,replace)
        for xit in self.exitrequest_queue:
            xit._replace(find,replace)
        


    def _urlencode(self,string_to_encode,decode_first=1):
        """
        Url encode string.
        If the string is a list encode each string within the list
        """
        
        if isinstance(string_to_encode,str):
            if decode_first ==1:
                string_to_encode = urllib.unquote(string_to_encode)
            return urllib.quote(string_to_encode)

        elif isinstance(string_to_encode,list):
            encoded = []
            for s in string_to_encode:
                if decode_first ==1:
                    s = urllib.unquote(s)
                encoded.append(urllib.quote(s))
            return encoded


    def update_tuple_param(self,param_list,param_name,new_value):
        """Update paramater values after query string has been decoded into a
        python list of tuples [(param,value),(param2,value2)]"""
        for param in range(len(param_list)):
            if param_list[param][0] == param_name:
                param_list[param] = (param_list[param][0],new_value)
        return param_list

    def update_querystring(self,param,value,delim="&",encode=1):
        "Update a query string paramater value with a given value"
        try:
            paramList = self.urlencode_to_list(self._get_querystring())
            if encode ==1:
                self.update_tuple_param(paramList,param_name=param,new_value=self._urlencode(value))
            else:
                self.update_tuple_param(paramList,param_name=param,new_value=value)
            self._set_querystring(delim.join(["=".join(x) for x in paramList]))
            return "Success"
        except Exception as err:
            print "[i] Update Querystring Data Error", err
            return "Failed: %s " % err



    def update_postdata(self,param,value,delim="&",encode=1):
        "Update a post data paramater value with a given value"
        try:
            paramList = self.urlencode_to_list(self.postdata)
            # Dont need to check, we may be adding
            #if param not in paramDict:
            #    return "Failed: Could not find param in post data"
            if encode ==1:
                if re.match("__VIEW|__EVENT",param):
                    self.update_tuple_param(paramList,param_name=param,new_value=self._viewstate_encode(value))
                    #paramDict[param] = self._viewstate_encode(value)
                else:
                    self.update_tuple_param(paramList,param_name=param,new_value=self._urlencode(value))
            else:
                self.update_tuple_param(paramList,param_name=param,new_value=value)

            self.postdata = delim.join(["=".join(x) for x in paramList])
            return "Success"
        except Exception as err:
            print "[i] Update Post Data Error", err
            return "Failed: %s " % err


    def reset_redircount(self):
        try:
            #print self.redirect_dict
            self.redirect_dict = {}
        except:
            pass


    def execute_prerequests(self):
        """
        Sends requests within the prerequest_queue before the main
        request.
        """
        #Send pre-requests if there are any
        if len(self.prerequest_queue) > 0:
            if Requestobj.verbose == 1:
                print "[i] Executing Prerequest queue. %s requests" % len(self.prerequest_queue)
        for prereq in self.prerequest_queue:
            if Requestobj.verbose == 1:
                print "[->] Executing Pre-Request:%s" % prereq.url
            
            # If we have set "extract_from_previous" then lets check the history
            #prereq.update_data_from_previous(self.request_history)
            prereq.request_history = self.request_history

            pre_response = prereq.makerequest(clear_history=0)
            #self.request_history.append(pre_response)
            #self.viewstate = pre_response.extractViewState()
        return pre_response


    def update_data_from_previous(self,request_history,walk_request_history=1,onsite_only=1):
        """Update the current request with data from a previous request or external function.

        This function is designed to update the current Query String or Form Data based on
        the web servers response to a previous request or the return value of an external
        function.


        This function is not generally called directly. To add a lookback regular expression
        or callback function use add_extract_from_previous_regex() and add_extract_from_previous_callback()
        functions.

        Arguments:

        request_history         --  A Python list containing responses for each request in the sequence
        walk_request_history    --  When set to 1 the regex or function is applied recursivley until there is a match
                                    or when all items within the request_history have been checked.
        onsite_only             --  When set to 1, only content from the same host is parsed.
        
        """
        d = None

        # If we have set "extract_from_previous" then lets check the history
        if len(self.extract_from_previous) > 0:# and len(request_history) > 0:
        
            for update_dict in self.extract_from_previous:
                param_to_update = update_dict[0]
                data_extracter_function = update_dict[1]
                method = update_dict[2]
                try:
                    if len(request_history) > 0:
                        
                        self.headers["Referer"] = request_history[-1].url

                        # See if we should walk the request history or just
                        # the previous request. If we have ajax crap in there
                        # its best to walk back. onsite_only specifies that the
                        # the host name of the previous request must match the
                        # current request. i.e. dont search for tokens within
                        # requests to external sites
                        if walk_request_history == 0:
                            d = data_extracter_function(previous=request_history[-1],current=self)

                        elif walk_request_history ==1:
                            # Cycle back through the domains
                            for req in request_history[-1::-1]:
                                if onsite_only==1:
                                    #print req._get_host()
                                    #print self._get_host()
                                    if not checkscope(req._get_host()):
                                        if Requestobj.verbose == 1:
                                            print "[i] Skipping analysis of {0} for lookback data. Does not match:{1}".format(req._get_host(),self._get_host())
                                            d = None
                                        continue # Skip if the host is different
                                #print "Walking History"
                                d = data_extracter_function(previous=req,current=self)
                                if d:
                                    break
                                #if param_to_update =="":
                                #    break
                    else:
                        #print "called",update_dict
                        d = data_extracter_function(previous=Responseobj(code=200,body="",status="Blank"),current=self)
                        

                    if d and param_to_update and method =="POST":
                        if param_to_update:
                            self.update_postdata(param_to_update ,d)

                    if d and param_to_update and method =="GET":
                        qs = self.urlencode_to_dict(self.url.split("?")[1])
                        qs[param_to_update] = d
                        newqs = "&".join(["=".join(x) for x in qs.items()])
                        self.url = self.url.split("?")[0] + "?" + newqs

                except Exception as err:
                #except urllib2.URLError as err:
                    print "[i] update_data_from_previous error",err
                    print self.request_history
                    #print p(req)

        return self

    def add_extract_from_previous_regex(self,extract_dict,targetmethod="POST"):
        """
        Add a regular expression to extract data from the previous request
        when using a pre-request or exit request queue.
        Accepts a dict.e.g. {"updatepostparam":regex}
        Useful for maintaining CSRF tokens stored in the body
        """

        for param_to_update,regex_pattern in extract_dict.items():
            self.extract_from_previous.append((param_to_update,self.data_extract_function_factory(regex_pattern),targetmethod))
            


    def add_extract_from_previous_callback(self,extract_dict,targetmethod="POST"):
        """Add a callback function to extract data from the previous request
        when using a pre-request or exit request queue.

        Accepts a dict.e.g. {"paramtoupdate":function}

        The function will be supplied with two key word paramaters:

        previous        --      The response object from the previous request.
        current         --      The current requst object

        By specifying "" as the paramater to update no params are update but
        the callback is still executed.

        See data_extract_function_factory for an example.

        Function should return a single string value which will then be supplied
        within the supplied paramater value.

        If the function returns nothing the function will be called response
        objects recursivley (moving backwards) until all responses have been
        passed in or we run out of responses.

        """

        for param_to_update,funct in extract_dict.items():
            self.extract_from_previous.append((param_to_update,funct,targetmethod))
            



    def data_extract_function_factory(self,regex):
        """
        Factory function to generate simple data extractor functions
        """
        regex = re.compile(regex,re.DOTALL | re.IGNORECASE)
        def extractor(previous=None,current=None):
            data = re.findall(regex,previous.body)
            if data:
                return data[0]
       
        return extractor


    def makerequest(self,retries=0,clear_history=1):
        try:
            # If we have specified a function to execute before the request
            # then execute it here
            if self.execute_before_request:
                self.execute_before_request()

            # Reset the history before executing the request queue
            if clear_history==1:
                self.request_history = []


            #execute pre requests
            if len(self.prerequest_queue) > 0:
                #print "Executing Pre Req Q"
                self.execute_prerequests()

            # Update data from previous request or callback if applicable
            self.update_data_from_previous(self.request_history)

            
            timerstart = time.clock()

            # If we have set dynamic cookie mods then obtain
            # custom cookie jar.
            custom_cookielib = self.apply_dynamic_cookiemod()

            if custom_cookielib:
                request_response = execute_httpreq(self,cookielib=custom_cookielib,disable_cookies=self.disable_cookies)    
            else:
                request_response = execute_httpreq(self,cookielib=self.get_cookiejar(),disable_cookies=self.disable_cookies)
            
            request_response.request_time = time.clock() - timerstart

            # If we have a valid response put it in the queue
            if request_response:
                self.request_history.append(request_response)

            if len(self.exitrequest_queue) > 0:
                self.execute_exitrequests()
            
        except Exception as err:
            print "[i] A makerequest error occured:",err


        # Append reference to request queue in response
        request_response.request_history = self.request_history
      
        # Add a pointer to the request object within the response
        request_response.request = self

        # Check to see if we have indicate fail enabled and retry if applicable
        
        if self.indicate_fail_regex and request_response.find_data(self.indicate_fail_regex):
            print "[i] Indicate Fail Patterm Matched attempting retry:%s" % retries
            if self.maxretries > retries:
                # Optionally execute a function on fail
                if self.execute_on_fail:
                    self.execute_on_fail(self)
                # Retry the request
                self.makerequest(retries+1)
            else:
                # If we have exceeded the max retry count then make as failed
                print "[i] Failed"
                request_response.failed = 1

        return request_response


    def execute_exitrequests(self):
        # set the default response
        exit_response = Responseobj(body="No Response")

        for exitreq in self.exitrequest_queue:
            #Set Viewstate if we need to
            #exitreq.update_data_from_previous(self.request_history)
            exitreq.request_history = self.request_history
            if Requestobj.verbose == 1:
                print "[->] Exit request:%s" % exitreq.url
            
            exit_response = exitreq.makerequest(clear_history=0)
            #self.request_history.append(exit_response)
            #self.viewstate = exit_response.extractViewState()
        if exit_response:
            return exit_response
        else:
            return Responseobj(body="No Response")
        # if request queue is empty
        #return Responseobj(body="No Response")



if __name__ == "__main__":
    a = Requestobj("http://www.sec-1.com/")
    x = a.makerequest()
    p(Requestobj.cj)
