package vulnerability

import (
	log "github.com/sirupsen/logrus"
	"github.com/urfave/cli/v2"
)

// Vulnerability command flags
const (
	// Shared Flags
	FlagCVE      = "cve"
	FlagCVEUsage = "Target vulnerability CVE ID"

	// EPSS Flags
	FlagDate      = "date"
	FlagDateUsage = "Date for the EPSS information (YYYY-MM-DD format)"

	FlagOp      = "op"
	FlagOpUsage = "EPSS operation ('lookup' | 'list')"

	FlagWithHistory      = "with-history"
	FlagWithHistoryUsage = "Return EPSS results with historical data"

	FlagLimit      = "limit"
	FlagLimitUsage = "Limit the number of returned records"

	FlagOffset      = "offset"
	FlagOffsetUsage = "Offset where to start returning records"

	FlagFilterCveIDPattern      = "filter-cve-id-pattern"
	FlagFilterCveIDPatternUsage = "'CVE ID pattern' ESPP list operation filter"

	FlagFilterDaysSinceAdded      = "filter-days-since-added"
	FlagFilterDaysSinceAddedUsage = "'days since added' ESPP list operation filter"

	FlagFilterScoreGt      = "filter-score-gt"
	FlagFilterScoreGtUsage = "'score is greater than' ESPP list operation filter"

	FlagFilterScoreLt      = "filter-score-lt"
	FlagFilterScoreLtUsage = "'score is less than' ESPP list operation filter"

	FlagFilterPercentileGt      = "filter-percentile-gt"
	FlagFilterPercentileGtUsage = "'percentile is greater than' ESPP list operation filter"

	FlagFilterPercentileLt      = "filter-percentile-lt"
	FlagFilterPercentileLtUsage = "'percentile is less than' ESPP list operation filter"

	FlagFilterOrderRecords      = "filter-order-records"
	FlagFilterOrderRecordsUsage = "'order returned records' ESPP list operation filter ('score-desc' | 'score-asc' | 'percentile-desc' | 'percentile-asc')"
)

const (
	EpssOpLookup = "lookup"
	EpssOpList   = "list"
)

var Flags = map[string]cli.Flag{
	FlagCVE: &cli.StringSliceFlag{
		Name:    FlagCVE,
		Value:   cli.NewStringSlice(),
		Usage:   FlagCVEUsage,
		EnvVars: []string{"DSLIM_VULN_CVE"},
	},
	FlagDate: &cli.StringFlag{
		Name:    FlagDate,
		Value:   "",
		Usage:   FlagDateUsage,
		EnvVars: []string{"DSLIM_VULN_EPSS_DATE"},
	},
	FlagOp: &cli.StringFlag{
		Name:    FlagOp,
		Value:   EpssOpLookup,
		Usage:   FlagOpUsage,
		EnvVars: []string{"DSLIM_VULN_EPSS_OP"},
	},
	FlagWithHistory: &cli.BoolFlag{
		Name:    FlagWithHistory,
		Value:   false, //defaults to false
		Usage:   FlagWithHistoryUsage,
		EnvVars: []string{"DSLIM_VULN_EPSS_HISTORY"},
	},
	FlagLimit: &cli.Uint64Flag{
		Name:    FlagLimit,
		Value:   10,
		Usage:   FlagLimitUsage,
		EnvVars: []string{"DSLIM_VULN_LIMIT"},
	},
	FlagOffset: &cli.Uint64Flag{
		Name:    FlagOffset,
		Value:   0,
		Usage:   FlagOffsetUsage,
		EnvVars: []string{"DSLIM_VULN_OFFSET"},
	},
	FlagFilterCveIDPattern: &cli.StringFlag{
		Name:    FlagFilterCveIDPattern,
		Value:   "",
		Usage:   FlagFilterCveIDPatternUsage,
		EnvVars: []string{"DSLIM_VULN_EPSS_FILTER_CVEID_PAT"},
	},
	FlagFilterDaysSinceAdded: &cli.UintFlag{
		Name:    FlagFilterDaysSinceAdded,
		Value:   0,
		Usage:   FlagFilterDaysSinceAddedUsage,
		EnvVars: []string{"DSLIM_VULN_EPSS_FILTER_DAYS_SINCE"},
	},
	FlagFilterScoreGt: &cli.Float64Flag{
		Name:    FlagFilterScoreGt,
		Value:   0,
		Usage:   FlagFilterScoreGtUsage,
		EnvVars: []string{"DSLIM_VULN_EPSS_FILTER_SCORE_GT"},
	},
	FlagFilterScoreLt: &cli.Float64Flag{
		Name:    FlagFilterScoreLt,
		Value:   0,
		Usage:   FlagFilterScoreLtUsage,
		EnvVars: []string{"DSLIM_VULN_EPSS_FILTER_SCORE_LT"},
	},
	FlagFilterPercentileGt: &cli.Float64Flag{
		Name:    FlagFilterPercentileGt,
		Value:   0,
		Usage:   FlagFilterPercentileGtUsage,
		EnvVars: []string{"DSLIM_VULN_EPSS_FILTER_PERC_GT"},
	},
	FlagFilterPercentileLt: &cli.Float64Flag{
		Name:    FlagFilterPercentileLt,
		Value:   0,
		Usage:   FlagFilterPercentileLtUsage,
		EnvVars: []string{"DSLIM_VULN_EPSS_FILTER_PERC_LT"},
	},
	FlagFilterOrderRecords: &cli.StringFlag{
		Name:    FlagFilterOrderRecords,
		Value:   "",
		Usage:   FlagFilterOrderRecordsUsage,
		EnvVars: []string{"DSLIM_VULN_EPSS_FILTER_ORDER"},
	},
}

func cflag(name string) cli.Flag {
	cf, ok := Flags[name]
	if !ok {
		log.Fatalf("unknown flag='%s'", name)
	}

	return cf
}
