/***************************************************************
 * Name:      engine.h
 * Author:    David Vachulka (arch_dvx@users.sourceforge.net)
 * Copyright: 2013
 * License:   GPL3
 **************************************************************/

#ifndef ENGINE_H
#define ENGINE_H

#ifndef WX_PRECOMP
#include <wx/wx.h>
#endif
#include "event.h"
#include "holiday.h"
#include "calendarday.h"
#include "dxdefs.h"
#include "wx/wxsqlite3.h"
#include <algorithm>
#include <vector>
#include <map>

wxDECLARE_EVENT(ENGINE_UPDATED, wxCommandEvent);
wxDECLARE_EVENT(ENGINE_MSG, wxCommandEvent);

#define iengine Engine::instance()

class Engine
{
public:
    explicit Engine(wxWindow *parent) : m_parent(parent), m_tooltipPlugin(_("No reminders today")), m_lastid(-1)
    {
        m_this = this;
    }
    virtual ~Engine();

    static Engine* instance();
    bool loadData();
    void loadCalendarData(bool showhidden);
    bool overwriteEvents(const wxString& filename, bool encypted, const wxString &key);
    bool mergeEvents(const wxString& filename, bool encypted, const wxString &key);
    bool overwriteHolidays(const wxString& filename, bool encypted, const wxString &key);
    bool overwriteNote(const wxString& filename, bool encypted, const wxString &key);
    void clearDB();
    void reEncrypt();
    wxInt64 addEvent(const Event& event, bool fireUpdate = true);
    void editEvent(const Event& event);
    void editEvent(wxInt64 id, const wxDateTime& date);
    void editEvent(wxInt64 id, wxInt64 reminder);
    void editEvent(wxInt64 id, const wxString &name);
    void removeEvent(wxInt64 id, bool fireUpdate = true);
    void hideEvent(wxInt64 id, bool fireUpdate = true);
    void unhideEvent(wxInt64 id, bool fireUpdate = true);
    bool someEvents();
    bool someDonotdeleteEvent(bool showhidden);
    void removeEvents();
    void recurrentEvent(wxInt64 id, bool fireUpdate = true);
    wxDateTime recurrentEventDate(const Event &event, const wxDateTime &rdate);
    Event getEvent(wxInt64 id);
    wxInt64 getEventReminder(wxInt64 id);
    int getEventRecurrence(wxInt64 id);
    std::vector<Event> eventsByReminder(wxDateTime time);
    bool someEventsByReminder(wxDateTime time);
    void setEventReminded(wxInt64 id, bool reminded = true);
    std::vector<Event> eventsForCalendarDate(const wxDateTime& date);
    bool hasEventsForCalendarDate(const wxDateTime& date);
    bool hasRecurrence(wxInt64 id);
    std::vector<Event> mergeEvents() const;
    wxString noteText();
    void updateNote(const wxString &text);
    bool mergeAllEvents(const wxString &filename, bool encypted, const wxString &key);
    std::vector<Holiday> holidays();
    void setHolidays(const std::vector<Holiday> &holidays);
    Holiday addHoliday(const wxString &name, int day, wxDateTime::Month month);
    bool isHoliday(const wxDateTime &date);
    bool someHolidayToFire();
    wxString holidayTextsForDate();
    wxString holidayTipTexts(const wxDateTime &date);
    void setHolidayReminded(wxInt64 id);
    std::vector<EventView> eventViews(bool onlyhidden, bool showhidden);
    wxDateTime plusBusinessDay(const wxDateTime &eventday, wxDateTime::wxDateTime_t days);
    wxDateTime montlyAtGivenBusinessDay(const wxDateTime &eventday, wxDateTime::wxDateTime_t day);
    wxDateTime weeklyAtGivenBusinessDay(const wxDateTime &eventday, wxDateTime::wxDateTime_t day);
    wxString getTooltipPlugin() const;
    bool someEventOnHiddenList();
    bool isEventOnHiddenList(wxInt64 id);
    void removeEventFromHiddenList(wxInt64 id);
    void clearListOfHiddenEvents();
    bool someEventOnHighlightList();
    int eventHighlightColor(wxInt64 id);
    void removeEventFromHighlightList(wxInt64 id);
    void clearListOfHighlightEvents();
    void highlightEvent(wxInt64 id, int color, bool fireUpdate = true);
    void alwaysshowEvent(wxInt64 id, bool always, bool fireUpdate = true);
    bool isNonworkingDay(const wxDateTime &date);
    wxDateTime nextWorkingDay(const wxDateTime &date);
    void fireUpdated(bool updateToolTip = true);
    void fireMsg(const wxString &msg);
    std::vector<Calendarday> updateCalendarDays(wxDateTime::Month month, int year, bool showhidden);
    void tooltipPluginForDate();
    bool isRecurrentedEventForDate(const wxDateTime &date, const Event &event);
    /**
     * @brief getLastid
     * @return last added or edited event ID
     */
    wxInt64 getLastid() const;
    void setLastid(const wxInt64 &lastid);

private:
    static Engine* m_this;
    wxWindow *m_parent;
    wxSQLite3Database m_db;
    std::vector<Event> m_events;
    std::vector<Event> m_mergeEvents;
    wxString m_tooltipPlugin;
    wxInt64 m_lastid;

    void createEventsTable();
    void createNoteTable();
    bool hasTableColumn(wxSQLite3Database *db, const wxString& name, const wxString& column);
    void tryExecuteUpdate(wxSQLite3Database *db, const wxString& sql);
    void createHolidaysTable();
    bool isMonthQuarterly(wxDateTime::Month month, wxDateTime::Month eventMonth);
    wxDateTime nextQuarterDay(wxDateTime::wxDateTime_t eventDay, wxDateTime::Month eventMonth);
    bool isWeekly(wxDateTime::WeekDay eventWeekDay, wxDateTime::WeekDay dateWeekDay);
    bool isWeeklyBusiness(wxDateTime::WeekDay eventWeekDay, const wxDateTime &date);
    bool isWeeklyPlusBusinessDay(const wxDateTime &event, const wxDateTime &date, wxDateTime::WeekDay dateWeekDay, int monthlyday);
    bool isWeeklyGivenBusinessDay(const wxDateTime &event, const wxDateTime &date, int monthlyday);
    bool is2Weekly(const wxDateTime &event, const wxDateTime &date);
    bool is2WeeklyBusiness(const wxDateTime &event, wxDateTime::WeekDay eventWeekDay, const wxDateTime &date);
    bool is2WeeklyPlusBusinessDay(const wxDateTime &event, const wxDateTime &date, wxDateTime::WeekDay dateWeekDay, int monthlyday);
    bool is2WeeklyGivenBusinessDay(const wxDateTime &event, const wxDateTime &date, int monthlyday);
    bool isMontly(wxDateTime::wxDateTime_t eventDay, wxDateTime::wxDateTime_t dateDay);
    bool isOwn(const wxDateTime &event, const wxDateTime &date, int days, int months, int hours, int minutes, bool all=false);
    bool isMonthlyAtDay(const wxDateTime &event, const wxDateTime &date, int monthlyday, int monthlyweek);
    bool isMontlyBusiness(const wxDateTime &event, const wxDateTime &date, int eventDay);
    bool isMontlyPlusBusinessDay(const wxDateTime &event, const wxDateTime &date, int eventDay, int monthlyday);
    bool isMontlyGivenBusinessDay(const wxDateTime &event, const wxDateTime &date, int monthlyday);
    bool isQuarterlyBusiness(const wxDateTime &event, const wxDateTime &date, int eventDay, wxDateTime::Month eventMonth);
    bool isQuarterlyPlusBusinessDay(const wxDateTime &event, const wxDateTime &date, int eventDay, int monthlyday, wxDateTime::Month eventMonth);
    bool isQuarterlyGivenBusinessDay(const wxDateTime &event, const wxDateTime &date, int monthlyday,  wxDateTime::Month eventMonth);
    bool isYearly(const wxDateTime &event, const wxDateTime &date, int eventDay);
    bool isQuarterly(const wxDateTime &event, const wxDateTime &date, int eventDay, wxDateTime::Month eventMonth);
    bool checkTableEvents(wxSQLite3Database *db);
    bool checkTableHolidays(wxSQLite3Database *db);
    bool checkTableNote(wxSQLite3Database *db);
    wxArrayString tableColumns(wxSQLite3Database *db, const wxString &name);
};

#endif // ENGINE_H
