# CMake code required to properly build Qt projects.

if(CMAKE_BINARY_DIR PATH_EQUAL CMAKE_SOURCE_DIR)
	message(FATAL_ERROR
		"Build is in the same directory as the sources. Source tree will become heavily polluted.\n"
		"Delete the CMakeCache.txt file and CMakeFiles folder and use one of the supplied presets."
	)
endif()

if(CMAKE_GENERATOR STREQUAL "Ninja Multi-Config")
	set(CMAKE_DEFAULT_BUILD_TYPE "Release")
endif()
set(CMAKE_CXX_STANDARD_REQUIRED ON)
set(CMAKE_CXX_EXTENSIONS OFF)

# Ensure autogenerated UI files can find source headers
set(CMAKE_INCLUDE_CURRENT_DIR ON)

# Enable compile_commands.json export for IDEs and tools
set(CMAKE_EXPORT_COMPILE_COMMANDS ON)
# Diagnostics decorations
add_compile_options(
	$<$<CXX_COMPILER_ID:Clang>:-fcolor-diagnostics>
	$<$<CXX_COMPILER_ID:GNU>:-fdiagnostics-color=always>
)

cmake_language(DEFER CALL build_deferred_setup)

function(build_compilation_setup target)
	set(is_release $<OR:$<CONFIG:Release>,$<CONFIG:RelWithDebInfo>,$<CONFIG:MinSizeRel>>)
	set(CMAKE_CXX_FLAGS_DEBUG "")
	set(CMAKE_CXX_FLAGS_RELEASE "")

	target_compile_definitions(${target} PRIVATE
		$<${is_release}:NDEBUG>
	)
	target_compile_options(${target} PRIVATE
		-Wall
		-Wextra
		-Wpedantic
		-O2
		$<$<OR:$<CONFIG:Debug>,$<CONFIG:RelWithDebInfo>>:-gdwarf-4>
		$<${is_release}:
			$<$<CXX_COMPILER_ID:Clang>:-flto=thin>
			$<$<CXX_COMPILER_ID:GNU>:-flto=auto -fno-fat-lto-objects>
		>
	)
	target_link_options(${target} PRIVATE
		-O2
		$<${is_release}:
			$<$<CXX_COMPILER_ID:Clang>:-flto=thin>
			$<$<CXX_COMPILER_ID:GNU>:-flto=auto -fno-fat-lto-objects>
		>
	)
endfunction()

function(build_version_definition target definition)
	# Get version from debian/changelog
	execute_process(
		COMMAND dpkg-parsechangelog -SVersion
		WORKING_DIRECTORY ${CMAKE_SOURCE_DIR}
		OUTPUT_VARIABLE pkg_version
		OUTPUT_STRIP_TRAILING_WHITESPACE
		RESULT_VARIABLE dpkg_result
	)
	if((NOT dpkg_result EQUAL 0) OR (pkg_version STREQUAL ""))
		message(WARNING "Failed to get version from debian/changelog using dpkg-parsechangelog")
		set(pkg_version "${PROJECT_VERSION}")
	endif()
	if(pkg_version STREQUAL "")
		message(FATAL_ERROR "Cannot add ${definition} compile definition: PROJECT_VERSION empty")
	endif()
	target_compile_definitions(${target} PRIVATE
		${definition}="${pkg_version}"
	)
endfunction()

function(build_translation_files target ts_dir)
	file(GLOB TRANSLATION_FILES ${ts_dir}/*.ts)
	qt_add_translations(${target}
		TS_FILES ${TRANSLATION_FILES}
		LRELEASE_OPTIONS -compress -nounfinished -removeidentical -silent
		QM_FILES_OUTPUT_VARIABLE qm_files # Stop qm files being embedded into the executable.
	)
endfunction()

function(build_deferred_setup)
	# Cannot be run before project() call
	if(CMAKE_INSTALL_PREFIX_INITIALIZED_TO_DEFAULT)
		set_property(CACHE CMAKE_INSTALL_PREFIX PROPERTY VALUE "/usr")
	endif()
endfunction()
