/*
 * Copyright (c) 1998,1999 Michael Elizabeth Chastain.
 * Copyright (c) 2001 Christoph Hellwig.
 * All rights resered.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#define _GNU_SOURCE

#include "config.h"

#include <sys/utsname.h>
#include <sys/stat.h>

#include <errno.h>
#include <getopt.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "mconfig.h"
#include "parser.tab.h"


static const char *seed = NULL;

#if defined(HAVE_LIBCURSES)
argument_type argument = {
	.version = VERSION,
	.mode =	mode_menu,
};
#else
argument_type argument = {
	.version = VERSION,
	.mode =	mode_line,
};
#endif

/*
 *	Save original command for output.
 */
static void
copy_cmdline(int argc, char **argv)
{
	char *cmdline;
	int iarg;
	int len = 1;

	for (iarg = 0; iarg < argc; ++iarg)
		len = len + 1 + strlen(argv[iarg]);

	cmdline = grab_memory(len);
	cmdline[0] = '\0';

	for (iarg = 0; iarg < argc; ++iarg) {
		if (iarg > 0)
			strcat(cmdline, " ");
		strcat(cmdline, argv[iarg]);
	}

	argument.cmdline = cmdline;
}

/*
 *	Print usage information.
 */
static void
print_usage(char **argv)
{
	printf("Usage: %s: options ...\n", argv[0]);
	printf("  -h,          --help\n");
	printf("  -V,          --version\n");
	printf("  -m MODE,     --mode MODE\n");
	printf("     modes: line old syntax random mutate "
#if defined(HAVE_LIBCURSES)
		"menu "
#endif
		"minimum maximum\n");
	printf("  -a ARCH,     --architecture ARCH\n");
	printf("  -s SEED,     --seed SEED\n");
	printf("  -t TITLE,    --title TITLE\n");
	printf("  -S DIR,      --dir-source DIR\n");
	printf("  -T DIR,      --dir-target DIR\n");
	printf("  -I FILE,     --file-config-in FILE\n");
	printf("  -O FILE,     --file-config-out FILE\n");
	printf("  -A FILE,     --file-autoconf-h FILE\n");
};

/*
 *	Print version information.
 */
static void
print_version(char **argv)
{
	printf("%s version %s\n", PACKAGE, VERSION);
}

/*
 *	Set host architecture.
 */
static void
set_host_architecture()
{
	struct utsname buf;

	if (uname(&buf) != 0)
		error_system_call("uname");

	if (buf.machine[0] == 'i' && buf.machine[1] != '\0' &&
	    buf.machine[2] == '8' && buf.machine[3] == '6' &&
	    buf.machine[4] == '\0')
		argument.arch = "i386";
	else
		argument.arch = strdup(buf.machine);
}

/*
 *	Make a nice default title.
 */
static void
set_default_title()
{
	const char *title;
	char version[64] = "";
	char patchlevel[64] = "";
	char sublevel[64] = "";
	char extra[64] = "";
	char name[64] = "";
	char *filename;
	FILE *fp;
	char *buf;

	title = "Linux Kernel Configuration (unknown version)";

	filename = grab_memory(strlen(argument.ds) + 16 + 1);
	sprintf(filename, "%s%s", argument.ds, "Makefile");
	fp = fopen(filename, "r");
	if (fp == NULL) {
		argument.title = title;
		return;
	}

	if (fscanf(fp, "VERSION = %63s\n", version) == 1
	    && fscanf(fp, "PATCHLEVEL = %63s\n", patchlevel) == 1
	    && fscanf(fp, "SUBLEVEL = %63s\n", sublevel) == 1) {
		/* EXTRAVERSION is optional */
		if (fscanf(fp, "EXTRAVERSION =%*[ ]%63s", extra) != 1)
			extra[0] = '\0';
	}
	if (fclose(fp) != 0)
		error_system_call(filename);

	sprintf(filename, "%s%s", argument.ds, ".name");
	fp = fopen(filename, "r");
	if (fp != NULL) {
		if (fgets(name, 63, fp) != name)
			error_system_call(filename);
		if (fclose(fp) != 0)
			error_system_call(filename);
		if (name[0] != '\0' && name[strlen(name) - 1] == '\n')
			name[strlen(name) - 1] = '\0';
	}
	buf = grab_memory(48 + strlen(argument.arch) + strlen(version) +
			strlen(patchlevel) + strlen(sublevel) +
			strlen(extra) + strlen(name));
	sprintf(buf, "Linux/%s v%s.%s.%s%s%s%s Kernel Configuration",
		argument.arch, version, patchlevel, sublevel, extra,
		name[0] ? "-" : "", name);

	argument.title = buf;
}

/*
 *	Set default config.in file.
 */
static void
set_default_config()
{
	size_t len = strlen(argument.ds) + strlen(argument.arch) + 16;
	char *name = grab_memory(len);

	sprintf(name, "%sarch/%s/config.in", argument.ds, argument.arch);
	argument.fs = name;
}

/*
 *	Set default defaults file.
 */
static void
set_default_configin()
{
	char *name1 = grab_memory(strlen(argument.dt) + 32);
	struct stat stat_buf;

	sprintf(name1, "%s.config", argument.dt);
	if (stat(name1, &stat_buf) == 0 || errno != ENOENT) {
		argument.fci = name1;
	} else {
		size_t len = strlen(argument.ds) + strlen(argument.arch) + 32;
		char *name2 = grab_memory(len);

		sprintf(name2, "%sarch/%s/defconfig",
				argument.ds, argument.arch);

		if (stat(name2, &stat_buf) == 0 || errno != ENOENT)
			argument.fci = name2;
	}
}

/*
 *	Set default output file.
 */
static void
set_default_configout()
{
	char *name = grab_memory(strlen(argument.dt) + 8);

	sprintf(name, "%s.config", argument.dt);
	argument.fco = name;
}

/*
 *	Set default autoconf header.
 */
static void
set_default_autoconf()
{
	char *name = grab_memory(strlen(argument.dt) + 32);

	sprintf(name, "%sinclude/linux/autoconf.h", argument.dt);
	argument.fac = name;
}

/*
 *	Parse -m/--mode arguments.
 */
static void
parse_mode(char *optarg)
{
	static struct {char *arg; int val;} modetab[] = {
#if defined(HAVE_LIBCURSES)
		{"menu", mode_menu},
#endif
		{"line", mode_line},
		{"old", mode_old},
		{"syntax", mode_syntax},
		{"random", mode_random},
		{"mutate", mode_mutate},
		{"minimum", mode_minimum},
		{"maximum", mode_maximum},
		{NULL, 0}
	};
	int i;

	if (optarg == NULL || optarg[0] == '\0')
		error_exit("mode name required");

	for (i = 0; modetab[i].arg; i++) {
		if (strncmp(optarg, modetab[i].arg, strlen(optarg)))
			continue;
		argument.mode = modetab[i].val;
		return;
	}

	error_exit("unknown mode");
}

/*
 *	Parse -a/--architecture arguments.
 */
static void
parse_architecture(char *optarg)
{
	if (optarg == NULL || optarg[0] == '\0')
		error_exit("architecture name required");

	argument.arch = grab_string(optarg, strlen(optarg));
}

/*
 *	Parse -s/--seed arguments.
 */
static void
parse_seed(char *optarg)
{
	char *endptr;
	long lseed;

	if (optarg == NULL || optarg[0] == '\0')
		error_exit("seed required");

	seed = grab_string(optarg, strlen(optarg));
	argument.useed = lseed = strtol(seed, &endptr, 0);

	if (argument.useed != lseed || *endptr != '\0')
		error_exit("invalid seed");
}

/*
 *	Parse -t/--title arguments.
 */
static void
parse_title(char *optarg)
{
	if (optarg == NULL || optarg[0] == '\0')
		error_exit("title required");

	argument.title = grab_string(optarg, strlen(optarg));
}

/*
 *	Parse -S/--dir-source arguments.
 */
static void
parse_dirsource(char *optarg)
{
	int len = strlen(optarg);
	char *ptr = grab_memory(len + 2);

	if (optarg == NULL || optarg[0] == '\0')
		error_exit("directory name required");
	
	memcpy(ptr, optarg, len + 1);
	if (len > 0 && ptr[len] != '/') {
		ptr[len] = '/';
		ptr[len + 1] = '\0';
	}
	
	argument.ds = ptr;
}

/*
 *	Parse -T/--dir-target arguments.
 */
static void
parse_dirtarget(char *optarg)
{
	int len = strlen(optarg);
	char *ptr = grab_memory(len + 2);

	if (optarg == NULL || optarg[0] == '\0')
		error_exit("directory name required");
	
	memcpy(ptr, optarg, len + 1);
	if (len > 0 && ptr[len] != '/') {
		ptr[len] = '/';
		ptr[len + 1] = '\0';
	}
	
	argument.dt = ptr;
}

/*
 *	Parse -I/--config-in arguments.
 */
static void
parse_configin(char *optarg)
{
	if (optarg == NULL || optarg[0] == '\0')
		error_exit("file name required");

	argument.fci = grab_string(optarg, strlen(optarg));
}

/*
 *	Parse -O/--config-out arguments.
 */
static void
parse_configout(char *optarg)
{
	if (optarg == NULL || optarg[0] == '\0')
		error_exit("file name required");

	argument.fco = grab_string(optarg, strlen(optarg));
}

/*
 *	Parce -C/--config arguments.
 */
static void
parse_config(char *optarg)
{
	if (optarg == NULL || optarg[0] == '\0')
		error_exit("file name required");
	
	argument.fs = grab_string(optarg, strlen(optarg));
}

/*
 *	Parse -A/--autoconf-h arguments.
 */
static void
parse_autoconf(char *optarg)
{
	if (optarg == NULL || optarg[0] == '\0')
		error_exit("file name required");
	argument.fac = grab_string(optarg, strlen(optarg));
}

static const char opts[] = "hVm:a:s:t:S:T:I:O:A:";

static const struct option longopts[] = {
	{"help", 0, NULL, 'h'},
	{"version", 0, NULL, 'V'},
	{"mode", 1, NULL, 'm'},
	{"architecture", 1, NULL, 'a'},
	{"seed", 1, NULL, 's'},
	{"title", 1, NULL, 't'},
	{"dir-source", 1, NULL, 'S'},
	{"dir-target", 1, NULL, 'T'},
	{"config-in", 1, NULL, 'I' },
	{"config-out", 1, NULL, 'O'},
	{"config", 1, NULL, 'C'},
	{"autoconf-h", 1, NULL, 'A'},
	{NULL, 0, NULL, 0}
};

static void
argument_parse(int argc, char **argv)
{
	int c, index;

	/*
         * Set the command name.  This is first so that error reports come
         * out nice.  And yes, I am paranoid here, I want a nice message if
         * the very first grab_string fails; and I want to call grab_string
         * because getopt_long may modify the argv block later.
         */
	argument.cmdname = argv[0];
	argument.cmdname = grab_string(argv[0], strlen(argv[0]));

	copy_cmdline(argc, argv);
	
	while ((c = getopt_long(argc, argv, opts, longopts, &index)) != EOF) {
		switch (c) {
		case 'h':
			print_usage(argv);
			exit(0);
		case 'V':
			print_version(argv);
			exit(0);
		case 'm':
			parse_mode(&optarg[0]);
			break;
		case 'a':
			parse_architecture(&optarg[0]);
			break;
		case 's':
			parse_seed(&optarg[0]);
			break;
		case 't':
			parse_title(&optarg[0]);
			break;
		case 'S':
			parse_dirsource(&optarg[0]);
			break;
		case 'T':
			parse_dirtarget(&optarg[0]);
			break;
		case 'I':
			parse_configin(&optarg[0]);
			break;
		case 'O':
			parse_configout(&optarg[0]);
			break;
		case 'C':
			parse_config(&optarg[0]);
			break;
		case 'A':
			parse_autoconf(&optarg[0]);
			break;
		case ':':
		case '?':
			exit(1);
		default:
			error_internal("argument_parse: getopt_long failed");
		}
	}

	if (optind + 1 == argc) {
		argument.fci = argv[optind];
		argument.fco = argv[optind];
	} else if (optind < argc)
		error_exit("extra arguments not allowed");

	/* arch defaults to host architecture */
	if (argument.arch == NULL)
		set_host_architecture();

	/* ds defaults to current directory */
	if (argument.ds == NULL)
		argument.ds = "";

	/* dt defaults to $ds */
	if (argument.dt == NULL)
		argument.dt = argument.ds;

	/* title defaults to a nice title */
	if (argument.title == NULL)
		set_default_title();

	/* seed defaults to something random */
	if (seed == NULL)
		/* FIXME: use /dev/random, /dev/urandom */
		argument.useed = time(NULL) + getpid();

	/* fs defaults to $ds/arch/$arch/config.in */
	if (argument.fs == NULL)
		set_default_config();

	/* fci defaults to $dt/.config if it exists followed by
	   $ds/$ARCH/defconfig if it exists.  */
	if (argument.fci == NULL)
		set_default_configin();

	/* fco defaults to $dt/.config */
	if (argument.fco == NULL)
		set_default_configout();

	/* fac defaults to $dt/include/linux/autoconf.h */
	if (argument.fac == NULL)
		set_default_autoconf();

	/* turn off some options in some non-interactive modes */
	switch (argument.mode) {
	case mode_random:
	case mode_minimum:
	case mode_maximum:
		argument.fci = NULL;
		argument.no_backup = 1;
	}
}


int 
main(int argc, char *argv[])
{
	block_type *block_top = NULL;

	/*
         * Parse arguments.
         * Initialize symbol table.
         */
	argument_parse(argc, argv);
	symbol_table_init(argument.arch);

	/*
         * Load the config.in file (and everything it sources).
         */
	if (argument.fs) {
		const char     *error_string;

		if (input_push_file(argument.fs, &error_string) < 0)
			error_exit(error_string);

		parser_magic_cookie = COOKIE_CONFIG_LANGUAGE;
		if (yyparse() != 0 || parser_error_count > 0) {
			char *buf = grab_memory(strlen(argument.fs) + 32);
			sprintf(buf, "%s: failed to parse", argument.fs);

			if (argument.mode != mode_syntax)
				error_exit(buf);
			else
				exit(1);
		}
		if (parser_warning_count > 0 && argument.mode != mode_syntax)
			 /* sleep( 2 ) */ ;

		block_top = parser_block_top;
	}

	/*
         * Load old values from $file-config-in.
         */
	if (argument.fci == NULL)
		 /* load_emptyconfig( block_top ) */ ;
	else
		load_defconfig(block_top, argument.fci);



	/*
         * Run the major mode.
         */
	switch (argument.mode) {
#if defined(HAVE_LIBCURSES)
	case mode_menu:
		do_mode_menu(block_top);
		break;
#endif
	case mode_line:
		do_mode_line(block_top);
		break;
	case mode_old:
		do_mode_old(block_top);
		break;
	case mode_syntax:
		do_mode_syntax(block_top);
		break;
	case mode_random:
		do_mode_random(block_top);
		break;
	case mode_mutate:
		do_mode_mutate(block_top);
		break;
	case mode_minimum:
		do_mode_minimum(block_top);
		break;
	case mode_maximum:
		do_mode_maximum(block_top);
		break;
	default:
		error_enum_bogus();
		break;
	}

	return 0;
}
