/**************************************************** */
/* Rule Set Based Access Control                      */
/* Implementation of the Access Control Decision      */
/* Facility (ADF) - System Resources (RES)            */
/* File: rsbac/adf/res/main.c                         */
/*                                                    */
/* Author and (c) 2002: Amon Ott <ao@rsbac.org>       */
/*                                                    */
/* Last modified: 22/Nov/2002                         */
/**************************************************** */

#include <linux/string.h>
#include <rsbac/types.h>
#include <rsbac/aci.h>
#include <rsbac/adf_main.h>
#include <rsbac/error.h>
#include <rsbac/helpers.h>
#include <rsbac/getname.h>
#include <rsbac/debug.h>

/************************************************* */
/*           Global Variables                      */
/************************************************* */

/************************************************* */
/*          Internal Help functions                */
/************************************************* */

/************************************************* */
/*          Externally visible functions           */
/************************************************* */

enum rsbac_adf_req_ret_t
   rsbac_adf_request_res (enum  rsbac_adf_request_t     request,
                                rsbac_pid_t             caller_pid,
                          enum  rsbac_target_t          target,
                          union rsbac_target_id_t       tid,
                          enum  rsbac_attribute_t       attr,
                          union rsbac_attribute_value_t attr_val,
                                rsbac_uid_t             owner)
  {
    union rsbac_target_id_t       i_tid;
    union rsbac_attribute_value_t i_attr_val1;

    switch (request)
      {
        case R_MODIFY_ATTRIBUTE:
            switch(attr)
              {
                case A_system_role:
                case A_auth_role:
                case A_res_min:
                case A_res_max:
                #ifdef CONFIG_RSBAC_RES_AUTH_PROT
                case A_auth_may_setuid:
                case A_auth_may_set_cap:
                case A_auth_add_f_cap:
                case A_auth_remove_f_cap:
                #endif
                /* All attributes (remove target!) */
                case A_none:
                  /* Security Officer? */
                  i_tid.user = owner;
                  if (rsbac_get_attr(RES,
                                     T_USER,
                                     i_tid,
                                     A_res_role,
                                     &i_attr_val1,
                                     TRUE))
                    {
                      printk(KERN_WARNING
                             "rsbac_adf_request_res(): rsbac_get_attr() returned error!\n");
                      return(NOT_GRANTED);
                    }
                  /* if sec_officer, then grant */
                  if (i_attr_val1.system_role == SR_security_officer)
                    return(GRANTED);
                  else
                    return(NOT_GRANTED);

                default:
                  return(DO_NOT_CARE);
              }

        case R_READ_ATTRIBUTE:
            switch(attr)
              {
                case A_system_role:
                case A_auth_role:
                case A_res_min:
                case A_res_max:
                /* All attributes (remove target!) */
                case A_none:
                  /* Security Officer or Admin? */
                  i_tid.user = owner;
                  if (rsbac_get_attr(RES,
                                     T_USER,
                                     i_tid,
                                     A_res_role,
                                     &i_attr_val1,
                                     TRUE))
                    {
                      printk(KERN_WARNING
                             "rsbac_adf_request_res(): rsbac_get_attr() returned error!\n");
                      return(NOT_GRANTED);
                    }
                  /* if sec_officer, then grant */
                  if(   (i_attr_val1.system_role == SR_security_officer)
                     || (i_attr_val1.system_role == SR_administrator)
                    )
                    return(GRANTED);
                  else
                    return(NOT_GRANTED);

                default:
                  return(DO_NOT_CARE);
              }

        case R_SWITCH_LOG:
            switch(target)
              {
                case T_NONE:
                  /* test owner's res_role */
                  i_tid.user = owner;
                  if (rsbac_get_attr(RES,
                                     T_USER,
                                     i_tid,
                                     A_res_role,
                                     &i_attr_val1,
                                     TRUE))
                    {
                      printk(KERN_WARNING "rsbac_adf_request_res(): rsbac_get_attr() returned error!\n");
                      return(NOT_GRANTED);
                    }
                  /* security officer? -> grant  */
                  if (i_attr_val1.system_role == SR_security_officer)
                    return(GRANTED);
                  else
                    return(NOT_GRANTED);

                /* all other cases are unknown */
                default: return(DO_NOT_CARE);
              }

        case R_SWITCH_MODULE:
            switch(target)
              {
                case T_NONE:
                  /* we need the switch_target */
                  if(attr != A_switch_target)
                    return(UNDEFINED);
                  /* do not care for other modules */
                  if(   (attr_val.switch_target != RES)
                     #ifdef CONFIG_RSBAC_RES_AUTH_PROT
                     && (attr_val.switch_target != AUTH)
                     #endif
                     #ifdef CONFIG_RSBAC_SOFTMODE
                     && (attr_val.switch_target != SOFTMODE)
                     #endif
                    )
                    return(DO_NOT_CARE);
                  /* test owner's res_role */
                  i_tid.user = owner;
                  if (rsbac_get_attr(RES,
                                     T_USER,
                                     i_tid,
                                     A_res_role,
                                     &i_attr_val1,
                                     TRUE))
                    {
                      printk(KERN_WARNING "rsbac_adf_request_res(): rsbac_get_attr() returned error!\n");
                      return(NOT_GRANTED);
                    }
                  /* security officer? -> grant  */
                  if (i_attr_val1.system_role == SR_security_officer)
                    return(GRANTED);
                  else
                    return(NOT_GRANTED);

                /* all other cases are unknown */
                default: return(DO_NOT_CARE);
              }


/*********************/
        default: return DO_NOT_CARE;
      }

    return(DO_NOT_CARE);
  }; /* end of rsbac_adf_request_res() */


/*****************************************************************************/
/* If the request returned granted and the operation is performed,           */
/* the following function can be called by the AEF to get all aci set        */
/* correctly. For write accesses that are performed fully within the kernel, */
/* this is usually not done to prevent extra calls, including R_CLOSE for    */
/* cleaning up.                                                              */
/* The second instance of target specification is the new target, if one has */
/* been created, otherwise its values are ignored.                           */
/* On success, 0 is returned, and an error from rsbac/error.h otherwise.     */

int  rsbac_adf_set_attr_res(
                      enum  rsbac_adf_request_t     request,
                            rsbac_pid_t             caller_pid,
                      enum  rsbac_target_t          target,
                      union rsbac_target_id_t       tid,
                      enum  rsbac_target_t          new_target,
                      union rsbac_target_id_t       new_tid,
                      enum  rsbac_attribute_t       attr,
                      union rsbac_attribute_value_t attr_val,
                            rsbac_uid_t             owner)
  {
    union rsbac_target_id_t       i_tid;
    union rsbac_attribute_value_t i_attr_val1;

    switch (request)
      {
        case R_CHANGE_OWNER:
            switch(target)
              {
                case T_PROCESS:
                  if(attr != A_owner)
                    return(-RSBAC_EINVALIDATTR);
                  /* Adjust Linux resources */
                  i_tid.user = attr_val.owner;
                  #ifdef CONFIG_RSBAC_SOFTMODE
                  if(!rsbac_softmode)
                  #endif
                    {
                      int maxval = rsbac_min(RLIM_NLIMITS - 1, RSBAC_RES_MAX);
                      int i;

                      if (rsbac_get_attr(RES,
                                         T_USER,
                                         i_tid,
                                         A_res_max,
                                         &i_attr_val1,
                                         FALSE))
                        {
                          printk(KERN_WARNING
                                 "rsbac_adf_set_attr_res(): rsbac_get_attr() for res_user_full returned error!\n");
                          return -RSBAC_EREADFAILED;
                        }
                      for(i = 0; i <= maxval ; i++)
                        {
                          if(i_attr_val1.res_array[i])
                            {
                              if(current->rlim[i].rlim_max > i_attr_val1.res_array[i])
                                current->rlim[i].rlim_max = i_attr_val1.res_array[i];
                              if(current->rlim[i].rlim_cur > i_attr_val1.res_array[i])
                                current->rlim[i].rlim_cur = i_attr_val1.res_array[i];
                            }
                        }
                      if (rsbac_get_attr(RES,
                                         T_USER,
                                         i_tid,
                                         A_res_min,
                                         &i_attr_val1,
                                         FALSE))
                        {
                          printk(KERN_WARNING
                                 "rsbac_adf_set_attr_res(): rsbac_get_attr() for res_user_full returned error!\n");
                          return -RSBAC_EREADFAILED;
                        }
                      if(i_attr_val1.res_array[RLIMIT_NOFILE] > NR_OPEN)
                        i_attr_val1.res_array[RLIMIT_NOFILE] = NR_OPEN;
                      for(i = 0; i <= maxval ; i++)
                        {
                          if(i_attr_val1.res_array[i])
                            {
                              if(current->rlim[i].rlim_max < i_attr_val1.res_array[i])
                                current->rlim[i].rlim_max = i_attr_val1.res_array[i];
                              if(current->rlim[i].rlim_cur < i_attr_val1.res_array[i])
                                current->rlim[i].rlim_cur = i_attr_val1.res_array[i];
                            }
                        }
                    }
                  return 0;

                /* all other cases are unknown */
                default:
                  return(0);
              }
            break;

        case R_EXECUTE:
            switch(target)
              {
                case T_FILE:
                  #ifdef CONFIG_RSBAC_SOFTMODE
                  if(!rsbac_softmode)
                  #endif
                    {
                      int maxval = rsbac_min(RLIM_NLIMITS - 1, RSBAC_RES_MAX);
                      int i;

                      if (rsbac_get_attr(RES,
                                         target,
                                         tid,
                                         A_res_max,
                                         &i_attr_val1,
                                         FALSE))
                        {
                          printk(KERN_WARNING
                                 "rsbac_adf_set_attr_res(): rsbac_get_attr() for res_user_full returned error!\n");
                          return -RSBAC_EREADFAILED;
                        }
                      for(i = 0; i <= maxval ; i++)
                        {
                          if(i_attr_val1.res_array[i])
                            {
                              if(current->rlim[i].rlim_max > i_attr_val1.res_array[i])
                                current->rlim[i].rlim_max = i_attr_val1.res_array[i];
                              if(current->rlim[i].rlim_cur > i_attr_val1.res_array[i])
                                current->rlim[i].rlim_cur = i_attr_val1.res_array[i];
                            }
                        }
                      if (rsbac_get_attr(RES,
                                         target,
                                         tid,
                                         A_res_min,
                                         &i_attr_val1,
                                         FALSE))
                        {
                          printk(KERN_WARNING
                                 "rsbac_adf_set_attr_res(): rsbac_get_attr() for res_user_full returned error!\n");
                          return -RSBAC_EREADFAILED;
                        }
                      for(i = 0; i <= maxval ; i++)
                        {
                          if(i_attr_val1.res_array[i])
                            {
                              if(current->rlim[i].rlim_max < i_attr_val1.res_array[i])
                                current->rlim[i].rlim_max = i_attr_val1.res_array[i];
                              if(current->rlim[i].rlim_cur < i_attr_val1.res_array[i])
                                current->rlim[i].rlim_cur = i_attr_val1.res_array[i];
                            }
                        }
                    }
                  return 0;

                /* all other cases are unknown */
                default:
                  return(0);
              }
            break;

/*********************/
        default: return(0);
      }

    return(0);
  }; /* end of rsbac_adf_set_attr_res() */

/* end of rsbac/adf/res/main.c */
