/* $Id: veth.c,v 1.33 2004/03/02 12:13:02 yumo Exp $ */
/* VETH: A virtual ethernet device driver for Linux. 
 *
 * This device driver is an implementation of Aggregation of Multiple
 * Link Segments (IEEE 802.3ad).
 *
 * Author: Yumo (Katsuyuki Yumoto) 2000-2004
 *         yumo@st.rim.or.jp
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
#include <linux/config.h>
#include <linux/version.h>
#include <linux/module.h>

#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/types.h>
#include <linux/fcntl.h>
#include <linux/interrupt.h>
#include <linux/string.h>
#include <linux/ptrace.h>
#include <linux/if_ether.h>
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
#include <net/sock.h>
#endif
#include <asm/system.h>
#include <asm/uaccess.h>
#include <asm/io.h>
#include <linux/in.h>
#include <linux/errno.h>
#include <linux/delay.h>
#include <linux/lp.h>
#include <linux/init.h>

#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/inetdevice.h>
#include <linux/skbuff.h>

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
#include <linux/tqueue.h>
#endif
#include <linux/ioport.h>
#include <asm/bitops.h>
#include <asm/irq.h>
#include <asm/byteorder.h>
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,4,0)
#include <asm/spinlock.h>
#else
#include <linux/spinlock.h>
#endif
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,4,5)
#include "miicompat.h"
#else
#include <linux/mii.h>
#endif
#include <linux/proc_fs.h>
#include <linux/smp_lock.h>

#include <linux/timer.h>

#include "vether.h"

/* If it is sure all the LAN adapters report IFF_RUNNING correctly,
 * define following line. E.g. eepro100 device driver report link
 * status by using IFF_RUNNING.  */
#undef VETH_USE_IFF_RUNNING

int nveth = 1;
int nport = MAX_PHYIF;

/* Aggregator wildcard. This shall be changed to 0 after most of
 * Ethernet device drivers report their status correctly */
int agg_wc = 1;

MODULE_PARM(nveth, "1-" __MODULE_STRING(MAX_VETH) "i");
MODULE_PARM(agg_wc, "1-" __MODULE_STRING(1) "i");
MODULE_PARM(nport, "1-" __MODULE_STRING(MAX_PHYIF) "i");
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,0)
MODULE_PARM_DESC(nveth, "veth" " Number of aggregators (1-MAX_PHYIF)");
MODULE_PARM_DESC(agg_wc, "veth" "Aggregator wildcard 1=enabled(default), 0=disabled");
MODULE_PARM_DESC(nport, "veth" "Number of ports (physical i/f's)");
#endif

/* Round robing distribution. !!!Attention!!! IEEE 802.3ad does not
 * allow round robbing on frame distribution. Distributed frame order
 * may change.  Don't use this feature if you don't want to allow
 * frame order changing. This is disabled by default. */
int rr = 0;
MODULE_PARM(rr, "1-" __MODULE_STRING(1) "i");
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,0)
MODULE_PARM_DESC(rr, "veth" "Round robbing. 1=enabled, 0=disabled(default)");
#endif

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,4,0)
/* 
 * Linux module subsystem does not provide module handling
 * facility by corresponding device name.
 * I don't want to make veth driver depend on module name
 * but it depends on device name (e.g. eth0).
 * However there are no way to lock/unlock the module by device name.
 * Therefore I need to convert device name into module name, and
 * then lock/unlock the module...
 */
/* find_module() is exported by patch */
extern struct module *find_module(const char *name);

#define MAX_MODNAME 16
#endif

#define MAX_NAME 7

static u16 get_link_stat(struct net_device *dev)
{
	static int (* ioctl)(struct net_device *, struct ifreq *, int);
	struct ifreq ifr;
	struct mii_ioctl_data *data = (struct mii_ioctl_data *)&ifr.ifr_data;

	if (!dev)
		return 0;

	/* Status register is mapped at address 1 in 
	 * MII management register set space.
	 * (IEEE Std 802.3, 1998)*/
	data->reg_num = MII_BMSR; 

	ioctl = dev->do_ioctl;
	if (ioctl &&
		((ioctl(dev, &ifr, SIOCGMIIPHY) == 0) ||
		 (ioctl(dev, &ifr, SIOCDEVPRIVATE) == 0))) {
		/* m->val_out contains status
		 * bit 2: 1 = link is up (BMSR_LSTATUS)
		 * bit 5: 1 = auto-negotiation process completed (BMSR_ANEGCOMPLETE)
		 */
		return data->val_out & (BMSR_LSTATUS | BMSR_ANEGCOMPLETE);
	}
	else {
		return BMSR_LSTATUS | BMSR_ANEGCOMPLETE;  /* regard as link up... */
	}
}

static u16 get_agg_link_stat(struct net_device *dev)
{
	struct veth_private *vp;
	struct agg_var *av;
	int i;
	u16 ret = 0;

	if (!dev)
		return 0;

	vp = (struct veth_private *)dev->priv;
	av = &(vp->agg->av);

	for (i = 0; i < MAX_PHYIF; i++){
		if (av->lag_ports[i]){
			ret |= get_link_stat(veth_ports[av->lag_ports[i]-1].dev);
		}
	}
	return ret;
}
#if 0
static u16 get_media(struct net_device *dev)
{
	static int (* ioctl)(struct net_device *, struct ifreq *, int);
	struct ifreq ifr;
	struct mii_ioctl_data *data = (struct mii_ioctl_data *)&ifr.ifr_data;
		
	/* Link partner ability register is mapped at address 5 in 
	 * MII management register set space.
	 * (IEEE Std 802.3, 1998)*/
	data->reg_num = MII_LPA; 

	ioctl = dev->do_ioctl;
	if (ioctl &&
		((ioctl(dev, &ifr, SIOCGMIIPHY) == 0) ||
		 (ioctl(dev, &ifr, SIOCDEVPRIVATE) == 0))) {
		/* m->val_out contains auto-negotiation result
		 */
		return mii_nway_result(data->val_out);
	}
	else {
		return LPA_SLCT;
	}
}
#endif

static int dev_to_port_num(struct net_device *dev)
{
	int i;
	int pn;

	for (i = 0, pn = 0; i < MAX_PHYIF; i++){
		if (veth_ports[i].dev == dev){
			pn = i + 1;
			break;
		}
	}

	return pn;
}

static int veth_check_new_mtu(struct net_device *dev, int new_mtu)
{
	int old_mtu;

	old_mtu = dev->mtu;
	if (dev->change_mtu(dev, new_mtu)){
		return _FALSE;
	}
	dev->change_mtu(dev, old_mtu);
	return _TRUE;
}

/********************************************************
 *
 *   Link Aggregation Control Protocol
 *
 ********************************************************/


/* 43.4.9 */
static void veth_record_pdu(struct lacpdu *buf, struct veth_port *p)
{
	int param_match;
	int is_active_link;

	p->pv.partner_oper_port_num = ntohs(buf->actor_port);
	p->pv.partner_oper_port_pri = ntohs(buf->actor_port_pri);
	memcpy(p->pv.partner_oper_sys, buf->actor_sys, ETH_ALEN);

	if (VETH_DEBUG(VETH_DEBUG_RCVM, 1)){
		printk(KERN_INFO "veth_record_pdu(): port:%d, partn:%02x%02x%02x-%02x%02x%02x\n",
			   p->pv.actor_port_num,
			   p->pv.partner_oper_sys[0],
			   p->pv.partner_oper_sys[1],
			   p->pv.partner_oper_sys[2],
			   p->pv.partner_oper_sys[3],
			   p->pv.partner_oper_sys[4],
			   p->pv.partner_oper_sys[5]);
	}

	p->pv.partner_oper_sys_pri = ntohs(buf->actor_sys_pri);
	p->pv.partner_oper_key = ntohs(buf->actor_key);
	p->pv.partner_oper_port_state = buf->actor_state;

	p->pv.actor_oper_port_state &= ~PS_DEFAULTED;


	/* param_match shows whether partner recognizes actor's parameters
       correctly */

	param_match = 
		(p->pv.actor_port_num == ntohs(buf->partner_port)) &&
		(p->pv.actor_port_pri == ntohs(buf->partner_port_pri)) &&
		(memcmp(veth_sysvar.actor_sys, buf->partner_sys, ETH_ALEN) == 0) &&
		(veth_sysvar.actor_sys_pri == ntohs(buf->partner_sys_pri)) &&
		(p->pv.actor_oper_port_key == ntohs(buf->partner_key)) &&
		((p->pv.actor_oper_port_state & PS_AGGREGATION) ==
		 (buf->partner_state & PS_AGGREGATION));

	is_active_link = (buf->actor_state & PS_LACP_ACTIVITY) ||
		((p->pv.actor_oper_port_state & PS_LACP_ACTIVITY) && 
		(buf->partner_state & PS_LACP_ACTIVITY));


	if (is_active_link &&
		((param_match && (buf->actor_state & PS_SYNCHRONIZATION)) ||
		((buf->actor_state & (PS_AGGREGATION | PS_SYNCHRONIZATION)) ==
		 PS_SYNCHRONIZATION))){
		p->pv.partner_oper_port_state |= PS_SYNCHRONIZATION;
	} else {
		p->pv.partner_oper_port_state &= ~PS_SYNCHRONIZATION;
	}


	/* update aggregator's partner_sys if this port attaches an
     * aggregator */
	if (p->pv.actor_port_agg_id){

		VP(p->pv.actor_port_agg_id-1);

		memcpy(vp->agg->av.partner_sys, p->pv.partner_oper_sys, ETH_ALEN);
		vp->agg->av.partner_sys_pri = p->pv.partner_oper_sys_pri;
	}
	
	return;
}


/* copy operational partner's parameter values from administrative
   parameter values */
static void veth_record_default(struct veth_port *p)
{
	p->pv.partner_oper_port_num = p->pv.partner_admin_port_num;
	p->pv.partner_oper_port_pri = p->pv.partner_admin_port_pri;
	memcpy(p->pv.partner_oper_sys, p->pv.partner_admin_sys, ETH_ALEN);
	p->pv.partner_oper_sys_pri = p->pv.partner_admin_sys_pri;
	p->pv.partner_oper_key = p->pv.partner_admin_key;
	p->pv.partner_oper_port_state = p->pv.partner_admin_port_state;

	p->pv.actor_oper_port_state |= PS_DEFAULTED;
	return;
}

static void veth_update_selected(struct lacpdu *buf, struct veth_port *p)
{
	int param_match;

	/* param_match shows whether actor recognizes partner's parameters
       correctly */
	param_match = 
		(p->pv.partner_oper_port_num == ntohs(buf->actor_port)) &&
		(p->pv.partner_oper_port_pri == ntohs(buf->actor_port_pri)) &&
		(memcmp(p->pv.partner_oper_sys, buf->actor_sys, ETH_ALEN) == 0) &&
		(p->pv.partner_oper_sys_pri == ntohs(buf->actor_sys_pri)) &&
		(p->pv.partner_oper_key == ntohs(buf->actor_key)) &&
		((p->pv.partner_oper_port_state & PS_AGGREGATION) ==
		(buf->actor_state & PS_AGGREGATION));

	if (!param_match){
		if (VETH_DEBUG(VETH_DEBUG_RCVM, 1)){
			printk (KERN_INFO "p_num:%d p_pri:%d sys:%d s_pri:%d key:%d p_st:%d\n",
					(p->pv.partner_oper_port_num == ntohs(buf->actor_port))?1:0,
					(p->pv.partner_oper_port_pri == ntohs(buf->actor_port_pri))?1:0,
					(memcmp(p->pv.partner_oper_sys, buf->actor_sys, ETH_ALEN) == 0)?1:0,
					(p->pv.partner_oper_sys_pri == ntohs(buf->actor_sys_pri))?1:0,
					(p->pv.partner_oper_key == ntohs(buf->actor_key))?1:0,
					((p->pv.partner_oper_port_state & PS_AGGREGATION) ==
					 (buf->actor_state & PS_AGGREGATION))?1:0
					);
		}
		p->pv.selected = UNSELECTED;
	}

	return;
}

static void veth_update_default_selected(struct veth_port *p)
{
	int param_match;

	/* param_match shows whether administrative parameter values match
       partner's parameters which actor recognizes */
	param_match = 
		(p->pv.partner_admin_port_num == p->pv.partner_oper_port_num) &&
		(p->pv.partner_admin_port_pri == p->pv.partner_oper_port_pri) &&
		(memcmp(p->pv.partner_admin_sys, p->pv.partner_oper_sys, ETH_ALEN) == 0) &&
		(p->pv.partner_admin_sys_pri == p->pv.partner_oper_sys_pri) &&
		(p->pv.partner_admin_key == p->pv.partner_oper_key) &&
		((p->pv.partner_admin_port_state & PS_AGGREGATION) ==
		(p->pv.partner_oper_port_state & PS_AGGREGATION));

	if (!param_match){
		p->pv.selected = UNSELECTED;
	} else {
		/* 43.4.12 
		 *
		 * Since all operational parameters are now set to locally
		 * administrated values, there can be no disagreement as to
		 * the Link Aggregation Group, so the
		 * partner_oper_port_state.synchronization variable is set to
		 * _TRUE.*/
		p->pv.partner_oper_port_state |= PS_SYNCHRONIZATION;
	}

	return;
}

static void veth_update_ntt(struct lacpdu *buf, struct veth_port *p)
{
	int param_match;

	/* param_match shows partner recognizes actor's parameters correctly */
	param_match = 
		(p->pv.actor_port_num == ntohs(buf->partner_port)) &&
		(p->pv.actor_port_pri == ntohs(buf->partner_port_pri)) &&
		(memcmp(veth_sysvar.actor_sys, buf->partner_sys, ETH_ALEN) == 0) &&
		(veth_sysvar.actor_sys_pri == ntohs(buf->partner_sys_pri)) &&
		(p->pv.actor_oper_port_key == ntohs(buf->partner_key)) &&
		((p->pv.actor_oper_port_state & (PS_LACP_ACTIVITY | PS_LACP_TIMEOUT | PS_AGGREGATION)) ==
		(buf->partner_state & (PS_LACP_ACTIVITY | PS_LACP_TIMEOUT | PS_AGGREGATION)));

	if (!param_match){
		p->pv.need_to_xmit = _TRUE;
	}
	return;
}

static void veth_attach_mux_to_agg(struct veth_port *p)
{
	p->pv.attached = _TRUE;
	return;
}

static void veth_detach_mux_from_agg(struct veth_port *p)
{
	int agg, port, i, m, n;
	int *lp;
	struct veth_agg *ag;

	p->pv.attached = _FALSE;

	/* get aggregator selected for this port */
	agg = p->pv.actor_port_agg_id;

	if (!agg){
		/* already detached */
		return;
	}

	/* get port id */
	port = p->pv.actor_port_num;

	if (!port){
		/* it shall not come here... */
		return;
	}

	agg--;

	/* find the port from lag_port list */
	ag = veth_aggs+agg;
	lp = ag->av.lag_ports;
	for (i = 0; i < MAX_PHYIF; i++){
		if (lp[i] == port && p->pv.selected == UNSELECTED){
			/* found. detach it */
			p->pv.actor_port_agg_id = 0;
			for (m = i, n = i+1; n < MAX_PHYIF; m++, n++){
				lp[m] = lp[n];
			}
			lp[m] = 0;
			if (m == 0){ /* last one */
				memset(ag->av.partner_sys, 0, ETH_ALEN);
				ag->av.partner_sys_pri = 0;
			}
			break;
		}
	}

}

static void veth_ena_collect(struct veth_port *p)
{
	p->pv.actor_oper_port_state |= PS_COLLECTING;
	return;
}

static void veth_dis_collect(struct veth_port *p)
{
	p->pv.actor_oper_port_state &= ~PS_COLLECTING;
	return;
}


/* Search for proper device (port) for transmitting this skb.
 *
 * Distribution algorithm shall not cause mis-ordering of frames
 * that are part of any given conversation (43.2.4). So this 
 * implementation uses hash by destination MAC address */
static void veth_find_distrib_port(struct veth_agg *ag)
{
	int i, j;
	int k; /* num of ports which attaches this aggregator */
	int ports[MAX_PHYIF]; /* keeps port which state is really "distributing" */
	struct veth_port *p;

	for (i = 0, k = 0; i < MAX_PHYIF; i++){
		if (!ag->av.lag_ports[i]){ /* lag_ports is built by selection logic */
			ports[k] = 0;
			break;
		} else {
			j = ag->av.lag_ports[i]-1;
			p = veth_ports+j;

			if (p->pv.attached && (p->pv.actor_oper_port_state & PS_DISTRIBUTING)){
				ports[k++] = j;
			}
		}
	}

	spin_lock(&ag->lock);
	memcpy(ag->ports, ports, k*sizeof(int));
	ag->n_ports = k;
	spin_unlock(&ag->lock);
}

static void veth_ena_distrib(struct veth_port *p)
{
	struct veth_agg *ag;
	int agg_id;

	agg_id = p->pv.actor_port_agg_id;
	ag = veth_aggs + agg_id - 1;

	p->pv.actor_oper_port_state |= PS_DISTRIBUTING;

	if (agg_id){
		veth_find_distrib_port(ag);

		/* MTU check */
		if (!veth_check_new_mtu(p->dev, ag->dev->mtu)){
			printk(KERN_INFO "veth: Port does not accept aggregator's MTU.\n"
				"veth: Attempt to make aggregator's MTU %d.\n", p->dev->mtu);
			ag->dev->change_mtu(ag->dev, p->dev->mtu);
		} else {
			p->dev->change_mtu(p->dev, ag->dev->mtu);
		}
	}

	return;
}

static void veth_dis_distrib(struct veth_port *p)
{
	struct veth_agg *ag;
	int agg_id;

	agg_id = p->pv.actor_port_agg_id;
	ag = veth_aggs + agg_id - 1;

	p->pv.actor_oper_port_state &= ~PS_DISTRIBUTING;

	if (agg_id)
		veth_find_distrib_port(ag);

	return;
}

#if 0
/* These two functions are not used for independent control model */
static void veth_ena_dist_coll(struct sk_buff *skb)
{
	veth_ena_collect(skb);
	veth_ena_distrib(skb);
}

static void veth_dis_dist_coll(struct sk_buff *skb)
{
	veth_dis_collect(skb);
	veth_dis_distrib(skb);
}
#endif
/* end of 43.4.9 */

/* 43.2.4 */
#include <linux/if_ether.h>
static int hash_mac(unsigned char *mac_h, int k)
{
	int ret;
	__u32 hash;
	struct ethhdr *eh;

	eh = (struct ethhdr *)mac_h;
	hash = *(__u32 *)(&eh->h_dest[ETH_ALEN-4]) ^
		*(__u32 *)(&eh->h_source[ETH_ALEN-4]);
	hash ^= hash >> 16;
	hash ^= hash >> 8;
	ret = hash%k;
	return ret;
}

#include <linux/ip.h>
#include <linux/tcp.h>
#include <linux/udp.h>

/* This function is used for solving correct hash value of IP fragments */
/* For TCP/UDP port number based hashing, it needs to distribute a datagrams
 * which consists of fragments via single port. This function unifies hash
 * value through all the fragments */ 

#define IPH_MF 0x2000

static int veth_get_previous_hash(struct iphdr *iph, int hash, struct veth_agg *ag)
{
	struct dist_hash_table **dt, *dp, *dpp;
	int ret = hash;

	dt = &ag->dht[iph->id%32];
	for (dp = *dt; dp; dp = dp->next){
		if (dp->id == iph->id &&
		    dp->daddr == iph->daddr &&
		    dp->saddr == iph->saddr){
			/* may be a 2'nd or later fragment */
			if (dp->hash >= ag->n_ports)
				dp->hash = 0;
			ret = dp->hash;

			if (!(iph->frag_off & IPH_MF))
				/* delete entry */
				for (dpp = *dt; dpp; dpp = dpp->next)
					if (dpp->next == dp){
						dpp->next = dp->next;
						kfree(dp);
						break;
					}
			return ret;
		}
	}

	/* not found in the table. register it if some fragments follow it. */
	if (iph->frag_off & IPH_MF){
		dp = *dt;
		*dt = kmalloc(sizeof(struct dist_hash_table), GFP_ATOMIC);
		(*dt)->next = dp;
		(*dt)->hash = hash;
		(*dt)->id = iph->id;
		(*dt)->saddr = iph->saddr;
		(*dt)->daddr = iph->daddr;
	}

	return ret;
}

static int veth_sel_distport_balanced(int len, struct veth_agg *ag)
{
	struct veth_port *p;
	int min_port = 0;
	int min = 1500;
	int i, j;

	for (i = 0; i < ag->n_ports; i++){
		j = ag->ports[i];
		p = veth_ports + j;

		if (p->avrlen < min){
			min = p->avrlen;
			min_port = j;
		}
	}

	for (i = 0; i < ag->n_ports; i++){
		j = ag->ports[i];
		p = veth_ports + j;

		p->avrlen -= (p->avrlen >> 3);
		if (j == min_port){
			p->avrlen += (len >> 3);
		}
	}
	return min_port;
}

static int veth_hash(unsigned char *mac, int len, struct veth_agg *ag)
{
	int ret;
	__u32 hash;
	struct ethhdr *eh;
	struct iphdr *iph;
	struct tcphdr *tcph;
	struct udphdr *udph;

	if ((ag->distmode & DIST_ROUNDROBIN) || rr){
	/*
	 * Round robbing distribution (disabled by default. because the
	 * standard does not allow this)
	 */
#if 1
		ret = veth_sel_distport_balanced(len, ag);
#else
		++ag->count;
		ag->count %= ag->n_ports;
		ret = ag->count;
#endif
		return ret;
	}

	eh = (struct ethhdr *)mac;

	/* check if IP datagram is contained */
	if (ntohs(eh->h_proto) != ETH_P_IP ||
	    !(ag->distmode & DIST_TCPUDPHASH)){
		return hash_mac(mac, ag->n_ports);
	}

	iph = (struct iphdr *)(mac+ETH_HLEN);

	/* check if TCP segment or UDP datagram is contained */
	if (iph->protocol == IPPROTO_TCP){
		tcph = (struct tcphdr *)(mac+ETH_HLEN+iph->ihl*4);
		hash = tcph->source ^ tcph->dest;
		hash ^= hash >> 8;
		ret = hash%ag->n_ports;
		return veth_get_previous_hash(iph, ret, ag);
	} else if (iph->protocol == IPPROTO_UDP){
		udph = (struct udphdr *)(mac+ETH_HLEN+iph->ihl*4);
		hash = udph->source ^ udph->dest;
		hash ^= hash >> 8;
		ret = hash%ag->n_ports;
		return veth_get_previous_hash(iph, ret, ag);
	}

	return hash_mac(mac, ag->n_ports);
}

static int veth_frame_distribution(struct sk_buff *skb, struct net_device *dev)
{
	struct net_device *tdev;
	struct veth_private *vp;
	unsigned char *mac_h;
	struct veth_agg *ag;
	struct veth_port *p;
	struct sk_buff *nskb;
	int len;
	unsigned long flags;

	vp = dev->priv;
	ag = vp->agg;

	/* There may be no ports for distributing this skb. so we need
	 * to clone skb to remove such skb */
	nskb = skb_clone(skb, GFP_ATOMIC);
	kfree_skb(skb);

	if (!nskb){
		return 0;
	}


	spin_lock_irqsave(&ag->lock, flags);

	switch (ag->n_ports){
	case 0:
		/* There are no ports for distributing this skb. */
		goto freeandout;
	case 1:
		p = veth_ports + ag->ports[0];
		tdev = p->dev;
		break;
	default:
		/* don't use skb->mac.* */
		mac_h = skb->data;
		p = veth_ports + ag->ports[veth_hash(mac_h, nskb->len, ag)];
		tdev = p->dev;
		break;
	}

	spin_unlock_irqrestore(&ag->lock, flags);

	if (!tdev || !(tdev->flags & IFF_UP))
		goto freeandout;

	nskb->priority = 1;

	nskb->dev = tdev;

	len = nskb->len;
	vp->enet_stats.tx_bytes += len;
	vp->enet_stats.tx_packets++;

	dev_queue_xmit(nskb);

	return 0;

freeandout:
	kfree_skb(nskb);
	return 0;
}

/* end of 43.2.4 */



static void veth_frame_collector(struct sk_buff *skb, struct net_device *dev)
{
	bool is_local_skb;
	struct veth_private *vp;
	struct ethhdr *eh;

	vp = (struct veth_private *)(dev->priv);

	eh = (struct ethhdr *)skb->mac.raw;
	is_local_skb = (!memcmp(dev->dev_addr, eh->h_dest, ETH_ALEN) ||
		eh->h_dest[0]&1)?_TRUE:_FALSE;

	if (is_local_skb)
		skb->pkt_type = PACKET_HOST; /* all frames go to upper layer */
	else
		skb->pkt_type = PACKET_OTHERHOST;

	if (is_local_skb || dev->flags&IFF_PROMISC){
		skb->dev = dev;
		netif_rx(skb);
		vp->enet_stats.rx_bytes += skb->len;
		vp->enet_stats.rx_packets++;
	} else 
		kfree_skb(skb);

	return;
	
}

/* This is an optional function */
static void veth_marker_receiver(struct sk_buff *skb, struct net_device *dev)
{
	/* veth_marker_receiver is not implemented yet */
	/* discard */
	kfree_skb(skb);
	return;
}

static void veth_marker_responder(struct sk_buff *skb, struct net_device *dev)
{

	struct lacpdu *buf;

	if (!skb){
		printk(KERN_CRIT "veth: veth_rx_frame: skb is null!\n");
		return;
	}

	buf = (struct lacpdu *)skb->mac.raw;

	/* modify TLV_type */
	buf->first_tlv_type = 0x02; /* marker response info */
	skb_push(skb, skb->data - skb->mac.raw);
	skb->protocol = eth_type_trans(skb, skb->dev);
	dev_queue_xmit(skb);

	return;
}




/*-------------------- Selection logic ---------------------*/

static void sort_ports_by_pri(void)
{
	int i, j, max;
	struct veth_port **p, **q, **t;
	static struct veth_port *t_opl[MAX_PHYIF];

	for (j = 0; j < MAX_PHYIF; j++){
		t_opl[j] = NULL;
	}

	p = t_opl;
	for (j = 0, max = 0; j < MAX_PHYIF; j++){
		if (veth_ports[j].dev){
			*(p++) = veth_ports+j;
			max++;
		}
	}

	for (j = 0, q = t_opl; j < max; j++, q++){
		for (i = j+1, p = q+1; i < max; i++, p++){
			if ((*p)->pv.actor_port_pri < (*q)->pv.actor_port_pri){
				t = p; p = q; q = t;
			}
		}
	}

	memcpy(opl, t_opl, MAX_PHYIF*sizeof(struct veth_port *));
	need_to_rebuild_opl = _FALSE;

	return;
}

static int get_num_of_port_attached(int an)
{
	int i;
	int ret = -1;

	if (!an){
		return ret;
	}

	if (!veth_aggs[an-1].dev){
		return ret;
	}

	for (i = 0, ret = 0; i < MAX_PHYIF; i++){
		if (veth_ports[i].pv.actor_port_agg_id == an &&
			veth_ports[i].pv.attached){
			ret++;
		}
	}

	return ret;
}


static bool not_assigned_mac(unsigned char *ad)
{
	bool ret = _FALSE;
	int i;

	for (i = 0; i < ETH_ALEN/2; i++){
		if (ad[i] == 0xfc){
			ret = _TRUE;
		}
	}

	return ret;
}



/* This function search for ports which ready_n is _TRUE, then
   caluculate each aggregator's Ready variable */

/* NOTE.  At this time point, there are no ethernet device driver report
 * media speed, link status and duplex mode by standard way. Therefore
 * veth cannot recognize correct status of such devices. On the other
 * hand, if veth follows the standard, veth cannot handle such
 * "unknown" status devices. i.e. there are no places where veth can
 * work. Therefore I decided to implement a wildcard option (agg_wc)
 * for aggregator until most of ethernet devices report their status
 * correctly... I've introduced media, link_status and duplex
 * mode field in the net_device_stats structure with veth. I hope
 * ethernet device drivers report their status using it in the
 * future... */

static void veth_scan_ready_n(struct veth_port *p)
{
	int i, j, k;
	int key;
	struct veth_agg *ag;

	/* loopback detection (links may be on the shared media) */
	if (memcmp(veth_sysvar.actor_sys, p->pv.partner_oper_sys, ETH_ALEN) == 0 &&
		(p->pv.partner_oper_key & SUBKEY_MASK_UPPER) == (p->pv.actor_oper_port_key & SUBKEY_MASK_UPPER)){
		i = p->pv.actor_port_agg_id;
		if (i){
			p->pv.selected = UNSELECTED;
			p->pv.loop = _TRUE; /* 43.6.4 */
			p->pv.actor_oper_port_state &= ~PS_AGGREGATION; /* 43.6.4 */
		}
		return;
	}

	if (!p->dev || p->pv.tbd_cnt){
		return;
	}

	if (p->muxm.state == DETACHED &&
		p->pv.selected == UNSELECTED &&
		p->pv.ready_n &&
		!p->pv.actor_port_agg_id){
		
		/* If agg_wc is disabled, veth does not allow unknown media
		 * speed device */
		if (!agg_wc && ((p->pv.actor_oper_port_key & SUBKEY_MASK_LOWER) == SUBKEY_UNKNOWN)){
			return;
		}

		/* apply admin params to oper params */
		p->pv.actor_oper_port_key &= 
			SUBKEY_MASK_LOWER; /* comes from veth_port_mon() */
		p->pv.actor_oper_port_key |= 
			(p->pv.actor_admin_port_key & SUBKEY_MASK_UPPER);
		key = p->pv.actor_oper_port_key;

		/* search for proper aggregator */
		for (i = 0, ag = veth_aggs; i < MAX_VETH && ag->dev; i++, ag++){

			if (!ag->av.lag_ports[0] &&
				ag->av.actor_oper_agg_key != ag->av.actor_admin_agg_key){
				/* There are no ports belong to this
				 * aggregator. So it is safe to change
				 * operational aggregator key.  */
				ag->av.actor_oper_agg_key = ag->av.actor_admin_agg_key;
			}

			if ((ag->av.actor_oper_agg_key & SUBKEY_MASK_UPPER) == (key & SUBKEY_MASK_UPPER)){


				/* goto tail (k never reach MAX_PHYIF) */
				for (k = 0; k < MAX_PHYIF; k++){
					if (!ag->av.lag_ports[k]){
						break;
					}
				}

				if (k > 0){
					struct veth_port *q;
					unsigned char s0, sn;


					/* If an individual port has been assigned, or if
					 * this port is an individual port and a port has
					 * been assigned, then not select. */
					q = veth_ports + ag->av.lag_ports[0] - 1;
					s0 = (q->pv.actor_oper_port_state & q->pv.partner_oper_port_state)
						& PS_AGGREGATION;
					sn = (p->pv.actor_oper_port_state & p->pv.partner_oper_port_state)
						& PS_AGGREGATION;
					if (!s0 || !sn){
						continue;
					}

					/* If LAG_ID is different from first
					 * port's one, not aggregate each other */
					if ((q->pv.partner_oper_key != p->pv.partner_oper_key) ||
						(q->pv.actor_oper_port_key != p->pv.actor_oper_port_key) ||
						(memcmp(q->pv.partner_oper_sys, p->pv.partner_oper_sys, ETH_ALEN) != 0)){
						continue;
					}

				}

				/* media speed of the first port is applied to the aggregator.
				 * It will be used until all the ports detach MUX from the aggregator. */
				if (!k){
					ag->av.actor_oper_agg_key &= SUBKEY_MASK_UPPER;
					ag->av.actor_oper_agg_key |= (key & SUBKEY_MASK_LOWER);
				}

				if (not_assigned_mac(ag->dev->dev_addr)){
					if (!veth_ports[i].dev){
#if 0
						printk(KERN_INFO "veth: %s does not have correct MAC address! (Too many aggregators)\n",
							   ag->dev->name);
#endif
						break;
					} else {
						memcpy(ag->dev->dev_addr, veth_ports[i].dev->dev_addr, ETH_ALEN);
						memcpy(ag->av.agg_mac, ag->dev->dev_addr, ETH_ALEN);
					}
				}

				ag->av.lag_ports[k] = p->pv.actor_port_num;
				p->pv.actor_port_agg_id = i+1;
					
				/* make it SELECTED after make sure no restriction for
				   attaching */
				p->pv.selected = STANDBY;
				break;
			}
		}
	}


	if (p->pv.selected == STANDBY){
		int n = get_num_of_port_attached(p->pv.actor_port_agg_id);

		if (n < 0){
			printk(KERN_CRIT "veth_scan_ready_n(): unexpected agg_id\n");
			return;
		}

		if (n < nport){
			if (VETH_DEBUG(VETH_DEBUG_SEL, 1)){
				printk(KERN_INFO "port:%d act_sys:%02x%02x%02x-%02x%02x%02x, partn_sys:%02x%02x%02x-%02x%02x%02x\n",
					   p->pv.actor_port_num,
					   veth_sysvar.actor_sys[0],
					   veth_sysvar.actor_sys[1],
					   veth_sysvar.actor_sys[2],
					   veth_sysvar.actor_sys[3],
					   veth_sysvar.actor_sys[4],
					   veth_sysvar.actor_sys[5],
					   p->pv.partner_oper_sys[0],
					   p->pv.partner_oper_sys[1],
					   p->pv.partner_oper_sys[2],
					   p->pv.partner_oper_sys[3],
					   p->pv.partner_oper_sys[4],
					   p->pv.partner_oper_sys[5]);
			}
			p->pv.selected = SELECTED;
			p->pv.need_to_xmit = _TRUE;
		} else {
			/* remain STANDBY because of restriction (43.6.1) */
		}
	}


	/* calculate ready value from ready_N value of each port which
       belongs to the aggregator. */

	i = p->pv.actor_port_agg_id - 1;
	if (i < 0){
		return;
	}

	ag = veth_aggs + i;

	for (j = 0; j < MAX_PHYIF && ag->av.lag_ports[j]; j++){
		p = veth_ports + (ag->av.lag_ports[j]-1);
		if (p->pv.ready_n){
			ag->av.ready = _TRUE;
			break;
		}
	}

	return;
}


/*-------------------- End of Selection logic ---------------------*/


/*-------------------- Timers ---------------------*/

static void vt_start(struct veth_timer *vt, int exp)
{
	vt->active = _TRUE;
	vt->exp = exp;
	vt->tim = 0;

	return;
}

static void vt_stop(struct veth_timer *vt)
{
	vt->tim = 0;
	vt->active = _FALSE;

	return;
}

static int is_vt_active(struct veth_timer *vt)
{
	return vt->active;
}




static void vt_count(struct veth_timer *vt)
{
	if (vt->active){
		if (vt->tim == vt->exp){
			vt->tim = 0;					
			vt->active = _FALSE;
		} else {
			vt->tim++;
		}
	}

	return;
}


static void veth_clear_timer(struct veth_timer *vt)
{
	memset(vt, 0, sizeof(struct veth_timer));
	return;
}

static void veth_unregister_timer(struct veth_timer *vt)
{
	struct veth_timer *p, *q;

	if (!vt){
		return;
	}

	if (vth == vt){
		p = vth->next;
		vth->next = 0;
		vth = p;
		return;
	}

	p = vth->next;
	q = vth;
	for (; p; p = p->next, q = q->next){
		if (vt == p){
			q->next = p->next;
			p->next = 0;
		}
	}
	return;
}

static void veth_register_timer(struct veth_timer *vt)
{
	if (!vt){
		return;
	}

	vt->tick_func = vt_count;
	vt->next = vth;
	vth = vt;
	return;
}

/*-------------------- end of Timers --------------*/


/*-------------- Periodic Xmit Machine ---------------*/
/* 43.4.13 */
static void veth_prm_tick(struct veth_port *p)
{
	struct perio_mach *m = &(p->prm);
	bool go_no_perio;

	go_no_perio = p->pv.begin || !p->pv.lacp_ena || !p->pv.port_enabled ||
		(!(p->pv.actor_oper_port_state & PS_LACP_ACTIVITY) &&
		 !(p->pv.partner_oper_port_state & PS_LACP_ACTIVITY));

	/* state machine */
	if (m->state == NO_PERIODIC){
		if (m->do_it){
			vt_stop(&(p->periodic_timer));
			m->do_it = _FALSE;
		}
		if (!go_no_perio){
			m->state = FAST_PERIODIC;
			m->do_it = _TRUE;
		}
	} else if (m->state == FAST_PERIODIC){
		if (go_no_perio){
			m->state = NO_PERIODIC;
			m->do_it = _FALSE;
		} else {
			if (m->do_it){
				vt_start(&(p->periodic_timer), Fast_Periodic_Time);
				/* 43.4.16 */
				p->pv.xmit_tim_update = Fast_Periodic_Time;
				p->pv.xmit_ena = _TRUE;
				m->do_it = _FALSE;
			}
			if (!(p->pv.partner_oper_port_state & PS_LACP_TIMEOUT)){
				/* long timeout */
				m->state = SLOW_PERIODIC;
				m->do_it = _TRUE;
			}
			/* short timeout */
			if (!is_vt_active(&(p->periodic_timer))){
				m->state = PERIODIC_TX;
				m->do_it = _TRUE;
			}
		}
	} else if (m->state == SLOW_PERIODIC){
		if (go_no_perio){
			m->state = NO_PERIODIC;
			m->do_it = _FALSE;
		} else {
			if (m->do_it){
				vt_start(&(p->periodic_timer), Slow_Periodic_Time);
				/* 43.4.16 */
				p->pv.xmit_tim_update = Fast_Periodic_Time;
				p->pv.xmit_ena = _TRUE;
				m->do_it = _FALSE;
			}
			if ((p->pv.partner_oper_port_state & PS_LACP_TIMEOUT) ||
				!is_vt_active(&(p->periodic_timer))){
				m->state = PERIODIC_TX;
				m->do_it = _TRUE;
			}
		}
	} else if (m->state == PERIODIC_TX){
		if (go_no_perio){
			m->state = NO_PERIODIC;
			m->do_it = _FALSE;
		} else {
			if (m->do_it){
				p->pv.need_to_xmit = _TRUE;
				m->do_it = _FALSE;
			}
			if (p->pv.partner_oper_port_state & PS_LACP_TIMEOUT){
				/* short timeout */
				m->state = FAST_PERIODIC;
				m->do_it = _TRUE;
			} else {
				m->state = SLOW_PERIODIC;
				m->do_it = _TRUE;
			}
		}
	}
	return;
}
/*-------------- Periodic Xmit Machine ---------------*/


/*-------------- receive machine -------------------*/

/* 
 * 43.4.12 (receive machine) 
 * This describes state machine for each port
 */

/* action at INITIALIZE state */
static void veth_rcvm_init(struct veth_port *p)
{
	p->pv.selected = UNSELECTED;
	veth_record_default(p);
	p->pv.port_moved = _FALSE;
	p->pv.actor_oper_port_state &= ~PS_EXPIRED;

	return;
}

/* action at PORT_DISABLED state */
static void veth_rcvm_port_dis(struct veth_port *p)
{

	p->pv.partner_oper_port_state &= ~PS_SYNCHRONIZATION;

	/* The standard does not mention explicitly where PS_LACP_ACTIVITY
	 * shall be set...  */
	p->pv.actor_oper_port_state |= PS_LACP_ACTIVITY;

	/* reset "loop condition" */
	p->pv.loop = _FALSE;
	p->pv.actor_oper_port_state |= PS_AGGREGATION;

	return;
}

static void veth_check_if_port_moved(struct veth_port *p)
{
	int i;

	for (i = 0; i < MAX_PHYIF; i++){
		if (veth_ports[i].dev &&
			p != veth_ports+i &&
			(veth_ports[i].pv.partner_oper_port_num == p->pv.partner_oper_port_num) &&
			(memcmp(veth_ports[i].pv.partner_oper_sys, p->pv.partner_oper_sys, ETH_ALEN) == 0)){
			p->pv.port_moved = _TRUE;
		}
	}
	return;
}

/* action at EXPIRED state */
static void veth_rcvm_exp(struct veth_port *p)
{
	p->pv.partner_oper_port_state &= ~PS_SYNCHRONIZATION;
	p->pv.partner_oper_port_state |= PS_LACP_TIMEOUT; /* short */
	vt_start(&(p->current_while_timer), Short_Timeout_Time);
	p->pv.actor_oper_port_state |= PS_EXPIRED;
	return;
}

/* action at LACP_DISABLED state */
static void veth_rcvm_lacp_dis(struct veth_port *p)
{
	p->pv.selected = UNSELECTED;
	veth_record_default(p);
	p->pv.partner_oper_port_state &= ~(PS_AGGREGATION | PS_EXPIRED);

	/* The standard does not mention explicitly when PS_LACP_ACTIVITY
	 * shall be set...  */
	p->pv.actor_oper_port_state &= ~PS_LACP_ACTIVITY;
	return;
}

/* action at DEFAULTED state */
static void veth_rcvm_def(struct veth_port *p)
{
	veth_update_default_selected(p);
	veth_record_default(p);
	p->pv.actor_oper_port_state &= ~PS_EXPIRED;
	return;
}

/* action at CURRENT state */
static void veth_rcvm_cur(struct veth_port *p)
{
	if (p->pv.actor_oper_port_state & PS_LACP_TIMEOUT){
		vt_start(&(p->current_while_timer), Short_Timeout_Time);
	} else {
		vt_start(&(p->current_while_timer), Long_Timeout_Time);
	}
	p->pv.actor_oper_port_state &= ~PS_EXPIRED;
	return;
}

static unsigned long get_phyif_link_stats(struct net_device *dev)
{
#if 0
	struct net_device_stats *r;
#endif

	/* Recently, some Ethernet device drivers report link
	 * status using MII library. */
	if (get_link_stat(dev))
		return IF_PORT_LINK_UP; /* This includes unknown link status..*/
	else
		return IF_PORT_LINK_DOWN;
#if 0
	if (dev->get_stats){
		r = dev->get_stats(dev);
		return r->link_stat;
	}
	return IF_PORT_LINK_UNKNOWN;
#endif
}

static unsigned long get_phyif_media(struct net_device *dev)
{
	struct net_device_stats *r;

	if (dev->get_stats){
		r = dev->get_stats(dev);
		return r->media;
	}
	return IF_PORT_UNKNOWN;
}


static unsigned long get_phyif_duplex_mode(struct net_device *dev)
{
	struct net_device_stats *r;

	if (dev->get_stats){
		r = dev->get_stats(dev);
		return r->duplex_mode;
	}
	return IF_PORT_DUPLEX_UNKNOWN;
}

static void veth_drop_lacpdu_buf(struct veth_port *p)
{
	for (;p->rcvm.ring.r != p->rcvm.ring.w;){
		/* LACPDU arrives (drop it)*/
		kfree(p->rcvm.ring.buf[p->rcvm.ring.r]);
		p->rcvm.ring.buf[p->rcvm.ring.r] = NULL;
		p->rcvm.ring.r++;
		if (p->rcvm.ring.r == MAX_RING){
			p->rcvm.ring.r = 0;
		}
	}
	return;
}

static void veth_rcvm_tick(struct veth_port *p)
{
	struct rcv_mach *m = &(p->rcvm);

	bool go_init;
	bool go_port_dis;

	go_init = p->pv.begin;
	go_port_dis = !p->pv.begin &&
		!p->pv.port_enabled &&
		!p->pv.port_moved;

	if (m->state == INITIALIZE){
		if (m->do_it){
			veth_rcvm_init(p);
			m->do_it = _FALSE;
		}
		if (!go_init){
			m->do_it = _TRUE;
			m->state = PORT_DISABLED;
		}
		veth_drop_lacpdu_buf(p);
	} else if (m->state == PORT_DISABLED){
		if (go_init){
			m->do_it = _TRUE;
			m->state = INITIALIZE;
		} else {
			if (m->do_it){
				veth_rcvm_port_dis(p);
				m->do_it = _FALSE;
			}

			if (!go_port_dis){
				if (p->pv.port_enabled &&
					p->pv.lacp_ena){
					m->do_it = _TRUE;
					m->state = EXPIRED;
				} else if (p->pv.port_moved){
					m->do_it = _TRUE;
					m->state = INITIALIZE;
				} else if (p->pv.port_enabled &&
						   !p->pv.lacp_ena){
					m->do_it = _TRUE;
					m->state = LACP_DISABLED;
				}
			}
			veth_check_if_port_moved(p);
			veth_drop_lacpdu_buf(p);
		}
	} else if (m->state == EXPIRED){
		if (go_init){
			m->do_it = _TRUE;
			m->state = INITIALIZE;
		} else if (go_port_dis){
			m->do_it = _TRUE;
			m->state = PORT_DISABLED;
		} else {
			if (m->do_it){
				veth_rcvm_exp(p);
				m->do_it = _FALSE;

				if (VETH_DEBUG(VETH_DEBUG_RCVM, 1)){
					printk(KERN_INFO "veth_rcvm_tick(): vt started\n");
				}

			}
			if (!is_vt_active(&p->current_while_timer)){
				/* stay this state until current_while_timer expires */

				if (VETH_DEBUG(VETH_DEBUG_RCVM, 1)){
					printk(KERN_INFO "veth_rcvm_tick(): vt expired\n");
				}

				m->state = DEFAULTED;
				m->do_it = _TRUE;
			}
			if (p->rcvm.ring.r != p->rcvm.ring.w){
				/* LACPDU arrives */
				m->state = CURRENT;
				m->do_it = _TRUE;
			}

		}
		/* stay this state until LACPDU received */
	} else if (m->state == LACP_DISABLED){
		if (go_init){
			m->do_it = _TRUE;
			m->state = INITIALIZE;
		} else if (go_port_dis){
			m->do_it = _TRUE;
			m->state = PORT_DISABLED;
		} else {
			if (m->do_it){
				veth_rcvm_lacp_dis(p);
				m->do_it = _FALSE;
			}
			veth_drop_lacpdu_buf(p);
		}
	} else if (m->state == DEFAULTED){
		if (go_init){
			m->do_it = _TRUE;
			m->state = INITIALIZE;
		} else if (go_port_dis){
			m->do_it = _TRUE;
			m->state = PORT_DISABLED;
		} else {
			if (m->do_it){
				veth_rcvm_def(p);
				m->do_it = _FALSE;
			}
			if (p->rcvm.ring.r != p->rcvm.ring.w){
				/* LACPDU arrives */
				m->state = CURRENT;
				m->do_it = _TRUE;
			}
		}
	} else if (m->state == CURRENT){
		if (go_init){
			m->do_it = _TRUE;
			m->state = INITIALIZE;
		} else if (go_port_dis){
			m->do_it = _TRUE;
			m->state = PORT_DISABLED;
		} else {

			if (m->do_it){
				struct lacpdu *buf;
				
				for (; p->rcvm.ring.r != p->rcvm.ring.w;){
					buf = (struct lacpdu *)p->rcvm.ring.buf[p->rcvm.ring.r];
					
					/* update_selected() should be done before
					 * record_pdu(). Because update_selected()
					 * compares actor's "partner information" with
					 * PDU's "actor information". If record_pdu() is
					 * done before update_selected(), above comparison
					 * cannot detect PDU's (partner's) information
					 * does not match one which actor holds. */
					veth_update_selected(buf, p);
					veth_update_ntt(buf, p);
					veth_record_pdu(buf, p);
					
					kfree(p->rcvm.ring.buf[p->rcvm.ring.r]);
					p->rcvm.ring.buf[p->rcvm.ring.r] = NULL;
					if (p->rcvm.ring.r == MAX_RING-1){
						p->rcvm.ring.r = 0;
					} else {
						p->rcvm.ring.r++;
					}
				}
				
				veth_rcvm_cur(p);
				m->do_it = _FALSE;
			}

			if (p->rcvm.ring.r != p->rcvm.ring.w){
				/* LACPDU arrives */
				m->do_it = _TRUE;
			}

			if (!is_vt_active(&p->current_while_timer)){
				m->state = EXPIRED;
				m->do_it = _TRUE;
			}
		}
	}

	return;
}

/*--------------- end of receive machine -----------*/

/*------ xmit machine ----------*/

static void xmit_lacpdu(struct veth_port *p)
{
	struct sk_buff *skb;
	struct lacpdu buf;
	struct ethhdr *eh;

	int pdu_len = LACPDU_LEN;
	int len = LACPDU_LEN + ETH_HLEN;

	memset(&buf, 0x0, pdu_len);

	/* build LACPDU */
	buf.subtype = LACP_Subtype;
	buf.version_number = LACP_Version;

	buf.first_tlv_type = 0x1; /* Actor Information */
	buf.actor_info_len = 0x14;
	buf.actor_sys_pri = htons(veth_sysvar.actor_sys_pri);
	memcpy(buf.actor_sys, veth_sysvar.actor_sys, ETH_ALEN);
	buf.actor_key = htons(p->pv.actor_oper_port_key);
	buf.actor_port_pri = htons(p->pv.actor_port_pri);
	buf.actor_port = htons(p->pv.actor_port_num);
	buf.actor_state = p->pv.actor_oper_port_state;

	buf.second_tlv_type = 0x2; /* Partner Information */
	buf.partner_info_len = 0x14;
	buf.partner_sys_pri = htons(p->pv.partner_oper_sys_pri);
	memcpy(buf.partner_sys, p->pv.partner_oper_sys, ETH_ALEN);
	buf.partner_key = htons(p->pv.partner_oper_key);
	buf.partner_port_pri = htons(p->pv.partner_oper_port_pri);
	buf.partner_port = htons(p->pv.partner_oper_port_num);
	buf.partner_state = p->pv.partner_oper_port_state;

	buf.third_tlv_type = 0x3; /* Collector Information */
	buf.collector_info_len = 0x10;
	buf.collector_max_del = htons(veth_sysvar.collector_max_del);

	buf.fourth_tlv_type = 0x0; /* Terminator */
	buf.terminator_len = 0x0;

	skb = dev_alloc_skb(len);

	if (!skb){
		return;
	}
	skb->dev = p->dev;
	skb->mac.raw = skb_put(skb, len);

	memcpy(skb->mac.raw, Slow_Protocols_Multicast, ETH_ALEN);
	memcpy(skb->mac.raw+ETH_ALEN, p->dev->dev_addr, ETH_ALEN);
	eh = (struct ethhdr *)skb->mac.raw;
	eh->h_proto = Slow_Protocols_Type;

	skb->nh.raw = skb->mac.raw + ETH_HLEN;
	memcpy(skb->nh.raw, &buf, pdu_len);

	dev_queue_xmit(skb);

	return;
}


static void veth_xmit_tick(struct veth_port *p)
{
	/* pv.xmit_ena is controlled by periodic machine */
	/* If lacp_ena is _FALSE, do not xmit LACPDU (43.4.16)*/
	if (!p->pv.xmit_ena || !p->pv.lacp_ena){
		p->pv.need_to_xmit = _FALSE;
		p->xmit_timer.tim = 0;
		return;
	}

	/* pv.xmit_update is controlled by periodic machine */
	/* 43.4.16 */
	if (p->pv.xmit_tim_update){
		p->xmit_timer.exp = p->pv.xmit_tim_update;
		p->pv.xmit_tim_update = 0;
		p->xmit_timer.tim = 0;
	}

	/* pv.xmit_count indicates LACPDU xmit count in a timer lifetime */
	if (p->xmit_timer.tim == p->xmit_timer.exp){
		p->pv.xmit_count = 0;
		p->xmit_timer.tim = 0;
	}

	if (p->pv.need_to_xmit && p->pv.xmit_count < 3){
		p->pv.xmit_count++;
		xmit_lacpdu(p);
	}
	p->pv.need_to_xmit = _FALSE;

	p->xmit_timer.tim++;

	return;
}
/*------ end of xmit machine ---*/


/*--------------- mux machine -----------*/

/* DETACHED state */
static void veth_muxm_detached(struct veth_port *p)
{
	veth_detach_mux_from_agg(p);
	p->pv.actor_oper_port_state &= ~PS_SYNCHRONIZATION;
	veth_dis_distrib(p);
	veth_dis_collect(p);
	p->pv.need_to_xmit = _TRUE;

	/*
	 * Tell selection logic this port is waiting for selection
	 * if this port is under STANDBY state, do nothing because
	 * selection is already done.
	 */
	if (p->pv.selected == UNSELECTED){
		p->pv.ready_n = _TRUE;
	}
	return;
}

/* WAITING state */
static void veth_muxm_waiting(struct veth_port *p)
{
	vt_start(&(p->wait_while_timer), Aggregate_Wait_Time);
	return;
}

/* ATTACHED state */
static void veth_muxm_attached(struct veth_port *p)
{
	veth_attach_mux_to_agg(p);
	p->pv.actor_oper_port_state |= PS_SYNCHRONIZATION;
	p->pv.actor_oper_port_state &= ~PS_COLLECTING;
	veth_dis_collect(p);
	p->pv.need_to_xmit = _TRUE;
	p->pv.ready_n = _FALSE; /* because this port attached to the aggregator */
	return;
}

/* COLLECTING state */
static void veth_muxm_collect(struct veth_port *p)
{
	veth_ena_collect(p);
	veth_dis_distrib(p);
	p->pv.need_to_xmit = _TRUE;
	return;
}

/* DISTRIBUTING state */
static void veth_muxm_distrib(struct veth_port *p)
{
	veth_ena_distrib(p);
}


static void veth_muxm_tick(struct veth_port *p)
{
	struct mux_mach *m = &(p->muxm);
	struct veth_agg *ag;
	bool go_detached;

	go_detached = p->pv.begin;

	if (m->state == DETACHED){
		if (m->do_it){
			veth_muxm_detached(p);
			m->do_it = _FALSE;
		}
		if (!go_detached){
			if (p->pv.selected == SELECTED ||
				p->pv.selected == STANDBY){
				m->state = WAITING;
				m->do_it = _TRUE;
			}
		}
	} else if (m->state == WAITING){
		if (go_detached){
			m->state = DETACHED;
			m->do_it = _TRUE;
		} else {
			if (m->do_it){
				veth_muxm_waiting(p);
				m->do_it = _FALSE;
			}
			if (!is_vt_active(&p->wait_while_timer)){

				ag = veth_aggs + p->pv.actor_port_agg_id-1;

				if (p->pv.selected == SELECTED &&
					ag->av.ready == _TRUE){
					m->state = ATTACHED;
					m->do_it = _TRUE;
				} else if (p->pv.selected == UNSELECTED){
					m->state = DETACHED;
					m->do_it = _TRUE;
				}
			}
		}
		/* stay this state while wait_while_timer does not expire */
	} else if (m->state == ATTACHED){
		if (go_detached){
			m->state = DETACHED;
			m->do_it = _TRUE;
		} else {
			if (m->do_it){
				veth_muxm_attached(p);
				m->do_it = _FALSE;
			}
			if (p->pv.selected == SELECTED &&
				p->pv.partner_oper_port_state & PS_SYNCHRONIZATION){
				m->state = COLLECTING;
				m->do_it = _TRUE;
			} else if (p->pv.selected == UNSELECTED ||
					   p->pv.selected == STANDBY){
				m->state = DETACHED;
				m->do_it = _TRUE;
			}
		}
	} else if (m->state == COLLECTING){
		if (go_detached){
			m->state = DETACHED;
			m->do_it = _TRUE;
		} else {
			if (m->do_it){
				veth_muxm_collect(p);
				m->do_it = _FALSE;
			}
			if (p->pv.selected == UNSELECTED ||
				p->pv.selected == STANDBY ||
				!(p->pv.partner_oper_port_state & PS_SYNCHRONIZATION)){
				m->state = ATTACHED;
				m->do_it = _TRUE;
			} else if (p->pv.selected == SELECTED &&
					   p->pv.partner_oper_port_state & PS_SYNCHRONIZATION &&
					   p->pv.partner_oper_port_state & PS_COLLECTING ){
				m->state = DISTRIBUTING;
				m->do_it = _TRUE;
			}
		}
	} else if (m->state == DISTRIBUTING){
		if (go_detached){
			m->state = DETACHED;
			m->do_it = _TRUE;
		} else {
			if (m->do_it){
				veth_muxm_distrib(p);
				m->do_it = _FALSE;
			}
			if (p->pv.selected == UNSELECTED ||
				p->pv.selected == STANDBY ||
				!(p->pv.partner_oper_port_state & PS_SYNCHRONIZATION) ||
				!(p->pv.partner_oper_port_state & PS_COLLECTING)){
				m->state = COLLECTING;
				m->do_it = _TRUE;
			}		
		}
	}
	return;
}


/*--------------- end of mux machine -----------*/

/*--------------- actor churn machine -----------*/

static void veth_acm_no_actor_churn(struct veth_port *p)
{
		p->pv.actor_churn = _FALSE;
		return;
}

static void veth_acm_actor_churn_mon(struct veth_port *p)
{
		p->pv.actor_churn = _FALSE;
		vt_start(&(p->actor_churn_timer), Churn_Detection_Time);
		return;
}

static void veth_acm_actor_churn(struct veth_port *p)
{
		p->pv.actor_churn = _TRUE;
		return;
}

static void veth_acm_tick(struct veth_port *p)
{
	struct ac_mach *m = &(p->acm);
	bool go_churn_mon;

	go_churn_mon = p->pv.begin ||
		!p->pv.port_enabled;

	if (m->state == NO_ACTOR_CHURN){
		if (m->do_it){
			veth_acm_no_actor_churn(p);
			m->do_it = _FALSE;
		}
		if (go_churn_mon){
			m->state = ACTOR_CHURN_MONITOR;
			m->do_it = _TRUE;
		} else {
			if (!p->pv.actor_oper_port_state & PS_SYNCHRONIZATION){
				m->state = ACTOR_CHURN_MONITOR;
				m->do_it = _TRUE;
			}
		}
	} else if (m->state == ACTOR_CHURN_MONITOR){
		if (m->do_it){
			veth_acm_actor_churn_mon(p);
			m->do_it = _FALSE;
		}
		if (!go_churn_mon){
			if (p->pv.actor_oper_port_state & PS_SYNCHRONIZATION){
				m->state = NO_ACTOR_CHURN;
				m->do_it = _TRUE;
			}
			if (!is_vt_active(&p->actor_churn_timer)){
				m->state = ACTOR_CHURN;
				m->do_it = _TRUE;
			}
		}
		/* stay this state while wait_while_timer does not expire */
	} else if (m->state == ACTOR_CHURN){
		if (m->do_it){
			veth_acm_actor_churn(p);
			m->do_it = _FALSE;
		}
		if (go_churn_mon){
			m->state = ACTOR_CHURN_MONITOR;
			m->do_it = _TRUE;
		} else {
			if (p->pv.actor_oper_port_state & PS_SYNCHRONIZATION){
				m->state = NO_ACTOR_CHURN;
				m->do_it = _TRUE;
			}
		}
	}
	return;
}
/*--------------- end of actor churn machine -----------*/

/*--------------- partner churn machine -----------*/

static void veth_pcm_no_partner_churn(struct veth_port *p)
{
		p->pv.partner_churn = _FALSE;
		return;
}

static void veth_pcm_partner_churn_mon(struct veth_port *p)
{
		p->pv.partner_churn = _FALSE;
		vt_start(&(p->partner_churn_timer), Churn_Detection_Time);
		return;
}

static void veth_pcm_partner_churn(struct veth_port *p)
{
		p->pv.partner_churn = _TRUE;
		return;
}

static void veth_pcm_tick(struct veth_port *p)
{
	struct pc_mach *m = &(p->pcm);
	bool go_churn_mon;

	go_churn_mon = p->pv.begin ||
		!p->pv.port_enabled;

	if (m->state == NO_PARTNER_CHURN){
		if (m->do_it){
			veth_pcm_no_partner_churn(p);
			m->do_it = _FALSE;
		}
		if (go_churn_mon){
			m->state = PARTNER_CHURN_MONITOR;
			m->do_it = _TRUE;
		} else {
			if (!p->pv.partner_oper_port_state & PS_SYNCHRONIZATION){
				m->state = PARTNER_CHURN_MONITOR;
				m->do_it = _TRUE;
			}
		}
	} else if (m->state == PARTNER_CHURN_MONITOR){
		if (m->do_it){
			veth_pcm_partner_churn_mon(p);
			m->do_it = _FALSE;
		}
		if (!go_churn_mon){
			if (p->pv.partner_oper_port_state & PS_SYNCHRONIZATION){
				m->state = NO_PARTNER_CHURN;
				m->do_it = _TRUE;
			}
			if (!is_vt_active(&p->partner_churn_timer)){
				m->state = PARTNER_CHURN;
				m->do_it = _TRUE;
			}
		}
		/* stay this state while wait_while_timer does not expire */
	} else if (m->state == PARTNER_CHURN){
		if (m->do_it){
			veth_pcm_partner_churn(p);
			m->do_it = _FALSE;
		}
		if (go_churn_mon){
			m->state = PARTNER_CHURN_MONITOR;
			m->do_it = _TRUE;
		} else {
			if (p->pv.partner_oper_port_state & PS_SYNCHRONIZATION){
				m->state = NO_PARTNER_CHURN;
				m->do_it = _TRUE;
			}
		}
	}
	return;
}

/*--------------- end of partner churn machine -----------*/

/*--------------- port deletion machine -----------*/
/* This machine is not defined by the standard */

static void veth_port_del(struct veth_port *p)
{
	/* unregister timers */
	veth_unregister_timer(&(p->current_while_timer));
	veth_unregister_timer(&(p->actor_churn_timer));
	veth_unregister_timer(&(p->periodic_timer));
	veth_unregister_timer(&(p->partner_churn_timer));
	veth_unregister_timer(&(p->wait_while_timer));
	veth_unregister_timer(&(p->xmit_timer));

	/* make the physical i/f's non-promiscuous mode */
	dev_set_promiscuity(p->dev, -1);

	p->dev->intercepted_ptype = 0;

	if (p->dev->set_multicast_list != NULL){
		p->dev->set_multicast_list(p->dev);
	}

	p->pv.actor_port_num = 0;

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,4,0)
	if (p->module_name[0]){
		__MOD_DEC_USE_COUNT(find_module(p->module_name));
		p->module_name[0] = '\0';
	}
#else
	dev_put(p->dev);
#endif

	p->dev = NULL;

	return;
}


static void veth_port_del_tick(struct veth_port *p)
{
	if (!p->pv.tbd_cnt){
		return;
	}

	p->pv.tbd_cnt--;

	if (p->pv.tbd_cnt < 3){
		p->pv.begin = _TRUE; /* reset this port */

		vt_stop(&(p->current_while_timer));
		vt_stop(&(p->actor_churn_timer));
		vt_stop(&(p->periodic_timer));
		vt_stop(&(p->partner_churn_timer));
		vt_stop(&(p->wait_while_timer));
		vt_stop(&(p->xmit_timer));
		
		if (p->pv.tbd_cnt){
			return;
		} else {
			veth_port_del(p);
			need_to_rebuild_opl = _TRUE;
		}
	}
}
/*--------------- end of port deletion machine -----------*/


/*--------- port status monitor -----------*/
/* Purpose of this block is periodic calculation of operational key
 * values and monitoring link status */

static void veth_port_mon(struct veth_port *p)
{
	unsigned long ls;
	unsigned short key;

	ls = get_phyif_link_stats(p->dev);

	/* Since most of Ethernet device drivers do not report link
	 * status, IF_PORT_LINK_UNKNOWN is regarded as IF_PORT_LINK_UP at
	 * this time point...  */
	if (ls == IF_PORT_LINK_UNKNOWN){
		p->pv.port_enabled = (p->dev->flags & IFF_UP)
#ifdef VETH_USE_IFF_RUNNING
					  &&(p->dev->flags & IFF_RUNNING)
#endif
			;
	} else {
		p->pv.port_enabled =
			(p->dev->flags&IFF_UP) && (ls == IF_PORT_LINK_UP);
	}

	if (!p->pv.loop){
		/* If the link is half duplex mode, we cannot enable LACP on
		 * the link... 43.1.2 (p) */
		switch(get_phyif_duplex_mode(p->dev)){
		case IF_PORT_DUPLEX_UNKNOWN:
			if (!agg_wc){
				p->pv.lacp_ena = _FALSE;
			} else {
				p->pv.lacp_ena = _TRUE;
			}
			break;
		case IF_PORT_DUPLEX_FULL:
			p->pv.lacp_ena = _TRUE;
			break;
		default:
			p->pv.lacp_ena = _FALSE;
			break;
		}
	} else { /* 43.6.3 */
			p->pv.lacp_ena = _FALSE;
	}

	/* In this implementation, Lower 8-bit of key means media speed. If
     * lower 8-bit of an aggregator's key is zero, it means the
     * aggregator accepts any media speed. See selection logic
     * comments...  */

	switch(get_phyif_media(p->dev)){
	case IF_PORT_10BASET:
		key = SUBKEY_10M;
		break;
	case IF_PORT_100BASET:
	case IF_PORT_100BASETX:
	case IF_PORT_100BASEFX:
		key = SUBKEY_100M;
		break;
	case IF_PORT_1000BASET:
	case IF_PORT_1000BASESX:
	case IF_PORT_1000BASELX:
		key = SUBKEY_1000M;
		break;
	default: /* unknown */
		key = SUBKEY_UNKNOWN;
		break;
	}

	/* agg_wc comes from module parameter */
	if (agg_wc == 1){
		key = SUBKEY_UNKNOWN;
	}
		
	if (key == SUBKEY_UNKNOWN && agg_wc){
		p->pv.actor_oper_port_key &= SUBKEY_MASK_UPPER;
		return;
	}

	/* if port's media speed is known and key has changed, then
     * unselects aggregator */
	if ((p->pv.actor_oper_port_key & SUBKEY_MASK_LOWER) != key){
		p->pv.actor_oper_port_key &= SUBKEY_MASK_UPPER;
		p->pv.actor_oper_port_key |= key;
		p->pv.selected = UNSELECTED;
	}

	return;
}

static void veth_port_mon_tick(struct veth_port *p)
{
	if (p->pv.pm_intval > VETH_PORT_MON_INTERVAL){
		p->pv.pm_intval = VETH_PORT_MON_INTERVAL;
	}

	if (!p->pv.pm_intval){
		p->pv.pm_intval = VETH_PORT_MON_INTERVAL;
		veth_port_mon(p);
	} else {
		p->pv.pm_intval--;
	}
	return;
}

/*--------- end of port status monitor ----*/

static void veth_tick(unsigned long data)
{
	struct veth_timer *vt;
	struct veth_port *p;
	int i;


	/* periodic machine and other timers */
	for (i = 0, vt = vth; vt;i++, vt = vt->next){
		if (vt->tick_func){
			vt->tick_func(vt);
		}
	}

	if (need_to_rebuild_opl){
		sort_ports_by_pri();
	}

	for (i = 0; opl[i] && i < MAX_PHYIF; i++){

		p = opl[i];

		/* port status monitor */
		veth_port_mon_tick(p);

		/* receive machine */
		veth_rcvm_tick(p);

		/* mux machine */
		veth_muxm_tick(p);

		/* actor churn machine */
		veth_acm_tick(p);

		/* partner churn machine */
		veth_pcm_tick(p);

		/* periodic xmit machine */
		veth_prm_tick(p);

		/* xmit machine */
		veth_xmit_tick(p);

		/* selection logic */
		veth_scan_ready_n(p);
	}


	for (p = veth_ports, i = 0; i < MAX_PHYIF; p++, i++){
		if (!veth_ports[i].dev){
			continue;
		}
		/* port deletion machine */
		veth_port_del_tick(p);
	}

	tl.expires = jiffies + HZ/10; /* 100 msec */
	add_timer(&tl);

	return;
}



/*----------- aggregator parser (43.2.8.1) --------------------*/

static struct veth_port *get_port_by_dev(struct net_device *dev)
{
	struct veth_port *p;
	int i;
	/* calculate port num */
	for (i = 0, p = NULL; i < MAX_PHYIF; i++){
		if (veth_ports[i].dev == dev){
			p = veth_ports+i;
			break;
		}
	}
	return p;
}
		
static void veth_aggregator_parser(struct sk_buff *skb, int b_slowp, struct net_device *dev)
{
	unsigned char *buf;
	int b_subtype;
	int agg;
	struct veth_port *p;


	buf = skb->mac.raw + ETH_HLEN;
	b_subtype = (buf[0] == Marker_Subtype);

	if (b_slowp && b_subtype && buf[2] == Marker_Information){
		veth_marker_responder(skb, dev);
	} else if (b_slowp && b_subtype && buf[2] == Marker_Response_Information){
		veth_marker_receiver(skb, dev);
	} else if (b_slowp && (buf[0] < 1 || 10 < buf[0])){
		goto freeandout;
	} else {
		p = get_port_by_dev(dev);
		if (!p)
			/* corresponding port not found */
			goto freeandout;
	
		/* calculate agg num */
		agg = p->pv.actor_port_agg_id - 1;
		if (agg < 0)
			/* proper aggregator not assigned */
			goto freeandout;

		/* If the port state is not COLLECTING then discards this skb */
		if (!p->pv.actor_oper_port_state & PS_COLLECTING)
			goto freeandout;

		veth_frame_collector(skb, veth_aggs[agg].dev);
	}
	return;

freeandout:
	kfree_skb(skb);
	return;
}

static void veth_lacp(struct sk_buff *skb, struct net_device *dev)
{

	struct veth_port *p;
	unsigned char *buf;
	int pn = 0;

	buf = skb->mac.raw + ETH_HLEN;

	pn = dev_to_port_num(skb->dev);
	if (!pn)
		goto freeandout;

	p = veth_ports + pn - 1;

	if (p->pv.tbd_cnt)
		goto freeandout;

	p->rcvm.ring.buf[p->rcvm.ring.w] = kmalloc(LACPDU_LEN+ETH_HLEN+5, GFP_ATOMIC);
	memcpy(p->rcvm.ring.buf[p->rcvm.ring.w++], buf, LACPDU_LEN+ETH_HLEN);
	if (p->rcvm.ring.w == MAX_RING){
		p->rcvm.ring.w = 0;
		if (p->rcvm.ring.r == p->rcvm.ring.w){
			/* Overwriting oldest entry.. This should not occur! */
			p->rcvm.ring.r++;
		}
	}

freeandout:
	kfree_skb(skb);
	return;
}

/* control parser (43.2.9.1.4) */
static int veth_control_parser(struct sk_buff *skb, struct net_device *dev, struct packet_type *pt)
{
	unsigned char *buf;
	struct sk_buff *nskb;
	int b_mac_addr;
	int b_mac_type;

	if (!skb)
		return 1;

	nskb = skb;
	skb = skb_clone(nskb, GFP_ATOMIC);
	kfree_skb(nskb);

	if (!skb)
		return 0;

	buf = skb->mac.raw + ETH_HLEN;

	if (VETH_DEBUG(VETH_DEBUG_CTLP, 1)){
		printk(KERN_INFO "h_dest:%02x%02x%02x-%02x%02x%02x, h_proto:%04x, lacp-sub:%02x\n",
			   skb->mac.ethernet->h_dest[0],
			   skb->mac.ethernet->h_dest[1],
			   skb->mac.ethernet->h_dest[2],
			   skb->mac.ethernet->h_dest[3],
			   skb->mac.ethernet->h_dest[4],
			   skb->mac.ethernet->h_dest[5],
			   skb->mac.ethernet->h_proto,
			   buf[0]);
	}

	b_mac_addr = (memcmp(skb->mac.ethernet->h_dest, Slow_Protocols_Multicast, ETH_ALEN) == 0)?_TRUE:_FALSE;
	b_mac_type = (skb->mac.ethernet->h_proto == Slow_Protocols_Type);

	if (b_mac_addr && b_mac_type)
		if (buf[0] == LACP_Subtype)
			veth_lacp(skb, dev);
		else
			veth_aggregator_parser(skb, _TRUE, dev);
	else
		veth_aggregator_parser(skb, _FALSE, dev);

	return 0;
}

/***************************************************************
 *              End of LACP
 ***************************************************************/

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
static struct packet_type veth_packet_type __initdata = {
	__constant_htons(ETH_P_VETH),
	NULL,           /* Listen to all devices */
	veth_control_parser,
	NULL,
	NULL
};
#else
static struct packet_type veth_packet_type = {
	.type = __constant_htons(ETH_P_VETH),
	.func = veth_control_parser,
};
#endif

static inline void ic_veth_init(void)
{
	dev_add_pack(&veth_packet_type);
}
static inline void ic_veth_cleanup(void)
{
	dev_remove_pack(&veth_packet_type);
}



/*
 *  Open/initialize the i/f.  This is called (in the current kernel)
 *  sometime after booting when the 'ifconfig' program is run.
 */
static int veth_open(struct net_device *dev)
{
	/* start transmitting */
	netif_start_queue(dev);

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
	MOD_INC_USE_COUNT;
#endif

	return 0;
}

/* The inverse routine to veth_open (). */
static int veth_close(struct net_device *dev)
{

	/* can't transmit any more */
	netif_stop_queue(dev);

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
	MOD_DEC_USE_COUNT;
#endif

	return 0;
}



static struct net_device_stats *veth_get_stats(struct net_device *dev)
{
	struct veth_private *vp = (struct veth_private *)dev->priv;
	struct net_device_stats *r = &vp->enet_stats;

	return r;
}

static int veth_config(struct net_device *dev, struct ifmap *map)
{
	/*struct veth_private *vp = (struct veth_private *) dev->priv;*/

	if (dev->flags & IFF_UP)
		return -EBUSY;

	return 0;
}

static int veth_change_mtu(struct net_device *dev, int new_mtu)
{
	struct veth_private *vp = (struct veth_private *) dev->priv;
	struct veth_agg *ag;
	struct veth_port *p;
	int err;
	int i, *pn;

	ag = vp->agg;
	pn = ag->av.lag_ports;

	/* Check if all the ports allow new_mtu value */
	for (err = 0, i = 0; *pn && (i < MAX_PHYIF); pn++, i++){
		p = veth_ports + *pn - 1;
		if (!veth_check_new_mtu(p->dev, new_mtu)){
			printk(KERN_INFO "veth: Port \"%s\" does not allow MTU %d\n",
				   dev->name, new_mtu);
			err |= 1;
		}
	}

	if (err){
		printk(KERN_INFO "veth: Cannot change MTU. "
			   "Because at least one port does not allow new MTU\n");
		return -EINVAL;
	}

	/* Now make each port's MTU same as aggregator's one */
	for (err = 0, i = 0; *pn && (i < MAX_PHYIF); pn++, i++){
		p = veth_ports + *pn - 1;
		p->dev->change_mtu(p->dev, new_mtu);
	}

	/* Now change given aggregator's MTU. */
	ag->dev->mtu = new_mtu;
	return 0;
}


static void veth_set_admin_port_vars(struct veth_port *p, unsigned short key)
{
	p->pv.actor_admin_port_key = p->pv.actor_oper_port_key = key;
	p->pv.actor_admin_port_state = PS_DEFAULTED | PS_AGGREGATION;
	p->pv.actor_oper_port_state |= PS_AGGREGATION;
	p->pv.partner_admin_key = key;
	memset(p->pv.partner_admin_sys, 0, ETH_ALEN);
	p->pv.partner_admin_sys_pri = 0x8000;
	p->pv.partner_admin_port_num = p->pv.actor_port_num;
	p->pv.partner_admin_port_pri = 0x8000;
	p->pv.partner_admin_port_state = PS_DEFAULTED | PS_SYNCHRONIZATION | PS_COLLECTING | PS_AGGREGATION | PS_LACP_TIMEOUT;
	return;
}

static void veth_port_vars_init(struct veth_port *p)
{
	/* state machines */
	p->rcvm.state = INITIALIZE;
	p->acm.state = NO_ACTOR_CHURN;
	p->pcm.state = NO_PARTNER_CHURN;
	p->muxm.state = DETACHED;
	p->prm.state = NO_PERIODIC;

	p->rcvm.do_it = _TRUE;
	p->acm.do_it = _TRUE;
	p->pcm.do_it = _TRUE;
	p->muxm.do_it = _TRUE;
	p->prm.do_it = _TRUE;

	p->pv.tbd_cnt = 0;
	p->pv.selected = UNSELECTED;

	p->avrlen = 750;
}


/*
 * ioctl() support functions
 */
static int veth_get_group_id_by_name(char *name)
{
	int i;

	if (!name)
		goto exit;

	for (i = 0; i < MAX_PHYIF; i++)
		if (veth_aggs[i].dev && veth_aggs[i].dev->name)
			if ((strcmp(veth_aggs[i].dev->name, name) == 0))
				return i+1;

exit:
	return 0; /* fail */
}

static int veth_get_port_id_by_name(char *ifname)
{
	int i;

	if (!ifname || strlen(ifname) > MAX_NAME)
		goto exit;

	/* scan physical i/f name */
	for (i = 0; i < MAX_PHYIF; i++)
		if (veth_ports[i].dev && veth_ports[i].dev->name)
			if (strcmp(veth_ports[i].dev->name, ifname) == 0)
				return i+1; /* found */

exit:
	return 0; /* not found */
}


static int veth_get_vacant_veth_port(void)
{
	int i;

	for (i = 0; i < MAX_PHYIF; i++)
		if (!veth_ports[i].dev)
			return i+1;

	return 0;
}


#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,0)
static int veth_catch_device(struct veth_port *p, char *ifname)
#else
static int veth_catch_device(struct veth_port *p, char *ifname, char *modname)
#endif
{
	int ret;

	ret = -EFAULT;
	if (!p || !ifname){
		goto exit;
	}
	ret = -ENODEV;
	if (strlen(ifname) > MAX_NAME)
		goto exit;

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,4,0)
	ret = -EFAULT;
	if (!modname){
		goto exit;
	}
	ret = -ENODEV;
	if (strlen(modname) > MAX_MODNAME)
		goto exit;
#endif

	memset(p, 0, sizeof(struct veth_port));

	/* remember this locks device in 2.4 */
	p->dev = dev_get_by_name(ifname);
	if (!p->dev)
		goto exit;		/* invalid ifname */

	if (p->dev->intercepted_ptype ||
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,0)
		p->dev->br_port){
		dev_put(p->dev);
#else
		p->dev->bridge_port_id){
#endif
		p->dev = NULL;
		goto exit;		/* already used by a virtual device driver */
	}

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,4,0)
	if (modname && modname[0]){
		strcpy(p->module_name, modname);
		__MOD_INC_USE_COUNT(find_module(p->module_name));
	} else {
		p->module_name[0] = '\0';
	}
#endif

	p->dev->intercepted_ptype = ETH_P_VETH;

	/* make the interface to promiscuous mode */
	dev_set_promiscuity(p->dev, 1);
#if 0
	/* Demand loaded and "unused" device driver 
	 * module causes OOPS even though 
	 * set_multicast_list is not NULL!!! */
	if (p->dev->set_multicast_list != NULL){
		p->dev->set_multicast_list(p->dev);
	}
#endif

	ret = 0;
exit:
	return ret;
}

static int veth_release_device(struct veth_port *p)
{
	if (!p)
		return -EFAULT;

	p->pv.selected = UNSELECTED;
	p->pv.tbd_cnt = 7; /* delete this phyif 7 ticks later... */

	return 0;
}

/* add physical i/f to pool */
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,4,0)
static int veth_add_phyif(char *ifname, char *modname, unsigned short key)
#else
static int veth_add_phyif(char *ifname, unsigned short key)
#endif
{
	int i;
	struct veth_port *p;
	int ret;

	ret = -EFAULT;
	if (!ifname)
		goto exit;

	ret = -ENODEV;
	if (strlen(ifname) > MAX_NAME)
		goto exit;

	/* We cannot add a veth i/f to veth
	 * We cannot add a phy i/f already registered
	 * search for vacant veth_ports entry */
	if (veth_get_group_id_by_name(ifname) ||
		veth_get_port_id_by_name(ifname) ||
		!(i = veth_get_vacant_veth_port()))
		goto exit;

	p = veth_ports+i-1;

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,4,0)
	if (veth_catch_device(p, ifname, modname))
#else
	if (veth_catch_device(p, ifname))
#endif
		goto exit;

	if (i == 1){
		memcpy(veth_sysvar.actor_sys, p->dev->dev_addr, ETH_ALEN);
		veth_sysvar.actor_sys_pri = 0x8000;
	}

	p->pv.begin = _TRUE;

	/* register timers */

	veth_clear_timer(&(p->current_while_timer));
	veth_clear_timer(&(p->actor_churn_timer));
	veth_clear_timer(&(p->periodic_timer));
	veth_clear_timer(&(p->partner_churn_timer));
	veth_clear_timer(&(p->wait_while_timer));
	veth_clear_timer(&(p->xmit_timer));
	
	veth_register_timer(&(p->current_while_timer));
	veth_register_timer(&(p->actor_churn_timer));
	veth_register_timer(&(p->periodic_timer));
	veth_register_timer(&(p->partner_churn_timer));
	veth_register_timer(&(p->wait_while_timer));
	veth_register_timer(&(p->xmit_timer));

	veth_port_vars_init(p);
	
	p->pv.actor_port_num = i;
	p->pv.actor_port_pri = 0x8000;
	need_to_rebuild_opl = _TRUE;
	
	p->pv.lacp_ena = _TRUE;
	
	veth_set_admin_port_vars(p, key);
	
	/* start state machines */
	p->pv.begin = _FALSE;
	
	ret = 0;

exit:
	return ret;
}


/* delete physical i/f from pool */
static int veth_del_phyif(char *ifname)
{
	int i;
	struct veth_port *p;
	int ret;

	ret = -EFAULT;
	if (!ifname)
		goto exit;

	ret = -ENODEV;
	if (strlen(ifname) > MAX_NAME)
		goto exit;

	/* We cannot delete a veth i/f 
	 * We cannot delete a phy i/f not registered */
	if (veth_get_group_id_by_name(ifname) ||
		!(i = veth_get_port_id_by_name(ifname)))
		goto exit;

	p = veth_ports+i-1;
	veth_release_device(p);

	ret = 0;

exit:
	return ret;
}

static int veth_get_agg_info(struct u_agg_info *u, char *vif_name)
{
	int i;
	struct veth_agg *ag;
	struct u_agg_info ku;
	int ret;

	ret = -EFAULT;
	if (!vif_name || !u)
		goto exit;

	ret = -ENODEV;
	if (strlen(vif_name) > MAX_NAME)
		goto exit;

	ret = -EFAULT;
	for (i = 0, ag = veth_aggs; i < MAX_VETH && ag->dev; i++, ag++){
		if (strcmp(ag->dev->name, vif_name) == 0){
			memcpy(ku.agg_mac, ag->av.agg_mac, ETH_ALEN);
			ku.agg_id = ag->av.agg_id;
			ku.ind_agg = ag->av.ind_agg;
			ku.actor_admin_agg_key = ag->av.actor_admin_agg_key;
			ku.actor_oper_agg_key = ag->av.actor_oper_agg_key;
			memcpy(ku.partner_sys, ag->av.partner_sys, ETH_ALEN);
			ku.partner_sys_pri = ag->av.partner_sys_pri;
			ku.partner_oper_agg_key = ag->av.partner_oper_agg_key;
			ku.ready = ag->av.ready;
			ku.distmode = ag->distmode;

			if (copy_to_user(u, &ku, sizeof(struct u_agg_info)))
				goto exit;

			ret = sizeof(struct u_agg_info);
			goto exit;
		}
	}
	ret = -ENODEV;

exit:
	return ret;
}

static int veth_set_agg_info(struct u_agg_info *u, char *vif_name)
{
	int i;
	struct veth_agg *ag;
	struct u_agg_info ku;
	int ret;

	ret = -EFAULT;
	if (!vif_name || !u)
		goto exit;

	ret = -ENODEV;
	if (strlen(vif_name) > MAX_NAME)
		goto exit;

	ret = -EFAULT;
	if (copy_from_user(&ku, u, sizeof(struct u_agg_info)))
		goto exit;

	for (i = 0, ag = veth_aggs; i < MAX_VETH && ag->dev; i++, ag++){
		if (strcmp(ag->dev->name, vif_name) == 0){
			ag->av.actor_admin_agg_key = ku.actor_admin_agg_key;
			ag->distmode = ku.distmode;

			ret = sizeof(struct u_agg_info);
			goto exit;
		}
	}
	ret = -ENODEV;

exit:
	return ret;
}




static int veth_get_port_info(struct u_port_info *u, char *pif_name)
{
	int i;
	struct veth_port *p;
	struct u_port_info ku;
	int ret;

	ret = -EFAULT;
	if (!pif_name || !u)
		goto exit;

	ret = -ENODEV;
	if (strlen(pif_name) > MAX_NAME)
		goto exit;

	ret = -EFAULT;
	for (i = 0, p = veth_ports; i < MAX_PHYIF; i++, p++){
		if (p->dev && (strcmp(p->dev->name, pif_name) == 0)){
			ku.actor_port_num = p->pv.actor_port_num;
			ku.actor_port_pri = p->pv.actor_port_pri;
			ku.actor_port_agg_id = p->pv.actor_port_agg_id;
			ku.actor_admin_port_key = p->pv.actor_admin_port_key;
			ku.actor_oper_port_key = p->pv.actor_oper_port_key;
			ku.actor_admin_port_state = p->pv.actor_admin_port_state;
			ku.actor_oper_port_state = p->pv.actor_oper_port_state;

			memcpy(ku.partner_admin_sys, p->pv.partner_admin_sys, ETH_ALEN);
			memcpy(ku.partner_oper_sys, p->pv.partner_oper_sys, ETH_ALEN);
			ku.partner_admin_sys_pri = p->pv.partner_admin_sys_pri;
			ku.partner_oper_sys_pri = p->pv.partner_oper_sys_pri;
			ku.partner_admin_key = p->pv.partner_admin_key;
			ku.partner_oper_key = p->pv.partner_oper_key;
			ku.partner_admin_port_num = p->pv.partner_admin_port_num;
			ku.partner_oper_port_num = p->pv.partner_oper_port_num;
			ku.partner_admin_port_pri = p->pv.partner_admin_port_pri;
			ku.partner_oper_port_pri = p->pv.partner_oper_port_pri;
			ku.partner_admin_port_state = p->pv.partner_admin_port_state;
			ku.partner_oper_port_state = p->pv.partner_oper_port_state;

			ku.selected = p->pv.selected;

			if (copy_to_user(u, &ku, sizeof(struct u_port_info)))
				goto exit;

			ret = sizeof(struct u_port_info);
			goto exit;
		}
	}
	ret = -ENODEV;

exit:
	return ret;
}


static int veth_set_port_info(struct u_port_info *u, char *pif_name)
{
	int i;
	struct veth_port *p;
	struct u_port_info ku;
	int ret;

	ret = -EFAULT;
	if (!pif_name || !u)
		goto exit;

	ret = -ENODEV;
	if (strlen(pif_name) > MAX_NAME)
		goto exit;

	ret = -EFAULT;
	if (copy_from_user(&ku, u, sizeof(struct u_port_info)))
		goto exit;

	for (i = 0, p = veth_ports; i < MAX_PHYIF; i++, p++){
		if (p->dev && (strcmp(p->dev->name, pif_name) == 0)){
			p->pv.actor_admin_port_key = ku.actor_admin_port_key;
			p->pv.selected = UNSELECTED;
			p->pv.actor_admin_port_state = ku.actor_admin_port_state;
			memcpy(p->pv.partner_admin_sys, ku.partner_admin_sys, ETH_ALEN);
			p->pv.partner_admin_sys_pri = ku.partner_admin_sys_pri;
			p->pv.partner_admin_key = ku.partner_admin_key;
			p->pv.partner_admin_port_num = ku.partner_admin_port_num;
			p->pv.partner_admin_port_pri = ku.partner_admin_port_pri;
			p->pv.partner_admin_port_state = ku.partner_admin_port_state;

			ret = sizeof(struct u_port_info); /* success */
			goto exit;
		}
	}
	ret = -ENODEV;

exit:
	return ret;
}


static int veth_get_version(struct u_drv_info *u)
{
	struct u_drv_info ku;

	if (!u)
		return -EFAULT;

	ku.major = VETH_VER_MAJOR;
	ku.minor = VETH_VER_MINOR;
	ku.patch = VETH_VER_PATCH;
	strcpy(ku.name, "VETH");
	strcpy(ku.author, "Yumo (Katsuyuki Yumoto)");

	if (copy_to_user(u, &ku, sizeof(struct u_drv_info)))
		return -EFAULT;

	return sizeof(struct u_drv_info);
}

static int veth_ioctl_dispatch(struct vethconf *vc)
{
	int n;
	int ret = 0;

	if (!vc)
		return -EFAULT;

	switch (vc->action){
	case VETHCTL_VERSION:
		n = veth_get_version((struct u_drv_info *)vc->buf);
		(n < 0)?  (ret = n) : (vc->size = n);
		break;

	case VETHCTL_ADD_PHYIF:
		{
			struct u_port_info *u = (struct u_port_info *)vc->buf;
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,4,0)
			ret = veth_add_phyif(vc->pif_name, vc->pif_module, u->actor_admin_port_key);
#else
			ret = veth_add_phyif(vc->pif_name, u->actor_admin_port_key);
#endif
		}
		break;

	case VETHCTL_DEL_PHYIF:
		ret = veth_del_phyif(vc->pif_name);
		break;

	case VETHCTL_GET_AGG_INFO:
		n = veth_get_agg_info((struct u_agg_info *)vc->buf, vc->vif_name);
		(n < 0)? (ret = n) : (vc->size = n);
		break;

	case VETHCTL_SET_AGG_INFO:
		n = veth_set_agg_info((struct u_agg_info *)vc->buf, vc->vif_name);
		(n < 0)? (ret = n) : (vc->size = n);
		break;

	case VETHCTL_GET_PORT_INFO:
		n = veth_get_port_info((struct u_port_info *)vc->buf, vc->pif_name);
		(n < 0)? (ret = n) : (vc->size = n);
		break;

	case VETHCTL_SET_PORT_INFO:
		n = veth_set_port_info((struct u_port_info *)vc->buf, vc->pif_name);
		(n < 0)? (ret = n) : (vc->size = n);
		break;

	default:
		ret = -EOPNOTSUPP;
		break;
	}

	return ret;
}

static int veth_ioctl(struct net_device *adev, struct ifreq *rq, int cmd)
{
	struct mii_ioctl_data *data;

	if (!adev || !rq || !rq->ifr_data)
		return -EFAULT;

	data = (struct mii_ioctl_data *)&rq->ifr_data;

	if (cmd == SIOCDEVPRIVATE){
		struct vethconf *vc = (struct vethconf *) rq->ifr_data;
		struct vethconf kvc;
		if (!copy_from_user(&kvc, vc, sizeof(struct vethconf)) &&
			(kvc.magic == VETH_MAGIC))
			return veth_ioctl_dispatch(vc);
		/* else
		 *    fall through to keep ioctl() compatibility
		 *    with many ethernet device drivers. They provides
		 *    MII functions commonly using SIOCDEVPRIVATE space. */
	}
	switch (cmd){
		case SIOCGMIIPHY:
			data->phy_id = 0;	
		case SIOCGMIIREG:
			printk("adev->name = %s\n", adev->name);
			data->val_out = get_agg_link_stat(adev);
			break;
		default:
			return -EOPNOTSUPP;
	}
	return 0;
}

static int inline veth_searchfor(int list[], int a)
{
	int i;
	for (i = 0; i < MAX_VETH && list[i] != -1; i++) {
		if (list[i] == a) return 1;
	}
	return 0;
}




/* Write the buffer (contents of the port statuses) to a PROCfs file */

static char *state2str_mux(enum muxm_state s)
{
	switch (s){
	case DETACHED:
		return "DETCH";
	case WAITING:
		return "WAIT";
	case ATTACHED:
		return "ATTACH";
	case COLLECTING:
		return "COLL";
	case DISTRIBUTING:
		return "DIST";
	}
	return NULL;
}

static char *state2str_rcv(enum rcvm_state s)
{
	switch (s){
	case INITIALIZE:
		return "INIT";
	case PORT_DISABLED:
		return "P_DIS";
	case EXPIRED:
		return "EXPIRE";
	case LACP_DISABLED:
		return "L_DIS";
	case DEFAULTED:
		return "DEFLT";
	case CURRENT:
		return "CURRNT";
	}
	return NULL;
}

static char *state2str_pr(enum perio_state s)
{
	switch (s){
	case NO_PERIODIC:
		return "NO_PER";
	case FAST_PERIODIC:
		return "FAST_P";
	case SLOW_PERIODIC:
		return "SLOW_P";
	case PERIODIC_TX:
		return "PER_TX";
	}
	return NULL;
}

#if 0
static char *state2str_ac(enum acm_state s)
{
	switch (s){
	case NO_ACTOR_CHURN:
		return "NO_ACT";
	case ACTOR_CHURN_MONITOR:
		return "AC_MON";
	case ACTOR_CHURN:
		return "ACT_CHU";
	}
	return NULL;
}


static char *state2str_pc(enum pcm_state s)
{
	switch (s){
	case NO_PARTNER_CHURN:
		return "NO_PTN";
	case PARTNER_CHURN_MONITOR:
		return "PC_MON";
	case PARTNER_CHURN:
		return "PTN_CHU";
	}
	return NULL;
}
#endif

static int sprintf_a_line(char *buf, struct veth_port *p)
{
	int len;
	int j;
	unsigned char uc;
	char c;

	j = p->pv.actor_port_agg_id-1;

	if (p->pv.selected == UNSELECTED){
		c = 'U';
	} else if (p->pv.selected == STANDBY){
		c = 'B';
	} else if (p->pv.selected == SELECTED){
		c = 'S';
	} else {
		c = '?';
	}
	len =  sprintf(buf, "%c%5s", c, p->dev->name);
	len += sprintf(buf+len, "%6s", (j >= 0)?veth_aggs[j].dev->name:"none");
	len += sprintf(buf+len, "  %04x", p->pv.actor_oper_port_key);
	len += sprintf(buf+len, "  %04x", p->pv.partner_oper_key);
	len += sprintf(buf+len, " %02x%02x%02x%02x%02x%02x",
				p->pv.partner_oper_sys[0],
				p->pv.partner_oper_sys[1],
				p->pv.partner_oper_sys[2],
				p->pv.partner_oper_sys[3],
				p->pv.partner_oper_sys[4],
				p->pv.partner_oper_sys[5]);
	len += sprintf(buf+len, "%8s", state2str_mux(p->muxm.state));
	len += sprintf(buf+len, "%8s", state2str_rcv(p->rcvm.state));
	len += sprintf(buf+len, "%8s", state2str_pr(p->prm.state));

	uc = p->pv.actor_oper_port_state;
	len += sprintf(buf+len, " %c%c%c%c%c%c%c%c",
				   uc&1?'L':'_',
				   uc&2?'T':'_',
				   uc&4?'A':'_',
				   uc&8?'S':'_',
				   uc&16?'C':'_',
				   uc&32?'D':'_',
				   uc&64?'F':'_',
				   uc&128?'E':'_'
				   );
	uc = p->pv.partner_oper_port_state;
	len += sprintf(buf+len, " %c%c%c%c%c%c%c%c\n",
				   uc&1?'L':'_',
				   uc&2?'T':'_',
				   uc&4?'A':'_',
				   uc&8?'S':'_',
				   uc&16?'C':'_',
				   uc&32?'D':'_',
				   uc&64?'F':'_',
				   uc&128?'E':'_'
				   );
	return len;
}

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,4,0)
static int veth_get_info(char *pbuf, char **start, off_t offset, int length, int unused)
#else
static int veth_get_info(char *pbuf, char **start, off_t offset, int length)
#endif
{
#define LIMIT (PAGE_SIZE-80);

	int size;
	int len = 0;
	int i;

	if(!offset){
		/* first time write the header */
		size = sprintf(pbuf,"%s","S Port   Agg A_KEY P_KEY  PARTNER_SYS     MUX    RECV   PERIO   A_STAT   P_STAT\n");
		len = size;
	}

	for (i = 0; i < MAX_PHYIF; i++){
		if (veth_ports[i].dev){
			len += sprintf_a_line(pbuf+len, veth_ports+i);
		}
	}

	if (len > PAGE_SIZE-80){
		len = PAGE_SIZE-80;
	}

	return len;
}

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,4,0)
#ifdef CONFIG_PROC_FS
struct proc_dir_entry proc_net_veth= {
	0, 4, "veth",
	S_IFREG | S_IRUGO, 1, 0, 0,
	0, &proc_net_inode_operations,
	veth_get_info
};
#endif
#endif



#if LINUX_VERSION_CODE < KERNEL_VERSION(2,4,0)
void cleanup_module(void)
#else
static void __exit veth_cleanup_module(void)
#endif
{
	int i;

	for (i = 0; i < MAX_PHYIF; i++){
		if (veth_ports[i].dev){
			/* make the physical i/f's non-promiscuous mode */
			dev_set_promiscuity(veth_ports[i].dev, -1);
			
			if (veth_ports[i].dev->set_multicast_list != NULL){
				veth_ports[i].dev->set_multicast_list(veth_ports[i].dev);
			}
			veth_ports[i].dev->intercepted_ptype = 0;
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,0)
			dev_put(veth_ports[i].dev);
#endif
		}
	}

	for (i = 0; i < MAX_VETH; i++) {
		if (veth_aggs[i].dev) {
			unregister_netdev(veth_aggs[i].dev);

			kfree(veth_aggs[i].dev->priv);
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,4,0)
			kfree(veth_aggs[i].dev->name);
#elif LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
			kfree(veth_aggs[i].dev);
#else
			free_netdev(veth_aggs[i].dev);
#endif
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,4,0)
			if (veth_ports[i].module_name[0]){
				__MOD_DEC_USE_COUNT(find_module(veth_ports[i].module_name));
				veth_ports[i].module_name[0] = '\0';
			}
#endif
			veth_aggs[i].dev = NULL;
		}
	}

	del_timer(&tl);
#ifdef CONFIG_PROC_FS
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,4,0)
	proc_net_unregister(proc_net_veth.low_ino);
#else
	proc_net_remove("veth");
#endif
#endif

	/* remove packet handler */
	ic_veth_cleanup();

}

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,4,0)
#define veth_init  init_module
#endif

static int veth_init_agg(int agn)
{
	struct veth_private *vp;
	struct veth_agg *ag;
	int i;

	i = agn;
	ag = veth_aggs + i;

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
	ag->dev = kmalloc(sizeof(struct net_device),  GFP_KERNEL);
#else
	ag->dev = alloc_etherdev(sizeof(struct net_device));
#endif
	if (!ag->dev) {
		printk(KERN_ERR "veth: out of memory.\n");
		return -ENOMEM;
	}
	
	memset(ag->dev, 0, sizeof(struct net_device));
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,4,0)
	ag->dev->name = kmalloc(strlen("vethXXX"), GFP_KERNEL);
	
	if (!ag->dev->name) {
		printk(KERN_ERR "veth: out of memory.\n");
		kfree(ag->dev);
		ag->dev = NULL;
		return -ENOMEM;
	}
#endif	
	sprintf(ag->dev->name, "veth%d", i);


	/* Set the private structure */

	vp = ag->dev->priv = 
		(struct veth_private *)kmalloc(sizeof(struct veth_private), GFP_KERNEL);
	
	if (vp == NULL) {
		printk(KERN_ERR "%s: out of memory.\n", ag->dev->name);
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,4,0)
		kfree(ag->dev->name);
#endif
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
		kfree(ag->dev);
#else
		free_netdev(ag->dev);
#endif
		ag->dev = NULL;
		return -ENOMEM;
	}
	memset(vp, 0, sizeof(struct veth_private));
	vp->agg = veth_aggs+i;
	ag->av.actor_admin_agg_key = (i+1) << 8;


	/* Fill in the generic fields of the device structure. */
	ether_setup(ag->dev);

	/* Then, override parts of it */
	ag->dev->hard_start_xmit	= veth_frame_distribution;
	ag->dev->open		= veth_open;
	ag->dev->stop		= veth_close;
	ag->dev->get_stats 		= veth_get_stats;
	ag->dev->set_config		= veth_config;
	ag->dev->change_mtu	= veth_change_mtu;
	ag->dev->do_ioctl		= veth_ioctl;
	memset(ag->dev->dev_addr, 0xfc, ETH_ALEN/2);
	ag->dev->dev_addr[ETH_ALEN-1] = i+1;
	memcpy(ag->av.agg_mac, ag->dev->dev_addr, ETH_ALEN);
	ag->av.agg_id = i+1;
	ag->dev->tx_queue_len = 0;

	ag->count = 0;
	ag->distmode = DIST_TCPUDPHASH;
	memset(ag->dht, 0, 32*sizeof(struct dist_hash_table *));
	ag->lock = SPIN_LOCK_UNLOCKED;

	return 0;
}

/* 
 * Entry point of VETH driver.
 * Register/initialize the driver.
 */

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,4,0)
__initfunc(int veth_init(void))
#else
static int __init veth_init_module(void)
#endif
{
	int i;
	int ret;

	printk(KERN_INFO "VETH %d.%d.%d Yumo (Katsuyuki Yumoto) yumo@st.rim.or.jp\n",
		   VETH_VER_MAJOR, VETH_VER_MINOR, VETH_VER_PATCH);

	for (i = 0; i < MAX_VETH; i++){
		veth_aggs[i].dev = NULL;
	}

	for (i = 0; i < MAX_PHYIF; i++){
		veth_ports[i].dev = NULL;
	}

	/* nveth comes from module parameter and it is default to 1 */
	for (i = 0; i < nveth; i++){

		ret = veth_init_agg(i);
		if (ret){
			printk(KERN_INFO "veth: no devices registered\n");
			return ret;
		}

		ret = register_netdev(veth_aggs[i].dev);
		if (ret){
			printk(KERN_INFO "veth: no devices registered\n");
			return ret;
		}
	}

	init_timer(&tl);
	tl.expires = jiffies + HZ/10; /* 100 msec */
	tl.function = veth_tick;
	add_timer(&tl);
	
#ifdef CONFIG_PROC_FS
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,4,0)
	proc_net_register(&proc_net_veth);
#else
	proc_net_create("veth", 0, veth_get_info);
#endif
#endif

	/* packet handler registration for receiving */
	ic_veth_init();

	return 0;
}

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,0)
module_init(veth_init_module);
module_exit(veth_cleanup_module);
MODULE_AUTHOR("Katsuyuki Yumoto");
MODULE_DESCRIPTION("IEEE 802.3ad link aggregation driver");
MODULE_LICENSE("GPL");
#endif

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,0)
#include <linux/vermagic.h>
#include <linux/compiler.h>
MODULE_INFO(vermagic, VERMAGIC_STRING);
MODULE_DESCRIPTION("IEEE 802.3ad link aggregation driver");
MODULE_AUTHOR("Katsuyuki Yumoto");
#endif
