#
# @(#) $Id: sections.py,v 1.28 2003/09/26 19:21:02 ivm Exp $
#
# $Log: sections.py,v $
# Revision 1.28  2003/09/26 19:21:02  ivm
# Fixed handling Username in -q
#
# Revision 1.27  2003/08/25 19:16:49  ivm
# In ststus, ls and lj, do not convert the username specified with -u
# to numeric UID. Instead, compare text Username directly
#
# Revision 1.26  2001/01/07 16:47:46  ivm
# HoldTime, not HeldTime
#
# Revision 1.25  2000/11/20 18:57:52  ivm
# Added -e and -o exec options
# Display Sub time for pending sections
#
# Revision 1.24  2000/11/06 17:06:55  ivm
# Added nrunning/ntotal to sections.py
#
# Revision 1.23  2000/10/13 20:40:20  ivm
# Implemented -r <repeat time> option
#
# Revision 1.22  2000/10/13 20:00:05  ivm
# Cache uid->username translation
# Add section status to jobs.py output
#
# Revision 1.21  2000/10/13 18:42:42  ivm
# Added RCS log
#
# Revision 1.20  2000/10/13 18:42:05  ivm
# Re-wrote inefficient code
# Use FBSQueueInfo features to better handle -q
# Added -s <states> option
# Cleaned out references to "farms" and "status"
# Added RCS log
#

from FBS_API import *
import sys
import getopt
import time
import string

fbs_client = FBSClient()
Usage="""USAGE:
python sections.py [--help]
python sections.py [-q <queue>] [-u (<user>|all)] [-s <states>]
python sections.py (<jobid>|<sectionid>) ...
"""

#############################################################################
#                                                                           #
# convert_time()                                                            #
#                                                                           #
#############################################################################

def convert_time(tm_secs):
	if tm_secs < 0:
		return 'Forever'

	tup = time.localtime(tm_secs)[1:6]	# mm,dd,hh,mm,ss
	return '%02d/%02d %02d:%02d:%02d' % tup

############################################################################
#                                                                          #
# print_sectionInfo()                                                      #
#    Print out the section specific info.                                  #
#                                                                          #
############################################################################

SectStateCodes = {
	'running':'*',
	'done':'d',
	'zombie':'z',
	'ready':'r',
	'failed':'f',
	'canceled':'c',
	'waiting':'w',
	'held':'h',
	'unknown':'?'
	}

def decodeSectionState(si):
# returns state including "held" and state code
	state = si.State
	if state == 'waiting' and si.HoldTime:
		state = 'held'
	try:	code = SectStateCodes[state]
	except KeyError:
		code = SectStateCodes['unknown']
	return state, code
	
def print_sectionInfo(sectionInfo):

    #
    # Time to display: 
    #   Submit time if job has not started.
    #   Start time if job is running.
    #
	state, state_code = decodeSectionState(sectionInfo)

	if state == 'running':
		printTime = '%s-Str' % convert_time(sectionInfo.StartTime)
	elif state in ['done','failed']:
		printTime = '%s-End' % convert_time(sectionInfo.EndTime)
	elif state == 'held':
		printTime = '%s-Hld' % convert_time(sectionInfo.HoldTime)
	else:	# state in ['zombie','ready','canceled','waiting'] and others
		printTime = '%s-Sub' % convert_time(sectionInfo.SubTime)

	sect_id = sectionInfo.ID
	if len(sect_id) > 15:
	   sect_id = sect_id[0:14] + '*'

	queue_name = sectionInfo.Queue
	if len(queue_name) > 9:
	   queue_name = queue_name[0:9] + '*'

	usr = sectionInfo.Username
	if len(usr) > 8:
	   usr = usr[0:7] + '*'

	ntot = sectionInfo.NProc
	nrun = 0
	for i in range(sectionInfo.NProc):
		if sectionInfo.ProcStats[i+1] == 'running':
			nrun = nrun + 1
	nprcstr = '%2d/%-2d' % (nrun, ntot)
	ptype = sectionInfo.ProcType
	if len(ptype) > 10:
		ptype = ptype[:9] + '*'
	print '%-15s %-8s %-2s %-10s %-3d %-10.10s %-5.5s %-20s' % \
    	  (sect_id, 
    	  usr,
    	  state_code,
    	  queue_name,
    	  sectionInfo.Prio,
    	  ptype,
    	  nprcstr,
    	  printTime)

	return 0

def makeSectionList(args, user, queue, states):
	slst = []
	if args:
		# ignore uid, queue
		for xid in args:
			tup = fbs_misc.decodeDotID(xid)
			if len(tup) == 1:
				# jid
				try:	ji = fbs_client.getJob(jid)
				except:
					continue
				for sn in ji.sections():
					try:	s = ji.getSection(sn)
					except:
						continue
					slst.append(s)
			elif len(tup) == 2:
				# sid
				try:	s = fbs_client.getSection(xid)
				except:
					continue
				slst.append(s)
	elif queue:
		try:	qi = fbs_client.getQueue(queue)
		except KeyError:
			print 'Unknown queue <%s>' % queue
			sys.exit(1)
		except:
			print 'Error communicating with BMGR'
			sys.exit(1)
		for sid in qi.Sections:
			try:	s = fbs_client.getSection(sid)
			except:
				continue
			state, code = decodeSectionState(s)
			if (not user or user == s.Username) and (not states or code in states):
				slst.append(s)
	else:
		jlst = fbs_client.getJobList(username=user)
		jlst.sort()
		for jid in jlst:
			try:	ji = fbs_client.getJob(jid)
			except KeyError:
				continue
			except:
				print 'Error communicating with BMGR'
				sys.exit(1)
			for sn in ji.sections():
				try:	s = ji.getSection(sn)
				except:
					continue
				state, code = decodeSectionState(s)
				if (not queue or queue == s.Queue) and (not states or code in states):
					slst.append(s)
	return slst			

def printSections(args, user, queue, states, print_time = 0):
	sectlist = makeSectionList(args, user, queue, states)

	if print_time:
		print ''
		print time.ctime(time.time())
	if not sectlist:
		return 0
	print '%-15s %-8s %-2s %-10s %-3s %-10s %5s %-20s' % \
    	  ('SectID', 
    	   'User',
    	   'St',
    	   'Queue',
    	   'Pr',
    	   'Proc Type',
    	   'Nproc',
    	   'Time')

	print '%-15s %-8s %-2s %-10s %-3s %-10s %5s %-20s' % \
    	  ('-----', 
    	   '----',
    	   '--',
    	   '-----',
    	   '--',
    	   '---------',
    	   '-----',
    	   '----')

	for s in sectlist:
		#print sid
		print_sectionInfo(s)
	return 0



#
# main starts here
#

def main(argv):

	try:
		optlist, args = getopt.getopt(argv[1:],"u:q:s:r:",['help'])
	except getopt.error,error_msg:
		print error_msg
		print "Try \'fbs ls help\' for more information."
		return 1

	user = 'all'
	queue = None
	user = None
	states = None
	repeat = -1
	for opt, val in optlist:
		if opt == '-u':
			user = val
		elif opt == '-q':
			queue = val
		elif opt == '--help':
			print Usage
			return 0
		elif opt == '-s':
			states = val
		elif opt == '-r':
			repeat = string.atoi(val)
						
	while 1:
		printSections(args, user, queue, states, print_time = repeat > 0)
		if repeat < 0:
			return 0
		time.sleep(repeat)

if __name__ == '__main__':
	import sys
	sys.exit(main(sys.argv))
		
