#!/bin/bash
#
#
# This script is used once every week for executing the debian packaging pipeline CI.
# It uses the debian source repository branch `upstream-ci/master` branch. This specific branch
# should be updated to follow the change in the CamiTK `develop` branch and always produce
# a viable debian package. The main benefit is that, once a new CamiTK release is out, the
# debian package should be not far behind (everything hat is done in `upstream-ci/master` can
# be cherry picked / used as a patch to the debian source repository `master` branch.
# At any time during the CamiTK life cycle, you may check that debian packaging is correct.
# The scheduled "package" CI stage uses the `upstream-ci/master` branch on the salsa remote to track change and progress of the debian files in between release.
#
# Normally a new package is built each time the "package" stage is ran. If there is a problem at one stage, here are the step to
# - get into a sid docker image that can build the package
# - reproduce the scheduled "package" CI stage
# - correct and fix `debian/*` files so that the package can be built
# - correct and fix upstream source if needed (e.g. patch or lintian warnings)
# - push the changes to `upstream-ci/master` branch on the salsa remote so that the next scheduled "package" CI stage works.
#
# ### Create a sid docker image
#
# ```bash
# # cleanup docker
# docker rm $(sudo docker ps -a -q)
# docker rmi debian:sid # remove debian:sid current image
# docker rmi camitk:sid
#
# # cleanup workbench
# cd ~/Dev/CamiTK/packaging
# sudo rm -rf camitk-build-* camitk_* log camitk-debian-ci
#
# # build a sid image with all basic stuff to build CamiTK
# cd ~/Dev/CamiTK/packaging
# docker build --rm -t camitk:sid docker/sid-packaging-ci
# ```
#
# The last step during the docker build is to copy the current version of `package.sh` (i.e., the latest `develop` branch version of `~/.gitlab/package.sh` to `/tmp/package.sh`.
#
# ### Test packaging
#
# Get into the camitk:sid docker container:
#
# ```bash
# XSOCK=/tmp/.X11-unix/X0 && docker run --privileged=true --rm -v $XSOCK:$XSOCK -v ~/Dev/CamiTK/packaging:/tmp/packaging -w /tmp -ti camitk:sid /bin/bash
# # you can create an alias by adding this line in your ~/.bashrc
# # alias sidocker='XSOCK=/tmp/.X11-unix/X0 && docker run --privileged=true --rm -v $XSOCK:$XSOCK -v ~/Dev/CamiTK/packaging:/tmp/packaging -w /tmp -ti camitk:sid /bin/bash'
# # Then you just have to type `sidocker`
# ```
#
# From inside de docker, test the packaging using the `package.sh` script:
# ```bash
# cd /tmp/packaging # just to make sure, bash should be there already
#
# # In a rare case that you are currently modifying package.sh, copy the current version (on the host) to ~/Dev/CamiTK/packaging/package.sh, and then
# # cp package.sh /tmp
#
# # Run packaging test
# PROJECT_LOG_DIR=log bash /tmp/package.sh
# ```
#
# ### Expected results
#
# Once the script is executed and if everything went well, the current directory (i.e., `/tmp/packaging`) should contains
#
# - a `camitk-debian-ci` with the clone of the debian salsa git repository
# - the source archive (develop tarball as well as generated orig tarball)
# - the following log in the `log` subdirectory:
#     - check-environment.log
#     - prepare-package-environment.log
#     - build-package.log
#     - lintian.log
#     - autopkgtest.log
# - all the `.deb` package, the `.changes` file, the `.dsc` and the `.build` and `.buildinfo` files
#
# If the packaging did not succeed, then you can check the `log/` directory.
#
# ### Problem solving
#
# In order to relaunch a packaging after you have made some modifications in `develop` or locally, you may stay in the same container, but you need to cleanup the files generated by the previous packaging attempt:
#
# ```bash
# # -- Cleanup previous attempt generated files
# rm -rf camitk-build-* camitk_* log camitk-debian-ci/camitk-build/ camitk-debian-ci/debian/debhelper-build-stamp camitk-debian-ci/debian/.debhelper camitk-debian-ci/debian/changelog.dch camitk-debian-ci/debian/tmp *_amd64.deb *_all.deb ~/.config/CamiTK
#
# # -- Cleanup build
# # Revert d/changelog (save the file if you manually edited it first!)
# cd camitk-debian-ci && git checkout debian/changelog && make clean -f debian/rules && git status && cd ..
#
# # -- Check / cleanup the upstream-ci/master branch (BE CAREFUL)
# # In the camitk-debian-ci, check git log
# cd camitk-debian-ci && git log --oneline && cd ..
# # the merge should still be there and to avoid pushing all the merged sources as well
# # just revert the last commit (which should be the merge from upstream-ci/upstream to upstream-ci/master
# :warning: cd camitk-debian-ci && git reset HEAD~1 && cd .. # BE CAREFUL not to revert an actual commit of an important modification in the debian/files!
# # or
# cd camitk-debian-ci && git restore . && cd ..
# # cleanup if needed
# git restore .
#
# # -- Run packaging test
# PROJECT_LOG_DIR=log bash /tmp/package.sh
# ```
#
# ### Note on RAM usage
#
# To follow memory/CPU usage during the build, you can type (on the host):
# ```bash
# docker ps | grep camitk:sid
# docker container stats <container ID>
# ```
#
# Note that the swap memory should be at least 4Gb for 16Gb of RAM and that it might still requires more. In this case, if you are on your own desktop/laptop, you might have to quit all other applications, such as `thunderbird` or `firefox` in order to leave some space for the packaging process.
# The ITK actions in the imaging CEP are particularly glutons in RAM.
# To check swap emptiness, type: `free -m`
#
# :warning: if at one stage the packaging process fails during compilation because
# :warning: the compiler was abrutply interrupted by the kernel, and you ran no other
# :warning: big applications that eat your memory,
# :warning: DO NOT try to purge the swap partition on the host
# :warning: with "sudo swapoff -a; sudo swapon -a" while your docker container is alive
# :warning: as your live docker container won't be happy about it.
# :warning: just reboot and restart!





# Uncomment next line to debug
# set -x

# cleanup apt state
function cleanup-apt-pre() {
    echo "## apt update"                > >(tee --append ${PACKAGE_LOG}) 2>&1
    apt-get -qq update                  > /dev/null
    echo "## apt clean"                 > >(tee --append ${PACKAGE_LOG}) 2>&1
    apt-get -qq clean                   > /dev/null
    echo "## apt upgrade"               > >(tee --append ${PACKAGE_LOG}) 2>&1
    apt-get -qq --assume-yes upgrade    > >(tee --append ${PACKAGE_LOG} | grep --line-buffered -e "^Get" -e "Setting up") 2>&1
    echo "## apt update"                > >(tee --append ${PACKAGE_LOG}) 2>&1
    apt-get -qq update                  > >(tee --append ${PACKAGE_LOG} | grep --line-buffered -e "^Get" -e "Setting up") 2>&1
    echo "## apt fix broken"            > >(tee --append ${PACKAGE_LOG}) 2>&1
    apt-get -qq --fix-broken install    > /dev/null
}

function cleanup-apt-post() {
    echo "## apt autoremove" > >(tee --append ${PACKAGE_LOG}) 2>&1
    apt-get -qq --assume-yes autoremove > /dev/null
    echo "## apt clean" > >(tee --append ${PACKAGE_LOG}) 2>&1
    apt-get -qq clean  > /dev/null
}

if [ "$OS" == "win10" ]; then
    # there is no docker on windows
    CONTAINER_ID=0
else
    CONTAINER_ID=$(cat /proc/self/cgroup | head -n 1 | cut -d '/' -f3)
fi

# default stage is coverage
if [[ "$CAMITK_CI_STAGE" == "" ]]; then
    CAMITK_CI_STAGE="$STAGE_PACKAGE";
fi

# default mode is Continuous for develop, Experimental for any other branches
if [[ "$CAMITK_CI_MODE" == "" ]]; then
    if [[ "$CI_COMMIT_REF_NAME" == "develop" ]]; then
        CAMITK_CI_MODE="Continuous";
    else
        CAMITK_CI_MODE="Experimental";
    fi
fi

# cleanup log directory
if [ -d ${PROJECT_LOG_DIR} ] ; then
    rm -rf ${PROJECT_LOG_DIR};
fi
mkdir -p ${PROJECT_LOG_DIR}

DEBIAN_VERSION=$(cat /proc/version)

# define log filenames
CHECK_LOG=$(pwd)/${PROJECT_LOG_DIR}/check-environment.log
PACKAGE_LOG=$(pwd)/${PROJECT_LOG_DIR}/prepare-package-environment.log
BUILD_PACKAGE_LOG=$(pwd)/${PROJECT_LOG_DIR}/build-package.log
SUMMARY_LOG=$(pwd)/${PROJECT_LOG_DIR}/summary.log
LOG_DIR=$(dirname ${BUILD_PACKAGE_LOG})
ARTIFACT_URL="https://gricad-gitlab.univ-grenoble-alpes.fr/CamiTK/CamiTK/-/jobs/$CI_JOB_ID/artifacts/raw/packaging-debian-unstable-develop/log"

# output all information to report file
echo "Docker container id....... $CONTAINER_ID"         > ${CHECK_LOG}
echo "Job....................... $CI_JOB_NAME"         >> ${CHECK_LOG}
echo "Job id.................... $CI_JOB_ID"           >> ${CHECK_LOG}
echo "Build Directory........... $PROJECT_BUILD_DIR"   >> ${CHECK_LOG}
echo "Log Directory............. $PROJECT_LOG_DIR"     >> ${CHECK_LOG}
ecno "Log Directory (full)...... $LOG_DIR"             >> ${CHECK_LOG}
echo "Branch.................... $CI_COMMIT_REF_NAME"  >> ${CHECK_LOG}
echo "CAMITK_CI_MODE............ $CAMITK_CI_MODE"      >> ${CHECK_LOG}
echo "CAMITK_CI_STAGE........... $CAMITK_CI_STAGE"     >> ${CHECK_LOG}
echo "Debian version............ $DEBIAN_VERSION"      >> ${CHECK_LOG}
echo "Artifact URL.............. $ARTIFACT_URL"        >> ${CHECK_LOG}

# show report
cat ${CHECK_LOG}

# prepare Summary
echo "BUILD PACKAGE CI" > ${SUMMARY_LOG}
echo $(date) > >(tee --append ${SUMMARY_LOG}) 2>&1
echo > >(tee --append ${SUMMARY_LOG}) 2>&1
cat ${CHECK_LOG} > >(tee --append ${SUMMARY_LOG}) 2>&1

echo > >(tee --append ${SUMMARY_LOG}) 2>&1
echo "- Preparing packaging environment..." > >(tee --append ${SUMMARY_LOG}) 2>&1

echo "\n"                                         > >(tee --append ${PACKAGE_LOG}) 2>&1
echo "#################################"          > >(tee --append ${PACKAGE_LOG}) 2>&1
echo "#         Prepare image         #"          > >(tee --append ${PACKAGE_LOG}) 2>&1
echo "#################################\n"        > >(tee --append ${PACKAGE_LOG}) 2>&1
echo "## id=$(id)"                                > >(tee --append ${PACKAGE_LOG}) 2>&1
echo "## Working Directory=$(pwd)"                > >(tee --append ${PACKAGE_LOG}) 2>&1

# DEBIAN_FRONTEND=noninteractive is to prevent question asked on interactive shell (e.g. "Please select the layout matching the keyboard for this machine"
export DEBIAN_FRONTEND=noninteractive

cleanup-apt-pre
echo "## Modifying /etc/apt/sources.list"         > >(tee --append ${PACKAGE_LOG}) 2>&1
apt-get -qq --assume-yes install netselect-apt    > >(tee --append ${PACKAGE_LOG} | grep --line-buffered -e "^Get" -e "^Setting up") 2>&1
netselect-apt -c france -t 15 -a amd64 -n sid
echo "## apt install wget git-buildpackage lintian autopkgtest cme" > >(tee --append ${PACKAGE_LOG}) 2>&1
apt-get -qq --assume-yes install wget git-buildpackage lintian autopkgtest cme > >(tee --append ${PACKAGE_LOG} | grep --line-buffered -e "^Get" -e "^Setting up") 2>&1
cleanup-apt-post

echo "## Create ~/.gbp.conf" > >(tee --append ${PACKAGE_LOG}) 2>&1
cat > ~/.gbp.conf << 'EOF'
[DEFAULT]
# color everything
color = auto
pristine-tar-commit = True
pristine-tar = True
# Add a very verbose lintian after the build and normal lintian check
postbuild = ~/.gbp-postbuild.sh
# display information
prebuild = echo "=========================" && echo "GBP_GIT_DIR=$GBP_GIT_DIR" && echo "GBP_BUILD_DIR=$GBP_BUILD_DIR" && echo "========================="
EOF

echo "## Create ~/.gbp-postbuild.sh" > >(tee --append ${PACKAGE_LOG}) 2>&1
cat > ~/.gbp-postbuild.sh << 'EOF'
#!/bin/bash
# works when gbp exports for instance, the following variables
# GBP_BUILD_DIR=/root/packaging/tmp/camitk-debian-ci
# GBP_CHANGES_FILE=/root/packaging/tmp/camitk_4.2.201807251601-1_amd64.changes

set -e

# ---------------------- variables ----------------------
echo "========================="
echo "GBP_BUILD_DIR=$GBP_BUILD_DIR"
echo "GBP_CHANGES_FILE=$GBP_CHANGES_FILE"
echo "Changes files:"
ls -la $GBP_CHANGES_FILE
echo "All files:"
ls -la
echo "Log files in $LOG_DIR:"
ls -la $LOG_DIR

# ---------------------- lintian ----------------------
echo "========================="
lintianHost=$(lintian --version)
lintian -iIE --pedantic $GBP_CHANGES_FILE > >(tee $LOG_DIR/lintian.log) 2>&1

# ---------------------- autopkgtest ----------------------
echo "========================="
# we need to write a bash script in order to run autopkgtest inside the sid-dev docker
autopkgtest -l $LOG_DIR/autopkgtest.log $GBP_CHANGES_FILE -- null
EOF

echo "## Make ~/.gbp-postbuild.sh executable" > >(tee --append ${PACKAGE_LOG}) 2>&1
chmod a+x ~/.gbp-postbuild.sh
ls -la ~ > >(tee --append ${PACKAGE_LOG}) 2>&1

echo > >(tee --append ${SUMMARY_LOG}) 2>&1
echo "- Preparing source code..." > >(tee --append ${SUMMARY_LOG}) 2>&1

echo "\n"                                         > >(tee --append ${PACKAGE_LOG}) 2>&1
echo "#################################"          > >(tee --append ${PACKAGE_LOG}) 2>&1
echo "#      Prepare source code      #"          > >(tee --append ${PACKAGE_LOG}) 2>&1
echo "#################################\n"        > >(tee --append ${PACKAGE_LOG}) 2>&1

echo "## Get current develop branch version" > >(tee --append ${PACKAGE_LOG}) 2>&1
wget --quiet https://gricad-gitlab.univ-grenoble-alpes.fr/CamiTK/CamiTK/raw/develop/CamiTKVersion.cmake --output-document=/tmp/CamiTKVersion.cmake
export VERSION_MAJOR=$(grep CAMITK_VERSION_MAJOR /tmp/CamiTKVersion.cmake | cut -f2 -d'"')
export VERSION_MINOR=$(grep CAMITK_VERSION_MINOR /tmp/CamiTKVersion.cmake | cut -f2 -d'"')
export VERSION_PATCH=$(grep CAMITK_VERSION_PATCH /tmp/CamiTKVersion.cmake | cut -f2 -d'"')
export NEW_VERSION="$VERSION_MAJOR.$VERSION_MINOR.$(date +"%Y%m%d%H%M")"
echo "→ Detected version: $NEW_VERSION" > >(tee --append ${PACKAGE_LOG} ${SUMMARY_LOG}) 2>&1

# copy the clone to another workspace to work on
echo "## Clone debian git"  > >(tee --append ${PACKAGE_LOG}) 2>&1
gbp clone https://salsa.debian.org/med-team/camitk.git camitk-debian-ci  > >(tee --append ${PACKAGE_LOG}) 2>&1

echo "## Create local upstream-ci branches"                             > >(tee --append ${PACKAGE_LOG}) 2>&1
cd camitk-debian-ci
git pull                                                                > >(tee --append ${PACKAGE_LOG}) 2>&1
git checkout --track origin/upstream                                    > >(tee --append ${PACKAGE_LOG}) 2>&1
git checkout --track origin/upstream-ci/master                          > >(tee --append ${PACKAGE_LOG}) 2>&1
git checkout upstream                                                   > >(tee --append ${PACKAGE_LOG}) 2>&1
git checkout -b upstream-ci/upstream                                    > >(tee --append ${PACKAGE_LOG}) 2>&1
git checkout master                                                     > >(tee --append ${PACKAGE_LOG}) 2>&1
git config --local user.email Emmanuel.Promayon@univ-grenoble-alpes.fr  > >(tee --append ${PACKAGE_LOG}) 2>&1
git config --local user.name "Emmanuel Promayon"                        > >(tee --append ${PACKAGE_LOG}) 2>&1
echo "## git status"                                                    > >(tee --append ${PACKAGE_LOG}) 2>&1
git status                                                              > >(tee --append ${PACKAGE_LOG}) 2>&1

echo "## Download the current develop version tarball"                  > >(tee --append ${PACKAGE_LOG}) 2>&1
rm -f ../*.tar.bz2
gbp import-orig \
    --upstream-branch=upstream-ci/upstream \
    --debian-branch=upstream-ci/master \
    --no-pristine-tar \
    --no-interactive \
    --upstream-version=$NEW_VERSION \
    https://gricad-gitlab.univ-grenoble-alpes.fr/CamiTK/CamiTK/-/archive/develop/CamiTK-develop.tar.bz2  > >(tee --append ${PACKAGE_LOG}) 2>&1

echo "## Add a snapshot in history" > >(tee --append ${PACKAGE_LOG}) 2>&1
git checkout upstream-ci/master > >(tee --append ${PACKAGE_LOG}) 2>&1
gbp dch --git-author --upstream-branch=upstream-ci/upstream --debian-branch=upstream-ci/master --new-version=$NEW_VERSION-1  > >(tee --append ${PACKAGE_LOG}) 2>&1
echo "## debian/changelog (last entry)" > >(tee --append ${PACKAGE_LOG}) 2>&1
sed -n '1,/^ --/ p' debian/changelog > >(tee --append ${PACKAGE_LOG}) 2>&1

echo "## Install deps" > >(tee --append ${PACKAGE_LOG}) 2>&1
cd ..
cleanup-apt-pre
mk-build-deps -t "apt-get --assume-yes --no-install-recommends -o Debug::pkgProblemResolver=yes" --install "camitk-debian-ci/debian/control"  > >(tee --append ${PACKAGE_LOG} | grep --line-buffered -e "^Get" -e "^Setting up") 2>&1
cleanup-apt-post

echo > >(tee --append ${SUMMARY_LOG}) 2>&1
echo "- Building package..." > >(tee --append ${SUMMARY_LOG}) 2>&1

echo "\n"                                         > >(tee --append ${PACKAGE_LOG}) 2>&1
echo "#################################"          > >(tee --append ${BUILD_PACKAGE_LOG}) 2>&1
echo "#         Build Package         #"          > >(tee --append ${BUILD_PACKAGE_LOG}) 2>&1
echo "#################################\n"        > >(tee --append ${BUILD_PACKAGE_LOG}) 2>&1
cd camitk-debian-ci

echo "## Try to build the package (directly on docker image, without pbuilder)" > >(tee --append ${BUILD_PACKAGE_LOG}) 2>&1
{ time gbp buildpackage --git-no-create-orig --git-ignore-new --git-upstream-branch=upstream-ci/upstream --git-debian-branch=upstream-ci/master ; } > >(tee --append ${BUILD_PACKAGE_LOG} | grep --line-buffered -e "^dpkg" -e "Built target" -e "Test \#" -e "^W:" -e "^P:" -e "^E:" -e "^I:" -e "^X:" -e "^=====" -e "^cepgenerator-test.sh" -e "^config-test.sh" -e "CMake.*Warning") 2>&1

echo > >(tee --append ${SUMMARY_LOG}) 2>&1
echo "- Checking debian files..." > >(tee --append ${SUMMARY_LOG}) 2>&1

echo "\n"                                         > >(tee --append ${PACKAGE_LOG}) 2>&1
echo "#################################"          > >(tee --append ${BUILD_PACKAGE_LOG}) 2>&1
echo "#      Debian files check       #"          > >(tee --append ${BUILD_PACKAGE_LOG}) 2>&1
echo "#################################\n"        > >(tee --append ${BUILD_PACKAGE_LOG}) 2>&1

echo "## Checking out upstream-ci/master" > >(tee --append ${PACKAGE_LOG}) 2>&1
git checkout upstream-ci/master

echo "## Check using cme"                                  > >(tee --append ${PACKAGE_LOG}) 2>&1
DEBIAN_FRONTEND=noninteractive cme check dpkg              > >(tee --append ${PACKAGE_LOG}) 2>&1
DEBIAN_FRONTEND=noninteractive cme check dpkg-control      > >(tee --append ${PACKAGE_LOG}) 2>&1
DEBIAN_FRONTEND=noninteractive cme check dpkg-copyright    > >(tee --append ${PACKAGE_LOG}) 2>&1

echo "## Check current version of debian-policy" > >(tee --append ${PACKAGE_LOG}) 2>&1
apt-get update
apt-cache policy debian-policy > >(tee --append ${PACKAGE_LOG}) 2>&1
DEB_POLICY_CURRENT=$(apt-cache policy debian-policy | grep "^     [0-9]" | sed -e "s/ [0-9]*$//g" | sed -e "s/\.[0.9]$//g" | xargs)
DEB_POLICY_IN_USE=$(grep "Standards-Version" debian/control | cut -f2 -d" " | xargs)
echo "Debian policy version: current=$DEB_POLICY_CURRENT used=$DEB_POLICY_IN_USE"
if [[ "$DEB_POLICY_CURRENT" =~ "$DEB_POLICY_IN_USE" ]]; then
    echo "→ Debian standards version OK!" > >(tee --append ${SUMMARY_LOG}) 2>&1
else
    echo "→ Please modify debian/control and update 'Standards-Version' field to $DEB_POLICY_CURRENT (instead of $DEB_POLICY_IN_USE)" > >(tee --append ${SUMMARY_LOG}) 2>&1
    echo "→ apt-cache policy debian-policy gives:" > >(tee --append ${SUMMARY_LOG}) 2>&1
    echo $(apt-cache policy debian-policy) > >(tee --append ${SUMMARY_LOG}) 2>&1
fi

echo "## Check current version of debhelper" > >(tee --append ${PACKAGE_LOG}) 2>&1
apt-get update
apt-cache policy debhelper > >(tee --append ${PACKAGE_LOG}) 2>&1
DEBHELPER_CURRENT_FULL=$(apt-cache policy debhelper | grep "Installed:" | cut -c14- | xargs)
DEBHELPER_CURRENT=$(echo $DEBHELPER_CURRENT_FULL | cut -f1 -d. | xargs)
DEBHELPER_IN_USE=$(grep debhelper debian/control | sed "s/.*(= \([0-9.]*\)).*/\1/" | xargs)

echo "Debian debhelper version: current=$DEBHELPER_CURRENT (full version is $DEBHELPER_CURRENT_FULL) used=$DEBHELPER_IN_USE"
if [ "$DEBHELPER_CURRENT" == "$DEBHELPER_IN_USE" ]; then
    echo "→ Debian debhelper version OK!" > >(tee --append ${SUMMARY_LOG}) 2>&1
else
    echo "→ Please update 'debhelper-compat' in debian/control to $DEBHELPER_CURRENT (instead of $DEBHELPER_IN_USE)" > >(tee --append ${SUMMARY_LOG}) 2>&1
fi

# check package directory
echo "## ls current dir"                > >(tee --append ${BUILD_PACKAGE_LOG}) 2>&1
ls -la .                                > >(tee --append ${BUILD_PACKAGE_LOG}) 2>&1
echo "## ls .."                         > >(tee --append ${BUILD_PACKAGE_LOG}) 2>&1
ls -la ..                               > >(tee --append ${BUILD_PACKAGE_LOG}) 2>&1
echo "## ls LOG_DIR ($LOG_DIR)"         > >(tee --append ${BUILD_PACKAGE_LOG}) 2>&1
ls -la $LOG_DIR                         > >(tee --append ${BUILD_PACKAGE_LOG}) 2>&1
echo "## ls camitk-build"               > >(tee --append ${BUILD_PACKAGE_LOG}) 2>&1
ls -la camitk-build                     > >(tee --append ${BUILD_PACKAGE_LOG}) 2>&1
echo "## lib directory"                 > >(tee --append ${BUILD_PACKAGE_LOG}) 2>&1
ls -la camitk-build/lib                 > >(tee --append ${BUILD_PACKAGE_LOG}) 2>&1
echo "## so file version $VERSION_MAJOR.$VERSION_MINOR.$VERSION_PATCH"                       > >(tee --append ${BUILD_PACKAGE_LOG}) 2>&1
ls -la camitk-build/lib/x86_64-linux-gnu > >(tee --append ${BUILD_PACKAGE_LOG}) 2>&1

# check overrides
echo "## check overrides for libcamitkcore.so with with objdump"      > >(tee --append ${BUILD_PACKAGE_LOG}) 2>&1
LIB_FILENAME=$(ls camitk-build/lib/x86_64-linux-gnu/libcamitkcore.so.$VERSION_MAJOR.$VERSION_MINOR.$VERSION_PATCH)
echo "checking $LIB_FILENAME"
objdump -p $LIB_FILENAME | sed -n -e's/^[[:space:]]*SONAME[[:space:]]*//p' | sed -r -e's/([0-9])\.so\./\1-/; s/\.so(\.|$)//; y/_/-/; s/(.*)/\L&/'                                   > >(tee --append ${BUILD_PACKAGE_LOG}) 2>&1
echo "## check overrides for libmonitoring.so with with objdump "      > >(tee --append ${BUILD_PACKAGE_LOG}) 2>&1
LIB_FILENAME=$(ls camitk-build/lib/x86_64-linux-gnu/libmonitoring.so.$VERSION_MAJOR.$VERSION_MINOR.$VERSION_PATCH)
echo "checking $LIB_FILENAME"
objdump -p $LIB_FILENAME | sed -n -e's/^[[:space:]]*SONAME[[:space:]]*//p' | sed -r -e's/([0-9])\.so\./\1-/; s/\.so(\.|$)//; y/_/-/; s/(.*)/\L&/'                                   > >(tee --append ${BUILD_PACKAGE_LOG}) 2>&1

# Check for failure of the packaging process
if [ -f ../camitk_$NEW_VERSION-1_amd64.changes ]; then
    echo "Everything is OK, packages were build!" > >(tee --append ${SUMMARY_LOG}) 2>&1
else
    echo > >(tee --append ${SUMMARY_LOG}) 2>&1
    echo "'changes' file not found in $(pwd)/.." > >(tee --append ${SUMMARY_LOG}) 2>&1
    echo "The package was not generated, please check the artifacts and fix the problem:" > >(tee --append ${SUMMARY_LOG}) 2>&1
    echo "- summary: $ARTIFACT_URL/summary.log" > >(tee --append ${SUMMARY_LOG}) 2>&1
    echo "- check environment: $ARTIFACT_URL/check-environment.log" > >(tee --append ${SUMMARY_LOG}) 2>&1
    echo "- prepare package environment: $ARTIFACT_URL/prepare-package-environment.log" > >(tee --append ${SUMMARY_LOG}) 2>&1
    echo "- build package: $ARTIFACT_URL/build-package.log" > >(tee --append ${SUMMARY_LOG}) 2>&1
    echo > >(tee --append ${SUMMARY_LOG}) 2>&1
    echo "To update the debian files:" > >(tee --append ${SUMMARY_LOG}) 2>&1
    echo "gbp clone https://salsa.debian.org/med-team/camitk.git camitk-debian-ci" > >(tee --append ${SUMMARY_LOG}) 2>&1
    echo "cd camitk-debian-ci" > >(tee --append ${SUMMARY_LOG}) 2>&1
    echo "git config --local user.email Emmanuel.Promayon@univ-grenoble-alpes.fr" > >(tee --append ${SUMMARY_LOG}) 2>&1
    echo "git config --local user.name \"Emmanuel Promayon\"" > >(tee --append ${SUMMARY_LOG}) 2>&1
    echo "git checkout --track origin/upstream-ci/master" > >(tee --append ${SUMMARY_LOG}) 2>&1
    echo "git status" > >(tee --append ${SUMMARY_LOG}) 2>&1
    echo "And then (for instance to fix standards version):" > >(tee --append ${SUMMARY_LOG}) 2>&1
    echo "cme fix dpkg" > >(tee --append ${SUMMARY_LOG}) 2>&1
    echo > >(tee --append ${SUMMARY_LOG}) 2>&1
    echo "Packaging failed" > >(tee --append ${SUMMARY_LOG}) 2>&1
    exit 1
fi
