package com.carrotsearch.ant.tasks.junit4.events;

import java.io.*;
import java.lang.annotation.Annotation;
import java.security.AccessController;
import java.security.PrivilegedExceptionAction;
import java.util.ArrayDeque;

import org.junit.runner.Description;

import com.carrotsearch.ant.tasks.junit4.events.json.*;
import com.carrotsearch.ant.tasks.junit4.slave.SlaveMain;
import com.google.common.base.Charsets;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.LongSerializationPolicy;
import com.google.gson.stream.JsonWriter;

/**
 * Event serializer.
 */
public class Serializer implements Closeable {
  /**
   * Should help in ensuring the right order of stream writes.
   */
  private final Object lock = new Object();

  private Writer writer;
  private Gson gson;

  private final ArrayDeque<IEvent> events = new ArrayDeque<IEvent>();
  
  private volatile Throwable doForcedShutdown;
  private Thread forceCloseDaemon;

  public Serializer(OutputStream os) throws IOException {
    this.writer = new OutputStreamWriter(os, Charsets.UTF_8);
    this.gson = createGSon(Thread.currentThread().getContextClassLoader());

    this.forceCloseDaemon = new Thread("JUnit4-serializer-daemon") {
      {
        this.setDaemon(true);
      }
      
      @Override
      public void run() {
        try {
          while (true) {
            Thread.sleep(1000);
            Throwable reason = doForcedShutdown;
            if (reason != null) {
              try {
                SlaveMain.warn("Unhandled exception in event serialization.", reason);
              } finally {
                Runtime.getRuntime().halt(0);
              }
            }
          }
        } catch (InterruptedException e) {
          // Ignore and exit.
        }
      }
    };
    forceCloseDaemon.start();
  }

  public Serializer serialize(IEvent event) throws IOException {
    synchronized (lock) {
      if (writer == null) {
        throw new IOException("Serializer already closed.");
      }

      // An alternative way of solving GH-92 and GH-110. Instead of buffering
      // serialized json we emit directly. If a recursive call occurs to serialize()
      // we enqueue the event and continue, serializing them in order.
      events.addLast(event);
      if (events.size() > 1) {
        // SlaveMain.warn("Serializing " + event.getType() + " (postponed, " + events.size() + " in queue)", null);
        return this;
      }

      // SlaveMain.warn("Serializing " + event.getType(), null);
      flushQueue();

      return this;
    }
  }

  private void flushQueue() throws IOException {
    synchronized (lock) {
      while (!events.isEmpty()) {
        if (writer == null) {
          throw new IOException("Serializer already closed, with " + events.size() + " events on queue.");
        }

        final IEvent event = events.removeFirst();
        try {
          final JsonWriter jsonWriter = new JsonWriter(writer);
          jsonWriter.setIndent("  ");
          jsonWriter.beginArray();
          jsonWriter.value(event.getType().name());
          // serialization requires suppressing access checks!
          final Gson gson = this.gson;
          AccessController.doPrivileged(new PrivilegedExceptionAction<Void>() {
            @Override
            public Void run() throws Exception {
              gson.toJson(event, event.getClass(), jsonWriter);
              return null;
            }
          });
          jsonWriter.endArray();
          writer.write("\n\n");
        } catch (Throwable t) {
          // We can't do a stack bang here so any call is a risk of hitting SOE again.
          while (true) {
            doForcedShutdown = t;
            try {
              forceCloseDaemon.join();
            } catch (Throwable ignored) {
              // Ignore.
            }
          }
        }
      }
    }
  }

  public Serializer flush() throws IOException {
    synchronized (lock) {
      if (writer != null) {
        // SlaveMain.warn("flushing...", null);
        flushQueue();
        writer.flush();
      } else {
        // SlaveMain.warn("flushing failed (serializer closed)", null);
      }
      return this;
    }
  }

  public void close() throws IOException {
    synchronized (lock) {
      // SlaveMain.warn("closing...", null);
      if (writer != null) {
        serialize(new QuitEvent());
        flushQueue();

        writer.close();
        writer = null;
      }

      try {
        forceCloseDaemon.interrupt();
        forceCloseDaemon.join();
      } catch (InterruptedException e) {
        // Ignore, can't do much about it (shouldn't happen).
      }
    }
  }

  public static Gson createGSon(ClassLoader refLoader) {
    return new GsonBuilder()
      .registerTypeAdapter(byte[].class, new JsonByteArrayAdapter())
      .registerTypeHierarchyAdapter(Annotation.class, new JsonAnnotationAdapter(refLoader))
      .registerTypeHierarchyAdapter(Class.class, new JsonClassAdapter(refLoader))
      .registerTypeAdapter(Description.class, new JsonDescriptionAdapter())
      .setLongSerializationPolicy(LongSerializationPolicy.DEFAULT)
      .disableHtmlEscaping()
      .create();
  }  
}