/**
 * Copyright (C) 2009-2013 Paul Fretwell - aka 'Footleg' (drfootleg@gmail.com)
 * 
 * This file is part of Cave Converter.
 * 
 * Cave Converter is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * Cave Converter is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with Cave Converter.  If not, see <http://www.gnu.org/licenses/>.
 */
package footleg.cavesurvey.data.reader;

import java.text.ParseException;
import java.util.ArrayList;
import java.util.Date;
import java.util.Iterator;
import java.util.List;

import footleg.cavesurvey.cmdline.CaveConverter;
import footleg.cavesurvey.data.model.CaveSurvey;
import footleg.cavesurvey.data.model.Equate;
import footleg.cavesurvey.data.model.SurveyLeg;
import footleg.cavesurvey.data.model.SurveySeries;
import footleg.cavesurvey.tools.UtilityFunctions;

/**
 * Parser for Survex format text data files.
 * 
 * @author      Footleg <drfootleg@gmail.com>
 * @version     2013.07.17                                (ISO 8601 YYYY.MM.DD)
 * @since       1.6                                       (The Java version used)
 * 
 * @to.do
 * TODO Parse entrance flags
 * TODO Parse fix flags
 * TODO Parse calibration comments fields
 * TODO Parse team fields
 * TODO Parse instrument fields
 * TODO Parse units declarations
 * TODO Add support for reading multiple files using includes to build up the survey
 */
public class SurvexParser {


	/**
	 * Parse survex format data into the cave data model
	 * 
	 * @param surveyFileData ListArray of data lines from a survex file
     * @return Cave Survey object
	 * @throws ParseException 
	 */
	public CaveSurvey parseSurvexFile( List<String> surveyFileData ) throws ParseException {
		/**
		 * Read state codes:
		 * 0=starting new file
		 * 1=inside begin/end block
		 * 2=LRUD block
		 */
		
		int state = 0;
		
		//Create new list of survey series to hold data
		CaveSurvey allSeries = new CaveSurvey();
		
		//Create stack to hold open series while processing data lines
		List<SurveySeries> seriesStack = new ArrayList<SurveySeries>();
		
		//Create stack to hold open series names while processing data lines
		List<String> nameStack = new ArrayList<String>();
		
		//Create list for equates
		List<Equate> equates = new ArrayList<Equate>();
		
		//Create a series instance to use as a pointer to the active series data is being read from
		SurveySeries liveSeries = null;
		
		//Declare flags to store status of legs being read
		boolean duplicateFlag = false;
		boolean splayFlag = false;
		boolean surfaceFlag = false;

		//Loop through all data lines
		for ( int i=0; i < surveyFileData.size(); i++ ) {
			int lineNo = i + 1;
			String dataLine = surveyFileData.get(i);
			
			//Discard text after comment character 
			//TODO Keep comments and add into data model for legs, series and file heading
			int commentPos = dataLine.indexOf(';');
			if ( commentPos > -1 ) {
				dataLine = dataLine.substring(0, commentPos);
			}
			//Trim whitespace off line ends
			dataLine = dataLine.trim();
			//Skip blank lines
			if ( dataLine.length() > 0 ) {
				//Check line for commands
				if ( dataLine.charAt(0) == '*' ) {
					//Process line into individual items
					String[] data = cleanAndSplitData(dataLine);
					//Get command keyword
					String cmd = data[0].substring(1);
					
					//Check for expected commands
					if ( cmd.compareToIgnoreCase("BEGIN") == 0 ) {
						//Start of new block
						state = 1;
						if (data.length == 2) {
							//Create series
							liveSeries = new SurveySeries(data[1]);
							//Reset flags for new series
							duplicateFlag = false;
							splayFlag = false;
							surfaceFlag = false;
							//Add name to stack
							nameStack.add(data[1]);
							//Get calibrations from last series and apply to new child series (if present)
							Double tapeCal = 0.0;
							Double compassCal = 0.0;
							Double clinoCal = 0.0;
							Double declinationCal = 0.0;
							//Check if a parent series exists
							if (seriesStack.size() > 0) {
								SurveySeries series = seriesStack.get( seriesStack.size() - 1 );
								tapeCal = series.getTapeCalibration();
								compassCal = series.getCompassCalibration();
								clinoCal = series.getClinoCalibration();
								declinationCal = series.getDeclination();
							}
							//Put new series onto stack
							seriesStack.add(liveSeries);
							//Apply calibrations from parent series. 
							//These will get overwritten if this series has it's own calibrations.
							liveSeries.setTapeCalibration(tapeCal);
							liveSeries.setCompassCalibration(compassCal);
							liveSeries.setClinoCalibration(clinoCal);
							liveSeries.setDeclination(declinationCal);
						}
						else if (data.length < 2) {
							//TODO support begin/end blocks without names
							throw new ParseException( formatError("BEGIN/END blocks without names are not supported.", lineNo ), lineNo );
						}
						else {
							//Do not support begin/end blocks names with white space
							throw new ParseException( formatError("BEGIN/END blocks names containing spaces are not supported.", lineNo ), lineNo );
						}
					}
					else if ( cmd.compareToIgnoreCase("END") == 0 ) {
						//End block
						String blockEndName = data[1];
						//Check end matches end of section name
						String currentBlockName = nameStack.get( nameStack.size() - 1 );
						if ( currentBlockName.compareToIgnoreCase(blockEndName) == 0 ) {
							//Found matching end block, so close series
							//Remove live series from stack, as it is closed
							SurveySeries endedSeries = seriesStack.remove( seriesStack.size() - 1 );
							nameStack.remove( nameStack.size() - 1 );
							if ( seriesStack.size() > 0) {
								//Series is inside another, so make that live and add finished series to it
								liveSeries = seriesStack.get( seriesStack.size() - 1 );
								liveSeries.addSeries( endedSeries );
								//Return state to 1
								state = 1;
							}
							else {
								//No other series on stack, so add to main cave survey
								allSeries.add( endedSeries );
								//Clear reference to live series
								liveSeries = null;
								//Return state to 0
								state = 0;
							}
						}
						else {
							//Names of begin end blocks do not match
							throw new ParseException( formatError("Names of begin end blocks do not match. Begin=" + 
									currentBlockName + " End=" + blockEndName + ".", lineNo ), lineNo );
						}
					}
					else if ( cmd.compareToIgnoreCase("EQUATE") == 0 ) {
						//Get two parts of the equate, expanding to full nested series names
						String fullSeriesPrefix = fullNestedSeriesName(nameStack);
						Equate equate = new Equate( fullSeriesPrefix, data[1], fullSeriesPrefix, data[2] );
						//Add to cache
						equates.add(equate);
					}
					else if ( cmd.compareToIgnoreCase("DATA") == 0 ) {
						//Check data command type
						if ( data[1].compareToIgnoreCase("PASSAGE") == 0 ) {
							//LRUD data block
							state = 2;
						}
						else if ( data[1].compareToIgnoreCase("NORMAL") == 0 ) {
							state = 1;
							//Check data order
							//TODO Add support for different data line ordering than default
							CaveConverter.logMessage( "Found normal data line. Checking format." );
							boolean dataOrderOk = false;
							if ( data.length > 6 ) {
								//Check supported order
								if ( data[2].compareToIgnoreCase("FROM") == 0 ) {
									if ( data[3].compareToIgnoreCase("TO") == 0 ) {
										if ( ( data[4].compareToIgnoreCase("TAPE") == 0 )
										|| ( data[4].compareToIgnoreCase("LENGTH") == 0 ) ) {
											if ( ( data[5].compareToIgnoreCase("COMPASS") == 0 )
											|| ( data[5].compareToIgnoreCase("BEARING") == 0 ) ) {
												if ( ( data[6].compareToIgnoreCase("CLINO") == 0 )
												|| ( data[6].compareToIgnoreCase("GRADIENT") == 0 ) ) {
													//Data order is fine, so we can support this line
													dataOrderOk = true;
												}
											}
										}
									}
								}
								
							}
							if ( dataOrderOk == false ) {
								//Other data order is not supported.
								throw new ParseException( formatError("Unsupported survex data order. Only '*data normal from to tape compass clino' ordering is supported.", lineNo ), lineNo );
							}
						}
						else {
							//Other data settings not currently supported (assumes file use default order)
							throw new ParseException( formatError("Unsupported survex data command: " + data[2], lineNo ), lineNo );
						}
					}
					else if ( cmd.compareToIgnoreCase("CALIBRATE") == 0 ) {
						//Process calibration command
						if (data.length == 3) {
							String type = data[1];
							Double value = Double.valueOf( data[2] );
							if ( type.compareToIgnoreCase("tape") == 0 ) {
								//Set tape calibration in active series
								liveSeries.setTapeCalibration(value);
							}
							else if ( type.compareToIgnoreCase("declination") == 0 ) {
								//Set declination calibration in active series
								liveSeries.setDeclination(value);
							}
							else if ( type.compareToIgnoreCase("compass") == 0 ) {
								//Set compass calibration in active series
								liveSeries.setCompassCalibration(value);
							}
							else if ( type.compareToIgnoreCase("clino") == 0 ) {
								//Set compass calibration in active series
								liveSeries.setClinoCalibration(value);
							}
							//TODO Add support for calibration scale factors
						}
						else {
							//Invalid calibration lie
							throw new ParseException( formatError("CALIBRATE command did not contain a single instrument type plus value.", lineNo ), lineNo );
						}
					}
					else if ( cmd.compareToIgnoreCase("DATE") == 0 ) {
						//Process date
						if (data.length == 2) {
							Date value = UtilityFunctions.stringToDate(data[1], "yyyy.MM.dd");
							liveSeries.setSurveyDate(value);
						}
					}
					else if ( cmd.compareToIgnoreCase("FLAGS") == 0 ) {
						//Process flags
						boolean notPrefixed = false;
						for (int iFlags = 1; iFlags < data.length; iFlags++) {
							//Read all flags settings to determine what is being turned on or off
							if (data[iFlags].compareToIgnoreCase("NOT") == 0 ) { 
								notPrefixed = true;
							}
							else if (data[iFlags].compareToIgnoreCase("DUPLICATE") == 0 ) { 
								duplicateFlag = (notPrefixed == false);
								notPrefixed = false;
							}
							else if (data[iFlags].compareToIgnoreCase("SPLAY") == 0 ) { 
								splayFlag = (notPrefixed == false);
								notPrefixed = false;
							}
							else if (data[iFlags].compareToIgnoreCase("SURFACE") == 0 ) { 
								surfaceFlag = (notPrefixed == false);
								notPrefixed = false;
							}
							else { 
								//Reset notPrefixed flag if any other value
								notPrefixed = false;
							}
						}
					}
					else {
						//Ignore other commands inside begin end block
						//TODO Add support for FIX stations
						//TODO Add support for ENTRANCE stations
						//TODO Add support for UNITS grads, feet, other common non-default units including topofil clino range
						CaveConverter.logMessage("Unsupported Survex command ignored: " + cmd);
					}
				}
				else {
					//Data line
					//CaveConverter.logMessage("Data line " + CaveConverter.padNumber(lineNo, 4) + ": " + dataLine);
					
					if ( liveSeries != null ) {
						//Process line into individual items
						String[] data = cleanAndSplitData(dataLine);
						
						switch (state) {
						case 1:
							//Create new survey leg
							SurveyLeg leg = new SurveyLeg();
							
							//Create record from the items
							int index = 0;
							while ( index < data.length ) {
								String item = data[index];
								//Check for end of line
								if ( item.charAt(0) == ';' ) {
									//Comments, so ignore rest of line
									//TODO Add support for leg comments
									index = data.length;
								}
								else {
									//Put item into appropriate value
									switch ( index ) {
									case 0:
										//TODO Add support for retaining station name when not a number
										leg.setFromStn( UtilityFunctions.createStationFromNameForSeries( data[index], liveSeries ) );
										break;
									case 1:
										//TODO Add support for retaining station name when not a number
										leg.setToStn( UtilityFunctions.createStationFromNameForSeries( data[index], liveSeries ) );
										break;
									case 2:
										leg.setLength( Double.parseDouble( data[index] ) );
										break;
									case 3:
										if ( data[index].compareTo("-") == 0 ) {
											leg.setCompass(0);
										}
										else {
											leg.setCompass( Double.parseDouble( data[index] ) );
										}
										break;
									case 4:
										//Trim data item after ';' if present
										String val = data[index];
										int commentCharPos = val.indexOf(';');
										if ( commentCharPos > 0 ) {
											val = val.substring(0, commentCharPos);
										}
										
										if ( val.compareToIgnoreCase("-V") == 0 ) {
											leg.setClino(-90);
										}
										else if ( val.compareToIgnoreCase("down") == 0 ) {
											leg.setClino(-90);
										}
										else if ( val.compareToIgnoreCase("d") == 0 ) {
											leg.setClino(-90);
										}
										else if ( val.compareToIgnoreCase("+V") == 0 ) {
											leg.setClino(90);
										}
										else if ( val.compareToIgnoreCase("up") == 0 ) {
											leg.setClino(90);
										}
										else if ( val.compareToIgnoreCase("u") == 0 ) {
											leg.setClino(90);
										}
										else if ( val.compareToIgnoreCase("-") == 0 ) {
											leg.setClino(0);
										}
										else if ( val.compareToIgnoreCase("level") == 0 ) {
											leg.setClino(0);
										}
										else {
											leg.setClino( Double.parseDouble( val ) );
										}
										break;
									}
								}
								
								index++;
							}
							
							//Check leg was found
							if ( leg.getLength() > -1 ) {
								//Set flags for leg
								leg.setDuplicate(duplicateFlag);
								leg.setSplay(splayFlag);
								leg.setSurface(surfaceFlag);
								//Add leg to series
								liveSeries.addLeg(leg);
							}
							break;
						case 2:
							//Add data to LRUD cache
							/**
							 * TODO Need to store all the LRUD lines in groups to match with 
							 * the legs once the series is complete. Need to match legs to lrud
							 * from two consecutive lines to be sure the LRUD is for that leg, and
							 * not another leg from the same station. Create the LRUD group in
							 * the command parsing switch, as that is where we know a new LRUD group has 
							 * been started.
							 */
							break;
						}
						
					}
					else {
						//Data line outside of series
						throw new ParseException( formatError("Data line found outside of any begin/end block.", lineNo ), lineNo );
					}
					
				}
			}
		}
		
		//Process equates
		UtilityFunctions.processEquates(equates, allSeries);
		
		//Debug dump
		UtilityFunctions.logSurveyDebugData(allSeries);
	
		//Completed file parsing
		return allSeries;
	}

	//Get series names in stack to generate full series name
	private String fullNestedSeriesName(List<String> stack) {
		String name = "";
		
		Iterator<String> stackIterator = stack.listIterator();
		while ( stackIterator.hasNext() ) {
			name += "." + stackIterator.next();
		}
		//Remove initial dot
		return name.substring(1);
	}
	
	/*
	 * cleanAndSplitData
	 * 
	 * Removes whitespace and splits a line into items separated by whitespace
	 * Returns and array of items from the input data
	 */
	private String[] cleanAndSplitData( String dataIn ) {
		//Process all white space down to single space chars
		String dataLine = dataIn.replace('\t', ' ');
		while ( dataLine.contains("  ") ) {
			dataLine = dataLine.replaceAll("  ", " ");
		}
		return dataLine.split(" ");
	}

	private String formatError( String message, int lineNo ) {
		String msg = message;
		
		msg += " At line: " + lineNo;
		return msg; 
	}
	

}
