#
# Copyright (C) 2005  Robert Collins  <robertc@squid-cache.org>
# 
# 
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

import os
import shutil
import tempfile

try:
    import pybaz
except ImportError:
    pybaz = None

import config_manager.implementations


class ArchVCS(object):
    """An Arch VCS implementation."""

    def __init__(self):
        self.schemes = ["arch"]

    def checkout(self, url, to):
        if not url.startswith('arch://'):
            return False
        try:
            pybaz.get(url[7:], to)
            return True
        except pybaz.errors.ExecProblem, e:
            shutil.rmtree(to, ignore_errors=True)
            raise config_manager.implementations.UnusableURL(
                "Error retrieving %r: '%s'." % (url, e))
        except pybaz.errors.NamespaceError:
            #os.rmdir(to)
            return False

    def get_test_repo(self):
        return ArchVCSRepo()


class ArchVCSRepo(object):
    """A stub repo for the arch: scheme, tested implicitly by use."""

    def setUp(self):
        """Prepare the repo for use."""
        self.tempdir = tempfile.mkdtemp()
        self.oldhome = None
        if "HOME" in os.environ:
            self.oldhome = os.environ["HOME"]
        os.environ["HOME"] = self.tempdir
        pybaz.set_my_id("<demo@example.com>")
        archive = pybaz.make_archive('anarchive@example.com',
                                     os.path.join(self.tempdir, 'archive'))
        import_tree = os.path.join (self.tempdir, 'import')
        os.mkdir(import_tree)
        tree = pybaz.init_tree(import_tree,
                               'anarchive@example.com/foo--0')
        tree.import_()

    def tearDown(self):
        """Delete any disk resources used by the repo."""
        if self.oldhome is None:
            del os.environ["HOME"]
        else:
            os.environ["HOME"] = self.oldhome
        shutil.rmtree(self.tempdir, ignore_errors=True)

    def get_branch_url(self):
        return "arch://anarchive@example.com/foo--0"

    def get_not_branch_url(self):
        # we classify unregistered branches as 
        # not branches, and bad namespaces likewise.
        return "arch://illegal/foo--0"

    def get_bad_url(self):
        return "arch://anarchive@example.com/bar--0"
        

if pybaz is not None:
    config_manager.implementations.register(ArchVCS())
