/* $Id: kmo_flat-test.c,v 1.12 2013-10-08 11:18:56 aagudo Exp $
 *
 * This file is part of the KMOS Library
 * Copyright (C) 2002-2006 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

/*
 * $Author: aagudo $
 * $Date: 2013-10-08 11:18:56 $
 * $Revision: 1.12 $
 * $Name: not supported by cvs2svn $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <string.h>
#include <sys/stat.h>
#include <math.h>

#include <cpl.h>

#include "kmclipm_functions.h"
#include "kmclipm_constants.h"

#include "kmo_dfs.h"

const char  *test_global_path_test_data;
float   test_global_seed_data, test_global_seed_noise;
int     test_global_size_x, test_global_size_y, test_global_size_z, test_global_nr_frames;
void kmo_get_pipe_command(char*, const char*, const char*, int);
int kmo_test_file_exists(const char*);
void kmo_test_verbose_off();
void kmo_test_verbose_on();
double kmo_test_esorex_data(const char*, int);
const char* kmo_test_cat_strings(char*, const char*, const char*);
int  kmo_test_create_RAW_data(int, int, const char*, char**, char**, int*, int, char**, char**, int*, int, float, float);
int  kmo_test_create_RAW_data_zero(int, int, const char*);
int  kmo_test_create_F2D_data(int, int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F2D_data_noise(int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F2I_data(int, int, int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F2I_data_noise(int, int, int, const char*, char *, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F3I_data(int, int, int, int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F3I_data_infinite(int, int, int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F3I_data2(int, int, int, int, int*, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F3I_data_noise(int, int, int, int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F1I_data(int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F1I_data_noise(int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_B2D_data(int, int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);

const char      *path_recipe        = "kmo_flat/";

const char      *valid_files[]      = {"v_raw_123_on.fits",
                                       "v_raw_234_on.fits",
                                       "v_raw_345_on.fits",
                                       "v_raw_123_off.fits",
                                       "v_raw_234_off.fits",
                                       "v_raw_345_off.fits",
                                       "flat_on_123.fits",
                                       "flat_on_231.fits",
                                       "flat_on_312.fits",
                                       "flat_off_123.fits",
                                       "flat_off_231.fits",
                                       "flat_off_312.fits"};

const char      *invalid_files[]    = {"i_data_F2D.fits"};

/**
    @defgroup kmo_flat   kmo_flat unit tests

    @{
*/

/**
 * @brief test with data and mask
 */
static void test_flat(const char *path_op_on1,
                      const char *path_op_on2,
                      const char *path_op_on3,
                      const char *path_op_off1,
                      const char *path_op_off2,
                      const char *path_op_off3,
                      int small,
                      int ret_val)
{
    // create sof-file
    const char *sof_path = "test_flat.sof";
    FILE *fh = fopen(sof_path, "w");
    fprintf (fh, "%s   FLAT_ON\n%s   FLAT_ON\n%s   FLAT_ON\n"
                 "%s   FLAT_OFF\n%s   FLAT_OFF\n%s   FLAT_OFF\n",
                 path_op_on1, path_op_on2, path_op_on3,
                 path_op_off1, path_op_off2, path_op_off3);

    if (small == 1) {
        fprintf (fh, "badpixel_dark_small.fits   BADPIXEL_DARK\n");
    } else  {
        fprintf (fh, "badpixel_dark.fits   BADPIXEL_DARK\n");
    }
    fclose(fh);

    // create esorex-command
    char esorex_command[256];
    sprintf(esorex_command,
            "esorex --suppress-prefix=TRUE kmo_flat %s",
            sof_path);

    kmo_get_pipe_command(esorex_command,
                         "log_kmo_flat.txt", sof_path, TRUE);

    // execute esorex-command
    if (ret_val == 0) {
        cpl_test_eq(0, system(esorex_command));
    } else {
        cpl_test_noneq(0, system(esorex_command));
    }
}

static void test_flat1(const char *path_op_on1,
                       const char *path_op_off1,
                       int small,
                       int ret_val)
{
    // create sof-file
    const char *sof_path = "test_flat1.sof";
    FILE *fh = fopen(sof_path, "w");
    fprintf (fh, "%s   FLAT_ON\n"
                 "%s   FLAT_OFF\n",
                 path_op_on1,
                 path_op_off1);

    if (small == 1) {
        fprintf (fh, "badpixel_dark_small.fits   BADPIXEL_DARK\n");
    } else  {
        fprintf (fh, "badpixel_dark.fits   BADPIXEL_DARK\n");
    }
    fclose(fh);

    // create esorex-command
    char esorex_command[256];
    sprintf(esorex_command,
            "esorex --suppress-prefix=TRUE kmo_flat %s", sof_path);


    kmo_get_pipe_command(esorex_command,
                         "log_kmo_flat.txt", sof_path, TRUE);

    // execute esorex-command
    if (ret_val == 0) {
        cpl_test_eq(0, system(esorex_command));
    } else {
        cpl_test_noneq(0, system(esorex_command));
    }
}

static void test_flat2(const char *path_op_on1,
                      const char *path_op_on2,
                      const char *path_op_off1,
                      const char *path_op_off2,
                      int small,
                      int ret_val)
{
    // create sof-file
    const char *sof_path = "test_flat2.sof";
    FILE *fh = fopen(sof_path, "w");
    fprintf (fh, "%s   FLAT_ON\n%s   FLAT_ON\n"
                 "%s   FLAT_OFF\n%s   FLAT_OFF\n",
                 path_op_on1, path_op_on2,
                 path_op_off1, path_op_off2);

    if (small == 1) {
        fprintf (fh, "badpixel_dark_small.fits   BADPIXEL_DARK\n");
    } else  {
        fprintf (fh, "badpixel_dark.fits   BADPIXEL_DARK\n");
    }
    fclose(fh);

    // create esorex-command
    char esorex_command[256];
    sprintf(esorex_command,
            "esorex --suppress-prefix=TRUE kmo_flat -cmethod=sum %s", sof_path);

    kmo_get_pipe_command(esorex_command,
                         "log_kmo_flat.txt", sof_path, TRUE);

    // execute esorex-command
    if (ret_val == 0) {
        cpl_test_eq(0, system(esorex_command));
    } else {
        cpl_test_noneq(0, system(esorex_command));
    }
}

static void test_flat23(const char *path_op_on1,
                      const char *path_op_on2,
                      const char *path_op_off1,
                      const char *path_op_off2,
                      const char *path_op_off3,
                      int small,
                      int ret_val)
{
    // create sof-file
    const char *sof_path = "test_flat23.sof";
    FILE *fh = fopen(sof_path, "w");
    fprintf (fh, "%s   FLAT_ON\n%s   FLAT_ON\n"
                 "%s   FLAT_OFF\n%s   FLAT_OFF\n%s   FLAT_OFF\n",
                 path_op_on1, path_op_on2,
                 path_op_off1, path_op_off2, path_op_off3);

    if (small == 1) {
        fprintf (fh, "badpixel_dark_small.fits   BADPIXEL_DARK\n");
    } else  {
        fprintf (fh, "badpixel_dark.fits   BADPIXEL_DARK\n");
    }
    fclose(fh);

    // create esorex-command
    char esorex_command[256];
    sprintf(esorex_command,
            "esorex --suppress-prefix=TRUE kmo_flat -cmethod=sum %s", sof_path);

    kmo_get_pipe_command(esorex_command,
                         "log_kmo_flat.txt", sof_path, TRUE);

    // execute esorex-command
    if (ret_val == 0) {
        cpl_test_eq(0, system(esorex_command));
    } else {
        cpl_test_noneq(0, system(esorex_command));
    }
}

static void test_flat32(const char *path_op_on1,
                      const char *path_op_on2,
                      const char *path_op_on3,
                      const char *path_op_off1,
                      const char *path_op_off2,
                      int small,
                      int ret_val)
{
    // create sof-file
    const char *sof_path = "test_flat32.sof";
    FILE *fh = fopen(sof_path, "w");
    fprintf (fh, "%s   FLAT_ON\n%s   FLAT_ON\n%s   FLAT_ON\n"
                 "%s   FLAT_OFF\n%s   FLAT_OFF\n",
                 path_op_on1, path_op_on2, path_op_on3,
                 path_op_off1, path_op_off2);

    if (small == 1) {
        fprintf (fh, "badpixel_dark_small.fits   BADPIXEL_DARK\n");
    } else  {
        fprintf (fh, "badpixel_dark.fits   BADPIXEL_DARK\n");
    }
    fclose(fh);

    // create esorex-command
    char esorex_command[256];
    sprintf(esorex_command,
            "esorex --suppress-prefix=TRUE kmo_flat -cmethod=sum %s", sof_path);

    kmo_get_pipe_command(esorex_command,
                         "log_kmo_flat.txt", sof_path, TRUE);

    // execute esorex-command
    if (ret_val == 0) {
        cpl_test_eq(0, system(esorex_command));
    } else {
        cpl_test_noneq(0, system(esorex_command));
    }
}

void check_headers(const char *path, int ext, double tol) {
    cpl_propertylist *h = kmclipm_propertylist_load(path, ext);

    cpl_test_eq(cpl_propertylist_get_int(h, QC_FLAT_SAT), 3);
    cpl_test_abs(cpl_propertylist_get_double(h, QC_FLAT_EFF),
                 26.6846, tol*10);
    cpl_test_abs(cpl_propertylist_get_double(h, QC_FLAT_SN), 5579.42, tol*400);
    cpl_test_abs(cpl_propertylist_get_double(h, QC_GAP_MEAN),
                 4.93731, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, QC_GAP_SDV),
                 0.0698977, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, QC_GAP_MAXDEV),
                 0.187894, tol*40);
    cpl_test_abs(cpl_propertylist_get_double(h, QC_SLIT_MEAN),
                 12.8606, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, QC_SLIT_SDV),
                 0.0653653, tol);
    cpl_test_abs(cpl_propertylist_get_double(h, QC_SLIT_MAXDEV),
                 0.190067, tol*20);

    cpl_propertylist_delete(h); h = NULL;
}

/**
    @brief
        Generates test data for kmo_flat.

    @param path Path where the generated test date should be saved to.
*/
static int kmo_generate_test_data_flat(const char *path)
{
    char        file_path[256];

    int         nr_primary_keys     = 7,
                nr_sub_keys         = 1;

    int         size_x              = 300,
                size_y              = 509;

    char        *primary_keys[nr_primary_keys],
                *primary_vals[nr_primary_keys],
                *sub_keys[nr_sub_keys],
                *sub_vals[nr_sub_keys];
    int         primary_types[nr_primary_keys],
                sub_types[nr_primary_keys];

    /* ----- valid test data ----- */
    primary_keys[0] = INS_LAMP1_ST;
    primary_types[0] = CPL_TYPE_BOOL;
    primary_vals[0] = "0";

    primary_keys[1] = INS_LAMP2_ST;
    primary_types[1] = CPL_TYPE_BOOL;
    primary_vals[1] = "0";

    sub_keys[0] = EXPTIME;
    sub_types[0] = CPL_TYPE_DOUBLE;
    sub_vals[0] = "1.0";

    // Lamp on
    primary_keys[2] = INS_LAMP3_ST;
    primary_types[2] = CPL_TYPE_BOOL;
    primary_vals[2] = "1";

    primary_keys[4] = EXPTIME;
    primary_types[4] = CPL_TYPE_DOUBLE;
    primary_vals[4] = "1.0";

    primary_keys[5] = "ESO INS FILT1 ID";
    primary_types[5] = CPL_TYPE_STRING;
    primary_vals[5] = "K";

    primary_keys[6] = "ESO INS FILT2 ID";
    primary_types[6] = CPL_TYPE_STRING;
    primary_vals[6] = "K";

    primary_keys[3] = "ESO INS FILT3 ID";
    primary_types[3] = CPL_TYPE_STRING;
    primary_vals[3] = "K";

    cpl_test_eq(0,
        kmo_test_create_RAW_data(size_x,
                                 size_y,
                               kmo_test_cat_strings(file_path, path, valid_files[0]),
                                 primary_keys,
                                 primary_vals,
                                 primary_types,
                                 nr_primary_keys,
                                 sub_keys,
                                 sub_vals,
                                 sub_types,
                                 nr_sub_keys,
                                 2.1, 0.1));

    cpl_test_eq(0,
        kmo_test_create_RAW_data(size_x,
                                 size_y,
                               kmo_test_cat_strings(file_path, path, valid_files[1]),
                                 primary_keys,
                                 primary_vals,
                                 primary_types,
                                 nr_primary_keys,
                                 sub_keys,
                                 sub_vals,
                                 sub_types,
                                 nr_sub_keys,
                                 3.3, 0.2));

    cpl_test_eq(0,
        kmo_test_create_RAW_data(size_x,
                                 size_y,
                               kmo_test_cat_strings(file_path, path, valid_files[2]),
                                 primary_keys,
                                 primary_vals,
                                 primary_types,
                                 nr_primary_keys,
                                 sub_keys,
                                 sub_vals,
                                 sub_types,
                                 nr_sub_keys,
                                 4.5, 0.3));

    // Lamp off
    primary_keys[2] = INS_LAMP3_ST;
    primary_types[2] = CPL_TYPE_BOOL;
    primary_vals[2] = "0";

    primary_keys[3] = INS_LAMP4_ST;
    primary_types[3] = CPL_TYPE_BOOL;
    primary_vals[3] = "0";

    primary_keys[4] = EXPTIME;
    primary_types[4] = CPL_TYPE_DOUBLE;
    primary_vals[4] = "1.0";

    cpl_test_eq(0,
        kmo_test_create_RAW_data(size_x,
                                 size_y,
                               kmo_test_cat_strings(file_path, path, valid_files[3]),
                                 primary_keys,
                                 primary_vals,
                                 primary_types,
                                 nr_primary_keys-3, // set no filter id
                                 sub_keys,
                                 sub_vals,
                                 sub_types,
                                 nr_sub_keys,
                                 1.1, 0.4));

    cpl_test_eq(0,
        kmo_test_create_RAW_data(size_x,
                                 size_y,
                               kmo_test_cat_strings(file_path, path, valid_files[4]),
                                 primary_keys,
                                 primary_vals,
                                 primary_types,
                                 nr_primary_keys-3, // set no filter id
                                 sub_keys,
                                 sub_vals,
                                 sub_types,
                                 nr_sub_keys,
                                 1.3, 0.5));

    cpl_test_eq(0,
        kmo_test_create_RAW_data(size_x,
                                 size_y,
                               kmo_test_cat_strings(file_path, path, valid_files[5]),
                                 primary_keys,
                                 primary_vals,
                                 primary_types,
                                 nr_primary_keys-3, // set no filter id
                                 sub_keys,
                                 sub_vals,
                                 sub_types,
                                 nr_sub_keys,
                                 1.5, 0.6));

    /* ----- invalid test data ----- */
    // F2D
    cpl_test_eq(0,
                kmo_test_create_F2D_data(size_x, size_y, FALSE,
                        kmo_test_cat_strings(file_path, path, invalid_files[0]),
                        NULL, NULL, NULL, 0,
                        NULL, NULL, NULL, 0));

    return 0;
}


/**
  @brief    Test of kmo_flat recipe.
  @param    argc   the number of parameters
  @param    argv   the parameter list

  Call @c kmo_flat-test @c --generate when only the test data for this
  recipe should be generated.
  Call @c kmo_flat-test @c --no-gen when only the tests should be
  executed and no data should be generated.

 */
int main(int argc, char *argv[])
{
    float   tol     = 0.01;

    char    test_path[256],
            file_path[256],
            tmp[256],
            op1_file_on[256],
            op2_file_on[256],
            op3_file_on[256],
            op1_file_off[256],
            op2_file_off[256],
            op3_file_off[256];

    cpl_propertylist *h = NULL;

    cpl_test_init("<kmos-spark@mpe.mpg.de>", CPL_MSG_WARNING);

    strcpy(test_path, test_global_path_test_data);
    strcat(test_path, path_recipe);

    if (kmo_test_file_exists(test_path) == FALSE) {
        mkdir(test_path, 0777);
    }

    if ((argc == 1) || ((argc > 1) && (strcmp(argv[1], "--no-gen") != 0))) {
        cpl_test_zero(kmo_generate_test_data_flat(test_path));
    }

    if ((argc > 1) && (strcmp(argv[1], "--generate") == 0)) {
        // just generate the data
        return cpl_test_end(0);
    }

    /* ----- run the tests ----- */
    kmo_test_cat_strings(op1_file_on, test_path, valid_files[0]);
    kmo_test_cat_strings(op2_file_on, test_path, valid_files[1]);
    kmo_test_cat_strings(op3_file_on, test_path, valid_files[2]);
    kmo_test_cat_strings(op1_file_off, test_path, valid_files[3]);
    kmo_test_cat_strings(op3_file_off, test_path, valid_files[5]);

    FILE *fd = fopen("log_kmo_flat.txt", "w");
    fprintf(fd, "%s\n", " ");
    fclose(fd);

    /* ----- invalid tests ----- */
    cpl_msg_info(cpl_func, "Testing kmo_flat with invalid data.");

    // invalid data (F2D)
    kmo_test_cat_strings(op2_file_off, test_path, invalid_files[0]);
    test_flat(op1_file_on, op2_file_on, op3_file_on,
              op1_file_off, op2_file_off, op3_file_off, 1, -1);

    // invalid data (doesn't conatin any slitlets)
    kmo_test_cat_strings(op2_file_off, test_path, valid_files[4]);
    test_flat(op1_file_on, op2_file_on, op3_file_on,
              op1_file_off, op2_file_off, op3_file_off, 1, -1);

    /* ----- valid tests ----- */
    //
    //  PIPELINE TEST (must be called after kmo_dark-test, since the
    //  following recipes rely on this output)
    //
    cpl_msg_info(cpl_func, "Testing kmo_flat with real pipeline data.");
    sprintf(test_path, "%s%s", test_global_path_test_data, "pipeline/");

    kmo_test_cat_strings(op1_file_on, test_path, valid_files[6]);
    kmo_test_cat_strings(op2_file_on, test_path, valid_files[7]);
    kmo_test_cat_strings(op3_file_on, test_path, valid_files[8]);
    kmo_test_cat_strings(op1_file_off, test_path, valid_files[9]);
    kmo_test_cat_strings(op2_file_off, test_path, valid_files[10]);
    kmo_test_cat_strings(op3_file_off, test_path, valid_files[11]);

    //
    // test with 1 FLAT/DARK frame
    //
    test_flat1(op1_file_on,
               op1_file_off, 0,  0);

    strcpy(tmp, MASTER_FLAT);
    sprintf(file_path, "%s_KKK.fits", kmo_strlower(tmp));

    h = kmclipm_propertylist_load(file_path, 1);
    cpl_test_eq(strcmp(cpl_propertylist_get_string(h, EXTNAME), "DET.1.DATA"), 0);
    cpl_test_abs(0.0, cpl_propertylist_get_double(h, QC_FLAT_SN), tol);
    cpl_test_eq(0, cpl_propertylist_get_int(h, QC_FLAT_SAT));
    cpl_test_abs(26.6846, cpl_propertylist_get_double(h, QC_FLAT_EFF), tol*10);
    cpl_test_abs(4.93827, cpl_propertylist_get_double(h, QC_GAP_MEAN), tol);
    cpl_test_abs(0.0698976, cpl_propertylist_get_double(h, QC_GAP_SDV), tol);
    cpl_test_abs(0.187887, cpl_propertylist_get_double(h, QC_GAP_MAXDEV), tol);
    cpl_test_abs(12.865489, cpl_propertylist_get_double(h, QC_SLIT_MEAN), tol);
    cpl_test_abs(0.0621559, cpl_propertylist_get_double(h, QC_SLIT_SDV), tol);
    cpl_test_abs(0.19006, cpl_propertylist_get_double(h, QC_SLIT_MAXDEV), tol*2.1);
    cpl_propertylist_delete(h); h = NULL;
    h = kmclipm_propertylist_load(file_path, 2);
    cpl_test_eq(strcmp(cpl_propertylist_get_string(h, EXTNAME), "DET.1.NOISE"), 0);
    cpl_propertylist_delete(h); h = NULL;
    h = kmclipm_propertylist_load(file_path, 3);
    cpl_test_eq(strcmp(cpl_propertylist_get_string(h, EXTNAME), "DET.2.DATA"), 0);
    cpl_propertylist_delete(h); h = NULL;
    h = kmclipm_propertylist_load(file_path, 4);
    cpl_test_eq(strcmp(cpl_propertylist_get_string(h, EXTNAME), "DET.2.NOISE"), 0);
    cpl_propertylist_delete(h); h = NULL;
    h = kmclipm_propertylist_load(file_path, 5);
    cpl_test_eq(strcmp(cpl_propertylist_get_string(h, EXTNAME), "DET.3.DATA"), 0);
    cpl_propertylist_delete(h); h = NULL;
    h = kmclipm_propertylist_load(file_path, 6);
    cpl_test_eq(strcmp(cpl_propertylist_get_string(h, EXTNAME), "DET.3.NOISE"), 0);
    cpl_propertylist_delete(h); h = NULL;

    //
    // test with 2 FLAT/DARK frames
    //
    test_flat2(op1_file_on, op2_file_on,
               op1_file_off, op2_file_off, 0, 0);

    strcpy(tmp, MASTER_FLAT);
    sprintf(file_path, "%s_KKK.fits", kmo_strlower(tmp));

    h = kmclipm_propertylist_load(file_path, 1);
    cpl_test_eq(strcmp(cpl_propertylist_get_string(h, EXTNAME), "DET.1.DATA"), 0);
    cpl_test_abs(6958.76, cpl_propertylist_get_double(h, QC_FLAT_SN), tol*1000);
    cpl_test_eq(3, cpl_propertylist_get_int(h, QC_FLAT_SAT));
    cpl_test_abs(53.3692, cpl_propertylist_get_double(h, QC_FLAT_EFF), tol*10);
    cpl_test_abs(4.93827, cpl_propertylist_get_double(h, QC_GAP_MEAN), tol);
    cpl_test_abs(0.0698928, cpl_propertylist_get_double(h, QC_GAP_SDV), tol);
    cpl_test_abs(0.187897, cpl_propertylist_get_double(h, QC_GAP_MAXDEV), tol);
    cpl_test_abs(12.865489, cpl_propertylist_get_double(h, QC_SLIT_MEAN), tol);
    cpl_test_abs(0.0590724, cpl_propertylist_get_double(h, QC_SLIT_SDV), tol);
    cpl_test_abs(0.190071, cpl_propertylist_get_double(h, QC_SLIT_MAXDEV), tol*2.1);
    cpl_propertylist_delete(h); h = NULL;
    h = kmclipm_propertylist_load(file_path, 2);
    cpl_test_eq(strcmp(cpl_propertylist_get_string(h, EXTNAME), "DET.1.NOISE"), 0);
    cpl_propertylist_delete(h); h = NULL;
    h = kmclipm_propertylist_load(file_path, 3);
    cpl_test_eq(strcmp(cpl_propertylist_get_string(h, EXTNAME), "DET.2.DATA"), 0);
    cpl_propertylist_delete(h); h = NULL;
    h = kmclipm_propertylist_load(file_path, 4);
    cpl_test_eq(strcmp(cpl_propertylist_get_string(h, EXTNAME), "DET.2.NOISE"), 0);
    cpl_propertylist_delete(h); h = NULL;
    h = kmclipm_propertylist_load(file_path, 5);
    cpl_test_eq(strcmp(cpl_propertylist_get_string(h, EXTNAME), "DET.3.DATA"), 0);
    cpl_propertylist_delete(h); h = NULL;
    h = kmclipm_propertylist_load(file_path, 6);
    cpl_test_eq(strcmp(cpl_propertylist_get_string(h, EXTNAME), "DET.3.NOISE"), 0);
    cpl_propertylist_delete(h); h = NULL;

    //
    // test with 2 FLAT/3 DARK frames
    //
    test_flat23(op1_file_on, op2_file_on,
               op1_file_off, op2_file_off, op3_file_off, 0, 0);

    strcpy(tmp, MASTER_FLAT);
    sprintf(file_path, "%s_KKK.fits", kmo_strlower(tmp));

    h = kmclipm_propertylist_load(file_path, 1);
    cpl_test_eq(strcmp(cpl_propertylist_get_string(h, EXTNAME), "DET.1.DATA"), 0);
    cpl_propertylist_delete(h); h = NULL;
    h = kmclipm_propertylist_load(file_path, 2);
    cpl_test_eq(strcmp(cpl_propertylist_get_string(h, EXTNAME), "DET.1.NOISE"), 0);
    cpl_propertylist_delete(h); h = NULL;
    h = kmclipm_propertylist_load(file_path, 3);
    cpl_test_eq(strcmp(cpl_propertylist_get_string(h, EXTNAME), "DET.2.DATA"), 0);
    cpl_propertylist_delete(h); h = NULL;
    h = kmclipm_propertylist_load(file_path, 4);
    cpl_test_eq(strcmp(cpl_propertylist_get_string(h, EXTNAME), "DET.2.NOISE"), 0);
    cpl_propertylist_delete(h); h = NULL;
    h = kmclipm_propertylist_load(file_path, 5);
    cpl_test_eq(strcmp(cpl_propertylist_get_string(h, EXTNAME), "DET.3.DATA"), 0);
    cpl_propertylist_delete(h); h = NULL;
    h = kmclipm_propertylist_load(file_path, 6);
    cpl_test_eq(strcmp(cpl_propertylist_get_string(h, EXTNAME), "DET.3.NOISE"), 0);
    cpl_propertylist_delete(h); h = NULL;

    //
    // test with 3 FLAT/2 DARK frames
    //
    test_flat32(op1_file_on, op2_file_on, op3_file_on,
               op1_file_off, op2_file_off, 0, 0);

    strcpy(tmp, MASTER_FLAT);
    sprintf(file_path, "%s_KKK.fits", kmo_strlower(tmp));

    h = kmclipm_propertylist_load(file_path, 1);
    cpl_test_eq(strcmp(cpl_propertylist_get_string(h, EXTNAME), "DET.1.DATA"), 0);
    cpl_propertylist_delete(h); h = NULL;
    h = kmclipm_propertylist_load(file_path, 2);
    cpl_test_eq(strcmp(cpl_propertylist_get_string(h, EXTNAME), "DET.1.NOISE"), 0);
    cpl_propertylist_delete(h); h = NULL;
    h = kmclipm_propertylist_load(file_path, 3);
    cpl_test_eq(strcmp(cpl_propertylist_get_string(h, EXTNAME), "DET.2.DATA"), 0);
    cpl_propertylist_delete(h); h = NULL;
    h = kmclipm_propertylist_load(file_path, 4);
    cpl_test_eq(strcmp(cpl_propertylist_get_string(h, EXTNAME), "DET.2.NOISE"), 0);
    cpl_propertylist_delete(h); h = NULL;
    h = kmclipm_propertylist_load(file_path, 5);
    cpl_test_eq(strcmp(cpl_propertylist_get_string(h, EXTNAME), "DET.3.DATA"), 0);
    cpl_propertylist_delete(h); h = NULL;
    h = kmclipm_propertylist_load(file_path, 6);
    cpl_test_eq(strcmp(cpl_propertylist_get_string(h, EXTNAME), "DET.3.NOISE"), 0);
    cpl_propertylist_delete(h); h = NULL;

    //
    // test with 3 frames
    //
    test_flat(op1_file_on, op2_file_on, op3_file_on,
              op1_file_off, op2_file_off, op3_file_off, 0, 0);

    // check BADPIXEL_FLAT
    strcpy(tmp, BADPIXEL_FLAT);
    sprintf(file_path, "%s_KKK", kmo_strlower(tmp));
    cpl_test_abs(kmo_test_esorex_data(file_path, 1), 2.09073, tol);
    strcat(file_path, ".fits");
    int i = 0;
    for (i = 1; i <= KMOS_NR_DETECTORS; i++) {
        check_headers(file_path, i, tol);
    }

    // check XCAL
    strcpy(tmp, XCAL);
    sprintf(file_path, "%s_KKK", kmo_strlower(tmp));
    cpl_test_abs(kmo_test_esorex_data(file_path, 1), -0.900974, tol*60);
    strcat(file_path, ".fits");
    for (i = 1; i <= KMOS_NR_DETECTORS; i++) {
        check_headers(file_path, i, tol);
    }

    // check YCAL
    strcpy(tmp, YCAL);
    sprintf(file_path, "%s_KKK", kmo_strlower(tmp));
    cpl_test_abs(kmo_test_esorex_data(file_path, 1), 0.875022, tol*6);
    strcat(file_path, ".fits");
    for (i = 1; i <= KMOS_NR_DETECTORS; i++) {
        check_headers(file_path, i, tol);
    }

    // check FLAT_EDGE
    strcpy(tmp, FLAT_EDGE);
    sprintf(file_path, "%s_KKK", kmo_strlower(tmp));
    cpl_test_abs(kmo_test_esorex_data(file_path, 1), 24554.7, 50*tol);
    strcat(file_path, ".fits");
    for (i = 1; i <= KMOS_NR_DETECTORS*KMOS_IFUS_PER_DETECTOR; i++) {
        check_headers(file_path, i, tol);
    }

    // check MASTER_FLAT
    strcpy(tmp, MASTER_FLAT);
    sprintf(file_path, "%s_KKK.fits", kmo_strlower(tmp));

    h = kmclipm_propertylist_load(file_path, 1);
    cpl_test_eq(strcmp(cpl_propertylist_get_string(h, EXTNAME), "DET.1.DATA"), 0);
    cpl_test_abs(5579.42, cpl_propertylist_get_double(h, QC_FLAT_SN), tol*1000);
    cpl_test_eq(3, cpl_propertylist_get_int(h, QC_FLAT_SAT));
    cpl_test_abs(26.6846, cpl_propertylist_get_double(h, QC_FLAT_EFF), tol*10);
    cpl_test_abs(4.93827, cpl_propertylist_get_double(h, QC_GAP_MEAN), tol);
    cpl_test_abs(0.0698977, cpl_propertylist_get_double(h, QC_GAP_SDV), tol/8);
    cpl_test_abs(0.187894, cpl_propertylist_get_double(h, QC_GAP_MAXDEV), tol);
    cpl_test_abs(12.865489, cpl_propertylist_get_double(h, QC_SLIT_MEAN), tol);
    cpl_test_abs(0.0590769, cpl_propertylist_get_double(h, QC_SLIT_SDV), tol/4);
    cpl_test_abs(0.190067, cpl_propertylist_get_double(h, QC_SLIT_MAXDEV), tol*2.1);
    cpl_propertylist_delete(h); h = NULL;
    h = kmclipm_propertylist_load(file_path, 2);
    cpl_test_eq(strcmp(cpl_propertylist_get_string(h, EXTNAME), "DET.1.NOISE"), 0);
    cpl_propertylist_delete(h); h = NULL;
    h = kmclipm_propertylist_load(file_path, 3);
    cpl_test_eq(strcmp(cpl_propertylist_get_string(h, EXTNAME), "DET.2.DATA"), 0);
    cpl_propertylist_delete(h); h = NULL;
    h = kmclipm_propertylist_load(file_path, 4);
    cpl_test_eq(strcmp(cpl_propertylist_get_string(h, EXTNAME), "DET.2.NOISE"), 0);
    cpl_propertylist_delete(h); h = NULL;
    h = kmclipm_propertylist_load(file_path, 5);
    cpl_test_eq(strcmp(cpl_propertylist_get_string(h, EXTNAME), "DET.3.DATA"), 0);
    cpl_propertylist_delete(h); h = NULL;
    h = kmclipm_propertylist_load(file_path, 6);
    cpl_test_eq(strcmp(cpl_propertylist_get_string(h, EXTNAME), "DET.3.NOISE"), 0);
    cpl_propertylist_delete(h); h = NULL;

    strcpy(tmp, MASTER_FLAT);
    sprintf(file_path, "%s_KKK", kmo_strlower(tmp));
    cpl_test_abs(kmo_test_esorex_data(file_path, 1), 3.00055, tol);
    strcat(file_path, ".fits");
    for (i = 1; i <= 2*KMOS_NR_DETECTORS; i++) {
        check_headers(file_path, i, tol);
    }

    return cpl_test_end(0);
}

/** @} */
