/*
   FALCON - The Falcon Programming Language.
   FILE: sequence_ext.cpp

   Implementation of the RTL Sequence abstract Falcon class.
   -------------------------------------------------------------------
   Author: Giancarlo Niccolai
   Begin: Sat, 08 Aug 2009 11:07:57 +0200

   -------------------------------------------------------------------
   (C) Copyright 2009: the FALCON developers (see list in AUTHORS file)

   See LICENSE file for licensing details.
*/

/*#
   @beginmodule core
*/

/** \file
   Implementation of the RTL Sequence Falcon class.
*/


#include <falcon/setup.h>
#include <falcon/sequence.h>
#include <falcon/iterator.h>
#include <falcon/vm.h>
#include <falcon/eng_messages.h>
#include "core_module.h"

namespace Falcon {
namespace core {

/*#
   @class Sequence
   @brief Base abstract class for VM assisted sequences.

   This class is meant to provide common methods to VM-assisted (that is,
   language level) sequence classes. You can't derive directly from
   this class unless you're writing a native module, but you can
   derive from script-visible classes children of this one.
 */

/*#
   @method comp Sequence
   @brief Appends elements to this sequence through a filter.
   @param source A sequence, a range or a callable generating items.
   @optparam filter A filtering function receiving one item at a time.
   @return This sequence.

   This method adds one item at a time to this Sequence.

   If @b source is a range,
   (must not be open), all the values generated by the range will be appended,
   considering range direction and step.

   If @b source is a sequence (array,
   dictionary, or any other object providing the sequence interface), all the values
   in the item will be appended to this Sequence.

   If @b source is a callable item, it is called repeatedly to generate the sequence.
   All the items it returns will be appended, until
   it declares being terminated by returning an oob(0). Continuation objects are
   also supported.

   If a @b filter callable is provided, all the items that should be appended
   are first passed to to it; the item returned by the callable will be used
   instead of the item provided in @b source. The @b filter may return an
   out of band 1 to skip the item from @b source, or an out of band 0 to
   stop the processing altogether. The  @b filter callable receives also the
   forming sequence as the second parameter so that it account for it or manage
   it dynamically during the filter step.

   For example, the following comprehension creates a dictionary associating
   a letter of the alphabet to each element in the source sequence, discarding
   elements with spaces and terminating when a "<end>" mark is found. The @b filter
   function uses the second parameter to determine how many items have been added,
   and return a different element each time.

   @code
      dict = [=>].comp(
         // the source
         .[ 'bananas' 'skip me' 'apples' 'oranges' '<end>' 'melons' ],
         // the filter
         { element, dict =>
           if " " in element: return oob(1)
           if "<end>" == element: return oob(0)
           return [ "A" / len(dict), element ]   // (1)
         }
      )

      // (1): "A" / number == chr( ord("A") + number )
   @endcode

   This method actually adds each item in the comprehension to the sequence or
   sequence-compatible item in self. This means that comprehension needs not to
   be performed on a new, empty sequence; it may be also used to integrate more
   data in already existing sequences.

   @see Array.comp
   @see Dictionary.comp
   @see Object.comp
*/
FALCON_FUNC  Sequence_comp ( ::Falcon::VMachine *vm )
{
   if ( vm->param(0) == 0 )
   {
      throw new ParamError( ErrorParam( e_inv_params, __LINE__ )
         .extra( "R|A|C|Sequence, [C]" ) );
   }

   // Save the parameters as the stack may change greatly.
   Sequence* seq = vm->self().asObject()->getSequence();
   fassert( seq != 0 );

   Item i_gen = *vm->param(0);
   Item i_check = vm->param(1) == 0 ? Item() : *vm->param(1);

   seq->comprehension_start( vm, vm->self(), i_check );
   vm->pushParam( i_gen );
}


/*#
   @method mcomp Sequence
   @brief Appends elements to this sequence from multiple sources.
   @param ... One or more sequences, ranges or callables generating items.
   @return This sequence.

   This method works as @a Sequence.comp but it's possible to specify more
   items and sequences. Each element in the result set is an array of
   items in which each element extracted from a source or returned by
   a generator is combined with all the others.

   For example, the following operation:
   @code
      [].mcomp( [1,2], [3,4] )
   @endcode

   results in:

   @code
      [ [1,3], [1,4], [2,3], [2,4] ]
   @endcode

   Generators are called repeatedly until they exhaust all the items they can
   generate, in the same order as they are declared in the @b mcomp call.

   For example:
   @code
      [].mcomp( alphagen, betagen )
   @endcode

   will first call alphagen until it returns an oob(0), and then call betagen
   repeatedly.

   @note Calls in this context are not atomic. Suspension, sleep, I/O, and
   continuations are allowed and supported.

   After all the generators are called, the collected data is mixed with static
   data coming from other sources. For example:

   @code
      function make_gen( count, name )
         i = 0
         return {=>
            if i == count: return oob(0)
            > name, ": ", i
            return i++
            }
      end

      > [].mcomp( ["a","b"], make_gen(2, "first"), make_gen(2, "second") ).describe()
   @endcode

   will generate the following output:

   @code
      first: 0
      first: 1
      second: 0
      second: 1
      [ [ "a", 0, 0], [ "a", 0, 1], [ "a", 1, 0], [ "a", 1, 1],
         [ "b", 0, 0], [ "b", 0, 1], [ "b", 1, 0], [ "b", 1, 1]]
   @endcode

   @note The @a Sequence.mfcomp provides a more flexible approach.

   @see Array.mcomp
   @see Dictionary.mcomp
   @see Object.mcomp
*/

FALCON_FUNC  Sequence_mcomp ( ::Falcon::VMachine *vm )
{
   // Save the parameters as the stack may change greatly.
   Sequence* seq = vm->self().asObject()->getSequence();
   fassert( seq != 0 );

   StackFrame* current = vm->currentFrame();
   seq->comprehension_start( vm, vm->self(), Item() );

   for( uint32 i = 0; i < current->m_param_count; ++i )
   {
      vm->pushParam( current->m_params[i] );
   }
}


/*#
   @method mfcomp Sequence
   @brief Appends elements to this sequence from multiple sources through a filter.
   @param filter A filtering function receiving one item at a time.
   @param ... One or more sequences, ranges or callables generating items.
   @return This sequence.

   This function works exactly as @a Sequence.mcomp, with the difference that
   the elements generated are passed to a filter function for final delivery
   to the target sequence.

   @note The @b filter parameter is optional; if @b nil is passed to it, this
   method works exactly as @a Sequence.mcomp.

   For example, the math set operation
   @code
      { x*y for x in 1,2,3 and y in 4,5,6 }
   @endcode

   can be written using mfcomp like the following:

   @code
      [].mfcomp( {x, y => x*y}, .[1 2 3], .[4 5 6] )
   @endcode

   which results in

   @code
        [ 4, 5, 6, 8, 10, 12, 12, 15, 18]
   @endcode

   The as for @a Sequence.comp, filter receives an extra parameter which is the
   sequence itself. For example, the following code:

   @code
      > [].mfcomp( {x, y, seq =>
                     printl( "Seq is now long: " + seq.len() )
                     return [seq.len(), x*y]
                     },
                  .[1 2 3], .[4 5 6]
                  ).describe()
   @endcode

   generates this output:

   @code
      Seq is now long: 0
      Seq is now long: 1
      Seq is now long: 2
      Seq is now long: 3
      Seq is now long: 4
      Seq is now long: 5
      Seq is now long: 6
      Seq is now long: 7
      Seq is now long: 8
      [ [ 0, 4], [ 1, 5], [ 2, 6], [ 3, 8], [ 4, 10], [ 5, 12], [ 6, 12], [ 7, 15], [ 8, 18]]
   @endcode

   Notice that it is possible to modify the sequence inside the filter, in case it's needed.

   The filter may return an oob(1) to skip the value, and an oob(0) to terminate the
   operation. For example, the following code s

   @note The call Sequence.mfcomp( filter, seq ) is equivalent to
         Sequence.comp( seq, filter ).

   @see Array.mfcomp
   @see Dictionary.mfcomp
   @see Object.mfcomp
*/
FALCON_FUNC  Sequence_mfcomp ( ::Falcon::VMachine *vm )
{
   if ( vm->param(0) == 0 )
   {
      throw new ParamError( ErrorParam( e_inv_params, __LINE__ )
         .extra( "C, ..." ) );
   }

   // Save the parameters as the stack may change greatly.
   Sequence* seq = vm->self().asObject()->getSequence();
   fassert( seq != 0 );

   Item i_check = *vm->param(0);

   StackFrame* current = vm->currentFrame();
   seq->comprehension_start( vm, vm->self(), i_check );

   for( uint32 i = 1; i < current->m_param_count; ++i )
   {
      vm->pushParam( current->m_params[i] );
   }
}


/*#
   @method front Sequence
   @brief Returns the first item in the Sequence.
   @raise AccessError if the Sequence is empty.
   @return The first item in the Sequence.

   This method overloads the BOM method @b front. If the Sequence
   is not empty, it returns the first element.
*/
FALCON_FUNC  Sequence_front ( ::Falcon::VMachine *vm )
{
   Sequence* seq = vm->self().asObject()->getSequence();
   fassert( seq != 0 );

   if( seq->empty() )
   {
      throw new AccessError( ErrorParam( e_arracc, __LINE__ ).
         origin( e_orig_runtime ) );
      return;
   }

   vm->retval( seq->front() );
}

/*#
   @method back Sequence
   @brief Returns the last item in the Sequence.
   @raise AccessError if the Sequence is empty.
   @return The last item in the Sequence.

   This method overloads the BOM method @b back. If the Sequence
   is not empty, it returns the last element.
*/
FALCON_FUNC  Sequence_back ( ::Falcon::VMachine *vm )
{
   Sequence* seq = vm->self().asObject()->getSequence();
   fassert( seq != 0 );

   if( seq->empty() )
   {
      throw new AccessError( ErrorParam( e_arracc, __LINE__ ).
         origin( e_orig_runtime ) );
      return;
   }

   vm->retval( seq->back() );
}

/*#
   @method first Sequence
   @brief Returns an iterator to the first element of the Sequence.
   @return An iterator.

   Returns an iterator to the first element of the Sequence. If the
   Sequence is empty, an invalid iterator will be returned, but an
   insertion on that iterator will succeed and append an item to the Sequence.
*/
FALCON_FUNC  Sequence_first ( ::Falcon::VMachine *vm )
{
   Sequence* seq = vm->self().asObject()->getSequence();
   fassert( seq != 0 );

   Item *i_iclass = vm->findWKI( "Iterator" );
   fassert( i_iclass != 0 );

   CoreObject *iobj = i_iclass->asClass()->createInstance();
   // we need it separated to activate the FalconData bit
   iobj->setUserData( new Iterator( seq ) );
   vm->retval( iobj );
}

/*#
   @method last Sequence
   @brief Returns an iterator to the last element of the Sequence.
   @return An iterator.

   Returns an iterator to the last element of the Sequence. If the
   Sequence is empty, an invalid iterator will be returned, but an
   insertion on that iterator will succeed and append an item to the Sequence.
*/
FALCON_FUNC  Sequence_last ( ::Falcon::VMachine *vm )
{
   Sequence* seq = vm->self().asObject()->getSequence();
   fassert( seq != 0 );

   Item *i_iclass = vm->findWKI( "Iterator" );
   fassert( i_iclass != 0 );

   CoreObject *iobj = i_iclass->asClass()->createInstance();
   // we need it separated to activate the FalconData bit
   iobj->setUserData( new Iterator( seq, true ) );
   vm->retval( iobj );
}


/*#
   @method empty Sequence
   @brief Checks if the Sequence is empty or not.
   @return True if the Sequence is empty, false if contains some elements.
*/
FALCON_FUNC  Sequence_empty( ::Falcon::VMachine *vm )
{
   Sequence* seq = vm->self().asObject()->getSequence();
   fassert( seq != 0 );
   vm->regA().setBoolean( seq->empty() );
}

/*#
   @method clear Sequence
   @brief Removes all the items from the Sequence.
*/
FALCON_FUNC  Sequence_clear( ::Falcon::VMachine *vm )
{
   Sequence* seq = vm->self().asObject()->getSequence();
   fassert( seq != 0 );
   seq->clear();
}

/*#
   @method prepend Sequence
   @brief Adds an item in front of the sequence
   @param item The item to be added.

   If the sequence is sorted, the position at which
   the @b item is inserted is determined by the internal
   ordering; otherwise the @b item is prepended in front
   of the sequence.
*/
FALCON_FUNC  Sequence_prepend ( ::Falcon::VMachine *vm )
{
   Item *data = vm->param( 0 );

   if( data == 0 )
   {
      throw new ParamError( ErrorParam( e_inv_params, __LINE__ ).
         origin( e_orig_runtime ).
         extra("X") );
      return;
   }

   Sequence* seq = vm->self().asObject()->getSequence();
   fassert( seq != 0 );
   seq->prepend( *data );
}

/*#
   @method append Sequence
   @brief Adds an item at the end of the sequence.
   @param item The item to be added.

   If the sequence is sorted, the position at which
   the @b item is inserted is determined by the internal
   ordering; otherwise the @b item is appended at the end
   of the sequence.
*/
FALCON_FUNC  Sequence_append ( ::Falcon::VMachine *vm )
{
   Item *data = vm->param( 0 );

   if( data == 0 )
   {
      throw new ParamError( ErrorParam( e_inv_params, __LINE__ ).
         origin( e_orig_runtime ).
         extra("X") );
      return;
   }

   Sequence* seq = vm->self().asObject()->getSequence();
   fassert( seq != 0 );
   seq->append( *data );
}


}
}

/* end of sequence_ext.cpp */
