/*
    Copyright (C) 2014 Fredrik Johansson

    This file is part of FLINT.

    FLINT is free software: you can redistribute it and/or modify it under
    the terms of the GNU Lesser General Public License (LGPL) as published
    by the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.  See <https://www.gnu.org/licenses/>.
*/

#include "mag.h"

#if !FLINT64
# include "longlong.h"
#endif

void
mag_div(mag_t z, const mag_t x, const mag_t y)
{
    if (mag_is_special(x) || mag_is_special(y))
    {
        if (mag_is_zero(y) || mag_is_inf(x))
            mag_inf(z);
        else
            mag_zero(z);
    }
    else
    {
        ulong q;
        slong fix;

#if FLINT_BITS == 64
        q = (MAG_MAN(x) << MAG_BITS) / MAG_MAN(y) + 1;
#else
        ulong hi, lo, r;
        lo = MAG_MAN(x) << MAG_BITS;
        hi = MAG_MAN(x) >> (FLINT_BITS - MAG_BITS);
        udiv_qrnnd(q, r, hi, lo, MAG_MAN(y));
        q += 1;
#endif
        fix = q >> MAG_BITS;
        q = (q >> fix) + fix;

        /* could overflow to next power of two */
        if (q >> MAG_BITS)
        {
            q >>= 1;
            fix += 1;
        }

        MAG_MAN(z) = q;
        _fmpz_sub2_fast(MAG_EXPREF(z), MAG_EXPREF(x), MAG_EXPREF(y), fix);
    }
}
