/*
 * This program source code file is part of KiCad, a free EDA CAD application.
 *
 * Copyright (C) 2004 Jean-Pierre Charras, jaen-pierre.charras@gipsa-lab.inpg.com
 * Copyright (C) 2004-2011 KiCad Developers, see change_log.txt for contributors.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, you may find one here:
 * http://www.gnu.org/licenses/old-licenses/gpl-2.0.html
 * or you may search the http://www.gnu.org website for the version 2 license,
 * or you may write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 */

/**
 * @file sch_polyline.h
 *
 */

#ifndef _SCH_POLYLINE_H_
#define _SCH_POLYLINE_H_


#include <sch_item_struct.h>


class SCH_POLYLINE : public SCH_ITEM
{
    int m_width;                        // Thickness
    std::vector<wxPoint> m_points;      // list of points (>= 2)

public:
    SCH_POLYLINE( int layer = LAYER_NOTES );

    // Do not create a copy constructor.  The one generated by the compiler is adequate.

    ~SCH_POLYLINE();

    wxString GetClass() const
    {
        return wxT( "SCH_POLYLINE" );
    }

    void Draw( EDA_DRAW_PANEL* aPanel, wxDC* aDC, const wxPoint& aOffset,
               GR_DRAWMODE aDrawMode, EDA_COLOR_T aColor = UNSPECIFIED_COLOR );

    bool Save( FILE* aFile ) const;

    bool Load( LINE_READER& aLine, wxString& aErrorMsg );

    /**
     * Function AddPoint
     * add a corner to m_points
     */
    void AddPoint( const wxPoint& point )
    {
        m_points.push_back( point );
    }


    /**
     * Function SetPoint
     * sets the point at \a aIndex in the list to \a aPoint.
     *
     * @param aIndex The index in the point list.
     * @param aPoint The new point value.
     */
    void SetPoint( int aIndex, const wxPoint& aPoint )
    {
        // (unsigned) excludes aIndex<0 also
        wxCHECK_RET( (unsigned)aIndex < m_points.size(),
                     wxT( "Invalid SCH_POLYLINE point list index." ) );

        m_points[ aIndex ] = aPoint;
    }


    /**
     * Function GetCornerCount
     * @return the number of corners
     */
    unsigned GetCornerCount() const { return m_points.size(); }

    int GetPenSize() const;

    void Move( const wxPoint& aMoveVector )
    {
        for( unsigned ii = 0; ii < GetCornerCount(); ii++ )
            m_points[ii] += aMoveVector;
    }

    void MirrorY( int aYaxis_position );

    void MirrorX( int aXaxis_position );

    void Rotate( wxPoint aPosition );

    wxString GetSelectMenuText() const;

    BITMAP_DEF GetMenuImage() const;

    /**
     * Function operator[]
     * is used for read only access and returns the point at \a aIndex.
     * @param aIndex The index of the list of points to return.
     * @return A wxPoint object containing the point at \a aIndex.
     */
    wxPoint operator[]( int aIndex ) const
    {
        // (unsigned) excludes aIndex<0 also
        wxCHECK_MSG( (unsigned)aIndex < m_points.size(), wxDefaultPosition,
                     wxT( "Invalid SCH_POLYLINE point list index." ) );

        return m_points[ aIndex ];
    }

    wxPoint GetPosition() const { return m_points[0]; }

    void SetPosition( const wxPoint& aPosition );

    bool HitTest( const wxPoint& aPosition, int aAccuracy ) const;

    bool HitTest( const EDA_RECT& aRect, bool aContained = false,
                          int aAccuracy = 0 ) const;

    EDA_ITEM* Clone() const;

#if defined(DEBUG)
    void Show( int nestLevel, std::ostream& os ) const { ShowDummy( os ); } // override
#endif
};


#endif    // _SCH_POLYLINE_H_
