/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.netbeans.modules.bugtracking.tasks;

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.util.ArrayList;
import java.util.List;
import javax.swing.DefaultComboBoxModel;
import org.openide.util.NbBundle;

/**
 *
 * @author jpeska
 */
public class SortAttributePanel extends javax.swing.JPanel {

    private final List<TaskAttribute> attributes;
    private TaskAttribute selected;
    private final int index;
    private ArrayList<TaskAttribute> availableAttributes;
    private final List<ActionListener> changeListeners = new ArrayList<ActionListener>();
    private final ActionListener comboActionListener = new ComboActionListener();
    private final ItemListener ascendingListener = new AscendingListener();

    /**
     * Creates new form SortAttributePanel
     */
    public SortAttributePanel(List<TaskAttribute> attributes, TaskAttribute selected, int index) {
        this.attributes = attributes;
        this.selected = selected;
        this.index = index;
        initComponents();
        initCombo();
    }

    public void addSortingChangeListener(ActionListener listener) {
        changeListeners.add(listener);
    }

    public void removeSortingChangeListener(ActionListener listener) {
        changeListeners.remove(listener);
    }

    public TaskAttribute getSelectedAttribute() {
        return selected;
    }

    public int getIndex() {
        return index;
    }

    public void updateModel(TaskAttribute selected) {
        this.selected = selected;
        initCombo();
    }

    /**
     * This method is called from within the constructor to initialize the form. WARNING: Do NOT modify this code. The content of this method is always regenerated by the Form Editor.
     */
    @SuppressWarnings("unchecked")
    // <editor-fold defaultstate="collapsed" desc="Generated Code">//GEN-BEGIN:initComponents
    private void initComponents() {
        java.awt.GridBagConstraints gridBagConstraints;

        buttonGroup1 = new javax.swing.ButtonGroup();
        comboAttributes = new javax.swing.JComboBox();
        rbAscending = new javax.swing.JRadioButton();
        rbDescending = new javax.swing.JRadioButton();

        setLayout(new java.awt.GridBagLayout());

        comboAttributes.setModel(new javax.swing.DefaultComboBoxModel(new String[] { "Item 1", "Item 2", "Item 3", "Item 4" }));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.weightx = 0.3;
        gridBagConstraints.insets = new java.awt.Insets(1, 0, 0, 0);
        add(comboAttributes, gridBagConstraints);

        buttonGroup1.add(rbAscending);
        org.openide.awt.Mnemonics.setLocalizedText(rbAscending, NbBundle.getMessage(SortAttributePanel.class, "SortAttributePanel.rbAscending.text")); // NOI18N
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.insets = new java.awt.Insets(0, 18, 0, 0);
        add(rbAscending, gridBagConstraints);

        buttonGroup1.add(rbDescending);
        org.openide.awt.Mnemonics.setLocalizedText(rbDescending, NbBundle.getMessage(SortAttributePanel.class, "SortAttributePanel.rbDescending.text")); // NOI18N
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 2;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.insets = new java.awt.Insets(0, 18, 0, 0);
        add(rbDescending, gridBagConstraints);
    }// </editor-fold>//GEN-END:initComponents


    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.ButtonGroup buttonGroup1;
    private javax.swing.JComboBox comboAttributes;
    private javax.swing.JRadioButton rbAscending;
    private javax.swing.JRadioButton rbDescending;
    // End of variables declaration//GEN-END:variables

    private void initCombo() {
        comboAttributes.removeActionListener(comboActionListener);
        rbAscending.removeItemListener(ascendingListener);
        availableAttributes = new ArrayList<TaskAttribute>();

        for (TaskAttribute attribute : attributes) {
            if (attribute.getRank() >= index) {
                availableAttributes.add(attribute);
            }
        }
        String[] values = new String[attributes.size() + 1];
        values[0] = "<" + NbBundle.getMessage(SortAttributePanel.class, "LBL_NoCategory") + ">"; //NOI18N
        for (int i = 0; i < availableAttributes.size(); i++) {
            values[i + 1] = availableAttributes.get(i).getDisplayName();
        }
        comboAttributes.setModel(new DefaultComboBoxModel(values));
        comboAttributes.setSelectedIndex(availableAttributes.indexOf(selected) + 1);
        if (selected != null) {
            rbAscending.setEnabled(true);
            rbDescending.setEnabled(true);
            if (selected.isAsceding()) {
                rbAscending.setSelected(true);
            } else {
                rbDescending.setSelected(true);
            }
        } else {
            rbAscending.setEnabled(false);
            rbDescending.setEnabled(false);
        }
        rbAscending.addItemListener(ascendingListener);
        comboAttributes.addActionListener(comboActionListener);
    }

    void setComponentsEnabled(boolean enabled) {
        comboAttributes.setEnabled(enabled);
        rbAscending.setEnabled(enabled);
        rbDescending.setEnabled(enabled);
    }

    private class ComboActionListener implements ActionListener {

        @Override
        public void actionPerformed(ActionEvent e) {
            TaskAttribute lastSelected = selected;
            int i = comboAttributes.getSelectedIndex() - 1;
            selected = i == -1 ? null : availableAttributes.get(i);
            if (lastSelected != null) {
                lastSelected.setRank(selected != null ? selected.getRank() : TaskAttribute.NO_RANK);
            }
            if (selected != null) {
                selected.setRank(index);
            }
            fireEvent(e);
        }

        private void fireEvent(ActionEvent e) {
            for (ActionListener listener : changeListeners) {
                listener.actionPerformed(e);
            }
        }
    }

    private class AscendingListener implements ItemListener {

        @Override
        public void itemStateChanged(ItemEvent e) {
            selected.setAsceding(rbAscending.isSelected());
        }

    }
}
