/* xm.c: s7/Ruby/Forth bindings for X/Xt/Xpm/Xm/Xext
 *   needs xen.h
 *   for tests and examples see snd-motif.scm, bess.scm|rb, and snd-test.scm
 */

#include "mus-config.h"
#include <stdlib.h>


#define HAVE_XSHAPEQUERYEXTENSION 1
/* this dates from X11 R6.4 -- set to 0 if you're running a 20 year old version of X
 */
#define HAVE_XP 0


#define XM_DATE "29-Oct-15"

/* HISTORY: 
 *
 *   29-Oct-15: removed ->string.
 *   --------
 *   6-Mar:     more macro name changes.
 *   21-Feb-14: _P changed to use _is_.
 *   --------
 *   7-Jul-13:  removed the non-standard extra widgets.
 *   --------
 *   20-Oct:    changed INT64_T to LONG_LONG.
 *   20-Mar-11: X/Xpm/Motif are assumed now.
 *   --------
 *   23-Dec:    removed XmPrint/libXp support.
 *   16-Dec:    removed Guile support.
 *   16-Nov:    XM_XTPOINTER resource type for 64-bit systems.
 *   7-Aug:     s7 extended type change.
 *   27-Jul:    changed OFF_T to INT64_T.
 *   24-Feb:    and then changed some back to ULONGs!
 *   16-Jan:    changed some ULONG's to C_POINTER's
 *   12-Jan-09: changed strdup to xen_strdup.
 *   --------
 *   11-Dec:    removed all the stuff on the XM_DISABLE_DEPRECATED switch.
 *   16-Oct:    removed Gauche support.
 *   1-Oct:     XtAppAddInput condition arg is a mess.
 *   10-Sep:    XtAppAddInput condition arg is an int.
 *   1-Sep-08:  S7 support.
 *   --------
 *   26-Aug:    removed WITH_GTK_AND_X11 switch.
 *   21-Apr:    Gauche additions.
 *   30-Jan:    removed XmMultiList.
 *   1-Jan-06:  XmNpopupEnabled resource type changed from boolean to int (enum) (Motif docs bug).
 *   --------
 *   16-Sep:    XmUNSPECIFIED_PIXEL and friends should be unsigned long (not int).
 *   17-Aug:    XtSetArg 3rd arg should be 0, not NULL (type mismatch if 64-bit).
 *   14-June:   various xen-related updates (Xen_DEFINE).
 *   13-June:   fold xm-ruby.c into xm.c.
 *   15-Apr:    XGetWindowProperty free bugfix.
 *   28-Mar:    fix some Ruby error strings (#f->false).
 *   31-Jan:    remove Motif 1 support, and Lesstif.
 *   4-Jan-05:  replace Xen_VECTOR_ELEMENTS usages.
 *   --------
 *   30-Dec:    plug various memory leaks.
 *   23-Nov:    resource type lookup indexing bugfix.
 *   22-Sep:    various minor cleanups.
 *   23-Aug:    more changes for new Guile.
 *   12-Aug:    some changes to accommodate new Guile names.
 *   22-June:   XmColorSelector resources.
 *   19-May:    plug several memory leaks.
 *   21-Apr:    XmMultiList, XmTabStack.
 *   19-Apr:    XmDataField.
 *   12-Apr:    XmDropDown, XmColumn.
 *   7-Apr:     XmButtonBox.
 *   22-Mar:    added feature 'Xp to indicate that the Xp stuff is included.
 *   8-Mar:     XtAppAddActionHook arity bugfix.
 *   12-Jan:    resources for XmFontSelector.
 *   8-Jan:     various changes for the SGI C compiler, thanks to Avi Bercovich.
 *   5-Jan-04:  added (Motif 2.2.3) XmCreateFontSelector, XmCreateColorSelector.
 *   --------
 *   1-Dec:     XShapeGetRectangles XRectangle array needed local allocation.
 *              removed (unusable) XtCallCallbackList.
 *   25-Nov:    more GC protection. Removed XInitThreads, XLockDisplay, XUnlockDisplay. Fixed XmTextBlock allocation bug.
 *   15-Oct:    XFontsOfFontSet indexing bugfix and several more struct field accessors from MS.
 *   14-Oct:    XShapeQueryExtension from Michael Scholz, plus other extensions/shape.h functions and constants.
 *              Also XSizeHints, XSet[Standard|WM]Properties and accessors for input and initial_state.
 *   3-Oct:     removed some macros that were intended only for testing, added XmTOP_LEFT etc.
 *   29-Sep:    changed: XSetWindowBorder, XSetWindowBackground, XCreatePixmapFromBitmapData, XSetWindowAttributes,
 *                 .backing_pixel, .border_pixel, .background_pixel, .base_pixel; these now use the "wrapped"
 *                 form of pixels, not the bare unsigned long.
 *   26-Sep:    .event -> #f if event is NULL.
 *   3-Sep:     XmRenderTableGetRenditions was incorrectly freeing the XmRenditions.
 *   20-Aug:    XtEventHandler *flag set to false if handler returns 'done.
 *   11-Aug:    int -> bool.
 *   17-July:   XpmAttributes .colorsymbols is a list.
 *   15-July:   type check cleanups.
 *   14-July:   .depths returns a list of Depth pointers; similar change for .visuals.
 *   11-July:   Several int->Dimension|Position|short resource type changes.
 *              removed several more undocumented resource names.
 *   23-June:   Motif 1 fixups.
 *   10-June:   added XmCvtByteStreamToXmString, XmCvtXmStringToByteStream, XmStringByteStreamLength.
 *              removed XFreeStringList (a no-op).
 *   20-May:    showValue resource is int (enumeration) in Motif 2.  resizeWidth|Height are booleans, not ints.
 *   9-May:     Ruby fixups (some functions accidentally omitted earlier).
 *   8-Apr:     XSetErrorHandler proc takes XErrorEvent, not XEvent 2nd arg (thanks Friedrich Delgado Friedrichs)
 *   7-Apr:     more changes for the WITH_GTK_AND_X11 switch.
 *   1-Apr:     XGetWindowProperty uses mem2string if not XA_STRING.
 *   31-Mar:    added WITH_GTK_AND_X11 switch for xg+local X funcs.
 *   4-Mar:     xm-ruby XM_DEFINE_ACCESSOR quoted SetName bugfix (Michael Scholz).
 *   1-Feb-03:  XChangeProperty data (arg7) can be list of ints as well as string.
 *   ----------
 *   19-Dec:    more Ruby fixups.
 *   6-Nov:     Ruby XmWMProtocols bugfixes thanks to Michael Scholz.
 *   17-Oct:    XtAppSetFallbackResources and fallbacks added to XtAppInitialize etc.
 *   15-Oct:    XtGetResourceList.
 *   11-Oct:    xm-ruby XM_DEFINE* cleaned up (thanks to Michael Scholz).
 *              removed all (ignored) XrmOptionDesc args (XtAppInitialize etc).
 *   8-Oct:     added [MUS_]WITH_EDITRES to include _XEditResCheckMessages.
 *   1-Oct:     some int args are now KeyCodes (Modifiermap stuff).
 *   23-Sep:    X ScreenSaver constants omitted earlier.
 *   18-Sep:    properties XFontStruct field now returns list of all properties.
 *              removed XmFontListCreate_r, XmFontListEntryCreate_r
 *   9-Sep:     added Motif WMProtocol convenience macros.
 *   30-Aug:    added add-resource for user-defined extensions to the resources.
 *   29-Aug:    added ten resources accidentally omitted earlier (thanks to Michael Scholz).
 *   2-Aug:     some Lesstif-related compile-time switches.
 *   26-Jul:    removed wrappers Widget Pixel GC and XtAppContext.
 *   24-Jul:    removed "|" prefix, use "." as default struct field prefix.
 *   19-Jul:    XM_FIELD_PREFIX for change from using vertical-bar ("|" is reserved in R5RS).
 *   17-Jun:    removed XtSetWMColormapWindows.
 *   29-Apr:    minor 64-bit fixups.
 *   29-Mar:    XmParseProc.
 *   20-Mar:    XpmGetErrorString omitted inadvertently earlier.
 *   4-Mar:     XWindowChanges and XSetWindowAttributes struct creators.
 *   1-Mar:     XmTabListFree, various ->* conversions(->strings etc).
 *   25-Feb:    XmTextBlock fields
 *   22-Feb:    #f = NULL and vice-versa throughout
 *   21-Feb:    added various callback struct makers, changed XtCallCallbacks to be compatible with them.
 *   18-Feb:    removed undocumented functions: XmCvtFromHorizontalPixels, XmCvtFromVerticalPixels, XmCvtToHorizontalPixels, XInitImage
 *                XmCvtToVerticalPixels, XmGetIconFileName, XmStringCreateFontList, XmStringCreateFontList_r, XmStringLtoRCreate
 *              added XM_DISABLE_DEPRECATED which affects:
 *                XtError, XtSetErrorHandler, XtSetWarningHandler, XtSetErrorMsgHandler, XtSetWarningMsgHandler, XtWarningMsg, XtAppWarning, XtErrorMsg
 *                XtSetSelectionTimeout, XtInitialize , XtAddActions, XtAddInput, XtAddTimeout, XtAddWorkProc, XtCreateApplicationShell
 *                XtNextEvent, XtPeekEvent, XtPending, XtProcessEvent, XtMainLoop, XtGetSelectionTimeout, XtWarning,
 *                XmNdefaultFontList, XmNshellUnitType, XmNfontList, XmFontList*, XmNchildType, XmGetMenuCursor, 
 *                XmNstringDirection, XmMainWindowSep1, XmMainWindowSep2, XmMainWindowSep3, XmMainWindowSetAreas, 
 *                XmNwhichButton, XmScrolledWindowSetAreas, XmSetFontUnit, XmSetFontUnits, XmNdefaultFontList, 
 *                XmSetMenuCursor, XmStringByteCompare, XmStringCreateLtoR, XmSTRING_COMPONENT_CHARSET, XmSTRING_COMPONENT_FONTLIST_ELEMENT_TAG,
 *                XmStringCreateSimple, XmStringGetLtoR, XmStringGetNextComponent, XmStringGetNextSegment, XmStringLength, XmStringNConcat
 *                XmStringNCopy, XmStringPeekNextComponent, XmStringSegmentCreate, XmTrackingLocate, XmRemoveFrom|AddTo|PostFromList
 *                XGet|SetStandardColormap
 *              added XmWidgetGetDisplayRect inadvertently omitted earlier.
 *   14-Feb:    XUniqueContext added, XExtentsOfFontSet and XGetErrorDatabaseText deleted.
 *              X save-set and X host-address stuff deleted.
 *              XVisualInfo fields added. 
 *              Cursor type added.
 *   1-Feb:     Motif 2.2 additions (tooltip).
 *   21-Jan:    Ruby fixups (Xen_COPY_ARG to protect lists)
 *   7-Jan-02:  XEvent fields settable. added XtCallCallbacks-raw.
 *   ----------
 *   12-Sep:    xm-version.
 *   13-Aug:    Xp bindings, X11 predefined Atoms.
 *   6-Aug:     XmTransfer functions inadvertently omitted earlier.
 *   3-Aug:     type checks in XtSetValues.
 *   23-Jul-01: use lists rather than vectors.
 */

#if HAVE_EXTENSION_LANGUAGE

  #include <Xm/XmAll.h>
  #include <X11/keysym.h>
  #include <X11/cursorfont.h>
  #include <stdio.h>

#if HAVE_XSHAPEQUERYEXTENSION
#include <X11/extensions/shape.h>
#endif

#if ((!__NetBSD__) && ((_MSC_VER) || (!defined(__STC__)) || (defined(__STDC_VERSION__) && (__STDC_VERSION__ < 199901L))))
  #define __func__ __FUNCTION__
#endif

/* compile-time flags are HAVE_RUBY|SCHEME WITH_EDITRES */

/* if you're using g++ and it complains about XmRemoveFromPostFromList, update Motif (you need 2.1.30) */

#if USE_SND
  /* USE_SND causes xm to use Snd's error handlers which are much smarter than xen's fallback versions */
  #include "snd.h"
#else
  #include "xen.h"
  /* in snd-0.h: */
  #if HAVE_RUBY
    #define S_set "set_"
    #define PROC_FALSE "false"
    #define PROC_TRUE "true"
  #endif
  #if HAVE_SCHEME
    #define S_set "set! "
    #define PROC_FALSE "#f"
    #define PROC_TRUE "#t"
  #endif
  #if HAVE_FORTH
    #define S_set "set-"
    #define PROC_FALSE "#f" 
    #define PROC_TRUE "#t"
 #endif
  #define NOT_A_GC_LOC -1
#endif

#include <X11/xpm.h>
#if WITH_EDITRES
  #include <X11/Xmu/Editres.h>
#endif

/* prefix for all names */
#if HAVE_SCHEME
  #define XM_PREFIX ""
  #define XM_POSTFIX ""
  #define XM_FIELD_PREFIX "."
#endif
#if HAVE_RUBY
/* for Ruby, XM PREFIX needs to be uppercase */
  #define XM_PREFIX "R"
  #define XM_POSTFIX ""
  #define XM_FIELD_PREFIX "R"
#endif
#if HAVE_FORTH
  #define XM_PREFIX "F"
  #define XM_POSTFIX ""
  #define XM_FIELD_PREFIX "F"
#endif

#define XM_field_assert_type(Assertion, Arg, Position, Caller, Correct_Type) \
  Xen_check_type(Assertion, Arg, Position, XM_FIELD_PREFIX Caller XM_POSTFIX, Correct_Type)

#if HAVE_RUBY
  #define XM_set_field_assert_type(Assertion, Arg, Position, Caller, Correct_Type) \
    Xen_check_type(Assertion, Arg, Position, XM_FIELD_PREFIX S_set Caller XM_POSTFIX, Correct_Type)
#endif
#if HAVE_SCHEME || HAVE_FORTH
  #define XM_set_field_assert_type(Assertion, Arg, Position, Caller, Correct_Type) \
    Xen_check_type(Assertion, Arg, Position, S_set XM_FIELD_PREFIX Caller XM_POSTFIX, Correct_Type)
#endif

#define XtIsSubClass XtIsSubclass
/* kludge around a bug in some Motif versions */

/* layout of file:
 *    preliminaries
 *    type manglers
 *    arglist handlers
 *    Motif procedures
 *    X procedures
 *    Xt procedures
 *    Xp procedures
 *    Xpm procedures
 *    struct handlers
 *    string constants
 *    integer constants
 *    pointer constants
 *    atom constants
 */

/* --------------------------------------------------------------------------------
 * a sample program

(let* ((shell-app (XtVaOpenApplication 
		    "Test" 0 () applicationShellWidgetClass
		    (list XmNallowShellResize #t)))
       (app (cadr shell-app))
       (shell (car shell-app))
       (black (BlackPixelOfScreen 
		(DefaultScreenOfDisplay 
		  (XtDisplay shell)))))
  (if (not (XtIsApplicationShell shell))
      (display "not appshell"?))
  (XtSetValues shell (list XmNtitle "Hi!"))
  (let* ((main-pane 
	  (XtVaCreateManagedWidget 
	    "main-pane" xmFormWidgetClass shell
	    (list XmNforeground       black
		  XmNtopAttachment    XmATTACH_FORM
		  XmNbottomAttachment XmATTACH_FORM
		  XmNleftAttachment   XmATTACH_FORM
		  XmNrightAttachment  XmATTACH_FORM
		  XmNallowResize      #t)))
	 (button (XtCreateManagedWidget 
		   "push me" xmPushButtonWidgetClass main-pane () 0)))
    (XtAddCallback button XmNactivateCallback 
		    (lambda (widget context event-info)
		      (display widget)
		      (display (.reason event-info))
		      (display context))
		    123)
    (XtRealizeWidget shell)
    (XtAppMainLoop app)))
 */


/* --------------------------------------------------------------------------------
 * differences from C:
 *
 *    Arg list is lisp list of name/value pairs and the "len" arg associated with it is optional
 *    ref args are usually returned by proc, and not passed in unless init val is needed
 *    array args are passed as lists, returned as lists
 *    pointers to structs are '(type val) where val is opaque except via accessors 
 *      that is, all non-simple types are lists in xm where the first element is a symbol describing the type
 *      and the second is usually the C value stored directly as an unsigned long
 *    "Va" args are passed and returned as lists
 *    XtCallback procedure args are passed by value
 *    various "client data" args are optional
 *    XtCallbackLists are passed as lists of procedure/data pairs
 *    where explicit NULL is needed as arg, use #f (or () for list args)
 *
 * omitted:       
 *
 *    Xwc*, Xrm*, XIM*, XOM*, XIC*, XOC*, Xcms*, 
 *    "resource" handlers and "quark" stuff (XtAppSetTypeConverter etc)
 *    all XTextProperty, XClassHint, XSizeHints support, XSetIconSizes
 *    XConnection internal stuff, XVaCreateNestedList, XSetAuthorization
 *    XtQueryGeometry and XtMakeGeometryRequest, all text16 stuff
 *    XtGetKeySymTable, XtChangeManagedSet (undocumented), XtAppGet|SetExitFlag
 *    XtSetEventDispatcher (undoc), XtSignal stuff (undoc), XtGetErrorDatabaseText
 *    XtBlockHook stuff (undoc), XtRegisterExtensionSelector (undoc)
 *    XmResolvePartOffsets, XmResolveAllPartOffsets
 *    XpSet|GetLocaleHinter, XFreeStringList
 *    XtHooksOfDisplay, XtRegiserDrawable, XtUnregisterDrawable
 *
 * added:
 *
 *    XGCValues -> a blank XGCValues struct (for XCreateGC etc)
 *    XColor pixel red green blue flags pad
 *    XArc XRectangle XPoint XSegment XEvent XWindowChanges XSetWindowAttributes
 *    XTextItem XpmImage XpmColorSymbol
 *    XDrawLinesDirect same as XDrawLines but takes (opaque) ptr to XPoint array
 *    vector->XPoints vect packages point data in vector as (opaque) array of XPoints 
 *    freeXPoints to free (opaque) XPoint array created by vector->Xpoints
 *    <name>? -> #t if arg is of type <name>
 *    <name> -> empty struct of type <name>
 *
 * Structs are accessed by the field name and the lisp variable (which contains the struct type)
 *   (.pixel color) for example, or (.foreground gcvalue)
 */
 


/* -------------------------------- GC -------------------------------- */

static Xen_object_type_t xm_obj_tag;

#if HAVE_RUBY
static void *xm_obj_free(Xen obj)
{
  void *vobj;
  vobj = (void *)obj;
  free(vobj);
  return(NULL);
}
#endif

#if HAVE_FORTH
static void xm_obj_free(Xen obj)
{
  void *val;
  val = (void *)Xen_object_ref(obj);
  free(val);
}
#endif

#if HAVE_SCHEME
static void xm_obj_free(void *val)
{
  free(val);
}

static bool s7_equalp_xm(void *x1, void *x2)
{
  return(x1 == x2);
}
#endif

static Xen make_xm_obj(void *ptr)
{
  return(Xen_make_object(xm_obj_tag, ptr, 0, xm_obj_free));
}

static void define_xm_obj(void)
{
#if HAVE_SCHEME
  xm_obj_tag = s7_new_type_x(s7, "<XmObj>", NULL, xm_obj_free, s7_equalp_xm, NULL, NULL, NULL, NULL, NULL, NULL, NULL);
#else
  xm_obj_tag = Xen_make_object_type("XmObj", sizeof(void *));
#endif

#if HAVE_FORTH
  fth_set_object_free(xm_obj_tag, xm_obj_free);
#endif
}  


/* -------------------------------- type manglers -------------------------------- */

/* most non-simple vars are handled as a list: (type ptr) where ptr is the (un-interpreted) C value
 *   the _OBJ form is used where we need to eventually free the memory
 *   the _PTR form treats NULL as #f and vice-versa
 */

#define wrap_for_Xen(Name, Value) Xen_list_2(C_string_to_Xen_symbol(Name), Xen_wrap_C_pointer(Value))
#define wrap_for_Xen_obj(Name, Value) Xen_list_3(C_string_to_Xen_symbol(Name), Xen_wrap_C_pointer(Value), make_xm_obj(Value))
#define is_wrapped(Name, Value) (Xen_is_list(Value) && \
                            (Xen_list_length(Value) >= 2) && \
                            (Xen_is_symbol(Xen_car(Value))) && \
                            (strcmp(Name, Xen_symbol_to_C_string(Xen_car(Value))) == 0))

/* Xm_type is used for non-pointers (XID mainly) */
#define Xm_type(Name, XType) \
  static Xen C_to_Xen_ ## Name (XType val) {return(Xen_list_2(C_string_to_Xen_symbol(#Name), C_ulong_to_Xen_ulong(val)));} \
  static XType Xen_to_C_ ## Name (Xen val) {return((XType)Xen_ulong_to_C_ulong(Xen_cadr(val)));} \
  static bool Xen_is_ ## Name (Xen val) {return(is_wrapped(#Name, val));} \
  static Xen g_is_ ## Name (Xen val) {return(C_bool_to_Xen_boolean(is_wrapped(#Name, val)));}

#define Xm_type_no_p(Name, XType) \
  static Xen C_to_Xen_ ## Name (XType val) {return(Xen_list_2(C_string_to_Xen_symbol(#Name), C_ulong_to_Xen_ulong(val)));} \
  static XType Xen_to_C_ ## Name (Xen val) {return((XType)Xen_ulong_to_C_ulong(Xen_cadr(val)));} \
  static bool Xen_is_ ## Name (Xen val) {return(is_wrapped(#Name, val));}

#define Xm_type_int(Name, XType) \
  static Xen C_to_Xen_ ## Name (XType val) {return(Xen_list_2(C_string_to_Xen_symbol(#Name), C_int_to_Xen_integer(val)));} \
  static XType Xen_to_C_ ## Name (Xen val) {return((XType)Xen_integer_to_C_int(Xen_cadr(val)));} \
  static bool Xen_is_ ## Name (Xen val) {return(is_wrapped(#Name, val));} \
  static Xen g_is_ ## Name (Xen val) {return(C_bool_to_Xen_boolean(is_wrapped(#Name, val)));}

#define Xm_type_ptr(Name, XType) \
  static Xen C_to_Xen_ ## Name (XType val) {if (val) return(wrap_for_Xen(#Name, val)); return(Xen_false);} \
  static XType Xen_to_C_ ## Name (Xen val) {if (Xen_is_false(val)) return((XType)NULL); return((XType)Xen_unwrap_C_pointer(Xen_cadr(val)));} \
  static bool Xen_is_ ## Name (Xen val) {return(is_wrapped(#Name, val));} \
  static Xen g_is_ ## Name (Xen val) {return(C_bool_to_Xen_boolean(is_wrapped(#Name, val)));}

#define Xm_type_ptr_no_p(Name, XType) \
  static Xen C_to_Xen_ ## Name (XType val) {if (val) return(wrap_for_Xen(#Name, val)); return(Xen_false);} \
  static XType Xen_to_C_ ## Name (Xen val) {if (Xen_is_false(val)) return((XType)NULL); return((XType)Xen_unwrap_C_pointer(Xen_cadr(val)));} \
  static bool Xen_is_ ## Name (Xen val) {return(is_wrapped(#Name, val));}

#define Xm_type_ptr_no_p_NO_P(Name, XType) \
  static Xen C_to_Xen_ ## Name (XType val) {if (val) return(wrap_for_Xen(#Name, val)); return(Xen_false);} \
  static XType Xen_to_C_ ## Name (Xen val) {if (Xen_is_false(val)) return((XType)NULL); return((XType)Xen_unwrap_C_pointer(Xen_cadr(val)));}

#define Xm_type_ptr_no_c2x(Name, XType) \
  static XType Xen_to_C_ ## Name (Xen val) {if (Xen_is_false(val)) return((XType)NULL); return((XType)Xen_unwrap_C_pointer(Xen_cadr(val)));} \
  static bool Xen_is_ ## Name (Xen val) {return(is_wrapped(#Name, val));} \
  static Xen g_is_ ## Name (Xen val) {return(C_bool_to_Xen_boolean(is_wrapped(#Name, val)));}

#define Xm_type_ptr_no_c2x_no_p(Name, XType) \
  static XType Xen_to_C_ ## Name (Xen val) {if (Xen_is_false(val)) return((XType)NULL); return((XType)Xen_unwrap_C_pointer(Xen_cadr(val)));} \
  static bool Xen_is_ ## Name (Xen val) {return(is_wrapped(#Name, val));}

#define Xm_type_ptr_obj(Name, XType) \
  static Xen C_to_Xen_ ## Name (XType val) {if (val) return(wrap_for_Xen_obj(#Name, val)); return(Xen_false);} \
  static XType Xen_to_C_ ## Name (Xen val) {if (Xen_is_false(val)) return(NULL); return((XType)Xen_unwrap_C_pointer(Xen_cadr(val)));} \
  static bool Xen_is_ ## Name (Xen val) {return(is_wrapped(#Name, val));} \
  static Xen g_is_ ## Name (Xen val) {return(C_bool_to_Xen_boolean(is_wrapped(#Name, val)));}



#define Xen_to_C_Dimension(Arg)  (Dimension)(Xen_integer_to_C_int(Arg))
#define C_to_Xen_Dimension(Arg)  (C_int_to_Xen_integer((Dimension)Arg))
#define Xen_is_Dimension(Arg)     (Xen_is_integer(Arg))

#define Xen_to_C_Position(Arg)   (Position)(Xen_integer_to_C_int(Arg))
#define C_to_Xen_Position(Arg)   (C_int_to_Xen_integer((Position)Arg))
#define Xen_is_Position(Arg)      (Xen_is_integer(Arg))

#define Xen_to_C_Modifiers(Arg)  (Modifiers)(Xen_ulong_to_C_ulong(Arg))
#define C_to_Xen_Modifiers(Arg)  (C_ulong_to_Xen_ulong(Arg))
#define Xen_is_Modifiers(Arg)     (Xen_is_ulong(Arg))

Xm_type(Cursor, Cursor)
Xm_type_ptr(Screen, Screen *)
Xm_type_ptr_obj(XRectangle, XRectangle *)
Xm_type_ptr_obj(XArc, XArc *)
Xm_type_ptr_obj(XPoint, XPoint *)
Xm_type_ptr_obj(XSegment, XSegment *)
Xm_type_ptr_obj(XColor, XColor *)
Xm_type(Atom, Atom)
Xm_type(Colormap, Colormap)  /* XID */
Xm_type_ptr_no_c2x(Depth, Depth *)
Xm_type_ptr(Display, Display *)
Xm_type(Font, Font)
Xm_type_ptr(GC, GC)
Xm_type(KeySym, KeySym)
Xm_type(Pixel, Pixel)
Xm_type(Pixmap, Pixmap)
Xm_type(Region, Region)
Xm_type(Time, Time)
Xm_type_ptr(Visual, Visual *)
Xm_type(Window, Window) /* this is XID = long I think */
Xm_type_ptr(XCharStruct, XCharStruct *)
Xm_type_ptr(XFontProp, XFontProp *)
Xm_type(XFontSet, XFontSet)
Xm_type_ptr(XFontStruct, XFontStruct *)
Xm_type_ptr(XGCValues, XGCValues *)
Xm_type_ptr_no_c2x(XTextItem, XTextItem *)
Xm_type_ptr(XModifierKeymap, XModifierKeymap *) /* opaque in this implementation */
Xm_type_ptr(XImage, XImage *)
static XAnyEvent *Xen_to_C_XAnyEvent(Xen val) {return((XAnyEvent *)Xen_unwrap_C_pointer(Xen_cadr(val)));}
Xm_type_ptr_no_c2x(XButtonEvent, XButtonEvent *)
Xm_type_ptr_no_c2x(XCirculateEvent, XCirculateEvent *)
Xm_type_ptr_no_c2x(XCirculateRequestEvent, XCirculateRequestEvent *)
Xm_type_ptr_no_c2x(XClientMessageEvent, XClientMessageEvent *)
Xm_type_ptr_no_c2x(XColormapEvent, XColormapEvent *)
Xm_type_ptr_no_c2x(XConfigureEvent, XConfigureEvent *)
Xm_type_ptr_no_c2x(XConfigureRequestEvent, XConfigureRequestEvent *)
Xm_type_ptr_no_c2x(XCreateWindowEvent, XCreateWindowEvent *)
Xm_type_ptr_no_c2x(XCrossingEvent, XCrossingEvent *)
Xm_type_ptr_no_c2x(XDestroyWindowEvent, XDestroyWindowEvent *)
Xm_type_ptr(XErrorEvent, XErrorEvent *)
Xm_type_ptr_no_c2x(XExposeEvent, XExposeEvent *)
Xm_type_ptr_no_c2x(XFocusChangeEvent, XFocusChangeEvent *)
Xm_type_ptr_no_c2x(XGraphicsExposeEvent, XGraphicsExposeEvent *)
Xm_type_ptr_no_c2x(XGravityEvent, XGravityEvent *)
Xm_type_ptr_no_c2x(XKeyEvent, XKeyEvent *)
Xm_type_ptr_no_c2x(XKeymapEvent, XKeymapEvent *)
Xm_type_ptr_no_c2x(XMapEvent, XMapEvent *)
Xm_type_ptr_no_c2x(XMapRequestEvent, XMapRequestEvent *)
Xm_type_ptr_no_c2x(XMappingEvent, XMappingEvent *)
Xm_type_ptr_no_c2x(XMotionEvent, XMotionEvent *)
Xm_type_ptr_no_c2x(XNoExposeEvent, XNoExposeEvent *)
Xm_type_ptr_no_c2x(XPropertyEvent, XPropertyEvent *)
Xm_type_ptr_no_c2x(XReparentEvent, XReparentEvent *)
Xm_type_ptr_no_c2x(XResizeRequestEvent, XResizeRequestEvent *)
Xm_type_ptr_no_c2x(XSelectionClearEvent, XSelectionClearEvent *)
Xm_type_ptr_no_c2x(XSelectionEvent, XSelectionEvent *)
Xm_type_ptr(XSelectionRequestEvent, XSelectionRequestEvent *)
Xm_type_ptr_no_c2x(XUnmapEvent, XUnmapEvent *)
Xm_type_ptr_no_c2x(XVisibilityEvent, XVisibilityEvent *)
Xm_type_ptr_obj(XSetWindowAttributes, XSetWindowAttributes *)
Xm_type_ptr(XVisualInfo, XVisualInfo *)
Xm_type_ptr(XWMHints, XWMHints *)
Xm_type_ptr_no_c2x_no_p(XSizeHints, XSizeHints *)
Xm_type_ptr(XWindowAttributes, XWindowAttributes *)
Xm_type_ptr_obj(XWindowChanges, XWindowChanges *)
Xm_type_ptr(XStandardColormap, XStandardColormap *)
Xm_type_int(KeyCode, KeyCode)
Xm_type_int(XContext, XContext)
Xm_type_ptr(XIconSize, XIconSize *)

Xm_type_ptr(Widget, Widget)
Xm_type(WidgetClass, WidgetClass)
Xm_type(XtAppContext, XtAppContext)
Xm_type(XtRequestId, XtRequestId)
Xm_type(XtWorkProcId, XtWorkProcId)
Xm_type(XtInputId, XtInputId)
Xm_type(XtIntervalId, XtIntervalId)
Xm_type_no_p(XtActionHookId, XtActionHookId)
Xm_type_no_p(XtTranslations, XtTranslations) /* opaque */
Xm_type_ptr(XmString, XmString)
Xm_type_ptr_no_p_NO_P(XmAnyCallbackStruct, XmAnyCallbackStruct *)
Xm_type_ptr_no_p(XmArrowButtonCallbackStruct, XmArrowButtonCallbackStruct *)
Xm_type_ptr_no_p(XmCommandCallbackStruct, XmCommandCallbackStruct *)
Xm_type_ptr_no_p(XmDragDropFinishCallbackStruct, XmDragDropFinishCallbackStruct *)
Xm_type_ptr_no_p(XmDragMotionCallbackStruct, XmDragMotionCallbackStruct *)
Xm_type_ptr_no_p(XmDragProcCallbackStruct, XmDragProcCallbackStruct *)
Xm_type_ptr_no_p(XmDrawingAreaCallbackStruct, XmDrawingAreaCallbackStruct *)
Xm_type_ptr_no_p(XmDrawnButtonCallbackStruct, XmDrawnButtonCallbackStruct *)
Xm_type_ptr_no_p(XmDropFinishCallbackStruct, XmDropFinishCallbackStruct *)
Xm_type_ptr_no_p(XmDropProcCallbackStruct, XmDropProcCallbackStruct *)
Xm_type_ptr_no_p(XmDropSiteEnterCallbackStruct, XmDropSiteEnterCallbackStruct *)
Xm_type_ptr_no_p(XmDropSiteLeaveCallbackStruct, XmDropSiteLeaveCallbackStruct *)
Xm_type_ptr_no_p(XmDropStartCallbackStruct, XmDropStartCallbackStruct *)
Xm_type_ptr_no_p(XmFileSelectionBoxCallbackStruct, XmFileSelectionBoxCallbackStruct *)
Xm_type_ptr_no_p(XmListCallbackStruct, XmListCallbackStruct *)
Xm_type(XmTab, XmTab) /* opaque */
Xm_type_ptr_no_p(XmDragStartCallbackStruct, XmDragStartCallbackStruct *)
Xm_type_ptr_no_p(XmDisplayCallbackStruct, XmDisplayCallbackStruct *)
Xm_type_ptr_no_p(XmDestinationCallbackStruct, XmDestinationCallbackStruct *)
Xm_type_ptr_no_p(XmConvertCallbackStruct, XmConvertCallbackStruct *)
Xm_type_ptr_no_p(XmComboBoxCallbackStruct, XmComboBoxCallbackStruct *)
Xm_type_ptr_no_p(XmContainerOutlineCallbackStruct, XmContainerOutlineCallbackStruct *)
Xm_type_ptr_no_p(XmContainerSelectCallbackStruct, XmContainerSelectCallbackStruct *)
Xm_type_ptr_no_p(XmNotebookCallbackStruct, XmNotebookCallbackStruct *)
Xm_type_ptr_no_p(XmNotebookPageInfo, XmNotebookPageInfo *)
Xm_type_ptr(XmRenderTable, XmRenderTable)
Xm_type_ptr(XmRendition, XmRendition)
Xm_type_ptr_no_p(XmSpinBoxCallbackStruct, XmSpinBoxCallbackStruct *)
Xm_type_ptr_no_p(XmTraverseObscuredCallbackStruct, XmTraverseObscuredCallbackStruct *)
Xm_type_ptr_no_p(XmTopLevelLeaveCallbackStruct, XmTopLevelLeaveCallbackStruct *)
Xm_type_ptr_no_p(XmTopLevelEnterCallbackStruct, XmTopLevelEnterCallbackStruct *)
Xm_type_ptr_no_p(XmPopupHandlerCallbackStruct, XmPopupHandlerCallbackStruct *)
Xm_type_ptr_no_p(XmSelectionCallbackStruct, XmSelectionCallbackStruct *)
Xm_type_ptr_no_c2x_no_p(XmTransferDoneCallbackStruct, XmTransferDoneCallbackStruct *)
Xm_type_ptr(XmTabList, XmTabList) /* opaque */
Xm_type(XmParseMapping, XmParseMapping)
Xm_type_ptr_no_p(XmOperationChangedCallbackStruct, XmOperationChangedCallbackStruct *)
Xm_type_ptr_no_p(XmPushButtonCallbackStruct, XmPushButtonCallbackStruct *)
Xm_type_ptr_no_p(XmRowColumnCallbackStruct, XmRowColumnCallbackStruct *)
Xm_type_ptr_no_p(XmScaleCallbackStruct, XmScaleCallbackStruct *)
Xm_type_ptr_no_p(XmScrollBarCallbackStruct, XmScrollBarCallbackStruct *)
Xm_type_ptr_no_p(XmSelectionBoxCallbackStruct, XmSelectionBoxCallbackStruct *)
Xm_type_ptr_no_p(XmTextVerifyCallbackStruct, XmTextVerifyCallbackStruct *)
Xm_type_ptr_no_c2x_no_p(XmTextBlock, XmTextBlock)
Xm_type_ptr_no_p(XmToggleButtonCallbackStruct, XmToggleButtonCallbackStruct *)
#define Xen_to_C_XmFontList(Arg) Xen_to_C_XmRenderTable(Arg)
Xm_type(XmTextSource, XmTextSource)
Xm_type(XmStringContext, XmStringContext)

static int Xen_is_XmFontList_or_XmRenderTable(Xen arg)
{
  return(Xen_is_XmRenderTable(arg));
}

static Xen type_to_event_symbol(int utype)
{
  const char *type;
  type = "XErrorEvent"; /* -1 for an error event? */
  switch (utype)
    {
    case 0:                type = "XAnyEvent";              break;
    case KeyPress:
    case KeyRelease:       type = "XKeyEvent";              break;
    case ButtonPress:
    case ButtonRelease:    type = "XButtonEvent";           break;
    case MotionNotify:     type = "XMotionEvent";           break;
    case EnterNotify:
    case LeaveNotify:      type = "XCrossingEvent";         break;
    case FocusIn:
    case FocusOut:         type = "XFocusChangeEvent";      break;
    case KeymapNotify:     type = "XKeymapEvent";           break;
    case Expose:           type = "XExposeEvent";           break;
    case GraphicsExpose:   type = "XGraphicsExposeEvent";   break;
    case NoExpose:         type = "XNoExposeEvent";         break;
    case VisibilityNotify: type = "XVisibilityEvent";       break;
    case CreateNotify:     type = "XCreateWindowEvent";     break;
    case DestroyNotify:    type = "XDestroyWindowEvent";    break;
    case UnmapNotify:      type = "XUnmapEvent";            break;
    case MapNotify:        type = "XMapEvent";              break;
    case MapRequest:       type = "XMapRequestEvent";       break;
    case ReparentNotify:   type = "XReparentEvent";         break;
    case ConfigureNotify:  type = "XConfigureEvent";        break;
    case ConfigureRequest: type = "XConfigureRequestEvent"; break;
    case GravityNotify:    type = "XGravityEvent";          break;
    case ResizeRequest:    type = "XResizeRequestEvent";    break;
    case CirculateNotify:  type = "XCirculateEvent";        break;
    case CirculateRequest: type = "XCirculateRequestEvent"; break;
    case PropertyNotify:   type = "XPropertyEvent";         break;
    case SelectionClear:   type = "XSelectionClearEvent";   break;
    case SelectionRequest: type = "XSelectionRequestEvent"; break;
    case SelectionNotify:  type = "XSelectionEvent";        break;
    case ColormapNotify:   type = "XColormapEvent";         break;
    case ClientMessage:    type = "XClientMessageEvent";    break;
    case MappingNotify:    type = "XMappingEvent";          break;
    }
  return(C_string_to_Xen_symbol(type));
}

static Xen C_to_Xen_XEvent_1(XEvent *e, int need_free)
{
  if (e == NULL) return(Xen_false); /* synthetic callback may have no event */
  if (need_free)
    return(Xen_list_4(type_to_event_symbol(e->type),
		      Xen_wrap_C_pointer(e),
		      make_xm_obj(e),
		      C_string_to_Xen_symbol("XEvent")));
  return(Xen_list_4(type_to_event_symbol(e->type),
		    Xen_wrap_C_pointer(e),
		    Xen_false,
		    C_string_to_Xen_symbol("XEvent")));
}


#define C_to_Xen_XEvent(e)     C_to_Xen_XEvent_1(e, false)
#define C_to_Xen_XEvent_OBJ(e) C_to_Xen_XEvent_1(e, true)
#define Xen_to_C_XEvent(Arg)   (XEvent *)Xen_unwrap_C_pointer(Xen_cadr(Arg))
#define Xen_is_XEvent(Value)    (Xen_is_list(Value) &&\
                               (Xen_list_length(Value) == 4) &&\
                               (Xen_is_symbol(Xen_cadddr(Value))) &&\
                               (strcmp("XEvent", Xen_symbol_to_C_string(Xen_cadddr(Value))) == 0))

static Xen g_is_XEvent(Xen val) 
{
  return(C_bool_to_Xen_boolean(Xen_is_XEvent(val)));
}

static Xen gxm_XEvent(Xen type)
{
  XEvent *e;
  Xen_check_type(Xen_is_integer_or_unbound(type), type, 1, "XEvent", "an X event type (integer)");
  e = (XEvent *)calloc(1, sizeof(XEvent));
  if (Xen_is_integer(type))
    e->type = Xen_integer_to_C_int(type);
  return(Xen_list_4(type_to_event_symbol(e->type),
		    Xen_wrap_C_pointer(e),
		    make_xm_obj(e),
		    C_string_to_Xen_symbol("XEvent")));
}

static Xen gxm_XGCValues(void) 
{
  XGCValues *e;
  e = (XGCValues *)calloc(1, sizeof(XGCValues));
  return(wrap_for_Xen_obj("XGCValues", e));
}

#define XM_Make(Name) \
  static Xen gxm_ ## Name (void) {Name *e; e = (Name *)calloc(1, sizeof(Name)); return(wrap_for_Xen_obj(#Name, e));} \
  Xen_wrap_no_args(gxm_ ## Name ## _w, gxm_ ## Name)
#define XM_Declare(Name) \
  Xen_define_safe_procedure(XM_PREFIX #Name XM_POSTFIX, gxm_ ## Name ## _w, 0, 0, 0, "Make an " #Name " struct")

XM_Make(XmAnyCallbackStruct)
XM_Make(XmArrowButtonCallbackStruct)
XM_Make(XmCommandCallbackStruct)
XM_Make(XmDragDropFinishCallbackStruct)
XM_Make(XmDragMotionCallbackStruct)
XM_Make(XmDragProcCallbackStruct)
XM_Make(XmDrawingAreaCallbackStruct)
XM_Make(XmDrawnButtonCallbackStruct)
XM_Make(XmDropFinishCallbackStruct)
XM_Make(XmDropProcCallbackStruct)
XM_Make(XmDropSiteEnterCallbackStruct)
XM_Make(XmDropSiteLeaveCallbackStruct)
XM_Make(XmDropStartCallbackStruct)
XM_Make(XmFileSelectionBoxCallbackStruct)
XM_Make(XmListCallbackStruct)
XM_Make(XmOperationChangedCallbackStruct)
XM_Make(XmPushButtonCallbackStruct)
XM_Make(XmRowColumnCallbackStruct)
XM_Make(XmScaleCallbackStruct)
XM_Make(XmScrollBarCallbackStruct)
XM_Make(XmSelectionBoxCallbackStruct)
XM_Make(XmTextVerifyCallbackStruct)
XM_Make(XmToggleButtonCallbackStruct)
XM_Make(XmDestinationCallbackStruct)
XM_Make(XmConvertCallbackStruct)
XM_Make(XmComboBoxCallbackStruct)
XM_Make(XmContainerOutlineCallbackStruct)
XM_Make(XmContainerSelectCallbackStruct)
XM_Make(XmNotebookCallbackStruct)
XM_Make(XmSpinBoxCallbackStruct)
XM_Make(XmTraverseObscuredCallbackStruct)
XM_Make(XmTopLevelLeaveCallbackStruct)
XM_Make(XmTopLevelEnterCallbackStruct)
XM_Make(XmPopupHandlerCallbackStruct)
XM_Make(XmSelectionCallbackStruct)
XM_Make(XmTransferDoneCallbackStruct)
XM_Make(XmDisplayCallbackStruct)
XM_Make(XmDragStartCallbackStruct)

static Xen gxm_XmTextBlock(void) 
{
  XmTextBlockRec *e; 
  e = (XmTextBlockRec *)calloc(1, sizeof(XmTextBlockRec)); 
  return(wrap_for_Xen_obj("XmTextBlock", e));
}

Xen_wrap_no_args(gxm_XmTextBlock_w, gxm_XmTextBlock)
  
static void define_makes(void)
{
  XM_Declare(XmAnyCallbackStruct);
  XM_Declare(XmArrowButtonCallbackStruct);
  XM_Declare(XmCommandCallbackStruct);
  XM_Declare(XmDragDropFinishCallbackStruct);
  XM_Declare(XmDragMotionCallbackStruct);
  XM_Declare(XmDragProcCallbackStruct);
  XM_Declare(XmDrawingAreaCallbackStruct);
  XM_Declare(XmDrawnButtonCallbackStruct);
  XM_Declare(XmDropFinishCallbackStruct);
  XM_Declare(XmDropProcCallbackStruct);
  XM_Declare(XmDropSiteEnterCallbackStruct);
  XM_Declare(XmDropSiteLeaveCallbackStruct);
  XM_Declare(XmDropStartCallbackStruct);
  XM_Declare(XmFileSelectionBoxCallbackStruct);
  XM_Declare(XmListCallbackStruct);
  XM_Declare(XmOperationChangedCallbackStruct);
  XM_Declare(XmPushButtonCallbackStruct);
  XM_Declare(XmRowColumnCallbackStruct);
  XM_Declare(XmScaleCallbackStruct);
  XM_Declare(XmScrollBarCallbackStruct);
  XM_Declare(XmSelectionBoxCallbackStruct);
  XM_Declare(XmTextVerifyCallbackStruct);
  XM_Declare(XmToggleButtonCallbackStruct);
  Xen_define_safe_procedure(XM_PREFIX "XmTextBlock" XM_POSTFIX, gxm_XmTextBlock_w, 0, 0, 0, "Make an XmTextBlock struct");
  XM_Declare(XmDestinationCallbackStruct);
  XM_Declare(XmConvertCallbackStruct);
  XM_Declare(XmComboBoxCallbackStruct);
  XM_Declare(XmContainerOutlineCallbackStruct);
  XM_Declare(XmContainerSelectCallbackStruct);
  XM_Declare(XmNotebookCallbackStruct);
  XM_Declare(XmSpinBoxCallbackStruct);
  XM_Declare(XmTraverseObscuredCallbackStruct);
  XM_Declare(XmTopLevelLeaveCallbackStruct);
  XM_Declare(XmTopLevelEnterCallbackStruct);
  XM_Declare(XmPopupHandlerCallbackStruct);
  XM_Declare(XmSelectionCallbackStruct);
  XM_Declare(XmTransferDoneCallbackStruct);
  XM_Declare(XmDisplayCallbackStruct);
  XM_Declare(XmDragStartCallbackStruct);
}

static int its_a_callbackstruct(const char *name)
{
  if (name)
    {
      int len;
      len = strlen(name);
      if (len > 16)
	{
	  char *end_name;
	  end_name = (char *)(name + len - 14);
	  return(strcmp(end_name, "CallbackStruct") == 0);
	}
    }
  return(0);
}

#define Xen_is_AnyCallbackStruct(Value) (Xen_is_list(Value) && \
                                       (Xen_list_length(Value) >= 2) && \
                                       (Xen_is_symbol(Xen_car(Value))) && \
                                       (its_a_callbackstruct(Xen_symbol_to_C_string(Xen_car(Value)))))

static int xm_protect(Xen obj);
static void xm_unprotect_at(int ind);


/* in XtGetValues we need to return tagged-types (etc) for arbitrarily named resources,
 *   and in XtSetValues we need to do type checks, so resources are hashed by
 *   name and type:
 */

typedef enum {XM_INT, XM_ULONG, XM_UCHAR, XM_FLOAT, XM_STRING, XM_XMSTRING, XM_STRING_TABLE,
	      XM_INT_TABLE, XM_RENDER_TABLE, XM_TAB_LIST, XM_WIDGET, XM_WIDGET_LIST, 
	      XM_BOOLEAN, XM_CALLBACK, XM_PIXEL, XM_PIXMAP, XM_XFONTSTRUCT, XM_DIMENSION,
	      XM_ATOM, XM_ATOM_LIST, XM_STRING_LIST, XM_CHARSET_TABLE, XM_TEXT_SOURCE,
	      XM_COLORMAP, XM_KEYSYM, XM_KEYSYM_TABLE, XM_SCREEN, XM_WINDOW,
	      XM_VISUAL, XM_RECTANGLE_LIST, XM_WIDGET_CLASS, XM_STRING_OR_INT,
	      XM_TRANSFER_CALLBACK, XM_CONVERT_CALLBACK, XM_SEARCH_CALLBACK, XM_ORDER_CALLBACK,
	      XM_QUALIFY_CALLBACK, XM_ALLOC_COLOR_CALLBACK, XM_POPUP_CALLBACK, XM_SCREEN_COLOR_CALLBACK,
	      XM_DROP_CALLBACK, XM_TRANSFER_ENTRY_LIST, XM_DRAG_CALLBACK, XM_STRING_OR_XMSTRING, XM_PARSE_CALLBACK,
	      XM_BOOLEAN_OR_INT, XM_POSITION, XM_SHORT, XM_CURSOR, XM_XTPOINTER,
	      XM_NOT_A_RESOURCE
} xm_resource_t;

static xm_resource_t resource_type(const char *resource);

static Xen C_to_Xen_Widgets(Widget *array, int len)
{
  Xen lst = Xen_empty_list;
  if (array)
    {
      int i, loc;
      loc = xm_protect(lst);
      for (i = len - 1; i >= 0; i--)
	lst = Xen_cons(C_to_Xen_Widget(array[i]), lst);
      xm_unprotect_at(loc);
    }
  return(lst);
}

static Xen C_to_Xen_XmStringTable(XmStringTable array, int len)
{
  Xen lst = Xen_empty_list;
  if (array)
    {
      int i, loc;
      loc = xm_protect(lst);
      for (i = len - 1; i >= 0; i--)
	lst = Xen_cons(C_to_Xen_XmString(array[i]), lst);
      xm_unprotect_at(loc);
    }
  return(lst);
}

static Widget *Xen_to_C_Widgets(Xen lst_1, int n)
{
  Widget *ws;
  int i;
  Xen lst;
  lst = Xen_copy_arg(lst_1);
  ws = (Widget *)calloc(n, sizeof(Widget));
  for (i = 0; (i < n) && (!Xen_is_null(lst)); i++, lst = Xen_cdr(lst))
    if (Xen_is_Widget(Xen_car(lst)))
      ws[i] = Xen_to_C_Widget(Xen_car(lst));
    else 
      {
	free(ws);
	ws = NULL;
	Xen_check_type(0, Xen_car(lst), i, __func__, "a Widget");
	break;
      }
  return(ws);
}

static XmString *Xen_to_C_XmStrings(Xen v_1, int len)
{
  XmString *str;
  int i;
  Xen v;
  v = Xen_copy_arg(v_1);
  str = (XmString *)calloc(len, sizeof(XmString));
  for (i = 0; (i < len) && (!Xen_is_null(v)); i++, v = Xen_cdr(v))
    if (Xen_is_XmString(Xen_car(v)))
      str[i] = (XmString)Xen_to_C_XmString(Xen_car(v));
    else 
      {
	free(str);
	str = NULL;
	Xen_check_type(0, Xen_car(v), i, __func__, "an XmString");
	break;
      }
  return(str);
}

static XmDropTransferEntryRec *Xen_to_C_XmDropTransferEntryRecs(Xen v_1, int len)
{
  XmDropTransferEntryRec *ps;
  int i;
  Xen v;
  v = Xen_copy_arg(v_1);
  ps = (XmDropTransferEntryRec *)calloc(len, sizeof(XmDropTransferEntryRec));
  for (i = 0; (i < len) && (!Xen_is_null(v)); i++, v = Xen_cdr(v))
    {
      if (Xen_is_Atom(Xen_car(v)))
	ps[i].target = Xen_to_C_Atom(Xen_car(v));
      else 
	{
	  if (Xen_is_list(Xen_car(v)))
	    {
	      if (Xen_is_Atom(Xen_car(Xen_car(v))))
		ps[i].target = Xen_to_C_Atom(Xen_car(Xen_car(v)));
	      else Xen_check_type(0, Xen_car(v), i, __func__, "an Atom");
	      ps[i].client_data = (XtPointer)Xen_cadr(Xen_car(v));
	    }
	}
    }
  return(ps);
}

static XmStringTable Xen_to_C_XmStringTable(Xen v_1, int len)
{
  Xen v;
  XmStringTable str;
  int i;
  v = Xen_copy_arg(v_1);
  str = (XmStringTable)calloc(len + 1, sizeof(XmString));
  for (i = 0; (i < len) && (!Xen_is_null(v)); i++, v = Xen_cdr(v))
    if (Xen_is_XmString(Xen_car(v)))
      str[i] = Xen_to_C_XmString(Xen_car(v));
    else 
      {
	free(str);
	str = NULL;
	Xen_check_type(0, Xen_car(v), i, __func__, "an XmString");
	break;
      }
  return(str);
}

static Xen C_to_Xen_Ints(int *array, int len)
{
  Xen lst = Xen_empty_list;
  if (array)
    {
      int i, loc;
      loc = xm_protect(lst);
      for (i = len - 1; i >= 0; i--)
	lst = Xen_cons(C_int_to_Xen_integer(array[i]), lst);
      xm_unprotect_at(loc);
    }
  return(lst);
}

#if HAVE_SCHEME
static Xen c_to_xen_ints(Xen array, Xen len)
{
  #define H_to_ints "->ints translates a Motif int array (from a .value reference for example) into a scheme list of ints"
  Xen_check_type(Xen_is_wrapped_c_pointer(array), array, 1, "->ints", "int*");
  return(C_to_Xen_Ints((int *)Xen_unwrap_C_pointer(array), Xen_integer_to_C_int(len)));
}
#endif

static Xen C_to_Xen_Atoms(Atom *array, int len)
{
  Xen lst = Xen_empty_list;
  if (array)
    {
      int i, loc;
      loc = xm_protect(lst);
      for (i = len - 1; i >= 0; i--)
	lst = Xen_cons(C_to_Xen_Atom(array[i]), lst);
      xm_unprotect_at(loc);
    }
  return(lst);
}

#if HAVE_SCHEME
static Xen c_to_xen_atoms(Xen array, Xen len)
{
  #define H_to_Atoms "->Atoms translates a Motif Atoms array (from a .value reference for example) into a scheme list of Atoms"
  Xen_check_type(Xen_is_wrapped_c_pointer(array), array, 1, "->Atoms", "Atom*");
  return(C_to_Xen_Atoms((Atom *)Xen_unwrap_C_pointer(array), Xen_integer_to_C_int(len)));
}
#endif

static Xen C_to_Xen_Strings(char **array, int len)
{
  Xen lst = Xen_empty_list;
  if (array)
    {
      int i, loc;
      loc = xm_protect(lst);
      for (i = len - 1; i >= 0; i--)
	lst = Xen_cons(C_string_to_Xen_string(array[i]), lst);
      xm_unprotect_at(loc);
    }
  return(lst);
}

#if HAVE_SCHEME
static Xen c_to_xen_strings(Xen array, Xen len)
{
  #define H_to_strings "->strings translates a Motif string array (from a .value reference for example) into a scheme list of strings"
  Xen_check_type(Xen_is_wrapped_c_pointer(array), array, 1, "->strings", "char**");
  return(C_to_Xen_Strings((char **)Xen_unwrap_C_pointer(array), Xen_integer_to_C_int(len)));
}
#endif

static Xen copy_xrectangle(XRectangle *old_r)
{
  XRectangle *r;
  r = (XRectangle *)calloc(1, sizeof(XRectangle));
  r->x = old_r->x;
  r->y = old_r->y;
  r->width = old_r->width;
  r->height = old_r->height;
  return(C_to_Xen_XRectangle(r));
}

static Xen C_to_Xen_XRectangles(XRectangle *array, int len)
{
  Xen lst = Xen_empty_list;
  if (array)
    {
      int i, loc;
      loc = xm_protect(lst);
      for (i = len - 1; i >= 0; i--)
	lst = Xen_cons(copy_xrectangle(&(array[i])), lst);
      xm_unprotect_at(loc);
    }
  return(lst);
}

#if HAVE_SCHEME
static Xen c_to_xen_xrectangles(Xen array, Xen len)
{
  #define H_to_XRectangles "->XRectangles translates a Motif rectangle array (from a .value reference for example) into a scheme list of rectangles"
  Xen_check_type(Xen_is_wrapped_c_pointer(array), array, 1, "->XRectangles", "Motif rectangle array");
  return(C_to_Xen_XRectangles((XRectangle *)Xen_unwrap_C_pointer(array), Xen_integer_to_C_int(len)));
}
#endif

static Xen C_to_Xen_KeySyms(KeySym *array, int len)
{
  Xen lst = Xen_empty_list;
  if (array)
    {
      int i, loc;
      loc = xm_protect(lst);
      for (i = len - 1; i >= 0; i--)
	lst = Xen_cons(C_to_Xen_KeySym(array[i]), lst);
      xm_unprotect_at(loc);
    }
  return(lst);
}

static Window *Xen_to_C_Windows(Xen lst_1, int n)
{
  Window *ws;
  int i;
  Xen lst;
  if (n == 0) return(NULL);
  lst = Xen_copy_arg(lst_1);
  ws = (Window *)calloc(n, sizeof(Window));
  for (i = 0; (i < n) && (!Xen_is_null(lst)); i++, lst = Xen_cdr(lst))
    if (Xen_is_Window(Xen_car(lst)))
      ws[i] = Xen_to_C_Window(Xen_car(lst));
    else 
      {
	free(ws);
	ws = NULL;
	Xen_check_type(0, Xen_car(lst), i, __func__, "a Window");
	break;
      }
  return(ws);
}

static XmRendition *Xen_to_C_XmRenditions(Xen lst_1, int n)
{
  Xen lst;
  XmRendition *ws;
  int i;
  if (n == 0) return(NULL);
  lst = Xen_copy_arg(lst_1);
  ws = (XmRendition *)calloc(n, sizeof(XmRendition));
  for (i = 0; (i < n) && (!Xen_is_null(lst)); i++, lst = Xen_cdr(lst))
    if (Xen_is_XmRendition(Xen_car(lst)))
      ws[i] = Xen_to_C_XmRendition(Xen_car(lst));
    else 
      {
	free(ws);
	ws = NULL;
	Xen_check_type(0, Xen_car(lst), i, __func__, "an XmRendition");
	break;
      }
  return(ws);
}

static XmTab *Xen_to_C_XmTabs(Xen v_1, int len)
{
  Xen v;
  XmTab *str;
  int i;
  if (len == 0) return(NULL);
  v = Xen_copy_arg(v_1);
  str = (XmTab *)calloc(len, sizeof(XmTab));
  for (i = 0; (i < len) && (!Xen_is_null(v)); i++, v = Xen_cdr(v))
    if (Xen_is_XmTab(Xen_car(v)))
      str[i] = (XmTab)Xen_to_C_XmTab(Xen_car(v));
    else 
      {
	free(str);
	str = NULL;
	Xen_check_type(0, Xen_car(v), i, __func__, "an XmTab");
	break;
      }
  return(str);
}

static Atom *Xen_to_C_Atoms(Xen v_1, int len)
{
  Xen v;
  Atom *str;
  int i;
  if (len == 0) return(NULL);
  v = Xen_copy_arg(v_1);
  str = (Atom *)calloc(len, sizeof(Atom));
  for (i = 0; (i < len) && (!Xen_is_null(v)); i++, v = Xen_cdr(v))
    if (Xen_is_Atom(Xen_car(v)))
      str[i] = (Atom)Xen_to_C_Atom(Xen_car(v));
    else 
      {
	free(str);
	str = NULL;
	Xen_check_type(0, Xen_car(v), i, __func__, "an Atom");
	break;
      }
  return(str);
}

static Pixel *Xen_to_C_Pixels(Xen v_1, int len)
{
  Xen v;
  Pixel *str;
  int i;
  if (len == 0) return(NULL);
  v = Xen_copy_arg(v_1);
  str = (Pixel *)calloc(len, sizeof(Pixel));
  for (i = 0; (i < len) && (!Xen_is_null(v)); i++, v = Xen_cdr(v))
    if (Xen_is_Pixel(Xen_car(v)))
      str[i] = (Pixel)Xen_to_C_Pixel(Xen_car(v));
    else 
      {
	free(str);
	str = NULL;
	Xen_check_type(0, Xen_car(v), i, __func__, "a Pixel");
	break;
      }
  return(str);
}

static KeySym *Xen_to_C_KeySyms(Xen v_1, int len)
{
  Xen v;
  KeySym *str;
  int i;
  if (len == 0) return(NULL);
  v = Xen_copy_arg(v_1);
  str = (KeySym *)calloc(len, sizeof(KeySym));
  for (i = 0; (i < len) && (!Xen_is_null(v)); i++, v = Xen_cdr(v))
    if (Xen_is_KeySym(Xen_car(v)))
      str[i] = (KeySym)Xen_to_C_KeySym(Xen_car(v));
    else 
      {
	free(str);
	str = NULL;
	Xen_check_type(0, Xen_car(v), i, __func__, "a KeySym");
	break;
      }
  return(str);
}

static char **Xen_to_C_Strings(Xen v_1, int len)
{
  Xen v;
  char **str;
  int i;
  if (len == 0) return(NULL);
  v = Xen_copy_arg(v_1);
  str = (char **)calloc(len, sizeof(char *));
  for (i = 0; (i < len) && (!Xen_is_null(v)); i++, v = Xen_cdr(v))
    if (Xen_is_string(Xen_car(v)))
      str[i] = (char *)Xen_string_to_C_string(Xen_car(v)); /* should this be protected? */
    else 
      {
	free(str);
	str = NULL;
	Xen_check_type(0, Xen_car(v), i, __func__, "a char*");
	break;
      }
  return(str);
}

static int *Xen_to_C_Ints(Xen v_1, int len)
{
  Xen v;
  int *ps;
  int i;
  if (len == 0) return(NULL);
  v = Xen_copy_arg(v_1);
  ps = (int *)calloc(len, sizeof(int));
  for (i = 0; (i < len) && (!Xen_is_null(v)); i++, v = Xen_cdr(v))
    if (Xen_is_integer(Xen_car(v)))
      ps[i] = Xen_integer_to_C_int(Xen_car(v));
    else 
      {
	free(ps);
	ps = NULL;
	Xen_check_type(0, Xen_car(v), i, __func__, "an int");
	break;
      }
  return(ps);
}

static Cardinal *Xen_to_C_Cardinals(Xen v_1, int len)
{
  Xen v;
  Cardinal *ps;
  int i;
  if (len == 0) return(NULL);
  v = Xen_copy_arg(v_1);
  ps = (Cardinal *)calloc(len, sizeof(int));
  for (i = 0; (i < len) && (!Xen_is_null(v)); i++, v = Xen_cdr(v))
    if (Xen_is_integer(Xen_car(v)))
      ps[i] = (Cardinal)Xen_integer_to_C_int(Xen_car(v));
    else 
      {
	free(ps);
	ps = NULL;
	Xen_check_type(0, Xen_car(v), i, __func__, "a Cardinal");
	break;
      }
  return(ps);
}

static XRectangle *Xen_to_C_XRectangles(Xen v_1, int len)
{
  Xen v;
  XRectangle *str, *dat;
  int i;
  if (len == 0) return(NULL);
  v = Xen_copy_arg(v_1);
  str = (XRectangle *)calloc(len, sizeof(XRectangle));
  for (i = 0; (i < len) && (!Xen_is_null(v)); i++, v = Xen_cdr(v))
    {
      if (Xen_is_XRectangle(Xen_car(v)))
	dat = (XRectangle *)Xen_to_C_XRectangle(Xen_car(v));
      else 
	{
	  free(str);
	  str = NULL;
	  Xen_check_type(0, Xen_car(v), i, __func__, "an XRectangle");
	  break;
	}
      str[i].x = dat->x;
      str[i].y = dat->y;
      str[i].width = dat->width;
      str[i].height = dat->height;
    }
  return(str);
}



/* -------- arglists -------- */

static Xen wrap_callback_struct(int type, XtPointer info);
static int map_over_protected_elements(bool (*func)(Xen val, int loc, unsigned long fid), unsigned long id);
static Xen xm_protected_element(int loc);
static int callback_struct_type(Widget w, const char *name);
static void xm_unprotect(Xen obj);

enum {CALLBACK_TYPE, CALLBACK_FUNC, CALLBACK_DATA, CALLBACK_STRUCT_TYPE, CALLBACK_GC_LOC};

#define C_to_Xen_XM_XtCallback(Code, Context) \
  Xen_list_5(C_string_to_Xen_symbol("XtCallback"), Code, Context, Xen_integer_zero, Xen_integer_zero)

static void gxm_XtCallbackProc(Widget w, XtPointer context, XtPointer info)
{
  Xen descr = (Xen)context;
  Xen_call_with_3_args(Xen_list_ref(descr, CALLBACK_FUNC),    /* descr: (list "XtCallback" func user-data struct-type gc-loc) */
	     C_to_Xen_Widget(w),
	     Xen_list_ref(descr, CALLBACK_DATA),
	     wrap_callback_struct(Xen_integer_to_C_int(Xen_list_ref(descr, CALLBACK_STRUCT_TYPE)), info),
	     __func__);
}

#define C_to_Xen_XM_Drop_Callback(Code) \
  Xen_list_5(C_string_to_Xen_symbol("Drop_Callback"), Code, Xen_false, Xen_integer_zero, Xen_integer_zero)
#define XM_is_Drop_Callback(Arg) is_wrapped("Drop_Callback", Arg)

static bool find_dropproc(Xen val, int loc, unsigned long w)
{
  return((XM_is_Drop_Callback(val)) &&
	 (((Xen_is_false((Xen)w)) && 
	   (Xen_is_false(Xen_list_ref(val, CALLBACK_DATA)))) ||
	  ((Xen_is_Widget(Xen_list_ref(val, CALLBACK_DATA))) &&
	   (Xen_ulong_to_C_ulong(Xen_cadr(Xen_list_ref(val, CALLBACK_DATA))) == w))));
}

static void gxm_Drop_Callback(Widget w, XtPointer context, XtPointer info)
{
  int i;
  XmDropProcCallbackStruct *cb = (XmDropProcCallbackStruct *)info;
  i = map_over_protected_elements(find_dropproc, (unsigned long)w);
  if (i >= 0)
    {
      Xen code;
      code = Xen_list_ref(xm_protected_element(i), CALLBACK_FUNC);
      if (Xen_is_procedure(code))
	Xen_call_with_3_args(code,
		   C_to_Xen_Widget(w),
		   Xen_false,
		   C_to_Xen_XmDropProcCallbackStruct(cb),
		   __func__);
    }
  cb->dropSiteStatus = XmINVALID_DROP_SITE; /* try to exit cleanly from on-going drop */
}

#define C_to_Xen_XM_Drag_Callback(Code) \
  Xen_list_5(C_string_to_Xen_symbol("Drag_Callback"), Code, Xen_false, Xen_integer_zero, Xen_integer_zero)
#define XM_is_Drag_Callback(Arg) is_wrapped("Drag_Callback", Arg)

static bool find_dragproc(Xen val, int loc, unsigned long w)
{
  return((XM_is_Drag_Callback(val)) &&
	 (((Xen_is_false((Xen)w)) && 
	   (Xen_is_false(Xen_list_ref(val, CALLBACK_DATA)))) ||
	  ((Xen_is_Widget(Xen_list_ref(val, CALLBACK_DATA))) &&
	   (Xen_ulong_to_C_ulong(Xen_cadr(Xen_list_ref(val, CALLBACK_DATA))) == w))));
}

static void gxm_Drag_Callback(Widget w, XtPointer context, XtPointer info)
{
  int i;
  XmDragProcCallbackStruct *cb = (XmDragProcCallbackStruct *)info;
  i = map_over_protected_elements(find_dragproc, (unsigned long)w);
  if (i >= 0)
    {
      Xen code;
      code = Xen_list_ref(xm_protected_element(i), CALLBACK_FUNC);
      if (Xen_is_procedure(code))
	Xen_call_with_3_args(code,
		   C_to_Xen_Widget(w),
		   Xen_false,
		   C_to_Xen_XmDragProcCallbackStruct(cb),
		   __func__);
    }
}

#define C_to_Xen_XM_XtPopupChild(Code) \
  Xen_list_5(C_string_to_Xen_symbol("XtPopupChild"), Code, Xen_false, Xen_integer_zero, Xen_integer_zero)
#define XM_is_XtPopupChild(Arg) is_wrapped("XtPopupChild", Arg)

static bool find_popupchild(Xen val, int loc, unsigned long w)
{
  return((XM_is_XtPopupChild(val)) &&
	 (((Xen_is_false((Xen)w)) && 
	   (Xen_is_false(Xen_list_ref(val, CALLBACK_DATA)))) ||
	  ((Xen_is_Widget(Xen_list_ref(val, CALLBACK_DATA))) &&
	   (Xen_ulong_to_C_ulong(Xen_cadr(Xen_list_ref(val, CALLBACK_DATA))) == w))));
}

static void gxm_XtPopupChild(Widget w)
{
  /* since all we have to go by is the widget, we have to scan the gc table for any associated code */
  int i;
  i = map_over_protected_elements(find_popupchild, (unsigned long)w);
  if (i >= 0)
    {
      Xen code;
      code = Xen_list_ref(xm_protected_element(i), CALLBACK_FUNC);
      if (Xen_is_procedure(code))
	Xen_call_with_1_arg(code,
		   C_to_Xen_Widget(w),
		   __func__);
    }
}

#define C_to_Xen_XM_XmSearchProc(Code) \
  Xen_list_5(C_string_to_Xen_symbol("XmSearchProc"), Code, Xen_false, Xen_integer_zero, Xen_integer_zero)
#define XM_is_XmSearchProc(Arg) is_wrapped("XmSearchProc", Arg)

static bool find_searchproc(Xen val, int loc, unsigned long w)
{
  return((XM_is_XmSearchProc(val)) &&
	 (((Xen_is_false((Xen)w)) && 
	   (Xen_is_false(Xen_list_ref(val, CALLBACK_DATA)))) ||
	  ((Xen_is_Widget(Xen_list_ref(val, CALLBACK_DATA))) &&
	   (Xen_ulong_to_C_ulong(Xen_cadr(Xen_list_ref(val, CALLBACK_DATA))) == w))));
}

static void gxm_XmSearchProc(Widget w, XmFileSelectionBoxCallbackStruct *info)
{
  /* here we again have to go by the widget */
  int i;
  i = map_over_protected_elements(find_searchproc, (unsigned long)w);
  if (i >= 0)
    {
      Xen code;
      code = Xen_list_ref(xm_protected_element(i), CALLBACK_FUNC);
      if (Xen_is_procedure(code))
	Xen_call_with_2_args(code,
		   C_to_Xen_Widget(w),
		   C_to_Xen_XmFileSelectionBoxCallbackStruct(info),
		   __func__);
    }
}

#define C_to_Xen_XM_XmQualifyProc(Code) \
  Xen_list_5(C_string_to_Xen_symbol("XmQualifyProc"), Code, Xen_false, Xen_integer_zero, Xen_integer_zero)
#define XM_is_XmQualifyProc(Arg) is_wrapped("XmQualifyProc", Arg)

static bool find_qualifyproc(Xen val, int loc, unsigned long w)
{
  return((XM_is_XmQualifyProc(val)) &&
	 (((Xen_is_false((Xen)w)) && 
	   (Xen_is_false(Xen_list_ref(val, CALLBACK_DATA)))) ||
	  ((Xen_is_Widget(Xen_list_ref(val, CALLBACK_DATA))) &&
	   (Xen_ulong_to_C_ulong(Xen_cadr(Xen_list_ref(val, CALLBACK_DATA))) == w))));
}

static void gxm_XmQualifyProc(Widget w, XtPointer indata, XtPointer outdata)
{
  /* here we again have to go by the widget */
  int i;
  i = map_over_protected_elements(find_qualifyproc, (unsigned long)w);
  if (i >= 0)
    {
      Xen code;
      code = Xen_list_ref(xm_protected_element(i), CALLBACK_FUNC);
      if (Xen_is_procedure(code))
	Xen_call_with_3_args(code,
		   C_to_Xen_Widget(w),
		   C_to_Xen_XmFileSelectionBoxCallbackStruct((XmFileSelectionBoxCallbackStruct *)indata),
		   C_to_Xen_XmFileSelectionBoxCallbackStruct((XmFileSelectionBoxCallbackStruct *)outdata),
		   __func__);
    }
}

#define C_to_Xen_XM_XtOrderProc(Code) \
  Xen_list_5(C_string_to_Xen_symbol("XtOrderProc"), Code, Xen_false, Xen_integer_zero, Xen_integer_zero)
#define XM_is_XtOrderProc(Arg) is_wrapped("XtOrderProc", Arg)

static bool find_orderproc(Xen val, int loc, unsigned long w)
{
  return((XM_is_XtOrderProc(val)) &&
	 (((Xen_is_false((Xen)w)) && 
	   (Xen_is_false(Xen_list_ref(val, CALLBACK_DATA)))) ||
	  ((Xen_is_Widget(Xen_list_ref(val, CALLBACK_DATA))) &&
	   (Xen_ulong_to_C_ulong(Xen_cadr(Xen_list_ref(val, CALLBACK_DATA))) == w))));
}

static Cardinal gxm_XtOrderProc(Widget w)
{
  /* here we again have to go by the widget */
  int i, result = 0;
  i = map_over_protected_elements(find_orderproc, (unsigned long)w);
  if (i >= 0)
    {
      Xen code;
      code = Xen_list_ref(xm_protected_element(i), CALLBACK_FUNC);
      if (Xen_is_procedure(code))
	result = Xen_integer_to_C_int(Xen_call_with_1_arg(code,
					 C_to_Xen_Widget(w),
					 __func__));
    }
  return((Cardinal)result);
}


#define C_to_Xen_XM_Parse_Callback(Code) \
  Xen_list_5(C_string_to_Xen_symbol("Parse_Callback"), Code, Xen_false, Xen_integer_zero, Xen_integer_zero)
#define XM_is_Parse_Callback(Arg) is_wrapped("Parse_Callback", Arg)

static bool find_parseproc(Xen val, int loc, unsigned long w)
{
  return((XM_is_Parse_Callback(val)) &&
	 (((Xen_is_false((Xen)w)) && 
	   (Xen_is_false(Xen_list_ref(val, CALLBACK_DATA)))) ||
	  ((Xen_is_XmParseMapping(Xen_list_ref(val, CALLBACK_DATA))) &&
	   (Xen_ulong_to_C_ulong(Xen_cadr(Xen_list_ref(val, CALLBACK_DATA))) == w))));
}

static XmIncludeStatus gxm_Parse_Callback(XtPointer *in_out, XtPointer text_end, XmTextType type, XmStringTag locale_tag,
					  XmParseMapping entry, int pattern_length, XmString *str_include, XtPointer call_data)
{
  int i;
  i = map_over_protected_elements(find_parseproc, (unsigned long)entry);
  if (i >= 0)
    {
      Xen code;
      code = Xen_list_ref(xm_protected_element(i), CALLBACK_FUNC);
      if (Xen_is_procedure(code))
	return(Xen_integer_to_C_int(Xen_apply(code,
				      Xen_list_8(C_string_to_Xen_string((char *)(*in_out)),
						 Xen_wrap_C_pointer(text_end), /* can't work... */
						 C_int_to_Xen_integer(type),
						 C_string_to_Xen_string(locale_tag),
						 C_to_Xen_XmParseMapping(entry),
						 C_int_to_Xen_integer(pattern_length),
						 C_to_Xen_XmString((*str_include)), /* can't work... */
						 (Xen)call_data),
				      __func__)));
    }
  return(0);
}

static Xen xm_XmColorAllocationProc;

static void gxm_XmAllocColorProc(Display *dpy, Colormap color, XColor *bs)
{
  /* DIFF: XmAllocColorProc should return new XColor value
   */
  Xen val;
  val = Xen_call_with_3_args(xm_XmColorAllocationProc,
		   C_to_Xen_Display(dpy),
		   C_to_Xen_Colormap(color),
		   C_to_Xen_XColor(bs),
		   __func__);
  (*bs) = (*(Xen_to_C_XColor(val)));
}

static Xen xm_XmColorCalculationProc;

static void gxm_XmColorCalculationProc(Screen *scr, XColor *bg, XColor *fg, XColor *sel, XColor *ts, XColor *bs)
{
  /* DIFF: XmColorCalculationProc takes 2 args, returns list of 4 colors
   */
  Xen lst;
  int loc;
  lst = Xen_call_with_2_args(xm_XmColorCalculationProc,
		   C_to_Xen_Screen(scr),
		   C_to_Xen_XColor(bg),
		   __func__);
  loc = xm_protect(lst);
  if (Xen_is_list(lst))
    {
      (*fg) = (*(Xen_to_C_XColor(Xen_list_ref(lst, 0)))); 
      (*sel) = (*(Xen_to_C_XColor(Xen_list_ref(lst, 1))));
      (*ts) = (*(Xen_to_C_XColor(Xen_list_ref(lst, 2))));
      (*bs) = (*(Xen_to_C_XColor(Xen_list_ref(lst, 3))));
   }
  xm_unprotect_at(loc);
}

static Xen xm_XmColorProc; /* XmColorProc is not the same as XmScreen color calculation proc */

static void gxm_XmColorProc(XColor *bg, XColor *fg, XColor *sel, XColor *ts, XColor *bs)
{
  Xen lst;
  int loc;
  lst = Xen_call_with_1_arg(xm_XmColorProc,
		   C_to_Xen_XColor(bg),
		   __func__);
  loc = xm_protect(lst);
  if (Xen_is_list(lst))
    {
      (*fg) = (*(Xen_to_C_XColor(Xen_list_ref(lst, 0)))); 
      (*sel) = (*(Xen_to_C_XColor(Xen_list_ref(lst, 1))));
      (*ts) = (*(Xen_to_C_XColor(Xen_list_ref(lst, 2))));
      (*bs) = (*(Xen_to_C_XColor(Xen_list_ref(lst, 3))));
   }
  xm_unprotect_at(loc);
}

static XtCallbackList Xen_to_C_XtCallbackList(Xen call_list1)
{
  Xen call_list;
  int call_i, call_len;
  XtCallbackRec *cl = NULL;
  call_list = Xen_copy_arg(call_list1);
  call_len = Xen_list_length(call_list) / 2;
  if (call_len == 0) return(NULL);
  cl = (XtCallbackRec *)calloc(call_len + 1, sizeof(XtCallbackRec));
  for (call_i = 0; call_i < call_len; call_i++, call_list = Xen_cddr(call_list))
    {
      Xen func, data;
      func = Xen_car(call_list);
      if (Xen_list_length(call_list) == 2)
	data = Xen_cadr(call_list);
      else data = Xen_false;
      if ((Xen_is_procedure(func)) && (Xen_is_aritable(func, 3)))
	{
	  Xen descr;
	  cl[call_i].callback = gxm_XtCallbackProc;
	  descr = C_to_Xen_XM_XtCallback(Xen_car(call_list), data);
	  cl[call_i].closure = (XtPointer)descr;
	  Xen_list_set(descr, CALLBACK_GC_LOC, C_int_to_Xen_integer(xm_protect(descr)));
	}
    }
  return(cl);
}

static Xen C_to_Xen_STRING_WITH_TERMINATION(char *str, unsigned long len)
{
  if ((len == 0) || (str == NULL)) return(Xen_false);
  str[len] = '\0';
  return(C_string_to_Xen_string(str));
}

static Xen xm_XtSelectionCallback_Descr;

static void gxm_XtSelectionCallbackProc(Widget w, XtPointer x, Atom *a1, Atom *a2, XtPointer x1, unsigned long *l, int *i)
{
  Xen_apply(Xen_car(xm_XtSelectionCallback_Descr),
	    Xen_list_7(C_to_Xen_Widget(w),
		       Xen_cadr(xm_XtSelectionCallback_Descr),
		       C_to_Xen_Atom(*a1),
		       C_to_Xen_Atom(*a2),
		       C_to_Xen_STRING_WITH_TERMINATION((char *)x1, *l), /* should we handle Atom -> Lisp type conversions? */
		       C_ulong_to_Xen_ulong(*l),
		       C_int_to_Xen_integer(*i)),
	    __func__);
}

static Xen xm_XtConvertSelectionIncr_Descr;

static Boolean gxm_XtConvertSelectionIncrProc(Widget w, Atom *selection, Atom *target, 
					   Atom *type_return, XtPointer *value_return, 
					   unsigned long *length_return, int *format_return, 
					   unsigned long *max_length, XtPointer client_data, 
					   XtRequestId *request_id)
{
  /* DIFF: user callback here takes 6 args (w selection target max_length client_data request_id)
   *       should return (list ...) if ok, #f if not
   *       the list should be (type value length format)
   */
  Xen result;
  result = Xen_apply(xm_XtConvertSelectionIncr_Descr,
		     Xen_list_6(C_to_Xen_Widget(w),
				C_to_Xen_Atom(*selection),
				C_to_Xen_Atom(*target),
				C_int_to_Xen_integer(*max_length),
				Xen_wrap_C_pointer(client_data),
				C_ulong_to_Xen_ulong(*request_id)), /* XtRequestId is XtPointer */
		     __func__);
  if (Xen_is_false(result))
    return(0);
  (*type_return) = Xen_to_C_Atom(Xen_list_ref(result, 0));
  (*value_return) = (XtPointer)Xen_unwrap_C_pointer(Xen_list_ref(result, 1));
  (*length_return) = (unsigned long)Xen_integer_to_C_int(Xen_list_ref(result, 2));
  (*format_return) = Xen_integer_to_C_int(Xen_list_ref(result, 3));
  return(1);
}

static Arg *protect_args(Arg *args, int len)
{
  /* probably too clever... trying to put off the copy until after all error checking to reduce memory leakage */
  int i;
  for (i = 0; i < len; i++)
    if (args[i].name)
      args[i].name = xen_strdup(args[i].name); /* Xen_to_C_STRING will eventually gc, so protect against that until we're done with args */
  return(args);
}

static Arg *free_args(Arg *args, int len)
{
  int i;
  for (i = 0; i < len; i++)
    if (args[i].name)
      free(args[i].name); /* free what we xen_strdup'd above */
  free(args);
  return(NULL);
}

static Arg *Xen_to_C_Args(Xen inargl)
{
  /* an Arg array in xm is a list of name value pairs */
  Arg *args = NULL;
  int i, len, gcloc;
  Xen descr, inarg;
  /* if XtVaNestedList supported, scan for it here, and increase length as needed,
   *   then make recursive call to Xen_to_C_Args in that branch, unloading afterwards
   *   this is not actually needed in xm -- just use append!
   */
  inarg = Xen_copy_arg(inargl);
  len = Xen_list_length(inarg) / 2;
  if (len == 0) return(NULL);
  gcloc = xm_protect(inarg);
  args = (Arg *)calloc(len, sizeof(Arg));
  for (i = 0; i < len; i++, inarg = Xen_cddr(inarg))
    {
      XtCallbackRec *cl = NULL;
      xm_resource_t type;
      Xen xname, value;
      char *name;
      xname = Xen_car(inarg);
      Xen_check_type(Xen_is_string(xname), xname, 0, __func__, "string");
      name = (char *)Xen_string_to_C_string(xname);
      type = resource_type(name);
      value = Xen_cadr(inarg);
      switch (type)
	{
	  /* here the XtSetArg call wants an XtCallbackList, the incoming Xen type is a list of callback data pairs
	   *   the new callback rec.callback = gxm_XtCallback (etc -- chosen by resource type)
	   *   the rec.closure will be the wrapped func/data info
	   *   we don't have all the data we need for the actual wrapped list here since we need the widget to choose the callback struct type
	   *   so fixup_args should always follow if there's a possible callback list in the resource settings 
	   * also, which callback is invoked may depend on callback type 
	   */
	case XM_CALLBACK:
	  cl = Xen_to_C_XtCallbackList(value);
	  if (cl) XtSetArg(args[i], name, cl);
	  break;
	case XM_PARSE_CALLBACK:
	  if ((Xen_is_procedure(value)) && (Xen_is_aritable(value, 8)))
	    {
	      XtSetArg(args[i], name, (unsigned long)gxm_Parse_Callback);
	      descr = C_to_Xen_XM_Parse_Callback(value);
	      Xen_list_set(descr, CALLBACK_GC_LOC, C_int_to_Xen_integer(xm_protect(descr)));
	    }
	  else 
	    {
	      if (Xen_is_false(value))
		XtSetArg(args[i], name, 0);
	      else Xen_check_type(0, value, 0, name, "procedure of 8 args");
	    }
	  break;
	case XM_DROP_CALLBACK:
	  if ((Xen_is_procedure(value)) && (Xen_is_aritable(value, 3)))
	    {
	      XtSetArg(args[i], name, (unsigned long)gxm_Drop_Callback);
	      descr = C_to_Xen_XM_Drop_Callback(value);
	      Xen_list_set(descr, CALLBACK_GC_LOC, C_int_to_Xen_integer(xm_protect(descr)));
	    }
	  else 
	    {
	      if (Xen_is_false(value))
		XtSetArg(args[i], name, 0);
	      else Xen_check_type(0, value, 0, name, "procedure of 3 args");
	    }
	  break;
	case XM_DRAG_CALLBACK:
	  if ((Xen_is_procedure(value)) && (Xen_is_aritable(value, 3)))
	    {
	      XtSetArg(args[i], name, (unsigned long)gxm_Drag_Callback);
	      descr = C_to_Xen_XM_Drag_Callback(value);
	      Xen_list_set(descr, CALLBACK_GC_LOC, C_int_to_Xen_integer(xm_protect(descr)));
	    }
	  else 
	    {
	      if (Xen_is_false(value))
		XtSetArg(args[i], name, 0);
	      else Xen_check_type(0, value, 0, name, "procedure of 3 args");
	    }
	  break;
	case XM_SEARCH_CALLBACK:    /* XmNfileSearchProc and XmNdirSearchProc, XmSearchProc XmFileSelectionBox 756 */
	  if ((Xen_is_procedure(value)) && (Xen_is_aritable(value, 2)))
	    {
	      XtSetArg(args[i], name, (unsigned long)gxm_XmSearchProc);
	      descr = C_to_Xen_XM_XmSearchProc(value);
	      Xen_list_set(descr, CALLBACK_GC_LOC, C_int_to_Xen_integer(xm_protect(descr)));
	    }
	  else 
	    {
	      if (Xen_is_false(value))
		XtSetArg(args[i], name, 0);
	      else Xen_check_type(0, value, 0, name, "procedure of 2 args");
	    }
	  break;
	case XM_QUALIFY_CALLBACK:   /* XmNqualifySearchDataProc, XmQualifyProc */
	  if ((Xen_is_procedure(value)) && (Xen_is_aritable(value, 3)))
	    {
	      XtSetArg(args[i], name, (unsigned long)gxm_XmQualifyProc);
	      descr = C_to_Xen_XM_XmQualifyProc(value);
	      Xen_list_set(descr, CALLBACK_GC_LOC, C_int_to_Xen_integer(xm_protect(descr)));
	    }
	  else 
	    {
	      if (Xen_is_false(value))
		XtSetArg(args[i], name, 0);
	      else Xen_check_type(0, value, 0, name, "procedure of 3 args");
	    }
	  break;
	case XM_ORDER_CALLBACK:
	  if ((Xen_is_procedure(value)) && (Xen_is_aritable(value, 1)))
	    {
	      XtSetArg(args[i], name, (unsigned long)gxm_XtOrderProc);
	      descr = C_to_Xen_XM_XtOrderProc(value);
	      Xen_list_set(descr, CALLBACK_GC_LOC, C_int_to_Xen_integer(xm_protect(descr)));
	    }
	  else 
	    {
	      if (Xen_is_false(value))
		XtSetArg(args[i], name, 0);
	      else Xen_check_type(0, value, 0, name, "procedure of 1 arg");
	    }
	  break;
	case XM_TRANSFER_CALLBACK:  /* XmNtransferProc, XtSelectionCallbackProc, XmDropTransfer */
	  /* for now I'll assume no collisions here */
	  xm_XtSelectionCallback_Descr = Xen_list_2(value, Xen_false);
	  xm_protect(xm_XtSelectionCallback_Descr);
	  XtSetArg(args[i], name, (unsigned long)gxm_XtSelectionCallbackProc);
	  break;
	case XM_CONVERT_CALLBACK:   /* XmNconvertProc, XtConvertSelectionIncrProc, XmDragContext */
	  xm_XtConvertSelectionIncr_Descr = value;
	  xm_protect(xm_XtConvertSelectionIncr_Descr);
	  XtSetArg(args[i], name, (unsigned long)gxm_XtConvertSelectionIncrProc);
	  break;
	case XM_ALLOC_COLOR_CALLBACK:     /* XmNcolorAllocationProc, XmAllocColorProc XmScreen 921 */
	  if ((Xen_is_procedure(value)) && (Xen_is_aritable(value, 3)))
	    {
	      XtSetArg(args[i], name, (unsigned long)gxm_XmAllocColorProc);
	      if (Xen_is_procedure(xm_XmColorAllocationProc)) xm_unprotect(xm_XmColorAllocationProc);
	      xm_protect(value);
	      xm_XmColorAllocationProc = value;
	    }
	  else 
	    {
	      if (Xen_is_false(value))
		XtSetArg(args[i], name, 0);
	      else Xen_check_type(0, value, 0, name, "procedure of 3 args");
	    }
	  break;
	case XM_SCREEN_COLOR_CALLBACK:     /* XmNcolorCalculationProc, XmScreen 921 */
	  if ((Xen_is_procedure(value)) && (Xen_is_aritable(value, 2)))
	    {
	      XtSetArg(args[i], name, (unsigned long)gxm_XmColorCalculationProc);
	      if (Xen_is_procedure(xm_XmColorCalculationProc)) xm_unprotect(xm_XmColorCalculationProc);
	      xm_protect(value);
	      xm_XmColorCalculationProc = value;
	    }
	  else 
	    {
	      if (Xen_is_false(value))
		XtSetArg(args[i], name, 0);
	      else Xen_check_type(0, value, 0, name, "procedure of 2 args");
	    }
	  break;
	case XM_POPUP_CALLBACK:     /* XmNcreatePopupChildProc, XtCreatePopupChildProc */
	  if ((Xen_is_procedure(value)) && (Xen_is_aritable(value, 1)))
	    {
	      XtSetArg(args[i], name, (unsigned long)gxm_XtPopupChild);
	      descr = C_to_Xen_XM_XtPopupChild(value);
	      Xen_list_set(descr, CALLBACK_GC_LOC, C_int_to_Xen_integer(xm_protect(descr)));
	    }
	  else 
	    {
	      if (Xen_is_false(value))
		XtSetArg(args[i], name, 0);
	      else Xen_check_type(0, value, 0, name, "procedure of 1 arg");
	    }
	  break;

	  /* the rest are just doing type checks before the conversion to C */

	case XM_INT:
	  Xen_check_type(Xen_is_integer(value), value, 1, name, "an integer");
	  XtSetArg(args[i], name, (XtArgVal)(Xen_integer_to_C_int(value)));
	  break;
	case XM_FLOAT:
	  Xen_check_type(Xen_is_double(value), value, 1, name, "a float");
	  XtSetArg(args[i], name, (XtArgVal)(Xen_real_to_C_double(value)));
	  break;
	case XM_STRING:	      
	  Xen_check_type(Xen_is_string(value), value, 1, name, "a string");      
	  XtSetArg(args[i], name, (XtArgVal)(Xen_string_to_C_string(value)));
	  break;
	case XM_STRING_OR_INT:
	  Xen_check_type(Xen_is_string(value) || Xen_is_integer(value), value, 1, name, "an integer or a string");      
	  if (Xen_is_string(value))
	    XtSetArg(args[i], name, (XtArgVal)(Xen_string_to_C_string(value)));
	  else XtSetArg(args[i], name, (XtArgVal)(Xen_integer_to_C_int(value)));
	  break;
	case XM_XMSTRING:
	  Xen_check_type(Xen_is_XmString(value), value, 1, name, "an XmString");      	      
	  XtSetArg(args[i], name, (XtArgVal)(Xen_to_C_XmString(value)));
	  break;
	case XM_STRING_OR_XMSTRING:
	  Xen_check_type(Xen_is_XmString(value) || Xen_is_string(value), value, 1, name, "a string or an XmString");  
	  if (Xen_is_string(value))
	    XtSetArg(args[i], name, (XtArgVal)(Xen_string_to_C_string(value)));
	  else XtSetArg(args[i], name, (XtArgVal)(Xen_to_C_XmString(value)));
	  break;
	case XM_STRING_TABLE:
	  Xen_check_type(Xen_is_list(value), value, 1, name, "an XmStringTable");      	           
	  XtSetArg(args[i], name, (XtArgVal)(Xen_to_C_XmStringTable(value, Xen_list_length(value))));
	  break;
	case XM_TRANSFER_ENTRY_LIST:
	  Xen_check_type(Xen_is_list(value), value, 1, name, "a list");      	           
	  XtSetArg(args[i], name, (XtArgVal)(Xen_to_C_XmDropTransferEntryRecs(value, Xen_list_length(value))));
	  break;
	case XM_RENDER_TABLE: 
	  Xen_check_type(Xen_is_XmRenderTable(value), value, 1, name, "an XmRenderTable");
	  XtSetArg(args[i], name, (XtArgVal)(Xen_to_C_XmRenderTable(value)));
	  break;
	case XM_TAB_LIST:
	  Xen_check_type(Xen_is_XmTabList(value), value, 1, name, "an XmTabList"); 
	  XtSetArg(args[i], name, (XtArgVal)(Xen_to_C_XmTabList(value)));
	  break;
	case XM_WIDGET:
	  Xen_check_type(Xen_is_Widget(value), value, 1, name, "a Widget"); 
	  XtSetArg(args[i], name, (XtArgVal)(Xen_to_C_Widget(value)));
	  break;
	case XM_BOOLEAN:
	  Xen_check_type(Xen_is_boolean(value), value, 1, name, "a boolean"); 
	  XtSetArg(args[i], name, (XtArgVal)(Xen_boolean_to_C_bool(value)));
	  break;
	case XM_BOOLEAN_OR_INT:
	  Xen_check_type(Xen_is_boolean(value) || Xen_is_integer(value), value, 1, name, "a boolean or int"); 
	  if (Xen_is_boolean(value))
	    XtSetArg(args[i], name, (XtArgVal)(Xen_boolean_to_C_bool(value)));
	  else XtSetArg(args[i], name, (XtArgVal)(Xen_integer_to_C_int(value)));
	  break;
	case XM_PIXEL:
	  Xen_check_type(Xen_is_Pixel(value), value, 1, name, "a pixel"); 
	  XtSetArg(args[i], name, (XtArgVal)(Xen_to_C_Pixel(value)));
	  break;
	case XM_PIXMAP:
	  Xen_check_type(Xen_is_Pixmap(value), value, 1, name, "a pixmap"); 
	  XtSetArg(args[i], name, (XtArgVal)(Xen_to_C_Pixmap(value)));
	  break;
	case XM_DIMENSION:
	  Xen_check_type(Xen_is_integer(value), value, 1, name, "a Dimension (integer)");      
	  XtSetArg(args[i], name, (XtArgVal)(Xen_to_C_Dimension(value)));
	  break;
	case XM_POSITION:
	  Xen_check_type(Xen_is_integer(value), value, 1, name, "a Position (integer)");      
	  XtSetArg(args[i], name, (XtArgVal)(Xen_to_C_Position(value)));
	  break;
	case XM_SHORT:
	  Xen_check_type(Xen_is_integer(value), value, 1, name, "a short");      
	  XtSetArg(args[i], name, (XtArgVal)(Xen_integer_to_C_int(value)));
	  break;
	case XM_ATOM:
	  Xen_check_type(Xen_is_Atom(value), value, 1, name, "an Atom");      
	  XtSetArg(args[i], name, (XtArgVal)(Xen_to_C_Atom(value)));
	  break;
	case XM_TEXT_SOURCE:
	  Xen_check_type(Xen_is_XmTextSource(value), value, 1, name, "an XmTextSource");      
	  XtSetArg(args[i], name, (XtArgVal)(Xen_to_C_XmTextSource(value)));
	  break;
	case XM_COLORMAP:
	  Xen_check_type(Xen_is_Colormap(value), value, 1, name, "a Colormap");      
	  XtSetArg(args[i], name, (XtArgVal)(Xen_to_C_Colormap(value)));
	  break;
	case XM_KEYSYM:
	  Xen_check_type(Xen_is_KeySym(value), value, 1, name, "a KeySym");      
	  XtSetArg(args[i], name, (XtArgVal)(Xen_to_C_KeySym(value)));
	  break;
	case XM_SCREEN:
	  Xen_check_type(Xen_is_Screen(value), value, 1, name, "a Screen");      
	  XtSetArg(args[i], name, (XtArgVal)(Xen_to_C_Screen(value)));
	  break;
	case XM_WINDOW:
	  Xen_check_type(Xen_is_Window(value), value, 1, name, "a Window");      
	  XtSetArg(args[i], name, (XtArgVal)(Xen_to_C_Window(value)));
	  break;
	case XM_VISUAL:
	  Xen_check_type(Xen_is_Visual(value), value, 1, name, "a Visual");      
	  XtSetArg(args[i], name, (XtArgVal)(Xen_to_C_Visual(value)));
	  break;
	case XM_WIDGET_CLASS:
	  Xen_check_type(Xen_is_WidgetClass(value), value, 1, name, "a WidgetClass");      
	  XtSetArg(args[i], name, (XtArgVal)(Xen_to_C_WidgetClass(value)));
	  break;

	case XM_ATOM_LIST:
	  Xen_check_type(Xen_is_list(value), value, 1, name, "a list of Atoms");
	  XtSetArg(args[i], name, (XtArgVal)(Xen_to_C_Atoms(value, Xen_list_length(value))));
	  break;
	case XM_INT_TABLE:
	  Xen_check_type(Xen_is_list(value), value, 1, name, "a list of ints");
	  XtSetArg(args[i], name, (XtArgVal)(Xen_to_C_Ints(value, Xen_list_length(value))));
	  break;
	case XM_WIDGET_LIST:
	  Xen_check_type(Xen_is_list(value), value, 1, name, "a list of Widgets");
	  XtSetArg(args[i], name, (XtArgVal)(Xen_to_C_Widgets(value, Xen_list_length(value))));
	  break;
	case XM_KEYSYM_TABLE:
	  Xen_check_type(Xen_is_list(value), value, 1, name, "a list of KeySyms");
	  XtSetArg(args[i], name, (XtArgVal)(Xen_to_C_KeySyms(value, Xen_list_length(value))));
	  break;
	case XM_STRING_LIST:
	case XM_CHARSET_TABLE:
	  Xen_check_type(Xen_is_list(value), value, 1, name, "a list of char *");
	  XtSetArg(args[i], name, (XtArgVal)(Xen_to_C_Strings(value, Xen_list_length(value))));
	  break;
	case XM_RECTANGLE_LIST:
	  Xen_check_type(Xen_is_list(value), value, 1, name, "a list of XRectangles");
	  XtSetArg(args[i], name, (XtArgVal)(Xen_to_C_XRectangles(value, Xen_list_length(value))));
	  break;
	case XM_CURSOR:
	  Xen_check_type(Xen_is_Cursor(value), value, 1, name, "a cursor"); 
	  XtSetArg(args[i], name, (XtArgVal)(Xen_to_C_Cursor(value)));
	  break;

	default:
	  if (Xen_is_ulong(value))
	    XtSetArg(args[i], name, (XtArgVal)(Xen_ulong_to_C_ulong(value)));
	  else if (Xen_is_integer(value))
	    XtSetArg(args[i], name, (XtArgVal)(Xen_integer_to_C_int(value)));
	  else if (Xen_is_boolean(value))
	    XtSetArg(args[i], name, (XtArgVal)(Xen_boolean_to_C_bool(value)));
	  else if (Xen_is_string(value))
	    XtSetArg(args[i], name, (XtArgVal)(Xen_string_to_C_string(value)));
	  /* these are bare pointers -- we can't assume they can be "unwrapped" in xen jargon */
	  else if (Xen_is_list(value))
	    XtSetArg(args[i], name, (XtArgVal)(Xen_llong_to_C_llong(Xen_cadr(value))));  /* all tagged types */
	  else 
	    XtSetArg(args[i], name, (XtArgVal)(Xen_llong_to_C_llong(value)));
	  break;
	}
    }
  xm_unprotect_at(gcloc);
  return(protect_args(args, len));
}

static void fixup_args(Widget w, Arg *args, int len)
{
  /* search for and fill-in incompletely specified callback info now that we have the widget type */
  int i;
  for (i = 0; i < len; i++)
    {
      char *name;
      name = args[i].name;
      if (name)
	{
	  XtCallbackRec *cl = NULL;
	  int j;
	  Xen data;
	  switch (resource_type(name))
	    {
	    case XM_STRING_TABLE:
	      {
		XmStringTable *st;
		st = (XmStringTable *)(args[i].value);
		free(st);
	      }
	      break;
	    case XM_WIDGET_LIST:
	      {
		Widget *w;
		w =(Widget *)(args[i].value); 
		free(w);
	      }
	      break;
	    case XM_CALLBACK:
	      cl = (XtCallbackRec *)(args[i].value);
	      for (j = 0 ;; j++)
		{
		  if (cl[j].callback == NULL) break;
		  data = (Xen)(cl[j].closure);
		  Xen_list_set(data, CALLBACK_STRUCT_TYPE, C_int_to_Xen_integer(callback_struct_type(w, name)));
		}
	      free(cl);
	      break;

	    case XM_DROP_CALLBACK:
	      j = map_over_protected_elements(find_dropproc, (unsigned long)w);
	      if (j >= 0)
		{
		  data = xm_protected_element(j);
		  xm_unprotect_at(Xen_integer_to_C_int(Xen_list_ref(data, CALLBACK_GC_LOC)));
		}
	      j = map_over_protected_elements(find_dropproc, (unsigned long)Xen_false);
	      if (j >= 0)
		{
		  data = xm_protected_element(j);
		  Xen_list_set(data, CALLBACK_DATA, C_to_Xen_Widget(w));
		}
	      else fprintf(stderr,"can't fixup drop proc!");
	      break;
	    case XM_DRAG_CALLBACK:
	      j = map_over_protected_elements(find_dragproc, (unsigned long)w);
	      if (j >= 0)
		{
		  data = xm_protected_element(j);
		  xm_unprotect_at(Xen_integer_to_C_int(Xen_list_ref(data, CALLBACK_GC_LOC)));
		}
	      j = map_over_protected_elements(find_dragproc, (unsigned long)Xen_false);
	      if (j >= 0)
		{
		  data = xm_protected_element(j);
		  Xen_list_set(data, CALLBACK_DATA, C_to_Xen_Widget(w));
		}
	      else fprintf(stderr,"can't fixup drag proc!");
	      break;
	    case XM_QUALIFY_CALLBACK:
	      j = map_over_protected_elements(find_qualifyproc, (unsigned long)w); /* see if one already exists */
	      if (j >= 0)
		{
		  data = xm_protected_element(j);
		  xm_unprotect_at(Xen_integer_to_C_int(Xen_list_ref(data, CALLBACK_GC_LOC)));
		}
	      j = map_over_protected_elements(find_qualifyproc, (unsigned long)Xen_false); /* i.e. find the unset one */
	      if (j >= 0)
		{
		  data = xm_protected_element(j);
		  Xen_list_set(data, CALLBACK_DATA, C_to_Xen_Widget(w));
		}
	      break;
	    case XM_SEARCH_CALLBACK:
	      j = map_over_protected_elements(find_searchproc, (unsigned long)w);
	      if (j >= 0)
		{
		  data = xm_protected_element(j);
		  xm_unprotect_at(Xen_integer_to_C_int(Xen_list_ref(data, CALLBACK_GC_LOC)));
		}
	      j = map_over_protected_elements(find_searchproc, (unsigned long)Xen_false);
	      if (j >= 0)
		{
		  data = xm_protected_element(j);
		  Xen_list_set(data, CALLBACK_DATA, C_to_Xen_Widget(w));
		}
	      break;
	    case XM_ORDER_CALLBACK:
	      j = map_over_protected_elements(find_orderproc, (unsigned long)w);
	      if (j >= 0)
		{
		  data = xm_protected_element(j);
		  xm_unprotect_at(Xen_integer_to_C_int(Xen_list_ref(data, CALLBACK_GC_LOC)));
		}
	      j = map_over_protected_elements(find_orderproc, (unsigned long)Xen_false);
	      if (j >= 0)
		{
		  data = xm_protected_element(j);
		  Xen_list_set(data, CALLBACK_DATA, C_to_Xen_Widget(w));
		}
	      break;
	    case XM_POPUP_CALLBACK:
	      j = map_over_protected_elements(find_popupchild, (unsigned long)w);
	      if (j >= 0)
		{
		  data = xm_protected_element(j);
		  xm_unprotect_at(Xen_integer_to_C_int(Xen_list_ref(data, CALLBACK_GC_LOC)));
		}
	      j = map_over_protected_elements(find_popupchild, (unsigned long)Xen_false);
	      if (j >= 0)
		{
		  data = xm_protected_element(j);
		  Xen_list_set(data, CALLBACK_DATA, C_to_Xen_Widget(w));
		}
	      break;
	    default:
	      break;
	    }
	}
    }
}

static int xmstringtable_length(Widget w, const char *name)
{
  /* it's not safe here to search for 0 as end of table, so where possible we'll match resources with counts */
  int len = 0;
  if (strcmp(name, XmNitems) == 0) XtVaGetValues(w, XmNitemCount, &len, NULL);
  else if (strcmp(name, XmNlistItems) == 0) XtVaGetValues(w, XmNlistItemCount, &len, NULL);
  else if (strcmp(name, XmNselectedItems) == 0) XtVaGetValues(w, XmNselectedItemCount, &len, NULL);
  else if (strcmp(name, XmNhistoryItems) == 0) XtVaGetValues(w, XmNhistoryItemCount, &len, NULL);
  else if (strcmp(name, XmNfileListItems) == 0) XtVaGetValues(w, XmNfileListItemCount, &len, NULL);
  else if (strcmp(name, XmNdirListItems) == 0) XtVaGetValues(w, XmNdirListItemCount, &len, NULL);
  else if (strcmp(name, XmNdetail) == 0) XtVaGetValues(w, XmNdetailCount, &len, NULL);
  else if (strcmp(name, XmNvalues) == 0) XtVaGetValues(w, XmNnumValues, &len, NULL);
  else XtVaGetValues(w, XmNbuttonCount, &len, NULL);
  return(len);
}

static Xen C_to_Xen_ANY(Widget w, Arg arg)
{
  /* XtGetValues -- a list of pairs: resource-name place-holder where we fill in the 2nd element */
  /* this has to wait until the last moment to decide what arg.value is -- if we assume unsigned long*
   *   (which works on the Pentium/Linux), the Sun gets confused by Dimensions (unsigned short)
   */
  Arg a[1];
  int j, ilen = 0;
  switch (resource_type(arg.name))
    {
    case XM_INT:	      return(C_int_to_Xen_integer((*((int *)(arg.value)))));
    case XM_ULONG:	      return(C_ulong_to_Xen_ulong((*((unsigned long *)(arg.value)))));
    case XM_XTPOINTER:	      return(Xen_wrap_C_pointer((XtPointer)(arg.value)));
    case XM_UCHAR:	      return(C_int_to_Xen_integer((*((unsigned char *)(arg.value)))));
    case XM_FLOAT:	      return(C_double_to_Xen_real((*((float *)(arg.value))))); /* the resource values are floats */
    case XM_STRING:	      return(C_string_to_Xen_string((char *)(*((char **)(arg.value)))));
    case XM_STRING_OR_XMSTRING: /* fileselectionbox here , not parsemapping */
    case XM_XMSTRING:	      return(C_to_Xen_XmString((XmString)(*((XmString *)(arg.value)))));
    case XM_STRING_TABLE:     return(C_to_Xen_XmStringTable((XmStringTable)(*((XmStringTable *)(arg.value))), xmstringtable_length(w, arg.name)));
    case XM_RENDER_TABLE:     return(C_to_Xen_XmRenderTable((XmRenderTable)(*((XmRenderTable *)(arg.value)))));
    case XM_TAB_LIST:	      return(C_to_Xen_XmTabList((XmTabList)(*((XmTabList *)(arg.value)))));
    case XM_WIDGET:	      return(C_to_Xen_Widget((Widget)(*((Widget *)(arg.value)))));
    case XM_WIDGET_LIST:      /* (XtGetValues c1 (list XmNchildren 0) 1) */
      if (strcmp(arg.name, XmNchildren) == 0)
	XtSetArg(a[0], XmNnumChildren, &ilen);             /* Composite */
      else XtSetArg(a[0], XmNselectedObjectCount, &ilen);  /* Container */
      XtGetValues(w, a, 1);
      return(C_to_Xen_Widgets((Widget *)(*((Widget **)(arg.value))), ilen));
      break;
    case XM_BOOLEAN:	      return(C_bool_to_Xen_boolean((*((Boolean *)(arg.value)))));
    case XM_BOOLEAN_OR_INT:   return(C_int_to_Xen_integer((*((int *)(arg.value)))));
    case XM_SEARCH_CALLBACK:
      j = map_over_protected_elements(find_searchproc, (unsigned long)w);
      if (j >= 0) return(Xen_list_ref(xm_protected_element(j), CALLBACK_FUNC));
      break;
    case XM_DROP_CALLBACK:
      j = map_over_protected_elements(find_dropproc, (unsigned long)w);
      if (j >= 0) return(Xen_list_ref(xm_protected_element(j), CALLBACK_FUNC));
      break;
    case XM_DRAG_CALLBACK:
      j = map_over_protected_elements(find_dragproc, (unsigned long)w);
      if (j >= 0) return(Xen_list_ref(xm_protected_element(j), CALLBACK_FUNC));
      break;
    case XM_ORDER_CALLBACK:
      j = map_over_protected_elements(find_orderproc, (unsigned long)w);
      if (j >= 0) return(Xen_list_ref(xm_protected_element(j), CALLBACK_FUNC));
      break;
    case XM_QUALIFY_CALLBACK:
      j = map_over_protected_elements(find_qualifyproc, (unsigned long)w);
      if (j >= 0) return(Xen_list_ref(xm_protected_element(j), CALLBACK_FUNC));
      break;
    case XM_ALLOC_COLOR_CALLBACK:
      return(xm_XmColorAllocationProc);
      break;
    case XM_SCREEN_COLOR_CALLBACK:
      return(xm_XmColorCalculationProc);
      break;
    case XM_POPUP_CALLBACK:
      j = map_over_protected_elements(find_popupchild, (unsigned long)w);
      if (j >= 0) return(Xen_list_ref(xm_protected_element(j), CALLBACK_FUNC));
      break;
    case XM_TRANSFER_CALLBACK:
      return(xm_XtSelectionCallback_Descr);
    case XM_CONVERT_CALLBACK:
      return(xm_XtConvertSelectionIncr_Descr);
    case XM_CALLBACK:	   
      /* this can't work because we don't know what the desired callback was, so we can't search the gc table */
      /*   and if the callback was a C procedure, what good would its address be in this context? */
      /*   (XtGetValues_1 would need to pass the entire Arg or something) */
      return(C_ulong_to_Xen_ulong((*((unsigned long *)(arg.value)))));
      break;

    case XM_PIXEL:	      return(C_to_Xen_Pixel((*((Pixel *)(arg.value)))));
    case XM_PIXMAP:	      return(C_to_Xen_Pixmap((*((Pixmap *)(arg.value)))));
    case XM_XFONTSTRUCT:      return(C_to_Xen_XFontStruct((XFontStruct *)(*((XFontStruct **)(arg.value)))));
    case XM_DIMENSION:	      return(C_to_Xen_Dimension((*((Dimension *)(arg.value)))));
    case XM_POSITION:	      return(C_to_Xen_Position((*((Position *)(arg.value)))));
    case XM_SHORT:	      return(C_int_to_Xen_integer((*((short *)(arg.value)))));
    case XM_ATOM:	      return(C_to_Xen_Atom((*((Atom *)(arg.value)))));
    case XM_TEXT_SOURCE:      return(C_to_Xen_XmTextSource((XmTextSource)(*((XmTextSource *)(arg.value)))));
    case XM_ATOM_LIST:	      
      if (strcmp(arg.name, XmNexportTargets) == 0)            /* DragContext */
	XtSetArg(a[0], XmNnumExportTargets, &ilen);        /* DropSite */
      else XtSetArg(a[0], XmNnumImportTargets, &ilen);
      XtGetValues(w, a, 1);
      if ((ilen > 0) && (ilen < 100))
	return(C_to_Xen_Atoms((Atom *)(*((Atom **)(arg.value))), ilen));
      else return(Xen_false);
      break;
    case XM_STRING_LIST:                                  /* ApplicationShell */
      XtSetArg(a[0], XmNargc, &ilen);
      XtGetValues(w, a, 1);
      return(C_to_Xen_Strings((char **)(*((char ***)(arg.value))), ilen));
      break;
    case XM_CHARSET_TABLE:    
      XtSetArg(a[0], XmNbuttonCount, &ilen); /* may not be long enough... */
      XtGetValues(w, a, 1);
      return(C_to_Xen_Strings((char **)(*((char ***)(arg.value))), ilen));
      break;
    case XM_COLORMAP:	      return(C_to_Xen_Colormap((*((Colormap *)(arg.value)))));
    case XM_KEYSYM:	      return(C_to_Xen_KeySym((*((KeySym *)(arg.value)))));
    case XM_KEYSYM_TABLE:     
      XtSetArg(a[0], XmNbuttonCount, &ilen);
      XtGetValues(w, a, 1);
      return(C_to_Xen_KeySyms((KeySym *)(*((KeySym **)(arg.value))), ilen));
      break;
    case XM_SCREEN:	      return(C_to_Xen_Screen((Screen *)(*((Screen **)(arg.value)))));
    case XM_WINDOW:	      return(C_to_Xen_Window((Window)(*((Window *)(arg.value)))));
    case XM_VISUAL:	      return(C_to_Xen_Visual((Visual *)(*((Visual **)(arg.value)))));
    case XM_INT_TABLE:	      
      if (strcmp(arg.name, XmNdetailOrder) == 0) 
	XtSetArg(a[0], XmNdetailOrderCount, &ilen); 
      else
	{
	  if (strcmp(arg.name, XmNselectionArray) == 0) 
	    XtSetArg(a[0], XmNselectionArrayCount, &ilen); 
	  else XtSetArg(a[0], XmNselectedPositionCount, &ilen);
	}
      XtGetValues(w, a, 1);
      return(C_to_Xen_Ints((int *)(*((int **)(arg.value))), ilen));
      break;
    case XM_RECTANGLE_LIST:                              /* XmNrectangles exists but is not documented */
      XtSetArg(a[0], XmNnumDropRectangles, &ilen);
      XtGetValues(w, a, 1);
      return(C_to_Xen_XRectangles((XRectangle *)(*((XRectangle **)(arg.value))), ilen));
      break;
    case XM_WIDGET_CLASS:     return(C_to_Xen_WidgetClass((WidgetClass)(*((WidgetClass *)(arg.value)))));
    case XM_STRING_OR_INT:
      if ((w) &&
	  ((XmIsText(w)) || (XmIsTextField(w))))
	return(C_string_to_Xen_string((char *)(*((char **)(arg.value)))));
      else return(C_int_to_Xen_integer((int)(*((int *)(arg.value)))));
      break;
    default:
      break;
    }
  return(C_ulong_to_Xen_ulong((*((unsigned long *)(arg.value))))); /* fallback */
}

static Xen C_to_Xen_Args(Widget w, Arg *args, int len)
{
  /* here XtGetValues (or equivalent) has filled in the resource values and we need to pass them back
   *   to scheme properly wrapped
   */
  int i, loc;
  Xen lst = Xen_empty_list;
  loc = xm_protect(lst);
  for (i = len - 1; i >= 0; i--) 
    lst = Xen_cons_2(C_string_to_Xen_string(args[i].name),
		     C_to_Xen_ANY(w, args[i]),
		     lst);
  xm_unprotect_at(loc);
  return(lst);
}

/* (XtGetValues c1 (list XmNx 0) 1) */

static Xen gxm_XtGetValues_1(Xen arg1, Xen larg2, int len)
{
  Arg *args;
  unsigned long *locs;
  Widget w;
  Xen val, arg2;
  int i, gcloc;
  /* here we need to make sure the ref args are ok from C's point of view */
  if (len <= 0) return(Xen_false);
  w = Xen_to_C_Widget(arg1);
  arg2 = Xen_copy_arg(larg2);
  gcloc = xm_protect(arg2);
  args = (Arg *)calloc(len, sizeof(Arg));
  locs = (unsigned long *)calloc(len, sizeof(unsigned long));
  for (i = 0; i < len; i++, arg2 = Xen_cddr(arg2))
    {
      char *name;
      name = xen_strdup(Xen_string_to_C_string(Xen_car(arg2)));
      XtSetArg(args[i], name, &(locs[i]));
    }
  XtGetValues(w, args, len);
  val = C_to_Xen_Args(w, args, len);
  free_args(args, len);
  free(locs);
  xm_unprotect_at(gcloc);
  return(val);
}



/* -------------------------------- gc protection -------------------------------- */

static Xen xm_protected;
static int xm_protected_size = 0;
static Xen xm_gc_table;
static int last_xm_unprotect = NOT_A_GC_LOC;

static int xm_protect(Xen obj)
{
  int i, new_size;
  Xen new_table;
  if (last_xm_unprotect >= 0)
    {
      i = last_xm_unprotect;
      if (Xen_is_false(Xen_vector_ref(xm_protected, i)))
	{
	  Xen_vector_set(xm_protected, i, obj);
	  last_xm_unprotect = NOT_A_GC_LOC;
	  return(i);
	}
      last_xm_unprotect = NOT_A_GC_LOC;
    }
  for (i = 0; i < xm_protected_size; i++)
    if (Xen_is_false(Xen_vector_ref(xm_protected, i)))
      {
	Xen_vector_set(xm_protected, i, obj);
	return(i);
      }
  new_size = xm_protected_size * 2;
  new_table = Xen_make_vector(new_size, Xen_false);
  for (i = 0; i < xm_protected_size; i++)
    {
      Xen_vector_set(new_table, i, Xen_vector_ref(xm_protected, i));
      Xen_vector_set(xm_protected, i, Xen_false);
    }
  Xen_vector_set(new_table, xm_protected_size, obj);
  Xen_vector_set(xm_gc_table, 0, new_table);
  i = xm_protected_size;
  xm_protected_size = new_size;
  xm_protected = new_table;
  return(i);
}

static void xm_unprotect(Xen obj)
{
  int i;
  for (i = 0; i < xm_protected_size; i++)
    if (Xen_is_eq(Xen_vector_ref(xm_protected, i), obj))
      {
	Xen_vector_set(xm_protected, i, Xen_false);
	last_xm_unprotect = i;
	return;
      }
}

static void xm_unprotect_at(int ind)
{
  Xen_vector_set(xm_protected, ind, Xen_false);
  last_xm_unprotect = ind;
}

static int map_over_protected_elements(bool (*func)(Xen val, int loc, unsigned long fid), unsigned long id)
{
  int i;
  for (i = 0; i < xm_protected_size; i++)
    if (func(Xen_vector_ref(xm_protected, i), i, id))
      return(i);
  return(-1);
}

static Xen xm_protected_element(int loc)
{
  return(Xen_vector_ref(xm_protected, loc));
}


/* ----------------------------------------------------------------------------------------------------
 *
 *                            *         *  *         *
 *                             *       *   **       **
 *                              *     *    * *     * *
 *                               *   *     *  *   *  *
 *                                * *      *   * *   *
 *                                 *       *    *    *
 *                                * *      *         *
 *                               *   *     *         *
 *                              *     *    *         *
 *                             *       *   *         *
 *                            *         *  *         *
 * ----------------------------------------------------------------------------------------------------
 */

/* Motif */

/* weird order of procedures here and throughout caused by the C-header scripts --
 *   basically I forgot to reverse the main lists before pouring out the code,
 *   so everything is slightly backwards
 */

static int Xen_to_C_INT_DEF(Xen len, Xen lst) 
{
  if (Xen_is_integer(len))
    return(Xen_integer_to_C_int(len));
  else
    {
      int list_len;
      list_len = Xen_list_length(lst);
      return((int)(list_len / 2));
    }
}

static Xen gxm_XmTransferDone(Xen arg1, Xen arg2)
{
  #define H_XmTransferDone "void XmTransferDone(XtPointer transfer_id, XmTransferStatus status) completes a data transfer"
  Xen_check_type(Xen_is_wrapped_c_pointer(arg1), arg1, 1, "XmTransferDone", "XtPointer");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XmTransferDone", "XmTransferStatus");
  XmTransferDone((XtPointer)Xen_unwrap_C_pointer(arg1), (XmTransferStatus)Xen_integer_to_C_int(arg2));
  return(arg1);
}

static Xen gxm_XmTransferSendRequest(Xen arg1, Xen arg2)
{
  #define H_XmTransferSendRequest "void XmTransferSendRequest(XtPointer transfer_id, Time time) transfers a MULTIPLE request"
  Xen_check_type(Xen_is_wrapped_c_pointer(arg1), arg1, 1, "XmTransferSendRequest", "XtPointer");
  Xen_check_type(Xen_is_Time(arg2), arg2, 2, "XmTransferSendRequest", "Time");
  XmTransferSendRequest((XtPointer)Xen_unwrap_C_pointer(arg1), Xen_to_C_Time(arg2));
  return(arg1);
}

static Xen gxm_XmTransferStartRequest(Xen arg1)
{
  #define H_XmTransferStartRequest "void XmTransferStartRequest(XtPointer transfer_id) begins a MULTIPLE transfer"
  Xen_check_type(Xen_is_wrapped_c_pointer(arg1), arg1, 1, "XmTransferStartRequest", "XtPointer");
  XmTransferStartRequest((XtPointer)Xen_unwrap_C_pointer(arg1));
  return(arg1);
}

static Xen gxm_XmTransferSetParameters(Xen arg1, Xen arg2, Xen arg3, Xen arg4, Xen arg5)
{
  #define H_XmTransferSetParameters "void XmTransferSetParameters(XtPointer transfer_id, XtPointer parm, int parm_fmt, \
unsigned long parm_length, Atom parm_type)  establishes parameters to be passed by the next call to XmTransferValue"
  Xen_check_type(Xen_is_wrapped_c_pointer(arg1), arg1, 1, "XmTransferSetParameters", "XtPointer");
  Xen_check_type(Xen_is_wrapped_c_pointer(arg2), arg2, 2, "XmTransferSetParameters", "XtPointer");
  Xen_check_type(Xen_is_integer(arg3), arg3, 3, "XmTransferSetParameters", "an integer");
  Xen_check_type(Xen_is_ulong(arg4), arg4, 4, "XmTransferSetParameters", "unsigned long");
  Xen_check_type(Xen_is_Atom(arg5), arg5, 5, "XmTransferSetParameters", "an Atom");
  XmTransferSetParameters((XtPointer)Xen_unwrap_C_pointer(arg1), 
			  (XtPointer)Xen_unwrap_C_pointer(arg2), 
			  Xen_integer_to_C_int(arg3),
			  Xen_ulong_to_C_ulong(arg4), 
			  Xen_to_C_Atom(arg5));
  return(arg1);
}

static void gxm_TransferValueProc(Widget w, XtPointer context, XtPointer info)
{
  Xen descr = (Xen)context;
  Xen_call_with_3_args(Xen_list_ref(descr, 0),
	     C_to_Xen_Widget(w),
	     Xen_list_ref(descr, 1),
	     C_to_Xen_XmSelectionCallbackStruct((XmSelectionCallbackStruct *)info),
	     __func__);
}

static Xen gxm_XmTransferValue(Xen arg1, Xen arg2, Xen arg3, Xen arg4, Xen arg5)
{
  #define H_XmTransferValue "void XmTransferValue(XtPointer transfer_id, Atom target, XtCallbackProc proc, XtPointer client_data, Time time) \
transfers data to a destination"
  Xen_check_type(Xen_is_wrapped_c_pointer(arg1), arg1, 1, "XmTransferValue", "XtPointer");
  Xen_check_type(Xen_is_Atom(arg2), arg2, 2, "XmTransferValue", "an Atom");
  Xen_check_type(Xen_is_procedure(arg3) && (Xen_is_aritable(arg3, 3)), arg3, 3, "XmTransferValue", "XtCallbackProc (3 args)");
  Xen_check_type(Xen_is_Time(arg5), arg5, 5, "XmTransferValue", "Time");
  XmTransferValue((XtPointer)Xen_unwrap_C_pointer(arg1), 
		  Xen_to_C_Atom(arg2), 
		  gxm_TransferValueProc,
		  (XtPointer)Xen_list_2(arg3, arg4),
		  Xen_to_C_Time(arg5));
  return(arg1);
}

static Xen gxm_new_widget(const char *caller, Widget (*func)(Widget parent, char *name, ArgList al, Cardinal ac), Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  Widget w;
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, caller, "Widget");
  Xen_check_type(Xen_is_string(arg2), arg2, 2, caller, "char*");
  Xen_check_type(Xen_is_list(arg3), arg3, 3, caller, "ArgList");
  Xen_check_type(Xen_is_integer_or_unbound(arg4), arg4, 4, caller, "int");
  {
    Arg *args;
    int arglen;
    args = Xen_to_C_Args(arg3);
    w = (*func)(Xen_to_C_Widget(arg1), 
		(char *)Xen_string_to_C_string(arg2), 
		args, 
		arglen = Xen_to_C_INT_DEF(arg4, arg3));
    if (args)
      {
	fixup_args(w, args, arglen);
	free_args(args, arglen);
      }
  }
  return(C_to_Xen_Widget(w));
}

static Xen gxm_XmCreateMenuShell(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XmCreateMenuShell "Widget XmCreateMenuShell(Widget parent, String name, ArgList arglist, Cardinal argcount) \
The MenuShell widget creation function"
  return(gxm_new_widget("XmCreateMenuShell", XmCreateMenuShell, arg1, arg2, arg3, arg4));
}

static Xen gxm_XmProcessTraversal(Xen arg1, Xen arg2)
{
  #define H_XmProcessTraversal "Boolean XmProcessTraversal(Widget widget, XmTraversalDirection direction) determines which \
component receives keyboard events when a widget has the focus"
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XmProcessTraversal", "Widget");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XmProcessTraversal", "XmTraversalDirection");
  return(C_bool_to_Xen_boolean(XmProcessTraversal(Xen_to_C_Widget(arg1), (XmTraversalDirection)Xen_integer_to_C_int(arg2))));
}

static Xen gxm_XmGetFocusWidget(Xen arg1)
{
  #define H_XmGetFocusWidget "Widget XmGetFocusWidget(Widget widget): returns the ID of the widget that has keyboard focus"
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XmGetFocusWidget", "Widget");
  return(C_to_Xen_Widget(XmGetFocusWidget(Xen_to_C_Widget(arg1))));
}

static Xen gxm_XmGetTabGroup(Xen arg1)
{
  #define H_XmGetTabGroup "Widget XmGetTabGroup(Widget widget): returns the widget ID of a tab group"
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XmGetTabGroup", "Widget");
  return(C_to_Xen_Widget(XmGetTabGroup(Xen_to_C_Widget(arg1))));
}

static Xen gxm_XmGetVisibility(Xen arg1)
{
  #define H_XmGetVisibility "XmVisibility XmGetVisibility(Widget widget) determines if a widget is visible"
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XmGetVisibility", "Widget");
  return(C_int_to_Xen_integer(XmGetVisibility(Xen_to_C_Widget(arg1))));
}

static Xen gxm_XmIsTraversable(Xen arg1)
{
  #define H_XmIsTraversable "Boolean XmIsTraversable(Widget widget) identifies whether a widget can be traversed"
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XmIsTraversable", "Widget");
  return(C_bool_to_Xen_boolean(XmIsTraversable(Xen_to_C_Widget(arg1))));
}

static Xen gxm_XmGetDestination(Xen arg1)
{
  #define H_XmGetDestination "Widget XmGetDestination(Display *display) gets the current destination widget for paste etc"
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XmGetDestination", "Display*");
  return(C_to_Xen_Widget(XmGetDestination(Xen_to_C_Display(arg1))));
}

static Xen gxm_XmRenderTableAddRenditions(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XmRenderTableAddRenditions "XmRenderTable XmRenderTableAddRenditions(XmRenderTable oldtable, \
XmRendition *renditions, Cardinal rendition_count, XmMergeMode merge_mode) adds renditions to a render table"
  /* DIFF: XmRenderTableAddRenditions arg2 is list of Renditions, arg1 can be #f = NULL
   */
  int len, listlen = 0;
  XmRendition *rs;
  XmRenderTable res;
  Xen_check_type(Xen_is_XmRenderTable(arg1) || Xen_is_false(arg1), arg1, 1, "XmRenderTableAddRenditions", "XmRenderTable");
  Xen_check_type(Xen_is_list(arg2), arg2, 2, "XmRenderTableAddRenditions", "list of XmRendition");
  Xen_check_type(Xen_is_integer(arg3), arg3, 3, "XmRenderTableAddRenditions", "int");
  Xen_check_type(Xen_is_integer(arg4), arg4, 4, "XmRenderTableAddRenditions", "XmMergeMode");
  listlen = Xen_list_length(arg2);
  len = Xen_integer_to_C_int(arg3);
  if (len > listlen) len = listlen;
  if (len <= 0) return(Xen_false);
  rs = Xen_to_C_XmRenditions(arg2, len);
  res = XmRenderTableAddRenditions(Xen_is_false(arg1) ? NULL : Xen_to_C_XmRenderTable(arg1), 
				   rs, (Cardinal)len, 
				   (XmMergeMode)Xen_integer_to_C_int(arg4));
  free(rs);
  return(C_to_Xen_XmRenderTable(res));
}

static Xen gxm_XmRenderTableRemoveRenditions(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XmRenderTableRemoveRenditions "XmRenderTable XmRenderTableRemoveRenditions(XmRenderTable oldtable, XmStringTag *tags, int tag_count) \
removes renditions"
  /* DIFF: XmRenderTableRemoveRenditions arg2 is list of strings
   */
  int len = 0;
  XmStringTag *tags = NULL;
  XmRenderTable rt = NULL;
  if (!Xen_is_bound(arg1) || Xen_is_false(arg1)) return(Xen_false);
  Xen_check_type(Xen_is_XmRenderTable(arg1) || Xen_is_false(arg1), arg1, 1, "XmRenderTableRemoveRenditions", "XmRenderTable");
  Xen_check_type(Xen_is_list(arg2) || Xen_is_false(arg2) || !Xen_is_bound(arg2), arg2, 2, "XmRenderTableRemoveRenditions", "XmStringTag*");
  Xen_check_type(Xen_is_integer_or_unbound(arg3), arg3, 3, "XmRenderTableRemoveRenditions", "int");
  if (Xen_is_integer(arg3)) 
    {
      len = Xen_integer_to_C_int(arg3); 
      if (len <= 0) return(Xen_false);
    }
  else 
    {
      if (Xen_is_list(arg2))
	len = Xen_list_length(arg2);
    }
  if (Xen_is_bound(arg2)) tags = (XmStringTag *)Xen_to_C_Strings(arg2, len);
  rt = XmRenderTableRemoveRenditions(Xen_to_C_XmRenderTable(arg1), tags, len);
  if (tags) free(tags);
  return(C_to_Xen_XmRenderTable(rt));
}

static Xen gxm_XmRenderTableCopy(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XmRenderTableCopy "XmRenderTable XmRenderTableCopy(XmRenderTable table, XmStringTag *tags, int tag_count) copies renditions"
  /* DIFF: XmRenderTableCopy arg2 is list of strings
   */
  int len = 0;
  XmStringTag *tags = NULL;
  XmRenderTable rt = NULL;
  if (!Xen_is_bound(arg1) || Xen_is_false(arg1)) return(Xen_false);
  Xen_check_type(Xen_is_XmRenderTable(arg1), arg1, 1, "XmRenderTableCopy", "XmRenderTable");
  Xen_check_type(Xen_is_list(arg2) || Xen_is_false(arg2) || !Xen_is_bound(arg2), arg2, 2, "XmRenderTableCopy", "XmStringTag*");
  Xen_check_type(Xen_is_integer_or_unbound(arg3), arg3, 3, "XmRenderTableCopy", "int");
  if (Xen_is_integer(arg3)) 
    {
      len = Xen_integer_to_C_int(arg3); 
      if (len <= 0) return(Xen_false);
    }
  else 
    {
      if (Xen_is_list(arg2))
	len = Xen_list_length(arg2);
    }
  if (Xen_is_bound(arg2)) tags = (XmStringTag *)Xen_to_C_Strings(arg2, len);
  rt = XmRenderTableCopy(Xen_to_C_XmRenderTable(arg1), tags, len);
  if (tags) free(tags);
  return(C_to_Xen_XmRenderTable(rt));
}

static Xen gxm_XmRenderTableFree(Xen arg1)
{
  #define H_XmRenderTableFree "void XmRenderTableFree(XmRenderTable table)  recovers memory"
  Xen_check_type(Xen_is_XmRenderTable(arg1), arg1, 1, "XmRenderTableFree", "XmRenderTable");
  XmRenderTableFree(Xen_to_C_XmRenderTable(arg1));
  return(Xen_false);
}

static Xen gxm_XmRenderTableGetTags(Xen arg1)
{
  #define H_XmRenderTableGetTags "int XmRenderTableGetTags(XmRenderTable table) gets rendition tags (list of strings)"
  /* DIFF: XmRenderTableGetTags omits arg2, returns list of strings
   */
  int len, loc;
  Xen lst = Xen_empty_list;
  XmStringTag *str;
  Xen_check_type(Xen_is_XmRenderTable(arg1), arg1, 1, "XmRenderTableGetTags", "XmRenderTable");
  loc = xm_protect(lst);
  len = XmRenderTableGetTags(Xen_to_C_XmRenderTable(arg1), &str);
  if (str)
    {
      int i;
      for (i = len - 1; i >= 0; i--)
	{
	  lst = Xen_cons(C_string_to_Xen_string(str[i]), lst);
	  free(str[i]);
	}
      free(str);
    }
  xm_unprotect_at(loc);
  return(lst);
}

static Xen gxm_XmRenderTableGetRendition(Xen arg1, Xen arg2)
{
  #define H_XmRenderTableGetRendition "XmRendition XmRenderTableGetRendition(XmRenderTable table, XmStringTag tag) \
matches a rendition tag"
  Xen_check_type(Xen_is_XmRenderTable(arg1), arg1, 1, "XmRenderTableGetRendition", "XmRenderTable");
  Xen_check_type(Xen_is_string(arg2), arg2, 2, "XmRenderTableGetRendition", "XmStringTag");
  return(C_to_Xen_XmRendition(XmRenderTableGetRendition(Xen_to_C_XmRenderTable(arg1), (char *)Xen_string_to_C_string(arg2))));
}

static Xen gxm_XmRenderTableGetRenditions(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XmRenderTableGetRenditions "XmRendition *XmRenderTableGetRenditions(XmRenderTable table, XmStringTag *tags, Cardinal tag_count) \
matches rendition tags"
  /* DIFF: XmRenderTableGetRenditions returns list of XmRenditions, arg2 is list of strings
   */
  int i, len = 0, loc;
  Xen lst = Xen_empty_list;
  XmRendition *rs;
  XmStringTag *tags;
  if ((!Xen_is_bound(arg1)) || !Xen_is_bound(arg2) || Xen_is_false(arg2)) return(Xen_false);
  Xen_check_type(Xen_is_XmRenderTable(arg1), arg1, 1, "XmRenderTableGetRenditions", "XmRenderTable");
  Xen_check_type(Xen_is_list(arg2), arg2, 2, "XmRenderTableGetRenditions", "list of String");
  Xen_check_type(Xen_is_integer_or_unbound(arg3), arg3, 3, "XmRenderTableGetRenditions", "int");
  if (Xen_is_bound(arg3))
    len = Xen_integer_to_C_int(arg3);
  else len = Xen_list_length(arg2);
  if (len <= 0) return(Xen_false);
  loc = xm_protect(lst);
  tags = (XmStringTag *)Xen_to_C_Strings(arg2, len);
  rs = XmRenderTableGetRenditions(Xen_to_C_XmRenderTable(arg1), tags, (Cardinal)len);
  free(tags);
  for (i = len - 1; i >= 0; i--)
    lst = Xen_cons(C_to_Xen_XmRendition(rs[i]), lst);
  free(rs);
  xm_unprotect_at(loc);
  return(lst);
}

static Xen gxm_XmRenditionCreate(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XmRenditionCreate "XmRendition XmRenditionCreate(Widget widget, XmStringTag tag, ArgList arglist, Cardinal argcount) \
creates a rendition"
  XmRendition w;
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XmRenditionCreate", "Widget");
  Xen_check_type(Xen_is_string(arg2), arg2, 2, "XmRenditionCreate", "XmStringTag");
  Xen_check_type(Xen_is_list(arg3), arg3, 3, "XmRenditionCreate", "ArgList");
  Xen_check_type(Xen_is_integer_or_unbound(arg4), arg4, 4, "XmRenditionCreate", "int");
  {
    Arg *args;
    int arglen;
    args = Xen_to_C_Args(arg3);
    arglen = Xen_to_C_INT_DEF(arg4, arg3);
    w = XmRenditionCreate(Xen_to_C_Widget(arg1), (char *)Xen_string_to_C_string(arg2), args, arglen);
    if (args) free_args(args, arglen);
  }
  return(C_to_Xen_XmRendition(w));
}

static Xen gxm_XmRenditionFree(Xen arg1)
{
  #define H_XmRenditionFree "void XmRenditionFree(XmRendition rendition) frees a rendition"
  Xen_check_type(Xen_is_XmRendition(arg1), arg1, 1, "XmRenditionFree", "XmRendition");
  XmRenditionFree(Xen_to_C_XmRendition(arg1));
  return(Xen_false);
}

static Xen gxm_XmRenditionRetrieve(Xen arg1, Xen larg2, Xen arg3)
{
  #define H_XmRenditionRetrieve "void XmRenditionRetrieve(XmRendition rendition, ArgList arglist, Cardinal argcount) \
retrieves rendition resources"
  Xen_check_type(Xen_is_XmRendition(arg1), arg1, 1, "XmRenditionRetrieve", "XmRendition");
  Xen_check_type(Xen_is_list(larg2), larg2, 2, "XmRenditionRetrieve", "ArgList");
  Xen_check_type(Xen_is_integer_or_unbound(arg3), arg3, 3, "XmRenditionRetrieve", "int");
  {
    /* this is a kind of XtGetvalues, not set
     */
    Arg *args;
    unsigned long *locs;
    Xen val;
    int i, len, gcloc;
    XmRendition r;
    Xen arg2;
    arg2 = Xen_copy_arg(larg2);
    gcloc = xm_protect(arg2);
    /* here we need to make sure the ref args are ok from C's point of view */
    r = Xen_to_C_XmRendition(arg1);
    len = Xen_to_C_INT_DEF(arg3, arg2);
    if (len <= 0) Xen_check_type(0, arg3, 3, "XmRenditionRetrieve", "positive integer");
    args = (Arg *)calloc(len, sizeof(Arg));
    locs = (unsigned long *)calloc(len, sizeof(unsigned long));
    for (i = 0; i < len; i++, arg2 = Xen_cddr(arg2))
      {
	char *name;
	name = xen_strdup(Xen_string_to_C_string(Xen_car(arg2)));
	XtSetArg(args[i], name, &(locs[i]));
      }
    XmRenditionRetrieve(r, args, len);
    val = C_to_Xen_Args((Widget)r, args, len);
    free_args(args, len);
    free(locs);
    xm_unprotect_at(gcloc);
    return(val);
  }
}

static Xen gxm_XmRenditionUpdate(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XmRenditionUpdate "void XmRenditionUpdate(XmRendition rendition, ArgList arglist, Cardinal argcount) \
modifies resources"
  Xen_check_type(Xen_is_XmRendition(arg1), arg1, 1, "XmRenditionUpdate", "XmRendition");
  Xen_check_type(Xen_is_list(arg2), arg2, 2, "XmRenditionUpdate", "ArgList");
  Xen_check_type(Xen_is_integer_or_unbound(arg3), arg3, 3, "XmRenditionUpdate", "int");
  {
    Arg *args;
    int arglen;
    args = Xen_to_C_Args(arg2);
    arglen = Xen_to_C_INT_DEF(arg3, arg2);
    XmRenditionUpdate(Xen_to_C_XmRendition(arg1), args, arglen);
    if (args) free_args(args, arglen);
  }
  return(Xen_false);
}

static Xen gxm_XmRenderTableCvtToProp(Xen arg1, Xen arg2)
{
  #define H_XmRenderTableCvtToProp "unsigned int XmRenderTableCvtToProp(Widget widget, XmRenderTable table) \
converts a render table to a string representation -> (val props)"
  /* DIFF: XmRenderTableCvtToProp omits and rtns arg3
   */
  char *buf;
  unsigned int val;
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XmRenderTableCvtToProp", "Widget");
  Xen_check_type(Xen_is_XmRenderTable(arg2), arg2, 2, "XmRenderTableCvtToProp", "XmRenderTable");
  val = XmRenderTableCvtToProp(Xen_to_C_Widget(arg1), Xen_to_C_XmRenderTable(arg2), &buf);
  return(Xen_list_2(C_ulong_to_Xen_ulong(val),
		    C_string_to_Xen_string(buf)));
}

static Xen gxm_XmRenderTableCvtFromProp(Xen arg1, Xen arg2, Xen arg3)
{
  char *str;
  int len;
  #define H_XmRenderTableCvtFromProp "XmRenderTable XmRenderTableCvtFromProp(Widget widget, char *property, unsigned int length) \
converts from a string representation to a render table"
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XmRenderTableCvtFromProp", "Widget");
  Xen_check_type(Xen_is_string(arg2), arg2, 2, "XmRenderTableCvtFromProp", "char*");
  Xen_check_type(Xen_is_ulong(arg3), arg3, 3, "XmRenderTableCvtFromProp", "unsigned int");
  str = (char *)Xen_string_to_C_string(arg2);
  len = Xen_ulong_to_C_ulong(arg3);
  if ((str) && ((int)strlen(str) == len))
    return(C_to_Xen_XmRenderTable(XmRenderTableCvtFromProp(Xen_to_C_Widget(arg1), str, len)));
  return(Xen_false);
}

static Xen gxm_XmTabListInsertTabs(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XmTabListInsertTabs "XmTabList XmTabListInsertTabs(XmTabList oldlist, XmTab *tabs, Cardinal tab_count, int position) \
inserts tabs into a tab list"
  /* DIFF: XmTabListInsertTabs arg2 is list of XmTabs (can be #f = NULL)
   */
  int len, listlen = 0;
  XmTab *tabs;
  XmTabList tl;
  Xen_check_type(Xen_is_XmTabList(arg1) || Xen_is_false(arg1), arg1, 1, "XmTabListInsertTabs", "XmTabList");
  Xen_check_type(Xen_is_list(arg2), arg2, 2, "XmTabListInsertTabs", "list of XmTab");
  Xen_check_type(Xen_is_integer(arg3), arg3, 3, "XmTabListInsertTabs", "int");
  Xen_check_type(Xen_is_integer(arg4), arg4, 4, "XmTabListInsertTabs", "int");
  listlen = Xen_list_length(arg2);
  len = Xen_integer_to_C_int(arg3);
  if (len > listlen) len = listlen;
  if (len <= 0) return(Xen_false);
  tabs = Xen_to_C_XmTabs(arg2, len);
  tl = XmTabListInsertTabs(Xen_is_false(arg1) ? NULL : Xen_to_C_XmTabList(arg1), 
			   tabs, len, Xen_integer_to_C_int(arg4));
  free(tabs);
  return(C_to_Xen_XmTabList(tl));
}

static Xen gxm_XmTabListCopy(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XmTabListCopy "XmTabList XmTabListCopy(XmTabList tablist, int offset, Cardinal count) creates \
a new tab list from an existing list"
  /* Motif documentation incorrectly calls this "XmTabListTabCopy" */
  Xen_check_type(Xen_is_XmTabList(arg1), arg1, 1, "XmTabListCopy", "XmTabList");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XmTabListCopy", "int");
  Xen_check_type(Xen_is_integer(arg3), arg3, 3, "XmTabListCopy", "int");
  return(C_to_Xen_XmTabList(XmTabListCopy(Xen_to_C_XmTabList(arg1), Xen_integer_to_C_int(arg2), Xen_integer_to_C_int(arg3))));
}

static Xen gxm_XmTabListTabCount(Xen arg1)
{
  #define H_XmTabListTabCount "Cardinal XmTabListTabCount(XmTabList tablist): returns length of tablist"
  Xen_check_type(Xen_is_XmTabList(arg1), arg1, 1, "XmTabListTabCount", "XmTabList");  
  return(C_int_to_Xen_integer(XmTabListTabCount(Xen_to_C_XmTabList(arg1))));
}


static Xen gxm_XmTabListGetTab(Xen arg1, Xen arg2)
{
  #define H_XmTabListGetTab "XmTab XmTabListGetTab(XmTabList tablist, Cardinal position): returns a copy of a tab"
  Xen_check_type(Xen_is_XmTabList(arg1), arg1, 1, "XmTabListGetTab", "XmTabList");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XmTabListGetTab", "int");
  return(C_to_Xen_XmTab(XmTabListGetTab(Xen_to_C_XmTabList(arg1), Xen_integer_to_C_int(arg2))));
}

static Xen gxm_XmTabListReplacePositions(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XmTabListReplacePositions "XmTabList XmTabListReplacePositions(XmTabList oldlist, Cardinal *position_list, XmTab *tabs, Cardinal tab_count) \
creates a new tab list with replacement tabs"
  /* DIFF: XmTabListReplacePositions arg2 is list of ints, arg3 is list of XmTabs
   */
  Cardinal *ts;
  int len;
  Xen res;
  XmTab *tabs;
  Xen_check_type(Xen_is_XmTabList(arg1), arg1, 1, "XmTabListReplacePositions", "XmTabList");
  Xen_check_type(Xen_is_list(arg2), arg2, 2, "XmTabListReplacePositions", "list of ints");
  Xen_check_type(Xen_is_list(arg3), arg3, 3, "XmTabListReplacePositions", "list of XmTab");
  Xen_check_type(Xen_is_integer_or_unbound(arg4), arg4, 4, "XmTabListReplacePositions", "int");
  if (Xen_is_integer(arg4)) len = Xen_integer_to_C_int(arg4); else len = Xen_list_length(arg3);
  if (len <= 0) return(Xen_false);
  ts = Xen_to_C_Cardinals(arg2, len);
  tabs = Xen_to_C_XmTabs(arg3, len);
  res = C_to_Xen_XmTabList(XmTabListReplacePositions(Xen_to_C_XmTabList(arg1), ts, tabs, len));
  free(ts);
  free(tabs);
  return(res);
}

static Xen gxm_XmTabListRemoveTabs(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XmTabListRemoveTabs "XmTabList XmTabListRemoveTabs(XmTabList oldlist, Cardinal *position_list, Cardinal position_count) \
removes noncontiguous tabs"
  /* DIFF: XmTabListRemoveTabs arg2 is list of ints
   */
  Cardinal *ts;
  int len;
  Xen res;
  Xen_check_type(Xen_is_XmTabList(arg1), arg1, 1, "XmTabListRemoveTabs", "XmTabList");
  Xen_check_type(Xen_is_list(arg2), arg2, 2, "XmTabListRemoveTabs", "list of int");
  Xen_check_type(Xen_is_integer_or_unbound(arg3), arg3, 3, "XmTabListRemoveTabs", "int");
  if (Xen_is_integer(arg3)) len = Xen_integer_to_C_int(arg3); else len = Xen_list_length(arg2);
  if (len <= 0) return(Xen_false);
  ts = Xen_to_C_Cardinals(arg2, len);
  res = C_to_Xen_XmTabList(XmTabListRemoveTabs(Xen_to_C_XmTabList(arg1), ts, len));
  free(ts);
  return(res);
}

static Xen gxm_XmTabCreate(Xen arg1, Xen arg2, Xen arg3, Xen arg4, Xen arg5)
{
  #define H_XmTabCreate "XmTab XmTabCreate(float value, unsigned char units, XmOffsetModel offset_model, unsigned char alignment, char *decimal) \
creates a tab stop"
  Xen_check_type(Xen_is_double(arg1), arg1, 1, "XmTabCreate", "float");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XmTabCreate", "unsigned char");
  Xen_check_type(Xen_is_integer(arg3), arg3, 3, "XmTabCreate", "XmOffsetModel"); 
  Xen_check_type(Xen_is_integer(arg4), arg4, 4, "XmTabCreate", "unsigned char");
  Xen_check_type(Xen_is_string(arg5), arg5, 5, "XmTabCreate", "char*");
  return(C_to_Xen_XmTab(XmTabCreate((float)Xen_real_to_C_double(arg1), 
				    (unsigned char)(Xen_integer_to_C_int(arg2)),
				    (XmOffsetModel)Xen_integer_to_C_int(arg3), 
				    (unsigned char)(Xen_integer_to_C_int(arg4)),
				    (char *)Xen_string_to_C_string(arg5))));
}

static Xen gxm_XmTabListFree(Xen arg1)
{
  #define H_XmTabListFree "void XmTabListFree(XmTabList tab) frees a tab list"
  Xen_check_type(Xen_is_XmTabList(arg1), arg1, 1, "XmTabListFree", "XmTabList");
  XmTabListFree(Xen_to_C_XmTabList(arg1));
  return(Xen_false);
}

static Xen gxm_XmTabFree(Xen arg1)
{
  #define H_XmTabFree "void XmTabFree(XmTab tab) frees a tab"
  Xen_check_type(Xen_is_XmTab(arg1), arg1, 1, "XmTabFree", "XmTab");
  XmTabFree(Xen_to_C_XmTab(arg1));
  return(Xen_false);
}

static Xen gxm_XmTabGetValues(Xen arg1)
{
  #define H_XmTabGetValues "float XmTabGetValues(XmTab tab): returns tab values"
  /* DIFF: XmTabGetValues tab [units offset align decimal] -> (list val units ofset align decimal)
     arg2/3/4/5 omit rtn with float 
  */
  XmOffsetModel off;
  unsigned char a1, a2;
  char **a3;
  float res;
  Xen val;
  Xen_check_type(Xen_is_XmTab(arg1), arg1, 1, "XmTabGetValues", "XmTab");
  a3 = (char **)calloc(1, sizeof(char *));
  res = XmTabGetValues(Xen_to_C_XmTab(arg1), &a1, &off, &a2, a3);
  val = Xen_list_5(C_double_to_Xen_real((double)res),
		   C_int_to_Xen_integer((int)a1),
		   C_int_to_Xen_integer((int)off),
		   C_int_to_Xen_integer((int)a2),
		   C_string_to_Xen_string(a3[0]));
  free(a3);
  return(val);
}

static Xen gxm_XmTabSetValue(Xen arg1, Xen arg2)
{
  #define H_XmTabSetValue "void XmTabSetValue(XmTab tab, float value) sets a tab stop"
  Xen_check_type(Xen_is_XmTab(arg1), arg1, 1, "XmTabSetValue", "XmTab");
  Xen_check_type(Xen_is_double(arg2), arg2, 2, "XmTabSetValue", "float");
  XmTabSetValue(Xen_to_C_XmTab(arg1), (float)Xen_real_to_C_double(arg2));
  return(Xen_false);
}

static Xen gxm_XmStringTableProposeTablist(Xen arg1, Xen arg2, Xen arg3, Xen arg4, Xen arg5)
{
  #define H_XmStringTableProposeTablist "XmTabList XmStringTableProposeTablist(XmStringTable strings, Cardinal num_strings, Widget widget, \
float pad_value, XmOffsetModel offset_model): returns a tab list"
  /* DIFF: XmStringTableProposetablist first arg is list of XmStrings
   */
  XmStringTable tab;
  XmTabList tabl;
  Xen_check_type(Xen_is_list(arg1), arg1, 1, "XmStringTableProposeTablist", "XmStringTable");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XmStringTableProposeTablist", "int");
  Xen_check_type(Xen_is_Widget(arg3), arg3, 3, "XmStringTableProposeTablist", "Widget");
  Xen_check_type(Xen_is_double(arg4), arg4, 4, "XmStringTableProposeTablist", "float");
  Xen_check_type(Xen_is_integer(arg5), arg5, 5, "XmStringTableProposeTablist", "XmOffsetModel");
  tab = Xen_to_C_XmStringTable(arg1, Xen_integer_to_C_int(arg2)); 
  tabl = XmStringTableProposeTablist(tab,
				     Xen_integer_to_C_int(arg2), 
				     Xen_to_C_Widget(arg3),
				     (float)Xen_real_to_C_double(arg4), 
				     (XmOffsetModel)Xen_integer_to_C_int(arg5));
  free(tab);
  return(C_to_Xen_XmTabList(tabl));
}

#define Xen_is_XmParseTable(Arg) Xen_is_list(Arg)

static XmParseTable Xen_to_C_XmParseTable(Xen lst, int size)
{
  int i;
  XmParseTable pt;
  if (size <= 0) return(NULL);
  pt = (XmParseTable)XtCalloc(size, sizeof(XmParseMapping));
  for (i = 0; i < size; i++)
    {
      Xen val;
      val = Xen_list_ref(lst, i);
      if (Xen_is_XmParseMapping(val))
	pt[i] = Xen_to_C_XmParseMapping(val);
      else Xen_check_type(0, val, i, __func__, "an XmParseMapping");
    }
  return(pt);
}

static Xen gxm_XmParseTableFree(Xen arg1, Xen arg2)
{
  #define H_XmParseTableFree "void XmParseTableFree(XmParseTable parse_table, Cardinal count) recovers memory"
  Xen_check_type(Xen_is_XmParseTable(arg1), arg1, 1, "XmParseTableFree", "XmParseTable");
  Xen_check_type(Xen_is_integer_or_unbound(arg2), arg2, 2, "XmParseTableFree", "int");
  /* can't happen -- a no-op */
  return(Xen_false);
}

static Xen gxm_XmParseMappingFree(Xen arg1)
{
  #define H_XmParseMappingFree "void XmParseMappingFree(XmParseMapping parse_mapping) frees a parse mapping"
  Xen_check_type(Xen_is_XmParseMapping(arg1), arg1, 1, "XmParseMappingFree", "XmParseMapping");
  XmParseMappingFree(Xen_to_C_XmParseMapping(arg1));
  return(Xen_false);
}

static Xen gxm_XmParseMappingGetValues(Xen arg1, Xen larg2, Xen arg3)
{
  #define H_XmParseMappingGetValues "void XmParseMappingGetValues(XmParseMapping parse_mapping, ArgList arglist, Cardinal argcount) \
retrieves attributes of a parse mapping"
  Xen_check_type(Xen_is_XmParseMapping(arg1), arg1, 1, "XmParseMappingGetValues", "XmParseMapping");
  Xen_check_type(Xen_is_list(larg2), larg2, 2, "XmParseMappingGetValues", "ArgList");
  Xen_check_type(Xen_is_integer_or_unbound(arg3), arg3, 3, "XmParseMappingGetValues", "int");
  {
    Arg *args;
    unsigned long *locs;
    int len, gcloc;
    Xen val, arg2;
    int i;
    len = Xen_to_C_INT_DEF(arg3, larg2);
    arg2 = Xen_copy_arg(larg2);
    gcloc = xm_protect(arg2);
    args = (Arg *)calloc(len, sizeof(Arg));
    locs = (unsigned long *)calloc(len, sizeof(unsigned long));
    for (i = 0; i < len; i++, arg2 = Xen_cddr(arg2))
      {
	char *name;
	name = xen_strdup(Xen_string_to_C_string(Xen_car(arg2)));
	XtSetArg(args[i], name, &(locs[i]));
      }
    XmParseMappingGetValues(Xen_to_C_XmParseMapping(arg1), args, len);
    val = C_to_Xen_Args(NULL, args, len);
    free_args(args, len);
    free(locs);
    xm_unprotect_at(gcloc);
    return(val);
  }
}

static Xen gxm_XmParseMappingSetValues(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XmParseMappingSetValues "void XmParseMappingSetValues(XmParseMapping parse_mapping, ArgList arglist, Cardinal argcount) \
sets attributes of a parse mapping"
  Xen_check_type(Xen_is_XmParseMapping(arg1), arg1, 1, "XmParseMappingSetValues", "XmParseMapping");
  Xen_check_type(Xen_is_list(arg2), arg2, 2, "XmParseMappingSetValues", "ArgList");
  Xen_check_type(Xen_is_integer_or_unbound(arg3), arg3, 3, "XmParseMappingSetValues", "int");
  {
    Arg *args;
    int arglen;
    args = Xen_to_C_Args(arg2);
    arglen = Xen_to_C_INT_DEF(arg3, arg2);
    XmParseMappingSetValues(Xen_to_C_XmParseMapping(arg1), args, arglen);
    if (args) free_args(args, arglen);
    return(Xen_false);
  }
}

static Xen gxm_XmParseMappingCreate(Xen arg1, Xen arg2)
{
  #define H_XmParseMappingCreate "XmParseMapping XmParseMappingCreate(ArgList arglist, Cardinal argcount) create a parse mapping"
  XmParseMapping w;
  Xen_check_type(Xen_is_list(arg1), arg1, 1, "XmParseMappingCreate", "ArgList");
  Xen_check_type(Xen_is_integer_or_unbound(arg2), arg2, 2, "XmParseMappingCreate", "int");
  {
    Arg *args;
    int arglen;
    args = Xen_to_C_Args(arg1);
    arglen = Xen_to_C_INT_DEF(arg2, arg1);
    w = XmParseMappingCreate(args, arglen);
    if (args) free_args(args, arglen);
  }
  return(C_to_Xen_XmParseMapping(w));
}

static Xen gxm_XmCvtXmStringToByteStream(Xen str)
{
  #define H_XmCvtXmStringToByteStream "int XmCvtXmStringToByteStream(XmString str) converts an XmString into a byte stream."
  unsigned char *prop = NULL;
  int res = 0; /* #bytes in returned val */
  XmString xstr;
  Xen_check_type(Xen_is_XmString(str), str, 1, "XmCvtXmStringToByteStream", "XmString");
  xstr = Xen_to_C_XmString(str);
  if (!(XmStringEmpty(xstr)))
    res = XmCvtXmStringToByteStream(xstr, &prop);
  /* here there is apparently no trailing 0, so this function worries Valgrind */
  return(Xen_list_2(C_int_to_Xen_integer(res), C_string_to_Xen_string((const char *)prop)));
}

static Xen gxm_XmCvtByteStreamToXmString(Xen str)
{
  #define H_XmCvtByteStreamToXmString "XmString XmCvtByteStreamToXmString(char *str) converts a byte stream into an XmString."
  XmString res = NULL;
  const char *bstr;
  Xen_check_type(Xen_is_string(str), str, 1, "XmCvtByteStreamToXmString", "char *");
  bstr = Xen_string_to_C_string(str);
  if (bstr)
    res = XmCvtByteStreamToXmString((unsigned char *)(Xen_string_to_C_string(str)));
  return(C_to_Xen_XmString(res));
}

static Xen gxm_XmStringByteStreamLength(Xen str)
{
  #define H_XmStringByteStreamLength "int XmStringByteStreamLength(char *str): returns the length of the byte stream."
  unsigned char *stream;

  Xen_check_type(Xen_is_string(str), str, 1, "XmStringByteStreamLength", "char *");
  stream = (unsigned char *)(Xen_string_to_C_string(str));
  if (!stream)
    Xen_out_of_range_error("XmStringByteStreamLength", 1, str, "a null stream?");
  return(C_int_to_Xen_integer((int)XmStringByteStreamLength(stream)));
}

static Xen gxm_XmStringPutRendition(Xen arg1, Xen arg2)
{
  #define H_XmStringPutRendition "XmString XmStringPutRendition(XmString string, XmStringTag rendition) places \
renditions around strings"
  Xen_check_type(Xen_is_XmString(arg1), arg1, 1, "XmStringPutRendition", "XmString");
  Xen_check_type(Xen_is_string(arg2), arg2, 2, "XmStringPutRendition", "XmStringTag");
  return(C_to_Xen_XmString(XmStringPutRendition(Xen_to_C_XmString(arg1), (char *)Xen_string_to_C_string(arg2))));
}

static Xen gxm_XmStringGenerate(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XmStringGenerate "XmString XmStringGenerate(XtPointer text, XmStringTag tag, XmTextType type, XmStringTag rendition) \
generates a compound string"

  XmTextType type;
  XmStringTag rendition;
  Xen_check_type(Xen_is_string(arg1), arg1, 1, "XmStringGenerate", "XtPointer");
  Xen_check_type(Xen_is_string(arg2) || Xen_is_false(arg2), arg2, 2, "XmStringGenerate", "XmStringTag");
  Xen_check_type(Xen_is_integer(arg3), arg3, 3, "XmStringGenerate", "XmTextType");
  Xen_check_type(Xen_is_string(arg4), arg4, 4, "XmStringGenerate", "XmStringTag");
  type = (XmTextType)Xen_integer_to_C_int(arg3);
  if (type > 1) Xen_check_type(0, arg3, 3, "XmStringGenerate", "XmTextType");
  rendition = (XmStringTag)Xen_string_to_C_string(arg4);
  if (!rendition)
    Xen_out_of_range_error("XmStringGenerate", 4, arg4, "a null rendition?");
  return(C_to_Xen_XmString(XmStringGenerate((XtPointer)Xen_string_to_C_string(arg1), 
					    Xen_is_false(arg2) ? NULL : (char *)Xen_string_to_C_string(arg2), 
					    type,
					    rendition)));
}

static Xen gxm_XmStringDirectionToDirection(Xen arg1)
{
  #define H_XmStringDirectionToDirection "XmDirection XmStringDirectionToDirection(XmStringDirection direction) converts \
from XmStringDirection to XmDirection"
  Xen_check_type(Xen_is_integer(arg1), arg1, 1, "XmStringDirectionToDirection", "XmStringDirection");
  return(C_int_to_Xen_integer(XmStringDirectionToDirection(Xen_integer_to_C_int(arg1))));
}

static Xen gxm_XmDirectionToStringDirection(Xen arg1)
{
  #define H_XmDirectionToStringDirection "XmStringDirection XmDirectionToStringDirection (dir) converts an XmDirection \
value to an XmStringDirection value"
  Xen_check_type(Xen_is_integer(arg1), arg1, 1, "XmDirectionToStringDirection", "XmDirection");
  return(C_int_to_Xen_integer(XmDirectionToStringDirection(Xen_integer_to_C_int(arg1))));
}

static Xen gxm_XmStringTableParseStringArray(Xen arg1, Xen arg2, Xen arg3, Xen arg4, Xen arg5, Xen arg6, Xen arg7)
{
  #define H_XmStringTableParseStringArray "XmStringTable XmStringTableParseStringArray(XtPointer *strings, Cardinal count, XmStringTag tag, \
XmTextType type, XmParseTable parse, Cardinal parse_count, XtPointer call_data) converts an array of strings \
to a compound string table"
  /* DIFF: XmStringTableParseStringArray arg1 is list of strings
   */
  char **strs;
  int len;
  XmStringTable val;
  XmParseTable pt = NULL;
  Xen lst;
  XmTextType type;
  Xen_check_type(Xen_is_list(arg1), arg1, 1, "XmStringTableParseStringArray", "list of strings");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XmStringTableParseStringArray", "int");
  Xen_check_type(Xen_is_false(arg3) || Xen_is_string(arg3), arg3, 3, "XmStringTableParseStringArray", "XmStringTag");
  Xen_check_type(Xen_is_integer(arg4), arg4, 4, "XmStringTableParseStringArray", "XmTextType");
  Xen_check_type(Xen_is_false(arg5) || Xen_is_XmParseTable(arg5), arg5, 5, "XmStringTableParseStringArray", "XmParseTable");
  Xen_check_type(Xen_is_integer(arg6), arg6, 6, "XmStringTableParseStringArray", "int");
  type = (XmTextType)Xen_integer_to_C_int(arg4);
  if (type > 1) Xen_check_type(0, arg4, 4, "XmStringTableParseStringArray", "XmTextType");
  len = Xen_integer_to_C_int(arg2);
  if (len <= 0) return(Xen_false);
  strs = Xen_to_C_Strings(arg1, len);
  if (Xen_is_XmParseTable(arg5)) pt = Xen_to_C_XmParseTable(arg5, len);
  val = XmStringTableParseStringArray((XtPointer *)strs, 
				      (Cardinal)len, 
				      (Xen_is_false(arg3)) ? NULL : (char *)Xen_string_to_C_string(arg3),
				      type,
				      pt,
				      Xen_integer_to_C_int(arg6),
				      (XtPointer)arg7);
  free(strs);
  lst = C_to_Xen_XmStringTable(val, len);
  free(val);
  if (pt) free(pt); 
  return(lst);
}

static Xen gxm_XmStringTableUnparse(Xen arg1, Xen arg2, Xen arg3, Xen arg4, Xen arg5, Xen arg6, Xen arg7, Xen arg8)
{
  #define H_XmStringTableUnparse "XtPointer *XmStringTableUnparse(XmStringTable table, Cardinal count, XmStringTag tag, XmTextType tag_type, \
XmTextType output_type, XmParseTable parse, Cardinal parse_count, XmParseModel parse_model) converts a table of \
compound strings to an array of text"
  /* DIFF: XmStringTableUnparse returns list of strings, first arg is list of XmStrings
   */
  Xen lst = Xen_empty_list;
  char **tab;
  XmStringTable tb;
  int i, len, loc;
  XmTextType type1, type2;
  XmParseTable pt = NULL;
  Xen_check_type(Xen_is_list(arg1), arg1, 1, "XmStringTableUnparse", "XmStringTable");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XmStringTableUnparse", "int");
  Xen_check_type(Xen_is_false(arg3) || Xen_is_string(arg3), arg3, 3, "XmStringTableUnparse", "XmStringTag");
  Xen_check_type(Xen_is_integer(arg4), arg4, 4, "XmStringTableUnparse", "XmTextType");
  Xen_check_type(Xen_is_integer(arg5), arg5, 5, "XmStringTableUnparse", "XmTextType");
  Xen_check_type(Xen_is_false(arg6) || Xen_is_XmParseTable(arg6), arg6, 6, "XmStringTableUnparse", "XmParseTable");
  Xen_check_type(Xen_is_integer(arg7), arg7, 7, "XmStringTableUnparse", "int");
  Xen_check_type(Xen_is_integer(arg8), arg8, 8, "XmStringTableUnparse", "XmParseModel");
  type1 = (XmTextType)Xen_integer_to_C_int(arg4);
  if (type1 > 1) Xen_check_type(0, arg4, 4, "XmStringTableUnparse", "XmTextType");
  type2 = (XmTextType)Xen_integer_to_C_int(arg5);
  if (type2 > 1) Xen_check_type(0, arg5, 5, "XmStringTableUnparse", "XmTextType");
  len = Xen_integer_to_C_int(arg2);
  if (len <= 0) return(Xen_false);
  loc = xm_protect(lst);
  tb = Xen_to_C_XmStringTable(arg1, len);
  if (Xen_is_XmParseTable(arg6)) pt = Xen_to_C_XmParseTable(arg6, len);
  tab = (char **)XmStringTableUnparse(tb,
				      len, 
				      (Xen_is_false(arg3)) ? NULL : (char *)Xen_string_to_C_string(arg3), 
				      type1,
				      type2,
				      pt,
				      Xen_integer_to_C_int(arg7), 
				      (XmParseModel)Xen_integer_to_C_int(arg8));
  free(tb);
  for (i = len - 1; i >= 0; i--)
    {
      lst = Xen_cons(C_string_to_Xen_string(tab[i]), lst);
      free(tab[i]);
    }
  free(tab);
  if (pt) free(pt);
  xm_unprotect_at(loc);
  return(lst);
}

static Xen gxm_XmStringTableToXmString(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XmStringTableToXmString "XmString XmStringTableToXmString(XmStringTable table, Cardinal count, XmString break_component) \
converts a compound string table to a single compound string"
  /* DIFF: XmStringTableToXmString first arg is list of XmStrings
   */
  int count;
  XmStringTable tab;
  XmString val;
  Xen_check_type(Xen_is_list(arg1), arg1, 1, "XmStringTableToXmString", "XmStringTable");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XmStringTableToXmString", "int");
  Xen_check_type(Xen_is_XmString(arg3) || Xen_is_false(arg3), arg3, 3, "XmStringTableToXmString", "XmString");
  count = Xen_integer_to_C_int(arg2);
  if (count <= 0) return(Xen_false);
  tab = Xen_to_C_XmStringTable(arg1, count);
  val = XmStringTableToXmString(tab,
				count,
				(Xen_is_XmString(arg3)) ? Xen_to_C_XmString(arg3) : NULL);
  free(tab);
  return(C_to_Xen_XmString(val));
}

static Xen gxm_XmStringToXmStringTable(Xen arg1, Xen arg2)
{
  #define H_XmStringToXmStringTable "Cardinal XmStringToXmStringTable(XmString string, XmString break_component) \
converts a single compound string to a table of compound strings"
  /* DIFF: XmStringToXmStringTable omits and rtns (list len table)
   */
  XmStringTable tab;
  Cardinal val;
  Xen_check_type(Xen_is_XmString(arg1), arg1, 1, "XmStringToXmStringTable", "XmString");
  Xen_check_type(Xen_is_XmString(arg2) || Xen_is_false(arg2), arg2, 2, "XmStringToXmStringTable", "XmString");
  val = XmStringToXmStringTable(Xen_to_C_XmString(arg1), 
				(Xen_is_XmString(arg2)) ? Xen_to_C_XmString(arg2) : NULL,
				&tab);
  return(Xen_list_2(C_int_to_Xen_integer(val), C_to_Xen_XmStringTable(tab, val)));
}

static Xen gxm_XmStringParseText(Xen arg1, Xen arg2, Xen arg3, Xen arg4, Xen arg5, Xen arg6, Xen arg7)
{
  #define H_XmStringParseText "XmString XmStringParseText(XtPointer text, XtPointer *text_end, XmStringTag tag, XmTextType type, \
XmParseTable parse_table, Cardinal parse_count, XtPointer call_data) converts a character string to a compound string"
  /* DIFF: XmStringParseText arg1 is string, arg2 is int
   */
  int len;
  const char *str, *tag = NULL;
  XmTextType type;
  XtPointer *intext = NULL;
  XmParseTable pt = NULL;
  Xen rtn;
  Xen_check_type(Xen_is_string(arg1), arg1, 1, "XmStringParseText", "string");
  Xen_check_type(Xen_is_integer(arg2) || Xen_is_false(arg2), arg2, 2, "XmStringParseText", "int");
  Xen_check_type(Xen_is_string(arg3) || Xen_is_false(arg3), arg3, 3, "XmStringParseText", "XmStringTag");
  Xen_check_type(Xen_is_integer(arg4), arg4, 4, "XmStringParseText", "XmTextType");
  Xen_check_type(Xen_is_XmParseTable(arg5), arg5, 5, "XmStringParseText", "XmParseTable");
  Xen_check_type(Xen_is_integer(arg6), arg6, 6, "XmStringParseText", "int");
  type = (XmTextType)Xen_integer_to_C_int(arg4);
  if (type > 1) Xen_check_type(0, arg4, 4, "XmStringParseText", "XmTextType");
  str = Xen_string_to_C_string(arg1);
  if (Xen_is_integer(arg2)) 
    {
      int loc;
      loc = Xen_integer_to_C_int(arg2);
      intext = (XtPointer *)(str + loc);
    }
  if (Xen_is_string(arg3)) tag = Xen_string_to_C_string(arg3);
  len = Xen_integer_to_C_int(arg6);
  if (Xen_is_XmParseTable(arg5)) pt = Xen_to_C_XmParseTable(arg5, len);
  rtn = C_to_Xen_XmString(XmStringParseText((char *)str, intext, (char *)tag, type, pt, len, (XtPointer)arg7));
  if (pt) free(pt);
  return(rtn);
}

static Xen gxm_XmStringUnparse(Xen arg1, Xen arg2, Xen arg3, Xen arg4, Xen arg5, Xen arg6, Xen arg7)
{
  #define H_XmStringUnparse "XtPointer XmStringUnparse(XmString string, XmStringTag tag, XmTextType tag_type, XmTextType output_type, \
XmParseTable parse_table, Cardinal parse_count, XmParseModel parse_model) unparses text"
  XmTextType type1, type2;
  Xen rtn;
  char *str;
  XmParseTable pt = NULL;
  int len;
  Xen_check_type(Xen_is_XmString(arg1), arg1, 1, "XmStringUnparse", "XmString");
  Xen_check_type(Xen_is_string(arg2) || Xen_is_false(arg2), arg2, 2, "XmStringUnparse", "XmStringTag or " PROC_FALSE);
  Xen_check_type(Xen_is_integer(arg3), arg3, 3, "XmStringUnparse", "XmTextType");
  Xen_check_type(Xen_is_integer(arg4), arg4, 4, "XmStringUnparse", "XmTextType");
  Xen_check_type(Xen_is_XmParseTable(arg5) || Xen_is_false(arg5), arg5, 5, "XmStringUnparse", "XmParseTable");
  Xen_check_type(Xen_is_integer(arg6), arg6, 6, "XmStringUnparse", "int");
  Xen_check_type(Xen_is_integer(arg7), arg7, 7, "XmStringUnparse", "XmParseModel");
  type1 = (XmTextType)Xen_integer_to_C_int(arg3);
  if (type1 > 1) Xen_check_type(0, arg3, 3, "XmStringUnparse", "XmTextType");
  type2 = (XmTextType)Xen_integer_to_C_int(arg4);
  if (type2 > 1) Xen_check_type(0, arg4, 4, "XmStringUnparse", "XmTextType");
  len = Xen_integer_to_C_int(arg6);
  if (Xen_is_XmParseTable(arg5)) pt = Xen_to_C_XmParseTable(arg5, len);
  str = (char *)XmStringUnparse(Xen_to_C_XmString(arg1), 
				(Xen_is_string(arg2)) ? (char *)Xen_string_to_C_string(arg2) : NULL,
				type1, type2, pt, len,
				(XmParseModel)Xen_integer_to_C_int(arg7));
  rtn = C_string_to_Xen_string(str);
  if (str) XtFree(str);
  if (pt) free(pt);
  return(rtn);
}

static Xen gxm_XmStringComponentCreate(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XmStringComponentCreate "XmString XmStringComponentCreate(XmStringComponentType c_type, unsigned int length, XtPointer value) \
creates arbitrary components"
  Xen_check_type(Xen_is_integer(arg1), arg1, 1, "XmStringComponentCreate", "XmStringComponentType");
  Xen_check_type(Xen_is_ulong(arg2), arg2, 2, "XmStringComponentCreate", "unsigned int");
  Xen_check_type(Xen_is_string(arg3) || Xen_is_false(arg3), arg3, 3, "XmStringComponentCreate", "XtPointer");
  return(C_to_Xen_XmString(XmStringComponentCreate(Xen_integer_to_C_int(arg1), 
						   Xen_ulong_to_C_ulong(arg2), 
						   Xen_is_false(arg3) ? NULL : (char *)Xen_string_to_C_string(arg3))));
}

static Xen gxm_XmStringGetNextTriple(Xen arg1)
{
  #define H_XmStringGetNextTriple "XmStringComponentType XmStringGetNextTriple(XmStringContext context) \
returns the type, length, and value of the next component in the compound string"
  /* DIFF: XmStringGetNextTriple omits last 2 args, returns val
   */
  unsigned int len;
  int val;
  XtPointer *ptr = NULL;
  Xen_check_type(Xen_is_XmStringContext(arg1), arg1, 1, "XmStringGetNextTriple", "XmStringContext");
  val = XmStringGetNextTriple(Xen_to_C_XmStringContext(arg1), &len, ptr);
  return(Xen_list_3(C_int_to_Xen_integer(val),
		    C_int_to_Xen_integer((int)len),
		    (val == XmSTRING_COMPONENT_TEXT) ? C_string_to_Xen_string((char *)(*ptr)) : Xen_wrap_C_pointer(ptr)));
}

static Xen gxm_XmStringPeekNextTriple(Xen arg1)
{
  #define H_XmStringPeekNextTriple "XmStringComponentType XmStringPeekNextTriple(XmStringContext context): returns the \
component type of the next component"
  Xen_check_type(Xen_is_XmStringContext(arg1), arg1, 1, "XmStringPeekNextTriple", "XmStringContext");
  return(C_int_to_Xen_integer(XmStringPeekNextTriple(Xen_to_C_XmStringContext(arg1))));
}

static Xen gxm_XmStringDrawUnderline(Xen args)
{
  #define H_XmStringDrawUnderline "void XmStringDrawUnderline(Display *d, Window w, XmRenderTable rendertable, XmString string, GC gc, \
Position x, Position y, Dimension width, unsigned char alignment, unsigned char layout_direction, XRectangle *clip, XmString underline) \
underlines a string drawn in an X Window"
  Xen arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8, arg9, arg10, arg11, arg12;
  arg1 = Xen_list_ref(args, 0);
  arg2 = Xen_list_ref(args, 1);
  arg3 = Xen_list_ref(args, 2);
  arg4 = Xen_list_ref(args, 3);
  arg5 = Xen_list_ref(args, 4);
  arg6 = Xen_list_ref(args, 5);
  arg7 = Xen_list_ref(args, 6);
  arg8 = Xen_list_ref(args, 7);
  arg9 = Xen_list_ref(args, 8);
  arg10 = Xen_list_ref(args, 9);
  arg11 = Xen_list_ref(args, 10);
  arg12 = Xen_list_ref(args, 11);
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XmStringDrawUnderline", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XmStringDrawUnderline", "Window");
  Xen_check_type(Xen_is_XmFontList_or_XmRenderTable(arg3), arg3, 3, "XmStringDrawUnderline", "XmFontList");
  Xen_check_type(Xen_is_XmString(arg4), arg4, 4, "XmStringDrawUnderline", "XmString");
  Xen_check_type(Xen_is_GC(arg5), arg5, 5, "XmStringDrawUnderline", "GC");
  Xen_check_type(Xen_is_integer(arg6), arg6, 6, "XmStringDrawUnderline", "int");
  Xen_check_type(Xen_is_integer(arg7), arg7, 7, "XmStringDrawUnderline", "int");
  Xen_check_type(Xen_is_integer(arg8), arg8, 8, "XmStringDrawUnderline", "int");
  Xen_check_type(Xen_is_ulong(arg9), arg9, 9, "XmStringDrawUnderline", "unsigned int");
  Xen_check_type(Xen_is_ulong(arg10), arg10, 10, "XmStringDrawUnderline", "unsigned int");
  Xen_check_type(Xen_is_XRectangle(arg11), arg11, 11, "XmStringDrawUnderline", "XRectangle");
  Xen_check_type(Xen_is_XmString(arg12), arg12, 12, "XmStringDrawUnderline", "XmString");
  XmStringDrawUnderline(Xen_to_C_Display(arg1), 
			Xen_to_C_Window(arg2), 
			Xen_to_C_XmFontList(arg3), 
			Xen_to_C_XmString(arg4), 
			Xen_to_C_GC(arg5), 
			Xen_integer_to_C_int(arg6), Xen_integer_to_C_int(arg7), Xen_integer_to_C_int(arg8), 
			Xen_ulong_to_C_ulong(arg9), Xen_ulong_to_C_ulong(arg10), 
			Xen_to_C_XRectangle(arg11), 
			Xen_to_C_XmString(arg12));
  return(Xen_false);
}

static Xen gxm_XmStringDrawImage(Xen args)
{
  #define H_XmStringDrawImage "void XmStringDrawImage(Display *d, Window w, XmRenderTable rendertable, XmString string, GC gc, Position x, \
Position y, Dimension width, unsigned char alignment, unsigned char layout_direction, XRectangle *clip) \
draws a compound string in an X Window and creates an image"
  Xen arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8, arg9, arg10, arg11;
  arg1 = Xen_list_ref(args, 0);
  arg2 = Xen_list_ref(args, 1);
  arg3 = Xen_list_ref(args, 2);
  arg4 = Xen_list_ref(args, 3);
  arg5 = Xen_list_ref(args, 4);
  arg6 = Xen_list_ref(args, 5);
  arg7 = Xen_list_ref(args, 6);
  arg8 = Xen_list_ref(args, 7);
  arg9 = Xen_list_ref(args, 8);
  arg10 = Xen_list_ref(args, 9);
  arg11 = Xen_list_ref(args, 10);
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XmStringDrawImage", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XmStringDrawImage", "Window");
  Xen_check_type(Xen_is_XmFontList_or_XmRenderTable(arg3), arg3, 3, "XmStringDrawImage", "XmFontList");
  Xen_check_type(Xen_is_XmString(arg4), arg4, 4, "XmStringDrawImage", "XmString");
  Xen_check_type(Xen_is_GC(arg5), arg5, 5, "XmStringDrawImage", "GC");
  Xen_check_type(Xen_is_integer(arg6), arg6, 6, "XmStringDrawImage", "int");
  Xen_check_type(Xen_is_integer(arg7), arg7, 7, "XmStringDrawImage", "int");
  Xen_check_type(Xen_is_integer(arg8), arg8, 8, "XmStringDrawImage", "int");
  Xen_check_type(Xen_is_ulong(arg9), arg9, 9, "XmStringDrawImage", "unsigned int");
  Xen_check_type(Xen_is_ulong(arg10), arg10, 10, "XmStringDrawImage", "unsigned int");
  Xen_check_type(Xen_is_XRectangle(arg11), arg11, 11, "XmStringDrawImage", "XRectangle*");
  XmStringDrawImage(Xen_to_C_Display(arg1), 
		    Xen_to_C_Window(arg2), 
		    Xen_to_C_XmFontList(arg3), 
		    Xen_to_C_XmString(arg4), 
		    Xen_to_C_GC(arg5), 
		    Xen_integer_to_C_int(arg6), Xen_integer_to_C_int(arg7), Xen_integer_to_C_int(arg8), 
		    Xen_ulong_to_C_ulong(arg9), Xen_ulong_to_C_ulong(arg10), 
		    Xen_to_C_XRectangle(arg11));
  return(Xen_false);
}

static Xen gxm_XmStringDraw(Xen args)
{
  #define H_XmStringDraw "void XmStringDraw(Display *d, Window w, XmRenderTable rendertable, XmString string, GC gc, Position x, Position y, \
Dimension width, unsigned char alignment, unsigned char layout_direction, XRectangle *clip) draws a compound \
string in an X window"
  Xen arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8, arg9, arg10, arg11;
  arg1 = Xen_list_ref(args, 0);
  arg2 = Xen_list_ref(args, 1);
  arg3 = Xen_list_ref(args, 2);
  arg4 = Xen_list_ref(args, 3);
  arg5 = Xen_list_ref(args, 4);
  arg6 = Xen_list_ref(args, 5);
  arg7 = Xen_list_ref(args, 6);
  arg8 = Xen_list_ref(args, 7);
  arg9 = Xen_list_ref(args, 8);
  arg10 = Xen_list_ref(args, 9);
  arg11 = Xen_list_ref(args, 10);
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XmStringDraw", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XmStringDraw", "Window");
  Xen_check_type(Xen_is_XmFontList_or_XmRenderTable(arg3), arg3, 3, "XmStringDraw", "XmFontList");
  Xen_check_type(Xen_is_XmString(arg4), arg4, 4, "XmStringDraw", "XmString");
  Xen_check_type(Xen_is_GC(arg5), arg5, 5, "XmStringDraw", "GC");
  Xen_check_type(Xen_is_integer(arg6), arg6, 6, "XmStringDraw", "int");
  Xen_check_type(Xen_is_integer(arg7), arg7, 7, "XmStringDraw", "int");
  Xen_check_type(Xen_is_integer(arg8), arg8, 8, "XmStringDraw", "int");
  Xen_check_type(Xen_is_ulong(arg9), arg9, 9, "XmStringDraw", "unsigned int");
  Xen_check_type(Xen_is_ulong(arg10), arg10, 10, "XmStringDraw", "unsigned int");
  Xen_check_type(Xen_is_XRectangle(arg11), arg11, 11, "XmStringDraw", "XRectangle*");
  XmStringDraw(Xen_to_C_Display(arg1), 
	       Xen_to_C_Window(arg2), 
	       Xen_to_C_XmFontList(arg3), 
	       Xen_to_C_XmString(arg4), 
	       Xen_to_C_GC(arg5), 
	       Xen_integer_to_C_int(arg6), Xen_integer_to_C_int(arg7), Xen_integer_to_C_int(arg8), 
	       Xen_ulong_to_C_ulong(arg9), Xen_ulong_to_C_ulong(arg10), 
	       Xen_to_C_XRectangle(arg11));
  return(Xen_false);
}

static Xen gxm_XmStringLineCount(Xen arg1)
{
  #define H_XmStringLineCount "int XmStringLineCount(XmString string): returns the number of separators plus one \
in the provided compound string"
  Xen_check_type(Xen_is_XmString(arg1), arg1, 1, "XmStringLineCount", "XmString");
  return(C_int_to_Xen_integer(XmStringLineCount(Xen_to_C_XmString(arg1))));
}

static Xen gxm_XmStringExtent(Xen arg1, Xen arg2)
{
  #define H_XmStringExtent "void XmStringExtent(XmRenderTable rendertable, XmString string) \
determines the size of the smallest rectangle that will enclose the compound string"
  /* DIFF: XmStringExtent omits and returns the last 2 args
   */
  Dimension w, h;
  Xen_check_type(Xen_is_XmFontList_or_XmRenderTable(arg1), arg1, 1, "XmStringExtent", "XmFontList");
  Xen_check_type(Xen_is_XmString(arg2), arg2, 2, "XmStringExtent", "XmString");
  XmStringExtent(Xen_to_C_XmFontList(arg1), Xen_to_C_XmString(arg2), &w, &h);
  return(Xen_list_2(C_to_Xen_Dimension(w),
		    C_to_Xen_Dimension(h)));
}

static Xen gxm_XmStringHeight(Xen arg1, Xen arg2)
{
  #define H_XmStringHeight "Dimension XmStringHeight(XmRenderTable rendertable, XmString string): returns the line \
height of the given compound string"
  Xen_check_type(Xen_is_XmFontList_or_XmRenderTable(arg1), arg1, 1, "XmStringHeight", "XmFontList");
  Xen_check_type(Xen_is_XmString(arg2), arg2, 2, "XmStringHeight", "XmString");
  return(C_to_Xen_Dimension(XmStringHeight(Xen_to_C_XmFontList(arg1), Xen_to_C_XmString(arg2))));
}

static Xen gxm_XmStringWidth(Xen arg1, Xen arg2)
{
  #define H_XmStringWidth "Dimension XmStringWidth(XmRenderTable rendertable, XmString string): returns \
the width of the widest line in a compound string"
  Xen_check_type(Xen_is_XmFontList_or_XmRenderTable(arg1), arg1, 1, "XmStringWidth", "XmFontList");
  Xen_check_type(Xen_is_XmString(arg2), arg2, 2, "XmStringWidth", "XmString");
  return(C_to_Xen_Dimension(XmStringWidth(Xen_to_C_XmFontList(arg1), Xen_to_C_XmString(arg2))));
}

static Xen gxm_XmStringBaseline(Xen arg1, Xen arg2)
{
  #define H_XmStringBaseline "Dimension XmStringBaseline(XmRenderTable rendertable, XmString string): returns \
the number of pixels between \
the top of the character box and the baseline of the first line of text"
  Xen_check_type(Xen_is_XmFontList_or_XmRenderTable(arg1), arg1, 1, "XmStringBaseline", "XmFontList");
  Xen_check_type(Xen_is_XmString(arg2), arg2, 2, "XmStringBaseline", "XmString");
  return(C_to_Xen_Dimension(XmStringBaseline(Xen_to_C_XmFontList(arg1), Xen_to_C_XmString(arg2))));
}

static Xen gxm_XmStringFree(Xen arg1)
{
  #define H_XmStringFree "void XmStringFree(XmString string) conditionally deallocates memory"
  Xen_check_type(Xen_is_XmString(arg1), arg1, 1, "XmStringFree", "XmString");
  XmStringFree(Xen_to_C_XmString(arg1));
  return(Xen_false);
}

static Xen gxm_XmStringHasSubstring(Xen arg1, Xen arg2)
{
  #define H_XmStringHasSubstring "Boolean XmStringHasSubstring(XmString string, XmString substring) indicates \
whether one compound string is contained within another"
  Xen_check_type(Xen_is_XmString(arg1), arg1, 1, "XmStringHasSubstring", "XmString");
  Xen_check_type(Xen_is_XmString(arg2), arg2, 2, "XmStringHasSubstring", "XmString");
  return(C_bool_to_Xen_boolean(XmStringHasSubstring(Xen_to_C_XmString(arg1), Xen_to_C_XmString(arg2))));
}

static Xen gxm_XmStringIsVoid(Xen arg1)
{
  #define H_XmStringIsVoid "Boolean XmStringIsVoid(XmString s1) provides information on the existence of non-zero-length text \
components, tab components, or separator components"
  Xen_check_type(Xen_is_XmString(arg1), arg1, 1, "XmStringIsVoid", "XmString");
  return(C_bool_to_Xen_boolean(XmStringIsVoid(Xen_to_C_XmString(arg1))));
}

static Xen gxm_XmStringEmpty(Xen arg1)
{
  #define H_XmStringEmpty "Boolean XmStringEmpty(XmString s1) provides information on the existence of \
non-zero-length text components"
  Xen_check_type(Xen_is_XmString(arg1), arg1, 1, "XmStringEmpty", "XmString");
  return(C_bool_to_Xen_boolean(XmStringEmpty(Xen_to_C_XmString(arg1))));
}

static Xen gxm_XmStringCompare(Xen arg1, Xen arg2)
{
  #define H_XmStringCompare "Boolean XmStringCompare(XmString s1, XmString s2) compares two strings"
  Xen_check_type(Xen_is_XmString(arg1), arg1, 1, "XmStringCompare", "XmString");
  Xen_check_type(Xen_is_XmString(arg2), arg2, 2, "XmStringCompare", "XmString");
  return(C_bool_to_Xen_boolean(XmStringCompare(Xen_to_C_XmString(arg1), Xen_to_C_XmString(arg2))));
}

static Xen gxm_XmStringCopy(Xen arg1)
{
  #define H_XmStringCopy "XmString XmStringCopy(XmString s1)  makes a copy of a string"
  Xen_check_type(Xen_is_XmString(arg1), arg1, 1, "XmStringCopy", "XmString");
  return(C_to_Xen_XmString(XmStringCopy(Xen_to_C_XmString(arg1))));
}

static Xen gxm_XmStringConcatAndFree(Xen arg1, Xen arg2)
{
  #define H_XmStringConcatAndFree "XmString XmStringConcatAndFree(XmString s1, XmString s2) appends one string to \
another and frees the original strings"
  Xen_check_type(Xen_is_XmString(arg1), arg1, 1, "XmStringConcatAndFree", "XmString");
  Xen_check_type(Xen_is_XmString(arg2), arg2, 2, "XmStringConcatAndFree", "XmString");
  return(C_to_Xen_XmString(XmStringConcatAndFree(Xen_to_C_XmString(arg1), Xen_to_C_XmString(arg2))));
}

static Xen gxm_XmStringConcat(Xen arg1, Xen arg2)
{
  #define H_XmStringConcat "XmString XmStringConcat(XmString s1, XmString s2) appends one string to another"
  Xen_check_type(Xen_is_XmString(arg1), arg1, 1, "XmStringConcat", "XmString");
  Xen_check_type(Xen_is_XmString(arg2), arg2, 2, "XmStringConcat", "XmString");
  return(C_to_Xen_XmString(XmStringConcat(Xen_to_C_XmString(arg1), Xen_to_C_XmString(arg2))));
}

static Xen gxm_XmStringFreeContext(Xen arg1)
{
  #define H_XmStringFreeContext "void XmStringFreeContext(XmStringContext context) releases the string scanning \
context data structure"
  Xen_check_type(Xen_is_XmStringContext(arg1), arg1, 1, "XmStringFreeContext", "XmStringContext");
  XmStringFreeContext(Xen_to_C_XmStringContext(arg1));
  return(Xen_false);
}

static XmStringContext gxm_initxmsc;
static Xen gxm_XmStringInitContext(Xen arg2)
{
  #define H_XmStringInitContext "Boolean XmStringInitContext(XmString string) creates \
a data structure for scanning an XmString component by component"
  /* DIFF XmStringInitContext first arg omitted and rtn
   */
  int val;
  Xen_check_type(Xen_is_XmString(arg2), arg2, 1, "XmStringInitContext", "XmString");
  val = XmStringInitContext(&gxm_initxmsc, Xen_to_C_XmString(arg2));
  return(Xen_list_2(C_bool_to_Xen_boolean(val), C_to_Xen_XmStringContext(gxm_initxmsc)));
}

static Xen gxm_XmStringSeparatorCreate(void)
{
  #define H_XmStringSeparatorCreate "XmString XmStringSeparatorCreate(void) creates a compound string"
  return(C_to_Xen_XmString(XmStringSeparatorCreate()));
}

static Xen gxm_XmStringDirectionCreate(Xen arg1)
{
  #define H_XmStringDirectionCreate "XmString XmStringDirectionCreate(XmStringDirection direction) creates a compound string"
  Xen_check_type(Xen_is_integer(arg1), arg1, 1, "XmStringDirectionCreate", "int");
  return(C_to_Xen_XmString(XmStringDirectionCreate(Xen_integer_to_C_int(arg1))));
}

static Xen gxm_XmStringCreateLocalized(Xen arg1)
{
  #define H_XmStringCreateLocalized "XmString XmStringCreateLocalized(char *text) creates a compound string in the current locale"
  Xen_check_type(Xen_is_string(arg1), arg1, 1, "XmStringCreateLocalized", "String");
  return(C_to_Xen_XmString(XmStringCreateLocalized((char *)Xen_string_to_C_string(arg1))));
}

static Xen gxm_XmStringCreate(Xen arg1, Xen arg2)
{
  #define H_XmStringCreate "XmString XmStringCreate(char *text, char *tag) creates a compound string"
  Xen_check_type(Xen_is_string(arg1), arg1, 1, "XmStringCreate", "char*");
  Xen_check_type(Xen_is_string(arg2), arg2, 2, "XmStringCreate", "XmStringCharSet");
  return(C_to_Xen_XmString(XmStringCreate((char *)Xen_string_to_C_string(arg1), (char *)Xen_string_to_C_string(arg2))));
}

static Xen gxm_XmChangeColor(Xen arg1, Xen arg2)
{
  #define H_XmChangeColor "void XmChangeColor(Widget widget, Pixel background) recalculates all associated colors of a widget"
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XmChangeColor", "Widget");
  Xen_check_type(Xen_is_Pixel(arg2), arg2, 2, "XmChangeColor", "Pixel");
  XmChangeColor(Xen_to_C_Widget(arg1), Xen_to_C_Pixel(arg2));
  return(Xen_false);
}

static Xen gxm_XmGetColors(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XmGetColors "void XmGetColors(Screen *screen, Colormap colormap, Pixel background) generates foreground, select, and shadow colors"
  /* DIFF: XmGetColors omits trailing 4 args and returns them
   */
  Pixel fg,ts, bs, sr;
  Xen_check_type(Xen_is_Screen(arg1), arg1, 1, "XmGetColors", "Screen*");
  Xen_check_type(Xen_is_Colormap(arg2), arg2, 2, "XmGetColors", "Colormap");
  Xen_check_type(Xen_is_Pixel(arg3), arg3, 3, "XmGetColors", "Pixel");
  XmGetColors(Xen_to_C_Screen(arg1), 
	      Xen_to_C_Colormap(arg2), 
	      Xen_to_C_Pixel(arg3),
	      &fg, &ts, &bs, &sr);
  return(Xen_list_4(C_to_Xen_Pixel(fg),
		    C_to_Xen_Pixel(ts),
		    C_to_Xen_Pixel(bs),
		    C_to_Xen_Pixel(sr)));
}

static Xen gxm_XmGetColorCalculation(void)
{
  #define H_XmGetColorCalculation "XmColorProc XmGetColorCalculation(void) get the procedure used for default color calculation"
  return(xm_XmColorProc);
}

static Xen gxm_XmSetColorCalculation(Xen arg1)
{
  #define H_XmSetColorCalculation "XmColorProc XmSetColorCalculation(XmColorProc color_proc) set the procedure used for default color calculation"
  /* DIFF: XmSetColorCalculation NULL -> #f
   */
  if (Xen_is_procedure(xm_XmColorProc)) xm_unprotect(xm_XmColorProc);
  if (Xen_is_false(arg1))
    {
      xm_XmColorProc = Xen_false;
      XmSetColorCalculation(NULL);
    }
  else
    {
      Xen_check_type(Xen_is_procedure(arg1) && (Xen_is_aritable(arg1, 1)), arg1, 1, "XmSetColorCalculation", "(XmColorProc but 1 arg)");
      xm_protect(arg1);
      xm_XmColorProc = arg1;
      XmSetColorCalculation((XmColorProc)gxm_XmColorProc);
    }
  return(arg1);
}

static Xen gxm_XmTrackingEvent(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XmTrackingEvent "Widget XmTrackingEvent(Widget widget, Cursor cursor, Boolean confine_to): (widget event)"
  /* DIFF: XmTrackingEvent widget cursor confine [event] -> (list widget event)
   */
  XEvent *e; /* do we need to allocate? */
  Widget w;
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XmTrackingEvent", "Widget");
  Xen_check_type(Xen_is_Cursor(arg2), arg2, 2, "XmTrackingEvent", "Cursor");
  Xen_check_type(Xen_is_boolean(arg3), arg3, 3, "XmTrackingEvent", "boolean");
  e = (XEvent *)calloc(1, sizeof(XEvent));
  w = XmTrackingEvent(Xen_to_C_Widget(arg1), Xen_ulong_to_C_ulong(arg2), Xen_boolean_to_C_bool(arg3), e);
  return(Xen_list_2(C_to_Xen_Widget(w), C_to_Xen_XEvent_OBJ(e)));
}


/* The various XmVaCreateSimple{object} require special arg handling --
     these are not "normal" resource lists, but can also contain special indicators
     -- not immediately clear how to handle these things!
     Apparently we need to run through the args, split out the special args,
     collect arglists, add to arglist buttons, buttonCount etc, then call the
     underlying simple creator. (see lesstif lib/Xm/VaSimple.c)

   rowcol = XmVaCreateSimpleCheckBox(toplevel, "checkBox", cb,
				     XmNspacing, 2,
				     XmNmarginHeight, 4,
				     XmVaCHECKBUTTON, s1, 0, NULL, NULL,
				     XmVaCHECKBUTTON, s2, 0, NULL, NULL,
				     NULL);

   option = XmVaCreateSimpleOptionMenu(form, "option", cs("MyLabel"), 
				      (KeySym) 'M', 1, SimpleCallbackProc,
				      XmVaPUSHBUTTON, cs("Red"), (int) 'R', 
				      "", XmNULL,
				      XmVaCASCADEBUTTON, cs("Help"), (int) 'H',
				      XmVaSEPARATOR,
				      XmVaPUSHBUTTON, cs("Green"), (int) 'G', 
				      "", XmNULL,
				      XmVaSEPARATOR,
				      XmVaPUSHBUTTON, cs("Blue"), (int) 'B', 
				      "", XmNULL,
				      NULL);				    
*/

static Xen gxm_XmVaCreateSimpleCheckBox(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XmVaCreateSimpleCheckBox "Widget XmVaCreateSimpleCheckBox(Widget parent, String name, XtCallbackProc callback, args)"
  Widget w;
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XmVaCreateSimpleCheckBox", "Widget");
  Xen_check_type(Xen_is_string(arg2), arg2, 2, "XmVaCreateSimpleCheckBox", "String");
  Xen_check_type(Xen_is_procedure(arg3) && (Xen_is_aritable(arg3, 3)), arg3, 3, "XmVaCreateSimpleCheckBox", "XtCallbackProc (3 args)");
  Xen_check_type(Xen_is_list(arg4), arg4, 4, "XmVaCreateSimpleCheckBox", "List");
  arg4 = Xen_cons_2(C_string_to_Xen_string(XmNvalueChangedCallback),
		    Xen_list_2(arg3, Xen_false), /* XtCallbackList technically */
		    arg4);
  {
    int arglen;
    Arg *args;
    args = Xen_to_C_Args(arg4);
    arglen = Xen_list_length(arg4) / 2;
    w = XmCreateSimpleCheckBox(Xen_to_C_Widget(arg1), 
			       (char *)Xen_string_to_C_string(arg2), 
			       args,
			       arglen);
    if (args)
      {
	fixup_args(w, args, arglen);
	free_args(args, arglen);
      }
  }
  return(C_to_Xen_Widget(w));

}

static Xen gxm_XmVaCreateSimpleRadioBox(Xen arg1, Xen arg2, Xen arg3, Xen arg4, Xen arg5)
{
  #define H_XmVaCreateSimpleRadioBox "Widget XmVaCreateSimpleRadioBox(Widget parent, String name, int button_set, XtCallbackProc callback, args)"
  Widget w;
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XmVaCreateSimpleRadioBox", "Widget");
  Xen_check_type(Xen_is_string(arg2), arg2, 2, "XmVaCreateSimpleRadioBox", "String");
  Xen_check_type(Xen_is_integer(arg3), arg3, 3, "XmVaCreateSimpleRadioBox", "int");
  Xen_check_type(Xen_is_procedure(arg4) && (Xen_is_aritable(arg4, 3)), arg4, 4, "XmVaCreateSimpleRadioBox", "XtCallbackProc (3 args");
  Xen_check_type(Xen_is_list(arg5), arg5, 5, "XmVaCreateSimpleRadioBox", "List");
  arg5 = Xen_cons_2(C_string_to_Xen_string(XmNvalueChangedCallback),
		    Xen_list_2(arg4, Xen_false),
		    arg5);
  /* can't handle button set here since the menuHistory resource is the widget allocated later */
  {
    int arglen;
    Arg *args;
    args = Xen_to_C_Args(arg5);
    arglen = Xen_list_length(arg5) / 2;
    w = XmCreateSimpleRadioBox(Xen_to_C_Widget(arg1), 
			       (char *)Xen_string_to_C_string(arg2), 
			       args,
			       arglen);
    if (args)
      {
	fixup_args(w, args, arglen);
	free_args(args, arglen);
      }
  }
  return(C_to_Xen_Widget(w));
}

static Xen gxm_XmVaCreateSimpleOptionMenu(Xen arg1, Xen arg2, Xen arg3, Xen arg4, Xen arg5, Xen arg6, Xen arg7)
{
  #define H_XmVaCreateSimpleOptionMenu "Widget XmVaCreateSimpleOptionMenu(Widget parent, String name, XmString option_label, \
KeySym option_mnemonic, int button_set, XtCallbackProc callback, args)"
  Widget w;
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XmVaCreateSimpleOptionMenu", "Widget");
  Xen_check_type(Xen_is_string(arg2), arg2, 2, "XmVaCreateSimpleOptionMenu", "String");
  Xen_check_type(Xen_is_XmString(arg3), arg3, 3, "XmVaCreateSimpleOptionMenu", "XmString");
  Xen_check_type(Xen_is_KeySym(arg4), arg4, 4, "XmVaCreateSimpleOptionMenu", "KeySym");
  Xen_check_type(Xen_is_integer(arg5), arg5, 5, "XmVaCreateSimpleOptionMenu", "int");
  Xen_check_type(Xen_is_procedure(arg6) && (Xen_is_aritable(arg6, 3)), arg6, 6, "XmVaCreateSimpleOptionMenu", "XtCallbackProc (3 args)");
  Xen_check_type(Xen_is_list(arg7), arg7, 7, "XmVaCreateSimpleOptionMenu", "List");
  arg7 = Xen_cons_2(C_string_to_Xen_string(XmNlabelString),
		    arg3,
		    Xen_cons_2(C_string_to_Xen_string(XmNmnemonic),
			       arg4,
			       Xen_cons_2(C_string_to_Xen_string(XmNvalueChangedCallback),
					  Xen_list_2(arg6, Xen_false),
					  arg7)));
  {
    int arglen;
    Arg *args;
    args = Xen_to_C_Args(arg7);
    arglen = Xen_list_length(arg7) / 2;
    w = XmCreateSimpleOptionMenu(Xen_to_C_Widget(arg1), 
				 (char *)Xen_string_to_C_string(arg2), 
				 args,
				 arglen);
    if (args)
      {
	fixup_args(w, args, arglen);
	free_args(args, arglen);
      }
  }
  return(C_to_Xen_Widget(w));
}

static Xen gxm_XmVaCreateSimplePulldownMenu(Xen arg1, Xen arg2, Xen arg3, Xen arg4, Xen arg5)
{
  #define H_XmVaCreateSimplePulldownMenu "Widget XmVaCreateSimplePulldownMenu(Widget parent, String name, int post_from_button, XtCallbackProc callback, args)"
  Widget w;
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XmVaCreateSimplePulldownMenu", "Widget");
  Xen_check_type(Xen_is_string(arg2), arg2, 2, "XmVaCreateSimplePulldownMenu", "String");
  Xen_check_type(Xen_is_integer(arg3), arg3, 3, "XmVaCreateSimplePulldownMenu", "int");
  Xen_check_type(Xen_is_procedure(arg4) && (Xen_is_aritable(arg4, 3)), arg4, 4, "XmVaCreateSimplePulldownMenu", "XtCallbackProc (3 args)");
  Xen_check_type(Xen_is_list(arg5), arg5, 5, "XmVaCreateSimplePulldownMenu", "List");
  arg5 = Xen_cons_2(C_string_to_Xen_string(XmNvalueChangedCallback),
		    Xen_list_2(arg4, Xen_false),
		    arg5);
  {
    int arglen;
    Arg *args;
    args = Xen_to_C_Args(arg5);
    arglen = Xen_list_length(arg5) / 2;
    w = XmCreateSimplePulldownMenu(Xen_to_C_Widget(arg1), 
				   (char *)Xen_string_to_C_string(arg2), 
				   args,
				   arglen);
    if (args)
      {
	fixup_args(w, args, arglen);
	free_args(args, arglen);
      }
  }
  return(C_to_Xen_Widget(w));
}

static Xen gxm_XmVaCreateSimplePopupMenu(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XmVaCreateSimplePopupMenu "Widget XmVaCreateSimplePopupMenu(Widget parent, String name, XtCallbackProc callback, args)"
  Widget w;
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XmVaCreateSimplePopupMenu", "Widget");
  Xen_check_type(Xen_is_string(arg2), arg2, 2, "XmVaCreateSimplePopupMenu", "String");
  Xen_check_type(Xen_is_procedure(arg3) && (Xen_is_aritable(arg3, 3)), arg3, 3, "XmVaCreateSimplePopupMenu", "XtCallbackProc (3 args)");
  Xen_check_type(Xen_is_list(arg4), arg4, 4, "XmVaCreateSimplePopupMenu", "List");
  arg4 = Xen_cons_2(C_string_to_Xen_string(XmNvalueChangedCallback),
		    Xen_list_2(arg3, Xen_false),
		    arg4);
  {
    int arglen;
    Arg *args;
    args = Xen_to_C_Args(arg4);
    arglen = Xen_list_length(arg4) / 2;
    w = XmCreateSimplePopupMenu(Xen_to_C_Widget(arg1), 
				(char *)Xen_string_to_C_string(arg2), 
				args,
				arglen);
    if (args)
      {
	fixup_args(w, args, arglen);
	free_args(args, arglen);
      }
  }
  return(C_to_Xen_Widget(w));
}

static Xen gxm_XmVaCreateSimpleMenuBar(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XmVaCreateSimpleMenuBar "Widget XmVaCreateSimpleMenuBar(Widget parent, String name, args)"
  Widget w;
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XmVaCreateSimpleMenuBar", "Widget");
  Xen_check_type(Xen_is_string(arg2), arg2, 2, "XmVaCreateSimpleMenuBar", "String");
  Xen_check_type(Xen_is_list(arg3), arg3, 3, "XmVaCreateSimpleMenuBar", "List");
  {
    int arglen;
    Arg *args;
    args = Xen_to_C_Args(arg3);
    arglen = Xen_list_length(arg3) / 2;
    w = XmVaCreateSimpleMenuBar(Xen_to_C_Widget(arg1), 
				(char *)Xen_string_to_C_string(arg2),
				args,
				arglen);
    if (args)
      {
	fixup_args(w, args, arglen);
	free_args(args, arglen);
      }
  }
  return(C_to_Xen_Widget(w));
}

static Xen gxm_XmCreateSimpleCheckBox(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XmCreateSimpleCheckBox "Widget XmCreateSimpleCheckBox(Widget parent, String name, ArgList arglist, Cardinal argcount)"
  return(gxm_new_widget("XmCreateSimpleCheckBox", XmCreateSimpleCheckBox, arg1, arg2, arg3, arg4));
}

static Xen gxm_XmCreateSimpleRadioBox(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XmCreateSimpleRadioBox "Widget XmCreateSimpleRadioBox(Widget parent, String name, ArgList arglist, Cardinal argcount)"
  return(gxm_new_widget("XmCreateSimpleRadioBox", XmCreateSimpleRadioBox, arg1, arg2, arg3, arg4));
}

static Xen gxm_XmCreateSimpleOptionMenu(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XmCreateSimpleOptionMenu "Widget XmCreateSimpleOptionMenu(Widget parent, String name, ArgList arglist, Cardinal argcount)"
  return(gxm_new_widget("XmCreateSimpleOptionMenu", XmCreateSimpleOptionMenu, arg1, arg2, arg3, arg4));
}

static Xen gxm_XmCreateSimplePulldownMenu(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XmCreateSimplePulldownMenu "Widget XmCreateSimplePulldownMenu(Widget parent, String name, ArgList arglist, Cardinal argcount)"
  return(gxm_new_widget("XmCreateSimplePulldownMenu", XmCreateSimplePulldownMenu, arg1, arg2, arg3, arg4));
}

static Xen gxm_XmCreateSimplePopupMenu(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XmCreateSimplePopupMenu "Widget XmCreateSimplePopupMenu(Widget parent, String name, ArgList arglist, Cardinal argcount)"
  return(gxm_new_widget("XmCreateSimplePopupMenu", XmCreateSimplePopupMenu, arg1, arg2, arg3, arg4));
}

static Xen gxm_XmCreateSimpleMenuBar(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XmCreateSimpleMenuBar "Widget XmCreateSimpleMenuBar(Widget parent, String name, ArgList arglist, Cardinal argcount)"
  return(gxm_new_widget("XmCreateSimpleMenuBar", XmCreateSimpleMenuBar, arg1, arg2, arg3, arg4));
}

static Xen gxm_XmConvertUnits(Xen arg1, Xen arg2, Xen arg3, Xen arg4, Xen arg5)
{
  #define H_XmConvertUnits "int XmConvertUnits(Widget widget, int orientation, int from_unit_type, int from_value, int to_unit_type) \
converts a value in one unit type to another unit type"
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XmConvertUnits", "Widget");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XmConvertUnits", "int");
  Xen_check_type(Xen_is_integer(arg3), arg3, 3, "XmConvertUnits", "register");
  Xen_check_type(Xen_is_integer(arg4), arg4, 4, "XmConvertUnits", "register");
  Xen_check_type(Xen_is_integer(arg5), arg5, 5, "XmConvertUnits", "register");
  return(C_int_to_Xen_integer(XmConvertUnits(Xen_to_C_Widget(arg1), Xen_integer_to_C_int(arg2), Xen_integer_to_C_int(arg3), Xen_integer_to_C_int(arg4), Xen_integer_to_C_int(arg5))));
}

static Xen gxm_XmConvertStringToUnits(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XmConvertStringToUnits "int XmConvertStringToUnits(Screen *screen, String spec, int orientation, int to_type, XtEnum *parse_error) \
converts a string specification to a unit value"
  /* DIFF: XmConvertStringToUnits scr spec orient type [err] -> #f (err) or val
   */
  XtEnum err = 0;
  int val;
  Xen_check_type(Xen_is_Screen(arg1), arg1, 1, "XmConvertStringToUnits", "Screen*");
  Xen_check_type(Xen_is_string(arg2), arg2, 2, "XmConvertStringToUnits", "String");
  Xen_check_type(Xen_is_integer(arg3), arg3, 3, "XmConvertStringToUnits", "int");
  Xen_check_type(Xen_is_integer(arg4), arg4, 4, "XmConvertStringToUnits", "int");
  val = XmConvertStringToUnits(Xen_to_C_Screen(arg1), (char *)Xen_string_to_C_string(arg2), Xen_integer_to_C_int(arg3), Xen_integer_to_C_int(arg4), &err);
  if (err == 0)
    return(C_int_to_Xen_integer(val));
  return(Xen_false);
}

static Xen gxm_XmCvtXmStringToCT(Xen arg1)
{
  #define H_XmCvtXmStringToCT "char *XmCvtXmStringToCT(XmString string) converts a compound string to compound text"
  Xen_check_type(Xen_is_XmString(arg1), arg1, 1, "XmCvtXmStringToCT", "XmString");
  return(C_string_to_Xen_string(XmCvtXmStringToCT(Xen_to_C_XmString(arg1))));
  /* memory leak here, but the docs don't say that I should free the string of "compound text" */
}

static Xen gxm_XmCvtCTToXmString(Xen arg1)
{
  #define H_XmCvtCTToXmString "XmString XmCvtCTToXmString(char *text) converts compound text to a compound string"
  Xen_check_type(Xen_is_string(arg1), arg1, 1, "XmCvtCTToXmString", "char*");
  return(C_to_Xen_XmString(XmCvtCTToXmString((char *)Xen_string_to_C_string(arg1))));
}

static Xen gxm_XmMapSegmentEncoding(Xen arg1)
{
  char *str;
  Xen res;
  #define H_XmMapSegmentEncoding "char *XmMapSegmentEncoding(char *fontlist_tag): returns the compound text \
encoding format associated with the specified font list tag"
  Xen_check_type(Xen_is_string(arg1), arg1, 1, "XmMapSegmentEncoding", "char*");
  str = XmMapSegmentEncoding((char *)Xen_string_to_C_string(arg1));
  res = C_string_to_Xen_string(str);
  if (str) XtFree(str);
  return(res);
}

static Xen gxm_XmRegisterSegmentEncoding(Xen arg1, Xen arg2)
{
  char *str;
  Xen res;
  #define H_XmRegisterSegmentEncoding "char *XmRegisterSegmentEncoding(char *fontlist_tag, char *ct_encoding) \
registers a compound text encoding format for a specified font list element tag"
  Xen_check_type(Xen_is_string(arg1), arg1, 1, "XmRegisterSegmentEncoding", "char*");
  Xen_check_type(Xen_is_string(arg2), arg2, 2, "XmRegisterSegmentEncoding", "char*");
  str = XmRegisterSegmentEncoding((char *)Xen_string_to_C_string(arg1), (char *)Xen_string_to_C_string(arg2));
  res = C_string_to_Xen_string(str);
  if (str) XtFree(str);
  return(res);
}

static Xen gxm_XmWidgetGetBaselines(Xen arg1)
{
  #define H_XmWidgetGetBaselines "Boolean XmWidgetGetBaselines(Widget widget) retrieves baseline information for a widget"
  /* DIFF: XmWidgetGetBaselines omits args 2 and 3, returns list of Dimensions
   */
  Dimension *ds;
  int len, i, loc;
  Boolean b;
  Xen lst = Xen_empty_list;
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XmWidgetGetBaselines", "Widget");
  b = XmWidgetGetBaselines(Xen_to_C_Widget(arg1), &ds, &len);
  if (!b)
    return(Xen_false);
  loc = xm_protect(lst);
  for (i = len - 1; i >= 0; i--)
    lst = Xen_cons(C_to_Xen_Dimension(ds[i]), lst);
  free(ds);
  xm_unprotect_at(loc);
  return(lst);
}

static Xen gxm_XmWidgetGetDisplayRect(Xen arg1)
{
  #define H_XmWidgetGetDisplayRect "Boolean XmWidgetGetDisplayRect(Widget widget): returns widget's bounding box as XRectangle"
  XRectangle *r;
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XmWidgetGetDisplayRect", "Widget");
  r = (XRectangle *)calloc(1, sizeof(XRectangle));
  if (!(XmWidgetGetDisplayRect(Xen_to_C_Widget(arg1), r)))
    {
      free(r);
      return(Xen_false);
    }
  return(C_to_Xen_XRectangle(r));
}

static Xen gxm_XmObjectAtPoint(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XmObjectAtPoint "Widget XmObjectAtPoint(Widget widget, Position x, Position y) determines which child \
intersects or comes closest to a specified point"
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XmObjectAtPoint", "Widget");
  Xen_check_type(Xen_is_Position(arg2), arg2, 2, "XmObjectAtPoint", "Position");
  Xen_check_type(Xen_is_Position(arg3), arg3, 3, "XmObjectAtPoint", "Position");
  return(C_to_Xen_Widget(XmObjectAtPoint(Xen_to_C_Widget(arg1), Xen_to_C_Position(arg2), Xen_to_C_Position(arg3))));
}

static Xen gxm_XmUpdateDisplay(Xen arg1)
{
  #define H_XmUpdateDisplay "void XmUpdateDisplay (widget) processes all pending exposure events immediately"
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XmUpdateDisplay", "Widget");
  XmUpdateDisplay(Xen_to_C_Widget(arg1));
  return(Xen_false);
}

static Xen gxm_XmDestroyPixmap(Xen arg1, Xen arg2)
{
  #define H_XmDestroyPixmap "Boolean XmDestroyPixmap(Screen *screen, Pixmap pixmap) removes a pixmap from the pixmap cache"
  Xen_check_type(Xen_is_Screen(arg1), arg1, 1, "XmDestroyPixmap", "Screen*");
  Xen_check_type(Xen_is_Pixmap(arg2), arg2, 2, "XmDestroyPixmap", "Pixmap");
  return(C_bool_to_Xen_boolean(XmDestroyPixmap(Xen_to_C_Screen(arg1), 
					  Xen_to_C_Pixmap(arg2))));
}

static Xen gxm_XmGetPixmapByDepth(Xen arg1, Xen arg2, Xen arg3, Xen arg4, Xen arg5)
{
  #define H_XmGetPixmapByDepth "Pixmap XmGetPixmapByDepth(Screen *screen, char *image_name, Pixel foreground, Pixel background, int depth) \
generates a pixmap, stores it in a pixmap cache, and returns the pixmap"
  Xen_check_type(Xen_is_Screen(arg1), arg1, 1, "XmGetPixmapByDepth", "Screen*");
  Xen_check_type(Xen_is_string(arg2), arg2, 2, "XmGetPixmapByDepth", "char*");
  Xen_check_type(Xen_is_Pixel(arg3), arg3, 3, "XmGetPixmapByDepth", "Pixel");
  Xen_check_type(Xen_is_Pixel(arg4), arg4, 4, "XmGetPixmapByDepth", "Pixel");
  Xen_check_type(Xen_is_integer(arg5), arg5, 5, "XmGetPixmapByDepth", "int");
  return(C_to_Xen_Pixmap(XmGetPixmapByDepth(Xen_to_C_Screen(arg1), 
					    (char *)Xen_string_to_C_string(arg2), 
					    Xen_to_C_Pixel(arg3), 
					    Xen_to_C_Pixel(arg4), 
					    Xen_integer_to_C_int(arg5))));
}

static Xen gxm_XmGetPixmap(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XmGetPixmap "Pixmap XmGetPixmap(Screen *screen, char *image_name, Pixel foreground, Pixel background) A pixmap caching function \
that generates a pixmap, stores it in a pixmap cache, and returns the pixmap"
  Xen_check_type(Xen_is_Screen(arg1), arg1, 1, "XmGetPixmap", "Screen*");
  Xen_check_type(Xen_is_string(arg2), arg2, 2, "XmGetPixmap", "char*");
  Xen_check_type(Xen_is_Pixel(arg3), arg3, 3, "XmGetPixmap", "Pixel");
  Xen_check_type(Xen_is_Pixel(arg4), arg4, 4, "XmGetPixmap", "Pixel");
  return(C_to_Xen_Pixmap(XmGetPixmap(Xen_to_C_Screen(arg1), 
				     (char *)Xen_string_to_C_string(arg2), 
				     Xen_to_C_Pixel(arg3), 
				     Xen_to_C_Pixel(arg4))));
}

static Xen gxm_XmUninstallImage(Xen arg1)
{
  #define H_XmUninstallImage "Boolean XmUninstallImage(XImage *image) removes an image from the image cache"
  Xen_check_type(Xen_is_XImage(arg1), arg1, 1, "XmUninstallImage", "XImage*");
  return(C_bool_to_Xen_boolean(XmUninstallImage(Xen_to_C_XImage(arg1))));
}

static Xen gxm_XmInstallImage(Xen arg1, Xen arg2)
{
  #define H_XmInstallImage "Boolean XmInstallImage(XImage *image, char *image_name) adds an image to the image cache"
  Xen_check_type(Xen_is_XImage(arg1), arg1, 1, "XmInstallImage", "XImage*");
  Xen_check_type(Xen_is_string(arg2), arg2, 2, "XmInstallImage", "char*");
  return(C_bool_to_Xen_boolean(XmInstallImage(Xen_to_C_XImage(arg1), 
					 (char *)Xen_string_to_C_string(arg2))));
}

static Xen gxm_XmCreateMainWindow(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XmCreateMainWindow "Widget XmCreateMainWindow(Widget parent, String name, ArgList arglist, Cardinal argcount) \
The MainWindow widget creation function"
  return(gxm_new_widget("XmCreateMainWindow", XmCreateMainWindow, arg1, arg2, arg3, arg4));
}

static Xen gxm_XmTranslateKey(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XmTranslateKey "void XmTranslateKey(Display *display, KeyCode keycode, Modifiers modifiers) \
The default keycode-to-keysym translator -> (modifiers keysym)"
  /* DIFF: XmTranslateKey omit and rtn last 2 args
   */
  Modifiers m;
  KeySym k;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XmTranslateKey", "Display*");
  Xen_check_type(Xen_is_KeyCode(arg2), arg2, 2, "XmTranslateKey", "KeyCode");
  Xen_check_type(Xen_is_Modifiers(arg3), arg3, 3, "XmTranslateKey", "Modifiers");
  XmTranslateKey(Xen_to_C_Display(arg1), Xen_to_C_KeyCode(arg2), Xen_to_C_Modifiers(arg3), &m, &k);
  return(Xen_list_2(C_to_Xen_Modifiers(m),
		    C_to_Xen_KeySym(k)));
}

static Xen gxm_XmCreateScrolledList(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XmCreateScrolledList "Widget XmCreateScrolledList(Widget parent, String name, ArgList arglist, Cardinal argcount) The List \
ScrolledList creation function"
  return(gxm_new_widget("XmCreateScrolledList", XmCreateScrolledList, arg1, arg2, arg3, arg4));
}

static Xen gxm_XmCreateList(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XmCreateList "Widget XmCreateList(Widget parent, String name, ArgList arglist, Cardinal argcount) The List widget creation function"
  return(gxm_new_widget("XmCreateList", XmCreateList, arg1, arg2, arg3, arg4));
}

#define Xen_is_ListWidget(Arg) (Xen_is_Widget(Arg) && XmIsList(Xen_to_C_Widget(Arg)))

static Xen gxm_XmListPosSelected(Xen arg1, Xen arg2)
{
  #define H_XmListPosSelected "Boolean XmListPosSelected(Widget widget, int position) determines if the list item at a \
specified position is selected"
  Xen_check_type(Xen_is_ListWidget(arg1), arg1, 1, "XmListPosSelected", "List Widget");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XmListPosSelected", "int");
  return(C_bool_to_Xen_boolean(XmListPosSelected(Xen_to_C_Widget(arg1), Xen_integer_to_C_int(arg2))));
}

static Xen gxm_XmListUpdateSelectedList(Xen arg1)
{
  #define H_XmListUpdateSelectedList "void XmListUpdateSelectedList(Widget widget) updates the XmNselectedItems resource"
  Xen_check_type(Xen_is_ListWidget(arg1), arg1, 1, "XmListUpdateSelectedList", "List Widget");
  XmListUpdateSelectedList(Xen_to_C_Widget(arg1));
  return(Xen_false);
}

static Xen gxm_XmListSetHorizPos(Xen arg1, Xen arg2)
{
  #define H_XmListSetHorizPos "void XmListSetHorizPos(Widget widget, int position) scrolls to the specified position in the list"
  Xen_check_type(Xen_is_ListWidget(arg1), arg1, 1, "XmListSetHorizPos", "List Widget");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XmListSetHorizPos", "int");
  XmListSetHorizPos(Xen_to_C_Widget(arg1), Xen_integer_to_C_int(arg2));
  return(Xen_false);
}

static Xen gxm_XmListGetSelectedPos(Xen arg1)
{
  #define H_XmListGetSelectedPos "Boolean XmListGetSelectedPos(Widget widget) A List function that returns the position of every selected item in the list"
  /* DIFF: XmListGetSelectedPos omits args 2 and 3, returns list of positions
   */
  int *ps;
  int i, len, loc;
  Boolean b;
  Xen lst = Xen_empty_list;
  Xen_check_type(Xen_is_ListWidget(arg1), arg1, 1, "XmListGetSelectedPos", "List Widget");
  b = XmListGetSelectedPos(Xen_to_C_Widget(arg1), &ps, &len);
  if (!b)
    return(Xen_false);
  loc = xm_protect(lst);
  for (i = len - 1; i >= 0; i--)
    lst = Xen_cons(C_int_to_Xen_integer(ps[i]), lst);
  free(ps);
  xm_unprotect_at(loc);
  return(lst);
}

static Xen gxm_XmListGetMatchPos(Xen arg1, Xen arg2)
{
  #define H_XmListGetMatchPos "Boolean XmListGetMatchPos(Widget widget, XmString item): returns all instances of an item in the list"
  /* DIFF: XmListGetSelectedPos omits args 3 and 4, returns list of positions
   */
  int *ps;
  int i, len, loc;
  Boolean b;
  Xen lst = Xen_empty_list;
  Xen_check_type(Xen_is_ListWidget(arg1), arg1, 1, "XmListGetMatchPos", "List Widget");
  Xen_check_type(Xen_is_XmString(arg2), arg2, 2, "XmListGetMatchPos", "XmString");
  b = XmListGetMatchPos(Xen_to_C_Widget(arg1), Xen_to_C_XmString(arg2), &ps, &len);
  if (!b)
    return(Xen_false);
  loc = xm_protect(lst);
  for (i = len - 1; i >= 0; i--)
    lst = Xen_cons(C_int_to_Xen_integer(ps[i]), lst);
  free(ps);
  xm_unprotect_at(loc);
  return(lst);
}

static Xen gxm_XmListPosToBounds(Xen arg1, Xen arg2)
{
  #define H_XmListPosToBounds "Boolean XmListPosToBounds(Widget widget, int position): returns the bounding box of an item at a specified position in a list"
  /* DIFF: XmListPosToBounds last 4 args omitted and returned 
   */
  Position x, y;
  Dimension w, h;
  int val;
  Xen_check_type(Xen_is_ListWidget(arg1), arg1, 1, "XmListPosToBounds", "List Widget");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XmListPosToBounds", "int");
  val = XmListPosToBounds(Xen_to_C_Widget(arg1), Xen_integer_to_C_int(arg2), &x, &y, &w, &h);
  return(Xen_list_5(C_bool_to_Xen_boolean(val),
		    C_to_Xen_Position(x),
		    C_to_Xen_Position(y),
		    C_to_Xen_Dimension(w),
		    C_to_Xen_Dimension(h)));
}


static Xen gxm_XmListYToPos(Xen arg1, Xen arg2)
{
  #define H_XmListYToPos "int XmListYToPos(Widget widget, Position y): returns the position of the item at a specified y-coordinate"
  Xen_check_type(Xen_is_ListWidget(arg1), arg1, 1, "XmListYToPos", "List Widget");
  Xen_check_type(Xen_is_Position(arg2), arg2, 2, "XmListYToPos", "Position");
  return(C_int_to_Xen_integer(XmListYToPos(Xen_to_C_Widget(arg1), Xen_to_C_Position(arg2))));
}

static Xen gxm_XmListSetKbdItemPos(Xen arg1, Xen arg2)
{
  #define H_XmListSetKbdItemPos "Boolean XmListSetKbdItemPos(Widget widget, int position) sets the location cursor at a specified position"
  Xen_check_type(Xen_is_ListWidget(arg1), arg1, 1, "XmListSetKbdItemPos", "List Widget");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XmListSetKbdItemPos", "int");
  return(C_bool_to_Xen_boolean(XmListSetKbdItemPos(Xen_to_C_Widget(arg1), Xen_integer_to_C_int(arg2))));
}

static Xen gxm_XmListGetKbdItemPos(Xen arg1)
{
  #define H_XmListGetKbdItemPos "int XmListGetKbdItemPos(Widget widget): returns the position of the item at the location cursor"
  Xen_check_type(Xen_is_ListWidget(arg1), arg1, 1, "XmListGetKbdItemPos", "List Widget");
  return(C_int_to_Xen_integer(XmListGetKbdItemPos(Xen_to_C_Widget(arg1))));
}

static Xen gxm_XmListItemPos(Xen arg1, Xen arg2)
{
  #define H_XmListItemPos "int XmListItemPos(Widget widget, XmString item): returns the position of an item in the list"
  Xen_check_type(Xen_is_ListWidget(arg1), arg1, 1, "XmListItemPos", "List Widget");
  Xen_check_type(Xen_is_XmString(arg2), arg2, 2, "XmListItemPos", "XmString");
  return(C_int_to_Xen_integer(XmListItemPos(Xen_to_C_Widget(arg1), Xen_to_C_XmString(arg2))));
}

static Xen gxm_XmListItemExists(Xen arg1, Xen arg2)
{
  #define H_XmListItemExists "Boolean XmListItemExists(Widget widget, XmString item) checks if a specified item is in the list"
  Xen_check_type(Xen_is_ListWidget(arg1), arg1, 1, "XmListItemExists", "List Widget");
  Xen_check_type(Xen_is_XmString(arg2), arg2, 2, "XmListItemExists", "XmString");
  return(C_bool_to_Xen_boolean(XmListItemExists(Xen_to_C_Widget(arg1), Xen_to_C_XmString(arg2))));
}

static Xen gxm_XmListSetAddMode(Xen arg1, Xen arg2)
{
  #define H_XmListSetAddMode "void XmListSetAddMode(Widget widget, Boolean state) sets add mode in the list"
  Xen_check_type(Xen_is_ListWidget(arg1), arg1, 1, "XmListSetAddMode", "List Widget");
  Xen_check_type(Xen_is_boolean(arg2), arg2, 2, "XmListSetAddMode", "boolean");
  XmListSetAddMode(Xen_to_C_Widget(arg1), Xen_boolean_to_C_bool(arg2));
  return(Xen_false);
}

static Xen gxm_XmListSetBottomItem(Xen arg1, Xen arg2)
{
  #define H_XmListSetBottomItem "void XmListSetBottomItem(Widget widget, XmString item) makes an existing item the last \
visible item in the list"
  Xen_check_type(Xen_is_ListWidget(arg1), arg1, 1, "XmListSetBottomItem", "List Widget");
  Xen_check_type(Xen_is_XmString(arg2), arg2, 2, "XmListSetBottomItem", "XmString");
  XmListSetBottomItem(Xen_to_C_Widget(arg1), Xen_to_C_XmString(arg2));
  return(Xen_false);
}

static Xen gxm_XmListSetItem(Xen arg1, Xen arg2)
{
  #define H_XmListSetItem "void XmListSetItem(Widget widget, XmString item) makes an existing item the first visible item in the list"
  Xen_check_type(Xen_is_ListWidget(arg1), arg1, 1, "XmListSetItem", "List Widget");
  Xen_check_type(Xen_is_XmString(arg2), arg2, 2, "XmListSetItem", "XmString");
  XmListSetItem(Xen_to_C_Widget(arg1), Xen_to_C_XmString(arg2));
  return(Xen_false);
}

static Xen gxm_XmListSetBottomPos(Xen arg1, Xen arg2)
{
  #define H_XmListSetBottomPos "void XmListSetBottomPos(Widget widget, int position) makes a specified item the last visible item in the list"
  Xen_check_type(Xen_is_ListWidget(arg1), arg1, 1, "XmListSetBottomPos", "List Widget");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XmListSetBottomPos", "int");
  XmListSetBottomPos(Xen_to_C_Widget(arg1), Xen_integer_to_C_int(arg2));
  return(Xen_false);
}

static Xen gxm_XmListSetPos(Xen arg1, Xen arg2)
{
  #define H_XmListSetPos "void XmListSetPos(Widget widget, int position) makes the item at the given position the first \
visible position in the list"
  Xen_check_type(Xen_is_ListWidget(arg1), arg1, 1, "XmListSetPos", "List Widget");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XmListSetPos", "int");
  XmListSetPos(Xen_to_C_Widget(arg1), Xen_integer_to_C_int(arg2));
  return(Xen_false);
}

static Xen gxm_XmListDeselectAllItems(Xen arg1)
{
  #define H_XmListDeselectAllItems "void XmListDeselectAllItems(Widget widget) unhighlights and removes all items from the selected list"
  Xen_check_type(Xen_is_ListWidget(arg1), arg1, 1, "XmListDeselectAllItems", "List Widget");
  XmListDeselectAllItems(Xen_to_C_Widget(arg1));
  return(Xen_false);
}

static Xen gxm_XmListDeselectPos(Xen arg1, Xen arg2)
{
  #define H_XmListDeselectPos "void XmListDeselectPos(Widget widget, int position) deselects an item at a specified position in the list"
  Xen_check_type(Xen_is_ListWidget(arg1), arg1, 1, "XmListDeselectPos", "List Widget");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XmListDeselectPos", "int");
  XmListDeselectPos(Xen_to_C_Widget(arg1), Xen_integer_to_C_int(arg2));
  return(Xen_false);
}

static Xen gxm_XmListDeselectItem(Xen arg1, Xen arg2)
{
  #define H_XmListDeselectItem "void XmListDeselectItem(Widget widget, XmString item) deselects the specified item from the selected list"
  Xen_check_type(Xen_is_ListWidget(arg1), arg1, 1, "XmListDeselectItem", "List Widget");
  Xen_check_type(Xen_is_XmString(arg2), arg2, 2, "XmListDeselectItem", "XmString");
  XmListDeselectItem(Xen_to_C_Widget(arg1), Xen_to_C_XmString(arg2));
  return(Xen_false);
}

static Xen gxm_XmListSelectPos(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XmListSelectPos "void XmListSelectPos(Widget widget, int position, Boolean notify) selects an item at a specified \
position in the list"
  Xen_check_type(Xen_is_ListWidget(arg1), arg1, 1, "XmListSelectPos", "List Widget");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XmListSelectPos", "int");
  Xen_check_type(Xen_is_boolean(arg3), arg3, 3, "XmListSelectPos", "boolean");
  XmListSelectPos(Xen_to_C_Widget(arg1), Xen_integer_to_C_int(arg2), Xen_boolean_to_C_bool(arg3));
  return(Xen_false);
}

static Xen gxm_XmListSelectItem(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XmListSelectItem "void XmListSelectItem(Widget widget, XmString item, Boolean notify) selects an item in the list"
  Xen_check_type(Xen_is_ListWidget(arg1), arg1, 1, "XmListSelectItem", "List Widget");
  Xen_check_type(Xen_is_XmString(arg2), arg2, 2, "XmListSelectItem", "XmString");
  Xen_check_type(Xen_is_boolean(arg3), arg3, 3, "XmListSelectItem", "boolean");
  XmListSelectItem(Xen_to_C_Widget(arg1), Xen_to_C_XmString(arg2), Xen_boolean_to_C_bool(arg3));
  return(Xen_false);
}

static Xen gxm_XmListReplacePositions(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XmListReplacePositions "void XmListReplacePositions(Widget widget, int *position_list, XmString *item_list, int item_count) \
replaces items in a list based on position"
  /* DIFF: XmListReplacePositions arg 2 is list of ints, arg3 is list of XmStrings
   */
  int *ps;
  XmString *str;
  int len;
  Xen_check_type(Xen_is_ListWidget(arg1), arg1, 1, "XmListReplacePositions", "List Widget");
  Xen_check_type(Xen_is_list(arg2), arg2, 2, "XmListReplacePositions", "list of int");
  Xen_check_type(Xen_is_list(arg3), arg3, 3, "XmListReplacePositions", "list of XmString");
  Xen_check_type(Xen_is_integer_or_unbound(arg4), arg4, 4, "XmListReplacePositions", "int");
  if (Xen_is_integer(arg4)) len = Xen_integer_to_C_int(arg4); else len = Xen_list_length(arg3);
  if (len <= 0) return(Xen_false);
  ps = Xen_to_C_Ints(arg2, len);
  str = Xen_to_C_XmStrings(arg3, len);
  XmListReplacePositions(Xen_to_C_Widget(arg1), ps, str, len);
  free(ps);
  free(str);
  return(Xen_false);
}

static Xen gxm_XmListReplaceItemsPosUnselected(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XmListReplaceItemsPosUnselected "void XmListReplaceItemsPosUnselected(Widget widget, XmString *new_items, int item_count, int position) \
replaces items in a list without selecting the replacement items"
  /* DIFF: XmListReplaceItemsPosUnselected arg 2 is list of XmStrings
   */
  XmString *str;
  int len;
  Xen_check_type(Xen_is_ListWidget(arg1), arg1, 1, "XmListReplaceItemsPosUnselected", "List Widget");
  Xen_check_type(Xen_is_list(arg2), arg2, 2, "XmListReplaceItemsPosUnselected", "list of XmString");
  Xen_check_type(Xen_is_integer(arg3), arg3, 3, "XmListReplaceItemsPosUnselected", "int");
  Xen_check_type(Xen_is_integer(arg4), arg4, 4, "XmListReplaceItemsPosUnselected", "int");
  len = Xen_integer_to_C_int(arg3);
  if (len <= 0) return(Xen_false);
  str = Xen_to_C_XmStrings(arg2, len);
  XmListReplaceItemsPosUnselected(Xen_to_C_Widget(arg1), str, len, Xen_integer_to_C_int(arg4));
  free(str);
  return(Xen_false);
}

static Xen gxm_XmListReplaceItemsUnselected(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XmListReplaceItemsUnselected "void XmListReplaceItemsUnselected(Widget widget, XmString *old_items, int item_count, XmString *new_items) \
replaces items in a list"
  /* DIFF: XmListReplaceItemsUnselected args 2 and 4 are lists of XmStrings
   */
  XmString *str1, *str2;
  int len;
  Xen_check_type(Xen_is_ListWidget(arg1), arg1, 1, "XmListReplaceItemsUnselected", "List Widget");
  Xen_check_type(Xen_is_list(arg2), arg2, 2, "XmListReplaceItemsUnselected", "list of XmString");
  Xen_check_type(Xen_is_integer(arg3), arg3, 3, "XmListReplaceItemsUnselected", "int");
  Xen_check_type(Xen_is_list(arg4), arg4, 4, "XmListReplaceItemsUnselected", "list of XmString");
  len = Xen_integer_to_C_int(arg3);
  if (len <= 0) return(Xen_false);
  str1 = Xen_to_C_XmStrings(arg2, len);
  str2 = Xen_to_C_XmStrings(arg4, len);
  XmListReplaceItemsUnselected(Xen_to_C_Widget(arg1), str1, len, str2);
  free(str1);
  free(str2);
  return(Xen_false);
}

static Xen gxm_XmListReplaceItemsPos(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XmListReplaceItemsPos "void XmListReplaceItemsPos(Widget widget, XmString *new_items, int item_count, int position) \
replaces the specified elements in the list"
  /* DIFF: XmListReplaceItemsPos arg 2 is list of XmStrings
   */
  XmString *str;
  int len;
  Xen_check_type(Xen_is_ListWidget(arg1), arg1, 1, "XmListReplaceItemsPos", "List Widget");
  Xen_check_type(Xen_is_list(arg2), arg2, 2, "XmListReplaceItemsPos", "list of XmString");
  Xen_check_type(Xen_is_integer(arg3), arg3, 3, "XmListReplaceItemsPos", "int");
  Xen_check_type(Xen_is_integer(arg4), arg4, 4, "XmListReplaceItemsPos", "int");
  len = Xen_integer_to_C_int(arg3);
  if (len <= 0) return(Xen_false);
  str = Xen_to_C_XmStrings(arg2, len);
  XmListReplaceItemsPos(Xen_to_C_Widget(arg1), str, len, Xen_integer_to_C_int(arg4));
  free(str);
  return(Xen_false);
}

static Xen gxm_XmListReplaceItems(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XmListReplaceItems "void XmListReplaceItems(Widget widget, XmString *old_items, int item_count, XmString *new_items) \
replaces the specified elements in the list"
  /* DIFF: XmListReplaceItems args 2 and 4 are lists of XmStrings
   */
  XmString *str1, *str2;
  int len;
  Xen_check_type(Xen_is_ListWidget(arg1), arg1, 1, "XmListReplaceItems", "List Widget");
  Xen_check_type(Xen_is_list(arg2), arg2, 2, "XmListReplaceItems", "list of XmString");
  Xen_check_type(Xen_is_integer(arg3), arg3, 3, "XmListReplaceItems", "int");
  Xen_check_type(Xen_is_list(arg4), arg4, 4, "XmListReplaceItems", "XmString*");
  len = Xen_integer_to_C_int(arg3);
  if (len <= 0) return(Xen_false);
  str1 = Xen_to_C_XmStrings(arg2, len);
  str2 = Xen_to_C_XmStrings(arg4, len);
  XmListReplaceItems(Xen_to_C_Widget(arg1), str1, len, str2);
  free(str1);
  free(str2);
  return(Xen_false);
}

static Xen gxm_XmListDeleteAllItems(Xen arg1)
{
  #define H_XmListDeleteAllItems "void XmListDeleteAllItems(Widget widget) deletes all items from the list"
  Xen_check_type(Xen_is_ListWidget(arg1), arg1, 1, "XmListDeleteAllItems", "List Widget");
  XmListDeleteAllItems(Xen_to_C_Widget(arg1));
  return(Xen_false);
}

static Xen gxm_XmListDeleteItemsPos(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XmListDeleteItemsPos "void XmListDeleteItemsPos(Widget widget, int item_count, int position) deletes \
items from the list starting at the given position"
  Xen_check_type(Xen_is_ListWidget(arg1), arg1, 1, "XmListDeleteItemsPos", "List Widget");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XmListDeleteItemsPos", "int");
  Xen_check_type(Xen_is_integer(arg3), arg3, 3, "XmListDeleteItemsPos", "int");
  XmListDeleteItemsPos(Xen_to_C_Widget(arg1), Xen_integer_to_C_int(arg2), Xen_integer_to_C_int(arg3));
  return(Xen_false);
}

static Xen gxm_XmListDeletePos(Xen arg1, Xen arg2)
{
  #define H_XmListDeletePos "void XmListDeletePos(Widget widget, int position) deletes an item from a list at a specified position"
  Xen_check_type(Xen_is_ListWidget(arg1), arg1, 1, "XmListDeletePos", "List Widget");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XmListDeletePos", "int");
  XmListDeletePos(Xen_to_C_Widget(arg1), Xen_integer_to_C_int(arg2));
  return(Xen_false);
}

static Xen gxm_XmListDeletePositions(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XmListDeletePositions "void XmListDeletePositions(Widget widget, int *position_list, int position_count) deletes \
items from a list based on an array of positions"
  /* DIFF: XmListDeletePositions arg2 is list of ints
   */
  int *pos;
  int len;
  Xen_check_type(Xen_is_ListWidget(arg1), arg1, 1, "XmListDeletePositions", "List Widget");
  Xen_check_type(Xen_is_list(arg2), arg2, 2, "XmListDeletePositions", "list of int");
  Xen_check_type(Xen_is_integer_or_unbound(arg3), arg3, 3, "XmListDeletePositions", "int");
  if (Xen_is_integer(arg3)) len = Xen_integer_to_C_int(arg3); else len = Xen_list_length(arg2);
  if (len <= 0) return(Xen_false);
  pos = Xen_to_C_Ints(arg2, len);
  XmListDeletePositions(Xen_to_C_Widget(arg1), pos, len);
  free(pos);
  return(Xen_false);
}

static Xen gxm_XmListDeleteItems(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XmListDeleteItems "void XmListDeleteItems(Widget widget, XmString *items, int item_count) deletes items from the list"
  /* DIFF: XmListDeleteItems arg 2 is list of XmStrings
   */
  XmString *str;
  int len;
  Xen_check_type(Xen_is_ListWidget(arg1), arg1, 1, "XmListDeleteItems", "List Widget");
  Xen_check_type(Xen_is_list(arg2), arg2, 2, "XmListDeleteItems", "list of XmString");
  Xen_check_type(Xen_is_integer_or_unbound(arg3), arg3, 3, "XmListDeleteItems", "int");
  if (Xen_is_integer(arg3)) len = Xen_integer_to_C_int(arg3); else len = Xen_list_length(arg2);
  if (len <= 0) return(Xen_false);
  str = Xen_to_C_XmStrings(arg2, len);
  XmListDeleteItems(Xen_to_C_Widget(arg1), str, len);
  free(str);
  return(Xen_false);
}

static Xen gxm_XmListDeleteItem(Xen arg1, Xen arg2)
{
  #define H_XmListDeleteItem "void XmListDeleteItem(Widget widget, XmString item) deletes an item from the list"
  Xen_check_type(Xen_is_ListWidget(arg1), arg1, 1, "XmListDeleteItem", "List Widget");
  Xen_check_type(Xen_is_XmString(arg2), arg2, 2, "XmListDeleteItem", "XmString");
  XmListDeleteItem(Xen_to_C_Widget(arg1), Xen_to_C_XmString(arg2));
  return(Xen_false);
}

static Xen gxm_XmListAddItemUnselected(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XmListAddItemUnselected "void XmListAddItemUnselected(Widget widget, XmString item, int position) adds an item to the list"
  Xen_check_type(Xen_is_ListWidget(arg1), arg1, 1, "XmListAddItemUnselected", "List Widget");
  Xen_check_type(Xen_is_XmString(arg2), arg2, 2, "XmListAddItemUnselected", "XmString");
  Xen_check_type(Xen_is_integer(arg3), arg3, 3, "XmListAddItemUnselected", "int");
  XmListAddItemUnselected(Xen_to_C_Widget(arg1), Xen_to_C_XmString(arg2), Xen_integer_to_C_int(arg3));
  return(Xen_false);
}

static Xen gxm_XmListAddItemsUnselected(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XmListAddItemsUnselected "void XmListAddItemsUnselected(Widget widget, XmString *items, int item_count, int position) \
adds items to a list"
  /* DIFF: XmListAddItemsUnselected arg 2 is list of XmStrings
   */
  XmString *str;
  int len;
  Xen_check_type(Xen_is_ListWidget(arg1), arg1, 1, "XmListAddItemsUnselected", "List Widget");
  Xen_check_type(Xen_is_list(arg2), arg2, 2, "XmListAddItemsUnselected", "list of XmString");
  Xen_check_type(Xen_is_integer(arg3), arg3, 3, "XmListAddItemsUnselected", "int");
  Xen_check_type(Xen_is_integer(arg4), arg4, 4, "XmListAddItemsUnselected", "int");
  len = Xen_integer_to_C_int(arg3);
  if (len <= 0) return(Xen_false);
  str = Xen_to_C_XmStrings(arg2, len);
  XmListAddItemsUnselected(Xen_to_C_Widget(arg1), str, len, Xen_integer_to_C_int(arg4));
  free(str);
  return(Xen_false);
}

static Xen gxm_XmListAddItems(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XmListAddItems "void XmListAddItems(Widget widget, XmString *items, int item_count, int position) adds items to the list"
  /* DIFF: XmListAddItems arg 2 is list of XmStrings
   */
  XmString *str;
  int len;
  Xen_check_type(Xen_is_ListWidget(arg1), arg1, 1, "XmListAddItems", "List Widget");
  Xen_check_type(Xen_is_list(arg2), arg2, 2, "XmListAddItems", "XmString*");
  Xen_check_type(Xen_is_integer(arg3), arg3, 3, "XmListAddItems", "int");
  Xen_check_type(Xen_is_integer(arg4), arg4, 4, "XmListAddItems", "int");
  len = Xen_integer_to_C_int(arg3);
  if (len <= 0) return(Xen_false);
  str = Xen_to_C_XmStrings(arg2, len);
  XmListAddItems(Xen_to_C_Widget(arg1), str, len, Xen_integer_to_C_int(arg4));
  free(str);
  return(Xen_false);
}

static Xen gxm_XmListAddItem(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XmListAddItem "void XmListAddItem(Widget widget, XmString item, int position) adds an item to the list"
  Xen_check_type(Xen_is_ListWidget(arg1), arg1, 1, "XmListAddItem", "List Widget");
  Xen_check_type(Xen_is_XmString(arg2), arg2, 2, "XmListAddItem", "XmString");
  Xen_check_type(Xen_is_integer(arg3), arg3, 3, "XmListAddItem", "int");
  XmListAddItem(Xen_to_C_Widget(arg1), Xen_to_C_XmString(arg2), Xen_integer_to_C_int(arg3));
  return(Xen_false);
}

static Xen gxm_XmIsMotifWMRunning(Xen arg1)
{
  #define H_XmIsMotifWMRunning "Boolean XmIsMotifWMRunning(Widget shell)  determines whether the window manager is running"
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XmIsMotifWMRunning", "Widget");
  return(C_bool_to_Xen_boolean(XmIsMotifWMRunning(Xen_to_C_Widget(arg1))));
}

/* DIFF: all XmCreate<Obj> arglist is a list of args 
 */
static Xen gxm_XmCreateLabel(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XmCreateLabel "Widget XmCreateLabel(Widget parent, String name, ArgList arglist, Cardinal argcount) The Label widget creation function"
  return(gxm_new_widget("XmCreateLabel", XmCreateLabel, arg1, arg2, arg3, arg4));
}

static Xen gxm_XmCreateLabelGadget(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XmCreateLabelGadget "Widget XmCreateLabelGadget(Widget parent, String name, ArgList arglist, Cardinal argcount) The LabelGadget creation function"
  return(gxm_new_widget("XmCreateLabelGadget", XmCreateLabelGadget, arg1, arg2, arg3, arg4));
}

static Xen gxm_XmCreateIconHeader(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XmCreateIconHeader "Widget XmCreateIconHeader(Widget parent, String name, ArgList arglist, Cardinal argcount)"
  return(gxm_new_widget("XmCreateIconHeader", XmCreateIconHeader, arg1, arg2, arg3, arg4));
}

static Xen gxm_XmCreateIconGadget(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XmCreateIconGadget "Widget XmCreateIconGadget(Widget parent, String name, ArgList arglist, Cardinal argcount) \
The IconGadget widget creation function"
  return(gxm_new_widget("XmCreateIconGadget", XmCreateIconGadget, arg1, arg2, arg3, arg4));
}

static Xen gxm_XmCreateToggleButton(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XmCreateToggleButton "Widget XmCreateToggleButton(Widget parent, String name, ArgList arglist, Cardinal argcount) \
The ToggleButton widget creation function"
  return(gxm_new_widget("XmCreateToggleButton", XmCreateToggleButton, arg1, arg2, arg3, arg4));
}

#define Xen_is_ToggleButtonWidget(Arg) (Xen_is_Widget(Arg) && (XmIsToggleButton(Xen_to_C_Widget(Arg)) || XmIsToggleButtonGadget(Xen_to_C_Widget(Arg))))

static Xen gxm_XmToggleButtonSetValue(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XmToggleButtonSetValue "void XmToggleButtonSetValue(Widget widget, XmToggleButtonState state, Boolean notify) \
sets or changes the current state"
  Xen_check_type(Xen_is_ToggleButtonWidget(arg1), arg1, 1, "XmToggleButtonSetValue", "ToggleButton Widget");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XmToggleButtonSetValue", "int (actually XmToggleButtonState)");
  Xen_check_type(Xen_is_boolean(arg3), arg3, 3, "XmToggleButtonSetValue", "boolean");
  return(C_bool_to_Xen_boolean(XmToggleButtonSetValue(Xen_to_C_Widget(arg1), Xen_integer_to_C_int(arg2), Xen_boolean_to_C_bool(arg3))));
}

static Xen gxm_XmToggleButtonSetState(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XmToggleButtonSetState "void XmToggleButtonSetState(Widget widget, Boolean state, Boolean notify) \
sets or changes the current state"
  Xen_check_type(Xen_is_ToggleButtonWidget(arg1), arg1, 1, "XmToggleButtonSetState", "ToggleButton Widget");
  Xen_check_type(Xen_is_boolean(arg2), arg2, 2, "XmToggleButtonSetState", "boolean");
  Xen_check_type(Xen_is_boolean(arg3), arg3, 3, "XmToggleButtonSetState", "boolean");
  XmToggleButtonSetState(Xen_to_C_Widget(arg1), Xen_boolean_to_C_bool(arg2), Xen_boolean_to_C_bool(arg3));
  return(Xen_false);
}

static Xen gxm_XmToggleButtonGetState(Xen arg1)
{
  #define H_XmToggleButtonGetState "Boolean XmToggleButtonGetState(Widget widget) obtains the state of a ToggleButton"
  Xen_check_type(Xen_is_ToggleButtonWidget(arg1), arg1, 1, "XmToggleButtonGetState", "ToggleButton Widget");
  return(C_bool_to_Xen_boolean(XmToggleButtonGetState(Xen_to_C_Widget(arg1))));
}


static Xen gxm_XmCreateToggleButtonGadget(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XmCreateToggleButtonGadget "Widget XmCreateToggleButtonGadget(Widget parent, String name, ArgList arglist, Cardinal argcount) \
The ToggleButtonGadget creation function"
  return(gxm_new_widget("XmCreateToggleButtonGadget", XmCreateToggleButtonGadget, arg1, arg2, arg3, arg4));
}

static Xen gxm_XmToggleButtonGadgetSetValue(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XmToggleButtonGadgetSetValue "Boolean XmToggleButtonGadgetSetValue(Widget w, XmToggleButtonState newstate, Boolean notify)"
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XmToggleButtonGadgetSetValue", "Widget");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XmToggleButtonGadgetSetValue", "int");
  Xen_check_type(Xen_is_boolean(arg3), arg3, 3, "XmToggleButtonGadgetSetValue", "boolean");
  return(C_bool_to_Xen_boolean(XmToggleButtonGadgetSetValue(Xen_to_C_Widget(arg1), Xen_integer_to_C_int(arg2), Xen_boolean_to_C_bool(arg3))));
}

static Xen gxm_XmCreateGrabShell(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XmCreateGrabShell "Widget XmCreateGrabShell(Widget parent, char *name, ArgList al, Cardinal ac): a new GrabShell"
  return(gxm_new_widget("XmCreateGrabShell", XmCreateGrabShell, arg1, arg2, arg3, arg4));
}

static Xen gxm_XmToggleButtonGadgetSetState(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XmToggleButtonGadgetSetState "void XmToggleButtonGadgetSetState(Widget widget, Boolean state, Boolean notify) \
sets or changes the current state"
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XmToggleButtonGadgetSetState", "Widget");
  Xen_check_type(Xen_is_boolean(arg2), arg2, 2, "XmToggleButtonGadgetSetState", "boolean");
  Xen_check_type(Xen_is_boolean(arg3), arg3, 3, "XmToggleButtonGadgetSetState", "boolean");
  XmToggleButtonGadgetSetState(Xen_to_C_Widget(arg1), Xen_boolean_to_C_bool(arg2), Xen_boolean_to_C_bool(arg3));
  return(Xen_false);
}

static Xen gxm_XmToggleButtonGadgetGetState(Xen arg1)
{
  #define H_XmToggleButtonGadgetGetState "Boolean XmToggleButtonGadgetGetState(Widget widget) obtains \
the state of a ToggleButtonGadget"
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XmToggleButtonGadgetGetState", "Widget");
  return(C_bool_to_Xen_boolean(XmToggleButtonGadgetGetState(Xen_to_C_Widget(arg1))));
}

static Xen gxm_XmCreateFrame(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XmCreateFrame "Widget XmCreateFrame(Widget parent, String name, ArgList arglist, Cardinal argcount) The Frame widget creation function"
  return(gxm_new_widget("XmCreateFrame", XmCreateFrame, arg1, arg2, arg3, arg4));
}

static Xen gxm_XmCreateFormDialog(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XmCreateFormDialog "Widget XmCreateFormDialog(Widget parent, String name, ArgList arglist, Cardinal argcount) \
A Form FormDialog creation function"
  return(gxm_new_widget("XmCreateFormDialog", XmCreateFormDialog, arg1, arg2, arg3, arg4));
}

static Xen gxm_XmCreateForm(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XmCreateForm "Widget XmCreateForm(Widget parent, String name, ArgList arglist, Cardinal argcount) The Form widget creation function"
  return(gxm_new_widget("XmCreateForm", XmCreateForm, arg1, arg2, arg3, arg4));
}

#define Xen_is_TextWidget(Arg) (Xen_is_Widget(Arg) && (XmIsText(Xen_to_C_Widget(Arg)) || XmIsTextField(Xen_to_C_Widget(Arg))))
#define Xen_is_JustTextWidget(Arg) (Xen_is_Widget(Arg) && XmIsText(Xen_to_C_Widget(Arg)))

static Xen gxm_XmTextFindString(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XmTextFindString "Boolean XmTextFindString(Widget widget, XmTextPosition start, char *string, XmTextDirection direction) \
finds the beginning position of a text string"
  /* DIFF: XmTextFindString widget start string dir [pos] -> pos or #f
  */
  XmTextPosition pos;
  int res;
  Xen_check_type(Xen_is_JustTextWidget(arg1), arg1, 1, "XmTextFindString", "Text or TextField Widget");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XmTextFindString", "XmTextPosition");
  Xen_check_type(Xen_is_string(arg3), arg3, 3, "XmTextFindString", "char*");
  Xen_check_type(Xen_is_integer(arg4), arg4, 4, "XmTextFindString", "XmTextDirection");
  res = XmTextFindString(Xen_to_C_Widget(arg1), 
			 (XmTextPosition)Xen_integer_to_C_int(arg2), 
			 (char *)Xen_string_to_C_string(arg3), 
			 (XmTextDirection)Xen_integer_to_C_int(arg4), 
			 &pos);
  if (res)
    return(C_int_to_Xen_integer(pos));
  return(Xen_false);
}

static Xen gxm_XmTextEnableRedisplay(Xen arg1)
{
  #define H_XmTextEnableRedisplay "void XmTextEnableRedisplay(Widget widget) forces the visual update of a Text widget"
  Xen_check_type(Xen_is_JustTextWidget(arg1), arg1, 1, "XmTextEnableRedisplay", "Text or TextField Widget");
  XmTextEnableRedisplay(Xen_to_C_Widget(arg1));
  return(Xen_false);
}

static Xen gxm_XmTextDisableRedisplay(Xen arg1)
{
  #define H_XmTextDisableRedisplay "void XmTextDisableRedisplay(Widget widget) temporarily prevents visual update of the Text widget"
  Xen_check_type(Xen_is_JustTextWidget(arg1), arg1, 1, "XmTextDisableRedisplay", "Text or TextField Widget");
  XmTextDisableRedisplay(Xen_to_C_Widget(arg1));
  return(Xen_false);
}

static Xen gxm_XmTextGetCenterline(Xen arg1)
{
  #define H_XmTextGetCenterline "int XmTextGetCenterline(Widget widget) Return the height (length) of a character string when the writing direction is vertical"
  Xen_check_type(Xen_is_TextWidget(arg1), arg1, 1, "XmTextGetCenterline", "Text or TextField Widget");
  return(C_int_to_Xen_integer(XmTextGetCenterline(Xen_to_C_Widget(arg1))));
}

static Xen gxm_XmTextGetBaseline(Xen arg1)
{
  #define H_XmTextGetBaseline "int XmTextGetBaseline(Widget widget) accesses the y position of the baseline"
  Xen_check_type(Xen_is_TextWidget(arg1), arg1, 1, "XmTextGetBaseline", "Text or TextField Widget");
  return(C_int_to_Xen_integer(XmTextGetBaseline(Xen_to_C_Widget(arg1))));
}

static Xen gxm_XmTextScroll(Xen arg1, Xen arg2)
{
  #define H_XmTextScroll "void XmTextScroll(Widget widget, int lines) scrolls text"
  Xen_check_type(Xen_is_JustTextWidget(arg1), arg1, 1, "XmTextScroll", "Text or TextField Widget");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XmTextScroll", "int");
  XmTextScroll(Xen_to_C_Widget(arg1), Xen_integer_to_C_int(arg2));
  return(Xen_false);
}

static Xen gxm_XmTextShowPosition(Xen arg1, Xen arg2)
{
  #define H_XmTextShowPosition "void XmTextShowPosition(Widget widget, XmTextPosition position) forces text at a given position to be displayed"
  Xen_check_type(Xen_is_TextWidget(arg1), arg1, 1, "XmTextShowPosition", "Text or TextField Widget");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XmTextShowPosition", "XmTextPosition");
  XmTextShowPosition(Xen_to_C_Widget(arg1), Xen_integer_to_C_int(arg2));
  return(Xen_false);
}

static Xen gxm_XmTextSetSource(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XmTextSetSource "void XmTextSetSource(Widget widget, XmTextSource source, XmTextPosition top_character, \
XmTextPosition cursor_position) sets the source of the widget"
  Xen_check_type(Xen_is_JustTextWidget(arg1), arg1, 1, "XmTextSetSource", "Text or TextField Widget");
  Xen_check_type(Xen_is_XmTextSource(arg2), arg2, 2, "XmTextSetSource", "XmTextSource");
  Xen_check_type(Xen_is_integer(arg3), arg3, 3, "XmTextSetSource", "XmTextPosition");
  Xen_check_type(Xen_is_integer(arg4), arg4, 4, "XmTextSetSource", "XmTextPosition");
  XmTextSetSource(Xen_to_C_Widget(arg1), Xen_to_C_XmTextSource(arg2), Xen_integer_to_C_int(arg3), Xen_integer_to_C_int(arg4));
  return(Xen_false);
}

static Xen gxm_XmTextGetSource(Xen arg1)
{
  #define H_XmTextGetSource "XmTextSource XmTextGetSource(Widget widget) accesses the source of the widget"
  Xen_check_type(Xen_is_JustTextWidget(arg1), arg1, 1, "XmTextGetSource", "Text or TextField Widget");
  return(C_to_Xen_XmTextSource(XmTextGetSource(Xen_to_C_Widget(arg1))));
}

static Xen gxm_XmTextPosToXY(Xen arg1, Xen arg2)
{
  #define H_XmTextPosToXY "Boolean XmTextPosToXY(Widget widget, XmTextPosition position) A Text function \
that returns the x and y position of a character position"
  /* DIFF: XmTextPosToXY omits last 2 args and returns them
   */
  Position x, y;
  int val;
  Xen_check_type(Xen_is_TextWidget(arg1), arg1, 1, "XmTextPosToXY", "Text or TextField Widget");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XmTextPosToXY", "XmTextPosition");
  val = XmTextPosToXY(Xen_to_C_Widget(arg1), Xen_integer_to_C_int(arg2), &x, &y);
  return(Xen_list_3(C_bool_to_Xen_boolean(val),
		    C_to_Xen_Position(x),
		    C_to_Xen_Position(y)));
}

static Xen gxm_XmTextXYToPos(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XmTextXYToPos "XmTextPosition XmTextXYToPos(Widget widget, Position x, Position y) accesses \
the character position nearest an x and y position"
  Xen_check_type(Xen_is_TextWidget(arg1), arg1, 1, "XmTextXYToPos", "Text or TextField Widget");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XmTextXYToPos", "int");
  Xen_check_type(Xen_is_integer(arg3), arg3, 3, "XmTextXYToPos", "int");
  return(C_int_to_Xen_integer(XmTextXYToPos(Xen_to_C_Widget(arg1), Xen_integer_to_C_int(arg2), Xen_integer_to_C_int(arg3))));
}

static Xen gxm_XmTextGetSelectionPosition(Xen arg1)
{
  #define H_XmTextGetSelectionPosition "Boolean XmTextGetSelectionPosition(Widget widget): returns the position of the primary selection"
  /* DIFF: XmTextGetSelectionPosition widget [left right] -> (list res left right)
     no arg2 arg3, returns (owner left right) instead */
  XmTextPosition pos1, pos2;
  int res;
  Xen_check_type(Xen_is_TextWidget(arg1), arg1, 1, "XmTextGetSelectionPosition", "Text or TextField Widget");
  res = XmTextGetSelectionPosition(Xen_to_C_Widget(arg1), &pos1, &pos2);
  return(Xen_list_3(C_bool_to_Xen_boolean(res),
		    C_int_to_Xen_integer(pos1),
		    C_int_to_Xen_integer(pos2)));
}

static Xen gxm_XmTextClearSelection(Xen arg1, Xen arg2)
{
  #define H_XmTextClearSelection "void XmTextClearSelection(Widget widget, Time time) clears the primary selection"
  Xen_check_type(Xen_is_TextWidget(arg1), arg1, 1, "XmTextClearSelection", "Text or TextField Widget");
  Xen_check_type(Xen_is_Time(arg2), arg2, 2, "XmTextClearSelection", "Time");
  XmTextClearSelection(Xen_to_C_Widget(arg1), Xen_to_C_Time(arg2));
  return(Xen_false);
}

static Xen gxm_XmTextSetSelection(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XmTextSetSelection "void XmTextSetSelection(Widget widget, XmTextPosition first, XmTextPosition last, Time time) \
sets the primary selection of the text"
  Xen_check_type(Xen_is_TextWidget(arg1), arg1, 1, "XmTextSetSelection", "Text or TextField Widget");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XmTextSetSelection", "XmTextPosition");
  Xen_check_type(Xen_is_integer(arg3), arg3, 3, "XmTextSetSelection", "XmTextPosition");
  Xen_check_type(Xen_is_Time(arg4), arg4, 4, "XmTextSetSelection", "Time");
  XmTextSetSelection(Xen_to_C_Widget(arg1), Xen_integer_to_C_int(arg2), Xen_integer_to_C_int(arg3), Xen_to_C_Time(arg4));
  return(Xen_false);
}

static Xen gxm_XmTextGetSelection(Xen arg1)
{
  char *str;
  Xen res;
  #define H_XmTextGetSelection "char *XmTextGetSelection(Widget widget) retrieves the value of the primary selection"
  Xen_check_type(Xen_is_TextWidget(arg1), arg1, 1, "XmTextGetSelection", "Text or TextField Widget");
  str = XmTextGetSelection(Xen_to_C_Widget(arg1));
  res = C_string_to_Xen_string(str);
  if (str) XtFree(str);
  return(res);
}

static Xen gxm_XmTextPasteLink(Xen arg1)
{
  #define H_XmTextPasteLink "Boolean XmTextPasteLink(Widget widget) inserts a link to the clipboard selection"
  Xen_check_type(Xen_is_TextWidget(arg1), arg1, 1, "XmTextPasteLink", "Text or TextField Widget");
  return(C_bool_to_Xen_boolean(XmTextPasteLink(Xen_to_C_Widget(arg1))));
}

static Xen gxm_XmTextPaste(Xen arg1)
{
  #define H_XmTextPaste "Boolean XmTextPaste(Widget widget) inserts the clipboard selection"
  Xen_check_type(Xen_is_TextWidget(arg1), arg1, 1, "XmTextPaste", "Text or TextField Widget");
  return(C_bool_to_Xen_boolean(XmTextPaste(Xen_to_C_Widget(arg1))));
}

static Xen gxm_XmTextCut(Xen arg1, Xen arg2)
{
  #define H_XmTextCut "Boolean XmTextCut(Widget widget, Time time) copies the primary selection to the clipboard and deletes the selected text"
  Xen_check_type(Xen_is_TextWidget(arg1), arg1, 1, "XmTextCut", "Text or TextField Widget");
  Xen_check_type(Xen_is_Time(arg2), arg2, 2, "XmTextCut", "Time");
  return(C_bool_to_Xen_boolean(XmTextCut(Xen_to_C_Widget(arg1), Xen_to_C_Time(arg2))));
}

static Xen gxm_XmTextCopyLink(Xen arg1, Xen arg2)
{
  #define H_XmTextCopyLink "Boolean XmTextCopyLink(Widget widget, Time time) copies a link to the primary selection to the clipboard"
  Xen_check_type(Xen_is_TextWidget(arg1), arg1, 1, "XmTextCopyLink", "Text or TextField Widget");
  Xen_check_type(Xen_is_Time(arg2), arg2, 2, "XmTextCopyLink", "Time");
  return(C_bool_to_Xen_boolean(XmTextCopyLink(Xen_to_C_Widget(arg1), Xen_to_C_Time(arg2))));
}

static Xen gxm_XmTextCopy(Xen arg1, Xen arg2)
{
  #define H_XmTextCopy "Boolean XmTextCopy(Widget widget, Time time) copies the primary selection to the clipboard"
  Xen_check_type(Xen_is_TextWidget(arg1), arg1, 1, "XmTextCopy", "Text or TextField Widget");
  Xen_check_type(Xen_is_Time(arg2), arg2, 2, "XmTextCopy", "Time");
  return(C_bool_to_Xen_boolean(XmTextCopy(Xen_to_C_Widget(arg1), Xen_to_C_Time(arg2))));
}

static Xen gxm_XmTextRemove(Xen arg1)
{
  #define H_XmTextRemove "Boolean XmTextRemove(Widget widget) deletes the primary selection"
  Xen_check_type(Xen_is_TextWidget(arg1), arg1, 1, "XmTextRemove", "Text or TextField Widget");
  return(C_bool_to_Xen_boolean(XmTextRemove(Xen_to_C_Widget(arg1))));
}

static Xen gxm_XmTextSetCursorPosition(Xen arg1, Xen arg2)
{
  #define H_XmTextSetCursorPosition "void XmTextSetCursorPosition(Widget w, int position) sets the insertion cursor position"
  Xen_check_type(Xen_is_TextWidget(arg1), arg1, 1, "XmTextSetCursorPosition", "Text or TextField Widget");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XmTextSetCursorPosition", "XmTextPosition");
  XmTextSetCursorPosition(Xen_to_C_Widget(arg1), Xen_integer_to_C_int(arg2));
  return(Xen_false);
}

static Xen gxm_XmTextSetInsertionPosition(Xen arg1, Xen arg2)
{
  #define H_XmTextSetInsertionPosition "void XmTextSetInsertionPosition(Widget widget, XmTextPosition position) \
sets the position of the insert cursor"
  Xen_check_type(Xen_is_TextWidget(arg1), arg1, 1, "XmTextSetInsertionPosition", "Text or TextField Widget");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XmTextSetInsertionPosition", "XmTextPosition");
  XmTextSetInsertionPosition(Xen_to_C_Widget(arg1), Xen_integer_to_C_int(arg2));
  return(Xen_false);
}

static Xen gxm_XmTextGetInsertionPosition(Xen arg1)
{
  #define H_XmTextGetInsertionPosition "XmTextPosition XmTextGetInsertionPosition(Widget widget) accesses the \
position of the insert cursor"
  Xen_check_type(Xen_is_TextWidget(arg1), arg1, 1, "XmTextGetInsertionPosition", "Text or TextField Widget");
  return(C_int_to_Xen_integer(XmTextGetInsertionPosition(Xen_to_C_Widget(arg1))));
}

static Xen gxm_XmTextGetCursorPosition(Xen arg1)
{
  #define H_XmTextGetCursorPosition "int XmTextGetCursorPosition(Widget w) presumably returns the widget's insertion cursor position"
  Xen_check_type(Xen_is_TextWidget(arg1), arg1, 1, "XmTextGetCursorPosition", "Text or TextField Widget");
  return(C_int_to_Xen_integer(XmTextGetCursorPosition(Xen_to_C_Widget(arg1))));
}

static Xen gxm_XmTextSetTopCharacter(Xen arg1, Xen arg2)
{
  #define H_XmTextSetTopCharacter "void XmTextSetTopCharacter(Widget widget, XmTextPosition top_character) sets \
the position of the first character displayed"
  Xen_check_type(Xen_is_JustTextWidget(arg1), arg1, 1, "XmTextSetTopCharacter", "Text or TextField Widget");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XmTextSetTopCharacter", "XmTextPosition");
  XmTextSetTopCharacter(Xen_to_C_Widget(arg1), Xen_integer_to_C_int(arg2));
  return(Xen_false);
}

static Xen gxm_XmTextGetTopCharacter(Xen arg1)
{
  #define H_XmTextGetTopCharacter "XmTextPosition XmTextGetTopCharacter(Widget widget) accesses the position of the first character displayed"
  Xen_check_type(Xen_is_JustTextWidget(arg1), arg1, 1, "XmTextGetTopCharacter", "Text or TextField Widget");
  return(C_int_to_Xen_integer(XmTextGetTopCharacter(Xen_to_C_Widget(arg1))));
}

static Xen gxm_XmTextSetMaxLength(Xen arg1, Xen arg2)
{
  #define H_XmTextSetMaxLength "void XmTextSetMaxLength(Widget widget, int max_length) sets the value of the current \
maximum allowable length of a text string entered from the keyboard"
  Xen_check_type(Xen_is_TextWidget(arg1), arg1, 1, "XmTextSetMaxLength", "Text or TextField Widget");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XmTextSetMaxLength", "int");
  XmTextSetMaxLength(Xen_to_C_Widget(arg1), Xen_integer_to_C_int(arg2));
  return(Xen_false);
}

static Xen gxm_XmTextGetMaxLength(Xen arg1)
{
  #define H_XmTextGetMaxLength "int XmTextGetMaxLength(Widget widget) accesses the value of the current maximum allowable \
length of a text string entered from the keyboard"
  Xen_check_type(Xen_is_TextWidget(arg1), arg1, 1, "XmTextGetMaxLength", "Text or TextField Widget");
  return(C_int_to_Xen_integer(XmTextGetMaxLength(Xen_to_C_Widget(arg1))));
}

static Xen gxm_XmTextSetEditable(Xen arg1, Xen arg2)
{
  #define H_XmTextSetEditable "void XmTextSetEditable(Widget widget, Boolean editable) sets the edit permission"
  Xen_check_type(Xen_is_TextWidget(arg1), arg1, 1, "XmTextSetEditable", "Text or TextField Widget");
  Xen_check_type(Xen_is_boolean(arg2), arg2, 2, "XmTextSetEditable", "boolean");
  XmTextSetEditable(Xen_to_C_Widget(arg1), Xen_boolean_to_C_bool(arg2));
  return(Xen_false);
}

static Xen gxm_XmTextGetEditable(Xen arg1)
{
  #define H_XmTextGetEditable "Boolean XmTextGetEditable(Widget widget) accesses the edit permission state"
  Xen_check_type(Xen_is_TextWidget(arg1), arg1, 1, "XmTextGetEditable", "Text or TextField Widget");
  return(C_bool_to_Xen_boolean(XmTextGetEditable(Xen_to_C_Widget(arg1))));
}

static Xen gxm_XmTextGetAddMode(Xen arg1)
{
  #define H_XmTextGetAddMode "Boolean XmTextGetAddMode(Widget w) presumably returns the widget's current add mode."
  Xen_check_type(Xen_is_TextWidget(arg1), arg1, 1, "XmTextGetAddMode", "Text or TextField Widget");
  return(C_bool_to_Xen_boolean(XmTextGetAddMode(Xen_to_C_Widget(arg1))));
}

static Xen gxm_XmTextSetAddMode(Xen arg1, Xen arg2)
{
  #define H_XmTextSetAddMode "void XmTextSetAddMode(Widget widget, Boolean state) sets the widget's add mode -- \
this determines whether you can move the insertion cursor without changing the primary selection"
  Xen_check_type(Xen_is_TextWidget(arg1), arg1, 1, "XmTextSetAddMode", "Text or TextField Widget");
  Xen_check_type(Xen_is_boolean(arg2), arg2, 2, "XmTextSetAddMode", "boolean");
  XmTextSetAddMode(Xen_to_C_Widget(arg1), Xen_boolean_to_C_bool(arg2));
  return(Xen_false);
}

static Xen gxm_XmTextInsert(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XmTextInsert "void XmTextInsert(Widget widget, XmTextPosition position, char *value) inserts a character \
string into a text string"
  Xen_check_type(Xen_is_TextWidget(arg1), arg1, 1, "XmTextInsert", "Text or TextField Widget");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XmTextInsert", "XmTextPosition");
  Xen_check_type(Xen_is_string(arg3), arg3, 3, "XmTextInsert", "char*");
  XmTextInsert(Xen_to_C_Widget(arg1), Xen_integer_to_C_int(arg2), (char *)Xen_string_to_C_string(arg3));
  return(Xen_false);
}

static Xen gxm_XmTextReplace(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XmTextReplace "void XmTextReplace(Widget widget, XmTextPosition from_pos, XmTextPosition to_pos, char *value) \
replaces part of a text string"
  Xen_check_type(Xen_is_TextWidget(arg1), arg1, 1, "XmTextReplace", "Text or TextField Widget");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XmTextReplace", "XmTextPosition");
  Xen_check_type(Xen_is_integer(arg3), arg3, 3, "XmTextReplace", "XmTextPosition");
  Xen_check_type(Xen_is_string(arg4), arg4, 4, "XmTextReplace", "char*");
  XmTextReplace(Xen_to_C_Widget(arg1), Xen_integer_to_C_int(arg2), Xen_integer_to_C_int(arg3), (char *)Xen_string_to_C_string(arg4));
  return(Xen_false);
}

static Xen gxm_XmTextSetString(Xen arg1, Xen arg2)
{
  #define H_XmTextSetString "void XmTextSetString(Widget widget, char *value) sets the string value"
  Xen_check_type(Xen_is_TextWidget(arg1), arg1, 1, "XmTextSetString", "Text or TextField Widget");
  Xen_check_type(Xen_is_string(arg2), arg2, 2, "XmTextSetString", "char*");
  XmTextSetString(Xen_to_C_Widget(arg1), (char *)Xen_string_to_C_string(arg2));
  return(Xen_false);
}

static Xen gxm_XmTextGetLastPosition(Xen arg1)
{
  #define H_XmTextGetLastPosition "XmTextPosition XmTextGetLastPosition(Widget widget) accesses the last position in the text"
  Xen_check_type(Xen_is_TextWidget(arg1), arg1, 1, "XmTextGetLastPosition", "Text or TextField Widget");
  return(C_int_to_Xen_integer(XmTextGetLastPosition(Xen_to_C_Widget(arg1))));
}

static Xen gxm_XmTextGetString(Xen arg1)
{
  char *str;
  Xen res;
  #define H_XmTextGetString "char *XmTextGetString(Widget widget) accesses the string value"
  Xen_check_type(Xen_is_TextWidget(arg1), arg1, 1, "XmTextGetString", "Text or TextField Widget");
  str = XmTextGetString(Xen_to_C_Widget(arg1));
  res = C_string_to_Xen_string(str);
  if (str) XtFree(str);
  return(res);
}

static Xen gxm_XmTextGetSubstring(Xen arg1, Xen arg2, Xen arg3)
{
  /* DIFF: omit and rtn last 2 args */
  #define H_XmTextGetSubstring "int XmTextGetSubstring(Widget widget, XmTextPosition start, int num_chars) \
retrieves a copy of a portion of the internal text buffer"
  int rtn, len;
  char *buf;
  Xen str = Xen_false;
  Xen_check_type(Xen_is_TextWidget(arg1), arg1, 1, "XmTextGetSubstring", "Text or TextField Widget");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XmTextGetSubstring", "XmTextPosition");
  Xen_check_type(Xen_is_integer(arg3), arg3, 3, "XmTextGetSubstring", "int");
  len = Xen_integer_to_C_int(arg3);
  buf = (char *)calloc(len + 1, sizeof(char));
  rtn = XmTextGetSubstring(Xen_to_C_Widget(arg1), Xen_integer_to_C_int(arg2), len, len + 1, buf);
  if (rtn != XmCOPY_FAILED)
    str = C_string_to_Xen_string(buf);
  free(buf);
  return(str);
}

static Xen gxm_XmCreateText(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XmCreateText "Widget XmCreateText(Widget parent, String name, ArgList arglist, Cardinal argcount) The Text widget creation function"
  return(gxm_new_widget("XmCreateText", XmCreateText, arg1, arg2, arg3, arg4));
}

static Xen gxm_XmCreateScrolledText(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XmCreateScrolledText "Widget XmCreateScrolledText(Widget parent, String name, ArgList arglist, Cardinal argcount) \
The Text ScrolledText creation function"
  return(gxm_new_widget("XmCreateScrolledText", XmCreateScrolledText, arg1, arg2, arg3, arg4));
}

static Xen gxm_XmTextSetHighlight(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XmTextSetHighlight "void XmTextSetHighlight(Widget widget, XmTextPosition left, XmTextPosition right, XmHighlightMode mode) \
highlights text"
  Xen_check_type(Xen_is_TextWidget(arg1), arg1, 1, "XmTextSetHighlight", "Text or TextField Widget");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XmTextSetHighlight", "XmTextPosition");
  Xen_check_type(Xen_is_integer(arg3), arg3, 3, "XmTextSetHighlight", "XmTextPosition");
  Xen_check_type(Xen_is_integer(arg4), arg4, 4, "XmTextSetHighlight", "XmHighlightMode");
  XmTextSetHighlight(Xen_to_C_Widget(arg1), 
		     (XmTextPosition)Xen_integer_to_C_int(arg2), 
		     (XmTextPosition)Xen_integer_to_C_int(arg3), 
		     (XmHighlightMode)Xen_integer_to_C_int(arg4));
  return(Xen_false);
}

static Xen gxm_XmCreateFileSelectionDialog(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XmCreateFileSelectionDialog "Widget XmCreateFileSelectionDialog(Widget parent, String name, ArgList arglist, Cardinal argcount) \
The FileSelectionBox FileSelectionDialog creation function"
  return(gxm_new_widget("XmCreateFileSelectionDialog", XmCreateFileSelectionDialog, arg1, arg2, arg3, arg4));
}

static Xen gxm_XmCreateFileSelectionBox(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XmCreateFileSelectionBox "Widget XmCreateFileSelectionBox(Widget parent, String name, ArgList arglist, Cardinal argcount) \
The FileSelectionBox widget creation function"
  return(gxm_new_widget("XmCreateFileSelectionBox", XmCreateFileSelectionBox, arg1, arg2, arg3, arg4));
}

static Xen gxm_XmFileSelectionDoSearch(Xen arg1, Xen arg2)
{
  #define H_XmFileSelectionDoSearch "void XmFileSelectionDoSearch(Widget widget, XmString dirmask) \
initiates a directory search"
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XmFileSelectionDoSearch", "Widget");
  Xen_check_type(Xen_is_XmString(arg2) || Xen_is_false(arg2), arg2, 2, "XmFileSelectionDoSearch", "XmString");
  XmFileSelectionDoSearch(Xen_to_C_Widget(arg1), Xen_is_false(arg2) ? NULL : Xen_to_C_XmString(arg2));
  return(Xen_false);
}

static Xen gxm_XmFileSelectionBoxGetChild(Xen arg1, Xen arg2)
{
  #define H_XmFileSelectionBoxGetChild "Widget XmFileSelectionBoxGetChild(Widget widget, unsigned char child) \
used to access a component"
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XmFileSelectionBoxGetChild", "Widget");
  Xen_check_type(Xen_is_ulong(arg2), arg2, 2, "XmFileSelectionBoxGetChild", "unsigned int");
  return(C_to_Xen_Widget(XmFileSelectionBoxGetChild(Xen_to_C_Widget(arg1), Xen_ulong_to_C_ulong(arg2))));
}

static Xen gxm_XmCreateTextField(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XmCreateTextField "Widget XmCreateTextField(Widget parent, String name, ArgList arglist, Cardinal argcount) \
The TextField widget creation function"
  return(gxm_new_widget("XmCreateTextField", XmCreateTextField, arg1, arg2, arg3, arg4));
}

#define Xen_is_TextFieldWidget(Arg) (Xen_is_Widget(Arg) && XmIsTextField(Xen_to_C_Widget(Arg)))

static Xen gxm_XmTextFieldGetBaseline(Xen arg1)
{
  #define H_XmTextFieldGetBaseline "int XmTextFieldGetBaseline(Widget widget) accesses the y position of the baseline"
  Xen_check_type(Xen_is_TextFieldWidget(arg1), arg1, 1, "XmTextFieldGetBaseline", "TextField Widget");
  return(C_int_to_Xen_integer(XmTextFieldGetBaseline(Xen_to_C_Widget(arg1))));
}

static Xen gxm_XmTextFieldSetHighlight(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XmTextFieldSetHighlight "void XmTextFieldSetHighlight(Widget widget, XmTextPosition left, XmTextPosition right, XmHighlightMode mode) \
highlights text"
  Xen_check_type(Xen_is_TextFieldWidget(arg1), arg1, 1, "XmTextFieldSetHighlight", "TextField Widget");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XmTextFieldSetHighlight", "XmTextPosition");
  Xen_check_type(Xen_is_integer(arg3), arg3, 3, "XmTextFieldSetHighlight", "XmTextPosition");
  Xen_check_type(Xen_is_integer(arg4), arg4, 4, "XmTextFieldSetHighlight", "XmHighlightMode");
  XmTextFieldSetHighlight(Xen_to_C_Widget(arg1), 
			  (XmTextPosition)Xen_integer_to_C_int(arg2), 
			  (XmTextPosition)Xen_integer_to_C_int(arg3), 
			  (XmHighlightMode)Xen_integer_to_C_int(arg4));
  return(Xen_false);
}

static Xen gxm_XmTextFieldShowPosition(Xen arg1, Xen arg2)
{
  #define H_XmTextFieldShowPosition "void XmTextFieldShowPosition(Widget widget, XmTextPosition position) \
forces text at a given position to be displayed"
  Xen_check_type(Xen_is_TextFieldWidget(arg1), arg1, 1, "XmTextFieldShowPosition", "TextField Widget");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XmTextFieldShowPosition", "XmTextPosition");
  XmTextFieldShowPosition(Xen_to_C_Widget(arg1), 
			  (XmTextPosition)Xen_integer_to_C_int(arg2));
  return(Xen_false);
}

static Xen gxm_XmTextFieldPosToXY(Xen arg1, Xen arg2)
{
  #define H_XmTextFieldPosToXY "Boolean XmTextFieldPosToXY(Widget widget, XmTextPosition position): returns the x and y position of a character position"
  /* DIFF: XmTextFieldPosToXY omits last 2 args and returns them
   */
  Position x, y;
  int val;
  Xen_check_type(Xen_is_TextFieldWidget(arg1), arg1, 1, "XmTextFieldPosToXY", "TextField Widget");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XmTextFieldPosToXY", "XmTextPosition");
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XmTextPosToXY", "Widget");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XmTextPosToXY", "XmTextPosition");
  val = XmTextFieldPosToXY(Xen_to_C_Widget(arg1), Xen_integer_to_C_int(arg2), &x, &y);
  return(Xen_list_3(C_bool_to_Xen_boolean(val),
		    C_to_Xen_Position(x),
		    C_to_Xen_Position(y)));
}

static Xen gxm_XmTextFieldXYToPos(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XmTextFieldXYToPos "XmTextPosition XmTextFieldXYToPos(Widget widget, Position x, Position y) \
accesses the character position nearest an x and y position"
  Xen_check_type(Xen_is_TextFieldWidget(arg1), arg1, 1, "XmTextFieldXYToPos", "TextField Widget");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XmTextFieldXYToPos", "int");
  Xen_check_type(Xen_is_integer(arg3), arg3, 3, "XmTextFieldXYToPos", "int");
  return(C_int_to_Xen_integer(XmTextFieldXYToPos(Xen_to_C_Widget(arg1), Xen_integer_to_C_int(arg2), Xen_integer_to_C_int(arg3))));
}

static Xen gxm_XmTextFieldSetSelection(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XmTextFieldSetSelection "void XmTextFieldSetSelection(Widget widget, XmTextPosition first, XmTextPosition last, Time time) \
sets the primary selection of the text"
  Xen_check_type(Xen_is_TextFieldWidget(arg1), arg1, 1, "XmTextFieldSetSelection", "TextField Widget");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XmTextFieldSetSelection", "XmTextPosition");
  Xen_check_type(Xen_is_integer(arg3), arg3, 3, "XmTextFieldSetSelection", "XmTextPosition");
  Xen_check_type(Xen_is_Time(arg4), arg4, 4, "XmTextFieldSetSelection", "Time");
  XmTextFieldSetSelection(Xen_to_C_Widget(arg1), Xen_integer_to_C_int(arg2), Xen_integer_to_C_int(arg3), Xen_to_C_Time(arg4));
  return(Xen_false);
}

static Xen gxm_XmTextFieldClearSelection(Xen arg1, Xen arg2)
{
  #define H_XmTextFieldClearSelection "void XmTextFieldClearSelection(Widget widget, Time time) clears the primary selection"
  Xen_check_type(Xen_is_TextFieldWidget(arg1), arg1, 1, "XmTextFieldClearSelection", "TextField Widget");
  Xen_check_type(Xen_is_Time(arg2), arg2, 2, "XmTextFieldClearSelection", "Time");
  XmTextFieldClearSelection(Xen_to_C_Widget(arg1), Xen_to_C_Time(arg2));
  return(Xen_false);
}

static Xen gxm_XmTextFieldPasteLink(Xen arg1)
{
  #define H_XmTextFieldPasteLink "Boolean XmTextFieldPasteLink(Widget widget) inserts a link to the clipboard selection"
  Xen_check_type(Xen_is_TextFieldWidget(arg1), arg1, 1, "XmTextFieldPasteLink", "TextField Widget");
  return(C_bool_to_Xen_boolean(XmTextFieldPasteLink(Xen_to_C_Widget(arg1))));
}

static Xen gxm_XmTextFieldCopyLink(Xen arg1, Xen arg2)
{
  #define H_XmTextFieldCopyLink "Boolean XmTextFieldCopyLink(Widget widget, Time time) copies a link to the \
primary selection to the clipboard"
  Xen_check_type(Xen_is_TextFieldWidget(arg1), arg1, 1, "XmTextFieldCopyLink", "TextField Widget");
  Xen_check_type(Xen_is_Time(arg2), arg2, 2, "XmTextFieldCopyLink", "Time");
  return(C_bool_to_Xen_boolean(XmTextFieldCopyLink(Xen_to_C_Widget(arg1), Xen_to_C_Time(arg2))));
}

static Xen gxm_XmTextFieldPaste(Xen arg1)
{
  #define H_XmTextFieldPaste "Boolean XmTextFieldPaste(Widget widget) inserts the clipboard selection"
  Xen_check_type(Xen_is_TextFieldWidget(arg1), arg1, 1, "XmTextFieldPaste", "TextField Widget");
  return(C_bool_to_Xen_boolean(XmTextFieldPaste(Xen_to_C_Widget(arg1))));
}

static Xen gxm_XmTextFieldCut(Xen arg1, Xen arg2)
{
  #define H_XmTextFieldCut "Boolean XmTextFieldCut(Widget widget, Time time) copies the primary selection \
to the clipboard and deletes the selected text"
  Xen_check_type(Xen_is_TextFieldWidget(arg1), arg1, 1, "XmTextFieldCut", "TextField Widget");
  Xen_check_type(Xen_is_Time(arg2), arg2, 2, "XmTextFieldCut", "Time");
  return(C_bool_to_Xen_boolean(XmTextFieldCut(Xen_to_C_Widget(arg1), Xen_to_C_Time(arg2))));
}

static Xen gxm_XmTextFieldCopy(Xen arg1, Xen arg2)
{
  #define H_XmTextFieldCopy "Boolean XmTextFieldCopy(Widget widget, Time time) copies the primary selection to the clipboard"
  Xen_check_type(Xen_is_TextFieldWidget(arg1), arg1, 1, "XmTextFieldCopy", "TextField Widget");
  Xen_check_type(Xen_is_Time(arg2), arg2, 2, "XmTextFieldCopy", "Time");
  return(C_bool_to_Xen_boolean(XmTextFieldCopy(Xen_to_C_Widget(arg1), Xen_to_C_Time(arg2))));
}

static Xen gxm_XmTextFieldRemove(Xen arg1)
{
  #define H_XmTextFieldRemove "Boolean XmTextFieldRemove(Widget widget) deletes the primary selection"
  Xen_check_type(Xen_is_TextFieldWidget(arg1), arg1, 1, "XmTextFieldRemove", "TextField Widget");
  return(C_bool_to_Xen_boolean(XmTextFieldRemove(Xen_to_C_Widget(arg1))));
}

static Xen gxm_XmTextFieldGetSelection(Xen arg1)
{
  char *str;
  Xen res;
  #define H_XmTextFieldGetSelection "char *XmTextFieldGetSelection(Widget widget) retrieves the value of the primary selection"
  Xen_check_type(Xen_is_TextFieldWidget(arg1), arg1, 1, "XmTextFieldGetSelection", "TextField Widget");
  str = XmTextFieldGetSelection(Xen_to_C_Widget(arg1));
  res = C_string_to_Xen_string(str);
  if (str) XtFree(str);
  return(res);
}

static Xen gxm_XmTextFieldGetSelectionPosition(Xen arg1)
{
  #define H_XmTextFieldGetSelectionPosition "Boolean XmTextFieldGetSelectionPosition(Widget widget) \
returns the position of the primary selection"
  /* DIFF: XmTextFieldGetSelectionPosition widget [left right] -> (list res left right)
     no arg2 arg3, returns (owner left right) instead 
  */
  XmTextPosition pos1, pos2;
  int res;
  Xen_check_type(Xen_is_TextFieldWidget(arg1), arg1, 1, "XmTextFieldGetSelectionPosition", "TextField Widget");
  res = XmTextFieldGetSelectionPosition(Xen_to_C_Widget(arg1), &pos1, &pos2);
  return(Xen_list_3(C_bool_to_Xen_boolean(res),
		    C_int_to_Xen_integer(pos1),
		    C_int_to_Xen_integer(pos2)));
}

static Xen gxm_XmTextFieldSetInsertionPosition(Xen arg1, Xen arg2)
{
  #define H_XmTextFieldSetInsertionPosition "void XmTextFieldSetInsertionPosition(Widget widget, XmTextPosition position) \
sets the position of the insertion cursor"
  Xen_check_type(Xen_is_TextFieldWidget(arg1), arg1, 1, "XmTextFieldSetInsertionPosition", "TextField Widget");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XmTextFieldSetInsertionPosition", "XmTextPosition");
  XmTextFieldSetInsertionPosition(Xen_to_C_Widget(arg1), Xen_integer_to_C_int(arg2));
  return(Xen_false);
}

static Xen gxm_XmTextFieldSetCursorPosition(Xen arg1, Xen arg2)
{
  #define H_XmTextFieldSetCursorPosition "void XmTextFieldSetCursorPosition(Widget w, int position) sets the insertion cursor position"
  Xen_check_type(Xen_is_TextFieldWidget(arg1), arg1, 1, "XmTextFieldSetCursorPosition", "TextField Widget");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XmTextFieldSetCursorPosition", "XmTextPosition");
  XmTextFieldSetCursorPosition(Xen_to_C_Widget(arg1), Xen_integer_to_C_int(arg2));
  return(Xen_false);
}

static Xen gxm_XmTextFieldGetInsertionPosition(Xen arg1)
{
  #define H_XmTextFieldGetInsertionPosition "XmTextPosition XmTextFieldGetInsertionPosition(Widget widget) \
accesses the position of the insertion cursor"
  Xen_check_type(Xen_is_TextFieldWidget(arg1), arg1, 1, "XmTextFieldGetInsertionPosition", "TextField Widget");
  return(C_int_to_Xen_integer(XmTextFieldGetInsertionPosition(Xen_to_C_Widget(arg1))));
}

static Xen gxm_XmTextFieldGetCursorPosition(Xen arg1)
{
  #define H_XmTextFieldGetCursorPosition "int XmTextFieldGetCursorPosition(Widget w) presumably returns the widget's insertion cursor position"
  Xen_check_type(Xen_is_TextFieldWidget(arg1), arg1, 1, "XmTextFieldGetCursorPosition", "TextField Widget");
  return(C_int_to_Xen_integer(XmTextFieldGetCursorPosition(Xen_to_C_Widget(arg1))));
}

static Xen gxm_XmTextFieldSetMaxLength(Xen arg1, Xen arg2)
{
  #define H_XmTextFieldSetMaxLength "void XmTextFieldSetMaxLength(Widget widget, int max_length) sets the \
value of the current maximum allowable length of a text string \
entered from the keyboard"
  Xen_check_type(Xen_is_TextFieldWidget(arg1), arg1, 1, "XmTextFieldSetMaxLength", "TextField Widget");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XmTextFieldSetMaxLength", "int");
  XmTextFieldSetMaxLength(Xen_to_C_Widget(arg1), Xen_integer_to_C_int(arg2));
  return(Xen_false);
}

static Xen gxm_XmTextFieldGetMaxLength(Xen arg1)
{
  #define H_XmTextFieldGetMaxLength "int XmTextFieldGetMaxLength(Widget widget) accesses the value of the \
current maximum allowable length of a text string \
entered from the keyboard"
  Xen_check_type(Xen_is_TextFieldWidget(arg1), arg1, 1, "XmTextFieldGetMaxLength", "TextField Widget");
  return(C_int_to_Xen_integer(XmTextFieldGetMaxLength(Xen_to_C_Widget(arg1))));
}

static Xen gxm_XmTextFieldSetEditable(Xen arg1, Xen arg2)
{
  #define H_XmTextFieldSetEditable "void XmTextFieldSetEditable(Widget widget, Boolean editable) sets the edit permission"
  Xen_check_type(Xen_is_TextFieldWidget(arg1), arg1, 1, "XmTextFieldSetEditable", "TextField Widget");
  Xen_check_type(Xen_is_boolean(arg2), arg2, 2, "XmTextFieldSetEditable", "boolean");
  XmTextFieldSetEditable(Xen_to_C_Widget(arg1), Xen_boolean_to_C_bool(arg2));
  return(Xen_false);
}

static Xen gxm_XmTextFieldGetEditable(Xen arg1)
{
  #define H_XmTextFieldGetEditable "Boolean XmTextFieldGetEditable(Widget widget) accesses the edit permission state"
  Xen_check_type(Xen_is_TextFieldWidget(arg1), arg1, 1, "XmTextFieldGetEditable", "TextField Widget");
  return(C_bool_to_Xen_boolean(XmTextFieldGetEditable(Xen_to_C_Widget(arg1))));
}

static Xen gxm_XmTextFieldGetAddMode(Xen arg1)
{
  #define H_XmTextFieldGetAddMode "Boolean XmTextFieldGetAddMode(Widget w) presumably returns the widget's current add mode."
  Xen_check_type(Xen_is_TextFieldWidget(arg1), arg1, 1, "XmTextFieldGetAddMode", "TextField Widget");
  return(C_bool_to_Xen_boolean(XmTextFieldGetAddMode(Xen_to_C_Widget(arg1))));
}

static Xen gxm_XmTextFieldSetAddMode(Xen arg1, Xen arg2)
{
  #define H_XmTextFieldSetAddMode "void XmTextFieldSetAddMode(Widget widget, Boolean state) sets the state of Add mode"
  Xen_check_type(Xen_is_TextFieldWidget(arg1), arg1, 1, "XmTextFieldSetAddMode", "TextField Widget");
  Xen_check_type(Xen_is_boolean(arg2), arg2, 2, "XmTextFieldSetAddMode", "boolean");
  XmTextFieldSetAddMode(Xen_to_C_Widget(arg1), Xen_boolean_to_C_bool(arg2));
  return(Xen_false);
}

static Xen gxm_XmTextFieldInsert(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XmTextFieldInsert "void XmTextFieldInsert(Widget widget, XmTextPosition position, char *value) \
inserts a character string into a text string"
  Xen_check_type(Xen_is_TextFieldWidget(arg1), arg1, 1, "XmTextFieldInsert", "TextField Widget");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XmTextFieldInsert", "XmTextPosition");
  Xen_check_type(Xen_is_string(arg3), arg3, 3, "XmTextFieldInsert", "char*");
  XmTextFieldInsert(Xen_to_C_Widget(arg1), Xen_integer_to_C_int(arg2), (char *)Xen_string_to_C_string(arg3));
  return(Xen_false);
}

static Xen gxm_XmTextFieldReplace(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XmTextFieldReplace "void XmTextFieldReplace(Widget widget, XmTextPosition from_pos, XmTextPosition to_pos, char *value) \
replaces part of a text string"
  Xen_check_type(Xen_is_TextFieldWidget(arg1), arg1, 1, "XmTextFieldReplace", "TextField Widget");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XmTextFieldReplace", "XmTextPosition");
  Xen_check_type(Xen_is_integer(arg3), arg3, 3, "XmTextFieldReplace", "XmTextPosition");
  Xen_check_type(Xen_is_string(arg4), arg4, 4, "XmTextFieldReplace", "char*");
  XmTextFieldReplace(Xen_to_C_Widget(arg1), Xen_integer_to_C_int(arg2), Xen_integer_to_C_int(arg3), (char *)Xen_string_to_C_string(arg4));
  return(Xen_false);
}

static Xen gxm_XmTextFieldSetString(Xen arg1, Xen arg2)
{
  #define H_XmTextFieldSetString "void XmTextFieldSetString(Widget widget, char *value) sets the string value"
  Xen_check_type(Xen_is_TextFieldWidget(arg1), arg1, 1, "XmTextFieldSetString", "TextField Widget");
  Xen_check_type(Xen_is_string(arg2), arg2, 2, "XmTextFieldSetString", "char*");
  XmTextFieldSetString(Xen_to_C_Widget(arg1), (char *)Xen_string_to_C_string(arg2));
  return(Xen_false);
}

static Xen gxm_XmTextFieldGetLastPosition(Xen arg1)
{
  #define H_XmTextFieldGetLastPosition "XmTextPosition XmTextFieldGetLastPosition(Widget widget) accesses \
the position of the last text character"
  Xen_check_type(Xen_is_TextFieldWidget(arg1), arg1, 1, "XmTextFieldGetLastPosition", "TextField Widget");
  return(C_int_to_Xen_integer(XmTextFieldGetLastPosition(Xen_to_C_Widget(arg1))));
}

static Xen gxm_XmTextFieldGetSubstring(Xen arg1, Xen arg2, Xen arg3)
{
  /* DIFF: omit and rtn last 2 args */
  #define H_XmTextFieldGetSubstring "int XmTextFieldGetSubstring(Widget widget, XmTextPosition start, int num_chars) \
retrieves a copy of a portion of the internal text buffer"
  int rtn, len;
  char *buf;
  Xen str = Xen_false;
  Xen_check_type(Xen_is_TextFieldWidget(arg1), arg1, 1, "XmTextFieldGetSubstring", "TextField Widget");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XmTextFieldGetSubstring", "XmTextPosition");
  Xen_check_type(Xen_is_integer(arg3), arg3, 3, "XmTextFieldGetSubstring", "int");
  len = Xen_integer_to_C_int(arg3);
  buf = (char *)calloc(len + 1, sizeof(char));
  rtn = XmTextFieldGetSubstring(Xen_to_C_Widget(arg1), Xen_integer_to_C_int(arg2), len, len + 1, buf);
  if (rtn != XmCOPY_FAILED)
    str = C_string_to_Xen_string(buf);
  free(buf);
  return(str);
}

static Xen gxm_XmTextFieldGetString(Xen arg1)
{
  char *str;
  Xen res;
  #define H_XmTextFieldGetString "char *XmTextFieldGetString(Widget widget) accesses the string value"
  Xen_check_type(Xen_is_TextFieldWidget(arg1), arg1, 1, "XmTextFieldGetString", "TextField Widget");
  str = XmTextFieldGetString(Xen_to_C_Widget(arg1));
  res = C_string_to_Xen_string(str);
  if (str) XtFree(str);
  return(res);
}

static Xen gxm_XmDropTransferAdd(Xen arg1, Xen arg2)
{
  #define H_XmDropTransferAdd "void XmDropTransferAdd(Widget drop_transfer, XmDropTransferEntryRec *transfers) \
enables additional drop transfer entries to be processed after initiating a drop transfer"
  /* DIFF: XmDropTransferAdd  arg2 is list of (atom data) lists, arg3 omitted
   */
  int len;
  XmDropTransferEntryRec *entries;
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XmDropTransferAdd", "Widget");
  Xen_check_type(Xen_is_list(arg2), arg2, 2, "XmDropTransferAdd", "XmDropTransferEntry");
  len = Xen_list_length(arg2);
  if (len <= 0) return(Xen_false);
  entries = Xen_to_C_XmDropTransferEntryRecs(arg2, len);
  XmDropTransferAdd(Xen_to_C_Widget(arg1), entries, len);
  free(entries);
  return(Xen_false);
}

static Xen gxm_XmDropTransferStart(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XmDropTransferStart "Widget XmDropTransferStart(Widget widget, ArgList arglist, Cardinal argcount) \
initiates a drop transfer"
  Widget w;
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XmDropTransferStart", "Widget");
  Xen_check_type(Xen_is_list(arg2), arg2, 2, "XmDropTransferStart", "ArgList");
  Xen_check_type(Xen_is_integer_or_unbound(arg3), arg3, 3, "XmDropTransferStart", "int");
  {
    Arg *args;
    int arglen;
    args = Xen_to_C_Args(arg2);
    arglen = Xen_to_C_INT_DEF(arg3, arg2);
    w = XmDropTransferStart(Xen_to_C_Widget(arg1), args, arglen);
    if (args)
      {
	fixup_args(w, args, arglen);
	free_args(args, arglen);
      }
  }
  return(C_to_Xen_Widget(w));
}

static Xen gxm_XmDropSiteConfigureStackingOrder(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XmDropSiteConfigureStackingOrder "void XmDropSiteConfigureStackingOrder(Widget widget, Widget sibling, Cardinal stack_mode) \
reorders a stack of widgets that are registered drop sites"
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XmDropSiteConfigureStackingOrder", "Widget");
  Xen_check_type(Xen_is_Widget(arg2), arg2, 2, "XmDropSiteConfigureStackingOrder", "Widget");
  Xen_check_type(Xen_is_integer(arg3), arg3, 3, "XmDropSiteConfigureStackingOrder", "int");
  XmDropSiteConfigureStackingOrder(Xen_to_C_Widget(arg1), Xen_to_C_Widget(arg2), Xen_integer_to_C_int(arg3));
  return(Xen_false);
}

static Xen gxm_XmDropSiteQueryStackingOrder(Xen arg1)
{
  #define H_XmDropSiteQueryStackingOrder "Status XmDropSiteQueryStackingOrder(Widget widget): (list parent child ...)"
  /* DIFF: XmDropSiteQueryStackingOrder widget [parent child numchild] -> (list parent child ...)
     no parent children num_children -- returned as '(parent child1...) or #f 
  */
  Widget parent;
  Widget *children;
  Xen lst = Xen_empty_list;
  unsigned int num_children;
  int res, i, loc;
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XmDropSiteQueryStackingOrder", "Widget");
  res = XmDropSiteQueryStackingOrder(Xen_to_C_Widget(arg1), &parent, &children, &num_children);
  if (res == 0)
    return(Xen_false);
  loc = xm_protect(lst);
  for (i = num_children - 1; i >= 0; i--)
    lst = Xen_cons(C_to_Xen_Widget(children[i]), lst);
  xm_unprotect_at(loc);
  return(Xen_cons(C_to_Xen_Widget(parent), lst));
}

static Xen gxm_XmDropSiteRetrieve(Xen arg1, Xen larg2, Xen arg3)
{
  #define H_XmDropSiteRetrieve "void XmDropSiteRetrieve(Widget widget, ArgList arglist, Cardinal argcount) \
retrieves resource values set on a drop site"

  Arg *args;
  unsigned long *locs;
  Xen val = Xen_false;
  int i, len, gcloc;
  Xen arg2;
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XmDropSiteRetrieve", "Widget");
  Xen_check_type(Xen_is_list(larg2), larg2, 2, "XmDropSiteRetrieve", "ArgList");
  Xen_check_type(Xen_is_integer_or_unbound(arg3), arg3, 3, "XmDropSiteRetrieve", "int");
  arg2 = Xen_copy_arg(larg2);
  gcloc = xm_protect(arg2);
  len = Xen_to_C_INT_DEF(arg3, larg2);
  if (len <= 0) Xen_check_type(0, arg3, 3, "XmDropSiteRetrieve", "positive integer");
  args = (Arg *)calloc(len, sizeof(Arg));
  locs = (unsigned long *)calloc(len, sizeof(unsigned long));
  for (i = 0; i < len; i++, arg2 = Xen_cddr(arg2))
    {
      char *name;
      name = xen_strdup(Xen_string_to_C_string(Xen_car(arg2)));
      XtSetArg(args[i], name, &(locs[i]));
    }
  XmDropSiteRetrieve(Xen_to_C_Widget(arg1), args, len);
  val = C_to_Xen_Args((Widget)(Xen_to_C_Widget(arg1)), args, len);
  free_args(args, len);
  free(locs);
  xm_unprotect_at(gcloc);
  return(val);
}

static Xen gxm_XmDropSiteEndUpdate(Xen arg1)
{
  #define H_XmDropSiteEndUpdate "void XmDropSiteEndUpdate(Widget widget) facilitates processing updates to multiple drop sites"
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XmDropSiteEndUpdate", "Widget");
  XmDropSiteEndUpdate(Xen_to_C_Widget(arg1));
  return(Xen_false);
}

static Xen gxm_XmDropSiteUpdate(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XmDropSiteUpdate "void XmDropSiteUpdate(Widget widget, ArgList arglist, Cardinal argcount) sets \
resource values for a drop site"
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XmDropSiteUpdate", "Widget");
  Xen_check_type(Xen_is_list(arg2), arg2, 2, "XmDropSiteUpdate", "ArgList");
  Xen_check_type(Xen_is_integer_or_unbound(arg3), arg3, 3, "XmDropSiteUpdate", "int");
  {
    Widget w;
    Arg *args;
    int arglen;
    w = Xen_to_C_Widget(arg1);
    args = Xen_to_C_Args(arg2);
    arglen = Xen_to_C_INT_DEF(arg3, arg2);
    XmDropSiteUpdate(w, args, arglen);
    if (args)
      {
	fixup_args(w, args, arglen);
	free_args(args, arglen);
      }
  }
  return(Xen_false);
}

static Xen gxm_XmDropSiteStartUpdate(Xen arg1)
{
  #define H_XmDropSiteStartUpdate "void XmDropSiteStartUpdate(Widget widget) facilitates processing updates \
to multiple drop sites"
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XmDropSiteStartUpdate", "Widget");
  XmDropSiteStartUpdate(Xen_to_C_Widget(arg1));
  return(Xen_false);
}

static Xen gxm_XmDropSiteRegistered(Xen arg1)
{
  #define H_XmDropSiteRegistered "Boolean XmDropSiteRegistered(Widget widget) determines if a drop site has been registered"
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XmDropSiteRegistered", "Widget");
  return(C_bool_to_Xen_boolean(XmDropSiteRegistered(Xen_to_C_Widget(arg1))));
}

static Xen gxm_XmDropSiteUnregister(Xen arg1)
{
  #define H_XmDropSiteUnregister "void XmDropSiteUnregister(Widget widget) frees drop site information"
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XmDropSiteUnregister", "Widget");
  XmDropSiteUnregister(Xen_to_C_Widget(arg1));
  return(Xen_false);
}

static Xen gxm_XmDropSiteRegister(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XmDropSiteRegister "void XmDropSiteRegister(Widget widget, ArgList arglist, Cardinal argcount) \
identifies a drop site and assigns resources that specify its behavior"
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XmDropSiteRegister", "Widget");
  Xen_check_type(Xen_is_list(arg2), arg2, 2, "XmDropSiteRegister", "ArgList");
  Xen_check_type(Xen_is_integer_or_unbound(arg3), arg3, 3, "XmDropSiteRegister", "int");
  {
    Widget w;
    Arg *args;
    int arglen;
    w = Xen_to_C_Widget(arg1);
    args = Xen_to_C_Args(arg2);
    arglen = Xen_to_C_INT_DEF(arg3, arg2);
    XmDropSiteRegister(w, args, arglen);
    if (args)
      {
	fixup_args(w, args, arglen);
	free_args(args, arglen);
      }
  }
  return(Xen_false);
}

static Xen gxm_XmSimpleSpinBoxSetItem(Xen arg1, Xen arg2)
{
  #define H_XmSimpleSpinBoxSetItem "void XmSimpleSpinBoxSetItem(Widget w, XmString item) set an item in the XmSimpleSpinBox list"
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XmSimpleSpinBoxSetItem", "Widget");
  Xen_check_type(Xen_is_XmString(arg2), arg2, 2, "XmSimpleSpinBoxSetItem", "XmString");
  XmSimpleSpinBoxSetItem(Xen_to_C_Widget(arg1), Xen_to_C_XmString(arg2));
  return(Xen_false);
}

static Xen gxm_XmSimpleSpinBoxDeletePos(Xen arg1, Xen arg2)
{
  #define H_XmSimpleSpinBoxDeletePos "void XmSimpleSpinBoxDeletePos(Widget w, int pos) delete a XmSimpleSpinBox item"
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XmSimpleSpinBoxDeletePos", "Widget");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XmSimpleSpinBoxDeletePos", "int");
  XmSimpleSpinBoxDeletePos(Xen_to_C_Widget(arg1), Xen_integer_to_C_int(arg2));
  return(Xen_false);
}

static Xen gxm_XmSimpleSpinBoxAddItem(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XmSimpleSpinBoxAddItem "void XmSimpleSpinBoxAddItem(Widget w, XmString item, int pos) add an item to the XmSimpleSpinBox"
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XmSimpleSpinBoxAddItem", "Widget");
  Xen_check_type(Xen_is_XmString(arg2), arg2, 2, "XmSimpleSpinBoxAddItem", "XmString");
  Xen_check_type(Xen_is_integer(arg3), arg3, 3, "XmSimpleSpinBoxAddItem", "int");
  XmSimpleSpinBoxAddItem(Xen_to_C_Widget(arg1), Xen_to_C_XmString(arg2), Xen_integer_to_C_int(arg3));
  return(Xen_false);
}

static Xen gxm_XmCreateSimpleSpinBox(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XmCreateSimpleSpinBox "Widget XmCreateSimpleSpinBox(Widget parent, String name, ArgList arglist, Cardinal argcount) \
The SimpleSpinBox widget creation function"
  return(gxm_new_widget("XmCreateSimpleSpinBox", XmCreateSimpleSpinBox, arg1, arg2, arg3, arg4));
}

static Xen gxm_XmCreateDrawnButton(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XmCreateDrawnButton "Widget XmCreateDrawnButton(Widget parent, String name, ArgList arglist, Cardinal argcount) \
The DrawnButton widget creation function"
  return(gxm_new_widget("XmCreateDrawnButton", XmCreateDrawnButton, arg1, arg2, arg3, arg4));
}

static Xen gxm_XmSpinBoxValidatePosition(Xen arg1)
{
  #define H_XmSpinBoxValidatePosition "int XmSpinBoxValidatePosition(Widget textfield) translate the current value of \
the specified XmSpinBox child into a valid position"
  /* DIFF: XmSpinBoxValidatePosition omits arg2, returns pos
   */
  int pos;
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XmSpinBoxValidatePosition", "Widget");
  XmSpinBoxValidatePosition(Xen_to_C_Widget(arg1), &pos);
  return(C_int_to_Xen_integer(pos));
}

static Xen gxm_XmCreateSpinBox(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XmCreateSpinBox "The SpinBox creation function"
  return(gxm_new_widget("XmCreateSpinBox", XmCreateSpinBox, arg1, arg2, arg3, arg4));
}

static Xen gxm_XmCreateDrawingArea(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XmCreateDrawingArea "Widget XmCreateDrawingArea(Widget parent, String name, ArgList arglist, Cardinal argcount) \
The DrawingArea widget creation function"
  return(gxm_new_widget("XmCreateDrawingArea", XmCreateDrawingArea, arg1, arg2, arg3, arg4));
}

static Xen gxm_XmCreateSeparator(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XmCreateSeparator "Widget XmCreateSeparator(Widget parent, String name, ArgList arglist, Cardinal argcount) \
The Separator widget creation function"
  return(gxm_new_widget("XmCreateSeparator", XmCreateSeparator, arg1, arg2, arg3, arg4));
}

static Xen gxm_XmCreateDragIcon(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XmCreateDragIcon "Widget XmCreateDragIcon(Widget widget, String name, ArgList arglist, Cardinal argcount) \
creates a DragIcon widget"
  return(gxm_new_widget("XmCreateDragIcon", XmCreateDragIcon, arg1, arg2, arg3, arg4));
}

static Xen gxm_XmCreateSeparatorGadget(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XmCreateSeparatorGadget "Widget XmCreateSeparatorGadget(Widget parent, String name, ArgList arglist, Cardinal argcount) \
The SeparatorGadget creation function"
  return(gxm_new_widget("XmCreateSeparatorGadget", XmCreateSeparatorGadget, arg1, arg2, arg3, arg4));
}

static Xen gxm_XmTargetsAreCompatible(Xen arg1, Xen arg2, Xen arg3, Xen arg4, Xen arg5)
{
  #define H_XmTargetsAreCompatible "Boolean XmTargetsAreCompatible(Display *display, Atom *export_targets, Cardinal num_export_targets, \
Atom *import_targets, Cardinal num_import_targets)  tests whether the target types match between a drop site and source object"
  /* DIFF: XmTargetsAreCompatible arg2 and arg4 are lists of Atoms
   */
  Atom *outs, *ins;
  int val, len1, len2;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XmTargetsAreCompatible", "Display*");
  Xen_check_type(Xen_is_list(arg2), arg2, 2, "XmTargetsAreCompatible", "list of Atom");
  Xen_check_type(Xen_is_integer(arg3), arg3, 3, "XmTargetsAreCompatible", "int");
  Xen_check_type(Xen_is_list(arg4), arg4, 4, "XmTargetsAreCompatible", "list of Atom");
  Xen_check_type(Xen_is_integer(arg5), arg5, 5, "XmTargetsAreCompatible", "int");
  len1 = Xen_integer_to_C_int(arg3);
  outs = Xen_to_C_Atoms(arg2, len1);
  len2 = Xen_integer_to_C_int(arg5);
  ins = Xen_to_C_Atoms(arg4, len2);
  val = XmTargetsAreCompatible(Xen_to_C_Display(arg1), outs, len1, ins, len2);
  free(outs);
  free(ins);
  return(C_bool_to_Xen_boolean(val));
}

static Xen gxm_XmDragCancel(Xen arg1)
{
  #define H_XmDragCancel "void XmDragCancel(Widget dragcontext) terminates a drag transaction"
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XmDragCancel", "Widget");
  XmDragCancel(Xen_to_C_Widget(arg1));
  return(Xen_false);
}

static Xen gxm_XmDragStart(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XmDragStart "Widget XmDragStart(Widget widget, XEvent *event, ArgList arglist, Cardinal argcount) \
initiates a drag and drop transaction"
  Widget w;
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XmDragStart", "Widget");
  Xen_check_type(Xen_is_XEvent(arg2), arg2, 2, "XmDragStart", "XEvent*");
  Xen_check_type(Xen_is_list(arg3), arg3, 3, "XmDragStart", "ArgList");
  Xen_check_type(Xen_is_integer_or_unbound(arg4), arg4, 4, "XmDragStart", "int");
  {
    Arg *args;
    int arglen;
    args = Xen_to_C_Args(arg3);
    arglen = Xen_to_C_INT_DEF(arg4, arg3);
    w = XmDragStart(Xen_to_C_Widget(arg1), 
		    Xen_to_C_XEvent(arg2), 
		    args, arglen);
    if (args)
      {
	fixup_args(w, args, arglen);
	free_args(args, arglen);
      }
  }
  return(C_to_Xen_Widget(w));
}

static Xen gxm_XmCreatePromptDialog(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XmCreatePromptDialog "Widget XmCreatePromptDialog(Widget parent, String name, ArgList arglist, Cardinal argcount) \
The SelectionBox PromptDialog creation function"
  return(gxm_new_widget("XmCreatePromptDialog", XmCreatePromptDialog, arg1, arg2, arg3, arg4));
}

static Xen gxm_XmCreateSelectionDialog(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XmCreateSelectionDialog "Widget XmCreateSelectionDialog(Widget parent, String name, ArgList arglist, Cardinal argcount) \
The SelectionBox SelectionDialog creation function"
  return(gxm_new_widget("XmCreateSelectionDialog", XmCreateSelectionDialog, arg1, arg2, arg3, arg4));
}

static Xen gxm_XmCreateSelectionBox(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XmCreateSelectionBox "Widget XmCreateSelectionBox(Widget parent, String name, ArgList arglist, Cardinal argcount) \
The SelectionBox widget creation function"
  return(gxm_new_widget("XmCreateSelectionBox", XmCreateSelectionBox, arg1, arg2, arg3, arg4));
}

static Xen gxm_XmSelectionBoxGetChild(Xen arg1, Xen arg2)
{
  #define H_XmSelectionBoxGetChild "Widget XmSelectionBoxGetChild(Widget widget, unsigned char child) used to access a SelectionBox component"
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XmSelectionBoxGetChild", "Widget");
  Xen_check_type(Xen_is_ulong(arg2), arg2, 2, "XmSelectionBoxGetChild", "unsigned int");
  return(C_to_Xen_Widget(XmSelectionBoxGetChild(Xen_to_C_Widget(arg1), Xen_ulong_to_C_ulong(arg2))));
}

static Xen gxm_XmGetXmDisplay(Xen arg1)
{
  #define H_XmGetXmDisplay "Widget XmGetXmDisplay(Display *display) A Display function that returns the"
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XmGetXmDisplay", "Display*");
  return(C_to_Xen_Widget(XmGetXmDisplay(Xen_to_C_Display(arg1))));
}

static Xen gxm_XmGetDragContext(Xen arg1, Xen arg2)
{
  #define H_XmGetDragContext "Widget XmGetDragContext(Widget refwidget, Time timestamp) retrieves \
the DragContext widget ID associated with a timestamp"
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XmGetDragContext", "Widget");
  Xen_check_type(Xen_is_Time(arg2), arg2, 2, "XmGetDragContext", "Time");
  return(C_to_Xen_Widget(XmGetDragContext(Xen_to_C_Widget(arg1), Xen_to_C_Time(arg2))));
}

static Xen gxm_XmScrollVisible(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XmScrollVisible "void XmScrollVisible(Widget scrollw_widget, Widget widget, Dimension left_right_margin, Dimension top_bottom_margin) \
makes an invisible  descendant of a ScrolledWindow work area visible"
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XmScrollVisible", "Widget");
  Xen_check_type(Xen_is_Widget(arg2), arg2, 2, "XmScrollVisible", "Widget");
  Xen_check_type(Xen_is_Dimension(arg3), arg3, 3, "XmScrollVisible", "Dimension");
  Xen_check_type(Xen_is_Dimension(arg4), arg4, 4, "XmScrollVisible", "Dimension");
  XmScrollVisible(Xen_to_C_Widget(arg1), Xen_to_C_Widget(arg2), 
		  Xen_to_C_Dimension(arg3), Xen_to_C_Dimension(arg4));
  return(Xen_false);
}

static Xen gxm_XmCreateScrolledWindow(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XmCreateScrolledWindow "Widget XmCreateScrolledWindow(Widget parent, String name, ArgList arglist, Cardinal argcount) \
The ScrolledWindow widget creation function"
  return(gxm_new_widget("XmCreateScrolledWindow", XmCreateScrolledWindow, arg1, arg2, arg3, arg4));
}

static Xen gxm_XmCreateDialogShell(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XmCreateDialogShell "Widget XmCreateDialogShell(Widget parent, String name, ArgList arglist, Cardinal argcount) \
The DialogShell widget creation function"
  return(gxm_new_widget("XmCreateDialogShell", XmCreateDialogShell, arg1, arg2, arg3, arg4));
}

static Xen gxm_XmScrollBarSetValues(Xen arg1, Xen arg2, Xen arg3, Xen arg4, Xen arg5, Xen arg6)
{
  #define H_XmScrollBarSetValues "void XmScrollBarSetValues (widget, value, slider_size, increment, page_increment, notify) \
changes ScrollBar's increment values and the slider's size and position"
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XmScrollBarSetValues", "Widget");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XmScrollBarSetValues", "int");
  Xen_check_type(Xen_is_integer(arg3), arg3, 3, "XmScrollBarSetValues", "int");
  Xen_check_type(Xen_is_integer(arg4), arg4, 4, "XmScrollBarSetValues", "int");
  Xen_check_type(Xen_is_integer(arg5), arg5, 5, "XmScrollBarSetValues", "int");
  Xen_check_type(Xen_is_boolean(arg6), arg6, 6, "XmScrollBarSetValues", "boolean");
  XmScrollBarSetValues(Xen_to_C_Widget(arg1), 
		       Xen_integer_to_C_int(arg2), Xen_integer_to_C_int(arg3), 
		       Xen_integer_to_C_int(arg4), Xen_integer_to_C_int(arg5), 
		       Xen_boolean_to_C_bool(arg6));
  return(Xen_false);
}

static Xen gxm_XmScrollBarGetValues(Xen arg1)
{
  #define H_XmScrollBarGetValues "void XmScrollBarGetValues (widget): returns the ScrollBar's increment values (list val size incr page)"
  /* DIFF: XmScrollBarGetValues omits and returns last 4 args
   */
  int val, size, incr, page;
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XmScrollBarGetValues", "Widget");
  XmScrollBarGetValues(Xen_to_C_Widget(arg1), &val, &size, &incr, &page);
  return(Xen_list_4(C_int_to_Xen_integer(val),
		    C_int_to_Xen_integer(size),
		    C_int_to_Xen_integer(incr),
		    C_int_to_Xen_integer(page)));
}

static Xen gxm_XmCreateScrollBar(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XmCreateScrollBar "Widget XmCreateScrollBar(Widget parent, String name, ArgList arglist, Cardinal argcount) \
The ScrollBar widget creation function"
  return(gxm_new_widget("XmCreateScrollBar", XmCreateScrollBar, arg1, arg2, arg3, arg4));
}

static Xen gxm_XmGetXmScreen(Xen arg1)
{
  #define H_XmGetXmScreen "Widget XmGetXmScreen(Screen *screen): returns the XmScreen object ID for a specified screen"
  /* this is the Motif Screen "widget" */
  Xen_check_type(Xen_is_Screen(arg1), arg1, 1, "XmGetXmScreen", "Screen*");
  return(C_to_Xen_Widget(XmGetXmScreen(Xen_to_C_Screen(arg1))));
}

static Xen gxm_XmClipboardRegisterFormat(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XmClipboardRegisterFormat "int XmClipboardRegisterFormat (display, format_name, format_length) registers a new format"
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XmClipboardRegisterFormat", "Display*");
  Xen_check_type(Xen_is_string(arg2), arg2, 2, "XmClipboardRegisterFormat", "char*");
  Xen_check_type(Xen_is_integer(arg3), arg3, 3, "XmClipboardRegisterFormat", "int");
  return(C_int_to_Xen_integer(XmClipboardRegisterFormat(Xen_to_C_Display(arg1), 
						(char *)Xen_string_to_C_string(arg2), 
						Xen_integer_to_C_int(arg3))));
}

static Xen gxm_XmClipboardInquirePendingItems(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XmClipboardInquirePendingItems "int XmClipboardInquirePendingItems (display, window, format_name) \
returns a list of data ID/private ID pairs"
  /* DIFF: XmClipboardInquirePendingItems omits last 2 args, returns list of pairs
   */
  unsigned long len;
  XmClipboardPendingList clst;
  int i, loc, rtn;
  Xen lst = Xen_empty_list;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XmClipboardInquirePendingItems", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XmClipboardInquirePendingItems", "Window");
  Xen_check_type(Xen_is_string(arg3), arg3, 3, "XmClipboardInquirePendingItems", "char*");
  rtn = XmClipboardInquirePendingItems(Xen_to_C_Display(arg1), 
				       Xen_to_C_Window(arg2), 
				       (char *)Xen_string_to_C_string(arg3), 
				       &clst,
				       &len);
  loc = xm_protect(lst);
  for (i = len - 1; i >= 0; i--)
    lst = Xen_cons(Xen_list_2(C_int_to_Xen_integer(clst[i].DataId), 
			      C_int_to_Xen_integer(clst[i].PrivateId)),
		   lst);
  lst = Xen_cons(C_int_to_Xen_integer(rtn), lst);
  xm_unprotect_at(loc);
  return(lst);
}

static Xen gxm_XmClipboardInquireLength(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XmClipboardInquireLength "int XmClipboardInquireLength (display, window, format_name): returns the length of the stored data"
  /* DIFF: XmClipboardInquireLength omit and rtn last arg
   */
  unsigned long len;
  int val;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XmClipboardInquireLength", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XmClipboardInquireLength", "Window");
  Xen_check_type(Xen_is_string(arg3), arg3, 3, "XmClipboardInquireLength", "char*");
  val = XmClipboardInquireLength(Xen_to_C_Display(arg1), Xen_to_C_Window(arg2), (char *)Xen_string_to_C_string(arg3), &len);
  return(Xen_list_2(C_int_to_Xen_integer(val),
		    C_ulong_to_Xen_ulong(len)));
}

static Xen gxm_XmClipboardInquireFormat(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XmClipboardInquireFormat "int XmClipboardInquireFormat (display, window, index, buffer_len) \
returns a specified format name"
  /* DIFF: XmClipboardInquireFormat omits arg4 (XtPointer buffer) and arg6, returns them
   */
  Xen res;
  XtPointer buf;
  unsigned long len, n;
  int val;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XmClipboardInquireFormat", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XmClipboardInquireFormat", "Window");
  Xen_check_type(Xen_is_integer(arg3), arg3, 3, "XmClipboardInquireFormat", "int");
  Xen_check_type(Xen_is_ulong(arg4), arg4, 4, "XmClipboardInquireFormat", "ulong");
  len = Xen_ulong_to_C_ulong(arg4);
  if (len == 0) return(Xen_false);
  buf = (XtPointer)calloc(len + 1, sizeof(char));
  val = XmClipboardInquireFormat(Xen_to_C_Display(arg1), 
				 Xen_to_C_Window(arg2), 
				 Xen_integer_to_C_int(arg3), 
				 buf, len, &n);
  res = C_string_to_Xen_string((char *)buf);
  free(buf);
  return(Xen_list_2(C_int_to_Xen_integer(val), res));
}

static Xen gxm_XmClipboardInquireCount(Xen arg1, Xen arg2)
{
  #define H_XmClipboardInquireCount "int XmClipboardInquireCount (display, window): returns the number of data item formats"
  /* DIFF: XmClipboardInquireCount omits and rtns last 2 args
   */
  int count, val;
  unsigned long len;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XmClipboardInquireCount", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XmClipboardInquireCount", "Window");
  val = XmClipboardInquireCount(Xen_to_C_Display(arg1), 
				Xen_to_C_Window(arg2),
				&count, &len);
  return(Xen_list_3(C_int_to_Xen_integer(val),
		    C_int_to_Xen_integer(count),
		    C_ulong_to_Xen_ulong(len)));
}

static Xen gxm_XmClipboardRetrieve(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XmClipboardRetrieve "int XmClipboardRetrieve (display, window, format_name, length) retrieves a data item from the clipboard"
  /* DIFF: XmClipboardRetrieve omits buf arg, and last 2, returning them and a list of ulongs
   */
  unsigned long n;
  long id;
  XtPointer buf;
  int len, val;
  Xen str;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XmClipboardRetrieve", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XmClipboardRetrieve", "Window");
  Xen_check_type(Xen_is_string(arg3), arg3, 3, "XmClipboardRetrieve", "char*");
  Xen_check_type(Xen_is_ulong(arg4), arg4, 4, "XmClipboardRetrieve", "ulong");
  len = Xen_ulong_to_C_ulong(arg4);
  if (len <= 0) return(Xen_false);
  buf = (XtPointer)malloc(len);
  val = XmClipboardRetrieve(Xen_to_C_Display(arg1), 
			    Xen_to_C_Window(arg2), 
			    (char *)Xen_string_to_C_string(arg3), 
			    buf, len, &n, &id);
  str = C_string_to_Xen_string((char *)buf);
  free(buf);
  return(Xen_list_3(C_int_to_Xen_integer(val),
		    str,
		    C_ulong_to_Xen_ulong(id)));
}

static Xen gxm_XmClipboardEndRetrieve(Xen arg1, Xen arg2)
{
  #define H_XmClipboardEndRetrieve "int XmClipboardEndRetrieve (display, window) completes retrieval of \
data from the clipboard"
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XmClipboardEndRetrieve", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XmClipboardEndRetrieve", "Window");
  return(C_int_to_Xen_integer(XmClipboardEndRetrieve(Xen_to_C_Display(arg1), Xen_to_C_Window(arg2))));
}

static Xen gxm_XmClipboardStartRetrieve(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XmClipboardStartRetrieve "int XmClipboardStartRetrieve (display, window, timestamp) \
prepares to retrieve data from the clipboard"
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XmClipboardStartRetrieve", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XmClipboardStartRetrieve", "Window");
  Xen_check_type(Xen_is_Time(arg3), arg3, 3, "XmClipboardStartRetrieve", "Time");
  return(C_int_to_Xen_integer(XmClipboardStartRetrieve(Xen_to_C_Display(arg1), Xen_to_C_Window(arg2), Xen_to_C_Time(arg3))));
}

static Xen gxm_XmClipboardUnlock(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XmClipboardUnlock "int XmClipboardUnlock (display, window, remove_all_locks) unlocks the clipboard"
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XmClipboardUnlock", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XmClipboardUnlock", "Window");
  Xen_check_type(Xen_is_boolean(arg3), arg3, 3, "XmClipboardUnlock", "boolean");
  return(C_int_to_Xen_integer(XmClipboardUnlock(Xen_to_C_Display(arg1), Xen_to_C_Window(arg2), Xen_boolean_to_C_bool(arg3))));
}

static Xen gxm_XmClipboardLock(Xen arg1, Xen arg2)
{
  #define H_XmClipboardLock "int XmClipboardLock (display, window) locks the clipboard"
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XmClipboardLock", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XmClipboardLock", "Window");
  return(C_int_to_Xen_integer(XmClipboardLock(Xen_to_C_Display(arg1), Xen_to_C_Window(arg2))));
}

static Xen gxm_XmClipboardUndoCopy(Xen arg1, Xen arg2)
{
  #define H_XmClipboardUndoCopy "int XmClipboardUndoCopy (display, window) deletes the last item placed on the clipboard"
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XmClipboardUndoCopy", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XmClipboardUndoCopy", "Window");
  return(C_int_to_Xen_integer(XmClipboardUndoCopy(Xen_to_C_Display(arg1), Xen_to_C_Window(arg2))));
}

static Xen gxm_XmClipboardCopyByName(Xen arg1, Xen arg2, Xen arg3, Xen arg4, Xen arg5, Xen arg6)
{
  #define H_XmClipboardCopyByName "int XmClipboardCopyByName (display, window, data_id,  buf, len, id) copies a data item passed by name"
  /* DIFF: XmClipboardCopyByName arg4 is string
   */
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XmClipboardCopyByName", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XmClipboardCopyByName", "Window");
  Xen_check_type(Xen_is_integer(arg3), arg3, 3, "XmClipboardCopyByName", "long");
  Xen_check_type(Xen_is_string(arg4), arg4, 4, "XmClipboardCopyByName", "XtPointer");
  Xen_check_type(Xen_is_ulong(arg5), arg5, 5, "XmClipboardCopyByName", "ulong");
  return(C_int_to_Xen_integer(XmClipboardCopyByName(Xen_to_C_Display(arg1), 
					    Xen_to_C_Window(arg2), Xen_integer_to_C_int(arg3), 
					    (XtPointer)Xen_string_to_C_string(arg4), 
					    Xen_ulong_to_C_ulong(arg5), 
					    (unsigned long)arg6)));
}

static Xen gxm_XmClipboardWithdrawFormat(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XmClipboardWithdrawFormat "int XmClipboardWithdrawFormat (display, window, data_id) \
indicates that the application no longer wants to supply a data item"
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XmClipboardWithdrawFormat", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XmClipboardWithdrawFormat", "Window");
  Xen_check_type(Xen_is_integer(arg3), arg3, 3, "XmClipboardWithdrawFormat", "long");
  return(C_int_to_Xen_integer(XmClipboardWithdrawFormat(Xen_to_C_Display(arg1), Xen_to_C_Window(arg2), Xen_integer_to_C_int(arg3))));
}

static Xen gxm_XmClipboardCancelCopy(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XmClipboardCancelCopy "int XmClipboardCancelCopy (display, window, item_id) cancels a copy to the clipboard"
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XmClipboardCancelCopy", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XmClipboardCancelCopy", "Window");
  Xen_check_type(Xen_is_integer(arg3), arg3, 3, "XmClipboardCancelCopy", "long");
  return(C_int_to_Xen_integer(XmClipboardCancelCopy(Xen_to_C_Display(arg1), Xen_to_C_Window(arg2), Xen_integer_to_C_int(arg3))));
}

static Xen gxm_XmClipboardEndCopy(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XmClipboardEndCopy "int XmClipboardEndCopy (display, window, item_id) completes the \
copying of data to the clipboard"
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XmClipboardEndCopy", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XmClipboardEndCopy", "Window");
  Xen_check_type(Xen_is_integer(arg3), arg3, 3, "XmClipboardEndCopy", "long");
  return(C_int_to_Xen_integer(XmClipboardEndCopy(Xen_to_C_Display(arg1), Xen_to_C_Window(arg2), Xen_integer_to_C_int(arg3))));
}

static Xen gxm_XmClipboardCopy(Xen arg1, Xen arg2, Xen arg3, Xen arg4, Xen arg5, Xen arg6, Xen arg7)
{
  #define H_XmClipboardCopy "int XmClipboardCopy (display, window, item_id, format_name, buffer, len, id) copies a data item \
to temporary storage for later copying to clipboard"
  /* DIFF: XmClipboardCopy omits and returns last arg, arg5 is string
   */
  long id;
  int val;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XmClipboardCopy", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XmClipboardCopy", "Window");
  Xen_check_type(Xen_is_integer(arg3), arg3, 3, "XmClipboardCopy", "long");
  Xen_check_type(Xen_is_string(arg4), arg4, 4, "XmClipboardCopy", "char*");
  Xen_check_type(Xen_is_string(arg5), arg5, 5, "XmClipboardCopy", "XtPointer");
  Xen_check_type(Xen_is_ulong(arg6), arg6, 6, "XmClipboardCopy", "ulong");
  Xen_check_type(Xen_is_integer(arg7), arg7, 7, "XmClipboardCopy", "long");
  val = XmClipboardCopy(Xen_to_C_Display(arg1), Xen_to_C_Window(arg2), 
			Xen_integer_to_C_int(arg3), (char *)Xen_string_to_C_string(arg4), 
			(XtPointer)Xen_string_to_C_string(arg5), Xen_ulong_to_C_ulong(arg6), 
			Xen_integer_to_C_int(arg7), &id);
  return(Xen_list_2(C_int_to_Xen_integer(val),
		    C_int_to_Xen_integer(id)));
}

/* There's just one clipboard, I think, so these callbacks must be globals */
static Xen xm_XmCutPasteProc;

static void gxm_XmCutPasteProc(Widget w, long *data, long *privater, int *reason)
{
  Xen_call_with_4_args(xm_XmCutPasteProc, 
	     C_to_Xen_Widget(w),
	     C_ulong_to_Xen_ulong(*data),
	     C_ulong_to_Xen_ulong(*privater),
	     C_int_to_Xen_integer(*reason),
	     __func__);
}

static Xen gxm_XmClipboardStartCopy(Xen arg1, Xen arg2, Xen arg3, Xen arg4, Xen arg5, Xen arg6)
{
  #define H_XmClipboardStartCopy "int XmClipboardStartCopy (display, window, clip_label, timestamp, widget, callback) \
sets up a storage and data structure, returns id"
  /* DIFF: XmClipboardStartCopy omits and returns last arg
   */
  long id;
  int val;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XmClipboardStartCopy", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XmClipboardStartCopy", "Window");
  Xen_check_type(Xen_is_XmString(arg3), arg3, 3, "XmClipboardStartCopy", "XmString");
  Xen_check_type(Xen_is_Time(arg4), arg4, 4, "XmClipboardStartCopy", "Time");
  Xen_check_type(Xen_is_Widget(arg5), arg5, 5, "XmClipboardStartCopy", "Widget");
  Xen_check_type(Xen_is_procedure(arg6) && (Xen_is_aritable(arg6, 4)), arg6, 6, "XmClipboardStartCopy", "(XmCutPasteProc widget data priv reason)");
  xm_protect(arg6);
  if (Xen_is_procedure(xm_XmCutPasteProc)) xm_unprotect(xm_XmCutPasteProc);
  xm_XmCutPasteProc = arg6;
  val = XmClipboardStartCopy(Xen_to_C_Display(arg1), Xen_to_C_Window(arg2), Xen_to_C_XmString(arg3), 
			     Xen_to_C_Time(arg4), Xen_to_C_Widget(arg5), (XmCutPasteProc)gxm_XmCutPasteProc, &id);
  return(Xen_list_2(C_int_to_Xen_integer(val),
		    C_int_to_Xen_integer(id)));
}

static Xen xm_XmVoidProc;

static void gxm_XmVoidProc(Widget w, int *data, int *privater, int *reason)
{
  Xen_call_with_4_args(xm_XmVoidProc, 
	     C_to_Xen_Widget(w),
	     C_ulong_to_Xen_ulong(*data),
	     C_ulong_to_Xen_ulong(*privater),
	     C_int_to_Xen_integer(*reason),
	     __func__);
}

static Xen gxm_XmClipboardBeginCopy(Xen arg1, Xen arg2, Xen arg3, Xen arg4, Xen arg5)
{
  #define H_XmClipboardBeginCopy "int XmClipboardBeginCopy(display, window, XmString label, widget, callback)"
  /* DIFF: XmClipboardBeinCopy omits and returns last arg
   */
  long id;
  int val;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XmClipboardBeginCopy", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XmClipboardBeginCopy", "Window");
  Xen_check_type(Xen_is_XmString(arg3), arg3, 3, "XmClipboardBeginCopy", "XmString");
  Xen_check_type(Xen_is_Widget(arg4), arg4, 4, "XmClipboardBeginCopy", "Widget");
  Xen_check_type(Xen_is_procedure(arg5) && (Xen_is_aritable(arg5, 4)), arg5, 5, "XmClipboardBeginCopy", "(XmVoidProc widget data priv reason)");
  xm_protect(arg5);
  if (Xen_is_procedure(xm_XmVoidProc)) xm_unprotect(xm_XmVoidProc);
  xm_XmVoidProc = arg5;
  val = XmClipboardBeginCopy(Xen_to_C_Display(arg1), Xen_to_C_Window(arg2), 
			     Xen_to_C_XmString(arg3), Xen_to_C_Widget(arg4), 
			     gxm_XmVoidProc, &id);
  return(Xen_list_2(C_int_to_Xen_integer(val),
		    C_int_to_Xen_integer(id)));
}

#define Xen_is_ScaleWidget(Arg) (Xen_is_Widget(Arg) && XmIsScale(Xen_to_C_Widget(Arg)))

static Xen gxm_XmScaleSetTicks(Xen arg1, Xen arg2, Xen arg3, Xen arg4, Xen arg5, Xen arg6, Xen arg7)
{
  #define H_XmScaleSetTicks "void XmScaleSetTicks(Widget scale, int big_every, Cardinal num_medium, Cardinal num_small, Dimension size_big, \
Dimension size_medium, Dimension size_small) controls Scale tick marks"
  Xen_check_type(Xen_is_ScaleWidget(arg1), arg1, 1, "XmScaleSetTicks", "Scale Widget");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XmScaleSetTicks", "int");
  Xen_check_type(Xen_is_integer(arg3), arg3, 3, "XmScaleSetTicks", "int");
  Xen_check_type(Xen_is_integer(arg4), arg4, 4, "XmScaleSetTicks", "int");
  Xen_check_type(Xen_is_Dimension(arg5), arg5, 5, "XmScaleSetTicks", "Dimension");
  Xen_check_type(Xen_is_Dimension(arg6), arg6, 6, "XmScaleSetTicks", "Dimension");
  Xen_check_type(Xen_is_Dimension(arg7), arg7, 7, "XmScaleSetTicks", "Dimension");
  XmScaleSetTicks(Xen_to_C_Widget(arg1), 
		  Xen_integer_to_C_int(arg2), Xen_integer_to_C_int(arg3), Xen_integer_to_C_int(arg4), 
		  Xen_to_C_Dimension(arg5), Xen_to_C_Dimension(arg6), Xen_to_C_Dimension(arg7));
  return(Xen_false);
}

static Xen gxm_XmCreateScale(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XmCreateScale "Widget XmCreateScale(Widget parent, String name, ArgList arglist, Cardinal argcount) \
The Scale widget creation function"
  return(gxm_new_widget("XmCreateScale", XmCreateScale, arg1, arg2, arg3, arg4));
}

static Xen gxm_XmScaleGetValue(Xen arg1)
{
  #define H_XmScaleGetValue "void XmScaleGetValue(Widget widget): returns the current (scale) slider position"
  /* DIFF: XmScaleGetValue omits and returns arg2
   */
  int val;
  Xen_check_type(Xen_is_ScaleWidget(arg1), arg1, 1, "XmScaleGetValue", "Scale Widget");
  XmScaleGetValue(Xen_to_C_Widget(arg1), &val);
  return(C_int_to_Xen_integer(val));
}

static Xen gxm_XmScaleSetValue(Xen arg1, Xen arg2)
{
  #define H_XmScaleSetValue "void XmScaleSetValue(Widget widget, int value) sets a Scale slider value"
  Xen_check_type(Xen_is_ScaleWidget(arg1), arg1, 1, "XmScaleSetValue", "Scale Widget");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XmScaleSetValue", "int");
  XmScaleSetValue(Xen_to_C_Widget(arg1), Xen_integer_to_C_int(arg2));
  return(Xen_false);
}

#define Xen_is_ContainerWidget(Arg) (Xen_is_Widget(Arg) && XmIsContainer(Xen_to_C_Widget(Arg)))

static Xen gxm_XmContainerPasteLink(Xen arg1)
{
  #define H_XmContainerPasteLink "Boolean XmContainerPasteLink(Widget container) container function to insert links from the clipboard"
  Xen_check_type(Xen_is_ContainerWidget(arg1), arg1, 1, "XmContainerPasteLink", "Container Widget");
  return(C_bool_to_Xen_boolean(XmContainerPasteLink(Xen_to_C_Widget(arg1))));
}

static Xen gxm_XmContainerCopyLink(Xen arg1, Xen arg2)
{
  #define H_XmContainerCopyLink "Boolean XmContainerCopyLink(Widget container, Time timestamp) Container function to copy links to the clipboard"
  Xen_check_type(Xen_is_ContainerWidget(arg1), arg1, 1, "XmContainerCopyLink", "Container Widget");
  Xen_check_type(Xen_is_Time(arg2), arg2, 2, "XmContainerCopyLink", "Time");
  return(C_bool_to_Xen_boolean(XmContainerCopyLink(Xen_to_C_Widget(arg1), Xen_to_C_Time(arg2))));
}

static Xen gxm_XmContainerPaste(Xen arg1)
{
  #define H_XmContainerPaste "Boolean XmContainerPaste(Widget container) Container function to insert items from the clipboard"
  Xen_check_type(Xen_is_ContainerWidget(arg1), arg1, 1, "XmContainerPaste", "Container Widget");
  return(C_bool_to_Xen_boolean(XmContainerPaste(Xen_to_C_Widget(arg1))));
}

static Xen gxm_XmContainerCopy(Xen arg1, Xen arg2)
{
  #define H_XmContainerCopy "Boolean XmContainerCopy(Widget container, Time timestamp) Container function to copy primary selection to the clipboard"
  Xen_check_type(Xen_is_ContainerWidget(arg1), arg1, 1, "XmContainerCopy", "Container Widget");
  Xen_check_type(Xen_is_Time(arg2), arg2, 2, "XmContainerCopy", "Time");
  return(C_bool_to_Xen_boolean(XmContainerCopy(Xen_to_C_Widget(arg1), Xen_to_C_Time(arg2))));
}

static Xen gxm_XmContainerCut(Xen arg1, Xen arg2)
{
  #define H_XmContainerCut "Boolean XmContainerCut(Widget container, Time timestamp) Container function to move items to the clipboard"
  Xen_check_type(Xen_is_ContainerWidget(arg1), arg1, 1, "XmContainerCut", "Container Widget");
  Xen_check_type(Xen_is_Time(arg2), arg2, 2, "XmContainerCut", "Time");
  return(C_bool_to_Xen_boolean(XmContainerCut(Xen_to_C_Widget(arg1), Xen_to_C_Time(arg2))));
}

static Xen gxm_XmContainerReorder(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XmContainerReorder "void XmContainerReorder(Widget container, WidgetList widgets, int num_widgets) Container function to reorder children"
  /* DIFF: XmContainerReorder arg2 is list of Widgets
   */
  int len;
  Xen_check_type(Xen_is_ContainerWidget(arg1), arg1, 1, "XmContainerReorder", "Container Widget");
  Xen_check_type(Xen_is_list(arg2), arg2, 2, "XmContainerReorder", "WidgetList");
  Xen_check_type(Xen_is_integer(arg3), arg3, 3, "XmContainerReorder", "int");
  len = Xen_integer_to_C_int(arg3);
  if (len > 0)
    {
      if (len > Xen_list_length(arg2))
	Xen_out_of_range_error("XmContainerReorder", 3, arg3, "len too large");
      else
	{
	  WidgetList ws;
	  ws = Xen_to_C_Widgets(arg2, len);
	  XmContainerReorder(Xen_to_C_Widget(arg1), ws, len);
	  if (ws) free(ws);
	}
    }
  return(Xen_false);
}

static Xen gxm_XmContainerRelayout(Xen arg1)
{
  #define H_XmContainerRelayout "void XmContainerRelayout(Widget container) Container relayout function"
  Xen_check_type(Xen_is_ContainerWidget(arg1), arg1, 1, "XmContainerRelayout", "Container Widget");
  XmContainerRelayout(Xen_to_C_Widget(arg1));
  return(Xen_false);
}

static Xen gxm_XmContainerGetItemChildren(Xen arg1, Xen arg2)
{
  #define H_XmContainerGetItemChildren "int XmContainerGetItemChildren(Widget container, Widget item) \
returns a list of all children of an item"
  /* DIFF: XmContainerGetItemChildren omits arg3, rtns widget list
   */
  Widget *ws;
  int len;
  Xen_check_type(Xen_is_ContainerWidget(arg1), arg1, 1, "XmContainerGetItemChildren", "Container Widget");
  Xen_check_type(Xen_is_Widget(arg2), arg2, 2, "XmContainerGetItemChildren", "Widget");
  len = XmContainerGetItemChildren(Xen_to_C_Widget(arg1), Xen_to_C_Widget(arg2), &ws);
  return(C_to_Xen_Widgets(ws, len));
}

static Xen gxm_XmCreateContainer(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XmCreateContainer "Widget XmCreateContainer(Widget parent, String name, ArgList arglist, Cardinal argcount) \
The Container creation function"
  return(gxm_new_widget("XmCreateContainer", XmCreateContainer, arg1, arg2, arg3, arg4));
}

static Xen gxm_XmGetTearOffControl(Xen arg1)
{
  #define H_XmGetTearOffControl "Widget XmGetTearOffControl(Widget menu) obtains the widget ID for the tear-off control in a menu"
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XmGetTearOffControl", "Widget");
  return(C_to_Xen_Widget(XmGetTearOffControl(Xen_to_C_Widget(arg1))));
}

static Xen gxm_XmGetPostedFromWidget(Xen arg1)
{
  #define H_XmGetPostedFromWidget "Widget XmGetPostedFromWidget(Widget menu): returns the widget from which a menu was posted"
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XmGetPostedFromWidget", "Widget");
  return(C_to_Xen_Widget(XmGetPostedFromWidget(Xen_to_C_Widget(arg1))));
}

static Xen gxm_XmCreatePulldownMenu(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XmCreatePulldownMenu "Widget XmCreatePulldownMenu(Widget parent, String name, ArgList arglist, Cardinal argcount) \
A RowColumn widget creation function"
  return(gxm_new_widget("XmCreatePulldownMenu", XmCreatePulldownMenu, arg1, arg2, arg3, arg4));
}

static Xen gxm_XmCreatePopupMenu(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XmCreatePopupMenu "Widget XmCreatePopupMenu(Widget parent, String name, ArgList arglist, Cardinal argcount) \
A RowColumn widget creation function"
  return(gxm_new_widget("XmCreatePopupMenu", XmCreatePopupMenu, arg1, arg2, arg3, arg4));
}

static Xen gxm_XmCreateMenuBar(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XmCreateMenuBar "Widget XmCreateMenuBar(Widget parent, String name, ArgList arglist, Cardinal argcount) \
A RowColumn widget creation function"
  return(gxm_new_widget("XmCreateMenuBar", XmCreateMenuBar, arg1, arg2, arg3, arg4));
}

static Xen gxm_XmOptionButtonGadget(Xen arg1)
{
  #define H_XmOptionButtonGadget "Widget XmOptionButtonGadget(Widget option_menu) obtains \
the widget ID for the CascadeButtonGadget in an OptionMenu"
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XmOptionButtonGadget", "Widget");
  return(C_to_Xen_Widget(XmOptionButtonGadget(Xen_to_C_Widget(arg1))));
}

static Xen gxm_XmOptionLabelGadget(Xen arg1)
{
  #define H_XmOptionLabelGadget "Widget XmOptionLabelGadget(Widget option_menu) obtains the \
widget ID for the LabelGadget in an OptionMenu"
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XmOptionLabelGadget", "Widget");
  return(C_to_Xen_Widget(XmOptionLabelGadget(Xen_to_C_Widget(arg1))));
}

static Xen gxm_XmCreateOptionMenu(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XmCreateOptionMenu "Widget XmCreateOptionMenu(Widget parent, String name, ArgList arglist, Cardinal argcount) \
A RowColumn widget creation function"
  return(gxm_new_widget("XmCreateOptionMenu", XmCreateOptionMenu, arg1, arg2, arg3, arg4));
}

static Xen gxm_XmCreateRadioBox(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XmCreateRadioBox "Widget XmCreateRadioBox(Widget parent, String name, ArgList arglist, Cardinal argcount) \
A RowColumn widget creation function"
  return(gxm_new_widget("XmCreateRadioBox", XmCreateRadioBox, arg1, arg2, arg3, arg4));
}

static Xen gxm_XmCreateWorkArea(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XmCreateWorkArea "Widget XmCreateWorkArea(Widget parent, String name, ArgList arglist, Cardinal argcount) \
creates a RowColumn WorkArea"
  return(gxm_new_widget("XmCreateWorkArea", XmCreateWorkArea, arg1, arg2, arg3, arg4));
}

static Xen gxm_XmCreateRowColumn(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XmCreateRowColumn "Widget XmCreateRowColumn(Widget parent, String name, ArgList arglist, Cardinal argcount) \
The RowColumn widget creation function"
  return(gxm_new_widget("XmCreateRowColumn", XmCreateRowColumn, arg1, arg2, arg3, arg4));
}

static Xen gxm_XmMenuPosition(Xen arg1, Xen arg2)
{
  #define H_XmMenuPosition "void XmMenuPosition(Widget menu, XButtonPressedEvent *event) positions a Popup menu pane"
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XmMenuPosition", "Widget");
  Xen_check_type(Xen_is_XButtonEvent(arg2), arg2, 2, "XmMenuPosition", "XButtonPressedEvent*");
  XmMenuPosition(Xen_to_C_Widget(arg1), Xen_to_C_XButtonEvent(arg2));
  return(Xen_false);
}

static Xen gxm_XmCreateCommandDialog(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XmCreateCommandDialog "Widget XmCreateCommandDialog(Widget parent, String name, ArgList arglist, Cardinal argcount) \
The Command CommandDialog creation function"
  return(gxm_new_widget("XmCreateCommandDialog", XmCreateCommandDialog, arg1, arg2, arg3, arg4));
}

#define Xen_is_CommandWidget(Arg) (Xen_is_Widget(Arg) && XmIsCommand(Xen_to_C_Widget(Arg)))

static Xen gxm_XmCommandError(Xen arg1, Xen arg2)
{
  #define H_XmCommandError "void XmCommandError(Widget widget, XmString error) Command function that displays an error message"
  Xen_check_type(Xen_is_CommandWidget(arg1), arg1, 1, "XmCommandError", "Command Widget");
  Xen_check_type(Xen_is_XmString(arg2), arg2, 2, "XmCommandError", "XmString");
  XmCommandError(Xen_to_C_Widget(arg1), Xen_to_C_XmString(arg2));
  return(Xen_false);
}

static Xen gxm_XmCommandAppendValue(Xen arg1, Xen arg2)
{
  #define H_XmCommandAppendValue "void XmCommandAppendValue(Widget widget, XmString command) \
appends the passed XmString to the end of the string displayed in the command area of the widget"
  Xen_check_type(Xen_is_CommandWidget(arg1), arg1, 1, "XmCommandAppendValue", "Command Widget");
  Xen_check_type(Xen_is_XmString(arg2), arg2, 2, "XmCommandAppendValue", "XmString");
  XmCommandAppendValue(Xen_to_C_Widget(arg1), Xen_to_C_XmString(arg2));
  return(Xen_false);
}

static Xen gxm_XmCommandSetValue(Xen arg1, Xen arg2)
{
  #define H_XmCommandSetValue "void XmCommandSetValue(Widget widget, XmString command) A Command function that replaces a displayed string"
  Xen_check_type(Xen_is_CommandWidget(arg1), arg1, 1, "XmCommandSetValue", "Command Widget");
  Xen_check_type(Xen_is_XmString(arg2), arg2, 2, "XmCommandSetValue", "XmString");
  XmCommandSetValue(Xen_to_C_Widget(arg1), Xen_to_C_XmString(arg2));
  return(Xen_false);
}

static Xen gxm_XmCommandGetChild(Xen arg1, Xen arg2)
{
  #define H_XmCommandGetChild "Widget XmCommandGetChild(Widget widget, unsigned char child) A Command function that is used to access a component"
  Xen_check_type(Xen_is_CommandWidget(arg1), arg1, 1, "XmCommandGetChild", "Command Widget");
  Xen_check_type(Xen_is_ulong(arg2), arg2, 2, "XmCommandGetChild", "unsigned int");
  return(C_to_Xen_Widget(XmCommandGetChild(Xen_to_C_Widget(arg1), Xen_ulong_to_C_ulong(arg2))));
}

static Xen gxm_XmCreateCommand(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XmCreateCommand "Widget XmCreateCommand(Widget parent, String name, ArgList arglist, Cardinal argcount) \
The Command widget creation function"
  return(gxm_new_widget("XmCreateCommand", XmCreateCommand, arg1, arg2, arg3, arg4));
}

#define Xen_is_ComboBoxWidget(Arg) (Xen_is_Widget(Arg) && XmIsComboBox(Xen_to_C_Widget(Arg)))

static Xen gxm_XmComboBoxUpdate(Xen arg1)
{
  #define H_XmComboBoxUpdate "void XmComboBoxUpdate(Widget widget) A ComboBox function that resynchronizes data"
  Xen_check_type(Xen_is_ComboBoxWidget(arg1), arg1, 1, "XmComboBoxUpdate", "ComboBox Widget");
  XmComboBoxUpdate(Xen_to_C_Widget(arg1));
  return(Xen_false);
}

static Xen gxm_XmComboBoxSetItem(Xen arg1, Xen arg2)
{
  #define H_XmComboBoxSetItem "void XmComboBoxSetItem(Widget w, XmString item) set an item in the XmComboBox list"
  Xen_check_type(Xen_is_ComboBoxWidget(arg1), arg1, 1, "XmComboBoxSetItem", "ComboBox Widget");
  Xen_check_type(Xen_is_XmString(arg2), arg2, 2, "XmComboBoxSetItem", "XmString");
  XmComboBoxSetItem(Xen_to_C_Widget(arg1), Xen_to_C_XmString(arg2));
  return(Xen_false);
}

static Xen gxm_XmComboBoxSelectItem(Xen arg1, Xen arg2)
{
  #define H_XmComboBoxSelectItem "void XmComboBoxSelectItem(Widget w, XmString item) select a XmComboBox item"
  Xen_check_type(Xen_is_ComboBoxWidget(arg1), arg1, 1, "XmComboBoxSelectItem", "ComboBox Widget");
  Xen_check_type(Xen_is_XmString(arg2), arg2, 2, "XmComboBoxSelectItem", "XmString");
  XmComboBoxSelectItem(Xen_to_C_Widget(arg1), Xen_to_C_XmString(arg2));
  return(Xen_false);
}

static Xen gxm_XmComboBoxDeletePos(Xen arg1, Xen arg2)
{
  #define H_XmComboBoxDeletePos "void XmComboBoxDeletePos(Widget w, int pos)  Delete a XmComboBox item"
  Xen_check_type(Xen_is_ComboBoxWidget(arg1), arg1, 1, "XmComboBoxDeletePos", "ComboBox Widget");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XmComboBoxDeletePos", "int");
  XmComboBoxDeletePos(Xen_to_C_Widget(arg1), Xen_integer_to_C_int(arg2));
  return(Xen_false);
}

static Xen gxm_XmComboBoxAddItem(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XmComboBoxAddItem "void XmComboBoxAddItem(Widget w, XmString item, int pos, Boolean unique) add an item to the ComboBox widget"
  Xen_check_type(Xen_is_ComboBoxWidget(arg1), arg1, 1, "XmComboBoxAddItem", "ComboBox Widget");
  Xen_check_type(Xen_is_XmString(arg2), arg2, 2, "XmComboBoxAddItem", "XmString");
  Xen_check_type(Xen_is_integer(arg3), arg3, 3, "XmComboBoxAddItem", "int");
  Xen_check_type(Xen_is_boolean(arg4), arg4, 4, "XmComboBoxAddItem", "Boolean");
  XmComboBoxAddItem(Xen_to_C_Widget(arg1), Xen_to_C_XmString(arg2), Xen_integer_to_C_int(arg3), Xen_boolean_to_C_bool(arg4));
  return(Xen_false);
}

static Xen gxm_XmCreateDropDownList(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XmCreateDropDownList "Widget XmCreateDropDownList(Widget parent, String name, ArgList arglist, Cardinal arg_count) \
The Drop-down list ComboBox widget creation function"
  return(gxm_new_widget("XmCreateDropDownList", XmCreateDropDownList, arg1, arg2, arg3, arg4));
}

static Xen gxm_XmCreateDropDownComboBox(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XmCreateDropDownComboBox "Widget XmCreateDropDownComboBox(Widget parent, String name, ArgList arglist, Cardinal arg_count) \
The Drop-down ComboBox widget creation function"
  return(gxm_new_widget("XmCreateDropDownComboBox", XmCreateDropDownComboBox, arg1, arg2, arg3, arg4));
}

static Xen gxm_XmCreateComboBox(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XmCreateComboBox "Widget XmCreateComboBox(Widget parent, String name, ArgList arglist, Cardinal arg_count) \
The default ComboBox widget creation function"
  return(gxm_new_widget("XmCreateComboBox", XmCreateComboBox, arg1, arg2, arg3, arg4));
}

static Xen gxm_XmCreatePushButton(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XmCreatePushButton "Widget XmCreatePushButton(Widget parent, String name, ArgList arglist, Cardinal argcount) \
The PushButton widget creation function"
  return(gxm_new_widget("XmCreatePushButton", XmCreatePushButton, arg1, arg2, arg3, arg4));
}

static Xen gxm_XmCreatePushButtonGadget(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XmCreatePushButtonGadget "Widget XmCreatePushButtonGadget(Widget parent, String name, ArgList arglist, Cardinal argcount) \
The PushButtonGadget creation function"
  return(gxm_new_widget("XmCreatePushButtonGadget", XmCreatePushButtonGadget, arg1, arg2, arg3, arg4));
}

static Xen gxm_XmCascadeButtonHighlight(Xen arg1, Xen arg2)
{
  #define H_XmCascadeButtonHighlight "void XmCascadeButtonHighlight(Widget cascadeButton, Boolean highlight) A CascadeButton and \
CascadeButtonGadget function that sets the highlight state"
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XmCascadeButtonHighlight", "Widget");
  Xen_check_type(Xen_is_boolean(arg2), arg2, 2, "XmCascadeButtonHighlight", "boolean");
  XmCascadeButtonHighlight(Xen_to_C_Widget(arg1), Xen_boolean_to_C_bool(arg2));
  return(Xen_false);
}

static Xen gxm_XmCreateCascadeButton(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XmCreateCascadeButton "Widget XmCreateCascadeButton(Widget parent, String name, ArgList arglist, Cardinal argcount) \
The CascadeButton widget creation function"
  return(gxm_new_widget("XmCreateCascadeButton", XmCreateCascadeButton, arg1, arg2, arg3, arg4));
}

static void gxm_ProtocolProc(Widget w, XtPointer context, XtPointer info)
{
  Xen descr = (Xen)context;
  Xen_call_with_3_args(Xen_cadr(descr),
	     C_to_Xen_Widget(w),
	     Xen_caddr(descr),
	     Xen_wrap_C_pointer(info), /* what's this? */
	     __func__);
}

#define C_to_Xen_XM_ProtocolHook(Code, Context, PropertyAtom, ProtocolAtom) \
  Xen_list_5(C_string_to_Xen_symbol("ProtocolHook"), Code, Context, PropertyAtom, ProtocolAtom)
/* #define XM_ProtocolHook_P(Arg) is_wrapped("ProtocolHook", Arg) */

#define C_to_Xen_XM_ProtocolProc(Code, Context, PropertyAtom, ProtocolAtom) \
  Xen_list_5(C_string_to_Xen_symbol("ProtocolProc"), Code, Context, PropertyAtom, ProtocolAtom)
#define XM_is_ProtocolProc(Arg) is_wrapped("ProtocolProc", Arg)

static Xen gxm_XmSetProtocolHooks(Xen arg1, Xen arg2, Xen arg3, Xen arg4, Xen arg5, Xen arg6, Xen arg7)
{
  #define H_XmSetProtocolHooks "void XmSetProtocolHooks(Widget shell, Atom property, Atom protocol, XtCallbackProc prehook, \
XtPointer pre_closure, XtCallbackProc posthook, XtPointer post_closure) A VendorShell function that allows preactions and postactions \
to be executed when a protocol message is received from MWM"
  Xen descr1, descr2;
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XmSetProtocolHooks", "Widget");
  Xen_check_type(Xen_is_Atom(arg2), arg2, 2, "XmSetProtocolHooks", "Atom");
  Xen_check_type(Xen_is_Atom(arg3), arg3, 3, "XmSetProtocolHooks", "Atom");
  Xen_check_type(Xen_is_procedure(arg4) && (Xen_is_aritable(arg4, 3)), arg4, 4, "XmSetProtocolHooks", "(XtCallbackProc widget data callb)");
  Xen_check_type(Xen_is_procedure(arg6) && (Xen_is_aritable(arg6, 3)), arg6, 6, "XmSetProtocolHooks", "(XtCallbackProc widget data callb)");
  descr1 = C_to_Xen_XM_ProtocolHook(arg4, arg5, arg2, arg3);
  descr2 = C_to_Xen_XM_ProtocolHook(arg6, arg7, arg2, arg3);
  xm_protect(descr1);
  xm_protect(descr2);
  XmSetProtocolHooks(Xen_to_C_Widget(arg1), Xen_to_C_Atom(arg2), Xen_to_C_Atom(arg3),
		     gxm_ProtocolProc, (XtPointer)descr1,
		     gxm_ProtocolProc, (XtPointer)descr2);
  return(Xen_false);
}

static Xen gxm_XmSetWMProtocolHooks(Xen arg1, Xen arg2, Xen arg3, Xen arg4, Xen arg5, Xen arg6)
{
  #define H_XmSetWMProtocolHooks "void XmSetWMProtocolHooks(Widget shell, Atom property, XtCallbackProc prehook, \
XtPointer pre_closure, XtCallbackProc posthook, XtPointer post_closure) A VendorShell function that allows preactions and postactions \
to be executed when a protocol message is received from MWM"
  Xen descr1, descr2, wm_atom;
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XmSetWMProtocolHooks", "Widget");
  Xen_check_type(Xen_is_Atom(arg2), arg2, 2, "XmSetWMProtocolHooks", "Atom");
  Xen_check_type(Xen_is_procedure(arg3) && (Xen_is_aritable(arg3, 3)), arg3, 3, "XmSetWMProtocolHooks", "(XtCallbackProc widget data callb)");
  Xen_check_type(Xen_is_procedure(arg5) && (Xen_is_aritable(arg5, 3)), arg5, 5, "XmSetWMProtocolHooks", "(XtCallbackProc widget data callb)");
  wm_atom = C_to_Xen_Atom(XInternAtom(XtDisplay(Xen_to_C_Widget(arg1)), "WM_PROTOCOLS", false));
  descr1 = C_to_Xen_XM_ProtocolHook(arg3, arg4, arg2, wm_atom);
  descr2 = C_to_Xen_XM_ProtocolHook(arg5, arg6, arg2, wm_atom);
  xm_protect(descr1);
  xm_protect(descr2);
  XmSetWMProtocolHooks(Xen_to_C_Widget(arg1), Xen_to_C_Atom(arg2),
		       gxm_ProtocolProc, (XtPointer)descr1,
		       gxm_ProtocolProc, (XtPointer)descr2);
  return(Xen_false);
}

static Xen gxm_XmDeactivateProtocol(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XmDeactivateProtocol "void XmDeactivateProtocol(Widget shell, Atom property, Atom protocol) \
deactivates a protocol without removing it"
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XmDeactivateProtocol", "Widget");
  Xen_check_type(Xen_is_Atom(arg2), arg2, 2, "XmDeactivateProtocol", "Atom");
  Xen_check_type(Xen_is_Atom(arg3), arg3, 3, "XmDeactivateProtocol", "Atom");
  XmDeactivateProtocol(Xen_to_C_Widget(arg1), Xen_to_C_Atom(arg2), Xen_to_C_Atom(arg3));
  return(Xen_false);
}

static Xen gxm_XmDeactivateWMProtocol(Xen arg1, Xen arg2)
{
  #define H_XmDeactivateWMProtocol "void XmDeactivateWMProtocol(Widget shell, Atom property) \
deactivates a property in WM_PROTOCOLS without removing it"
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XmDeactivateProtocol", "Widget");
  Xen_check_type(Xen_is_Atom(arg2), arg2, 2, "XmDeactivateProtocol", "Atom");
  XmDeactivateWMProtocol(Xen_to_C_Widget(arg1), Xen_to_C_Atom(arg2));
  return(Xen_false);
}

static Xen gxm_XmActivateProtocol(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XmActivateProtocol "oid XmActivateProtocol(Widget shell, Atom property, Atom protocol) activates a protocol"
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XmActivateProtocol", "Widget");
  Xen_check_type(Xen_is_Atom(arg2), arg2, 2, "XmActivateProtocol", "Atom");
  Xen_check_type(Xen_is_Atom(arg3), arg3, 3, "XmActivateProtocol", "Atom");
  XmActivateProtocol(Xen_to_C_Widget(arg1), Xen_to_C_Atom(arg2), Xen_to_C_Atom(arg3));
  return(Xen_false);
}

static Xen gxm_XmActivateWMProtocol(Xen arg1, Xen arg2)
{
  #define H_XmActivateWMProtocol "oid XmActivateWMProtocol(Widget shell, Atom property) activates a property in the WM_PROTOCOLS"
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XmActivateProtocol", "Widget");
  Xen_check_type(Xen_is_Atom(arg2), arg2, 2, "XmActivateProtocol", "Atom");
  XmActivateWMProtocol(Xen_to_C_Widget(arg1), Xen_to_C_Atom(arg2));
  return(Xen_false);
}

static bool unprotect_protocolproc(Xen val, int loc, unsigned long udescr)
{
  Xen descr = (Xen)udescr; /* ('protocolproc func data propatom protoatom) */
  return((XM_is_ProtocolProc(val)) &&
	 (Xen_ulong_to_C_ulong(Xen_cadr(val)) == Xen_ulong_to_C_ulong(Xen_cadr(descr))) &&
	 (Xen_ulong_to_C_ulong(Xen_caddr(val)) == Xen_ulong_to_C_ulong(Xen_caddr(descr))) &&
	 (Xen_to_C_Atom(Xen_list_ref(val, 3)) == Xen_to_C_Atom(Xen_list_ref(val,3))) &&
	 (Xen_to_C_Atom(Xen_list_ref(val, 4)) == Xen_to_C_Atom(Xen_list_ref(val,4))));
}

static Xen gxm_XmRemoveProtocolCallback(Xen arg1, Xen arg2, Xen arg3, Xen arg4, Xen arg5)
{
  #define H_XmRemoveProtocolCallback "void XmRemoveProtocolCallback(Widget shell, Atom property, Atom protocol, XtCallbackProc callback, \
XtPointer closure) removes a callback from the internal list"
  Xen descr;
  int loc, dloc;
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XmRemoveProtocolCallback", "Widget");
  Xen_check_type(Xen_is_Atom(arg2), arg2, 2, "XmRemoveProtocolCallback", "Atom");
  Xen_check_type(Xen_is_Atom(arg3), arg3, 3, "XmRemoveProtocolCallback", "Atom"); 
  Xen_check_type(Xen_is_procedure(arg4) && (Xen_is_aritable(arg4, 3)), arg4, 4, "XmRemoveProtocolCallback", "XtCallbackProc (3 args)");
  descr = C_to_Xen_XM_ProtocolProc(arg4, arg5, arg2, arg3);
  dloc = xm_protect(descr);
  loc = map_over_protected_elements(unprotect_protocolproc, (unsigned long)descr);
  XmRemoveProtocolCallback(Xen_to_C_Widget(arg1), 
			   Xen_to_C_Atom(arg2), 
			   Xen_to_C_Atom(arg3), 
			   gxm_ProtocolProc,
			   (XtPointer)(Xen_vector_ref(xm_protected, loc))); /* this was the original tag passed in */
  /* now unprotect the proc and our descr */
  xm_unprotect_at(dloc);
  xm_unprotect_at(loc);
  return(Xen_false);
}

static Xen gxm_XmRemoveWMProtocolCallback(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XmRemoveWMProtocolCallback "void XmRemoveWMProtocolCallback(Widget shell, Atom property, XtCallbackProc callback, \
XtPointer closure) removes a callback from the WM_PROTOCOLS list"
  Xen descr;
  int loc, dloc;
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XmRemoveWMProtocolCallback", "Widget");
  Xen_check_type(Xen_is_Atom(arg2), arg2, 2, "XmRemoveWMProtocolCallback", "Atom");
  Xen_check_type(Xen_is_procedure(arg3) && (Xen_is_aritable(arg3, 3)), arg3, 3, "XmRemoveWMProtocolCallback", "XtCallbackProc (3 args)");
  descr = C_to_Xen_XM_ProtocolProc(arg3, arg4, arg2, C_to_Xen_Atom(XInternAtom(XtDisplay(Xen_to_C_Widget(arg1)), "WM_PROTOCOLS", false)));
  dloc = xm_protect(descr);
  loc = map_over_protected_elements(unprotect_protocolproc, (unsigned long)descr);
  XmRemoveWMProtocolCallback(Xen_to_C_Widget(arg1), 
			     Xen_to_C_Atom(arg2), 
			     gxm_ProtocolProc,
			     (XtPointer)(Xen_vector_ref(xm_protected, loc))); /* this was the original tag passed in */
  /* now unprotect the proc and our descr */
  xm_unprotect_at(dloc);
  xm_unprotect_at(loc);
  return(Xen_false);
}

static Xen gxm_XmAddProtocolCallback(Xen arg1, Xen arg2, Xen arg3, Xen arg4, Xen arg5)
{
  #define H_XmAddProtocolCallback "void XmAddProtocolCallback(Widget shell, Atom property, Atom protocol, XtCallbackProc callback, \
XtPointer closure) adds client callbacks for a protocol"
  Xen descr;
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XmAddProtocolCallback", "Widget");
  Xen_check_type(Xen_is_Atom(arg2), arg2, 2, "XmAddProtocolCallback", "Atom");
  Xen_check_type(Xen_is_Atom(arg3), arg3, 3, "XmAddProtocolCallback", "Atom");
  Xen_check_type(Xen_is_procedure(arg4) && (Xen_is_aritable(arg4, 3)), arg4, 4, "XmAddProtocolCallback", "(XtCallbackProc widget data callb)");
  descr = C_to_Xen_XM_ProtocolProc(arg4, arg5, arg2, arg3);
  xm_protect(descr);
  XmAddProtocolCallback(Xen_to_C_Widget(arg1), 
			Xen_to_C_Atom(arg2),
			Xen_to_C_Atom(arg3),
			gxm_ProtocolProc,
			(XtPointer)descr);
  return(Xen_false);
}

static Xen gxm_XmAddWMProtocolCallback(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XmAddWMProtocolCallback "void XmAddWMProtocolCallback(Widget shell, Atom property, XtCallbackProc callback, \
XtPointer closure) adds client callbacks for the WM_PROTOCOLS"
  Xen descr;
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XmAddWMProtocolCallback", "Widget");
  Xen_check_type(Xen_is_Atom(arg2), arg2, 2, "XmAddWMProtocolCallback", "Atom");
  Xen_check_type(Xen_is_procedure(arg3) && (Xen_is_aritable(arg3, 3)), arg3, 3, "XmAddWMProtocolCallback", "(XtCallbackProc widget data callb)");
  descr = C_to_Xen_XM_ProtocolProc(arg3, arg4, arg2, C_to_Xen_Atom(XInternAtom(XtDisplay(Xen_to_C_Widget(arg1)), "WM_PROTOCOLS", false)));
  xm_protect(descr);
  XmAddWMProtocolCallback(Xen_to_C_Widget(arg1), 
			  Xen_to_C_Atom(arg2), 
			  gxm_ProtocolProc,
			  (XtPointer)descr);
  return(Xen_false);
}

static Xen gxm_XmRemoveProtocols(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XmRemoveProtocols "void XmRemoveProtocols(Widget shell, Atom property, Atom *protocols, Cardinal num_protocols) \
removes the protocols from the protocol manager and deallocates the internal tables"
  /* DIFF: XmRemoveProtocols takes list of Atoms
   */
  Atom *outs;
  int len;
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XmRemoveProtocols", "Widget");
  Xen_check_type(Xen_is_Atom(arg2), arg2, 2, "XmRemoveProtocols", "Atom");
  Xen_check_type(Xen_is_list(arg3), arg3, 3, "XmRemoveProtocols", "list of Atom");
  Xen_check_type(Xen_is_integer_or_unbound(arg4), arg4, 4, "XmRemoveProtocols", "int");
  if (Xen_is_integer(arg4)) len = Xen_integer_to_C_int(arg4); else len = Xen_list_length(arg3);
  if (len <= 0) return(Xen_false);
  outs = Xen_to_C_Atoms(arg3, len);
  XmRemoveProtocols(Xen_to_C_Widget(arg1), Xen_to_C_Atom(arg2), outs, len);
  free(outs);
  return(Xen_false);
}

static Xen gxm_XmRemoveWMProtocols(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XmRemoveWMProtocols "void XmRemoveWMProtocols(Widget shell, Atom *protocols, Cardinal num_protocols) \
removes the protocols from WM_PROTOCOLS"
  Atom *outs;
  int len;
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XmRemoveWMProtocols", "Widget");
  Xen_check_type(Xen_is_list(arg2), arg2, 2, "XmRemoveWMProtocols", "list of Atom");
  Xen_check_type(Xen_is_integer_or_unbound(arg3), arg3, 3, "XmRemoveWMProtocols", "int");
  if (Xen_is_integer(arg3)) len = Xen_integer_to_C_int(arg3); else len = Xen_list_length(arg2);
  if (len <= 0) return(Xen_false);
  outs = Xen_to_C_Atoms(arg2, len);
  XmRemoveWMProtocols(Xen_to_C_Widget(arg1), outs, len);
  free(outs);
  return(Xen_false);
}

static Xen gxm_XmAddProtocols(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XmAddProtocols "void XmAddProtocols(Widget shell, Atom property, Atom *protocols, Cardinal num_protocols) \
adds the protocols to the protocol manager and allocates the internal tables"
  /* DIFF: XmAddProtocols takes list of Atoms
   */
  Atom *outs;
  int len;
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XmAddProtocols", "Widget");
  Xen_check_type(Xen_is_Atom(arg2), arg2, 2, "XmAddProtocols", "Atom");
  Xen_check_type(Xen_is_list(arg3), arg3, 3, "XmAddProtocols", "list of Atom");
  Xen_check_type(Xen_is_integer_or_unbound(arg4), arg4, 4, "XmAddProtocols", "int");
  if (Xen_is_integer(arg4)) len = Xen_integer_to_C_int(arg4); else len = Xen_list_length(arg3);
  if (len <= 0) return(Xen_false);
  outs = Xen_to_C_Atoms(arg3, len);
  XmAddProtocols(Xen_to_C_Widget(arg1), Xen_to_C_Atom(arg2), outs, len);
  free(outs);
  return(Xen_false);
}

static Xen gxm_XmAddWMProtocols(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XmAddWMProtocols "void XmAddWMProtocols(Widget shell, Atom *protocols, Cardinal num_protocols) \
adds the protocols to WM_PROTOCOLS"
  Atom *outs;
  int len;
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XmAddWMProtocols", "Widget");
  Xen_check_type(Xen_is_list(arg2), arg2, 2, "XmAddWMProtocols", "list of Atom");
  Xen_check_type(Xen_is_integer_or_unbound(arg3), arg3, 3, "XmAddWMProtocols", "int");
  if (Xen_is_integer(arg3)) len = Xen_integer_to_C_int(arg3); else len = Xen_list_length(arg2);
  if (len <= 0) return(Xen_false);
  outs = Xen_to_C_Atoms(arg2, len);
  XmAddWMProtocols(Xen_to_C_Widget(arg1), outs, len);
  free(outs);
  return(Xen_false);
}

static Xen gxm_XmCascadeButtonGadgetHighlight(Xen arg1, Xen arg2)
{
  #define H_XmCascadeButtonGadgetHighlight "void XmCascadeButtonGadgetHighlight(Widget cascadeButtonGadget, Boolean highlight) \
A CascadeButtonGadget function that sets the highlight state"
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XmCascadeButtonGadgetHighlight", "Widget");
  Xen_check_type(Xen_is_boolean(arg2), arg2, 2, "XmCascadeButtonGadgetHighlight", "boolean");
  XmCascadeButtonGadgetHighlight(Xen_to_C_Widget(arg1), Xen_boolean_to_C_bool(arg2));
  return(Xen_false);
}

static Xen gxm_XmCreateCascadeButtonGadget(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XmCreateCascadeButtonGadget "Widget XmCreateCascadeButtonGadget(Widget parent, String name, ArgList arglist, Cardinal argcount) \
The CascadeButtonGadget creation function"
  return(gxm_new_widget("XmCreateCascadeButtonGadget", XmCreateCascadeButtonGadget, arg1, arg2, arg3, arg4));
}

static Xen gxm_XmCreateBulletinBoardDialog(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XmCreateBulletinBoardDialog "Widget XmCreateBulletinBoardDialog(Widget parent, String name, ArgList arglist, Cardinal argcount) \
The BulletinBoard BulletinBoardDialog creation function"
  return(gxm_new_widget("XmCreateBulletinBoardDialog", XmCreateBulletinBoardDialog, arg1, arg2, arg3, arg4));
}

static Xen gxm_XmCreateBulletinBoard(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XmCreateBulletinBoard "Widget XmCreateBulletinBoard(Widget parent, String name, ArgList arglist, Cardinal argcount) \
The BulletinBoard widget creation function"
  return(gxm_new_widget("XmCreateBulletinBoard", XmCreateBulletinBoard, arg1, arg2, arg3, arg4));
}

static Xen gxm_XmCreatePanedWindow(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XmCreatePanedWindow "Widget XmCreatePanedWindow(Widget parent, String name, ArgList arglist, Cardinal argcount) \
The PanedWindow widget creation function"
  return(gxm_new_widget("XmCreatePanedWindow", XmCreatePanedWindow, arg1, arg2, arg3, arg4));
}

static Xen gxm_XmGetAtomName(Xen arg1, Xen arg2)
{
  char *str;
  Xen res;
  #define H_XmGetAtomName "String XmGetAtomName(Display *display, Atom atom)  returns the string representation for an atom"
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XmGetAtomName", "Display*");
  Xen_check_type(Xen_is_Atom(arg2), arg2, 2, "XmGetAtomName", "Atom");
  str = XmGetAtomName(Xen_to_C_Display(arg1), Xen_to_C_Atom(arg2));
  res = C_string_to_Xen_string(str);
  if (str) XtFree(str);
  return(res);
}

static Xen gxm_XmInternAtom(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XmInternAtom "A macro that returns an atom for a given name"
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XmInternAtom", "Display*");
  Xen_check_type(Xen_is_string(arg2), arg2, 2, "XmInternAtom", "String");
  Xen_check_type(Xen_is_boolean(arg3), arg3, 3, "XmInternAtom", "Boolean");
  return(C_to_Xen_Atom(XmInternAtom(Xen_to_C_Display(arg1), (char *)Xen_string_to_C_string(arg2), Xen_boolean_to_C_bool(arg3))));
}

static Xen gxm_XmNotebookGetPageInfo(Xen arg1, Xen arg2)
{
  #define H_XmNotebookGetPageInfo "XmNotebookPageStatus XmNotebookGetPageInfo(Widget notebook, int page_number) \
A Notebook function that returns page information"
  XmNotebookPageInfo *info;
  int err;
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XmNotebookGetPageInfo", "Widget");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XmNotebookGetPageInfo", "int");
  info = (XmNotebookPageInfo *)calloc(1, sizeof(XmNotebookPageInfo));
  err = XmNotebookGetPageInfo(Xen_to_C_Widget(arg1), Xen_integer_to_C_int(arg2), info);
  return(Xen_list_2(C_int_to_Xen_integer(err), C_to_Xen_XmNotebookPageInfo(info)));
}

static Xen gxm_XmCreateNotebook(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XmCreateNotebook "void XmCreateNotebook(Widget parent, String name, ArgList arglist, Cardinal argcount) The Notebook widget creation function"
  return(gxm_new_widget("XmCreateNotebook", XmCreateNotebook, arg1, arg2, arg3, arg4));
}

static Xen gxm_XmCreateArrowButton(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XmCreateArrowButton "Widget XmCreateArrowButton(Widget parent, String name, ArgList arglist, Cardinal argcount) \
The ArrowButton widget creation function"
  return(gxm_new_widget("XmCreateArrowButton", XmCreateArrowButton, arg1, arg2, arg3, arg4));
}

static Xen gxm_XmCreateArrowButtonGadget(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XmCreateArrowButtonGadget "Widget XmCreateArrowButtonGadget(Widget parent, String name, ArgList arglist, Cardinal argcount) \
The ArrowButtonGadget creation function"
  return(gxm_new_widget("XmCreateArrowButtonGadget", XmCreateArrowButtonGadget, arg1, arg2, arg3, arg4));
}

static Xen gxm_XmMessageBoxGetChild(Xen arg1, Xen arg2)
{
  #define H_XmMessageBoxGetChild "Widget XmMessageBoxGetChild(Widget widget, unsigned char child) A MessageBox function that is used to access a component"
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XmMessageBoxGetChild", "Widget");
  Xen_check_type(Xen_is_ulong(arg2), arg2, 2, "XmMessageBoxGetChild", "unsigned int");
  return(C_to_Xen_Widget(XmMessageBoxGetChild(Xen_to_C_Widget(arg1), Xen_ulong_to_C_ulong(arg2))));
}

static Xen gxm_XmCreateTemplateDialog(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XmCreateTemplateDialog "Widget XmCreateTemplateDialog(Widget parent, String name, ArgList arglist, Cardinal argcount) \
A MessageBox TemplateDialog creation function"
  return(gxm_new_widget("XmCreateTemplateDialog", XmCreateTemplateDialog, arg1, arg2, arg3, arg4));
}

static Xen gxm_XmCreateWorkingDialog(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XmCreateWorkingDialog "Widget XmCreateWorkingDialog(Widget parent, String name, ArgList arglist, Cardinal argcount) \
The MessageBox WorkingDialog creation function"
  return(gxm_new_widget("XmCreateWorkingDialog", XmCreateWorkingDialog, arg1, arg2, arg3, arg4));
}

static Xen gxm_XmCreateWarningDialog(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XmCreateWarningDialog "Widget XmCreateWarningDialog(Widget parent, String name, ArgList arglist, Cardinal argcount) \
The MessageBox WarningDialog creation function"
  return(gxm_new_widget("XmCreateWarningDialog", XmCreateWarningDialog, arg1, arg2, arg3, arg4));
}

static Xen gxm_XmCreateQuestionDialog(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XmCreateQuestionDialog "Widget XmCreateQuestionDialog(Widget parent, String name, ArgList arglist, Cardinal argcount) \
The MessageBox QuestionDialog creation function"
  return(gxm_new_widget("XmCreateQuestionDialog", XmCreateQuestionDialog, arg1, arg2, arg3, arg4));
}

static Xen gxm_XmCreateInformationDialog(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XmCreateInformationDialog "Widget XmCreateInformationDialog(Widget parent, String name, ArgList arglist, Cardinal argcount) \
The MessageBox InformationDialog creation function"
  return(gxm_new_widget("XmCreateInformationDialog", XmCreateInformationDialog, arg1, arg2, arg3, arg4));
}

static Xen gxm_XmCreateErrorDialog(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XmCreateErrorDialog "Widget XmCreateErrorDialog(Widget parent, String name, ArgList arglist, Cardinal argcount) \
The MessageBox ErrorDialog creation function"
  return(gxm_new_widget("XmCreateErrorDialog", XmCreateErrorDialog, arg1, arg2, arg3, arg4));
}

static Xen gxm_XmCreateMessageDialog(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XmCreateMessageDialog "Widget XmCreateMessageDialog(Widget parent, String name, ArgList arglist, Cardinal argcount) \
The MessageBox MessageDialog creation function"
  return(gxm_new_widget("XmCreateMessageDialog", XmCreateMessageDialog, arg1, arg2, arg3, arg4));
}

static Xen gxm_XmCreateMessageBox(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XmCreateMessageBox "Widget XmCreateMessageBox(Widget parent, String name, ArgList arglist, Cardinal argcount) \
The MessageBox widget creation function"
  return(gxm_new_widget("XmCreateMessageBox", XmCreateMessageBox, arg1, arg2, arg3, arg4));
}

static Xen gxm_XmIsMessageBox(Xen arg)
{
  #define H_XmIsMessageBox "XmIsMessageBox(arg): " PROC_TRUE " if arg is a MessageBox widget"
  Xen_check_type(Xen_is_Widget(arg), arg, 0, "XmIsMessageBox", "Widget");
  return(C_bool_to_Xen_boolean(XmIsMessageBox(Xen_to_C_Widget(arg))));
}

static Xen gxm_XmIsArrowButtonGadget(Xen arg)
{
  #define H_XmIsArrowButtonGadget "XmIsArrowButtonGadget(arg): " PROC_TRUE " if arg is an ArrowButton gadget"
  Xen_check_type(Xen_is_Widget(arg), arg, 0, "XmIsArrowButtonGadget", "Widget");
  return(C_bool_to_Xen_boolean(XmIsArrowButtonGadget(Xen_to_C_Widget(arg))));
}

static Xen gxm_XmIsArrowButton(Xen arg)
{
  #define H_XmIsArrowButton "XmIsArrowButton(arg): " PROC_TRUE " if arg is an ArrowButton widget"
  Xen_check_type(Xen_is_Widget(arg), arg, 0, "XmIsArrowButton", "Widget");
  return(C_bool_to_Xen_boolean(XmIsArrowButton(Xen_to_C_Widget(arg))));
}

static Xen gxm_XmIsNotebook(Xen arg)
{
  #define H_XmIsNotebook "XmIsNotebook(arg): " PROC_TRUE " if arg is a Notebook widget"
  Xen_check_type(Xen_is_Widget(arg), arg, 0, "XmIsNotebook", "Widget");
  return(C_bool_to_Xen_boolean(XmIsNotebook(Xen_to_C_Widget(arg))));
}

static Xen gxm_XmIsPanedWindow(Xen arg)
{
  #define H_XmIsPanedWindow "XmIsPanedWindow(arg): " PROC_TRUE " if arg is a PanedWindow widget"
  Xen_check_type(Xen_is_Widget(arg), arg, 0, "XmIsPanedWindow", "Widget");
  return(C_bool_to_Xen_boolean(XmIsPanedWindow(Xen_to_C_Widget(arg))));
}

static Xen gxm_XmIsBulletinBoard(Xen arg)
{
  #define H_XmIsBulletinBoard "XmIsBulletinBoard(arg): " PROC_TRUE " if arg is a BulletinBoard widget"
  Xen_check_type(Xen_is_Widget(arg), arg, 0, "XmIsBulletinBoard", "Widget");
  return(C_bool_to_Xen_boolean(XmIsBulletinBoard(Xen_to_C_Widget(arg))));
}

static Xen gxm_XmIsPrimitive(Xen arg)
{
  #define H_XmIsPrimitive "XmIsPrimitive(arg): " PROC_TRUE " if arg is a Primitive widget"
  Xen_check_type(Xen_is_Widget(arg), arg, 0, "XmIsPrimitive", "Widget");
  return(C_bool_to_Xen_boolean(XmIsPrimitive(Xen_to_C_Widget(arg))));
}

static Xen gxm_XmIsCascadeButtonGadget(Xen arg)
{
  #define H_XmIsCascadeButtonGadget "XmIsCascadeButtonGadget(arg): " PROC_TRUE " if arg is a CascadeButtonGadget gadget"
  Xen_check_type(Xen_is_Widget(arg), arg, 0, "XmIsCascadeButtonGadget", "Widget");
  return(C_bool_to_Xen_boolean(XmIsCascadeButtonGadget(Xen_to_C_Widget(arg))));
}

static Xen gxm_XmIsCascadeButton(Xen arg)
{
  #define H_XmIsCascadeButton "XmIsCascadeButton(arg): " PROC_TRUE " if arg is a CascadeButton widget"
  Xen_check_type(Xen_is_Widget(arg), arg, 0, "XmIsCascadeButton", "Widget");
  return(C_bool_to_Xen_boolean(XmIsCascadeButton(Xen_to_C_Widget(arg))));
}

static Xen gxm_XmIsPushButtonGadget(Xen arg)
{
  #define H_XmIsPushButtonGadget "XmIsPushButtonGadget(arg): " PROC_TRUE " if arg is a PushButtonGadget gadget"
  Xen_check_type(Xen_is_Widget(arg), arg, 0, "XmIsPushButtonGadget", "Widget");
  return(C_bool_to_Xen_boolean(XmIsPushButtonGadget(Xen_to_C_Widget(arg))));
}

static Xen gxm_XmIsPushButton(Xen arg)
{
  #define H_XmIsPushButton "XmIsPushButton(arg): " PROC_TRUE " if arg is a PushButton widget"
  Xen_check_type(Xen_is_Widget(arg), arg, 0, "XmIsPushButton", "Widget");
  return(C_bool_to_Xen_boolean(XmIsPushButton(Xen_to_C_Widget(arg))));
}

static Xen gxm_XmIsComboBox(Xen arg)
{
  #define H_XmIsComboBox "XmIsComboBox(arg): " PROC_TRUE " if arg is a ComboBox widget"
  Xen_check_type(Xen_is_Widget(arg), arg, 0, "XmIsComboBox", "Widget");
  return(C_bool_to_Xen_boolean(XmIsComboBox(Xen_to_C_Widget(arg))));
}

static Xen gxm_XmIsCommand(Xen arg)
{
  #define H_XmIsCommand "XmIsCommand(arg): " PROC_TRUE " if arg is a Command widget"
  Xen_check_type(Xen_is_Widget(arg), arg, 0, "XmIsCommand", "Widget");
  return(C_bool_to_Xen_boolean(XmIsCommand(Xen_to_C_Widget(arg))));
}

static Xen gxm_XmIsRowColumn(Xen arg)
{
  #define H_XmIsRowColumn "XmIsRowColumn(arg): " PROC_TRUE " if arg is a RowColumn widget"
  Xen_check_type(Xen_is_Widget(arg), arg, 0, "XmIsRowColumn", "Widget");
  return(C_bool_to_Xen_boolean(XmIsRowColumn(Xen_to_C_Widget(arg))));
}

static Xen gxm_XmIsContainer(Xen arg)
{
  #define H_XmIsContainer "XmIsContainer(arg): " PROC_TRUE " if arg is a Container widget"
  Xen_check_type(Xen_is_Widget(arg), arg, 0, "XmIsContainer", "Widget");
  return(C_bool_to_Xen_boolean(XmIsContainer(Xen_to_C_Widget(arg))));
}

static Xen gxm_XmIsScreen(Xen arg)
{
  #define H_XmIsScreen "XmIsScreen(arg): " PROC_TRUE " if arg is a Screen widget"
  Xen_check_type(Xen_is_Widget(arg), arg, 0, "XmIsScreen", "Widget");
  return(C_bool_to_Xen_boolean(XmIsScreen(Xen_to_C_Widget(arg))));
}

static Xen gxm_XmIsScale(Xen arg)
{
  #define H_XmIsScale "XmIsScale(arg): " PROC_TRUE " if arg is a Scale widget"
  Xen_check_type(Xen_is_Widget(arg), arg, 0, "XmIsScale", "Widget");
  return(C_bool_to_Xen_boolean(XmIsScale(Xen_to_C_Widget(arg))));
}

static Xen gxm_XmIsScrollBar(Xen arg)
{
  #define H_XmIsScrollBar "XmIsScrollBar(arg): " PROC_TRUE " if arg is a ScrollBar widget"
  Xen_check_type(Xen_is_Widget(arg), arg, 0, "XmIsScrollBar", "Widget");
  return(C_bool_to_Xen_boolean(XmIsScrollBar(Xen_to_C_Widget(arg))));
}

static Xen gxm_XmIsDialogShell(Xen arg)
{
  #define H_XmIsDialogShell "XmIsDialogShell(arg): " PROC_TRUE " if arg is a DialogShell widget"
  Xen_check_type(Xen_is_Widget(arg), arg, 0, "XmIsDialogShell", "Widget");
  return(C_bool_to_Xen_boolean(XmIsDialogShell(Xen_to_C_Widget(arg))));
}

static Xen gxm_XmIsScrolledWindow(Xen arg)
{
  #define H_XmIsScrolledWindow "XmIsScrolledWindow(arg): " PROC_TRUE " if arg is a ScrolledWindow widget"
  Xen_check_type(Xen_is_Widget(arg), arg, 0, "XmIsScrolledWindow", "Widget");
  return(C_bool_to_Xen_boolean(XmIsScrolledWindow(Xen_to_C_Widget(arg))));
}

static Xen gxm_XmIsDisplay(Xen arg)
{
  #define H_XmIsDisplay "XmIsDisplay(arg): " PROC_TRUE " if arg is a Display widget"
  Xen_check_type(Xen_is_Widget(arg), arg, 0, "XmIsDisplay", "Widget");
  return(C_bool_to_Xen_boolean(XmIsDisplay(Xen_to_C_Widget(arg))));
}

static Xen gxm_XmIsSelectionBox(Xen arg)
{
  #define H_XmIsSelectionBox "XmIsSelectionBox(arg): " PROC_TRUE " if arg is a SelectionBox widget"
  Xen_check_type(Xen_is_Widget(arg), arg, 0, "XmIsSelectionBox", "Widget");
  return(C_bool_to_Xen_boolean(XmIsSelectionBox(Xen_to_C_Widget(arg))));
}

static Xen gxm_XmIsDragContext(Xen arg)
{
  #define H_XmIsDragContext "XmIsDragContext(arg): " PROC_TRUE " if arg is a DragContext widget"
  Xen_check_type(Xen_is_Widget(arg), arg, 0, "XmIsDragContext", "Widget");
  return(C_bool_to_Xen_boolean(XmIsDragContext(Xen_to_C_Widget(arg))));
}

static Xen gxm_XmIsSeparatorGadget(Xen arg)
{
  #define H_XmIsSeparatorGadget "XmIsSeparatorGadget(arg): " PROC_TRUE " if arg is a SeparatorGadget gadget"
  Xen_check_type(Xen_is_Widget(arg), arg, 0, "XmIsSeparatorGadget", "Widget");
  return(C_bool_to_Xen_boolean(XmIsSeparatorGadget(Xen_to_C_Widget(arg))));
}

static Xen gxm_XmIsDragIconObjectClass(Xen arg)
{
  #define H_XmIsDragIconObjectClass "XmIsDragIconObjectClass(arg): " PROC_TRUE " if arg is a DragIconObjectClass widget"
  Xen_check_type(Xen_is_Widget(arg), arg, 0, "XmIsDragIconObjectClass", "Widget");
  return(C_bool_to_Xen_boolean(XmIsDragIconObjectClass(Xen_to_C_Widget(arg))));
}

static Xen gxm_XmIsSeparator(Xen arg)
{
  #define H_XmIsSeparator "XmIsSeparator(arg): " PROC_TRUE " if arg is a Separator widget"
  Xen_check_type(Xen_is_Widget(arg), arg, 0, "XmIsSeparator", "Widget");
  return(C_bool_to_Xen_boolean(XmIsSeparator(Xen_to_C_Widget(arg))));
}

static Xen gxm_XmIsDrawingArea(Xen arg)
{
  #define H_XmIsDrawingArea "XmIsDrawingArea(arg): " PROC_TRUE " if arg is a DrawingArea widget"
  Xen_check_type(Xen_is_Widget(arg), arg, 0, "XmIsDrawingArea", "Widget");
  return(C_bool_to_Xen_boolean(XmIsDrawingArea(Xen_to_C_Widget(arg))));
}

static Xen gxm_XmIsDrawnButton(Xen arg)
{
  #define H_XmIsDrawnButton "XmIsDrawnButton(arg): " PROC_TRUE " if arg is a DrawnButton widget"
  Xen_check_type(Xen_is_Widget(arg), arg, 0, "XmIsDrawnButton", "Widget");
  return(C_bool_to_Xen_boolean(XmIsDrawnButton(Xen_to_C_Widget(arg))));
}

static Xen gxm_XmIsDropSiteManager(Xen arg)
{
  #define H_XmIsDropSiteManager "XmIsDropSiteManager(arg): " PROC_TRUE " if arg is a DropSiteManager widget"
  Xen_check_type(Xen_is_Widget(arg), arg, 0, "XmIsDropSiteManager", "Widget");
  return(C_bool_to_Xen_boolean(XmIsDropSiteManager(Xen_to_C_Widget(arg))));
}

static Xen gxm_XmIsDropTransfer(Xen arg)
{
  #define H_XmIsDropTransfer "XmIsDropTransfer(arg): " PROC_TRUE " if arg is a DropTransfer widget"
  Xen_check_type(Xen_is_Widget(arg), arg, 0, "XmIsDropTransfer", "Widget");
  return(C_bool_to_Xen_boolean(XmIsDropTransfer(Xen_to_C_Widget(arg))));
}

static Xen gxm_XmIsTextField(Xen arg)
{
  #define H_XmIsTextField "XmIsTextField(arg): " PROC_TRUE " if arg is a TextField widget"
  Xen_check_type(Xen_is_Widget(arg), arg, 0, "XmIsTextField", "Widget");
  return(C_bool_to_Xen_boolean(XmIsTextField(Xen_to_C_Widget(arg))));
}

static Xen gxm_XmIsFileSelectionBox(Xen arg)
{
  #define H_XmIsFileSelectionBox "XmIsFileSelectionBox(arg): " PROC_TRUE " if arg is a FileSelectionBox widget"
  Xen_check_type(Xen_is_Widget(arg), arg, 0, "XmIsFileSelectionBox", "Widget");
  return(C_bool_to_Xen_boolean(XmIsFileSelectionBox(Xen_to_C_Widget(arg))));
}

static Xen gxm_XmIsText(Xen arg)
{
  #define H_XmIsText "XmIsText(arg): " PROC_TRUE " if arg is a Text widget"
  Xen_check_type(Xen_is_Widget(arg), arg, 0, "XmIsText", "Widget");
  return(C_bool_to_Xen_boolean(XmIsText(Xen_to_C_Widget(arg))));
}

static Xen gxm_XmIsForm(Xen arg)
{
  #define H_XmIsForm "XmIsForm(arg): " PROC_TRUE " if arg is a Form widget"
  Xen_check_type(Xen_is_Widget(arg), arg, 0, "XmIsForm", "Widget");
  return(C_bool_to_Xen_boolean(XmIsForm(Xen_to_C_Widget(arg))));
}

static Xen gxm_XmIsFrame(Xen arg)
{
  #define H_XmIsFrame "XmIsFrame(arg): " PROC_TRUE " if arg is a Frame widget"
  Xen_check_type(Xen_is_Widget(arg), arg, 0, "XmIsFrame", "Widget");
  return(C_bool_to_Xen_boolean(XmIsFrame(Xen_to_C_Widget(arg))));
}

static Xen gxm_XmIsGadget(Xen arg)
{
  #define H_XmIsGadget "XmIsGadget(arg): " PROC_TRUE " if arg is a Gadget"
  Xen_check_type(Xen_is_Widget(arg), arg, 0, "XmIsGadget", "Widget");
  return(C_bool_to_Xen_boolean(XmIsGadget(Xen_to_C_Widget(arg))));
}

static Xen gxm_XmIsToggleButtonGadget(Xen arg)
{
  #define H_XmIsToggleButtonGadget "XmIsToggleButtonGadget(arg): " PROC_TRUE " if arg is a ToggleButtonGadget gadget"
  Xen_check_type(Xen_is_Widget(arg), arg, 0, "XmIsToggleButtonGadget", "Widget");
  return(C_bool_to_Xen_boolean(XmIsToggleButtonGadget(Xen_to_C_Widget(arg))));
}

static Xen gxm_XmIsGrabShell(Xen arg)
{
  #define H_XmIsGrabShell "XmIsGrabShell(arg): " PROC_TRUE " if arg is a GrabShell widget"
  Xen_check_type(Xen_is_Widget(arg), arg, 0, "XmIsGrabShell", "Widget");
  return(C_bool_to_Xen_boolean(XmIsGrabShell(Xen_to_C_Widget(arg))));
}

static Xen gxm_XmIsToggleButton(Xen arg)
{
  #define H_XmIsToggleButton "XmIsToggleButton(arg): " PROC_TRUE " if arg is a ToggleButton widget"
  Xen_check_type(Xen_is_Widget(arg), arg, 0, "XmIsToggleButton", "Widget");
  return(C_bool_to_Xen_boolean(XmIsToggleButton(Xen_to_C_Widget(arg))));
}

static Xen gxm_XmIsIconGadget(Xen arg)
{
  #define H_XmIsIconGadget "XmIsIconGadget(arg): " PROC_TRUE " if arg is a IconGadget widget"
  Xen_check_type(Xen_is_Widget(arg), arg, 0, "XmIsIconGadget", "Widget");
  return(C_bool_to_Xen_boolean(XmIsIconGadget(Xen_to_C_Widget(arg))));
}

static Xen gxm_XmIsIconHeader(Xen arg)
{
  #define H_XmIsIconHeader "XmIsIconHeader(arg): " PROC_TRUE " if arg is a IconHeader widget"
  Xen_check_type(Xen_is_Widget(arg), arg, 0, "XmIsIconHeader", "Widget");
  return(C_bool_to_Xen_boolean(XmIsIconHeader(Xen_to_C_Widget(arg))));
}

static Xen gxm_XmIsLabelGadget(Xen arg)
{
  #define H_XmIsLabelGadget "XmIsLabelGadget(arg): " PROC_TRUE " if arg is a LabelGadget widget"
  Xen_check_type(Xen_is_Widget(arg), arg, 0, "XmIsLabelGadget", "Widget");
  return(C_bool_to_Xen_boolean(XmIsLabelGadget(Xen_to_C_Widget(arg))));
}

static Xen gxm_XmIsLabel(Xen arg)
{
  #define H_XmIsLabel "XmIsLabel(arg): " PROC_TRUE " if arg is a Label widget"
  Xen_check_type(Xen_is_Widget(arg), arg, 0, "XmIsLabel", "Widget");
  return(C_bool_to_Xen_boolean(XmIsLabel(Xen_to_C_Widget(arg))));
}

static Xen gxm_XmIsVendorShell(Xen arg)
{
  #define H_XmIsVendorShell "XmIsVendorShell(arg): " PROC_TRUE " if arg is a VendorShell widget"
  Xen_check_type(Xen_is_Widget(arg), arg, 0, "XmIsVendorShell", "Widget");
  return(C_bool_to_Xen_boolean(XmIsVendorShell(Xen_to_C_Widget(arg))));
}

static Xen gxm_XmIsList(Xen arg)
{
  #define H_XmIsList "XmIsList(arg): " PROC_TRUE " if arg is a List widget"
  Xen_check_type(Xen_is_Widget(arg), arg, 0, "XmIsList", "Widget");
  return(C_bool_to_Xen_boolean(XmIsList(Xen_to_C_Widget(arg))));
}

static Xen gxm_XmIsMainWindow(Xen arg)
{
  #define H_XmIsMainWindow "XmIsMainWindow(arg): " PROC_TRUE " if arg is a MainWindow widget"
  Xen_check_type(Xen_is_Widget(arg), arg, 0, "XmIsMainWindow", "Widget");
  return(C_bool_to_Xen_boolean(XmIsMainWindow(Xen_to_C_Widget(arg))));
}

static Xen gxm_XmIsManager(Xen arg)
{
  #define H_XmIsManager "XmIsManager(arg): " PROC_TRUE " if arg is a Manager widget"
  Xen_check_type(Xen_is_Widget(arg), arg, 0, "XmIsManager", "Widget");
  return(C_bool_to_Xen_boolean(XmIsManager(Xen_to_C_Widget(arg))));
}

static Xen gxm_XmIsMenuShell(Xen arg)
{
  #define H_XmIsMenuShell "XmIsMenuShell(arg): " PROC_TRUE " if arg is a MenuShell widget"
  Xen_check_type(Xen_is_Widget(arg), arg, 0, "XmIsMenuShell", "Widget");
  return(C_bool_to_Xen_boolean(XmIsMenuShell(Xen_to_C_Widget(arg))));
}





/* ----------------------------------------------------------------------------------------------------
 *
 *                            *         *
 *                             *       *
 *                              *     *
 *                               *   *
 *                                * *
 *                                 *
 *                                * *
 *                               *   *
 *                              *     *
 *                             *       *
 *                            *         *
 * ----------------------------------------------------------------------------------------------------
 */

static Xen gxm_XXorRegion(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XXorRegion "XXorRegion(sra, srb, dr_return) calculates the difference between the union and intersection of two regions."
  Xen_check_type(Xen_is_Region(arg1), arg1, 1, "XXorRegion", "Region");
  Xen_check_type(Xen_is_Region(arg2), arg2, 2, "XXorRegion", "Region");
  Xen_check_type(Xen_is_Region(arg3), arg3, 3, "XXorRegion", "Region");
  return(C_int_to_Xen_integer(XXorRegion(Xen_to_C_Region(arg1), Xen_to_C_Region(arg2), Xen_to_C_Region(arg3))));
}

static Xen gxm_XUnionRegion(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XUnionRegion "XUnionRegion(sra, srb, dr_return) computes the union of two regions."
  Xen_check_type(Xen_is_Region(arg1), arg1, 1, "XUnionRegion", "Region");
  Xen_check_type(Xen_is_Region(arg2), arg2, 2, "XUnionRegion", "Region");
  Xen_check_type(Xen_is_Region(arg3), arg3, 3, "XUnionRegion", "Region");
  return(C_int_to_Xen_integer(XUnionRegion(Xen_to_C_Region(arg1), Xen_to_C_Region(arg2), Xen_to_C_Region(arg3))));
}

static Xen gxm_XUnionRectWithRegion(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XUnionRectWithRegion "XUnionRectWithRegion(rectangle, src_region, dest_region_return) updates the destination region from a \
union of the specified rectangle and the specified source region."
  Xen_check_type(Xen_is_XRectangle(arg1), arg1, 1, "XUnionRectWithRegion", "XRectangle*");
  Xen_check_type(Xen_is_Region(arg2), arg2, 2, "XUnionRectWithRegion", "Region");
  Xen_check_type(Xen_is_Region(arg3), arg3, 3, "XUnionRectWithRegion", "Region");
  return(C_int_to_Xen_integer(XUnionRectWithRegion(Xen_to_C_XRectangle(arg1), Xen_to_C_Region(arg2), Xen_to_C_Region(arg3))));
}

static Xen gxm_XSubtractRegion(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XSubtractRegion "XSubtractRegion(sra, srb, dr_return) subtracts srb from sra and stores the results in dr_return."
  Xen_check_type(Xen_is_Region(arg1), arg1, 1, "XSubtractRegion", "Region");
  Xen_check_type(Xen_is_Region(arg2), arg2, 2, "XSubtractRegion", "Region");
  Xen_check_type(Xen_is_Region(arg3), arg3, 3, "XSubtractRegion", "Region");
  return(C_int_to_Xen_integer(XSubtractRegion(Xen_to_C_Region(arg1), Xen_to_C_Region(arg2), Xen_to_C_Region(arg3))));
}

static Xen gxm_XShrinkRegion(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XShrinkRegion "XShrinkRegion(r, dx, dy) reduces the specified region by a specified amount."
  Xen_check_type(Xen_is_Region(arg1), arg1, 1, "XShrinkRegion", "Region");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XShrinkRegion", "int");
  Xen_check_type(Xen_is_integer(arg3), arg3, 3, "XShrinkRegion", "int");
  return(C_int_to_Xen_integer(XShrinkRegion(Xen_to_C_Region(arg1), Xen_integer_to_C_int(arg2), Xen_integer_to_C_int(arg3))));
}

static Xen gxm_XSetWMProperties(Xen dpy, Xen win, Xen win_name, Xen icon_name, Xen argv, Xen argc, Xen normal_hints, Xen wm_hints)
{
  /* last arg omitted -- XClassHint not supported */
  #define H_XSetWMProperties "XSetWMProperties(dpy, win, win_name, icon_name, argv, argc, normal_hints wm_hints) sets the window properties"
  int c_argc = 0;
  char **c_argv = NULL;
  XTextProperty w_name, i_name;
  bool use_w_name = false, use_i_name = false;
  Xen_check_type(Xen_is_Display(dpy), dpy, 1, "XSetWMProperties", "Display*");
  Xen_check_type(Xen_is_Window(win), win, 2, "XSetWMProperties", "Window");
  Xen_check_type(Xen_is_string(win_name) || Xen_is_null(win_name) || Xen_is_false(win_name), win_name, 3, "XSetWMProperties", "char*");
  Xen_check_type(Xen_is_string(icon_name) || Xen_is_null(icon_name) || Xen_is_false(icon_name), icon_name, 4, "XSetWMProperties", "char*");
  Xen_check_type(Xen_is_list(argv), argv, 5, "XSetWMProperties", "list of char*");
  Xen_check_type(Xen_is_integer(argc), argc, 6, "XSetWMProperties", "int");
  Xen_check_type(Xen_is_false(normal_hints) || Xen_is_XSizeHints(normal_hints), normal_hints, 7, "XSetWMProperties", "XSizeHints* or false");
  Xen_check_type(Xen_is_false(wm_hints) || Xen_is_XWMHints(wm_hints), wm_hints, 8, "XSetWMProperties", "XWMHints* or false");
  c_argc = Xen_integer_to_C_int(argc);
  if (c_argc > 0) c_argv = Xen_to_C_Strings(argv, c_argc);
  if (Xen_is_string(win_name))
    {
      char *name;
      use_w_name = true;
      name = (char *)Xen_string_to_C_string(win_name);
      XStringListToTextProperty(&name, 1, &w_name);
    }
  if (Xen_is_string(icon_name))
    {
      char *name;
      use_i_name = true; 
      name = (char *)Xen_string_to_C_string(icon_name);
      XStringListToTextProperty(&name, 1, &i_name);
   }
  XSetWMProperties(Xen_to_C_Display(dpy),
		   Xen_to_C_Window(win),
		   (use_w_name) ? &w_name : NULL,
		   (use_i_name) ? &i_name : NULL,
		   c_argv,
		   c_argc,
		   Xen_to_C_XSizeHints(normal_hints),
		   Xen_to_C_XWMHints(wm_hints),
		   NULL);
  if (c_argv) free(c_argv);
  return(Xen_false);
}

static Xen gxm_XSetRegion(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XSetRegion "XSetRegion(display, gc, r) sets the clip-mask in the GC to the specified region."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XSetRegion", "Display*");
  Xen_check_type(Xen_is_GC(arg2), arg2, 2, "XSetRegion", "GC");
  Xen_check_type(Xen_is_Region(arg3), arg3, 3, "XSetRegion", "Region");
  return(C_int_to_Xen_integer(XSetRegion(Xen_to_C_Display(arg1), Xen_to_C_GC(arg2), Xen_to_C_Region(arg3))));
}

static Xen gxm_XSetWMHints(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XSetWMHints "XSetWMHints(display, w, wmhints) sets the window manager hints that include icon information and location, the \
initial state of the window, and whether the application relies on the window manager to get keyboard input."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XSetWMHints", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XSetWMHints", "Window");
  Xen_check_type(Xen_is_XWMHints(arg3), arg3, 3, "XSetWMHints", "XWMHints*");
  return(C_int_to_Xen_integer(XSetWMHints(Xen_to_C_Display(arg1), Xen_to_C_Window(arg2), Xen_to_C_XWMHints(arg3))));
}

static Xen gxm_XSetRGBColormaps(Xen arg1, Xen arg2, Xen arg3, Xen arg4, Xen arg5)
{
  #define H_XSetRGBColormaps "void XSetRGBColormaps(display, w, std_colormap, count, property) replaces the RGB colormap definition in the \
specified property on the named window."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XSetRGBColormaps", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XSetRGBColormaps", "Window");
  Xen_check_type(Xen_is_XStandardColormap(arg3), arg3, 3, "XSetRGBColormaps", "XStandardColormap*");
  Xen_check_type(Xen_is_integer(arg4), arg4, 4, "XSetRGBColormaps", "int");
  Xen_check_type(Xen_is_Atom(arg5), arg5, 5, "XSetRGBColormaps", "Atom");
  XSetRGBColormaps(Xen_to_C_Display(arg1), 
		   Xen_to_C_Window(arg2), 
		   Xen_to_C_XStandardColormap(arg3), 
		   Xen_integer_to_C_int(arg4), 
		   Xen_to_C_Atom(arg5));
  return(Xen_false);
}

static Xen gxm_XUniqueContext(void)
{
  #define H_XUniqueContext "XContext XUniqueContext() creates a unique context type that may be used in subsequent calls to XSaveContext."
  return(C_to_Xen_XContext(XUniqueContext()));
}

static Xen gxm_XSaveContext(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XSaveContext "XSaveContext(dpy, rid, context) saves a context"
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XSaveContext", "Display*");
  Xen_check_type(Xen_is_ulong(arg2), arg2, 2, "XSaveContext", "XID");
  Xen_check_type(Xen_is_XContext(arg3), arg3, 3, "XSaveContext", "XContext");
  return(C_int_to_Xen_integer(XSaveContext(Xen_to_C_Display(arg1), Xen_ulong_to_C_ulong(arg2), Xen_to_C_XContext(arg3), (caddr_t)arg4)));
}

static Xen gxm_XRectInRegion(Xen arg1, Xen arg2, Xen arg3, Xen arg4, Xen arg5)
{
  #define H_XRectInRegion "int XRectInRegion(r, x, y, width, height): returns RectangleIn if the rectangle is entirely in the specified region, \
RectangleOut if the rectangle is entirely out of the specified region, and RectanglePart if the rectangle is partially in the specified region. "
  Xen_check_type(Xen_is_Region(arg1), arg1, 1, "XRectInRegion", "Region");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XRectInRegion", "int");
  Xen_check_type(Xen_is_integer(arg3), arg3, 3, "XRectInRegion", "int");
  Xen_check_type(Xen_is_ulong(arg4), arg4, 4, "XRectInRegion", "unsigned int");
  Xen_check_type(Xen_is_ulong(arg5), arg5, 5, "XRectInRegion", "unsigned int");
  return(C_int_to_Xen_integer(XRectInRegion(Xen_to_C_Region(arg1), Xen_integer_to_C_int(arg2), Xen_integer_to_C_int(arg3), Xen_ulong_to_C_ulong(arg4), Xen_ulong_to_C_ulong(arg5))));
}

static Xen gxm_XPolygonRegion(Xen larg1, Xen arg2, Xen arg3)
{
  #define H_XPolygonRegion "Region XPolygonRegion(points, n, fill_rule): returns a region for the polygon defined by the points list."
  /* DIFF: XPolygonRegion XPoint* arg (arg 1) is list of XPoints
   */
  XPoint *pt, *pt1;
  int i, len;
  Region res;
  Xen arg1;
  Xen_check_type(Xen_is_list(larg1), larg1, 1, "XPolygonRegion", "list of XPoints");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XPolygonRegion", "int");
  Xen_check_type(Xen_is_integer(arg3), arg3, 3, "XPolygonRegion", "int");
  arg1 = Xen_copy_arg(larg1);
  len = Xen_integer_to_C_int(arg2);
  if (len <= 0) Xen_check_type(0, arg2, 2, "XPolygonRegion", "positive integer");
  pt = (XPoint *)calloc(len, sizeof(XPoint));
  for (i = 0; (i < len) && (!Xen_is_null(arg1)); i++, arg1 = Xen_cdr(arg1))
    {
      Xen xp;
      xp = Xen_car(arg1);
      if (!(Xen_is_XPoint(xp))) {free(pt); Xen_check_type(0, xp, i, "XPolygonRegion", "XPoint"); return(Xen_false);}
      pt1 = Xen_to_C_XPoint(Xen_car(arg1));
      pt[i].x = pt1->x;
      pt[i].y = pt1->y;
    }
  res = XPolygonRegion(pt, len, Xen_integer_to_C_int(arg3));
  free(pt);
  return(C_to_Xen_Region(res));
}

static Xen gxm_XPointInRegion(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XPointInRegion "Bool XPointInRegion(r, x, y): returns " PROC_TRUE " if the point (x, y) is contained in the region r."
  Xen_check_type(Xen_is_Region(arg1), arg1, 1, "XPointInRegion", "Region");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XPointInRegion", "int");
  Xen_check_type(Xen_is_integer(arg3), arg3, 3, "XPointInRegion", "int");
  return(C_bool_to_Xen_boolean(XPointInRegion(Xen_to_C_Region(arg1), Xen_integer_to_C_int(arg2), Xen_integer_to_C_int(arg3))));
}

static Xen gxm_XOffsetRegion(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XOffsetRegion "XOffsetRegion(r, dx, dy) moves the specified region by a specified amount."
  Xen_check_type(Xen_is_Region(arg1), arg1, 1, "XOffsetRegion", "Region");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XOffsetRegion", "int");
  Xen_check_type(Xen_is_integer(arg3), arg3, 3, "XOffsetRegion", "int");
  return(C_int_to_Xen_integer(XOffsetRegion(Xen_to_C_Region(arg1), Xen_integer_to_C_int(arg2), Xen_integer_to_C_int(arg3))));
}

static XVisualInfo *match_visual_info;
static Xen gxm_XMatchVisualInfo(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XMatchVisualInfo "Status XMatchVisualInfo(display, screen, depth, class): returns the visual information for a \
visual that matches the specified depth and class for a screen."
  /* DIFF: XMatchVisualInfo dpy scr dep class [visual] -> #f or visual
   */
  int val;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XMatchVisualInfo", "Display*");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XMatchVisualInfo", "int");
  Xen_check_type(Xen_is_integer(arg3), arg3, 3, "XMatchVisualInfo", "int");
  Xen_check_type(Xen_is_integer(arg4), arg4, 4, "XMatchVisualInfo", "int");
  match_visual_info = (XVisualInfo *)calloc(1, sizeof(XVisualInfo));
  val = XMatchVisualInfo(Xen_to_C_Display(arg1), Xen_integer_to_C_int(arg2), Xen_integer_to_C_int(arg3), Xen_integer_to_C_int(arg4), match_visual_info);
  if (val)
    return(C_to_Xen_XVisualInfo(match_visual_info));
  return(Xen_false);
}

static Xen gxm_XLookupString(Xen arg1)
{
  #define H_XLookupString "int XLookupString(event_struct) translates a key event to a KeySym and a string -> (len str keysym)."
  /* DIFF: XLookupString last arg is ignored, keyevent and all but last omitted -> (list len str keysym)
   */
  KeySym key;
  char *str;
  int len;
  Xen res;
  Xen_check_type(Xen_is_XKeyEvent(arg1), arg1, 1, "XLookupString", "XKeyEvent*");/* user-created */
  str = (char *)calloc(16, sizeof(char));
  len = XLookupString(Xen_to_C_XKeyEvent(arg1), str, 16, &key, NULL);
  res = Xen_list_3(C_int_to_Xen_integer(len),
		   C_string_to_Xen_string(str),
		   C_to_Xen_KeySym(key));
  free(str);
  return(res);
}

static Xen gxm_XConvertCase(Xen arg1)
{
  #define H_XConvertCase "void XConvertCase(keysym): returns the uppercase and lowercase forms of the specified \
Keysym, if the KeySym is subject to case conversion; otherwise, the specified KeySym is returned to both lower_return and upper_return."
  /* DIFF: XConvertCase keysym [k1 k2] -> (list k1 k2)
   */
  KeySym k1, k2;
  Xen_check_type(Xen_is_KeySym(arg1), arg1, 1, "XConvertCase", "KeySym");
  XConvertCase(Xen_to_C_KeySym(arg1), &k1, &k2);
  return(Xen_list_2(C_to_Xen_KeySym(k1),
		    C_to_Xen_KeySym(k2)));
}

static Xen gxm_XIntersectRegion(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XIntersectRegion "XIntersectRegion(sra, srb, dr_return) computes the intersection of two regions."
  Xen_check_type(Xen_is_Region(arg1), arg1, 1, "XIntersectRegion", "Region");
  Xen_check_type(Xen_is_Region(arg2), arg2, 2, "XIntersectRegion", "Region");
  Xen_check_type(Xen_is_Region(arg3), arg3, 3, "XIntersectRegion", "Region");
  return(C_int_to_Xen_integer(XIntersectRegion(Xen_to_C_Region(arg1), Xen_to_C_Region(arg2), Xen_to_C_Region(arg3))));
}

static Xen gxm_XGetWMHints(Xen arg1, Xen arg2)
{
  #define H_XGetWMHints "XWMHints *XGetWMHints(display, w) reads the window manager hints and returns NULL if no WM_HINTS property was \
set on the window or returns a pointer to a XWMHints structure if it succeeds."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XGetWMHints", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XGetWMHints", "Window");
  return(C_to_Xen_XWMHints(XGetWMHints(Xen_to_C_Display(arg1), Xen_to_C_Window(arg2))));
}

static Xen gxm_XGetVisualInfo(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XGetVisualInfo "XVisualInfo *XGetVisualInfo(display, vinfo_mask, vinfo_template): returns a list of visual \
structures that have attributes equal to the attributes specified by vinfo_template."
  /* DIFF: XGetVisualInfo dpy mask template [nitems] -> () or (list visual...)
   */
  XVisualInfo *v;
  int len;
  Xen lst = Xen_empty_list;

  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XGetVisualInfo", "Display*");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XGetVisualInfo", "long");
  Xen_check_type(Xen_is_XVisualInfo(arg3), arg3, 3, "XGetVisualInfo", "XVisualInfo*");

  v = XGetVisualInfo(Xen_to_C_Display(arg1), Xen_integer_to_C_int(arg2), Xen_to_C_XVisualInfo(arg3), &len);
  if (v)
    {
      int i, loc;
      loc = xm_protect(lst);
      for (i = len - 1; i >= 0; i--)
	lst = Xen_cons(C_to_Xen_XVisualInfo(v + i), lst);
      xm_unprotect_at(loc);
      /* XFree(v); */ /* valgrind says this is a bad idea */
    }
  return(lst);
}

static Xen gxm_XGetRGBColormaps(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XGetRGBColormaps "Status XGetRGBColormaps(display, w, property): returns the RGB colormap \
definitions stored in the specified property on the named window."
  /* DIFF: XGetRGBColormaps omits arg 3 and 4, returns list of XStandardColormaps 
   */
  int len, i, loc;
  Status val;
  Xen lst = Xen_empty_list;
  XStandardColormap **cs = NULL; /* do I allocate this?? */
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XGetRGBColormaps", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XGetRGBColormaps", "Window");
  Xen_check_type(Xen_is_Atom(arg3), arg3, 3, "XGetRGBColormaps", "Atom");
  val = XGetRGBColormaps(Xen_to_C_Display(arg1), 
			 Xen_to_C_Window(arg2), 
			 cs, &len,
			 Xen_to_C_Atom(arg3));
  if (val == 0)
    return(Xen_false);
  loc = xm_protect(lst);
  for (i = len - 1; i >= 0; i--)
    lst = Xen_cons(C_to_Xen_XStandardColormap(cs[i]), lst);
  xm_unprotect_at(loc);
  return(lst);
}

static Xen gxm_XGetIconSizes(Xen arg1, Xen arg2)
{
  #define H_XGetIconSizes "Status XGetIconSizes(display, w): returns " PROC_FALSE " if the window manager has not \
set icon sizes; otherwise, it return nonzero and a list of XIconSize structs."
  /* DIFF: XGetIconSizes omit last 2 args, return list of XIconSizes
   */
  XIconSize *sizes;
  int i, len, val, loc;
  Xen lst = Xen_empty_list;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XGetIconSizes", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XGetIconSizes", "Window");
  val = XGetIconSizes(Xen_to_C_Display(arg1), Xen_to_C_Window(arg2), &sizes, &len);
  if (val == 0)
    return(Xen_false);
  loc = xm_protect(lst);
  for (i = len - 1; i >= 0; i--)
    lst = Xen_cons(C_to_Xen_XIconSize(&(sizes[i])), lst);
  XFree(sizes);
  xm_unprotect_at(loc);
  return(Xen_list_2(C_int_to_Xen_integer(val),
		    lst));
}

static Xen gxm_XFindContext(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XFindContext "XFindContext(dpy, rid, context) gets data from the context manager"
  /* DIFF: XFindContext last arg omitted, val returned
   */
  caddr_t x;
  int val;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XFindContext", "Display*");
  Xen_check_type(Xen_is_ulong(arg2), arg2, 2, "XFindContext", "XID");
  Xen_check_type(Xen_is_XContext(arg3), arg3, 3, "XFindContext", "XContext");
  val = XFindContext(Xen_to_C_Display(arg1), Xen_ulong_to_C_ulong(arg2), Xen_to_C_XContext(arg3), &x);
  return(Xen_list_2(C_int_to_Xen_integer(val),
		    (Xen)x));
}

static Xen gxm_XEqualRegion(Xen arg1, Xen arg2)
{
  #define H_XEqualRegion "Bool XEqualRegion(r1, r2): returns " PROC_TRUE " if the two regions have the same offset, size, and shape."
  Xen_check_type(Xen_is_Region(arg1), arg1, 1, "XEqualRegion", "Region");
  Xen_check_type(Xen_is_Region(arg2), arg2, 2, "XEqualRegion", "Region");
  return(C_bool_to_Xen_boolean(XEqualRegion(Xen_to_C_Region(arg1), Xen_to_C_Region(arg2))));
}

static Xen gxm_XEmptyRegion(Xen arg1)
{
  #define H_XEmptyRegion "Bool XEmptyRegion(r): returns " PROC_TRUE " if the region is empty."
  Xen_check_type(Xen_is_Region(arg1), arg1, 1, "XEmptyRegion", "Region");
  return(C_bool_to_Xen_boolean(XEmptyRegion(Xen_to_C_Region(arg1))));
}

static Xen gxm_XDestroyRegion(Xen arg1)
{
  #define H_XDestroyRegion "XDestroyRegion(r) deallocates the storage associated with a specified region."
  Xen_check_type(Xen_is_Region(arg1), arg1, 1, "XDestroyRegion", "Region");
  return(C_int_to_Xen_integer(XDestroyRegion(Xen_to_C_Region(arg1))));
}

static Xen gxm_XDeleteContext(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XDeleteContext "int XDeleteContext(display, rid, context) deletes the entry for the given resource ID and type from the data structure."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XDeleteContext", "Display*");
  Xen_check_type(Xen_is_ulong(arg2), arg2, 2, "XDeleteContext", "XID");
  Xen_check_type(Xen_is_XContext(arg3), arg3, 3, "XDeleteContext", "XContext");
  return(C_int_to_Xen_integer(XDeleteContext(Xen_to_C_Display(arg1), Xen_ulong_to_C_ulong(arg2), Xen_to_C_XContext(arg3))));
}

static Xen gxm_XDefaultString(void)
{
  #define H_XDefaultString "char *XDefaultString(): returns the default string used by Xlib for text conversion"
  return(C_string_to_Xen_string(XDefaultString()));
}

static Xen gxm_XCreateRegion(void)
{
  #define H_XCreateRegion "Region XCreateRegion()"
  return(C_to_Xen_Region(XCreateRegion()));
}

static Xen gxm_XClipBox(Xen arg1)
{
  #define H_XClipBox "XClipBox(r): returns the smallest rectangle enclosing the specified region."
  /* DIFF: XClipBox region [rectangle] -> (list val rectangle)
   */
  XRectangle *r;
  int val;
  Xen_check_type(Xen_is_Region(arg1), arg1, 1, "XClipBox", "Region");
  r = (XRectangle *)calloc(1, sizeof(XRectangle));
  val = XClipBox(Xen_to_C_Region(arg1), r);
  return(Xen_list_2(C_int_to_Xen_integer(val), C_to_Xen_XRectangle(r)));
}

static Xen gxm_XAllocWMHints(void)
{
  #define H_XAllocWMHints "XAllocWMHints() allocates a window manager hints structure"
  return(C_to_Xen_XWMHints(XAllocWMHints()));
}

static Xen gxm_XAllocStandardColormap(void)
{
  #define H_XAllocStandardColormap "XStandardColormap *XAllocStandardColormap() allocates and returns a pointer to a XStandardColormap structure."
  return(C_to_Xen_XStandardColormap(XAllocStandardColormap()));
}

static Xen gxm_XAllocIconSize(void)
{
  #define H_XAllocIconSize "XIconSize *XAllocIconSize() allocates and returns a pointer to a XIconSize structure."
  return(C_to_Xen_XIconSize(XAllocIconSize()));
}

static Xen gxm_XFilterEvent(Xen arg1, Xen arg2)
{
  #define H_XFilterEvent "Bool XFilterEvent(event, w) passes the event to any filters registered for it in the given window"
  Xen_check_type(Xen_is_XEvent(arg1), arg1, 1, "XFilterEvent", "XEvent*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XFilterEvent", "Window");
  return(C_bool_to_Xen_boolean(XFilterEvent(Xen_to_C_XEvent(arg1), Xen_to_C_Window(arg2))));
}

static Xen gxm_XContextualDrawing(Xen arg1)
{
  #define H_XContextualDrawing "Bool XContextualDrawing(font_set): returns " PROC_TRUE " if text drawn with the font set might include context-dependent drawing."
  Xen_check_type(Xen_is_XFontSet(arg1), arg1, 1, "XContextualDrawing", "XFontSet");
  return(C_bool_to_Xen_boolean(XContextualDrawing(Xen_to_C_XFontSet(arg1))));
}

static Xen gxm_XDirectionalDependentDrawing(Xen arg1)
{
  #define H_XDirectionalDependentDrawing "Bool XDirectionalDependentDrawing(font_set): returns " PROC_TRUE " if the drawing functions implement \
implicit text directionality."
  Xen_check_type(Xen_is_XFontSet(arg1), arg1, 1, "XDirectionalDependentDrawing", "XFontSet");
  return(C_bool_to_Xen_boolean(XDirectionalDependentDrawing(Xen_to_C_XFontSet(arg1))));
}

static Xen gxm_XContextDependentDrawing(Xen arg1)
{
  #define H_XContextDependentDrawing "Bool XContextDependentDrawing(font_set): returns " PROC_TRUE " if the drawing functions implement implicit \
text directionality or if text drawn with the font_set might include context-dependent drawing."
  Xen_check_type(Xen_is_XFontSet(arg1), arg1, 1, "XContextDependentDrawing", "XFontSet");
  return(C_bool_to_Xen_boolean(XContextDependentDrawing(Xen_to_C_XFontSet(arg1))));
}

static Xen gxm_XLocaleOfFontSet(Xen arg1)
{
  #define H_XLocaleOfFontSet "char *XLocaleOfFontSet(font_set): returns the name of the locale bound to the specified XFontSet, as a \
null-terminated string."
  Xen_check_type(Xen_is_XFontSet(arg1), arg1, 1, "XLocaleOfFontSet", "XFontSet");
  return(C_string_to_Xen_string(XLocaleOfFontSet(Xen_to_C_XFontSet(arg1))));
}

static Xen gxm_XBaseFontNameListOfFontSet(Xen arg1)
{
  #define H_XBaseFontNameListOfFontSet "char *XBaseFontNameListOfFontSet(font_set): returns the original base font name list supplied \
by the client when the  XFontSet was created."
  Xen_check_type(Xen_is_XFontSet(arg1), arg1, 1, "XBaseFontNameListOfFontSet", "XFontSet");
  return(C_string_to_Xen_string(XBaseFontNameListOfFontSet(Xen_to_C_XFontSet(arg1))));
}

static Xen gxm_XFontsOfFontSet(Xen arg1)
{
  #define H_XFontsOfFontSet "int XFontsOfFontSet(font_set): returns a list of one or more \
XFontStructs and font names for the fonts used by the Xmb and Xwc layers, for the given font set."
  /* DIFF: XFontsOfFontSet omit last 2 args, return 2 lists
   */
  int i, len, loc1, loc2;
  char **names;
  Xen lst1 = Xen_empty_list, lst2 = Xen_empty_list;
  XFontStruct **fs;
  Xen_check_type(Xen_is_XFontSet(arg1), arg1, 1, "XFontsOfFontSet", "XFontSet");
  len = XFontsOfFontSet(Xen_to_C_XFontSet(arg1), &fs, &names);
  loc1 = xm_protect(lst1);
  loc2 = xm_protect(lst2);
  for (i = len - 1; i >= 0; i--)
    {
      lst1 = Xen_cons(C_to_Xen_XFontStruct(fs[i]), lst1);
      lst2 = Xen_cons(C_string_to_Xen_string(names[i]), lst2);
    }
  xm_unprotect_at(loc1);
  xm_unprotect_at(loc2);
  return(Xen_list_2(lst1, lst2));
}

static Xen gxm_XFreeFontSet(Xen arg1, Xen arg2)
{
  #define H_XFreeFontSet "void XFreeFontSet(display, font_set) frees the specified font set."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XFreeFontSet", "Display*");
  Xen_check_type(Xen_is_XFontSet(arg2), arg2, 2, "XFreeFontSet", "XFontSet");
  XFreeFontSet(Xen_to_C_Display(arg1), Xen_to_C_XFontSet(arg2));
  return(Xen_false);
}

static Xen gxm_XCreateFontSet(Xen arg1, Xen arg2)
{
  #define H_XCreateFontSet "XFontSet XCreateFontSet(display, base_font_name_list) creates a font set for the specified display."
  /* DIFF: XCreateFontSet ignores (omits) the 3 trailing missing glyph args
   */
  char **cs;
  int len;
  char *str;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XCreateFontSet", "Display*");
  Xen_check_type(Xen_is_string(arg2), arg2, 2, "XCreateFontSet", "char*");
  return(C_to_Xen_XFontSet(XCreateFontSet(Xen_to_C_Display(arg1), (char *)Xen_string_to_C_string(arg2), &cs, &len, &str)));
}

static Xen gxm_XSetLocaleModifiers(Xen arg1)
{
  #define H_XSetLocaleModifiers "char *XSetLocaleModifiers(modifier_list) sets the X modifiers for the current locale setting."
  Xen_check_type(Xen_is_string(arg1), arg1, 1, "XSetLocaleModifiers", "char*");
  return(C_string_to_Xen_string(XSetLocaleModifiers(Xen_string_to_C_string(arg1))));
}

static Xen gxm_XSupportsLocale(void)
{
  #define H_XSupportsLocale "Bool XSupportsLocale(): returns " PROC_TRUE " if Xlib functions are capable of operating under the current locale."
  return(C_bool_to_Xen_boolean(XSupportsLocale()));
}

static Xen gxm_XWriteBitmapFile(Xen arg1, Xen arg2, Xen arg3, Xen arg4, Xen arg5, Xen arg6, Xen arg7)
{
  #define H_XWriteBitmapFile "int XWriteBitmapFile(display, filename, bitmap, width, height, x_hot, y_hot) writes a bitmap out to a file in \
the X Version 11 format."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XWriteBitmapFile", "Display*");
  Xen_check_type(Xen_is_string(arg2), arg2, 2, "XWriteBitmapFile", "char*");
  Xen_check_type(Xen_is_Pixmap(arg3), arg3, 3, "XWriteBitmapFile", "Pixmap");
  Xen_check_type(Xen_is_ulong(arg4), arg4, 4, "XWriteBitmapFile", "unsigned int");
  Xen_check_type(Xen_is_ulong(arg5), arg5, 5, "XWriteBitmapFile", "unsigned int");
  Xen_check_type(Xen_is_integer(arg6), arg6, 6, "XWriteBitmapFile", "int");
  Xen_check_type(Xen_is_integer(arg7), arg7, 7, "XWriteBitmapFile", "int");
  return(C_int_to_Xen_integer(XWriteBitmapFile(Xen_to_C_Display(arg1), 
				       (char *)Xen_string_to_C_string(arg2), 
				       Xen_to_C_Pixmap(arg3), 
				       Xen_ulong_to_C_ulong(arg4), Xen_ulong_to_C_ulong(arg5), 
				       Xen_integer_to_C_int(arg6), Xen_integer_to_C_int(arg7))));
}

static Xen gxm_XWindowEvent(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XWindowEvent "XWindowEvent(display, w, event_mask) searches the event queue for an event that matches both \
the specified window and event mask, and removes it or waits until it arrives."
  /* DIFF: XWindowEvent dpy win ev [evrtn] -> (list val evrtn)
   */
  XEvent *e;
  int val;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XWindowEvent", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XWindowEvent", "Window");
  Xen_check_type(Xen_is_integer(arg3), arg3, 3, "XWindowEvent", "long");
  e = (XEvent *)calloc(1, sizeof(XEvent));
  val = XWindowEvent(Xen_to_C_Display(arg1), Xen_to_C_Window(arg2), Xen_integer_to_C_int(arg3), e);
  return(Xen_list_2(C_int_to_Xen_integer(val), C_to_Xen_XEvent_OBJ(e)));
}

static Xen gxm_XWidthOfScreen(Xen arg1)
{
  #define H_WidthOfScreen "XWidthOfScreen(screen): returns the width of the specified screen."
  Xen_check_type(Xen_is_Screen(arg1), arg1, 1, "XWidthOfScreen", "Screen*");
  return(C_int_to_Xen_integer(XWidthOfScreen(Xen_to_C_Screen(arg1))));
}

static Xen gxm_XWidthMMOfScreen(Xen arg1)
{
  #define H_WidthMMOfScreen "XWidthMMOfScreen(screen): returns the width of the specified screen in millimeters."
  Xen_check_type(Xen_is_Screen(arg1), arg1, 1, "XWidthMMOfScreen", "Screen*");
  return(C_int_to_Xen_integer(XWidthMMOfScreen(Xen_to_C_Screen(arg1))));
}

static Xen gxm_XWarpPointer(Xen arg1, Xen arg2, Xen arg3, Xen arg4, Xen arg5, Xen arg6, Xen arg7, Xen arg8, Xen arg9)
{
  #define H_XWarpPointer "XWarpPointer(display, src_w, dest_w, src_x, src_y, src_width, src_height, dest_x, dest_y)"
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XWarpPointer", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XWarpPointer", "Window");
  Xen_check_type(Xen_is_Window(arg3), arg3, 3, "XWarpPointer", "Window");
  Xen_check_type(Xen_is_integer(arg4), arg4, 4, "XWarpPointer", "int");
  Xen_check_type(Xen_is_integer(arg5), arg5, 5, "XWarpPointer", "int");
  Xen_check_type(Xen_is_ulong(arg6), arg6, 6, "XWarpPointer", "unsigned int");
  Xen_check_type(Xen_is_ulong(arg7), arg7, 7, "XWarpPointer", "unsigned int");
  Xen_check_type(Xen_is_integer(arg8), arg8, 8, "XWarpPointer", "int");
  Xen_check_type(Xen_is_integer(arg9), arg9, 9, "XWarpPointer", "int");
  return(C_int_to_Xen_integer(XWarpPointer(Xen_to_C_Display(arg1), 
				   Xen_to_C_Window(arg2), Xen_to_C_Window(arg3), 
				   Xen_integer_to_C_int(arg4), Xen_integer_to_C_int(arg5), 
				   Xen_ulong_to_C_ulong(arg6), Xen_ulong_to_C_ulong(arg7), 
				   Xen_integer_to_C_int(arg8), Xen_integer_to_C_int(arg9))));
}

static Xen gxm_XVendorRelease(Xen arg1)
{
  #define H_VendorRelease "VendorRelease(display): returns a number related to a vendor's release of the X server."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XVendorRelease", "Display*");
  return(C_int_to_Xen_integer(XVendorRelease(Xen_to_C_Display(arg1))));
}

static Xen gxm_XUnmapWindow(Xen arg1, Xen arg2)
{
  #define H_XUnmapWindow "XUnmapWindow(display, w) unmaps the specified window and causes the X server to generate an UnmapNotify event."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XUnmapWindow", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XUnmapWindow", "Window");
  return(C_int_to_Xen_integer(XUnmapWindow(Xen_to_C_Display(arg1), Xen_to_C_Window(arg2))));
}

static Xen gxm_XUnmapSubwindows(Xen arg1, Xen arg2)
{
  #define H_XUnmapSubwindows "XUnmapSubwindows(display, w) unmaps all subwindows for the specified window in bottom-to-top stacking order."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XUnmapSubwindows", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XUnmapSubwindows", "Window");
  return(C_int_to_Xen_integer(XUnmapSubwindows(Xen_to_C_Display(arg1), Xen_to_C_Window(arg2))));
}

static Xen gxm_XUnloadFont(Xen arg1, Xen arg2)
{
  #define H_XUnloadFont "XUnloadFont(display, font) deletes the association between the font resource ID and the specified font."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XUnloadFont", "Display*");
  Xen_check_type(Xen_is_Font(arg2), arg2, 2, "XUnloadFont", "Font");
  return(C_int_to_Xen_integer(XUnloadFont(Xen_to_C_Display(arg1), Xen_to_C_Font(arg2))));
}

static Xen gxm_XUninstallColormap(Xen arg1, Xen arg2)
{
  #define H_XUninstallColormap "XUninstallColormap(display, colormap) removes the specified colormap from the required list for its screen."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XUninstallColormap", "Display*");
  Xen_check_type(Xen_is_Colormap(arg2), arg2, 2, "XUninstallColormap", "Colormap");
  return(C_int_to_Xen_integer(XUninstallColormap(Xen_to_C_Display(arg1), Xen_to_C_Colormap(arg2))));
}

static Xen gxm_XUngrabServer(Xen arg1)
{
  #define H_XUngrabServer "XUngrabServer(display) restarts processing of requests and close downs on other connections."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XUngrabServer", "Display*");
  return(C_int_to_Xen_integer(XUngrabServer(Xen_to_C_Display(arg1))));
}

static Xen gxm_XUngrabPointer(Xen arg1, Xen arg2)
{
  #define H_XUngrabPointer "XUngrabPointer(display, time) releases the pointer and any queued events if this client has actively grabbed the \
pointer from XGrabPointer, XGrabButton,or from a normal button press."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XUngrabPointer", "Display*");
  Xen_check_type(Xen_is_Time(arg2), arg2, 2, "XUngrabPointer", "Time");
  return(C_int_to_Xen_integer(XUngrabPointer(Xen_to_C_Display(arg1), Xen_to_C_Time(arg2))));
}

static Xen gxm_XUngrabKeyboard(Xen arg1, Xen arg2)
{
  #define H_XUngrabKeyboard "XUngrabKeyboard(display, time) releases the keyboard and any queued events if this client has it actively grabbed \
from either XGrabKeyboard or XGrabKey."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XUngrabKeyboard", "Display*");
  Xen_check_type(Xen_is_Time(arg2), arg2, 2, "XUngrabKeyboard", "Time");
  return(C_int_to_Xen_integer(XUngrabKeyboard(Xen_to_C_Display(arg1), Xen_to_C_Time(arg2))));
}

static Xen gxm_XUngrabKey(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XUngrabKey "XUngrabKey(display, keycode, modifiers, grab_window) releases the key combination on the specified window if it was \
grabbed by this client."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XUngrabKey", "Display*");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XUngrabKey", "int");
  Xen_check_type(Xen_is_ulong(arg3), arg3, 3, "XUngrabKey", "unsigned int");
  Xen_check_type(Xen_is_Window(arg4), arg4, 4, "XUngrabKey", "Window");
  return(C_int_to_Xen_integer(XUngrabKey(Xen_to_C_Display(arg1), Xen_integer_to_C_int(arg2), Xen_ulong_to_C_ulong(arg3), Xen_to_C_Window(arg4))));
}

static Xen gxm_XUngrabButton(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XUngrabButton "XUngrabButton(display, button, modifiers, grab_window) releases the passive button/key combination on the specified \
window if it was grabbed by this client."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XUngrabButton", "Display*");
  Xen_check_type(Xen_is_ulong(arg2), arg2, 2, "XUngrabButton", "unsigned int");
  Xen_check_type(Xen_is_ulong(arg3), arg3, 3, "XUngrabButton", "unsigned int");
  Xen_check_type(Xen_is_Window(arg4), arg4, 4, "XUngrabButton", "Window");
  return(C_int_to_Xen_integer(XUngrabButton(Xen_to_C_Display(arg1), Xen_ulong_to_C_ulong(arg2), Xen_ulong_to_C_ulong(arg3), Xen_to_C_Window(arg4))));
}

static Xen gxm_XUndefineCursor(Xen arg1, Xen arg2)
{
  #define H_XUndefineCursor "XUndefineCursor(display, w) undoes the effect of a previous XDefineCursor for this window."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XUndefineCursor", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XUndefineCursor", "Window");
  return(C_int_to_Xen_integer(XUndefineCursor(Xen_to_C_Display(arg1), Xen_to_C_Window(arg2))));
}

static Xen gxm_XTranslateCoordinates(Xen arg1, Xen arg2, Xen arg3, Xen arg4, Xen arg5)
{
  #define H_XTranslateCoordinates "Bool XTranslateCoordinates(display, src_w, dest_w, src_x, src_y) \
takes the src_x and src_y coordinates relative to the source window's origin and returns these coordinates to dest_x_return and dest_y_return \
relative to the destination window's origin -> (rtn x y win)."
  /* DIFF: XTranslateCoordinates omit last 3 args
   */
  Window w;
  int x, y, rtn;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XTranslateCoordinates", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XTranslateCoordinates", "Window");
  Xen_check_type(Xen_is_Window(arg3), arg3, 3, "XTranslateCoordinates", "Window");
  Xen_check_type(Xen_is_integer(arg4), arg4, 4, "XTranslateCoordinates", "int");
  Xen_check_type(Xen_is_integer(arg5), arg5, 5, "XTranslateCoordinates", "int");
  rtn = XTranslateCoordinates(Xen_to_C_Display(arg1), 
			      Xen_to_C_Window(arg2), 
			      Xen_to_C_Window(arg3), 
			      Xen_integer_to_C_int(arg4), Xen_integer_to_C_int(arg5), 
			      &x, &y, &w);
  return(Xen_list_4(C_bool_to_Xen_boolean(rtn),
		    C_int_to_Xen_integer(x),
		    C_int_to_Xen_integer(y),
		    C_to_Xen_Window(w)));
}

static Xen gxm_XTextWidth(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XTextWidth "int XTextWidth(font_struct, string, count): returns the width of the specified 8-bit string."
  Xen_check_type(Xen_is_XFontStruct(arg1), arg1, 1, "XTextWidth", "XFontStruct*");
  Xen_check_type(Xen_is_string(arg2), arg2, 2, "XTextWidth", "char*");
  Xen_check_type(Xen_is_integer(arg3), arg3, 3, "XTextWidth", "int");
  return(C_int_to_Xen_integer(XTextWidth(Xen_to_C_XFontStruct(arg1), (char *)Xen_string_to_C_string(arg2), Xen_integer_to_C_int(arg3))));
}

static Xen gxm_XTextExtents(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XTextExtents "XTextExtents(font_struct, string, nchars): returns an XCharStruct structure describing the text."
  /* DIFF: XTextExtents omit final 4 args and returns them [Xcharset returned as embedded list)
   */
  int dir, fa, fd, rtn;
  XCharStruct val;
  Xen_check_type(Xen_is_XFontStruct(arg1), arg1, 1, "XTextExtents", "XFontStruct*");
  Xen_check_type(Xen_is_string(arg2), arg2, 2, "XTextExtents", "char*");
  Xen_check_type(Xen_is_integer(arg3), arg3, 3, "XTextExtents", "int");
  rtn = XTextExtents(Xen_to_C_XFontStruct(arg1), 
		     (char *)Xen_string_to_C_string(arg2), 
		     Xen_integer_to_C_int(arg3), 
		     &dir, &fa, &fd, &val);
  return(Xen_list_5(C_int_to_Xen_integer(rtn),
		    C_int_to_Xen_integer(dir),
		    C_int_to_Xen_integer(fa),
		    C_int_to_Xen_integer(fd),
		    Xen_list_6(C_int_to_Xen_integer((int)(val.lbearing)),
			       C_int_to_Xen_integer((int)(val.rbearing)),
			       C_int_to_Xen_integer((int)(val.width)),
			       C_int_to_Xen_integer((int)(val.ascent)),
			       C_int_to_Xen_integer((int)(val.descent)),
			       C_int_to_Xen_integer((int)(val.attributes)))));
}

static Xen gxm_XSync(Xen arg1, Xen arg2)
{
  #define H_XSync "XSync(display, discard) flushes the output buffer and then waits until all requests have been received and processed by the X server."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XSync", "Display*");
  Xen_check_type(Xen_is_boolean(arg2), arg2, 2, "XSync", "Bool");
  return(C_int_to_Xen_integer(XSync(Xen_to_C_Display(arg1), Xen_boolean_to_C_bool(arg2))));
}

static Xen gxm_XStoreNamedColor(Xen arg1, Xen arg2, Xen arg3, Xen arg4, Xen arg5)
{
  #define H_XStoreNamedColor "XStoreNamedColor(display, colormap, color, pixel, flags) looks up the named color with respect to the screen \
associated with the colormap and stores the result in the specified colormap."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XStoreNamedColor", "Display*");
  Xen_check_type(Xen_is_Colormap(arg2), arg2, 2, "XStoreNamedColor", "Colormap");
  Xen_check_type(Xen_is_string(arg3), arg3, 3, "XStoreNamedColor", "char*");
  Xen_check_type(Xen_is_ulong(arg4), arg4, 4, "XStoreNamedColor", "ulong"); /* this is explicitly an index into the colormap, so I'll leave it as is */
  Xen_check_type(Xen_is_integer(arg5), arg5, 5, "XStoreNamedColor", "int");
  return(C_int_to_Xen_integer(XStoreNamedColor(Xen_to_C_Display(arg1), 
				       Xen_to_C_Colormap(arg2), 
				       (char *)Xen_string_to_C_string(arg3), Xen_ulong_to_C_ulong(arg4), Xen_integer_to_C_int(arg5))));
}

static Xen gxm_XStoreName(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XStoreName "XStoreName(display, w, window_name) assigns the name passed to window_name to the specified window."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XStoreName", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XStoreName", "Window");
  Xen_check_type(Xen_is_string(arg3), arg3, 3, "XStoreName", "char*");
  return(C_int_to_Xen_integer(XStoreName(Xen_to_C_Display(arg1), Xen_to_C_Window(arg2), (char *)Xen_string_to_C_string(arg3))));
}

static Xen gxm_XStoreColors(Xen arg1, Xen arg2, Xen larg3, Xen arg4)
{
  #define H_XStoreColors "XStoreColors(display, colormap, color, ncolors) changes the colormap entries of the pixel values specified in the \
pixel members of the XColor structures."
  /* DIFF: XStoreColors arg 3 is list of XColor 
   */
  XColor *xc;
  int i, len;
  Xen arg3;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XStoreColors", "Display*");
  Xen_check_type(Xen_is_Colormap(arg2), arg2, 2, "XStoreColors", "Colormap");
  Xen_check_type(Xen_is_list(larg3), larg3, 3, "XStoreColors", "list of XColor");
  Xen_check_type(Xen_is_integer(arg4), arg4, 4, "XStoreColors", "int");
  arg3 = Xen_copy_arg(larg3);
  len = Xen_integer_to_C_int(arg4);
  if (len <= 0) Xen_check_type(0, arg4, 4, "XStoreColors", "positive integer");
  xc = (XColor *)calloc(len, sizeof(XColor));
  for (i = 0; (i < len) && (!Xen_is_null(arg3)); i++, arg3 = Xen_cdr(arg3))
    {
      XColor *xc1;
      xc1 = Xen_to_C_XColor(Xen_car(arg3));
      xc[i].pixel = xc1->pixel;
      xc[i].red = xc1->red;
      xc[i].green = xc1->green;
      xc[i].blue = xc1->blue;
      xc[i].flags = xc1->flags;
      xc[i].pad = xc1->pad;
    }
  XStoreColors(Xen_to_C_Display(arg1), 
	       Xen_to_C_Colormap(arg2),
	       xc, len);
  free(xc);
  return(C_int_to_Xen_integer(len));
}

static Xen gxm_XStoreColor(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XStoreColor "XStoreColor(display, colormap, color) changes the colormap entry of the pixel value specified in the pixel member \
of the XColor structure."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XStoreColor", "Display*");
  Xen_check_type(Xen_is_Colormap(arg2), arg2, 2, "XStoreColor", "Colormap");
  Xen_check_type(Xen_is_XColor(arg3), arg3, 3, "XStoreColor", "XColor"); 
  return(C_int_to_Xen_integer(XStoreColor(Xen_to_C_Display(arg1), Xen_to_C_Colormap(arg2), Xen_to_C_XColor(arg3))));
}

static Xen gxm_XStoreBytes(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XStoreBytes "XStoreBytes(display, bytes, nbytes)"
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XStoreBytes", "Display*");
  Xen_check_type(Xen_is_string(arg2), arg2, 2, "XStoreBytes", "char*");
  Xen_check_type(Xen_is_integer(arg3), arg3, 3, "XStoreBytes", "int");
  return(C_int_to_Xen_integer(XStoreBytes(Xen_to_C_Display(arg1), (char *)Xen_string_to_C_string(arg2), Xen_integer_to_C_int(arg3) + 1)));
}

static Xen gxm_XStoreBuffer(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XStoreBuffer "XStoreBuffer(display, bytes, nbytes, buffer)"
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XStoreBuffer", "Display*");
  Xen_check_type(Xen_is_string(arg2), arg2, 2, "XStoreBuffer", "char*");
  Xen_check_type(Xen_is_integer(arg3), arg3, 3, "XStoreBuffer", "int");
  Xen_check_type(Xen_is_integer(arg4), arg4, 4, "XStoreBuffer", "int");
  return(C_int_to_Xen_integer(XStoreBuffer(Xen_to_C_Display(arg1), (char *)Xen_string_to_C_string(arg2), Xen_integer_to_C_int(arg3) + 1, Xen_integer_to_C_int(arg4))));
}

static Xen gxm_XSetWindowColormap(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XSetWindowColormap "XSetWindowColormap(display, w, colormap) sets the specified colormap of the specified window."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XSetWindowColormap", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XSetWindowColormap", "Window");
  Xen_check_type(Xen_is_Colormap(arg3), arg3, 3, "XSetWindowColormap", "Colormap");
  return(C_int_to_Xen_integer(XSetWindowColormap(Xen_to_C_Display(arg1), Xen_to_C_Window(arg2), Xen_to_C_Colormap(arg3))));
}

static Xen gxm_XSetWindowBorderWidth(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XSetWindowBorderWidth "XSetWindowBorderWidth(display, w, width) sets the specified window's border width to the specified width."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XSetWindowBorderWidth", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XSetWindowBorderWidth", "Window");
  Xen_check_type(Xen_is_ulong(arg3), arg3, 3, "XSetWindowBorderWidth", "unsigned int");
  return(C_int_to_Xen_integer(XSetWindowBorderWidth(Xen_to_C_Display(arg1), Xen_to_C_Window(arg2), Xen_ulong_to_C_ulong(arg3))));
}

static Xen gxm_XSetWindowBorderPixmap(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XSetWindowBorderPixmap "XSetWindowBorderPixmap(display, w, border_pixmap) sets the border pixmap of the window to the pixmap you specify."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XSetWindowBorderPixmap", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XSetWindowBorderPixmap", "Window");
  Xen_check_type(Xen_is_Pixmap(arg3) || Xen_is_integer(arg3), arg3, 3, "XSetWindowBorderPixmap", "Pixmap");
  return(C_int_to_Xen_integer(XSetWindowBorderPixmap(Xen_to_C_Display(arg1), 
					     Xen_to_C_Window(arg2),
					     (Xen_is_Pixmap(arg3)) ? Xen_to_C_Pixmap(arg3) : CopyFromParent)));
}

static Xen gxm_XSetWindowBorder(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XSetWindowBorder "XSetWindowBorder(display, w, border_pixel) sets the border of the window to the pixel value you specify."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XSetWindowBorder", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XSetWindowBorder", "Window");
  Xen_check_type(Xen_is_Pixel(arg3), arg3, 3, "XSetWindowBorder", "pixel"); 
  return(C_int_to_Xen_integer(XSetWindowBorder(Xen_to_C_Display(arg1), Xen_to_C_Window(arg2), Xen_to_C_Pixel(arg3))));
}

static Xen gxm_XSetWindowBackgroundPixmap(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XSetWindowBackgroundPixmap "XSetWindowBackgroundPixmap(display, w, background_pixmap) sets the background pixmap of the window to \
the specified pixmap."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XSetWindowBackgroundPixmap", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XSetWindowBackgroundPixmap", "Window");
  Xen_check_type(Xen_is_Pixmap(arg3) || Xen_is_integer(arg3), arg3, 3, "XSetWindowBackgroundPixmap", "Pixmap");
  return(C_int_to_Xen_integer(XSetWindowBackgroundPixmap(Xen_to_C_Display(arg1), 
						 Xen_to_C_Window(arg2), 
						 (Xen_is_Pixmap(arg3)) ? Xen_to_C_Pixmap(arg3) : Xen_ulong_to_C_ulong(arg3))));
}

static Xen gxm_XSetWindowBackground(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XSetWindowBackground "XSetWindowBackground(display, w, background_pixel) sets the background of the window to the specified pixel value."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XSetWindowBackground", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XSetWindowBackground", "Window");
  Xen_check_type(Xen_is_Pixel(arg3), arg3, 3, "XSetWindowBackground", "pixel"); 
  return(C_int_to_Xen_integer(XSetWindowBackground(Xen_to_C_Display(arg1), Xen_to_C_Window(arg2), Xen_to_C_Pixel(arg3))));
}

static Xen gxm_XSetTile(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XSetTile "XSetTile(display, gc, tile) sets the fill tile in the specified GC."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XSetTile", "Display*");
  Xen_check_type(Xen_is_GC(arg2), arg2, 2, "XSetTile", "GC");
  Xen_check_type(Xen_is_Pixmap(arg3), arg3, 3, "XSetTile", "Pixmap");
  return(C_int_to_Xen_integer(XSetTile(Xen_to_C_Display(arg1), Xen_to_C_GC(arg2), Xen_to_C_Pixmap(arg3))));
}

static Xen gxm_XSetTSOrigin(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XSetTSOrigin "XSetTSOrigin(display, gc, ts_x_origin, ts_y_origin) sets the tile/stipple origin in the specified GC."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XSetTSOrigin", "Display*");
  Xen_check_type(Xen_is_GC(arg2), arg2, 2, "XSetTSOrigin", "GC");
  Xen_check_type(Xen_is_integer(arg3), arg3, 3, "XSetTSOrigin", "int");
  Xen_check_type(Xen_is_integer(arg4), arg4, 4, "XSetTSOrigin", "int");
  return(C_int_to_Xen_integer(XSetTSOrigin(Xen_to_C_Display(arg1), Xen_to_C_GC(arg2), Xen_integer_to_C_int(arg3), Xen_integer_to_C_int(arg4))));
}

static Xen gxm_XSetSubwindowMode(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XSetSubwindowMode "XSetSubwindowMode(display, gc, subwindow_mode) sets the subwindow mode in the specified GC."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XSetSubwindowMode", "Display*");
  Xen_check_type(Xen_is_GC(arg2), arg2, 2, "XSetSubwindowMode", "GC");
  Xen_check_type(Xen_is_integer(arg3), arg3, 3, "XSetSubwindowMode", "int");
  return(C_int_to_Xen_integer(XSetSubwindowMode(Xen_to_C_Display(arg1), Xen_to_C_GC(arg2), Xen_integer_to_C_int(arg3))));
}

static Xen gxm_XSetStipple(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XSetStipple "XSetStipple(display, gc, stipple) sets the stipple in the specified GC."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XSetStipple", "Display*");
  Xen_check_type(Xen_is_GC(arg2), arg2, 2, "XSetStipple", "GC");
  Xen_check_type(Xen_is_Pixmap(arg3), arg3, 3, "XSetStipple", "Pixmap");
  return(C_int_to_Xen_integer(XSetStipple(Xen_to_C_Display(arg1), Xen_to_C_GC(arg2), Xen_to_C_Pixmap(arg3))));
}

static Xen gxm_XSetState(Xen arg1, Xen arg2, Xen arg3, Xen arg4, Xen arg5, Xen arg6)
{
  #define H_XSetState "XSetState(display, gc, foreground, background, function, plane_mask) sets the foreground, background, plane mask, and \
function components for the specified GC."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XSetState", "Display*");
  Xen_check_type(Xen_is_GC(arg2), arg2, 2, "XSetState", "GC");
  Xen_check_type(Xen_is_Pixel(arg3), arg3, 3, "XSetState", "Pixel");
  Xen_check_type(Xen_is_Pixel(arg4), arg4, 4, "XSetState", "Pixel");
  Xen_check_type(Xen_is_integer(arg5), arg5, 5, "XSetState", "int");
  Xen_check_type(Xen_is_ulong(arg6), arg6, 6, "XSetState", "ulong");
  return(C_int_to_Xen_integer(XSetState(Xen_to_C_Display(arg1), 
				Xen_to_C_GC(arg2), 
				Xen_to_C_Pixel(arg3), Xen_to_C_Pixel(arg4), 
				Xen_integer_to_C_int(arg5), Xen_ulong_to_C_ulong(arg6))));
}

static Xen gxm_XSetSelectionOwner(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XSetSelectionOwner "XSetSelectionOwner(display, selection, owner, time) changes the owner and last-change time for the specified selection"
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XSetSelectionOwner", "Display*");
  Xen_check_type(Xen_is_Atom(arg2), arg2, 2, "XSetSelectionOwner", "Atom");
  Xen_check_type(Xen_is_Window(arg3), arg3, 3, "XSetSelectionOwner", "Window");
  Xen_check_type(Xen_is_Time(arg4), arg4, 4, "XSetSelectionOwner", "Time");
  return(C_int_to_Xen_integer(XSetSelectionOwner(Xen_to_C_Display(arg1), Xen_to_C_Atom(arg2), Xen_to_C_Window(arg3), Xen_to_C_Time(arg4))));
}

static Xen gxm_XSetScreenSaver(Xen arg1, Xen arg2, Xen arg3, Xen arg4, Xen arg5)
{
  #define H_XSetScreenSaver "XSetScreenSaver(display, timeout, interval, prefer_blanking, allow_exposures) enables the screen saver."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XSetScreenSaver", "Display*");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XSetScreenSaver", "int");
  Xen_check_type(Xen_is_integer(arg3), arg3, 3, "XSetScreenSaver", "int");
  Xen_check_type(Xen_is_integer(arg4), arg4, 4, "XSetScreenSaver", "int");
  Xen_check_type(Xen_is_integer(arg5), arg5, 5, "XSetScreenSaver", "int");
  return(C_int_to_Xen_integer(XSetScreenSaver(Xen_to_C_Display(arg1), Xen_integer_to_C_int(arg2), Xen_integer_to_C_int(arg3), Xen_integer_to_C_int(arg4), Xen_integer_to_C_int(arg5))));
}

static Xen gxm_XSetPointerMapping(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XSetPointerMapping "int XSetPointerMapping(display, map, nmap) sets the mapping of the pointer."
  int i, len, rtn;
  unsigned char *map;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XSetPointerMapping", "Display*");
  Xen_check_type(Xen_is_list(arg2), arg2, 2, "XSetPointerMapping", "list of ints");
  Xen_check_type(Xen_is_integer_or_unbound(arg3), arg3, 3, "XSetPointerMapping", "int");
  if (Xen_is_integer(arg3)) len = Xen_integer_to_C_int(arg3); else len = Xen_list_length(arg2);
  map = (unsigned char *)calloc(len, sizeof(unsigned char));
  for (i = 0; i < len; i++)
    map[i] = (unsigned char)Xen_integer_to_C_int(Xen_list_ref(arg2, i));
  rtn = XSetPointerMapping(Xen_to_C_Display(arg1), map, len);
  free(map);
  return(C_int_to_Xen_integer(rtn));
}

static Xen gxm_XSetPlaneMask(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XSetPlaneMask "XSetPlaneMask(display, gc, plane_mask) sets the plane mask in the specified GC."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XSetPlaneMask", "Display*");
  Xen_check_type(Xen_is_GC(arg2), arg2, 2, "XSetPlaneMask", "GC");
  Xen_check_type(Xen_is_ulong(arg3), arg3, 3, "XSetPlaneMask", "ulong");
  return(C_int_to_Xen_integer(XSetPlaneMask(Xen_to_C_Display(arg1), Xen_to_C_GC(arg2), Xen_ulong_to_C_ulong(arg3))));
}

static Xen gxm_XSetModifierMapping(Xen arg1, Xen arg2)
{
  #define H_XSetModifierMapping "int XSetModifierMapping(display, modmap) specifies the KeyCodes of the keys (if any) that are to be used as modifiers."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XSetModifierMapping", "Display*");
  Xen_check_type(Xen_is_XModifierKeymap(arg2), arg2, 2, "XSetModifierMapping", "XModifierKeymap*");
  return(C_int_to_Xen_integer(XSetModifierMapping(Xen_to_C_Display(arg1), Xen_to_C_XModifierKeymap(arg2))));
}

static Xen gxm_XSetLineAttributes(Xen arg1, Xen arg2, Xen arg3, Xen arg4, Xen arg5, Xen arg6)
{
  #define H_XSetLineAttributes "XSetLineAttributes(display, gc, line_width, line_style, cap_style, join_style) sets the line drawing components \
in the specified GC."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XSetLineAttributes", "Display*");
  Xen_check_type(Xen_is_GC(arg2), arg2, 2, "XSetLineAttributes", "GC");
  Xen_check_type(Xen_is_ulong(arg3), arg3, 3, "XSetLineAttributes", "unsigned int");
  Xen_check_type(Xen_is_integer(arg4), arg4, 4, "XSetLineAttributes", "int");
  Xen_check_type(Xen_is_integer(arg5), arg5, 5, "XSetLineAttributes", "int");
  Xen_check_type(Xen_is_integer(arg6), arg6, 6, "XSetLineAttributes", "int");
  return(C_int_to_Xen_integer(XSetLineAttributes(Xen_to_C_Display(arg1), 
					 Xen_to_C_GC(arg2), 
					 Xen_ulong_to_C_ulong(arg3), Xen_integer_to_C_int(arg4), Xen_integer_to_C_int(arg5), Xen_integer_to_C_int(arg6))));
}

static Xen gxm_XSetInputFocus(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XSetInputFocus "XSetInputFocus(display, focus, revert_to, time) changes the input focus and the last-focus-change time."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XSetInputFocus", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XSetInputFocus", "Window");
  Xen_check_type(Xen_is_integer(arg3), arg3, 3, "XSetInputFocus", "int");
  Xen_check_type(Xen_is_Time(arg4), arg4, 4, "XSetInputFocus", "Time");
  return(C_int_to_Xen_integer(XSetInputFocus(Xen_to_C_Display(arg1), Xen_to_C_Window(arg2), Xen_integer_to_C_int(arg3), Xen_to_C_Time(arg4))));
}

static Xen gxm_XSetIconName(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XSetIconName "XSetIconName(display, w, icon_name) sets the name to be displayed in a window's icon."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XSetIconName", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XSetIconName", "Window");
  Xen_check_type(Xen_is_string(arg3), arg3, 3, "XSetIconName", "char*");
  return(C_int_to_Xen_integer(XSetIconName(Xen_to_C_Display(arg1), Xen_to_C_Window(arg2), (char *)Xen_string_to_C_string(arg3))));
}

static Xen gxm_XSetGraphicsExposures(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XSetGraphicsExposures "XSetGraphicsExposures(display, gc, graphics_exposures) sets the graphics-exposures flag in the specified GC."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XSetGraphicsExposures", "Display*");
  Xen_check_type(Xen_is_GC(arg2), arg2, 2, "XSetGraphicsExposures", "GC");
  Xen_check_type(Xen_is_boolean(arg3), arg3, 3, "XSetGraphicsExposures", "Bool");
  return(C_int_to_Xen_integer(XSetGraphicsExposures(Xen_to_C_Display(arg1), Xen_to_C_GC(arg2), Xen_boolean_to_C_bool(arg3))));
}

static Xen gxm_XSetFunction(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XSetFunction "XSetFunction(display, gc, function) sets a specified value in the specified GC"
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XSetFunction", "Display*");
  Xen_check_type(Xen_is_GC(arg2), arg2, 2, "XSetFunction", "GC");
  Xen_check_type(Xen_is_integer(arg3), arg3, 3, "XSetFunction", "int");
  return(C_int_to_Xen_integer(XSetFunction(Xen_to_C_Display(arg1), Xen_to_C_GC(arg2), Xen_integer_to_C_int(arg3))));
}

static Xen gxm_XSetForeground(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XSetForeground "XSetForeground(display, gc, foreground) sets the foreground in the specified GC."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XSetForeground", "Display*");
  Xen_check_type(Xen_is_GC(arg2), arg2, 2, "XSetForeground", "GC");
  Xen_check_type(Xen_is_Pixel(arg3), arg3, 3, "XSetForeground", "Pixel");
  return(C_int_to_Xen_integer(XSetForeground(Xen_to_C_Display(arg1), Xen_to_C_GC(arg2), Xen_to_C_Pixel(arg3))));
}

static Xen gxm_XSetFontPath(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XSetFontPath "XSetFontPath(display, directories, ndirs) defines the directory search path for font lookup."
  /* DIFF: XSetFontPath arg2 is list of strings
   */
  char **paths;
  int len, rtn;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XSetFontPath", "Display*");
  Xen_check_type(Xen_is_list(arg2), arg2, 2, "XSetFontPath", "list of char*");
  Xen_check_type(Xen_is_integer(arg3), arg3, 3, "XSetFontPath", "int");
  len = Xen_integer_to_C_int(arg3);
  if (len > 0)
    paths = Xen_to_C_Strings(arg2, len);
  else paths = NULL;
  rtn = XSetFontPath(Xen_to_C_Display(arg1), paths, len);
  if (paths) free(paths);
  return(C_int_to_Xen_integer(rtn));
}

static Xen gxm_XSetFont(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XSetFont "XSetFont(display, gc, font) sets the current font in the specified GC."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XSetFont", "Display*");
  Xen_check_type(Xen_is_GC(arg2), arg2, 2, "XSetFont", "GC");
  Xen_check_type(Xen_is_Font(arg3), arg3, 3, "XSetFont", "Font");
  return(C_int_to_Xen_integer(XSetFont(Xen_to_C_Display(arg1), Xen_to_C_GC(arg2), Xen_to_C_Font(arg3))));
}

static Xen gxm_XSetFillStyle(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XSetFillStyle "XSetFillStyle(display, gc, fill_style) sets the fill-style in the specified GC."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XSetFillStyle", "Display*");
  Xen_check_type(Xen_is_GC(arg2), arg2, 2, "XSetFillStyle", "GC");
  Xen_check_type(Xen_is_integer(arg3), arg3, 3, "XSetFillStyle", "int");
  return(C_int_to_Xen_integer(XSetFillStyle(Xen_to_C_Display(arg1), Xen_to_C_GC(arg2), Xen_integer_to_C_int(arg3))));
}

static Xen gxm_XSetFillRule(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XSetFillRule "XSetFillRule(display, gc, fill_rule) sets the fill-rule in the specified GC."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XSetFillRule", "Display*");
  Xen_check_type(Xen_is_GC(arg2), arg2, 2, "XSetFillRule", "GC");
  Xen_check_type(Xen_is_integer(arg3), arg3, 3, "XSetFillRule", "int");
  return(C_int_to_Xen_integer(XSetFillRule(Xen_to_C_Display(arg1), Xen_to_C_GC(arg2), Xen_integer_to_C_int(arg3))));
}

static Xen gxm_XSetDashes(Xen arg1, Xen arg2, Xen arg3, Xen arg4, Xen arg5)
{
  #define H_XSetDashes "XSetDashes(display, gc, dash_offset, dash_list, n) sets the dash-offset and dash-list attributes for dashed line styles \
in the specified GC."
  char *dashes;
  int i, len = 0, val;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XSetDashes", "Display*");
  Xen_check_type(Xen_is_GC(arg2), arg2, 2, "XSetDashes", "GC");
  Xen_check_type(Xen_is_integer(arg3), arg3, 3, "XSetDashes", "int");
  Xen_check_type(Xen_is_list(arg4), arg4, 4, "XSetDashes", "list of ints");
  Xen_check_type(Xen_is_integer_or_unbound(arg5), arg5, 5, "XSetDashes", "optional length of list (int)");
  if (Xen_is_integer(arg5)) len = Xen_integer_to_C_int(arg5);
  if (len <= 0) len = Xen_list_length(arg4);
  dashes = (char *)calloc(len, sizeof(char));
  for (i = 0; i < len; i++) dashes[i] = (char)(Xen_integer_to_C_int(Xen_list_ref(arg4, i)));
  val = XSetDashes(Xen_to_C_Display(arg1), 
		   Xen_to_C_GC(arg2), 
		   Xen_integer_to_C_int(arg3),
		   dashes, len);
  free(dashes);
  return(C_int_to_Xen_integer(val));
}

static Xen gxm_XSetCommand(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XSetCommand "XSetCommand(display, w, argv, argc) sets the command and arguments used to invoke the application."
  /* DIFF: XSetCommand argv is list of strings
   */
  int len, val;
  char **str;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XSetCommand", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XSetCommand", "Window");
  Xen_check_type(Xen_is_list(arg3), arg3, 3, "XSetCommand", "list of char*");
  Xen_check_type(Xen_is_integer(arg4), arg4, 4, "XSetCommand", "int");
  len = Xen_integer_to_C_int(arg4);
  if (len <= 0) return(Xen_false);
  str = Xen_to_C_Strings(arg3, len);
  val = XSetCommand(Xen_to_C_Display(arg1), Xen_to_C_Window(arg2), str, len);
  free(str);
  return(C_int_to_Xen_integer(val));
}

static Xen gxm_XSetCloseDownMode(Xen arg1, Xen arg2)
{
  #define H_XSetCloseDownMode "XSetCloseDownMode(display, close_mode) defines what will happen to the client's resources at connection close."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XSetCloseDownMode", "Display*");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XSetCloseDownMode", "int");
  return(C_int_to_Xen_integer(XSetCloseDownMode(Xen_to_C_Display(arg1), Xen_integer_to_C_int(arg2))));
}

static Xen gxm_XSetClipRectangles(Xen arg1, Xen arg2, Xen arg3, Xen arg4, Xen larg5, Xen arg6, Xen arg7)
{
  #define H_XSetClipRectangles "XSetClipRectangles(display, gc, clip_x_origin, clip_y_origin, rectangles, n, ordering) changes the clip-mask in \
the specified GC to the specified list of rectangles and sets the clip origin."
  /* DIFF: XSetClipRectangles XRectangle* arg (arg 5) is list of XRectangles
   */
  XRectangle *pt;
  int i, len;
  Xen arg5;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XSetClipRectangles", "Display*");
  Xen_check_type(Xen_is_GC(arg2), arg2, 2, "XSetClipRectangles", "GC");
  Xen_check_type(Xen_is_integer(arg3), arg3, 3, "XSetClipRectangles", "int");
  Xen_check_type(Xen_is_integer(arg4), arg4, 4, "XSetClipRectangles", "int");
  Xen_check_type(Xen_is_list(larg5), larg5, 5, "XSetClipRectangles", "list of XRectangles");
  Xen_check_type(Xen_is_integer(arg6), arg6, 6, "XSetClipRectangles", "int");
  Xen_check_type(Xen_is_integer(arg7), arg7, 7, "XSetClipRectangles", "int");
  arg5 = Xen_copy_arg(larg5);
  len = Xen_integer_to_C_int(arg6);
  if (len <= 0) Xen_check_type(0, arg6, 6, "XSetClipRectangles", "positive integer");
  pt = (XRectangle *)calloc(len, sizeof(XRectangle));
  for (i = 0; (i < len) && (!Xen_is_null(arg5)); i++, arg5 = Xen_cdr(arg5))
    {
      XRectangle *pt1;
      pt1 = Xen_to_C_XRectangle(Xen_car(arg5));
      pt[i].x = pt1->x;
      pt[i].y = pt1->y;
      pt[i].width = pt1->width;
      pt[i].height = pt1->height;
    }
  XSetClipRectangles(Xen_to_C_Display(arg1), 
		     Xen_to_C_GC(arg2), 
		     Xen_integer_to_C_int(arg3), 
		     Xen_integer_to_C_int(arg4), 
		     pt, len,
		     Xen_integer_to_C_int(arg7));
  free(pt);
  return(C_int_to_Xen_integer(len));
}

static Xen gxm_XSetClipOrigin(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XSetClipOrigin "XSetClipOrigin(display, gc, clip_x_origin, clip_y_origin) sets the clip origin in the specified GC."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XSetClipOrigin", "Display*");
  Xen_check_type(Xen_is_GC(arg2), arg2, 2, "XSetClipOrigin", "GC");
  Xen_check_type(Xen_is_integer(arg3), arg3, 3, "XSetClipOrigin", "int");
  Xen_check_type(Xen_is_integer(arg4), arg4, 4, "XSetClipOrigin", "int");
  return(C_int_to_Xen_integer(XSetClipOrigin(Xen_to_C_Display(arg1), Xen_to_C_GC(arg2), Xen_integer_to_C_int(arg3), Xen_integer_to_C_int(arg4))));
}

static Xen gxm_XSetClipMask(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XSetClipMask "XSetClipMask(display, gc, pixmap) sets the clip-mask in the specified GC to the specified pixmap."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XSetClipMask", "Display*");
  Xen_check_type(Xen_is_GC(arg2), arg2, 2, "XSetClipMask", "GC");
  Xen_check_type(Xen_is_Pixmap(arg3) || Xen_is_integer(arg3), arg3, 3, "XSetClipMask", "Pixmap or None");
  return(C_int_to_Xen_integer(XSetClipMask(Xen_to_C_Display(arg1), 
				   Xen_to_C_GC(arg2),
				   (Xen_is_Pixmap(arg3)) ? Xen_to_C_Pixmap(arg3) : None)));
}

static Xen gxm_XSetBackground(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XSetBackground "XSetBackground(display, gc, background) sets the background in the specified GC."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XSetBackground", "Display*");
  Xen_check_type(Xen_is_GC(arg2), arg2, 2, "XSetBackground", "GC");
  Xen_check_type(Xen_is_Pixel(arg3), arg3, 3, "XSetBackground", "Pixel");
  return(C_int_to_Xen_integer(XSetBackground(Xen_to_C_Display(arg1), Xen_to_C_GC(arg2), Xen_to_C_Pixel(arg3))));
}

static Xen gxm_XSetArcMode(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XSetArcMode "XSetArcMode(display, gc, arc_mode)"
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XSetArcMode", "Display*");
  Xen_check_type(Xen_is_GC(arg2), arg2, 2, "XSetArcMode", "GC");
  Xen_check_type(Xen_is_integer(arg3), arg3, 3, "XSetArcMode", "int");
  return(C_int_to_Xen_integer(XSetArcMode(Xen_to_C_Display(arg1), Xen_to_C_GC(arg2), Xen_integer_to_C_int(arg3))));
}

static Xen gxm_XSetAccessControl(Xen arg1, Xen arg2)
{
  #define H_XSetAccessControl "XSetAccessControl(display, mode) either enables or disables the use of the access control list at each connection setup."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XSetAccessControl", "Display*");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XSetAccessControl", "int");
  return(C_int_to_Xen_integer(XSetAccessControl(Xen_to_C_Display(arg1), Xen_integer_to_C_int(arg2))));
}

static Xen gxm_XSendEvent(Xen arg1, Xen arg2, Xen arg3, Xen arg4, Xen arg5)
{
  #define H_XSendEvent "Status XSendEvent(display, w, propagate, event_mask, event_send) identifies the destination window, determines which \
clients should receive the specified events, "
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XSendEvent", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XSendEvent", "Window");
  Xen_check_type(Xen_is_boolean(arg3), arg3, 3, "XSendEvent", "Bool");
  Xen_check_type(Xen_is_integer(arg4), arg4, 4, "XSendEvent", "long");
  Xen_check_type(Xen_is_XEvent(arg5), arg5, 5, "XSendEvent", "XEvent*");
  return(C_int_to_Xen_integer(XSendEvent(Xen_to_C_Display(arg1), 
				 Xen_to_C_Window(arg2), 
				 Xen_boolean_to_C_bool(arg3), Xen_integer_to_C_int(arg4), Xen_to_C_XEvent(arg5))));
}

static Xen gxm_XSelectInput(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XSelectInput "XSelectInput(dpy, window, event_mask) selects input events"
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XSelectInput", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XSelectInput", "Window");
  Xen_check_type(Xen_is_integer(arg3), arg3, 3, "XSelectInput", "long");
  return(C_int_to_Xen_integer(XSelectInput(Xen_to_C_Display(arg1), Xen_to_C_Window(arg2), Xen_integer_to_C_int(arg3))));
}

static Xen gxm_XScreenCount(Xen arg1)
{
  #define H_ScreenCount "returns the number of available screens."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XScreenCount", "Display*");
  return(C_int_to_Xen_integer(XScreenCount(Xen_to_C_Display(arg1))));
}

static Xen gxm_XRotateWindowProperties(Xen arg1, Xen arg2, Xen arg3, Xen arg4, Xen arg5)
{
  #define H_XRotateWindowProperties "XRotateWindowProperties(display, w, properties, num_prop, npositions) allows you to rotate properties on a \
window and causes the X server to generate PropertyNotify events."
  /* DIFF: XRotateWindowProperties arg3 is list of atoms
   */
  Atom *outs;
  int val, len;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XRotateWindowProperties", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XRotateWindowProperties", "Window");
  Xen_check_type(Xen_is_list(arg3), arg3, 3, "XRotateWindowProperties", "list of Atom");
  Xen_check_type(Xen_is_integer(arg4), arg4, 4, "XRotateWindowProperties", "int");
  Xen_check_type(Xen_is_integer(arg5), arg5, 5, "XRotateWindowProperties", "int");
  len = Xen_integer_to_C_int(arg4);
  if (len <= 0) return(Xen_false);
  outs = Xen_to_C_Atoms(arg3, len);
  val = XRotateWindowProperties(Xen_to_C_Display(arg1), Xen_to_C_Window(arg2), outs, len, Xen_integer_to_C_int(arg5));
  free(outs);
  return(C_int_to_Xen_integer(val));
}

static Xen gxm_XRotateBuffers(Xen arg1, Xen arg2)
{
  #define H_XRotateBuffers "XRotateBuffers(display, rotate) rotates the cut buffers, such that buffer 0 becomes buffer n, buffer 1 becomes n + 1 \
mod 8, and so on. "
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XRotateBuffers", "Display*");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XRotateBuffers", "int");
  return(C_int_to_Xen_integer(XRotateBuffers(Xen_to_C_Display(arg1), Xen_integer_to_C_int(arg2))));
}

static Xen gxm_XRestackWindows(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XRestackWindows "XRestackWindows(display, windows, nwindows) restacks the windows in the order specified, from top to bottom."
  /* DIFF: XRestackWindows arg2 is list of Windows
   */
  int len, rtn;
  Window *ws;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XRestackWindows", "Display*");
  Xen_check_type(Xen_is_list(arg2), arg2, 2, "XRestackWindows", "list of Windows");
  Xen_check_type(Xen_is_integer(arg3), arg3, 3, "XRestackWindows", "int");
  len = Xen_integer_to_C_int(arg3);
  if (len <= 0) return(Xen_false);
  ws = Xen_to_C_Windows(arg2, len);
  rtn = XRestackWindows(Xen_to_C_Display(arg1), ws, len);
  free(ws);
  return(C_int_to_Xen_integer(rtn));
}

static Xen gxm_XResizeWindow(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XResizeWindow "XResizeWindow(display, w, width, height) changes the inside dimensions of the specified window, not including its borders."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XResizeWindow", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XResizeWindow", "Window");
  Xen_check_type(Xen_is_ulong(arg3), arg3, 3, "XResizeWindow", "unsigned int");
  Xen_check_type(Xen_is_ulong(arg4), arg4, 4, "XResizeWindow", "unsigned int");
  return(C_int_to_Xen_integer(XResizeWindow(Xen_to_C_Display(arg1), Xen_to_C_Window(arg2), Xen_ulong_to_C_ulong(arg3), Xen_ulong_to_C_ulong(arg4))));
}

static Xen gxm_XResetScreenSaver(Xen arg1)
{
  #define H_XResetScreenSaver "XResetScreenSaver(display) resets the screen saver."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XResetScreenSaver", "Display*");
  return(C_int_to_Xen_integer(XResetScreenSaver(Xen_to_C_Display(arg1))));
}

static Xen gxm_XReparentWindow(Xen arg1, Xen arg2, Xen arg3, Xen arg4, Xen arg5)
{
  #define H_XReparentWindow "XReparentWindow(display, w, parent, x, y)"
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XReparentWindow", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XReparentWindow", "Window");
  Xen_check_type(Xen_is_Window(arg3), arg3, 3, "XReparentWindow", "Window");
  Xen_check_type(Xen_is_integer(arg4), arg4, 4, "XReparentWindow", "int");
  Xen_check_type(Xen_is_integer(arg5), arg5, 5, "XReparentWindow", "int");
  return(C_int_to_Xen_integer(XReparentWindow(Xen_to_C_Display(arg1), 
				      Xen_to_C_Window(arg2), 
				      Xen_to_C_Window(arg3), 
				      Xen_integer_to_C_int(arg4), Xen_integer_to_C_int(arg5))));
}

static Xen gxm_XRefreshKeyboardMapping(Xen arg1)
{
  #define H_XRefreshKeyboardMapping "XRefreshKeyboardMapping(event_map) refreshes the stored modifier and keymap information."
  Xen_check_type(Xen_is_XMappingEvent(arg1), arg1, 1, "XRefreshKeyboardMapping", "XMappingEvent*");
  return(C_int_to_Xen_integer(XRefreshKeyboardMapping(Xen_to_C_XMappingEvent(arg1))));
}

static Xen gxm_XRecolorCursor(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XRecolorCursor "XRecolorCursor(display, cursor, foreground_color, background_color) changes the color of the specified cursor, and if \
the cursor is being displayed on a screen, the change is visible immediately."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XRecolorCursor", "Display*");
  Xen_check_type(Xen_is_Cursor(arg2), arg2, 2, "XRecolorCursor", "Cursor");
  Xen_check_type(Xen_is_XColor(arg3), arg3, 3, "XRecolorCursor", "XColor");
  Xen_check_type(Xen_is_XColor(arg4), arg4, 4, "XRecolorCursor", "XColor");
  return(C_int_to_Xen_integer(XRecolorCursor(Xen_to_C_Display(arg1), Xen_to_C_Cursor(arg2), 
				     Xen_to_C_XColor(arg3), Xen_to_C_XColor(arg4))));
}

static Xen gxm_XRebindKeysym(Xen arg1, Xen arg2, Xen arg3, Xen arg4, Xen arg5, Xen arg6)
{
  #define H_XRebindKeysym "XRebindKeysym(display, keysym, list, mod_count, string, num_bytes) can be used to rebind the meaning of a KeySym for the client."
  /* DIFF: XRebindKeysym mod_list is list of keysyms
   */
  KeySym *ks;
  int len, val;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XRebindKeysym", "Display*");
  Xen_check_type(Xen_is_KeySym(arg2), arg2, 2, "XRebindKeysym", "KeySym");
  Xen_check_type(Xen_is_list(arg3), arg3, 3, "XRebindKeysym", "list of KeySym");
  Xen_check_type(Xen_is_integer(arg4), arg4, 4, "XRebindKeysym", "int");
  Xen_check_type(Xen_is_string(arg5), arg5, 5, "XRebindKeysym", "string");
  Xen_check_type(Xen_is_integer(arg6), arg6, 6, "XRebindKeysym", "int");
  len = Xen_integer_to_C_int(arg4);
  if (len <= 0) return(Xen_false);
  ks = Xen_to_C_KeySyms(arg3, len);
  val = XRebindKeysym(Xen_to_C_Display(arg1), 
		      Xen_to_C_KeySym(arg2),
		      ks, len, 
		      (unsigned char *)Xen_string_to_C_string(arg5), Xen_integer_to_C_int(arg6));
  free(ks);
  return(C_int_to_Xen_integer(val));
}

static Xen gxm_XReadBitmapFileData(Xen arg1)
{
  #define H_XReadBitmapFileData "int XReadBitmapFileData(filename) reads in a \
file containing a bitmap, in the same manner as XReadBitmapFile, but returns the data directly rather than creating a pixmap in the server."
  /* DIFF: XReadBitmapFileData omits last 5 args, returns as list
   */
  unsigned int w = 0, h = 0, i, j;
  int x, y;
  unsigned char **str = NULL; /* allocated by X? */
  int val, loc;
  Xen bits = Xen_empty_list;
  Xen_check_type(Xen_is_string(arg1), arg1, 1, "XReadBitmapFileData", "char*");
  val = XReadBitmapFileData((char *)Xen_string_to_C_string(arg1), &w, &h, str, &x, &y);
  loc = xm_protect(bits);
  for (i = 0; i < h; i++)
    for (j = 0; j < w; j++)
      bits = Xen_cons(C_int_to_Xen_integer((int)(str[i][j])), bits);
  if (str) free(str);
  xm_unprotect_at(loc);
  return(Xen_list_6(C_int_to_Xen_integer(val),
		    C_int_to_Xen_integer((int)w),
		    C_int_to_Xen_integer((int)h),
		    bits,
		    C_int_to_Xen_integer((int)x),
		    C_int_to_Xen_integer((int)y)));
}

static Xen gxm_XReadBitmapFile(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XReadBitmapFile "int XReadBitmapFile(display, d, filename) reads in a file containing a bitmap."
  /* DIFF: XReadBitmapFile omits last 5 args, returns as list
   */
  unsigned int w, h;
  int x, y;
  int val;
  Pixmap *p = NULL; /* allocated by X? */
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XReadBitmapFile", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XReadBitmapFile", "Drawable");
  Xen_check_type(Xen_is_string(arg3), arg3, 3, "XReadBitmapFile", "char*");
  val = XReadBitmapFile(Xen_to_C_Display(arg1), Xen_to_C_Window(arg2), (char *)Xen_string_to_C_string(arg3), &w, &h, p, &x, &y);
  return(Xen_list_6(C_int_to_Xen_integer(val),
		    C_int_to_Xen_integer((int)w),
		    C_int_to_Xen_integer((int)h),
		    C_to_Xen_Pixmap(*p),
		    C_int_to_Xen_integer((int)x),
		    C_int_to_Xen_integer((int)y)));
}

static Xen gxm_XRaiseWindow(Xen arg1, Xen arg2)
{
  #define H_XRaiseWindow "XRaiseWindow(display, w)"
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XRaiseWindow", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XRaiseWindow", "Window");
  return(C_int_to_Xen_integer(XRaiseWindow(Xen_to_C_Display(arg1), Xen_to_C_Window(arg2))));
}

static Xen gxm_XQueryTree(Xen arg1, Xen arg2)
{
  #define H_XQueryTree "Status XQueryTree(display, w): returns the root ID, the \
parent window ID, a pointer to the list of children windows and the number of children in the list for the specified window."
  /* DIFF: XQueryTree last 4 arg omit, returns (list val root parent (list children))
   */
  unsigned int arrlen;
  Window *ws;
  Window root, parent;
  int i, val, loc;
  Xen lst = Xen_empty_list;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XQueryTree", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XQueryTree", "Window");
  val = XQueryTree(Xen_to_C_Display(arg1), 
		   Xen_to_C_Window(arg2), 
		   &root, &parent, &ws, &arrlen);
  loc = xm_protect(lst);
  for (i = arrlen - 1; i >= 0; i--)
    lst = Xen_cons(C_to_Xen_Window(ws[i]), lst);
  XFree(ws);
  xm_unprotect_at(loc);
  return(Xen_list_4(C_int_to_Xen_integer(val),
		    C_to_Xen_Window(root),
		    C_to_Xen_Window(parent),
		    lst));
}

static Xen gxm_XQueryTextExtents(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XQueryTextExtents "XQueryTextExtents(display, font, string): returns the bounding box of the specified 8-bit string."
  /* DIFF: XQueryTextExtents omits last 5 args, returns list 
   */
  XCharStruct *c;
  int fa, fd, dr, val;
  char *str;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XQueryTextExtents", "Display*");
  Xen_check_type(Xen_is_Font(arg2), arg2, 2, "XQueryTextExtents", "Font");
  Xen_check_type(Xen_is_string(arg3), arg3, 3, "XQueryTextExtents", "char*");
  str = (char *)Xen_string_to_C_string(arg3);
  c = (XCharStruct *)calloc(1, sizeof(XCharStruct));
  val = XQueryTextExtents(Xen_to_C_Display(arg1), 
			  Xen_to_C_Font(arg2), 
			  str, strlen(str), &dr, &fa, &fd, c); 
  return(Xen_list_5(C_int_to_Xen_integer(val),
		    C_int_to_Xen_integer(dr),
		    C_int_to_Xen_integer(fa),
		    C_int_to_Xen_integer(fd),
		    wrap_for_Xen_obj("XCharStruct", c)));
}

static Xen gxm_XQueryPointer(Xen arg1, Xen arg2)
{
  #define H_XQueryPointer "Bool XQueryPointer(display, w): returns the root window the pointer is logically on and the pointer \
coordinates relative to the root window's origin."
  /* DIFF: XQueryPointer last 7 args omit and rtn
   */
  Window w1, w2;
  int rx, ry, wx, wy;
  unsigned int mask;
  int rtn;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XQueryPointer", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XQueryPointer", "Window");
  rtn = XQueryPointer(Xen_to_C_Display(arg1), 
		      Xen_to_C_Window(arg2), 
		      &w1, &w2, &rx, &ry, &wx, &wy, &mask);
  return(Xen_list_8(C_bool_to_Xen_boolean(rtn),
		    C_to_Xen_Window(w1),
		    C_to_Xen_Window(w2),
		    C_int_to_Xen_integer(rx),
		    C_int_to_Xen_integer(ry),
		    C_int_to_Xen_integer(wx),
		    C_int_to_Xen_integer(wy),
		    C_ulong_to_Xen_ulong((unsigned long)mask)));
}

static Xen gxm_XQueryKeymap(Xen arg1)
{
  #define H_XQueryKeymap "XQueryKeymap(display): returns a bit vector for the logical state of the keyboard, where each bit \
set to 1 indicates that the corresponding key is currently pressed down."
  /* DIFF: XQueryKeymap omits keys -> (list val keys)
   */
  char keys[32];
  int val, i, loc;
  Xen lst = Xen_empty_list;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XQueryKeymap", "Display*");
  val = XQueryKeymap(Xen_to_C_Display(arg1), keys);
  loc = xm_protect(lst);
  for (i = 31; i >= 0; i--)
    lst = Xen_cons(C_int_to_Xen_integer((int)keys[i]), lst);
  xm_unprotect_at(loc);
  return(Xen_cons(C_int_to_Xen_integer(val), lst));
}

static Xen gxm_XQueryExtension(Xen arg1, Xen arg2)
{
  #define H_XQueryExtension "Bool XQueryExtension(dpy, name) gets extension version information"
  /* DIFF: XQueryExtension dpy name [op er er] -> (list val op er er)
   */
  int op, err1, err2, val;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XQueryExtension", "Display*");
  Xen_check_type(Xen_is_string(arg2), arg2, 2, "XQueryExtension", "char*");
  val = XQueryExtension(Xen_to_C_Display(arg1), (char *)Xen_string_to_C_string(arg2), &op, &err1, &err2);
  return(Xen_list_4(C_bool_to_Xen_boolean(val),
		    C_int_to_Xen_integer(op),
		    C_int_to_Xen_integer(err1),
		    C_int_to_Xen_integer(err2)));
}

static Xen gxm_XQueryColors(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XQueryColors "XQueryColors(display, colormap, defs_in_out, ncolors)"
  int i, len, rtn;
  Xen lst;
  XColor *cols;
  XColor *col;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XQueryColors", "Display*");
  Xen_check_type(Xen_is_Colormap(arg2), arg2, 2, "XQueryColors", "Colormap");
  Xen_check_type(Xen_is_list(arg3), arg3, 3, "XQueryColors", "XColor list");
  Xen_check_type(Xen_is_integer_or_unbound(arg4), arg4, 4, "XQueryColors", "int");
  if (Xen_is_integer(arg4)) len = Xen_integer_to_C_int(arg4); else len = Xen_list_length(arg3);
  lst = Xen_copy_arg(arg3);
  cols = (XColor *)calloc(len, sizeof(XColor));
  for (i = 0; (i < len) && (!Xen_is_null(lst)); i++, lst = Xen_cdr(lst))
    {
      if (Xen_is_XColor(Xen_car(lst)))
	{
	  col = Xen_to_C_XColor(Xen_car(lst));
	  cols[i].pixel = col->pixel;
	}
      else Xen_check_type(0, Xen_car(lst), i, __func__, "an XColor");
    }
  rtn = XQueryColors(Xen_to_C_Display(arg1), Xen_to_C_Colormap(arg2), cols, len);
  for (i = 0; i < len; i++)
    {
      col = Xen_to_C_XColor(Xen_list_ref(arg3, i));
      col->red = cols[i].red;
      col->green = cols[i].green;
      col->blue = cols[i].blue;
      col->flags = cols[i].flags;
    }
  free(cols);
  return(C_int_to_Xen_integer(rtn));
}

static Xen gxm_XQueryColor(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XQueryColor "XQueryColor(display, colormap, def_in_out): returns the current RGB value for the pixel in the XColor structure \
and sets the DoRed, DoGreen, DoBlue flags."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XQueryColor", "Display*");
  Xen_check_type(Xen_is_Colormap(arg2), arg2, 2, "XQueryColor", "Colormap");
  Xen_check_type(Xen_is_XColor(arg3), arg3, 3, "XQueryColor", "XColor*");
  return(C_int_to_Xen_integer(XQueryColor(Xen_to_C_Display(arg1), Xen_to_C_Colormap(arg2), Xen_to_C_XColor(arg3))));
}

static Xen gxm_XQueryBestTile(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XQueryBestTile "Status XQueryBestTile(display, which_screen, width, height)  returns the best or \
closest size, that is, the size that can be tiled fastest on the screen specified by which_screen."
  /* DIFF: XQueryBestTile dpy scr wid hgt [wd hg] -> (list status wd hg)
   */
  unsigned int w, h;
  int val;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XQueryBestTile", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XQueryBestTile", "Drawable");
  Xen_check_type(Xen_is_ulong(arg3), arg3, 3, "XQueryBestTile", "unsigned int");
  Xen_check_type(Xen_is_ulong(arg4), arg4, 4, "XQueryBestTile", "unsigned int");
  val = XQueryBestTile(Xen_to_C_Display(arg1), Xen_to_C_Window(arg2), 
		       Xen_ulong_to_C_ulong(arg3), Xen_ulong_to_C_ulong(arg4), &w, &h);
  return(Xen_list_3(C_int_to_Xen_integer(val),
		    C_ulong_to_Xen_ulong(w),
		    C_ulong_to_Xen_ulong(h)));
}

static Xen gxm_XQueryBestStipple(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XQueryBestStipple "Status XQueryBestStipple(display, which_screen, width, height)  returns the \
best or closest size, that is, the size that can be stippled fastest on the screen specified by which_screen."
  /* DIFF: XQueryBestStipple dpy scr wid hgt [wd hg] -> (list status wd hg)
   */
  unsigned int w, h;
  int val;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XQueryBestStipple", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XQueryBestStipple", "Drawable");
  Xen_check_type(Xen_is_ulong(arg3), arg3, 3, "XQueryBestStipple", "unsigned int");
  Xen_check_type(Xen_is_ulong(arg4), arg4, 4, "XQueryBestStipple", "unsigned int");
  val = XQueryBestStipple(Xen_to_C_Display(arg1), Xen_to_C_Window(arg2), 
			  Xen_ulong_to_C_ulong(arg3), Xen_ulong_to_C_ulong(arg4), 
			  &w, &h);
  return(Xen_list_3(C_int_to_Xen_integer(val),
		    C_ulong_to_Xen_ulong(w),
		    C_ulong_to_Xen_ulong(h)));
}

static Xen gxm_XQueryBestSize(Xen arg1, Xen arg2, Xen arg3, Xen arg4, Xen arg5)
{
  #define H_XQueryBestSize "Status XQueryBestSize(display, class, which_screen, width, height)  returns the \
best or closest size to the specified size."
  /* DIFF: XQueryBestSize dpy cls scr wid hgt [wd hg] -> (list status wd hg)
   */
  unsigned int w, h;
  int val;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XQueryBestSize", "Display*");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XQueryBestSize", "int");
  Xen_check_type(Xen_is_Window(arg3), arg3, 3, "XQueryBestSize", "Drawable");
  Xen_check_type(Xen_is_ulong(arg4), arg4, 4, "XQueryBestSize", "unsigned int");
  Xen_check_type(Xen_is_ulong(arg5), arg5, 5, "XQueryBestSize", "unsigned int");
  val = XQueryBestSize(Xen_to_C_Display(arg1), Xen_integer_to_C_int(arg2), Xen_to_C_Window(arg3), 
		       Xen_ulong_to_C_ulong(arg4), Xen_ulong_to_C_ulong(arg5), 
		       &w, &h);
  return(Xen_list_3(C_int_to_Xen_integer(val),
		    C_ulong_to_Xen_ulong(w),
		    C_ulong_to_Xen_ulong(h)));
}

static Xen gxm_XQueryBestCursor(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XQueryBestCursor "Status XQueryBestCursor(display, d, width, height) provides a way to find \
out what size cursors are actually possible on the display."
  /* DIFF: XQueryBestCursor dpy d wid hgt [wd hg] -> (list status wd hg)
   */
  unsigned int w, h;
  int val;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XQueryBestCursor", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XQueryBestCursor", "Drawable");
  Xen_check_type(Xen_is_ulong(arg3), arg3, 3, "XQueryBestCursor", "unsigned int");
  Xen_check_type(Xen_is_ulong(arg4), arg4, 4, "XQueryBestCursor", "unsigned int");
  val = XQueryBestCursor(Xen_to_C_Display(arg1), Xen_to_C_Window(arg2), 
			 Xen_ulong_to_C_ulong(arg3), Xen_ulong_to_C_ulong(arg4), 
			 &w, &h);
  return(Xen_list_3(C_int_to_Xen_integer(val),
		    C_ulong_to_Xen_ulong(w),
		    C_ulong_to_Xen_ulong(h)));
}

static Xen gxm_XQLength(Xen arg1)
{
  #define H_QLength "returns the length of the event queue for the connected display."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XQLength", "Display*");
  return(C_int_to_Xen_integer(XQLength(Xen_to_C_Display(arg1))));
}

static Xen gxm_XPutImage(Xen args)
{
  #define H_XPutImage "XPutImage(display, d, gc, image, src_x, src_y, dest_x, dest_y, width, height) combines an image with a rectangle \
of the specified drawable."
  Xen arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8, arg9, arg10;
  arg1 = Xen_list_ref(args, 0);
  arg2 = Xen_list_ref(args, 1);
  arg3 = Xen_list_ref(args, 2);
  arg4 = Xen_list_ref(args, 3);
  arg5 = Xen_list_ref(args, 4);
  arg6 = Xen_list_ref(args, 5);
  arg7 = Xen_list_ref(args, 6);
  arg8 = Xen_list_ref(args, 7);
  arg9 = Xen_list_ref(args, 8);
  arg10 = Xen_list_ref(args, 9);
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XPutImage", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XPutImage", "Drawable");
  Xen_check_type(Xen_is_GC(arg3), arg3, 3, "XPutImage", "GC");
  Xen_check_type(Xen_is_XImage(arg4), arg4, 4, "XPutImage", "XImage*");
  Xen_check_type(Xen_is_integer(arg5), arg5, 5, "XPutImage", "int");
  Xen_check_type(Xen_is_integer(arg6), arg6, 6, "XPutImage", "int");
  Xen_check_type(Xen_is_integer(arg7), arg7, 7, "XPutImage", "int");
  Xen_check_type(Xen_is_integer(arg8), arg8, 8, "XPutImage", "int");
  Xen_check_type(Xen_is_ulong(arg9), arg9, 9, "XPutImage", "unsigned int");
  Xen_check_type(Xen_is_ulong(arg10), arg10, 10, "XPutImage", "unsigned int");
  return(C_int_to_Xen_integer(XPutImage(Xen_to_C_Display(arg1), Xen_to_C_Window(arg2), 
				Xen_to_C_GC(arg3), Xen_to_C_XImage(arg4), Xen_integer_to_C_int(arg5), Xen_integer_to_C_int(arg6), 
				Xen_integer_to_C_int(arg7), Xen_integer_to_C_int(arg8), Xen_ulong_to_C_ulong(arg9), Xen_ulong_to_C_ulong(arg10))));
}

static Xen gxm_XPutBackEvent(Xen arg1, Xen arg2)
{
  #define H_XPutBackEvent "XPutBackEvent(display, event) pushes an event back onto the head of the display's event queue by copying the event into the queue."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XPutBackEvent", "Display*");
  Xen_check_type(Xen_is_XEvent(arg2), arg2, 2, "XPutBackEvent", "XEvent*");
  return(C_int_to_Xen_integer(XPutBackEvent(Xen_to_C_Display(arg1), Xen_to_C_XEvent(arg2))));
}

static Xen gxm_XProtocolVersion(Xen arg1)
{
  #define H_ProtocolVersion "ProtocolVersion(display): returns the major version number (11) of the X protocol associated with the connected display."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XProtocolVersion", "Display*");
  return(C_int_to_Xen_integer(XProtocolVersion(Xen_to_C_Display(arg1))));
}

static Xen gxm_XProtocolRevision(Xen arg1)
{
  #define H_ProtocolRevision "ProtocolRevision(display): returns the minor protocol revision number of the X server."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XProtocolRevision", "Display*");
  return(C_int_to_Xen_integer(XProtocolRevision(Xen_to_C_Display(arg1))));
}

static Xen gxm_XPlanesOfScreen(Xen arg1)
{
  #define H_PlanesOfScreen "PlanesOfScreen(screen): returns the number of planes in the root window of the specified screen."
  Xen_check_type(Xen_is_Screen(arg1), arg1, 1, "XPlanesOfScreen", "Screen*");
  return(C_int_to_Xen_integer(XPlanesOfScreen(Xen_to_C_Screen(arg1))));
}

static Xen gxm_XPending(Xen arg1)
{
  #define H_XPending "int XPending(display): returns the number of events that have been received from the X server but have not been removed \
from the event queue."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XPending", "Display*");
  return(C_int_to_Xen_integer(XPending(Xen_to_C_Display(arg1))));
}

static Xen xm_XPeekIfEventProc;

static Bool gxm_XPeekIfEventProc(Display *dpy, XEvent *e, XtPointer p)
{
  return(Xen_boolean_to_C_bool(Xen_call_with_3_args(xm_XPeekIfEventProc, C_to_Xen_Display(dpy), C_to_Xen_XEvent(e), (Xen)(p), __func__)));
}

static Xen gxm_XPeekIfEvent(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XPeekIfEvent "XPeekIfEvent(display, predicate, arg): returns only when the specified predicate procedure returns " PROC_TRUE " for an event."
  /* DIFF: XPeekIfEvent dpy [evrtn] proc ptr -> (list val evrtn)
   */
  XEvent *e;
  int val;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XPeekIfEvent", "Display*");
  Xen_check_type(Xen_is_procedure(arg2) && (Xen_is_aritable(arg2, 3)), arg2, 2, "XPeekIfEvent", "(Bool_Proc dpy ev data)");
  e = (XEvent *)calloc(1, sizeof(XEvent));
  val = XPeekIfEvent(Xen_to_C_Display(arg1), 
		     e, 
		     (Bool (*)(Display *d, XEvent *ev, char *p))gxm_XPeekIfEventProc, /* C++ insists on the cast */
		     (XPointer)arg3);
  return(Xen_list_2(C_int_to_Xen_integer(val), C_to_Xen_XEvent_OBJ(e)));
}

static Xen gxm_XPeekEvent(Xen arg1)
{
  #define H_XPeekEvent "XPeekEvent(display): returns the first event from the event queue, but it does not remove the event from the queue."
  /* DIFF: XPeekEvent dpy [ev] -> (list int event)
   */
  XEvent *e;
  int val;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XPeekEvent", "Display*");
  e = (XEvent *)calloc(1, sizeof(XEvent));
  val = XPeekEvent(Xen_to_C_Display(arg1), e);
  return(Xen_list_2(C_int_to_Xen_integer(val), C_to_Xen_XEvent_OBJ(e)));
}

static Xen gxm_XParseGeometry(Xen arg1)
{
  #define H_XParseGeometry "int XParseGeometry(parsestring): returns a bitmask that \
indicates which of the four values (width, height, xoffset, and yoffset) were actually found in the string and whether the x and y values are negative. "
  /* DIFF: XParseGeometry str [x y w h] -> (list bit x y w h)
   */
  int x = 0, y = 0, val = 0;
  unsigned int w = 0, h = 0;
  Xen_check_type(Xen_is_string(arg1), arg1, 1, "XParseGeometry", "char*");
  val = XParseGeometry((char *)Xen_string_to_C_string(arg1), &x, &y, &w, &h);
  return(Xen_list_5(C_int_to_Xen_integer(val),
		    C_int_to_Xen_integer(x),
		    C_int_to_Xen_integer(y),
		    C_int_to_Xen_integer(w),
		    C_int_to_Xen_integer(h)));
}

static Xen gxm_XParseColor(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XParseColor "Status XParseColor(display, colormap, spec, exact_def_return) looks up the string name of a color with respect \
to the screen associated with the specified colormap."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XParseColor", "Display*");
  Xen_check_type(Xen_is_Colormap(arg2), arg2, 2, "XParseColor", "Colormap");
  Xen_check_type(Xen_is_string(arg3), arg3, 3, "XParseColor", "char*");
  Xen_check_type(Xen_is_XColor(arg4), arg4, 4, "XParseColor", "XColor");
  return(C_int_to_Xen_integer(XParseColor(Xen_to_C_Display(arg1), Xen_to_C_Colormap(arg2), (char *)Xen_string_to_C_string(arg3), Xen_to_C_XColor(arg4))));
}

static Xen gxm_XNoOp(Xen arg1)
{
  #define H_XNoOp "XNoOp(dpy) sends the server a no-op for exercise"
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XNoOp", "Display*");
  return(C_int_to_Xen_integer(XNoOp(Xen_to_C_Display(arg1))));
}

static Xen gxm_XNextEvent(Xen arg1)
{
  #define H_XNextEvent "XNextEvent(display) copies the first event from the event queue into the specified XEvent structure \
and then removes it from the queue."
  /* DIFF: XNextEvent dpy [ev] -> (list int event)
   */
  XEvent *e;
  int val;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XNextEvent", "Display*");
  e = (XEvent *)calloc(1, sizeof(XEvent));
  val = XNextEvent(Xen_to_C_Display(arg1), e);
  return(Xen_list_2(C_int_to_Xen_integer(val), C_to_Xen_XEvent_OBJ(e)));
}

static Xen gxm_XMoveWindow(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XMoveWindow "XMoveWindow(display, w, x, y) moves the specified window to the specified x and y coordinates, but it does not \
change the window's size, raise the window, or change the mapping state of the window."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XMoveWindow", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XMoveWindow", "Window");
  Xen_check_type(Xen_is_integer(arg3), arg3, 3, "XMoveWindow", "int");
  Xen_check_type(Xen_is_integer(arg4), arg4, 4, "XMoveWindow", "int");
  return(C_int_to_Xen_integer(XMoveWindow(Xen_to_C_Display(arg1), Xen_to_C_Window(arg2), Xen_integer_to_C_int(arg3), Xen_integer_to_C_int(arg4))));
}

static Xen gxm_XMoveResizeWindow(Xen arg1, Xen arg2, Xen arg3, Xen arg4, Xen arg5, Xen arg6)
{
  #define H_XMoveResizeWindow "XMoveResizeWindow(display, w, x, y, width, height) changes the size and location of the specified window without raising it."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XMoveResizeWindow", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XMoveResizeWindow", "Window");
  Xen_check_type(Xen_is_integer(arg3), arg3, 3, "XMoveResizeWindow", "int");
  Xen_check_type(Xen_is_integer(arg4), arg4, 4, "XMoveResizeWindow", "int");
  Xen_check_type(Xen_is_ulong(arg5), arg5, 5, "XMoveResizeWindow", "unsigned int");
  Xen_check_type(Xen_is_ulong(arg6), arg6, 6, "XMoveResizeWindow", "unsigned int");
  return(C_int_to_Xen_integer(XMoveResizeWindow(Xen_to_C_Display(arg1), 
					Xen_to_C_Window(arg2), 
					Xen_integer_to_C_int(arg3), Xen_integer_to_C_int(arg4), 
					Xen_ulong_to_C_ulong(arg5), Xen_ulong_to_C_ulong(arg6))));
}

static Xen gxm_XMinCmapsOfScreen(Xen arg1)
{
  #define H_MinCmapsOfScreen "returns the minimum number of installed colormaps supported by the specified screen."
  Xen_check_type(Xen_is_Screen(arg1), arg1, 1, "XMinCmapsOfScreen", "Screen*");
  return(C_int_to_Xen_integer(XMinCmapsOfScreen(Xen_to_C_Screen(arg1))));
}

static Xen gxm_XMaxCmapsOfScreen(Xen arg1)
{
  #define H_MaxCmapsOfScreen "returns the maximum number of installed colormaps supported by the specified screen."
  Xen_check_type(Xen_is_Screen(arg1), arg1, 1, "XMaxCmapsOfScreen", "Screen*");
  return(C_int_to_Xen_integer(XMaxCmapsOfScreen(Xen_to_C_Screen(arg1))));
}

static Xen gxm_XMaskEvent(Xen arg1, Xen arg2)
{
  #define H_XMaskEvent "XMaskEvent(display, event_mask) searches the event queue for the events associated with the specified mask."
  /* DIFF: XMaskEvent dpy mask [ev] -> (list val ev)
   */
  XEvent *e;
  int val;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XMaskEvent", "Display*");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XMaskEvent", "long");
  e = (XEvent *)calloc(1, sizeof(XEvent));
  val = XMaskEvent(Xen_to_C_Display(arg1), Xen_integer_to_C_int(arg2), e);
  return(Xen_list_2(C_int_to_Xen_integer(val), C_to_Xen_XEvent_OBJ(e)));
}

static Xen gxm_XMapWindow(Xen arg1, Xen arg2)
{
  #define H_XMapWindow "XMapWindow(display, w) maps the window and all of its subwindows that have had map requests."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XMapWindow", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XMapWindow", "Window");
  return(C_int_to_Xen_integer(XMapWindow(Xen_to_C_Display(arg1), Xen_to_C_Window(arg2))));
}

static Xen gxm_XMapSubwindows(Xen arg1, Xen arg2)
{
  #define H_XMapSubwindows "XMapSubwindows(display, w) maps all subwindows for a specified window in top-to-bottom stacking order."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XMapSubwindows", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XMapSubwindows", "Window");
  return(C_int_to_Xen_integer(XMapSubwindows(Xen_to_C_Display(arg1), Xen_to_C_Window(arg2))));
}

static Xen gxm_XMapRaised(Xen arg1, Xen arg2)
{
  #define H_XMapRaised "XMapRaised(display, w) maps the window and all of its subwindows that have had map requests, and raises the \
specified window to the top of the stack."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XMapRaised", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XMapRaised", "Window");
  return(C_int_to_Xen_integer(XMapRaised(Xen_to_C_Display(arg1), Xen_to_C_Window(arg2))));
}

static Xen gxm_XLowerWindow(Xen arg1, Xen arg2)
{
  #define H_XLowerWindow "XLowerWindow(display, w) lowers the specified window to the bottom of the stack so that it does not obscure any sibling windows."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XLowerWindow", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XLowerWindow", "Window");
  return(C_int_to_Xen_integer(XLowerWindow(Xen_to_C_Display(arg1), Xen_to_C_Window(arg2))));
}

static Xen gxm_XLookupColor(Xen arg1, Xen arg2, Xen arg3, Xen arg4, Xen arg5)
{
  #define H_XLookupColor "Status XLookupColor(display, colormap, color_name, exact_def_return, screen_def_return) looks up the string name \
of a color with respect to the screen associated with the specified colormap."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XLookupColor", "Display*");
  Xen_check_type(Xen_is_Colormap(arg2), arg2, 2, "XLookupColor", "Colormap");
  Xen_check_type(Xen_is_string(arg3), arg3, 3, "XLookupColor", "char*");
  Xen_check_type(Xen_is_XColor(arg4), arg4, 4, "XLookupColor", "XColor*");
  Xen_check_type(Xen_is_XColor(arg5), arg5, 5, "XLookupColor", "XColor*");
  return(C_int_to_Xen_integer(XLookupColor(Xen_to_C_Display(arg1), 
				   Xen_to_C_Colormap(arg2), 
				   (char *)Xen_string_to_C_string(arg3), 
				   Xen_to_C_XColor(arg4), Xen_to_C_XColor(arg5))));
}

static Xen gxm_XKillClient(Xen arg1, Xen arg2)
{
  #define H_XKillClient "XKillClient(display, resource) forces a close-down of the client that created the resource"
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XKillClient", "Display*");
  Xen_check_type(Xen_is_ulong(arg2), arg2, 2, "XKillClient", "XID");
  return(C_int_to_Xen_integer(XKillClient(Xen_to_C_Display(arg1), Xen_ulong_to_C_ulong(arg2))));
}

static Xen gxm_XKeysymToKeycode(Xen arg1, Xen arg2)
{
  #define H_XKeysymToKeycode "KeyCode XKeysymToKeycode(display, keysym)"
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XKeysymToKeycode", "Display*");
  Xen_check_type(Xen_is_KeySym(arg2), arg2, 2, "XKeysymToKeycode", "KeySym");
  return(C_to_Xen_KeyCode(XKeysymToKeycode(Xen_to_C_Display(arg1), Xen_to_C_KeySym(arg2))));
}

static Xen gxm_XInstallColormap(Xen arg1, Xen arg2)
{
  #define H_XInstallColormap "XInstallColormap(display, colormap) installs the specified colormap for its associated screen."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XInstallColormap", "Display*");
  Xen_check_type(Xen_is_Colormap(arg2), arg2, 2, "XInstallColormap", "Colormap");
  return(C_int_to_Xen_integer(XInstallColormap(Xen_to_C_Display(arg1), Xen_to_C_Colormap(arg2))));
}

static Xen gxm_XImageByteOrder(Xen arg1)
{
  #define H_ImageByteOrder "specifies the required byte order for images for each scanline unit in XY format (bitmap) or for each pixel value in Z format."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XImageByteOrder", "Display*");
  return(C_int_to_Xen_integer(XImageByteOrder(Xen_to_C_Display(arg1))));
}

static Xen gxm_XIfEvent(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XIfEvent "XIfEvent(display, predicate, arg) completes only when the specified predicate procedure returns " PROC_TRUE " for \
an event, which indicates an event in the queue matches."
  /* DIFF: XIfEvent dpy [ev] proc ptr -> (list val ev)
   */
  XEvent *e;
  int val;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XIfEvent", "Display*");
  Xen_check_type(Xen_is_procedure(arg2) && (Xen_is_aritable(arg2, 3)), arg2, 2, "XIfEvent", "(Bool_Proc dpy ev data)");
  e = (XEvent *)calloc(1, sizeof(XEvent));
  val = XIfEvent(Xen_to_C_Display(arg1), 
		 e, 
		 (Bool (*)(Display *d1, XEvent *e1, char *p1))gxm_XPeekIfEventProc, 
		 (char *)arg3);
  return(Xen_list_2(C_int_to_Xen_integer(val), C_to_Xen_XEvent_OBJ(e)));
}

static Xen gxm_XHeightOfScreen(Xen arg1)
{
  #define H_HeightOfScreen "XHeightOfScreen(screen): returns the height of the specified screen."
  Xen_check_type(Xen_is_Screen(arg1), arg1, 1, "XHeightOfScreen", "Screen*");
  return(C_int_to_Xen_integer(XHeightOfScreen(Xen_to_C_Screen(arg1))));
}

static Xen gxm_XHeightMMOfScreen(Xen arg1)
{
  #define H_HeightMMOfScreen "XHeightMMOfScreen(screen): returns the height of the specified screen in millimeters."
  Xen_check_type(Xen_is_Screen(arg1), arg1, 1, "XHeightMMOfScreen", "Screen*");
  return(C_int_to_Xen_integer(XHeightMMOfScreen(Xen_to_C_Screen(arg1))));
}

static Xen gxm_XGrabServer(Xen arg1)
{
  #define H_XGrabServer "XGrabServer(display) disables processing of requests and close downs on all other connections than the one this request arrived on."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XGrabServer", "Display*");
  return(C_int_to_Xen_integer(XGrabServer(Xen_to_C_Display(arg1))));
}

static Xen gxm_XGrabPointer(Xen arg1, Xen arg2, Xen arg3, Xen arg4, Xen arg5, Xen arg6, Xen arg7, Xen arg8, Xen arg9)
{
  #define H_XGrabPointer "int XGrabPointer(display, grab_window, owner_events, event_mask, pointer_mode, keyboard_mode, confine_to, cursor, time) \
actively grabs control of the pointer and returns GrabSuccess if the grab was successful."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XGrabPointer", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XGrabPointer", "Window");
  Xen_check_type(Xen_is_boolean(arg3), arg3, 3, "XGrabPointer", "Bool");
  Xen_check_type(Xen_is_ulong(arg4), arg4, 4, "XGrabPointer", "unsigned int");
  Xen_check_type(Xen_is_integer(arg5), arg5, 5, "XGrabPointer", "int");
  Xen_check_type(Xen_is_integer(arg6), arg6, 6, "XGrabPointer", "int");
  Xen_check_type(Xen_is_Window(arg7), arg7, 7, "XGrabPointer", "Window");
  Xen_check_type(Xen_is_Cursor(arg8), arg8, 8, "XGrabPointer", "Cursor");
  Xen_check_type(Xen_is_Time(arg9), arg9, 9, "XGrabPointer", "Time");
  return(C_int_to_Xen_integer(XGrabPointer(Xen_to_C_Display(arg1), 
				   Xen_to_C_Window(arg2), 
				   Xen_boolean_to_C_bool(arg3), 
				   Xen_ulong_to_C_ulong(arg4), Xen_integer_to_C_int(arg5), Xen_integer_to_C_int(arg6), 
				   Xen_to_C_Window(arg7), Xen_to_C_Cursor(arg8), 
				   Xen_to_C_Time(arg9))));
}

static Xen gxm_XGrabKeyboard(Xen arg1, Xen arg2, Xen arg3, Xen arg4, Xen arg5, Xen arg6)
{
  #define H_XGrabKeyboard "int XGrabKeyboard(display, grab_window, owner_events, pointer_mode, keyboard_mode, time) actively grabs control of \
the keyboard and generates FocusIn and FocusOut events."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XGrabKeyboard", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XGrabKeyboard", "Window");
  Xen_check_type(Xen_is_boolean(arg3), arg3, 3, "XGrabKeyboard", "Bool");
  Xen_check_type(Xen_is_integer(arg4), arg4, 4, "XGrabKeyboard", "int");
  Xen_check_type(Xen_is_integer(arg5), arg5, 5, "XGrabKeyboard", "int");
  Xen_check_type(Xen_is_Time(arg6), arg6, 6, "XGrabKeyboard", "Time");
  return(C_int_to_Xen_integer(XGrabKeyboard(Xen_to_C_Display(arg1), 
				    Xen_to_C_Window(arg2), 
				    Xen_boolean_to_C_bool(arg3), Xen_integer_to_C_int(arg4), Xen_integer_to_C_int(arg5), 
				    Xen_to_C_Time(arg6))));
}

static Xen gxm_XGrabKey(Xen arg1, Xen arg2, Xen arg3, Xen arg4, Xen arg5, Xen arg6, Xen arg7)
{
  #define H_XGrabKey "XGrabKey(display, keycode, modifiers, grab_window, owner_events, pointer_mode, keyboard_mode) establishes a passive \
grab on the keyboard."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XGrabKey", "Display*");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XGrabKey", "int");
  Xen_check_type(Xen_is_ulong(arg3), arg3, 3, "XGrabKey", "unsigned int");
  Xen_check_type(Xen_is_Window(arg4), arg4, 4, "XGrabKey", "Window");
  Xen_check_type(Xen_is_boolean(arg5), arg5, 5, "XGrabKey", "Bool");
  Xen_check_type(Xen_is_integer(arg6), arg6, 6, "XGrabKey", "int");
  Xen_check_type(Xen_is_integer(arg7), arg7, 7, "XGrabKey", "int");
  return(C_int_to_Xen_integer(XGrabKey(Xen_to_C_Display(arg1), 
			       Xen_integer_to_C_int(arg2), Xen_ulong_to_C_ulong(arg3), 
			       Xen_to_C_Window(arg4), 
			       Xen_boolean_to_C_bool(arg5), Xen_integer_to_C_int(arg6), Xen_integer_to_C_int(arg7))));
}

static Xen gxm_XGrabButton(Xen args)
{
  #define H_XGrabButton "XGrabButton(display, button, modifiers, grab_window, owner_events, event_mask, pointer_mode, keyboard_mode, confine_to, cursor) \
establishes a passive grab."
  Xen arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8, arg9, arg10;
  arg1 = Xen_list_ref(args, 0);
  arg2 = Xen_list_ref(args, 1);
  arg3 = Xen_list_ref(args, 2);
  arg4 = Xen_list_ref(args, 3);
  arg5 = Xen_list_ref(args, 4);
  arg6 = Xen_list_ref(args, 5);
  arg7 = Xen_list_ref(args, 6);
  arg8 = Xen_list_ref(args, 7);
  arg9 = Xen_list_ref(args, 8);
  arg10 = Xen_list_ref(args, 9);
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XGrabButton", "Display*");
  Xen_check_type(Xen_is_ulong(arg2), arg2, 2, "XGrabButton", "unsigned int");
  Xen_check_type(Xen_is_ulong(arg3), arg3, 3, "XGrabButton", "unsigned int");
  Xen_check_type(Xen_is_Window(arg4), arg4, 4, "XGrabButton", "Window");
  Xen_check_type(Xen_is_boolean(arg5), arg5, 5, "XGrabButton", "Bool");
  Xen_check_type(Xen_is_ulong(arg6), arg6, 6, "XGrabButton", "unsigned int");
  Xen_check_type(Xen_is_integer(arg7), arg7, 7, "XGrabButton", "int");
  Xen_check_type(Xen_is_integer(arg8), arg8, 8, "XGrabButton", "int");
  Xen_check_type(Xen_is_Window(arg9), arg9, 9, "XGrabButton", "Window");
  Xen_check_type(Xen_is_Cursor(arg10), arg10, 10, "XGrabButton", "Cursor");
  return(C_int_to_Xen_integer(XGrabButton(Xen_to_C_Display(arg1), 
				  Xen_ulong_to_C_ulong(arg2), Xen_ulong_to_C_ulong(arg3), 
				  Xen_to_C_Window(arg4), Xen_boolean_to_C_bool(arg5), 
				  Xen_ulong_to_C_ulong(arg6), Xen_integer_to_C_int(arg7), 
				  Xen_integer_to_C_int(arg8), 
				  Xen_to_C_Window(arg9), Xen_to_C_Cursor(arg10))));
}

static Xen gxm_XGetWindowAttributes(Xen arg1, Xen arg2)
{ 
  #define H_XGetWindowAttributes "Status XGetWindowAttributes(display, w): returns the current attributes for the \
specified window to an XWindowAttributes structure."
  /* DIFF: XGetWindowAttributes omits and rtns arg3 
   */
  XWindowAttributes *w;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XGetWindowAttributes", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XGetWindowAttributes", "Window");
  w = (XWindowAttributes *)calloc(1, sizeof(XWindowAttributes));
  XGetWindowAttributes(Xen_to_C_Display(arg1), Xen_to_C_Window(arg2), w);
  return(C_to_Xen_XWindowAttributes(w));
}

static Xen gxm_XGetWindowProperty(Xen args)
{
  #define H_XGetWindowProperty "int XGetWindowProperty(display, w, property, long_offset, long_length, delete, req_type) \
returns the actual type of the property; the actual format of the property; the \
number of 8-bit, 16-bit, or 32-bit items transferred; the number of bytes remaining to be read in the property; and a pointer to the data \
actually returned."
  /* DIFF: XGetWindowProperty omit trailing 5 args, rtn as list
   */
  Xen arg1, arg2, arg3, arg4, arg5, arg6, arg7; 
  Xen result = Xen_false;
  Atom a;
  int ret, val;
  unsigned long len = 0, bytes;
  unsigned char *data[1];
  arg1 = Xen_list_ref(args, 0);
  arg2 = Xen_list_ref(args, 1);
  arg3 = Xen_list_ref(args, 2);
  arg4 = Xen_list_ref(args, 3);
  arg5 = Xen_list_ref(args, 4);
  arg6 = Xen_list_ref(args, 5);
  arg7 = Xen_list_ref(args, 6);
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XGetWindowProperty", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XGetWindowProperty", "Window");
  Xen_check_type(Xen_is_Atom(arg3), arg3, 3, "XGetWindowProperty", "Atom");
  Xen_check_type(Xen_is_integer(arg4), arg4, 4, "XGetWindowProperty", "long");
  Xen_check_type(Xen_is_integer(arg5), arg5, 5, "XGetWindowProperty", "long");
  Xen_check_type(Xen_is_boolean(arg6), arg6, 6, "XGetWindowProperty", "Bool");
  Xen_check_type(Xen_is_Atom(arg7), arg7, 7, "XGetWindowProperty", "Atom");
  val = XGetWindowProperty(Xen_to_C_Display(arg1), 
			   Xen_to_C_Window(arg2), 
			   Xen_to_C_Atom(arg3), 
			   Xen_integer_to_C_int(arg4), 
			   Xen_integer_to_C_int(arg5), 
			   Xen_boolean_to_C_bool(arg6), 
			   Xen_to_C_Atom(arg7), 
			   &a, &ret, &len, &bytes, (unsigned char **)data);
  if ((a != (Atom)None) && (len > 0))
    {
      if (a == XA_STRING)
	result = C_string_to_Xen_string((char *)data[0]);
      else result = C_string_to_Xen_string_with_length((char *)data[0], len * ret / 8); /* is this a good idea? -- perhaps a void pointer here? */
      if (data[0]) XFree(data[0]);
    }
  return(Xen_list_6(C_int_to_Xen_integer(val),
		    C_to_Xen_Atom(a),
		    C_int_to_Xen_integer(ret),
		    C_ulong_to_Xen_ulong(len),
		    C_ulong_to_Xen_ulong(bytes),
		    result));
}

static Xen gxm_XGetTransientForHint(Xen arg1, Xen arg2)
{
  #define H_XGetTransientForHint "Status XGetTransientForHint(display, w): returns the WM_TRANSIENT_FOR property for the specified window."
  /* DIFF: XGetTransientForHint omit and rtn last arg
   */
  Window w;
  int val;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XGetTransientForHint", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XGetTransientForHint", "Window");
  val = XGetTransientForHint(Xen_to_C_Display(arg1), Xen_to_C_Window(arg2), &w);
  return(Xen_list_2(C_int_to_Xen_integer(val),
		    C_to_Xen_Window(w)));
}

static Xen gxm_XGetScreenSaver(Xen arg1)
{
  #define H_XGetScreenSaver "XGetScreenSaver(display) gets the current screen saver values."
  /* DIFF: XGetScreenSaver omit and rtn last 4 args
   */
  int a, b, c ,d, val;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XGetScreenSaver", "Display*");
  val = XGetScreenSaver(Xen_to_C_Display(arg1), &a, &b, &c, &d);
  return(Xen_list_5(C_int_to_Xen_integer(val),
		    C_int_to_Xen_integer(a),
		    C_int_to_Xen_integer(b),
		    C_int_to_Xen_integer(c),
		    C_int_to_Xen_integer(d)));
}

static Xen gxm_XGetPointerMapping(Xen arg1, Xen ignore, Xen arg3)
{
  #define H_XGetPointerMapping "int XGetPointerMapping(display, ignored, len): returns the current mapping of the pointer."
  /* DIFF: XGetPointerMapping ignores arg2, returns list
   * (XGetPointerMapping (XtDisplay (cadr (main-widgets))) 0 3)
   */
  int i, len, loc, rtn;
  unsigned char *map;
  Xen lst = Xen_empty_list;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XGetPointerMapping", "Display*");
  Xen_check_type(Xen_is_integer(arg3), arg3, 3, "XGetPointerMapping", "int");
  len = Xen_integer_to_C_int(arg3);
  if (len <= 0) Xen_check_type(0, arg3, 3, "XGetPointerMapping", "positive integer");
  map = (unsigned char *)calloc(len, sizeof(unsigned char));
  rtn = XGetPointerMapping(Xen_to_C_Display(arg1), map, len);
  if (len > rtn) len = rtn;
  loc = xm_protect(lst);
  for (i = len - 1; i >= 0; i--)
    lst = Xen_cons(C_int_to_Xen_integer((int)(map[i])), lst);
  free(map);
  xm_unprotect_at(loc);
  return(lst);
}

static Xen gxm_XGetPointerControl(Xen arg1)
{
  #define H_XGetPointerControl "XGetPointerControl(display) \
returns the pointer's current acceleration multiplier and acceleration threshold."
  /* DIFF: XGetPointerControl omits and return last 3 args
   */
  int val, num, den, thresh;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XGetPointerControl", "Display*");
  val = XGetPointerControl(Xen_to_C_Display(arg1), &num, &den, &thresh);
  return(Xen_list_4(C_int_to_Xen_integer(val),
		    C_int_to_Xen_integer(num),
		    C_int_to_Xen_integer(den),
		    C_int_to_Xen_integer(thresh)));
}

static Xen gxm_XGetKeyboardControl(Xen arg1)
{
  #define H_XGetKeyboardControl "XGetKeyboardControl(display): returns the current control values for the keyboard \
to the XKeyboardState structure."
  /* DIFF: XGetKeyboardControl omits arg2 and rtns list of fields
   */
  XKeyboardState ks;
  Xen v;
  int i, loc;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XGetKeyboardControl", "Display*");
  XGetKeyboardControl(Xen_to_C_Display(arg1), &ks);
  v = Xen_make_vector(32, Xen_integer_zero);
  loc = xm_protect(v);
  for (i = 0; i < 32; i++)
    Xen_vector_set(v, i, C_int_to_Xen_integer((int)(ks.auto_repeats[i])));
  xm_unprotect_at(loc);
  return(Xen_list_7(C_int_to_Xen_integer(ks.key_click_percent),
		    C_int_to_Xen_integer(ks.bell_percent),
		    C_int_to_Xen_integer(ks.bell_pitch),
		    C_int_to_Xen_integer(ks.bell_duration),
		    C_int_to_Xen_integer(ks.led_mask),
		    C_int_to_Xen_integer(ks.global_auto_repeat),
		    v));
}

static Xen gxm_XGetInputFocus(Xen arg1)
{
  #define H_XGetInputFocus "XGetInputFocus(display): returns the focus window and the current focus state."
  /* DIFF: XGetInputFocus omit and rtn last 2 args
   */
  Window w;
  int r, val;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XGetInputFocus", "Display*");
  val = XGetInputFocus(Xen_to_C_Display(arg1), &w, &r);
  return(Xen_list_3(C_int_to_Xen_integer(val),
		    C_to_Xen_Window(w),
		    C_int_to_Xen_integer(r)));
}

static Xen gxm_XGetIconName(Xen arg1, Xen arg2)
{
  #define H_XGetIconName "Status XGetIconName(display, w): returns the name to be displayed in the specified window's icon."
  /* DIFF: XGetIconName omits and returns arg3
   */
  char *str;
  int val;
  Xen res = Xen_false;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XGetIconName", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XGetIconName", "Window");
  val = XGetIconName(Xen_to_C_Display(arg1), Xen_to_C_Window(arg2), &str);
  if (val != 0)
    {
      res = C_string_to_Xen_string(str);
      XFree(str);
    }
  return(res);
}

static Xen gxm_XGetGeometry(Xen arg1, Xen arg2)
{
  #define H_XGetGeometry "Status XGetGeometry(display, d): returns the root window and the current geometry of the drawable."
  /* DIFF: XGetGeometry omits last 7 args and returns list
   */
  unsigned int wr, hr, br, dr;
  int xr, yr, val;
  Window root;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XGetGeometry", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XGetGeometry", "Drawable");
  val = XGetGeometry(Xen_to_C_Display(arg1), Xen_to_C_Window(arg2), 
		     &root, &xr, &yr, &wr, &hr, &br, &dr);
  return(Xen_list_8(C_int_to_Xen_integer(val),
		    C_to_Xen_Window(root),
		    C_int_to_Xen_integer(xr),
		    C_int_to_Xen_integer(yr),
		    C_ulong_to_Xen_ulong(wr),
		    C_ulong_to_Xen_ulong(hr),
		    C_ulong_to_Xen_ulong(br),
		    C_ulong_to_Xen_ulong(dr)));
}

static Xen gxm_XGetGCValues(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XGetGCValues "Status XGetGCValues(display, gc, valuemask): returns the components specified by valuemask for the specified GC."
  /* DIFF: XGetGCValues omits and returns last arg
   */
  XGCValues *val;
  int rtn;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XGetGCValues", "Display*");
  Xen_check_type(Xen_is_GC(arg2), arg2, 2, "XGetGCValues", "GC");
  Xen_check_type(Xen_is_ulong(arg3), arg3, 3, "XGetGCValues", "ulong");
  val = (XGCValues *)calloc(1, sizeof(XGCValues));
  rtn = XGetGCValues(Xen_to_C_Display(arg1), Xen_to_C_GC(arg2), Xen_ulong_to_C_ulong(arg3), val);
  return(Xen_list_2(C_int_to_Xen_integer(rtn),
		    C_to_Xen_XGCValues(val)));
}

static Xen gxm_XGetFontProperty(Xen arg1, Xen arg2)
{
  #define H_XGetFontProperty "Bool XGetFontProperty(font_struct, atom): returns the value of the specified font property. "
  /* DIFF: XGetFontProperty omits and rtns last arg
   */
  Bool val = False;
  unsigned long prop = 0;
  Xen_check_type(Xen_is_XFontStruct(arg1), arg1, 1, "XGetFontProperty", "XFontStruct*");
  Xen_check_type(Xen_is_Atom(arg2), arg2, 2, "XGetFontProperty", "Atom");
  val = XGetFontProperty(Xen_to_C_XFontStruct(arg1), Xen_to_C_Atom(arg2), &prop);
  return(Xen_list_2(C_bool_to_Xen_boolean(val),
		    C_ulong_to_Xen_ulong(prop)));
}

static Xen gxm_XGetErrorText(Xen arg1, Xen arg2, Xen ignore, Xen arg4)
{
  #define H_XGetErrorText "XGetErrorText(display, code, buffer_return, length) copies a null-terminated string describing the specified error \
code into the specified buffer."
  /* DIFF: XGetErrorText ignores arg3, returns text
   */
  char *buf;
  int len, val;
  Xen str;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XGetErrorText", "Display*");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XGetErrorText", "int");
  Xen_check_type(Xen_is_integer(arg4), arg4, 4, "XGetErrorText", "int");
  len = Xen_integer_to_C_int(arg4);
  if (len <= 0) Xen_check_type(0, arg4, 4, "XGetErrorText", "positive integer");
  buf = (char *)calloc(len, sizeof(char));
  val = XGetErrorText(Xen_to_C_Display(arg1), Xen_integer_to_C_int(arg2), buf, len);
  str = C_string_to_Xen_string(buf);
  free(buf);
  return(Xen_list_2(C_int_to_Xen_integer(val),
		    str));
}

static Xen gxm_XGeometry(Xen args)
{
  #define H_XGeometry "int XGeometry(dpy, screen, position, default_position, bwidth, fwidth, fheight, xadder, yadder) calculates \
window geometry given user geometry string and default geometry"
  /* DIFF: XGetGeometry omits trailing 4 args and returns them
   */
  int x, y, w, h, val;
  Xen arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8, arg9;
  arg1 = Xen_list_ref(args, 0);
  arg2 = Xen_list_ref(args, 1);
  arg3 = Xen_list_ref(args, 2);
  arg4 = Xen_list_ref(args, 3);
  arg5 = Xen_list_ref(args, 4);
  arg6 = Xen_list_ref(args, 5);
  arg7 = Xen_list_ref(args, 6);
  arg8 = Xen_list_ref(args, 7);
  arg9 = Xen_list_ref(args, 8);
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XGeometry", "Display*");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XGeometry", "int");
  Xen_check_type(Xen_is_string(arg3), arg3, 3, "XGeometry", "char*");
  Xen_check_type(Xen_is_string(arg4), arg4, 4, "XGeometry", "char*");
  Xen_check_type(Xen_is_ulong(arg5), arg5, 5, "XGeometry", "unsigned int");
  Xen_check_type(Xen_is_ulong(arg6), arg6, 6, "XGeometry", "unsigned int");
  Xen_check_type(Xen_is_ulong(arg7), arg7, 7, "XGeometry", "unsigned int");
  Xen_check_type(Xen_is_integer(arg8), arg8, 8, "XGeometry", "int");
  Xen_check_type(Xen_is_integer(arg9), arg9, 9, "XGeometry", "int");
  val = XGeometry(Xen_to_C_Display(arg1), 
		  Xen_integer_to_C_int(arg2), 
		  (char *)Xen_string_to_C_string(arg3), 
		  (char *)Xen_string_to_C_string(arg4), 
		  Xen_ulong_to_C_ulong(arg5), 
		  Xen_ulong_to_C_ulong(arg6), 
		  Xen_ulong_to_C_ulong(arg7), 
		  Xen_integer_to_C_int(arg8), 
		  Xen_integer_to_C_int(arg9), 
		  &x, &y, &w, &h);
  return(Xen_list_5(C_int_to_Xen_integer(val),
		    C_int_to_Xen_integer(x),
		    C_int_to_Xen_integer(y),
		    C_int_to_Xen_integer(w),
		    C_int_to_Xen_integer(h)));
}

static Xen gxm_XFreePixmap(Xen arg1, Xen arg2)
{
  #define H_XFreePixmap "XFreePixmap(display, pixmap) first deletes the association between the pixmap ID and the pixmap."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XFreePixmap", "Display*");
  Xen_check_type(Xen_is_Pixmap(arg2), arg2, 2, "XFreePixmap", "Pixmap");
  return(C_int_to_Xen_integer(XFreePixmap(Xen_to_C_Display(arg1), Xen_to_C_Pixmap(arg2))));
}

static Xen gxm_XFreeModifiermap(Xen arg1)
{
  #define H_XFreeModifiermap "XFreeModifiermap(modmap) frees the specified XModifierKeymap structure."
  Xen_check_type(Xen_is_XModifierKeymap(arg1), arg1, 1, "XFreeModifiermap", "XModifierKeymap*");
  return(C_int_to_Xen_integer(XFreeModifiermap(Xen_to_C_XModifierKeymap(arg1))));
}

static Xen gxm_XFreeGC(Xen arg1, Xen arg2)
{
  #define H_XFreeGC "XFreeGC(display, gc) destroys the specified GC as well as all the associated storage."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XFreeGC", "Display*");
  Xen_check_type(Xen_is_GC(arg2), arg2, 2, "XFreeGC", "GC");
  return(C_int_to_Xen_integer(XFreeGC(Xen_to_C_Display(arg1), Xen_to_C_GC(arg2))));
}

static Xen gxm_XFreeFontPath(Xen ignore)
{
  #define H_XFreeFontPath "XFreeFontPath(list) frees the data allocated by XGetFontPath (a no-op in xm)."
  /* DIFF: XFreeFontPath is no-op
   */
  return(Xen_false);
}

static Xen gxm_XFreeFontNames(Xen ignore)
{
  #define H_XFreeFontNames "XFreeFontNames(list) frees the array and strings returned by XListFonts or XListFontsWithInfo (a no-op in xm)."
  /* DIFF: XFreeFontNames is no-op
   */
  return(Xen_false);
}

static Xen gxm_XFreeFontInfo(Xen ignore1, Xen ignore2, Xen ignore3)
{
  #define H_XFreeFontInfo "XFreeFontInfo(names, free_info, actual_count) frees a font structure or an array of font structures, and \
optionally an array of font names (a no-op in xm)."
  /* DIFF: XFreeFontInfo is a no-op 
   */
  return(Xen_false);
}

static Xen gxm_XFreeFont(Xen arg1, Xen arg2)
{
  #define H_XFreeFont "XFreeFont(display, font_struct) deletes the association between the font resource ID and the specified font and \
frees the XFontStruct structure."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XFreeFont", "Display*");
  Xen_check_type(Xen_is_XFontStruct(arg2), arg2, 2, "XFreeFont", "XFontStruct*");
  return(C_int_to_Xen_integer(XFreeFont(Xen_to_C_Display(arg1), Xen_to_C_XFontStruct(arg2))));
}

static Xen gxm_XFreeExtensionList(Xen ignore)
{
  /* DIFF: XFreeExtensionList is a no-op
   */
  return(Xen_false);
}

static Xen gxm_XFreeCursor(Xen arg1, Xen arg2)
{
  #define H_XFreeCursor "XFreeCursor(display, cursor) deletes the association between the cursor resource ID and the specified cursor."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XFreeCursor", "Display*");
  Xen_check_type(Xen_is_Cursor(arg2), arg2, 2, "XFreeCursor", "Cursor");
  return(C_int_to_Xen_integer(XFreeCursor(Xen_to_C_Display(arg1), Xen_to_C_Cursor(arg2))));
}

static Xen gxm_XFreeColors(Xen arg1, Xen arg2, Xen arg3, Xen arg4, Xen arg5)
{
  #define H_XFreeColors "XFreeColors(display, colormap, pixels, npixels, planes) frees the cells represented by pixels whose values are in the pixels array."
  /* DIFF: XFreeColors pixel array (arg3) is list of pixels
   */
  unsigned long *ps;
  int len, val;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XFreeColors", "Display*");
  Xen_check_type(Xen_is_Colormap(arg2), arg2, 2, "XFreeColors", "Colormap");
  Xen_check_type(Xen_is_list(arg3), arg3, 3, "XFreeColors", "list of pixel");
  Xen_check_type(Xen_is_integer(arg4), arg4, 4, "XFreeColors", "int");
  Xen_check_type(Xen_is_ulong(arg5), arg5, 5, "XFreeColors", "ulong");
  len = Xen_integer_to_C_int(arg4);
  if (len <= 0) return(Xen_false);
  ps = Xen_to_C_Pixels(arg3, len);
  val = XFreeColors(Xen_to_C_Display(arg1), 
		    Xen_to_C_Colormap(arg2), 
		    ps, len, 
		    Xen_ulong_to_C_ulong(arg5));
  free(ps);
  return(C_int_to_Xen_integer(val));
}

static Xen gxm_XFreeColormap(Xen arg1, Xen arg2)
{
  #define H_XFreeColormap "XFreeColormap(display, colormap) deletes the association between the colormap resource ID and the colormap and \
frees the colormap storage."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XFreeColormap", "Display*");
  Xen_check_type(Xen_is_Colormap(arg2), arg2, 2, "XFreeColormap", "Colormap");
  return(C_int_to_Xen_integer(XFreeColormap(Xen_to_C_Display(arg1), Xen_to_C_Colormap(arg2))));
}

static Xen gxm_XFree(Xen arg1)
{
  #define H_XFree "XFree(data) is a general-purpose Xlib routine that frees the specified data."
  Xen_check_type(Xen_is_wrapped_c_pointer(arg1) || Xen_is_list(arg1), arg1, 1, "XFree", "void* or xm entity");

  if (Xen_is_list(arg1))
    XFree((void *)Xen_unwrap_C_pointer(Xen_cadr(arg1)));
  else XFree((void *)Xen_unwrap_C_pointer(arg1));
  return(Xen_false);
}

static Xen gxm_XForceScreenSaver(Xen arg1, Xen arg2)
{
  #define H_XForceScreenSaver "XForceScreenSaver(display, mode) activates the screen saver"
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XForceScreenSaver", "Display*");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XForceScreenSaver", "int");
  return(C_int_to_Xen_integer(XForceScreenSaver(Xen_to_C_Display(arg1), Xen_integer_to_C_int(arg2))));
}

static Xen gxm_XFlush(Xen arg1)
{
  #define H_XFlush "XFlush(display) flushes the output buffer."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XFlush", "Display*");
  return(C_int_to_Xen_integer(XFlush(Xen_to_C_Display(arg1))));
}


static Xen gxm_XFillRectangles(Xen arg1, Xen arg2, Xen arg3, Xen larg4, Xen arg5)
{
  #define H_XFillRectangles "XFillRectangles(display, d, gc, rectangles, nrectangles)"
  /* DIFF: XFillRectangles XRectangle* arg (arg 4) is list of XRectangles
   */
  XRectangle *pt;
  int i, len;
  Xen arg4;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XFillRectangles", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XFillRectangles", "Drawable");
  Xen_check_type(Xen_is_GC(arg3), arg3, 3, "XFillRectangles", "GC");
  Xen_check_type(Xen_is_list(larg4), larg4, 4, "XFillRectangles", "list of XRectangle");
  Xen_check_type(Xen_is_integer(arg5), arg5, 5, "XFillRectangles", "int");
  arg4 = Xen_copy_arg(larg4);
  len = Xen_integer_to_C_int(arg5);
  if (len <= 0) Xen_check_type(0, arg5, 5, "XFillRectangles", "positive integer");
  pt = (XRectangle *)calloc(len, sizeof(XRectangle));
  for (i = 0; (i < len) && (!Xen_is_null(arg4)); i++, arg4 = Xen_cdr(arg4))
    {
      XRectangle *pt1;
      pt1 = Xen_to_C_XRectangle(Xen_car(arg4));
      pt[i].x = pt1->x;
      pt[i].y = pt1->y;
      pt[i].width = pt1->width;
      pt[i].height = pt1->height;
    }
  XFillRectangles(Xen_to_C_Display(arg1), 
		  Xen_to_C_Window(arg2), 
		  Xen_to_C_GC(arg3),
		  pt, len);
  free(pt);
  return(C_int_to_Xen_integer(len));
}

static Xen gxm_XFillRectangle(Xen arg1, Xen arg2, Xen arg3, Xen arg4, Xen arg5, Xen arg6, Xen arg7)
{
  #define H_XFillRectangle "XFillRectangle(display, d, gc, x, y, width, height)"
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XFillRectangle", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XFillRectangle", "Drawable");
  Xen_check_type(Xen_is_GC(arg3), arg3, 3, "XFillRectangle", "GC");
  Xen_check_type(Xen_is_integer(arg4), arg4, 4, "XFillRectangle", "int");
  Xen_check_type(Xen_is_integer(arg5), arg5, 5, "XFillRectangle", "int");
  Xen_check_type(Xen_is_ulong(arg6), arg6, 6, "XFillRectangle", "unsigned int");
  Xen_check_type(Xen_is_ulong(arg7), arg7, 7, "XFillRectangle", "unsigned int");
  return(C_int_to_Xen_integer(XFillRectangle(Xen_to_C_Display(arg1), Xen_to_C_Window(arg2), 
				     Xen_to_C_GC(arg3), Xen_integer_to_C_int(arg4), Xen_integer_to_C_int(arg5), 
				     Xen_ulong_to_C_ulong(arg6), Xen_ulong_to_C_ulong(arg7))));
}

static Xen gxm_XFillPolygon(Xen arg1, Xen arg2, Xen arg3, Xen larg4, Xen arg5, Xen arg6, Xen arg7)
{
  #define H_XFillPolygon "XFillPolygon(display, d, gc, points, npoints, shape, mode)"
  /* DIFF: XFillPolygon Point* arg (arg 4) is list of XPoint
   */
  XPoint *pt, *pt1;
  int i, len;
  Xen arg4;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XFillPolygon", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XFillPolygon", "Drawable");
  Xen_check_type(Xen_is_GC(arg3), arg3, 3, "XFillPolygon", "GC");
  Xen_check_type(Xen_is_list(larg4), larg4, 4, "XFillPolygon", "list of XPoints");
  Xen_check_type(Xen_is_integer(arg5), arg5, 5, "XFillPolygon", "int");
  Xen_check_type(Xen_is_integer(arg6), arg6, 6, "XFillPolygon", "int");
  Xen_check_type(Xen_is_integer(arg7), arg7, 7, "XFillPolygon", "int");
  arg4 = Xen_copy_arg(larg4);
  len = Xen_integer_to_C_int(arg5);
  if (len <= 0) Xen_check_type(0, arg5, 5, "XFillPolygon", "positive integer");
  pt = (XPoint *)calloc(len, sizeof(XPoint));
  for (i = 0; (i < len) && (!Xen_is_null(arg4)); i++, arg4 = Xen_cdr(arg4))
    {
      Xen xp;
      xp = Xen_car(arg4);
      if (!(Xen_is_XPoint(xp))) Xen_check_type(0, xp, i, "XFillRegion", "XPoint");
      pt1 = Xen_to_C_XPoint(xp);
      pt[i].x = pt1->x;
      pt[i].y = pt1->y;
    }
  XFillPolygon(Xen_to_C_Display(arg1), 
	       Xen_to_C_Window(arg2), 
	       Xen_to_C_GC(arg3), 
	       pt, len, 
	       Xen_integer_to_C_int(arg6), 
	       Xen_integer_to_C_int(arg7));
  free(pt);
  return(C_int_to_Xen_integer(len));
}

static Xen gxm_XFillArcs(Xen arg1, Xen arg2, Xen arg3, Xen larg4, Xen arg5)
{
  #define H_XFillArcs "XFillArcs(display, d, gc, arcs, narcs)"
  /* DIFF: XFillArcs Arc* arg (arg 4) is list of XArcs
   */
  int i, len;
  Display *dpy;
  Drawable draw;
  GC gc;
  Xen arg4;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XFillArcs", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XFillArcs", "Drawable");
  Xen_check_type(Xen_is_GC(arg3), arg3, 3, "XFillArcs", "GC");
  Xen_check_type(Xen_is_list(larg4), larg4, 4, "XFillArcs", "list of XArcs");
  Xen_check_type(Xen_is_integer(arg5), arg5, 5, "XFillArcs", "int");
  arg4 = Xen_copy_arg(larg4);
  len = Xen_integer_to_C_int(arg5);
  if (len <= 0) return(Xen_false);
  dpy = Xen_to_C_Display(arg1);
  draw = Xen_to_C_Window(arg2);
  gc = Xen_to_C_GC(arg3);
  for (i = 0; (i < len) && (!Xen_is_null(arg4)); i++, arg4 = Xen_cdr(arg4))
    {
      XArc *arc;
      arc = Xen_to_C_XArc(Xen_car(arg4));
      XFillArc(dpy, draw, gc, arc->x, arc->y, arc->width, arc->height, arc->angle1, arc->angle2);
    }
  return(C_int_to_Xen_integer(len));
}

static Xen gxm_XFillArc(Xen arg1, Xen arg2, Xen arg3, Xen arg4, Xen arg5, Xen arg6, Xen arg7, Xen arg8, Xen arg9)
{
  #define H_XFillArc "XFillArc(display, d, gc, x, y, width, height, angle1, angle2) fills the region described by the specified arc."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XFillArc", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XFillArc", "Drawable");
  Xen_check_type(Xen_is_GC(arg3), arg3, 3, "XFillArc", "GC");
  Xen_check_type(Xen_is_integer(arg4), arg4, 4, "XFillArc", "int");
  Xen_check_type(Xen_is_integer(arg5), arg5, 5, "XFillArc", "int");
  Xen_check_type(Xen_is_ulong(arg6), arg6, 6, "XFillArc", "unsigned int");
  Xen_check_type(Xen_is_ulong(arg7), arg7, 7, "XFillArc", "unsigned int");
  Xen_check_type(Xen_is_integer(arg8), arg8, 8, "XFillArc", "int");
  Xen_check_type(Xen_is_integer(arg9), arg9, 9, "XFillArc", "int");
  return(C_int_to_Xen_integer(XFillArc(Xen_to_C_Display(arg1), Xen_to_C_Window(arg2), 
			       Xen_to_C_GC(arg3), Xen_integer_to_C_int(arg4), Xen_integer_to_C_int(arg5), 
			       Xen_ulong_to_C_ulong(arg6), Xen_ulong_to_C_ulong(arg7), 
			       Xen_integer_to_C_int(arg8), Xen_integer_to_C_int(arg9))));
}

static Xen gxm_XFetchName(Xen arg1, Xen arg2)
{
  #define H_XFetchName "Status XFetchName(display, w): returns the name of the specified window."
  /* DIFF: XFetchName omits and rtns arg3
   */
  char *name;
  int val;
  Xen str;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XFetchName", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XFetchName", "Window");
  val = XFetchName(Xen_to_C_Display(arg1), Xen_to_C_Window(arg2), &name);
  if (val == 0)
    return(Xen_false);
  str = C_string_to_Xen_string(name);
  free(name);
  return(str);
}

static Xen gxm_XEventsQueued(Xen arg1, Xen arg2)
{
  #define H_XEventsQueued "int XEventsQueued(display, mode)"
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XEventsQueued", "Display*");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XEventsQueued", "int");
  return(C_int_to_Xen_integer(XEventsQueued(Xen_to_C_Display(arg1), Xen_integer_to_C_int(arg2))));
}

static Xen gxm_XEnableAccessControl(Xen arg1)
{
  #define H_XEnableAccessControl "XEnableAccessControl(display) enables the use of the access control list at each connection setup."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XEnableAccessControl", "Display*");
  return(C_int_to_Xen_integer(XEnableAccessControl(Xen_to_C_Display(arg1))));
}

static Xen gxm_XDrawText(Xen arg1, Xen arg2, Xen arg3, Xen arg4, Xen arg5, Xen arg6, Xen arg7)
{
  #define H_XDrawText "XDrawText(display, d, gc, x, y, items, nitems) draws text"
  int i, len = 0, res;
  XTextItem *items;
  Xen lst;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XDrawText", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XDrawText", "Drawable");
  Xen_check_type(Xen_is_GC(arg3), arg3, 3, "XDrawText", "GC");
  Xen_check_type(Xen_is_integer(arg4), arg4, 4, "XDrawText", "int");
  Xen_check_type(Xen_is_integer(arg5), arg5, 5, "XDrawText", "int");
  Xen_check_type(Xen_is_list(arg6), arg6, 6, "XDrawText", "list of XTextItem");
  Xen_check_type(Xen_is_integer_or_unbound(arg7), arg7, 7, "XDrawText", "int");
  lst = Xen_copy_arg(arg6);
  if (Xen_is_integer(arg7)) len = Xen_integer_to_C_int(arg7); else len = Xen_list_length(arg6);
  items = (XTextItem *)calloc(len, sizeof(XTextItem));
  for (i = 0; (i < len) && (!Xen_is_null(lst)); i++, lst = Xen_cdr(lst))
    {
      XTextItem *val;
      val = Xen_to_C_XTextItem(Xen_car(lst));
      items[i].chars = val->chars;
      items[i].nchars = val->nchars;
      items[i].delta = val->delta;
      items[i].font = val->font;
    }
  res = XDrawText(Xen_to_C_Display(arg1), 
		  Xen_to_C_Window(arg2), 
		  Xen_to_C_GC(arg3), 
		  Xen_integer_to_C_int(arg4), Xen_integer_to_C_int(arg5), 
		  items, len);
  free(items);
  return(C_int_to_Xen_integer(res));
}

static Xen gxm_XDrawString(Xen arg1, Xen arg2, Xen arg3, Xen arg4, Xen arg5, Xen arg6, Xen arg7)
{
  #define H_XDrawString "XDrawString(display, d, gc, x, y, string, length)"
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XDrawString", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XDrawString", "Drawable");
  Xen_check_type(Xen_is_GC(arg3), arg3, 3, "XDrawString", "GC");
  Xen_check_type(Xen_is_integer(arg4), arg4, 4, "XDrawString", "int");
  Xen_check_type(Xen_is_integer(arg5), arg5, 5, "XDrawString", "int");
  Xen_check_type(Xen_is_string(arg6), arg6, 6, "XDrawString", "char*");
  Xen_check_type(Xen_is_integer(arg7), arg7, 7, "XDrawString", "int");
  return(C_int_to_Xen_integer(XDrawString(Xen_to_C_Display(arg1), 
				  Xen_to_C_Window(arg2), 
				  Xen_to_C_GC(arg3), 
				  Xen_integer_to_C_int(arg4), Xen_integer_to_C_int(arg5), 
				  (char *)Xen_string_to_C_string(arg6), Xen_integer_to_C_int(arg7))));
}

static Xen gxm_XDrawSegments(Xen arg1, Xen arg2, Xen arg3, Xen larg4, Xen arg5)
{
  #define H_XDrawSegments "XDrawSegments(display, d, gc, segments, nsegments) draws multiple, unconnected lines. "
  /* DIFF: XDrawSegments XSegment* arg (arg 4) is list of XSegments
   */
  XSegment *pt;
  int i, len;
  Xen arg4;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XDrawSegments", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XDrawSegments", "Drawable");
  Xen_check_type(Xen_is_GC(arg3), arg3, 3, "XDrawSegments", "GC");
  Xen_check_type(Xen_is_list(larg4), larg4, 4, "XDrawSegments", "list of XSegments");
  Xen_check_type(Xen_is_integer(arg5), arg5, 5, "XDrawSegments", "int");
  arg4 = Xen_copy_arg(larg4);
  len = Xen_integer_to_C_int(arg5);
  if (len <= 0) Xen_check_type(0, arg5, 5, "XDrawSegments", "positive integer");
  pt = (XSegment *)calloc(len, sizeof(XSegment));
  for (i = 0; (i < len) && (!Xen_is_null(arg4)); i++, arg4 = Xen_cdr(arg4))
    {
      XSegment *pt1;
      pt1 = Xen_to_C_XSegment(Xen_car(arg4));
      pt[i].x1 = pt1->x1;
      pt[i].y1 = pt1->y1;
      pt[i].x2 = pt1->x2;
      pt[i].y2 = pt1->y2;
    }
  XDrawSegments(Xen_to_C_Display(arg1), 
		Xen_to_C_Window(arg2), 
		Xen_to_C_GC(arg3),
		pt, len);
  free(pt);
  return(C_int_to_Xen_integer(len));
}

static Xen gxm_XDrawRectangles(Xen arg1, Xen arg2, Xen arg3, Xen larg4, Xen arg5)
{
  #define H_XDrawRectangles "XDrawRectangles(display, d, gc, rectangles, nrectangles) draws the outlines of the specified rectangles."
  /* DIFF: XDrawRectangles XRectangle* arg (arg 4) is list of XRectangles
   */
  XRectangle *pt;
  int i, len;
  Xen arg4;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XDrawRectangles", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XDrawRectangles", "Drawable");
  Xen_check_type(Xen_is_GC(arg3), arg3, 3, "XDrawRectangles", "GC");
  Xen_check_type(Xen_is_list(larg4), larg4, 4, "XDrawRectangles", "list of XRectangles");
  Xen_check_type(Xen_is_integer(arg5), arg5, 5, "XDrawRectangles", "int");
  arg4 = Xen_copy_arg(larg4);
  len = Xen_integer_to_C_int(arg5);
  if (len <= 0) Xen_check_type(0, arg5, 5, "XDrawRectangles", "positive integer");
  pt = (XRectangle *)calloc(len, sizeof(XRectangle));
  for (i = 0; (i < len) && (!Xen_is_null(arg4)); i++, arg4 = Xen_cdr(arg4))
    {
      XRectangle *pt1;
      pt1 = Xen_to_C_XRectangle(Xen_car(arg4));
      pt[i].x = pt1->x;
      pt[i].y = pt1->y;
      pt[i].width = pt1->width;
      pt[i].height = pt1->height;
    }
  XDrawRectangles(Xen_to_C_Display(arg1), 
		  Xen_to_C_Window(arg2), 
		  Xen_to_C_GC(arg3),
		  pt, len);
  free(pt);
  return(C_int_to_Xen_integer(len));
}

static Xen gxm_XDrawRectangle(Xen arg1, Xen arg2, Xen arg3, Xen arg4, Xen arg5, Xen arg6, Xen arg7)
{
  #define H_XDrawRectangle "XDrawRectangle(display, d, gc, x, y, width, height) draws the outlines of the specified rectangle."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XDrawRectangle", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XDrawRectangle", "Drawable");
  Xen_check_type(Xen_is_GC(arg3), arg3, 3, "XDrawRectangle", "GC");
  Xen_check_type(Xen_is_integer(arg4), arg4, 4, "XDrawRectangle", "int");
  Xen_check_type(Xen_is_integer(arg5), arg5, 5, "XDrawRectangle", "int");
  Xen_check_type(Xen_is_ulong(arg6), arg6, 6, "XDrawRectangle", "unsigned int");
  Xen_check_type(Xen_is_ulong(arg7), arg7, 7, "XDrawRectangle", "unsigned int");
  return(C_int_to_Xen_integer(XDrawRectangle(Xen_to_C_Display(arg1), 
				     Xen_to_C_Window(arg2), 
				     Xen_to_C_GC(arg3), 
				     Xen_integer_to_C_int(arg4), Xen_integer_to_C_int(arg5), Xen_ulong_to_C_ulong(arg6), Xen_ulong_to_C_ulong(arg7))));
}

static Xen gxm_XDrawPoints(Xen arg1, Xen arg2, Xen arg3, Xen larg4, Xen arg5, Xen arg6)
{
  #define H_XDrawPoints "XDrawPoints(display, d, gc, points, npoints, mode) draws multiple points."
  /* DIFF: XDrawPoints XPoint* arg (arg 4) is list of XPoints
   */
  XPoint *pt, *pt1;
  int i, len;
  Xen arg4;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XDrawPoints", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XDrawPoints", "Drawable");
  Xen_check_type(Xen_is_GC(arg3), arg3, 3, "XDrawPoints", "GC");
  Xen_check_type(Xen_is_list(larg4), larg4, 4, "XDrawPoints", "list of XPoints");
  Xen_check_type(Xen_is_integer(arg5), arg5, 5, "XDrawPoints", "int");
  Xen_check_type(Xen_is_integer(arg6), arg6, 6, "XDrawPoints", "int");
  arg4 = Xen_copy_arg(larg4);
  len = Xen_integer_to_C_int(arg5);
  if (len <= 0) Xen_check_type(0, arg5, 5, "XDrawPoints", "positive integer");
  pt = (XPoint *)calloc(len, sizeof(XPoint));
  for (i = 0; (i < len) && (!Xen_is_null(arg4)); i++, arg4 = Xen_cdr(arg4))
    {
      Xen xp;
      xp = Xen_car(arg4);
      if (!(Xen_is_XPoint(xp))) Xen_check_type(0, xp, i, "XDrawPoints", "XPoint");
      pt1 = Xen_to_C_XPoint(xp);
      pt[i].x = pt1->x;
      pt[i].y = pt1->y;
    }
  XDrawPoints(Xen_to_C_Display(arg1), 
	      Xen_to_C_Window(arg2), 
	      Xen_to_C_GC(arg3), 
	      pt, len, 
	      Xen_integer_to_C_int(arg6));
  free(pt);
  return(C_int_to_Xen_integer(len));
}

static Xen gxm_XDrawPoint(Xen arg1, Xen arg2, Xen arg3, Xen arg4, Xen arg5)
{
  #define H_XDrawPoint "XDrawPoint(display, d, gc, x, y) uses the foreground pixel and function components of the GC to draw a single \
point into the specified drawable."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XDrawPoint", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XDrawPoint", "Drawable");
  Xen_check_type(Xen_is_GC(arg3), arg3, 3, "XDrawPoint", "GC");
  Xen_check_type(Xen_is_integer(arg4), arg4, 4, "XDrawPoint", "int");
  Xen_check_type(Xen_is_integer(arg5), arg5, 5, "XDrawPoint", "int");
  return(C_int_to_Xen_integer(XDrawPoint(Xen_to_C_Display(arg1), 
				 Xen_to_C_Window(arg2), 
				 Xen_to_C_GC(arg3), 
				 Xen_integer_to_C_int(arg4), Xen_integer_to_C_int(arg5))));
}

static Xen gxm_XDrawLines(Xen arg1, Xen arg2, Xen arg3, Xen larg4, Xen arg5, Xen arg6)
{
  #define H_XDrawLines "XDrawLines(display, d, gc, points, npoints, mode) uses the components of the specified GC to draw npoints lines \
between each pair of points (point[i], point[i+1]) in the array of XPoint structures."
  /* DIFF: XDrawLines XPoint* arg (arg 4) is list of XPoints
   */
  XPoint *pt, *pt1;
  int i, len;
  Xen arg4;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XDrawLines", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XDrawLines", "Drawable");
  Xen_check_type(Xen_is_GC(arg3), arg3, 3, "XDrawLines", "GC");
  Xen_check_type(Xen_is_list(larg4), larg4, 4, "XDrawLines", "list of XPoints");
  Xen_check_type(Xen_is_integer(arg5), arg5, 5, "XDrawLines", "int");
  Xen_check_type(Xen_is_integer(arg6), arg6, 6, "XDrawLines", "int");
  arg4 = Xen_copy_arg(larg4);
  len = Xen_integer_to_C_int(arg5);
  if (len <= 0) Xen_check_type(0, arg5, 5, "XDrawLines", "positive integer");
  pt = (XPoint *)calloc(len, sizeof(XPoint));
  for (i = 0; (i < len) && (!Xen_is_null(arg4)); i++, arg4 = Xen_cdr(arg4))
    {
      Xen xp;
      xp = Xen_car(arg4);
      if (!(Xen_is_XPoint(xp))) Xen_check_type(0, xp, i, "XDrawLines", "XPoint");
      pt1 = Xen_to_C_XPoint(xp);
      pt[i].x = pt1->x;
      pt[i].y = pt1->y;
    }
  XDrawLines(Xen_to_C_Display(arg1), 
	     Xen_to_C_Window(arg2), 
	     Xen_to_C_GC(arg3), 
	     pt, len, 
	     Xen_integer_to_C_int(arg6));
  free(pt);
  return(C_int_to_Xen_integer(len));
}

static Xen gxm_XDrawLinesDirect(Xen arg1, Xen arg2, Xen arg3, Xen arg4, Xen arg5, Xen arg6)
{
  #define H_XDrawLinesDirect "XDrawLinesDirect is the same as XDrawLines but takes an (opaque) pointer to an XPoint array"
  XPoint *pt;
  int len;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XDrawLinesDirect", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XDrawLinesDirect", "Drawable");
  Xen_check_type(Xen_is_GC(arg3), arg3, 3, "XDrawLinesDirect", "GC");
  Xen_check_type(Xen_is_wrapped_c_pointer(arg4), arg4, 4, "XDrawLinesDirect", "array of XPoints");
  Xen_check_type(Xen_is_integer(arg5), arg5, 5, "XDrawLines", "int");
  Xen_check_type(Xen_is_integer(arg6), arg6, 6, "XDrawLines", "int");
  len = Xen_integer_to_C_int(arg5);
  if (len <= 0) return(Xen_false);
  pt = (XPoint *)Xen_unwrap_C_pointer(arg4);
  XDrawLines(Xen_to_C_Display(arg1), 
	     Xen_to_C_Window(arg2), 
	     Xen_to_C_GC(arg3), 
	     pt, len, 
	     Xen_integer_to_C_int(arg6));
  return(C_int_to_Xen_integer(len));
}

static Xen gxm_Vector2XPoints(Xen arg1)
{
  #define H_vector2XPoints "(vector->XPoints vect) packages point data in vect as (opaque) array of XPoints"
  int i, j, len;
  /* vector assumed to be sequence of x y pairs (not XPoints from local view)
   */
  XPoint *pt;
  Xen_check_type(Xen_is_vector(arg1), arg1, 1, "vector->XPoints", "vector of x,y values");
  len = Xen_vector_length(arg1) / 2;
  if (len <= 0) Xen_check_type(0, arg1, 1, "vector->XPoints", "positive integer");
  pt = (XPoint *)calloc(len, sizeof(XPoint));
  for (i = 0, j = 0; i < len; i++, j += 2)
    {
      pt[i].x = Xen_integer_to_C_int(Xen_vector_ref(arg1, j));
      pt[i].y = Xen_integer_to_C_int(Xen_vector_ref(arg1, j + 1));
    }
  return(Xen_wrap_C_pointer(pt));
}

static Xen gxm_FreeXPoints(Xen arg1)
{
  void *pts;
  #define H_freeXPoints "(freeXPoints vect) frees an (opaque) XPoint array created by vector->Xpoints"
  Xen_check_type(Xen_is_wrapped_c_pointer(arg1), arg1, 1, "freeXPoints", "opaque XPoint array");
  pts = (void *)(Xen_unwrap_C_pointer(arg1));
  free(pts);
  return(Xen_false);
}


static Xen gxm_XDrawLine(Xen arg1, Xen arg2, Xen arg3, Xen arg4, Xen arg5, Xen arg6, Xen arg7)
{
  #define H_XDrawLine "XDrawLine(display, d, gc, x1, y1, x2, y2) uses the components of the specified GC to draw a line between the \
specified set of points (x1, y1) and (x2, y2)."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XDrawLine", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XDrawLine", "Drawable");
  Xen_check_type(Xen_is_GC(arg3), arg3, 3, "XDrawLine", "GC");
  Xen_check_type(Xen_is_integer(arg4), arg4, 4, "XDrawLine", "int");
  Xen_check_type(Xen_is_integer(arg5), arg5, 5, "XDrawLine", "int");
  Xen_check_type(Xen_is_integer(arg6), arg6, 6, "XDrawLine", "int");
  Xen_check_type(Xen_is_integer(arg7), arg7, 7, "XDrawLine", "int");
  return(C_int_to_Xen_integer(XDrawLine(Xen_to_C_Display(arg1), 
				Xen_to_C_Window(arg2), 
				Xen_to_C_GC(arg3), 
				Xen_integer_to_C_int(arg4), Xen_integer_to_C_int(arg5), Xen_integer_to_C_int(arg6), Xen_integer_to_C_int(arg7))));
}

static Xen gxm_XDrawImageString(Xen arg1, Xen arg2, Xen arg3, Xen arg4, Xen arg5, Xen arg6, Xen arg7)
{
  #define H_XDrawImageString "XDrawImageString(display, d, gc, x, y, string, length)"
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XDrawImageString", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XDrawImageString", "Drawable");
  Xen_check_type(Xen_is_GC(arg3), arg3, 3, "XDrawImageString", "GC");
  Xen_check_type(Xen_is_integer(arg4), arg4, 4, "XDrawImageString", "int");
  Xen_check_type(Xen_is_integer(arg5), arg5, 5, "XDrawImageString", "int");
  Xen_check_type(Xen_is_string(arg6), arg6, 6, "XDrawImageString", "char*");
  Xen_check_type(Xen_is_integer(arg7), arg7, 7, "XDrawImageString", "int");
  return(C_int_to_Xen_integer(XDrawImageString(Xen_to_C_Display(arg1), 
				       Xen_to_C_Window(arg2), 
				       Xen_to_C_GC(arg3), 
				       Xen_integer_to_C_int(arg4), Xen_integer_to_C_int(arg5), (char *)Xen_string_to_C_string(arg6), Xen_integer_to_C_int(arg7))));
}

static Xen gxm_XDrawArcs(Xen arg1, Xen arg2, Xen arg3, Xen larg4, Xen arg5)
{
  #define H_XDrawArcs "XDrawArcs(display, d, gc, arcs, narcs) draws multiple circular or elliptical arcs."
  /* DIFF: XDrawArcs Arc* arg (arg 4) is list of XArcs
   */
  int i, len;
  Display *dpy;
  Drawable draw;
  GC gc;
  Xen arg4;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XDrawArcs", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XDrawArcs", "Drawable");
  Xen_check_type(Xen_is_GC(arg3), arg3, 3, "XDrawArcs", "GC");
  Xen_check_type(Xen_is_list(larg4), larg4, 4, "XDrawArcs", "list of XArcs");
  Xen_check_type(Xen_is_integer(arg5), arg5, 5, "XDrawArcs", "int");
  arg4 = Xen_copy_arg(larg4);
  len = Xen_integer_to_C_int(arg5);
  dpy = Xen_to_C_Display(arg1);
  draw = Xen_to_C_Window(arg2);
  gc = Xen_to_C_GC(arg3);
  for (i = 0; (i < len) && (!Xen_is_null(arg4)); i++, arg4 = Xen_cdr(arg4))
    {
      XArc *arc;
      arc = Xen_to_C_XArc(Xen_car(arg4));
      XDrawArc(dpy, draw, gc, arc->x, arc->y, arc->width, arc->height, arc->angle1, arc->angle2);
    }
  return(C_int_to_Xen_integer(len));
}

static Xen gxm_XDrawArc(Xen arg1, Xen arg2, Xen arg3, Xen arg4, Xen arg5, Xen arg6, Xen arg7, Xen arg8, Xen arg9)
{
  #define H_XDrawArc "XDrawArc(display, d, gc, x, y, width, height, angle1, angle2) draws a single circular or elliptical arc."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XDrawArc", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XDrawArc", "Drawable");
  Xen_check_type(Xen_is_GC(arg3), arg3, 3, "XDrawArc", "GC");
  Xen_check_type(Xen_is_integer(arg4), arg4, 4, "XDrawArc", "int");
  Xen_check_type(Xen_is_integer(arg5), arg5, 5, "XDrawArc", "int");
  Xen_check_type(Xen_is_ulong(arg6), arg6, 6, "XDrawArc", "unsigned int");
  Xen_check_type(Xen_is_ulong(arg7), arg7, 7, "XDrawArc", "unsigned int");
  Xen_check_type(Xen_is_integer(arg8), arg8, 8, "XDrawArc", "int");
  Xen_check_type(Xen_is_integer(arg9), arg9, 9, "XDrawArc", "int");
  return(C_int_to_Xen_integer(XDrawArc(Xen_to_C_Display(arg1), 
			       Xen_to_C_Window(arg2), 
			       Xen_to_C_GC(arg3), 
			       Xen_integer_to_C_int(arg4), Xen_integer_to_C_int(arg5), 
			       Xen_ulong_to_C_ulong(arg6), Xen_ulong_to_C_ulong(arg7), Xen_integer_to_C_int(arg8), Xen_integer_to_C_int(arg9))));
}

static Xen gxm_XDisplayWidthMM(Xen arg1, Xen arg2)
{
  #define H_DisplayWidthMM "DisplayWidthMM(display, screen_number): returns the width of the specified screen in millimeters."
  #define H_XDisplayWidthMM "XDisplayWidthMM(display, screen_number): returns the width of the specified screen in millimeters."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XDisplayWidthMM", "Display*");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XDisplayWidthMM", "int");
  return(C_int_to_Xen_integer(XDisplayWidthMM(Xen_to_C_Display(arg1), Xen_integer_to_C_int(arg2))));
}

static Xen gxm_XDisplayWidth(Xen arg1, Xen arg2)
{
  #define H_DisplayWidth "DisplayWidth(display, screen_number): returns the width of the screen in pixels."
  #define H_XDisplayWidth "XDisplayWidth(display, screen_number): returns the width of the screen in pixels."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XDisplayWidth", "Display*");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XDisplayWidth", "int");
  return(C_int_to_Xen_integer(XDisplayWidth(Xen_to_C_Display(arg1), Xen_integer_to_C_int(arg2))));
}

static Xen gxm_XDisplayPlanes(Xen arg1, Xen arg2)
{
  #define H_DisplayPlanes "DisplayPlanes(display, screen_number): returns the depth of the root window of the specified screen."
  #define H_XDisplayPlanes "XDisplayPlanes(display, screen_number): returns the depth of the root window of the specified screen."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XDisplayPlanes", "Display*");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XDisplayPlanes", "int");
  return(C_int_to_Xen_integer(XDisplayPlanes(Xen_to_C_Display(arg1), Xen_integer_to_C_int(arg2))));
}

static Xen gxm_XDisplayKeycodes(Xen arg1)
{
  #define H_XDisplayKeycodes "XDisplayKeycodes(display): returns the min-keycodes and max-keycodes supported by the specified display."
  /* DIFF: XDisplayKeycodes omit and rtn arg 2 and 3
   */
  int m1, m2, val;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XDisplayKeycodes", "Display*");
  val = XDisplayKeycodes(Xen_to_C_Display(arg1), &m1, &m2);
  return(Xen_list_3(C_int_to_Xen_integer(val),
		    C_int_to_Xen_integer(m1),
		    C_int_to_Xen_integer(m2)));
}

static Xen gxm_XDisplayHeightMM(Xen arg1, Xen arg2)
{
  #define H_DisplayHeightMM "DisplayHeightMM(display, screen_number): returns the height of the specified screen in millimeters."
  #define H_XDisplayHeightMM "XDisplayHeightMM(display, screen_number): returns the height of the specified screen in millimeters."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XDisplayHeightMM", "Display*");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XDisplayHeightMM", "int");
  return(C_int_to_Xen_integer(XDisplayHeightMM(Xen_to_C_Display(arg1), Xen_integer_to_C_int(arg2))));
}

static Xen gxm_XDisplayHeight(Xen arg1, Xen arg2)
{
  #define H_DisplayHeight "DisplayHeight(display, screen_number): returns the height of the specified screen in pixels."
  #define H_XDisplayHeight "XDisplayHeight(display, screen_number): returns the height of the specified screen in pixels."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XDisplayHeight", "Display*");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XDisplayHeight", "int");
  return(C_int_to_Xen_integer(XDisplayHeight(Xen_to_C_Display(arg1), Xen_integer_to_C_int(arg2))));
}

static Xen gxm_XDisplayCells(Xen arg1, Xen arg2)
{
  #define H_DisplayCells "DisplayCells(display, screen_number): returns the number of entries in the default colormap."
  #define H_XDisplayCells "XDisplayCells(display, screen_number): returns the number of entries in the default colormap."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XDisplayCells", "Display*");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XDisplayCells", "int");
  return(C_int_to_Xen_integer(XDisplayCells(Xen_to_C_Display(arg1), Xen_integer_to_C_int(arg2))));
}

static Xen gxm_XDisableAccessControl(Xen arg1)
{
  #define H_XDisableAccessControl "XDisableAccessControl(display) disables the use of the access control list at each connection setup."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XDisableAccessControl", "Display*");
  return(C_int_to_Xen_integer(XDisableAccessControl(Xen_to_C_Display(arg1))));
}

static Xen gxm_XDoesSaveUnders(Xen arg1)
{
  #define H_DoesSaveUnders "DoesSaveUnders(screen): returns a Boolean value indicating whether the screen supports save unders."
  #define H_XDoesSaveUnders "XDoesSaveUnders(screen): returns a Boolean value indicating whether the screen supports save unders."
  Xen_check_type(Xen_is_Screen(arg1), arg1, 1, "XDoesSaveUnders", "Screen*");
  return(C_bool_to_Xen_boolean(XDoesSaveUnders(Xen_to_C_Screen(arg1))));
}

static Xen gxm_XDoesBackingStore(Xen arg1)
{
  #define H_DoesBackingStore "DoesBackingStore(screen): returns WhenMapped, NotUseful,or Always,which indicate whether the screen supports backing stores."
  #define H_XDoesBackingStore "XDoesBackingStore(screen): returns WhenMapped, NotUseful,or Always,which indicate whether the screen supports backing stores."
  Xen_check_type(Xen_is_Screen(arg1), arg1, 1, "XDoesBackingStore", "Screen*");
  return(C_bool_to_Xen_boolean(XDoesBackingStore(Xen_to_C_Screen(arg1))));
}

static Xen gxm_XDestroySubwindows(Xen arg1, Xen arg2)
{
  #define H_XDestroySubwindows "XDestroySubwindows(display, w) destroys all inferior windows of the specified window, in bottom-to-top stacking order."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XDestroySubwindows", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XDestroySubwindows", "Window");
  return(C_int_to_Xen_integer(XDestroySubwindows(Xen_to_C_Display(arg1), Xen_to_C_Window(arg2))));
}

static Xen gxm_XDestroyWindow(Xen arg1, Xen arg2)
{
  #define H_XDestroyWindow "XDestroyWindow(display, w) destroys the specified window as well as all of its subwindows and causes the X server \
to generate a DestroyNotify event for each window."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XDestroyWindow", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XDestroyWindow", "Window");
  return(C_int_to_Xen_integer(XDestroyWindow(Xen_to_C_Display(arg1), Xen_to_C_Window(arg2))));
}

static Xen gxm_XDeleteProperty(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XDeleteProperty "XDeleteProperty(display, w, property) deletes the specified property only if the property was defined on the specified \
window and causes the X server to generate a PropertyNotify event on the window unless the property does not exist."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XDeleteProperty", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XDeleteProperty", "Window");
  Xen_check_type(Xen_is_Atom(arg3), arg3, 3, "XDeleteProperty", "Atom");
  return(C_int_to_Xen_integer(XDeleteProperty(Xen_to_C_Display(arg1), Xen_to_C_Window(arg2), Xen_to_C_Atom(arg3))));
}

static Xen gxm_XDefineCursor(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XDefineCursor "XDefineCursor(display, w, cursor)"
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XDefineCursor", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XDefineCursor", "Window");
  Xen_check_type(Xen_is_Cursor(arg3), arg3, 3, "XDefineCursor", "Cursor");
  return(C_int_to_Xen_integer(XDefineCursor(Xen_to_C_Display(arg1), Xen_to_C_Window(arg2), Xen_to_C_Cursor(arg3))));
}

static Xen gxm_XDefaultScreen(Xen arg1)
{
  #define H_XDefaultScreen "XDefaultScreen(display)"
  #define H_XDefaultScreenOfDisplay "returns the default screen of the specified display."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XDefaultScreen", "Display*");
  return(C_int_to_Xen_integer(XDefaultScreen(Xen_to_C_Display(arg1))));
}

static Xen gxm_XDefaultDepthOfScreen(Xen arg1)
{
  #define H_DefaultDepthOfScreen "returns the default depth of the root window of the specified screen."
  #define H_XDefaultDepthOfScreen "returns the default depth of the root window of the specified screen."
  Xen_check_type(Xen_is_Screen(arg1), arg1, 1, "XDefaultDepthOfScreen", "Screen*");
  return(C_int_to_Xen_integer(XDefaultDepthOfScreen(Xen_to_C_Screen(arg1))));
}

static Xen gxm_XDefaultDepth(Xen arg1, Xen arg2)
{
  #define H_DefaultDepth "returns the depth (number of planes) of the default root window for the specified screen."
  #define H_XDefaultDepth "returns the depth (number of planes) of the default root window for the specified screen."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XDefaultDepth", "Display*");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XDefaultDepth", "int");
  return(C_int_to_Xen_integer(XDefaultDepth(Xen_to_C_Display(arg1), Xen_integer_to_C_int(arg2))));
}

static Xen gxm_XCopyPlane(Xen args)
{
  #define H_XCopyPlane "XCopyPlane(display, src, dest, gc, src_x, src_y, width, height, dest_x, dest_y, plane) uses a single bit plane of the \
specified source rectangle combined with the specified GC to modify the specified rectangle of dest."
  Xen arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8, arg9, arg10, arg11;
  arg1 = Xen_list_ref(args, 0);
  arg2 = Xen_list_ref(args, 1);
  arg3 = Xen_list_ref(args, 2);
  arg4 = Xen_list_ref(args, 3);
  arg5 = Xen_list_ref(args, 4);
  arg6 = Xen_list_ref(args, 5);
  arg7 = Xen_list_ref(args, 6);
  arg8 = Xen_list_ref(args, 7);
  arg9 = Xen_list_ref(args, 8);
  arg10 = Xen_list_ref(args, 9);
  arg11 = Xen_list_ref(args, 10);
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XCopyPlane", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XCopyPlane", "Drawable");
  Xen_check_type(Xen_is_Window(arg3), arg3, 3, "XCopyPlane", "Drawable");
  Xen_check_type(Xen_is_GC(arg4), arg4, 4, "XCopyPlane", "GC");
  Xen_check_type(Xen_is_integer(arg5), arg5, 5, "XCopyPlane", "int");
  Xen_check_type(Xen_is_integer(arg6), arg6, 6, "XCopyPlane", "int");
  Xen_check_type(Xen_is_ulong(arg7), arg7, 7, "XCopyPlane", "unsigned int");
  Xen_check_type(Xen_is_ulong(arg8), arg8, 8, "XCopyPlane", "unsigned int");
  Xen_check_type(Xen_is_integer(arg9), arg9, 9, "XCopyPlane", "int");
  Xen_check_type(Xen_is_integer(arg10), arg10, 10, "XCopyPlane", "int");
  Xen_check_type(Xen_is_ulong(arg11), arg11, 11, "XCopyPlane", "ulong");
  return(C_int_to_Xen_integer(XCopyPlane(Xen_to_C_Display(arg1), 
				 Xen_to_C_Window(arg2), 
				 Xen_to_C_Window(arg3), 
				 Xen_to_C_GC(arg4), 
				 Xen_integer_to_C_int(arg5), Xen_integer_to_C_int(arg6), Xen_ulong_to_C_ulong(arg7), 
				 Xen_ulong_to_C_ulong(arg8), Xen_integer_to_C_int(arg9), 
				 Xen_integer_to_C_int(arg10), Xen_ulong_to_C_ulong(arg11))));
}

static Xen gxm_XCopyGC(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XCopyGC "XCopyGC(display, src, valuemask, dest) copies the specified components from the source GC to the destination GC."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XCopyGC", "Display*");
  Xen_check_type(Xen_is_GC(arg2), arg2, 2, "XCopyGC", "GC");
  Xen_check_type(Xen_is_ulong(arg3), arg3, 3, "XCopyGC", "ulong");
  Xen_check_type(Xen_is_GC(arg4), arg4, 4, "XCopyGC", "GC");
  return(C_int_to_Xen_integer(XCopyGC(Xen_to_C_Display(arg1), Xen_to_C_GC(arg2), Xen_ulong_to_C_ulong(arg3), Xen_to_C_GC(arg4))));
}

static Xen gxm_XCopyArea(Xen args)
{
  #define H_XCopyArea "XCopyArea(display, src, dest, gc, src_x, src_y, width, height, dest_x, dest_y) combines the specified rectangle of src \
with the specified rectangle of dest."
  Xen arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8, arg9, arg10;
  arg1 = Xen_list_ref(args, 0);
  arg2 = Xen_list_ref(args, 1);
  arg3 = Xen_list_ref(args, 2);
  arg4 = Xen_list_ref(args, 3);
  arg5 = Xen_list_ref(args, 4);
  arg6 = Xen_list_ref(args, 5);
  arg7 = Xen_list_ref(args, 6);
  arg8 = Xen_list_ref(args, 7);
  arg9 = Xen_list_ref(args, 8);
  arg10 = Xen_list_ref(args, 9);
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XCopyArea", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XCopyArea", "Drawable");
  Xen_check_type(Xen_is_Window(arg3), arg3, 3, "XCopyArea", "Drawable");
  Xen_check_type(Xen_is_GC(arg4), arg4, 4, "XCopyArea", "GC");
  Xen_check_type(Xen_is_integer(arg5), arg5, 5, "XCopyArea", "int");
  Xen_check_type(Xen_is_integer(arg6), arg6, 6, "XCopyArea", "int");
  Xen_check_type(Xen_is_ulong(arg7), arg7, 7, "XCopyArea", "unsigned int");
  Xen_check_type(Xen_is_ulong(arg8), arg8, 8, "XCopyArea", "unsigned int");
  Xen_check_type(Xen_is_integer(arg9), arg9, 9, "XCopyArea", "int");
  Xen_check_type(Xen_is_integer(arg10), arg10, 10, "XCopyArea", "int");
  return(C_int_to_Xen_integer(XCopyArea(Xen_to_C_Display(arg1), 
				Xen_to_C_Window(arg2), 
				Xen_to_C_Window(arg3), 
				Xen_to_C_GC(arg4), 
				Xen_integer_to_C_int(arg5), Xen_integer_to_C_int(arg6), Xen_ulong_to_C_ulong(arg7), 
				Xen_ulong_to_C_ulong(arg8), Xen_integer_to_C_int(arg9), Xen_integer_to_C_int(arg10))));
}

static Xen gxm_XConvertSelection(Xen arg1, Xen arg2, Xen arg3, Xen arg4, Xen arg5, Xen arg6)
{
  #define H_XConvertSelection "void XConvertSelection(display, selection, target, property, requestor, time)"
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XConvertSelection", "Display*");
  Xen_check_type(Xen_is_Atom(arg2), arg2, 2, "XConvertSelection", "Atom");
  Xen_check_type(Xen_is_Atom(arg3), arg3, 3, "XConvertSelection", "Atom");
  Xen_check_type(Xen_is_Atom(arg4), arg4, 4, "XConvertSelection", "Atom");
  Xen_check_type(Xen_is_Window(arg5), arg5, 5, "XConvertSelection", "Window");
  Xen_check_type(Xen_is_Time(arg6), arg6, 6, "XConvertSelection", "Time");
  return(C_int_to_Xen_integer(XConvertSelection(Xen_to_C_Display(arg1), 
					Xen_to_C_Atom(arg2), Xen_to_C_Atom(arg3), Xen_to_C_Atom(arg4), 
					Xen_to_C_Window(arg5), Xen_to_C_Time(arg6))));
}

static Xen gxm_XConnectionNumber(Xen arg1)
{
  #define H_XConnectionNumber "returns a connection number for the specified display."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XConnectionNumber", "Display*");
  return(C_int_to_Xen_integer(XConnectionNumber(Xen_to_C_Display(arg1))));
}

static Xen gxm_XConfigureWindow(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XConfigureWindow "XConfigureWindow(display, w, value_mask, values) uses the values specified in the XWindowChanges structure to \
reconfigure a window's size, position, border, and stacking order."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XConfigureWindow", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XConfigureWindow", "Window");
  Xen_check_type(Xen_is_ulong(arg3), arg3, 3, "XConfigureWindow", "unsigned int");
  Xen_check_type(Xen_is_XWindowChanges(arg4), arg4, 4, "XConfigureWindow", "XWindowChanges*");
  return(C_int_to_Xen_integer(XConfigureWindow(Xen_to_C_Display(arg1), 
				       Xen_to_C_Window(arg2), Xen_ulong_to_C_ulong(arg3), Xen_to_C_XWindowChanges(arg4))));
}

static Xen gxm_XCloseDisplay(Xen arg1)
{
  #define H_XCloseDisplay "XCloseDisplay(display) closes the connection to the X server for the display specified in the Display structure and \
destroys all windows, resource IDs etc."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XCloseDisplay", "Display*");
  return(C_int_to_Xen_integer(XCloseDisplay(Xen_to_C_Display(arg1))));
}

static Xen gxm_XClearWindow(Xen arg1, Xen arg2)
{
  #define H_XClearWindow "XClearWindow(display, w) clears the entire area in the specified window."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XClearWindow", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XClearWindow", "Window");
  return(C_int_to_Xen_integer(XClearWindow(Xen_to_C_Display(arg1), Xen_to_C_Window(arg2))));
}

static Xen gxm_XClearArea(Xen arg1, Xen arg2, Xen arg3, Xen arg4, Xen arg5, Xen arg6, Xen arg7)
{
  #define H_XClearArea "XClearArea(display, w, x, y, width, height, exposures) paints a rectangular area in the specified window according to the \
specified dimensions with the window's background pixel or pixmap."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XClearArea", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XClearArea", "Window");
  Xen_check_type(Xen_is_integer(arg3), arg3, 3, "XClearArea", "int");
  Xen_check_type(Xen_is_integer(arg4), arg4, 4, "XClearArea", "int");
  Xen_check_type(Xen_is_ulong(arg5), arg5, 5, "XClearArea", "unsigned int");
  Xen_check_type(Xen_is_ulong(arg6), arg6, 6, "XClearArea", "unsigned int");
  Xen_check_type(Xen_is_boolean(arg7), arg7, 7, "XClearArea", "Bool");
  return(C_int_to_Xen_integer(XClearArea(Xen_to_C_Display(arg1), 
				 Xen_to_C_Window(arg2), 
				 Xen_integer_to_C_int(arg3), Xen_integer_to_C_int(arg4), 
				 Xen_ulong_to_C_ulong(arg5), Xen_ulong_to_C_ulong(arg6), Xen_boolean_to_C_bool(arg7))));
}

static Xen gxm_XCirculateSubwindowsUp(Xen arg1, Xen arg2)
{
  #define H_XCirculateSubwindowsUp "XCirculateSubwindowsUp(display, w) raises the lowest mapped child of the specified window that is partially or \
completely occluded by another child."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XCirculateSubwindowsUp", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XCirculateSubwindowsUp", "Window");
  return(C_int_to_Xen_integer(XCirculateSubwindowsUp(Xen_to_C_Display(arg1), Xen_to_C_Window(arg2))));
}

static Xen gxm_XCirculateSubwindowsDown(Xen arg1, Xen arg2)
{
  #define H_XCirculateSubwindowsDown "XCirculateSubwindowsDown(display, w) lowers the highest mapped child of the specified window that partially or \
completely occludes another child."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XCirculateSubwindowsDown", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XCirculateSubwindowsDown", "Window");
  return(C_int_to_Xen_integer(XCirculateSubwindowsDown(Xen_to_C_Display(arg1), Xen_to_C_Window(arg2))));
}

static Xen gxm_XCirculateSubwindows(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XCirculateSubwindows "XCirculateSubwindows(display, w, direction) circulates children of the specified window in the specified direction."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XCirculateSubwindows", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XCirculateSubwindows", "Window");
  Xen_check_type(Xen_is_integer(arg3), arg3, 3, "XCirculateSubwindows", "int");
  return(C_int_to_Xen_integer(XCirculateSubwindows(Xen_to_C_Display(arg1), Xen_to_C_Window(arg2), Xen_integer_to_C_int(arg3))));
}

static Xen gxm_XCheckWindowEvent(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XCheckWindowEvent "Bool XCheckWindowEvent(display, w, event_mask) searches the event queue and then the events available \
on the server connection for the first event that matches the specified window and event mask."
  /* DIFF: XCheckWindowEvent dpy win mask [ev] -> (list val ev)
   */
  XEvent *e;
  int val;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XCheckWindowEvent", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XCheckWindowEvent", "Window");
  Xen_check_type(Xen_is_integer(arg3), arg3, 3, "XCheckWindowEvent", "long");
  e = (XEvent *)calloc(1, sizeof(XEvent));
  val = XCheckWindowEvent(Xen_to_C_Display(arg1), Xen_to_C_Window(arg2), Xen_integer_to_C_int(arg3), e);
  if (val)
    return(C_to_Xen_XEvent_OBJ(e));
  else
    {
      free(e);
      return(Xen_false);
    }
}

static Xen gxm_XCheckTypedWindowEvent(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XCheckTypedWindowEvent "Bool XCheckTypedWindowEvent(display, w, event_type) searches the event queue and then any events \
available on the server connection for the first event that matches the specified event mask"
  /* DIFF: XCheckTypedWindowEvent dpy win mask [ev] -> (list val ev)
   */
  XEvent *e;
  int val;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XCheckTypedWindowEvent", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XCheckTypedWindowEvent", "Window");
  Xen_check_type(Xen_is_integer(arg3), arg3, 3, "XCheckTypedWindowEvent", "int");
  e = (XEvent *)calloc(1, sizeof(XEvent));
  val = XCheckTypedWindowEvent(Xen_to_C_Display(arg1), Xen_to_C_Window(arg2), Xen_integer_to_C_int(arg3), e);
  if (val)
    return(C_to_Xen_XEvent_OBJ(e));
  else
    {
      free(e);
      return(Xen_false);
    }
}

static Xen gxm_XCheckTypedEvent(Xen arg1, Xen arg2)
{
  #define H_XCheckTypedEvent "Bool XCheckTypedEvent(display, event_type) searches the event queue and then any events available  \
on the server connection for the first event that matches the specified type."
  /* DIFF: XCheckTypedEvent dpy mask [ev] -> (list val ev)
   */
  XEvent *e;
  int val;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XCheckTypedEvent", "Display*");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XCheckTypedEvent", "int");
  e = (XEvent *)calloc(1, sizeof(XEvent));
  val = XCheckTypedEvent(Xen_to_C_Display(arg1), Xen_integer_to_C_int(arg2), e);
  if (val)
    return(C_to_Xen_XEvent_OBJ(e));
  else
    {
      free(e);
      return(Xen_false);
    }
}

static Xen gxm_XCheckMaskEvent(Xen arg1, Xen arg2)
{
  #define H_XCheckMaskEvent "Bool XCheckMaskEvent(display, event_mask) searches the event queue and then any events available on \
the server connection for the first event that matches the specified mask."
  /* DIFF: XCheckMaskEvent dpy mask [ev] -> (list val ev)
   */
  XEvent *e;
  int val;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XCheckMaskEvent", "Display*");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XCheckMaskEvent", "long");
  e = (XEvent *)calloc(1, sizeof(XEvent));
  val = XCheckMaskEvent(Xen_to_C_Display(arg1), Xen_integer_to_C_int(arg2), e);
  if (val)
    return(C_to_Xen_XEvent_OBJ(e));
  else
    {
      free(e);
      return(Xen_false);
    }
}

static Xen gxm_XCheckIfEvent(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XCheckIfEvent "Bool XCheckIfEvent(display, predicate, arg)"
  /* DIFF: XCheckIfEvent dpy [ev] proc ptr -> (list val ev)
   */
  XEvent *e;
  int val;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XCheckIfEvent", "Display*");
  Xen_check_type(Xen_is_procedure(arg2) && (Xen_is_aritable(arg2, 3)), arg2, 2, "XCheckIfEvent", "(Bool_Proc dpy ev data)");
  e = (XEvent *)calloc(1, sizeof(XEvent));
  val = XCheckIfEvent(Xen_to_C_Display(arg1), 
		      e, 
		      (Bool (*)(Display *d, XEvent *ev, char *p))gxm_XPeekIfEventProc, 
		      (char*)arg3);
  if (val)
    return(C_to_Xen_XEvent_OBJ(e));
  else
    {
      free(e);
      return(Xen_false);
    }
}

static Xen gxm_XChangeWindowAttributes(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XChangeWindowAttributes "XChangeWindowAttributes(display, w, valuemask, attributes) uses the window attributes in the XSetWindowAttributes \
structure to change the specified window attributes."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XChangeWindowAttributes", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XChangeWindowAttributes", "Window");
  Xen_check_type(Xen_is_ulong(arg3), arg3, 3, "XChangeWindowAttributes", "ulong");
  Xen_check_type(Xen_is_XSetWindowAttributes(arg4), arg4, 4, "XChangeWindowAttributes", "XSetWindowAttributes*");
  return(C_int_to_Xen_integer(XChangeWindowAttributes(Xen_to_C_Display(arg1), Xen_to_C_Window(arg2), Xen_ulong_to_C_ulong(arg3), 
					      Xen_to_C_XSetWindowAttributes(arg4))));
}

static Xen gxm_XChangeProperty(Xen arg1, Xen arg2, Xen arg3, Xen arg4, Xen arg5, Xen arg6, Xen arg7, Xen arg8)
{
  #define H_XChangeProperty "XChangeProperty(display, w, property, type, format, mode, data, nelements) alters the property for the specified \
window and causes the X server to generate a PropertyNotify event on that window."
  unsigned char *command;
  int len;
  int *data = NULL;
  Xen rtn;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XChangeProperty", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XChangeProperty", "Window");
  Xen_check_type(Xen_is_Atom(arg3), arg3, 3, "XChangeProperty", "Atom");
  Xen_check_type(Xen_is_Atom(arg4), arg4, 4, "XChangeProperty", "Atom");
  Xen_check_type(Xen_is_integer(arg5), arg5, 5, "XChangeProperty", "int");
  Xen_check_type(Xen_is_integer(arg6), arg6, 6, "XChangeProperty", "int");
  Xen_check_type(Xen_is_string(arg7) || Xen_is_list(arg7), arg7, 7, "XChangeProperty", "string or list of int");
  Xen_check_type(Xen_is_integer_or_unbound(arg8), arg8, 8, "XChangeProperty", "int");
  if (Xen_is_string(arg7))
    {
      command = (unsigned char *)(Xen_string_to_C_string(arg7));
      if (Xen_is_integer(arg8)) len = Xen_integer_to_C_int(arg8); else len = strlen((const char *)command) + 1;
    }
  else 
    {
      if (Xen_is_integer(arg8)) len = Xen_integer_to_C_int(arg8); else len = Xen_list_length(arg7);
      data = Xen_to_C_Ints(arg7, len);
      command = (unsigned char *)data;
    }
  rtn = C_int_to_Xen_integer(XChangeProperty(Xen_to_C_Display(arg1), Xen_to_C_Window(arg2), 
				     Xen_to_C_Atom(arg3), Xen_to_C_Atom(arg4), 
				     Xen_integer_to_C_int(arg5), Xen_integer_to_C_int(arg6), 
				     (const unsigned char *)command, len));
  if (data) free(data);
  return(rtn);
}

static Xen gxm_XChangePointerControl(Xen arg1, Xen arg2, Xen arg3, Xen arg4, Xen arg5, Xen arg6)
{
  #define H_XChangePointerControl "XChangePointerControl(display, do_accel, do_threshold, accel_numerator, accel_denominator, threshold) \
defines how the pointing device moves."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XChangePointerControl", "Display*");
  Xen_check_type(Xen_is_boolean(arg2), arg2, 2, "XChangePointerControl", "Bool");
  Xen_check_type(Xen_is_boolean(arg3), arg3, 3, "XChangePointerControl", "Bool");
  Xen_check_type(Xen_is_integer(arg4), arg4, 4, "XChangePointerControl", "int");
  Xen_check_type(Xen_is_integer(arg5), arg5, 5, "XChangePointerControl", "int");
  Xen_check_type(Xen_is_integer(arg6), arg6, 6, "XChangePointerControl", "int");
  return(C_int_to_Xen_integer(XChangePointerControl(Xen_to_C_Display(arg1), 
					    Xen_boolean_to_C_bool(arg2), Xen_boolean_to_C_bool(arg3), 
					    Xen_integer_to_C_int(arg4), Xen_integer_to_C_int(arg5), Xen_integer_to_C_int(arg6))));
}

static Xen gxm_XChangeKeyboardMapping(Xen arg1, Xen arg2, Xen arg3, Xen arg4, Xen arg5)
{
  #define H_XChangeKeyboardMapping "XChangeKeyboardMapping(display, first_keycode, keysyms_per_keycode, keysyms, num_codes) defines the \
symbols for the specified number of KeyCodes starting with first_keycode."
  /* DIFF: XChangeKeyboardMapping takes list of KeySyms
   */
  KeySym *ks;
  int len, val;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XChangeKeyboardMapping", "Display*");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XChangeKeyboardMapping", "int");
  Xen_check_type(Xen_is_integer(arg3), arg3, 3, "XChangeKeyboardMapping", "int");
  Xen_check_type(Xen_is_list(arg4), arg4, 4, "XChangeKeyboardMapping", "list of KeySym");
  Xen_check_type(Xen_is_integer(arg5), arg5, 5, "XChangeKeyboardMapping", "int");
  len = Xen_integer_to_C_int(arg5);
  if (len <= 0) return(Xen_false);
  ks = Xen_to_C_KeySyms(arg4, len);
  val = XChangeKeyboardMapping(Xen_to_C_Display(arg1), Xen_integer_to_C_int(arg2), Xen_integer_to_C_int(arg3), ks, len);
  free(ks);
  return(C_int_to_Xen_integer(val));
}

static Xen gxm_XChangeKeyboardControl(Xen arg1, Xen arg2, Xen larg3)
{
  #define H_XChangeKeyboardControl "XChangeKeyboardControl(display, value_mask, values) controls the keyboard characteristics defined by \
the XKeyboardControl structure."
  /* DIFF: XChangeKeyboardControl arg3 is list of XKeyboardControl fields
   */
  Xen arg3;
  XKeyboardControl kc;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XChangeKeyboardControl", "Display*");
  Xen_check_type(Xen_is_ulong(arg2), arg2, 2, "XChangeKeyboardControl", "ulong");
  Xen_check_type(Xen_is_list(larg3), larg3, 3, "XChangeKeyboardControl", "XKeyboardControl*");
  arg3 = Xen_copy_arg(larg3);
  kc.key_click_percent = Xen_integer_to_C_int(Xen_car(arg3)); arg3 = Xen_cdr(arg3);
  if (!(Xen_is_null(arg3))) 
    {
      kc.bell_percent = Xen_integer_to_C_int(Xen_car(arg3)); arg3 = Xen_cdr(arg3);
      if (!(Xen_is_null(arg3)))
	{
	  kc.bell_pitch = Xen_integer_to_C_int(Xen_car(arg3)); arg3 = Xen_cdr(arg3);
	  if (!(Xen_is_null(arg3)))
	    {
	      kc.bell_duration = Xen_integer_to_C_int(Xen_car(arg3)); arg3 = Xen_cdr(arg3);
	      if (!(Xen_is_null(arg3)))
		{
		  kc.led = Xen_integer_to_C_int(Xen_car(arg3)); arg3 = Xen_cdr(arg3);
		  if (!(Xen_is_null(arg3)))
		    {
		      kc.led_mode = Xen_integer_to_C_int(Xen_car(arg3)); arg3 = Xen_cdr(arg3);
		      if (!(Xen_is_null(arg3)))
			{
			  kc.key = Xen_integer_to_C_int(Xen_car(arg3)); arg3 = Xen_cdr(arg3);
			  if (!(Xen_is_null(arg3)))
			    {
			      kc.auto_repeat_mode = Xen_integer_to_C_int(Xen_car(arg3));
			    }}}}}}}
  return(C_int_to_Xen_integer(XChangeKeyboardControl(Xen_to_C_Display(arg1), Xen_ulong_to_C_ulong(arg2), &kc)));
}

static Xen gxm_XChangeGC(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XChangeGC "XChangeGC(display, gc, valuemask, values) changes the components specified by valuemask for the specified GC."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XChangeGC", "Display*");
  Xen_check_type(Xen_is_GC(arg2), arg2, 2, "XChangeGC", "GC");
  Xen_check_type(Xen_is_ulong(arg3), arg3, 3, "XChangeGC", "ulong");
  Xen_check_type(Xen_is_XGCValues(arg4), arg4, 4, "XChangeGC", "XGCValues*");
  return(C_int_to_Xen_integer(XChangeGC(Xen_to_C_Display(arg1), Xen_to_C_GC(arg2), Xen_ulong_to_C_ulong(arg3), Xen_to_C_XGCValues(arg4))));
}

static Xen gxm_XChangeActivePointerGrab(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XChangeActivePointerGrab "XChangeActivePointerGrab(display, event_mask, cursor, time) changes the specified dynamic parameters \
if the pointer is actively grabbed by the client and if the specified time is no earlier than the last-pointer-grab time and no later than \
the current X server time."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XChangeActivePointerGrab", "Display*");
  Xen_check_type(Xen_is_ulong(arg2), arg2, 2, "XChangeActivePointerGrab", "unsigned int");
  Xen_check_type(Xen_is_Cursor(arg3), arg3, 3, "XChangeActivePointerGrab", "Cursor");
  Xen_check_type(Xen_is_Time(arg4), arg4, 4, "XChangeActivePointerGrab", "Time");
  return(C_int_to_Xen_integer(XChangeActivePointerGrab(Xen_to_C_Display(arg1), Xen_ulong_to_C_ulong(arg2), Xen_to_C_Cursor(arg3), Xen_to_C_Time(arg4))));
}

static Xen gxm_XCellsOfScreen(Xen arg1)
{
  #define H_CellsOfScreen "CellsOfScreen(screen): returns the number of colormap cells in the default colormap of the specified screen."
  Xen_check_type(Xen_is_Screen(arg1), arg1, 1, "XCellsOfScreen", "Screen*");
  return(C_int_to_Xen_integer(XCellsOfScreen(Xen_to_C_Screen(arg1))));
}

static Xen gxm_XBitmapUnit(Xen arg1)
{
  #define H_BitmapUnit "BitmapUnit(display): returns the size of a bitmap's scanline unit in bits."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XBitmapUnit", "Display*");
  return(C_int_to_Xen_integer(XBitmapUnit(Xen_to_C_Display(arg1))));
}

static Xen gxm_XBitmapPad(Xen arg1)
{
  #define H_BitmapPad "BitmapPad(display): returns the number of bits that each scanline must be padded."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XBitmapPad", "Display*");
  return(C_int_to_Xen_integer(XBitmapPad(Xen_to_C_Display(arg1))));
}

static Xen gxm_XBitmapBitOrder(Xen arg1)
{
  #define H_BitmapBitOrder "BitmapBitOrder(display): returns LSBFirst or MSBFirst to indicate whether the leftmost bit in the bitmap as \
displayed on the screen is the least or most significant bit in the unit."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XBitmapBitOrder", "Display*");
  return(C_int_to_Xen_integer(XBitmapBitOrder(Xen_to_C_Display(arg1))));
}

static Xen gxm_XBell(Xen arg1, Xen arg2)
{
  #define H_XBell "XBell(display, percent) rings the bell on the keyboard on the specified display, if possible."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XBell", "Display*");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XBell", "int");
  return(C_int_to_Xen_integer(XBell(Xen_to_C_Display(arg1), Xen_integer_to_C_int(arg2))));
}

static Xen gxm_XAutoRepeatOn(Xen arg1)
{
  #define H_XAutoRepeatOn "XAutoRepeatOn(display) turns on auto-repeat for the keyboard on the specified display."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XAutoRepeatOn", "Display*");
  return(C_int_to_Xen_integer(XAutoRepeatOn(Xen_to_C_Display(arg1))));
}

static Xen gxm_XAutoRepeatOff(Xen arg1)
{
  #define H_XAutoRepeatOff "XAutoRepeatOff(display) turns off auto-repeat for the keyboard on the specified display."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XAutoRepeatOff", "Display*");
  return(C_int_to_Xen_integer(XAutoRepeatOff(Xen_to_C_Display(arg1))));
}

static Xen gxm_XAllowEvents(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XAllowEvents "XAllowEvents(display, event_mode, time)"
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XAllowEvents", "Display*");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XAllowEvents", "int");
  Xen_check_type(Xen_is_Time(arg3), arg3, 3, "XAllowEvents", "Time");
  return(C_int_to_Xen_integer(XAllowEvents(Xen_to_C_Display(arg1), Xen_integer_to_C_int(arg2), Xen_to_C_Time(arg3))));
}

static Xen gxm_XAllocNamedColor(Xen arg1, Xen arg2, Xen arg3, Xen arg4, Xen arg5)
{
  #define H_XAllocNamedColor "Status XAllocNamedColor(display, colormap, color_name, screen_def_return, exact_def_return) looks up the \
named color with respect to the screen that is associated with the specified colormap."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XAllocNamedColor", "Display*");
  Xen_check_type(Xen_is_Colormap(arg2), arg2, 2, "XAllocNamedColor", "Colormap");
  Xen_check_type(Xen_is_string(arg3), arg3, 3, "XAllocNamedColor", "char*");
  Xen_check_type(Xen_is_XColor(arg4), arg4, 4, "XAllocNamedColor", "XColor");
  Xen_check_type(Xen_is_XColor(arg5), arg5, 5, "XAllocNamedColor", "XColor");
  return(C_int_to_Xen_integer(XAllocNamedColor(Xen_to_C_Display(arg1), 
				       Xen_to_C_Colormap(arg2), 
				       (char *)Xen_string_to_C_string(arg3), 
				       Xen_to_C_XColor(arg4), Xen_to_C_XColor(arg5))));
}

static Xen gxm_XAllocColorPlanes(Xen args)
{
  #define H_XAllocColorPlanes "Status XAllocColorPlanes(display, colormap, contig, ncolors, nreds, ngreens, nblues)"
  /* DIFF: XAllocColorPlanes omits pixel array (arg4) and trailing 3 args, returns them and embedded list of pixels
   */
  unsigned long r,g,b;
  unsigned long *ps;
  int len, val;
  Xen lst = Xen_false;
  Xen arg1, arg2, arg3, arg5, arg6, arg7, arg8;
  arg1 = Xen_list_ref(args, 0);
  arg2 = Xen_list_ref(args, 1);
  arg3 = Xen_list_ref(args, 2);
  arg5 = Xen_list_ref(args, 3);
  arg6 = Xen_list_ref(args, 4);
  arg7 = Xen_list_ref(args, 5);
  arg8 = Xen_list_ref(args, 6);
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XAllocColorPlanes", "Display*");
  Xen_check_type(Xen_is_Colormap(arg2), arg2, 2, "XAllocColorPlanes", "Colormap");
  Xen_check_type(Xen_is_boolean(arg3), arg3, 3, "XAllocColorPlanes", "Bool");
  Xen_check_type(Xen_is_integer(arg5), arg5, 5, "XAllocColorPlanes", "int");
  Xen_check_type(Xen_is_integer(arg6), arg6, 6, "XAllocColorPlanes", "int");
  Xen_check_type(Xen_is_integer(arg7), arg7, 7, "XAllocColorPlanes", "int");
  Xen_check_type(Xen_is_integer(arg8), arg8, 8, "XAllocColorPlanes", "int");
  len = Xen_integer_to_C_int(arg5);
  if (len <= 0) Xen_check_type(0, arg5, 5, "XAllocColorPlanes", "positive integer");
  ps = (unsigned long *)calloc(len, sizeof(unsigned long));
  val = XAllocColorPlanes(Xen_to_C_Display(arg1), 
			  Xen_to_C_Colormap(arg2), 
			  Xen_boolean_to_C_bool(arg3), 
			  ps, len, 
			  Xen_integer_to_C_int(arg6), Xen_integer_to_C_int(arg7), Xen_integer_to_C_int(arg8), 
			  &r, &g, &b);
  if (val != 0)
    {
      Xen plist = Xen_empty_list;
      int i, loc;
      loc = xm_protect(plist);
      for (i = len - 1; i >= 0; i--)
	plist = Xen_cons(C_ulong_to_Xen_ulong(ps[i]), plist);
      xm_unprotect_at(loc);
      lst = Xen_list_5(C_int_to_Xen_integer(val),
		       plist,
		       C_ulong_to_Xen_ulong(r),
		       C_ulong_to_Xen_ulong(g),
		       C_ulong_to_Xen_ulong(b));
    }
  free(ps);
  return(lst);
}

static Xen gxm_XAllocColorCells(Xen arg1, Xen arg2, Xen arg3, Xen arg4, Xen arg5)
{
  #define H_XAllocColorCells "Status XAllocColorCells(display, colormap, contig, nplanes, npixels) \
allocates read/write color cells."
  /* DIFF: XAllocColorCells arg 4 and 6 omitted and returned as (embedded) lists 
   */
  int mlen, plen, val;
  Xen mlst = Xen_empty_list, plst = Xen_empty_list;
  unsigned long *ms, *ps;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XAllocColorCells", "Display*");
  Xen_check_type(Xen_is_Colormap(arg2), arg2, 2, "XAllocColorCells", "Colormap");
  Xen_check_type(Xen_is_boolean(arg3), arg3, 3, "XAllocColorCells", "Bool");
  Xen_check_type(Xen_is_ulong(arg4), arg4, 4, "XAllocColorCells", "unsigned int");
  Xen_check_type(Xen_is_ulong(arg5), arg5, 5, "XAllocColorCells", "unsigned int");
  mlen = Xen_integer_to_C_int(arg4);
  if (mlen <= 0) return(Xen_false);
  plen = Xen_integer_to_C_int(arg5);
  if (plen <= 0) return(Xen_false);
  ms = (unsigned long *)calloc(mlen, sizeof(unsigned long));
  ps = (unsigned long *)calloc(plen, sizeof(unsigned long));
  val = XAllocColorCells(Xen_to_C_Display(arg1), 
			 Xen_to_C_Colormap(arg2), 
			 Xen_boolean_to_C_bool(arg3), 
			 ms, mlen, 
			 ps, plen);
  if (val != 0)
    {
      int i, loc1, loc2;
      loc1 = xm_protect(mlst);
      loc2 = xm_protect(plst);
      for (i = mlen - 1; i >= 0; i--) 
	mlst = Xen_cons(C_ulong_to_Xen_ulong(ms[i]), mlst);
      for (i = plen - 1; i >= 0; i--) 
	mlst = Xen_cons(C_ulong_to_Xen_ulong(ps[i]), plst);
      xm_unprotect_at(loc1);
      xm_unprotect_at(loc2);
    }
  free(ms);
  free(ps);
  if (val != 0)
    return(Xen_list_3(C_int_to_Xen_integer(val),
		      mlst,
		      plst));
  return(Xen_false);
}

static Xen gxm_XAllocColor(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XAllocColor "Status XAllocColor(display, colormap, screen_in_out) allocates a read-only colormap entry corresponding to the \
closest RGB value supported by the hardware."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XAllocColor", "Display*");
  Xen_check_type(Xen_is_Colormap(arg2), arg2, 2, "XAllocColor", "Colormap");
  Xen_check_type(Xen_is_XColor(arg3), arg3, 3, "XAllocColor", "XColor"); 
  return(C_int_to_Xen_integer(XAllocColor(Xen_to_C_Display(arg1), Xen_to_C_Colormap(arg2), Xen_to_C_XColor(arg3))));
}

static Xen gxm_XActivateScreenSaver(Xen arg1)
{
  #define H_XActivateScreenSaver "XActivateScreenSaver(display) activates the screen saver."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XActivateScreenSaver", "Display*");
  return(C_int_to_Xen_integer(XActivateScreenSaver(Xen_to_C_Display(arg1))));
}

static Xen gxm_XSetTransientForHint(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XSetTransientForHint "XSetTransientForHint(display, w, prop_window) sets the WM_TRANSIENT_FOR property of the specified \
window to the specified prop_window."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XSetTransientForHint", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XSetTransientForHint", "Window");
  Xen_check_type(Xen_is_Window(arg3), arg3, 3, "XSetTransientForHint", "Window");
  return(C_int_to_Xen_integer(XSetTransientForHint(Xen_to_C_Display(arg1), Xen_to_C_Window(arg2), Xen_to_C_Window(arg3))));
}

static Xen gxm_XSetWMColormapWindows(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XSetWMColormapWindows "Status XSetWMColormapWindows(display, w, colormap_windows, count) replaces the WM_COLORMAP_WINDOWS property \
on the specified window with the list of windows specified by the colormap_windows argument."
  /* DIFF: XSetWMColormapWindows arg 3 is list of Windows
   */
  int len, rtn;
  Window *ws;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XSetWMColormapWindows", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XSetWMColormapWindows", "Window");
  Xen_check_type(Xen_is_list(arg3), arg3, 3, "XSetWMColormapWindows", "list of Windows");
  Xen_check_type(Xen_is_integer(arg4), arg4, 4, "XSetWMColormapWindows", "int");
  len = Xen_integer_to_C_int(arg4);
  if (len <= 0) return(Xen_false);
  ws = Xen_to_C_Windows(arg3, len);
  rtn = XSetWMColormapWindows(Xen_to_C_Display(arg1), 
			      Xen_to_C_Window(arg2),
			      ws, len);
  free(ws);
  return(C_int_to_Xen_integer(rtn));
}

static Xen gxm_XGetWMColormapWindows(Xen arg1, Xen arg2)
{
  #define H_XGetWMColormapWindows "Status XGetWMColormapWindows(display, w): returns the list of \
window identifiers stored in the WM_COLORMAP_WINDOWS property on the specified window."
  /* DIFF: XGetWMColormapWindows omit last 2 args, return list of windows
   */
  Xen lst = Xen_empty_list;
  int i, len, rtn, loc;
  Window *ws;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XGetWMColormapWindows", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XGetWMColormapWindows", "Window");
  rtn = XGetWMColormapWindows(Xen_to_C_Display(arg1), 
			      Xen_to_C_Window(arg2), 
			      &ws, &len);
  if (rtn == 0)
    return(Xen_false);
  loc = xm_protect(lst);
  for (i = len - 1; i >= 0; i--)
    lst = Xen_cons(C_to_Xen_Window(ws[i]), lst);
  xm_unprotect_at(loc);
  XFree(ws);
  return(lst);
}

static Xen gxm_XGetCommand(Xen arg1, Xen arg2)
{
  #define H_XGetCommand "Status XGetCommand(display, w) reads the WM_COMMAND property from the specified window \
and returns a string list."
  /* DIFF: XGetCommand omits last 2 args, returns list
   */
  char **argv;
  int argc;
  Xen lst = Xen_empty_list;
  Status err;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XGetCommand", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XGetCommand", "Window");
  err = XGetCommand(Xen_to_C_Display(arg1), Xen_to_C_Window(arg2), &argv, &argc);
  if (err != 0)
    {
      int i, loc;
      loc = xm_protect(lst);
      for (i = argc - 1; i >= 0; i--)
	lst = Xen_cons(C_string_to_Xen_string(argv[i]), lst);
      XFreeStringList(argv);
      xm_unprotect_at(loc);
    }
  return(lst);
}

static Xen gxm_XWithdrawWindow(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XWithdrawWindow "Status XWithdrawWindow(display, w, screen_number) unmaps the specified window and sends a synthetic  \
UnmapNotify event to the root window of the specified screen."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XWithdrawWindow", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XWithdrawWindow", "Window");
  Xen_check_type(Xen_is_integer(arg3), arg3, 3, "XWithdrawWindow", "int");
  return(C_int_to_Xen_integer(XWithdrawWindow(Xen_to_C_Display(arg1), Xen_to_C_Window(arg2), Xen_integer_to_C_int(arg3))));
}

static Xen gxm_XIconifyWindow(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XIconifyWindow "Status XIconifyWindow(display, w, screen_number) sends a WM_CHANGE_STATE  ClientMessage event with a format of \
32 and a first data element of IconicState to the root window of the specified screen with an event mask set to \
SubstructureNotifyMask | SubstructureRedirectMask."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XIconifyWindow", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XIconifyWindow", "Window");
  Xen_check_type(Xen_is_integer(arg3), arg3, 3, "XIconifyWindow", "int");
  return(C_int_to_Xen_integer(XIconifyWindow(Xen_to_C_Display(arg1), Xen_to_C_Window(arg2), Xen_integer_to_C_int(arg3))));
}

static Xen gxm_XSetWMProtocols(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XSetWMProtocols "Status XSetWMProtocols(display, w, protocols, count) replaces the WM_PROTOCOLS property on the \
specified window with the list of atoms specified by the protocols argument."
  /* DIFF: XSetWMProtocols arg3 is list of atoms
   */
  Atom *outs;
  int val, len;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XSetWMProtocols", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XSetWMProtocols", "Window");
  Xen_check_type(Xen_is_list(arg3), arg3, 3, "XSetWMProtocols", "list of Atom");
  Xen_check_type(Xen_is_integer(arg4), arg4, 4, "XSetWMProtocols", "int");
  len = Xen_integer_to_C_int(arg4);
  if (len <= 0) return(Xen_false);
  outs = Xen_to_C_Atoms(arg3, len);
  val = XSetWMProtocols(Xen_to_C_Display(arg1), Xen_to_C_Window(arg2), outs, len);
  free(outs);
  return(C_int_to_Xen_integer(val));
}

static Xen gxm_XGetWMProtocols(Xen arg1, Xen arg2)
{
  #define H_XGetWMProtocols "Status XGetWMProtocols(display, w): returns the list of atoms stored in the \
WM_PROTOCOLS property on the specified window."
  /* DIFF: XGetWMProtocols omits last 2 args, returns list
   */
  Atom *ats;
  int len, i, val, loc;
  Xen lst = Xen_empty_list;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XGetWMProtocols", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XGetWMProtocols", "Window");
  val = XGetWMProtocols(Xen_to_C_Display(arg1), Xen_to_C_Window(arg2), &ats, &len);
  if (val == 0)
    return(Xen_false);
  loc = xm_protect(lst);
  for (i = len - 1; i >= 0; i--)
    lst = Xen_cons(C_to_Xen_Atom(ats[i]), lst);
  xm_unprotect_at(loc);
  XFree((void *)ats);
  return(lst);
}

static Xen gxm_XReconfigureWMWindow(Xen arg1, Xen arg2, Xen arg3, Xen arg4, Xen arg5)
{
  #define H_XReconfigureWMWindow "Status XReconfigureWMWindow(display, w, screen_number, value_mask, values) issues a  ConfigureWindow \
request on the specified top-level window."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XReconfigureWMWindow", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XReconfigureWMWindow", "Window");
  Xen_check_type(Xen_is_integer(arg3), arg3, 3, "XReconfigureWMWindow", "int");
  Xen_check_type(Xen_is_ulong(arg4), arg4, 4, "XReconfigureWMWindow", "unsigned int");
  Xen_check_type(Xen_is_XWindowChanges(arg5), arg5, 5, "XReconfigureWMWindow", "XWindowChanges*");
  return(C_int_to_Xen_integer(XReconfigureWMWindow(Xen_to_C_Display(arg1), Xen_to_C_Window(arg2), Xen_integer_to_C_int(arg3), Xen_ulong_to_C_ulong(arg4), 
					   Xen_to_C_XWindowChanges(arg5))));
}

static Xen gxm_XListDepths(Xen arg1, Xen arg2)
{
  #define H_XListDepths "int *XListDepths(display, screen_number): returns the array of depths that are available on the \
specified screen."
  /* DIFF: XListDepths omits last arg, returns list of depths
   */
  Xen lst = Xen_empty_list;
  int i, len, loc;
  int *ds;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XListDepths", "Display*");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XListDepths", "int");
  ds = XListDepths(Xen_to_C_Display(arg1), Xen_integer_to_C_int(arg2), &len);
  loc = xm_protect(lst);
  for (i = len - 1; i >= 0; i--)
    lst = Xen_cons(C_int_to_Xen_integer(ds[i]), lst);
  XFree(ds);
  xm_unprotect_at(loc);
  return(lst);
}

static Xen gxm_XListPixmapFormats(Xen arg1)
{
  #define H_XListPixmapFormats "XPixmapFormatValues *XListPixmapFormats(display): returns an array of XPixmapFormatValues \
structures that describe the types of Z format images supported by the specified display."
  /* DIFF: XListPixmapFormats omits arg2, rtns list of lists, each holding XPixmapFormatValues data
   */
  XPixmapFormatValues *ps;
  int len, loc;
  Xen lst = Xen_empty_list;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XListPixmapFormats", "Display*");
  ps = XListPixmapFormats(Xen_to_C_Display(arg1), &len);
  loc = xm_protect(lst);
  if (ps)
    {
      int i;
      for (i = len - 1; i >= 0; i--)
	lst = Xen_cons(Xen_list_3(C_int_to_Xen_integer(ps[i].depth),
				  C_int_to_Xen_integer(ps[i].bits_per_pixel),
				  C_int_to_Xen_integer(ps[i].scanline_pad)),
		       lst);
      free(ps);
    }
  xm_unprotect_at(loc);
  return(lst);
}


static Xen xm_XIOErrorHandler;
static int gxm_XIOErrorHandler(Display *dpy)
{
  Xen_call_with_1_arg(xm_XIOErrorHandler, C_to_Xen_Display(dpy), __func__);
  return(0); /* never happens */
}

static Xen gxm_XSetIOErrorHandler(Xen arg1)
{
  #define H_XSetIOErrorHandler "int (*XSetIOErrorHandler(handler))() sets the fatal I/O error handler. "
  Xen old_val;
  Xen_check_type(Xen_is_false(arg1) || Xen_is_procedure(arg1), arg1, 1, "XSetIOErrorHandler", PROC_FALSE "=null or function of 1 arg");
  xm_protect(arg1);
  old_val = xm_XIOErrorHandler;
  xm_XIOErrorHandler = arg1;
  if (Xen_is_false(arg1))
    XSetIOErrorHandler(NULL);
  else XSetIOErrorHandler(gxm_XIOErrorHandler);
  if (Xen_is_procedure(old_val)) xm_unprotect(old_val); /* hmmm... what if we're gc'd on the way back? */
  return(old_val);
}

static Xen xm_XErrorHandler;

static int gxm_XErrorHandler(Display *dpy, XErrorEvent *e)
{
  Xen_call_with_2_args(xm_XErrorHandler, C_to_Xen_Display(dpy), C_to_Xen_XErrorEvent((XErrorEvent *)e), __func__);
  return(0); /* never happens */
}

static Xen gxm_XSetErrorHandler(Xen arg1)
{
  #define H_XSetErrorHandler "XSetErrorHandler(proc) causes proc to be called if an error occurs"
  Xen old_val;
  Xen_check_type(Xen_is_false(arg1) || Xen_is_procedure(arg1), arg1, 1, "XSetErrorHandler", PROC_FALSE "=null or function of 2 args");
  xm_protect(arg1);
  old_val = xm_XErrorHandler;
  xm_XErrorHandler = arg1;
  if (Xen_is_false(arg1))
    XSetErrorHandler(NULL);
  else XSetErrorHandler(gxm_XErrorHandler);
  if (Xen_is_procedure(old_val)) xm_unprotect(old_val); /* hmmm... what if we're gc'd on the way back? */
  return(old_val);
}

static Xen gxm_XScreenNumberOfScreen(Xen arg1)
{
  #define H_XScreenNumberOfScreen "int XScreenNumberOfScreen(screen): returns the screen index number of the specified screen."
  Xen_check_type(Xen_is_Screen(arg1), arg1, 1, "XScreenNumberOfScreen", "Screen*");
  return(C_int_to_Xen_integer(XScreenNumberOfScreen(Xen_to_C_Screen(arg1))));
}

static Xen gxm_XEventMaskOfScreen(Xen arg1)
{
  #define H_EventMaskOfScreen "EventMaskOfScreen(screen): returns the root event mask of the root window for the specified screen at connection setup."
  Xen_check_type(Xen_is_Screen(arg1), arg1, 1, "XEventMaskOfScreen", "Screen*");
  return(C_ulong_to_Xen_ulong(XEventMaskOfScreen(Xen_to_C_Screen(arg1))));
}

static Xen gxm_XDefaultScreenOfDisplay(Xen arg1)
{
  /* #define DefaultScreenOfDisplay(display) */
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XDefaultScreenOfDisplay", "Display*");
  return(C_to_Xen_Screen(XDefaultScreenOfDisplay(Xen_to_C_Display(arg1))));
}

static Xen gxm_XScreenOfDisplay(Xen arg1, Xen arg2)
{
  #define H_ScreenOfDisplay "ScreenOfDisplay(display, screen_number): returns a pointer to the screen of the specified display."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XScreenOfDisplay", "Display*");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XScreenOfDisplay", "int");
  return(C_to_Xen_Screen(XScreenOfDisplay(Xen_to_C_Display(arg1), Xen_integer_to_C_int(arg2))));
}

static Xen gxm_XDisplayOfScreen(Xen arg1)
{
  #define H_DisplayOfScreen "DisplayOfScreen(screen): returns the display of the specified screen."
  Xen_check_type(Xen_is_Screen(arg1), arg1, 1, "XDisplayOfScreen", "Screen*");
  return(C_to_Xen_Display(XDisplayOfScreen(Xen_to_C_Screen(arg1))));
}

static Xen gxm_XDefaultColormapOfScreen(Xen arg1)
{
  #define H_DefaultColormapOfScreen "DefaultColormapOfScreen(screen): returns the default colormap of the specified screen."
  Xen_check_type(Xen_is_Screen(arg1), arg1, 1, "XDefaultColormapOfScreen", "Screen*");
  return(C_to_Xen_Colormap(XDefaultColormapOfScreen(Xen_to_C_Screen(arg1))));
}

static Xen gxm_XDefaultColormap(Xen arg1, Xen arg2)
{
  #define H_DefaultColormap "DefaultColormap(display, screen_number): returns the default colormap ID for allocation on the specified screen."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XDefaultColormap", "Display*");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XDefaultColormap", "int");
  return(C_to_Xen_Colormap(XDefaultColormap(Xen_to_C_Display(arg1), Xen_integer_to_C_int(arg2))));
}

static Xen gxm_XDisplayString(Xen arg1)
{
  #define H_DisplayString "DisplayString(display): returns the string that was passed to XOpenDisplay when the current display was opened."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XDisplayString", "Display*");
  return(C_string_to_Xen_string(XDisplayString(Xen_to_C_Display(arg1))));
}

static Xen gxm_XServerVendor(Xen arg1)
{
  #define H_ServerVendor "ServerVendor(display): returns a pointer to a null-terminated string that provides some identification of the \
owner of the X server implementation."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XServerVendor", "Display*");
  return(C_string_to_Xen_string(XServerVendor(Xen_to_C_Display(arg1))));
}

static Xen gxm_XLastKnownRequestProcessed(Xen arg1)
{
  #define H_LastKnownRequestProcessed "LastKnownRequestProcessed(display) extracts the full serial number of the last request known by Xlib \
to have been processed by the X server."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XLastKnownRequestProcessed", "Display*");
  return(C_ulong_to_Xen_ulong(XLastKnownRequestProcessed(Xen_to_C_Display(arg1))));
}

static Xen gxm_XNextRequest(Xen arg1)
{
  #define H_NextRequest "NextRequest(display) extracts the full serial number that is to be used for the next request."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XNextRequest", "Display*");
  return(C_ulong_to_Xen_ulong(XNextRequest(Xen_to_C_Display(arg1))));
}

static Xen gxm_XWhitePixelOfScreen(Xen arg1)
{
  #define H_WhitePixelOfScreen "WhitePixelOfScreen(screen): returns the white pixel value of the specified screen."
  Xen_check_type(Xen_is_Screen(arg1), arg1, 1, "XWhitePixelOfScreen", "Screen*");
  return(C_to_Xen_Pixel(XWhitePixelOfScreen(Xen_to_C_Screen(arg1))));
}

static Xen gxm_XBlackPixelOfScreen(Xen arg1)
{
  #define H_BlackPixelOfScreen "BlackPixelOfScreen(screen): returns the black pixel value of the specified screen."
  Xen_check_type(Xen_is_Screen(arg1), arg1, 1, "XBlackPixelOfScreen", "Screen*");
  return(C_to_Xen_Pixel(XBlackPixelOfScreen(Xen_to_C_Screen(arg1))));
}

static Xen gxm_XAllPlanes(void)
{
  #define H_AllPlanes "Allplanes(): returns a value with all bits set to 1 suitable for use in a plane argument to a procedure."
  return(C_ulong_to_Xen_ulong(XAllPlanes()));
}

static Xen gxm_XWhitePixel(Xen arg1, Xen arg2)
{
  #define H_WhitePixel "WhitePixel(display, screen_number): returns the white pixel value for the specified screen."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XWhitePixel", "Display*");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XWhitePixel", "int");
  return(C_to_Xen_Pixel(XWhitePixel(Xen_to_C_Display(arg1), Xen_integer_to_C_int(arg2))));
}

static Xen gxm_XBlackPixel(Xen arg1, Xen arg2)
{
  #define H_BlackPixel "BlackPixel(display, screen_number): returns the black pixel value for the specified screen."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XBlackPixel", "Display*");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XBlackPixel", "int");
  return(C_to_Xen_Pixel(XBlackPixel(Xen_to_C_Display(arg1), Xen_integer_to_C_int(arg2))));
}

static Xen gxm_XDefaultGCOfScreen(Xen arg1)
{
  #define H_DefaultGCOfScreen "DefaultGCOfScreen(screen): returns the default GC of the specified screen, which has the same depth as the root \
window of the screen."
  Xen_check_type(Xen_is_Screen(arg1), arg1, 1, "XDefaultGCOfScreen", "Screen*");
  return(C_to_Xen_GC(XDefaultGCOfScreen(Xen_to_C_Screen(arg1))));
}

static Xen gxm_XDefaultGC(Xen arg1, Xen arg2)
{
  #define H_DefaultGC "DefaultGC(display, screen_number): returns the default GC for the root window of the "
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XDefaultGC", "Display*");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XDefaultGC", "int");
  return(C_to_Xen_GC(XDefaultGC(Xen_to_C_Display(arg1), Xen_integer_to_C_int(arg2))));
}

static Xen gxm_XDefaultVisualOfScreen(Xen arg1)
{
  #define H_DefaultVisualOfScreen "DefaultVisualOfScreen(screen): returns the default visual of the specified screen."
  Xen_check_type(Xen_is_Screen(arg1), arg1, 1, "XDefaultVisualOfScreen", "Screen*");
  return(C_to_Xen_Visual(XDefaultVisualOfScreen(Xen_to_C_Screen(arg1))));
}

static Xen gxm_XDefaultVisual(Xen arg1, Xen arg2)
{
  #define H_DefaultVisual "DefaultVisual(display, screen_number): returns the default visual type for the specified screen."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XDefaultVisual", "Display*");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XDefaultVisual", "int");
  return(C_to_Xen_Visual(XDefaultVisual(Xen_to_C_Display(arg1), Xen_integer_to_C_int(arg2))));
}

static Xen gxm_XRootWindowOfScreen(Xen arg1)
{
  #define H_RootWindowOfScreen "RootWindowOfScreen(screen): returns the root window of the specified screen."
  Xen_check_type(Xen_is_Screen(arg1), arg1, 1, "XRootWindowOfScreen", "Screen*");
  return(C_to_Xen_Window(XRootWindowOfScreen(Xen_to_C_Screen(arg1))));
}

static Xen gxm_XDefaultRootWindow(Xen arg1)
{
  #define H_DefaultRootWindow "DefaultRootWindow(display): returns the root window for the default screen."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XDefaultRootWindow", "Display*");
  return(C_to_Xen_Window(XDefaultRootWindow(Xen_to_C_Display(arg1))));
}

static Xen gxm_XRootWindow(Xen arg1, Xen arg2)
{
  #define H_RootWindow "RootWindow(display, screen_number): returns the root window."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XRootWindow", "Display*");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XRootWindow", "int");
  return(C_to_Xen_Window(XRootWindow(Xen_to_C_Display(arg1), Xen_integer_to_C_int(arg2))));
}

static Xen gxm_XVisualIDFromVisual(Xen arg1)
{
  #define H_XVisualIDFromVisual "VisualID XVisualIDFromVisual(visual): returns the visual ID for the specified visual type."
  Xen_check_type(Xen_is_Visual(arg1), arg1, 1, "XVisualIDFromVisual", "Visual*");
  return(C_ulong_to_Xen_ulong(XVisualIDFromVisual(Xen_to_C_Visual(arg1))));
}

static Xen gxm_XDisplayMotionBufferSize(Xen arg1)
{
  #define H_XDisplayMotionBufferSize "unsigned long XDisplayMotionBufferSize(display)"
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XDisplayMotionBufferSize", "Display*");
  return(C_ulong_to_Xen_ulong(XDisplayMotionBufferSize(Xen_to_C_Display(arg1))));
}

static Xen gxm_XExtendedMaxRequestSize(Xen arg1)
{
  #define H_XExtendedMaxRequestSize "long XExtendedMaxRequestSize(display): returns zero if the specified display does not support an \
extended-length protocol encoding; otherwise, it returns the maximum request size (in 4-byte units) supported by the server using the \
extended-length encoding."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XExtendedMaxRequestSize", "Display*");
  return(C_int_to_Xen_integer(XExtendedMaxRequestSize(Xen_to_C_Display(arg1))));
}

static Xen gxm_XMaxRequestSize(Xen arg1)
{
  #define H_XMaxRequestSize "long XMaxRequestSize(display): returns the maximum request size (in 4-byte units) supported by the server \
without using an extended-length protocol encoding."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XMaxRequestSize", "Display*");
  return(C_int_to_Xen_integer(XMaxRequestSize(Xen_to_C_Display(arg1))));
}

static Xen gxm_XStringToKeysym(Xen arg1)
{
  #define H_XStringToKeysym "KeySym XStringToKeysym(string)"
  Xen_check_type(Xen_is_string(arg1), arg1, 1, "XStringToKeysym", "char*");
  return(C_to_Xen_KeySym(XStringToKeysym((char *)Xen_string_to_C_string(arg1))));
}

static Xen gxm_XGetKeyboardMapping(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XGetKeyboardMapping "KeySym *XGetKeyboardMapping(display, first_keycode, keycode_count): returns \
the symbols for the specified number of KeyCodes starting with first_keycode."
  /* DIFF: XGetKeyboardMapping omits last arg, returns list of keys
   */
  int n, i, len, count, loc;
  KeySym *keys;
  Xen lst = Xen_empty_list;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XGetKeyboardMapping", "Display*");
  Xen_check_type(Xen_is_KeyCode(arg2), arg2, 2, "XGetKeyboardMapping", "KeyCode");
  Xen_check_type(Xen_is_integer(arg3), arg3, 3, "XGetKeyboardMapping", "int");
  count = Xen_integer_to_C_int(arg3);
  keys = XGetKeyboardMapping(Xen_to_C_Display(arg1), Xen_to_C_KeyCode(arg2), count, &n);
  len = count * n;
  loc = xm_protect(lst);
  for (i = len - 1; i >= 0; i--)
    lst = Xen_cons(C_to_Xen_KeySym(keys[i]), lst);
  XFree(keys);
  xm_unprotect_at(loc);
  return(lst);
}

static Xen gxm_XLookupKeysym(Xen arg1, Xen arg2)
{
  #define H_XLookupKeysym "KeySym XLookupKeysym(key_event, index) uses a given keyboard event and the index you specified to return \
the KeySym from the list that corresponds to the KeyCode member in the XKeyPressedEvent or XKeyReleasedEvent structure."
  Xen_check_type(Xen_is_XKeyEvent(arg1), arg1, 1, "XLookupKeysym", "XKeyEvent*");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XLookupKeysym", "int");
  return(C_to_Xen_KeySym(XLookupKeysym(Xen_to_C_XKeyEvent(arg1), Xen_integer_to_C_int(arg2))));
}

#if 0
/* this has been deprecated for XkbKeycodeToKeysym */
static Xen gxm_XKeycodeToKeysym(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XKeycodeToKeysym "KeySym XKeycodeToKeysym(display, keycode, index) uses internal Xlib tables and returns the KeySym defined \
for the specified KeyCode and the element of the KeyCode vector."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XKeycodeToKeysym", "Display*");
  Xen_check_type(Xen_is_KeyCode(arg2), arg2, 2, "XKeycodeToKeysym", "KeyCode");
  Xen_check_type(Xen_is_integer(arg3), arg3, 3, "XKeycodeToKeysym", "int");
  return(C_to_Xen_KeySym(XKeycodeToKeysym(Xen_to_C_Display(arg1), Xen_to_C_KeyCode(arg2), Xen_integer_to_C_int(arg3))));
}
#endif

static Xen gxm_XListProperties(Xen arg1, Xen arg2)
{
  #define H_XListProperties "Atom *XListProperties(display, w): returns a pointer to an array of atom properties that \
are defined for the specified window or returns NULL if no properties were found."
  /* DIFF: XListProperties returns list, no arg3
   */
  int i, len, loc;
  Xen lst = Xen_empty_list;
  Atom *ats;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XListProperties", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XListProperties", "Window");
  ats = XListProperties(Xen_to_C_Display(arg1), Xen_to_C_Window(arg2), &len);
  if (ats == NULL)
    return(Xen_false);
  loc = xm_protect(lst);
  for (i = len - 1; i >= 0; i--)
    lst = Xen_cons(C_to_Xen_Atom(ats[i]), lst);
  xm_unprotect_at(loc);
  XFree(ats);
  return(lst);
}

static Xen gxm_XListExtensions(Xen arg1)
{
  #define H_XListExtensions "XListExtensions(dpy): list of strings describing available extensions"
  /* DIFF: XListExtensions omits arg2, returns list
   */
  int i, len, loc;
  Xen lst = Xen_empty_list;
  char **str;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XListExtensions", "Display*");
  str = XListExtensions(Xen_to_C_Display(arg1), &len);
  loc = xm_protect(lst);
  for (i = len - 1; i >= 0; i--)
    lst = Xen_cons(C_string_to_Xen_string(str[i]), lst);
  XFreeExtensionList(str);
  xm_unprotect_at(loc);
  return(lst);
}

static Xen gxm_XGetFontPath(Xen arg1)
{
  #define H_XGetFontPath "char **XGetFontPath(display) allocates and returns an array of strings containing the search path."
  /* DIFF: XGetFontPath omits arg2, returns list
   */
  int i, len, loc;
  Xen lst = Xen_empty_list;
  char **str;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XGetFontPath", "Display*");
  str = XGetFontPath(Xen_to_C_Display(arg1), &len);
  loc = xm_protect(lst);
  for (i = len - 1; i >= 0; i--)
    lst = Xen_cons(C_string_to_Xen_string(str[i]), lst);
  XFreeFontPath(str);
  xm_unprotect_at(loc);
  return(lst);
}

static Xen gxm_XListFontsWithInfo(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XListFontsWithInfo "char **XListFontsWithInfo(display, pattern, maxnames): returns a list of \
font names that match the specified pattern and their associated font information."
  /* DIFF: XListFontsWithInfo omit last 2 args, returns list of lists
   */
  int i, count, loc;
  XFontStruct *info;
  char **val;
  Xen lst = Xen_empty_list;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XListFontsWithInfo", "Display*");
  Xen_check_type(Xen_is_string(arg2), arg2, 2, "XListFontsWithInfo", "char*");
  Xen_check_type(Xen_is_integer(arg3), arg3, 3, "XListFontsWithInfo", "int");
  val = XListFontsWithInfo(Xen_to_C_Display(arg1), (char *)Xen_string_to_C_string(arg2), Xen_integer_to_C_int(arg3), &count, &info);
  loc = xm_protect(lst);
  for (i = count - 1; i >= 0; i--)
    lst = Xen_cons(Xen_list_2(C_string_to_Xen_string(val[i]), 
			      C_to_Xen_XFontStruct(&(info[i]))),
		   lst);
  XFreeFontInfo(val, info, count);
  /* XFreeFontNames(val); */
  xm_unprotect_at(loc);
  return(lst);
}

static Xen gxm_XListFonts(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XListFonts "char **XListFonts(display, pattern, maxnames): returns an array of available font names that match \
the string you passed to the pattern argument."
  /* DIFF: XListFonts omits arg4, returns list
   */
  int i, len, loc;
  Xen lst = Xen_empty_list;
  char **str;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XListFonts", "Display*");
  Xen_check_type(Xen_is_string(arg2), arg2, 2, "XListFonts", "char*");
  Xen_check_type(Xen_is_integer(arg3), arg3, 3, "XListFonts", "int");
  str = XListFonts(Xen_to_C_Display(arg1), (char *)Xen_string_to_C_string(arg2), Xen_integer_to_C_int(arg3), &len);
  loc = xm_protect(lst);
  for (i = len - 1; i >= 0; i--)
    lst = Xen_cons(C_string_to_Xen_string(str[i]), lst);
  XFreeFontNames(str);
  xm_unprotect_at(loc);
  return(lst);
}

static Xen gxm_XListInstalledColormaps(Xen arg1, Xen arg2)
{
  #define H_XListInstalledColormaps "Colormap *XListInstalledColormaps(display, w): returns a list of the currently installed \
colormaps for the screen of the specified window."
  /* DIFF: XListInstalledColormaps omits last arg, rtns list of XColormaps
   */
  Colormap *cm;
  int i, len, loc;
  Xen lst = Xen_empty_list;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XListInstalledColormaps", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XListInstalledColormaps", "Window");
  cm = XListInstalledColormaps(Xen_to_C_Display(arg1), Xen_to_C_Window(arg2), &len);
  loc = xm_protect(lst);
  for (i = len - 1; i >= 0; i--)
    lst = Xen_cons(C_to_Xen_Colormap(cm[i]), lst);
  free(cm);
  xm_unprotect_at(loc);
  return(lst);
}

static Xen gxm_XCreateWindow(Xen args)
{
  #define H_XCreateWindow "Window XCreateWindow(display, parent, x, y, width, height, border_width, depth, class, visual, valuemask, attributes) \
creates an unmapped subwindow for a specified parent window, returns the window ID of the created window, and causes the X server to generate \
a CreateNotify event."
  Xen arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8, arg9, arg10, arg11, arg12;
  arg1 = Xen_list_ref(args, 0);
  arg2 = Xen_list_ref(args, 1);
  arg3 = Xen_list_ref(args, 2);
  arg4 = Xen_list_ref(args, 3);
  arg5 = Xen_list_ref(args, 4);
  arg6 = Xen_list_ref(args, 5);
  arg7 = Xen_list_ref(args, 6);
  arg8 = Xen_list_ref(args, 7);
  arg9 = Xen_list_ref(args, 8);
  arg10 = Xen_list_ref(args, 9);
  arg11 = Xen_list_ref(args, 10);
  arg12 = Xen_list_ref(args, 11);
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XCreateWindow", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XCreateWindow", "Window");
  Xen_check_type(Xen_is_integer(arg3), arg3, 3, "XCreateWindow", "int");
  Xen_check_type(Xen_is_integer(arg4), arg4, 4, "XCreateWindow", "int");
  Xen_check_type(Xen_is_ulong(arg5), arg5, 5, "XCreateWindow", "unsigned int");
  Xen_check_type(Xen_is_ulong(arg6), arg6, 6, "XCreateWindow", "unsigned int");
  Xen_check_type(Xen_is_ulong(arg7), arg7, 7, "XCreateWindow", "unsigned int");
  Xen_check_type(Xen_is_integer(arg8), arg8, 8, "XCreateWindow", "int");
  Xen_check_type(Xen_is_ulong(arg9), arg9, 9, "XCreateWindow", "unsigned int");
  Xen_check_type(Xen_is_Visual(arg10), arg10, 10, "XCreateWindow", "Visual*");
  Xen_check_type(Xen_is_ulong(arg11), arg11, 11, "XCreateWindow", "ulong");
  Xen_check_type(Xen_is_XSetWindowAttributes(arg12), arg12, 12, "XCreateWindow", "XSetWindowAttributes*");
  return(C_to_Xen_Window(XCreateWindow(Xen_to_C_Display(arg1), 
				       Xen_to_C_Window(arg2), 
				       Xen_integer_to_C_int(arg3), Xen_integer_to_C_int(arg4), 
				       Xen_ulong_to_C_ulong(arg5), Xen_ulong_to_C_ulong(arg6), 
				       Xen_ulong_to_C_ulong(arg7), Xen_integer_to_C_int(arg8), 
				       Xen_ulong_to_C_ulong(arg9), Xen_to_C_Visual(arg10), 
				       Xen_ulong_to_C_ulong(arg11), 
				       Xen_to_C_XSetWindowAttributes(arg12))));
}

static Xen gxm_XGetSelectionOwner(Xen arg1, Xen arg2)
{
  #define H_XGetSelectionOwner "Window XGetSelectionOwner(display, selection): returns the window ID associated with the window that \
currently owns the specified selection."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XGetSelectionOwner", "Display*");
  Xen_check_type(Xen_is_Atom(arg2), arg2, 2, "XGetSelectionOwner", "Atom");
  return(C_to_Xen_Window(XGetSelectionOwner(Xen_to_C_Display(arg1), Xen_to_C_Atom(arg2))));
}

static Xen gxm_XCreateSimpleWindow(Xen arg1, Xen arg2, Xen arg3, Xen arg4, Xen arg5, Xen arg6, Xen arg7, Xen arg8, Xen arg9)
{
  #define H_XCreateSimpleWindow "Window XCreateSimpleWindow(display, parent, x, y, width, height, border_width, border, background) \
creates an unmapped InputOutput subwindow for a specified parent window, returns the window ID of the created window, and causes the X \
server to generate a CreateNotify event."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XCreateSimpleWindow", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XCreateSimpleWindow", "Window");
  Xen_check_type(Xen_is_integer(arg3), arg3, 3, "XCreateSimpleWindow", "int");
  Xen_check_type(Xen_is_integer(arg4), arg4, 4, "XCreateSimpleWindow", "int");
  Xen_check_type(Xen_is_ulong(arg5), arg5, 5, "XCreateSimpleWindow", "unsigned int");
  Xen_check_type(Xen_is_ulong(arg6), arg6, 6, "XCreateSimpleWindow", "unsigned int");
  Xen_check_type(Xen_is_ulong(arg7), arg7, 7, "XCreateSimpleWindow", "unsigned int");
  Xen_check_type(Xen_is_Pixel(arg8), arg8, 8, "XCreateSimpleWindow", "Pixel");
  Xen_check_type(Xen_is_Pixel(arg9), arg9, 9, "XCreateSimpleWindow", "Pixel");
  return(C_to_Xen_Window(XCreateSimpleWindow(Xen_to_C_Display(arg1), 
					     Xen_to_C_Window(arg2), 
					     Xen_integer_to_C_int(arg3), Xen_integer_to_C_int(arg4), 
					     Xen_ulong_to_C_ulong(arg5), Xen_ulong_to_C_ulong(arg6), 
					     Xen_ulong_to_C_ulong(arg7), Xen_to_C_Pixel(arg8), 
					     Xen_to_C_Pixel(arg9))));
}

static Xen gxm_XCreatePixmapFromBitmapData(Xen arg1, Xen arg2, Xen larg3, Xen arg4, Xen arg5, Xen arg6, Xen arg7, Xen arg8)
{
  #define H_XCreatePixmapFromBitmapData "Pixmap XCreatePixmapFromBitmapData(display, d, data, width, height, fg, bg, depth) creates a \
pixmap of the given depth and then does a bitmap-format XPutImage of the data into it."
  /* DIFF: XCreatePixmapFromBitmapData takes list of chars as arg3 (not char *)
   */
  char *bits;
  int i, len;
  Pixmap p;
  Xen arg3;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XCreatePixmapFromBitmapData", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XCreatePixmapFromBitmapData", "Drawable");
  Xen_check_type(Xen_is_list(larg3), larg3, 3, "XCreatePixmapFromBitmapData", "list of char");
  Xen_check_type(Xen_is_ulong(arg4), arg4, 4, "XCreatePixmapFromBitmapData", "unsigned int");
  Xen_check_type(Xen_is_ulong(arg5), arg5, 5, "XCreatePixmapFromBitmapData", "unsigned int");
  Xen_check_type(Xen_is_Pixel(arg6), arg6, 6, "XCreatePixmapFromBitmapData", "pixel");
  Xen_check_type(Xen_is_Pixel(arg7), arg7, 7, "XCreatePixmapFromBitmapData", "pixel");
  Xen_check_type(Xen_is_ulong(arg8), arg8, 8, "XCreatePixmapFromBitmapData", "unsigned int");
  len = Xen_list_length(larg3);
  if (len <= 0) Xen_check_type(0, larg3, 3, "XCreatePixmapFromBitmapData", "positive integer");
  arg3 = Xen_copy_arg(larg3);
  bits = (char *)calloc(len, sizeof(char));
  for (i = 0; i < len; i++, arg3 = Xen_cdr(arg3))
    bits[i] = (char)Xen_integer_to_C_int(Xen_car(arg3));
  p = XCreatePixmapFromBitmapData(Xen_to_C_Display(arg1), 
				  Xen_to_C_Window(arg2), 
				  bits,
				  Xen_ulong_to_C_ulong(arg4), Xen_ulong_to_C_ulong(arg5), 
				  Xen_to_C_Pixel(arg6), Xen_to_C_Pixel(arg7), 
				  Xen_ulong_to_C_ulong(arg8));
  free(bits);
  return(C_to_Xen_Pixmap(p));
}

static Xen gxm_XCreateBitmapFromData(Xen arg1, Xen arg2, Xen larg3, Xen arg4, Xen arg5)
{
  #define H_XCreateBitmapFromData "Pixmap XCreateBitmapFromData(display, d, data, width, height) allows you to include in your C \
program a bitmap file that was written out by XWriteBitmapFile"
  /* DIFF: XCreateBitmapFromData takes list of chars as arg3 (not char *)
   */
  char *bits;
  int i, len;
  Pixmap p;
  Xen arg3;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XCreateBitmapFromData", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XCreateBitmapFromData", "Drawable");
  Xen_check_type(Xen_is_list(larg3), larg3, 3, "XCreateBitmapFromData", "list of char");
  Xen_check_type(Xen_is_ulong(arg4), arg4, 4, "XCreateBitmapFromData", "unsigned int");
  Xen_check_type(Xen_is_ulong(arg5), arg5, 5, "XCreateBitmapFromData", "unsigned int");
  len = Xen_list_length(larg3);
  if (len <= 0) Xen_check_type(0, larg3, 3, "XCreateBitmapFromData", "positive integer");
  arg3 = Xen_copy_arg(larg3);
  bits = (char *)calloc(len, sizeof(char));
  for (i = 0; i < len; i++, arg3 = Xen_cdr(arg3))
    bits[i] = (char)Xen_integer_to_C_int(Xen_car(arg3));
  p = XCreateBitmapFromData(Xen_to_C_Display(arg1), 
			    Xen_to_C_Window(arg2),
			    bits, 
			    Xen_ulong_to_C_ulong(arg4), 
			    Xen_ulong_to_C_ulong(arg5));
  free(bits);
  return(C_to_Xen_Pixmap(p));
}

static Xen gxm_XCreatePixmap(Xen arg1, Xen arg2, Xen arg3, Xen arg4, Xen arg5)
{
  #define H_XCreatePixmap "Pixmap XCreatePixmap(display, d, width, height, depth)"
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XCreatePixmap", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XCreatePixmap", "Drawable");
  Xen_check_type(Xen_is_ulong(arg3), arg3, 3, "XCreatePixmap", "unsigned int");
  Xen_check_type(Xen_is_ulong(arg4), arg4, 4, "XCreatePixmap", "unsigned int");
  Xen_check_type(Xen_is_ulong(arg5), arg5, 5, "XCreatePixmap", "unsigned int");
  return(C_to_Xen_Pixmap(XCreatePixmap(Xen_to_C_Display(arg1), Xen_to_C_Window(arg2), 
				       Xen_ulong_to_C_ulong(arg3), Xen_ulong_to_C_ulong(arg4), Xen_ulong_to_C_ulong(arg5))));
}

static Xen gxm_XFlushGC(Xen arg1, Xen arg2)
{
  #define H_XFlushGC "XFlushGC(dpy, gc) forces cached GC changes to X server"
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XFlushGC", "Display*");
  Xen_check_type(Xen_is_GC(arg2), arg2, 2, "XFlushGC", "GC");
  XFlushGC(Xen_to_C_Display(arg1), Xen_to_C_GC(arg2));
  return(Xen_false);
}

static Xen gxm_XCreateGC(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XCreateGC "GC XCreateGC(display, d, valuemask, values) creates a graphics context and returns a GC."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XCreateGC", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XCreateGC", "Drawable");
  Xen_check_type(Xen_is_ulong(arg3), arg3, 3, "XCreateGC", "ulong");
  Xen_check_type(Xen_is_XGCValues(arg4), arg4, 4, "XCreateGC", "XGCValues*");
  return(C_to_Xen_GC(XCreateGC(Xen_to_C_Display(arg1), Xen_to_C_Window(arg2), Xen_ulong_to_C_ulong(arg3), Xen_to_C_XGCValues(arg4))));
}

static Xen gxm_XLoadFont(Xen arg1, Xen arg2)
{
  #define H_XLoadFont "Font XLoadFont(display, name) loads the specified font and returns its associated font ID."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XLoadFont", "Display*");
  Xen_check_type(Xen_is_string(arg2), arg2, 2, "XLoadFont", "char*");
  return(C_to_Xen_Font(XLoadFont(Xen_to_C_Display(arg1), (char *)Xen_string_to_C_string(arg2))));
}

static Xen gxm_XCreateFontCursor(Xen arg1, Xen arg2)
{
  #define H_XCreateFontCursor "Cursor XCreateFontCursor(display, shape)"
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XCreateFontCursor", "Display*");
  Xen_check_type(Xen_is_ulong(arg2), arg2, 2, "XCreateFontCursor", "unsigned int");
  return(C_to_Xen_Cursor(XCreateFontCursor(Xen_to_C_Display(arg1), Xen_ulong_to_C_ulong(arg2))));
}

static Xen gxm_XCreateGlyphCursor(Xen arg1, Xen arg2, Xen arg3, Xen arg4, Xen arg5, Xen arg6, Xen arg7)
{
  #define H_XCreateGlyphCursor "Cursor XCreateGlyphCursor(display, source_font, mask_font, source_char, mask_char, foreground_color, \
background_color) is similar to XCreatePixmapCursor except that the source and mask bitmaps are obtained from the specified font glyphs."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XCreateGlyphCursor", "Display*");
  Xen_check_type(Xen_is_Font(arg2), arg2, 2, "XCreateGlyphCursor", "Font");
  Xen_check_type(Xen_is_Font(arg3) || Xen_is_integer(arg3) || Xen_is_false(arg3), arg3, 3, "XCreateGlyphCursor", "Font");
  Xen_check_type(Xen_is_ulong(arg4), arg4, 4, "XCreateGlyphCursor", "unsigned int");
  Xen_check_type(Xen_is_ulong(arg5), arg5, 5, "XCreateGlyphCursor", "unsigned int");
  Xen_check_type(Xen_is_XColor(arg6), arg6, 6, "XCreateGlyphCursor", "XColor");
  Xen_check_type(Xen_is_XColor(arg7), arg7, 7, "XCreateGlyphCursor", "XColor");
  return(C_to_Xen_Cursor(XCreateGlyphCursor(Xen_to_C_Display(arg1), Xen_to_C_Font(arg2), 
					    (Xen_is_Font(arg3)) ? Xen_to_C_Font(arg3) : None,
					    Xen_ulong_to_C_ulong(arg4),
					    Xen_ulong_to_C_ulong(arg5), 
					    Xen_to_C_XColor(arg6), Xen_to_C_XColor(arg7))));
}

static Xen gxm_XCreatePixmapCursor(Xen arg1, Xen arg2, Xen arg3, Xen arg4, Xen arg5, Xen arg6, Xen arg7)
{
  #define H_XCreatePixmapCursor "Cursor XCreatePixmapCursor(display, source, mask, foreground_color, background_color, x, y) creates \
a cursor and returns the cursor ID associated with it."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XCreatePixmapCursor", "Display*");
  Xen_check_type(Xen_is_Pixmap(arg2), arg2, 2, "XCreatePixmapCursor", "Pixmap");
  Xen_check_type(Xen_is_Pixmap(arg3) || Xen_is_integer(arg3) || Xen_is_false(arg3), arg3, 3, "XCreatePixmapCursor", "Pixmap");
  Xen_check_type(Xen_is_XColor(arg4), arg4, 4, "XCreatePixmapCursor", "XColor");
  Xen_check_type(Xen_is_XColor(arg5), arg5, 5, "XCreatePixmapCursor", "XColor");
  Xen_check_type(Xen_is_ulong(arg6), arg6, 6, "XCreatePixmapCursor", "unsigned int");
  Xen_check_type(Xen_is_ulong(arg7), arg7, 7, "XCreatePixmapCursor", "unsigned int");
  return(C_to_Xen_Cursor(XCreatePixmapCursor(Xen_to_C_Display(arg1), 
					     Xen_to_C_Pixmap(arg2), 
					     (Xen_is_Pixmap(arg3)) ? Xen_to_C_Pixmap(arg3) : None,
					     Xen_to_C_XColor(arg4), 
					     Xen_to_C_XColor(arg5), 
					     Xen_ulong_to_C_ulong(arg6), Xen_ulong_to_C_ulong(arg7))));
}

static Xen gxm_XCreateColormap(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XCreateColormap "Colormap XCreateColormap(display, w, visual, alloc) creates a colormap of the specified visual type for \
the screen on which the specified window resides and returns the colormap ID associated with it."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XCreateColormap", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XCreateColormap", "Window");
  Xen_check_type(Xen_is_Visual(arg3), arg3, 3, "XCreateColormap", "Visual*");
  Xen_check_type(Xen_is_integer(arg4), arg4, 4, "XCreateColormap", "int");
  return(C_to_Xen_Colormap(XCreateColormap(Xen_to_C_Display(arg1), Xen_to_C_Window(arg2), Xen_to_C_Visual(arg3), Xen_integer_to_C_int(arg4))));
}

static Xen gxm_XCopyColormapAndFree(Xen arg1, Xen arg2)
{
  #define H_XCopyColormapAndFree "Colormap XCopyColormapAndFree(display, colormap) creates a colormap of the same visual type and \
for the same screen as the specified colormap and returns the new colormap ID."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XCopyColormapAndFree", "Display*");
  Xen_check_type(Xen_is_Colormap(arg2), arg2, 2, "XCopyColormapAndFree", "Colormap");
  return(C_to_Xen_Colormap(XCopyColormapAndFree(Xen_to_C_Display(arg1), Xen_to_C_Colormap(arg2))));
}

static Xen gxm_XInternAtom(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XInternAtom "Atom XInternAtom(display, atom_name, only_if_exists): returns the atom identifier associated with the specified atom_name string."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XInternAtom", "Display*");
  Xen_check_type(Xen_is_string(arg2), arg2, 2, "XInternAtom", "char*");
  Xen_check_type(Xen_is_boolean(arg3), arg3, 3, "XInternAtom", "Bool");
  return(C_to_Xen_Atom(XInternAtom(Xen_to_C_Display(arg1), (char *)Xen_string_to_C_string(arg2), Xen_boolean_to_C_bool(arg3))));
}

static Xen xm_AfterFunction;

static int default_after_function(Display *ignore) {return(0);}

static int gxm_AfterFunction(Display *dpy)
{
  return(Xen_integer_to_C_int(Xen_call_with_1_arg(xm_AfterFunction, 
				 C_to_Xen_Display(dpy),
				 __func__)));
}

static Xen gxm_XSetAfterFunction(Xen arg1, Xen arg2)
{
  #define H_XSetAfterFunction "XSetAfterFunction(dpy, proc) sets a function (one arg: dpy) to be called after every X function call"
  Xen old_func;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XSetAfterFunction", "Display*");
  xm_protect(arg2);
  old_func = xm_AfterFunction;
  xm_AfterFunction = arg2;
  if (Xen_is_procedure(arg2))
    XSetAfterFunction(Xen_to_C_Display(arg1), gxm_AfterFunction);
  else XSetAfterFunction(Xen_to_C_Display(arg1), default_after_function);
  if (Xen_is_procedure(old_func)) xm_unprotect(old_func);
  return(old_func);
}

static Xen gxm_XSynchronize(Xen arg1, Xen arg2)
{
  #define H_XSynchronize "int (*XSynchronize(display, onoff))() turns on/off synchronous behavior."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XSynchronize", "Display*");
  Xen_check_type(Xen_is_boolean(arg2), arg2, 2, "XSynchronize", "boolean");
  XSynchronize(Xen_to_C_Display(arg1),
	       Xen_boolean_to_C_bool(arg2));
  return(xm_AfterFunction);
}

static Xen gxm_XKeysymToString(Xen arg1)
{
  #define H_XKeysymToString "char *XKeysymToString(keysym)"
  Xen_check_type(Xen_is_KeySym(arg1), arg1, 1, "XKeysymToString", "KeySym");
  return(C_string_to_Xen_string(XKeysymToString(Xen_to_C_KeySym(arg1))));
}

static Xen gxm_XDisplayName(Xen arg1)
{
  #define H_XDisplayName "char *XDisplayName(string): returns the name of the display that  XOpenDisplay would attempt to use."
  Xen_check_type(Xen_is_string(arg1), arg1, 1, "XDisplayName", "char*");
  return(C_string_to_Xen_string(XDisplayName((char *)Xen_string_to_C_string(arg1))));
}

static Xen gxm_XGetAtomName(Xen arg1, Xen arg2)
{
  #define H_XGetAtomName "char *XGetAtomName(display, atom): returns the name associated with the specified atom."
  char *str;
  Xen res;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XGetAtomName", "Display*");
  Xen_check_type(Xen_is_Atom(arg2), arg2, 2, "XGetAtomName", "Atom");
  str = XGetAtomName(Xen_to_C_Display(arg1), Xen_to_C_Atom(arg2));
  res = C_string_to_Xen_string(str);
  XFree(str);
  return(res);
}

static Xen gxm_XFetchBuffer(Xen arg1, Xen arg2)
{
  #define H_XFetchBuffer "char *XFetchBuffer(display, buffer): returns " PROC_FALSE " if there \
is no data in the buffer or if an invalid buffer is specified, otherwise a string."
  int len = 0;
  char *buf;
  Xen lst = Xen_false;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XFetchBuffer", "Display*");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XFetchBuffer", "int");
  buf = XFetchBuffer(Xen_to_C_Display(arg1), &len, Xen_integer_to_C_int(arg2));
  if (len > 0) 
    {
      lst = C_string_to_Xen_string(buf);
      free(buf);
    }
  return(lst);
}

static Xen gxm_XFetchBytes(Xen arg1)
{
  #define H_XFetchBytes "char *XFetchBytes(display): returns the string in cut buffer 0"
  /* DIFF: XFetchBytes returns string, omits arg2
   */
  int len = 0;
  char *buf;
  Xen lst = Xen_false;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XFetchBytes", "Display*");
  buf = XFetchBytes(Xen_to_C_Display(arg1), &len);
  if (len > 0) 
    {
      lst = C_string_to_Xen_string(buf);
      free(buf);
    }
  return(lst);
}

static Xen gxm_XOpenDisplay(Xen arg1)
{
  #define H_XOpenDisplay "Display *XOpenDisplay(display_name): returns a Display structure that serves as the connection to the X server \
and that contains all the information about that X server."
  Display *dpy;
  Xen_check_type(Xen_is_string(arg1) || Xen_is_false(arg1), arg1, 1, "XOpenDisplay", "char*");
  dpy = XOpenDisplay(Xen_is_false(arg1) ? NULL : (char *)Xen_string_to_C_string(arg1));
  if (dpy)
    return(C_to_Xen_Display(dpy));
  return(Xen_false);
}

static Xen gxm_XGetSubImage(Xen args)
{
  #define H_XGetSubImage "XImage *XGetSubImage(display, d, x, y, width, height, plane_mask, format, dest_image, dest_x, dest_y) updates \
dest_image with the specified subimage in the same manner as "
  Xen arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8, arg9, arg10, arg11;
  arg1 = Xen_list_ref(args, 0);
  arg2 = Xen_list_ref(args, 1);
  arg3 = Xen_list_ref(args, 2);
  arg4 = Xen_list_ref(args, 3);
  arg5 = Xen_list_ref(args, 4);
  arg6 = Xen_list_ref(args, 5);
  arg7 = Xen_list_ref(args, 6);
  arg8 = Xen_list_ref(args, 7);
  arg9 = Xen_list_ref(args, 8);
  arg10 = Xen_list_ref(args, 9);
  arg11 = Xen_list_ref(args, 10);
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XGetSubImage", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XGetSubImage", "Drawable");
  Xen_check_type(Xen_is_integer(arg3), arg3, 3, "XGetSubImage", "int");
  Xen_check_type(Xen_is_integer(arg4), arg4, 4, "XGetSubImage", "int");
  Xen_check_type(Xen_is_ulong(arg5), arg5, 5, "XGetSubImage", "unsigned int");
  Xen_check_type(Xen_is_ulong(arg6), arg6, 6, "XGetSubImage", "unsigned int");
  Xen_check_type(Xen_is_ulong(arg7), arg7, 7, "XGetSubImage", "ulong");
  Xen_check_type(Xen_is_integer(arg8), arg8, 8, "XGetSubImage", "int");
  Xen_check_type(Xen_is_XImage(arg9), arg9, 9, "XGetSubImage", "XImage*");
  Xen_check_type(Xen_is_integer(arg10), arg10, 10, "XGetSubImage", "int");
  Xen_check_type(Xen_is_integer(arg11), arg11, 11, "XGetSubImage", "int");
  return(C_to_Xen_XImage(XGetSubImage(Xen_to_C_Display(arg1),
				      Xen_to_C_Window(arg2), 
				      Xen_integer_to_C_int(arg3), Xen_integer_to_C_int(arg4), 
				      Xen_ulong_to_C_ulong(arg5), Xen_ulong_to_C_ulong(arg6), 
				      Xen_ulong_to_C_ulong(arg7), Xen_integer_to_C_int(arg8), 
				      Xen_to_C_XImage(arg9), 
				      Xen_integer_to_C_int(arg10), Xen_integer_to_C_int(arg11))));
}

static Xen gxm_XGetImage(Xen arg1, Xen arg2, Xen arg3, Xen arg4, Xen arg5, Xen arg6, Xen arg7, Xen arg8)
{
  #define H_XGetImage "XImage *XGetImage(display, d, x, y, width, height, plane_mask, format): returns a pointer to an XImage structure."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XGetImage", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XGetImage", "Drawable");
  Xen_check_type(Xen_is_integer(arg3), arg3, 3, "XGetImage", "int");
  Xen_check_type(Xen_is_integer(arg4), arg4, 4, "XGetImage", "int");
  Xen_check_type(Xen_is_ulong(arg5), arg5, 5, "XGetImage", "unsigned int");
  Xen_check_type(Xen_is_ulong(arg6), arg6, 6, "XGetImage", "unsigned int");
  Xen_check_type(Xen_is_ulong(arg7), arg7, 7, "XGetImage", "ulong");
  Xen_check_type(Xen_is_integer(arg8), arg8, 8, "XGetImage", "int");
  return(C_to_Xen_XImage(XGetImage(Xen_to_C_Display(arg1), 
				   Xen_to_C_Window(arg2), 
				   Xen_integer_to_C_int(arg3), Xen_integer_to_C_int(arg4), 
				   Xen_ulong_to_C_ulong(arg5), Xen_ulong_to_C_ulong(arg6), 
				   Xen_ulong_to_C_ulong(arg7), Xen_integer_to_C_int(arg8))));
}

static Xen gxm_XCreateImage(Xen args)
{
  #define H_XCreateImage "XImage *XCreateImage(display, visual, depth, format, offset, data, width, height, bitmap_pad, bytes_per_line) \
allocates the memory needed for an XImage structure for the specified display but does not allocate space for the image itself."
  Xen arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8, arg9, arg10;
  arg1 = Xen_list_ref(args, 0);
  arg2 = Xen_list_ref(args, 1);
  arg3 = Xen_list_ref(args, 2);
  arg4 = Xen_list_ref(args, 3);
  arg5 = Xen_list_ref(args, 4);
  arg6 = Xen_list_ref(args, 5);
  arg7 = Xen_list_ref(args, 6);
  arg8 = Xen_list_ref(args, 7);
  arg9 = Xen_list_ref(args, 8);
  arg10 = Xen_list_ref(args, 9);
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XCreateImage", "Display*");
  Xen_check_type(Xen_is_Visual(arg2), arg2, 2, "XCreateImage", "Visual*");
  Xen_check_type(Xen_is_ulong(arg3), arg3, 3, "XCreateImage", "unsigned int");
  Xen_check_type(Xen_is_integer(arg4), arg4, 4, "XCreateImage", "int");
  Xen_check_type(Xen_is_integer(arg5), arg5, 5, "XCreateImage", "int");
  Xen_check_type(Xen_is_wrapped_c_pointer(arg6), arg6, 6, "XCreateImage", "pointer");
  Xen_check_type(Xen_is_ulong(arg7), arg7, 7, "XCreateImage", "unsigned int");
  Xen_check_type(Xen_is_ulong(arg8), arg8, 8, "XCreateImage", "unsigned int");
  Xen_check_type(Xen_is_integer(arg9), arg9, 9, "XCreateImage", "int");
  Xen_check_type(Xen_is_integer(arg10), arg10, 10, "XCreateImage", "int");
  return(C_to_Xen_XImage(XCreateImage(Xen_to_C_Display(arg1), 
				      Xen_to_C_Visual(arg2), 
				      Xen_ulong_to_C_ulong(arg3), Xen_integer_to_C_int(arg4), Xen_integer_to_C_int(arg5), 
				      (char *)Xen_unwrap_C_pointer(arg6),
				      Xen_ulong_to_C_ulong(arg7), Xen_ulong_to_C_ulong(arg8),
				      Xen_integer_to_C_int(arg9), Xen_integer_to_C_int(arg10))));
}

static Xen gxm_XNewModifiermap(Xen arg1)
{
  #define H_XNewModifiermap "XModifierKeymap *XNewModifiermap(max_keys_per_mod): returns a pointer to XModifierKeymap structure for later use."
  Xen_check_type(Xen_is_integer(arg1), arg1, 1, "XNewModifiermap", "int");
  return(C_to_Xen_XModifierKeymap(XNewModifiermap(Xen_integer_to_C_int(arg1))));
}

static Xen gxm_XInsertModifiermapEntry(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XInsertModifiermapEntry "XModifierKeymap *XInsertModifiermapEntry(modmap, keycode_entry, modifier) adds the specified KeyCode to \
the set that controls the specified modifier and returns the resulting XModifierKeymap structure (expanded as needed)."
  Xen_check_type(Xen_is_XModifierKeymap(arg1), arg1, 1, "XInsertModifiermapEntry", "XModifierKeymap*");
  Xen_check_type(Xen_is_KeyCode(arg2), arg2, 2, "XInsertModifiermapEntry", "KeyCode");
  Xen_check_type(Xen_is_integer(arg3), arg3, 3, "XInsertModifiermapEntry", "int");
  return(C_to_Xen_XModifierKeymap(XInsertModifiermapEntry(Xen_to_C_XModifierKeymap(arg1), Xen_to_C_KeyCode(arg2), Xen_integer_to_C_int(arg3))));
}

static Xen gxm_XGetModifierMapping(Xen arg1)
{
  #define H_XGetModifierMapping "XModifierKeymap *XGetModifierMapping(display): returns a pointer to a newly created XModifierKeymap structure \
that contains the keys being used as modifiers."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XGetModifierMapping", "Display*");
  return(C_to_Xen_XModifierKeymap(XGetModifierMapping(Xen_to_C_Display(arg1))));
}

static Xen gxm_XDeleteModifiermapEntry(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XDeleteModifiermapEntry "XModifierKeymap *XDeleteModifiermapEntry(modmap, keycode_entry, modifier) deletes the specified KeyCode \
from the set that controls the specified modifier and returns a pointer to the resulting XModifierKeymap structure."
  Xen_check_type(Xen_is_XModifierKeymap(arg1), arg1, 1, "XDeleteModifiermapEntry", "XModifierKeymap*");
  Xen_check_type(Xen_is_KeyCode(arg2), arg2, 2, "XDeleteModifiermapEntry", "KeyCode");
  Xen_check_type(Xen_is_integer(arg3), arg3, 3, "XDeleteModifiermapEntry", "int");
  return(C_to_Xen_XModifierKeymap(XDeleteModifiermapEntry(Xen_to_C_XModifierKeymap(arg1), Xen_to_C_KeyCode(arg2), Xen_integer_to_C_int(arg3))));
}

static Xen gxm_XGetMotionEvents(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XGetMotionEvents "XTimeCoord *XGetMotionEvents(display, w, start, stop): returns all events in the motion history \
buffer that fall between the specified start and stop times, inclusive, and that have coordinates that lie within the specified window \
(including its borders) at its present placement."
  /* DIFF: XGetMotionEvents omits last arg, return time coords as list of lists
   */
  int n, i, loc;
  XTimeCoord *tcs;
  Xen lst = Xen_empty_list;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XGetMotionEvents", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XGetMotionEvents", "Window");
  Xen_check_type(Xen_is_Time(arg3), arg3, 3, "XGetMotionEvents", "Time");
  Xen_check_type(Xen_is_Time(arg4), arg4, 4, "XGetMotionEvents", "Time");
  tcs = XGetMotionEvents(Xen_to_C_Display(arg1), Xen_to_C_Window(arg2), Xen_to_C_Time(arg3), Xen_to_C_Time(arg4), &n);
  loc = xm_protect(lst);
  for (i = n - 1; i > 0; i--)
    lst = Xen_cons(Xen_list_3(C_to_Xen_Time(tcs->time),
			      C_int_to_Xen_integer((int)(tcs->x)),
			      C_int_to_Xen_integer((int)(tcs->y))),
		   lst);
  XFree(tcs); /* free each as well? */
  xm_unprotect_at(loc);
  return(lst);
}

static Xen gxm_XQueryFont(Xen arg1, Xen arg2)
{
  #define H_XQueryFont "XFontStruct *XQueryFont(display, font): returns a pointer to the XFontStruct structure, which contains information \
associated with the font."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XQueryFont", "Display*");
  Xen_check_type(Xen_is_Font(arg2), arg2, 2, "XQueryFont", "Font");
  return(C_to_Xen_XFontStruct(XQueryFont(Xen_to_C_Display(arg1), Xen_to_C_Font(arg2))));
}

static Xen gxm_XLoadQueryFont(Xen arg1, Xen arg2)
{
  #define H_XLoadQueryFont "XFontStruct *XLoadQueryFont(display, name) provides the most common way for accessing a font. XLoadQueryFont \
both opens (loads) the specified font and returns a pointer to the appropriate XFontStruct structure."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XLoadQueryFont", "Display*");
  Xen_check_type(Xen_is_string(arg2), arg2, 2, "XLoadQueryFont", "char*");
  return(C_to_Xen_XFontStruct(XLoadQueryFont(Xen_to_C_Display(arg1), (char *)Xen_string_to_C_string(arg2))));
}

static Xen gxm_DefaultScreen(Xen arg)
{
  #define H_DefaultScreen "returns the default screen number referenced in the XOpenDisplay routine."
  Xen_check_type(Xen_is_Display(arg), arg, 0, "DefaultScreen", "Display*");
  return(C_int_to_Xen_integer(DefaultScreen(Xen_to_C_Display(arg))));
}

static Xen gxm_DefaultRootWindow(Xen arg)
{
  Xen_check_type(Xen_is_Display(arg), arg, 0, "DefaultRootWindow", "Display*");
  return(C_to_Xen_Window(DefaultRootWindow(Xen_to_C_Display(arg))));
}

static Xen gxm_QLength(Xen arg)
{
  /* QLength(display) */
  Xen_check_type(Xen_is_Display(arg), arg, 0, "QLength", "Display*");
  return(C_int_to_Xen_integer(QLength(Xen_to_C_Display(arg))));
}

static Xen gxm_ScreenCount(Xen arg1)
{
  /* ScreenCount(display) */
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "ScreenCount", "Display*");
  return(C_int_to_Xen_integer(XScreenCount(Xen_to_C_Display(arg1))));
}

static Xen gxm_ServerVendor(Xen arg)
{
  Xen_check_type(Xen_is_Display(arg), arg, 0, "ServerVendor", "Display*");
  return(C_string_to_Xen_string(ServerVendor(Xen_to_C_Display(arg))));
}

static Xen gxm_ProtocolVersion(Xen arg)
{
  Xen_check_type(Xen_is_Display(arg), arg, 0, "ProtocolVersion", "Display*");
  return(C_int_to_Xen_integer(ProtocolVersion(Xen_to_C_Display(arg))));
}

static Xen gxm_ProtocolRevision(Xen arg)
{
  Xen_check_type(Xen_is_Display(arg), arg, 0, "ProtocolRevision", "Display*");
  return(C_int_to_Xen_integer(ProtocolRevision(Xen_to_C_Display(arg))));
}

static Xen gxm_VendorRelease(Xen arg)
{
  Xen_check_type(Xen_is_Display(arg), arg, 0, "VendorRelease", "Display*");
  return(C_int_to_Xen_integer(VendorRelease(Xen_to_C_Display(arg))));
}

static Xen gxm_DisplayString(Xen arg)
{
  Xen_check_type(Xen_is_Display(arg), arg, 0, "DisplayString", "Display*");
  return(C_string_to_Xen_string(DisplayString(Xen_to_C_Display(arg))));
}

static Xen gxm_BitmapUnit(Xen arg)
{
  Xen_check_type(Xen_is_Display(arg), arg, 0, "BitmapUnit", "Display*");
  return(C_int_to_Xen_integer(BitmapUnit(Xen_to_C_Display(arg))));
}

static Xen gxm_BitmapBitOrder(Xen arg)
{
  Xen_check_type(Xen_is_Display(arg), arg, 0, "BitmapBitOrder", "Display*");
  return(C_int_to_Xen_integer(BitmapBitOrder(Xen_to_C_Display(arg))));
}

static Xen gxm_BitmapPad(Xen arg)
{
  Xen_check_type(Xen_is_Display(arg), arg, 0, "BitmapPad", "Display*");
  return(C_int_to_Xen_integer(BitmapPad(Xen_to_C_Display(arg))));
}

static Xen gxm_ImageByteOrder(Xen arg)
{
  Xen_check_type(Xen_is_Display(arg), arg, 0, "ImageByteOrder", "Display*");
  return(C_int_to_Xen_integer(ImageByteOrder(Xen_to_C_Display(arg))));
}

static Xen gxm_NextRequest(Xen arg)
{
  Xen_check_type(Xen_is_Display(arg), arg, 0, "NextRequest", "Display*");
  return(C_ulong_to_Xen_ulong(NextRequest(Xen_to_C_Display(arg))));
}

static Xen gxm_LastKnownRequestProcessed(Xen arg)
{
  Xen_check_type(Xen_is_Display(arg), arg, 0, "LastKnownRequestProcessed", "Display*");
  return(C_ulong_to_Xen_ulong(LastKnownRequestProcessed(Xen_to_C_Display(arg))));
}

static Xen gxm_DefaultScreenOfDisplay(Xen arg)
{
  #define H_DefaultScreenOfDisplay "returns the default screen of the specified display."
  Xen_check_type(Xen_is_Display(arg), arg, 0, "DefaultScreenOfDisplay", "Display");
  return(C_to_Xen_Screen(DefaultScreenOfDisplay(Xen_to_C_Display(arg))));
}

static Xen gxm_DisplayOfScreen(Xen arg)
{
  Xen_check_type(Xen_is_Screen(arg), arg, 0, "DisplayOfScreen", "Screen");
  return(C_to_Xen_Display(DisplayOfScreen(Xen_to_C_Screen(arg))));
}

static Xen gxm_RootWindowOfScreen(Xen arg)
{
  Xen_check_type(Xen_is_Screen(arg), arg, 0, "RootWindowOfScreen", "Screen");
  return(C_to_Xen_Window(RootWindowOfScreen(Xen_to_C_Screen(arg))));
}

static Xen gxm_BlackPixelOfScreen(Xen arg)
{
  Xen_check_type(Xen_is_Screen(arg), arg, 0, "BlackPixelOfScreen", "Screen");
  return(C_to_Xen_Pixel(BlackPixelOfScreen(Xen_to_C_Screen(arg))));
}

static Xen gxm_WhitePixelOfScreen(Xen arg)
{
  Xen_check_type(Xen_is_Screen(arg), arg, 0, "WhitePixelOfScreen", "Screen");
  return(C_to_Xen_Pixel(WhitePixelOfScreen(Xen_to_C_Screen(arg))));
}

static Xen gxm_DefaultColormapOfScreen(Xen arg)
{
  Xen_check_type(Xen_is_Screen(arg), arg, 0, "DefaultColormapOfScreen", "Screen");
  return(C_to_Xen_Colormap(DefaultColormapOfScreen(Xen_to_C_Screen(arg))));
}

static Xen gxm_DefaultDepthOfScreen(Xen arg)
{
  /* DefaultDepthOfScreen(screen) */
  Xen_check_type(Xen_is_Screen(arg), arg, 0, "DefaultDepthOfScreen", "Screen");
  return(C_int_to_Xen_integer(DefaultDepthOfScreen(Xen_to_C_Screen(arg))));
}

static Xen gxm_DefaultGCOfScreen(Xen arg)
{
  Xen_check_type(Xen_is_Screen(arg), arg, 0, "DefaultGCOfScreen", "Screen");
  return(C_to_Xen_GC(DefaultGCOfScreen(Xen_to_C_Screen(arg))));
}

static Xen gxm_DefaultVisualOfScreen(Xen arg)
{
  Xen_check_type(Xen_is_Screen(arg), arg, 0, "DefaultVisualOfScreen", "Screen");
  return(C_to_Xen_Visual(DefaultVisualOfScreen(Xen_to_C_Screen(arg))));
}

static Xen gxm_WidthOfScreen(Xen arg)
{
  Xen_check_type(Xen_is_Screen(arg), arg, 0, "WidthOfScreen", "Screen");
  return(C_int_to_Xen_integer(WidthOfScreen(Xen_to_C_Screen(arg))));
}

static Xen gxm_HeightOfScreen(Xen arg)
{
  /* HeightOfScreen(screen) */
  Xen_check_type(Xen_is_Screen(arg), arg, 0, "HeightOfScreen", "Screen");
  return(C_int_to_Xen_integer(HeightOfScreen(Xen_to_C_Screen(arg))));
}

static Xen gxm_WidthMMOfScreen(Xen arg)
{
  Xen_check_type(Xen_is_Screen(arg), arg, 0, "WidthMMOfScreen", "Screen");
  return(C_int_to_Xen_integer(WidthMMOfScreen(Xen_to_C_Screen(arg))));
}

static Xen gxm_HeightMMOfScreen(Xen arg)
{
  Xen_check_type(Xen_is_Screen(arg), arg, 0, "HeightMMOfScreen", "Screen");
  return(C_int_to_Xen_integer(HeightMMOfScreen(Xen_to_C_Screen(arg))));
}

static Xen gxm_PlanesOfScreen(Xen arg)
{
  Xen_check_type(Xen_is_Screen(arg), arg, 0, "PlanesOfScreen", "Screen");
  return(C_int_to_Xen_integer(PlanesOfScreen(Xen_to_C_Screen(arg))));
}

static Xen gxm_CellsOfScreen(Xen arg)
{
  Xen_check_type(Xen_is_Screen(arg), arg, 0, "CellsOfScreen", "Screen");
  return(C_int_to_Xen_integer(CellsOfScreen(Xen_to_C_Screen(arg))));
}

static Xen gxm_MinCmapsOfScreen(Xen arg)
{
  /* MinCmapsOfScreen(screen) */
  Xen_check_type(Xen_is_Screen(arg), arg, 0, "MinCmapsOfScreen", "Screen");
  return(C_int_to_Xen_integer(MinCmapsOfScreen(Xen_to_C_Screen(arg))));
}

static Xen gxm_MaxCmapsOfScreen(Xen arg)
{
  /* MaxCmapsOfScreen(screen) */
  Xen_check_type(Xen_is_Screen(arg), arg, 0, "MaxCmapsOfScreen", "Screen");
  return(C_int_to_Xen_integer(MaxCmapsOfScreen(Xen_to_C_Screen(arg))));
}

static Xen gxm_DoesSaveUnders(Xen arg)
{
  Xen_check_type(Xen_is_Screen(arg), arg, 0, "DoesSaveUnders", "Screen");
  return(C_bool_to_Xen_boolean(DoesSaveUnders(Xen_to_C_Screen(arg))));
}

static Xen gxm_DoesBackingStore(Xen arg)
{
  Xen_check_type(Xen_is_Screen(arg), arg, 0, "DoesBackingStore", "Screen");
  return(C_bool_to_Xen_boolean(DoesBackingStore(Xen_to_C_Screen(arg))));
}

static Xen gxm_EventMaskOfScreen(Xen arg)
{
  Xen_check_type(Xen_is_Screen(arg), arg, 0, "EventMaskOfScreen", "Screen");
  return(C_ulong_to_Xen_ulong(EventMaskOfScreen(Xen_to_C_Screen(arg))));
}

static Xen gxm_RootWindow(Xen arg1, Xen arg2)
{
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "RootWindow", "Display*");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "RootWindow", "int");
  return(C_to_Xen_Window(RootWindow(Xen_to_C_Display(arg1), Xen_integer_to_C_int(arg2))));
}

static Xen gxm_DefaultVisual(Xen arg1, Xen arg2)
{
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "DefaultVisual", "Display*");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "DefaultVisual", "int");
  return(C_to_Xen_Visual(DefaultVisual(Xen_to_C_Display(arg1), Xen_integer_to_C_int(arg2))));
}

static Xen gxm_DefaultGC(Xen arg1, Xen arg2)
{
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "DefaultGC", "Display*");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "DefaultGC", "int");
  return(C_to_Xen_GC(DefaultGC(Xen_to_C_Display(arg1), Xen_integer_to_C_int(arg2))));
}

static Xen gxm_BlackPixel(Xen arg1, Xen arg2)
{
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "BlackPixel", "Display*");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "BlackPixel", "int");
  return(C_to_Xen_Pixel(BlackPixel(Xen_to_C_Display(arg1), Xen_integer_to_C_int(arg2))));
}

static Xen gxm_WhitePixel(Xen arg1, Xen arg2)
{
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "WhitePixel", "Display*");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "WhitePixel", "int");
  return(C_to_Xen_Pixel(WhitePixel(Xen_to_C_Display(arg1), Xen_integer_to_C_int(arg2))));
}

static Xen gxm_DisplayWidth(Xen arg1, Xen arg2)
{
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "DisplayWidth", "Display*");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "DisplayWidth", "int");
  return(C_int_to_Xen_integer(DisplayWidth(Xen_to_C_Display(arg1), Xen_integer_to_C_int(arg2))));
}

static Xen gxm_DisplayHeight(Xen arg1, Xen arg2)
{
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "DisplayHeight", "Display*");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "DisplayHeight", "int");
  return(C_int_to_Xen_integer(DisplayHeight(Xen_to_C_Display(arg1), Xen_integer_to_C_int(arg2))));
}

static Xen gxm_DisplayWidthMM(Xen arg1, Xen arg2)
{
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "DisplayWidthMM", "Display*");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "DisplayWidthMM", "int");
  return(C_int_to_Xen_integer(DisplayWidthMM(Xen_to_C_Display(arg1), Xen_integer_to_C_int(arg2))));
}

static Xen gxm_DisplayHeightMM(Xen arg1, Xen arg2)
{
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "DisplayHeightMM", "Display*");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "DisplayHeightMM", "int");
  return(C_int_to_Xen_integer(DisplayHeightMM(Xen_to_C_Display(arg1), Xen_integer_to_C_int(arg2))));
}

static Xen gxm_DisplayPlanes(Xen arg1, Xen arg2)
{
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "DisplayPlanes", "Display*");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "DisplayPlanes", "int");
  return(C_int_to_Xen_integer(DisplayPlanes(Xen_to_C_Display(arg1), Xen_integer_to_C_int(arg2))));
}

static Xen gxm_DisplayCells(Xen arg1, Xen arg2)
{
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "DisplayCells", "Display*");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "DisplayCells", "int");
  return(C_int_to_Xen_integer(DisplayCells(Xen_to_C_Display(arg1), Xen_integer_to_C_int(arg2))));
}

static Xen gxm_DefaultColormap(Xen arg1, Xen arg2)
{
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "DefaultColormap", "Display*");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "DefaultColormap", "int");
  return(C_to_Xen_Colormap(DefaultColormap(Xen_to_C_Display(arg1), Xen_integer_to_C_int(arg2))));
}

static Xen gxm_ScreenOfDisplay(Xen arg1, Xen arg2)
{
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "ScreenOfDisplay", "Display*");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "ScreenOfDisplay", "int");
  return(C_to_Xen_Screen(ScreenOfDisplay(Xen_to_C_Display(arg1), Xen_integer_to_C_int(arg2))));
}

static Xen gxm_DefaultDepth(Xen arg1, Xen arg2)
{
  /* DefaultDepth(display, screen_number) */
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "DefaultDepth", "Display*");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "DefaultDepth", "int");
  return(C_int_to_Xen_integer(DefaultDepth(Xen_to_C_Display(arg1), Xen_integer_to_C_int(arg2))));
}

static Xen gxm_IsKeypadKey(Xen arg)
{
  #define H_IsKeypadKey "IsKeypadKey(keysym): returns " PROC_TRUE " if the specified KeySym is a keypad key."
  Xen_check_type(Xen_is_KeySym(arg), arg, 0, "IsKeypadKey", "KeySym");
  return(C_bool_to_Xen_boolean(IsKeypadKey(Xen_to_C_KeySym(arg))));
}

static Xen gxm_IsPrivateKeypadKey(Xen arg)
{
  #define H_IsPrivateKeypadKey "IsPrivateKeypadKey(keysym): returns " PROC_TRUE " if the specified KeySym is a vendor-private keypad key."
  Xen_check_type(Xen_is_KeySym(arg), arg, 0, "IsPrivateKeypadKey", "KeySym");
  return(C_bool_to_Xen_boolean(IsPrivateKeypadKey(Xen_to_C_KeySym(arg))));
}

static Xen gxm_IsCursorKey(Xen arg)
{
  #define H_IsCursorKey "IsCursorKey(keysym): returns " PROC_TRUE " if the specified KeySym is a cursor key."
  Xen_check_type(Xen_is_KeySym(arg), arg, 0, "IsCursorKey", "KeySym");
  return(C_bool_to_Xen_boolean(IsCursorKey(Xen_to_C_KeySym(arg))));
}

static Xen gxm_IsPFKey(Xen arg)
{
  #define H_IsPFKey "IsPFKey(keysym): returns " PROC_TRUE " if the specified KeySym is a PF key."
  Xen_check_type(Xen_is_KeySym(arg), arg, 0, "IsPFKey", "KeySym");
  return(C_bool_to_Xen_boolean(IsPFKey(Xen_to_C_KeySym(arg))));
}

static Xen gxm_IsFunctionKey(Xen arg)
{
  #define H_IsFunctionKey "IsFunctionKey(keysym): returns " PROC_TRUE " if the KeySym is a function key."
  Xen_check_type(Xen_is_KeySym(arg), arg, 0, "IsFunctionKey", "KeySym");
  return(C_bool_to_Xen_boolean(IsFunctionKey(Xen_to_C_KeySym(arg))));
}

static Xen gxm_IsMiscFunctionKey(Xen arg)
{
  #define H_IsMiscFunctionKey "IsMiscFunctionKey(keysym): returns " PROC_TRUE " if the specified KeySym is a miscellaneous function key."
  Xen_check_type(Xen_is_KeySym(arg), arg, 0, "IsMiscFunctionKey", "KeySym");
  return(C_bool_to_Xen_boolean(IsMiscFunctionKey(Xen_to_C_KeySym(arg))));
}

static Xen gxm_IsModifierKey(Xen arg)
{
  #define H_IsModifierKey "IsModifierKey(keysym): returns " PROC_TRUE " if the specified KeySym is a modifier key."
  Xen_check_type(Xen_is_KeySym(arg), arg, 0, "IsModifierKey", "KeySym");
  return(C_bool_to_Xen_boolean(IsModifierKey(Xen_to_C_KeySym(arg))));
}

static Xen gxm_XAddPixel(Xen arg1, Xen arg2)
{
  #define H_XAddPixel "XAddPixel(ximage, value) adds a constant value to every pixel in an image."
  Xen_check_type(Xen_is_XImage(arg1), arg1, 1, "XAddPixel", "XImage*");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XAddPixel", "long");
  return(C_int_to_Xen_integer(XAddPixel(Xen_to_C_XImage(arg1), Xen_integer_to_C_int(arg2))));
}

static Xen gxm_XSubImage(Xen arg1, Xen arg2, Xen arg3, Xen arg4, Xen arg5)
{
  #define H_XSubImage "XImage *XSubImage(ximage, x, y, subimage_width, subimage_height) creates a new image that is a subsection of an existing one."
  Xen_check_type(Xen_is_XImage(arg1), arg1, 1, "XSubImage", "XImage*");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XSubImage", "int");
  Xen_check_type(Xen_is_integer(arg3), arg3, 3, "XSubImage", "int");
  Xen_check_type(Xen_is_ulong(arg4), arg4, 4, "XSubImage", "unsigned int");
  Xen_check_type(Xen_is_ulong(arg5), arg5, 5, "XSubImage", "unsigned int");
  return(C_to_Xen_XImage(XSubImage(Xen_to_C_XImage(arg1), Xen_integer_to_C_int(arg2), Xen_integer_to_C_int(arg3), Xen_ulong_to_C_ulong(arg4), Xen_ulong_to_C_ulong(arg5))));
}

static Xen gxm_XPutPixel(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XPutPixel "XPutPixel(ximage, x, y, pixel) overwrites the pixel in the named image with the specified pixel value."
  Xen_check_type(Xen_is_XImage(arg1), arg1, 1, "XPutPixel", "XImage*");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XPutPixel", "int");
  Xen_check_type(Xen_is_integer(arg3), arg3, 3, "XPutPixel", "int");
  Xen_check_type(Xen_is_Pixel(arg4), arg4, 4, "XPutPixel", "Pixel");
  return(C_int_to_Xen_integer(XPutPixel(Xen_to_C_XImage(arg1), Xen_integer_to_C_int(arg2), Xen_integer_to_C_int(arg3), Xen_to_C_Pixel(arg4))));
}

static Xen gxm_XGetPixel(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XGetPixel "Pixel XGetPixel(ximage, x, y): returns the specified pixel from the named image."
  Xen_check_type(Xen_is_XImage(arg1), arg1, 1, "XGetPixel", "XImage*");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XGetPixel", "int");
  Xen_check_type(Xen_is_integer(arg3), arg3, 3, "XGetPixel", "int");
  return(C_to_Xen_Pixel(XGetPixel(Xen_to_C_XImage(arg1), Xen_integer_to_C_int(arg2), Xen_integer_to_C_int(arg3))));
}

static Xen gxm_XDestroyImage(Xen arg1)
{
  #define H_XDestroyImage "XDestroyImage(ximage) deallocates the memory associated with the XImage structure."
  Xen_check_type(Xen_is_XImage(arg1), arg1, 1, "XDestroyImage", "XImage*");
  return(C_int_to_Xen_integer(XDestroyImage(Xen_to_C_XImage(arg1))));
}



/* ----------------------------------------------------------------------------------------------------
 *
 *                            *         *  ***********  *         *  ***********
 *                             *       *   *             *       *        *
 *                              *     *    *              *     *         *
 *                               *   *     *               *   *          *
 *                                * *      *                * *           *
 *                                 *       *******           *            *
 *                                * *      *                * *           *
 *                               *   *     *               *   *          *
 *                              *     *    *              *     *         *
 *                             *       *   *             *       *        *
 *                            *         *  ***********  *         *       *
 * ----------------------------------------------------------------------------------------------------
 */

#if HAVE_XSHAPEQUERYEXTENSION

/* thanks to Michael Scholz! */

static Xen gxm_XShapeQueryExtension(Xen dpy)
{
  #define H_XShapeQueryExtension "(XShapeQueryExtension dpy): returns list of (Bool event_base error_base)"
  /* DIFF: (proposal)
     [C] Bool XShapeQueryExtension(Display *dpy, int *event_base, int *error_base)
     [XEN] (XShapeQueryExtension dpy): (list Bool event_base error_base)
  */
  int event_base = 0, error_base = 0;
  Bool ret = False;
  Xen_check_type(Xen_is_Display(dpy), dpy, 1, "XShapeQueryExtension", "Display*");
  ret = XShapeQueryExtension(Xen_to_C_Display(dpy), &event_base, &error_base);
  return(Xen_list_3(C_bool_to_Xen_boolean(ret), C_int_to_Xen_integer(event_base), C_int_to_Xen_integer(error_base)));
}

static Xen gxm_XShapeQueryVersion(Xen dpy)
{
  #define H_XShapeQueryVersion "(XShapeQueryVersion dpy): returns list of (Bool major_version minor_version)"
  int major = 0, minor = 0;
  Bool ret = False;
  Xen_check_type(Xen_is_Display(dpy), dpy, 1, "XShapeQueryVersion", "Display*");
  ret = XShapeQueryVersion(Xen_to_C_Display(dpy), &major, &minor);
  return(Xen_list_3(C_bool_to_Xen_boolean(ret), C_int_to_Xen_integer(major), C_int_to_Xen_integer(minor)));
}

static Xen gxm_XShapeQueryExtents(Xen dpy, Xen win)
{
  #define H_XShapeQueryExtents "(XShapeQueryExtents dpy win): returns list of (status bounding_shaped x_bound y_bound w_bound \
h_bound clip_shaped x_clip y_clip w_clip h_clip)"
  Bool bounding_shaped;
  int x_bounding;
  int y_bounding;
  unsigned int w_bounding;
  unsigned int h_bounding;
  Bool clip_shaped;
  int x_clip;
  int y_clip;
  unsigned int w_clip;
  unsigned int h_clip;
  Status ret;
  Xen_check_type(Xen_is_Display(dpy), dpy, 1, "XShapeQueryExtents", "Display*");
  Xen_check_type(Xen_is_Window(win), win, 2, "XShapeQueryExtents", "Window");
  ret = XShapeQueryExtents(Xen_to_C_Display(dpy), Xen_to_C_Window(win),
			   &bounding_shaped, &x_bounding, &y_bounding, &w_bounding, &h_bounding,
			   &clip_shaped, &x_clip, &y_clip, &w_clip, &h_clip);
  return(Xen_cons(C_int_to_Xen_integer(ret),
		  Xen_cons(C_int_to_Xen_integer(bounding_shaped),
			   Xen_list_9(C_int_to_Xen_integer(x_bounding),
				      C_int_to_Xen_integer(y_bounding),
				      C_int_to_Xen_integer(w_bounding),
				      C_int_to_Xen_integer(h_bounding),
				      C_int_to_Xen_integer(clip_shaped),
				      C_int_to_Xen_integer(x_clip),
				      C_int_to_Xen_integer(y_clip),
				      C_int_to_Xen_integer(w_clip),
				      C_int_to_Xen_integer(h_clip)))));
}

static Xen gxm_XShapeGetRectangles(Xen dpy, Xen win, Xen kind)
{
  #define H_XShapeGetRectangles "(XShapeGetRectangles dpy win kind): returns list of (xrectangles ordering)"
  int count = 0, ordering = 0;
  XRectangle *res;
  Xen_check_type(Xen_is_Display(dpy), dpy, 1, "XShapeGetRectangles", "Display*");
  Xen_check_type(Xen_is_Window(win), win, 2, "XShapeGetRectangles", "Window");
  Xen_check_type(Xen_is_integer(kind), kind, 3, "XShapeGetRectangles", "int");
  res = XShapeGetRectangles(Xen_to_C_Display(dpy), Xen_to_C_Window(win), Xen_integer_to_C_int(kind), &count, &ordering);
  return(Xen_list_2(C_to_Xen_XRectangles(res, count), C_int_to_Xen_integer(ordering)));
}

static Xen gxm_XShapeOffsetShape(Xen dpy, Xen win, Xen kind, Xen x, Xen y)
{
  #define H_XShapeOffsetShape "(XShapeOffsetShape dpy win kind x-off y-off)"
  Xen_check_type(Xen_is_Display(dpy), dpy, 1, "XShapeOffsetShape", "Display*");
  Xen_check_type(Xen_is_Window(win), win, 2, "XShapeOffsetShape", "Window");
  Xen_check_type(Xen_is_integer(kind), kind, 3, "XShapeOffsetShape", "int");
  Xen_check_type(Xen_is_integer(x), x, 4, "XShapeOffsetShape", "int");
  Xen_check_type(Xen_is_integer(y), y, 5, "XShapeOffsetShape", "int");
  XShapeOffsetShape(Xen_to_C_Display(dpy), Xen_to_C_Window(win), 
		    Xen_integer_to_C_int(kind), Xen_integer_to_C_int(x), Xen_integer_to_C_int(y));
  return(Xen_false);
}

static Xen gxm_XShapeCombineRegion(Xen dpy, Xen win, Xen kind, Xen x, Xen y, Xen reg, Xen op)
{
  #define H_XShapeCombineRegion "(XShapeCombineRegion dpy win kind x-off y-off region op)"
  Xen_check_type(Xen_is_Display(dpy), dpy, 1, "XShapeCombineRegion", "Display*");
  Xen_check_type(Xen_is_Window(win), win, 2, "XShapeCombineRegion", "Window");
  Xen_check_type(Xen_is_integer(kind), kind, 3, "XShapeCombineRegion", "int");
  Xen_check_type(Xen_is_integer(x), x, 4, "XShapeCombineRegion", "int");
  Xen_check_type(Xen_is_integer(y), y, 5, "XShapeCombineRegion", "int");
  Xen_check_type(Xen_is_Region(reg), reg, 6, "XShapeCombineRegion", "Region");
  Xen_check_type(Xen_is_integer(op), op, 6, "XShapeCombineRegion", "int");
  XShapeCombineRegion(Xen_to_C_Display(dpy), Xen_to_C_Window(win), 
		      Xen_integer_to_C_int(kind), Xen_integer_to_C_int(x), Xen_integer_to_C_int(y),
		      Xen_to_C_Region(reg),
		      Xen_integer_to_C_int(op));
  return(Xen_false);
}

static Xen gxm_XShapeCombineMask(Xen dpy, Xen win, Xen kind, Xen x, Xen y, Xen pix, Xen op)
{
  #define H_XShapeCombineMask "(XShapeCombineMask dpy win kind x-off y-off pixmap op)"
  Xen_check_type(Xen_is_Display(dpy), dpy, 1, "XShapeCombineMask", "Display*");
  Xen_check_type(Xen_is_Window(win), win, 2, "XShapeCombineMask", "Window");
  Xen_check_type(Xen_is_integer(kind), kind, 3, "XShapeCombineMask", "int");
  Xen_check_type(Xen_is_integer(x), x, 4, "XShapeCombineMask", "int");
  Xen_check_type(Xen_is_integer(y), y, 5, "XShapeCombineMask", "int");
  Xen_check_type(Xen_is_Pixmap(pix), pix, 6, "XShapeCombineMask", "Pixmap");
  Xen_check_type(Xen_is_integer(op), op, 6, "XShapeCombineMask", "int");
  XShapeCombineMask(Xen_to_C_Display(dpy), Xen_to_C_Window(win), 
		    Xen_integer_to_C_int(kind), Xen_integer_to_C_int(x), Xen_integer_to_C_int(y),
		    Xen_to_C_Pixmap(pix),
		    Xen_integer_to_C_int(op));
  return(Xen_false);
}

static Xen gxm_XShapeCombineShape(Xen dpy, Xen win, Xen kind, Xen x, Xen y, Xen src, Xen src_kind, Xen op)
{
  #define H_XShapeCombineShape "(XShapeCombineShape dpy win kind x-off y-off src src_kind op)"
  Xen_check_type(Xen_is_Display(dpy), dpy, 1, "XShapeCombineShape", "Display*");
  Xen_check_type(Xen_is_Window(win), win, 2, "XShapeCombineShape", "Window");
  Xen_check_type(Xen_is_integer(kind), kind, 3, "XShapeCombineShape", "int");
  Xen_check_type(Xen_is_integer(x), x, 4, "XShapeCombineShape", "int");
  Xen_check_type(Xen_is_integer(y), y, 5, "XShapeCombineShape", "int");
  Xen_check_type(Xen_is_Window(src), src, 6, "XShapeCombineShape", "Window");
  Xen_check_type(Xen_is_integer(src_kind), src_kind, 7, "XShapeCombineShape", "int");
  Xen_check_type(Xen_is_integer(op), op, 8, "XShapeCombineShape", "int");
  XShapeCombineShape(Xen_to_C_Display(dpy), Xen_to_C_Window(win), 
		     Xen_integer_to_C_int(kind), Xen_integer_to_C_int(x), Xen_integer_to_C_int(y),
		     Xen_to_C_Window(src), Xen_integer_to_C_int(src_kind),
		     Xen_integer_to_C_int(op));
  return(Xen_false);
}

static Xen gxm_XShapeCombineRectangles(Xen dpy, Xen win, Xen kind, Xen x, Xen y, Xen rects, Xen n_rects, Xen op, Xen ordering)
{
  #define H_XShapeCombineRectangles "(XShapeCombineRectangles dpy win kind x-off y-off rectangles n-rects op ordering)"
  XRectangle *cr = NULL;
  Xen_check_type(Xen_is_Display(dpy), dpy, 1, "XShapeCombineRectangles", "Display*");
  Xen_check_type(Xen_is_Window(win), win, 2, "XShapeCombineRectangles", "Window");
  Xen_check_type(Xen_is_integer(kind), kind, 3, "XShapeCombineRectangles", "int");
  Xen_check_type(Xen_is_integer(x), x, 4, "XShapeCombineRectangles", "int");
  Xen_check_type(Xen_is_integer(y), y, 5, "XShapeCombineRectangles", "int");
  Xen_check_type(Xen_is_list(rects), rects, 6, "XShapeCombineRectangles", "list of XRectangles");
  Xen_check_type(Xen_is_integer(n_rects), n_rects, 7, "XShapeCombineRectangles", "int");
  Xen_check_type(Xen_is_integer(op), op, 8, "XShapeCombineRectangles", "int");
  Xen_check_type(Xen_is_integer(ordering), ordering, 9, "XShapeCombineRectangles", "int");
  cr = Xen_to_C_XRectangles(rects, Xen_integer_to_C_int(n_rects));
  XShapeCombineRectangles(Xen_to_C_Display(dpy), Xen_to_C_Window(win), 
			  Xen_integer_to_C_int(kind), Xen_integer_to_C_int(x), Xen_integer_to_C_int(y),
			  cr, Xen_integer_to_C_int(n_rects),
			  Xen_integer_to_C_int(op), Xen_integer_to_C_int(ordering));
  if (cr) free(cr);
  return(Xen_false);
}
#endif



/* ----------------------------------------------------------------------------------------------------
 *
 *                            *         *  ***********
 *                             *       *        *
 *                              *     *         *
 *                               *   *          *
 *                                * *           *
 *                                 *            *
 *                                * *           *
 *                               *   *          *
 *                              *     *         *
 *                             *       *        *
 *                            *         *       *
 * ----------------------------------------------------------------------------------------------------
 */


typedef enum {CANCEL_CONVERT, CONVERT, LOSE, DONE, CONVERT_INCR, LOSE_INCR, DONE_INCR} xm_selmap_t;
/* need a way to map from widget to selection proc */

typedef struct {
  Widget w;
  xm_selmap_t type;
  Xen proc;
} selmap;

static selmap *selmaps = NULL;
static int selmap_size = 0;
static int selmap_ctr = 0;
static void add_selmap(Widget w, xm_selmap_t type, Xen proc)
{
  if (selmap_size == 0)
    selmaps = (selmap *)calloc(8, sizeof(selmap));
  else
    {
      if (selmap_size == selmap_ctr)
	{
	  int i;
	  selmap_size += 8;
	  selmaps = (selmap *)realloc(selmaps, selmap_size * sizeof(selmap));
	  for (i = selmap_ctr; i < selmap_size; i++) selmaps[i].w = NULL;
	}
    }
  selmaps[selmap_ctr].w = w;
  selmaps[selmap_ctr].type = type;
  xm_protect(proc);
  selmaps[selmap_ctr++].proc = proc;
}

static Xen unselmap(Widget w, xm_selmap_t type)
{
  int i;
  for (i = 0; i < selmap_ctr; i++)
    if ((selmaps[i].w == w) &&
	(selmaps[i].type == type))
      return(selmaps[i].proc);
  return(Xen_false);
}

static void gxm_XtCancelConvertSelectionProc(Widget w, Atom *a1, Atom *a2, XtRequestId *id, XtPointer x) 
{
  Xen proc;
  proc = unselmap(w, CANCEL_CONVERT);
  if (Xen_is_procedure(proc))
    Xen_call_with_5_args(proc,
	       C_to_Xen_Widget(w),
	       C_to_Xen_Atom(*a1),
	       C_to_Xen_Atom(*a2),
	       C_to_Xen_XtRequestId(*id),
	       (Xen)x,
	       "CancelConvert");
} 

static Boolean gxm_XtConvertSelectionProc(Widget w, Atom *a1, Atom *a2, Atom *a3, XtPointer* x, unsigned long *l, int *i) 
{
  Xen proc;
  proc = unselmap(w, CONVERT);
  if (Xen_is_procedure(proc))
    {
      Xen val;
      val = Xen_call_with_3_args(proc,
		       C_to_Xen_Widget(w),
		       C_to_Xen_Atom(*a1),
		       C_to_Xen_Atom(*a2),
		       "ConvertSelection");
      (*a3) = Xen_to_C_Atom(Xen_list_ref(val, 1));
      (*x) = (XtPointer)(Xen_list_ref(val, 2));
      (*l) = Xen_ulong_to_C_ulong(Xen_list_ref(val, 3));
      (*i) = Xen_integer_to_C_int(Xen_list_ref(val, 4));
      return(Xen_boolean_to_C_bool(Xen_car(val)));
    }
  return(0);
}

static void gxm_XtLoseSelectionIncrProc(Widget w, Atom *a, XtPointer x) 
{
  Xen proc;
  proc = unselmap(w, LOSE_INCR);
  if (Xen_is_procedure(proc))
    Xen_call_with_3_args(proc,
	       C_to_Xen_Widget(w),
	       C_to_Xen_Atom(*a),
	       (Xen)x,
	       "LoseSelectionIncr");
}

static void gxm_XtLoseSelectionProc(Widget w, Atom *a) 
{
  Xen proc;
  proc = unselmap(w, LOSE);
  if (Xen_is_procedure(proc))
    Xen_call_with_2_args(proc,
	       C_to_Xen_Widget(w),
	       C_to_Xen_Atom(*a),
	       "LoseSelection");
}

static void gxm_XtSelectionDoneProc(Widget w, Atom *a1, Atom *a2) 
{
  Xen proc;
  proc = unselmap(w, DONE);
  if (Xen_is_procedure(proc))
    Xen_call_with_3_args(proc,
	       C_to_Xen_Widget(w),
	       C_to_Xen_Atom(*a1),
	       C_to_Xen_Atom(*a2),
	       "DoneSelection");
}

static void gxm_XtSelectionDoneIncrProc(Widget w, Atom *a1, Atom *a2, XtRequestId *i, XtPointer x) 
{
  Xen proc;
  proc = unselmap(w, DONE_INCR);
  if (Xen_is_procedure(proc))
    Xen_call_with_5_args(proc,
	       C_to_Xen_Widget(w),
	       C_to_Xen_Atom(*a1),
	       C_to_Xen_Atom(*a2),
	       C_to_Xen_XtRequestId(*i),
	       (Xen)x,
	       "DoneSelectionIncr");
}

static Xen gxm_XtAppUnlock(Xen arg1)
{
  #define H_XtAppUnlock "void XtAppUnlock(app_context) unlocks the application context."
  Xen_check_type(Xen_is_XtAppContext(arg1), arg1, 1, "XtAppUnlock", "XtAppContext");
  XtAppUnlock(Xen_to_C_XtAppContext(arg1));
  return(Xen_false);
}

static Xen gxm_XtAppLock(Xen arg1)
{
  #define H_XtAppLock "void XtAppLock(app_context) locks the application context including all its related displays and widgets."
  Xen_check_type(Xen_is_XtAppContext(arg1), arg1, 1, "XtAppLock", "XtAppContext");
  XtAppLock(Xen_to_C_XtAppContext(arg1));
  return(Xen_false);
}

static Xen gxm_XtToolkitThreadInitialize(void)
{
  #define H_XtToolkitThreadInitialize "Boolean XtToolkitThreadInitialize()"
  return(C_bool_to_Xen_boolean(XtToolkitThreadInitialize()));
}

static Xen gxm_XtGetDisplays(Xen arg1)
{
  #define H_XtGetDisplays "void XtGetDisplays(app_context): list of displays"
  /* DIFF: XtGetDisplays not arg2 arg3 returns list
   */
  unsigned int x;
  Display **ds;
  int i, loc;
  Xen lst = Xen_empty_list;
  Xen_check_type(Xen_is_XtAppContext(arg1), arg1, 1, "XtGetDisplays", "XtAppContext");
  XtGetDisplays(Xen_to_C_XtAppContext(arg1), &ds, &x);
  loc = xm_protect(lst);
  for (i = x - 1; i >= 0; i--)
    lst = Xen_cons(C_to_Xen_Display(ds[i]), lst);
  xm_unprotect_at(loc);
  return(lst);
}

static Xen gxm_XtGetApplicationNameAndClass(Xen arg1)
{
  #define H_XtGetApplicationNameAndClass "void XtGetApplicationNameAndClass(display): returns the application name \
and class passed to XtDisplayInitialize for the specified display."
  /* DIFF: XtGetApplicationNameAndClass omits and rtns args 2 and 3
   */
  char *name, *clas;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XtGetApplicationNameAndClass", "Display*");
  XtGetApplicationNameAndClass(Xen_to_C_Display(arg1), &name, &clas);
  return(Xen_list_2(C_string_to_Xen_string(name),
		    C_string_to_Xen_string(clas)));
}

static Xen gxm_XtUngrabPointer(Xen arg1, Xen arg2)
{
  #define H_XtUngrabPointer "void XtUngrabPointer(widget, time) calls XUngrabPointer with the specified time."
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XtUngrabPointer", "Widget");
  Xen_check_type(Xen_is_Time(arg2), arg2, 2, "XtUngrabPointer", "Time");
  XtUngrabPointer(Xen_to_C_Widget(arg1), Xen_to_C_Time(arg2));
  return(Xen_false);
}

static Xen gxm_XtGrabPointer(Xen arg1, Xen arg2, Xen arg3, Xen arg4, Xen arg5, Xen arg6, Xen arg7, Xen arg8)
{
  #define H_XtGrabPointer "int XtGrabPointer(widget, owner_events, event_mask, pointer_mode, keyboard_mode, confine_to, cursor, time) calls \
XGrabPointer specifying the widget's window as the grab window."
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XtGrabPointer", "Widget");
  Xen_check_type(Xen_is_boolean(arg2), arg2, 2, "XtGrabPointer", "boolean");
  Xen_check_type(Xen_is_ulong(arg3), arg3, 3, "XtGrabPointer", "unsigned int");
  Xen_check_type(Xen_is_integer(arg4), arg4, 4, "XtGrabPointer", "int");
  Xen_check_type(Xen_is_integer(arg5), arg5, 5, "XtGrabPointer", "int");
  Xen_check_type(Xen_is_Window(arg6), arg6, 6, "XtGrabPointer", "Window");
  Xen_check_type(Xen_is_Cursor(arg7), arg7, 7, "XtGrabPointer", "Cursor");
  Xen_check_type(Xen_is_Time(arg8), arg8, 8, "XtGrabPointer", "Time");
  return(C_int_to_Xen_integer(XtGrabPointer(Xen_to_C_Widget(arg1), 
				    Xen_boolean_to_C_bool(arg2), Xen_ulong_to_C_ulong(arg3), 
				    Xen_integer_to_C_int(arg4), Xen_integer_to_C_int(arg5), 
				    Xen_to_C_Window(arg6), Xen_to_C_Cursor(arg7), 
				    Xen_to_C_Time(arg8))));
}

static Xen gxm_XtUngrabButton(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XtUngrabButton "void XtUngrabButton(widget, button, modifiers) calls XUngrabButton specifying the widget's window as the ungrab \
window if the widget is realized. "
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XtUngrabButton", "Widget");
  Xen_check_type(Xen_is_ulong(arg2), arg2, 2, "XtUngrabButton", "unsigned int");
  Xen_check_type(Xen_is_Modifiers(arg3), arg3, 3, "XtUngrabButton", "Modifiers");
  XtUngrabButton(Xen_to_C_Widget(arg1), Xen_ulong_to_C_ulong(arg2), Xen_to_C_Modifiers(arg3));
  return(Xen_false);
}

static Xen gxm_XtGrabButton(Xen arg1, Xen arg2, Xen arg3, Xen arg4, Xen arg5, Xen arg6, Xen arg7, Xen arg8, Xen arg9)
{
  #define H_XtGrabButton "void XtGrabButton(widget, button, modifiers, owner_events, event_mask, pointer_mode, keyboard_mode, confine_to, cursor) \
calls XGrabButton specifying the widget's window as the grab window if the widget is realized."
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XtGrabButton", "Widget");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XtGrabButton", "int");
  Xen_check_type(Xen_is_Modifiers(arg3), arg3, 3, "XtGrabButton", "Modifiers");
  Xen_check_type(Xen_is_boolean(arg4), arg4, 4, "XtGrabButton", "boolean");
  Xen_check_type(Xen_is_ulong(arg5), arg5, 5, "XtGrabButton", "unsigned int");
  Xen_check_type(Xen_is_integer(arg6), arg6, 6, "XtGrabButton", "int");
  Xen_check_type(Xen_is_integer(arg7), arg7, 7, "XtGrabButton", "int");
  Xen_check_type(Xen_is_Window(arg8), arg8, 8, "XtGrabButton", "Window");
  Xen_check_type(Xen_is_Cursor(arg9), arg9, 9, "XtGrabButton", "Cursor");
  XtGrabButton(Xen_to_C_Widget(arg1), Xen_integer_to_C_int(arg2), 
	       Xen_to_C_Modifiers(arg3), Xen_boolean_to_C_bool(arg4), Xen_ulong_to_C_ulong(arg5), Xen_integer_to_C_int(arg6), Xen_integer_to_C_int(arg7), 
	       Xen_to_C_Window(arg8), Xen_to_C_Cursor(arg9));
  return(Xen_false);
}

static Xen gxm_XtUngrabKeyboard(Xen arg1, Xen arg2)
{
  #define H_XtUngrabKeyboard "void XtUngrabKeyboard(widget, time) calls XUngrabKeyboard with the specified time."
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XtUngrabKeyboard", "Widget");
  Xen_check_type(Xen_is_Time(arg2), arg2, 2, "XtUngrabKeyboard", "Time");
  XtUngrabKeyboard(Xen_to_C_Widget(arg1), Xen_to_C_Time(arg2));
  return(Xen_false);
}

static Xen gxm_XtGrabKeyboard(Xen arg1, Xen arg2, Xen arg3, Xen arg4, Xen arg5)
{
  #define H_XtGrabKeyboard "int XtGrabKeyboard(widget, owner_events, pointer_mode, keyboard_mode, time)"
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XtGrabKeyboard", "Widget");
  Xen_check_type(Xen_is_boolean(arg2), arg2, 2, "XtGrabKeyboard", "boolean");
  Xen_check_type(Xen_is_integer(arg3), arg3, 3, "XtGrabKeyboard", "int");
  Xen_check_type(Xen_is_integer(arg4), arg4, 4, "XtGrabKeyboard", "int");
  Xen_check_type(Xen_is_Time(arg5), arg5, 5, "XtGrabKeyboard", "Time");
  return(C_int_to_Xen_integer(XtGrabKeyboard(Xen_to_C_Widget(arg1), Xen_boolean_to_C_bool(arg2), Xen_integer_to_C_int(arg3), Xen_integer_to_C_int(arg4), Xen_to_C_Time(arg5))));
}

static Xen gxm_XtUngrabKey(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XtUngrabKey "void XtUngrabKey(widget, keycode, modifiers) calls XUngrabKey specifying the widget's window as the ungrab window \
if the widget is realized."
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XtUngrabKey", "Widget");
  Xen_check_type(Xen_is_KeyCode(arg2), arg2, 2, "XtUngrabKey", "KeyCode");
  Xen_check_type(Xen_is_Modifiers(arg3), arg3, 3, "XtUngrabKey", "Modifiers");
  XtUngrabKey(Xen_to_C_Widget(arg1), Xen_to_C_KeyCode(arg2), Xen_to_C_Modifiers(arg3));
  return(Xen_false);
}

static Xen gxm_XtGrabKey(Xen arg1, Xen arg2, Xen arg3, Xen arg4, Xen arg5, Xen arg6)
{
  #define H_XtGrabKey "void XtGrabKey(widget, keycode, modifiers, owner_events, pointer_mode, keyboard_mode) calls XGrabKey specifying the \
widget's window as the grab window if the widget is realized."
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XtGrabKey", "Widget");
  Xen_check_type(Xen_is_KeyCode(arg2), arg2, 2, "XtGrabKey", "KeyCode");
  Xen_check_type(Xen_is_Modifiers(arg3), arg3, 3, "XtGrabKey", "Modifiers");
  Xen_check_type(Xen_is_boolean(arg4), arg4, 4, "XtGrabKey", "boolean");
  Xen_check_type(Xen_is_integer(arg5), arg5, 5, "XtGrabKey", "int");
  Xen_check_type(Xen_is_integer(arg6), arg6, 6, "XtGrabKey", "int");
  XtGrabKey(Xen_to_C_Widget(arg1), Xen_to_C_KeyCode(arg2), Xen_to_C_Modifiers(arg3), Xen_boolean_to_C_bool(arg4), Xen_integer_to_C_int(arg5), Xen_integer_to_C_int(arg6));
  return(Xen_false);
}

static Xen gxm_XtCancelSelectionRequest(Xen arg1, Xen arg2)
{
  #define H_XtCancelSelectionRequest "void XtCancelSelectionRequest(requestor, selection)"
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XtCancelSelectionRequest", "Widget");
  Xen_check_type(Xen_is_Atom(arg2), arg2, 2, "XtCancelSelectionRequest", "Atom");
  XtCancelSelectionRequest(Xen_to_C_Widget(arg1), Xen_to_C_Atom(arg2));
  return(Xen_false);
}

static Xen gxm_XtSendSelectionRequest(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XtSendSelectionRequest "void XtSendSelectionRequest(requestor, selection, time)"
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XtSendSelectionRequest", "Widget");
  Xen_check_type(Xen_is_Atom(arg2), arg2, 2, "XtSendSelectionRequest", "Atom");
  Xen_check_type(Xen_is_Time(arg3), arg3, 3, "XtSendSelectionRequest", "Time");
  XtSendSelectionRequest(Xen_to_C_Widget(arg1), Xen_to_C_Atom(arg2), Xen_to_C_Time(arg3));
  return(Xen_false);
}

static Xen gxm_XtCreateSelectionRequest(Xen arg1, Xen arg2)
{
  #define H_XtCreateSelectionRequest "void XtCreateSelectionRequest(requestor, selection)"
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XtCreateSelectionRequest", "Widget");
  Xen_check_type(Xen_is_Atom(arg2), arg2, 2, "XtCreateSelectionRequest", "Atom");
  XtCreateSelectionRequest(Xen_to_C_Widget(arg1), Xen_to_C_Atom(arg2));
  return(Xen_false);
}

static Xen gxm_XtGetSelectionValuesIncremental(Xen arg1, Xen arg2, Xen arg3, Xen arg4, Xen arg5, Xen arg6, Xen arg7)
{
  #define H_XtGetSelectionValuesIncremental "void XtGetSelectionValuesIncremental(w, selection, targets, count, callback, client_data, time) \
is similar to XtGetSelectionValueIncremental except that it takes a list of target types and a list of client data and obtains the current \
value of the selection converted to each of the targets."
  /* DIFF: XtGetSelectionValuesIncremental arg 3 is list of Atoms
   */
  Atom *outs;
  int len, loc;
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XtGetSelectionValuesIncremental", "Widget");
  Xen_check_type(Xen_is_Atom(arg2), arg2, 2, "XtGetSelectionValuesIncremental", "Atom");
  Xen_check_type(Xen_is_list(arg3), arg3, 3, "XtGetSelectionValuesIncremental", "list of Atom");
  Xen_check_type(Xen_is_integer(arg4), arg4, 4, "XtGetSelectionValuesIncremental", "int");
  Xen_check_type(Xen_is_procedure(arg5) && (Xen_is_aritable(arg5, 7)), arg5, 5, "XtGetSelectionValuesIncremental", "XtSelectionCallbackProc");
  Xen_check_type(Xen_is_Time(arg7), arg7, 7, "XtGetSelectionValuesIncremental", "Time");
  len = Xen_integer_to_C_int(arg4);
  if (len <= 0) return(Xen_false);
  outs = Xen_to_C_Atoms(arg3, len);
  xm_XtSelectionCallback_Descr = Xen_list_2(arg5, arg6);
  loc = xm_protect(xm_XtSelectionCallback_Descr);
  XtGetSelectionValuesIncremental(Xen_to_C_Widget(arg1), 
				  Xen_to_C_Atom(arg2), outs, len, 
				  gxm_XtSelectionCallbackProc, 
				  (XtPointer *)arg6, Xen_to_C_Time(arg7));
  xm_unprotect_at(loc);
  free(outs);
  return(Xen_false);
}  

static Xen gxm_XtGetSelectionValueIncremental(Xen arg1, Xen arg2, Xen arg3, Xen arg4, Xen arg5, Xen arg6)
{
  #define H_XtGetSelectionValueIncremental "void XtGetSelectionValueIncremental(w, selection, target, callback, client_data, time) is similar \
to XtGetSelectionValue except that the selection_callback procedure will be called repeatedly upon delivery of multiple segments of the selection value."
  int loc;
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XtGetSelectionValueIncremental", "Widget");
  Xen_check_type(Xen_is_Atom(arg2), arg2, 2, "XtGetSelectionValueIncremental", "Atom");
  Xen_check_type(Xen_is_Atom(arg3), arg3, 3, "XtGetSelectionValueIncremental", "Atom");
  Xen_check_type(Xen_is_procedure(arg4) && (Xen_is_aritable(arg4, 7)), arg4, 4, "XtGetSelectionValueIncremental", "XtSelectionCallbackProc");
  Xen_check_type(Xen_is_Time(arg6), arg6, 6, "XtGetSelectionValueIncremental", "Time");
  xm_XtSelectionCallback_Descr = Xen_list_2(arg4, arg5);
  loc = xm_protect(xm_XtSelectionCallback_Descr);
  XtGetSelectionValueIncremental(Xen_to_C_Widget(arg1), 
				 Xen_to_C_Atom(arg2), Xen_to_C_Atom(arg3), 
				 gxm_XtSelectionCallbackProc, 
				 (XtPointer)arg5, Xen_to_C_Time(arg6));
  xm_unprotect_at(loc);
  return(Xen_false);
}

static Xen gxm_XtGetSelectionRequest(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XtGetSelectionRequest "XSelectionRequestEvent* XtGetSelectionRequest(w, selection, request_id)"
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XtGetSelectionRequest", "Widget");
  Xen_check_type(Xen_is_Atom(arg2), arg2, 2, "XtGetSelectionRequest", "Atom");
  Xen_check_type(Xen_is_XtRequestId(arg3), arg3, 3, "XtGetSelectionRequest", "XtRequestId");
  return(C_to_Xen_XSelectionRequestEvent(XtGetSelectionRequest(Xen_to_C_Widget(arg1), Xen_to_C_Atom(arg2), Xen_to_C_XtRequestId(arg3))));
}

static Xen gxm_XtAppGetSelectionTimeout(Xen arg1)
{
  #define H_XtAppGetSelectionTimeout "unsigned long XtAppGetSelectionTimeout(app_context): returns the current selection timeout value, in milliseconds."
  Xen_check_type(Xen_is_XtAppContext(arg1), arg1, 1, "XtAppGetSelectionTimeout", "XtAppContext");
  return(C_ulong_to_Xen_ulong(XtAppGetSelectionTimeout(Xen_to_C_XtAppContext(arg1))));
}

static Xen gxm_XtAppSetSelectionTimeout(Xen arg1, Xen arg2)
{
  #define H_XtAppSetSelectionTimeout "void XtAppSetSelectionTimeout(app_context, timeout) sets the app's selection timeout mechanism."
  Xen_check_type(Xen_is_XtAppContext(arg1), arg1, 1, "XtAppSetSelectionTimeout", "XtAppContext");
  Xen_check_type(Xen_is_ulong(arg2), arg2, 2, "XtAppSetSelectionTimeout", "ulong");
  XtAppSetSelectionTimeout(Xen_to_C_XtAppContext(arg1), Xen_ulong_to_C_ulong(arg2));
  return(Xen_false);
}

static Xen gxm_XtGetSelectionValues(Xen arg1, Xen arg2, Xen arg3, Xen arg4, Xen arg5, Xen arg6, Xen arg7)
{
  #define H_XtGetSelectionValues "void XtGetSelectionValues(w, selection, targets, count, callback, client_data, time) is similar to \
XtGetSelectionValue except that it takes a list of target types and a list of client data and obtains the current value of the selection \
converted to each of the targets."
  /* DIFF: XtGetSelectionValues arg 3 is list of Atoms
   */
  Atom *outs;
  int len, loc;
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XtGetSelectionValues", "Widget");
  Xen_check_type(Xen_is_Atom(arg2), arg2, 2, "XtGetSelectionValues", "Atom");
  Xen_check_type(Xen_is_list(arg3), arg3, 3, "XtGetSelectionValues", "list of Atom");
  Xen_check_type(Xen_is_integer(arg4), arg4, 4, "XtGetSelectionValues", "int");
  Xen_check_type(Xen_is_procedure(arg5) && (Xen_is_aritable(arg5, 7)), arg5, 5, "XtGetSelectionValues", "XtSelectionCallbackProc");
  Xen_check_type(Xen_is_Time(arg7), arg7, 7, "XtGetSelectionValues", "Time");
  len = Xen_integer_to_C_int(arg4);
  if (len <= 0) return(Xen_false);
  outs = Xen_to_C_Atoms(arg3, len);
  xm_XtSelectionCallback_Descr = Xen_list_2(arg5, arg6);
  loc = xm_protect(xm_XtSelectionCallback_Descr);
  XtGetSelectionValues(Xen_to_C_Widget(arg1), 
		       Xen_to_C_Atom(arg2), outs, len, 
		       gxm_XtSelectionCallbackProc, 
		       (XtPointer *)arg6, Xen_to_C_Time(arg7));
  xm_unprotect_at(loc);
  free(outs);
  return(Xen_false);
}

static Xen gxm_XtGetSelectionValue(Xen arg1, Xen arg2, Xen arg3, Xen arg4, Xen arg5, Xen arg6)
{
  #define H_XtGetSelectionValue "void XtGetSelectionValue(w, selection, target, callback, client_data, time) requests the value of the \
selection that has been converted to the target type. "
  int loc;
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XtGetSelectionValue", "Widget");
  Xen_check_type(Xen_is_Atom(arg2), arg2, 2, "XtGetSelectionValue", "Atom");
  Xen_check_type(Xen_is_Atom(arg3), arg3, 3, "XtGetSelectionValue", "Atom");
  Xen_check_type(Xen_is_procedure(arg4) && (Xen_is_aritable(arg4, 7)), arg4, 4, "XtGetSelectionValue", "XtSelectionCallbackProc");
  Xen_check_type(Xen_is_Time(arg6), arg6, 6, "XtGetSelectionValue", "Time");
  xm_XtSelectionCallback_Descr = Xen_list_2(arg4, arg5);
  loc = xm_protect(xm_XtSelectionCallback_Descr);
  XtGetSelectionValue(Xen_to_C_Widget(arg1), 
		      Xen_to_C_Atom(arg2), 
		      Xen_to_C_Atom(arg3), 
		      gxm_XtSelectionCallbackProc, 
		      (XtPointer)arg5, Xen_to_C_Time(arg6));
  xm_unprotect_at(loc);
  return(Xen_false);
}

static Xen gxm_XtDisownSelection(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XtDisownSelection "void XtDisownSelection(w, selection, time) informs the selection mechanism that the specified widget is to \
lose ownership of the selection."
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XtDisownSelection", "Widget");
  Xen_check_type(Xen_is_Atom(arg2), arg2, 2, "XtDisownSelection", "Atom");
  Xen_check_type(Xen_is_Time(arg3), arg3, 3, "XtDisownSelection", "Time");
  XtDisownSelection(Xen_to_C_Widget(arg1), Xen_to_C_Atom(arg2), Xen_to_C_Time(arg3));
  return(Xen_false);
}


/*-------- file predicate -------- */

/* a "Substitution" is a list '(char substitute), so pass a list of such lists below where a substitution array is required */

static Xen xm_filepredicate_proc;

static Boolean gxm_XtFilePredicate(String filename) 
{
  return(Xen_boolean_to_C_bool(Xen_call_with_1_arg(xm_filepredicate_proc,
				     C_string_to_Xen_string(filename),
				     __func__)));
}

static SubstitutionRec *gxm_make_subs(Xen lst_1)
{
  int len;
  SubstitutionRec *subs = NULL;
  len = Xen_list_length(lst_1);
  if (len > 0)
    {
      int i;
      Xen lst;
      lst = Xen_copy_arg(lst_1);
      subs = (SubstitutionRec *)calloc(len, sizeof(SubstitutionRec));
      for (i = 0; i < len; i++, lst = Xen_cdr(lst))
	{
	  if (!(Xen_is_list(Xen_car(lst))))
	    {
	      free(subs);
	      return(NULL);
	    }
	  subs[i].match = Xen_char_to_C_char(Xen_car(Xen_car(lst)));
	  subs[i].substitution = xen_strdup(Xen_string_to_C_string(Xen_cadr(Xen_car(lst))));
	}
    }
  return(subs);
}

/* (XtFindFile "/lib/%N:/usr/lib/%N:/usr/local/lib/%N" (list (list #\N "libxm.so")) 1 file-exists?) */

static Xen gxm_XtResolvePathname(Xen arg1, Xen arg2, Xen arg3, Xen arg4, Xen arg5, Xen arg6, Xen arg7, Xen arg8)
{
  #define H_XtResolvePathname "String XtResolvePathname(display, type, filename, suffix, path, substitutions, num_substitutions, predicate)"
  /* DIFF: XtResolvePathname args use #f for NULL
   *       (XtResolvePathname (XtDisplay (cadr (main-widgets))) "app-defaults" #f #f #f #f 0 #f)
   */
  int arg8_loc = -1;
  Xen res;
  char *str;
  SubstitutionRec *subs = NULL;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XtResolvePathname", "Display*");
  Xen_check_type(Xen_is_false(arg2) || Xen_is_string(arg2), arg2, 2, "XtResolvePathname", "char*");
  Xen_check_type(Xen_is_false(arg3) || Xen_is_string(arg3), arg3, 3, "XtResolvePathname", "char*");
  Xen_check_type(Xen_is_false(arg4) || Xen_is_string(arg4), arg4, 4, "XtResolvePathname", "char*");
  Xen_check_type(Xen_is_false(arg5) || Xen_is_string(arg5), arg5, 5, "XtResolvePathname", "char*");
  Xen_check_type(Xen_is_false(arg6) || Xen_is_list(arg6), arg6, 6, "XtResolvePathname", "Substitution list");
  Xen_check_type(Xen_is_integer(arg7), arg7, 7, "XtResolvePathname", "int");
  Xen_check_type(Xen_is_false(arg8) || 
		  (Xen_is_procedure(arg8) && (Xen_is_aritable(arg8, 1))), 
		  arg8, 8, "XtResolvePathname", "XtFilePredicate (takes 1 arg)");
  if (Xen_is_list(arg6)) 
    {
      subs = gxm_make_subs(Xen_copy_arg(arg6));
      if (subs == NULL) return(Xen_false); /* type error? */
    }
  if (Xen_is_procedure(arg8))
    {
      arg8_loc = xm_protect(arg8);
      xm_filepredicate_proc = arg8;
    }
  str = XtResolvePathname(Xen_to_C_Display(arg1), 
			  (Xen_is_false(arg2)) ? NULL : (char *)Xen_string_to_C_string(arg2), 
			  (Xen_is_false(arg3)) ? NULL : (char *)Xen_string_to_C_string(arg3), 
			  (Xen_is_false(arg4)) ? NULL : (char *)Xen_string_to_C_string(arg4), 
			  (Xen_is_false(arg5)) ? NULL : (char *)Xen_string_to_C_string(arg5), 
			  subs,
			  Xen_integer_to_C_int(arg7), 
			  (Xen_is_false(arg8)) ? NULL : gxm_XtFilePredicate);
  if (Xen_is_procedure(arg8)) xm_unprotect_at(arg8_loc);
  if (subs) 
    {
      int i, len;
      len = Xen_list_length(arg6);
      for (i = 0; i < len; i++)
	if (subs[i].substitution) free(subs[i].substitution);
      free(subs);
    }
  res = C_string_to_Xen_string(str);
  if (str) XtFree(str);
  return(res);
}

static Xen gxm_XtFindFile(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XtFindFile "String XtFindFile(path, substitutions, num_substitutions, predicate) \
searches for a file using substitutions in the path list"
  char *str;
  Xen res;
  int arg4_loc = -1;
  SubstitutionRec *subs = NULL;
  Xen_check_type(Xen_is_string(arg1), arg1, 1, "XtFindFile", "char*");
  Xen_check_type(Xen_is_false(arg2) || Xen_is_list(arg2), arg2, 2, "XtFindFile", "Substitution list");
  Xen_check_type(Xen_is_integer(arg3), arg3, 3, "XtFindFile", "int");
  Xen_check_type(Xen_is_false(arg4) || 
		  (Xen_is_procedure(arg4) && (Xen_is_aritable(arg4, 1))), 
		  arg4, 4, "XtFindFile", "XtFilePredicate (takes 1 arg)");
  if (Xen_is_list(arg2)) 
    {
      subs = gxm_make_subs(Xen_copy_arg(arg2));
      if (subs == NULL) return(Xen_false); /* type error? */
    }
  if (Xen_is_procedure(arg4))
    {
      arg4_loc = xm_protect(arg4);
      xm_filepredicate_proc = arg4;
    }
  str = XtFindFile((char *)Xen_string_to_C_string(arg1), 
		   subs,
		   Xen_integer_to_C_int(arg3),
		   (Xen_is_false(arg4) ? NULL : gxm_XtFilePredicate));
  if (Xen_is_procedure(arg4)) xm_unprotect_at(arg4_loc);
  if (subs) 
    {
      int i, len;
      len = Xen_list_length(arg2);
      for (i = 0; i < len; i++)
	if (subs[i].substitution) free(subs[i].substitution);
      free(subs);
    }
  res = C_string_to_Xen_string(str);
  if (str) XtFree(str);
  return(res);
}

static Xen gxm_XtReleaseGC(Xen arg1, Xen arg2)
{
  #define H_XtReleaseGC "void XtReleaseGC(w, gc) deallocate the specified shared GC."
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XtReleaseGC", "Widget");
  Xen_check_type(Xen_is_GC(arg2), arg2, 2, "XtReleaseGC", "GC");
  XtReleaseGC(Xen_to_C_Widget(arg1), Xen_to_C_GC(arg2));
  return(Xen_false);
}

static Xen gxm_XtDestroyGC(Xen arg1)
{
  #define H_XtDestroyGC "XtDestroyGC(gc) is obsolete -- use XtReleaseGC"
  Xen_check_type(Xen_is_GC(arg1), arg1, 1, "XtDestroyGC", "GC");
  XtDestroyGC(Xen_to_C_GC(arg1));
  return(Xen_false);
}

static Xen gxm_XtAllocateGC(Xen arg1, Xen arg2, Xen arg3, Xen arg4, Xen arg5, Xen arg6)
{
  #define H_XtAllocateGC "GC XtAllocateGC(w, depth, value_mask, values, dynamic_mask, unused_mask): returns a sharable GC that may be modified by the client."
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XtAllocateGC", "Widget");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XtAllocateGC", "int");
  Xen_check_type(Xen_is_ulong(arg3), arg3, 3, "XtAllocateGC", "XtGCMask");
  Xen_check_type(Xen_is_XGCValues(arg4), arg4, 4, "XtAllocateGC", "XGCValues*");
  Xen_check_type(Xen_is_ulong(arg5), arg5, 5, "XtAllocateGC", "XtGCMask");
  Xen_check_type(Xen_is_ulong(arg6), arg6, 6, "XtAllocateGC", "XtGCMask");
  return(C_to_Xen_GC(XtAllocateGC(Xen_to_C_Widget(arg1), Xen_integer_to_C_int(arg2), 
				  Xen_ulong_to_C_ulong(arg3), Xen_to_C_XGCValues(arg4), 
				  Xen_ulong_to_C_ulong(arg5), Xen_ulong_to_C_ulong(arg6))));
}

static Xen gxm_XtGetGC(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XtGetGC "GC XtGetGC(w, value_mask, values): returns a sharable, read-only GC."
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XtGetGC", "Widget");
  Xen_check_type(Xen_is_ulong(arg2), arg2, 2, "XtGetGC", "XtGCMask");
  Xen_check_type(Xen_is_XGCValues(arg3), arg3, 3, "XtGetGC", "XGCValues*");
  return(C_to_Xen_GC(XtGetGC(Xen_to_C_Widget(arg1), Xen_ulong_to_C_ulong(arg2), Xen_to_C_XGCValues(arg3))));
}


/* -------- background process -------- */

/* (80) XtWorkProc called (protected) until quits or removed explicitly */

#define C_to_Xen_XM_Background(Code, Context) \
  Xen_list_5(C_string_to_Xen_symbol("Background"), Code, Context, Xen_integer_zero, Xen_integer_zero)
#define XM_is_Background(Arg) is_wrapped("Background", Arg)

static Boolean gxm_XtWorkProc(XtPointer cdata)
{
  /* if true, quits */
  int val;
  Xen descr = (Xen)cdata;
  /* (list 'Background function context gc-loc id) */
  val = Xen_boolean_to_C_bool(Xen_call_with_1_arg(Xen_cadr(descr), 
				    Xen_caddr(descr),
				    __func__));
  if (val) xm_unprotect_at(Xen_integer_to_C_int(Xen_list_ref(descr, 3)));
  return(val);
}

static bool unprotect_workproc(Xen val, int loc, unsigned long id)
{
  if ((XM_is_Background(val)) &&
      ((XtWorkProcId)Xen_ulong_to_C_ulong(Xen_list_ref(val, 4)) == id))
    {
      xm_unprotect_at(loc);
      return(true);
    }
  return(false);
}

static Xen gxm_XtRemoveWorkProc(Xen arg1)
{
  #define H_XtRemoveWorkProc "void XtRemoveWorkProc(id) explicitly removes the specified background work procedure."
  XtWorkProcId id;
  Xen_check_type(Xen_is_XtWorkProcId(arg1), arg1, 1, "XtRemoveWorkProc", "XtWorkProcId");
  id = Xen_to_C_XtWorkProcId(arg1);
  XtRemoveWorkProc(id);
  map_over_protected_elements(unprotect_workproc, id);
  return(Xen_false);
}

static Xen gxm_XtAppAddWorkProc(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XtAppAddWorkProc "XtWorkProcId XtAppAddWorkProc(app_context, proc, client_data) adds the specified work procedure for the \
application identified by app_context."
  XtWorkProcId id;
  int gc_loc;
  Xen descr;
  Xen_check_type(Xen_is_XtAppContext(arg1), arg1, 1, "XtAppAddWorkProc", "XtAppContext");
  Xen_check_type(Xen_is_procedure(arg2) && (Xen_is_aritable(arg2, 1)), arg2, 2, "XtAppAddWorkProc", "(XtWorkProc data)");
  
  descr = C_to_Xen_XM_Background(arg2, (Xen_is_bound(arg3)) ? arg3 : Xen_false);
  gc_loc = xm_protect(descr);
  id = XtAppAddWorkProc(Xen_to_C_XtAppContext(arg1), 
			gxm_XtWorkProc, 
			(XtPointer)descr);
  Xen_list_set(descr, 3, C_int_to_Xen_integer(gc_loc));
  Xen_list_set(descr, 4, C_ulong_to_Xen_ulong(id));
  return(C_to_Xen_XtWorkProcId(id));
}

/* the next 4 are needed where the caller allocates a block of memory, but X frees it (XCreateImage) --
 *   can't use Scheme-allocated memory here etc
 */
static Xen gxm_XtFree(Xen arg1)
{
  #define H_XtFree "void XtFree(ptr)"
  char *ptr;

  Xen_check_type(Xen_is_wrapped_c_pointer(arg1), arg1, 1, "XtFree", "pointer");

  ptr = (char *)Xen_unwrap_C_pointer(arg1);
  if (ptr) XtFree(ptr);
  return(Xen_false);
}

static Xen gxm_XtRealloc(Xen arg1, Xen arg2)
{
  #define H_XtRealloc "char *XtRealloc(ptr, num)"
  int num;
  char *ptr;

  Xen_check_type(Xen_is_wrapped_c_pointer(arg1), arg1, 1, "XtRealloc", "pointer");
  ptr = (char *)Xen_unwrap_C_pointer(arg1);
  if (!ptr) return(Xen_false);

  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XtRealloc", "int");
  num = Xen_integer_to_C_int(arg2);
  if (num <= 0)
    Xen_out_of_range_error("XtRealloc", 2, arg2, "num should be positive");

  return(C_ulong_to_Xen_ulong(XtRealloc(ptr, num)));
}

static Xen gxm_XtCalloc(Xen arg1, Xen arg2)
{
  #define H_XtCalloc "char *XtCalloc(num, size)"
  int num, size;

  Xen_check_type(Xen_is_integer(arg1), arg1, 1, "XtCalloc", "int");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XtCalloc", "int");
  num = Xen_integer_to_C_int(arg1);
  if (num <= 0)
    Xen_out_of_range_error("XtCalloc", 1, arg1, "num should be positive");
  size = Xen_integer_to_C_int(arg2);
  if (size <= 0)
    Xen_out_of_range_error("XtCalloc", 2, arg2, "size should be positive");

  return(Xen_wrap_C_pointer(XtCalloc(num, size))); /* dumb thing simply exits the main program on error! */
}

static Xen gxm_XtMalloc(Xen arg1)
{
  #define H_XtMalloc "char *XtMalloc(size)"
  int size;

  Xen_check_type(Xen_is_integer(arg1), arg1, 1, "XtMalloc", "int");
  size = Xen_integer_to_C_int(arg1);
  if (size <= 0)
    Xen_out_of_range_error("XtMalloc", 1, arg1, "size should be positive");

  return(Xen_wrap_C_pointer(XtMalloc(size)));
}

static Xen xm_XtErrorHandler;
static Xen xm_XtWarningHandler;

static void gxm_XtErrorHandler(String msg)
{
  if (Xen_is_procedure(xm_XtErrorHandler))
    Xen_call_with_1_arg(xm_XtErrorHandler, 
	       C_string_to_Xen_string(msg), 
	       __func__);
}

static Xen gxm_XtAppSetErrorHandler(Xen arg1, Xen arg2)
{
  #define H_XtAppSetErrorHandler "void XtAppSetErrorHandler(app_context, handler) registers the specified procedure, which is called when \
a fatal error condition occurs."
  Xen old_val;
  Xen_check_type(Xen_is_XtAppContext(arg1), arg1, 1, "XtAppSetErrorHandler", "XtAppContext");
  old_val = xm_XtErrorHandler;
  xm_protect(arg2);
  xm_XtErrorHandler = arg2;
  XtAppSetErrorHandler(Xen_to_C_XtAppContext(arg1), gxm_XtErrorHandler);
  if (Xen_is_procedure(old_val)) xm_unprotect(old_val);
  return(old_val);
}

static void gxm_XtWarningHandler(String msg)
{
  if (Xen_is_procedure(xm_XtWarningHandler))
    Xen_call_with_1_arg(xm_XtWarningHandler, 
	       C_string_to_Xen_string(msg), 
	       __func__);
}

static Xen gxm_XtAppSetWarningHandler(Xen arg1, Xen arg2)
{
  #define H_XtAppSetWarningHandler "void XtAppSetWarningHandler(app_context, handler) registers the specified procedure, which is called \
when a nonfatal error condition occurs."
  Xen old_val;
  Xen_check_type(Xen_is_XtAppContext(arg1), arg1, 1, "XtAppSetWarningHandler", "XtAppContext");
  old_val = xm_XtWarningHandler;
  xm_protect(arg2);
  xm_XtWarningHandler = arg2;
  XtAppSetWarningHandler(Xen_to_C_XtAppContext(arg1), gxm_XtWarningHandler);
  if (Xen_is_procedure(old_val)) xm_unprotect(old_val);
  return(old_val);
}

static Xen gxm_XtAppError(Xen arg1, Xen arg2)
{
  #define H_XtAppError "void XtAppError(app_context, message) calls the installed error procedure and passes the specified message."
  Xen_check_type(Xen_is_XtAppContext(arg1), arg1, 1, "XtAppError", "XtAppContext");
  Xen_check_type(Xen_is_string(arg2), arg2, 2, "XtAppError", "char*");
  XtAppError(Xen_to_C_XtAppContext(arg1), (char *)Xen_string_to_C_string(arg2));
  return(Xen_false);
}


static Xen gxm_XtAppWarningMsg(Xen arg1, Xen arg2, Xen arg3, Xen arg4, Xen arg5, Xen arg6, Xen arg7)
{
  #define H_XtAppWarningMsg "void XtAppWarningMsg(app_context, name, type, class, default, params, num_params) calls the high-level error \
handler and passes the specified information."
  /* DIFF: XtAppWarningMsg takes final int not int*, arg6 is list of strings
   */
  int size;
  Cardinal csize;
  char **pars;
  Xen_check_type(Xen_is_XtAppContext(arg1), arg1, 1, "XtAppWarningMsg", "XtAppContext");
  Xen_check_type(Xen_is_string(arg2), arg2, 2, "XtAppWarningMsg", "char*");
  Xen_check_type(Xen_is_string(arg3), arg3, 3, "XtAppWarningMsg", "char*");
  Xen_check_type(Xen_is_string(arg4), arg4, 4, "XtAppWarningMsg", "char*");
  Xen_check_type(Xen_is_string(arg5), arg5, 5, "XtAppWarningMsg", "char*");
  Xen_check_type(Xen_is_list(arg6), arg6, 6, "XtAppWarningMsg", "list of String");
  Xen_check_type(Xen_is_integer(arg7), arg7, 7, "XtAppWarningMsg", "int");
  size = Xen_integer_to_C_int(arg7);
  if (size <= 0) return(Xen_false);
  pars = Xen_to_C_Strings(arg6, size);
  csize = (Cardinal)size;
  XtAppWarningMsg(Xen_to_C_XtAppContext(arg1), (char *)Xen_string_to_C_string(arg2), 
		  (char *)Xen_string_to_C_string(arg3), (char *)Xen_string_to_C_string(arg4), (char *)Xen_string_to_C_string(arg5), 
		  pars, &csize);
  free(pars);
  return(Xen_false);
}

static Xen gxm_XtAppErrorMsg(Xen arg1, Xen arg2, Xen arg3, Xen arg4, Xen arg5, Xen arg6, Xen arg7)
{
  #define H_XtAppErrorMsg "void XtAppErrorMsg(app_context, name, type, class, default, params, num_params) calls the high-level error \
handler and passes the specified information."
  /* DIFF: XtAppErrorMsg takes final int not int*, arg6 is list of strings
   */
  int size;
  Cardinal csize;
  char **pars;
  Xen_check_type(Xen_is_XtAppContext(arg1), arg1, 1, "XtAppErrorMsg", "XtAppContext");
  Xen_check_type(Xen_is_string(arg2), arg2, 2, "XtAppErrorMsg", "char*");
  Xen_check_type(Xen_is_string(arg3), arg3, 3, "XtAppErrorMsg", "char*");
  Xen_check_type(Xen_is_string(arg4), arg4, 4, "XtAppErrorMsg", "char*");
  Xen_check_type(Xen_is_string(arg5), arg5, 5, "XtAppErrorMsg", "char*");
  Xen_check_type(Xen_is_list(arg6), arg6, 6, "XtAppErrorMsg", "list of String");
  Xen_check_type(Xen_is_integer(arg7), arg7, 7, "XtAppErrorMsg", "int");
  size = Xen_integer_to_C_int(arg7);
  if (size <= 0) return(Xen_false);
  pars = Xen_to_C_Strings(arg6, size);
  csize = (Cardinal)size;
  XtAppErrorMsg(Xen_to_C_XtAppContext(arg1), (char *)Xen_string_to_C_string(arg2), 
		(char *)Xen_string_to_C_string(arg3), (char *)Xen_string_to_C_string(arg4), 
		(char *)Xen_string_to_C_string(arg5), pars, &csize);
  free(pars);
  return(Xen_false);
}

static Xen xm_XtErrorMsgHandler;
static Xen xm_XtWarningMsgHandler;

static void gxm_XtErrorMsgHandler(String name, String type, String clas, String defp, String *pars, Cardinal *num)
{
  /* DIFF: XtErrorMsgHandler takes list of string pars
   */
  if ((Xen_is_procedure(xm_XtErrorMsgHandler)) && (num) && ((pars) || (*num == 0)))
    {
      Xen lst = Xen_empty_list;
      int i, len, loc;
      loc = xm_protect(lst);
      len = (*num);
      for (i = len - 1; i >= 0; i--)
	lst = Xen_cons(C_string_to_Xen_string(pars[i]), lst);
      Xen_call_with_6_args(xm_XtErrorMsgHandler, 
		 C_string_to_Xen_string(name),
		 C_string_to_Xen_string(type),
		 C_string_to_Xen_string(clas),
		 C_string_to_Xen_string(defp),
		 lst,
		 C_int_to_Xen_integer(*num),
		 __func__);
      xm_unprotect_at(loc);
    }
}

static Xen gxm_XtAppSetErrorMsgHandler(Xen arg1, Xen arg2)
{
  #define H_XtAppSetErrorMsgHandler "void XtAppSetErrorMsgHandler(app_context, msg_handler) registers the specified procedure, which is called \
when a fatal error occurs."
  Xen old_val;
  Xen_check_type(Xen_is_XtAppContext(arg1), arg1, 1, "XtAppSetErrorMsgHandler", "XtAppContext");
  old_val = xm_XtErrorMsgHandler;
  xm_protect(arg2);
  xm_XtErrorMsgHandler = arg2;
  XtAppSetErrorMsgHandler(Xen_to_C_XtAppContext(arg1), gxm_XtErrorMsgHandler);
  if (Xen_is_procedure(old_val)) xm_unprotect(old_val);
  return(old_val);
}


static void gxm_XtWarningMsgHandler(String name, String type, String clas, String defp, String *pars, Cardinal *num)
{
  /* DIFF: XtWarningMsgHandler takes list of string pars
   */
  if ((Xen_is_procedure(xm_XtWarningMsgHandler)) && (num) && ((pars) || (*num == 0)))
    {
      Xen lst = Xen_empty_list;
      int i, len, loc;
      loc = xm_protect(lst);
      len = (*num);
      for (i = len - 1; i >= 0; i--)
	lst = Xen_cons(C_string_to_Xen_string(pars[i]), lst);
      Xen_call_with_6_args(xm_XtWarningMsgHandler, 
		 C_string_to_Xen_string(name),
		 C_string_to_Xen_string(type),
		 C_string_to_Xen_string(clas),
		 C_string_to_Xen_string(defp),
		 lst,
		 C_int_to_Xen_integer(*num),
		 __func__);
      xm_unprotect_at(loc);
    }
}

static Xen gxm_XtAppSetWarningMsgHandler(Xen arg1, Xen arg2)
{
  #define H_XtAppSetWarningMsgHandler "void XtAppSetWarningMsgHandler(app_context, msg_handler) registers the specified procedure, which \
is called when a nonfatal error condition occurs."
  Xen old_val;
  Xen_check_type(Xen_is_XtAppContext(arg1), arg1, 1, "XtAppSetWarningMsgHandler", "XtAppContext");
  old_val = xm_XtWarningMsgHandler;
  xm_protect(arg2);
  xm_XtWarningMsgHandler = arg2;
  XtAppSetWarningMsgHandler(Xen_to_C_XtAppContext(arg1), gxm_XtWarningMsgHandler);
  if (Xen_is_procedure(old_val)) xm_unprotect(old_val);
  return(old_val);
}

static Xen gxm_XtGetValues(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XtGetValues "void XtGetValues(Widget w, ArgList args, Cardinal num_args): returns the values of the \
resources specified for the widget w.  In xm, 'args' is a list of resource names followed by a placeholder \
(normally 0), and the length of the arglist is optional.  For example, to get the current height and width \
of a widget, (" XM_PREFIX "XtGetValues" XM_POSTFIX " w (list " XM_PREFIX "XmNheight" XM_POSTFIX " 0 \
" XM_PREFIX "XmNwidth" XM_POSTFIX " 0)) which returns the list with the values filled in: (list height 123 \
width 321).  If the resource value is an array in C, it is returned as a list."

  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XtGetValues", "Widget");
  Xen_check_type(Xen_is_list(arg2), arg2, 2, "XtGetValues", "List");
  Xen_check_type(Xen_is_integer_or_unbound(arg3), arg3, 3, "XtGetValues", "int");
  return(gxm_XtGetValues_1(arg1, arg2, Xen_to_C_INT_DEF(arg3, arg2)));
}

static Xen gxm_XtVaGetValues(Xen arg1, Xen arg2)
{
  #define H_XtVaGetValues "void XtVaGetValues(w, ...) in xm is the same as XtGetValues."
  /* DIFF: XtVaGetValues -> returns original list with vals in place 
   */
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XtVaGetValues", "Widget");
  Xen_check_type(Xen_is_list(arg2), arg2, 2, "XtVaGetValues", "List");
  return(gxm_XtGetValues_1(arg1, arg2, Xen_list_length(arg2) / 2));
}

static Xen gxm_XtVaSetValues(Xen arg1, Xen arg2)
{
  #define H_XtVaSetValues "void XtVaSetValues(w, ...) in xm is the same as XtSetValues."
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XtVaSetValues", "Widget");
  Xen_check_type(Xen_is_list(arg2), arg2, 2, "XtVaSetValues", "List");
  {
    Widget w;
    Arg *args;
    int arglen;
    args = Xen_to_C_Args(arg2);
    w = Xen_to_C_Widget(arg1);
    arglen = Xen_list_length(arg2) / 2;
    XtSetValues(w, args, arglen);
    if (args)
      {
	fixup_args(w, args, arglen);
	free_args(args, arglen);
      }
  }
  return(Xen_false);
}

static Xen gxm_XtSetValues(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XtSetValues "void XtSetValues(Widget w, ArgList args, Cardinal num_args) sets the values of the \
resources specified for the widget w.  In xm, 'args' is a list of resource names followed by the new value, \
and the length of the arglist is optional.  For example, to set the current height and width \
of a widget, (" XM_PREFIX "XtSetValues" XM_POSTFIX " w (list " XM_PREFIX "XmNheight" XM_POSTFIX " 123 \
" XM_PREFIX "XmNwidth" XM_POSTFIX " 321)).  If the resource value in C is an array, it is a list in xm."

  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XtSetValues", "Widget");
  Xen_check_type(Xen_is_list(arg2), arg2, 2, "XtSetValues", "ArgList");
  Xen_check_type(Xen_is_integer_or_unbound(arg3), arg3, 3, "XtSetValues", "int");
  {
    Arg *args;
    int arglen;
    Widget w;
    w = Xen_to_C_Widget(arg1);
    args = Xen_to_C_Args(arg2);
    arglen = Xen_to_C_INT_DEF(arg3, arg2);
    XtSetValues(w, args, arglen);
    if (args)
      {
	fixup_args(w, args, arglen);
	free_args(args, arglen);
      }
  }
  return(Xen_false);
}

static Xen gxm_XtCloseDisplay(Xen arg1)
{
  #define H_XtCloseDisplay "void XtCloseDisplay(display) closes the specified display as soon as it is safe to do so."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XtCloseDisplay", "Display*");
  XtCloseDisplay(Xen_to_C_Display(arg1));
  return(Xen_false);
}

static Xen gxm_XtDisplayToApplicationContext(Xen arg1)
{
  #define H_XtDisplayToApplicationContext "XtAppContext XtDisplayToApplicationContext(dpy): returns the application context for the specified display."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XtDisplayToApplicationContext", "Display*");
  return(C_to_Xen_XtAppContext(XtDisplayToApplicationContext(Xen_to_C_Display(arg1))));
}

static Xen gxm_XtWidgetToApplicationContext(Xen arg1)
{
  #define H_XtWidgetToApplicationContext "XtAppContext XtWidgetToApplicationContext(w): returns the application context for the specified widget."
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XtWidgetToApplicationContext", "Widget");
  return(C_to_Xen_XtAppContext(XtWidgetToApplicationContext(Xen_to_C_Widget(arg1))));
}

static Xen gxm_XtInitializeWidgetClass(Xen arg1)
{
  #define H_XtInitializeWidgetClass "void XtInitializeWidgetClass(object_class)"
  Xen_check_type(Xen_is_WidgetClass(arg1), arg1, 1, "XtInitializeWidgetClass", "WidgetClass");
  XtInitializeWidgetClass(Xen_to_C_WidgetClass(arg1));
  return(Xen_false);
}

static Xen gxm_XtDestroyApplicationContext(Xen arg1)
{
  #define H_XtDestroyApplicationContext "void XtDestroyApplicationContext(app_context) destroys the specified application context as soon as it is safe to do so."
  Xen_check_type(Xen_is_XtAppContext(arg1), arg1, 1, "XtDestroyApplicationContext", "XtAppContext");
  XtDestroyApplicationContext(Xen_to_C_XtAppContext(arg1));
  return(Xen_false);
}

static Xen gxm_XtCreateApplicationContext(void)
{
  #define H_XtCreateApplicationContext "XtAppContext XtCreateApplicationContext()"
  return(C_to_Xen_XtAppContext(XtCreateApplicationContext()));
}

static Xen gxm_argv_to_list(Xen lst, int argc, char **argv)
{
  int i, loc;
  if (argc == 0) return(lst);
  loc = xm_protect(lst);
  for (i = argc - 1; i >= 0; i--)
    lst = Xen_cons(C_string_to_Xen_string(argv[i]), lst);
  free(argv);
  xm_unprotect_at(loc);
  return(lst);
}

static Xen gxm_XtOpenDisplay(Xen arg1, Xen arg2, Xen arg3, Xen arg4, Xen arg7, Xen arg8)
{
  #define H_XtOpenDisplay "Display *XtOpenDisplay(app_context, display_string, application_name, application_class, argc, argv) \
calls XOpenDisplay the specified display name."
  /* DIFF: XtOpenDisplay ignore arg5 6, argc is int, argv is list of strings, returns (list dpy argv ...)
   */
  char **argv = NULL;
  int argc;
  Xen lst = Xen_empty_list;
  Display *dpy;
  Xen_check_type(Xen_is_XtAppContext(arg1), arg1, 1, "XtOpenDisplay", "XtAppContext");
  Xen_check_type(Xen_is_string(arg2) || Xen_is_false(arg2), arg2, 2, "XtOpenDisplay", "char*");
  Xen_check_type(Xen_is_string(arg3), arg3, 3, "XtOpenDisplay", "char*");
  Xen_check_type(Xen_is_string(arg4), arg4, 4, "XtOpenDisplay", "char*");
  Xen_check_type(Xen_is_integer(arg7), arg7, 5, "XtOpenDisplay", "int");
  Xen_check_type(Xen_is_list(arg8), arg8, 6, "XtOpenDisplay", "list of char*");
  argc = Xen_integer_to_C_int(arg7);
  if (Xen_list_length(arg8) != argc) return(Xen_false); /* error? */
  if (argc > 0) argv = Xen_to_C_Strings(arg8, argc);
  dpy = XtOpenDisplay(Xen_to_C_XtAppContext(arg1), 
		      (Xen_is_false(arg2)) ? NULL : (char *)Xen_string_to_C_string(arg2), 
		      (char *)Xen_string_to_C_string(arg3), 
		      (char *)Xen_string_to_C_string(arg4), 
		      NULL, 0, &argc, argv);
  if (dpy)
    lst = Xen_cons(C_to_Xen_Display(dpy), lst);
  else lst = Xen_cons(Xen_false, lst);
  return(gxm_argv_to_list(lst, argc, argv));
}

static Xen gxm_XtAppSetFallbackResources(Xen app, Xen specs)
{
  #define H_XtAppSetFallbackResources "XtAppSetFallbackResources(app, list-of-strings) sets the app's default resource values \
from the list of strings"
  char **fallbacks;
  int i, len;
  Xen lst;
  Xen_check_type(Xen_is_XtAppContext(app), app, 1, "XtAppSetFallbackResources", "XtAppContext");
  Xen_check_type(Xen_is_list(specs), specs, 2, "XtAppSetFallbackResources", "list of char*");
  len = Xen_list_length(specs);
  lst = Xen_copy_arg(specs);
  fallbacks = (char **)calloc(len + 1, sizeof(char *)); /* +1 for null termination */
  for (i = 0; i < len; i++, lst = Xen_cdr(lst)) 
    fallbacks[i] = (char *)Xen_string_to_C_string(Xen_car(lst));
  XtAppSetFallbackResources(Xen_to_C_XtAppContext(app), fallbacks);
  free(fallbacks);
  return(app);
}

static Xen gxm_XtVaAppInitialize(Xen arg2, Xen arg5, Xen arg6, Xen arg8, Xen specs)
{
  #define H_XtVaAppInitialize "Widget XtVaAppInitialize(application_class, argc_in_out, argv_in_out, args, fallbacks) -- the order \
of the arguments is slightly different from the C Xt call.  The final arg is an (optional) list of strings."
  /* DIFF: XtVaAppInitialize [app] class {options numopts} {argc} argv resources -> (list widget app (new argv)), argc is int not int* options/num ignored
     Arg *args;
     app is returned not passed (list widget app) 
  */
  XtAppContext app;
  Arg *args;
  Widget res;
  int i, len = 0, argc, arglen;
  char **argv = NULL;
  char **fallbacks = NULL;
  Xen_check_type(Xen_is_string(arg2), arg2, 1, "XtVaAppInitialize", "char*");
  Xen_check_type(Xen_is_integer(arg5), arg5, 2, "XtVaAppInitialize", "int");
  Xen_check_type(Xen_is_list(arg6), arg6, 3, "XtVaAppInitialize", "list of String");
  Xen_check_type(Xen_is_list(arg8), arg8, 4, "XtVaAppInitialize", "arg list");
  Xen_check_type(Xen_is_list(specs) || !Xen_is_bound(specs), specs, 5, "XtVaAppInitialize", "list of char*");
  argc = Xen_integer_to_C_int(arg5);
  if (Xen_list_length(arg6) != argc) return(Xen_false); /* error? */
  if (argc > 0) argv = Xen_to_C_Strings(arg6, argc);
  if (Xen_is_list(specs))
    {
      Xen lst;
      int gcloc;
      len = Xen_list_length(specs);
      if (len <= 0) return(Xen_false);
      lst = Xen_copy_arg(specs);
      gcloc = xm_protect(lst);
      fallbacks = (char **)calloc(len + 1, sizeof(char *)); /* +1 for null termination */
      for (i = 0; i < len; i++, lst = Xen_cdr(lst)) 
	{
	  if (!Xen_is_string(Xen_car(lst)))
	    {
	      free(fallbacks);
	      xm_unprotect_at(gcloc);
	      return(Xen_false);
	    }
	  fallbacks[i] = xen_strdup(Xen_string_to_C_string(Xen_car(lst)));
	}
      xm_unprotect_at(gcloc);
    }
  args = Xen_to_C_Args(arg8);
  arglen = Xen_list_length(arg8) / 2;
  res = XtAppInitialize(&app, 
			(char *)Xen_string_to_C_string(arg2), 
			NULL,
			0,
			&argc, 
			argv,
			fallbacks,
			args,
			arglen);
  if (args)
    {
      fixup_args(res, args, arglen);
      free_args(args, arglen);
    }
  if (fallbacks) 
    {
      for (i = 0; i < len; i++)
	if (fallbacks[i]) free(fallbacks[i]);
      free(fallbacks);
    }
  return(Xen_list_3(C_to_Xen_Widget(res), 
		    C_to_Xen_XtAppContext(app),
		    gxm_argv_to_list(Xen_empty_list, argc, argv)));
}


static Xen gxm_XtAppInitialize(Xen arg2, Xen arg5, Xen arg6, Xen arg8, Xen arg9)
{
  #define H_XtAppInitialize "Widget XtAppInitialize(application_class, argc_in_out, argv_in_out, \
args, num_args) calls XtToolkitInitialize followed by XtCreateApplicationContext ,then calls XtOpenDisplay with \
display_string NULL and application_name NULL, and finally calls XtAppCreateShell with appcation_name NULL, widget_class applicationShellWidgetClass , \
and the specified args and num_args and returns the created shell.  The num_args argument can be list of strings = fallback resources"
  /* DIFF: XtAppInitialize [app] class {options numopts} {argc} argv resources args numargs -> (list widget app), argc is int not int* options/num ignored
     Arg *args;
     app is returned not passed (list widget app) 
  */
  XtAppContext app;
  Arg *args;
  Widget res;
  int argc, arglen;
  char **argv = NULL;
  char **fallbacks = NULL;
  int i, len = 0;
  Xen_check_type(Xen_is_string(arg2), arg2, 1, "XtAppInitialize", "char*");
  Xen_check_type(Xen_is_integer(arg5), arg5, 2, "XtAppInitialize", "int");
  Xen_check_type(Xen_is_list(arg6), arg6, 3, "XtAppInitialize", "list of String*");
  Xen_check_type(Xen_is_list(arg8), arg8, 4, "XtAppInitialize", "ArgList");
  Xen_check_type(Xen_is_integer(arg9) || Xen_is_list(arg9) || !Xen_is_bound(arg9), arg9, 5, "XtAppInitialize", "int or list of strings"); /* num_args */
  argc = Xen_integer_to_C_int(arg5);
  if (Xen_list_length(arg6) != argc) return(Xen_false); /* error? */
  if (argc > 0) argv = Xen_to_C_Strings(arg6, argc);
  args = Xen_to_C_Args(arg8);
  if (Xen_is_integer(arg9)) arglen = Xen_integer_to_C_int(arg9); else arglen = Xen_list_length(arg8) / 2;
  if (Xen_is_list(arg9))
    {
      Xen lst;
      int gcloc;
      len = Xen_list_length(arg9);
      lst = Xen_copy_arg(arg9);
      gcloc = xm_protect(lst);
      fallbacks = (char **)calloc(len + 1, sizeof(char *)); /* +1 for null termination */
      for (i = 0; i < len; i++, lst = Xen_cdr(lst)) 
	fallbacks[i] = xen_strdup(Xen_string_to_C_string(Xen_car(lst)));
      xm_unprotect_at(gcloc);
    }
  res = XtAppInitialize(&app,
			(char *)Xen_string_to_C_string(arg2), 
			NULL,
			0,
			&argc,
			argv,
			fallbacks,
			args,
			arglen);
  if (args)
    {
      fixup_args(res, args, arglen);
      free_args(args, arglen);
    }
  if (fallbacks) 
    {
      for (i = 0; i < len; i++)
	if (fallbacks[i]) free(fallbacks[i]);
      free(fallbacks);
    }
  return(Xen_list_3(C_to_Xen_Widget(res), 
		    C_to_Xen_XtAppContext(app),
		    gxm_argv_to_list(Xen_empty_list, argc, argv)));
}

static Xen gxm_XtVaOpenApplication(Xen arg1, Xen arg4, Xen arg5, Xen arg7, Xen arg8, Xen specs)
{
  #define H_XtVaOpenApplication "Widget XtVaOpenApplication(application_class, argc_in_out, argv_in_out, widget_class, args, fallbacks)"
  /* DIFF: XtVaOpenApplication [app] name {options numopts} {argc} argv resources class args -> (list widget app), argc is int not int* options/num ignored
     Arg *args;
     app is returned not passed (list widget app) 
  */
  XtAppContext app;
  Arg *args;
  Widget res;
  int argc, arglen;
  char **argv = NULL;
  char **fallbacks = NULL;
  int i, len = 0;
  Xen_check_type(Xen_is_string(arg1), arg1, 1, "XtVaOpenApplication", "char*");
  Xen_check_type(Xen_is_integer(arg4), arg4, 2, "XtVaOpenApplication", "int"); /* was arg3 by mistake, 11-Oct-02 */
  Xen_check_type(Xen_is_list(arg5), arg5, 3, "XtVaOpenApplication", "list of String");
  Xen_check_type(Xen_is_WidgetClass(arg7), arg7, 4, "XtVaOpenApplication", "WidgetClass");
  Xen_check_type(Xen_is_list(arg8), arg8, 5, "XtVaOpenApplication", "arg list");
  Xen_check_type(Xen_is_list(specs) || !Xen_is_bound(specs), specs, 5, "XtVaOpenApplication", "list of char*");
  argc = Xen_integer_to_C_int(arg4);
  if (Xen_list_length(arg5) != argc) return(Xen_false); /* error? */
  if (argc > 0) argv = Xen_to_C_Strings(arg5, argc);
  if (Xen_is_list(specs))
    {
      Xen lst;
      int gcloc;
      len = Xen_list_length(specs);
      lst = Xen_copy_arg(specs);
      gcloc = xm_protect(lst);
      fallbacks = (char **)calloc(len + 1, sizeof(char *)); /* +1 for null termination */
      for (i = 0; i < len; i++, lst = Xen_cdr(lst)) 
	fallbacks[i] = xen_strdup(Xen_string_to_C_string(Xen_car(lst)));
      xm_unprotect_at(gcloc);
    }
  args = Xen_to_C_Args(arg8);
  arglen = Xen_list_length(arg8) / 2;
  res = XtOpenApplication(&app, 
			  (char *)Xen_string_to_C_string(arg1), 
			  NULL,
			  0,
			  &argc,
			  argv,
			  fallbacks,
			  Xen_to_C_WidgetClass(arg7), 
			  args, 
			  arglen);
  if (args)
    {
      fixup_args(res, args, arglen);
      free_args(args, arglen);
    }
  if (fallbacks) 
    {
      for (i = 0; i < len; i++)
	if (fallbacks[i]) free(fallbacks[i]);
      free(fallbacks);
    }
  return(Xen_list_3(C_to_Xen_Widget(res), 
		    C_to_Xen_XtAppContext(app),
		    gxm_argv_to_list(Xen_empty_list, argc, argv)));
}

static Xen gxm_XtOpenApplication(Xen arg1, Xen arg4, Xen arg5, Xen arg7, Xen arg8, Xen arg9)
{
  #define H_XtOpenApplication "Widget XtOpenApplication(application_class, argc_in_out, argv_in_out, \
widget_class, args, num_args) calls XtToolkitInitialize followed by XtCreateApplicationContext , then calls XtOpenDisplay \
with display_string NULL and application_name NULL, and finally calls XtAppCreateShell with appcation_name NULL, widget_class \
applicationShellWidgetClass ,and the specified args and num_args and returns the created shell.  num_args can also be a list \
of fallback resources."
  /* DIFF: XtOpenApplication [app] name {options numopts} {argc} argv resources class args argnum -> (list widget app), argc is int not int* options/num ignored
  */
  XtAppContext app;
  Arg *args;
  Widget res;
  int argc, arglen;
  char **argv;
  char **fallbacks = NULL;
  int i, len = 0;
  Xen_check_type(Xen_is_string(arg1), arg1, 1, "XtOpenApplication", "char*");
  Xen_check_type(Xen_is_integer(arg4), arg4, 2, "XtOpenApplication", "int");
  Xen_check_type(Xen_is_list(arg5), arg5, 3, "XtOpenApplication", "list of String*");
  Xen_check_type(Xen_is_WidgetClass(arg7), arg7, 4, "XtOpenApplication", "WidgetClass");
  Xen_check_type(Xen_is_list(arg8), arg8, 8, "XtOpenApplication", "ArgList");
  Xen_check_type(Xen_is_integer(arg9) || Xen_is_list(arg9) || !Xen_is_bound(arg9), arg9, 5, "XtOpenApplication", "int or list of strings"); /* num_args */
  argc = Xen_integer_to_C_int(arg4);
  argv = Xen_to_C_Strings(arg5, argc);
  args = Xen_to_C_Args(arg8);
  if (Xen_is_integer(arg9)) arglen = Xen_integer_to_C_int(arg9); else arglen = Xen_list_length(arg8) / 2;
  if (Xen_is_list(arg9))
    {
      Xen lst;
      int gcloc;
      len = Xen_list_length(arg9);
      lst = Xen_copy_arg(arg9);
      gcloc = xm_protect(lst);
      fallbacks = (char **)calloc(len + 1, sizeof(char *)); /* +1 for null termination */
      for (i = 0; i < len; i++, lst = Xen_cdr(lst)) 
	fallbacks[i] = (char *)Xen_string_to_C_string(Xen_car(lst));
      xm_unprotect_at(gcloc);
    }
  res = XtOpenApplication(&app, (char *)Xen_string_to_C_string(arg1), 
			  NULL, 0, &argc,
			  argv, fallbacks, Xen_to_C_WidgetClass(arg7), 
			  args, arglen);
  if (args)
    {
      fixup_args(res, args, arglen);
      free_args(args, arglen);
    }
  if (fallbacks) 
    {
      for (i = 0; i < len; i++)
	if (fallbacks[i]) free(fallbacks[i]);
      free(fallbacks);
    }
  return(Xen_list_3(C_to_Xen_Widget(res), 
		    C_to_Xen_XtAppContext(app),
		    gxm_argv_to_list(Xen_empty_list, argc, argv)));
}

static Xen gxm_XtDisplayInitialize(Xen arg1, Xen arg2, Xen arg3, Xen arg4, Xen arg7, Xen arg8)
{
  #define H_XtDisplayInitialize "void XtDisplayInitialize(app_context, display, application_name, application_class, argc, argv) \
builds the resource database, calls the Xlib XrmParseCommand to parse the command line, and performs other per display initialization."
  /* DIFF: XtDisplayInitialize arg 5 6 ignored, argc is normal int, argv is list of strings, returns argv
   */
  char **argv;
  int argc;
  Xen_check_type(Xen_is_XtAppContext(arg1), arg1, 1, "XtDisplayInitialize", "XtAppContext");
  Xen_check_type(Xen_is_Display(arg2), arg2, 2, "XtDisplayInitialize", "Display*");
  Xen_check_type(Xen_is_string(arg3), arg3, 3, "XtDisplayInitialize", "char*");
  Xen_check_type(Xen_is_string(arg4), arg4, 4, "XtDisplayInitialize", "char*");
  Xen_check_type(Xen_is_integer(arg7), arg7, 5, "XtDisplayInitialize", "int");
  Xen_check_type(Xen_is_list(arg8), arg8, 6, "XtDisplayInitialize", "list of char*");
  argc = Xen_integer_to_C_int(arg7);
  argv = Xen_to_C_Strings(arg8, argc);
  XtDisplayInitialize(Xen_to_C_XtAppContext(arg1), 
		      Xen_to_C_Display(arg2), 
		      (char *)Xen_string_to_C_string(arg3), 
		      (char *)Xen_string_to_C_string(arg4), 
		      NULL, 0, 
		      &argc, argv);
  return(gxm_argv_to_list(Xen_empty_list, argc, argv));
}

/* -------- XtLanguage callback -------- */
/* (456) a global */

static Xen xm_language_proc;

static String gxm_XtLanguageProc(Display* d, String s, XtPointer context) 
{
  char *res;
  res = (char *)Xen_string_to_C_string(Xen_call_with_3_args(xm_language_proc,
				   C_to_Xen_Display(d),
				   C_string_to_Xen_string(s),
				   (Xen)context,
				   __func__));
  if (res)
    return(xen_strdup(res));
  return(NULL);
}

static Xen gxm_XtSetLanguageProc(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XtSetLanguageProc "Widget XtSetLanguageProc(app_context, proc, client_data)"
  /* arg can be null -> use default */
  /* DIFF: XtSetLanguageProc args1 and 2 use #f for NULL, return of #f means none was set
   */
  Xen previous_proc = Xen_false;
  Xen_check_type(Xen_is_false(arg1) || Xen_is_XtAppContext(arg1), arg1, 1, "XtSetLanguageProc", "XtAppContext");
  Xen_check_type(Xen_is_false(arg2) || (Xen_is_procedure(arg2) && (Xen_is_aritable(arg2, 3))), arg2, 2, "XtSetLanguageProc", "XtLanguageProc");
  previous_proc = xm_language_proc;
  if (Xen_is_procedure(previous_proc))
    xm_unprotect(previous_proc);
  if (Xen_is_procedure(arg2))
    xm_protect(arg2);
  if (Xen_is_false(arg1))
    {
      if (Xen_is_false(arg2))
	XtSetLanguageProc(NULL, NULL, NULL);
      else XtSetLanguageProc(NULL,
			     (XtLanguageProc)gxm_XtLanguageProc, 
			     (XtPointer)arg3);
    }
  else 
    {
      if (Xen_is_false(arg2))
	XtSetLanguageProc(Xen_to_C_XtAppContext(arg1), NULL, NULL);
      else  XtSetLanguageProc(Xen_to_C_XtAppContext(arg1), 
			      (XtLanguageProc)gxm_XtLanguageProc, 
			      (XtPointer)arg3);
    }
  return(previous_proc);
}

static Xen gxm_XtToolkitInitialize(void)
{
  #define H_XtToolkitInitialize "void XtToolkitInitialize()"
  XtToolkitInitialize();
  return(Xen_false);
}

static Xen gxm_XtVaAppCreateShell(Xen arg1, Xen arg2, Xen arg3, Xen arg4, Xen arg5)
{
  #define H_XtVaAppCreateShell "Widget XtVaAppCreateShell(application_name, application_class, widget_class, display, ...)"
  Arg *args;
  Widget w;
  int arglen;
  Xen_check_type(Xen_is_string(arg1), arg1, 1, "XtVaAppCreateShell", "char*");
  Xen_check_type(Xen_is_string(arg2), arg2, 2, "XtVaAppCreateShell", "char*");
  Xen_check_type(Xen_is_WidgetClass(arg3), arg3, 3, "XtVaAppCreateShell", "WidgetClass");
  Xen_check_type(Xen_is_Display(arg4), arg4, 4, "XtVaAppCreateShell", "Display*");
  Xen_check_type(Xen_is_list(arg5), arg5, 5, "XtVaAppCreateShell", "List");
  args = Xen_to_C_Args(arg5);
  arglen = Xen_list_length(arg5) / 2;
  w = XtAppCreateShell((char *)Xen_string_to_C_string(arg1), 
		       (char *)Xen_string_to_C_string(arg2), 
		       Xen_to_C_WidgetClass(arg3), 
		       Xen_to_C_Display(arg4), 
		       args,
		       arglen);
  if (args)
    {
      fixup_args(w, args, arglen);
      free_args(args, arglen);
    }
  return(C_to_Xen_Widget(w));
}

static Xen gxm_XtAppCreateShell(Xen arg1, Xen arg2, Xen arg3, Xen arg4, Xen arg5, Xen arg6)
{
  #define H_XtAppCreateShell "Widget XtAppCreateShell(application_name, application_class, widget_class, display, args, num_args) saves \
the specified application name and application class for qualifying all widget resource specifiers."
  Arg *args;
  Widget w;
  int arglen;
  Xen_check_type(Xen_is_string(arg1), arg1, 1, "XtAppCreateShell", "char*");
  Xen_check_type(Xen_is_string(arg2), arg2, 2, "XtAppCreateShell", "char*");
  Xen_check_type(Xen_is_WidgetClass(arg3), arg3, 3, "XtAppCreateShell", "WidgetClass");
  Xen_check_type(Xen_is_Display(arg4), arg4, 4, "XtAppCreateShell", "Display*");
  Xen_check_type(Xen_is_list(arg5), arg5, 5, "XtAppCreateShell", "ArgList");
  Xen_check_type(Xen_is_integer_or_unbound(arg6), arg6, 6, "XtAppCreateShell", "int");
  args = Xen_to_C_Args(arg5);
  arglen = Xen_to_C_INT_DEF(arg6, arg5);
  w = XtAppCreateShell((char *)Xen_string_to_C_string(arg1), 
		       (char *)Xen_string_to_C_string(arg2), 
		       Xen_to_C_WidgetClass(arg3), 
		       Xen_to_C_Display(arg4), 
		       args, 
		       arglen);
  if (args)
    {
      fixup_args(w, args, arglen);
      free_args(args, arglen);
    }
  return(C_to_Xen_Widget(w));
}

static Xen gxm_XtVaCreateManagedWidget(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XtVaCreateManagedWidget "Widget XtVaCreateManagedWidget(name, widget_class, parent, ...)"
  Arg *args;
  Widget w;
  int arglen;
  Xen_check_type(Xen_is_string(arg1), arg1, 1, "XtVaCreateManagedWidget", "char*");
  Xen_check_type(Xen_is_WidgetClass(arg2), arg2, 2, "XtVaCreateManagedWidget", "WidgetClass");
  Xen_check_type(Xen_is_Widget(arg3), arg3, 3, "XtVaCreateManagedWidget", "Widget");
  Xen_check_type(Xen_is_list(arg4), arg4, 4, "XtVaCreateManagedWidget", "List");
  args = Xen_to_C_Args(arg4);
  arglen = Xen_list_length(arg4) / 2;
  w = XtCreateManagedWidget((char *)Xen_string_to_C_string(arg1),
			    Xen_to_C_WidgetClass(arg2), 
			    Xen_to_C_Widget(arg3), 
			    args,
			    arglen);
  if (args)
    {
      fixup_args(w, args, arglen);
      free_args(args, arglen);
    }
  return(C_to_Xen_Widget(w));
}

static Xen gxm_XtVaCreateWidget(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XtVaCreateWidget "Widget XtVaCreateWidget(name, widget_class, parent, ...)"
  Arg *args;
  int arglen;
  Widget w;
  Xen_check_type(Xen_is_string(arg1), arg1, 1, "XtVaCreateWidget", "char*");
  Xen_check_type(Xen_is_WidgetClass(arg2), arg2, 2, "XtVaCreateWidget", "WidgetClass");
  Xen_check_type(Xen_is_Widget(arg3), arg3, 3, "XtVaCreateWidget", "Widget");
  Xen_check_type(Xen_is_list(arg4), arg4, 4, "XtVaCreateWidget", "List");
  args = Xen_to_C_Args(arg4);
  arglen = Xen_list_length(arg4) / 2;
  w = XtCreateWidget((char *)Xen_string_to_C_string(arg1), 
		     Xen_to_C_WidgetClass(arg2), 
		     Xen_to_C_Widget(arg3), 
		     args,
		     arglen);
  if (args)
    {
      fixup_args(w, args, arglen);
      free_args(args, arglen);
    }
  return(C_to_Xen_Widget(w));
}

static Xen gxm_XtCreateManagedWidget(Xen arg1, Xen arg2, Xen arg3, Xen arg4, Xen arg5)
{
  #define H_XtCreateManagedWidget "Widget XtCreateManagedWidget(name, widget_class, parent, args, num_args) is a routine \
that calls XtCreateWidget and XtManageChild."
  Arg *args;
  Widget w;
  int arglen;
  Xen_check_type(Xen_is_string(arg1), arg1, 1, "XtCreateManagedWidget", "char*");
  Xen_check_type(Xen_is_WidgetClass(arg2), arg2, 2, "XtCreateManagedWidget", "WidgetClass");
  Xen_check_type(Xen_is_Widget(arg3), arg3, 3, "XtCreateManagedWidget", "Widget");
  Xen_check_type(Xen_is_list(arg4), arg4, 4, "XtCreateManagedWidget", "ArgList");
  Xen_check_type(Xen_is_integer_or_unbound(arg5), arg5, 5, "XtCreateManagedWidget", "int");
  args = Xen_to_C_Args(arg4);
  arglen = Xen_to_C_INT_DEF(arg5, arg4);
  w = XtCreateManagedWidget((char *)Xen_string_to_C_string(arg1), 
			    Xen_to_C_WidgetClass(arg2), 
			    Xen_to_C_Widget(arg3), 
			    args, arglen);
  if (args)
    {
      fixup_args(w, args, arglen);
      free_args(args, arglen);
    }
  return(C_to_Xen_Widget(w));
}

static Xen gxm_XtCreateWidget(Xen arg1, Xen arg2, Xen arg3, Xen arg4, Xen arg5)
{
  #define H_XtCreateWidget "Widget XtCreateWidget(name, widget_class, parent, args, num_args) performs much of the boilerplate operations of widget creation."
  Arg *args;
  Widget w;
  int arglen;
  Xen_check_type(Xen_is_string(arg1), arg1, 1, "XtCreateWidget", "char*");
  Xen_check_type(Xen_is_WidgetClass(arg2), arg2, 2, "XtCreateWidget", "WidgetClass");
  Xen_check_type(Xen_is_Widget(arg3), arg3, 3, "XtCreateWidget", "Widget");
  Xen_check_type(Xen_is_list(arg4), arg4, 4, "XtCreateWidget", "ArgList");
  Xen_check_type(Xen_is_integer_or_unbound(arg5), arg5, 5, "XtCreateWidget", "int");
  args = Xen_to_C_Args(arg4);
  arglen = Xen_to_C_INT_DEF(arg5, arg4);
  w = XtCreateWidget((char *)Xen_string_to_C_string(arg1), 
		     Xen_to_C_WidgetClass(arg2), 
		     Xen_to_C_Widget(arg3), 
		     args, arglen);
  if (args)
    {
      fixup_args(w, args, arglen);
      free_args(args, arglen);
    }
  return(C_to_Xen_Widget(w));
}

static Xen gxm_XtCallbackPopdown(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XtCallbackPopdown "void XtCallbackPopdown(w, client_data, call_data) calls XtPopdown with the specified shell_widget and then \
calls XtSetSensitive to resensitize the enable_widget."
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XtCallbackPopdown", "Widget");
  XtCallbackPopdown(Xen_to_C_Widget(arg1), (XtPointer)arg2, (XtPointer)arg3);
  return(Xen_false);
}

static Xen gxm_XtPopdown(Xen arg1)
{
  #define H_XtPopdown "void XtPopdown(popup_shell)"
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XtPopdown", "Widget");
  XtPopdown(Xen_to_C_Widget(arg1));
  return(Xen_false);
}

static Xen gxm_XtCallbackExclusive(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XtCallbackExclusive "void XtCallbackExclusive(w, client_data, call_data)"
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XtCallbackExclusive", "Widget");
  XtCallbackExclusive(Xen_to_C_Widget(arg1), (XtPointer)arg2, (XtPointer)arg3);
  return(Xen_false);
}

static Xen gxm_XtCallbackNonexclusive(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XtCallbackNonexclusive "void XtCallbackNonexclusive(w, client_data, call_data)"
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XtCallbackNonexclusive", "Widget");
  XtCallbackNonexclusive(Xen_to_C_Widget(arg1), (XtPointer)arg2, (XtPointer)arg3);
  return(Xen_false);
}

static Xen gxm_XtCallbackNone(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XtCallbackNone "void XtCallbackNone(w, client_data, call_data)"
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XtCallbackNone", "Widget");
  XtCallbackNone(Xen_to_C_Widget(arg1), (XtPointer)arg2, (XtPointer)arg3);
  return(Xen_false);
}

static Xen gxm_XtPopupSpringLoaded(Xen arg1)
{
  #define H_XtPopupSpringLoaded "void XtPopupSpringLoaded(popup_shell)"
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XtPopupSpringLoaded", "Widget");
  XtPopupSpringLoaded(Xen_to_C_Widget(arg1));
  return(Xen_false);
}

static Xen gxm_XtPopup(Xen arg1, Xen arg2)
{
  #define H_XtPopup "void XtPopup(popup_shell, grab_kind)"
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XtPopup", "Widget");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XtPopup", "XtGrabKind");
  XtPopup(Xen_to_C_Widget(arg1), (XtGrabKind)Xen_integer_to_C_int(arg2));
  return(Xen_false);
}

static Xen gxm_XtVaCreatePopupShell(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XtVaCreatePopupShell "Widget XtVaCreatePopupShell(name, widget_class, parent, ...)"
  Arg *args;
  int arglen;
  Widget w;
  Xen_check_type(Xen_is_string(arg1), arg1, 1, "XtVaCreatePopupShell", "char*");
  Xen_check_type(Xen_is_WidgetClass(arg2), arg2, 2, "XtVaCreatePopupShell", "WidgetClass");
  Xen_check_type(Xen_is_Widget(arg3), arg3, 3, "XtVaCreatePopupShell", "Widget");
  Xen_check_type(Xen_is_list(arg4), arg4, 4, "XtVaCreatePopupShell", "List");
  args = Xen_to_C_Args(arg4);
  arglen = Xen_list_length(arg4) / 2;
  w = XtCreatePopupShell((char *)Xen_string_to_C_string(arg1), 
			 Xen_to_C_WidgetClass(arg2), 
			 Xen_to_C_Widget(arg3),
			 args,
			 arglen);
  if (args)
    {
      fixup_args(w, args, arglen);
      free_args(args, arglen);
    }
  return(C_to_Xen_Widget(w));
}

static Xen gxm_XtCreatePopupShell(Xen arg1, Xen arg2, Xen arg3, Xen arg4, Xen arg5)
{
  #define H_XtCreatePopupShell "Widget XtCreatePopupShell(name, widget_class, parent, args, num_args) ensures that the specified \
class is a subclass of Shell and, rather than using insert_child to attach the widget to the parent's children list, attaches the shell \
to the parent's pop-ups list directly."
  Widget w;
  Xen_check_type(Xen_is_string(arg1), arg1, 1, "XtCreatePopupShell", "char*");
  Xen_check_type(Xen_is_WidgetClass(arg2), arg2, 2, "XtCreatePopupShell", "WidgetClass");
  Xen_check_type(Xen_is_Widget(arg3), arg3, 3, "XtCreatePopupShell", "Widget");
  Xen_check_type(Xen_is_list(arg4), arg4, 4, "XtCreatePopupShell", "ArgList");
  Xen_check_type(Xen_is_integer_or_unbound(arg5), arg5, 5, "XtCreatePopupShell", "int");
  {
    Arg *args;
    int arglen;
    args = Xen_to_C_Args(arg4);
    arglen = Xen_to_C_INT_DEF(arg5, arg4);
    w = XtCreatePopupShell((char *)Xen_string_to_C_string(arg1), 
			   Xen_to_C_WidgetClass(arg2), 
			   Xen_to_C_Widget(arg3), 
			   args, arglen);
    if (args)
      {
	fixup_args(w, args, arglen);
	free_args(args, arglen);
      }
  }
  return(C_to_Xen_Widget(w));
}

static Xen gxm_XtHasCallbacks(Xen arg1, Xen arg2)
{
  #define H_XtHasCallbacks "XtCallbackStatus XtHasCallbacks(w, callback_name) first checks to see if the widget has a callback \
list identified by callback_name; returns XtCallbackNoList or XtCallbackHasNone if none, else XtCallbackHasSome"
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XtHasCallbacks", "Widget");
  Xen_check_type(Xen_is_string(arg2), arg2, 2, "XtHasCallbacks", "char*");
  return(C_int_to_Xen_integer(XtHasCallbacks(Xen_to_C_Widget(arg1), (char *)Xen_string_to_C_string(arg2))));
}

static Xen gxm_XtCallCallbacks(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XtCallCallbacks "void XtCallCallbacks(w, callback_name, call_data) calls each procedure that is registered in the \
specified widget's callback list. The call_data arg is assumed to be a callback struct reference"
  XtPointer val = NULL;
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XtCallCallbacks", "Widget");
  Xen_check_type(Xen_is_string(arg2), arg2, 2, "XtCallCallbacks", "char*");
  if (Xen_is_list(arg3)) val = (XtPointer)Xen_unwrap_C_pointer(Xen_cadr(arg3));
  XtCallCallbacks(Xen_to_C_Widget(arg1), (char *)Xen_string_to_C_string(arg2), val);
  return(Xen_false);
}

static Xen gxm_XtRemoveAllCallbacks(Xen arg1, Xen arg2)
{
  #define H_XtRemoveAllCallbacks "void XtRemoveAllCallbacks(w, callback_name) removes all the callback procedures from the specified widget's callback list."
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XtRemoveAllCallbacks", "Widget");
  Xen_check_type(Xen_is_string(arg2), arg2, 2, "XtRemoveAllCallbacks", "char*");
  XtRemoveAllCallbacks(Xen_to_C_Widget(arg1), (char *)Xen_string_to_C_string(arg2));
  return(Xen_false);
}

static Xen gxm_XtRemoveCallback(Xen arg1, Xen arg2, Xen arg4)
{
  #define H_XtRemoveCallback "void XtRemoveCallback(w, callback_name, client_data) removes a callback"
  /* DIFF: XtRemoveCallback omits proc arg and is passed whatever XtAddCallback returned
   */
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XtRemoveCallback", "Widget");
  Xen_check_type(Xen_is_string(arg2), arg2, 2, "XtRemoveCallback", "char*"); 
  xm_unprotect_at(Xen_integer_to_C_int(Xen_list_ref(arg4, CALLBACK_GC_LOC)));
  XtRemoveCallback(Xen_to_C_Widget(arg1), (char *)Xen_string_to_C_string(arg2), gxm_XtCallbackProc, (XtPointer)arg4);
  return(Xen_false);
}

static Xen gxm_XtRemoveCallbacks(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XtRemoveCallbacks "void XtRemoveCallbacks(w, callback_name, callbacks) removes the specified callback procedures from the \
specified widget's callback list. (The 3rd arg is a list of descriptors returned by XtAddCallback)."
  /* DIFF: XtRemoveCallbacks takes list of descriptors as arg3
   */
  Xen lst;
  int i, len;
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XtRemoveCallbacks", "Widget");
  Xen_check_type(Xen_is_string(arg2), arg2, 2, "XtRemoveCallbacks", "char*");
  Xen_check_type(Xen_is_list(arg3), arg3, 3, "XtRemoveCallbacks", "list of XtCallbacks");
  lst = Xen_copy_arg(arg3);
  len = Xen_list_length(lst);
  for (i = 0; i < len; i++, lst = Xen_cdr(lst))
    gxm_XtRemoveCallback(arg1, arg2, Xen_car(lst));
  return(Xen_false);
}

/* need these for all callback proc types, and locations to store the list of them */

enum {
  GXM_Any, GXM_Arrow, GXM_Combo, GXM_Command, GXM_Container_Outline, GXM_Container_Select,
  GXM_Convert, GXM_Destination, GXM_Display, GXM_Drag_Drop, GXM_Drag_Motion, GXM_Drag_Proc, GXM_Drag_Start,
  GXM_Drawing, GXM_Drawn, GXM_Drop_Finish, GXM_Drop_Proc, GXM_DropSite_Enter, GXM_DropSite_Leave, GXM_Drop_Start,
  GXM_File, GXM_List, GXM_Notebook, GXM_Operation, GXM_Popup, GXM_PushButton, GXM_RowColumn, GXM_Scale,
  GXM_ScrollBar, GXM_Selection, GXM_SpinBox, GXM_ToggleButton, GXM_TopLevel_Enter, GXM_TopLevel_Leave,
  GXM_Traverse, GXM_Verify
};

static Xen wrap_callback_struct(int type, XtPointer info)
{
  switch (type)
    {
    case GXM_Any:               return(C_to_Xen_XmAnyCallbackStruct((XmAnyCallbackStruct *)info));
    case GXM_Arrow:             return(C_to_Xen_XmArrowButtonCallbackStruct((XmArrowButtonCallbackStruct *)info));
    case GXM_Command:           return(C_to_Xen_XmCommandCallbackStruct((XmCommandCallbackStruct *)info));
    case GXM_Drag_Drop:         return(C_to_Xen_XmDragDropFinishCallbackStruct((XmDragDropFinishCallbackStruct *)info));
    case GXM_Drag_Motion:       return(C_to_Xen_XmDragMotionCallbackStruct((XmDragMotionCallbackStruct *)info));
    case GXM_Drag_Proc:         return(C_to_Xen_XmDragProcCallbackStruct((XmDragProcCallbackStruct *)info));
    case GXM_Drawing:           return(C_to_Xen_XmDrawingAreaCallbackStruct((XmDrawingAreaCallbackStruct *)info));
    case GXM_Drawn:             return(C_to_Xen_XmDrawnButtonCallbackStruct((XmDrawnButtonCallbackStruct *)info));
    case GXM_Drop_Finish:       return(C_to_Xen_XmDropFinishCallbackStruct((XmDropFinishCallbackStruct *)info));
    case GXM_Drop_Proc:         return(C_to_Xen_XmDropProcCallbackStruct((XmDropProcCallbackStruct *)info));
    case GXM_DropSite_Enter:    return(C_to_Xen_XmDropSiteEnterCallbackStruct((XmDropSiteEnterCallbackStruct *)info));
    case GXM_DropSite_Leave:    return(C_to_Xen_XmDropSiteLeaveCallbackStruct((XmDropSiteLeaveCallbackStruct *)info));
    case GXM_Drop_Start:        return(C_to_Xen_XmDropStartCallbackStruct((XmDropStartCallbackStruct *)info));
    case GXM_File:              return(C_to_Xen_XmFileSelectionBoxCallbackStruct((XmFileSelectionBoxCallbackStruct *)info));
    case GXM_List:              return(C_to_Xen_XmListCallbackStruct((XmListCallbackStruct *)info));
    case GXM_PushButton:        return(C_to_Xen_XmPushButtonCallbackStruct((XmPushButtonCallbackStruct *)info));
    case GXM_RowColumn:         return(C_to_Xen_XmRowColumnCallbackStruct((XmRowColumnCallbackStruct *)info));
    case GXM_Scale:             return(C_to_Xen_XmScaleCallbackStruct((XmScaleCallbackStruct *)info));
    case GXM_ScrollBar:         return(C_to_Xen_XmScrollBarCallbackStruct((XmScrollBarCallbackStruct *)info));
    case GXM_Selection:         return(C_to_Xen_XmSelectionBoxCallbackStruct((XmSelectionBoxCallbackStruct *)info));
    case GXM_ToggleButton:      return(C_to_Xen_XmToggleButtonCallbackStruct((XmToggleButtonCallbackStruct *)info));
    case GXM_Verify:            return(C_to_Xen_XmTextVerifyCallbackStruct((XmTextVerifyCallbackStruct *)info));
    case GXM_Popup:             return(C_to_Xen_XmPopupHandlerCallbackStruct((XmPopupHandlerCallbackStruct *)info));
    case GXM_Drag_Start:        return(C_to_Xen_XmDragStartCallbackStruct((XmDragStartCallbackStruct *)info));
    case GXM_Convert:           return(C_to_Xen_XmConvertCallbackStruct((XmConvertCallbackStruct *)info));
    case GXM_Container_Outline: return(C_to_Xen_XmContainerOutlineCallbackStruct((XmContainerOutlineCallbackStruct *)info));
    case GXM_Container_Select:  return(C_to_Xen_XmContainerSelectCallbackStruct((XmContainerSelectCallbackStruct *)info));
    case GXM_Destination:       return(C_to_Xen_XmDestinationCallbackStruct((XmDestinationCallbackStruct *)info));
    case GXM_Display:           return(C_to_Xen_XmDisplayCallbackStruct((XmDisplayCallbackStruct *)info));
    case GXM_Combo:             return(C_to_Xen_XmComboBoxCallbackStruct((XmComboBoxCallbackStruct *)info));
    case GXM_Notebook:          return(C_to_Xen_XmNotebookCallbackStruct((XmNotebookCallbackStruct *)info));
    case GXM_Operation:         return(C_to_Xen_XmOperationChangedCallbackStruct((XmOperationChangedCallbackStruct *)info));
    case GXM_SpinBox:           return(C_to_Xen_XmSpinBoxCallbackStruct((XmSpinBoxCallbackStruct *)info));
    case GXM_TopLevel_Enter:    return(C_to_Xen_XmTopLevelEnterCallbackStruct((XmTopLevelEnterCallbackStruct *)info));
    case GXM_TopLevel_Leave:    return(C_to_Xen_XmTopLevelLeaveCallbackStruct((XmTopLevelLeaveCallbackStruct *)info));
    case GXM_Traverse:          return(C_to_Xen_XmTraverseObscuredCallbackStruct((XmTraverseObscuredCallbackStruct *)info));
    }
  return(Xen_false);
}

static int callback_struct_type(Widget w, const char *name)
{
  if (strcmp(name, XmNdestinationCallback) == 0) return(GXM_Destination);
  if (strcmp(name, XmNpopupHandlerCallback) == 0) return(GXM_Popup);
  if (strcmp(name, XmNconvertCallback) == 0) return(GXM_Convert);
  if (strcmp(name, XmNdragStartCallback) == 0) return(GXM_Drag_Start);
  if (strcmp(name, XmNdropProc) == 0) return(GXM_Drop_Proc);      
  if (strcmp(name, XmNdragProc) == 0) return(GXM_Drag_Proc);      

  if (XmIsArrowButton(w)) return(GXM_Arrow);
  if (XmIsPushButton(w)) return(GXM_PushButton);
  if (XmIsScale(w)) return(GXM_Scale);
  if (XmIsRowColumn(w)) return(GXM_RowColumn);
  if (XmIsScrollBar(w)) return(GXM_ScrollBar);
  if (XmIsToggleButton(w)) return(GXM_ToggleButton);
  if (XmIsList(w)) return(GXM_List);
  if (XmIsFileSelectionBox(w)) return(GXM_File);
  if (XmIsDrawingArea(w)) return(GXM_Drawing);
  if (XmIsScrolledWindow(w)) return(GXM_Traverse);
  if (XmIsNotebook(w)) return(GXM_Notebook);
  if (XmIsComboBox(w)) return(GXM_Combo);
  if ((XmIsText(w)) || (XmIsTextField(w)))
    {
      if ((strcmp(name, XmNlosingFocusCallback) == 0) ||
	  (strcmp(name, XmNmodifyVerifyCallback) == 0) ||
	  (strcmp(name, XmNmotionVerifyCallback) == 0))
	return(GXM_Verify);
      return(GXM_Any);
    }
  if (XmIsContainer(w))
    {
      if (strcmp(name, XmNoutlineChangedCallback) == 0) return(GXM_Container_Outline);
      if ((strcmp(name, XmNselectionCallback) == 0) ||
	  (strcmp(name, XmNdefaultActionCallback) == 0))
	return(GXM_Container_Select);
      return(GXM_Destination); 
    }
  /* how to recognize a SpinBox? */
  if (XmIsCommand(w)) return(GXM_Command);
  if (XmIsDisplay(w)) return(GXM_Display);
  if (XmIsSelectionBox(w)) return(GXM_Selection);
  if (XmIsDragContext(w))
    {
      if (strcmp(name, XmNdragDropFinishCallback) == 0) return(GXM_Drag_Drop);
      if (strcmp(name, XmNdropFinishCallback) == 0) return(GXM_Drop_Finish);
      if (strcmp(name, XmNdragMotionCallback) == 0) return(GXM_Drag_Motion);
      if (strcmp(name, XmNdropSiteEnterCallback) == 0) return(GXM_DropSite_Enter);
      if (strcmp(name, XmNdropSiteLeaveCallback) == 0) return(GXM_DropSite_Leave);
      if (strcmp(name, XmNdropStartCallback) == 0) return(GXM_Drop_Start);
      if (strcmp(name, XmNoperationChangedCallback) == 0) return(GXM_Operation);
      if (strcmp(name, XmNtopLevelEnterCallback) == 0) return(GXM_TopLevel_Enter);
      if (strcmp(name, XmNtopLevelLeaveCallback) == 0) return(GXM_TopLevel_Leave);
    }
  if (XmIsDrawnButton(w)) return(GXM_Drawn);
  return(GXM_Any);
}

static Xen gxm_XtAddCallback(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XtAddCallback "void XtAddCallback(w, callback_name, callback, client_data) adds the specified callback procedure to \
the specified widget's callback list.  In xm, the client-data is optional, defaulting to " PROC_FALSE ". The callback procedure takes \
3 args: widget client-data callback-info. Returns a description of the callback suitable for use with XtRemoveCallback."
  /* DIFF: XtAddCallback returns the C-side "client-data" (for subsequent XtRemoveCallback)
   */
  char *name;
  Widget w;
  int gc_loc;
  Xen call_descr = Xen_empty_list;
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XtAddCallback", "Widget");
  Xen_check_type(Xen_is_string(arg2), arg2, 2, "XtAddCallback", "char*");
  Xen_check_type(Xen_is_procedure(arg3) && (Xen_is_aritable(arg3, 3)), arg3, 3, "XtAddCallback", "(XtCallbackProc widget data callb)");
  w = Xen_to_C_Widget(arg1);
  name = (char *)Xen_string_to_C_string(arg2);
  call_descr = C_to_Xen_XM_XtCallback(arg3, (Xen_is_bound(arg4)) ? arg4 : Xen_false);
  gc_loc = xm_protect(call_descr);
  Xen_list_set(call_descr, CALLBACK_GC_LOC, C_int_to_Xen_integer(gc_loc));
  Xen_list_set(call_descr, CALLBACK_STRUCT_TYPE, C_int_to_Xen_integer(callback_struct_type(w, name)));
  XtAddCallback(w, name, gxm_XtCallbackProc, (XtPointer)call_descr);
  return(call_descr);
}

static Xen gxm_XtAddCallbacks(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XtAddCallbacks "void XtAddCallbacks(w, callback_name, callbacks) add the specified list of callbacks to the specified widget's callback list. \
It returns a list of callback descriptors for use with XtRemoveCallback(s)."
  /* DIFF: XtAddCallbacks takes list of (func data) pairs as arg3
   */
  Xen res = Xen_empty_list, lst;
  int i, len;
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XtAddCallbacks", "Widget");
  Xen_check_type(Xen_is_string(arg2), arg2, 2, "XtAddCallbacks", "char*");
  Xen_check_type(Xen_is_list(arg3), arg3, 3, "XtAddCallbacks", "list of XtCallbacks");
  len = Xen_list_length(arg3);
  lst = Xen_copy_arg(arg3);
  for (i = len - 1; i >= 0; i--, lst = Xen_cdr(lst))
    res = Xen_cons(gxm_XtAddCallback(arg1, arg2, Xen_car(Xen_car(lst)), Xen_cadr(Xen_car(lst))),
		   res);
  return(res);
}

static Xen gxm_XtParent(Xen arg1)
{
  #define H_XtParent "Widget XtParent(w): returns the widget's parent widget ID."
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XtParent", "Widget");
  return(C_to_Xen_Widget(XtParent(Xen_to_C_Widget(arg1))));
}

static Xen gxm_XtClass(Xen arg1)
{
  #define H_XtClass "WidgetClass XtClass(w)"
  #define H_XtSuperclass "WidgetClass XtSuperclass(w): returns a pointer to the widget's superclass class structure."
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XtClass", "Widget");
  return(C_to_Xen_WidgetClass(XtClass(Xen_to_C_Widget(arg1))));
}

static Xen gxm_XtSuperclass(Xen arg1)
{
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XtSuperclass", "Widget");
  return(C_to_Xen_WidgetClass(XtSuperclass(Xen_to_C_Widget(arg1))));
}

static Xen gxm_XtName(Xen arg1)
{
  #define H_XtName "Widget XtName(w): returns the widget's name."
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XtName", "Widget");
  return(C_string_to_Xen_string(XtName(Xen_to_C_Widget(arg1))));
}

static Xen gxm_XtWindowOfObject(Xen arg1)
{
  #define H_XtWindowOfObject "Window XtWindowOfObject(object): returns the window of the specified object."
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XtWindowOfObject", "Widget");
  return(C_to_Xen_Window(XtWindowOfObject(Xen_to_C_Widget(arg1))));
}

static Xen gxm_XtWindow(Xen arg1)
{
  #define H_XtWindow "Window XtWindow(w): returns the window of the specified widget."
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XtWindow", "Widget");
  return(C_to_Xen_Window(XtWindow(Xen_to_C_Widget(arg1))));
}

static Xen gxm_XtScreenOfObject(Xen arg1)
{
  #define H_XtScreenOfObject "Screen *XtScreenOfObject(object): returns the screen pointer for the specified object."
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XtScreenOfObject", "Widget");
  return(C_to_Xen_Screen(XtScreenOfObject(Xen_to_C_Widget(arg1))));
}

static Xen gxm_XtScreen(Xen arg1)
{
  #define H_XtScreen "Screen* XtScreen(w): returns the screen pointer for the specified widget."
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XtScreen", "Widget");
  return(C_to_Xen_Screen(XtScreen(Xen_to_C_Widget(arg1))));
}

static Xen gxm_XtDisplayOfObject(Xen arg1)
{
  #define H_XtDisplayOfObject "Display *XtDisplayOfObject(object): returns the display pointer for the specified object."
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XtDisplayOfObject", "Widget");
  return(C_to_Xen_Display(XtDisplayOfObject(Xen_to_C_Widget(arg1))));
}

static Xen gxm_XtDisplay(Xen arg1)
{
  #define H_XtDisplay "Display* XtDisplay(w)"
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XtDisplay", "Widget");
  return(C_to_Xen_Display(XtDisplay(Xen_to_C_Widget(arg1))));
}

static Xen gxm_XtVaCreateArgsList(Xen arg1, Xen ignore2)
{
  #define H_XtVaCreateArgsList "XtVarArgsList XtVaCreateArgsList(unused, ...) allocates memory and copies its arguments into a single \
list pointer, which may be used with XtVaNestedList."
  /* DIFF: XtVaCreateArgsList just returns its first arg
   */
  return(arg1);
}

static Xen gxm_XtMergeArgLists(Xen arg1, Xen ignore2, Xen arg3, Xen ignore4)
{
  #define H_XtMergeArgLists "ArgList XtMergeArgLists(args1, num_args1, args2, num_args2) allocates enough storage to hold the combined \
ArgList structures and copies them into it."
  /* just merges, not duplicate check, kinda dumb to drop into C for that */
  Xen_check_type(Xen_is_list(arg1), arg1, 1, "XtMergeArgLists", "list");
  Xen_check_type(Xen_is_list(arg3), arg3, 3, "XtMergeArgLists", "list");
  return(Xen_append(arg1, arg3));
}

static Xen gxm_XtWindowToWidget(Xen arg1, Xen arg2)
{
  #define H_XtWindowToWidget "Widget XtWindowToWidget(display, window) translates the specified window and display pointer into the appropriate widget instance."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XtWindowToWidget", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XtWindowToWidget", "Window");
  return(C_to_Xen_Widget(XtWindowToWidget(Xen_to_C_Display(arg1), Xen_to_C_Window(arg2))));
}

static Xen gxm_XtNameToWidget(Xen arg1, Xen arg2)
{
  #define H_XtNameToWidget "Widget XtNameToWidget(reference, names) looks for a widget whose name is the first component in the specified names"
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XtNameToWidget", "Widget");
  Xen_check_type(Xen_is_string(arg2), arg2, 2, "XtNameToWidget", "char*");
  return(C_to_Xen_Widget(XtNameToWidget(Xen_to_C_Widget(arg1), (char *)Xen_string_to_C_string(arg2))));
}

static Xen gxm_XtSetSensitive(Xen arg1, Xen arg2)
{
  #define H_XtSetSensitive "void XtSetSensitive(w, sensitive)"
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XtSetSensitive", "Widget");
  Xen_check_type(Xen_is_boolean(arg2), arg2, 2, "XtSetSensitive", "boolean");
  XtSetSensitive(Xen_to_C_Widget(arg1), Xen_boolean_to_C_bool(arg2));
  return(Xen_false);
}

static Xen gxm_XtDestroyWidget(Xen arg1)
{
  #define H_XtDestroyWidget "void XtDestroyWidget(w) provides the only method of destroying a widget, including widgets that need to destroy themselves."
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XtDestroyWidget", "Widget");
  XtDestroyWidget(Xen_to_C_Widget(arg1));
  return(Xen_false);
}

static Xen gxm_XtUnrealizeWidget(Xen arg1)
{
  #define H_XtUnrealizeWidget "void XtUnrealizeWidget(w) destroys the windows of an existing widget and all of its children (recursively down the widget tree)."
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XtUnrealizeWidget", "Widget");
  XtUnrealizeWidget(Xen_to_C_Widget(arg1));
  return(Xen_false);
}

static Xen gxm_XtRealizeWidget(Xen arg1)
{
  #define H_XtRealizeWidget "void XtRealizeWidget(w) maps the widget window."
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XtRealizeWidget", "Widget");
  XtRealizeWidget(Xen_to_C_Widget(arg1));
  return(Xen_false);
}

static Xen gxm_XtAppPending(Xen arg1)
{
  #define H_XtAppPending "XtInputMask XtAppPending(app_context): returns a nonzero value if there are events pending from the X server, timer \
pending, or other input sources pending. "
  Xen_check_type(Xen_is_XtAppContext(arg1), arg1, 1, "XtAppPending", "XtAppContext");
  return(C_ulong_to_Xen_ulong(XtAppPending(Xen_to_C_XtAppContext(arg1))));
}

static Xen gxm_XtAppNextEvent(Xen arg1)
{
  #define H_XtAppNextEvent "void XtAppNextEvent(app_context) flushes the X output buffers of each Display in the application \
context and waits for an event while looking at the other input sources, timeout timeout values, and signal handlers and calling any callback \
procedures triggered by them -> event."
  /* DIFF: XtAppNextEvent app [ev] -> ev
   */
  XEvent *e;
  Xen_check_type(Xen_is_XtAppContext(arg1), arg1, 1, "XtAppNextEvent", "XtAppContext");
  e = (XEvent *)calloc(1, sizeof(XEvent));
  XtAppNextEvent(Xen_to_C_XtAppContext(arg1), e);
  return(C_to_Xen_XEvent_OBJ(e));
}



/* -------- Input callback -------- */

/* (77) explicitly removed via XtRemoveInput */

#define C_to_Xen_XM_Input(Code, Context) \
  Xen_list_5(C_string_to_Xen_symbol("Input"), Code, Context, Xen_integer_zero, Xen_integer_zero)
#define XM_is_Input(Arg) is_wrapped("Input", Arg)

static void gxm_XtInputCallbackProc(XtPointer cdata, int *fileno, XtInputId *id) 
{
  Xen descr = (Xen)cdata;
  /* (list 'Input function context gc-loc id) */
  Xen_call_with_3_args(Xen_cadr(descr), 
	     Xen_caddr(descr), 
	     C_ulong_to_Xen_ulong(*fileno), 
	     C_to_Xen_XtInputId(*id), 
	     __func__);
}

static bool unprotect_inputproc(Xen val, int loc, unsigned long id)
{
  if ((XM_is_Input(val)) &&
      ((XtInputId)Xen_ulong_to_C_ulong(Xen_list_ref(val, 4)) == id))
    {
      xm_unprotect_at(loc);
      return(true);
    }
  return(false);
}

static Xen gxm_XtRemoveInput(Xen arg1)
{
  #define H_XtRemoveInput "void XtRemoveInput(id) causes the read routine to stop watching for input from the input source."
  XtInputId id;
  Xen_check_type(Xen_is_XtInputId(arg1), arg1, 1, "XtRemoveInput", "XtInputId");
  id = Xen_to_C_XtInputId(arg1);
  XtRemoveInput(id);
  map_over_protected_elements(unprotect_inputproc, id);
  return(Xen_false);
}

static Xen gxm_XtAppAddInput(Xen arg1, Xen arg2, Xen arg3, Xen arg4, Xen arg5)
{
  #define H_XtAppAddInput "XtInputId XtAppAddInput(app_context, source, condition, proc, client_data) registers with the read routine a \
new source of events, which is usually file input but can also be file output."
  XtInputId id;
  int gc_loc;
  Xen descr;
  Xen_check_type(Xen_is_XtAppContext(arg1), arg1, 1, "XtAppAddInput", "XtAppContext");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XtAppAddInput", "int");
  Xen_check_type(Xen_is_integer(arg3), arg3, 3, "XtAppAddInput", "int");
  Xen_check_type(Xen_is_procedure(arg4) && (Xen_is_aritable(arg4, 3)), arg4, 4, "XtAppAddInput", "(XtInputCallbackProc data fileno id)");
  descr = C_to_Xen_XM_Input(arg4, (Xen_is_bound(arg5)) ? arg5 : Xen_false);
  gc_loc = xm_protect(descr);
  id = XtAppAddInput(Xen_to_C_XtAppContext(arg1), 
		     Xen_integer_to_C_int(arg2), 
#if (SIZEOF_VOID_P == 4)
		     (XtPointer)((int)Xen_ulong_to_C_ulong(arg3)),
#else
		     (XtPointer)Xen_llong_to_C_llong(arg3),
#endif
		     gxm_XtInputCallbackProc, 
		     (XtPointer)descr);
  Xen_list_set(descr, 3, C_int_to_Xen_integer(gc_loc));
  Xen_list_set(descr, 4, C_ulong_to_Xen_ulong(id));
  return(C_to_Xen_XtInputId(id));
}

/* -------- Timer Callback -------- */
/* (79) protect the function, then unprotect after invocation */

#define C_to_Xen_XM_TimeOut(Code, Context) \
  Xen_list_5(C_string_to_Xen_symbol("TimeOut"), Code, Context, Xen_integer_zero, Xen_integer_zero)
#define XM_is_TimeOut(Arg) is_wrapped("TimeOut", Arg)

static void gxm_XtTimerCallbackProc(XtPointer cdata, XtIntervalId* i) 
{
  Xen descr = (Xen)cdata;
  /* (list 'TimeOut function context gc-loc id) */
  int gc_loc;
  gc_loc = Xen_integer_to_C_int(Xen_list_ref(descr, 3));
  Xen_call_with_2_args(Xen_cadr(descr), 
	     Xen_caddr(descr),
	     C_to_Xen_XtIntervalId(*i),
	     __func__);
  xm_unprotect_at(gc_loc);
}

static bool unprotect_timeoutproc(Xen val, int loc, unsigned long id)
{
  if ((XM_is_TimeOut(val)) &&
      ((XtIntervalId)Xen_ulong_to_C_ulong(Xen_list_ref(val, 4)) == id))
    {
      xm_unprotect_at(loc);
      return(true);
    }
  return(false);
}

static Xen gxm_XtRemoveTimeOut(Xen arg1)
{
  #define H_XtRemoveTimeOut "void XtRemoveTimeOut(timer) removes the timeout."
  XtIntervalId id;
  Xen_check_type(Xen_is_XtIntervalId(arg1), arg1, 1, "XtRemoveTimeOut", "XtIntervalId");
  id = Xen_to_C_XtIntervalId(arg1);
  XtRemoveTimeOut(id);
  map_over_protected_elements(unprotect_timeoutproc, id);
  return(Xen_false);
}

static Xen gxm_XtAppAddTimeOut(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XtAppAddTimeOut "XtIntervalId XtAppAddTimeOut(app_context, interval, proc, client_data) creates a timeout and returns an identifier for it."
  XtIntervalId id;
  int gc_loc;
  Xen descr;
  Xen_check_type(Xen_is_XtAppContext(arg1), arg1, 1, "XtAppAddTimeOut", "XtAppContext");
  Xen_check_type(Xen_is_ulong(arg2), arg2, 2, "XtAppAddTimeOut", "ulong");
  Xen_check_type(Xen_is_procedure(arg3) && (Xen_is_aritable(arg3, 2)), arg3, 3, "XtAppAddTimeOut", "(XtTimerCallbackProc data id)");
  descr = C_to_Xen_XM_TimeOut(arg3, (Xen_is_bound(arg4)) ? arg4 : Xen_false);
  gc_loc = xm_protect(descr);
  id = XtAppAddTimeOut(Xen_to_C_XtAppContext(arg1), 
		       Xen_ulong_to_C_ulong(arg2), 
		       gxm_XtTimerCallbackProc, 
		       (XtPointer)descr);
  Xen_list_set(descr, 3, C_int_to_Xen_integer(gc_loc));
  Xen_list_set(descr, 4, C_ulong_to_Xen_ulong(id));
  return(C_to_Xen_XtIntervalId(id));
}

static Xen gxm_XtLastTimestampProcessed(Xen arg1)
{
  #define H_XtLastTimestampProcessed "Time XtLastTimestampProcessed(display): returns the timestamp of the last event"
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XtLastTimestampProcessed", "Display*");
  return(C_to_Xen_Time(XtLastTimestampProcessed(Xen_to_C_Display(arg1))));
}

static Xen gxm_XtLastEventProcessed(Xen arg1)
{
  #define H_XtLastEventProcessed "XEvent* XtLastEventProcessed(display): returns the last event passed to XtDispatchEvent for the \
specified display and NULL if there has been no event. "
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XtLastEventProcessed", "Display*");
  return(C_to_Xen_XEvent(XtLastEventProcessed(Xen_to_C_Display(arg1))));
}

static Xen gxm_XtGetKeyboardFocusWidget(Xen arg1)
{
  #define H_XtGetKeyboardFocusWidget "Widget XtGetKeyboardFocusWidget(widget): returns the widget that would be the end result of keyboard \
event forwarding for a keyboard event for the specified widget."
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XtGetKeyboardFocusWidget", "Widget");
  return(C_to_Xen_Widget(XtGetKeyboardFocusWidget(Xen_to_C_Widget(arg1))));
}

static Xen gxm_XtSetKeyboardFocus(Xen arg1, Xen arg2)
{
  #define H_XtSetKeyboardFocus "XtSetKeyboardFocus(subtree descendant) causes XtDispatchEvent to remap and send the event to the specified \
descendant widget."
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XtSetKeyboardFocus", "Widget");
  Xen_check_type(Xen_is_Widget(arg2), arg2, 2, "XtSetKeyboardFocus", "Widget");
  XtSetKeyboardFocus(Xen_to_C_Widget(arg1), Xen_to_C_Widget(arg2));
  return(Xen_false);
}

static Xen gxm_XtAddExposureToRegion(Xen arg1, Xen arg2)
{
  #define H_XtAddExposureToRegion "void XtAddExposureToRegion(event, region) computes the union of the rectangle defined by the exposure event \
and the specified region. Then, it stores the results back in region."
  Xen_check_type(Xen_is_XEvent(arg1), arg1, 1, "XtAddExposureToRegion", "XEvent*");
  Xen_check_type(Xen_is_Region(arg2), arg2, 2, "XtAddExposureToRegion", "Region");
  XtAddExposureToRegion(Xen_to_C_XEvent(arg1), Xen_to_C_Region(arg2));
  return(Xen_false);
}

static Xen gxm_XtAppMainLoop(Xen arg1)
{

  #define H_XtAppMainLoop "void XtAppMainLoop(app_context) first reads the next incoming X event by calling XtAppNextEvent and then it dispatches \
the event to the appropriate registered procedure by calling XtDispatchEvent."
  Xen_check_type(Xen_is_XtAppContext(arg1), arg1, 1, "XtAppMainLoop", "XtAppContext");
  XtAppMainLoop(Xen_to_C_XtAppContext(arg1));
  return(Xen_false);
}

static Xen gxm_XtAppProcessEvent(Xen arg1, Xen arg2)
{
  #define H_XtAppProcessEvent "void XtAppProcessEvent(app_context, mask) processes one timer, alternate input, signal source, or X event. \
If there is nothing of the appropriate type to process, XtAppProcessEvent blocks until there is."
  Xen_check_type(Xen_is_XtAppContext(arg1), arg1, 1, "XtAppProcessEvent", "XtAppContext");
  Xen_check_type(Xen_is_ulong(arg2), arg2, 2, "XtAppProcessEvent", "XtInputMask");
  XtAppProcessEvent(Xen_to_C_XtAppContext(arg1), Xen_ulong_to_C_ulong(arg2));
  return(Xen_false);
}

static Xen gxm_XtRemoveGrab(Xen arg1)
{
  #define H_XtRemoveGrab "void XtRemoveGrab(w) removes widgets from the modal cascade starting at the most recent widget up to and \
including the specified widget. "
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XtRemoveGrab", "Widget");
  XtRemoveGrab(Xen_to_C_Widget(arg1));
  return(Xen_false);
}

static Xen gxm_XtAddGrab(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XtAddGrab "void XtAddGrab(w, exclusive, spring_loaded) appends the widget (and associated parameters) to the modal cascade and \
checks that exclusive is " PROC_TRUE " if spring_loaded is " PROC_TRUE "."
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XtAddGrab", "Widget");
  Xen_check_type(Xen_is_boolean(arg2), arg2, 2, "XtAddGrab", "boolean");
  Xen_check_type(Xen_is_boolean(arg3), arg3, 3, "XtAddGrab", "boolean");
  XtAddGrab(Xen_to_C_Widget(arg1), Xen_boolean_to_C_bool(arg2), Xen_boolean_to_C_bool(arg3));
  return(Xen_false);
}

static Xen gxm_XtBuildEventMask(Xen arg1)
{
  #define H_XtBuildEventMask "EventMask XtBuildEventMask(w): widget's event mask"
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XtBuildEventMask", "Widget");
  return(C_ulong_to_Xen_ulong(XtBuildEventMask(Xen_to_C_Widget(arg1))));
}

static Xen gxm_XtDispatchEventToWidget(Xen arg1, Xen arg2)
{
  #define H_XtDispatchEventToWidget "Boolean XtDispatchEventToWidget(widget, event) scans the list of registered event handlers for the \
specified widget and calls each handler that has been registered for the specified event type, subject to the continue_to_dispatch value \
returned by each handler."
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XtDispatchEventToWidget", "Widget");
  Xen_check_type(Xen_is_XEvent(arg2), arg2, 2, "XtDispatchEventToWidget", "XEvent*");
  return(C_bool_to_Xen_boolean(XtDispatchEventToWidget(Xen_to_C_Widget(arg1), Xen_to_C_XEvent(arg2))));
}

enum {EVENT_HANDLER_TYPE, EVENT_HANDLER_FUNC, EVENT_HANDLER_DATA, EVENT_HANDLER_GC_LOC, EVENT_HANDLER_WIDGET, EVENT_HANDLER_MASK};

#define C_to_Xen_XM_XtEventHandler(Code, Context, Widget, Mask) \
  Xen_list_6(C_string_to_Xen_symbol("XtEventHandler"), Code, Context, Xen_integer_zero, Widget, Mask)
#define XM_is_XtEventHandler(Arg) is_wrapped("XtEventHandler", Arg)

static void gxm_XtEventHandler(Widget w, XtPointer context, XEvent *event, Boolean *flag)
{
  Xen result;
  Xen descr = (Xen)context;
  result = Xen_call_with_4_args(Xen_list_ref(descr, EVENT_HANDLER_FUNC),
		      C_to_Xen_Widget(w),
		      Xen_list_ref(descr, EVENT_HANDLER_DATA),
		      C_to_Xen_XEvent(event),
		      C_bool_to_Xen_boolean(*flag),
		      __func__);
  if ((Xen_is_symbol(result)) && (strcmp("done", Xen_symbol_to_C_string(result)) == 0))
    (*flag) = false;
}

static bool find_xteventproc_1(Xen val, int loc, unsigned long wd)
{
  Xen lst = (Xen)wd;
  unsigned long w;
  Xen code;
  Xen data;
  w = Xen_ulong_to_C_ulong(Xen_car(lst));
  code = Xen_cadr(lst);
  data = Xen_caddr(lst);
  return((XM_is_XtEventHandler(val)) &&
	 (Xen_is_Widget(Xen_list_ref(val, EVENT_HANDLER_WIDGET))) &&
	 (Xen_ulong_to_C_ulong(Xen_cadr(Xen_list_ref(val, EVENT_HANDLER_WIDGET))) == w) &&
	 (Xen_is_eq(code, Xen_list_ref(val, EVENT_HANDLER_FUNC))) &&
	 (Xen_is_eq(data, Xen_list_ref(val, EVENT_HANDLER_DATA))));
}

static Xen find_xteventproc(Widget w, Xen code, Xen data)
{
  /* here we again have to go by the widget */
  Xen lst;
  int i, loc;
  lst = Xen_list_3(C_ulong_to_Xen_ulong((unsigned long)w),code, data);
  loc = xm_protect(lst);
  i = map_over_protected_elements(find_xteventproc_1, (unsigned long)lst);
  xm_unprotect_at(loc);
  if (i >= 0)
    return(xm_protected_element(i));
  return(Xen_false);
}

static Xen gxm_XtInsertRawEventHandler(Xen arg1, Xen arg2, Xen arg3, Xen arg4, Xen arg5, Xen arg6)
{
  #define H_XtInsertRawEventHandler "void XtInsertRawEventHandler(w, event_mask, nonmaskable, proc, client_data, position) is similar to \
XtInsertEventHandler except that it does not modify the widget's event mask and never causes an XSelectInput for the specified events."
  Xen call_descr = Xen_empty_list;
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XtInsertRawEventHandler", "Widget");
  Xen_check_type(Xen_is_ulong(arg2), arg2, 2, "XtInsertRawEventHandler", "EventMask");
  Xen_check_type(Xen_is_boolean(arg3), arg3, 3, "XtInsertRawEventHandler", "boolean");
  Xen_check_type(Xen_is_procedure(arg4) && (Xen_is_aritable(arg4, 4)), arg4, 4, "XtInsertRawEventHandler", "XtEventHandler");
  Xen_check_type(Xen_is_integer(arg6), arg6, 6, "XtInsertRawEventHandler", "XtListPosition");
  call_descr = C_to_Xen_XM_XtEventHandler(arg4, (Xen_is_bound(arg5)) ? arg5 : Xen_false, arg1, arg2);
  Xen_list_set(call_descr, EVENT_HANDLER_GC_LOC, C_int_to_Xen_integer(xm_protect(call_descr)));
  XtInsertRawEventHandler(Xen_to_C_Widget(arg1), 
			  Xen_ulong_to_C_ulong(arg2), 
			  Xen_boolean_to_C_bool(arg3), 
			  gxm_XtEventHandler, 
			  (XtPointer)call_descr, 
			  (XtListPosition)Xen_integer_to_C_int(arg6));
  return(Xen_false);
}

static Xen gxm_XtInsertEventHandler(Xen arg1, Xen arg2, Xen arg3, Xen arg4, Xen arg5, Xen arg6)
{
  #define H_XtInsertEventHandler "void XtInsertEventHandler(w, event_mask, nonmaskable, proc, client_data, position) is identical to \
XtAddEventHandler with the additional position argument. "
  Xen call_descr = Xen_empty_list;
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XtInsertEventHandler", "Widget");
  Xen_check_type(Xen_is_ulong(arg2), arg2, 2, "XtInsertEventHandler", "EventMask");
  Xen_check_type(Xen_is_boolean(arg3), arg3, 3, "XtInsertEventHandler", "boolean");
  Xen_check_type(Xen_is_procedure(arg4) && (Xen_is_aritable(arg4, 4)), arg4, 4, "XtInsertEventHandler", "XtEventHandler");
  Xen_check_type(Xen_is_integer(arg6), arg6, 6, "XtInsertEventHandler", "XtListPosition");
  call_descr = C_to_Xen_XM_XtEventHandler(arg4, (Xen_is_bound(arg5)) ? arg5 : Xen_false, arg1, arg2);
  Xen_list_set(call_descr, EVENT_HANDLER_GC_LOC, C_int_to_Xen_integer(xm_protect(call_descr)));
  XtInsertEventHandler(Xen_to_C_Widget(arg1), 
		       Xen_ulong_to_C_ulong(arg2), 
		       Xen_boolean_to_C_bool(arg3), 
		       gxm_XtEventHandler, 
		       (XtPointer)call_descr, 
		       (XtListPosition)Xen_integer_to_C_int(arg6));
  return(Xen_false);
}

static Xen gxm_XtRemoveRawEventHandler(Xen arg1, Xen arg2, Xen arg3, Xen arg4, Xen arg5)
{
  #define H_XtRemoveRawEventHandler "void XtRemoveRawEventHandler(w, event_mask, nonmaskable, proc, client_data) stops the specified \
procedure from receiving the specified events."
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XtRemoveRawEventHandler", "Widget");
  Xen_check_type(Xen_is_ulong(arg2), arg2, 2, "XtRemoveRawEventHandler", "EventMask");
  Xen_check_type(Xen_is_boolean(arg3), arg3, 3, "XtRemoveRawEventHandler", "boolean");
  Xen_check_type(Xen_is_procedure(arg4) && (Xen_is_aritable(arg4, 4)), arg4, 4, "XtRemoveRawEventHandler", "XtEventHandler");
  XtRemoveRawEventHandler(Xen_to_C_Widget(arg1), 
			  Xen_ulong_to_C_ulong(arg2), 
			  Xen_boolean_to_C_bool(arg3),
			  gxm_XtEventHandler, 
			  (XtPointer)find_xteventproc(Xen_to_C_Widget(arg1), arg4, arg5));
  return(Xen_false);
}

static Xen gxm_XtAddRawEventHandler(Xen arg1, Xen arg2, Xen arg3, Xen arg4, Xen arg5)
{
  #define H_XtAddRawEventHandler "void XtAddRawEventHandler(w, event_mask, nonmaskable, proc, client_data) is similar to XtAddEventHandler \
except that it does not affect the widget's mask and never causes an XSelectInput for its events."
  Xen call_descr = Xen_empty_list;
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XtAddRawEventHandler", "Widget");
  Xen_check_type(Xen_is_ulong(arg2), arg2, 2, "XtAddRawEventHandler", "EventMask");
  Xen_check_type(Xen_is_boolean(arg3), arg3, 3, "XtAddRawEventHandler", "boolean");
  Xen_check_type(Xen_is_procedure(arg4) && (Xen_is_aritable(arg4, 4)), arg4, 4, "XtAddRawEventHandler", "XtEventHandler");
  call_descr = C_to_Xen_XM_XtEventHandler(arg4, (Xen_is_bound(arg5)) ? arg5 : Xen_false, arg1, arg2);
  Xen_list_set(call_descr, EVENT_HANDLER_GC_LOC, C_int_to_Xen_integer(xm_protect(call_descr)));
  XtAddRawEventHandler(Xen_to_C_Widget(arg1), 
		       Xen_ulong_to_C_ulong(arg2), 
		       Xen_boolean_to_C_bool(arg3), 
		       gxm_XtEventHandler,
		       (XtPointer)call_descr);
  return(Xen_false);
}

static Xen gxm_XtRemoveEventHandler(Xen arg1, Xen arg2, Xen arg3, Xen arg4, Xen arg5)
{
  #define H_XtRemoveEventHandler "XtRemoveEventHandler(w, event_mask, nonmaskable, proc, client_data)"
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XtRemoveEventHandler", "Widget");
  Xen_check_type(Xen_is_ulong(arg2), arg2, 2, "XtRemoveEventHandler", "EventMask");
  Xen_check_type(Xen_is_boolean(arg3), arg3, 3, "XtRemoveEventHandler", "boolean");
  Xen_check_type(Xen_is_procedure(arg4) && (Xen_is_aritable(arg4, 4)), arg4, 4, "XtRemoveEventHandler", "XtEventHandler");
  XtRemoveEventHandler(Xen_to_C_Widget(arg1), 
		       Xen_ulong_to_C_ulong(arg2), 
		       Xen_boolean_to_C_bool(arg3), 
		       gxm_XtEventHandler, 
		       (XtPointer)find_xteventproc(Xen_to_C_Widget(arg1), arg4, arg5));
  return(Xen_false);
}

static Xen gxm_XtAddEventHandler(Xen arg1, Xen arg2, Xen arg3, Xen arg4, Xen arg5)
{
  #define H_XtAddEventHandler "void XtAddEventHandler(w, event_mask, nonmaskable, proc, client_data) registers a procedure with the dispatch \
mechanism that is to be called when an event that matches the mask occurs on the specified widget.  To set the 'continue_to_dispatch' \
flag to 'false', return the symbol 'done from the event handler."
  Xen call_descr = Xen_empty_list;
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XtAddEventHandler", "Widget");
  Xen_check_type(Xen_is_ulong(arg2), arg2, 2, "XtAddEventHandler", "EventMask");
  Xen_check_type(Xen_is_boolean(arg3), arg3, 3, "XtAddEventHandler", "boolean");
  Xen_check_type(Xen_is_procedure(arg4) && (Xen_is_aritable(arg4, 4)), arg4, 4, "XtAddEventHandler", "XtEventHandler");
  call_descr = C_to_Xen_XM_XtEventHandler(arg4, (Xen_is_bound(arg5)) ? arg5 : Xen_false, arg1, arg2);
  Xen_list_set(call_descr, EVENT_HANDLER_GC_LOC, C_int_to_Xen_integer(xm_protect(call_descr)));
  XtAddEventHandler(Xen_to_C_Widget(arg1), 
		    Xen_ulong_to_C_ulong(arg2), 
		    Xen_boolean_to_C_bool(arg3), 
		    gxm_XtEventHandler,
		    (XtPointer)call_descr);
  return(Xen_false);
}

static Xen gxm_XtConvertCase(Xen arg1, Xen arg2)
{
  #define H_XtConvertCase "void XtConvertCase(display, keysym) calls the appropriate converter and returns the results."
  /* DIFF: XtConvertCase dpy keysym [k1 k2] -> (list k1 k2)
   */
  KeySym k1, k2;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XtConvertCase", "Display*");
  Xen_check_type(Xen_is_KeySym(arg2), arg2, 2, "XtConvertCase", "KeySym");
  XtConvertCase(Xen_to_C_Display(arg1), Xen_to_C_KeySym(arg2), &k1, &k2);
  return(Xen_list_2(C_to_Xen_KeySym(k1),
		    C_to_Xen_KeySym(k2)));
}

/* -------- case converter -------- */

/* (424) convert case XtRegisterCaseConverter global */

static Xen xm_XtCaseProc;

static void gxm_XtCaseProc(Display* d, KeySym k1, KeySym* k2, KeySym* k3)
{
  Xen val;
  int loc;
  val = Xen_call_with_2_args(xm_XtCaseProc,
		   C_to_Xen_Display(d),
		   C_to_Xen_KeySym(k1),
		   __func__);
  loc = xm_protect(val);
  if (Xen_is_list(val))
    {
      (*k2) = Xen_to_C_KeySym(Xen_car(val));
      (*k3) = Xen_to_C_KeySym(Xen_cadr(val));
    }
  xm_unprotect_at(loc);
}

static Xen gxm_XtRegisterCaseConverter(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  #define H_XtRegisterCaseConverter "void XtRegisterCaseConverter(display, proc, start, stop) registers the specified case converter."
  /* DIFF: XtRegisterCaseConverter user XtCaseProc should return the new KeySyms as a list (not as ref args)
   */
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XtRegisterCaseConverter", "Display*");
  Xen_check_type(Xen_is_procedure(arg2) && (Xen_is_aritable(arg2, 2)), arg2, 2, "XtRegisterCaseConverter", "(XtCaseProc dpy keysym)");
  Xen_check_type(Xen_is_KeySym(arg3), arg3, 3, "XtRegisterCaseConverter", "KeySym");
  Xen_check_type(Xen_is_KeySym(arg4), arg4, 4, "XtRegisterCaseConverter", "KeySym");
  if (Xen_is_procedure(xm_XtCaseProc)) xm_unprotect(xm_XtCaseProc);
  xm_protect(arg2);
  xm_XtCaseProc = arg2;
  XtRegisterCaseConverter(Xen_to_C_Display(arg1), gxm_XtCaseProc, Xen_to_C_KeySym(arg3), Xen_to_C_KeySym(arg4));
  return(Xen_false);
}


/* -------- keyproc -------- */
/* (454) XtSetKeyTranslator global */

static Xen xm_XtKeyProc;

static void gxm_XtKeyProc(Display *dpy, KeyCode c, Modifiers m, Modifiers *mp, KeySym *sym)
{
  Xen val;
  int loc;
  val = Xen_call_with_3_args(xm_XtKeyProc,
		   C_to_Xen_Display(dpy),
		   C_to_Xen_KeyCode(c),
		   C_to_Xen_Modifiers(m),
		   __func__);
  loc = xm_protect(val);
  if (Xen_is_list(val))
    {
      /* KeySym is long, Modifier(s) is int, so these can actually work, I guess */
      (*mp) = Xen_to_C_Modifiers(Xen_car(val));
      (*sym) = Xen_to_C_KeySym(Xen_cadr(val));
    }
  else XtTranslateKey(dpy, c, m, mp, sym);
  xm_unprotect_at(loc);
}

static Xen gxm_XtSetKeyTranslator(Xen arg1, Xen arg2)
{
  #define H_XtSetKeyTranslator "void XtSetKeyTranslator(display, proc) sets the specified procedure as the current key translator. "
  /* DIFF: XtSetKeyTranslator user XtKeyProc should return the new Modifiers and KeySym as a list (not as ref arg), set arg2 #f to get default proc
   */
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XtSetKeyTranslator", "Display*");
  Xen_check_type(Xen_is_false(arg2) || (Xen_is_procedure(arg2) && (Xen_is_aritable(arg2, 3))), arg2, 2, "XtSetKeyTranslator", "(XtKeyProc dpy key mod)");
  if (Xen_is_procedure(xm_XtKeyProc)) xm_unprotect(xm_XtKeyProc);
  xm_XtKeyProc = arg2;
  if (Xen_is_false(arg2))
    XtSetKeyTranslator(Xen_to_C_Display(arg1), XtTranslateKey);
  else
    {
      xm_protect(arg2);
      XtSetKeyTranslator(Xen_to_C_Display(arg1), (XtKeyProc)gxm_XtKeyProc);
    }
  return(Xen_false);
}

static Xen gxm_XtTranslateKey(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XtTranslateKey "XtTranslateKey(Display *, XtKeyCode, Modifiers): modifiers and keysym"
  /* DIFF: XtTranslateKey omit and rtn last 2 args
   */
  Modifiers m;
  KeySym k;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XtTranslateKey", "Display*");
  Xen_check_type(Xen_is_KeyCode(arg2), arg2, 2, "XtTranslateKey", "KeyCode");
  Xen_check_type(Xen_is_Modifiers(arg3), arg3, 3, "XtTranslateKey", "Modifiers");
  XtTranslateKey(Xen_to_C_Display(arg1), Xen_to_C_KeyCode(arg2), Xen_to_C_Modifiers(arg3), &m, &k);
  return(Xen_list_2(C_to_Xen_Modifiers(m),
		    C_to_Xen_KeySym(k)));
}

static Xen gxm_XtTranslateKeycode(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XtTranslateKeycode "void XtTranslateKeycode(display, keycode, modifiers) passes the specified \
arguments directly to the currently registered KeyCode to KeySym translator, returns (modifiers keysym)."
  /* DIFF: XtTranslateKeycode omit and rtn last 2 args
   */
  Modifiers m;
  KeySym k;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XtTranslateKeycode", "Display*");
  Xen_check_type(Xen_is_KeyCode(arg2), arg2, 2, "XtTranslateKeycode", "KeyCode");
  Xen_check_type(Xen_is_Modifiers(arg3), arg3, 3, "XtTranslateKeycode", "Modifiers");
  XtTranslateKeycode(Xen_to_C_Display(arg1), Xen_to_C_KeyCode(arg2), Xen_to_C_Modifiers(arg3), &m, &k);
  return(Xen_list_2(C_to_Xen_Modifiers(m),
		    C_to_Xen_KeySym(k)));
}

static Xen gxm_XtGetActionKeysym(Xen arg1)
{
  #define H_XtGetActionKeysym "KeySym XtGetActionKeysym(event): (keysym modifiers)"
  /* DIFF: XtGetActionKeysym omit and rtn last arg
   */
  Modifiers m;
  KeySym k;
  Xen_check_type(Xen_is_XEvent(arg1), arg1, 1, "XtGetActionKeysym", "XEvent*");
  k = XtGetActionKeysym(Xen_to_C_XEvent(arg1), &m);
  return(Xen_list_2(C_to_Xen_KeySym(k),
		    C_to_Xen_Modifiers(m)));
}

static Xen gxm_XtGetResourceList(Xen widget_class)
{
  #define H_XtGetResourceList "XtGetResourceList(widget-class): returns the widget class's resource list"
  Xen lst = Xen_empty_list;
  Cardinal len = 0;
  XtResourceList resources;
  Xen_check_type(Xen_is_WidgetClass(widget_class), widget_class, 1, "XtGetResourceList", "WidgetClass");
  XtGetResourceList(Xen_to_C_WidgetClass(widget_class), &resources, &len);
  if (len > 0)
    {
      int i;
      for (i = len - 1; i >= 0; i--)
	lst = Xen_cons(Xen_list_7(C_string_to_Xen_string(resources[i].resource_name),
				  C_string_to_Xen_string(resources[i].resource_class),
				  C_string_to_Xen_string(resources[i].resource_type),
				  C_int_to_Xen_integer((int)(resources[i].resource_size)),
				  C_int_to_Xen_integer((int)(resources[i].resource_offset)),
				  C_string_to_Xen_string(resources[i].default_type),
				  Xen_wrap_C_pointer(resources[i].default_addr)),
		       lst);
      XtFree((char *)resources);
    }
  return(lst);
}

static Xen gxm_XtGetMultiClickTime(Xen arg1)
{
  #define H_XtGetMultiClickTime "int XtGetMultiClickTime(display): returns the time in milliseconds that the translation manager uses to \
determine if multiple events are to be interpreted as a repeated event "
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XtGetMultiClickTime", "Display*");
  return(C_int_to_Xen_integer(XtGetMultiClickTime(Xen_to_C_Display(arg1))));
}

static Xen gxm_XtSetMultiClickTime(Xen arg1, Xen arg2)
{
  #define H_XtSetMultiClickTime "void XtSetMultiClickTime(display, time) sets the time interval used by the translation manager to determine \
when multiple events are interpreted as a repeated event."
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XtSetMultiClickTime", "Display*");
  Xen_check_type(Xen_is_integer(arg2), arg2, 2, "XtSetMultiClickTime", "int");
  XtSetMultiClickTime(Xen_to_C_Display(arg1), Xen_integer_to_C_int(arg2));
  return(Xen_false);
}

static Xen register_proc;

static void gxm_XtRegisterGrabActionProc(Widget w, XEvent* e, String* s, Cardinal* c) 
{
  Xen_call_with_3_args(register_proc,
	     C_to_Xen_Widget(w),
	     C_to_Xen_XEvent(e),
	     C_to_Xen_Strings(s, *c),
	     __func__);
}

static Xen gxm_XtRegisterGrabAction(Xen arg1, Xen arg2, Xen arg3, Xen arg4, Xen arg5)
{
  #define H_XtRegisterGrabAction "void XtRegisterGrabAction(action_proc, owner_events, event_mask, pointer_mode, keyboard_mode) adds the \
specified action_proc to a list known to the translation manager."
  Xen_check_type(Xen_is_procedure(arg1) && (Xen_is_aritable(arg1, 3)), arg1, 1, "XtRegisterGrabAction", "XtActionProc");
  Xen_check_type(Xen_is_boolean(arg2), arg2, 2, "XtRegisterGrabAction", "boolean");
  Xen_check_type(Xen_is_ulong(arg3), arg3, 3, "XtRegisterGrabAction", "unsigned int");
  Xen_check_type(Xen_is_integer(arg4), arg4, 4, "XtRegisterGrabAction", "int");
  Xen_check_type(Xen_is_integer(arg5), arg5, 5, "XtRegisterGrabAction", "int");
  xm_protect(arg1);
  register_proc = arg1;
  XtRegisterGrabAction(gxm_XtRegisterGrabActionProc, Xen_boolean_to_C_bool(arg2), Xen_ulong_to_C_ulong(arg3), Xen_integer_to_C_int(arg4), Xen_integer_to_C_int(arg5));
  return(Xen_false);
}

static Xen gxm_XtCallActionProc(Xen arg1, Xen arg2, Xen arg3, Xen arg4, Xen arg5)
{
  #define H_XtCallActionProc "void XtCallActionProc(widget, action, event, params, num_params) searches for the named action routine in the \
same manner and order as translation tables are bound. If found, the action routine is invoked with the specified widget, event pointer, \
and parameters."
  char **params = NULL;
  int i, len = 0;
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XtCallActionProc", "Widget");
  Xen_check_type(Xen_is_string(arg2), arg2, 2, "XtCallActionProc", "char*");
  Xen_check_type(Xen_is_XEvent(arg3) || Xen_is_false(arg3), arg3, 3, "XtCallActionProc", "XEvent*");
  Xen_check_type(Xen_is_list(arg4) || Xen_is_false(arg4), arg4, 4, "XtCallActionProc", "list of String");
  Xen_check_type(Xen_is_integer_or_unbound(arg5), arg5, 5, "XtCallActionProc", "int");
  if (Xen_is_list(arg4))
    {
      if (Xen_is_integer(arg5)) 
	len = Xen_integer_to_C_int(arg5); 
      else len = Xen_list_length(arg4);
    }
  if (len > 0) 
    {
      params = (char **)calloc(len, sizeof(char *));
      for (i = 0; i < len; i++)
	params[i] = xen_strdup(Xen_string_to_C_string(Xen_list_ref(arg4, i)));
    }
  XtCallActionProc(Xen_to_C_Widget(arg1), 
		   (char *)Xen_string_to_C_string(arg2), 
		   (Xen_is_XEvent(arg3)) ? Xen_to_C_XEvent(arg3) : NULL,
		   params, len);
  if (params)
    {
      for (i = 0; i < len; i++)
	if (params[i]) free(params[i]);
      free(params);
    }
  return(Xen_false);
}

static Xen gxm_XtGetActionList(Xen arg1)
{
  #define H_XtGetActionList "void XtGetActionList(widget_class): list of actions"
  /* DIFF: XtGetActionList omit arg2 and 3, return list of lists
   */
  unsigned int len;
  XtActionList act;
  Xen lst = Xen_empty_list;
  Xen_check_type(Xen_is_WidgetClass(arg1), arg1, 1, "XtGetActionList", "WidgetClass");
  XtGetActionList(Xen_to_C_WidgetClass(arg1), &act, &len);
  if (len > 0)
    {
      int i;
      for (i = len - 1; i >= 0; i--)
	lst = Xen_cons(Xen_list_2(C_string_to_Xen_string(act[i].string),
				  C_ulong_to_Xen_ulong(act[i].proc)),
		       lst);
      free(act);
    }
  return(lst);
}

#define C_to_Xen_XM_ActionHook(Code, Context) \
  Xen_list_5(C_string_to_Xen_symbol("ActionHook"), Code, Context, Xen_integer_zero, Xen_integer_zero)
#define XM_is_ActionHook(Arg) is_wrapped("ActionHook", Arg)

static bool unprotect_actionhook(Xen val, int loc, unsigned long id)
{
  if ((XM_is_ActionHook(val)) &&
      ((XtActionHookId)Xen_ulong_to_C_ulong(Xen_list_ref(val, 4)) == (XtActionHookId)id))
    {
      xm_unprotect_at(loc);
      return(true);
    }
  return(false);
}

static Xen gxm_XtRemoveActionHook(Xen arg1)
{
  #define H_XtRemoveActionHook "void XtRemoveActionHook(idP) removes the specified action hook procedure from the list in which it was registered."
  XtActionHookId id;
  Xen_check_type(Xen_is_XtActionHookId(arg1), arg1, 1, "XtRemoveActionHook", "XtActionHookId");
  id = Xen_to_C_XtActionHookId(arg1);
  XtRemoveActionHook(id);
  map_over_protected_elements(unprotect_actionhook, (unsigned long)id);
  return(Xen_false);
}

static void gxm_XtActionHookProc(Widget w, XtPointer p, String s, XEvent* e, String* s1, Cardinal* c) 
{
  /* DIFF: XtActionHookProc takes 5 args (last is string list)
   */
  Xen descr = (Xen)p;
  /* (list 'ActionHook function context gc-loc id) */
  Xen_call_with_5_args(Xen_cadr(descr), 
	     C_to_Xen_Widget(w),
	     Xen_caddr(descr),
	     C_string_to_Xen_string(s),
	     C_to_Xen_XEvent(e),
	     C_to_Xen_Strings(s1, *c),
	     __func__);
}

static Xen gxm_XtAppAddActionHook(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XtAppAddActionHook "XtActionHookId XtAppAddActionHook(app_context, proc, client_data) adds the specified procedure to the front of \
a list maintained in the application context."
  XtActionHookId id;
  int gc_loc;
  Xen descr;
  Xen_check_type(Xen_is_XtAppContext(arg1), arg1, 1, "XtAppAddActionHook", "XtAppContext");
  Xen_check_type(Xen_is_procedure(arg2) && (Xen_is_aritable(arg2, 5)), arg2, 2, "XtAppAddActionHook", "XtActionHookProc");
  descr = C_to_Xen_XM_ActionHook(arg2, (Xen_is_bound(arg3)) ? arg3 : Xen_false);
  gc_loc = xm_protect(descr);
  id = XtAppAddActionHook(Xen_to_C_XtAppContext(arg1), gxm_XtActionHookProc, (XtPointer)descr);
  Xen_list_set(descr, 3, C_int_to_Xen_integer(gc_loc));
  Xen_list_set(descr, 4, C_ulong_to_Xen_ulong(id));
  return(C_to_Xen_XtActionHookId(id));
}


/* (72) this is part of an actionrec */
/* this code is a horrible kludge, but what can I do? -- to vector to the procedure corresponding to
 *   the action that triggered the action proc, I need to have some way to find that procedure,
 *   and there's none I can see -- the args incoming (widget event args argn) give me no way to
 *   trace back to the caller, or even to save that data at declaration time via clientData or
 *   its equivalent -- I don't even get to see the affected widget.  By the time install|augment|override
 *   translations is called, my procedure info has been buried (or at least compiled into some unknown form).
 *
 *   XtAppContext->action_table is a list of ActionLists
 *   Widget struct has: XtTMRec tm:
 *     XtTranslations  translations;
 *     XtBoundActions  proc_table;  -- list of XtActionProcs		
 *     struct _XtStateRec *current_state;  
 */

static Xen xtactionprocs[9];
static void gxm_XtActionProc0(Widget w, XEvent *e, char **args, Cardinal *argn) 
{
  Xen_call_with_3_args(xtactionprocs[0], C_to_Xen_Widget(w), C_to_Xen_XEvent(e), C_to_Xen_Strings(args, *argn), __func__);
}
static void gxm_XtActionProc1(Widget w, XEvent *e, char **args, Cardinal *argn) 
{
  Xen_call_with_3_args(xtactionprocs[1], C_to_Xen_Widget(w), C_to_Xen_XEvent(e), C_to_Xen_Strings(args, *argn), __func__);
}
static void gxm_XtActionProc2(Widget w, XEvent *e, char **args, Cardinal *argn) 
{
  Xen_call_with_3_args(xtactionprocs[2], C_to_Xen_Widget(w), C_to_Xen_XEvent(e), C_to_Xen_Strings(args, *argn), __func__);
}
static void gxm_XtActionProc3(Widget w, XEvent *e, char **args, Cardinal *argn) 
{
  Xen_call_with_3_args(xtactionprocs[3], C_to_Xen_Widget(w), C_to_Xen_XEvent(e), C_to_Xen_Strings(args, *argn), __func__);
}
static void gxm_XtActionProc4(Widget w, XEvent *e, char **args, Cardinal *argn) 
{
  Xen_call_with_3_args(xtactionprocs[4], C_to_Xen_Widget(w), C_to_Xen_XEvent(e), C_to_Xen_Strings(args, *argn), __func__);
}
static void gxm_XtActionProc5(Widget w, XEvent *e, char **args, Cardinal *argn) 
{
  Xen_call_with_3_args(xtactionprocs[5], C_to_Xen_Widget(w), C_to_Xen_XEvent(e), C_to_Xen_Strings(args, *argn), __func__);
}
static void gxm_XtActionProc6(Widget w, XEvent *e, char **args, Cardinal *argn) 
{
  Xen_call_with_3_args(xtactionprocs[6], C_to_Xen_Widget(w), C_to_Xen_XEvent(e), C_to_Xen_Strings(args, *argn), __func__);
}
static void gxm_XtActionProc7(Widget w, XEvent *e, char **args, Cardinal *argn) 
{
  Xen_call_with_3_args(xtactionprocs[7], C_to_Xen_Widget(w), C_to_Xen_XEvent(e), C_to_Xen_Strings(args, *argn), __func__);
}
static void gxm_XtActionProc8(Widget w, XEvent *e, char **args, Cardinal *argn) 
{
  fprintf(stderr, "too many actions");
}
static int xm_action_ctr = 0;

static XtActionsRec *make_action_rec(int len, Xen larg2)
{
  int i, gcloc;
  XtActionsRec *act;
  Xen arg2;
  arg2 = Xen_copy_arg(larg2);
  gcloc = xm_protect(arg2);
  act = (XtActionsRec *)calloc(len, sizeof(XtActionsRec));
  for (i = 0; i < len; i++, arg2 = Xen_cdr(arg2))
    {
      Xen pair;
      pair = Xen_car(arg2);
      act[i].string = (String)xen_strdup(Xen_string_to_C_string(Xen_car(pair)));
      if (xm_action_ctr >= 8)
	{
	  fprintf(stderr,"too many actions...");
	  act[i].proc = (XtActionProc)gxm_XtActionProc8;
	  xm_protect(Xen_cadr(pair));
	  xtactionprocs[8] = Xen_cadr(pair);
	}
      else
	{
	  switch (xm_action_ctr)
	    {
	    case 0: act[i].proc = (XtActionProc)gxm_XtActionProc0; break;
	    case 1: act[i].proc = (XtActionProc)gxm_XtActionProc1; break;
	    case 2: act[i].proc = (XtActionProc)gxm_XtActionProc2; break;
	    case 3: act[i].proc = (XtActionProc)gxm_XtActionProc3; break;
	    case 4: act[i].proc = (XtActionProc)gxm_XtActionProc4; break;
	    case 5: act[i].proc = (XtActionProc)gxm_XtActionProc5; break;
	    case 6: act[i].proc = (XtActionProc)gxm_XtActionProc6; break;
	    case 7: act[i].proc = (XtActionProc)gxm_XtActionProc7; break;
	    }
	  xm_protect(Xen_cadr(pair));
	  xtactionprocs[xm_action_ctr++] = Xen_cadr(pair);
	}
    }
  xm_unprotect_at(gcloc);
  return(act);
}

static Xen gxm_XtAppAddActions(Xen arg1, Xen arg2)
{
  #define H_XtAppAddActions "void XtAppAddActions(app_context, actions, num_actions) adds the specified action table and registers it \
with the translation manager."
  /* DIFF: XtAddAppActions takes list of lists for arg2 (name proc) pairs, not XtActionList, omits arg3 (pointless)
   *        and action proc itself takes 3 args (no need for trailing count)
   */
  XtActionsRec *act;
  int i, len;
  Xen_check_type(Xen_is_XtAppContext(arg1), arg1, 1, "XtAppAddActions", "XtAppContext");
  Xen_check_type(Xen_is_list(arg2), arg2, 2, "XtAppAddActions", "list of XtActions");
  len = Xen_list_length(arg2);
  if (len <= 0) Xen_check_type(0, arg2, 2, "XtAppAddActions", "positive integer");
  act = make_action_rec(len, arg2);
  XtAppAddActions(Xen_to_C_XtAppContext(arg1), act, len);
  for (i = 0; i < len; i++)
    if (act[i].string) free(act[i].string);
  free(act);
  return(arg1);
}

static Xen gxm_XtUninstallTranslations(Xen arg1)
{
  #define H_XtUninstallTranslations "void XtUninstallTranslations(w) causes the entire translation table for widget to be removed."
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XtUninstallTranslations", "Widget");
  XtUninstallTranslations(Xen_to_C_Widget(arg1));
  return(Xen_false);
}

static Xen gxm_XtInstallAllAccelerators(Xen arg1, Xen arg2)
{
  #define H_XtInstallAllAccelerators "void XtInstallAllAccelerators(destination, source)"
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XtInstallAllAccelerators", "Widget");
  Xen_check_type(Xen_is_Widget(arg2), arg2, 2, "XtInstallAllAccelerators", "Widget");
  XtInstallAllAccelerators(Xen_to_C_Widget(arg1), Xen_to_C_Widget(arg2));
  return(Xen_false);
}

static Xen gxm_XtInstallAccelerators(Xen arg1, Xen arg2)
{
  #define H_XtInstallAccelerators "void XtInstallAccelerators(destination, source) installs the accelerators from source onto destination by \
augmenting the destination translations with the source accelerators."
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XtInstallAccelerators", "Widget");
  Xen_check_type(Xen_is_Widget(arg2), arg2, 2, "XtInstallAccelerators", "Widget");
  XtInstallAccelerators(Xen_to_C_Widget(arg1), Xen_to_C_Widget(arg2));
  return(Xen_false);
}

static Xen gxm_XtAugmentTranslations(Xen arg1, Xen arg2)
{
  #define H_XtAugmentTranslations "void XtAugmentTranslations(w, translations) nondestructively merges the new translations into the existing \
widget translations."
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XtAugmentTranslations", "Widget");
  Xen_check_type(Xen_is_XtTranslations(arg2), arg2, 2, "XtAugmentTranslations", "XtTranslations");
  XtAugmentTranslations(Xen_to_C_Widget(arg1), Xen_to_C_XtTranslations(arg2));
  return(Xen_false);
}

static Xen gxm_XtOverrideTranslations(Xen arg1, Xen arg2)
{
  #define H_XtOverrideTranslations "void XtOverrideTranslations(w, translations) destructively merges the new translations into the existing \
widget translations."
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XtOverrideTranslations", "Widget");
  Xen_check_type(Xen_is_XtTranslations(arg2), arg2, 2, "XtOverrideTranslations", "XtTranslations");
  XtOverrideTranslations(Xen_to_C_Widget(arg1), Xen_to_C_XtTranslations(arg2));
  return(Xen_false);
}

static Xen gxm_XtParseAcceleratorTable(Xen arg1)
{
  #define H_XtParseAcceleratorTable "XtAccelerators XtParseAcceleratorTable(const char *source) compiles source"
  Xen_check_type(Xen_is_string(arg1), arg1, 1, "XtParseAcceleratorTable", "char*");
  return(C_ulong_to_Xen_ulong(XtParseAcceleratorTable((char *)Xen_string_to_C_string(arg1))));
}

static Xen gxm_XtParseTranslationTable(Xen arg1)
{
  #define H_XtParseTranslationTable "XtTranslations XtParseTranslationTable(table) compiles the translation table into the opaque internal \
representation of type XtTranslations."
  Xen_check_type(Xen_is_string(arg1), arg1, 1, "XtParseTranslationTable", "char*");
  return(C_to_Xen_XtTranslations(XtParseTranslationTable((char *)Xen_string_to_C_string(arg1))));
}

static Xen gxm_XtKeysymToKeycodeList(Xen arg1, Xen arg2)
{
  #define H_XtKeysymToKeycodeList "void XtKeysymToKeycodeList(display, keysym) procedure returns all the \
KeyCodes that have keysym in their entry for the keyboard mapping table associated with display -> (keycodes)."
  /* DIFF: XtKeysymToKeycodeList final 2 args omit, returns list
   */
  unsigned int len;
  KeyCode *kr;
  Xen lst = Xen_empty_list;
  int loc;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XtKeysymToKeycodeList", "Display*");
  Xen_check_type(Xen_is_KeySym(arg2), arg2, 2, "XtKeysymToKeycodeList", "KeySym");
  XtKeysymToKeycodeList(Xen_to_C_Display(arg1), Xen_to_C_KeySym(arg2), &kr, &len);
  loc = xm_protect(lst);
  if (len > 0)
    {
      int i;
      for (i = len - 1; i >= 0; i--)
	lst = Xen_cons(C_to_Xen_KeyCode(kr[i]), lst);
      free(kr);
    }
  xm_unprotect_at(loc);
  return(lst);
}

static Xen gxm_XtTranslateCoords(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XtTranslateCoords "void XtTranslateCoords(w, x, y): (root_x root_y)"
  /* DIFF: XtTranslateCoords omits and returns last 2 args
   */
  Position x, y;
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XtTranslateCoords", "Widget");
  Xen_check_type(Xen_is_Position(arg2), arg2, 2, "XtTranslateCoords", "Position");
  Xen_check_type(Xen_is_Position(arg3), arg3, 3, "XtTranslateCoords", "Position");
  XtTranslateCoords(Xen_to_C_Widget(arg1), Xen_to_C_Position(arg2), Xen_to_C_Position(arg3), &x, &y);
  return(Xen_list_2(C_to_Xen_Position(x),
		    C_to_Xen_Position(y)));
}

static Xen gxm_XtMakeResizeRequest(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XtMakeResizeRequest "XtGeometryResult XtMakeResizeRequest(w, width, height): (res width height)"
  /* DIFF: XtMakeResizeRequest w wid hgt [rtnw rtn] -> (list res w h)
   */
  Dimension w, h;
  int val;
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XtMakeResizeRequest", "Widget");
  Xen_check_type(Xen_is_Dimension(arg2), arg2, 2, "XtMakeResizeRequest", "Dimension");
  Xen_check_type(Xen_is_Dimension(arg3), arg3, 3, "XtMakeResizeRequest", "Dimension");
  val = XtMakeResizeRequest(Xen_to_C_Widget(arg1), 
			    Xen_to_C_Dimension(arg2), 
			    Xen_to_C_Dimension(arg3), 
			    &w, &h);
  return(Xen_list_3(C_int_to_Xen_integer(val),
		    C_to_Xen_Dimension(w),
		    C_to_Xen_Dimension(h)));
}

static Xen gxm_XtOwnSelectionIncremental(Xen arg1, Xen arg2, Xen arg3, Xen arg4, Xen arg5, Xen arg6, Xen arg7, Xen arg8)
{
  #define H_XtOwnSelectionIncremental "Boolean XtOwnSelectionIncremental(w, selection, time, convert_callback, lose_callback, \
done_callback, cancel_callback, client_data)"
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XtOwnSelectionIncremental", "Widget");
  Xen_check_type(Xen_is_Atom(arg2), arg2, 2, "XtOwnSelectionIncremental", "Atom");
  Xen_check_type(Xen_is_Time(arg3), arg3, 3, "XtOwnSelectionIncremental", "Time");
  Xen_check_type(Xen_is_procedure(arg4) && (Xen_is_aritable(arg4, 10)), arg4, 4, "XtOwnSelectionIncremental", "XtConvertSelectionIncrProc");
  Xen_check_type(Xen_is_procedure(arg5) && (Xen_is_aritable(arg5, 3)), arg5, 5, "XtOwnSelectionIncremental", "XtLoseSelectionIncrProc");
  Xen_check_type(Xen_is_procedure(arg6) && (Xen_is_aritable(arg6, 5)), arg6, 6, "XtOwnSelectionIncremental", "XtSelectionDoneIncrProc");
  Xen_check_type(Xen_is_procedure(arg7) && (Xen_is_aritable(arg7, 5)), arg7, 7, "XtOwnSelectionIncremental", "XtCancelConvertSelectionProc");
  add_selmap(Xen_to_C_Widget(arg1), CONVERT_INCR, arg4);
  add_selmap(Xen_to_C_Widget(arg1), LOSE_INCR, arg5);
  add_selmap(Xen_to_C_Widget(arg1), DONE_INCR, arg6);
  add_selmap(Xen_to_C_Widget(arg1), CANCEL_CONVERT, arg7);	     
  return(C_bool_to_Xen_boolean(XtOwnSelectionIncremental(Xen_to_C_Widget(arg1), 
						    Xen_to_C_Atom(arg2), 
						    Xen_to_C_Time(arg3), 
						    (XtConvertSelectionIncrProc)gxm_XtConvertSelectionIncrProc, 
						    gxm_XtLoseSelectionIncrProc, 
						    gxm_XtSelectionDoneIncrProc, 
						    gxm_XtCancelConvertSelectionProc, 
						    (XtPointer)arg8)));
}

static Xen gxm_XtOwnSelection(Xen arg1, Xen arg2, Xen arg3, Xen arg4, Xen arg5, Xen arg6)
{
  #define H_XtOwnSelection "Boolean XtOwnSelection(w, selection, time, convert_proc, lose_selection, done_proc) informs the selection \
mechanism that a widget believes it owns a selection."
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XtOwnSelection", "Widget");
  Xen_check_type(Xen_is_Atom(arg2), arg2, 2, "XtOwnSelection", "Atom");
  Xen_check_type(Xen_is_Time(arg3), arg3, 3, "XtOwnSelection", "Time");
  Xen_check_type(Xen_is_procedure(arg4) && (Xen_is_aritable(arg4, 7)), arg4, 4, "XtOwnSelection", "XtConvertSelectionProc");
  Xen_check_type(Xen_is_procedure(arg5) && (Xen_is_aritable(arg5, 2)), arg5, 5, "XtOwnSelection", "XtLoseSelectionProc");
  Xen_check_type(Xen_is_procedure(arg6) && (Xen_is_aritable(arg6, 3)), arg6, 6, "XtOwnSelection", "XtSelectionDoneProc");
  add_selmap(Xen_to_C_Widget(arg1), CONVERT, arg4);
  add_selmap(Xen_to_C_Widget(arg1), LOSE, arg5);
  add_selmap(Xen_to_C_Widget(arg1), DONE, arg6);	     
  return(C_bool_to_Xen_boolean(XtOwnSelection(Xen_to_C_Widget(arg1), 
					 Xen_to_C_Atom(arg2), 
					 Xen_to_C_Time(arg3), 
					 (XtConvertSelectionProc)gxm_XtConvertSelectionProc, 
					 gxm_XtLoseSelectionProc, 
					 gxm_XtSelectionDoneProc)));
}

static Xen gxm_XtIsSensitive(Xen arg1)
{
  #define H_XtIsSensitive "Boolean XtIsSensitive(w): returns " PROC_TRUE " if user input events are being dispatched."
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XtIsSensitive", "Widget");
  return(C_bool_to_Xen_boolean(XtIsSensitive(Xen_to_C_Widget(arg1))));
}

static Xen gxm_XtIsRealized(Xen arg1)
{
  #define H_XtIsRealized "Boolean XtIsRealized(w): returns " PROC_TRUE " if the widget has been realized,"
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XtIsRealized", "Widget");
  return(C_bool_to_Xen_boolean(XtIsRealized(Xen_to_C_Widget(arg1))));
}

static Xen gxm_XtIsManaged(Xen arg1)
{
  #define H_XtIsManaged "Boolean XtIsManaged(widget): returns " PROC_TRUE " if the specified widget is of class RectObj or any subclass thereof and is managed."
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XtIsManaged", "Widget");
  return(C_bool_to_Xen_boolean(XtIsManaged(Xen_to_C_Widget(arg1))));
}

static Xen gxm_XtIsObject(Xen arg1)
{
  #define H_XtIsObject "Boolean XtIsObject(w)"
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XtIsObject", "Widget");
  return(C_bool_to_Xen_boolean(XtIsObject(Xen_to_C_Widget(arg1))));
}

static Xen gxm_XtIsSubclass(Xen arg1, Xen arg2)
{
  #define H_XtIsSubclass "Boolean XtIsSubclass(w, widget_class): returns " PROC_TRUE " if the class of the specified widget is equal to or is a \
subclass of the specified class."
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XtIsSubclass", "Widget");
  Xen_check_type(Xen_is_WidgetClass(arg2), arg2, 2, "XtIsSubclass", "WidgetClass");
  return(C_bool_to_Xen_boolean(XtIsSubclass(Xen_to_C_Widget(arg1), Xen_to_C_WidgetClass(arg2))));
}

static Xen gxm_XtAppPeekEvent(Xen arg1)
{
  #define H_XtAppPeekEvent "Boolean XtAppPeekEvent(app_context) fills in the event and returns a nonzero value. If no X \
input is on the queue, XtAppPeekEvent flushes the output buffer and blocks until input is available."
  /* DIFF: XtAppPeekEvent app [ev] -> (list val ev)
   */
  XEvent *e;
  int val;
  Xen_check_type(Xen_is_XtAppContext(arg1), arg1, 1, "XtAppPeekEvent", "XtAppContext");
  e = (XEvent *)calloc(1, sizeof(XEvent));
  val = XtAppPeekEvent(Xen_to_C_XtAppContext(arg1), e);
  return(Xen_list_2(C_bool_to_Xen_boolean(val), C_to_Xen_XEvent_OBJ(e)));
}

static Xen gxm_XtCallAcceptFocus(Xen arg1, Xen arg2)
{
  #define H_XtCallAcceptFocus "Boolean XtCallAcceptFocus(w, time) calls the specified widget's accept_focus procedure, passing it the \
specified widget and time, and returns what the accept_focus procedure returns."
  /* DIFF: XtCallAcceptFocus takes Time arg (not Time*)
   */
  Time tm;
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XtCallAcceptFocus", "Widget");
  Xen_check_type(Xen_is_Time(arg2), arg2, 2, "XtCallAcceptFocus", "Time");
  tm = Xen_to_C_Time(arg2);
  return(C_bool_to_Xen_boolean(XtCallAcceptFocus(Xen_to_C_Widget(arg1), &tm)));
}

static Xen gxm_XtDispatchEvent(Xen arg1)
{
  #define H_XtDispatchEvent "Boolean XtDispatchEvent(event) sends those events to the event handler functions that have been previously \
registered with the dispatch routine."
  Xen_check_type(Xen_is_XEvent(arg1), arg1, 1, "XtDispatchEvent", "XEvent*");
  return(C_bool_to_Xen_boolean(XtDispatchEvent(Xen_to_C_XEvent(arg1))));
}

static Xen gxm_XtUnmanageChild(Xen arg1)
{
  #define H_XtUnmanageChild "void XtUnmanageChild(child)"
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XtUnmanageChild", "Widget");
  XtUnmanageChild(Xen_to_C_Widget(arg1));
  return(Xen_false);
}

static Xen gxm_XtUnmanageChildren(Xen arg1, Xen arg2)
{
  #define H_XtUnmanageChildren "void XtUnmanageChildren(children, num_children)"
  /* DIFF: XtUnmanageChildren arg1 is list of widgets
   */
  int len;
  Xen_check_type(Xen_is_list(arg1), arg1, 1, "XtUnmanageChildren", "WidgetList");
  Xen_check_type(Xen_is_integer_or_unbound(arg2), arg2, 2, "XtUnmanageChildren", "int");
  if (Xen_is_integer(arg2)) len = Xen_integer_to_C_int(arg2); else len = Xen_list_length(arg1);
  if (len > 0)
    {
      if (len > Xen_list_length(arg1))
	Xen_out_of_range_error("XmContainerReorder", 2, arg2, "len too large");
      else
	{
	  WidgetList ws1;
	  ws1 = Xen_to_C_Widgets(arg1, len);
	  XtUnmanageChildren(ws1, len);
	  if (ws1) free(ws1);
	}
    }
  return(Xen_false);
}

static Xen gxm_XtManageChild(Xen arg1)
{
  #define H_XtManageChild "void XtManageChild(child)"
  Xen_check_type(Xen_is_Widget(arg1), arg1, 1, "XtManageChild", "Widget");
  XtManageChild(Xen_to_C_Widget(arg1));
  return(Xen_false);
}

static Xen gxm_XtManageChildren(Xen arg1, Xen arg2)
{
  #define H_XtManageChildren "void XtManageChildren(children, num_children)"
  /* DIFF: XtManageChildren arg1 is list of widgets
   */
  int len;
  Xen_check_type(Xen_is_list(arg1), arg1, 1, "XtManageChildren", "WidgetList");
  Xen_check_type(Xen_is_integer_or_unbound(arg2), arg2, 2, "XtManageChildren", "int");
  if (Xen_is_integer(arg2)) len = Xen_integer_to_C_int(arg2); else len = Xen_list_length(arg1);
  if (len > 0)
    {
      if (len > Xen_list_length(arg1))
	Xen_out_of_range_error("XmContainerReorder", 2, arg2, "len too large");
      else
	{
	  WidgetList ws1;
	  ws1 = Xen_to_C_Widgets(arg1, len);
	  XtManageChildren(ws1, len);
	  if (ws1) free(ws1);
	}
    }
  return(Xen_false);
}

static Xen gxm_XtIsRectObj(Xen arg)
{
  #define H_XtIsRectObj "Boolean XtIsRectObj(w)"
  Xen_check_type(Xen_is_Widget(arg), arg, 0, "XtIsRectObj", "Widget");
  return(C_bool_to_Xen_boolean(XtIsRectObj(Xen_to_C_Widget(arg))));
}

static Xen gxm_XtIsWidget(Xen arg)
{
  #define H_XtIsWidget "Boolean XtIsWidget(w)"
  return(C_bool_to_Xen_boolean(is_wrapped("Widget", arg) && 
			  (XtIsWidget(Xen_to_C_Widget(arg)))));
}

static Xen gxm_XtIsComposite(Xen arg)
{
  #define H_XtIsComposite "Boolean XtIsComposite(w)"
  Xen_check_type(Xen_is_Widget(arg), arg, 0, "XtIsComposite", "Widget");
  return(C_bool_to_Xen_boolean(XtIsComposite(Xen_to_C_Widget(arg))));
}

static Xen gxm_XtIsConstraint(Xen arg)
{
  #define H_XtIsConstraint "Boolean XtIsConstraint(w)"
  Xen_check_type(Xen_is_Widget(arg), arg, 0, "XtIsConstraint", "Widget");
  return(C_bool_to_Xen_boolean(XtIsConstraint(Xen_to_C_Widget(arg))));
}

static Xen gxm_XtIsShell(Xen arg)
{
  #define H_XtIsShell "Boolean XtIsShell(w)"
  Xen_check_type(Xen_is_Widget(arg), arg, 0, "XtIsShell", "Widget");
  return(C_bool_to_Xen_boolean(XtIsShell(Xen_to_C_Widget(arg))));
}

static Xen gxm_XtIsOverrideShell(Xen arg)
{
  #define H_XtIsOverrideShell "Boolean XtIsOverrideShell(w)"
  Xen_check_type(Xen_is_Widget(arg), arg, 0, "XtIsOverrideShell", "Widget");
  return(C_bool_to_Xen_boolean(XtIsOverrideShell(Xen_to_C_Widget(arg))));
}

static Xen gxm_XtIsWMShell(Xen arg)
{
  #define H_XtIsWMShell "Boolean XtIsWMShell(w)"
  Xen_check_type(Xen_is_Widget(arg), arg, 0, "XtIsWMShell", "Widget");
  return(C_bool_to_Xen_boolean(XtIsWMShell(Xen_to_C_Widget(arg))));
}

static Xen gxm_XtIsVendorShell(Xen arg)
{
  #define H_XtIsVendorShell "Boolean XtIsVendorShell(w)"
  Xen_check_type(Xen_is_Widget(arg), arg, 0, "XtIsVendorShell", "Widget");
  return(C_bool_to_Xen_boolean(XtIsVendorShell(Xen_to_C_Widget(arg))));
}

static Xen gxm_XtIsTransientShell(Xen arg)
{
  #define H_XtIsTransientShell "Boolean XtIsTransientShell(w)"
  Xen_check_type(Xen_is_Widget(arg), arg, 0, "XtIsTransientShell", "Widget");
  return(C_bool_to_Xen_boolean(XtIsTransientShell(Xen_to_C_Widget(arg))));
}

static Xen gxm_XtIsTopLevelShell(Xen arg)
{
  #define H_XtIsTopLevelShell "Boolean XtIsTopLevelShell(w)"
  Xen_check_type(Xen_is_Widget(arg), arg, 0, "XtIsTopLevelShell", "Widget");
  return(C_bool_to_Xen_boolean(XtIsTopLevelShell(Xen_to_C_Widget(arg))));
}

static Xen gxm_XtIsApplicationShell(Xen arg)
{
  #define H_XtIsApplicationShell "Boolean XtIsApplicationShell(w)"
  Xen_check_type(Xen_is_Widget(arg), arg, 0, "XtIsApplicationShell", "Widget");
  return(C_bool_to_Xen_boolean(XtIsApplicationShell(Xen_to_C_Widget(arg))));
}

static Xen gxm_XtIsSessionShell(Xen arg)
{
  #define H_XtIsSessionShell "Boolean XtIsSessionShell(w)"
  Xen_check_type(Xen_is_Widget(arg), arg, 0, "XtIsSessionShell", "Widget");
  return(C_bool_to_Xen_boolean(XtIsSessionShell(Xen_to_C_Widget(arg))));
}

static Xen gxm_XtMapWidget(Xen arg)
{
  #define H_XtMapWidget "XtMapWidget(w)"
  Xen_check_type(Xen_is_Widget(arg), arg, 0, "XtMapWidget", "Widget");
  XtMapWidget(Xen_to_C_Widget(arg));
  return(Xen_false);
}

static Xen gxm_XtUnmapWidget(Xen arg)
{
  #define H_XtUnmapWidget "XtUnmapWidget(w)"
  Xen_check_type(Xen_is_Widget(arg), arg, 0, "XtUnmapWidget", "Widget");
  XtUnmapWidget(Xen_to_C_Widget(arg));
  return(Xen_false);
}

static Xen gxm_XtSetArg(Xen arg1, Xen arg2, Xen arg3)
{
  #define H_XtSetArg "XtSetArg in xm is useless -- it returns its arguments as a list"
  return(Xen_list_3(arg1, arg2, arg3));
}




/* ---------------------------------------------------------------------------------------------------- */

Xm_type_ptr_no_c2x(XpmImage, XpmImage *)
Xm_type_ptr_no_c2x(XpmAttributes, XpmAttributes *) /* _OBJ?? */
Xm_type_ptr_no_c2x(XpmColorSymbol, XpmColorSymbol *)

static Xen gxm_XpmCreateXpmImageFromPixmap(Xen arg1, Xen arg2, Xen arg3, Xen arg5)
{
  XpmImage *image;
  int val;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XpmCreateXpmImageFromPixmap", "Display*");
  Xen_check_type(Xen_is_Pixmap(arg2), arg2, 2, "XpmCreateXpmImageFromPixmap", "Pixmap");
  Xen_check_type(Xen_is_Pixmap(arg3), arg3, 3, "XpmCreateXpmImageFromPixmap", "Pixmap");
  Xen_check_type(Xen_is_XpmAttributes(arg5) || Xen_is_false(arg5), arg5, 5, "XpmCreateXpmImageFromPixmap", "XpmAttributes*");
  image = (XpmImage *)calloc(1, sizeof(XpmImage));
  val = XpmCreateXpmImageFromPixmap(Xen_to_C_Display(arg1), 
				    Xen_to_C_Pixmap(arg2), 
				    Xen_to_C_Pixmap(arg3), 
				    image,
				    (Xen_is_false(arg5)) ? NULL : Xen_to_C_XpmAttributes(arg5));
  if (val == XpmSuccess)
    return(wrap_for_Xen_obj("XpmImage", image));
  free(image);
  return(C_int_to_Xen_integer(val));
}

static Xen gxm_XpmCreatePixmapFromXpmImage(Xen arg1, Xen arg2, Xen arg3, Xen arg4)
{
  /* DIFF: XpmCreatePixmapFromXpmImage omits and returns pixmap args
   */
  Pixmap p1, p2;
  int val;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XpmCreatePixmapFromXpmImage", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XpmCreatePixmapFromXpmImage", "Drawable");
  Xen_check_type(Xen_is_XpmImage(arg3), arg3, 3, "XpmCreatePixmapFromXpmImage", "XpmImage*");
  Xen_check_type(Xen_is_XpmAttributes(arg4) || Xen_is_false(arg4), arg4, 4, "XpmCreatePixmapFromXpmImage", "XpmAttributes*");
  val = XpmCreatePixmapFromXpmImage(Xen_to_C_Display(arg1), 
				    Xen_to_C_Window(arg2), 
				    Xen_to_C_XpmImage(arg3), 
				    &p1, &p2,
				    (Xen_is_false(arg4)) ? NULL : Xen_to_C_XpmAttributes(arg4));
  return(Xen_list_3(C_int_to_Xen_integer(val),
		    C_to_Xen_Pixmap(p1),
		    C_to_Xen_Pixmap(p2)));
}

static Xen gxm_XpmReadFileToPixmap(Xen arg1, Xen arg2, Xen arg3, Xen arg6)
{
  /* DIFF: XpmReadFileToPixmap omits and returns pixmap args
   */
  Pixmap p1, p2;
  int val;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XpmReadFileToPixmap", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XpmReadFileToPixmap", "Drawable");
  Xen_check_type(Xen_is_string(arg3), arg3, 3, "XpmReadFileToPixmap", "char*");
  Xen_check_type(Xen_is_XpmAttributes(arg6) || Xen_is_false(arg6), arg6, 6, "XpmReadFileToPixmap", "XpmAttributes*");
  val = XpmReadFileToPixmap(Xen_to_C_Display(arg1), 
			    Xen_to_C_Window(arg2), 
			    (char *)Xen_string_to_C_string(arg3), 
			    &p1, &p2,
			    (Xen_is_false(arg6)) ? NULL : Xen_to_C_XpmAttributes(arg6));
  return(Xen_list_3(C_int_to_Xen_integer(val),
		    C_to_Xen_Pixmap(p1),
		    C_to_Xen_Pixmap(p2)));
}

static Xen gxm_XpmReadFileToXpmImage(Xen arg1)
{
  int val;
  XpmImage *image;
  Xen_check_type(Xen_is_string(arg1), arg1, 1, "XpmReadFileToXpmImage", "char*");
  image = (XpmImage *)calloc(1, sizeof(XpmImage));
  val = XpmReadFileToXpmImage((char *)Xen_string_to_C_string(arg1), image, NULL);
  if (val == XpmSuccess)
    return(wrap_for_Xen_obj("XpmImage", image));
  free(image);
  return(C_int_to_Xen_integer(val));
}

static Xen gxm_XpmCreatePixmapFromData(Xen arg1, Xen arg2, Xen larg3, Xen arg6)
{
  /* DIFF: XpmCreatePixmapFromData omits and returns pixmap args, arg3 (bits) is list of strings
   */
  Pixmap p1, p2;
  int val, i, len;
  char **bits;
  Xen arg3;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XpmCreatePixmapFromData", "Display*");
  Xen_check_type(Xen_is_Window(arg2), arg2, 2, "XpmCreatePixmapFromData", "Drawable");
  Xen_check_type(Xen_is_list(larg3), larg3, 3, "XpmCreatePixmapFromData", "list of char*");
  Xen_check_type(Xen_is_XpmAttributes(arg6) || Xen_is_false(arg6), arg6, 6, "XpmCreatePixmapFromData", "XpmAttributes*");
  arg3 = Xen_copy_arg(larg3);
  len = Xen_list_length(arg3);
  if (len <= 0) Xen_check_type(0, arg3, 3, "XpmCreatePixmapFromData", "positive integer");
  bits = (char **)calloc(len, sizeof(char *));
  for (i = 0; i < len; i++, arg3 = Xen_cdr(arg3))
    bits[i] = xen_strdup(Xen_string_to_C_string(Xen_car(arg3)));
  val = XpmCreatePixmapFromData(Xen_to_C_Display(arg1), 
				Xen_to_C_Window(arg2), 
				bits,
				&p1, &p2,
				(Xen_is_false(arg6)) ? NULL : Xen_to_C_XpmAttributes(arg6));
  for (i = 0; i < len; i++)
    if (bits[i]) free(bits[i]);
  free(bits);
  return(Xen_list_3(C_int_to_Xen_integer(val),
		    C_to_Xen_Pixmap(p1),
		    C_to_Xen_Pixmap(p2)));
}

static Xen gxm_XpmWriteFileFromPixmap(Xen arg1, Xen arg2, Xen arg3, Xen arg4, Xen arg5)
{
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XpmWriteFileFromPixmap", "Display*");
  Xen_check_type(Xen_is_string(arg2), arg2, 2, "XpmWriteFileFromPixmap", "char*");
  Xen_check_type(Xen_is_Pixmap(arg3), arg3, 3, "XpmWriteFileFromPixmap", "Pixmap");
  Xen_check_type(Xen_is_Pixmap(arg4), arg4, 4, "XpmWriteFileFromPixmap", "Pixmap");
  Xen_check_type(Xen_is_XpmAttributes(arg5) || Xen_is_false(arg5), arg5, 5, "XpmWriteFileFromPixmap", "XpmAttributes*");
  return(C_int_to_Xen_integer(XpmWriteFileFromPixmap(Xen_to_C_Display(arg1), (char *)Xen_string_to_C_string(arg2), 
					     Xen_to_C_Pixmap(arg3), Xen_to_C_Pixmap(arg4), 
					     (Xen_is_false(arg5)) ? NULL : Xen_to_C_XpmAttributes(arg5))));
}

static Xen gxm_XpmCreateDataFromPixmap(Xen arg1, Xen arg3, Xen arg4, Xen arg5)
{
  /* DIFF: XpmCreateDataFromPixmap arg2 omitted and rtn'd
   */
  char **buf = NULL;
  int val;
  Xen_check_type(Xen_is_Display(arg1), arg1, 1, "XpmCreateDataFromPixmap", "Display*");
  Xen_check_type(Xen_is_Pixmap(arg3), arg3, 3, "XpmCreateDataFromPixmap", "Pixmap");
  Xen_check_type(Xen_is_Pixmap(arg4), arg4, 4, "XpmCreateDataFromPixmap", "Pixmap");
  Xen_check_type(Xen_is_XpmAttributes(arg5) || Xen_is_false(arg5), arg5, 5, "XpmCreateDataFromPixmap", "XpmAttributes*");
  val = XpmCreateDataFromPixmap(Xen_to_C_Display(arg1), &buf, 
				Xen_to_C_Pixmap(arg3), 
				Xen_to_C_Pixmap(arg4), 
				(Xen_is_false(arg5)) ? NULL : Xen_to_C_XpmAttributes(arg5));
  return(Xen_list_2(C_int_to_Xen_integer(val),
		    Xen_wrap_C_pointer(buf)));
}

static Xen gxm_XpmGetErrorString(Xen err)
{
  #define H_XpmGetErrorString "(XpmGetErrorString err): string describing error"
  Xen_check_type(Xen_is_integer(err), err, 1, "XpmGetErrorString", "an Xpm status code");
  return(C_string_to_Xen_string(XpmGetErrorString(Xen_integer_to_C_int(err))));
}

static Xen gxm_XpmColorSymbol(Xen name, Xen value, Xen pixel)
{
  XpmColorSymbol *r;
  #define H_XpmColorSymbol "(XpmColorSymbol name val pix): new XpmColorSymbol struct"
  Xen_check_type(Xen_is_string(name), name, 1, "XpmColorSymbol", "char*");
  Xen_check_type(Xen_is_false(value) || Xen_is_string(value), value, 2, "XpmColorSymbol", "char*");
  Xen_check_type(Xen_is_Pixel(pixel), pixel, 3, "XpmColorSymbol", "Pixel");
  r = (XpmColorSymbol *)calloc(1, sizeof(XpmColorSymbol));
  r->name = (char *)Xen_string_to_C_string(name);
  r->value = (Xen_is_false(value)) ? NULL : (char *)Xen_string_to_C_string(value);
  r->pixel = Xen_to_C_Pixel(pixel);
  return(wrap_for_Xen_obj("XpmColorSymbol",r));
}

static Xen gxm_XpmImage(Xen width, Xen height, Xen cpp, Xen ncolors, Xen data)
{
  XpmImage *r;
  #define H_XpmImage "(XpmImage w h cpp n data): new XpmImage struct"
  Xen_check_type(Xen_is_ulong(width), width, 1, "XpmImage", "ulong");
  Xen_check_type(Xen_is_ulong(height), height, 2, "XpmImage", "ulong");
  Xen_check_type(Xen_is_ulong(cpp), cpp, 3, "XpmImage", "ulong");
  Xen_check_type(Xen_is_ulong(ncolors), ncolors, 4, "XpmImage", "ulong");
  Xen_check_type(Xen_is_ulong(data), data, 5, "XpmImage", "ulong");
  r = (XpmImage *)calloc(1, sizeof(XpmImage));
  r->width = Xen_ulong_to_C_ulong(width);
  r->height = Xen_ulong_to_C_ulong(height);
  r->cpp = Xen_ulong_to_C_ulong(cpp);
  r->ncolors = Xen_ulong_to_C_ulong(ncolors);
  r->data = (unsigned int *)Xen_ulong_to_C_ulong(data);
  return(wrap_for_Xen_obj("XpmImage", r));
}

static Xen gxm_XpmAttributes(void)
{
  #define H_XpmAttributes "(XpmAttributes): new XpmAttributes struct"
  return(wrap_for_Xen_obj("XpmAttributes", 
			  (XpmAttributes *)calloc(1, sizeof(XpmAttributes))));
}

static Xen gxm_cpp(Xen ptr)
{
  XM_field_assert_type((Xen_is_XpmImage(ptr)) || (Xen_is_XpmAttributes(ptr)), ptr, 1, "cpp", "XpmImage");
  if (Xen_is_XpmImage(ptr)) return(C_ulong_to_Xen_ulong((unsigned long)((Xen_to_C_XpmImage(ptr))->cpp)));
  return(C_ulong_to_Xen_ulong((unsigned long)((Xen_to_C_XpmAttributes(ptr))->cpp)));
}

static Xen gxm_set_cpp(Xen ptr, Xen val)
{
  XM_set_field_assert_type((Xen_is_XpmImage(ptr)) || (Xen_is_XpmAttributes(ptr)), ptr, 1, "cpp", "XpmImage");
  if (Xen_is_XpmImage(ptr)) (Xen_to_C_XpmImage(ptr))->cpp = Xen_ulong_to_C_ulong(val);
  (Xen_to_C_XpmAttributes(ptr))->cpp = Xen_ulong_to_C_ulong(val);
  return(val);
}

static Xen gxm_ncolors(Xen ptr)
{
  XM_field_assert_type((Xen_is_XpmImage(ptr)) || (Xen_is_XpmAttributes(ptr)), ptr, 1, "ncolors", "XpmImage");
  if (Xen_is_XpmImage(ptr)) return(C_ulong_to_Xen_ulong((unsigned long)((Xen_to_C_XpmImage(ptr))->ncolors)));
  return(C_ulong_to_Xen_ulong((unsigned long)((Xen_to_C_XpmAttributes(ptr))->ncolors)));
}

static Xen gxm_set_ncolors(Xen ptr, Xen val)
{
  XM_set_field_assert_type((Xen_is_XpmImage(ptr)) || (Xen_is_XpmAttributes(ptr)), ptr, 1, "ncolors", "XpmImage");
  if (Xen_is_XpmImage(ptr)) (Xen_to_C_XpmImage(ptr))->ncolors = Xen_ulong_to_C_ulong(val);
  else (Xen_to_C_XpmAttributes(ptr))->ncolors = Xen_ulong_to_C_ulong(val);
  return(val);
}

#if 0
static Xen gxm_set_data(Xen ptr, Xen val)
{
  XM_set_field_assert_type(Xen_is_XpmImage(ptr), ptr, 1, "data", "XpmImage");
  (Xen_to_C_XpmImage(ptr))->data = (unsigned int *)Xen_unwrap_C_pointer(val);
  return(val);
}
#endif

static Xen gxm_valuemask(Xen ptr)
{
  XM_field_assert_type(Xen_is_XpmAttributes(ptr), ptr, 1, "valuemask", "XpmAttributes");
  return(C_ulong_to_Xen_ulong((unsigned long)((Xen_to_C_XpmAttributes(ptr))->valuemask)));
}

static Xen gxm_set_valuemask(Xen ptr, Xen val)
{
  XM_set_field_assert_type(Xen_is_XpmAttributes(ptr), ptr, 1, "valuemask", "XpmAttributes");
  (Xen_to_C_XpmAttributes(ptr))->valuemask = Xen_ulong_to_C_ulong(val);
  return(val);
}

static Xen gxm_x_hotspot(Xen ptr)
{
  XM_field_assert_type(Xen_is_XpmAttributes(ptr), ptr, 1, "x_hotspot", "XpmAttributes");
  return(C_ulong_to_Xen_ulong((unsigned long)((Xen_to_C_XpmAttributes(ptr))->x_hotspot)));
}

static Xen gxm_set_x_hotspot(Xen ptr, Xen val)
{
  XM_set_field_assert_type(Xen_is_XpmAttributes(ptr), ptr, 1, "x_hotspot", "XpmAttributes");
  (Xen_to_C_XpmAttributes(ptr))->x_hotspot = Xen_ulong_to_C_ulong(val);
  return(val);
}

static Xen gxm_y_hotspot(Xen ptr)
{
  XM_field_assert_type(Xen_is_XpmAttributes(ptr), ptr, 1, "y_hotspot", "XpmAttributes");
  return(C_ulong_to_Xen_ulong((unsigned long)((Xen_to_C_XpmAttributes(ptr))->y_hotspot)));
}

static Xen gxm_set_y_hotspot(Xen ptr, Xen val)
{
  XM_set_field_assert_type(Xen_is_XpmAttributes(ptr), ptr, 1, "y_hotspot", "XpmAttributes");
  (Xen_to_C_XpmAttributes(ptr))->y_hotspot = Xen_ulong_to_C_ulong(val);
  return(val);
}

static Xen gxm_colorsymbols(Xen ptr)
{
  XpmAttributes *atr;
  int len;
  Xen lst = Xen_empty_list;
  XM_field_assert_type(Xen_is_XpmAttributes(ptr), ptr, 1, "colorsymbols", "XpmAttributes");
  atr = Xen_to_C_XpmAttributes(ptr);
  len = atr->numsymbols;
  if (len > 0)
    {
      int i;
      XpmColorSymbol *cols;
      cols = atr->colorsymbols;
      for (i = len - 1; i >= 0; i--)
	lst = Xen_cons(wrap_for_Xen("XpmColorSymbol", &(cols[i])), lst);
    }
  return(lst);
}

static Xen gxm_set_colorsymbols(Xen ptr, Xen vals)
{
  XpmAttributes *atr;
  int len;
  XM_set_field_assert_type(Xen_is_XpmAttributes(ptr), ptr, 1, "colorsymbols", "XpmAttributes");
  XM_set_field_assert_type(Xen_is_list(vals), vals, 2, "colorsymbols", "list of XpmColorSymbols");
  atr = Xen_to_C_XpmAttributes(ptr);
  len = Xen_list_length(vals);
  if (len > 0)
    {
      Xen lst;
      int i;
      XpmColorSymbol *cols = NULL, *cur;
      cols = (XpmColorSymbol *)calloc(len, sizeof(XpmColorSymbol));
      for (lst = Xen_copy_arg(vals), i = 0; i < len; i++, lst = Xen_cdr(lst))
	{
	  cur = Xen_to_C_XpmColorSymbol(Xen_car(lst));
	  if (cur->name) cols[i].name = xen_strdup(cur->name);
	  if (cur->value) cols[i].value = xen_strdup(cur->value);
	  cols[i].pixel = cur->pixel;
	}
      atr->colorsymbols = cols;
    }
  return(vals);
}

static Xen gxm_numsymbols(Xen ptr)
{
  XM_field_assert_type(Xen_is_XpmAttributes(ptr), ptr, 1, "numsymbols", "XpmAttributes");
  return(C_ulong_to_Xen_ulong((unsigned long)((Xen_to_C_XpmAttributes(ptr))->numsymbols)));
}

static Xen gxm_set_numsymbols(Xen ptr, Xen val)
{
  XM_set_field_assert_type(Xen_is_XpmAttributes(ptr), ptr, 1, "numsymbols", "XpmAttributes");
  XM_set_field_assert_type(Xen_is_integer(val), val, 2, "numsymbols", "integer");
  (Xen_to_C_XpmAttributes(ptr))->numsymbols = Xen_integer_to_C_int(val);
  return(val);
}

/* pixels is the list -- not sure what good npixels is without pixels */
static Xen gxm_npixels(Xen ptr)
{
  XM_field_assert_type(Xen_is_XpmAttributes(ptr), ptr, 1, "npixels", "XpmAttributes");
  return(C_ulong_to_Xen_ulong((unsigned long)((Xen_to_C_XpmAttributes(ptr))->npixels)));
}

static Xen gxm_set_npixels(Xen ptr, Xen val)
{
  XM_set_field_assert_type(Xen_is_XpmAttributes(ptr), ptr, 1, "npixels", "XpmAttributes");
  XM_set_field_assert_type(Xen_is_integer(val), val, 2, "npixels", "integer");
  (Xen_to_C_XpmAttributes(ptr))->npixels = Xen_integer_to_C_int(val);
  return(val);
}


#if WITH_EDITRES
static Xen gxm_XEditResCheckMessages(Xen widget, Xen data, Xen event, Xen cont)
{
  Boolean flag;
  Xen_check_type(Xen_is_Widget(widget), widget, 1, "widget", "_XEditResCheckMessages");
  Xen_check_type(Xen_is_XEvent(event), event, 3, "XEvent", "_XEditResCheckMessages");
  Xen_check_type(Xen_is_boolean(cont), cont, 4, "boolean", "_XEditResCheckMessages");
  flag = Xen_boolean_to_C_bool(cont);
  _XEditResCheckMessages(Xen_to_C_Widget(widget),
			 (XtPointer)data,
			 Xen_to_C_XEvent(event),
			 &flag);
  return(Xen_false);
}
#endif



/* -------------------------------- struct handlers -------------------------------- */

/* XRectangle */

static Xen gxm_XRectangle(Xen x, Xen y, Xen width, Xen height)
{
  #define H_XRectangle "(XRectangle x y width height): returns the given XRectangle"
  XRectangle *r;
  Xen_check_type(Xen_is_integer(x), x, 1, "XRectangle", "short");
  Xen_check_type(Xen_is_integer(y), y, 2, "XRectangle", "short");
  Xen_check_type(Xen_is_integer(width), width, 3, "XRectangle", "INT");
  Xen_check_type(Xen_is_integer(height), height, 4, "XRectangle", "INT");
  r = (XRectangle *)calloc(1, sizeof(XRectangle));
  r->x = (short)Xen_integer_to_C_int(x);
  r->y = (short)Xen_integer_to_C_int(y);
  r->width = Xen_integer_to_C_int(width);
  r->height = Xen_integer_to_C_int(height);
  return(C_to_Xen_XRectangle(r));
}

/* XSegment */

static Xen gxm_XSegment(Xen x1, Xen y1, Xen x2, Xen y2)
{
  XSegment *r;
#define H_XSegment "(XSegment x1 y1 x2 y2): new XSegment struct"
  Xen_check_type(Xen_is_integer(x1), x1, 1, "XSegment", "short");
  Xen_check_type(Xen_is_integer(y1), y1, 2, "XSegment", "short");
  Xen_check_type(Xen_is_integer(x2), x2, 3, "XSegment", "short");
  Xen_check_type(Xen_is_integer(y2), y2, 4, "XSegment", "short");
  r = (XSegment *)calloc(1, sizeof(XSegment));
  r->x1 = (short)Xen_integer_to_C_int(x1);
  r->y1 = (short)Xen_integer_to_C_int(y1);
  r->x2 = (short)Xen_integer_to_C_int(x2);
  r->y2 = (short)Xen_integer_to_C_int(y2);
  return(C_to_Xen_XSegment(r));
}

static Xen gxm_y2(Xen ptr)
{
  if (Xen_is_XSegment(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XSegment(ptr))->y2)));
  XM_field_assert_type(0, ptr, 1, "y2", "XSegment");
  return(Xen_false);
}

static Xen gxm_set_y2(Xen ptr, Xen val)
{
  XM_set_field_assert_type(Xen_is_integer(val), val, 1, "y2", "an integer");
  XM_set_field_assert_type(Xen_is_XSegment(ptr), ptr, 2, "y2", "XSegment");
  (Xen_to_C_XSegment(ptr))->y2 = (short)Xen_integer_to_C_int(val);
  return(val);
}

static Xen gxm_x2(Xen ptr)
{
  if (Xen_is_XSegment(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XSegment(ptr))->x2)));
  XM_field_assert_type(0, ptr, 1, "x2", "XSegment");
  return(Xen_false);
}

static Xen gxm_set_x2(Xen ptr, Xen val)
{
  XM_set_field_assert_type(Xen_is_integer(val), val, 1, "x2", "an integer");
  XM_set_field_assert_type(Xen_is_XSegment(ptr), ptr, 2, "x2", "XSegment");
  (Xen_to_C_XSegment(ptr))->x2 = (short)Xen_integer_to_C_int(val);
  return(val);
}

static Xen gxm_y1(Xen ptr)
{
  if (Xen_is_XSegment(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XSegment(ptr))->y1)));
  XM_field_assert_type(0, ptr, 1, "y1", "XSegment");
  return(Xen_false);
}

static Xen gxm_set_y1(Xen ptr, Xen val)
{
  XM_set_field_assert_type(Xen_is_integer(val), val, 1, "y1", "an integer");
  XM_set_field_assert_type(Xen_is_XSegment(ptr), ptr, 2, "y1", "XSegment");
  (Xen_to_C_XSegment(ptr))->y1 = (short)Xen_integer_to_C_int(val);
  return(val);
}

static Xen gxm_x1(Xen ptr)
{
  if (Xen_is_XSegment(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XSegment(ptr))->x1)));
  XM_field_assert_type(0, ptr, 1, "x1", "XSegment");
  return(Xen_false);
}

static Xen gxm_set_x1(Xen ptr, Xen val)
{
  XM_set_field_assert_type(Xen_is_integer(val), val, 1, "x1", "an integer");
  XM_set_field_assert_type(Xen_is_XSegment(ptr), ptr, 2, "x1", "XSegment");
  (Xen_to_C_XSegment(ptr))->x1 = (short)Xen_integer_to_C_int(val);
  return(val);
}

/* XPoint */

static Xen gxm_XPoint(Xen x, Xen y)
{
  XPoint *r;
  #define H_XPoint "(XPoint x y): new XPoint struct"
  Xen_check_type(Xen_is_integer(x), x, 1, "XPoint", "short");
  Xen_check_type(Xen_is_integer(y), y, 2, "XPoint", "short");
  r = (XPoint *)calloc(1, sizeof(XPoint));
  r->x = (short)Xen_integer_to_C_int(x);
  r->y = (short)Xen_integer_to_C_int(y);
  return(C_to_Xen_XPoint(r));
}

/* XArc */

static Xen gxm_XArc(Xen x, Xen y, Xen width, Xen height, Xen angle1, Xen angle2)
{
  XArc *r;
  #define H_XArc "(XArc x y w h ang1 ang2): new XArc struct"
  Xen_check_type(Xen_is_integer(x), x, 1, "XArc", "short");
  Xen_check_type(Xen_is_integer(y), y, 2, "XArc", "short");
  Xen_check_type(Xen_is_integer(width), width, 3, "XArc", "INT");
  Xen_check_type(Xen_is_integer(height), height, 4, "XArc", "INT");
  Xen_check_type(Xen_is_integer(angle1), angle1, 5, "XArc", "short");
  Xen_check_type(Xen_is_integer(angle2), angle2, 6, "XArc", "short");
  r = (XArc *)calloc(1, sizeof(XArc));
  r->x = (short)Xen_integer_to_C_int(x);
  r->y = (short)Xen_integer_to_C_int(y);
  r->width = Xen_integer_to_C_int(width);
  r->height = Xen_integer_to_C_int(height);
  r->angle1 = (short)Xen_integer_to_C_int(angle1);
  r->angle2 = (short)Xen_integer_to_C_int(angle2);
  return(C_to_Xen_XArc(r));
}

static Xen gxm_angle2(Xen ptr)
{
  XM_field_assert_type(Xen_is_XArc(ptr), ptr, 1, "angle2", "XArc");
  return(C_int_to_Xen_integer((int)((Xen_to_C_XArc(ptr))->angle2)));
}

static Xen gxm_set_angle2(Xen ptr, Xen val)
{
  XM_set_field_assert_type(Xen_is_integer(val), val, 1, "angle2", "an integer");
  XM_set_field_assert_type(Xen_is_XArc(ptr), ptr, 2, "angle2", "XArc");
  (Xen_to_C_XArc(ptr))->angle2 = (short)Xen_integer_to_C_int(val);
  return(val);
}

static Xen gxm_angle1(Xen ptr)
{
  XM_field_assert_type(Xen_is_XArc(ptr), ptr, 1, "angle1", "XArc");
  return(C_int_to_Xen_integer((int)((Xen_to_C_XArc(ptr))->angle1)));
}

static Xen gxm_set_angle1(Xen ptr, Xen val)
{
  XM_set_field_assert_type(Xen_is_integer(val), val, 1, "angle1", "an integer");
  XM_set_field_assert_type(Xen_is_XArc(ptr), ptr, 2, "angle1", "XArc");
  (Xen_to_C_XArc(ptr))->angle1 = (short)Xen_integer_to_C_int(val);
  return(val);
}

/* XColor */

static Xen gxm_XColor(Xen pixel, Xen red, Xen green, Xen blue, Xen flags, Xen pad)
{
  XColor *r;
  #define H_XColor "(XColor pixel red green blue flags pad): new XColor struct"
  Xen_check_type(Xen_is_ulong(pixel) || !Xen_is_bound(pixel), pixel, 1, "XColor", "ulong");
  Xen_check_type(Xen_is_integer_or_unbound(red), red, 2, "XColor", "int");
  Xen_check_type(Xen_is_integer_or_unbound(green), green, 3, "XColor", "int");
  Xen_check_type(Xen_is_integer_or_unbound(blue), blue, 4, "XColor", "int");
  Xen_check_type(Xen_is_integer_or_unbound(flags), flags, 5, "XColor", "char");
  Xen_check_type(Xen_is_integer_or_unbound(pad), pad, 6, "XColor", "char");
  r = (XColor *)calloc(1, sizeof(XColor));
  if (Xen_is_bound(pixel)) r->pixel = Xen_ulong_to_C_ulong(pixel);
  if (Xen_is_bound(red)) r->red = Xen_integer_to_C_int(red);
  if (Xen_is_bound(green)) r->green = Xen_integer_to_C_int(green);
  if (Xen_is_bound(blue)) r->blue = Xen_integer_to_C_int(blue);
  if (Xen_is_bound(flags)) r->flags = (char)Xen_integer_to_C_int(flags);
  if (Xen_is_bound(pad)) r->pad = (char)Xen_integer_to_C_int(pad);
  return(C_to_Xen_XColor(r));
}

static Xen gxm_pad(Xen ptr)
{
  XM_field_assert_type(Xen_is_XColor(ptr), ptr, 1, "pad", "XColor");
  return(C_int_to_Xen_integer((char)((Xen_to_C_XColor(ptr))->pad)));
}

static Xen gxm_set_pad(Xen ptr, Xen val)
{
  XM_set_field_assert_type(Xen_is_XColor(ptr), ptr, 1, "pad", "XColor");
  XM_set_field_assert_type(Xen_is_integer(val), val, 2, "pad", "char");
  (Xen_to_C_XColor(ptr))->pad = (char)Xen_integer_to_C_int(val);
  return(val);
}

static Xen gxm_blue(Xen ptr)
{
  if (Xen_is_XColor(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XColor(ptr))->blue)));
  XM_field_assert_type(0, ptr, 1, "blue", "XColor");
  return(Xen_false);
}

static Xen gxm_set_blue(Xen ptr, Xen val)
{
  XM_set_field_assert_type(Xen_is_integer(val), val, 2, "blue", "an integer");
  if (Xen_is_XColor(ptr)) (Xen_to_C_XColor(ptr))->blue = Xen_integer_to_C_int(val);
  else XM_set_field_assert_type(0, ptr, 1, "blue", "XColor");
  return(val);
}

static Xen gxm_green(Xen ptr)
{
  if (Xen_is_XColor(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XColor(ptr))->green)));
  XM_field_assert_type(0, ptr, 1, "green", "XColor");
  return(Xen_false);
}

static Xen gxm_set_green(Xen ptr, Xen val)
{
  XM_set_field_assert_type(Xen_is_integer(val), val, 2, "green", "an integer");
  if (Xen_is_XColor(ptr)) (Xen_to_C_XColor(ptr))->green = Xen_integer_to_C_int(val);
  else XM_set_field_assert_type(0, ptr, 1, "green", "XColor");
  return(val);
}

static Xen gxm_red(Xen ptr)
{
  if (Xen_is_XColor(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XColor(ptr))->red)));
  XM_field_assert_type(0, ptr, 1, "red", "XColor");
  return(Xen_false);
}

static Xen gxm_set_red(Xen ptr, Xen val)
{
  XM_set_field_assert_type(Xen_is_integer(val), val, 2, "red", "an integer");
  if (Xen_is_XColor(ptr)) (Xen_to_C_XColor(ptr))->red = Xen_integer_to_C_int(val);
  else XM_set_field_assert_type(0, ptr, 1, "red", "XColor");
  return(val);
}

static Xen gxm_XWindowChanges(Xen x, Xen y, Xen width, Xen height, Xen border_width, Xen sibling, Xen stack_mode)
{
  XWindowChanges *r;
  #define H_XWindowChanges "(XWindowChanges x y w h border sib stack_mode): new XWindowChanges struct"
  Xen_check_type(Xen_is_integer_or_unbound(x), x, 1, "XWindowChanges", "int");
  Xen_check_type(Xen_is_integer_or_unbound(y), y, 2, "XWindowChanges", "int");
  Xen_check_type(Xen_is_integer_or_unbound(width), width, 3, "XWindowChanges", "int");
  Xen_check_type(Xen_is_integer_or_unbound(height), height, 4, "XWindowChanges", "int");
  Xen_check_type(Xen_is_integer_or_unbound(border_width), border_width, 5, "XWindowChanges", "int");
  Xen_check_type(Xen_is_Window(sibling) || !Xen_is_bound(sibling), sibling, 6, "XWindowChanges", "Window");
  Xen_check_type(Xen_is_integer_or_unbound(stack_mode), stack_mode, 7, "XWindowChanges", "int");
  r = (XWindowChanges *)calloc(1, sizeof(XWindowChanges));
  if (Xen_is_bound(x)) r->x = Xen_integer_to_C_int(x);
  if (Xen_is_bound(y)) r->y = Xen_integer_to_C_int(y);
  if (Xen_is_bound(width)) r->width = Xen_integer_to_C_int(width);
  if (Xen_is_bound(height)) r->height = Xen_integer_to_C_int(height);
  if (Xen_is_bound(border_width)) r->border_width = Xen_integer_to_C_int(border_width);
  if (Xen_is_bound(sibling)) r->sibling = Xen_to_C_Window(sibling);
  if (Xen_is_bound(stack_mode)) r->stack_mode = Xen_integer_to_C_int(stack_mode);
  return(C_to_Xen_XWindowChanges(r));
}

static Xen gxm_XSetWindowAttributes(Xen arglist)
{
  int len;
  XSetWindowAttributes *r;
  #define H_XSetWindowAttributes "(XSetWindowAttributes args): new XSetWindowAttributes struct (14 or more args!)"
  r = (XSetWindowAttributes *)calloc(1, sizeof(XSetWindowAttributes));
  len = Xen_list_length(arglist);
  if ((len > 0) && (Xen_is_Pixmap(Xen_list_ref(arglist, 0)))) r->background_pixmap = Xen_to_C_Pixmap(Xen_list_ref(arglist, 0));
  if ((len > 1) && (Xen_is_Pixel(Xen_list_ref(arglist, 1)))) r->background_pixel = Xen_to_C_Pixel(Xen_list_ref(arglist, 1));
  if ((len > 2) && (Xen_is_Pixmap(Xen_list_ref(arglist, 2)))) r->border_pixmap = Xen_to_C_Pixmap(Xen_list_ref(arglist, 2));
  if ((len > 3) && (Xen_is_Pixel(Xen_list_ref(arglist, 3)))) r->border_pixel = Xen_to_C_Pixel(Xen_list_ref(arglist, 3));
  if ((len > 4) && (Xen_is_integer(Xen_list_ref(arglist, 4)))) r->bit_gravity = Xen_integer_to_C_int(Xen_list_ref(arglist, 4));
  if ((len > 5) && (Xen_is_integer(Xen_list_ref(arglist, 5)))) r->win_gravity = Xen_integer_to_C_int(Xen_list_ref(arglist, 5));
  if ((len > 6) && (Xen_is_integer(Xen_list_ref(arglist, 6)))) r->backing_store = Xen_integer_to_C_int(Xen_list_ref(arglist, 6));
  if ((len > 7) && (Xen_is_ulong(Xen_list_ref(arglist, 7)))) r->backing_planes = Xen_ulong_to_C_ulong(Xen_list_ref(arglist, 7));
  if ((len > 8) && (Xen_is_Pixel(Xen_list_ref(arglist, 8)))) r->backing_pixel = Xen_to_C_Pixel(Xen_list_ref(arglist, 8));
  if ((len > 9) && (Xen_is_boolean(Xen_list_ref(arglist, 9)))) r->save_under = Xen_boolean_to_C_bool(Xen_list_ref(arglist, 9));
  if ((len > 10) && (Xen_is_integer(Xen_list_ref(arglist, 10)))) r->event_mask = Xen_integer_to_C_int(Xen_list_ref(arglist, 10));
  if ((len > 11) && (Xen_is_integer(Xen_list_ref(arglist, 11)))) r->do_not_propagate_mask = Xen_integer_to_C_int(Xen_list_ref(arglist, 11));
  if ((len > 12) && (Xen_is_boolean(Xen_list_ref(arglist, 12)))) r->override_redirect = Xen_boolean_to_C_bool(Xen_list_ref(arglist, 12));
  if ((len > 13) && (Xen_is_Colormap(Xen_list_ref(arglist, 13)))) r->colormap = Xen_to_C_Colormap(Xen_list_ref(arglist, 13));
  if ((len > 14) && (Xen_is_Cursor(Xen_list_ref(arglist, 14)))) r->cursor = Xen_to_C_Cursor(Xen_list_ref(arglist, 14));
  return(C_to_Xen_XSetWindowAttributes(r));
}

static Xen gxm_stack_mode(Xen ptr)
{
  XM_field_assert_type(Xen_is_XWindowChanges(ptr), ptr, 1, "stack_mode", "XWindowChanges");
  return(C_int_to_Xen_integer((int)((Xen_to_C_XWindowChanges(ptr))->stack_mode)));
}

static Xen gxm_sibling(Xen ptr)
{
  XM_field_assert_type(Xen_is_XWindowChanges(ptr), ptr, 1, "sibling", "XWindowChanges");
  return(C_to_Xen_Window((Window)((Xen_to_C_XWindowChanges(ptr))->sibling)));
}

static Xen gxm_obdata(Xen ptr)
{
  XM_field_assert_type(Xen_is_XImage(ptr), ptr, 1, "obdata", "XImage");
  return(Xen_wrap_C_pointer((XPointer)((Xen_to_C_XImage(ptr))->obdata)));
}

static Xen gxm_bytes_per_line(Xen ptr)
{
  XM_field_assert_type(Xen_is_XImage(ptr), ptr, 1, "bytes_per_line", "XImage");
  return(C_int_to_Xen_integer((int)((Xen_to_C_XImage(ptr))->bytes_per_line)));
}

static Xen gxm_bitmap_pad(Xen ptr)
{
  XM_field_assert_type(Xen_is_XImage(ptr), ptr, 1, "bitmap_pad", "XImage");
  return(C_int_to_Xen_integer((int)((Xen_to_C_XImage(ptr))->bitmap_pad)));
}

static Xen gxm_bitmap_bit_order(Xen ptr)
{
  XM_field_assert_type(Xen_is_XImage(ptr), ptr, 1, "bitmap_bit_order", "XImage");
  return(C_int_to_Xen_integer((int)((Xen_to_C_XImage(ptr))->bitmap_bit_order)));
}

static Xen gxm_bitmap_unit(Xen ptr)
{
  XM_field_assert_type(Xen_is_XImage(ptr), ptr, 1, "bitmap_unit", "XImage");
  return(C_int_to_Xen_integer((int)((Xen_to_C_XImage(ptr))->bitmap_unit)));
}

static Xen gxm_byte_order(Xen ptr)
{
  if (Xen_is_XImage(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XImage(ptr))->byte_order)));
  XM_field_assert_type(0, ptr, 1, "byte_order", "XImage");
  return(Xen_false);
}

static Xen gxm_xoffset(Xen ptr)
{
  XM_field_assert_type(Xen_is_XImage(ptr), ptr, 1, "xoffset", "XImage");
  return(C_int_to_Xen_integer((int)((Xen_to_C_XImage(ptr))->xoffset)));
}

static Xen gxm_screen(Xen ptr)
{
  if (Xen_is_XWindowAttributes(ptr)) return(C_to_Xen_Screen((Screen *)((Xen_to_C_XWindowAttributes(ptr))->screen)));
  if (Xen_is_XmTopLevelEnterCallbackStruct(ptr)) return(C_to_Xen_Screen((Screen *)((Xen_to_C_XmTopLevelEnterCallbackStruct(ptr))->screen)));
  if (Xen_is_XmTopLevelLeaveCallbackStruct(ptr)) return(C_to_Xen_Screen((Screen *)((Xen_to_C_XmTopLevelLeaveCallbackStruct(ptr))->screen)));
  if (Xen_is_XVisualInfo(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XVisualInfo(ptr))->screen)));
  XM_field_assert_type(0, ptr, 1, "screen", "a struct with a screen field");
  return(Xen_false);
}

static Xen gxm_your_event_mask(Xen ptr)
{
  XM_field_assert_type(Xen_is_XWindowAttributes(ptr), ptr, 1, "your_event_mask", "XWindowAttributes");
  return(C_int_to_Xen_integer((long)((Xen_to_C_XWindowAttributes(ptr))->your_event_mask)));
}

static Xen gxm_all_event_masks(Xen ptr)
{
  XM_field_assert_type(Xen_is_XWindowAttributes(ptr), ptr, 1, "all_event_masks", "XWindowAttributes");
  return(C_int_to_Xen_integer((long)((Xen_to_C_XWindowAttributes(ptr))->all_event_masks)));
}

static Xen gxm_map_state(Xen ptr)
{
  XM_field_assert_type(Xen_is_XWindowAttributes(ptr), ptr, 1, "map_state", "XWindowAttributes");
  return(C_int_to_Xen_integer((int)((Xen_to_C_XWindowAttributes(ptr))->map_state)));
}

static Xen gxm_map_installed(Xen ptr)
{
  XM_field_assert_type(Xen_is_XWindowAttributes(ptr), ptr, 1, "map_installed", "XWindowAttributes");
  return(C_bool_to_Xen_boolean((Bool)((Xen_to_C_XWindowAttributes(ptr))->map_installed)));
}

static Xen gxm_set_visual(Xen ptr, Xen val)
{
  XM_set_field_assert_type(Xen_is_Visual(val), val, 2, "visual", "Visual");
  XM_set_field_assert_type(Xen_is_XpmAttributes(ptr), ptr, 1, "visual", "Visual");
  (Xen_to_C_XpmAttributes(ptr))->visual = (Visual *)Xen_to_C_Visual(val);
  return(val);
}

static Xen gxm_visual(Xen ptr)
{
  if (Xen_is_XWindowAttributes(ptr)) return(C_to_Xen_Visual((Visual *)((Xen_to_C_XWindowAttributes(ptr))->visual)));
  if (Xen_is_XpmAttributes(ptr)) return(C_to_Xen_Visual((Visual *)((Xen_to_C_XpmAttributes(ptr))->visual)));
  XM_field_assert_type(0, ptr, 1, "visual", "a struct with a visual field");
  return(Xen_false);
}

static Xen gxm_cursor(Xen ptr)
{
  if (Xen_is_XSetWindowAttributes(ptr)) return(C_to_Xen_Cursor((Cursor)((Xen_to_C_XSetWindowAttributes(ptr))->cursor)));
  XM_field_assert_type(0, ptr, 1, "cursor", "XSetWindowAttributes");
  return(Xen_false);
}

static Xen gxm_set_cursor(Xen ptr, Xen val)
{
  XM_set_field_assert_type(Xen_is_XSetWindowAttributes(ptr), ptr, 1, "cursor", "XSetWindowAttributes");
  XM_set_field_assert_type(Xen_is_Cursor(val), val, 2, "cursor", "a Cursor");
  (Xen_to_C_XSetWindowAttributes(ptr))->cursor = Xen_to_C_Cursor(val);
  return(val);
}

static Xen gxm_do_not_propagate_mask(Xen ptr)
{
  XM_field_assert_type(Xen_is_XWindowAttributes(ptr) || Xen_is_XSetWindowAttributes(ptr), 
		       ptr, 1, "do_not_propagate_mask", "a struct with a do_not_propagate_mask field");
  if (Xen_is_XWindowAttributes(ptr)) return(C_int_to_Xen_integer((long)((Xen_to_C_XWindowAttributes(ptr))->do_not_propagate_mask)));
  return(C_int_to_Xen_integer((long)((Xen_to_C_XSetWindowAttributes(ptr))->do_not_propagate_mask)));
}

static Xen gxm_event_mask(Xen ptr)
{
  if (Xen_is_XSetWindowAttributes(ptr)) return(C_int_to_Xen_integer((long)((Xen_to_C_XSetWindowAttributes(ptr))->event_mask)));
  XM_field_assert_type(0, ptr, 1, "event_mask", "XSetWindowAttributes");
  return(Xen_false);
}

static Xen gxm_set_event_mask(Xen ptr, Xen val)
{
  XM_set_field_assert_type(Xen_is_XSetWindowAttributes(ptr), ptr, 1, "event_mask", "XSetWindowAttributes");
  XM_set_field_assert_type(Xen_is_integer(val), val, 2, "event_mask", "an integer");
  (Xen_to_C_XSetWindowAttributes(ptr))->event_mask = Xen_integer_to_C_int(val);
  return(val);
}

static Xen gxm_save_under(Xen ptr)
{
  XM_field_assert_type(Xen_is_XWindowAttributes(ptr) || Xen_is_XSetWindowAttributes(ptr), ptr, 1, "save_under", "a struct with a save_under field");
  if (Xen_is_XWindowAttributes(ptr)) return(C_bool_to_Xen_boolean((Bool)((Xen_to_C_XWindowAttributes(ptr))->save_under)));
  return(C_bool_to_Xen_boolean((Bool)((Xen_to_C_XSetWindowAttributes(ptr))->save_under)));
}

static Xen gxm_set_save_under(Xen ptr, Xen val)
{
  XM_set_field_assert_type(Xen_is_XWindowAttributes(ptr) || Xen_is_XSetWindowAttributes(ptr), ptr, 1, "save_under", "a struct with a save_under field");
  XM_set_field_assert_type(Xen_is_boolean(val), val, 2, "save_under", "a Boolean");
  if (Xen_is_XWindowAttributes(ptr))
      (Xen_to_C_XWindowAttributes(ptr))->save_under = Xen_boolean_to_C_bool(val);
  else
      (Xen_to_C_XSetWindowAttributes(ptr))->save_under = Xen_boolean_to_C_bool(val);
  return(val);
}

static Xen gxm_backing_pixel(Xen ptr)
{
  XM_field_assert_type(Xen_is_XWindowAttributes(ptr) || Xen_is_XSetWindowAttributes(ptr), ptr, 1, "backing_pixel", "a struct with a backing_pixel field");
  if (Xen_is_XWindowAttributes(ptr)) return(C_to_Xen_Pixel((unsigned long)((Xen_to_C_XWindowAttributes(ptr))->backing_pixel)));
  return(C_to_Xen_Pixel((unsigned long)((Xen_to_C_XSetWindowAttributes(ptr))->backing_pixel)));
}

static Xen gxm_backing_planes(Xen ptr)
{
  XM_field_assert_type(Xen_is_XWindowAttributes(ptr) || Xen_is_XSetWindowAttributes(ptr), ptr, 1, "backing_planes", "a struct with a backing_planes field");
  if (Xen_is_XWindowAttributes(ptr)) return(C_ulong_to_Xen_ulong((unsigned long)((Xen_to_C_XWindowAttributes(ptr))->backing_planes)));
  return(C_ulong_to_Xen_ulong((unsigned long)((Xen_to_C_XSetWindowAttributes(ptr))->backing_planes)));
}

static Xen gxm_win_gravity(Xen ptr)
{
  if (Xen_is_XWindowAttributes(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XWindowAttributes(ptr))->win_gravity)));
  if (Xen_is_XWindowAttributes(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XSetWindowAttributes(ptr))->win_gravity)));
  XM_field_assert_type(0, ptr, 1, "win_gravity", "a struct with a win_gravity field");
  return(Xen_false);
}

static Xen gxm_bit_gravity(Xen ptr)
{
  XM_field_assert_type(Xen_is_XWindowAttributes(ptr) || Xen_is_XSetWindowAttributes(ptr), ptr, 1, "bit_gravity", "a struct with a bit_gravity field");
  if (Xen_is_XWindowAttributes(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XWindowAttributes(ptr))->bit_gravity)));
  return(C_int_to_Xen_integer((int)((Xen_to_C_XSetWindowAttributes(ptr))->bit_gravity)));
}

static Xen gxm_set_bit_gravity(Xen ptr, Xen val)
{
  XM_set_field_assert_type(Xen_is_XWindowAttributes(ptr) || Xen_is_XSetWindowAttributes(ptr), ptr, 1, "bit_gravity", "a struct with a bit_gravity field");
  XM_set_field_assert_type(Xen_is_integer(val), val, 2, "bit_gravity", "an integer");
  if (Xen_is_XWindowAttributes(ptr))
      (Xen_to_C_XWindowAttributes(ptr))->bit_gravity = Xen_integer_to_C_int(val);
  else
      (Xen_to_C_XSetWindowAttributes(ptr))->bit_gravity = Xen_integer_to_C_int(val);
  return(val);
}

static Xen gxm_border_pixel(Xen ptr)
{
  XM_field_assert_type(Xen_is_XSetWindowAttributes(ptr), ptr, 1, "border_pixel", "XSetWindowAttributes");
  return(C_to_Xen_Pixel((unsigned long)((Xen_to_C_XSetWindowAttributes(ptr))->border_pixel)));
}

static Xen gxm_set_border_pixel(Xen ptr, Xen val)
{
  XM_set_field_assert_type(Xen_is_XSetWindowAttributes(ptr), ptr, 1, "border_pixel", "XSetWindowAttributes");
  XM_set_field_assert_type(Xen_is_Pixel(val), val, 2, "border_pixel", "a Pixel");
  (Xen_to_C_XSetWindowAttributes(ptr))->border_pixel = Xen_to_C_Pixel(val);
  return(val);
}

static Xen gxm_border_pixmap(Xen ptr)
{
  XM_field_assert_type(Xen_is_XSetWindowAttributes(ptr), ptr, 1, "border_pixmap", "XSetWindowAttributes");
  return(C_to_Xen_Pixmap((Pixmap)((Xen_to_C_XSetWindowAttributes(ptr))->border_pixmap)));
}

static Xen gxm_background_pixel(Xen ptr)
{
  XM_field_assert_type(Xen_is_XSetWindowAttributes(ptr), ptr, 1, "background_pixel", "XSetWindowAttributes");
  return(C_to_Xen_Pixel((unsigned long)((Xen_to_C_XSetWindowAttributes(ptr))->background_pixel)));
}

static Xen gxm_set_background_pixel(Xen ptr, Xen val)
{
  XM_set_field_assert_type(Xen_is_XSetWindowAttributes(ptr), ptr, 1, "background_pixel", "XSetWindowAttributes");
  XM_set_field_assert_type(Xen_is_Pixel(val), val, 2, "background_pixel", "a Pixel");
  (Xen_to_C_XSetWindowAttributes(ptr))->background_pixel = Xen_to_C_Pixel(val);
  return(val);
}

static Xen gxm_background_pixmap(Xen ptr)
{
  XM_field_assert_type(Xen_is_XSetWindowAttributes(ptr), ptr, 1, "background_pixmap", "XSetWindowAttributes");
  return(C_to_Xen_Pixmap((Pixmap)((Xen_to_C_XSetWindowAttributes(ptr))->background_pixmap)));
}

static Xen gxm_bits_per_pixel(Xen ptr)
{
  if (Xen_is_XImage(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XImage(ptr))->bits_per_pixel)));
  XM_field_assert_type(0, ptr, 1, "bits_per_pixel", "XImage");
  return(Xen_false);
}

static Xen gxm_visuals(Xen ptr)
{
  Depth *dps;
  int len;
  Xen lst = Xen_empty_list;
  XM_field_assert_type(Xen_is_Depth(ptr), ptr, 1, "visuals", "Depth");
  dps = Xen_to_C_Depth(ptr);
  len = dps->nvisuals;
  if (len > 0)
    {
      int i;
      Visual *vs;
      vs = dps->visuals;
      for (i = len - 1; i >= 0; i--)
	lst = Xen_cons(wrap_for_Xen("Visual", &(vs[i])), lst);
    }
  return(lst);
}

static Xen gxm_nvisuals(Xen ptr)
{
  XM_field_assert_type(Xen_is_Depth(ptr), ptr, 1, "nvisuals", "Depth");
  return(C_int_to_Xen_integer((int)((Xen_to_C_Depth(ptr))->nvisuals)));
}

static Xen gxm_set_depth(Xen ptr, Xen val)
{
  XM_set_field_assert_type(Xen_is_XpmAttributes(ptr), ptr, 1, "depth", "XpmAttributes");
  (Xen_to_C_XpmAttributes(ptr))->depth = Xen_ulong_to_C_ulong(val);
  return(val);
}

static Xen gxm_depth(Xen ptr)
{
  if (Xen_is_XImage(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XImage(ptr))->depth)));
  if (Xen_is_XWindowAttributes(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XWindowAttributes(ptr))->depth)));
  if (Xen_is_Depth(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_Depth(ptr))->depth)));
  if (Xen_is_XVisualInfo(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XVisualInfo(ptr))->depth)));
  if (Xen_is_XpmAttributes(ptr)) return(C_ulong_to_Xen_ulong((unsigned long)((Xen_to_C_XpmAttributes(ptr))->depth)));
  XM_field_assert_type(0, ptr, 1, "depth", "a struct with a depth field");
  return(Xen_false);
}

static Xen gxm_map_entries(Xen ptr)
{
  XM_field_assert_type(Xen_is_Visual(ptr), ptr, 1, "map_entries", "Visual");
  return(C_int_to_Xen_integer((int)((Xen_to_C_Visual(ptr))->map_entries)));
}

static Xen gxm_bits_per_rgb(Xen ptr)
{
  if (Xen_is_XVisualInfo(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XVisualInfo(ptr))->bits_per_rgb)));
  if (Xen_is_Visual(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_Visual(ptr))->bits_per_rgb)));
  XM_field_assert_type(0, ptr, 1, "bits_per_rgb", "Visual or XVisualInfo");
  return(Xen_false);
}

static Xen gxm_blue_mask(Xen ptr)
{
  if (Xen_is_XImage(ptr)) return(C_ulong_to_Xen_ulong((unsigned long)((Xen_to_C_XImage(ptr))->blue_mask)));
  if (Xen_is_XVisualInfo(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XVisualInfo(ptr))->blue_mask)));
  if (Xen_is_Visual(ptr)) return(C_int_to_Xen_integer((long)((Xen_to_C_Visual(ptr))->blue_mask)));
  XM_field_assert_type(0, ptr, 1, "blue_mask", "a struct with a blue_mask field");
  return(Xen_false);
}

static Xen gxm_green_mask(Xen ptr)
{
  if (Xen_is_XImage(ptr)) return(C_ulong_to_Xen_ulong((unsigned long)((Xen_to_C_XImage(ptr))->green_mask)));
  if (Xen_is_XVisualInfo(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XVisualInfo(ptr))->green_mask)));
  if (Xen_is_Visual(ptr)) return(C_int_to_Xen_integer((long)((Xen_to_C_Visual(ptr))->green_mask)));
  XM_field_assert_type(0, ptr, 1, "green_mask", "a struct with a green_mask field");
  return(Xen_false);
}

static Xen gxm_red_mask(Xen ptr)
{
  if (Xen_is_XImage(ptr)) return(C_ulong_to_Xen_ulong((unsigned long)((Xen_to_C_XImage(ptr))->red_mask)));
  if (Xen_is_XVisualInfo(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XVisualInfo(ptr))->red_mask)));
  if (Xen_is_Visual(ptr)) return(C_ulong_to_Xen_ulong((unsigned long)((Xen_to_C_Visual(ptr))->red_mask)));
  XM_field_assert_type(0, ptr, 1, "red_mask", "a struct with a red_mask field");
  return(Xen_false);
}

static Xen gxm_colormap_size(Xen ptr)
{
  if (Xen_is_XVisualInfo(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XVisualInfo(ptr))->colormap_size)));
  XM_field_assert_type(0, ptr, 1, "colormap_size", "XVisualInfo*");
  return(Xen_false);
}

static Xen gxm_class(Xen ptr)
{
#ifndef __cplusplus
  if (Xen_is_XWindowAttributes(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XWindowAttributes(ptr))->class)));
  if (Xen_is_Visual(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_Visual(ptr))->class)));
  if (Xen_is_XVisualInfo(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XVisualInfo(ptr))->class)));
  XM_field_assert_type(0, ptr, 1, "class", "a struct with a class field");
#endif
  return(Xen_false);
}

static Xen gxm_visualid(Xen ptr)
{
  if (Xen_is_Visual(ptr)) return(C_ulong_to_Xen_ulong((VisualID)((Xen_to_C_Visual(ptr))->visualid)));
  if (Xen_is_XStandardColormap(ptr)) return(C_ulong_to_Xen_ulong((VisualID)((Xen_to_C_XStandardColormap(ptr))->visualid)));
  if (Xen_is_XVisualInfo(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XVisualInfo(ptr))->visualid)));
  XM_field_assert_type(0, ptr, 1, "visualid", "a struct with a visualid field");
  return(Xen_false);
}

static Xen gxm_set_pixel(Xen ptr, Xen val)
{
  XM_set_field_assert_type(Xen_is_Pixel(val), val, 2, "pixel", "a Pixel");  
  if (Xen_is_XColor(ptr)) (Xen_to_C_XColor(ptr))->pixel = Xen_to_C_Pixel(val);
  else if (Xen_is_XpmColorSymbol(ptr)) (Xen_to_C_XpmColorSymbol(ptr))->pixel = Xen_to_C_Pixel(val);
  return(val);
}

static Xen gxm_pixel(Xen ptr)
{
  if (Xen_is_XColor(ptr)) return(C_to_Xen_Pixel((Pixel)((Xen_to_C_XColor(ptr))->pixel)));
  if (Xen_is_XmScrollBarCallbackStruct(ptr)) return(C_to_Xen_Pixel((Pixel)((Xen_to_C_XmScrollBarCallbackStruct(ptr))->pixel)));
  if (Xen_is_XpmColorSymbol(ptr)) return(C_to_Xen_Pixel((Pixel)((Xen_to_C_XpmColorSymbol(ptr))->pixel)));
  XM_field_assert_type(0, ptr, 1, "pixel", "a struct with a pixel field");
  return(Xen_false);
}

static Xen gxm_window_group(Xen ptr)
{
  XM_field_assert_type(Xen_is_XWMHints(ptr), ptr, 1, "window_group", "XWMHints");
  return(C_ulong_to_Xen_ulong((XID)((Xen_to_C_XWMHints(ptr))->window_group)));
}

static Xen gxm_icon_mask(Xen ptr)
{
  XM_field_assert_type(Xen_is_XWMHints(ptr), ptr, 1, "icon_mask", "XWMHints");
  return(C_to_Xen_Pixmap((Pixmap)((Xen_to_C_XWMHints(ptr))->icon_mask)));
}

static Xen gxm_icon_y(Xen ptr)
{
  XM_field_assert_type(Xen_is_XWMHints(ptr), ptr, 1, "icon_y", "XWMHints");
  return(C_int_to_Xen_integer((int)((Xen_to_C_XWMHints(ptr))->icon_y)));
}

static Xen gxm_icon_x(Xen ptr)
{
  XM_field_assert_type(Xen_is_XWMHints(ptr), ptr, 1, "icon_x", "XWMHints");
  return(C_int_to_Xen_integer((int)((Xen_to_C_XWMHints(ptr))->icon_x)));
}

static Xen gxm_icon_window(Xen ptr)
{
  XM_field_assert_type(Xen_is_XWMHints(ptr), ptr, 1, "icon_window", "XWMHints");
  return(C_to_Xen_Window((Window)((Xen_to_C_XWMHints(ptr))->icon_window)));
}

static Xen gxm_icon_pixmap(Xen ptr)
{
  XM_field_assert_type(Xen_is_XWMHints(ptr), ptr, 1, "icon_pixmap", "XWMHints");
  return(C_to_Xen_Pixmap((Pixmap)((Xen_to_C_XWMHints(ptr))->icon_pixmap)));
}

static Xen gxm_initial_state(Xen ptr)
{
  XM_field_assert_type(Xen_is_XWMHints(ptr), ptr, 1, "initial_state", "XWMHints");
  return(C_int_to_Xen_integer((int)((Xen_to_C_XWMHints(ptr))->initial_state)));
}

static Xen gxm_set_initial_state(Xen ptr, Xen val)
{
  XM_set_field_assert_type(Xen_is_XWMHints(ptr), ptr, 1, "initial_state", "XWMHints");
  XM_set_field_assert_type(Xen_is_integer(val), val, 2, "initial_state", "an integer");
  (Xen_to_C_XWMHints(ptr))->initial_state = Xen_integer_to_C_int(val);
  return(val);
}

static Xen gxm_input(Xen ptr)
{
  XM_field_assert_type(Xen_is_XWMHints(ptr), ptr, 1, "input", "XWMHints");
  return(C_bool_to_Xen_boolean((Bool)((Xen_to_C_XWMHints(ptr))->input)));
}

static Xen gxm_set_input(Xen ptr, Xen val)
{
  XM_set_field_assert_type(Xen_is_XWMHints(ptr), ptr, 1, "input", "XWMHints");
  XM_set_field_assert_type(Xen_is_boolean(val), val, 2, "input", "a boolean");
  (Xen_to_C_XWMHints(ptr))->input = Xen_boolean_to_C_bool(val);
  return(val);
}

static Xen gxm_flags(Xen ptr)
{
  if (Xen_is_XColor(ptr)) return(C_int_to_Xen_integer((char)((Xen_to_C_XColor(ptr))->flags)));
  if (Xen_is_XmSelectionCallbackStruct(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XmSelectionCallbackStruct(ptr))->flags)));
  if (Xen_is_XmDestinationCallbackStruct(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XmDestinationCallbackStruct(ptr))->flags)));
  if (Xen_is_XmConvertCallbackStruct(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XmConvertCallbackStruct(ptr))->flags)));
  if (Xen_is_XWMHints(ptr)) return(C_int_to_Xen_integer((long)((Xen_to_C_XWMHints(ptr))->flags)));
  XM_field_assert_type(0, ptr, 1, "flags", "a struct with a flags field");
  return(Xen_false);
}

static Xen gxm_set_flags(Xen ptr, Xen val)
{
  XM_set_field_assert_type(Xen_is_XColor(ptr) || Xen_is_XWMHints(ptr), ptr, 1, "flags", "XColor or XWMHints");
  XM_set_field_assert_type(Xen_is_integer(val), val, 2, "input", "a char (int)");
  if (Xen_is_XColor(ptr)) (Xen_to_C_XColor(ptr))->flags = (char)Xen_integer_to_C_int(val);
  else if (Xen_is_XWMHints(ptr)) (Xen_to_C_XWMHints(ptr))->flags = (char)Xen_integer_to_C_int(val);
  return(val);
}

static Xen gxm_per_char(Xen ptr)
{
  XM_field_assert_type(Xen_is_XFontStruct(ptr), ptr, 1, "per_char", "XFontStruct");
  return(C_to_Xen_XCharStruct((XCharStruct *)((Xen_to_C_XFontStruct(ptr))->per_char)));
}

static Xen gxm_max_bounds(Xen ptr)
{
  XM_field_assert_type(Xen_is_XFontStruct(ptr), ptr, 1, "max_bounds", "XFontStruct");
  return(C_to_Xen_XCharStruct((XCharStruct *)(&(Xen_to_C_XFontStruct(ptr))->max_bounds)));
}

static Xen gxm_min_bounds(Xen ptr)
{
  XM_field_assert_type(Xen_is_XFontStruct(ptr), ptr, 1, "min_bounds", "XFontStruct");
  return(C_to_Xen_XCharStruct((XCharStruct *)(&(Xen_to_C_XFontStruct(ptr))->min_bounds)));
}

static Xen gxm_min_height(Xen ptr)
{
  if (Xen_is_XIconSize(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XIconSize(ptr))->min_height)));
  XM_field_assert_type(0, ptr, 1, "min_height", "XIconSize");
  return(Xen_false);
}

static Xen gxm_min_width(Xen ptr)
{
  if (Xen_is_XIconSize(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XIconSize(ptr))->min_width)));
  XM_field_assert_type(0, ptr, 1, "min_width", "XIconSize");
  return(Xen_false);
}

static Xen gxm_max_height(Xen ptr)
{
  if (Xen_is_XIconSize(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XIconSize(ptr))->max_height)));
  XM_field_assert_type(0, ptr, 1, "max_height", "XIconSize");
  return(Xen_false);
}

static Xen gxm_max_width(Xen ptr)
{
  if (Xen_is_XIconSize(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XIconSize(ptr))->max_width)));
  XM_field_assert_type(0, ptr, 1, "max_width", "XIconSize");
  return(Xen_false);
}

static Xen gxm_height_inc(Xen ptr)
{
  if (Xen_is_XIconSize(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XIconSize(ptr))->height_inc)));
  XM_field_assert_type(0, ptr, 1, "height_inc", "XIconSize");
  return(Xen_false);
}

static Xen gxm_width_inc(Xen ptr)
{
  if (Xen_is_XIconSize(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XIconSize(ptr))->width_inc)));
  XM_field_assert_type(0, ptr, 1, "width_inc", "XIconSize");
  return(Xen_false);
}

static Xen gxm_properties(Xen ptr)
{
  int i, len;
  XFontStruct *fs;
  XFontProp *props;
  Xen lst = Xen_empty_list;
  XM_field_assert_type(Xen_is_XFontStruct(ptr), ptr, 1, "properties", "XFontStruct");
  fs = Xen_to_C_XFontStruct(ptr);
  len = fs->n_properties;
  props = fs->properties;
  for (i = 0; i < len; i++)
    lst = Xen_cons(C_to_Xen_XFontProp(&(props[i])), lst);
  return(lst);
}

static Xen gxm_fid(Xen ptr)
{
  XM_field_assert_type(Xen_is_XFontStruct(ptr), ptr, 1, "fid", "XFontStruct");
  return(C_to_Xen_Font((Font)((Xen_to_C_XFontStruct(ptr))->fid)));
}

static Xen gxm_card32(Xen ptr)
{
  XM_field_assert_type(Xen_is_XFontProp(ptr), ptr, 1, "card32", "XFontProp");
  return(C_ulong_to_Xen_ulong((unsigned long)((Xen_to_C_XFontProp(ptr))->card32)));
}

static Xen gxm_set_name(Xen ptr, Xen val)
{
  XM_set_field_assert_type(Xen_is_string(val), val, 2, "name", "a string");
  XM_set_field_assert_type(Xen_is_XpmColorSymbol(ptr), ptr, 1, "name", "XpmColorSymbol");
  (Xen_to_C_XpmColorSymbol(ptr))->name = xen_strdup(Xen_string_to_C_string(val));
  return(val);
}

static Xen gxm_name(Xen ptr)
{
  if (Xen_is_XFontProp(ptr)) return(C_to_Xen_Atom((Atom)((Xen_to_C_XFontProp(ptr))->name)));
  if (Xen_is_XpmColorSymbol(ptr)) return(C_string_to_Xen_string((char *)((Xen_to_C_XpmColorSymbol(ptr))->name)));
  XM_field_assert_type(0, ptr, 1, "name", "a struct with a name field");
  return(Xen_false);
}

static Xen gxm_attributes(Xen ptr)
{
  XM_field_assert_type(Xen_is_XCharStruct(ptr), ptr, 1, "attributes", "XCharStruct");
  return(C_int_to_Xen_integer((int)((Xen_to_C_XCharStruct(ptr))->attributes)));
}

static Xen gxm_descent(Xen ptr)
{
  if (Xen_is_XFontStruct(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XFontStruct(ptr))->descent)));
  if (Xen_is_XCharStruct(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XCharStruct(ptr))->descent)));
  XM_field_assert_type(0, ptr, 1, "descent", "a struct with a descent field");
  return(Xen_false);
}

static Xen gxm_ascent(Xen ptr)
{
  if (Xen_is_XFontStruct(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XFontStruct(ptr))->ascent)));
  if (Xen_is_XCharStruct(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XCharStruct(ptr))->ascent)));
  XM_field_assert_type(0, ptr, 1, "ascent", "a struct with an ascent field");
  return(Xen_false);
}

static Xen gxm_rbearing(Xen ptr)
{
  XM_field_assert_type(Xen_is_XCharStruct(ptr), ptr, 1, "rbearing", "XCharStruct");
  return(C_int_to_Xen_integer((int)((Xen_to_C_XCharStruct(ptr))->rbearing)));
}

static Xen gxm_lbearing(Xen ptr)
{
  XM_field_assert_type(Xen_is_XCharStruct(ptr), ptr, 1, "lbearing", "XCharStruct");
  return(C_int_to_Xen_integer((int)((Xen_to_C_XCharStruct(ptr))->lbearing)));
}

static Xen gxm_request_code(Xen ptr)
{
  XM_field_assert_type(Xen_is_XErrorEvent(ptr), ptr, 1, "request_code", "XErrorEvent");
  return(C_int_to_Xen_integer((int)((Xen_to_C_XErrorEvent(ptr))->request_code)));
}

static Xen gxm_set_request_code(Xen ptr, Xen val)
{
  XM_set_field_assert_type(Xen_is_integer(val), val, 2, "request_code", "an integer");
  XM_set_field_assert_type(Xen_is_XErrorEvent(ptr), ptr, 1, "request_code", "XErrorEvent");
  (Xen_to_C_XErrorEvent(ptr))->request_code = Xen_integer_to_C_int(val);
  return(val);
}

static Xen gxm_error_code(Xen ptr)
{
  XM_field_assert_type(Xen_is_XErrorEvent(ptr), ptr, 1, "error_code", "XErrorEvent");
  return(C_int_to_Xen_integer((int)((Xen_to_C_XErrorEvent(ptr))->error_code)));
}

static Xen gxm_set_error_code(Xen ptr, Xen val)
{
  XM_set_field_assert_type(Xen_is_integer(val), val, 2, "error_code", "an integer");
  XM_set_field_assert_type(Xen_is_XErrorEvent(ptr), ptr, 1, "error_code", "XErrorEvent");
  (Xen_to_C_XErrorEvent(ptr))->error_code = Xen_integer_to_C_int(val);
  return(val);
}

static Xen gxm_resourceid(Xen ptr)
{
  XM_field_assert_type(Xen_is_XErrorEvent(ptr), ptr, 1, "resourceid", "XErrorEvent");
  return(C_ulong_to_Xen_ulong((XID)((Xen_to_C_XErrorEvent(ptr))->resourceid)));
}

static Xen gxm_set_resourceid(Xen ptr, Xen val)
{
  XM_set_field_assert_type(Xen_is_ulong(val), val, 2, "resourceid", "an XID");
  XM_set_field_assert_type(Xen_is_XErrorEvent(ptr), ptr, 1, "resourceid", "XErrorEvent");
  (Xen_to_C_XErrorEvent(ptr))->resourceid = (XID)Xen_ulong_to_C_ulong(val);
  return(val);
}

static Xen gxm_first_keycode(Xen ptr)
{
  XM_field_assert_type(Xen_is_XMappingEvent(ptr), ptr, 1, "first_keycode", "XMappingEvent");
  return(C_int_to_Xen_integer((int)((Xen_to_C_XMappingEvent(ptr))->first_keycode)));
}

static Xen gxm_set_first_keycode(Xen ptr, Xen val)
{
  XM_set_field_assert_type(Xen_is_integer(val), val, 2, "first_keycode", "an integer");
  XM_set_field_assert_type(Xen_is_XMappingEvent(ptr), ptr, 1, "first_keycode", "XMappingEvent");
  (Xen_to_C_XMappingEvent(ptr))->first_keycode = Xen_integer_to_C_int(val);
  return(val);
}

static Xen gxm_request(Xen ptr)
{
  XM_field_assert_type(Xen_is_XMappingEvent(ptr), ptr, 1, "request", "XMappingEvent");
  return(C_int_to_Xen_integer((int)((Xen_to_C_XMappingEvent(ptr))->request)));
}

static Xen gxm_set_request(Xen ptr, Xen val)
{
  XM_set_field_assert_type(Xen_is_integer(val), val, 2, "request", "an integer");
  XM_set_field_assert_type(Xen_is_XMappingEvent(ptr), ptr, 1, "request", "XMappingEvent");
  (Xen_to_C_XMappingEvent(ptr))->request = Xen_integer_to_C_int(val);
  return(val);
}

static Xen gxm_format(Xen ptr)
{
  if (Xen_is_XImage(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XImage(ptr))->format)));
  if (Xen_is_XmSelectionCallbackStruct(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XmSelectionCallbackStruct(ptr))->format)));
  if (Xen_is_XmConvertCallbackStruct(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XmConvertCallbackStruct(ptr))->format)));
  if (Xen_is_XmTextBlock(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XmTextBlock(ptr))->format)));
  if (Xen_is_XClientMessageEvent(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XClientMessageEvent(ptr))->format)));
  XM_field_assert_type(0, ptr, 1, "format", "XClientMessageEvent");
  return(Xen_false);
}

static Xen gxm_set_format(Xen ptr, Xen val)
{
  XM_set_field_assert_type(Xen_is_integer(val), val, 2, "format", "an integer");
  if (Xen_is_XClientMessageEvent(ptr)) (Xen_to_C_XClientMessageEvent(ptr))->format = Xen_integer_to_C_int(val);
  else if (Xen_is_XmTextBlock(ptr)) (Xen_to_C_XmTextBlock(ptr))->format = Xen_integer_to_C_int(val);
  else XM_set_field_assert_type(0, ptr, 1, "format", "XClientMessageEvent or XmTextBlock");
  return(val);
}

static Xen gxm_message_type(Xen ptr)
{
  XM_field_assert_type(Xen_is_XClientMessageEvent(ptr), ptr, 1, "message_type", "XClientMessageEvent");
  return(C_to_Xen_Atom((Atom)((Xen_to_C_XClientMessageEvent(ptr))->message_type)));
}

static Xen gxm_set_message_type(Xen ptr, Xen val)
{
  XM_set_field_assert_type(Xen_is_Atom(val), val, 2, "message_type", "an Atom");
  XM_set_field_assert_type(Xen_is_XClientMessageEvent(ptr), ptr, 1, "message_type", "XClientMessageEvent");
  (Xen_to_C_XClientMessageEvent(ptr))->message_type = Xen_to_C_Atom(val);
  return(val);
}

static Xen gxm_new(Xen ptr)
{
#ifndef __cplusplus
  XM_field_assert_type(Xen_is_XColormapEvent(ptr), ptr, 1, "new", "XColormapEvent");
  return(C_bool_to_Xen_boolean((Bool)((Xen_to_C_XColormapEvent(ptr))->new)));
#endif
  return(Xen_false);
}

static Xen gxm_set_new(Xen ptr, Xen val)
{
  XM_set_field_assert_type(Xen_is_boolean(val), val, 2, "new", "a boolean");
#ifndef __cplusplus
  if (Xen_is_XColormapEvent(ptr)) (Xen_to_C_XColormapEvent(ptr))->new = Xen_boolean_to_C_bool(val);
  else XM_set_field_assert_type(0, ptr, 1, "new", "XColormapEvent");
#endif
  return(val);
}

static Xen gxm_colormap(Xen ptr)
{
  if (Xen_is_XWindowAttributes(ptr)) return(C_to_Xen_Colormap((Colormap)((Xen_to_C_XWindowAttributes(ptr))->colormap)));
  if (Xen_is_XSetWindowAttributes(ptr)) return(C_to_Xen_Colormap((Colormap)((Xen_to_C_XSetWindowAttributes(ptr))->colormap)));
  if (Xen_is_XColormapEvent(ptr)) return(C_to_Xen_Colormap((Colormap)((Xen_to_C_XColormapEvent(ptr))->colormap)));
  if (Xen_is_XStandardColormap(ptr)) return(C_to_Xen_Colormap((Colormap)((Xen_to_C_XStandardColormap(ptr))->colormap)));
  if (Xen_is_XpmAttributes(ptr)) return(C_to_Xen_Colormap((Colormap)((Xen_to_C_XpmAttributes(ptr))->colormap)));
  XM_field_assert_type(0, ptr, 1, "colormap", "a struct with a colormap field");
  return(Xen_false);
}

static Xen gxm_set_colormap(Xen ptr, Xen val)
{
  XM_set_field_assert_type(Xen_is_Colormap(val), val, 2, "colormap", "a Colormap");
  if (Xen_is_XpmAttributes(ptr)) (Xen_to_C_XpmAttributes(ptr))->colormap = Xen_to_C_Colormap(val);
  else 
    {
      if (Xen_is_XColormapEvent(ptr)) (Xen_to_C_XColormapEvent(ptr))->colormap = Xen_to_C_Colormap(val);
      else XM_set_field_assert_type(0, ptr, 1, "colormap", "XpmAttributes or XColormapEvent");
    }
  return(val);
}

static Xen gxm_property(Xen ptr)
{
  if (Xen_is_XSelectionEvent(ptr)) return(C_to_Xen_Atom((Atom)((Xen_to_C_XSelectionEvent(ptr))->property)));
  if (Xen_is_XSelectionRequestEvent(ptr)) return(C_to_Xen_Atom((Atom)((Xen_to_C_XSelectionRequestEvent(ptr))->property)));
  XM_field_assert_type(0, ptr, 1, "property", "a struct with a property field");
  return(Xen_false);
}

static Xen gxm_set_property(Xen ptr, Xen val)
{
  XM_set_field_assert_type(Xen_is_Atom(val), val, 2, "property", "an Atom");
  if (Xen_is_XSelectionEvent(ptr)) (Xen_to_C_XSelectionEvent(ptr))->property = Xen_to_C_Atom(val);
  else if (Xen_is_XSelectionRequestEvent(ptr)) (Xen_to_C_XSelectionRequestEvent(ptr))->property = Xen_to_C_Atom(val);
  else XM_set_field_assert_type(0, ptr, 1, "property", "XSelection(Request)Event");
  return(val);
}

static Xen gxm_target(Xen ptr)
{
  if (Xen_is_XmPopupHandlerCallbackStruct(ptr)) return(C_to_Xen_Widget((Widget)((Xen_to_C_XmPopupHandlerCallbackStruct(ptr))->target)));
  if (Xen_is_XmSelectionCallbackStruct(ptr)) return(C_to_Xen_Atom((Atom)((Xen_to_C_XmSelectionCallbackStruct(ptr))->target)));
  if (Xen_is_XmConvertCallbackStruct(ptr)) return(C_to_Xen_Atom((Atom)((Xen_to_C_XmConvertCallbackStruct(ptr))->target)));
  if (Xen_is_XSelectionEvent(ptr)) return(C_to_Xen_Atom((Atom)((Xen_to_C_XSelectionEvent(ptr))->target)));
  if (Xen_is_XSelectionRequestEvent(ptr)) return(C_to_Xen_Atom((Atom)((Xen_to_C_XSelectionRequestEvent(ptr))->target)));
  XM_field_assert_type(0, ptr, 1, "target", "a struct with a target field");
  return(Xen_false);
}

static Xen gxm_set_target(Xen ptr, Xen val)
{
  XM_set_field_assert_type(Xen_is_Atom(val), val, 2, "target", "an Atom");
  if (Xen_is_XSelectionEvent(ptr)) (Xen_to_C_XSelectionEvent(ptr))->target = Xen_to_C_Atom(val);
  else if (Xen_is_XSelectionRequestEvent(ptr)) (Xen_to_C_XSelectionRequestEvent(ptr))->target = Xen_to_C_Atom(val);
  else XM_set_field_assert_type(0, ptr, 1, "target", "XSelection(Request)Event");
  return(val);
}

static Xen gxm_requestor(Xen ptr)
{
  if (Xen_is_XSelectionEvent(ptr)) return(C_to_Xen_Window((Window)((Xen_to_C_XSelectionEvent(ptr))->requestor)));
  if (Xen_is_XSelectionRequestEvent(ptr)) return(C_to_Xen_Window((Window)((Xen_to_C_XSelectionRequestEvent(ptr))->requestor)));
  XM_field_assert_type(0, ptr, 1, "requestor", "a struct with a requestor field");
  return(Xen_false);
}

static Xen gxm_set_requestor(Xen ptr, Xen val)
{
  XM_set_field_assert_type(Xen_is_Window(val), val, 2, "requestor", "a Window");
  if (Xen_is_XSelectionEvent(ptr)) (Xen_to_C_XSelectionEvent(ptr))->requestor = Xen_to_C_Window(val);
  else if (Xen_is_XSelectionRequestEvent(ptr)) (Xen_to_C_XSelectionRequestEvent(ptr))->requestor = Xen_to_C_Window(val);
  else XM_set_field_assert_type(0, ptr, 1, "requestor", "XSelection(Request)Event");
  return(val);
}

static Xen gxm_owner(Xen ptr)
{
  XM_field_assert_type(Xen_is_XSelectionRequestEvent(ptr), ptr, 1, "owner", "XSelectionRequestEvent");
  return(C_to_Xen_Window((Window)((Xen_to_C_XSelectionRequestEvent(ptr))->owner)));
}

static Xen gxm_set_owner(Xen ptr, Xen val)
{
  XM_set_field_assert_type(Xen_is_Window(val), val, 2, "owner", "a Window");
  XM_set_field_assert_type(Xen_is_XSelectionRequestEvent(ptr), ptr, 1, "owner", "XSelectionRequestEvent");
  (Xen_to_C_XSelectionRequestEvent(ptr))->owner = Xen_to_C_Window(val);
  return(val);
}

static Xen gxm_selection(Xen ptr)
{
  if (Xen_is_XmTransferDoneCallbackStruct(ptr)) return(C_to_Xen_Atom((Atom)((Xen_to_C_XmTransferDoneCallbackStruct(ptr))->selection)));
  if (Xen_is_XmSelectionCallbackStruct(ptr)) return(C_to_Xen_Atom((Atom)((Xen_to_C_XmSelectionCallbackStruct(ptr))->selection)));
  if (Xen_is_XmDestinationCallbackStruct(ptr)) return(C_to_Xen_Atom((Atom)((Xen_to_C_XmDestinationCallbackStruct(ptr))->selection)));
  if (Xen_is_XmConvertCallbackStruct(ptr)) return(C_to_Xen_Atom((Atom)((Xen_to_C_XmConvertCallbackStruct(ptr))->selection)));
  if (Xen_is_XSelectionEvent(ptr)) return(C_to_Xen_Atom((Atom)((Xen_to_C_XSelectionEvent(ptr))->selection)));
  if (Xen_is_XSelectionRequestEvent(ptr)) return(C_to_Xen_Atom((Atom)((Xen_to_C_XSelectionRequestEvent(ptr))->selection)));
  if (Xen_is_XSelectionClearEvent(ptr)) return(C_to_Xen_Atom((Atom)((Xen_to_C_XSelectionClearEvent(ptr))->selection)));
  XM_field_assert_type(0, ptr, 1, "selection", "a struct with a selection field");
  return(Xen_false);
}

static Xen gxm_set_selection(Xen ptr, Xen val)
{
  XM_set_field_assert_type(Xen_is_Atom(val), val, 2, "selection", "an Atom");
  if (Xen_is_XSelectionEvent(ptr)) (Xen_to_C_XSelectionEvent(ptr))->selection = Xen_to_C_Atom(val);
  else if (Xen_is_XSelectionRequestEvent(ptr)) (Xen_to_C_XSelectionRequestEvent(ptr))->selection = Xen_to_C_Atom(val);
  else if (Xen_is_XSelectionClearEvent(ptr)) (Xen_to_C_XSelectionClearEvent(ptr))->selection = Xen_to_C_Atom(val);
  else XM_set_field_assert_type(0, ptr, 1, "selection", "XSelection(Request|Clear)Event");
  return(val);
}

static Xen gxm_atom(Xen ptr)
{
  if (Xen_is_XPropertyEvent(ptr)) return(C_to_Xen_Atom((Atom)((Xen_to_C_XPropertyEvent(ptr))->atom)));
  XM_field_assert_type(0, ptr, 1, "atom", "XPropertyEvent");
  return(Xen_false);
}

static Xen gxm_set_atom(Xen ptr, Xen val)
{
  XM_set_field_assert_type(Xen_is_Atom(val), val, 2, "atom", "an Atom");
  XM_set_field_assert_type(Xen_is_XPropertyEvent(ptr), ptr, 1, "atom", "XPropertyEvent");
  (Xen_to_C_XPropertyEvent(ptr))->atom = Xen_to_C_Atom(val);
  return(val);
}

static Xen gxm_place(Xen ptr)
{
  XM_field_assert_type(Xen_is_XCirculateRequestEvent(ptr) || Xen_is_XCirculateEvent(ptr), ptr, 1, "place", "a struct with a place field");
  if (Xen_is_XCirculateRequestEvent(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XCirculateRequestEvent(ptr))->place)));
  return(C_int_to_Xen_integer((int)((Xen_to_C_XCirculateEvent(ptr))->place)));
}

static Xen gxm_set_place(Xen ptr, Xen val)
{
  XM_set_field_assert_type(Xen_is_integer(val), val, 2, "place", "an integer");
  if (Xen_is_XCirculateRequestEvent(ptr)) (Xen_to_C_XCirculateRequestEvent(ptr))->place = Xen_integer_to_C_int(val);
  else if (Xen_is_XCirculateEvent(ptr)) (Xen_to_C_XCirculateEvent(ptr))->place = Xen_integer_to_C_int(val);
  else XM_set_field_assert_type(0, ptr, 1, "place", "a struct with a place field");
  return(val);
}

static Xen gxm_value_mask(Xen ptr)
{
  XM_field_assert_type(Xen_is_XConfigureRequestEvent(ptr), ptr, 1, "value_mask", "XConfigureRequestEvent");
  return(C_ulong_to_Xen_ulong((unsigned long)((Xen_to_C_XConfigureRequestEvent(ptr))->value_mask)));
}

static Xen gxm_set_value_mask(Xen ptr, Xen val)
{
  XM_set_field_assert_type(Xen_is_ulong(val), val, 2, "value_mask", "an unsigned long");
  XM_set_field_assert_type(Xen_is_XConfigureRequestEvent(ptr), ptr, 1, "value_mask", "XConfigureRequestEvent");
  (Xen_to_C_XConfigureRequestEvent(ptr))->value_mask = Xen_ulong_to_C_ulong(val);
  return(val);
}

static Xen gxm_above(Xen ptr)
{
  XM_field_assert_type(Xen_is_XConfigureRequestEvent(ptr) || Xen_is_XConfigureEvent(ptr), ptr, 1, "above", "a struct with an above field");
  if (Xen_is_XConfigureRequestEvent(ptr)) return(C_to_Xen_Window((Window)((Xen_to_C_XConfigureRequestEvent(ptr))->above)));
  return(C_to_Xen_Window((Window)((Xen_to_C_XConfigureEvent(ptr))->above)));
}

static Xen gxm_set_above(Xen ptr, Xen val)
{
  XM_set_field_assert_type(Xen_is_Window(val), val, 2, "above", "a Window");
  if (Xen_is_XConfigureRequestEvent(ptr)) (Xen_to_C_XConfigureRequestEvent(ptr))->above = Xen_to_C_Window(val);
  else if (Xen_is_XConfigureEvent(ptr)) (Xen_to_C_XConfigureEvent(ptr))->above = Xen_to_C_Window(val);
  else XM_set_field_assert_type(0, ptr, 1, "above", "a struct with an above field");
  return(val);
}

static Xen gxm_from_configure(Xen ptr)
{
  XM_field_assert_type(Xen_is_XUnmapEvent(ptr), ptr, 1, "from_configure", "XUnmapEvent");
  return(C_bool_to_Xen_boolean((Bool)((Xen_to_C_XUnmapEvent(ptr))->from_configure)));
}

static Xen gxm_set_from_configure(Xen ptr, Xen val)
{
  XM_set_field_assert_type(Xen_is_boolean(val), val, 2, "from_configure", "a boolean");
  XM_set_field_assert_type(Xen_is_XUnmapEvent(ptr), ptr, 1, "from_configure", "XUnmapEvent");
  (Xen_to_C_XUnmapEvent(ptr))->from_configure = Xen_boolean_to_C_bool(val);
  return(val);
}


static Xen gxm_event(Xen ptr)
{
  /* Xlib.h says event is a window in these cases -- kinda strange looking */
  if (Xen_is_XCirculateEvent(ptr)) return(C_to_Xen_Window((Window)((Xen_to_C_XCirculateEvent(ptr))->event)));
  if (Xen_is_XGravityEvent(ptr)) return(C_to_Xen_Window((Window)((Xen_to_C_XGravityEvent(ptr))->event)));
  if (Xen_is_XConfigureEvent(ptr)) return(C_to_Xen_Window((Window)((Xen_to_C_XConfigureEvent(ptr))->event)));
  if (Xen_is_XReparentEvent(ptr)) return(C_to_Xen_Window((Window)((Xen_to_C_XReparentEvent(ptr))->event)));
  if (Xen_is_XMapEvent(ptr)) return(C_to_Xen_Window((Window)((Xen_to_C_XMapEvent(ptr))->event)));
  if (Xen_is_XUnmapEvent(ptr)) return(C_to_Xen_Window((Window)((Xen_to_C_XUnmapEvent(ptr))->event)));
  if (Xen_is_XDestroyWindowEvent(ptr)) return(C_to_Xen_Window((Window)((Xen_to_C_XDestroyWindowEvent(ptr))->event)));
  if (Xen_is_AnyCallbackStruct(ptr)) return(C_to_Xen_XEvent((XEvent *)((Xen_to_C_XmAnyCallbackStruct(ptr))->event)));
  XM_field_assert_type(0, ptr, 1, "event", "a struct with an event field");
  return(Xen_false);
}

static Xen gxm_set_event(Xen ptr, Xen val)
{
  if (Xen_is_XCirculateEvent(ptr)) (Xen_to_C_XCirculateEvent(ptr))->event = Xen_to_C_Window(val); else
  if (Xen_is_XGravityEvent(ptr)) (Xen_to_C_XGravityEvent(ptr))->event = Xen_to_C_Window(val); else
  if (Xen_is_XConfigureEvent(ptr)) (Xen_to_C_XConfigureEvent(ptr))->event = Xen_to_C_Window(val); else
  if (Xen_is_XReparentEvent(ptr)) (Xen_to_C_XReparentEvent(ptr))->event = Xen_to_C_Window(val); else
  if (Xen_is_XMapEvent(ptr)) (Xen_to_C_XMapEvent(ptr))->event = Xen_to_C_Window(val); else
  if (Xen_is_XUnmapEvent(ptr)) (Xen_to_C_XUnmapEvent(ptr))->event = Xen_to_C_Window(val); else
  if (Xen_is_XDestroyWindowEvent(ptr)) (Xen_to_C_XDestroyWindowEvent(ptr))->event = Xen_to_C_Window(val); else
  if (Xen_is_AnyCallbackStruct(ptr)) (Xen_to_C_XmAnyCallbackStruct(ptr))->event = Xen_to_C_XEvent(val); else
  XM_set_field_assert_type(0, ptr, 1, "event", "a struct with an event field");
  return(val);
}

static Xen gxm_override_redirect(Xen ptr)
{
  if (Xen_is_XWindowAttributes(ptr)) return(C_bool_to_Xen_boolean((Bool)((Xen_to_C_XWindowAttributes(ptr))->override_redirect)));
  if (Xen_is_XSetWindowAttributes(ptr)) return(C_bool_to_Xen_boolean((Bool)((Xen_to_C_XSetWindowAttributes(ptr))->override_redirect)));
  if (Xen_is_XConfigureEvent(ptr)) return(C_bool_to_Xen_boolean((Bool)((Xen_to_C_XConfigureEvent(ptr))->override_redirect)));
  if (Xen_is_XReparentEvent(ptr)) return(C_bool_to_Xen_boolean((Bool)((Xen_to_C_XReparentEvent(ptr))->override_redirect)));
  if (Xen_is_XMapEvent(ptr)) return(C_bool_to_Xen_boolean((Bool)((Xen_to_C_XMapEvent(ptr))->override_redirect)));
  if (Xen_is_XCreateWindowEvent(ptr)) return(C_bool_to_Xen_boolean((Bool)((Xen_to_C_XCreateWindowEvent(ptr))->override_redirect)));
  XM_field_assert_type(0, ptr, 1, "override_redirect", "a struct with an override_redirect field");
  return(Xen_false);
}

static Xen gxm_set_override_redirect(Xen ptr, Xen val)
{
  Bool b;
  XM_set_field_assert_type(Xen_is_boolean(val), val, 2, "override_redirect", "a boolean");
  b = (Bool)Xen_boolean_to_C_bool(val);
  if (Xen_is_XConfigureEvent(ptr)) (Xen_to_C_XConfigureEvent(ptr))->override_redirect = b;
  else if (Xen_is_XReparentEvent(ptr)) (Xen_to_C_XReparentEvent(ptr))->override_redirect = b;
  else if (Xen_is_XMapEvent(ptr)) (Xen_to_C_XMapEvent(ptr))->override_redirect = b;
  else if (Xen_is_XCreateWindowEvent(ptr)) (Xen_to_C_XCreateWindowEvent(ptr))->override_redirect = b;
  else XM_set_field_assert_type(0, ptr, 1, "override_redirect", "a struct with an override_redirect field");
  return(val);
}

static Xen gxm_border_width(Xen ptr)
{
  if (Xen_is_XWindowChanges(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XWindowChanges(ptr))->border_width)));
  if (Xen_is_XWindowAttributes(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XWindowAttributes(ptr))->border_width)));
  if (Xen_is_XConfigureRequestEvent(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XConfigureRequestEvent(ptr))->border_width)));
  if (Xen_is_XConfigureEvent(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XConfigureEvent(ptr))->border_width)));
  if (Xen_is_XCreateWindowEvent(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XCreateWindowEvent(ptr))->border_width)));
  XM_field_assert_type(0, ptr, 1, "border_width", "a struct with a border_width field");
  return(Xen_false);
}

static Xen gxm_set_border_width(Xen ptr, Xen val)
{
  int wid;
  XM_set_field_assert_type(Xen_is_integer(val), val, 2, "border_width", "an integer");
  wid = Xen_integer_to_C_int(val);
  if (Xen_is_XConfigureRequestEvent(ptr)) (Xen_to_C_XConfigureRequestEvent(ptr))->border_width = wid;
  else if (Xen_is_XConfigureEvent(ptr)) (Xen_to_C_XConfigureEvent(ptr))->border_width = wid;
  else if (Xen_is_XCreateWindowEvent(ptr)) (Xen_to_C_XCreateWindowEvent(ptr))->border_width = wid;
  else XM_set_field_assert_type(0, ptr, 1, "border_width", "a struct with a border_width field");
  return(val);
}

static Xen gxm_parent(Xen ptr)
{
  if (Xen_is_XCirculateRequestEvent(ptr)) return(C_to_Xen_Window((Window)((Xen_to_C_XCirculateRequestEvent(ptr))->parent)));
  if (Xen_is_XConfigureRequestEvent(ptr)) return(C_to_Xen_Window((Window)((Xen_to_C_XConfigureRequestEvent(ptr))->parent)));
  if (Xen_is_XReparentEvent(ptr)) return(C_to_Xen_Window((Window)((Xen_to_C_XReparentEvent(ptr))->parent)));
  if (Xen_is_XMapRequestEvent(ptr)) return(C_to_Xen_Window((Window)((Xen_to_C_XMapRequestEvent(ptr))->parent)));
  if (Xen_is_XCreateWindowEvent(ptr)) return(C_to_Xen_Window((Window)((Xen_to_C_XCreateWindowEvent(ptr))->parent)));
  XM_field_assert_type(0, ptr, 1, "parent", "a struct with a parent field");
  return(Xen_false);
}

static Xen gxm_set_parent(Xen ptr, Xen val)
{
  Window w;
  XM_set_field_assert_type(Xen_is_Window(val), val, 2, "parent", "a Window");
  w = Xen_to_C_Window(val);
  if (Xen_is_XCirculateRequestEvent(ptr)) (Xen_to_C_XCirculateRequestEvent(ptr))->parent = w;
  else if (Xen_is_XConfigureRequestEvent(ptr)) (Xen_to_C_XConfigureRequestEvent(ptr))->parent = w;
  else if (Xen_is_XReparentEvent(ptr)) (Xen_to_C_XReparentEvent(ptr))->parent = w;
  else if (Xen_is_XMapRequestEvent(ptr)) (Xen_to_C_XMapRequestEvent(ptr))->parent = w;
  else if (Xen_is_XCreateWindowEvent(ptr)) (Xen_to_C_XCreateWindowEvent(ptr))->parent= w;
  else XM_set_field_assert_type(0, ptr, 1, "parent", "a struct with a parent field");
  return(val);
}

static Xen gxm_minor_code(Xen ptr)
{
  if (Xen_is_XErrorEvent(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XErrorEvent(ptr))->minor_code)));
  if (Xen_is_XNoExposeEvent(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XNoExposeEvent(ptr))->minor_code)));
  if (Xen_is_XGraphicsExposeEvent(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XGraphicsExposeEvent(ptr))->minor_code)));
  XM_field_assert_type(0, ptr, 1, "minor_code", "a struct with a minor_code field");
  return(Xen_false);
}

static Xen gxm_set_minor_code(Xen ptr, Xen val)
{
  XM_set_field_assert_type(Xen_is_integer(val), val, 2, "minor_code", "an integer");
  if (Xen_is_XErrorEvent(ptr)) (Xen_to_C_XErrorEvent(ptr))->minor_code = Xen_integer_to_C_int(val);
  else if (Xen_is_XNoExposeEvent(ptr)) (Xen_to_C_XNoExposeEvent(ptr))->minor_code = Xen_integer_to_C_int(val);
  else if (Xen_is_XGraphicsExposeEvent(ptr)) (Xen_to_C_XGraphicsExposeEvent(ptr))->minor_code = Xen_integer_to_C_int(val);
  else XM_set_field_assert_type(0, ptr, 1, "minor_code", "a struct with a minor_code field");
  return(val);
}

static Xen gxm_major_code(Xen ptr)
{
  XM_field_assert_type(Xen_is_XNoExposeEvent(ptr) || Xen_is_XGraphicsExposeEvent(ptr), ptr, 1, "major_code", "a struct with a major_code field");
  if (Xen_is_XNoExposeEvent(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XNoExposeEvent(ptr))->major_code)));
  return(C_int_to_Xen_integer((int)((Xen_to_C_XGraphicsExposeEvent(ptr))->major_code)));
}

static Xen gxm_set_major_code(Xen ptr, Xen val)
{
  XM_set_field_assert_type(Xen_is_integer(val), val, 2, "major_code", "an integer");
  if (Xen_is_XNoExposeEvent(ptr)) (Xen_to_C_XNoExposeEvent(ptr))->major_code = Xen_integer_to_C_int(val);
  else if (Xen_is_XGraphicsExposeEvent(ptr)) (Xen_to_C_XGraphicsExposeEvent(ptr))->major_code = Xen_integer_to_C_int(val);
  else XM_set_field_assert_type(0, ptr, 1, "major_code", "a struct with a major_code field");
  return(val);
}

static Xen gxm_drawable(Xen ptr)
{
  XM_field_assert_type(Xen_is_XNoExposeEvent(ptr) || Xen_is_XGraphicsExposeEvent(ptr), ptr, 1, "drawable", "a struct with a drawable field");
  if (Xen_is_XNoExposeEvent(ptr)) return(C_to_Xen_Window((Window)((Xen_to_C_XNoExposeEvent(ptr))->drawable)));
  return(C_to_Xen_Window((Window)((Xen_to_C_XGraphicsExposeEvent(ptr))->drawable)));
}

static Xen gxm_set_drawable(Xen ptr, Xen val)
{
  XM_set_field_assert_type(Xen_is_Window(val), val, 2, "drawable", "a Window");
  if (Xen_is_XNoExposeEvent(ptr)) (Xen_to_C_XNoExposeEvent(ptr))->drawable = Xen_to_C_Window(val);
  else if (Xen_is_XGraphicsExposeEvent(ptr)) (Xen_to_C_XGraphicsExposeEvent(ptr))->drawable = Xen_to_C_Window(val);
  else XM_set_field_assert_type(0, ptr, 1, "drawable", "a struct with a drawable field");
  return(val);
}

static Xen gxm_count(Xen ptr)
{
  if (Xen_is_XMappingEvent(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XMappingEvent(ptr))->count)));
  if (Xen_is_XGraphicsExposeEvent(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XGraphicsExposeEvent(ptr))->count)));
  if (Xen_is_XExposeEvent(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XExposeEvent(ptr))->count)));
  XM_field_assert_type(0, ptr, 1, "count", "a struct with a count field");
  return(Xen_false);
}

static Xen gxm_set_count(Xen ptr, Xen val)
{
  XM_set_field_assert_type(Xen_is_integer(val), val, 2, "count", "an integer");
  if (Xen_is_XMappingEvent(ptr)) (Xen_to_C_XMappingEvent(ptr))->count = Xen_integer_to_C_int(val);
  else if (Xen_is_XGraphicsExposeEvent(ptr)) (Xen_to_C_XGraphicsExposeEvent(ptr))->count = Xen_integer_to_C_int(val);
  else if (Xen_is_XExposeEvent(ptr)) (Xen_to_C_XExposeEvent(ptr))->count = Xen_integer_to_C_int(val);
  else XM_set_field_assert_type(0, ptr, 1, "count", "a struct with a count field");
  return(val);
}

static Xen gxm_key_vector(Xen ptr)
{
  XM_field_assert_type(Xen_is_XKeymapEvent(ptr), ptr, 1, "key_vector", "XKeymapEvent");
  return(C_string_to_Xen_string((char *)((Xen_to_C_XKeymapEvent(ptr))->key_vector)));
}

static Xen gxm_set_key_vector(Xen ptr, Xen val)
{
  char *keys;
  int lim = 0;
  XM_set_field_assert_type(Xen_is_string(val), val, 2, "key_vector", "a string");
  keys = (char *)Xen_string_to_C_string(val);
  if (keys) lim = strlen(keys);
  if (lim > 32) lim = 32;
  if (lim > 0)
    {
      if (Xen_is_XKeymapEvent(ptr))
	{
	  int i;
	  for (i = 0; i < lim; i++)
	    (Xen_to_C_XKeymapEvent(ptr))->key_vector[i] = keys[i];
	}
      else XM_set_field_assert_type(0, ptr, 1, "key_vector", "XKeymapEvent");
    }
  return(val);
}

static Xen gxm_focus(Xen ptr)
{
  if (Xen_is_XCrossingEvent(ptr)) return(C_bool_to_Xen_boolean((Bool)((Xen_to_C_XCrossingEvent(ptr))->focus)));
  XM_field_assert_type(0, ptr, 1, "focus", "XCrossingEvent");
  return(Xen_false);
}

static Xen gxm_set_focus(Xen ptr, Xen val)
{
  XM_set_field_assert_type(Xen_is_boolean(val), val, 2, "focus", "a boolean");
  XM_set_field_assert_type(Xen_is_XCrossingEvent(ptr), ptr, 1, "focus", "XCrossingEvent");
  (Xen_to_C_XCrossingEvent(ptr))->focus = (Bool)Xen_boolean_to_C_bool(val);
  return(val);
}

static Xen gxm_detail(Xen ptr)
{
  if (Xen_is_XConfigureRequestEvent(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XConfigureRequestEvent(ptr))->detail)));
  if (Xen_is_XFocusChangeEvent(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XFocusChangeEvent(ptr))->detail)));
  if (Xen_is_XCrossingEvent(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XCrossingEvent(ptr))->detail)));
  XM_field_assert_type(0, ptr, 1, "detail", "a struct with a detail field");
  return(Xen_false);
}

static Xen gxm_set_detail(Xen ptr, Xen val)
{
  XM_set_field_assert_type(Xen_is_integer(val), val, 2, "detail", "an integer");
  if (Xen_is_XConfigureRequestEvent(ptr)) (Xen_to_C_XConfigureRequestEvent(ptr))->detail = Xen_integer_to_C_int(val);
  else if (Xen_is_XFocusChangeEvent(ptr)) (Xen_to_C_XFocusChangeEvent(ptr))->detail = Xen_integer_to_C_int(val);
  else if (Xen_is_XCrossingEvent(ptr)) (Xen_to_C_XCrossingEvent(ptr))->detail = Xen_integer_to_C_int(val);
  else XM_set_field_assert_type(0, ptr, 1, "detail", "a struct with a detail field");
  return(val);
}

static Xen gxm_mode(Xen ptr)
{
  if (Xen_is_XFocusChangeEvent(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XFocusChangeEvent(ptr))->mode)));
  if (Xen_is_XCrossingEvent(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XCrossingEvent(ptr))->mode)));
  XM_field_assert_type(0, ptr, 1, "mode", "a struct with a mode field");
  return(Xen_false);
}

static Xen gxm_set_mode(Xen ptr, Xen val)
{
  XM_set_field_assert_type(Xen_is_integer(val), val, 2, "mode", "an integer");
  if (Xen_is_XFocusChangeEvent(ptr)) (Xen_to_C_XFocusChangeEvent(ptr))->mode = Xen_integer_to_C_int(val);
  else if (Xen_is_XCrossingEvent(ptr)) (Xen_to_C_XCrossingEvent(ptr))->mode  = Xen_integer_to_C_int(val);
  else XM_set_field_assert_type(0, ptr, 1, "mode", "a struct with a mode field");
  return(val);
}

static Xen gxm_is_hint(Xen ptr)
{
  if (Xen_is_XMotionEvent(ptr)) return(C_int_to_Xen_integer((char)((Xen_to_C_XMotionEvent(ptr))->is_hint)));
  XM_field_assert_type(0, ptr, 1, "is_hint", "XMotionEvent");
  return(Xen_false);
}

static Xen gxm_set_is_hint(Xen ptr, Xen val)
{
  XM_set_field_assert_type(Xen_is_integer(val), val, 2, "is_hint", "an integer");
  XM_set_field_assert_type(Xen_is_XMotionEvent(ptr), ptr, 1, "is_hint", "XMotionEvent");
  (Xen_to_C_XMotionEvent(ptr))->is_hint = (char)Xen_integer_to_C_int(val);
  return(val);
}

static Xen gxm_button(Xen ptr)
{
  if (Xen_is_XButtonEvent(ptr)) return(C_ulong_to_Xen_ulong((unsigned long)((Xen_to_C_XButtonEvent(ptr))->button)));
  XM_field_assert_type(0, ptr, 1, "button", "XButtonEvent");
  return(Xen_false);
}

static Xen gxm_set_button(Xen ptr, Xen val)
{
  XM_set_field_assert_type(Xen_is_ulong(val), val, 2, "button", "an unsigned long");
  XM_set_field_assert_type(Xen_is_XButtonEvent(ptr), ptr, 1, "button", "XButtonEvent");
  (Xen_to_C_XButtonEvent(ptr))->button = Xen_ulong_to_C_ulong(val);
  return(val);
}

static Xen gxm_same_screen(Xen ptr)
{
  if (Xen_is_XCrossingEvent(ptr)) return(C_bool_to_Xen_boolean((Bool)((Xen_to_C_XCrossingEvent(ptr))->same_screen)));
  if (Xen_is_XMotionEvent(ptr)) return(C_bool_to_Xen_boolean((Bool)((Xen_to_C_XMotionEvent(ptr))->same_screen)));
  if (Xen_is_XButtonEvent(ptr)) return(C_bool_to_Xen_boolean((Bool)((Xen_to_C_XButtonEvent(ptr))->same_screen)));
  if (Xen_is_XKeyEvent(ptr)) return(C_bool_to_Xen_boolean((Bool)((Xen_to_C_XKeyEvent(ptr))->same_screen)));
  XM_field_assert_type(0, ptr, 1, "same_screen", "a struct with a same_screen field");
  return(Xen_false);
}

static Xen gxm_set_same_screen(Xen ptr, Xen val)
{
  Bool b;
  XM_set_field_assert_type(Xen_is_boolean(val), val, 2, "same_screen", "a boolean");
  b = (Bool)Xen_boolean_to_C_bool(val);
  if (Xen_is_XCrossingEvent(ptr)) (Xen_to_C_XCrossingEvent(ptr))->same_screen = b;
  else if (Xen_is_XMotionEvent(ptr)) (Xen_to_C_XMotionEvent(ptr))->same_screen = b;
  else if (Xen_is_XButtonEvent(ptr)) (Xen_to_C_XButtonEvent(ptr))->same_screen = b;
  else if (Xen_is_XKeyEvent(ptr)) (Xen_to_C_XKeyEvent(ptr))->same_screen = b;
  else XM_set_field_assert_type(0, ptr, 1, "same_screen", "a struct with a same_screen field");
  return(val);
}

static Xen gxm_keycode(Xen ptr)
{
  if (Xen_is_XKeyEvent(ptr)) return(C_to_Xen_KeyCode((Xen_to_C_XKeyEvent(ptr))->keycode));
  XM_field_assert_type(0, ptr, 1, "keycode", "XKeyEvent");
  return(Xen_false);
}

static Xen gxm_set_keycode(Xen ptr, Xen val)
{
  XM_set_field_assert_type(Xen_is_KeyCode(val), val, 2, "keycode", "a KeyCode");
  XM_set_field_assert_type(Xen_is_XKeyEvent(ptr), ptr, 1, "keycode", "XKeyEvent");
  (Xen_to_C_XKeyEvent(ptr))->keycode = Xen_to_C_KeyCode(val);
  return(val);
}

static Xen gxm_state(Xen ptr)
{
  if (Xen_is_XColormapEvent(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XColormapEvent(ptr))->state)));
  if (Xen_is_XPropertyEvent(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XPropertyEvent(ptr))->state)));
  if (Xen_is_XVisibilityEvent(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XVisibilityEvent(ptr))->state)));
  if (Xen_is_XCrossingEvent(ptr)) return(C_ulong_to_Xen_ulong((unsigned long)((Xen_to_C_XCrossingEvent(ptr))->state)));
  if (Xen_is_XMotionEvent(ptr)) return(C_ulong_to_Xen_ulong((unsigned long)((Xen_to_C_XMotionEvent(ptr))->state)));
  if (Xen_is_XButtonEvent(ptr)) return(C_ulong_to_Xen_ulong((unsigned long)((Xen_to_C_XButtonEvent(ptr))->state)));
  if (Xen_is_XKeyEvent(ptr)) return(C_ulong_to_Xen_ulong((unsigned long)((Xen_to_C_XKeyEvent(ptr))->state)));
  XM_field_assert_type(0, ptr, 1, "state", "a struct with a state field");
  return(Xen_false);
}

static Xen gxm_set_state(Xen ptr, Xen val)
{
  XM_set_field_assert_type(Xen_is_integer(val) || Xen_is_ulong(val), val, 2, "state", "an integer");
  if (Xen_is_XColormapEvent(ptr)) (Xen_to_C_XColormapEvent(ptr))->state = Xen_integer_to_C_int(val);
  else if (Xen_is_XPropertyEvent(ptr)) (Xen_to_C_XPropertyEvent(ptr))->state = Xen_integer_to_C_int(val);
  else if (Xen_is_XVisibilityEvent(ptr)) (Xen_to_C_XVisibilityEvent(ptr))->state = Xen_integer_to_C_int(val);
  else if (Xen_is_XCrossingEvent(ptr)) (Xen_to_C_XCrossingEvent(ptr))->state = Xen_ulong_to_C_ulong(val);
  else if (Xen_is_XMotionEvent(ptr)) (Xen_to_C_XMotionEvent(ptr))->state = Xen_ulong_to_C_ulong(val);
  else if (Xen_is_XButtonEvent(ptr)) (Xen_to_C_XButtonEvent(ptr))->state = Xen_ulong_to_C_ulong(val);
  else if (Xen_is_XKeyEvent(ptr)) (Xen_to_C_XKeyEvent(ptr))->state = Xen_ulong_to_C_ulong(val);
  else XM_set_field_assert_type(0, ptr, 1, "state", "a struct with a state field");
  return(val);
}

static Xen gxm_y_root(Xen ptr)
{
  if (Xen_is_XCrossingEvent(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XCrossingEvent(ptr))->y_root)));
  if (Xen_is_XMotionEvent(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XMotionEvent(ptr))->y_root)));
  if (Xen_is_XButtonEvent(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XButtonEvent(ptr))->y_root)));
  if (Xen_is_XKeyEvent(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XKeyEvent(ptr))->y_root)));
  XM_field_assert_type(0, ptr, 1, "y_root", "a struct with a y_root field");
  return(Xen_false);
}

static Xen gxm_set_y_root(Xen ptr, Xen val)
{
  XM_set_field_assert_type(Xen_is_integer(val), val, 2, "y_root", "an integer");
  if (Xen_is_XCrossingEvent(ptr)) (Xen_to_C_XCrossingEvent(ptr))->y_root = Xen_integer_to_C_int(val);
  else if (Xen_is_XMotionEvent(ptr)) (Xen_to_C_XMotionEvent(ptr))->y_root = Xen_integer_to_C_int(val);
  else if (Xen_is_XButtonEvent(ptr)) (Xen_to_C_XButtonEvent(ptr))->y_root = Xen_integer_to_C_int(val);
  else if (Xen_is_XKeyEvent(ptr)) (Xen_to_C_XKeyEvent(ptr))->y_root = Xen_integer_to_C_int(val);
  else XM_set_field_assert_type(0, ptr, 1, "y_root", "a struct with a y_root field");
  return(val);
}

static Xen gxm_x_root(Xen ptr)
{
  if (Xen_is_XCrossingEvent(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XCrossingEvent(ptr))->x_root)));
  if (Xen_is_XMotionEvent(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XMotionEvent(ptr))->x_root)));
  if (Xen_is_XButtonEvent(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XButtonEvent(ptr))->x_root)));
  if (Xen_is_XKeyEvent(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XKeyEvent(ptr))->x_root)));
  XM_field_assert_type(0, ptr, 1, "x_root", "a struct with an x_root field");
  return(Xen_false);
}

static Xen gxm_set_x_root(Xen ptr, Xen val)
{
  XM_set_field_assert_type(Xen_is_integer(val), val, 2, "x_root", "an integer");
  if (Xen_is_XCrossingEvent(ptr)) (Xen_to_C_XCrossingEvent(ptr))->x_root = Xen_integer_to_C_int(val);
  else if (Xen_is_XMotionEvent(ptr)) (Xen_to_C_XMotionEvent(ptr))->x_root = Xen_integer_to_C_int(val);
  else if (Xen_is_XButtonEvent(ptr)) (Xen_to_C_XButtonEvent(ptr))->x_root = Xen_integer_to_C_int(val);
  else if (Xen_is_XKeyEvent(ptr)) (Xen_to_C_XKeyEvent(ptr))->x_root = Xen_integer_to_C_int(val);
  else XM_set_field_assert_type(0, ptr, 1, "x_root", "a struct with a x_root field");
  return(val);
}

static Xen gxm_set_x(Xen ptr, Xen val)
{
  XM_set_field_assert_type(Xen_is_integer(val), val, 2, "x", "an integer");
  if (Xen_is_XRectangle(ptr)) (Xen_to_C_XRectangle(ptr))->x = (short)Xen_integer_to_C_int(val);
  else if (Xen_is_XPoint(ptr)) (Xen_to_C_XPoint(ptr))->x = (short)Xen_integer_to_C_int(val);
  else if (Xen_is_XArc(ptr)) (Xen_to_C_XArc(ptr))->x = (short)Xen_integer_to_C_int(val);
  else if (Xen_is_XConfigureRequestEvent(ptr)) (Xen_to_C_XConfigureRequestEvent(ptr))->x = Xen_integer_to_C_int(val);
  else if (Xen_is_XGravityEvent(ptr)) (Xen_to_C_XGravityEvent(ptr))->x = Xen_integer_to_C_int(val);
  else if (Xen_is_XConfigureEvent(ptr)) (Xen_to_C_XConfigureEvent(ptr))->x = Xen_integer_to_C_int(val);
  else if (Xen_is_XReparentEvent(ptr)) (Xen_to_C_XReparentEvent(ptr))->x = Xen_integer_to_C_int(val);
  else if (Xen_is_XCreateWindowEvent(ptr)) (Xen_to_C_XCreateWindowEvent(ptr))->x = Xen_integer_to_C_int(val);
  else if (Xen_is_XGraphicsExposeEvent(ptr)) (Xen_to_C_XGraphicsExposeEvent(ptr))->x = Xen_integer_to_C_int(val);
  else if (Xen_is_XExposeEvent(ptr)) (Xen_to_C_XExposeEvent(ptr))->x = Xen_integer_to_C_int(val);
  else if (Xen_is_XCrossingEvent(ptr)) (Xen_to_C_XCrossingEvent(ptr))->x = Xen_integer_to_C_int(val);
  else if (Xen_is_XMotionEvent(ptr)) (Xen_to_C_XMotionEvent(ptr))->x = Xen_integer_to_C_int(val);
  else if (Xen_is_XButtonEvent(ptr)) (Xen_to_C_XButtonEvent(ptr))->x = Xen_integer_to_C_int(val);
  else if (Xen_is_XKeyEvent(ptr)) (Xen_to_C_XKeyEvent(ptr))->x = Xen_integer_to_C_int(val);
  else XM_set_field_assert_type(0, ptr, 1, "x", "a struct with an x field");
  return(val);
}

static Xen gxm_set_y(Xen ptr, Xen val)
{
  XM_set_field_assert_type(Xen_is_integer(val), val, 2, "y", "an integer");
  if (Xen_is_XRectangle(ptr)) (Xen_to_C_XRectangle(ptr))->y = (short)Xen_integer_to_C_int(val);
  else if (Xen_is_XPoint(ptr)) (Xen_to_C_XPoint(ptr))->y = (short)Xen_integer_to_C_int(val);
  else if (Xen_is_XArc(ptr)) (Xen_to_C_XArc(ptr))->y = (short)Xen_integer_to_C_int(val);
  else if (Xen_is_XConfigureRequestEvent(ptr)) (Xen_to_C_XConfigureRequestEvent(ptr))->y = Xen_integer_to_C_int(val);
  else if (Xen_is_XGravityEvent(ptr)) (Xen_to_C_XGravityEvent(ptr))->y = Xen_integer_to_C_int(val);
  else if (Xen_is_XConfigureEvent(ptr)) (Xen_to_C_XConfigureEvent(ptr))->y = Xen_integer_to_C_int(val);
  else if (Xen_is_XReparentEvent(ptr)) (Xen_to_C_XReparentEvent(ptr))->y = Xen_integer_to_C_int(val);
  else if (Xen_is_XCreateWindowEvent(ptr)) (Xen_to_C_XCreateWindowEvent(ptr))->y = Xen_integer_to_C_int(val);
  else if (Xen_is_XGraphicsExposeEvent(ptr)) (Xen_to_C_XGraphicsExposeEvent(ptr))->y = Xen_integer_to_C_int(val);
  else if (Xen_is_XExposeEvent(ptr)) (Xen_to_C_XExposeEvent(ptr))->y = Xen_integer_to_C_int(val);
  else if (Xen_is_XCrossingEvent(ptr)) (Xen_to_C_XCrossingEvent(ptr))->y = Xen_integer_to_C_int(val);
  else if (Xen_is_XMotionEvent(ptr)) (Xen_to_C_XMotionEvent(ptr))->y = Xen_integer_to_C_int(val);
  else if (Xen_is_XButtonEvent(ptr)) (Xen_to_C_XButtonEvent(ptr))->y = Xen_integer_to_C_int(val);
  else if (Xen_is_XKeyEvent(ptr)) (Xen_to_C_XKeyEvent(ptr))->y = Xen_integer_to_C_int(val);
  else XM_set_field_assert_type(0, ptr, 1, "y", "a struct with a y field");
  return(val);
}

static Xen gxm_y(Xen ptr)
{
  if (Xen_is_XRectangle(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XRectangle(ptr))->y)));
  if (Xen_is_XPoint(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XPoint(ptr))->y)));
  if (Xen_is_XArc(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XArc(ptr))->y)));
  if (Xen_is_XWindowChanges(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XWindowChanges(ptr))->y)));
  if (Xen_is_XWindowAttributes(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XWindowAttributes(ptr))->y)));
  if (Xen_is_XmDropProcCallbackStruct(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XmDropProcCallbackStruct(ptr))->y)));
  if (Xen_is_XmDragProcCallbackStruct(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XmDragProcCallbackStruct(ptr))->y)));
  if (Xen_is_XmDropStartCallbackStruct(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XmDropStartCallbackStruct(ptr))->y)));
  if (Xen_is_XmDragMotionCallbackStruct(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XmDragMotionCallbackStruct(ptr))->y)));
  if (Xen_is_XmDropSiteEnterCallbackStruct(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XmDropSiteEnterCallbackStruct(ptr))->y)));
  if (Xen_is_XmTopLevelEnterCallbackStruct(ptr)) return(C_to_Xen_Position((Position)((Xen_to_C_XmTopLevelEnterCallbackStruct(ptr))->y)));
  if (Xen_is_XConfigureRequestEvent(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XConfigureRequestEvent(ptr))->y)));
  if (Xen_is_XGravityEvent(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XGravityEvent(ptr))->y)));
  if (Xen_is_XConfigureEvent(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XConfigureEvent(ptr))->y)));
  if (Xen_is_XReparentEvent(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XReparentEvent(ptr))->y)));
  if (Xen_is_XCreateWindowEvent(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XCreateWindowEvent(ptr))->y)));
  if (Xen_is_XGraphicsExposeEvent(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XGraphicsExposeEvent(ptr))->y)));
  if (Xen_is_XExposeEvent(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XExposeEvent(ptr))->y)));
  if (Xen_is_XCrossingEvent(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XCrossingEvent(ptr))->y)));
  if (Xen_is_XMotionEvent(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XMotionEvent(ptr))->y)));
  if (Xen_is_XButtonEvent(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XButtonEvent(ptr))->y)));
  if (Xen_is_XKeyEvent(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XKeyEvent(ptr))->y)));
  XM_field_assert_type(0, ptr, 1, "y", "a struct with a y field");
  return(Xen_false);
}

static Xen gxm_x(Xen ptr)
{
  if (Xen_is_XRectangle(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XRectangle(ptr))->x)));
  if (Xen_is_XPoint(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XPoint(ptr))->x)));
  if (Xen_is_XArc(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XArc(ptr))->x)));
  if (Xen_is_XWindowChanges(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XWindowChanges(ptr))->x)));
  if (Xen_is_XWindowAttributes(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XWindowAttributes(ptr))->x)));
  if (Xen_is_XmDropProcCallbackStruct(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XmDropProcCallbackStruct(ptr))->x)));
  if (Xen_is_XmDragProcCallbackStruct(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XmDragProcCallbackStruct(ptr))->x)));
  if (Xen_is_XmDropStartCallbackStruct(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XmDropStartCallbackStruct(ptr))->x)));
  if (Xen_is_XmDragMotionCallbackStruct(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XmDragMotionCallbackStruct(ptr))->x)));
  if (Xen_is_XmDropSiteEnterCallbackStruct(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XmDropSiteEnterCallbackStruct(ptr))->x)));
  if (Xen_is_XmTopLevelEnterCallbackStruct(ptr)) return(C_to_Xen_Position((Position)((Xen_to_C_XmTopLevelEnterCallbackStruct(ptr))->x)));
  if (Xen_is_XConfigureRequestEvent(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XConfigureRequestEvent(ptr))->x)));
  if (Xen_is_XGravityEvent(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XGravityEvent(ptr))->x)));
  if (Xen_is_XConfigureEvent(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XConfigureEvent(ptr))->x)));
  if (Xen_is_XReparentEvent(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XReparentEvent(ptr))->x)));
  if (Xen_is_XCreateWindowEvent(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XCreateWindowEvent(ptr))->x)));
  if (Xen_is_XGraphicsExposeEvent(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XGraphicsExposeEvent(ptr))->x)));
  if (Xen_is_XExposeEvent(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XExposeEvent(ptr))->x)));
  if (Xen_is_XCrossingEvent(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XCrossingEvent(ptr))->x)));
  if (Xen_is_XMotionEvent(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XMotionEvent(ptr))->x)));
  if (Xen_is_XButtonEvent(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XButtonEvent(ptr))->x)));
  if (Xen_is_XKeyEvent(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XKeyEvent(ptr))->x)));
  XM_field_assert_type(0, ptr, 1, "x", "a struct with an x field");
  return(Xen_false);
}

static Xen gxm_time(Xen ptr)
{
  if (Xen_is_XmDestinationCallbackStruct(ptr)) return(C_to_Xen_Time((Time)((Xen_to_C_XmDestinationCallbackStruct(ptr))->time)));
  if (Xen_is_XSelectionEvent(ptr)) return(C_to_Xen_Time((Time)((Xen_to_C_XSelectionEvent(ptr))->time)));
  if (Xen_is_XSelectionRequestEvent(ptr)) return(C_to_Xen_Time((Time)((Xen_to_C_XSelectionRequestEvent(ptr))->time)));
  if (Xen_is_XSelectionClearEvent(ptr)) return(C_to_Xen_Time((Time)((Xen_to_C_XSelectionClearEvent(ptr))->time)));
  if (Xen_is_XPropertyEvent(ptr)) return(C_to_Xen_Time((Time)((Xen_to_C_XPropertyEvent(ptr))->time)));
  if (Xen_is_XCrossingEvent(ptr)) return(C_to_Xen_Time((Time)((Xen_to_C_XCrossingEvent(ptr))->time)));
  if (Xen_is_XMotionEvent(ptr)) return(C_to_Xen_Time((Time)((Xen_to_C_XMotionEvent(ptr))->time)));
  if (Xen_is_XButtonEvent(ptr)) return(C_to_Xen_Time((Time)((Xen_to_C_XButtonEvent(ptr))->time)));
  if (Xen_is_XKeyEvent(ptr)) return(C_to_Xen_Time((Time)((Xen_to_C_XKeyEvent(ptr))->time)));
  XM_field_assert_type(0, ptr, 1, "time", "a struct with a time field");
  return(Xen_false);
}

static Xen gxm_set_time(Xen ptr, Xen val)
{
  Time tm;
  XM_set_field_assert_type(Xen_is_Time(val), val, 2, "time", "Time");
  tm = Xen_to_C_Time(val);
  if (Xen_is_XSelectionEvent(ptr)) (Xen_to_C_XSelectionEvent(ptr))->time = tm;
  else if (Xen_is_XSelectionRequestEvent(ptr)) (Xen_to_C_XSelectionRequestEvent(ptr))->time = tm;
  else if (Xen_is_XSelectionClearEvent(ptr)) (Xen_to_C_XSelectionClearEvent(ptr))->time = tm;
  else if (Xen_is_XPropertyEvent(ptr)) (Xen_to_C_XPropertyEvent(ptr))->time = tm;
  else if (Xen_is_XCrossingEvent(ptr)) (Xen_to_C_XCrossingEvent(ptr))->time = tm;
  else if (Xen_is_XMotionEvent(ptr)) (Xen_to_C_XMotionEvent(ptr))->time = tm;
  else if (Xen_is_XButtonEvent(ptr)) (Xen_to_C_XButtonEvent(ptr))->time = tm;
  else if (Xen_is_XKeyEvent(ptr)) (Xen_to_C_XKeyEvent(ptr))->time = tm;
  else XM_set_field_assert_type(0, ptr, 1, "time", "a struct with a time field");
  return(val);
}

static Xen gxm_subwindow(Xen ptr)
{
  if (Xen_is_XCrossingEvent(ptr)) return(C_to_Xen_Window((Window)((Xen_to_C_XCrossingEvent(ptr))->subwindow)));
  if (Xen_is_XMotionEvent(ptr)) return(C_to_Xen_Window((Window)((Xen_to_C_XMotionEvent(ptr))->subwindow)));
  if (Xen_is_XButtonEvent(ptr)) return(C_to_Xen_Window((Window)((Xen_to_C_XButtonEvent(ptr))->subwindow)));
  if (Xen_is_XKeyEvent(ptr)) return(C_to_Xen_Window((Window)((Xen_to_C_XKeyEvent(ptr))->subwindow)));
  XM_field_assert_type(0, ptr, 1, "subwindow", "a struct with a subwindow field");
  return(Xen_false);
}

static Xen gxm_set_subwindow(Xen ptr, Xen val)
{
  XM_set_field_assert_type(Xen_is_Window(val), val, 2, "subwindow", "a Window");
  if (Xen_is_XCrossingEvent(ptr)) (Xen_to_C_XCrossingEvent(ptr))->subwindow = Xen_to_C_Window(val);
  else if (Xen_is_XMotionEvent(ptr)) (Xen_to_C_XMotionEvent(ptr))->subwindow = Xen_to_C_Window(val);
  else if (Xen_is_XButtonEvent(ptr)) (Xen_to_C_XButtonEvent(ptr))->subwindow = Xen_to_C_Window(val);
  else if (Xen_is_XKeyEvent(ptr)) (Xen_to_C_XKeyEvent(ptr))->subwindow = Xen_to_C_Window(val);
  else XM_set_field_assert_type(0, ptr, 1, "subwindow", "a struct with a subwindow field");
  return(val);
}

static Xen gxm_window(Xen ptr)
{
  if (Xen_is_XmDrawnButtonCallbackStruct(ptr)) return(C_to_Xen_Window((Window)((Xen_to_C_XmDrawnButtonCallbackStruct(ptr))->window)));
  if (Xen_is_XmDrawingAreaCallbackStruct(ptr)) return(C_to_Xen_Window((Window)((Xen_to_C_XmDrawingAreaCallbackStruct(ptr))->window)));
  if (Xen_is_XmDropStartCallbackStruct(ptr)) return(C_to_Xen_Window((Window)((Xen_to_C_XmDropStartCallbackStruct(ptr))->window)));
  if (Xen_is_XmTopLevelEnterCallbackStruct(ptr)) return(C_to_Xen_Window((Window)((Xen_to_C_XmTopLevelEnterCallbackStruct(ptr))->window)));
  if (Xen_is_XmTopLevelLeaveCallbackStruct(ptr)) return(C_to_Xen_Window((Window)((Xen_to_C_XmTopLevelLeaveCallbackStruct(ptr))->window)));
  if (Xen_is_XEvent(ptr)) return(C_to_Xen_Window((Window)((Xen_to_C_XAnyEvent(ptr))->window)));
  XM_field_assert_type(0, ptr, 1, "window", "a struct with a window field");
  return(Xen_false);
}

static Xen gxm_set_window(Xen ptr, Xen val)
{
  XM_set_field_assert_type(Xen_is_Window(val), val, 2, "window", "a Window");
  XM_set_field_assert_type(Xen_is_XEvent(ptr), ptr, 1, "window", "XEvent");
  (Xen_to_C_XAnyEvent(ptr))->window = Xen_to_C_Window(val);
  return(val);
}

static Xen gxm_send_event(Xen ptr)
{
  if (Xen_is_XEvent(ptr)) return(C_bool_to_Xen_boolean((Bool)((Xen_to_C_XAnyEvent(ptr))->send_event)));
  XM_field_assert_type(0, ptr, 1, "send_event", "XEvent");
  return(Xen_false);
}

static Xen gxm_set_send_event(Xen ptr, Xen val)
{
  XM_set_field_assert_type(Xen_is_boolean(val), val, 2, "send_event", "a boolean");
  XM_set_field_assert_type(Xen_is_XEvent(ptr), ptr, 1, "send_event", "XEvent");
  (Xen_to_C_XAnyEvent(ptr))->send_event = (Bool)Xen_boolean_to_C_bool(val);
  return(val);
}

static Xen gxm_serial(Xen ptr)
{
  XM_field_assert_type(Xen_is_XEvent(ptr), ptr, 1, "serial", "XEvent");
  return(C_ulong_to_Xen_ulong((unsigned long)((Xen_to_C_XAnyEvent(ptr))->serial)));
}

static Xen gxm_set_serial(Xen ptr, Xen val)
{
  XM_set_field_assert_type(Xen_is_XEvent(ptr), ptr, 1, "serial", "XEvent");
  (Xen_to_C_XAnyEvent(ptr))->serial = Xen_ulong_to_C_ulong(val);
  return(val);
}

static Xen gxm_type(Xen ptr)
{
  if (Xen_is_XmSelectionCallbackStruct(ptr)) return(C_to_Xen_Atom((Atom)((Xen_to_C_XmSelectionCallbackStruct(ptr))->type)));
  if (Xen_is_XmConvertCallbackStruct(ptr)) return(C_to_Xen_Atom((Atom)((Xen_to_C_XmConvertCallbackStruct(ptr))->type)));
  if (Xen_is_XEvent(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XAnyEvent(ptr))->type)));
  XM_field_assert_type(0, ptr, 1, "type", "a struct with a type field");
  return(Xen_false);
}

static Xen gxm_set_type(Xen ptr, Xen val)
{
  XM_set_field_assert_type(Xen_is_XEvent(ptr), ptr, 1, "type", "XEvent");
  XM_set_field_assert_type(Xen_is_integer(val), val, 2, "type", "integer");
  (Xen_to_C_XAnyEvent(ptr))->type = Xen_integer_to_C_int(val);
  return(val);
}

static Xen gxm_root_input_mask(Xen ptr)
{
  XM_field_assert_type(Xen_is_Screen(ptr), ptr, 1, "root_input_mask", "Screen");
  return(C_int_to_Xen_integer((long)((Xen_to_C_Screen(ptr))->root_input_mask)));
}

static Xen gxm_save_unders(Xen ptr)
{
  XM_field_assert_type(Xen_is_Screen(ptr), ptr, 1, "save_unders", "Screen");
  return(C_bool_to_Xen_boolean((Bool)((Xen_to_C_Screen(ptr))->save_unders)));
}

static Xen gxm_backing_store(Xen ptr)
{
  if (Xen_is_XWindowAttributes(ptr)) return(C_bool_to_Xen_boolean((int)((Xen_to_C_XWindowAttributes(ptr))->backing_store)));
  if (Xen_is_XSetWindowAttributes(ptr)) return(C_bool_to_Xen_boolean((int)((Xen_to_C_XSetWindowAttributes(ptr))->backing_store)));
  if (Xen_is_Screen(ptr)) return(C_bool_to_Xen_boolean((int)((Xen_to_C_Screen(ptr))->backing_store)));
  XM_field_assert_type(0, ptr, 1, "backing_store", "a struct with a backing_store field");
  return(Xen_false);
}

static Xen gxm_set_backing_store(Xen ptr, Xen val)
{
  XM_set_field_assert_type(Xen_is_XSetWindowAttributes(ptr) || Xen_is_XWindowAttributes(ptr) || Xen_is_Screen(ptr), 
			   ptr, 1, "backing_store", "a struct with a backing_store field");
  XM_set_field_assert_type(Xen_is_boolean(val), val, 2, "backing_store", "a Boolean");
  if (Xen_is_XWindowAttributes(ptr))
      (Xen_to_C_XWindowAttributes(ptr))->backing_store = Xen_boolean_to_C_bool(val);
  else if (Xen_is_XSetWindowAttributes(ptr))
      (Xen_to_C_XSetWindowAttributes(ptr))->backing_store = Xen_boolean_to_C_bool(val);
  else if (Xen_is_Screen(ptr))
      (Xen_to_C_Screen(ptr))->backing_store = Xen_boolean_to_C_bool(val);
  return(val);
}

static Xen gxm_min_maps(Xen ptr)
{
  XM_field_assert_type(Xen_is_Screen(ptr), ptr, 1, "min_maps", "Screen");
  return(C_int_to_Xen_integer((int)((Xen_to_C_Screen(ptr))->min_maps)));
}

static Xen gxm_max_maps(Xen ptr)
{
  XM_field_assert_type(Xen_is_Screen(ptr), ptr, 1, "max_maps", "Screen");
  return(C_int_to_Xen_integer((int)((Xen_to_C_Screen(ptr))->max_maps)));
}

static Xen gxm_black_pixel(Xen ptr)
{
  XM_field_assert_type(Xen_is_Screen(ptr), ptr, 1, "black_pixel", "Screen");
  return(C_to_Xen_Pixel((Xen_to_C_Screen(ptr))->black_pixel));
}

static Xen gxm_white_pixel(Xen ptr)
{
  XM_field_assert_type(Xen_is_Screen(ptr), ptr, 1, "white_pixel", "Screen");
  return(C_to_Xen_Pixel((Xen_to_C_Screen(ptr))->white_pixel));
}

static Xen gxm_cmap(Xen ptr)
{
  XM_field_assert_type(Xen_is_Screen(ptr), ptr, 1, "cmap", "Screen");
  return(C_to_Xen_Colormap((Colormap)((Xen_to_C_Screen(ptr))->cmap)));
}

static Xen gxm_default_gc(Xen ptr)
{
  XM_field_assert_type(Xen_is_Screen(ptr), ptr, 1, "default_gc", "Screen");
  return(C_to_Xen_GC((GC)((Xen_to_C_Screen(ptr))->default_gc)));
}

static Xen gxm_root_visual(Xen ptr)
{
  XM_field_assert_type(Xen_is_Screen(ptr), ptr, 1, "root_visual", "Screen");
  return(C_to_Xen_Visual((Visual *)((Xen_to_C_Screen(ptr))->root_visual)));
}

static Xen gxm_root_depth(Xen ptr)
{
  XM_field_assert_type(Xen_is_Screen(ptr), ptr, 1, "root_depth", "Screen");
  return(C_int_to_Xen_integer((int)((Xen_to_C_Screen(ptr))->root_depth)));
}

static Xen gxm_depths(Xen ptr)
{
  Screen *scr;
  int len;
  Xen lst = Xen_empty_list;
  XM_field_assert_type(Xen_is_Screen(ptr), ptr, 1, "depths", "Screen");
  scr = Xen_to_C_Screen(ptr);
  len = scr->ndepths;
  if (len > 0)
    {
      Depth *dps;
      int i;
      dps = scr->depths;
      for (i = len - 1; i >= 0; i--)
	lst = Xen_cons(wrap_for_Xen("Depth", &(dps[i])), lst);
    }
  return(lst);
}

static Xen gxm_ndepths(Xen ptr)
{
  XM_field_assert_type(Xen_is_Screen(ptr), ptr, 1, "ndepths", "Screen");
  return(C_int_to_Xen_integer((int)((Xen_to_C_Screen(ptr))->ndepths)));
}

static Xen gxm_mheight(Xen ptr)
{
  XM_field_assert_type(Xen_is_Screen(ptr), ptr, 1, "mheight", "Screen");
  return(C_int_to_Xen_integer((int)((Xen_to_C_Screen(ptr))->mheight)));
}

static Xen gxm_mwidth(Xen ptr)
{
  XM_field_assert_type(Xen_is_Screen(ptr), ptr, 1, "mwidth", "Screen");
  return(C_int_to_Xen_integer((int)((Xen_to_C_Screen(ptr))->mwidth)));
}


static Xen gxm_set_height(Xen ptr, Xen val)
{
  if (Xen_is_XRectangle(ptr)) (Xen_to_C_XRectangle(ptr))->height = Xen_integer_to_C_int(val);
  else if (Xen_is_XArc(ptr)) (Xen_to_C_XArc(ptr))->height = Xen_integer_to_C_int(val);
  else if (Xen_is_XpmImage(ptr)) (Xen_to_C_XpmImage(ptr))->height = Xen_ulong_to_C_ulong(val);
  else if (Xen_is_XpmAttributes(ptr)) (Xen_to_C_XpmAttributes(ptr))->height = Xen_ulong_to_C_ulong(val);
  else if (Xen_is_XConfigureRequestEvent(ptr)) (Xen_to_C_XConfigureRequestEvent(ptr))->height = Xen_integer_to_C_int(val);
  else if (Xen_is_XResizeRequestEvent(ptr)) (Xen_to_C_XResizeRequestEvent(ptr))->height = Xen_integer_to_C_int(val);
  else if (Xen_is_XConfigureEvent(ptr)) (Xen_to_C_XConfigureEvent(ptr))->height = Xen_integer_to_C_int(val);
  else if (Xen_is_XCreateWindowEvent(ptr)) (Xen_to_C_XCreateWindowEvent(ptr))->height = Xen_integer_to_C_int(val);
  else if (Xen_is_XGraphicsExposeEvent(ptr)) (Xen_to_C_XGraphicsExposeEvent(ptr))->height = Xen_integer_to_C_int(val);
  else if (Xen_is_XExposeEvent(ptr)) (Xen_to_C_XExposeEvent(ptr))->height = Xen_integer_to_C_int(val);
  else XM_set_field_assert_type(0, ptr, 1, "height", "a struct with a height field");
  return(val);
}

static Xen gxm_set_width(Xen ptr, Xen val)
{
  if (Xen_is_XRectangle(ptr)) (Xen_to_C_XRectangle(ptr))->width = Xen_integer_to_C_int(val);
  else if (Xen_is_XArc(ptr)) (Xen_to_C_XArc(ptr))->width = Xen_integer_to_C_int(val);
  else if (Xen_is_XpmImage(ptr)) (Xen_to_C_XpmImage(ptr))->width = Xen_ulong_to_C_ulong(val);
  else if (Xen_is_XpmAttributes(ptr)) (Xen_to_C_XpmAttributes(ptr))->width = Xen_ulong_to_C_ulong(val);
  else if (Xen_is_XConfigureRequestEvent(ptr)) (Xen_to_C_XConfigureRequestEvent(ptr))->width = Xen_integer_to_C_int(val);
  else if (Xen_is_XResizeRequestEvent(ptr)) (Xen_to_C_XResizeRequestEvent(ptr))->width = Xen_integer_to_C_int(val);
  else if (Xen_is_XConfigureEvent(ptr)) (Xen_to_C_XConfigureEvent(ptr))->width = Xen_integer_to_C_int(val);
  else if (Xen_is_XCreateWindowEvent(ptr)) (Xen_to_C_XCreateWindowEvent(ptr))->width = Xen_integer_to_C_int(val);
  else if (Xen_is_XGraphicsExposeEvent(ptr)) (Xen_to_C_XGraphicsExposeEvent(ptr))->width = Xen_integer_to_C_int(val);
  else if (Xen_is_XExposeEvent(ptr)) (Xen_to_C_XExposeEvent(ptr))->width = Xen_integer_to_C_int(val);
  else XM_set_field_assert_type(0, ptr, 1, "width", "a struct with a width field");
  return(val);
}

static Xen gxm_height(Xen ptr)
{
  if (Xen_is_XRectangle(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XRectangle(ptr))->height)));
  if (Xen_is_XArc(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XArc(ptr))->height)));
  if (Xen_is_XWindowChanges(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XWindowChanges(ptr))->height)));
  if (Xen_is_XImage(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XImage(ptr))->height)));
  if (Xen_is_XWindowAttributes(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XWindowAttributes(ptr))->height)));
  if (Xen_is_XConfigureRequestEvent(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XConfigureRequestEvent(ptr))->height)));
  if (Xen_is_XResizeRequestEvent(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XResizeRequestEvent(ptr))->height)));
  if (Xen_is_XConfigureEvent(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XConfigureEvent(ptr))->height)));
  if (Xen_is_XCreateWindowEvent(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XCreateWindowEvent(ptr))->height)));
  if (Xen_is_XGraphicsExposeEvent(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XGraphicsExposeEvent(ptr))->height)));
  if (Xen_is_XExposeEvent(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XExposeEvent(ptr))->height)));
  if (Xen_is_Screen(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_Screen(ptr))->height)));
  if (Xen_is_XpmImage(ptr)) return(C_ulong_to_Xen_ulong((unsigned long)((Xen_to_C_XpmImage(ptr))->height)));
  if (Xen_is_XpmAttributes(ptr)) return(C_ulong_to_Xen_ulong((unsigned long)((Xen_to_C_XpmAttributes(ptr))->height)));
  XM_field_assert_type(0, ptr, 1, "height", "a struct with a height field");
  return(Xen_false);
}

static Xen gxm_width(Xen ptr)
{
  if (Xen_is_XRectangle(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XRectangle(ptr))->width)));
  if (Xen_is_XArc(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XArc(ptr))->width)));
  if (Xen_is_XWindowChanges(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XWindowChanges(ptr))->width)));
  if (Xen_is_XImage(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XImage(ptr))->width)));
  if (Xen_is_XWindowAttributes(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XWindowAttributes(ptr))->width)));
  if (Xen_is_XCharStruct(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XCharStruct(ptr))->width)));
  if (Xen_is_XConfigureRequestEvent(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XConfigureRequestEvent(ptr))->width)));
  if (Xen_is_XResizeRequestEvent(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XResizeRequestEvent(ptr))->width)));
  if (Xen_is_XConfigureEvent(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XConfigureEvent(ptr))->width)));
  if (Xen_is_XCreateWindowEvent(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XCreateWindowEvent(ptr))->width)));
  if (Xen_is_XGraphicsExposeEvent(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XGraphicsExposeEvent(ptr))->width)));
  if (Xen_is_XExposeEvent(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XExposeEvent(ptr))->width)));
  if (Xen_is_Screen(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_Screen(ptr))->width)));
  if (Xen_is_XpmImage(ptr)) return(C_ulong_to_Xen_ulong((unsigned long)((Xen_to_C_XpmImage(ptr))->width)));
  if (Xen_is_XpmAttributes(ptr)) return(C_ulong_to_Xen_ulong((unsigned long)((Xen_to_C_XpmAttributes(ptr))->width)));
  XM_field_assert_type(0, ptr, 1, "width", "a struct with a width field");
  return(Xen_false);
}

static Xen gxm_root(Xen ptr)
{
  if (Xen_is_XWindowAttributes(ptr)) return(C_to_Xen_Window((Window)((Xen_to_C_XWindowAttributes(ptr))->root)));
  if (Xen_is_XCrossingEvent(ptr)) return(C_to_Xen_Window((Window)((Xen_to_C_XCrossingEvent(ptr))->root)));
  if (Xen_is_XMotionEvent(ptr)) return(C_to_Xen_Window((Window)((Xen_to_C_XMotionEvent(ptr))->root)));
  if (Xen_is_XButtonEvent(ptr)) return(C_to_Xen_Window((Window)((Xen_to_C_XButtonEvent(ptr))->root)));
  if (Xen_is_XKeyEvent(ptr)) return(C_to_Xen_Window((Window)((Xen_to_C_XKeyEvent(ptr))->root)));
  if (Xen_is_Screen(ptr)) return(C_to_Xen_Window((Window)((Xen_to_C_Screen(ptr))->root)));
  XM_field_assert_type(0, ptr, 1, "root", "a struct with a root field");
  return(Xen_false);
}

static Xen gxm_set_root(Xen ptr, Xen val)
{
  Window w;
  XM_set_field_assert_type(Xen_is_Window(val), val, 2, "root", "a Window");
  w = Xen_to_C_Window(val);
  if (Xen_is_XWindowAttributes(ptr)) (Xen_to_C_XWindowAttributes(ptr))->root = w;
  else if (Xen_is_XCrossingEvent(ptr)) (Xen_to_C_XCrossingEvent(ptr))->root = w;
  else if (Xen_is_XMotionEvent(ptr)) (Xen_to_C_XMotionEvent(ptr))->root = w;
  else if (Xen_is_XButtonEvent(ptr)) (Xen_to_C_XButtonEvent(ptr))->root = w;
  else if (Xen_is_XKeyEvent(ptr)) (Xen_to_C_XKeyEvent(ptr))->root = w;
  else if (Xen_is_Screen(ptr)) (Xen_to_C_Screen(ptr))->root = w;
  else XM_set_field_assert_type(0, ptr, 1, "root", "a struct with a root field");
  return(val);
}

static Xen gxm_display(Xen ptr)
{
  XM_field_assert_type(Xen_is_XEvent(ptr) || Xen_is_Screen(ptr), ptr, 1, "display", "XEvent or Screen");
  if (Xen_is_XEvent(ptr)) return(C_to_Xen_Display((Display *)((Xen_to_C_XAnyEvent(ptr))->display)));
  return(C_to_Xen_Display((Display *)((Xen_to_C_Screen(ptr))->display)));
}

static Xen gxm_set_display(Xen ptr, Xen val)
{
  XM_set_field_assert_type(Xen_is_Display(val), val, 2, "display", "a Display*");
  XM_set_field_assert_type(Xen_is_XEvent(ptr), ptr, 1, "display", "XEvent");
  (Xen_to_C_XAnyEvent(ptr))->display = Xen_to_C_Display(val);
  return(val);
}

static Xen gxm_function(Xen ptr)
{
  if (Xen_is_XGCValues(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XGCValues(ptr))->function)));
  XM_field_assert_type(0, ptr, 1, "function", "XGCValues");
  return(Xen_false);
}

static Xen gxm_set_function(Xen ptr, Xen val)
{
  XM_set_field_assert_type(Xen_is_integer(val), val, 2, "function", "an integer");
  XM_set_field_assert_type(Xen_is_XGCValues(ptr), ptr, 1, "function", "XGCValues");
  (Xen_to_C_XGCValues(ptr))->function = Xen_integer_to_C_int(val);
  return(val);
}

static Xen gxm_plane_mask(Xen ptr)
{
  XM_field_assert_type(Xen_is_XGCValues(ptr), ptr, 1, "plane_mask", "XGCValues");
  return(C_ulong_to_Xen_ulong((unsigned long)((Xen_to_C_XGCValues(ptr))->plane_mask)));
}

static Xen gxm_set_plane_mask(Xen ptr, Xen val)
{
  XM_set_field_assert_type(Xen_is_XGCValues(ptr), ptr, 1, "plane_mask", "XGCValues");
  (Xen_to_C_XGCValues(ptr))->plane_mask = Xen_ulong_to_C_ulong(val);
  return(val);
}

static Xen gxm_foreground(Xen ptr)
{
  if (Xen_is_XGCValues(ptr)) return(C_to_Xen_Pixel((Pixel)((Xen_to_C_XGCValues(ptr))->foreground)));
  XM_field_assert_type(0, ptr, 1, "foreground", "a struct with a foreground field");
  return(Xen_false);
}

static Xen gxm_set_foreground(Xen ptr, Xen val)
{
  XM_set_field_assert_type(Xen_is_Pixel(val), val, 2, "foreground", "a Pixel");  
  XM_field_assert_type(Xen_is_XGCValues(ptr), ptr, 1, "foreground", "XGCValues");
  (Xen_to_C_XGCValues(ptr))->foreground = Xen_to_C_Pixel(val);
  return(val);
}

static Xen gxm_background(Xen ptr)
{
  if (Xen_is_XGCValues(ptr)) return(C_to_Xen_Pixel((Pixel)((Xen_to_C_XGCValues(ptr))->background)));
  XM_field_assert_type(0, ptr, 1, "background", "a struct with a background field");
  return(Xen_false);
}

static Xen gxm_set_background(Xen ptr, Xen val)
{
  XM_set_field_assert_type(Xen_is_Pixel(val), val, 2, "background", "a Pixel");  
  XM_set_field_assert_type(Xen_is_XGCValues(ptr), ptr, 1, "background", "XGCValues");
  (Xen_to_C_XGCValues(ptr))->background = Xen_to_C_Pixel(val);
  return(val);
}

static Xen gxm_line_width(Xen ptr)
{
  if (Xen_is_XGCValues(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XGCValues(ptr))->line_width)));
  XM_field_assert_type(0, ptr, 1, "line_width", "XGCValues");
  return(Xen_false);
}

static Xen gxm_set_line_width(Xen ptr, Xen val)
{
  XM_set_field_assert_type(Xen_is_integer(val), val, 2, "line_width", "an integer");
  XM_set_field_assert_type(Xen_is_XGCValues(ptr), ptr, 1, "line_width", "XGCValues");
  (Xen_to_C_XGCValues(ptr))->line_width = Xen_integer_to_C_int(val);
  return(val);
}

static Xen gxm_line_style(Xen ptr)
{
  if (Xen_is_XGCValues(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XGCValues(ptr))->line_style)));
  XM_field_assert_type(0, ptr, 1, "line_style", "XGCValues");
  return(Xen_false);
}

static Xen gxm_set_line_style(Xen ptr, Xen val)
{
  XM_set_field_assert_type(Xen_is_integer(val), val, 2, "line_style", "an integer");
  XM_set_field_assert_type(Xen_is_XGCValues(ptr), ptr, 1, "line_style", "XGCValues");
  (Xen_to_C_XGCValues(ptr))->line_style = Xen_integer_to_C_int(val);
  return(val);
}

static Xen gxm_cap_style(Xen ptr)
{
  if (Xen_is_XGCValues(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XGCValues(ptr))->cap_style)));
  XM_field_assert_type(0, ptr, 1, "cap_style", "XGCValues");
  return(Xen_false);
}

static Xen gxm_set_cap_style(Xen ptr, Xen val)
{
  XM_set_field_assert_type(Xen_is_integer(val), val, 2, "cap_style", "an integer");
  XM_set_field_assert_type(Xen_is_XGCValues(ptr), ptr, 1, "cap_style", "XGCValues");
  (Xen_to_C_XGCValues(ptr))->cap_style = Xen_integer_to_C_int(val);
  return(val);
}

static Xen gxm_join_style(Xen ptr)
{
  if (Xen_is_XGCValues(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XGCValues(ptr))->join_style)));
  XM_field_assert_type(0, ptr, 1, "join_style", "XGCValues");
  return(Xen_false);
}

static Xen gxm_set_join_style(Xen ptr, Xen val)
{
  XM_set_field_assert_type(Xen_is_integer(val), val, 2, "join_style", "an integer");
  XM_set_field_assert_type(Xen_is_XGCValues(ptr), ptr, 1, "join_style", "XGCValues");
  (Xen_to_C_XGCValues(ptr))->join_style = Xen_integer_to_C_int(val);
  return(val);
}

static Xen gxm_fill_style(Xen ptr)
{
  XM_field_assert_type(Xen_is_XGCValues(ptr), ptr, 1, "fill_style", "XGCValues");
  return(C_int_to_Xen_integer((int)((Xen_to_C_XGCValues(ptr))->fill_style)));
}

static Xen gxm_set_fill_style(Xen ptr, Xen val)
{
  XM_set_field_assert_type(Xen_is_integer(val), val, 2, "fill_style", "an integer");
  XM_set_field_assert_type(Xen_is_XGCValues(ptr), ptr, 1, "fill_style", "XGCValues");
  (Xen_to_C_XGCValues(ptr))->fill_style = Xen_integer_to_C_int(val);
  return(val);
}

static Xen gxm_fill_rule(Xen ptr)
{
  XM_field_assert_type(Xen_is_XGCValues(ptr), ptr, 1, "fill_rule", "XGCValues");
  return(C_int_to_Xen_integer((int)((Xen_to_C_XGCValues(ptr))->fill_rule)));
}

static Xen gxm_set_fill_rule(Xen ptr, Xen val)
{
  XM_set_field_assert_type(Xen_is_integer(val), val, 2, "fill_rule", "an integer");
  XM_set_field_assert_type(Xen_is_XGCValues(ptr), ptr, 1, "fill_rule", "XGCValues");
  (Xen_to_C_XGCValues(ptr))->fill_rule = Xen_integer_to_C_int(val);
  return(val);
}

static Xen gxm_arc_mode(Xen ptr)
{
  XM_field_assert_type(Xen_is_XGCValues(ptr), ptr, 1, "arc_mode", "XGCValues");
  return(C_int_to_Xen_integer((int)((Xen_to_C_XGCValues(ptr))->arc_mode)));
}

static Xen gxm_set_arc_mode(Xen ptr, Xen val)
{
  XM_set_field_assert_type(Xen_is_integer(val), val, 2, "arc_mode", "an integer");
  XM_set_field_assert_type(Xen_is_XGCValues(ptr), ptr, 1, "arc_mode", "XGCValues");
  (Xen_to_C_XGCValues(ptr))->arc_mode = Xen_integer_to_C_int(val);
  return(val);
}

static Xen gxm_tile(Xen ptr)
{
  if (Xen_is_XGCValues(ptr)) return(C_to_Xen_Pixmap((Pixmap)((Xen_to_C_XGCValues(ptr))->tile)));
  XM_field_assert_type(0, ptr, 1, "tile", "XGCValues");
  return(Xen_false);
}

static Xen gxm_set_tile(Xen ptr, Xen val)
{
  XM_set_field_assert_type(Xen_is_Pixmap(val), val, 2, "tile", "a Pixmap");
  XM_set_field_assert_type(Xen_is_XGCValues(ptr), ptr, 1, "tile", "XGCValues");
  (Xen_to_C_XGCValues(ptr))->tile = Xen_to_C_Pixmap(val);
  return(val);
}

static Xen gxm_stipple(Xen ptr)
{
  if (Xen_is_XGCValues(ptr)) return(C_to_Xen_Pixmap((Pixmap)((Xen_to_C_XGCValues(ptr))->stipple)));
  XM_field_assert_type(0, ptr, 1, "stipple", "XGCValues");
  return(Xen_false);
}

static Xen gxm_set_stipple(Xen ptr, Xen val)
{
  XM_set_field_assert_type(Xen_is_Pixmap(val), val, 2, "stipple", "a Pixmap");
  XM_set_field_assert_type(Xen_is_XGCValues(ptr), ptr, 1, "stipple", "XGCValues");
  (Xen_to_C_XGCValues(ptr))->stipple = Xen_to_C_Pixmap(val);
  return(val);
}

static Xen gxm_ts_x_origin(Xen ptr)
{
  if (Xen_is_XGCValues(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XGCValues(ptr))->ts_x_origin)));
  XM_field_assert_type(0, ptr, 1, "ts_x_origin", "XGCValues");
  return(Xen_false);
}

static Xen gxm_set_ts_x_origin(Xen ptr, Xen val)
{
  XM_set_field_assert_type(Xen_is_integer(val), val, 2, "ts_x_origin", "an integer");
  XM_set_field_assert_type(Xen_is_XGCValues(ptr), ptr, 1, "ts_x_origin", "XGCValues");
  (Xen_to_C_XGCValues(ptr))->ts_x_origin = Xen_integer_to_C_int(val);
  return(val);
}

static Xen gxm_ts_y_origin(Xen ptr)
{
  if (Xen_is_XGCValues(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XGCValues(ptr))->ts_y_origin)));
  XM_field_assert_type(0, ptr, 1, "ts_y_origin", "XGCValues");
  return(Xen_false);
}

static Xen gxm_set_ts_y_origin(Xen ptr, Xen val)
{
  XM_set_field_assert_type(Xen_is_integer(val), val, 2, "ts_y_origin", "an integer");
  XM_set_field_assert_type(Xen_is_XGCValues(ptr), ptr, 1, "ts_y_origin", "XGCValues");
  (Xen_to_C_XGCValues(ptr))->ts_y_origin = Xen_integer_to_C_int(val);
  return(val);
}

static Xen gxm_font(Xen ptr)
{
  if (Xen_is_XGCValues(ptr)) return(C_to_Xen_Font((Font)((Xen_to_C_XGCValues(ptr))->font)));
  if (Xen_is_XTextItem(ptr)) return(C_to_Xen_Font((Font)((Xen_to_C_XTextItem(ptr))->font)));
  XM_field_assert_type(0, ptr, 1, "font", "a struct with a font field");
  return(Xen_false);
}

static Xen gxm_set_font(Xen ptr, Xen val)
{
  XM_set_field_assert_type(Xen_is_Font(val), val, 2, "font", "a font");
  if (Xen_is_XGCValues(ptr)) (Xen_to_C_XGCValues(ptr))->font = Xen_to_C_Font(val);
  else if (Xen_is_XTextItem(ptr)) (Xen_to_C_XTextItem(ptr))->font = Xen_to_C_Font(val);
  else XM_set_field_assert_type(0, ptr, 1, "font", "a struct with a font field");
  return(val);
}

static Xen gxm_subwindow_mode(Xen ptr)
{
  if (Xen_is_XGCValues(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XGCValues(ptr))->subwindow_mode)));
  XM_field_assert_type(0, ptr, 1, "subwindow_mode", "XGCValues");
  return(Xen_false);
}

static Xen gxm_set_subwindow_mode(Xen ptr, Xen val)
{
  XM_set_field_assert_type(Xen_is_integer(val), val, 2, "subwindow_mode", "an integer");
  XM_set_field_assert_type(Xen_is_XGCValues(ptr), ptr, 1, "subwindow_mode", "XGCValues");
  (Xen_to_C_XGCValues(ptr))->subwindow_mode = Xen_integer_to_C_int(val);
  return(val);
}

static Xen gxm_graphics_exposures(Xen ptr)
{
  if (Xen_is_XGCValues(ptr)) return(C_bool_to_Xen_boolean((Bool)((Xen_to_C_XGCValues(ptr))->graphics_exposures)));
  XM_field_assert_type(0, ptr, 1, "graphics_exposures", "XGCValues");
  return(Xen_false);
}

static Xen gxm_set_graphics_exposures(Xen ptr, Xen val)
{
  XM_set_field_assert_type(Xen_is_boolean(val), val, 2, "graphics_exposures", "a boolean");
  XM_set_field_assert_type(Xen_is_XGCValues(ptr), ptr, 1, "graphics_exposure", "XGCValues");
  (Xen_to_C_XGCValues(ptr))->graphics_exposures = Xen_boolean_to_C_bool(val);
  return(val);
}

static Xen gxm_clip_x_origin(Xen ptr)
{
  if (Xen_is_XGCValues(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XGCValues(ptr))->clip_x_origin)));
  XM_field_assert_type(0, ptr, 1, "clip_x_origin", "XGCValues");
  return(Xen_false);
}

static Xen gxm_set_clip_x_origin(Xen ptr, Xen val)
{
  XM_set_field_assert_type(Xen_is_integer(val), val, 2, "clip_x_origin", "an integer");
  XM_set_field_assert_type(Xen_is_XGCValues(ptr), ptr, 1, "clip_x_origin", "XGCValues");
  (Xen_to_C_XGCValues(ptr))->clip_x_origin = Xen_integer_to_C_int(val);
  return(val);
}

static Xen gxm_clip_y_origin(Xen ptr)
{
  if (Xen_is_XGCValues(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XGCValues(ptr))->clip_y_origin)));
  XM_field_assert_type(0, ptr, 1, "clip_y_origin", "XGCValues");
  return(Xen_false);
}

static Xen gxm_set_clip_y_origin(Xen ptr, Xen val)
{
  XM_set_field_assert_type(Xen_is_integer(val), val, 2, "clip_y_origin", "an integer");
  XM_set_field_assert_type(Xen_is_XGCValues(ptr), ptr, 1, "clip_y_origin", "XGCValues");
  (Xen_to_C_XGCValues(ptr))->clip_y_origin = Xen_integer_to_C_int(val);
  return(val);
}

static Xen gxm_clip_mask(Xen ptr)
{
  if (Xen_is_XGCValues(ptr)) return(C_to_Xen_Pixmap((Pixmap)((Xen_to_C_XGCValues(ptr))->clip_mask)));
  XM_field_assert_type(0, ptr, 1, "clip_mask", "XGCValues");
  return(Xen_false);
}

static Xen gxm_set_clip_mask(Xen ptr, Xen val)
{
  XM_set_field_assert_type(Xen_is_Pixmap(val), val, 2, "clip_mask", "a Pixmap");
  XM_set_field_assert_type(Xen_is_XGCValues(ptr), ptr, 1, "clip_mask", "XGCValues");
  (Xen_to_C_XGCValues(ptr))->clip_mask = Xen_to_C_Pixmap(val);
  return(val);
}

static Xen gxm_dash_offset(Xen ptr)
{
  XM_field_assert_type(Xen_is_XGCValues(ptr), ptr, 1, "dash_offset", "XGCValues");
  return(C_int_to_Xen_integer((int)((Xen_to_C_XGCValues(ptr))->dash_offset)));
}

static Xen gxm_set_dash_offset(Xen ptr, Xen val)
{
  XM_set_field_assert_type(Xen_is_integer(val), val, 2, "dash_offset", "an integer");
  XM_set_field_assert_type(Xen_is_XGCValues(ptr), ptr, 1, "dash_offset", "XGCValues");
  (Xen_to_C_XGCValues(ptr))->dash_offset = Xen_integer_to_C_int(val);
  return(val);
}

static Xen gxm_dashes(Xen ptr)
{
  XM_field_assert_type(Xen_is_XGCValues(ptr), ptr, 1, "dashes", "XGCValues");
  return(C_int_to_Xen_integer((char)((Xen_to_C_XGCValues(ptr))->dashes)));
}

static Xen gxm_set_dashes(Xen ptr, Xen val)
{
  XM_set_field_assert_type(Xen_is_XGCValues(ptr), ptr, 1, "dashes", "XGCValues");
  (Xen_to_C_XGCValues(ptr))->dashes = (char)Xen_integer_to_C_int(val);
  return(val);
}


static Xen gxm_killid(Xen ptr)
{
  XM_field_assert_type(Xen_is_XStandardColormap(ptr), ptr, 1, "killid", "XStandardColormap");
  return(C_int_to_Xen_integer((long)((Xen_to_C_XStandardColormap(ptr))->killid)));
}

static Xen gxm_base_pixel(Xen ptr)
{
  XM_field_assert_type(Xen_is_XStandardColormap(ptr), ptr, 1, "base_pixel", "XStandardColormap");
  return(C_to_Xen_Pixel((unsigned long)((Xen_to_C_XStandardColormap(ptr))->base_pixel)));
}

static Xen gxm_blue_mult(Xen ptr)
{
  XM_field_assert_type(Xen_is_XStandardColormap(ptr), ptr, 1, "blue_mult", "XStandardColormap");
  return(C_ulong_to_Xen_ulong((unsigned long)((Xen_to_C_XStandardColormap(ptr))->blue_mult)));
}

static Xen gxm_blue_max(Xen ptr)
{
  XM_field_assert_type(Xen_is_XStandardColormap(ptr), ptr, 1, "blue_max", "XStandardColormap");
  return(C_ulong_to_Xen_ulong((unsigned long)((Xen_to_C_XStandardColormap(ptr))->blue_max)));
}

static Xen gxm_green_mult(Xen ptr)
{
  XM_field_assert_type(Xen_is_XStandardColormap(ptr), ptr, 1, "green_mult", "XStandardColormap");
  return(C_ulong_to_Xen_ulong((unsigned long)((Xen_to_C_XStandardColormap(ptr))->green_mult)));
}

static Xen gxm_green_max(Xen ptr)
{
  XM_field_assert_type(Xen_is_XStandardColormap(ptr), ptr, 1, "green_max", "XStandardColormap");
  return(C_ulong_to_Xen_ulong((unsigned long)((Xen_to_C_XStandardColormap(ptr))->green_max)));
}

static Xen gxm_red_mult(Xen ptr)
{
  XM_field_assert_type(Xen_is_XStandardColormap(ptr), ptr, 1, "red_mult", "XStandardColormap");
  return(C_ulong_to_Xen_ulong((unsigned long)((Xen_to_C_XStandardColormap(ptr))->red_mult)));
}

static Xen gxm_red_max(Xen ptr)
{
  XM_field_assert_type(Xen_is_XStandardColormap(ptr), ptr, 1, "red_max", "XStandardColormap");
  return(C_ulong_to_Xen_ulong((unsigned long)((Xen_to_C_XStandardColormap(ptr))->red_max)));
}

static Xen gxm_XTextItem(Xen chars, Xen nchars, Xen delta, Xen font)
{
  XTextItem *r;
  #define H_XTextItem "(XTextItem chars len delta font): new XTextItem struct"
  Xen_check_type(Xen_is_string(chars), chars, 1, "XTextItem", "char*");
  Xen_check_type(Xen_is_integer(nchars), nchars, 2, "XTextItem", "int");
  Xen_check_type(Xen_is_integer(delta), delta, 3, "XTextItem", "int");
  Xen_check_type(Xen_is_Font(font), font, 4, "XTextItem", "Font");
  r = (XTextItem *)calloc(1, sizeof(XTextItem));
  r->chars = (char *)Xen_string_to_C_string(chars);
  r->nchars = Xen_integer_to_C_int(nchars);
  r->delta = Xen_integer_to_C_int(delta);
  r->font = Xen_to_C_Font(font);
  return(wrap_for_Xen_obj("XTextItem", (XTextItem *)r));
}

static Xen gxm_chars(Xen ptr)
{
  XM_field_assert_type(Xen_is_XTextItem(ptr), ptr, 1, "chars", "XTextItem");
  return(C_string_to_Xen_string((char *)((Xen_to_C_XTextItem(ptr))->chars)));
}

static Xen gxm_set_chars(Xen ptr, Xen val)
{
  XM_set_field_assert_type(Xen_is_string(val), val, 2, "chars", "a string");
  XM_set_field_assert_type(Xen_is_XTextItem(ptr), ptr, 1, "chars", "XTextItem");
  (Xen_to_C_XTextItem(ptr))->chars = xen_strdup(Xen_string_to_C_string(val));
  return(val);
}

static Xen gxm_nchars(Xen ptr)
{
  XM_field_assert_type(Xen_is_XTextItem(ptr), ptr, 1, "nchars", "XTextItem");
  return(C_int_to_Xen_integer((int)((Xen_to_C_XTextItem(ptr))->nchars)));
}

static Xen gxm_set_nchars(Xen ptr, Xen val)
{
  XM_set_field_assert_type(Xen_is_integer(val), val, 2, "nchars", "an integer");
  XM_set_field_assert_type(Xen_is_XTextItem(ptr), ptr, 1, "nchars", "XTextItem");
  (Xen_to_C_XTextItem(ptr))->nchars = Xen_integer_to_C_int(val);
  return(val);
}

static Xen gxm_delta(Xen ptr)
{
  XM_field_assert_type(Xen_is_XTextItem(ptr), ptr, 1, "delta", "XTextItem");
  return(C_int_to_Xen_integer((int)((Xen_to_C_XTextItem(ptr))->delta)));
}

static Xen gxm_set_delta(Xen ptr, Xen val)
{
  XM_set_field_assert_type(Xen_is_integer(val), val, 2, "delta", "an integer");
  XM_set_field_assert_type(Xen_is_XTextItem(ptr), ptr, 1, "delta", "XTextItem");
  (Xen_to_C_XTextItem(ptr))->delta = Xen_integer_to_C_int(val);
  return(val);
}

static Xen gxm_data(Xen ptr)
{
  if (Xen_is_XImage(ptr)) return(C_string_to_Xen_string((char *)((Xen_to_C_XImage(ptr))->data)));
  if (Xen_is_XClientMessageEvent(ptr)) return(C_string_to_Xen_string((char *)((Xen_to_C_XClientMessageEvent(ptr))->data.b)));
  if (Xen_is_XmRowColumnCallbackStruct(ptr)) return(C_string_to_Xen_string((char *)((Xen_to_C_XmRowColumnCallbackStruct(ptr))->data)));
  if (Xen_is_XpmImage(ptr)) return(Xen_wrap_C_pointer((Xen_to_C_XpmImage(ptr))->data));
  XM_field_assert_type(0, ptr, 1, "data", "XpmImage");
  return(Xen_false);
}

static Xen gxm_set_data(Xen ptr, Xen val)
{
  char *str;
  XM_set_field_assert_type(Xen_is_string(val) || Xen_is_list(val), val, 2, "data", "a string or a list of longs");
  XM_set_field_assert_type(Xen_is_XClientMessageEvent(ptr), ptr, 1, "data", "XClientMessageEvent");
  if (Xen_is_string(val))
    {
      int i, len = 0;
      str = (char *)Xen_string_to_C_string(val);
      if (str)
	len = strlen(str);
      if (len > 19) len = 19;
      for (i = 0; i < len; i++)
	(Xen_to_C_XClientMessageEvent(ptr))->data.b[i] = str[i];
    }
  else
    {
      int i, len;
      len = Xen_list_length(val);
      if (len > 5) len = 5;  /* only room here for 5 ints */
      for (i = 0; i < len; i++)
	(Xen_to_C_XClientMessageEvent(ptr))->data.l[i] = Xen_integer_to_C_int(Xen_list_ref(val, i));
    }
  return(val);
}



/* -------------------------------------------------------------------------------- */

static Xen gxm_text(Xen ptr)
{
  /* DIFF: TextVerifyCallbackStruct text field returns list (string format)
   */
  XmTextBlock tb;
  if (Xen_is_XmTextVerifyCallbackStruct(ptr)) 
    {
      tb = (XmTextBlock)(Xen_to_C_XmTextVerifyCallbackStruct(ptr))->text;
      if ((tb) && (tb->length > 0))
	return(Xen_list_2(C_string_to_Xen_string(tb->ptr),
			  C_ulong_to_Xen_ulong(tb->format)));
      return(Xen_false);
    }
  XM_field_assert_type(0, ptr, 1, "text", "an XmTextVerifyCallbackStruct");
  return(Xen_false);
}

static Xen gxm_endPos(Xen ptr)
{
  XM_field_assert_type(Xen_is_XmTextVerifyCallbackStruct(ptr), ptr, 1, "endPos", "XmTextVerifyCallbackStruct");
  return(C_int_to_Xen_integer((long)((Xen_to_C_XmTextVerifyCallbackStruct(ptr))->endPos)));
}

static Xen gxm_startPos(Xen ptr)
{
  XM_field_assert_type(Xen_is_XmTextVerifyCallbackStruct(ptr), ptr, 1, "startPos", "XmTextVerifyCallbackStruct");
  return(C_int_to_Xen_integer((long)((Xen_to_C_XmTextVerifyCallbackStruct(ptr))->startPos)));
}

static Xen gxm_newInsert(Xen ptr)
{
  XM_field_assert_type(Xen_is_XmTextVerifyCallbackStruct(ptr), ptr, 1, "newInsert", "XmTextVerifyCallbackStruct");
  return(C_int_to_Xen_integer((long)((Xen_to_C_XmTextVerifyCallbackStruct(ptr))->newInsert)));
}

static Xen gxm_currInsert(Xen ptr)
{
  XM_field_assert_type(Xen_is_XmTextVerifyCallbackStruct(ptr), ptr, 1, "currInsert", "XmTextVerifyCallbackStruct");
  return(C_int_to_Xen_integer((long)((Xen_to_C_XmTextVerifyCallbackStruct(ptr))->currInsert)));
}

static Xen gxm_crossed_boundary(Xen ptr)
{
  XM_field_assert_type(Xen_is_XmSpinBoxCallbackStruct(ptr), ptr, 1, "crossed_boundary", "XmSpinBoxCallbackStruct");
  return(C_bool_to_Xen_boolean((Boolean)((Xen_to_C_XmSpinBoxCallbackStruct(ptr))->crossed_boundary)));
}

static Xen gxm_position(Xen ptr)
{
  XM_field_assert_type(Xen_is_XmSpinBoxCallbackStruct(ptr), ptr, 1, "position", "XmSpinBoxCallbackStruct");
  return(C_int_to_Xen_integer((int)((Xen_to_C_XmSpinBoxCallbackStruct(ptr))->position)));
}

static Xen gxm_tag(Xen ptr)
{
  XM_field_assert_type(Xen_is_XmDisplayCallbackStruct(ptr), ptr, 1, "tag", "XmDisplayCallbackStruct");
  return(C_string_to_Xen_string((XmStringTag)((Xen_to_C_XmDisplayCallbackStruct(ptr))->tag)));
}

static Xen gxm_render_table(Xen ptr)
{
  XM_field_assert_type(Xen_is_XmDisplayCallbackStruct(ptr), ptr, 1, "render_table", "XmDisplayCallbackStruct");
  return(C_to_Xen_XmRenderTable((XmRenderTable)((Xen_to_C_XmDisplayCallbackStruct(ptr))->render_table)));
}

static Xen gxm_font_name(Xen ptr)
{
  XM_field_assert_type(Xen_is_XmDisplayCallbackStruct(ptr), ptr, 1, "font_name", "XmDisplayCallbackStruct");
  return(C_string_to_Xen_string((char *)((Xen_to_C_XmDisplayCallbackStruct(ptr))->font_name)));
}

static Xen gxm_rendition(Xen ptr)
{
  XM_field_assert_type(Xen_is_XmDisplayCallbackStruct(ptr), ptr, 1, "rendition", "XmDisplayCallbackStruct");
  return(C_to_Xen_XmRendition((XmRendition)((Xen_to_C_XmDisplayCallbackStruct(ptr))->rendition)));
}

static Xen gxm_prev_page_widget(Xen ptr)
{
  XM_field_assert_type(Xen_is_XmNotebookCallbackStruct(ptr), ptr, 1, "prev_page_widget", "XmNotebookCallbackStruct");
  return(C_to_Xen_Widget((Widget)((Xen_to_C_XmNotebookCallbackStruct(ptr))->prev_page_widget)));
}

static Xen gxm_prev_page_number(Xen ptr)
{
  XM_field_assert_type(Xen_is_XmNotebookCallbackStruct(ptr), ptr, 1, "prev_page_number", "XmNotebookCallbackStruct");
  return(C_int_to_Xen_integer((int)((Xen_to_C_XmNotebookCallbackStruct(ptr))->prev_page_number)));
}

static Xen gxm_new_outline_state(Xen ptr)
{
  XM_field_assert_type(Xen_is_XmContainerOutlineCallbackStruct(ptr), ptr, 1, "new_outline_state", "XmContainerOutlineCallbackStruct");
  return(C_int_to_Xen_integer((int)((Xen_to_C_XmContainerOutlineCallbackStruct(ptr))->new_outline_state)));
}

static Xen gxm_postIt(Xen ptr)
{
  XM_field_assert_type(Xen_is_XmPopupHandlerCallbackStruct(ptr), ptr, 1, "postIt", "XmPopupHandlerCallbackStruct");
  return(C_bool_to_Xen_boolean((Boolean)((Xen_to_C_XmPopupHandlerCallbackStruct(ptr))->postIt)));
}

static Xen gxm_menuToPost(Xen ptr)
{
  XM_field_assert_type(Xen_is_XmPopupHandlerCallbackStruct(ptr), ptr, 1, "menuToPost", "XmPopupHandlerCallbackStruct");
  return(C_to_Xen_Widget((Widget)((Xen_to_C_XmPopupHandlerCallbackStruct(ptr))->menuToPost)));
}

static Xen gxm_set_postIt(Xen ptr, Xen post)
{
  XM_set_field_assert_type(Xen_is_boolean(post), post, 2, "postIt", "a boolean");
  XM_set_field_assert_type(Xen_is_XmPopupHandlerCallbackStruct(ptr), ptr, 1, "postIt", "XmPopupHandler");
  (Xen_to_C_XmPopupHandlerCallbackStruct(ptr))->postIt = Xen_boolean_to_C_bool(post);
  return(post);
}

static Xen gxm_set_menuToPost(Xen ptr, Xen menu)
{
  XM_set_field_assert_type(Xen_is_Widget(menu), menu, 2, "menuToPost", "a Widget");
  XM_set_field_assert_type(Xen_is_XmPopupHandlerCallbackStruct(ptr), ptr, 1, "menuToPost", "XmPopupHandler");
  (Xen_to_C_XmPopupHandlerCallbackStruct(ptr))->menuToPost = (Widget)Xen_to_C_Widget(menu);
  return(menu);
}

static Xen gxm_pattern_length(Xen ptr)
{
  XM_field_assert_type(Xen_is_XmFileSelectionBoxCallbackStruct(ptr), ptr, 1, "pattern_length", "XmFileSelectionBoxCallbackStruct");
  return(C_int_to_Xen_integer((int)((Xen_to_C_XmFileSelectionBoxCallbackStruct(ptr))->pattern_length)));
}

static Xen gxm_pattern(Xen ptr)
{
  XM_field_assert_type(Xen_is_XmFileSelectionBoxCallbackStruct(ptr), ptr, 1, "pattern", "XmFileSelectionBoxCallbackStruct");
  return(C_to_Xen_XmString((XmString)((Xen_to_C_XmFileSelectionBoxCallbackStruct(ptr))->pattern)));
}

static Xen gxm_dir_length(Xen ptr)
{
  XM_field_assert_type(Xen_is_XmFileSelectionBoxCallbackStruct(ptr), ptr, 1, "dir_length", "XmFileSelectionBoxCallbackStruct");
  return(C_int_to_Xen_integer((int)((Xen_to_C_XmFileSelectionBoxCallbackStruct(ptr))->dir_length)));
}

static Xen gxm_dir(Xen ptr)
{
  XM_field_assert_type(Xen_is_XmFileSelectionBoxCallbackStruct(ptr), ptr, 1, "dir", "XmFileSelectionBoxCallbackStruct");
  return(C_to_Xen_XmString((XmString)((Xen_to_C_XmFileSelectionBoxCallbackStruct(ptr))->dir)));
}

static Xen gxm_mask_length(Xen ptr)
{
  XM_field_assert_type(Xen_is_XmFileSelectionBoxCallbackStruct(ptr), ptr, 1, "mask_length", "XmFileSelectionBoxCallbackStruct");
  return(C_int_to_Xen_integer((int)((Xen_to_C_XmFileSelectionBoxCallbackStruct(ptr))->mask_length)));
}

static Xen gxm_mask(Xen ptr)
{
  XM_field_assert_type(Xen_is_XmFileSelectionBoxCallbackStruct(ptr), ptr, 1, "mask", "XmFileSelectionBoxCallbackStruct");
  return(C_to_Xen_XmString((XmString)((Xen_to_C_XmFileSelectionBoxCallbackStruct(ptr))->mask)));
}

static Xen gxm_auto_selection_type(Xen ptr)
{
  XM_field_assert_type(Xen_is_XmContainerSelectCallbackStruct(ptr) || Xen_is_XmListCallbackStruct(ptr), 
		  ptr, 1, "auto_selection_type", "a struct with an auto_selection_type field");
  if (Xen_is_XmContainerSelectCallbackStruct(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XmContainerSelectCallbackStruct(ptr))->auto_selection_type)));
  return(C_int_to_Xen_integer((char)((Xen_to_C_XmListCallbackStruct(ptr))->auto_selection_type)));
}

static Xen gxm_selection_type(Xen ptr)
{
  XM_field_assert_type(Xen_is_XmListCallbackStruct(ptr), ptr, 1, "selection_type", "XmListCallbackStruct");
  return(C_int_to_Xen_integer((char)((Xen_to_C_XmListCallbackStruct(ptr))->selection_type)));
}

static Xen gxm_selected_item_positions(Xen ptr)
{
  /* DIFF: selected_item_positions is a list of ints
   */
  if (Xen_is_XmListCallbackStruct(ptr)) 
    return(C_to_Xen_Ints((int *)((Xen_to_C_XmListCallbackStruct(ptr))->selected_item_positions),
			 ((int)((Xen_to_C_XmListCallbackStruct(ptr))->selected_item_count))));
  XM_field_assert_type(0, ptr, 1, "selected_item_positions", "a struct with a selected_item_positions field");
  return(Xen_false);
}

static Xen gxm_selected_item_count(Xen ptr)
{
  if (Xen_is_XmContainerSelectCallbackStruct(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XmContainerSelectCallbackStruct(ptr))->selected_item_count)));
  if (Xen_is_XmListCallbackStruct(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XmListCallbackStruct(ptr))->selected_item_count)));
  XM_field_assert_type(0, ptr, 1, "selected_item_count", "XmListCallbackStruct");
  return(Xen_false);
}

static Xen gxm_selected_items(Xen ptr)
{
  /* DIFF selected_items field returns list
   */
  if (Xen_is_XmContainerSelectCallbackStruct(ptr)) 
    return(C_to_Xen_Widgets((WidgetList)((Xen_to_C_XmContainerSelectCallbackStruct(ptr))->selected_items),
			    (Xen_to_C_XmContainerSelectCallbackStruct(ptr))->selected_item_count));
  if (Xen_is_XmListCallbackStruct(ptr)) 
    return(C_to_Xen_XmStringTable((XmStringTable)((Xen_to_C_XmListCallbackStruct(ptr))->selected_items),
				  (Xen_to_C_XmListCallbackStruct(ptr))->selected_item_count));
  XM_field_assert_type(0, ptr, 1, "selected_items", "a struct with a selected_items field");
  return(Xen_false);
}

static Xen gxm_item_length(Xen ptr)
{
  XM_field_assert_type(Xen_is_XmListCallbackStruct(ptr), ptr, 1, "item_length", "XmListCallbackStruct");
  return(C_int_to_Xen_integer((int)((Xen_to_C_XmListCallbackStruct(ptr))->item_length)));
}

static Xen gxm_item(Xen ptr)
{
  if (Xen_is_XmContainerOutlineCallbackStruct(ptr)) return(C_to_Xen_Widget((Widget)((Xen_to_C_XmContainerOutlineCallbackStruct(ptr))->item)));
  if (Xen_is_XmListCallbackStruct(ptr)) return(C_to_Xen_XmString((XmString)((Xen_to_C_XmListCallbackStruct(ptr))->item)));
  XM_field_assert_type(0, ptr, 1, "item", "XmListCallbackStruct");
  return(Xen_false);
}

/* in Motif 2, this field is an int: XmSET to XmINDETERMINATE -- should we change? */
static Xen gxm_set(Xen ptr)
{
  XM_field_assert_type(Xen_is_XmToggleButtonCallbackStruct(ptr), ptr, 1, "set", "XmToggleButtonCallbackStruct");
  return(C_bool_to_Xen_boolean((int)((Xen_to_C_XmToggleButtonCallbackStruct(ptr))->set)));
}

static Xen gxm_set_set(Xen ptr, Xen val)
{
  XM_set_field_assert_type(Xen_is_boolean(val) || Xen_is_integer(val), val, 2, "set", "a boolean");
  XM_set_field_assert_type(Xen_is_XmToggleButtonCallbackStruct(ptr), ptr, 1, "set", "XmToggleButtonCallbackStruct");
  (Xen_to_C_XmToggleButtonCallbackStruct(ptr))->set = Xen_boolean_to_C_bool(val);
  return(val);
}

static Xen gxm_callbackstruct(Xen ptr)
{
  XM_field_assert_type(Xen_is_XmRowColumnCallbackStruct(ptr), ptr, 1, "callbackstruct", "XmRowColumnCallbackStruct");
  return(C_string_to_Xen_string((char *)((Xen_to_C_XmRowColumnCallbackStruct(ptr))->callbackstruct)));
}

static Xen gxm_item_position(Xen ptr)
{
  if (Xen_is_XmListCallbackStruct(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XmListCallbackStruct(ptr))->item_position)));
  if (Xen_is_XmComboBoxCallbackStruct(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XmComboBoxCallbackStruct(ptr))->item_position)));
  XM_field_assert_type(0, ptr, 1, "item_position", "a struct with an item_position field");
  return(Xen_false);
}

static Xen gxm_item_or_text(Xen ptr)
{
  XM_field_assert_type(Xen_is_XmComboBoxCallbackStruct(ptr), ptr, 1, "item_or_text", "XmComboBoxCallbackStruct");
  return(C_to_Xen_XmString((XmString)((Xen_to_C_XmComboBoxCallbackStruct(ptr))->item_or_text)));
}

static Xen gxm_doit(Xen ptr)
{
  if (Xen_is_XmTextVerifyCallbackStruct(ptr)) return(C_bool_to_Xen_boolean((Boolean)((Xen_to_C_XmTextVerifyCallbackStruct(ptr))->doit)));
  if (Xen_is_XmSpinBoxCallbackStruct(ptr)) return(C_bool_to_Xen_boolean((Boolean)((Xen_to_C_XmSpinBoxCallbackStruct(ptr))->doit)));
  if (Xen_is_XmDragStartCallbackStruct(ptr)) return(C_bool_to_Xen_boolean((Boolean)((Xen_to_C_XmDragStartCallbackStruct(ptr))->doit)));
  XM_field_assert_type(0, ptr, 1, "doit", "a struct with a doit field");
  return(Xen_false);
}

static Xen gxm_set_doit(Xen ptr, Xen val)
{
  XM_set_field_assert_type(Xen_is_boolean(val), val, 2, "doit", "a boolean");
  if (Xen_is_XmTextVerifyCallbackStruct(ptr)) (Xen_to_C_XmTextVerifyCallbackStruct(ptr))->doit = Xen_boolean_to_C_bool(val);
  else if (Xen_is_XmSpinBoxCallbackStruct(ptr)) (Xen_to_C_XmSpinBoxCallbackStruct(ptr))->doit = Xen_boolean_to_C_bool(val);
  else if (Xen_is_XmDragStartCallbackStruct(ptr)) (Xen_to_C_XmDragStartCallbackStruct(ptr))->doit = Xen_boolean_to_C_bool(val);
  else XM_set_field_assert_type(0, ptr, 1, "doit", "a struct with a doit field");
  return(val);
}

static Xen gxm_widget(Xen ptr)
{
  if (Xen_is_XmSpinBoxCallbackStruct(ptr)) return(C_to_Xen_Widget((Widget)((Xen_to_C_XmSpinBoxCallbackStruct(ptr))->widget)));
  if (Xen_is_XmDragStartCallbackStruct(ptr)) return(C_to_Xen_Widget((Widget)((Xen_to_C_XmDragStartCallbackStruct(ptr))->widget)));
  if (Xen_is_XmRowColumnCallbackStruct(ptr)) return(C_to_Xen_Widget((Widget)((Xen_to_C_XmRowColumnCallbackStruct(ptr))->widget)));
  XM_field_assert_type(0, ptr, 1, "widget", "a struct with a widget field");
  return(Xen_false);
}

static Xen gxm_click_count(Xen ptr)
{
  if (Xen_is_XmPushButtonCallbackStruct(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XmPushButtonCallbackStruct(ptr))->click_count)));
  if (Xen_is_XmDrawnButtonCallbackStruct(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XmDrawnButtonCallbackStruct(ptr))->click_count)));
  if (Xen_is_XmArrowButtonCallbackStruct(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XmArrowButtonCallbackStruct(ptr))->click_count)));
  XM_field_assert_type(0, ptr, 1, "click_count", "a struct with a click_count field");
  return(Xen_false);
}

static Xen gxm_set_click_count(Xen ptr, Xen val)
{
  int count;
  XM_set_field_assert_type(Xen_is_integer(val), val, 2, "click_count", "int");
  count = Xen_integer_to_C_int(val);
  if (Xen_is_XmPushButtonCallbackStruct(ptr)) (Xen_to_C_XmPushButtonCallbackStruct(ptr))->click_count = count;
  else if (Xen_is_XmDrawnButtonCallbackStruct(ptr)) (Xen_to_C_XmDrawnButtonCallbackStruct(ptr))->click_count = count;
  else if (Xen_is_XmArrowButtonCallbackStruct(ptr)) (Xen_to_C_XmArrowButtonCallbackStruct(ptr))->click_count = count;
  else XM_set_field_assert_type(0, ptr, 1, "click_count", "a struct with a click_count field");
  return(val);
}

static Xen gxm_remaining(Xen ptr)
{
  XM_field_assert_type(Xen_is_XmSelectionCallbackStruct(ptr), ptr, 1, "remaining", "XmSelectionCallbackStruct");
  return(C_int_to_Xen_integer((int)((Xen_to_C_XmSelectionCallbackStruct(ptr))->remaining)));
}

static Xen gxm_destination_data(Xen ptr)
{
  /* widget-class specific interpretation -- is it worth untangling? */
  XM_field_assert_type(Xen_is_XmDestinationCallbackStruct(ptr), ptr, 1, "destination_data", "XmDestinationCallbackStruct");
  return(Xen_wrap_C_pointer(((Xen_to_C_XmDestinationCallbackStruct(ptr))->destination_data)));
}

static Xen gxm_transfer_id(Xen ptr)
{
  if (Xen_is_XmTransferDoneCallbackStruct(ptr)) return(Xen_wrap_C_pointer(((Xen_to_C_XmTransferDoneCallbackStruct(ptr))->transfer_id)));
  if (Xen_is_XmSelectionCallbackStruct(ptr)) return(Xen_wrap_C_pointer(((Xen_to_C_XmSelectionCallbackStruct(ptr))->transfer_id)));
  if (Xen_is_XmDestinationCallbackStruct(ptr)) return(Xen_wrap_C_pointer(((Xen_to_C_XmDestinationCallbackStruct(ptr))->transfer_id)));
  XM_field_assert_type(0, ptr, 1, "transfer_id", "a struct with a transfer_id field");
  return(Xen_false);
}

static Xen gxm_length(Xen ptr)
{
  if (Xen_is_XmFileSelectionBoxCallbackStruct(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XmFileSelectionBoxCallbackStruct(ptr))->length)));
  if (Xen_is_XmCommandCallbackStruct(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XmCommandCallbackStruct(ptr))->length)));
  if (Xen_is_XmTextBlock(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XmTextBlock(ptr))->length)));
  if (Xen_is_XmSelectionBoxCallbackStruct(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XmSelectionBoxCallbackStruct(ptr))->length)));
  if (Xen_is_XmSelectionCallbackStruct(ptr)) return(C_ulong_to_Xen_ulong((unsigned long)((Xen_to_C_XmSelectionCallbackStruct(ptr))->length)));
  if (Xen_is_XmConvertCallbackStruct(ptr)) return(C_ulong_to_Xen_ulong((unsigned long)((Xen_to_C_XmConvertCallbackStruct(ptr))->length)));
  XM_field_assert_type(0, ptr, 1, "length", "a struct with a length field");
  return(Xen_false);
}

static Xen gxm_set_length(Xen ptr, Xen val)
{
  XM_set_field_assert_type(Xen_is_XmTextBlock(ptr), ptr, 1, "length", "XmTextBlock");
  XM_set_field_assert_type(Xen_is_integer(val), val, 2, "length", "int");
  (Xen_to_C_XmTextBlock(ptr))->length = Xen_integer_to_C_int(val);
  return(val);
}

static Xen gxm_ptr(Xen ptr)
{
  XM_field_assert_type(Xen_is_XmTextBlock(ptr), ptr, 1, "ptr", "XmTextBlock");
  return(C_string_to_Xen_string((Xen_to_C_XmTextBlock(ptr))->ptr));
}

static Xen gxm_set_ptr(Xen pt, Xen val)
{
  XM_set_field_assert_type(Xen_is_XmTextBlock(pt), pt, 1, "ptr", "XmTextBlock");
  XM_set_field_assert_type(Xen_is_string(val), val, 2, "length", "char*");
  (Xen_to_C_XmTextBlock(pt))->ptr = xen_strdup(Xen_string_to_C_string(val));
  return(val);
}

static Xen gxm_value(Xen ptr)
{
  if (Xen_is_XmSpinBoxCallbackStruct(ptr)) return(C_to_Xen_XmString((XmString)((Xen_to_C_XmSpinBoxCallbackStruct(ptr))->value)));
  if (Xen_is_XmSelectionBoxCallbackStruct(ptr)) return(C_to_Xen_XmString((XmString)((Xen_to_C_XmSelectionBoxCallbackStruct(ptr))->value)));
  if (Xen_is_XmConvertCallbackStruct(ptr)) return(Xen_wrap_C_pointer((XtPointer)((Xen_to_C_XmConvertCallbackStruct(ptr))->value)));
  if (Xen_is_XmSelectionCallbackStruct(ptr)) return(Xen_wrap_C_pointer((XtPointer)((Xen_to_C_XmSelectionCallbackStruct(ptr))->value)));
  if (Xen_is_XmScaleCallbackStruct(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XmScaleCallbackStruct(ptr))->value)));
  if (Xen_is_XmFileSelectionBoxCallbackStruct(ptr)) return(C_to_Xen_XmString((XmString)((Xen_to_C_XmFileSelectionBoxCallbackStruct(ptr))->value)));
  if (Xen_is_XmCommandCallbackStruct(ptr)) return(C_to_Xen_XmString((XmString)((Xen_to_C_XmCommandCallbackStruct(ptr))->value)));
  if (Xen_is_XmScrollBarCallbackStruct(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XmScrollBarCallbackStruct(ptr))->value)));
  if (Xen_is_XpmColorSymbol(ptr)) return(C_string_to_Xen_string((char *)((Xen_to_C_XpmColorSymbol(ptr))->value)));
  XM_field_assert_type(0, ptr, 1, "value", "a struct with a value field");
  return(Xen_false);
}

static Xen gxm_set_value(Xen ptr, Xen val)
{
  if (Xen_is_XmScaleCallbackStruct(ptr)) (Xen_to_C_XmScaleCallbackStruct(ptr))->value = Xen_integer_to_C_int(val); else
  if (Xen_is_XmFileSelectionBoxCallbackStruct(ptr)) (Xen_to_C_XmFileSelectionBoxCallbackStruct(ptr))->value = Xen_to_C_XmString(val); else
  if (Xen_is_XmCommandCallbackStruct(ptr)) (Xen_to_C_XmCommandCallbackStruct(ptr))->value = Xen_to_C_XmString(val); else
  if (Xen_is_XmScrollBarCallbackStruct(ptr)) (Xen_to_C_XmScrollBarCallbackStruct(ptr))->value = Xen_integer_to_C_int(val); else
  if (Xen_is_XmConvertCallbackStruct(ptr)) (Xen_to_C_XmConvertCallbackStruct(ptr))->value = (XtPointer)Xen_unwrap_C_pointer(val); else
  if (Xen_is_XmSpinBoxCallbackStruct(ptr)) (Xen_to_C_XmSpinBoxCallbackStruct(ptr))->value = Xen_to_C_XmString(val); else
  if (Xen_is_XmSelectionBoxCallbackStruct(ptr)) (Xen_to_C_XmSelectionBoxCallbackStruct(ptr))->value = Xen_to_C_XmString(val); else
  if (Xen_is_XmSelectionCallbackStruct(ptr)) (Xen_to_C_XmSelectionCallbackStruct(ptr))->value = (XtPointer)Xen_unwrap_C_pointer(val); else
  if (Xen_is_XpmColorSymbol(ptr)) (Xen_to_C_XpmColorSymbol(ptr))->value = xen_strdup(Xen_string_to_C_string(val)); else
  XM_set_field_assert_type(0, ptr, 1, "value", "a struct with a value field");
  return(val);
}

static Xen gxm_status(Xen ptr)
{
  XM_field_assert_type(Xen_is_XmTransferDoneCallbackStruct(ptr) || Xen_is_XmConvertCallbackStruct(ptr), ptr, 1, "status", "a struct with a status field");
  if (Xen_is_XmTransferDoneCallbackStruct(ptr)) return(C_int_to_Xen_integer((XmTransferStatus)((Xen_to_C_XmTransferDoneCallbackStruct(ptr))->status)));
  return(C_int_to_Xen_integer((int)((Xen_to_C_XmConvertCallbackStruct(ptr))->status)));
}

static Xen gxm_parm_type(Xen ptr)
{
  XM_field_assert_type(Xen_is_XmConvertCallbackStruct(ptr), ptr, 1, "parm_type", "XmConvertCallbackStruct");
  return(C_to_Xen_Atom((Atom)((Xen_to_C_XmConvertCallbackStruct(ptr))->parm_type)));
}

static Xen gxm_parm_length(Xen ptr)
{
  XM_field_assert_type(Xen_is_XmConvertCallbackStruct(ptr), ptr, 1, "parm_length", "XmConvertCallbackStruct");
  return(C_ulong_to_Xen_ulong((unsigned long)((Xen_to_C_XmConvertCallbackStruct(ptr))->parm_length)));
}

static Xen gxm_parm_format(Xen ptr)
{
  XM_field_assert_type(Xen_is_XmConvertCallbackStruct(ptr), ptr, 1, "parm_format", "XmConvertCallbackStruct");
  return(C_int_to_Xen_integer((int)((Xen_to_C_XmConvertCallbackStruct(ptr))->parm_format)));
}

static Xen gxm_parm(Xen ptr)
{
  XM_field_assert_type(Xen_is_XmConvertCallbackStruct(ptr), ptr, 1, "parm", "XmConvertCallbackStruct");
  return(Xen_wrap_C_pointer((XtPointer)((Xen_to_C_XmConvertCallbackStruct(ptr))->parm)));
}

static Xen gxm_location_data(Xen ptr)
{
  /* widget-class specific interpretation -- is it worth untangling? */
  XM_field_assert_type(Xen_is_XmDestinationCallbackStruct(ptr) || Xen_is_XmConvertCallbackStruct(ptr), 
		  ptr, 1, "location_data", "a struct with a location_data field");
  if (Xen_is_XmDestinationCallbackStruct(ptr)) return(Xen_wrap_C_pointer((XtPointer)((Xen_to_C_XmDestinationCallbackStruct(ptr))->location_data)));
  return(Xen_wrap_C_pointer((XtPointer)((Xen_to_C_XmConvertCallbackStruct(ptr))->location_data)));
}

static Xen gxm_source_data(Xen ptr)
{
  XM_field_assert_type(Xen_is_XmConvertCallbackStruct(ptr), ptr, 1, "source_data", "XmConvertCallbackStruct");
  return(Xen_wrap_C_pointer((XtPointer)((Xen_to_C_XmConvertCallbackStruct(ptr))->source_data)));
}

static Xen gxm_client_data(Xen ptr)
{
  XM_field_assert_type(Xen_is_XmTransferDoneCallbackStruct(ptr), ptr, 1, "client_data", "XmTransferDoneCallbackStruct");
  return(Xen_wrap_C_pointer((XtPointer)((Xen_to_C_XmTransferDoneCallbackStruct(ptr))->client_data)));
}

static Xen gxm_animate(Xen ptr)
{
  XM_field_assert_type(Xen_is_XmDragProcCallbackStruct(ptr), ptr, 1, "animate", "XmDragProcCallbackStruct");
  return(C_bool_to_Xen_boolean((Boolean)((Xen_to_C_XmDragProcCallbackStruct(ptr))->animate)));
}

static Xen gxm_dragContext(Xen ptr)
{
  XM_field_assert_type(Xen_is_XmDropProcCallbackStruct(ptr) || Xen_is_XmDragProcCallbackStruct(ptr), ptr, 1, "dragContext", "a struct with a dragContext field");
  if (Xen_is_XmDropProcCallbackStruct(ptr)) return(C_to_Xen_Widget((Widget)((Xen_to_C_XmDropProcCallbackStruct(ptr))->dragContext)));
  return(C_to_Xen_Widget((Widget)((Xen_to_C_XmDragProcCallbackStruct(ptr))->dragContext)));
}

static Xen gxm_completionStatus(Xen ptr)
{
  XM_field_assert_type(Xen_is_XmDropFinishCallbackStruct(ptr), ptr, 1, "completionStatus", "XmDropFinishCallbackStruct");
  return(C_int_to_Xen_integer((int)((Xen_to_C_XmDropFinishCallbackStruct(ptr))->completionStatus)));
}

static Xen gxm_iccHandle(Xen ptr)
{
  if (Xen_is_XmDropStartCallbackStruct(ptr)) return(C_to_Xen_Atom((Atom)((Xen_to_C_XmDropStartCallbackStruct(ptr))->iccHandle)));
  if (Xen_is_XmTopLevelEnterCallbackStruct(ptr)) return(C_to_Xen_Atom((Atom)((Xen_to_C_XmTopLevelEnterCallbackStruct(ptr))->iccHandle)));
  XM_field_assert_type(0, ptr, 1, "iccHandle", "a struct with an iccHandle field");
  return(Xen_false);
}

static Xen gxm_dropAction(Xen ptr)
{
  if (Xen_is_XmDropProcCallbackStruct(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XmDropProcCallbackStruct(ptr))->dropAction)));
  if (Xen_is_XmDropFinishCallbackStruct(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XmDropFinishCallbackStruct(ptr))->dropAction)));
  if (Xen_is_XmDropStartCallbackStruct(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XmDropStartCallbackStruct(ptr))->dropAction)));
  XM_field_assert_type(0, ptr, 1, "dropAction", "a struct with a dropAction field");
  return(Xen_false);
}

static Xen gxm_dropSiteStatus(Xen ptr)
{
  if (Xen_is_XmDropProcCallbackStruct(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XmDropProcCallbackStruct(ptr))->dropSiteStatus)));
  if (Xen_is_XmDragProcCallbackStruct(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XmDragProcCallbackStruct(ptr))->dropSiteStatus)));
  if (Xen_is_XmDropFinishCallbackStruct(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XmDropFinishCallbackStruct(ptr))->dropSiteStatus)));
  if (Xen_is_XmDropStartCallbackStruct(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XmDropStartCallbackStruct(ptr))->dropSiteStatus)));
  if (Xen_is_XmOperationChangedCallbackStruct(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XmOperationChangedCallbackStruct(ptr))->dropSiteStatus)));
  if (Xen_is_XmDragMotionCallbackStruct(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XmDragMotionCallbackStruct(ptr))->dropSiteStatus)));
  if (Xen_is_XmDropSiteEnterCallbackStruct(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XmDropSiteEnterCallbackStruct(ptr))->dropSiteStatus)));
  XM_field_assert_type(0, ptr, 1, "dropSiteStatus", "a struct with a dropSiteStatus field");
  return(Xen_false);
}

static Xen gxm_set_dropSiteStatus(Xen ptr, Xen val)
{
  XM_set_field_assert_type(Xen_is_integer(val), val, 2, "dropSiteStatus", "an integer");
  if (Xen_is_XmDropProcCallbackStruct(ptr)) (Xen_to_C_XmDropProcCallbackStruct(ptr))->dropSiteStatus = Xen_integer_to_C_int(val); else
  if (Xen_is_XmDragProcCallbackStruct(ptr)) (Xen_to_C_XmDragProcCallbackStruct(ptr))->dropSiteStatus = Xen_integer_to_C_int(val); else
  if (Xen_is_XmDropFinishCallbackStruct(ptr)) (Xen_to_C_XmDropFinishCallbackStruct(ptr))->dropSiteStatus = Xen_integer_to_C_int(val); else
  if (Xen_is_XmDropStartCallbackStruct(ptr)) (Xen_to_C_XmDropStartCallbackStruct(ptr))->dropSiteStatus = Xen_integer_to_C_int(val); else
  if (Xen_is_XmOperationChangedCallbackStruct(ptr)) (Xen_to_C_XmOperationChangedCallbackStruct(ptr))->dropSiteStatus = Xen_integer_to_C_int(val); else
  if (Xen_is_XmDragMotionCallbackStruct(ptr)) (Xen_to_C_XmDragMotionCallbackStruct(ptr))->dropSiteStatus = Xen_integer_to_C_int(val); else
  if (Xen_is_XmDropSiteEnterCallbackStruct(ptr)) (Xen_to_C_XmDropSiteEnterCallbackStruct(ptr))->dropSiteStatus = Xen_integer_to_C_int(val); else
  XM_set_field_assert_type(0, ptr, 1, "dropSiteStatus", "a struct with a dropSiteStatus field");
  return(val);
}

static Xen gxm_operations(Xen ptr)
{
  if (Xen_is_XmDropProcCallbackStruct(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XmDropProcCallbackStruct(ptr))->operations)));
  if (Xen_is_XmDragProcCallbackStruct(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XmDragProcCallbackStruct(ptr))->operations)));
  if (Xen_is_XmDropFinishCallbackStruct(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XmDropFinishCallbackStruct(ptr))->operations)));
  if (Xen_is_XmDropStartCallbackStruct(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XmDropStartCallbackStruct(ptr))->operations)));
  if (Xen_is_XmOperationChangedCallbackStruct(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XmOperationChangedCallbackStruct(ptr))->operations)));
  if (Xen_is_XmDragMotionCallbackStruct(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XmDragMotionCallbackStruct(ptr))->operations)));
  if (Xen_is_XmDropSiteEnterCallbackStruct(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XmDropSiteEnterCallbackStruct(ptr))->operations)));
  XM_field_assert_type(0, ptr, 1, "operations", "a struct with an operations field");
  return(Xen_false);
}

static Xen gxm_operation(Xen ptr)
{
  if (Xen_is_XmDestinationCallbackStruct(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XmDestinationCallbackStruct(ptr))->operation)));
  if (Xen_is_XmDropProcCallbackStruct(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XmDropProcCallbackStruct(ptr))->operation)));
  if (Xen_is_XmDragProcCallbackStruct(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XmDragProcCallbackStruct(ptr))->operation)));
  if (Xen_is_XmDropFinishCallbackStruct(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XmDropFinishCallbackStruct(ptr))->operation)));
  if (Xen_is_XmDropStartCallbackStruct(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XmDropStartCallbackStruct(ptr))->operation)));
  if (Xen_is_XmOperationChangedCallbackStruct(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XmOperationChangedCallbackStruct(ptr))->operation)));
  if (Xen_is_XmDragMotionCallbackStruct(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XmDragMotionCallbackStruct(ptr))->operation)));
  if (Xen_is_XmDropSiteEnterCallbackStruct(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XmDropSiteEnterCallbackStruct(ptr))->operation)));
  XM_field_assert_type(0, ptr, 1, "operation", "a struct with an operation field");
  return(Xen_false);
}

static Xen gxm_set_operation(Xen ptr, Xen val)
{
  XM_set_field_assert_type(Xen_is_integer(val), val, 2, "operation", "an integer");
  if (Xen_is_XmDestinationCallbackStruct(ptr)) (Xen_to_C_XmDestinationCallbackStruct(ptr))->operation = Xen_integer_to_C_int(val); else
  if (Xen_is_XmDropProcCallbackStruct(ptr)) (Xen_to_C_XmDropProcCallbackStruct(ptr))->operation = Xen_integer_to_C_int(val); else
  if (Xen_is_XmDragProcCallbackStruct(ptr)) (Xen_to_C_XmDragProcCallbackStruct(ptr))->operation = Xen_integer_to_C_int(val); else
  if (Xen_is_XmDropFinishCallbackStruct(ptr)) (Xen_to_C_XmDropFinishCallbackStruct(ptr))->operation = Xen_integer_to_C_int(val); else
  if (Xen_is_XmDropStartCallbackStruct(ptr)) (Xen_to_C_XmDropStartCallbackStruct(ptr))->operation = Xen_integer_to_C_int(val); else
  if (Xen_is_XmOperationChangedCallbackStruct(ptr)) (Xen_to_C_XmOperationChangedCallbackStruct(ptr))->operation = Xen_integer_to_C_int(val); else
  if (Xen_is_XmDragMotionCallbackStruct(ptr)) (Xen_to_C_XmDragMotionCallbackStruct(ptr))->operation = Xen_integer_to_C_int(val); else
  if (Xen_is_XmDropSiteEnterCallbackStruct(ptr)) (Xen_to_C_XmDropSiteEnterCallbackStruct(ptr))->operation = Xen_integer_to_C_int(val); else
  XM_set_field_assert_type(0, ptr, 1, "operation", "a struct with an operation field");
  return(val);
}

static Xen gxm_timeStamp(Xen ptr)
{
  if (Xen_is_XmDropProcCallbackStruct(ptr)) return(C_to_Xen_Time((Time)((Xen_to_C_XmDropProcCallbackStruct(ptr))->timeStamp)));
  if (Xen_is_XmDragProcCallbackStruct(ptr)) return(C_to_Xen_Time((Time)((Xen_to_C_XmDragProcCallbackStruct(ptr))->timeStamp)));
  if (Xen_is_XmDragDropFinishCallbackStruct(ptr)) return(C_to_Xen_Time((Time)((Xen_to_C_XmDragDropFinishCallbackStruct(ptr))->timeStamp)));
  if (Xen_is_XmDropFinishCallbackStruct(ptr)) return(C_to_Xen_Time((Time)((Xen_to_C_XmDropFinishCallbackStruct(ptr))->timeStamp)));
  if (Xen_is_XmDropStartCallbackStruct(ptr)) return(C_to_Xen_Time((Time)((Xen_to_C_XmDropStartCallbackStruct(ptr))->timeStamp)));
  if (Xen_is_XmOperationChangedCallbackStruct(ptr)) return(C_to_Xen_Time((Time)((Xen_to_C_XmOperationChangedCallbackStruct(ptr))->timeStamp)));
  if (Xen_is_XmDragMotionCallbackStruct(ptr)) return(C_to_Xen_Time((Time)((Xen_to_C_XmDragMotionCallbackStruct(ptr))->timeStamp)));
  if (Xen_is_XmDropSiteLeaveCallbackStruct(ptr)) return(C_to_Xen_Time((Time)((Xen_to_C_XmDropSiteLeaveCallbackStruct(ptr))->timeStamp)));
  if (Xen_is_XmDropSiteEnterCallbackStruct(ptr)) return(C_to_Xen_Time((Time)((Xen_to_C_XmDropSiteEnterCallbackStruct(ptr))->timeStamp)));
  if (Xen_is_XmTopLevelEnterCallbackStruct(ptr)) return(C_to_Xen_Time((Time)((Xen_to_C_XmTopLevelEnterCallbackStruct(ptr))->timeStamp)));
  if (Xen_is_XmTopLevelLeaveCallbackStruct(ptr)) return(C_to_Xen_Time((Time)((Xen_to_C_XmTopLevelLeaveCallbackStruct(ptr))->timeStamp)));
  XM_field_assert_type(0, ptr, 1, "timeStamp", "a struct with a timeStamp field");
  return(Xen_false);
}

static Xen gxm_reason(Xen ptr)
{
  XM_field_assert_type(Xen_is_AnyCallbackStruct(ptr), ptr, 1, "reason", "a callbackstruct");
  return(C_int_to_Xen_integer((int)((Xen_to_C_XmAnyCallbackStruct(ptr))->reason)));
}

static Xen gxm_set_reason(Xen ptr, Xen val)
{
  XM_set_field_assert_type(Xen_is_integer(val), val, 2, "reason", "integer");
  XM_set_field_assert_type(Xen_is_AnyCallbackStruct(ptr), ptr, 1, "reason", "a callbackstruct");
  (Xen_to_C_XmAnyCallbackStruct(ptr))->reason = Xen_integer_to_C_int(val);
  return(val);
}

static Xen gxm_direction(Xen ptr)
{
  XM_field_assert_type(Xen_is_XmTraverseObscuredCallbackStruct(ptr), ptr, 1, "direction", "XmTraverseObscuredCallbackStruct");
  return(C_int_to_Xen_integer((int)((Xen_to_C_XmTraverseObscuredCallbackStruct(ptr))->direction)));
}

static Xen gxm_dragProtocolStyle(Xen ptr)
{
  XM_field_assert_type(Xen_is_XmTopLevelEnterCallbackStruct(ptr), ptr, 1, "dragProtocolStyle", "XmTopLevelEnterCallbackStruct");
  return(C_int_to_Xen_integer((int)((Xen_to_C_XmTopLevelEnterCallbackStruct(ptr))->dragProtocolStyle)));
}

static Xen gxm_traversal_destination(Xen ptr)
{
  XM_field_assert_type(Xen_is_XmTraverseObscuredCallbackStruct(ptr), ptr, 1, "traversal_destination", "XmTraverseObscuredCallbackStruct");
  return(C_to_Xen_Widget((Widget)((Xen_to_C_XmTraverseObscuredCallbackStruct(ptr))->traversal_destination)));
}

static Xen gxm_minor_tab_widget(Xen ptr)
{
  XM_field_assert_type(Xen_is_XmNotebookPageInfo(ptr), ptr, 1, "minor_tab_widget", "XmNotebookPageInfo");
  return(C_to_Xen_Widget((Widget)((Xen_to_C_XmNotebookPageInfo(ptr))->minor_tab_widget)));
}

static Xen gxm_major_tab_widget(Xen ptr)
{
  XM_field_assert_type(Xen_is_XmNotebookPageInfo(ptr), ptr, 1, "major_tab_widget", "XmNotebookPageInfo");
  return(C_to_Xen_Widget((Widget)((Xen_to_C_XmNotebookPageInfo(ptr))->major_tab_widget)));
}

static Xen gxm_status_area_widget(Xen ptr)
{
  XM_field_assert_type(Xen_is_XmNotebookPageInfo(ptr), ptr, 1, "status_area_widget", "XmNotebookPageInfo");
  return(C_to_Xen_Widget((Widget)((Xen_to_C_XmNotebookPageInfo(ptr))->status_area_widget)));
}

static Xen gxm_page_widget(Xen ptr)
{
  XM_field_assert_type(Xen_is_XmNotebookCallbackStruct(ptr) || Xen_is_XmNotebookPageInfo(ptr), ptr, 1, "page_widget", "a struct with a page_widget field");
  if (Xen_is_XmNotebookCallbackStruct(ptr)) return(C_to_Xen_Widget((Widget)((Xen_to_C_XmNotebookCallbackStruct(ptr))->page_widget)));
  return(C_to_Xen_Widget((Widget)((Xen_to_C_XmNotebookPageInfo(ptr))->page_widget)));
}

static Xen gxm_page_number(Xen ptr)
{
  XM_field_assert_type(Xen_is_XmNotebookCallbackStruct(ptr) || Xen_is_XmNotebookPageInfo(ptr), ptr, 1, "page_number", "a struct with a page_number field");
  if (Xen_is_XmNotebookCallbackStruct(ptr)) return(C_int_to_Xen_integer((int)((Xen_to_C_XmNotebookCallbackStruct(ptr))->page_number)));
  return(C_int_to_Xen_integer((int)((Xen_to_C_XmNotebookPageInfo(ptr))->page_number)));
}


Xen_wrap_3_args(gxm_XtSetArg_w, gxm_XtSetArg)
Xen_wrap_2_optional_args(gxm_XtManageChildren_w, gxm_XtManageChildren)
Xen_wrap_1_arg(gxm_XtManageChild_w, gxm_XtManageChild)
Xen_wrap_2_optional_args(gxm_XtUnmanageChildren_w, gxm_XtUnmanageChildren)
Xen_wrap_1_arg(gxm_XtUnmanageChild_w, gxm_XtUnmanageChild)
Xen_wrap_1_arg(gxm_XtDispatchEvent_w, gxm_XtDispatchEvent)
Xen_wrap_2_args(gxm_XtCallAcceptFocus_w, gxm_XtCallAcceptFocus)
Xen_wrap_1_arg(gxm_XtAppPeekEvent_w, gxm_XtAppPeekEvent)
Xen_wrap_2_args(gxm_XtIsSubclass_w, gxm_XtIsSubclass)
Xen_wrap_2_args(gxm_XtAppSetFallbackResources_w, gxm_XtAppSetFallbackResources)
Xen_wrap_1_arg(gxm_XtIsObject_w, gxm_XtIsObject)
Xen_wrap_1_arg(gxm_XtIsManaged_w, gxm_XtIsManaged)
Xen_wrap_1_arg(gxm_XtIsRealized_w, gxm_XtIsRealized)
Xen_wrap_1_arg(gxm_XtIsSensitive_w, gxm_XtIsSensitive)
Xen_wrap_6_args(gxm_XtOwnSelection_w, gxm_XtOwnSelection)
Xen_wrap_8_args(gxm_XtOwnSelectionIncremental_w, gxm_XtOwnSelectionIncremental)
Xen_wrap_3_args(gxm_XtMakeResizeRequest_w, gxm_XtMakeResizeRequest)
Xen_wrap_3_args(gxm_XtTranslateCoords_w, gxm_XtTranslateCoords)
Xen_wrap_2_args(gxm_XtKeysymToKeycodeList_w, gxm_XtKeysymToKeycodeList)
Xen_wrap_1_arg(gxm_XtParseTranslationTable_w, gxm_XtParseTranslationTable)
Xen_wrap_1_arg(gxm_XtParseAcceleratorTable_w, gxm_XtParseAcceleratorTable)
Xen_wrap_2_args(gxm_XtOverrideTranslations_w, gxm_XtOverrideTranslations)
Xen_wrap_2_args(gxm_XtAugmentTranslations_w, gxm_XtAugmentTranslations)
Xen_wrap_2_args(gxm_XtInstallAccelerators_w, gxm_XtInstallAccelerators)
Xen_wrap_2_args(gxm_XtInstallAllAccelerators_w, gxm_XtInstallAllAccelerators)
Xen_wrap_1_arg(gxm_XtUninstallTranslations_w, gxm_XtUninstallTranslations)
Xen_wrap_2_args(gxm_XtAppAddActions_w, gxm_XtAppAddActions)
Xen_wrap_3_optional_args(gxm_XtAppAddActionHook_w, gxm_XtAppAddActionHook)
Xen_wrap_1_arg(gxm_XtRemoveActionHook_w, gxm_XtRemoveActionHook)
Xen_wrap_1_arg(gxm_XtGetActionList_w, gxm_XtGetActionList)
Xen_wrap_5_optional_args(gxm_XtCallActionProc_w, gxm_XtCallActionProc)
Xen_wrap_5_args(gxm_XtRegisterGrabAction_w, gxm_XtRegisterGrabAction)
Xen_wrap_2_args(gxm_XtSetMultiClickTime_w, gxm_XtSetMultiClickTime)
Xen_wrap_1_arg(gxm_XtGetMultiClickTime_w, gxm_XtGetMultiClickTime)
Xen_wrap_1_arg(gxm_XtGetResourceList_w, gxm_XtGetResourceList)
Xen_wrap_1_arg(gxm_XtGetActionKeysym_w, gxm_XtGetActionKeysym)
Xen_wrap_3_args(gxm_XtTranslateKeycode_w, gxm_XtTranslateKeycode)
Xen_wrap_3_args(gxm_XtTranslateKey_w, gxm_XtTranslateKey)
Xen_wrap_2_args(gxm_XtSetKeyTranslator_w, gxm_XtSetKeyTranslator)
Xen_wrap_4_args(gxm_XtRegisterCaseConverter_w, gxm_XtRegisterCaseConverter)
Xen_wrap_2_args(gxm_XtConvertCase_w, gxm_XtConvertCase)
Xen_wrap_5_optional_args(gxm_XtAddEventHandler_w, gxm_XtAddEventHandler)
Xen_wrap_5_args(gxm_XtRemoveEventHandler_w, gxm_XtRemoveEventHandler)
Xen_wrap_5_args(gxm_XtAddRawEventHandler_w, gxm_XtAddRawEventHandler)
Xen_wrap_5_args(gxm_XtRemoveRawEventHandler_w, gxm_XtRemoveRawEventHandler)
Xen_wrap_6_args(gxm_XtInsertEventHandler_w, gxm_XtInsertEventHandler)
Xen_wrap_6_args(gxm_XtInsertRawEventHandler_w, gxm_XtInsertRawEventHandler)
Xen_wrap_2_args(gxm_XtDispatchEventToWidget_w, gxm_XtDispatchEventToWidget)
Xen_wrap_1_arg(gxm_XtBuildEventMask_w, gxm_XtBuildEventMask)
Xen_wrap_3_args(gxm_XtAddGrab_w, gxm_XtAddGrab)
Xen_wrap_1_arg(gxm_XtRemoveGrab_w, gxm_XtRemoveGrab)
Xen_wrap_2_args(gxm_XtAppProcessEvent_w, gxm_XtAppProcessEvent)
Xen_wrap_1_arg(gxm_XtAppMainLoop_w, gxm_XtAppMainLoop)
Xen_wrap_2_args(gxm_XtAddExposureToRegion_w, gxm_XtAddExposureToRegion)
Xen_wrap_2_args(gxm_XtSetKeyboardFocus_w, gxm_XtSetKeyboardFocus)
Xen_wrap_1_arg(gxm_XtGetKeyboardFocusWidget_w, gxm_XtGetKeyboardFocusWidget)
Xen_wrap_1_arg(gxm_XtLastEventProcessed_w, gxm_XtLastEventProcessed)
Xen_wrap_1_arg(gxm_XtLastTimestampProcessed_w, gxm_XtLastTimestampProcessed)
Xen_wrap_4_optional_args(gxm_XtAppAddTimeOut_w, gxm_XtAppAddTimeOut)
Xen_wrap_1_arg(gxm_XtRemoveTimeOut_w, gxm_XtRemoveTimeOut)
Xen_wrap_5_optional_args(gxm_XtAppAddInput_w, gxm_XtAppAddInput)
Xen_wrap_1_arg(gxm_XtRemoveInput_w, gxm_XtRemoveInput)
Xen_wrap_1_arg(gxm_XtAppNextEvent_w, gxm_XtAppNextEvent)
Xen_wrap_1_arg(gxm_XtAppPending_w, gxm_XtAppPending)
Xen_wrap_1_arg(gxm_XtRealizeWidget_w, gxm_XtRealizeWidget)
Xen_wrap_1_arg(gxm_XtUnrealizeWidget_w, gxm_XtUnrealizeWidget)
Xen_wrap_1_arg(gxm_XtDestroyWidget_w, gxm_XtDestroyWidget)
Xen_wrap_2_args(gxm_XtSetSensitive_w, gxm_XtSetSensitive)
Xen_wrap_2_args(gxm_XtNameToWidget_w, gxm_XtNameToWidget)
Xen_wrap_2_args(gxm_XtWindowToWidget_w, gxm_XtWindowToWidget)
Xen_wrap_4_args(gxm_XtMergeArgLists_w, gxm_XtMergeArgLists)
Xen_wrap_2_args(gxm_XtVaCreateArgsList_w, gxm_XtVaCreateArgsList)
Xen_wrap_1_arg(gxm_XtDisplay_w, gxm_XtDisplay)
Xen_wrap_1_arg(gxm_XtDisplayOfObject_w, gxm_XtDisplayOfObject)
Xen_wrap_1_arg(gxm_XtScreen_w, gxm_XtScreen)
Xen_wrap_1_arg(gxm_XtScreenOfObject_w, gxm_XtScreenOfObject)
Xen_wrap_1_arg(gxm_XtWindow_w, gxm_XtWindow)
Xen_wrap_1_arg(gxm_XtWindowOfObject_w, gxm_XtWindowOfObject)
Xen_wrap_1_arg(gxm_XtName_w, gxm_XtName)
Xen_wrap_1_arg(gxm_XtSuperclass_w, gxm_XtSuperclass)
Xen_wrap_1_arg(gxm_XtClass_w, gxm_XtClass)
Xen_wrap_1_arg(gxm_XtParent_w, gxm_XtParent)
Xen_wrap_4_optional_args(gxm_XtAddCallback_w, gxm_XtAddCallback)
Xen_wrap_3_args(gxm_XtRemoveCallback_w, gxm_XtRemoveCallback)
Xen_wrap_3_args(gxm_XtAddCallbacks_w, gxm_XtAddCallbacks)
Xen_wrap_3_args(gxm_XtRemoveCallbacks_w, gxm_XtRemoveCallbacks)
Xen_wrap_2_args(gxm_XtRemoveAllCallbacks_w, gxm_XtRemoveAllCallbacks)
Xen_wrap_3_args(gxm_XtCallCallbacks_w, gxm_XtCallCallbacks)
Xen_wrap_2_args(gxm_XtHasCallbacks_w, gxm_XtHasCallbacks)
Xen_wrap_5_optional_args(gxm_XtCreatePopupShell_w, gxm_XtCreatePopupShell)
Xen_wrap_4_args(gxm_XtVaCreatePopupShell_w, gxm_XtVaCreatePopupShell)
Xen_wrap_2_args(gxm_XtPopup_w, gxm_XtPopup)
Xen_wrap_1_arg(gxm_XtPopupSpringLoaded_w, gxm_XtPopupSpringLoaded)
Xen_wrap_3_args(gxm_XtCallbackNone_w, gxm_XtCallbackNone)
Xen_wrap_3_args(gxm_XtCallbackNonexclusive_w, gxm_XtCallbackNonexclusive)
Xen_wrap_3_args(gxm_XtCallbackExclusive_w, gxm_XtCallbackExclusive)
Xen_wrap_1_arg(gxm_XtPopdown_w, gxm_XtPopdown)
Xen_wrap_3_args(gxm_XtCallbackPopdown_w, gxm_XtCallbackPopdown)
Xen_wrap_5_optional_args(gxm_XtCreateWidget_w, gxm_XtCreateWidget)
Xen_wrap_5_optional_args(gxm_XtCreateManagedWidget_w, gxm_XtCreateManagedWidget)
Xen_wrap_4_args(gxm_XtVaCreateWidget_w, gxm_XtVaCreateWidget)
Xen_wrap_4_args(gxm_XtVaCreateManagedWidget_w, gxm_XtVaCreateManagedWidget)
Xen_wrap_6_optional_args(gxm_XtAppCreateShell_w, gxm_XtAppCreateShell)
Xen_wrap_5_args(gxm_XtVaAppCreateShell_w, gxm_XtVaAppCreateShell)
Xen_wrap_no_args(gxm_XtToolkitInitialize_w, gxm_XtToolkitInitialize)
Xen_wrap_3_args(gxm_XtSetLanguageProc_w, gxm_XtSetLanguageProc)
Xen_wrap_6_args(gxm_XtDisplayInitialize_w, gxm_XtDisplayInitialize)
Xen_wrap_6_optional_args(gxm_XtOpenApplication_w, gxm_XtOpenApplication)
Xen_wrap_6_optional_args(gxm_XtVaOpenApplication_w, gxm_XtVaOpenApplication)
Xen_wrap_5_optional_args(gxm_XtAppInitialize_w, gxm_XtAppInitialize)
Xen_wrap_5_optional_args(gxm_XtVaAppInitialize_w, gxm_XtVaAppInitialize)
Xen_wrap_6_args(gxm_XtOpenDisplay_w, gxm_XtOpenDisplay)
Xen_wrap_no_args(gxm_XtCreateApplicationContext_w, gxm_XtCreateApplicationContext)
Xen_wrap_1_arg(gxm_XtDestroyApplicationContext_w, gxm_XtDestroyApplicationContext)
Xen_wrap_1_arg(gxm_XtInitializeWidgetClass_w, gxm_XtInitializeWidgetClass)
Xen_wrap_1_arg(gxm_XtWidgetToApplicationContext_w, gxm_XtWidgetToApplicationContext)
Xen_wrap_1_arg(gxm_XtDisplayToApplicationContext_w, gxm_XtDisplayToApplicationContext)
Xen_wrap_1_arg(gxm_XtCloseDisplay_w, gxm_XtCloseDisplay)
Xen_wrap_3_optional_args(gxm_XtSetValues_w, gxm_XtSetValues)
Xen_wrap_2_args(gxm_XtVaSetValues_w, gxm_XtVaSetValues)
Xen_wrap_3_optional_args(gxm_XtGetValues_w, gxm_XtGetValues)
Xen_wrap_2_args(gxm_XtVaGetValues_w, gxm_XtVaGetValues)
Xen_wrap_2_args(gxm_XtAppSetErrorMsgHandler_w, gxm_XtAppSetErrorMsgHandler)
Xen_wrap_2_args(gxm_XtAppSetWarningMsgHandler_w, gxm_XtAppSetWarningMsgHandler)
Xen_wrap_7_args(gxm_XtAppErrorMsg_w, gxm_XtAppErrorMsg)
Xen_wrap_7_args(gxm_XtAppWarningMsg_w, gxm_XtAppWarningMsg)
Xen_wrap_2_args(gxm_XtAppSetErrorHandler_w, gxm_XtAppSetErrorHandler)
Xen_wrap_2_args(gxm_XtAppSetWarningHandler_w, gxm_XtAppSetWarningHandler)
Xen_wrap_2_args(gxm_XtAppError_w, gxm_XtAppError)
Xen_wrap_1_arg(gxm_XtMalloc_w, gxm_XtMalloc)
Xen_wrap_2_args(gxm_XtCalloc_w, gxm_XtCalloc)
Xen_wrap_2_args(gxm_XtRealloc_w, gxm_XtRealloc)
Xen_wrap_1_arg(gxm_XtFree_w, gxm_XtFree)
Xen_wrap_3_optional_args(gxm_XtAppAddWorkProc_w, gxm_XtAppAddWorkProc)
Xen_wrap_1_arg(gxm_XtRemoveWorkProc_w, gxm_XtRemoveWorkProc)
Xen_wrap_3_args(gxm_XtGetGC_w, gxm_XtGetGC)
Xen_wrap_6_args(gxm_XtAllocateGC_w, gxm_XtAllocateGC)
Xen_wrap_1_arg(gxm_XtDestroyGC_w, gxm_XtDestroyGC)
Xen_wrap_2_args(gxm_XtReleaseGC_w, gxm_XtReleaseGC)
Xen_wrap_4_args(gxm_XtFindFile_w, gxm_XtFindFile)
Xen_wrap_8_args(gxm_XtResolvePathname_w, gxm_XtResolvePathname)
Xen_wrap_3_args(gxm_XtDisownSelection_w, gxm_XtDisownSelection)
Xen_wrap_6_args(gxm_XtGetSelectionValue_w, gxm_XtGetSelectionValue)
Xen_wrap_7_args(gxm_XtGetSelectionValues_w, gxm_XtGetSelectionValues)
Xen_wrap_2_args(gxm_XtAppSetSelectionTimeout_w, gxm_XtAppSetSelectionTimeout)
Xen_wrap_1_arg(gxm_XtAppGetSelectionTimeout_w, gxm_XtAppGetSelectionTimeout)
Xen_wrap_3_args(gxm_XtGetSelectionRequest_w, gxm_XtGetSelectionRequest)
Xen_wrap_6_args(gxm_XtGetSelectionValueIncremental_w, gxm_XtGetSelectionValueIncremental)
Xen_wrap_7_args(gxm_XtGetSelectionValuesIncremental_w, gxm_XtGetSelectionValuesIncremental)
Xen_wrap_2_args(gxm_XtCreateSelectionRequest_w, gxm_XtCreateSelectionRequest)
Xen_wrap_3_args(gxm_XtSendSelectionRequest_w, gxm_XtSendSelectionRequest)
Xen_wrap_2_args(gxm_XtCancelSelectionRequest_w, gxm_XtCancelSelectionRequest)
Xen_wrap_6_args(gxm_XtGrabKey_w, gxm_XtGrabKey)
Xen_wrap_3_args(gxm_XtUngrabKey_w, gxm_XtUngrabKey)
Xen_wrap_5_args(gxm_XtGrabKeyboard_w, gxm_XtGrabKeyboard)
Xen_wrap_2_args(gxm_XtUngrabKeyboard_w, gxm_XtUngrabKeyboard)
Xen_wrap_9_args(gxm_XtGrabButton_w, gxm_XtGrabButton)
Xen_wrap_3_args(gxm_XtUngrabButton_w, gxm_XtUngrabButton)
Xen_wrap_8_args(gxm_XtGrabPointer_w, gxm_XtGrabPointer)
Xen_wrap_2_args(gxm_XtUngrabPointer_w, gxm_XtUngrabPointer)
Xen_wrap_1_arg(gxm_XtGetApplicationNameAndClass_w, gxm_XtGetApplicationNameAndClass)
Xen_wrap_1_arg(gxm_XtGetDisplays_w, gxm_XtGetDisplays)
Xen_wrap_no_args(gxm_XtToolkitThreadInitialize_w, gxm_XtToolkitThreadInitialize)
Xen_wrap_1_arg(gxm_XtAppLock_w, gxm_XtAppLock)
Xen_wrap_1_arg(gxm_XtAppUnlock_w, gxm_XtAppUnlock)
Xen_wrap_1_arg(gxm_XtIsRectObj_w, gxm_XtIsRectObj)
Xen_wrap_1_arg(gxm_XtIsWidget_w, gxm_XtIsWidget)
Xen_wrap_1_arg(gxm_XtIsComposite_w, gxm_XtIsComposite)
Xen_wrap_1_arg(gxm_XtIsConstraint_w, gxm_XtIsConstraint)
Xen_wrap_1_arg(gxm_XtIsShell_w, gxm_XtIsShell)
Xen_wrap_1_arg(gxm_XtIsOverrideShell_w, gxm_XtIsOverrideShell)
Xen_wrap_1_arg(gxm_XtIsWMShell_w, gxm_XtIsWMShell)
Xen_wrap_1_arg(gxm_XtIsVendorShell_w, gxm_XtIsVendorShell)
Xen_wrap_1_arg(gxm_XtIsTransientShell_w, gxm_XtIsTransientShell)
Xen_wrap_1_arg(gxm_XtIsTopLevelShell_w, gxm_XtIsTopLevelShell)
Xen_wrap_1_arg(gxm_XtIsApplicationShell_w, gxm_XtIsApplicationShell)
Xen_wrap_1_arg(gxm_XtIsSessionShell_w, gxm_XtIsSessionShell)
Xen_wrap_1_arg(gxm_XtMapWidget_w, gxm_XtMapWidget)
Xen_wrap_1_arg(gxm_XtUnmapWidget_w, gxm_XtUnmapWidget)

Xen_wrap_2_args(gxm_XLoadQueryFont_w, gxm_XLoadQueryFont)
Xen_wrap_2_args(gxm_XQueryFont_w, gxm_XQueryFont)
Xen_wrap_4_args(gxm_XGetMotionEvents_w, gxm_XGetMotionEvents)
Xen_wrap_3_args(gxm_XDeleteModifiermapEntry_w, gxm_XDeleteModifiermapEntry)
Xen_wrap_1_arg(gxm_XGetModifierMapping_w, gxm_XGetModifierMapping)
Xen_wrap_3_args(gxm_XInsertModifiermapEntry_w, gxm_XInsertModifiermapEntry)
Xen_wrap_1_arg(gxm_XNewModifiermap_w, gxm_XNewModifiermap)
Xen_wrap_any_args(gxm_XCreateImage_w, gxm_XCreateImage)
Xen_wrap_8_args(gxm_XGetImage_w, gxm_XGetImage)
Xen_wrap_any_args(gxm_XGetSubImage_w, gxm_XGetSubImage)
Xen_wrap_1_arg(gxm_XOpenDisplay_w, gxm_XOpenDisplay)
Xen_wrap_1_arg(gxm_XFetchBytes_w, gxm_XFetchBytes)
Xen_wrap_2_args(gxm_XFetchBuffer_w, gxm_XFetchBuffer)
Xen_wrap_2_args(gxm_XGetAtomName_w, gxm_XGetAtomName)
Xen_wrap_1_arg(gxm_XDisplayName_w, gxm_XDisplayName)
Xen_wrap_1_arg(gxm_XKeysymToString_w, gxm_XKeysymToString)
Xen_wrap_2_args(gxm_XSynchronize_w, gxm_XSynchronize)
Xen_wrap_2_args(gxm_XSetAfterFunction_w, gxm_XSetAfterFunction)
Xen_wrap_3_args(gxm_XInternAtom_w, gxm_XInternAtom)
Xen_wrap_2_args(gxm_XCopyColormapAndFree_w, gxm_XCopyColormapAndFree)
Xen_wrap_4_args(gxm_XCreateColormap_w, gxm_XCreateColormap)
Xen_wrap_7_args(gxm_XCreatePixmapCursor_w, gxm_XCreatePixmapCursor)
Xen_wrap_7_args(gxm_XCreateGlyphCursor_w, gxm_XCreateGlyphCursor)
Xen_wrap_2_args(gxm_XCreateFontCursor_w, gxm_XCreateFontCursor)
Xen_wrap_2_args(gxm_XLoadFont_w, gxm_XLoadFont)
Xen_wrap_4_args(gxm_XCreateGC_w, gxm_XCreateGC)
Xen_wrap_2_args(gxm_XFlushGC_w, gxm_XFlushGC)
Xen_wrap_5_args(gxm_XCreatePixmap_w, gxm_XCreatePixmap)
Xen_wrap_5_args(gxm_XCreateBitmapFromData_w, gxm_XCreateBitmapFromData)
Xen_wrap_8_args(gxm_XCreatePixmapFromBitmapData_w, gxm_XCreatePixmapFromBitmapData)
Xen_wrap_9_args(gxm_XCreateSimpleWindow_w, gxm_XCreateSimpleWindow)
Xen_wrap_2_args(gxm_XGetSelectionOwner_w, gxm_XGetSelectionOwner)
Xen_wrap_any_args(gxm_XCreateWindow_w, gxm_XCreateWindow)
Xen_wrap_2_args(gxm_XListInstalledColormaps_w, gxm_XListInstalledColormaps)
Xen_wrap_3_args(gxm_XListFonts_w, gxm_XListFonts)
Xen_wrap_3_args(gxm_XListFontsWithInfo_w, gxm_XListFontsWithInfo)
Xen_wrap_1_arg(gxm_XGetFontPath_w, gxm_XGetFontPath)
Xen_wrap_1_arg(gxm_XListExtensions_w, gxm_XListExtensions)
Xen_wrap_2_args(gxm_XListProperties_w, gxm_XListProperties)
#if 0
Xen_wrap_3_args(gxm_XKeycodeToKeysym_w, gxm_XKeycodeToKeysym)
#endif
Xen_wrap_2_args(gxm_XLookupKeysym_w, gxm_XLookupKeysym)
Xen_wrap_3_args(gxm_XGetKeyboardMapping_w, gxm_XGetKeyboardMapping)
Xen_wrap_1_arg(gxm_XStringToKeysym_w, gxm_XStringToKeysym)
Xen_wrap_1_arg(gxm_XMaxRequestSize_w, gxm_XMaxRequestSize)
Xen_wrap_1_arg(gxm_XExtendedMaxRequestSize_w, gxm_XExtendedMaxRequestSize)
Xen_wrap_1_arg(gxm_XDisplayMotionBufferSize_w, gxm_XDisplayMotionBufferSize)
Xen_wrap_1_arg(gxm_XVisualIDFromVisual_w, gxm_XVisualIDFromVisual)
Xen_wrap_2_args(gxm_XRootWindow_w, gxm_XRootWindow)
Xen_wrap_1_arg(gxm_XDefaultRootWindow_w, gxm_XDefaultRootWindow)
Xen_wrap_1_arg(gxm_XRootWindowOfScreen_w, gxm_XRootWindowOfScreen)
Xen_wrap_2_args(gxm_XDefaultVisual_w, gxm_XDefaultVisual)
Xen_wrap_1_arg(gxm_XDefaultVisualOfScreen_w, gxm_XDefaultVisualOfScreen)
Xen_wrap_2_args(gxm_XDefaultGC_w, gxm_XDefaultGC)
Xen_wrap_1_arg(gxm_XDefaultGCOfScreen_w, gxm_XDefaultGCOfScreen)
Xen_wrap_2_args(gxm_XBlackPixel_w, gxm_XBlackPixel)
Xen_wrap_2_args(gxm_XWhitePixel_w, gxm_XWhitePixel)
Xen_wrap_no_args(gxm_XAllPlanes_w, gxm_XAllPlanes)
Xen_wrap_1_arg(gxm_XBlackPixelOfScreen_w, gxm_XBlackPixelOfScreen)
Xen_wrap_1_arg(gxm_XWhitePixelOfScreen_w, gxm_XWhitePixelOfScreen)
Xen_wrap_1_arg(gxm_XNextRequest_w, gxm_XNextRequest)
Xen_wrap_1_arg(gxm_XLastKnownRequestProcessed_w, gxm_XLastKnownRequestProcessed)
Xen_wrap_1_arg(gxm_XServerVendor_w, gxm_XServerVendor)
Xen_wrap_1_arg(gxm_XDisplayString_w, gxm_XDisplayString)
Xen_wrap_2_args(gxm_XDefaultColormap_w, gxm_XDefaultColormap)
Xen_wrap_1_arg(gxm_XDefaultColormapOfScreen_w, gxm_XDefaultColormapOfScreen)
Xen_wrap_1_arg(gxm_XDisplayOfScreen_w, gxm_XDisplayOfScreen)
Xen_wrap_2_args(gxm_XScreenOfDisplay_w, gxm_XScreenOfDisplay)
Xen_wrap_1_arg(gxm_XDefaultScreenOfDisplay_w, gxm_XDefaultScreenOfDisplay)
Xen_wrap_1_arg(gxm_XEventMaskOfScreen_w, gxm_XEventMaskOfScreen)
Xen_wrap_1_arg(gxm_XScreenNumberOfScreen_w, gxm_XScreenNumberOfScreen)
Xen_wrap_1_arg(gxm_XSetErrorHandler_w, gxm_XSetErrorHandler)
Xen_wrap_1_arg(gxm_XSetIOErrorHandler_w, gxm_XSetIOErrorHandler)
Xen_wrap_1_arg(gxm_XListPixmapFormats_w, gxm_XListPixmapFormats)
Xen_wrap_2_args(gxm_XListDepths_w, gxm_XListDepths)
Xen_wrap_5_args(gxm_XReconfigureWMWindow_w, gxm_XReconfigureWMWindow)
Xen_wrap_2_args(gxm_XGetWMProtocols_w, gxm_XGetWMProtocols)
Xen_wrap_4_args(gxm_XSetWMProtocols_w, gxm_XSetWMProtocols)
Xen_wrap_3_args(gxm_XIconifyWindow_w, gxm_XIconifyWindow)
Xen_wrap_3_args(gxm_XWithdrawWindow_w, gxm_XWithdrawWindow)
Xen_wrap_2_args(gxm_XGetCommand_w, gxm_XGetCommand)
Xen_wrap_2_args(gxm_XGetWMColormapWindows_w, gxm_XGetWMColormapWindows)
Xen_wrap_4_args(gxm_XSetWMColormapWindows_w, gxm_XSetWMColormapWindows)
Xen_wrap_3_args(gxm_XSetTransientForHint_w, gxm_XSetTransientForHint)
Xen_wrap_1_arg(gxm_XActivateScreenSaver_w, gxm_XActivateScreenSaver)
Xen_wrap_3_args(gxm_XAllocColor_w, gxm_XAllocColor)
Xen_wrap_5_args(gxm_XAllocColorCells_w, gxm_XAllocColorCells)
Xen_wrap_any_args(gxm_XAllocColorPlanes_w, gxm_XAllocColorPlanes)
Xen_wrap_5_args(gxm_XAllocNamedColor_w, gxm_XAllocNamedColor)
Xen_wrap_3_args(gxm_XAllowEvents_w, gxm_XAllowEvents)
Xen_wrap_1_arg(gxm_XAutoRepeatOff_w, gxm_XAutoRepeatOff)
Xen_wrap_1_arg(gxm_XAutoRepeatOn_w, gxm_XAutoRepeatOn)
Xen_wrap_2_args(gxm_XBell_w, gxm_XBell)
Xen_wrap_1_arg(gxm_XBitmapBitOrder_w, gxm_XBitmapBitOrder)
Xen_wrap_1_arg(gxm_XBitmapPad_w, gxm_XBitmapPad)
Xen_wrap_1_arg(gxm_XBitmapUnit_w, gxm_XBitmapUnit)
Xen_wrap_1_arg(gxm_XCellsOfScreen_w, gxm_XCellsOfScreen)
Xen_wrap_4_args(gxm_XChangeActivePointerGrab_w, gxm_XChangeActivePointerGrab)
Xen_wrap_4_args(gxm_XChangeGC_w, gxm_XChangeGC)
Xen_wrap_3_args(gxm_XChangeKeyboardControl_w, gxm_XChangeKeyboardControl)
Xen_wrap_5_args(gxm_XChangeKeyboardMapping_w, gxm_XChangeKeyboardMapping)
Xen_wrap_6_args(gxm_XChangePointerControl_w, gxm_XChangePointerControl)
Xen_wrap_8_optional_args(gxm_XChangeProperty_w, gxm_XChangeProperty)
Xen_wrap_4_args(gxm_XChangeWindowAttributes_w, gxm_XChangeWindowAttributes)
Xen_wrap_3_args(gxm_XCheckIfEvent_w, gxm_XCheckIfEvent)
Xen_wrap_2_args(gxm_XCheckMaskEvent_w, gxm_XCheckMaskEvent)
Xen_wrap_2_args(gxm_XCheckTypedEvent_w, gxm_XCheckTypedEvent)
Xen_wrap_3_args(gxm_XCheckTypedWindowEvent_w, gxm_XCheckTypedWindowEvent)
Xen_wrap_3_args(gxm_XCheckWindowEvent_w, gxm_XCheckWindowEvent)
Xen_wrap_3_args(gxm_XCirculateSubwindows_w, gxm_XCirculateSubwindows)
Xen_wrap_2_args(gxm_XCirculateSubwindowsDown_w, gxm_XCirculateSubwindowsDown)
Xen_wrap_2_args(gxm_XCirculateSubwindowsUp_w, gxm_XCirculateSubwindowsUp)
Xen_wrap_7_args(gxm_XClearArea_w, gxm_XClearArea)
Xen_wrap_2_args(gxm_XClearWindow_w, gxm_XClearWindow)
Xen_wrap_1_arg(gxm_XCloseDisplay_w, gxm_XCloseDisplay)
Xen_wrap_4_args(gxm_XConfigureWindow_w, gxm_XConfigureWindow)
Xen_wrap_1_arg(gxm_XConnectionNumber_w, gxm_XConnectionNumber)
Xen_wrap_6_args(gxm_XConvertSelection_w, gxm_XConvertSelection)
Xen_wrap_any_args(gxm_XCopyArea_w, gxm_XCopyArea)
Xen_wrap_4_args(gxm_XCopyGC_w, gxm_XCopyGC)
Xen_wrap_any_args(gxm_XCopyPlane_w, gxm_XCopyPlane)
Xen_wrap_2_args(gxm_XDefaultDepth_w, gxm_XDefaultDepth)
Xen_wrap_1_arg(gxm_XDefaultDepthOfScreen_w, gxm_XDefaultDepthOfScreen)
Xen_wrap_1_arg(gxm_XDefaultScreen_w, gxm_XDefaultScreen)
Xen_wrap_3_args(gxm_XDefineCursor_w, gxm_XDefineCursor)
Xen_wrap_3_args(gxm_XDeleteProperty_w, gxm_XDeleteProperty)
Xen_wrap_2_args(gxm_XDestroyWindow_w, gxm_XDestroyWindow)
Xen_wrap_2_args(gxm_XDestroySubwindows_w, gxm_XDestroySubwindows)
Xen_wrap_1_arg(gxm_XDoesBackingStore_w, gxm_XDoesBackingStore)
Xen_wrap_1_arg(gxm_XDoesSaveUnders_w, gxm_XDoesSaveUnders)
Xen_wrap_1_arg(gxm_XDisableAccessControl_w, gxm_XDisableAccessControl)
Xen_wrap_2_args(gxm_XDisplayCells_w, gxm_XDisplayCells)
Xen_wrap_2_args(gxm_XDisplayHeight_w, gxm_XDisplayHeight)
Xen_wrap_2_args(gxm_XDisplayHeightMM_w, gxm_XDisplayHeightMM)
Xen_wrap_1_arg(gxm_XDisplayKeycodes_w, gxm_XDisplayKeycodes)
Xen_wrap_2_args(gxm_XDisplayPlanes_w, gxm_XDisplayPlanes)
Xen_wrap_2_args(gxm_XDisplayWidth_w, gxm_XDisplayWidth)
Xen_wrap_2_args(gxm_XDisplayWidthMM_w, gxm_XDisplayWidthMM)
Xen_wrap_9_args(gxm_XDrawArc_w, gxm_XDrawArc)
Xen_wrap_5_args(gxm_XDrawArcs_w, gxm_XDrawArcs)
Xen_wrap_7_args(gxm_XDrawImageString_w, gxm_XDrawImageString)
Xen_wrap_7_args(gxm_XDrawLine_w, gxm_XDrawLine)
Xen_wrap_6_args(gxm_XDrawLines_w, gxm_XDrawLines)
Xen_wrap_6_args(gxm_XDrawLinesDirect_w, gxm_XDrawLinesDirect)
Xen_wrap_1_arg(gxm_FreeXPoints_w, gxm_FreeXPoints)
Xen_wrap_1_arg(gxm_Vector2XPoints_w, gxm_Vector2XPoints)
Xen_wrap_5_args(gxm_XDrawPoint_w, gxm_XDrawPoint)
Xen_wrap_6_args(gxm_XDrawPoints_w, gxm_XDrawPoints)
Xen_wrap_7_args(gxm_XDrawRectangle_w, gxm_XDrawRectangle)
Xen_wrap_5_args(gxm_XDrawRectangles_w, gxm_XDrawRectangles)
Xen_wrap_5_args(gxm_XDrawSegments_w, gxm_XDrawSegments)
Xen_wrap_7_args(gxm_XDrawString_w, gxm_XDrawString)
Xen_wrap_7_optional_args(gxm_XDrawText_w, gxm_XDrawText)
Xen_wrap_1_arg(gxm_XEnableAccessControl_w, gxm_XEnableAccessControl)
Xen_wrap_2_args(gxm_XEventsQueued_w, gxm_XEventsQueued)
Xen_wrap_2_args(gxm_XFetchName_w, gxm_XFetchName)
Xen_wrap_9_args(gxm_XFillArc_w, gxm_XFillArc)
Xen_wrap_5_args(gxm_XFillArcs_w, gxm_XFillArcs)
Xen_wrap_7_args(gxm_XFillPolygon_w, gxm_XFillPolygon)
Xen_wrap_7_args(gxm_XFillRectangle_w, gxm_XFillRectangle)
Xen_wrap_5_args(gxm_XFillRectangles_w, gxm_XFillRectangles)
Xen_wrap_1_arg(gxm_XFlush_w, gxm_XFlush)
Xen_wrap_2_args(gxm_XForceScreenSaver_w, gxm_XForceScreenSaver)
Xen_wrap_1_arg(gxm_XFree_w, gxm_XFree)
Xen_wrap_2_args(gxm_XFreeColormap_w, gxm_XFreeColormap)
Xen_wrap_5_args(gxm_XFreeColors_w, gxm_XFreeColors)
Xen_wrap_2_args(gxm_XFreeCursor_w, gxm_XFreeCursor)
Xen_wrap_1_arg(gxm_XFreeExtensionList_w, gxm_XFreeExtensionList)
Xen_wrap_2_args(gxm_XFreeFont_w, gxm_XFreeFont)
Xen_wrap_3_args(gxm_XFreeFontInfo_w, gxm_XFreeFontInfo)
Xen_wrap_1_arg(gxm_XFreeFontNames_w, gxm_XFreeFontNames)
Xen_wrap_1_arg(gxm_XFreeFontPath_w, gxm_XFreeFontPath)
Xen_wrap_2_args(gxm_XFreeGC_w, gxm_XFreeGC)
Xen_wrap_1_arg(gxm_XFreeModifiermap_w, gxm_XFreeModifiermap)
Xen_wrap_2_args(gxm_XFreePixmap_w, gxm_XFreePixmap)
Xen_wrap_any_args(gxm_XGeometry_w, gxm_XGeometry)
Xen_wrap_4_args(gxm_XGetErrorText_w, gxm_XGetErrorText)
Xen_wrap_2_args(gxm_XGetFontProperty_w, gxm_XGetFontProperty)
Xen_wrap_3_args(gxm_XGetGCValues_w, gxm_XGetGCValues)
Xen_wrap_no_args(gxm_XGCValues_w, gxm_XGCValues)
Xen_wrap_1_optional_arg(gxm_XEvent_w, gxm_XEvent)
Xen_wrap_2_args(gxm_XGetGeometry_w, gxm_XGetGeometry)
Xen_wrap_2_args(gxm_XGetIconName_w, gxm_XGetIconName)
Xen_wrap_1_arg(gxm_XGetInputFocus_w, gxm_XGetInputFocus)
Xen_wrap_1_arg(gxm_XGetKeyboardControl_w, gxm_XGetKeyboardControl)
Xen_wrap_1_arg(gxm_XGetPointerControl_w, gxm_XGetPointerControl)
Xen_wrap_3_args(gxm_XGetPointerMapping_w, gxm_XGetPointerMapping)
Xen_wrap_1_arg(gxm_XGetScreenSaver_w, gxm_XGetScreenSaver)
Xen_wrap_2_args(gxm_XGetTransientForHint_w, gxm_XGetTransientForHint)
Xen_wrap_any_args(gxm_XGetWindowProperty_w, gxm_XGetWindowProperty)
Xen_wrap_2_args(gxm_XGetWindowAttributes_w, gxm_XGetWindowAttributes)
Xen_wrap_any_args(gxm_XGrabButton_w, gxm_XGrabButton)
Xen_wrap_7_args(gxm_XGrabKey_w, gxm_XGrabKey)
Xen_wrap_6_args(gxm_XGrabKeyboard_w, gxm_XGrabKeyboard)
Xen_wrap_9_args(gxm_XGrabPointer_w, gxm_XGrabPointer)
Xen_wrap_1_arg(gxm_XGrabServer_w, gxm_XGrabServer)
Xen_wrap_1_arg(gxm_XHeightMMOfScreen_w, gxm_XHeightMMOfScreen)
Xen_wrap_1_arg(gxm_XHeightOfScreen_w, gxm_XHeightOfScreen)
Xen_wrap_3_args(gxm_XIfEvent_w, gxm_XIfEvent)
Xen_wrap_1_arg(gxm_XImageByteOrder_w, gxm_XImageByteOrder)
Xen_wrap_2_args(gxm_XInstallColormap_w, gxm_XInstallColormap)
Xen_wrap_2_args(gxm_XKeysymToKeycode_w, gxm_XKeysymToKeycode)
Xen_wrap_2_args(gxm_XKillClient_w, gxm_XKillClient)
Xen_wrap_5_args(gxm_XLookupColor_w, gxm_XLookupColor)
Xen_wrap_2_args(gxm_XLowerWindow_w, gxm_XLowerWindow)
Xen_wrap_2_args(gxm_XMapRaised_w, gxm_XMapRaised)
Xen_wrap_2_args(gxm_XMapSubwindows_w, gxm_XMapSubwindows)
Xen_wrap_2_args(gxm_XMapWindow_w, gxm_XMapWindow)
Xen_wrap_2_args(gxm_XMaskEvent_w, gxm_XMaskEvent)
Xen_wrap_1_arg(gxm_XMaxCmapsOfScreen_w, gxm_XMaxCmapsOfScreen)
Xen_wrap_1_arg(gxm_XMinCmapsOfScreen_w, gxm_XMinCmapsOfScreen)
Xen_wrap_6_args(gxm_XMoveResizeWindow_w, gxm_XMoveResizeWindow)
Xen_wrap_4_args(gxm_XMoveWindow_w, gxm_XMoveWindow)
Xen_wrap_1_arg(gxm_XNextEvent_w, gxm_XNextEvent)
Xen_wrap_1_arg(gxm_XNoOp_w, gxm_XNoOp)
Xen_wrap_4_args(gxm_XParseColor_w, gxm_XParseColor)
Xen_wrap_1_arg(gxm_XParseGeometry_w, gxm_XParseGeometry)
Xen_wrap_1_arg(gxm_XPeekEvent_w, gxm_XPeekEvent)
Xen_wrap_3_args(gxm_XPeekIfEvent_w, gxm_XPeekIfEvent)
Xen_wrap_1_arg(gxm_XPending_w, gxm_XPending)
Xen_wrap_1_arg(gxm_XPlanesOfScreen_w, gxm_XPlanesOfScreen)
Xen_wrap_1_arg(gxm_XProtocolRevision_w, gxm_XProtocolRevision)
Xen_wrap_1_arg(gxm_XProtocolVersion_w, gxm_XProtocolVersion)
Xen_wrap_2_args(gxm_XPutBackEvent_w, gxm_XPutBackEvent)
Xen_wrap_any_args(gxm_XPutImage_w, gxm_XPutImage)
Xen_wrap_1_arg(gxm_XQLength_w, gxm_XQLength)
Xen_wrap_4_args(gxm_XQueryBestCursor_w, gxm_XQueryBestCursor)
Xen_wrap_5_args(gxm_XQueryBestSize_w, gxm_XQueryBestSize)
Xen_wrap_4_args(gxm_XQueryBestStipple_w, gxm_XQueryBestStipple)
Xen_wrap_4_args(gxm_XQueryBestTile_w, gxm_XQueryBestTile)
Xen_wrap_3_args(gxm_XQueryColor_w, gxm_XQueryColor)
Xen_wrap_4_optional_args(gxm_XQueryColors_w, gxm_XQueryColors)
Xen_wrap_2_args(gxm_XQueryExtension_w, gxm_XQueryExtension)
Xen_wrap_1_arg(gxm_XQueryKeymap_w, gxm_XQueryKeymap)
Xen_wrap_2_args(gxm_XQueryPointer_w, gxm_XQueryPointer)
Xen_wrap_3_args(gxm_XQueryTextExtents_w, gxm_XQueryTextExtents)
Xen_wrap_2_args(gxm_XQueryTree_w, gxm_XQueryTree)
Xen_wrap_2_args(gxm_XRaiseWindow_w, gxm_XRaiseWindow)
Xen_wrap_3_args(gxm_XReadBitmapFile_w, gxm_XReadBitmapFile)
Xen_wrap_1_arg(gxm_XReadBitmapFileData_w, gxm_XReadBitmapFileData)
Xen_wrap_6_args(gxm_XRebindKeysym_w, gxm_XRebindKeysym)
Xen_wrap_4_args(gxm_XRecolorCursor_w, gxm_XRecolorCursor)
Xen_wrap_1_arg(gxm_XRefreshKeyboardMapping_w, gxm_XRefreshKeyboardMapping)
Xen_wrap_5_args(gxm_XReparentWindow_w, gxm_XReparentWindow)
Xen_wrap_1_arg(gxm_XResetScreenSaver_w, gxm_XResetScreenSaver)
Xen_wrap_4_args(gxm_XResizeWindow_w, gxm_XResizeWindow)
Xen_wrap_3_args(gxm_XRestackWindows_w, gxm_XRestackWindows)
Xen_wrap_2_args(gxm_XRotateBuffers_w, gxm_XRotateBuffers)
Xen_wrap_5_args(gxm_XRotateWindowProperties_w, gxm_XRotateWindowProperties)
Xen_wrap_1_arg(gxm_XScreenCount_w, gxm_XScreenCount)
Xen_wrap_3_args(gxm_XSelectInput_w, gxm_XSelectInput)
Xen_wrap_5_args(gxm_XSendEvent_w, gxm_XSendEvent)
Xen_wrap_2_args(gxm_XSetAccessControl_w, gxm_XSetAccessControl)
Xen_wrap_3_args(gxm_XSetArcMode_w, gxm_XSetArcMode)
Xen_wrap_3_args(gxm_XSetBackground_w, gxm_XSetBackground)
Xen_wrap_3_args(gxm_XSetClipMask_w, gxm_XSetClipMask)
Xen_wrap_4_args(gxm_XSetClipOrigin_w, gxm_XSetClipOrigin)
Xen_wrap_7_args(gxm_XSetClipRectangles_w, gxm_XSetClipRectangles)
Xen_wrap_2_args(gxm_XSetCloseDownMode_w, gxm_XSetCloseDownMode)
Xen_wrap_4_args(gxm_XSetCommand_w, gxm_XSetCommand)
Xen_wrap_5_optional_args(gxm_XSetDashes_w, gxm_XSetDashes)
Xen_wrap_3_args(gxm_XSetFillRule_w, gxm_XSetFillRule)
Xen_wrap_3_args(gxm_XSetFillStyle_w, gxm_XSetFillStyle)
Xen_wrap_3_args(gxm_XSetFont_w, gxm_XSetFont)
Xen_wrap_3_args(gxm_XSetFontPath_w, gxm_XSetFontPath)
Xen_wrap_3_args(gxm_XSetForeground_w, gxm_XSetForeground)
Xen_wrap_3_args(gxm_XSetFunction_w, gxm_XSetFunction)
Xen_wrap_3_args(gxm_XSetGraphicsExposures_w, gxm_XSetGraphicsExposures)
Xen_wrap_3_args(gxm_XSetIconName_w, gxm_XSetIconName)
Xen_wrap_4_args(gxm_XSetInputFocus_w, gxm_XSetInputFocus)
Xen_wrap_6_args(gxm_XSetLineAttributes_w, gxm_XSetLineAttributes)
Xen_wrap_2_args(gxm_XSetModifierMapping_w, gxm_XSetModifierMapping)
Xen_wrap_3_args(gxm_XSetPlaneMask_w, gxm_XSetPlaneMask)
Xen_wrap_3_optional_args(gxm_XSetPointerMapping_w, gxm_XSetPointerMapping)
Xen_wrap_5_args(gxm_XSetScreenSaver_w, gxm_XSetScreenSaver)
Xen_wrap_4_args(gxm_XSetSelectionOwner_w, gxm_XSetSelectionOwner)
Xen_wrap_6_args(gxm_XSetState_w, gxm_XSetState)
Xen_wrap_3_args(gxm_XSetStipple_w, gxm_XSetStipple)
Xen_wrap_3_args(gxm_XSetSubwindowMode_w, gxm_XSetSubwindowMode)
Xen_wrap_4_args(gxm_XSetTSOrigin_w, gxm_XSetTSOrigin)
Xen_wrap_3_args(gxm_XSetTile_w, gxm_XSetTile)
Xen_wrap_3_args(gxm_XSetWindowBackground_w, gxm_XSetWindowBackground)
Xen_wrap_3_args(gxm_XSetWindowBackgroundPixmap_w, gxm_XSetWindowBackgroundPixmap)
Xen_wrap_3_args(gxm_XSetWindowBorder_w, gxm_XSetWindowBorder)
Xen_wrap_3_args(gxm_XSetWindowBorderPixmap_w, gxm_XSetWindowBorderPixmap)
Xen_wrap_3_args(gxm_XSetWindowBorderWidth_w, gxm_XSetWindowBorderWidth)
Xen_wrap_3_args(gxm_XSetWindowColormap_w, gxm_XSetWindowColormap)
Xen_wrap_4_args(gxm_XStoreBuffer_w, gxm_XStoreBuffer)
Xen_wrap_3_args(gxm_XStoreBytes_w, gxm_XStoreBytes)
Xen_wrap_3_args(gxm_XStoreColor_w, gxm_XStoreColor)
Xen_wrap_4_args(gxm_XStoreColors_w, gxm_XStoreColors)
Xen_wrap_3_args(gxm_XStoreName_w, gxm_XStoreName)
Xen_wrap_5_args(gxm_XStoreNamedColor_w, gxm_XStoreNamedColor)
Xen_wrap_2_args(gxm_XSync_w, gxm_XSync)
Xen_wrap_3_args(gxm_XTextExtents_w, gxm_XTextExtents)
Xen_wrap_3_args(gxm_XTextWidth_w, gxm_XTextWidth)
Xen_wrap_5_args(gxm_XTranslateCoordinates_w, gxm_XTranslateCoordinates)
Xen_wrap_2_args(gxm_XUndefineCursor_w, gxm_XUndefineCursor)
Xen_wrap_4_args(gxm_XUngrabButton_w, gxm_XUngrabButton)
Xen_wrap_4_args(gxm_XUngrabKey_w, gxm_XUngrabKey)
Xen_wrap_2_args(gxm_XUngrabKeyboard_w, gxm_XUngrabKeyboard)
Xen_wrap_2_args(gxm_XUngrabPointer_w, gxm_XUngrabPointer)
Xen_wrap_1_arg(gxm_XUngrabServer_w, gxm_XUngrabServer)
Xen_wrap_2_args(gxm_XUninstallColormap_w, gxm_XUninstallColormap)
Xen_wrap_2_args(gxm_XUnloadFont_w, gxm_XUnloadFont)
Xen_wrap_2_args(gxm_XUnmapSubwindows_w, gxm_XUnmapSubwindows)
Xen_wrap_2_args(gxm_XUnmapWindow_w, gxm_XUnmapWindow)
Xen_wrap_1_arg(gxm_XVendorRelease_w, gxm_XVendorRelease)
Xen_wrap_9_args(gxm_XWarpPointer_w, gxm_XWarpPointer)
Xen_wrap_1_arg(gxm_XWidthMMOfScreen_w, gxm_XWidthMMOfScreen)
Xen_wrap_1_arg(gxm_XWidthOfScreen_w, gxm_XWidthOfScreen)
Xen_wrap_3_args(gxm_XWindowEvent_w, gxm_XWindowEvent)
Xen_wrap_7_args(gxm_XWriteBitmapFile_w, gxm_XWriteBitmapFile)
Xen_wrap_no_args(gxm_XSupportsLocale_w, gxm_XSupportsLocale)
Xen_wrap_1_arg(gxm_XSetLocaleModifiers_w, gxm_XSetLocaleModifiers)
Xen_wrap_2_args(gxm_XCreateFontSet_w, gxm_XCreateFontSet)
Xen_wrap_2_args(gxm_XFreeFontSet_w, gxm_XFreeFontSet)
Xen_wrap_1_arg(gxm_XFontsOfFontSet_w, gxm_XFontsOfFontSet)
Xen_wrap_1_arg(gxm_XBaseFontNameListOfFontSet_w, gxm_XBaseFontNameListOfFontSet)
Xen_wrap_1_arg(gxm_XLocaleOfFontSet_w, gxm_XLocaleOfFontSet)
Xen_wrap_1_arg(gxm_XContextDependentDrawing_w, gxm_XContextDependentDrawing)
Xen_wrap_1_arg(gxm_XDirectionalDependentDrawing_w, gxm_XDirectionalDependentDrawing)
Xen_wrap_1_arg(gxm_XContextualDrawing_w, gxm_XContextualDrawing)
Xen_wrap_2_args(gxm_XFilterEvent_w, gxm_XFilterEvent)
Xen_wrap_no_args(gxm_XAllocIconSize_w, gxm_XAllocIconSize)
Xen_wrap_no_args(gxm_XAllocStandardColormap_w, gxm_XAllocStandardColormap)
Xen_wrap_no_args(gxm_XAllocWMHints_w, gxm_XAllocWMHints)
Xen_wrap_1_arg(gxm_XClipBox_w, gxm_XClipBox)
Xen_wrap_no_args(gxm_XCreateRegion_w, gxm_XCreateRegion)
Xen_wrap_no_args(gxm_XDefaultString_w, gxm_XDefaultString)
Xen_wrap_3_args(gxm_XDeleteContext_w, gxm_XDeleteContext)
Xen_wrap_1_arg(gxm_XDestroyRegion_w, gxm_XDestroyRegion)
Xen_wrap_1_arg(gxm_XEmptyRegion_w, gxm_XEmptyRegion)
Xen_wrap_2_args(gxm_XEqualRegion_w, gxm_XEqualRegion)
Xen_wrap_3_args(gxm_XFindContext_w, gxm_XFindContext)
Xen_wrap_2_args(gxm_XGetIconSizes_w, gxm_XGetIconSizes)
Xen_wrap_3_args(gxm_XGetRGBColormaps_w, gxm_XGetRGBColormaps)
Xen_wrap_3_args(gxm_XGetVisualInfo_w, gxm_XGetVisualInfo)
Xen_wrap_2_args(gxm_XGetWMHints_w, gxm_XGetWMHints)
Xen_wrap_3_args(gxm_XIntersectRegion_w, gxm_XIntersectRegion)
Xen_wrap_1_arg(gxm_XConvertCase_w, gxm_XConvertCase)
Xen_wrap_1_arg(gxm_XLookupString_w, gxm_XLookupString)
Xen_wrap_4_args(gxm_XMatchVisualInfo_w, gxm_XMatchVisualInfo)
Xen_wrap_3_args(gxm_XOffsetRegion_w, gxm_XOffsetRegion)
Xen_wrap_3_args(gxm_XPointInRegion_w, gxm_XPointInRegion)
Xen_wrap_3_args(gxm_XPolygonRegion_w, gxm_XPolygonRegion)
Xen_wrap_5_args(gxm_XRectInRegion_w, gxm_XRectInRegion)
Xen_wrap_4_args(gxm_XSaveContext_w, gxm_XSaveContext)
Xen_wrap_no_args(gxm_XUniqueContext_w, gxm_XUniqueContext)
Xen_wrap_5_args(gxm_XSetRGBColormaps_w, gxm_XSetRGBColormaps)
Xen_wrap_3_args(gxm_XSetWMHints_w, gxm_XSetWMHints)
Xen_wrap_3_args(gxm_XSetRegion_w, gxm_XSetRegion)
Xen_wrap_8_args(gxm_XSetWMProperties_w, gxm_XSetWMProperties)
Xen_wrap_3_args(gxm_XShrinkRegion_w, gxm_XShrinkRegion)
Xen_wrap_3_args(gxm_XSubtractRegion_w, gxm_XSubtractRegion)
Xen_wrap_3_args(gxm_XUnionRectWithRegion_w, gxm_XUnionRectWithRegion)
Xen_wrap_3_args(gxm_XUnionRegion_w, gxm_XUnionRegion)
Xen_wrap_3_args(gxm_XXorRegion_w, gxm_XXorRegion)

Xen_wrap_1_arg(gxm_DefaultScreen_w, gxm_DefaultScreen)
Xen_wrap_1_arg(gxm_DefaultRootWindow_w, gxm_DefaultRootWindow)
Xen_wrap_1_arg(gxm_QLength_w, gxm_QLength)
Xen_wrap_1_arg(gxm_ScreenCount_w, gxm_ScreenCount)
Xen_wrap_1_arg(gxm_ServerVendor_w, gxm_ServerVendor)
Xen_wrap_1_arg(gxm_ProtocolVersion_w, gxm_ProtocolVersion)
Xen_wrap_1_arg(gxm_ProtocolRevision_w, gxm_ProtocolRevision)
Xen_wrap_1_arg(gxm_VendorRelease_w, gxm_VendorRelease)
Xen_wrap_1_arg(gxm_DisplayString_w, gxm_DisplayString)
Xen_wrap_1_arg(gxm_BitmapUnit_w, gxm_BitmapUnit)
Xen_wrap_1_arg(gxm_BitmapBitOrder_w, gxm_BitmapBitOrder)
Xen_wrap_1_arg(gxm_BitmapPad_w, gxm_BitmapPad)
Xen_wrap_1_arg(gxm_ImageByteOrder_w, gxm_ImageByteOrder)
Xen_wrap_1_arg(gxm_NextRequest_w, gxm_NextRequest)
Xen_wrap_1_arg(gxm_LastKnownRequestProcessed_w, gxm_LastKnownRequestProcessed)
Xen_wrap_1_arg(gxm_DefaultScreenOfDisplay_w, gxm_DefaultScreenOfDisplay)
Xen_wrap_1_arg(gxm_DisplayOfScreen_w, gxm_DisplayOfScreen)
Xen_wrap_1_arg(gxm_RootWindowOfScreen_w, gxm_RootWindowOfScreen)
Xen_wrap_1_arg(gxm_BlackPixelOfScreen_w, gxm_BlackPixelOfScreen)
Xen_wrap_1_arg(gxm_WhitePixelOfScreen_w, gxm_WhitePixelOfScreen)
Xen_wrap_1_arg(gxm_DefaultColormapOfScreen_w, gxm_DefaultColormapOfScreen)
Xen_wrap_1_arg(gxm_DefaultDepthOfScreen_w, gxm_DefaultDepthOfScreen)
Xen_wrap_1_arg(gxm_DefaultGCOfScreen_w, gxm_DefaultGCOfScreen)
Xen_wrap_1_arg(gxm_DefaultVisualOfScreen_w, gxm_DefaultVisualOfScreen)
Xen_wrap_1_arg(gxm_WidthOfScreen_w, gxm_WidthOfScreen)
Xen_wrap_1_arg(gxm_HeightOfScreen_w, gxm_HeightOfScreen)
Xen_wrap_1_arg(gxm_WidthMMOfScreen_w, gxm_WidthMMOfScreen)
Xen_wrap_1_arg(gxm_HeightMMOfScreen_w, gxm_HeightMMOfScreen)
Xen_wrap_1_arg(gxm_PlanesOfScreen_w, gxm_PlanesOfScreen)
Xen_wrap_1_arg(gxm_CellsOfScreen_w, gxm_CellsOfScreen)
Xen_wrap_1_arg(gxm_MinCmapsOfScreen_w, gxm_MinCmapsOfScreen)
Xen_wrap_1_arg(gxm_MaxCmapsOfScreen_w, gxm_MaxCmapsOfScreen)
Xen_wrap_1_arg(gxm_DoesSaveUnders_w, gxm_DoesSaveUnders)
Xen_wrap_1_arg(gxm_DoesBackingStore_w, gxm_DoesBackingStore)
Xen_wrap_1_arg(gxm_EventMaskOfScreen_w, gxm_EventMaskOfScreen)
Xen_wrap_2_args(gxm_RootWindow_w, gxm_RootWindow)
Xen_wrap_2_args(gxm_DefaultVisual_w, gxm_DefaultVisual)
Xen_wrap_2_args(gxm_DefaultGC_w, gxm_DefaultGC)
Xen_wrap_2_args(gxm_BlackPixel_w, gxm_BlackPixel)
Xen_wrap_2_args(gxm_WhitePixel_w, gxm_WhitePixel)
Xen_wrap_2_args(gxm_DisplayWidth_w, gxm_DisplayWidth)
Xen_wrap_2_args(gxm_DisplayHeight_w, gxm_DisplayHeight)
Xen_wrap_2_args(gxm_DisplayWidthMM_w, gxm_DisplayWidthMM)
Xen_wrap_2_args(gxm_DisplayHeightMM_w, gxm_DisplayHeightMM)
Xen_wrap_2_args(gxm_DisplayPlanes_w, gxm_DisplayPlanes)
Xen_wrap_2_args(gxm_DisplayCells_w, gxm_DisplayCells)
Xen_wrap_2_args(gxm_DefaultColormap_w, gxm_DefaultColormap)
Xen_wrap_2_args(gxm_ScreenOfDisplay_w, gxm_ScreenOfDisplay)
Xen_wrap_2_args(gxm_DefaultDepth_w, gxm_DefaultDepth)

Xen_wrap_1_arg(gxm_IsKeypadKey_w, gxm_IsKeypadKey)
Xen_wrap_1_arg(gxm_IsPrivateKeypadKey_w, gxm_IsPrivateKeypadKey)
Xen_wrap_1_arg(gxm_IsCursorKey_w, gxm_IsCursorKey)
Xen_wrap_1_arg(gxm_IsPFKey_w, gxm_IsPFKey)
Xen_wrap_1_arg(gxm_IsFunctionKey_w, gxm_IsFunctionKey)
Xen_wrap_1_arg(gxm_IsMiscFunctionKey_w, gxm_IsMiscFunctionKey)
Xen_wrap_1_arg(gxm_IsModifierKey_w, gxm_IsModifierKey)

Xen_wrap_1_arg(g_is_XButtonEvent_w, g_is_XButtonEvent)
Xen_wrap_1_arg(g_is_XCirculateEvent_w, g_is_XCirculateEvent)
Xen_wrap_1_arg(g_is_XCirculateRequestEvent_w, g_is_XCirculateRequestEvent)
Xen_wrap_1_arg(g_is_XClientMessageEvent_w, g_is_XClientMessageEvent)
Xen_wrap_1_arg(g_is_XColormapEvent_w, g_is_XColormapEvent)
Xen_wrap_1_arg(g_is_XConfigureEvent_w, g_is_XConfigureEvent)
Xen_wrap_1_arg(g_is_XConfigureRequestEvent_w, g_is_XConfigureRequestEvent)
Xen_wrap_1_arg(g_is_XCreateWindowEvent_w, g_is_XCreateWindowEvent)
Xen_wrap_1_arg(g_is_XCrossingEvent_w, g_is_XCrossingEvent)
Xen_wrap_1_arg(g_is_XDestroyWindowEvent_w, g_is_XDestroyWindowEvent)
Xen_wrap_1_arg(g_is_XErrorEvent_w, g_is_XErrorEvent)
Xen_wrap_1_arg(g_is_XExposeEvent_w, g_is_XExposeEvent)
Xen_wrap_1_arg(g_is_XFocusChangeEvent_w, g_is_XFocusChangeEvent)
Xen_wrap_1_arg(g_is_XGraphicsExposeEvent_w, g_is_XGraphicsExposeEvent)
Xen_wrap_1_arg(g_is_XGravityEvent_w, g_is_XGravityEvent)
Xen_wrap_1_arg(g_is_XKeyEvent_w, g_is_XKeyEvent)
Xen_wrap_1_arg(g_is_XKeymapEvent_w, g_is_XKeymapEvent)
Xen_wrap_1_arg(g_is_XMapEvent_w, g_is_XMapEvent)
Xen_wrap_1_arg(g_is_XMapRequestEvent_w, g_is_XMapRequestEvent)
Xen_wrap_1_arg(g_is_XMappingEvent_w, g_is_XMappingEvent)
Xen_wrap_1_arg(g_is_XMotionEvent_w, g_is_XMotionEvent)
Xen_wrap_1_arg(g_is_XNoExposeEvent_w, g_is_XNoExposeEvent)
Xen_wrap_1_arg(g_is_XPropertyEvent_w, g_is_XPropertyEvent)
Xen_wrap_1_arg(g_is_XReparentEvent_w, g_is_XReparentEvent)
Xen_wrap_1_arg(g_is_XResizeRequestEvent_w, g_is_XResizeRequestEvent)
Xen_wrap_1_arg(g_is_XSelectionClearEvent_w, g_is_XSelectionClearEvent)
Xen_wrap_1_arg(g_is_XSelectionEvent_w, g_is_XSelectionEvent)
Xen_wrap_1_arg(g_is_XSelectionRequestEvent_w, g_is_XSelectionRequestEvent)
Xen_wrap_1_arg(g_is_XSetWindowAttributes_w, g_is_XSetWindowAttributes)
Xen_wrap_1_arg(g_is_XUnmapEvent_w, g_is_XUnmapEvent)
Xen_wrap_1_arg(g_is_XVisibilityEvent_w, g_is_XVisibilityEvent)
Xen_wrap_1_arg(g_is_XIconSize_w, g_is_XIconSize)

Xen_wrap_4_optional_args(gxm_XmCreateMessageBox_w, gxm_XmCreateMessageBox)
Xen_wrap_4_optional_args(gxm_XmCreateMessageDialog_w, gxm_XmCreateMessageDialog)
Xen_wrap_4_optional_args(gxm_XmCreateErrorDialog_w, gxm_XmCreateErrorDialog)
Xen_wrap_4_optional_args(gxm_XmCreateInformationDialog_w, gxm_XmCreateInformationDialog)
Xen_wrap_4_optional_args(gxm_XmCreateQuestionDialog_w, gxm_XmCreateQuestionDialog)
Xen_wrap_4_optional_args(gxm_XmCreateWarningDialog_w, gxm_XmCreateWarningDialog)
Xen_wrap_4_optional_args(gxm_XmCreateWorkingDialog_w, gxm_XmCreateWorkingDialog)
Xen_wrap_4_optional_args(gxm_XmCreateTemplateDialog_w, gxm_XmCreateTemplateDialog)
Xen_wrap_2_args(gxm_XmMessageBoxGetChild_w, gxm_XmMessageBoxGetChild)
Xen_wrap_4_optional_args(gxm_XmCreateArrowButtonGadget_w, gxm_XmCreateArrowButtonGadget)
Xen_wrap_4_optional_args(gxm_XmCreateArrowButton_w, gxm_XmCreateArrowButton)
Xen_wrap_4_optional_args(gxm_XmCreateNotebook_w, gxm_XmCreateNotebook)
Xen_wrap_2_args(gxm_XmNotebookGetPageInfo_w, gxm_XmNotebookGetPageInfo)
Xen_wrap_5_args(gxm_XmTransferSetParameters_w, gxm_XmTransferSetParameters)
Xen_wrap_2_args(gxm_XmTransferDone_w, gxm_XmTransferDone)
Xen_wrap_5_args(gxm_XmTransferValue_w, gxm_XmTransferValue)
Xen_wrap_1_arg(gxm_XmTransferStartRequest_w, gxm_XmTransferStartRequest)
Xen_wrap_2_args(gxm_XmTransferSendRequest_w, gxm_XmTransferSendRequest)
Xen_wrap_4_optional_args(gxm_XmCreateComboBox_w, gxm_XmCreateComboBox)
Xen_wrap_4_optional_args(gxm_XmCreateDropDownComboBox_w, gxm_XmCreateDropDownComboBox)
Xen_wrap_4_optional_args(gxm_XmCreateDropDownList_w, gxm_XmCreateDropDownList)
Xen_wrap_4_args(gxm_XmComboBoxAddItem_w, gxm_XmComboBoxAddItem)
Xen_wrap_2_args(gxm_XmComboBoxDeletePos_w, gxm_XmComboBoxDeletePos)
Xen_wrap_2_args(gxm_XmComboBoxSelectItem_w, gxm_XmComboBoxSelectItem)
Xen_wrap_2_args(gxm_XmComboBoxSetItem_w, gxm_XmComboBoxSetItem)
Xen_wrap_1_arg(gxm_XmComboBoxUpdate_w, gxm_XmComboBoxUpdate)
Xen_wrap_4_optional_args(gxm_XmCreateContainer_w, gxm_XmCreateContainer)
Xen_wrap_2_args(gxm_XmContainerGetItemChildren_w, gxm_XmContainerGetItemChildren)
Xen_wrap_1_arg(gxm_XmContainerRelayout_w, gxm_XmContainerRelayout)
Xen_wrap_3_args(gxm_XmContainerReorder_w, gxm_XmContainerReorder)
Xen_wrap_2_args(gxm_XmContainerCut_w, gxm_XmContainerCut)
Xen_wrap_2_args(gxm_XmContainerCopy_w, gxm_XmContainerCopy)
Xen_wrap_1_arg(gxm_XmContainerPaste_w, gxm_XmContainerPaste)
Xen_wrap_2_args(gxm_XmContainerCopyLink_w, gxm_XmContainerCopyLink)
Xen_wrap_1_arg(gxm_XmContainerPasteLink_w, gxm_XmContainerPasteLink)
Xen_wrap_4_optional_args(gxm_XmCreateSpinBox_w, gxm_XmCreateSpinBox)
Xen_wrap_1_arg(gxm_XmSpinBoxValidatePosition_w, gxm_XmSpinBoxValidatePosition)
Xen_wrap_4_optional_args(gxm_XmCreateSimpleSpinBox_w, gxm_XmCreateSimpleSpinBox)
Xen_wrap_3_args(gxm_XmSimpleSpinBoxAddItem_w, gxm_XmSimpleSpinBoxAddItem)
Xen_wrap_2_args(gxm_XmSimpleSpinBoxDeletePos_w, gxm_XmSimpleSpinBoxDeletePos)
Xen_wrap_2_args(gxm_XmSimpleSpinBoxSetItem_w, gxm_XmSimpleSpinBoxSetItem)
Xen_wrap_1_arg(gxm_XmDropSiteRegistered_w, gxm_XmDropSiteRegistered)
Xen_wrap_2_args(gxm_XmTextFieldCopyLink_w, gxm_XmTextFieldCopyLink)
Xen_wrap_1_arg(gxm_XmTextFieldPasteLink_w, gxm_XmTextFieldPasteLink)
Xen_wrap_1_arg(gxm_XmTextGetCenterline_w, gxm_XmTextGetCenterline)
Xen_wrap_3_args(gxm_XmToggleButtonGadgetSetValue_w, gxm_XmToggleButtonGadgetSetValue)
Xen_wrap_4_optional_args(gxm_XmCreateIconGadget_w, gxm_XmCreateIconGadget)
Xen_wrap_4_optional_args(gxm_XmCreateIconHeader_w, gxm_XmCreateIconHeader)
Xen_wrap_3_args(gxm_XmObjectAtPoint_w, gxm_XmObjectAtPoint)
Xen_wrap_4_args(gxm_XmConvertStringToUnits_w, gxm_XmConvertStringToUnits)
Xen_wrap_4_optional_args(gxm_XmCreateGrabShell_w, gxm_XmCreateGrabShell)
Xen_wrap_3_args(gxm_XmToggleButtonSetValue_w, gxm_XmToggleButtonSetValue)
Xen_wrap_1_arg(gxm_XmTextPasteLink_w, gxm_XmTextPasteLink)
Xen_wrap_2_args(gxm_XmTextCopyLink_w, gxm_XmTextCopyLink)
Xen_wrap_7_args(gxm_XmScaleSetTicks_w, gxm_XmScaleSetTicks)
Xen_wrap_3_args(gxm_XmInternAtom_w, gxm_XmInternAtom)
Xen_wrap_2_args(gxm_XmGetAtomName_w, gxm_XmGetAtomName)
Xen_wrap_4_optional_args(gxm_XmCreatePanedWindow_w, gxm_XmCreatePanedWindow)
Xen_wrap_4_optional_args(gxm_XmCreateBulletinBoard_w, gxm_XmCreateBulletinBoard)
Xen_wrap_4_optional_args(gxm_XmCreateBulletinBoardDialog_w, gxm_XmCreateBulletinBoardDialog)
Xen_wrap_4_optional_args(gxm_XmCreateCascadeButtonGadget_w, gxm_XmCreateCascadeButtonGadget)
Xen_wrap_2_args(gxm_XmCascadeButtonGadgetHighlight_w, gxm_XmCascadeButtonGadgetHighlight)
Xen_wrap_4_optional_args(gxm_XmAddProtocols_w, gxm_XmAddProtocols)
Xen_wrap_3_optional_args(gxm_XmAddWMProtocols_w, gxm_XmAddWMProtocols)
Xen_wrap_4_optional_args(gxm_XmRemoveProtocols_w, gxm_XmRemoveProtocols)
Xen_wrap_3_optional_args(gxm_XmRemoveWMProtocols_w, gxm_XmRemoveWMProtocols)
Xen_wrap_5_args(gxm_XmAddProtocolCallback_w, gxm_XmAddProtocolCallback)
Xen_wrap_4_args(gxm_XmAddWMProtocolCallback_w, gxm_XmAddWMProtocolCallback)
Xen_wrap_5_args(gxm_XmRemoveProtocolCallback_w, gxm_XmRemoveProtocolCallback)
Xen_wrap_4_args(gxm_XmRemoveWMProtocolCallback_w, gxm_XmRemoveWMProtocolCallback)
Xen_wrap_3_args(gxm_XmActivateProtocol_w, gxm_XmActivateProtocol)
Xen_wrap_3_args(gxm_XmDeactivateProtocol_w, gxm_XmDeactivateProtocol)
Xen_wrap_2_args(gxm_XmActivateWMProtocol_w, gxm_XmActivateWMProtocol)
Xen_wrap_2_args(gxm_XmDeactivateWMProtocol_w, gxm_XmDeactivateWMProtocol)
Xen_wrap_7_args(gxm_XmSetProtocolHooks_w, gxm_XmSetProtocolHooks)
Xen_wrap_6_args(gxm_XmSetWMProtocolHooks_w, gxm_XmSetWMProtocolHooks)
Xen_wrap_4_optional_args(gxm_XmCreateCascadeButton_w, gxm_XmCreateCascadeButton)
Xen_wrap_2_args(gxm_XmCascadeButtonHighlight_w, gxm_XmCascadeButtonHighlight)
Xen_wrap_4_optional_args(gxm_XmCreatePushButtonGadget_w, gxm_XmCreatePushButtonGadget)
Xen_wrap_4_optional_args(gxm_XmCreatePushButton_w, gxm_XmCreatePushButton)
Xen_wrap_4_optional_args(gxm_XmCreateCommand_w, gxm_XmCreateCommand)
Xen_wrap_2_args(gxm_XmCommandGetChild_w, gxm_XmCommandGetChild)
Xen_wrap_2_args(gxm_XmCommandSetValue_w, gxm_XmCommandSetValue)
Xen_wrap_2_args(gxm_XmCommandAppendValue_w, gxm_XmCommandAppendValue)
Xen_wrap_2_args(gxm_XmCommandError_w, gxm_XmCommandError)
Xen_wrap_4_optional_args(gxm_XmCreateCommandDialog_w, gxm_XmCreateCommandDialog)
Xen_wrap_2_args(gxm_XmMenuPosition_w, gxm_XmMenuPosition)
Xen_wrap_4_optional_args(gxm_XmCreateRowColumn_w, gxm_XmCreateRowColumn)
Xen_wrap_4_optional_args(gxm_XmCreateWorkArea_w, gxm_XmCreateWorkArea)
Xen_wrap_4_optional_args(gxm_XmCreateRadioBox_w, gxm_XmCreateRadioBox)
Xen_wrap_4_optional_args(gxm_XmCreateOptionMenu_w, gxm_XmCreateOptionMenu)
Xen_wrap_1_arg(gxm_XmOptionLabelGadget_w, gxm_XmOptionLabelGadget)
Xen_wrap_1_arg(gxm_XmOptionButtonGadget_w, gxm_XmOptionButtonGadget)
Xen_wrap_4_optional_args(gxm_XmCreateMenuBar_w, gxm_XmCreateMenuBar)
Xen_wrap_4_optional_args(gxm_XmCreatePopupMenu_w, gxm_XmCreatePopupMenu)
Xen_wrap_4_optional_args(gxm_XmCreatePulldownMenu_w, gxm_XmCreatePulldownMenu)
Xen_wrap_1_arg(gxm_XmGetPostedFromWidget_w, gxm_XmGetPostedFromWidget)
Xen_wrap_1_arg(gxm_XmGetTearOffControl_w, gxm_XmGetTearOffControl)
Xen_wrap_2_args(gxm_XmScaleSetValue_w, gxm_XmScaleSetValue)
Xen_wrap_1_arg(gxm_XmScaleGetValue_w, gxm_XmScaleGetValue)
Xen_wrap_4_optional_args(gxm_XmCreateScale_w, gxm_XmCreateScale)
Xen_wrap_5_args(gxm_XmClipboardBeginCopy_w, gxm_XmClipboardBeginCopy)
Xen_wrap_6_args(gxm_XmClipboardStartCopy_w, gxm_XmClipboardStartCopy)
Xen_wrap_7_args(gxm_XmClipboardCopy_w, gxm_XmClipboardCopy)
Xen_wrap_3_args(gxm_XmClipboardEndCopy_w, gxm_XmClipboardEndCopy)
Xen_wrap_3_args(gxm_XmClipboardCancelCopy_w, gxm_XmClipboardCancelCopy)
Xen_wrap_3_args(gxm_XmClipboardWithdrawFormat_w, gxm_XmClipboardWithdrawFormat)
Xen_wrap_6_args(gxm_XmClipboardCopyByName_w, gxm_XmClipboardCopyByName)
Xen_wrap_2_args(gxm_XmClipboardUndoCopy_w, gxm_XmClipboardUndoCopy)
Xen_wrap_2_args(gxm_XmClipboardLock_w, gxm_XmClipboardLock)
Xen_wrap_3_args(gxm_XmClipboardUnlock_w, gxm_XmClipboardUnlock)
Xen_wrap_3_args(gxm_XmClipboardStartRetrieve_w, gxm_XmClipboardStartRetrieve)
Xen_wrap_2_args(gxm_XmClipboardEndRetrieve_w, gxm_XmClipboardEndRetrieve)
Xen_wrap_4_args(gxm_XmClipboardRetrieve_w, gxm_XmClipboardRetrieve)
Xen_wrap_2_args(gxm_XmClipboardInquireCount_w, gxm_XmClipboardInquireCount)
Xen_wrap_4_args(gxm_XmClipboardInquireFormat_w, gxm_XmClipboardInquireFormat)
Xen_wrap_3_args(gxm_XmClipboardInquireLength_w, gxm_XmClipboardInquireLength)
Xen_wrap_3_args(gxm_XmClipboardInquirePendingItems_w, gxm_XmClipboardInquirePendingItems)
Xen_wrap_3_args(gxm_XmClipboardRegisterFormat_w, gxm_XmClipboardRegisterFormat)
Xen_wrap_1_arg(gxm_XmGetXmScreen_w, gxm_XmGetXmScreen)
Xen_wrap_4_optional_args(gxm_XmCreateScrollBar_w, gxm_XmCreateScrollBar)
Xen_wrap_1_arg(gxm_XmScrollBarGetValues_w, gxm_XmScrollBarGetValues)
Xen_wrap_6_args(gxm_XmScrollBarSetValues_w, gxm_XmScrollBarSetValues)
Xen_wrap_4_optional_args(gxm_XmCreateDialogShell_w, gxm_XmCreateDialogShell)
Xen_wrap_4_optional_args(gxm_XmCreateScrolledWindow_w, gxm_XmCreateScrolledWindow)
Xen_wrap_4_args(gxm_XmScrollVisible_w, gxm_XmScrollVisible)
Xen_wrap_2_args(gxm_XmGetDragContext_w, gxm_XmGetDragContext)
Xen_wrap_1_arg(gxm_XmGetXmDisplay_w, gxm_XmGetXmDisplay)
Xen_wrap_2_args(gxm_XmSelectionBoxGetChild_w, gxm_XmSelectionBoxGetChild)
Xen_wrap_4_optional_args(gxm_XmCreateSelectionBox_w, gxm_XmCreateSelectionBox)
Xen_wrap_4_optional_args(gxm_XmCreateSelectionDialog_w, gxm_XmCreateSelectionDialog)
Xen_wrap_4_optional_args(gxm_XmCreatePromptDialog_w, gxm_XmCreatePromptDialog)
Xen_wrap_4_optional_args(gxm_XmDragStart_w, gxm_XmDragStart)
Xen_wrap_1_arg(gxm_XmDragCancel_w, gxm_XmDragCancel)
Xen_wrap_5_args(gxm_XmTargetsAreCompatible_w, gxm_XmTargetsAreCompatible)
Xen_wrap_4_optional_args(gxm_XmCreateSeparatorGadget_w, gxm_XmCreateSeparatorGadget)
Xen_wrap_4_optional_args(gxm_XmCreateDragIcon_w, gxm_XmCreateDragIcon)
Xen_wrap_4_optional_args(gxm_XmCreateSeparator_w, gxm_XmCreateSeparator)
Xen_wrap_4_optional_args(gxm_XmCreateDrawingArea_w, gxm_XmCreateDrawingArea)
Xen_wrap_4_optional_args(gxm_XmCreateDrawnButton_w, gxm_XmCreateDrawnButton)
Xen_wrap_3_optional_args(gxm_XmDropSiteRegister_w, gxm_XmDropSiteRegister)
Xen_wrap_1_arg(gxm_XmDropSiteUnregister_w, gxm_XmDropSiteUnregister)
Xen_wrap_1_arg(gxm_XmDropSiteStartUpdate_w, gxm_XmDropSiteStartUpdate)
Xen_wrap_3_optional_args(gxm_XmDropSiteUpdate_w, gxm_XmDropSiteUpdate)
Xen_wrap_1_arg(gxm_XmDropSiteEndUpdate_w, gxm_XmDropSiteEndUpdate)
Xen_wrap_3_optional_args(gxm_XmDropSiteRetrieve_w, gxm_XmDropSiteRetrieve)
Xen_wrap_1_arg(gxm_XmDropSiteQueryStackingOrder_w, gxm_XmDropSiteQueryStackingOrder)
Xen_wrap_3_args(gxm_XmDropSiteConfigureStackingOrder_w, gxm_XmDropSiteConfigureStackingOrder)
Xen_wrap_3_optional_args(gxm_XmDropTransferStart_w, gxm_XmDropTransferStart)
Xen_wrap_2_args(gxm_XmDropTransferAdd_w, gxm_XmDropTransferAdd)
Xen_wrap_1_arg(gxm_XmTextFieldGetString_w, gxm_XmTextFieldGetString)
Xen_wrap_3_args(gxm_XmTextFieldGetSubstring_w, gxm_XmTextFieldGetSubstring)
Xen_wrap_1_arg(gxm_XmTextFieldGetLastPosition_w, gxm_XmTextFieldGetLastPosition)
Xen_wrap_2_args(gxm_XmTextFieldSetString_w, gxm_XmTextFieldSetString)
Xen_wrap_4_args(gxm_XmTextFieldReplace_w, gxm_XmTextFieldReplace)
Xen_wrap_3_args(gxm_XmTextFieldInsert_w, gxm_XmTextFieldInsert)
Xen_wrap_2_args(gxm_XmTextFieldSetAddMode_w, gxm_XmTextFieldSetAddMode)
Xen_wrap_1_arg(gxm_XmTextFieldGetAddMode_w, gxm_XmTextFieldGetAddMode)
Xen_wrap_1_arg(gxm_XmTextFieldGetEditable_w, gxm_XmTextFieldGetEditable)
Xen_wrap_2_args(gxm_XmTextFieldSetEditable_w, gxm_XmTextFieldSetEditable)
Xen_wrap_1_arg(gxm_XmTextFieldGetMaxLength_w, gxm_XmTextFieldGetMaxLength)
Xen_wrap_2_args(gxm_XmTextFieldSetMaxLength_w, gxm_XmTextFieldSetMaxLength)
Xen_wrap_1_arg(gxm_XmTextFieldGetCursorPosition_w, gxm_XmTextFieldGetCursorPosition)
Xen_wrap_1_arg(gxm_XmTextFieldGetInsertionPosition_w, gxm_XmTextFieldGetInsertionPosition)
Xen_wrap_2_args(gxm_XmTextFieldSetCursorPosition_w, gxm_XmTextFieldSetCursorPosition)
Xen_wrap_2_args(gxm_XmTextFieldSetInsertionPosition_w, gxm_XmTextFieldSetInsertionPosition)
Xen_wrap_1_arg(gxm_XmTextFieldGetSelectionPosition_w, gxm_XmTextFieldGetSelectionPosition)
Xen_wrap_1_arg(gxm_XmTextFieldGetSelection_w, gxm_XmTextFieldGetSelection)
Xen_wrap_1_arg(gxm_XmTextFieldRemove_w, gxm_XmTextFieldRemove)
Xen_wrap_2_args(gxm_XmTextFieldCopy_w, gxm_XmTextFieldCopy)
Xen_wrap_2_args(gxm_XmTextFieldCut_w, gxm_XmTextFieldCut)
Xen_wrap_1_arg(gxm_XmTextFieldPaste_w, gxm_XmTextFieldPaste)
Xen_wrap_2_args(gxm_XmTextFieldClearSelection_w, gxm_XmTextFieldClearSelection)
Xen_wrap_4_args(gxm_XmTextFieldSetSelection_w, gxm_XmTextFieldSetSelection)
Xen_wrap_3_args(gxm_XmTextFieldXYToPos_w, gxm_XmTextFieldXYToPos)
Xen_wrap_2_args(gxm_XmTextFieldPosToXY_w, gxm_XmTextFieldPosToXY)
Xen_wrap_2_args(gxm_XmTextFieldShowPosition_w, gxm_XmTextFieldShowPosition)
Xen_wrap_4_args(gxm_XmTextFieldSetHighlight_w, gxm_XmTextFieldSetHighlight)
Xen_wrap_1_arg(gxm_XmTextFieldGetBaseline_w, gxm_XmTextFieldGetBaseline)
Xen_wrap_4_optional_args(gxm_XmCreateTextField_w, gxm_XmCreateTextField)
Xen_wrap_2_args(gxm_XmFileSelectionBoxGetChild_w, gxm_XmFileSelectionBoxGetChild)
Xen_wrap_2_args(gxm_XmFileSelectionDoSearch_w, gxm_XmFileSelectionDoSearch)
Xen_wrap_4_optional_args(gxm_XmCreateFileSelectionBox_w, gxm_XmCreateFileSelectionBox)
Xen_wrap_4_optional_args(gxm_XmCreateFileSelectionDialog_w, gxm_XmCreateFileSelectionDialog)
Xen_wrap_4_args(gxm_XmTextSetHighlight_w, gxm_XmTextSetHighlight)
Xen_wrap_4_optional_args(gxm_XmCreateScrolledText_w, gxm_XmCreateScrolledText)
Xen_wrap_4_optional_args(gxm_XmCreateText_w, gxm_XmCreateText)
Xen_wrap_3_args(gxm_XmTextGetSubstring_w, gxm_XmTextGetSubstring)
Xen_wrap_1_arg(gxm_XmTextGetString_w, gxm_XmTextGetString)
Xen_wrap_1_arg(gxm_XmTextGetLastPosition_w, gxm_XmTextGetLastPosition)
Xen_wrap_2_args(gxm_XmTextSetString_w, gxm_XmTextSetString)
Xen_wrap_4_args(gxm_XmTextReplace_w, gxm_XmTextReplace)
Xen_wrap_3_args(gxm_XmTextInsert_w, gxm_XmTextInsert)
Xen_wrap_2_args(gxm_XmTextSetAddMode_w, gxm_XmTextSetAddMode)
Xen_wrap_1_arg(gxm_XmTextGetAddMode_w, gxm_XmTextGetAddMode)
Xen_wrap_1_arg(gxm_XmTextGetEditable_w, gxm_XmTextGetEditable)
Xen_wrap_2_args(gxm_XmTextSetEditable_w, gxm_XmTextSetEditable)
Xen_wrap_1_arg(gxm_XmTextGetMaxLength_w, gxm_XmTextGetMaxLength)
Xen_wrap_2_args(gxm_XmTextSetMaxLength_w, gxm_XmTextSetMaxLength)
Xen_wrap_1_arg(gxm_XmTextGetTopCharacter_w, gxm_XmTextGetTopCharacter)
Xen_wrap_2_args(gxm_XmTextSetTopCharacter_w, gxm_XmTextSetTopCharacter)
Xen_wrap_1_arg(gxm_XmTextGetCursorPosition_w, gxm_XmTextGetCursorPosition)
Xen_wrap_1_arg(gxm_XmTextGetInsertionPosition_w, gxm_XmTextGetInsertionPosition)
Xen_wrap_2_args(gxm_XmTextSetInsertionPosition_w, gxm_XmTextSetInsertionPosition)
Xen_wrap_2_args(gxm_XmTextSetCursorPosition_w, gxm_XmTextSetCursorPosition)
Xen_wrap_1_arg(gxm_XmTextRemove_w, gxm_XmTextRemove)
Xen_wrap_2_args(gxm_XmTextCopy_w, gxm_XmTextCopy)
Xen_wrap_2_args(gxm_XmTextCut_w, gxm_XmTextCut)
Xen_wrap_1_arg(gxm_XmTextPaste_w, gxm_XmTextPaste)
Xen_wrap_1_arg(gxm_XmTextGetSelection_w, gxm_XmTextGetSelection)
Xen_wrap_4_args(gxm_XmTextSetSelection_w, gxm_XmTextSetSelection)
Xen_wrap_2_args(gxm_XmTextClearSelection_w, gxm_XmTextClearSelection)
Xen_wrap_1_arg(gxm_XmTextGetSelectionPosition_w, gxm_XmTextGetSelectionPosition)
Xen_wrap_3_args(gxm_XmTextXYToPos_w, gxm_XmTextXYToPos)
Xen_wrap_2_args(gxm_XmTextPosToXY_w, gxm_XmTextPosToXY)
Xen_wrap_1_arg(gxm_XmTextGetSource_w, gxm_XmTextGetSource)
Xen_wrap_4_args(gxm_XmTextSetSource_w, gxm_XmTextSetSource)
Xen_wrap_2_args(gxm_XmTextShowPosition_w, gxm_XmTextShowPosition)
Xen_wrap_2_args(gxm_XmTextScroll_w, gxm_XmTextScroll)
Xen_wrap_1_arg(gxm_XmTextGetBaseline_w, gxm_XmTextGetBaseline)
Xen_wrap_1_arg(gxm_XmTextDisableRedisplay_w, gxm_XmTextDisableRedisplay)
Xen_wrap_1_arg(gxm_XmTextEnableRedisplay_w, gxm_XmTextEnableRedisplay)
Xen_wrap_4_args(gxm_XmTextFindString_w, gxm_XmTextFindString)
Xen_wrap_4_optional_args(gxm_XmCreateForm_w, gxm_XmCreateForm)
Xen_wrap_4_optional_args(gxm_XmCreateFormDialog_w, gxm_XmCreateFormDialog)
Xen_wrap_4_optional_args(gxm_XmCreateFrame_w, gxm_XmCreateFrame)
Xen_wrap_1_arg(gxm_XmToggleButtonGadgetGetState_w, gxm_XmToggleButtonGadgetGetState)
Xen_wrap_3_args(gxm_XmToggleButtonGadgetSetState_w, gxm_XmToggleButtonGadgetSetState)
Xen_wrap_4_optional_args(gxm_XmCreateToggleButtonGadget_w, gxm_XmCreateToggleButtonGadget)
Xen_wrap_1_arg(gxm_XmToggleButtonGetState_w, gxm_XmToggleButtonGetState)
Xen_wrap_3_args(gxm_XmToggleButtonSetState_w, gxm_XmToggleButtonSetState)
Xen_wrap_4_optional_args(gxm_XmCreateToggleButton_w, gxm_XmCreateToggleButton)
Xen_wrap_4_optional_args(gxm_XmCreateLabelGadget_w, gxm_XmCreateLabelGadget)
Xen_wrap_4_optional_args(gxm_XmCreateLabel_w, gxm_XmCreateLabel)
Xen_wrap_1_arg(gxm_XmIsMotifWMRunning_w, gxm_XmIsMotifWMRunning)
Xen_wrap_3_args(gxm_XmListAddItem_w, gxm_XmListAddItem)
Xen_wrap_4_args(gxm_XmListAddItems_w, gxm_XmListAddItems)
Xen_wrap_4_args(gxm_XmListAddItemsUnselected_w, gxm_XmListAddItemsUnselected)
Xen_wrap_3_args(gxm_XmListAddItemUnselected_w, gxm_XmListAddItemUnselected)
Xen_wrap_2_args(gxm_XmListDeleteItem_w, gxm_XmListDeleteItem)
Xen_wrap_3_optional_args(gxm_XmListDeleteItems_w, gxm_XmListDeleteItems)
Xen_wrap_3_optional_args(gxm_XmListDeletePositions_w, gxm_XmListDeletePositions)
Xen_wrap_2_args(gxm_XmListDeletePos_w, gxm_XmListDeletePos)
Xen_wrap_3_args(gxm_XmListDeleteItemsPos_w, gxm_XmListDeleteItemsPos)
Xen_wrap_1_arg(gxm_XmListDeleteAllItems_w, gxm_XmListDeleteAllItems)
Xen_wrap_4_args(gxm_XmListReplaceItems_w, gxm_XmListReplaceItems)
Xen_wrap_4_args(gxm_XmListReplaceItemsPos_w, gxm_XmListReplaceItemsPos)
Xen_wrap_4_args(gxm_XmListReplaceItemsUnselected_w, gxm_XmListReplaceItemsUnselected)
Xen_wrap_4_args(gxm_XmListReplaceItemsPosUnselected_w, gxm_XmListReplaceItemsPosUnselected)
Xen_wrap_4_args(gxm_XmListReplacePositions_w, gxm_XmListReplacePositions)
Xen_wrap_3_args(gxm_XmListSelectItem_w, gxm_XmListSelectItem)
Xen_wrap_3_args(gxm_XmListSelectPos_w, gxm_XmListSelectPos)
Xen_wrap_2_args(gxm_XmListDeselectItem_w, gxm_XmListDeselectItem)
Xen_wrap_2_args(gxm_XmListDeselectPos_w, gxm_XmListDeselectPos)
Xen_wrap_1_arg(gxm_XmListDeselectAllItems_w, gxm_XmListDeselectAllItems)
Xen_wrap_2_args(gxm_XmListSetPos_w, gxm_XmListSetPos)
Xen_wrap_2_args(gxm_XmListSetBottomPos_w, gxm_XmListSetBottomPos)
Xen_wrap_2_args(gxm_XmListSetItem_w, gxm_XmListSetItem)
Xen_wrap_2_args(gxm_XmListSetBottomItem_w, gxm_XmListSetBottomItem)
Xen_wrap_2_args(gxm_XmListSetAddMode_w, gxm_XmListSetAddMode)
Xen_wrap_2_args(gxm_XmListItemExists_w, gxm_XmListItemExists)
Xen_wrap_2_args(gxm_XmListItemPos_w, gxm_XmListItemPos)
Xen_wrap_1_arg(gxm_XmListGetKbdItemPos_w, gxm_XmListGetKbdItemPos)
Xen_wrap_2_args(gxm_XmListSetKbdItemPos_w, gxm_XmListSetKbdItemPos)
Xen_wrap_2_args(gxm_XmListYToPos_w, gxm_XmListYToPos)
Xen_wrap_2_args(gxm_XmListPosToBounds_w, gxm_XmListPosToBounds)
Xen_wrap_2_args(gxm_XmListGetMatchPos_w, gxm_XmListGetMatchPos)
Xen_wrap_2_args(gxm_XmListSetHorizPos_w, gxm_XmListSetHorizPos)
Xen_wrap_1_arg(gxm_XmListUpdateSelectedList_w, gxm_XmListUpdateSelectedList)
Xen_wrap_2_args(gxm_XmListPosSelected_w, gxm_XmListPosSelected)
Xen_wrap_4_optional_args(gxm_XmCreateList_w, gxm_XmCreateList)
Xen_wrap_4_optional_args(gxm_XmCreateScrolledList_w, gxm_XmCreateScrolledList)
Xen_wrap_3_args(gxm_XmTranslateKey_w, gxm_XmTranslateKey)
Xen_wrap_4_optional_args(gxm_XmCreateMainWindow_w, gxm_XmCreateMainWindow)
Xen_wrap_2_args(gxm_XmInstallImage_w, gxm_XmInstallImage)
Xen_wrap_1_arg(gxm_XmUninstallImage_w, gxm_XmUninstallImage)
Xen_wrap_4_args(gxm_XmGetPixmap_w, gxm_XmGetPixmap)
Xen_wrap_5_args(gxm_XmGetPixmapByDepth_w, gxm_XmGetPixmapByDepth)
Xen_wrap_2_args(gxm_XmDestroyPixmap_w, gxm_XmDestroyPixmap)
Xen_wrap_1_arg(gxm_XmUpdateDisplay_w, gxm_XmUpdateDisplay)
Xen_wrap_1_arg(gxm_XmWidgetGetBaselines_w, gxm_XmWidgetGetBaselines)
Xen_wrap_2_args(gxm_XmRegisterSegmentEncoding_w, gxm_XmRegisterSegmentEncoding)
Xen_wrap_1_arg(gxm_XmMapSegmentEncoding_w, gxm_XmMapSegmentEncoding)
Xen_wrap_1_arg(gxm_XmCvtCTToXmString_w, gxm_XmCvtCTToXmString)
Xen_wrap_1_arg(gxm_XmCvtXmStringToCT_w, gxm_XmCvtXmStringToCT)
Xen_wrap_5_args(gxm_XmConvertUnits_w, gxm_XmConvertUnits)
Xen_wrap_4_optional_args(gxm_XmCreateSimpleMenuBar_w, gxm_XmCreateSimpleMenuBar)
Xen_wrap_4_optional_args(gxm_XmCreateSimplePopupMenu_w, gxm_XmCreateSimplePopupMenu)
Xen_wrap_4_optional_args(gxm_XmCreateSimplePulldownMenu_w, gxm_XmCreateSimplePulldownMenu)
Xen_wrap_4_optional_args(gxm_XmCreateSimpleOptionMenu_w, gxm_XmCreateSimpleOptionMenu)
Xen_wrap_4_optional_args(gxm_XmCreateSimpleRadioBox_w, gxm_XmCreateSimpleRadioBox)
Xen_wrap_4_optional_args(gxm_XmCreateSimpleCheckBox_w, gxm_XmCreateSimpleCheckBox)
Xen_wrap_3_args(gxm_XmVaCreateSimpleMenuBar_w, gxm_XmVaCreateSimpleMenuBar)
Xen_wrap_4_args(gxm_XmVaCreateSimplePopupMenu_w, gxm_XmVaCreateSimplePopupMenu)
Xen_wrap_5_args(gxm_XmVaCreateSimplePulldownMenu_w, gxm_XmVaCreateSimplePulldownMenu)
Xen_wrap_7_args(gxm_XmVaCreateSimpleOptionMenu_w, gxm_XmVaCreateSimpleOptionMenu)
Xen_wrap_5_args(gxm_XmVaCreateSimpleRadioBox_w, gxm_XmVaCreateSimpleRadioBox)
Xen_wrap_4_args(gxm_XmVaCreateSimpleCheckBox_w, gxm_XmVaCreateSimpleCheckBox)
Xen_wrap_3_args(gxm_XmTrackingEvent_w, gxm_XmTrackingEvent)
Xen_wrap_1_arg(gxm_XmSetColorCalculation_w, gxm_XmSetColorCalculation)
Xen_wrap_no_args(gxm_XmGetColorCalculation_w, gxm_XmGetColorCalculation)
Xen_wrap_3_args(gxm_XmGetColors_w, gxm_XmGetColors)
Xen_wrap_2_args(gxm_XmChangeColor_w, gxm_XmChangeColor)
Xen_wrap_2_args(gxm_XmStringCreate_w, gxm_XmStringCreate)
Xen_wrap_1_arg(gxm_XmStringCreateLocalized_w, gxm_XmStringCreateLocalized)
Xen_wrap_1_arg(gxm_XmStringDirectionCreate_w, gxm_XmStringDirectionCreate)
Xen_wrap_no_args(gxm_XmStringSeparatorCreate_w, gxm_XmStringSeparatorCreate)
Xen_wrap_1_arg(gxm_XmStringInitContext_w, gxm_XmStringInitContext)
Xen_wrap_1_arg(gxm_XmStringFreeContext_w, gxm_XmStringFreeContext)
Xen_wrap_2_args(gxm_XmStringConcatAndFree_w, gxm_XmStringConcatAndFree)
Xen_wrap_1_arg(gxm_XmStringIsVoid_w, gxm_XmStringIsVoid)
Xen_wrap_1_arg(gxm_XmStringPeekNextTriple_w, gxm_XmStringPeekNextTriple)
Xen_wrap_1_arg(gxm_XmStringGetNextTriple_w, gxm_XmStringGetNextTriple)
Xen_wrap_3_args(gxm_XmStringComponentCreate_w, gxm_XmStringComponentCreate)
Xen_wrap_7_args(gxm_XmStringUnparse_w, gxm_XmStringUnparse)
Xen_wrap_7_args(gxm_XmStringParseText_w, gxm_XmStringParseText)
Xen_wrap_2_args(gxm_XmStringToXmStringTable_w, gxm_XmStringToXmStringTable)
Xen_wrap_3_args(gxm_XmStringTableToXmString_w, gxm_XmStringTableToXmString)
Xen_wrap_8_args(gxm_XmStringTableUnparse_w, gxm_XmStringTableUnparse)
Xen_wrap_7_args(gxm_XmStringTableParseStringArray_w, gxm_XmStringTableParseStringArray)
Xen_wrap_1_arg(gxm_XmDirectionToStringDirection_w, gxm_XmDirectionToStringDirection)
Xen_wrap_1_arg(gxm_XmStringDirectionToDirection_w, gxm_XmStringDirectionToDirection)
Xen_wrap_4_args(gxm_XmStringGenerate_w, gxm_XmStringGenerate)
Xen_wrap_2_args(gxm_XmStringPutRendition_w, gxm_XmStringPutRendition)
Xen_wrap_2_optional_args(gxm_XmParseMappingCreate_w, gxm_XmParseMappingCreate)
Xen_wrap_3_optional_args(gxm_XmParseMappingSetValues_w, gxm_XmParseMappingSetValues)
Xen_wrap_3_optional_args(gxm_XmParseMappingGetValues_w, gxm_XmParseMappingGetValues)
Xen_wrap_1_arg(gxm_XmParseMappingFree_w, gxm_XmParseMappingFree)
Xen_wrap_2_optional_args(gxm_XmParseTableFree_w, gxm_XmParseTableFree)
Xen_wrap_5_args(gxm_XmStringTableProposeTablist_w, gxm_XmStringTableProposeTablist)
Xen_wrap_2_args(gxm_XmTabSetValue_w, gxm_XmTabSetValue)
Xen_wrap_1_arg(gxm_XmTabGetValues_w, gxm_XmTabGetValues)
Xen_wrap_1_arg(gxm_XmTabFree_w, gxm_XmTabFree)
Xen_wrap_1_arg(gxm_XmTabListFree_w, gxm_XmTabListFree)
Xen_wrap_5_args(gxm_XmTabCreate_w, gxm_XmTabCreate)
Xen_wrap_1_arg(gxm_XmTabListTabCount_w, gxm_XmTabListTabCount)
Xen_wrap_3_optional_args(gxm_XmTabListRemoveTabs_w, gxm_XmTabListRemoveTabs)
Xen_wrap_4_optional_args(gxm_XmTabListReplacePositions_w, gxm_XmTabListReplacePositions)
Xen_wrap_2_args(gxm_XmTabListGetTab_w, gxm_XmTabListGetTab)
Xen_wrap_3_args(gxm_XmTabListCopy_w, gxm_XmTabListCopy)
Xen_wrap_4_args(gxm_XmTabListInsertTabs_w, gxm_XmTabListInsertTabs)
Xen_wrap_3_args(gxm_XmRenderTableCvtFromProp_w, gxm_XmRenderTableCvtFromProp)
Xen_wrap_2_args(gxm_XmRenderTableCvtToProp_w, gxm_XmRenderTableCvtToProp)
Xen_wrap_3_optional_args(gxm_XmRenditionUpdate_w, gxm_XmRenditionUpdate)
Xen_wrap_3_optional_args(gxm_XmRenditionRetrieve_w, gxm_XmRenditionRetrieve)
Xen_wrap_1_arg(gxm_XmRenditionFree_w, gxm_XmRenditionFree)
Xen_wrap_4_optional_args(gxm_XmRenditionCreate_w, gxm_XmRenditionCreate)
Xen_wrap_3_optional_args(gxm_XmRenderTableGetRenditions_w, gxm_XmRenderTableGetRenditions)
Xen_wrap_2_args(gxm_XmRenderTableGetRendition_w, gxm_XmRenderTableGetRendition)
Xen_wrap_1_arg(gxm_XmRenderTableGetTags_w, gxm_XmRenderTableGetTags)
Xen_wrap_1_arg(gxm_XmRenderTableFree_w, gxm_XmRenderTableFree)
Xen_wrap_3_optional_args(gxm_XmRenderTableCopy_w, gxm_XmRenderTableCopy)
Xen_wrap_3_optional_args(gxm_XmRenderTableRemoveRenditions_w, gxm_XmRenderTableRemoveRenditions)
Xen_wrap_4_args(gxm_XmRenderTableAddRenditions_w, gxm_XmRenderTableAddRenditions)
Xen_wrap_2_args(gxm_XmStringConcat_w, gxm_XmStringConcat)
Xen_wrap_1_arg(gxm_XmStringCopy_w, gxm_XmStringCopy)
Xen_wrap_2_args(gxm_XmStringCompare_w, gxm_XmStringCompare)
Xen_wrap_1_arg(gxm_XmStringEmpty_w, gxm_XmStringEmpty)
Xen_wrap_2_args(gxm_XmStringHasSubstring_w, gxm_XmStringHasSubstring)
Xen_wrap_1_arg(gxm_XmStringFree_w, gxm_XmStringFree)
Xen_wrap_2_args(gxm_XmStringBaseline_w, gxm_XmStringBaseline)
Xen_wrap_2_args(gxm_XmStringWidth_w, gxm_XmStringWidth)
Xen_wrap_2_args(gxm_XmStringHeight_w, gxm_XmStringHeight)
Xen_wrap_2_args(gxm_XmStringExtent_w, gxm_XmStringExtent)
Xen_wrap_1_arg(gxm_XmStringLineCount_w, gxm_XmStringLineCount)
Xen_wrap_any_args(gxm_XmStringDraw_w, gxm_XmStringDraw)
Xen_wrap_any_args(gxm_XmStringDrawImage_w, gxm_XmStringDrawImage)
Xen_wrap_any_args(gxm_XmStringDrawUnderline_w, gxm_XmStringDrawUnderline)

Xen_wrap_1_arg(gxm_XmGetDestination_w, gxm_XmGetDestination)
Xen_wrap_1_arg(gxm_XmIsTraversable_w, gxm_XmIsTraversable)
Xen_wrap_1_arg(gxm_XmGetVisibility_w, gxm_XmGetVisibility)
Xen_wrap_1_arg(gxm_XmGetTabGroup_w, gxm_XmGetTabGroup)
Xen_wrap_1_arg(gxm_XmGetFocusWidget_w, gxm_XmGetFocusWidget)
Xen_wrap_2_args(gxm_XmProcessTraversal_w, gxm_XmProcessTraversal)
Xen_wrap_4_optional_args(gxm_XmCreateMenuShell_w, gxm_XmCreateMenuShell)

Xen_wrap_1_arg(gxm_XmIsMessageBox_w, gxm_XmIsMessageBox)
Xen_wrap_1_arg(gxm_XmIsArrowButtonGadget_w, gxm_XmIsArrowButtonGadget)
Xen_wrap_1_arg(gxm_XmIsArrowButton_w, gxm_XmIsArrowButton)
Xen_wrap_1_arg(gxm_XmCvtXmStringToByteStream_w, gxm_XmCvtXmStringToByteStream)
Xen_wrap_1_arg(gxm_XmCvtByteStreamToXmString_w, gxm_XmCvtByteStreamToXmString)
Xen_wrap_1_arg(gxm_XmStringByteStreamLength_w, gxm_XmStringByteStreamLength)
Xen_wrap_1_arg(gxm_XmIsNotebook_w, gxm_XmIsNotebook)
Xen_wrap_1_arg(gxm_XmIsComboBox_w, gxm_XmIsComboBox)
Xen_wrap_1_arg(gxm_XmIsContainer_w, gxm_XmIsContainer)
Xen_wrap_1_arg(gxm_XmIsGrabShell_w, gxm_XmIsGrabShell)
Xen_wrap_1_arg(gxm_XmIsIconGadget_w, gxm_XmIsIconGadget)
Xen_wrap_1_arg(gxm_XmIsIconHeader_w, gxm_XmIsIconHeader)
Xen_wrap_1_arg(gxm_XmIsPanedWindow_w, gxm_XmIsPanedWindow)
Xen_wrap_1_arg(gxm_XmIsBulletinBoard_w, gxm_XmIsBulletinBoard)
Xen_wrap_1_arg(gxm_XmIsPrimitive_w, gxm_XmIsPrimitive)
Xen_wrap_1_arg(gxm_XmIsCascadeButtonGadget_w, gxm_XmIsCascadeButtonGadget)
Xen_wrap_1_arg(gxm_XmIsCascadeButton_w, gxm_XmIsCascadeButton)
Xen_wrap_1_arg(gxm_XmIsPushButtonGadget_w, gxm_XmIsPushButtonGadget)
Xen_wrap_1_arg(gxm_XmIsPushButton_w, gxm_XmIsPushButton)
Xen_wrap_1_arg(gxm_XmIsCommand_w, gxm_XmIsCommand)
Xen_wrap_1_arg(gxm_XmIsRowColumn_w, gxm_XmIsRowColumn)
Xen_wrap_1_arg(gxm_XmIsScale_w, gxm_XmIsScale)
Xen_wrap_1_arg(gxm_XmIsScreen_w, gxm_XmIsScreen)
Xen_wrap_1_arg(gxm_XmIsScrollBar_w, gxm_XmIsScrollBar)
Xen_wrap_1_arg(gxm_XmIsDialogShell_w, gxm_XmIsDialogShell)
Xen_wrap_1_arg(gxm_XmIsScrolledWindow_w, gxm_XmIsScrolledWindow)
Xen_wrap_1_arg(gxm_XmIsDisplay_w, gxm_XmIsDisplay)
Xen_wrap_1_arg(gxm_XmIsSelectionBox_w, gxm_XmIsSelectionBox)
Xen_wrap_1_arg(gxm_XmIsDragContext_w, gxm_XmIsDragContext)
Xen_wrap_1_arg(gxm_XmIsSeparatorGadget_w, gxm_XmIsSeparatorGadget)
Xen_wrap_1_arg(gxm_XmIsDragIconObjectClass_w, gxm_XmIsDragIconObjectClass)
Xen_wrap_1_arg(gxm_XmIsSeparator_w, gxm_XmIsSeparator)
Xen_wrap_1_arg(gxm_XmIsDrawingArea_w, gxm_XmIsDrawingArea)
Xen_wrap_1_arg(gxm_XmIsDrawnButton_w, gxm_XmIsDrawnButton)
Xen_wrap_1_arg(gxm_XmIsDropSiteManager_w, gxm_XmIsDropSiteManager)
Xen_wrap_1_arg(gxm_XmIsDropTransfer_w, gxm_XmIsDropTransfer)
Xen_wrap_1_arg(gxm_XmIsTextField_w, gxm_XmIsTextField)
Xen_wrap_1_arg(gxm_XmIsFileSelectionBox_w, gxm_XmIsFileSelectionBox)
Xen_wrap_1_arg(gxm_XmIsText_w, gxm_XmIsText)
Xen_wrap_1_arg(gxm_XmIsForm_w, gxm_XmIsForm)
Xen_wrap_1_arg(gxm_XmIsFrame_w, gxm_XmIsFrame)
Xen_wrap_1_arg(gxm_XmIsGadget_w, gxm_XmIsGadget)
Xen_wrap_1_arg(gxm_XmIsToggleButtonGadget_w, gxm_XmIsToggleButtonGadget)
Xen_wrap_1_arg(gxm_XmIsToggleButton_w, gxm_XmIsToggleButton)
Xen_wrap_1_arg(gxm_XmIsLabelGadget_w, gxm_XmIsLabelGadget)
Xen_wrap_1_arg(gxm_XmIsLabel_w, gxm_XmIsLabel)
Xen_wrap_1_arg(gxm_XmIsVendorShell_w, gxm_XmIsVendorShell)
Xen_wrap_1_arg(gxm_XmIsList_w, gxm_XmIsList)
Xen_wrap_1_arg(gxm_XmIsMainWindow_w, gxm_XmIsMainWindow)
Xen_wrap_1_arg(gxm_XmIsManager_w, gxm_XmIsManager)
Xen_wrap_1_arg(gxm_XmIsMenuShell_w, gxm_XmIsMenuShell)
Xen_wrap_1_arg(gxm_XmListGetSelectedPos_w, gxm_XmListGetSelectedPos)
Xen_wrap_1_arg(gxm_XmWidgetGetDisplayRect_w, gxm_XmWidgetGetDisplayRect)

Xen_wrap_4_args(gxm_XpmCreatePixmapFromData_w, gxm_XpmCreatePixmapFromData)
Xen_wrap_4_args(gxm_XpmCreateDataFromPixmap_w, gxm_XpmCreateDataFromPixmap)
Xen_wrap_4_args(gxm_XpmReadFileToPixmap_w, gxm_XpmReadFileToPixmap)
Xen_wrap_1_arg(gxm_XpmReadFileToXpmImage_w, gxm_XpmReadFileToXpmImage)
Xen_wrap_5_args(gxm_XpmWriteFileFromPixmap_w, gxm_XpmWriteFileFromPixmap)
Xen_wrap_4_args(gxm_XpmCreatePixmapFromXpmImage_w, gxm_XpmCreatePixmapFromXpmImage)
Xen_wrap_4_args(gxm_XpmCreateXpmImageFromPixmap_w, gxm_XpmCreateXpmImageFromPixmap)
Xen_wrap_1_arg(gxm_XpmGetErrorString_w, gxm_XpmGetErrorString)

Xen_wrap_3_args(gxm_XGetPixel_w, gxm_XGetPixel)
Xen_wrap_1_arg(gxm_XDestroyImage_w, gxm_XDestroyImage)
Xen_wrap_4_args(gxm_XPutPixel_w, gxm_XPutPixel)
Xen_wrap_5_args(gxm_XSubImage_w, gxm_XSubImage)
Xen_wrap_2_args(gxm_XAddPixel_w, gxm_XAddPixel)

Xen_wrap_1_arg(g_is_XtAppContext_w, g_is_XtAppContext)
Xen_wrap_1_arg(g_is_XtRequestId_w, g_is_XtRequestId)
Xen_wrap_1_arg(g_is_XtWorkProcId_w, g_is_XtWorkProcId)
Xen_wrap_1_arg(g_is_XtInputId_w, g_is_XtInputId)
Xen_wrap_1_arg(g_is_XtIntervalId_w, g_is_XtIntervalId)

Xen_wrap_1_arg(g_is_Screen_w, g_is_Screen)
Xen_wrap_1_arg(g_is_XEvent_w, g_is_XEvent)
Xen_wrap_1_arg(g_is_XRectangle_w, g_is_XRectangle)
Xen_wrap_1_arg(g_is_XArc_w, g_is_XArc)
Xen_wrap_1_arg(g_is_XPoint_w, g_is_XPoint)
Xen_wrap_1_arg(g_is_XSegment_w, g_is_XSegment)
Xen_wrap_1_arg(g_is_XColor_w, g_is_XColor)
Xen_wrap_1_arg(g_is_Atom_w, g_is_Atom)
Xen_wrap_1_arg(g_is_Colormap_w, g_is_Colormap)
Xen_wrap_1_arg(g_is_XModifierKeymap_w, g_is_XModifierKeymap)
Xen_wrap_1_arg(g_is_Depth_w, g_is_Depth)
Xen_wrap_1_arg(g_is_Display_w, g_is_Display)
Xen_wrap_1_arg(g_is_Font_w, g_is_Font)
Xen_wrap_1_arg(g_is_GC_w, g_is_GC)
Xen_wrap_1_arg(g_is_KeySym_w, g_is_KeySym)
Xen_wrap_1_arg(g_is_Pixel_w, g_is_Pixel)
Xen_wrap_1_arg(g_is_Pixmap_w, g_is_Pixmap)
Xen_wrap_1_arg(g_is_Region_w, g_is_Region)
Xen_wrap_1_arg(g_is_Time_w, g_is_Time)
Xen_wrap_1_arg(g_is_Visual_w, g_is_Visual)
Xen_wrap_1_arg(g_is_Window_w, g_is_Window)
Xen_wrap_1_arg(g_is_Widget_w, g_is_Widget)
Xen_wrap_1_arg(g_is_XmStringContext_w, g_is_XmStringContext)
Xen_wrap_1_arg(g_is_XFontProp_w, g_is_XFontProp)
Xen_wrap_1_arg(g_is_XFontSet_w, g_is_XFontSet)
Xen_wrap_1_arg(g_is_XFontStruct_w, g_is_XFontStruct)
Xen_wrap_1_arg(g_is_XGCValues_w, g_is_XGCValues)
Xen_wrap_1_arg(g_is_XImage_w, g_is_XImage)
Xen_wrap_1_arg(g_is_XVisualInfo_w, g_is_XVisualInfo)
Xen_wrap_1_arg(g_is_XWMHints_w, g_is_XWMHints)
Xen_wrap_1_arg(g_is_XWindowAttributes_w, g_is_XWindowAttributes)
Xen_wrap_1_arg(g_is_XWindowChanges_w, g_is_XWindowChanges)
Xen_wrap_1_arg(g_is_KeyCode_w, g_is_KeyCode)
Xen_wrap_1_arg(g_is_XContext_w, g_is_XContext)
Xen_wrap_1_arg(g_is_XCharStruct_w, g_is_XCharStruct)
Xen_wrap_1_arg(g_is_XTextItem_w, g_is_XTextItem)
Xen_wrap_1_arg(g_is_XStandardColormap_w, g_is_XStandardColormap)
Xen_wrap_1_arg(g_is_Cursor_w, g_is_Cursor)
Xen_wrap_1_arg(g_is_WidgetClass_w, g_is_WidgetClass)
Xen_wrap_1_arg(g_is_XmString_w, g_is_XmString)
Xen_wrap_1_arg(g_is_XmTab_w, g_is_XmTab)
Xen_wrap_1_arg(g_is_XmRendition_w, g_is_XmRendition)
Xen_wrap_1_arg(g_is_XmRenderTable_w, g_is_XmRenderTable)
Xen_wrap_1_arg(g_is_XmTabList_w, g_is_XmTabList)
Xen_wrap_1_arg(g_is_XmParseMapping_w, g_is_XmParseMapping)
Xen_wrap_1_arg(g_is_XmTextSource_w, g_is_XmTextSource)

Xen_wrap_1_arg(g_is_XpmAttributes_w, g_is_XpmAttributes)
Xen_wrap_1_arg(g_is_XpmImage_w, g_is_XpmImage)
Xen_wrap_1_arg(g_is_XpmColorSymbol_w, g_is_XpmColorSymbol)

#if WITH_EDITRES
Xen_wrap_4_args(gxm_XEditResCheckMessages_w, gxm_XEditResCheckMessages)
#endif

#if HAVE_XSHAPEQUERYEXTENSION
Xen_wrap_1_arg(gxm_XShapeQueryExtension_w, gxm_XShapeQueryExtension)
Xen_wrap_1_arg(gxm_XShapeQueryVersion_w, gxm_XShapeQueryVersion)
Xen_wrap_2_args(gxm_XShapeQueryExtents_w, gxm_XShapeQueryExtents)
Xen_wrap_3_args(gxm_XShapeGetRectangles_w, gxm_XShapeGetRectangles)
Xen_wrap_5_args(gxm_XShapeOffsetShape_w, gxm_XShapeOffsetShape)
Xen_wrap_7_args(gxm_XShapeCombineRegion_w, gxm_XShapeCombineRegion)
Xen_wrap_7_args(gxm_XShapeCombineMask_w, gxm_XShapeCombineMask)
Xen_wrap_8_args(gxm_XShapeCombineShape_w, gxm_XShapeCombineShape)
Xen_wrap_9_args(gxm_XShapeCombineRectangles_w, gxm_XShapeCombineRectangles)
#endif

Xen_wrap_4_args(gxm_XSegment_w, gxm_XSegment)
Xen_wrap_4_args(gxm_XRectangle_w, gxm_XRectangle)
Xen_wrap_6_optional_args(gxm_XColor_w, gxm_XColor)
Xen_wrap_6_args(gxm_XArc_w, gxm_XArc)
Xen_wrap_7_args(gxm_XWindowChanges_w, gxm_XWindowChanges)
Xen_wrap_any_args(gxm_XSetWindowAttributes_w, gxm_XSetWindowAttributes)
Xen_wrap_2_args(gxm_XPoint_w, gxm_XPoint)
Xen_wrap_4_args(gxm_XTextItem_w, gxm_XTextItem)
Xen_wrap_1_arg(gxm_pixel_w, gxm_pixel)
Xen_wrap_2_args(gxm_set_pixel_w, gxm_set_pixel)
Xen_wrap_1_arg(gxm_red_w, gxm_red)
Xen_wrap_2_args(gxm_set_red_w, gxm_set_red)
Xen_wrap_1_arg(gxm_green_w, gxm_green)
Xen_wrap_2_args(gxm_set_green_w, gxm_set_green)
Xen_wrap_1_arg(gxm_blue_w, gxm_blue)
Xen_wrap_2_args(gxm_set_blue_w, gxm_set_blue)
Xen_wrap_1_arg(gxm_flags_w, gxm_flags)
Xen_wrap_2_args(gxm_set_flags_w, gxm_set_flags)
Xen_wrap_1_arg(gxm_pad_w, gxm_pad)
Xen_wrap_2_args(gxm_set_pad_w, gxm_set_pad)
Xen_wrap_1_arg(gxm_x_w, gxm_x)
Xen_wrap_2_args(gxm_set_x_w, gxm_set_x)
Xen_wrap_1_arg(gxm_y_w, gxm_y)
Xen_wrap_2_args(gxm_set_y_w, gxm_set_y)
Xen_wrap_1_arg(gxm_width_w, gxm_width)
Xen_wrap_2_args(gxm_set_width_w, gxm_set_width)
Xen_wrap_1_arg(gxm_height_w, gxm_height)
Xen_wrap_2_args(gxm_set_height_w, gxm_set_height)
Xen_wrap_1_arg(gxm_angle1_w, gxm_angle1)
Xen_wrap_2_args(gxm_set_angle1_w, gxm_set_angle1)
Xen_wrap_1_arg(gxm_angle2_w, gxm_angle2)
Xen_wrap_2_args(gxm_set_angle2_w, gxm_set_angle2)
Xen_wrap_1_arg(gxm_x1_w, gxm_x1)
Xen_wrap_2_args(gxm_set_x1_w, gxm_set_x1)
Xen_wrap_1_arg(gxm_y1_w, gxm_y1)
Xen_wrap_2_args(gxm_set_y1_w, gxm_set_y1)
Xen_wrap_1_arg(gxm_x2_w, gxm_x2)
Xen_wrap_2_args(gxm_set_x2_w, gxm_set_x2)
Xen_wrap_1_arg(gxm_y2_w, gxm_y2)
Xen_wrap_2_args(gxm_set_y2_w, gxm_set_y2)
Xen_wrap_1_arg(gxm_dashes_w, gxm_dashes)
Xen_wrap_2_args(gxm_set_dashes_w, gxm_set_dashes)
Xen_wrap_1_arg(gxm_dash_offset_w, gxm_dash_offset)
Xen_wrap_2_args(gxm_set_dash_offset_w, gxm_set_dash_offset)
Xen_wrap_1_arg(gxm_clip_mask_w, gxm_clip_mask)
Xen_wrap_2_args(gxm_set_clip_mask_w, gxm_set_clip_mask)
Xen_wrap_1_arg(gxm_clip_y_origin_w, gxm_clip_y_origin)
Xen_wrap_2_args(gxm_set_clip_y_origin_w, gxm_set_clip_y_origin)
Xen_wrap_1_arg(gxm_clip_x_origin_w, gxm_clip_x_origin)
Xen_wrap_2_args(gxm_set_clip_x_origin_w, gxm_set_clip_x_origin)
Xen_wrap_1_arg(gxm_graphics_exposures_w, gxm_graphics_exposures)
Xen_wrap_2_args(gxm_set_graphics_exposures_w, gxm_set_graphics_exposures)
Xen_wrap_1_arg(gxm_subwindow_mode_w, gxm_subwindow_mode)
Xen_wrap_2_args(gxm_set_subwindow_mode_w, gxm_set_subwindow_mode)
Xen_wrap_1_arg(gxm_font_w, gxm_font)
Xen_wrap_2_args(gxm_set_font_w, gxm_set_font)
Xen_wrap_1_arg(gxm_ts_y_origin_w, gxm_ts_y_origin)
Xen_wrap_2_args(gxm_set_ts_y_origin_w, gxm_set_ts_y_origin)
Xen_wrap_1_arg(gxm_ts_x_origin_w, gxm_ts_x_origin)
Xen_wrap_2_args(gxm_set_ts_x_origin_w, gxm_set_ts_x_origin)
Xen_wrap_1_arg(gxm_stipple_w, gxm_stipple)
Xen_wrap_2_args(gxm_set_stipple_w, gxm_set_stipple)
Xen_wrap_1_arg(gxm_tile_w, gxm_tile)
Xen_wrap_2_args(gxm_set_tile_w, gxm_set_tile)
Xen_wrap_1_arg(gxm_arc_mode_w, gxm_arc_mode)
Xen_wrap_2_args(gxm_set_arc_mode_w, gxm_set_arc_mode)
Xen_wrap_1_arg(gxm_fill_rule_w, gxm_fill_rule)
Xen_wrap_2_args(gxm_set_fill_rule_w, gxm_set_fill_rule)
Xen_wrap_1_arg(gxm_fill_style_w, gxm_fill_style)
Xen_wrap_2_args(gxm_set_fill_style_w, gxm_set_fill_style)
Xen_wrap_1_arg(gxm_join_style_w, gxm_join_style)
Xen_wrap_2_args(gxm_set_join_style_w, gxm_set_join_style)
Xen_wrap_1_arg(gxm_cap_style_w, gxm_cap_style)
Xen_wrap_2_args(gxm_set_cap_style_w, gxm_set_cap_style)
Xen_wrap_1_arg(gxm_line_style_w, gxm_line_style)
Xen_wrap_2_args(gxm_set_line_style_w, gxm_set_line_style)
Xen_wrap_1_arg(gxm_line_width_w, gxm_line_width)
Xen_wrap_2_args(gxm_set_line_width_w, gxm_set_line_width)
Xen_wrap_1_arg(gxm_background_w, gxm_background)
Xen_wrap_2_args(gxm_set_background_w, gxm_set_background)
Xen_wrap_1_arg(gxm_foreground_w, gxm_foreground)
Xen_wrap_2_args(gxm_set_foreground_w, gxm_set_foreground)
Xen_wrap_1_arg(gxm_plane_mask_w, gxm_plane_mask)
Xen_wrap_2_args(gxm_set_plane_mask_w, gxm_set_plane_mask)
Xen_wrap_1_arg(gxm_function_w, gxm_function)
Xen_wrap_2_args(gxm_set_function_w, gxm_set_function)
Xen_wrap_1_arg(gxm_delta_w, gxm_delta)
Xen_wrap_2_args(gxm_set_delta_w, gxm_set_delta)
Xen_wrap_1_arg(gxm_nchars_w, gxm_nchars)
Xen_wrap_2_args(gxm_set_nchars_w, gxm_set_nchars)
Xen_wrap_1_arg(gxm_chars_w, gxm_chars)
Xen_wrap_2_args(gxm_set_chars_w, gxm_set_chars)
Xen_wrap_1_arg(gxm_name_w, gxm_name)
Xen_wrap_2_args(gxm_set_name_w, gxm_set_name)
Xen_wrap_1_arg(gxm_depth_w, gxm_depth)
Xen_wrap_2_args(gxm_set_depth_w, gxm_set_depth)
Xen_wrap_1_arg(gxm_visual_w, gxm_visual)
Xen_wrap_2_args(gxm_set_visual_w, gxm_set_visual)

Xen_wrap_1_arg(gxm_display_w, gxm_display)
Xen_wrap_1_arg(gxm_root_w, gxm_root)
Xen_wrap_1_arg(gxm_mwidth_w, gxm_mwidth)
Xen_wrap_1_arg(gxm_mheight_w, gxm_mheight)
Xen_wrap_1_arg(gxm_ndepths_w, gxm_ndepths)
Xen_wrap_1_arg(gxm_depths_w, gxm_depths)
Xen_wrap_1_arg(gxm_root_depth_w, gxm_root_depth)
Xen_wrap_1_arg(gxm_root_visual_w, gxm_root_visual)
Xen_wrap_1_arg(gxm_default_gc_w, gxm_default_gc)
Xen_wrap_1_arg(gxm_cmap_w, gxm_cmap)
Xen_wrap_1_arg(gxm_white_pixel_w, gxm_white_pixel)
Xen_wrap_1_arg(gxm_black_pixel_w, gxm_black_pixel)
Xen_wrap_1_arg(gxm_max_maps_w, gxm_max_maps)
Xen_wrap_1_arg(gxm_min_maps_w, gxm_min_maps)
Xen_wrap_1_arg(gxm_backing_store_w, gxm_backing_store)
Xen_wrap_1_arg(gxm_save_unders_w, gxm_save_unders)
Xen_wrap_1_arg(gxm_root_input_mask_w, gxm_root_input_mask)
Xen_wrap_1_arg(gxm_type_w, gxm_type)
Xen_wrap_1_arg(gxm_serial_w, gxm_serial)
Xen_wrap_1_arg(gxm_send_event_w, gxm_send_event)
Xen_wrap_1_arg(gxm_window_w, gxm_window)
Xen_wrap_1_arg(gxm_subwindow_w, gxm_subwindow)
Xen_wrap_1_arg(gxm_time_w, gxm_time)
Xen_wrap_1_arg(gxm_x_root_w, gxm_x_root)
Xen_wrap_1_arg(gxm_y_root_w, gxm_y_root)
Xen_wrap_1_arg(gxm_state_w, gxm_state)
Xen_wrap_1_arg(gxm_keycode_w, gxm_keycode)
Xen_wrap_1_arg(gxm_same_screen_w, gxm_same_screen)
Xen_wrap_1_arg(gxm_button_w, gxm_button)
Xen_wrap_1_arg(gxm_is_hint_w, gxm_is_hint)
Xen_wrap_1_arg(gxm_mode_w, gxm_mode)
Xen_wrap_1_arg(gxm_detail_w, gxm_detail)
Xen_wrap_1_arg(gxm_focus_w, gxm_focus)
Xen_wrap_1_arg(gxm_key_vector_w, gxm_key_vector)
Xen_wrap_1_arg(gxm_count_w, gxm_count)
Xen_wrap_1_arg(gxm_drawable_w, gxm_drawable)
Xen_wrap_1_arg(gxm_major_code_w, gxm_major_code)
Xen_wrap_1_arg(gxm_minor_code_w, gxm_minor_code)
Xen_wrap_1_arg(gxm_parent_w, gxm_parent)
Xen_wrap_1_arg(gxm_border_width_w, gxm_border_width)
Xen_wrap_1_arg(gxm_override_redirect_w, gxm_override_redirect)
Xen_wrap_1_arg(gxm_event_w, gxm_event)
Xen_wrap_1_arg(gxm_from_configure_w, gxm_from_configure)
Xen_wrap_1_arg(gxm_above_w, gxm_above)
Xen_wrap_1_arg(gxm_value_mask_w, gxm_value_mask)
Xen_wrap_1_arg(gxm_place_w, gxm_place)
Xen_wrap_1_arg(gxm_atom_w, gxm_atom)
Xen_wrap_1_arg(gxm_selection_w, gxm_selection)
Xen_wrap_1_arg(gxm_owner_w, gxm_owner)
Xen_wrap_1_arg(gxm_requestor_w, gxm_requestor)
Xen_wrap_1_arg(gxm_target_w, gxm_target)
Xen_wrap_1_arg(gxm_property_w, gxm_property)
Xen_wrap_1_arg(gxm_new_w, gxm_new)
Xen_wrap_1_arg(gxm_message_type_w, gxm_message_type)
Xen_wrap_1_arg(gxm_format_w, gxm_format)
Xen_wrap_1_arg(gxm_request_w, gxm_request)
Xen_wrap_1_arg(gxm_first_keycode_w, gxm_first_keycode)
Xen_wrap_1_arg(gxm_resourceid_w, gxm_resourceid)
Xen_wrap_1_arg(gxm_error_code_w, gxm_error_code)
Xen_wrap_1_arg(gxm_request_code_w, gxm_request_code)
Xen_wrap_1_arg(gxm_lbearing_w, gxm_lbearing)
Xen_wrap_1_arg(gxm_rbearing_w, gxm_rbearing)
Xen_wrap_1_arg(gxm_ascent_w, gxm_ascent)
Xen_wrap_1_arg(gxm_descent_w, gxm_descent)
Xen_wrap_1_arg(gxm_attributes_w, gxm_attributes)
Xen_wrap_1_arg(gxm_card32_w, gxm_card32)
Xen_wrap_1_arg(gxm_fid_w, gxm_fid)
Xen_wrap_1_arg(gxm_properties_w, gxm_properties)
Xen_wrap_1_arg(gxm_min_bounds_w, gxm_min_bounds)
Xen_wrap_1_arg(gxm_max_bounds_w, gxm_max_bounds)
Xen_wrap_1_arg(gxm_per_char_w, gxm_per_char)
Xen_wrap_1_arg(gxm_input_w, gxm_input)
Xen_wrap_1_arg(gxm_initial_state_w, gxm_initial_state)
Xen_wrap_1_arg(gxm_icon_pixmap_w, gxm_icon_pixmap)
Xen_wrap_1_arg(gxm_icon_window_w, gxm_icon_window)
Xen_wrap_1_arg(gxm_icon_x_w, gxm_icon_x)
Xen_wrap_1_arg(gxm_icon_y_w, gxm_icon_y)
Xen_wrap_1_arg(gxm_icon_mask_w, gxm_icon_mask)
Xen_wrap_1_arg(gxm_window_group_w, gxm_window_group)
Xen_wrap_1_arg(gxm_visualid_w, gxm_visualid)
Xen_wrap_1_arg(gxm_class_w, gxm_class)
Xen_wrap_1_arg(gxm_red_mask_w, gxm_red_mask)
Xen_wrap_1_arg(gxm_green_mask_w, gxm_green_mask)
Xen_wrap_1_arg(gxm_blue_mask_w, gxm_blue_mask)
Xen_wrap_1_arg(gxm_bits_per_rgb_w, gxm_bits_per_rgb)
Xen_wrap_1_arg(gxm_map_entries_w, gxm_map_entries)
Xen_wrap_1_arg(gxm_colormap_size_w, gxm_colormap_size)
Xen_wrap_1_arg(gxm_nvisuals_w, gxm_nvisuals)
Xen_wrap_1_arg(gxm_visuals_w, gxm_visuals)
Xen_wrap_1_arg(gxm_bits_per_pixel_w, gxm_bits_per_pixel)
Xen_wrap_1_arg(gxm_background_pixmap_w, gxm_background_pixmap)
Xen_wrap_1_arg(gxm_background_pixel_w, gxm_background_pixel)
Xen_wrap_1_arg(gxm_border_pixmap_w, gxm_border_pixmap)
Xen_wrap_1_arg(gxm_border_pixel_w, gxm_border_pixel)
Xen_wrap_1_arg(gxm_bit_gravity_w, gxm_bit_gravity)
Xen_wrap_1_arg(gxm_win_gravity_w, gxm_win_gravity)
Xen_wrap_1_arg(gxm_backing_planes_w, gxm_backing_planes)
Xen_wrap_1_arg(gxm_backing_pixel_w, gxm_backing_pixel)
Xen_wrap_1_arg(gxm_save_under_w, gxm_save_under)
Xen_wrap_1_arg(gxm_event_mask_w, gxm_event_mask)
Xen_wrap_1_arg(gxm_do_not_propagate_mask_w, gxm_do_not_propagate_mask)
Xen_wrap_1_arg(gxm_cursor_w, gxm_cursor)
Xen_wrap_1_arg(gxm_map_installed_w, gxm_map_installed)
Xen_wrap_1_arg(gxm_map_state_w, gxm_map_state)
Xen_wrap_1_arg(gxm_all_event_masks_w, gxm_all_event_masks)
Xen_wrap_1_arg(gxm_your_event_mask_w, gxm_your_event_mask)
Xen_wrap_1_arg(gxm_screen_w, gxm_screen)
Xen_wrap_1_arg(gxm_xoffset_w, gxm_xoffset)
Xen_wrap_1_arg(gxm_byte_order_w, gxm_byte_order)
Xen_wrap_1_arg(gxm_bitmap_unit_w, gxm_bitmap_unit)
Xen_wrap_1_arg(gxm_bitmap_bit_order_w, gxm_bitmap_bit_order)
Xen_wrap_1_arg(gxm_bitmap_pad_w, gxm_bitmap_pad)
Xen_wrap_1_arg(gxm_bytes_per_line_w, gxm_bytes_per_line)
Xen_wrap_1_arg(gxm_obdata_w, gxm_obdata)
Xen_wrap_1_arg(gxm_sibling_w, gxm_sibling)
Xen_wrap_1_arg(gxm_stack_mode_w, gxm_stack_mode)
 
Xen_wrap_1_arg(gxm_red_max_w, gxm_red_max)
Xen_wrap_1_arg(gxm_red_mult_w, gxm_red_mult)
Xen_wrap_1_arg(gxm_green_max_w, gxm_green_max)
Xen_wrap_1_arg(gxm_green_mult_w, gxm_green_mult)
Xen_wrap_1_arg(gxm_blue_max_w, gxm_blue_max)
Xen_wrap_1_arg(gxm_blue_mult_w, gxm_blue_mult)
Xen_wrap_1_arg(gxm_base_pixel_w, gxm_base_pixel)
Xen_wrap_1_arg(gxm_killid_w, gxm_killid)
Xen_wrap_1_arg(gxm_data_w, gxm_data)

Xen_wrap_2_args(gxm_set_request_code_w, gxm_set_request_code)
Xen_wrap_2_args(gxm_set_error_code_w, gxm_set_error_code)
Xen_wrap_2_args(gxm_set_first_keycode_w, gxm_set_first_keycode)
Xen_wrap_2_args(gxm_set_request_w, gxm_set_request)
Xen_wrap_2_args(gxm_set_resourceid_w, gxm_set_resourceid)
Xen_wrap_2_args(gxm_set_format_w, gxm_set_format)
Xen_wrap_2_args(gxm_set_message_type_w, gxm_set_message_type)
Xen_wrap_2_args(gxm_set_new_w, gxm_set_new)
Xen_wrap_2_args(gxm_set_property_w, gxm_set_property)
Xen_wrap_2_args(gxm_set_display_w, gxm_set_display)
Xen_wrap_2_args(gxm_set_target_w, gxm_set_target)
Xen_wrap_2_args(gxm_set_requestor_w, gxm_set_requestor)
Xen_wrap_2_args(gxm_set_owner_w, gxm_set_owner)
Xen_wrap_2_args(gxm_set_selection_w, gxm_set_selection)
Xen_wrap_2_args(gxm_set_atom_w, gxm_set_atom)
Xen_wrap_2_args(gxm_set_place_w, gxm_set_place)
Xen_wrap_2_args(gxm_set_value_mask_w, gxm_set_value_mask)
Xen_wrap_2_args(gxm_set_above_w, gxm_set_above)
Xen_wrap_2_args(gxm_set_from_configure_w, gxm_set_from_configure)
Xen_wrap_2_args(gxm_set_event_w, gxm_set_event)
Xen_wrap_2_args(gxm_set_override_redirect_w, gxm_set_override_redirect)
Xen_wrap_2_args(gxm_set_border_width_w, gxm_set_border_width)
Xen_wrap_2_args(gxm_set_parent_w, gxm_set_parent)
Xen_wrap_2_args(gxm_set_minor_code_w, gxm_set_minor_code)
Xen_wrap_2_args(gxm_set_major_code_w, gxm_set_major_code)
Xen_wrap_2_args(gxm_set_drawable_w, gxm_set_drawable)
Xen_wrap_2_args(gxm_set_count_w, gxm_set_count)
Xen_wrap_2_args(gxm_set_key_vector_w, gxm_set_key_vector)
Xen_wrap_2_args(gxm_set_focus_w, gxm_set_focus)
Xen_wrap_2_args(gxm_set_detail_w, gxm_set_detail)
Xen_wrap_2_args(gxm_set_mode_w, gxm_set_mode)
Xen_wrap_2_args(gxm_set_is_hint_w, gxm_set_is_hint)
Xen_wrap_2_args(gxm_set_button_w, gxm_set_button)
Xen_wrap_2_args(gxm_set_same_screen_w, gxm_set_same_screen)
Xen_wrap_2_args(gxm_set_keycode_w, gxm_set_keycode)
Xen_wrap_2_args(gxm_set_state_w, gxm_set_state)
Xen_wrap_2_args(gxm_set_y_root_w, gxm_set_y_root)
Xen_wrap_2_args(gxm_set_x_root_w, gxm_set_x_root)
Xen_wrap_2_args(gxm_set_root_w, gxm_set_root)
Xen_wrap_2_args(gxm_set_time_w, gxm_set_time)
Xen_wrap_2_args(gxm_set_subwindow_w, gxm_set_subwindow)
Xen_wrap_2_args(gxm_set_window_w, gxm_set_window)
Xen_wrap_2_args(gxm_set_send_event_w, gxm_set_send_event)
Xen_wrap_2_args(gxm_set_serial_w, gxm_set_serial)
Xen_wrap_2_args(gxm_set_type_w, gxm_set_type)
Xen_wrap_1_arg(gxm_colormap_w, gxm_colormap)
Xen_wrap_2_args(gxm_set_colormap_w, gxm_set_colormap)

Xen_wrap_2_args(gxm_set_input_w, gxm_set_input)
Xen_wrap_2_args(gxm_set_initial_state_w, gxm_set_initial_state)

Xen_wrap_1_arg(gxm_min_height_w, gxm_min_height)
Xen_wrap_1_arg(gxm_max_height_w, gxm_max_height)
Xen_wrap_1_arg(gxm_min_width_w, gxm_min_width)
Xen_wrap_1_arg(gxm_max_width_w, gxm_max_width)
Xen_wrap_1_arg(gxm_height_inc_w, gxm_height_inc)
Xen_wrap_1_arg(gxm_width_inc_w, gxm_width_inc)

Xen_wrap_2_args(gxm_set_data_w, gxm_set_data)
Xen_wrap_2_args(gxm_set_backing_store_w, gxm_set_backing_store)
Xen_wrap_2_args(gxm_set_background_pixel_w, gxm_set_background_pixel)
Xen_wrap_2_args(gxm_set_border_pixel_w, gxm_set_border_pixel)
Xen_wrap_2_args(gxm_set_bit_gravity_w, gxm_set_bit_gravity)
Xen_wrap_2_args(gxm_set_save_under_w, gxm_set_save_under)
Xen_wrap_2_args(gxm_set_event_mask_w, gxm_set_event_mask)
Xen_wrap_2_args(gxm_set_cursor_w, gxm_set_cursor)

Xen_wrap_2_args(gxm_set_set_w, gxm_set_set)
Xen_wrap_2_args(gxm_set_click_count_w, gxm_set_click_count)
Xen_wrap_2_args(gxm_set_length_w, gxm_set_length)
Xen_wrap_1_arg(gxm_ptr_w, gxm_ptr)
Xen_wrap_2_args(gxm_set_ptr_w, gxm_set_ptr)
Xen_wrap_2_args(gxm_set_reason_w, gxm_set_reason)
Xen_wrap_1_arg(gxm_page_number_w, gxm_page_number)
Xen_wrap_1_arg(gxm_page_widget_w, gxm_page_widget)
Xen_wrap_1_arg(gxm_status_area_widget_w, gxm_status_area_widget)
Xen_wrap_1_arg(gxm_major_tab_widget_w, gxm_major_tab_widget)
Xen_wrap_1_arg(gxm_minor_tab_widget_w, gxm_minor_tab_widget)
Xen_wrap_1_arg(gxm_source_data_w, gxm_source_data)
Xen_wrap_1_arg(gxm_location_data_w, gxm_location_data)
Xen_wrap_1_arg(gxm_parm_w, gxm_parm)
Xen_wrap_1_arg(gxm_parm_format_w, gxm_parm_format)
Xen_wrap_1_arg(gxm_parm_length_w, gxm_parm_length)
Xen_wrap_1_arg(gxm_parm_type_w, gxm_parm_type)
Xen_wrap_1_arg(gxm_transfer_id_w, gxm_transfer_id)
Xen_wrap_1_arg(gxm_destination_data_w, gxm_destination_data)
Xen_wrap_1_arg(gxm_remaining_w, gxm_remaining)
Xen_wrap_1_arg(gxm_item_or_text_w, gxm_item_or_text)
Xen_wrap_1_arg(gxm_auto_selection_type_w, gxm_auto_selection_type)
Xen_wrap_1_arg(gxm_new_outline_state_w, gxm_new_outline_state)
Xen_wrap_1_arg(gxm_prev_page_number_w, gxm_prev_page_number)
Xen_wrap_1_arg(gxm_prev_page_widget_w, gxm_prev_page_widget)
Xen_wrap_1_arg(gxm_rendition_w, gxm_rendition)
Xen_wrap_1_arg(gxm_render_table_w, gxm_render_table)
Xen_wrap_1_arg(gxm_crossed_boundary_w, gxm_crossed_boundary)
Xen_wrap_1_arg(gxm_client_data_w, gxm_client_data)
Xen_wrap_1_arg(gxm_status_w, gxm_status)
Xen_wrap_1_arg(gxm_font_name_w, gxm_font_name)
Xen_wrap_1_arg(gxm_tag_w, gxm_tag)
Xen_wrap_1_arg(gxm_traversal_destination_w, gxm_traversal_destination)
Xen_wrap_1_arg(gxm_dragProtocolStyle_w, gxm_dragProtocolStyle)
Xen_wrap_1_arg(gxm_direction_w, gxm_direction)
Xen_wrap_1_arg(gxm_reason_w, gxm_reason)
Xen_wrap_1_arg(gxm_timeStamp_w, gxm_timeStamp)
Xen_wrap_1_arg(gxm_operation_w, gxm_operation )
Xen_wrap_2_args(gxm_set_operation_w, gxm_set_operation)
Xen_wrap_1_arg(gxm_operations_w, gxm_operations)
Xen_wrap_1_arg(gxm_dropSiteStatus_w, gxm_dropSiteStatus )
Xen_wrap_2_args(gxm_set_dropSiteStatus_w, gxm_set_dropSiteStatus)
Xen_wrap_1_arg(gxm_dropAction_w, gxm_dropAction)
Xen_wrap_1_arg(gxm_iccHandle_w, gxm_iccHandle)
Xen_wrap_1_arg(gxm_completionStatus_w, gxm_completionStatus)
Xen_wrap_1_arg(gxm_dragContext_w, gxm_dragContext)
Xen_wrap_1_arg(gxm_animate_w, gxm_animate)
Xen_wrap_1_arg(gxm_length_w, gxm_length)
Xen_wrap_1_arg(gxm_click_count_w, gxm_click_count)
Xen_wrap_1_arg(gxm_widget_w, gxm_widget)
Xen_wrap_1_arg(gxm_item_position_w, gxm_item_position)
Xen_wrap_1_arg(gxm_callbackstruct_w, gxm_callbackstruct)
Xen_wrap_1_arg(gxm_set_w, gxm_set)
Xen_wrap_1_arg(gxm_item_w, gxm_item)
Xen_wrap_1_arg(gxm_item_length_w, gxm_item_length)
Xen_wrap_1_arg(gxm_selected_items_w, gxm_selected_items)
Xen_wrap_1_arg(gxm_selected_item_count_w, gxm_selected_item_count)
Xen_wrap_1_arg(gxm_selected_item_positions_w, gxm_selected_item_positions)
Xen_wrap_1_arg(gxm_selection_type_w, gxm_selection_type)
Xen_wrap_1_arg(gxm_mask_w, gxm_mask)
Xen_wrap_1_arg(gxm_mask_length_w, gxm_mask_length)
Xen_wrap_1_arg(gxm_dir_w, gxm_dir)
Xen_wrap_1_arg(gxm_dir_length_w, gxm_dir_length)
Xen_wrap_1_arg(gxm_pattern_w, gxm_pattern)
Xen_wrap_1_arg(gxm_pattern_length_w, gxm_pattern_length)
Xen_wrap_1_arg(gxm_position_w, gxm_position)
Xen_wrap_1_arg(gxm_currInsert_w, gxm_currInsert)
Xen_wrap_1_arg(gxm_newInsert_w, gxm_newInsert)
Xen_wrap_1_arg(gxm_startPos_w, gxm_startPos)
Xen_wrap_1_arg(gxm_endPos_w, gxm_endPos)
Xen_wrap_1_arg(gxm_text_w, gxm_text)
Xen_wrap_1_arg(gxm_value_w, gxm_value)
Xen_wrap_2_args(gxm_set_value_w, gxm_set_value)
Xen_wrap_1_arg(gxm_doit_w, gxm_doit)
Xen_wrap_2_args(gxm_set_doit_w, gxm_set_doit)
Xen_wrap_1_arg(gxm_menuToPost_w, gxm_menuToPost)
Xen_wrap_2_args(gxm_set_menuToPost_w, gxm_set_menuToPost)
Xen_wrap_1_arg(gxm_postIt_w, gxm_postIt)
Xen_wrap_2_args(gxm_set_postIt_w, gxm_set_postIt)

Xen_wrap_1_arg(gxm_valuemask_w, gxm_valuemask)
Xen_wrap_2_args(gxm_set_valuemask_w, gxm_set_valuemask)
Xen_wrap_1_arg(gxm_ncolors_w, gxm_ncolors)
Xen_wrap_2_args(gxm_set_ncolors_w, gxm_set_ncolors)
Xen_wrap_1_arg(gxm_cpp_w, gxm_cpp)
Xen_wrap_2_args(gxm_set_cpp_w, gxm_set_cpp)
Xen_wrap_1_arg(gxm_numsymbols_w, gxm_numsymbols)
Xen_wrap_2_args(gxm_set_numsymbols_w, gxm_set_numsymbols)
Xen_wrap_1_arg(gxm_colorsymbols_w, gxm_colorsymbols)
Xen_wrap_2_args(gxm_set_colorsymbols_w, gxm_set_colorsymbols)
Xen_wrap_1_arg(gxm_npixels_w, gxm_npixels)
Xen_wrap_2_args(gxm_set_npixels_w, gxm_set_npixels)
Xen_wrap_1_arg(gxm_y_hotspot_w, gxm_y_hotspot)
Xen_wrap_2_args(gxm_set_y_hotspot_w, gxm_set_y_hotspot)
Xen_wrap_1_arg(gxm_x_hotspot_w, gxm_x_hotspot)
Xen_wrap_2_args(gxm_set_x_hotspot_w, gxm_set_x_hotspot)

Xen_wrap_5_args(gxm_XpmImage_w, gxm_XpmImage)
Xen_wrap_3_args(gxm_XpmColorSymbol_w, gxm_XpmColorSymbol)
Xen_wrap_no_args(gxm_XpmAttributes_w, gxm_XpmAttributes)

#if HAVE_SCHEME
Xen_wrap_2_args(c_to_xen_strings_w, c_to_xen_strings)
Xen_wrap_2_args(c_to_xen_ints_w, c_to_xen_ints)
Xen_wrap_2_args(c_to_xen_atoms_w, c_to_xen_atoms)
Xen_wrap_2_args(c_to_xen_xrectangles_w, c_to_xen_xrectangles)
#endif

static void define_procedures(void)
{
  #define XM_define_procedure(Name, Value, A1, A2, A3, Help) Xen_define_safe_procedure(XM_PREFIX #Name XM_POSTFIX, Value, A1, A2, A3, Help)

  xm_gc_table = Xen_make_vector(1, Xen_false);
  Xen_GC_protect(xm_gc_table);
  xm_protected_size = 512;
  xm_protected = Xen_make_vector(xm_protected_size, Xen_false);
  Xen_vector_set(xm_gc_table, 0, xm_protect