//@HEADER
// ************************************************************************
//
//            NOX: An Object-Oriented Nonlinear Solver Package
//                 Copyright (2002) Sandia Corporation
//
// Under terms of Contract DE-AC04-94AL85000, there is a non-exclusive
// license for use of this work by or on behalf of the U.S. Government.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are
// met:
//
// 1. Redistributions of source code must retain the above copyright
// notice, this list of conditions and the following disclaimer.
//
// 2. Redistributions in binary form must reproduce the above copyright
// notice, this list of conditions and the following disclaimer in the
// documentation and/or other materials provided with the distribution.
//
// 3. Neither the name of the Corporation nor the names of the
// contributors may be used to endorse or promote products derived from
// this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY SANDIA CORPORATION "AS IS" AND ANY
// EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
// PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL SANDIA CORPORATION OR THE
// CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
// EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
// PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
// PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
// LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
// NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
// SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//
// Questions? Contact Roger Pawlowski (rppawlo@sandia.gov) or
// Eric Phipps (etphipp@sandia.gov), Sandia National Laboratories.
// ************************************************************************
//  CVS Information
//  $Source$
//  $Author$
//  $Date$
//  $Revision$
// ************************************************************************
//@HEADER

// Finite Element Problem Class
/* Provides function (F) and Jacobian evaluations for the following equation
 * via a 1D linear finite element discretization with Epetra objects.
 *
 * d2u
 * --- - k * u**2 = 0
 * dx2
 *
 * subject to @ x=0, u=1
 */

#ifndef _NOX_EXAMPLE_EPETRA_NONLINEAR_FINITEELEMENTPROBLEM_H
#define _NOX_EXAMPLE_EPETRA_NONLINEAR_FINITEELEMENTPROBLEM_H

// This is needed to get the FillType enum values from NOX_Epetra_Interface
#include "NOX_Epetra.H"
#include "Teuchos_RCP.hpp"

// Forward Declarations
class Epetra_Comm;
class Epetra_Map;
class Epetra_Vector;
class Epetra_Import;
class Epetra_CrsGraph;
class Epetra_CrsMatrix;

// Flag to tell the evaluate routine what objects to fill
enum FillType { F_ONLY, MATRIX_ONLY, ALL };

// Flag to tell how to order the unknowns - Currently not used
enum StoreUnknowns {Interleaved, Staggered};

// Finite Element Problem Class
class Brusselator {

public:

  // Flag to tell the evaluate routine how to ghost shared information
  enum OverlapType { NODES, ELEMENTS };

  // Constructor
  Brusselator( int NumGlobalUnknowns, Epetra_Comm& Comm,
               OverlapType OType = ELEMENTS );

  // Destructor
  ~Brusselator();

  // Reset problem for next parameter (time) step.
  // For now, this simply updates oldsoln with the given Epetra_Vector
  void reset(const Epetra_Vector& x);

  // Set initial condition for solution vector
  void initializeSoln();

  // Evaluates the function (F) and/or the Jacobian using the solution
  // values in solnVector.
  bool evaluate(NOX::Epetra::Interface::Required::FillType fType,
                const Epetra_Vector *solnVector,
        Epetra_Vector *rhsVector, Epetra_RowMatrix *matrix);

  // Return a reference to the Epetra_Vector with the initial guess
  // that is generated by the Brusselator class.
  Teuchos::RCP<Epetra_Vector> getSolution();

  // Return a reference to the Epetra_Vector with the Jacobian
  // that is generated by the Brusselator class.
  Teuchos::RCP<Epetra_CrsMatrix> getJacobian();

  // Return a reference to the Epetra_Vector with the mesh positions
  Teuchos::RCP<Epetra_Vector> getMesh();

  // Accesor function for time step
  double getdt();

  // Return a reference to the Epetra_Vector with the old solution
  Epetra_Vector& getOldSoln();

  // Return a reference to the Epetra_CrsGraph that is generated by
  // the Brusselator problem class.
  Teuchos::RCP<Epetra_CrsGraph> getGraph();

private:

  // inserts the global column indices into the Graph
  Epetra_CrsGraph& generateGraphUsingNodes(Epetra_CrsGraph& AA);

  // inserts the global column indices into the Graph
  Epetra_CrsGraph& generateGraphUsingElements(Epetra_CrsGraph& AA);

private:

  double xmin;
  double xmax;
  double dx;
  double dt;

  FillType flag;
  OverlapType overlapType;
  Epetra_Map *StandardNodeMap;
  Epetra_Map *OverlapNodeMap;
  Epetra_Map *StandardMap;
  Epetra_Map *OverlapMap;
  Epetra_Import *Importer;
  Epetra_Import *nodeImporter;
  Epetra_Import *ColumnToOverlapImporter;
  Teuchos::RCP<Epetra_Vector> xptr;
  Teuchos::RCP<Epetra_Vector> initialSolution;
  Epetra_Vector *oldSolution;
  Epetra_Vector *rhs;
  Teuchos::RCP<Epetra_CrsGraph> AA;
  Teuchos::RCP<Epetra_CrsMatrix> A;
  Epetra_Comm *Comm;

  int MyPID;              // Process number
  int NumProc;            // Total number of processes

  int NumSpecies;         // Number of unknowns owned by this process
  enum { NUMSPECIES = 2}; // Needed for SGI build
  int NumMyNodes;         // Number of nodes owned by this process
  int NumGlobalNodes;     // Total Number of nodes
  int NumMyUnknowns;      // Number of unknowns owned by this process
  int NumGlobalUnknowns;  // Total Number of unknowns

};
#endif




